import os
from .QueryBuilder import QueryBuilder
from data_import.models import CSV_LOCATION, CSV_PROPERTY, CSV_OBSERVATION # noqa


class TableQueryBuilder(QueryBuilder):

    def get_statements(self):
        stmts = []

        with open(os.path.dirname(__file__) + '/importrecord_table.sql', encoding='utf-8') as sql:
            stmts.append(sql.read())

        stmts.append("""
            create table if not exists {feature_table}
            (
                id         integer generated always as identity constraint {feature_table}_pk primary key,
                name       varchar not null,
                lat        double precision,
                lon        double precision,
                geom       geometry,
                gtype      varchar,
                srid       varchar,
                country    varchar,
                community  varchar,
                import_id  integer not null,
                min_timerecord timestamp,
                max_timerecord timestamp,
                created_at timestamp
                {properties});
        """.format(
            feature_table=self.feature_table,
            properties=self.get_custom_column_defs(CSV_LOCATION)
        ))

        stmts.append("create unique index if not exists unique_{table} on {table} (name, lat, lon);".format(table=self.feature_table))

        stmts.append(""" 
            create table if not exists {property_table}
            (
                id           integer generated always as identity constraint {property_table}_pk primary key,
                name         varchar not null,
                technical_id varchar,
                min_lat      numeric,
                min_lon      numeric,
                max_lat      numeric,
                max_lon      numeric,
                geom         geometry,
                gtype        varchar,
                srid         varchar
                {properties});
        """.format(
            property_table=self.property_table,
            properties=self.get_custom_column_defs(CSV_PROPERTY)
        ))

        stmts.append("create unique index if not exists unique_{table} on {table} (name);".format(table=self.property_table))

        stmts.append("""
            create table if not exists {timeseries_table}
            (
                id          integer generated always as identity constraint {timeseries_table}_pk primary key,
                feature_id  integer not null constraint {timeseries_table}_{feature_table}_id_fk references {feature_table},
                property_id integer not null constraint {timeseries_table}_{property_table}_id_fk references {property_table},
                unit        varchar,
                lat         numeric,
                lon         numeric,
                geom        geometry,
                gtype       varchar,
                srid        varchar,
                min_value   double precision,
                max_value   double precision,
                min_date    timestamp,
                max_date    timestamp);
        """.format(
            timeseries_table=self.timeseries_table,
            feature_table=self.feature_table,
            property_table=self.property_table
        ))

        stmts.append("create unique index if not exists unique_{table} on {table} (feature_id, property_id, unit);".format(table=self.timeseries_table))

        stmts.append("create index if not exists {table}_timeseries_id_index on {table} (feature_id);".format(table=self.timeseries_table))

        stmts.append("""
            create table if not exists {timerecord_table}
            (
                id            integer generated always as identity constraint {timerecord_table}_pk primary key,
                timeseries_id integer not null constraint "{timerecord_table}_{timeseries_table}_id_fk" references {timeseries_table},
                date          timestamp not null,
                value_text    varchar,
                lat           numeric,
                lon           numeric,
                geom          geometry,
                gtype         varchar,
                import_id     integer   not null,
                value_number  double precision,
                srid          varchar
                {properties}
            );
        """.format(
                timerecord_table=self.timerecord_table,
                timeseries_table=self.timeseries_table,
                properties=self.get_custom_column_defs(CSV_OBSERVATION)
        ))

        stmts.append("create index if not exists {table}_timeseries_id_index on {table} (timeseries_id);".format(table=self.timerecord_table))

        tables = [
            "data_import_wfsimportrecord",
            self.feature_table,
            self.property_table,
            self.timeseries_table,
            self.timerecord_table
        ]

        for table in tables:
            stmts.append("alter table {table} owner to {owner};".format(table=table, owner=os.environ.get(
                "GEOSERVER_POSTGRES_USERNAME")))

        return stmts


    def get_custom_column_defs(self, entity_type: str):
        columns = []
        for extra_column in self.extra_columns:
            if extra_column.related_entity == entity_type:
                columns.append(extra_column.col_name + " " + extra_column.type_in_db)

        result = ""
        if len(columns) > 0:
            result = ",\n" + ",\n".join(columns)
        return result
