from .QueryBuilder import QueryBuilder
from data_import.models import CSV_LOCATION, CSV_PROPERTY, CSV_OBSERVATION # noqa

class InsertQueryBuilder(QueryBuilder):

    def get_statements(self):

        stmts = []

        stmts.append("""
            insert into {table} (name, lat, lon, geom, gtype, srid, country, import_id, created_at{custom_columns})
                select 
                    d.location_name,
                    cast(d.lat as double precision),
                    cast(d.lon as double precision),
                    ST_SetSRID(ST_MakePoint(cast(d.lon as double precision), cast(d.lat as double precision)),4326),
                    'Point',
                    '4326',
                    max(d.country),
                    max(d.import_job),
                    now()
                    {custom_column_selects}
                from data_import_wfsimportrecord d
                group by (d.location_name, d.lat, d.lon, d.location_properties)
                on conflict do nothing;
        """.format(
            table=self.feature_table,
            custom_columns=self.get_custom_columns(CSV_LOCATION),
            custom_column_selects=self.get_custom_column_selects(CSV_LOCATION, 'd.location_properties')
        ))

        stmts.append("""
            insert into {table} (name, technical_id, min_lat, min_lon, max_lat, max_lon{custom_columns})
                select 
                    d.property_name, 
                    d.property_id, 
                    MIN(cast(d.lat as double precision)), 
                    MIN(cast(d.lon as double precision)), 
                    MAX(cast(d.lat as double precision)), 
                    MAX(cast(d.lon as double precision))
                    {custom_column_selects}
                from data_import_wfsimportrecord d
                group by (d.property_name, d.property_id, d.property_properties)
                on conflict do nothing;
        """.format(
            table=self.property_table,
            custom_columns=self.get_custom_columns(CSV_PROPERTY),
            custom_column_selects=self.get_custom_column_selects(CSV_PROPERTY, 'd.property_properties')
        ))

        stmts.append("""
            insert into {timeseries_table} (feature_id, property_id, unit, min_value, max_value, min_date, max_date, lat, lon, geom, gtype, srid)
                select
                    (select f.id from {feature_table} f where f.name = d.location_name and f.lat = cast(d.lat as double precision) and f.lon = cast(d.lon as double precision)),
                    (select p.id from {property_table} p where p.name = d.property_name),
                    d.unit,
                    MIN(cast(d.value as double precision)),
                    MAX(cast(d.value as double precision)),
                    MIN(to_timestamp(d.date, 'YYYY-MM-DDThh24:mi:ss')::timestamp without time zone at time zone 'Etc/UTC'),
                    MAX(to_timestamp(d.date, 'YYYY-MM-DDThh24:mi:ss')::timestamp without time zone at time zone 'Etc/UTC'),
                    cast(d.lat as double precision),
                    cast(d.lon as double precision),
                    ST_SetSRID(ST_MakePoint(cast(d.lon as double precision), cast(d.lat as double precision)),4326),
                    'Point',
                    '4326'
                    from data_import_wfsimportrecord d
                    group by (
                        d.location_name,
                        d.lat,
                        d.lon,
                        d.property_name,
                        d.unit
                    )
                on conflict do nothing;
        """.format(
            timeseries_table=self.timeseries_table,
            feature_table=self.feature_table,
            property_table=self.property_table
        ))

        stmts.append("""
            insert into {timerecord_table} (timeseries_id, date, value_text, value_number, lat, lon, geom, gtype, srid, import_id{custom_columns})
                select
                    (select id from {timeseries_table} t where t.feature_id = cf.id and t.property_id = cp.id),
                    to_timestamp(d.date, 'YYYY-MM-DDThh24:mi:ss')::timestamp without time zone at time zone 'Etc/UTC',
                    d.value,
                    cast(d.value as double precision),
                    cast(d.lat as double precision),
                    cast(d.lon as double precision),
                    ST_SetSRID(ST_MakePoint(cast(d.lon as double precision), cast(d.lat as double precision)),4326),
                    'Point',
                    '4326',
                    d.import_job
                    {custom_column_selects}
                    from data_import_wfsimportrecord d
                    inner join {feature_table} cf on d.location_name = cf.name and cast(d.lat as double precision) = cf.lat and cast(d.lon as double precision) = cf.lon
                    inner join {property_table} cp on d.property_name = cp.name;
        """.format(
            timerecord_table=self.timerecord_table,
            feature_table=self.feature_table,
            property_table=self.property_table,
            timeseries_table=self.timeseries_table,
            custom_columns=self.get_custom_columns(CSV_OBSERVATION),
            custom_column_selects=self.get_custom_column_selects(CSV_OBSERVATION, 'd.record_properties')
        ))

        return stmts


    def get_custom_columns(self, entity_type: str):
        result = ""
        for extra_column in self.extra_columns:
            if extra_column.related_entity == entity_type:
                result += ", " + extra_column.col_name
        return result


    def get_custom_column_selects(self, entity_type: str, field: str):
        columns = []
        for extra_column in self.extra_columns:

            if extra_column.related_entity == entity_type:
                value = field + "->>'" + extra_column.col_name + "'"

                if extra_column.type_in_db == 'timestamp':
                    columns.append("to_timestamp({}, 'YYYY-MM-DDThh24:mi:ss')::timestamp without time zone at time zone 'Etc/UTC'".format(value))
                elif extra_column.type_in_db == 'double precision':
                    columns.append("(SELECT CASE WHEN {value}~E'[0-9]+\.?[0-9]*' THEN CAST({value} as double precision) ELSE NULL END)".format(value=value))
                elif extra_column.type_in_db == 'integer':
                    columns.append("(SELECT CASE WHEN {value}~E'[0-9]+' THEN CAST({value} as integer) ELSE NULL END)".format(value=value))
                else:
                    columns.append(value)

        result = ""
        if len(columns) > 0:
            result = ",\n" + ",\n".join(columns)
        return result
