import csv

from data_import.models import CsvParserExtraColumn  # noqa
from data_import.models import WfsImportRecord, CSV_LOCATION, CSV_PROPERTY, CSV_OBSERVATION # noqa

from .sql import TableQueryBuilder, InsertQueryBuilder, UpdateQueryBuilder
from .Importer import Importer
from .utils import create_wfs_data_store_name


class WfsImporter(Importer):

    def run_import_job(self, conn):

        data_store_name = create_wfs_data_store_name(self.job.bucket.name, self.job.s3_file)

        extra_columns = CsvParserExtraColumn.objects.filter(parser=self.job.bucket.wfs_csv_parser)

        print ("Create tables if not exist.")
        try:
            qb = TableQueryBuilder(data_store_name, extra_columns)
            for stmt in qb.get_statements():
                conn.execute(stmt)
        except Exception as e:
            print('Cannot create tables.')
            self.job.validation_error = str(e)
            self.job.is_success = False
            return

        try:
            print("Import raw data.")

            self.import_raw_data(extra_columns)

            print('Insert Features, Properties and TimeSeries.')

            qb = InsertQueryBuilder(data_store_name, extra_columns)
            for stmt in qb.get_statements():
                conn.execute(stmt)

            print('Create BoundingBoxes, set Min- and Max-Values.')

            qb = UpdateQueryBuilder(data_store_name)
            for stmt in qb.get_statements():
                conn.execute(stmt)

            print('Delete raw data.')
            conn.execute("DELETE FROM data_import_wfsimportrecord")

            print('Import done.')
            self.job.is_success = True

        except Exception as e:
            self.job.validation_error = str(e)
            self.job.is_success = False


    def import_raw_data(self, extra_columns: list[CsvParserExtraColumn]):

        p = self.job.bucket.wfs_csv_parser

        with open(self.get_file_name(), newline='') as csvfile:
            reader = csv.reader(csvfile, delimiter=',')

            batch_size = 1000
            batch = []
            count = 0

            if p.skip_first_row:
                next(reader)  # skip header

            for row in reader:

                if len(row) == 0:
                    continue
                if self.is_row_included(row):

                    location_props = {}
                    property_props = {}
                    record_props = {}

                    for ec in extra_columns:
                        if ec.related_entity == CSV_LOCATION:
                            location_props[ec.col_name] = row[ec.col_num]
                        if ec.related_entity == CSV_PROPERTY:
                            property_props[ec.col_name] = row[ec.col_num]
                        if ec.related_entity == CSV_OBSERVATION:
                            record_props[ec.col_name] = row[ec.col_num]

                    if row[p.time_col_num] == 'NA':
                        continue

                    # if is point data
                    # TODO: adapt also for polygon data

                    batch.append(
                        WfsImportRecord(
                            location_name=row[p.station_col_num],
                            lat=row[p.lat_col_num],
                            lon=row[p.lon_col_num],
                            value=row[p.value_col_num],
                            date=row[p.time_col_num],
                            unit=row[p.unit_col_num],
                            property_name=row[p.property_col_num],
                            bucket_name=self.job.bucket.name,
                            import_job=self.job.id,
                            location_id=row[p.location_id_col_num] if p.location_id_col_num else None,
                            property_id=row[p.property_id_col_num] if p.property_id_col_num else None,
                            country=row[p.country_col_num] if p.country_col_num else None,
                            community=row[p.community_col_num] if p.community_col_num else None,
                            location_properties=location_props,
                            property_properties=property_props,
                            record_properties=record_props
                        )
                    )

                    count += 1
                    if count % batch_size == 0:
                        WfsImportRecord.objects.using('geoserver').bulk_create(batch)
                        batch = []
