import os
import requests
import zipfile
from jinja2 import Template
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry  # noqa
from xml.etree import ElementTree
from geoserver.catalog import Catalog  # noqa

from main.models import WmsLayer, WfsLayer # noqa

GEOSERVER_MOSAIC_DIR = "/opt/geoserver/data_dir/mosaic"


class GeoServerApi:

    def __init__(self):

        self.url = os.environ.get("GEOSERVER_URL")

        self.username = os.environ.get("GEOSERVER_USERNAME")
        self.password = os.environ.get("GEOSERVER_PASSWORD")
        self.auth = (self.username, self.password)

        self.cat = Catalog(self.url + "/rest/", self.username, self.password)

        self.logs = []

        retry_strategy = Retry(
            total=3,
            backoff_factor=2,
            allowed_methods=['DELETE', 'GET', 'HEAD', 'OPTIONS', 'PUT', 'POST']
        )
        adapter = HTTPAdapter(max_retries=retry_strategy)
        self.session = requests.Session()
        self.session.mount("https://", adapter)
        self.session.mount("http://", adapter)


    def log(self, log):
        self.logs.append(str(log))


    def create_workspace_if_not_exists(self, workspace_name):

        workspace = self.cat.get_workspace(workspace_name)

        if workspace is None:
            try:
                self.cat.create_workspace(workspace_name, workspace_name)
                self.log("Workspace created: {}".format(workspace_name))
            except Exception as e:
                self.log("Workspace could not be created: {}".format(e))
        else:
            self.log("Workspace already exists. Existing Workspace will be used: {}".format(workspace_name))


    def create_postgis_datastore(self, layer: WfsLayer):
        self.create_workspace_if_not_exists(layer.workspace)
        url = '{0}/rest/workspaces/{1}/datastores'.format(self.url, layer.workspace)
        headers = {'Content-Type': 'application/json'}
        body = {
            'dataStore': {
                'name': layer.data_store_name,
                'connectionParameters': {
                    'entry': [
                        {"@key": "jndiReferenceName", "$": "jdbc/postgres"},
                        {"@key": "dbtype", "$": "postgis"},
                        {"@key": "min connections", "$": 1},
                        {"@key": "preparedStatements", "$": True},
                    ]
                }
            }
        }

        try:
            response = self.session.post(url, auth=self.auth, headers=headers, json=body, timeout=5)
            if response.status_code != 201:
                self.log("Create status not ok:")
                self.log(response.content)
            else:
                self.log("POSTGIS Datastore {} created successfully".format(layer.data_store_name))
        except requests.exceptions.RetryError:
            self.log("Timeout, datastore could not be created: {}".format(layer.data_store_name))
        except Exception as e:
            self.log("Error, datastore could not be created: {}".format(e))

        for table in ["feature", "property", "timerecord", "timeseries"]:
            self._publish_datastore_layer(layer.workspace, layer.data_store_name, layer.data_store_name + "_" + table)


    def _publish_datastore_layer(self, workspace, data_store_name, layer):
        url = "{0}/rest/workspaces/{1}/datastores/{2}/featuretypes".format(self.url, workspace, data_store_name)
        headers = {'Content-Type': 'application/json'}
        body = {
            'featureType': {
                'name': layer,
                'title': layer,
                'nativeName': layer,
                'enabled': True
            }
        }

        try:
            response = self.session.post(url, auth=self.auth, headers=headers, json=body, timeout=5)
            if response.status_code != 201:
                self.log("Create status not ok:")
                self.log(response.content)
            else:
                self.log("Layer {} created successfully".format(layer))
        except requests.exceptions.RetryError:
            self.log("Timeout, layer could not be created: {}".format(layer))
        except Exception as e:
            self.log("Error, layer could not be created: {}".format(e))


    def _configure_datastore_layer_name(self, workspace, data_store, native_name):
        url = "{0}/rest/workspaces/{1}/datastores/{2}/featuretypes/{3}".format(self.url, workspace, data_store, native_name)
        headers = {'Content-Type': 'application/json'}
        body = {
            'featureType': {
                'name': data_store,
                'title': data_store,
                'nativeName': native_name,
                'enabled': True
            }
        }
        try:
            response = self.session.put(url, auth=self.auth, headers=headers, json=body, timeout=5)
            if response.status_code not in [200, 201]:
                raise Exception(response.content)
            else:
                self.log("Setting name for featureType {} successfully".format(data_store))
        except Exception as e:
            self.log("Setting name for featureType {} not ok: {}".format(data_store, e))


    def modify_gwc(self, layer_name, style_name, caching_mode):
        url = '{0}/gwc/rest/seed/{1}.json'.format(self.url, layer_name)
        headers = {"content-type": "application/json"}
        seed_data = {
            "seedRequest": {
                "name": layer_name,
                "gridSetId": "EPSG:4326",
                "zoomStart": 0,
                "zoomStop": 10,
                "type": caching_mode,
                "threadCount": 4,
                "tileFormat": "image/png",
                "parameter": {
                    "STYLES": style_name
                }
            }
        }

        try:
            response = self.session.post(url, auth=self.auth, headers=headers, json=seed_data, timeout=5)
            if response.status_code not in [200, 201]:
                self.log("Create status not ok:")
                self.log(response.content)
            else:
                self.log("GWC created successfully")
        except Exception as e:
            print("GeoWebCache could not be created: {}".format(e))


    def _create_new_style(self, style_xml):
        try:
            headers = {"Content-type": "application/vnd.ogc.sld+xml"}
            url = '{0}/rest/styles?raw=true'.format(self.url)
            response = self.session.post(url=url, auth=self.auth, headers=headers, data=style_xml, timeout=5)
            if response.status_code != 201:
                print('Style could not be created: {}'.format(response.content))
            else:
                print('Style created successfully')
        except Exception as e:
            print('Style could not be filled/updated: {}'.format(str(e)))


    def _update_style(self, style_name, style_xml):
        try:
            headers = {"Content-type": "application/vnd.ogc.sld+xml"}
            url = '{0}/rest/styles/{1}?raw=true'.format(self.url, style_name)
            response = self.session.put(url=url, auth=self.auth, headers=headers, data=style_xml, timeout=5)
            if response.status_code != 200:
                print('Style could not be updated: {}'.format(response.content))
            else:
                print('Style updated successfully')
        except Exception as e:
            print('Style could not be filled/updated: {}'.format(str(e)))


    def create_style(self, style_name, style_xml):
        try:
            url = '{0}/rest/styles/{1}.sld'.format(self.url, style_name)
            response = self.session.get(url=url, auth=self.auth, timeout=5)

            style_xml = style_xml.encode('utf-8')

            if response.status_code != 200:
                print('Style does not exist yet. Style will be added as new.')
                self._create_new_style(style_xml)
            else:
                print('Update existing style')
                self._update_style(style_name, style_xml)
        except Exception as e:
            print('Could not retrieve style: {}'.format(str(e)))


    def add_style(self, style_name, layer_name, default=False):

        headers = {"content-type": "application/json"}
        url = '{0}/rest/layers/{1}/styles?default={2}'.format(self.url, layer_name, default)

        body = {
            "style": {
                "name": style_name,
                "filename": style_name + '.sld'
            }
        }

        try:
            response = self.session.post(url, json=body, auth=self.auth, headers=headers, timeout=5)

            if response.status_code != 201:
                print("Create status not ok: {}".format(response.content))
            else:
                print("Style linked successfully")

        except Exception as e:
            print("Style {0} could not be linked to layer {1}: {2}".format(style_name, layer_name, e))


    def configure_cog_layer(self, workspace, store_name, native_name):
        url = "{0}/rest/workspaces/{1}/coveragestores/{2}/coverages".format(
            self.url, workspace, store_name)
        headers = {"content-type": "application/xml"}
        data = (
            "<coverage>"
            "<name>" + store_name + "</name>"
            "<title>" + store_name  + "</title>"
            "<nativeName>" + native_name + "</nativeName>"
            "<enabled>true</enabled>"
            "</coverage>"
        )
        try:
            response = self.session.post(url, data=data, auth=self.auth, headers=headers, timeout=5)
            if response.status_code not in [200, 201]:
                raise Exception(response.content)
            else:
                print("Creating coverage {} successfully".format(store_name))
        except Exception as e:
            print("Error in creating coverage {}: {}".format(native_name, e))


    def get_wms_timesteps(self, layer: WmsLayer):
        wms_url = '{}/ows?service=wms&version=1.3.0&request=GetCapabilities&namespace={}'.format(self.url, layer.bucket.name)

        try:
            response = self.session.get(wms_url)
            xml = ElementTree.fromstring(response.text)

            for item in xml.findall(".//{http://www.opengis.net/wms}Layer"):
                if item.get("queryable") is not None:

                    name = item.find('{http://www.opengis.net/wms}Name').text
                    if name == layer.layer_name:
                        for dimension in item.findall("{http://www.opengis.net/wms}Dimension"):
                            if dimension.get('name') == 'time':
                                time_steps = dimension.text.strip()
                                self.log("Layer-timesteps fetched")
                                return time_steps.split(',')
            return []

        except Exception as e:
            self.log("Could not fetch timesteps {}: {}".format(layer.layer_name, str(e)))


    def delete_datastore(self, workspace, data_store, store_type='coveragestores'):
        url = "{0}/rest/workspaces/{1}/{2}/{3}?recurse=true".format(self.url, workspace, store_type, data_store)
        try:
            response = self.session.delete(url, auth=self.auth, timeout=5)
            if response.status_code not in [200, 201]:
                self.log("Error in deleting datastore {}: {}".format(data_store, response.content))
            else:
                self.log("Deleting datastore successful: {}".format(data_store))
        except Exception as e:
            self.log("Error in deleting datastore {}: {}".format(data_store, e))


    def add_allowed_url(self, title, allowed_url):

        url = "{}rest/urlchecks".format(self.url)
        headers = {"content-type": "application/xml"}

        body = (
            "<regexUrlCheck>"
            "<name>" + title + "</name>"
            "<description></description>"
            "<enabled>true</enabled>"
            "<regex>^" + allowed_url + ".*$</regex>"
            "</regexUrlCheck>"
        )

        try:
            response = self.session.post(
                url,
                data=body,
                auth=self.auth,
                headers=headers,
                timeout=5
            )
            if response.status_code not in [200, 201]:
                print("Error in setting url-check: {}".format(response.content))
            else:
                print("Adding URL-Check successful: {}".format(allowed_url))

        except Exception as e:
            print("Error in setting url-check: {}".format(e))


    def create_cog_coveragestore(self, workspace, store_name, url):

        self.create_workspace_if_not_exists(workspace)

        if not self.is_coveragestore_existing(workspace, store_name):

            create_url = f"{self.url}/rest/workspaces/{workspace}/coveragestores"
            headers = {'Content-Type': 'application/json'}
            body = {
                "coverageStore": {
                    "name": store_name,
                    "type": "GeoTIFF",
                    "url": url,
                    "enabled": True,
                    "workspace": workspace,
                    "metadata": [
                        {
                            "entry": {
                                "@key": "CogSettings.Key",
                                "cogSettings": {
                                    "useCachingStream": False,
                                    "rangeReaderSettings": "HTTP"
                                }
                            }
                        }
                    ]
                }
            }

            response = self.session.post(create_url, auth=self.auth, headers=headers, json=body)
            print(f"[POST] Create store '{store_name}': {response.status_code}")

            self.configure_cog_layer(workspace, store_name, 'geotiff_coverage')

        else:
            print(f"[✓] Store '{store_name}' already exists")


    def create_flatgeobuf_store(self, workspace, store_name, url):

        self.create_workspace_if_not_exists(workspace)

        if not self.is_datastore_existing(workspace, store_name):
            headers = {'Content-Type': 'application/json'}
            body = {
                'dataStore': {
                    'name': store_name,
                    'connectionParameters': {
                        'entry': [
                            {"@key": "url", "$": url},
                            {"@key": "namespace", "$": workspace},
                        ]
                    },
                    'enabled': True,
                    'type': 'FlatGeobuf'
                }
            }

            create_url = '{0}/rest/workspaces/{1}/datastores'.format(self.url, workspace)

            response = self.session.post(create_url, auth=self.auth, headers=headers, json=body, timeout=5)

            print(f"[POST] Create store '{store_name}': {response.status_code}")

            self._publish_datastore_layer(workspace, store_name, 'shapefile')
            self._configure_datastore_layer_name(workspace, store_name, 'shapefile')
        else:
            print(f"[✓] Store '{store_name}' already exists")


    def is_coveragestore_existing(self, workspace, store_name):
        check_url = f"{self.url}/rest/workspaces/{workspace}/coveragestores/{store_name}"
        response = requests.get(check_url, auth=self.auth)

        return response.status_code != 404


    def is_datastore_existing(self, workspace, store_name):
        check_url = f"{self.url}/rest/workspaces/{workspace}/datastores/{store_name}"
        response = requests.get(check_url, auth=self.auth)

        return response.status_code != 404


    def create_imagemosaic(self, workspace, store_name, first_cog_url):

        self.create_workspace_if_not_exists(workspace)

        # Creating a minimal store object of type ImageMosaic
        store_create_url = f"{self.url}/rest/workspaces/{workspace}/coveragestores"
        store_payload = f"""
        <coverageStore>
          <name>{store_name}</name>
          <type>ImageMosaic</type>
          <enabled>true</enabled>
          <workspace>{workspace}</workspace>
        </coverageStore>
        """
        r_create = self.session.post(store_create_url, auth=self.auth, headers={"Content-type": "text/xml"},
                                 data=store_payload)

        # 1. Create indexer.properties
        indexer_content = f"""Cog=true
    PropertyCollectors=TimestampFileNameExtractorSPI[timeregex](time)
    TimeAttribute=time
    Schema=*the_geom:Polygon,location:String,time:java.util.Date
    CanBeEmpty=true
    Name={store_name}
    """
        with open("indexer.properties", "w") as f:
            f.write(indexer_content)

        # 2. timeregex.properties
        timeregex_content = "regex=[0-9]{4}-[0-9]{2}-[0-9]{2},format=yyyy-MM-dd"
        with open("timeregex.properties", "w") as f:
            f.write(timeregex_content)

        # 3. datastore.properties
        with open("datastore.properties", "w") as f:
            f.write(create_datastore_properties())

        # 4. Zip the config files
        zip_name = f"{store_name}.zip"
        with zipfile.ZipFile(zip_name, "w") as zipf:
            zipf.write("indexer.properties")
            zipf.write("timeregex.properties")
            zipf.write("datastore.properties")

        print(f"[✓] Created zip: {zip_name}")


        # 5. Upload the zipped mosaic config
        url_upload = f"{self.url}/rest/workspaces/{workspace}/coveragestores/{store_name}/file.imagemosaic?configure=none"
        with open(zip_name, 'rb') as f:
            r = self.session.put(url_upload, auth=self.auth, headers={"Content-type": "application/zip"}, data=f)
        print(f"[PUT] Upload mosaic: {r.status_code}")

        # 6. Register the first granule
        url_remote = f"{self.url}/rest/workspaces/{workspace}/coveragestores/{store_name}/remote.imagemosaic"
        r2 = self.session.post(url_remote, auth=self.auth, headers={"Content-type": "text/plain"}, data=first_cog_url)
        print(f"[POST] Register first TIFF: {r2.status_code}")

        # 7. Initialize store
        url_init = f"{self.url}/rest/workspaces/{workspace}/coveragestores/{store_name}/coverages.xml?list=all"
        r3 = requests.get(url_init, auth=self.auth)
        print(f"[GET] Initialize store: {r3.status_code}")

        # 8. Create coverage.xml
        with open(os.path.dirname(__file__) + '/templates/coverage.xml', encoding='utf-8') as template:
            template = Template(template.read())

        with open("coverage.xml", "w") as f:
            f.write(template.render(store_name=store_name))

        # 9. Configure the coverage
        url_coverage = f"{self.url}/rest/workspaces/{workspace}/coveragestores/{store_name}/coverages"
        with open("coverage.xml", "rb") as f:
            r4 = self.session.post(url_coverage, auth=self.auth, headers={"Content-type": "text/xml"}, data=f)
        print(f"[POST] Configure coverage: {r4.status_code}")

        # 10. Cleanup local files
        files_to_remove = [
            "indexer.properties",
            "timeregex.properties",
            "datastore.properties",
            "coverage.xml",
            f"{store_name}.zip"
        ]

        for file in files_to_remove:
            if os.path.exists(file):
                os.remove(file)
                print(f"[🧹] Deleted: {file}")
            else:
                print(f"[!] File not found for cleanup: {file}")


    def add_granule(self, workspace, store_name, url):
        remote_url = f"{self.url}/rest/workspaces/{workspace}/coveragestores/{store_name}/remote.imagemosaic"

        try:
            response = self.session.post(remote_url, auth=self.auth, headers={"Content-type": "text/plain"}, data=url)
            print(f"[POST] Added granule: {url} -> {response.status_code}")
        except requests.exceptions.RetryError:
            self.log("Timeout, granule could not be added: {}".format(url))
        except Exception as e:
            self.log("Error, granule could not be added: {}".format(url))



def create_datastore_properties():

    with open(os.path.dirname(__file__) + '/templates/datastore.properties', encoding='utf-8') as template:
        template = Template(template.read())
    return template.render()
