import os

from django.contrib import admin  # noqa
from django_tabbed_changeform_admin.admin import DjangoTabbedChangeformAdmin    # noqa
from django.contrib.auth.models import Group, User  # noqa
from django import forms  # noqa
from django.core.exceptions import ValidationError  # noqa
from django.db.models import TextField  # noqa

from main.models import Bucket, BucketLog, BucketUser, S3User  # noqa
from main.lib.backend import get_user_groups  # noqa
from ..management.commands._api import GeonetworkApi

from main.lib.s3_utils import *  # noqa

from main.models import S3User, StaEndpoint  # noqa
from data_import.models import CsvParser  # noqa


class BucketUserInlineForm(forms.ModelForm):
    def __init__(self, *args, **kwargs):
        self.s3user = kwargs.pop('s3user')
        super(BucketUserInlineForm, self).__init__(*args, **kwargs)

    def clean(self):
        cleaned_data = super().clean()

        current_s3user = self.s3user

        if 's3user' not in cleaned_data:
            raise ValidationError("A bucket should have at least one assigned user.")

        connected_user = cleaned_data['s3user'].name

        if cleaned_data['id']:
            old_connected_user = self.instance.s3user.name
            if connected_user != current_s3user:
                if old_connected_user == current_s3user:
                    raise ValidationError("You cannot switch your bucket-assignment to another user.")

        if cleaned_data.get('DELETE', False):
            if connected_user == current_s3user:
                raise ValidationError("Your assignment to the bucket can only be removed by another administrator.")

        return cleaned_data


class BucketUserInlineFormset(forms.BaseInlineFormSet):
    model = BucketUser

    def clean(self):
        count_users = 0
        for form in self.forms:
            cleaned_data = form.cleaned_data

            if cleaned_data == {}:
                continue    # form is empty and will not be saved. Skip validation.

            if not cleaned_data.get('DELETE', False):
                count_users += 1

            if count_users == 0:
                raise ValidationError("A bucket should have at least one assigned user.")


class BucketUserInline(admin.TabularInline):
    model = BucketUser
    form = BucketUserInlineForm
    formset = BucketUserInlineFormset
    min_num = 1
    extra = 0
    classes = ["tab-users"]

    def formfield_for_foreignkey(self, db_field, request, **kwargs):

        if not request.user.is_superuser:
            # if not superuser, a user can only select users of his groups
            if db_field.name == "s3user":
                users = User.objects.filter(groups__in=get_user_groups(request))
                kwargs["queryset"] = S3User.objects.filter(user__in=users)

        return super().formfield_for_foreignkey(db_field, request, **kwargs)


class BucketLogInline(admin.StackedInline):
    model = BucketLog
    form = BucketUserInlineForm
    formset = BucketUserInlineFormset
    min_num = 0
    extra = 0
    classes = ["tab-logs"]

    def has_change_permission(self, request, obj=None):
        return False

    def has_add_permission(self, request, obj=None):
        return False

    def has_delete_permission(self, request, obj=None):
        if request.user.is_superuser:
            return True
        return False

class BucketForm(forms.ModelForm):
    model = Bucket

    def clean(self):
        cleaned_data = super().clean()

        group = cleaned_data.get("group")
        sta = cleaned_data.get("sta_endpoint")
        wfs_csv_parser = cleaned_data.get("wfs_csv_parser")
        sta_csv_parser = cleaned_data.get("sta_csv_parser")

        if group and sta:
            if group != sta.group:
                self.add_error("sta_endpoint", "Must select StaEndpoint of same Group as Bucket.")

        if group and wfs_csv_parser:
            if group != wfs_csv_parser.group:
                self.add_error("wfs_csv_parser", "Must select CsvParser of same Group as Bucket.")
            if wfs_csv_parser.target != 'WFS':
                self.add_error("wfs_csv_parser", "Must select CsvParser of type 'WFS'.")

        if group and sta_csv_parser:
            if group != sta_csv_parser.group:
                self.add_error("sta_csv_parser", "Must select CsvParser of same Group as Bucket.")
            if sta_csv_parser.target != 'STA':
                self.add_error("sta_csv_parser", "Must select CsvParser of type 'STA'.")


class BucketAdmin(DjangoTabbedChangeformAdmin, admin.ModelAdmin):
    model = Bucket
    form = BucketForm
    inlines = [BucketUserInline, BucketLogInline]
    list_display = ('name', 'group', 'next_update', 'is_update_running')
    ordering = ('name',)
    formfield_overrides = {
        TextField: dict(widget=forms.Textarea(attrs=dict(readonly=True, rows=25, cols=90)))
    }

    fieldsets = [
        (None, {
            'fields': ('name', 'next_update', 'is_update_running', 'update_interval'),
            'classes': ('tab-basic',),
        }),
        (None, {
            'fields': ('wms_is_integration_active', 'wms_file_suffix', 'wms_delete_integrated_layers_if_file_not_longer_exists'),
            'classes': ('tab-wms',),
        }),
        (None, {
            'fields': ('wfs_is_integration_active', 'wfs_csv_parser', ),
            'classes': ('tab-wfs',),
        }),
        (None, {
            'fields': ('sta_is_integration_active', 'sta_endpoint', 'sta_csv_parser', ),
            'classes': ('tab-sta',),
        }),
        (None, {
            'fields': ('shp_is_integration_active', 'shp_file_suffix', 'shp_delete_integrated_layers_if_file_not_longer_exists'),
            'classes': ('tab-shp',),
        }),
        (None, {
            'fields': ('group', 'public_folder', ('quota', 'size_unit')),
            'classes': ('tab-admin',),
        }),
    ]

    tabs = [
        ("Basic Information", ["tab-basic"]),
        ("WMS", ["tab-wms"]),
        ("WFS", ["tab-wfs"]),
        ("STA", ["tab-sta"]),
        ("Shapefile", ["tab-shp"]),
        ("Users", ["tab-users"]),
        ("Admin", ["tab-admin"]),
        ("Logs", ["tab-logs"])
    ]


    def get_formset_kwargs(self, request, obj, inline, prefix):
        try:
            s3user = S3User.objects.get(user=request.user)
            s3user = s3user.name
        except(S3User.DoesNotExist):
            s3user = ""
        return {
            **super().get_formset_kwargs(request, obj, inline, prefix),
            "form_kwargs": {"s3user": s3user},
        }


    def get_readonly_fields(self, request, obj=None):
        fields = ['is_update_running']

        if request.user.is_superuser:
            if not os.environ.get("MANAGE_S3_BUCKETS", False):
                fields += ["public_folder", "quota", "size_unit"]

            if obj:
                return fields + ["name", "public_folder"]
            else:
                return fields
        else:
            return fields + ["name", "group", "public_folder", "quota", "size_unit"]


    def formfield_for_foreignkey(self, db_field, request, **kwargs):

        if not request.user.is_superuser:
            # if not superuser, a user can only select parsers of his groups
            if db_field.name == "wfs_csv_parser":
                kwargs["queryset"] = CsvParser.objects.filter(group__in=get_user_groups(request))
            if db_field.name == "sta_csv_parser":
                kwargs["queryset"] = CsvParser.objects.filter(group__in=get_user_groups(request))
            # if not superuser, a user can only select StaEndpoints of his groups
            if db_field.name == "sta_endpoint":
                kwargs["queryset"] = StaEndpoint.objects.filter(group__in=get_user_groups(request))

        return super().formfield_for_foreignkey(db_field, request, **kwargs)


    def get_queryset(self, request):
        # if not superuser, a user can see only his buckets
        qs = super().get_queryset(request)

        if request.user.is_superuser:
            return qs
        return qs.filter(group__in=get_user_groups(request))


    def has_change_permission(self, request, obj=None):
        return has_admin_permission(request, obj)


    def has_delete_permission(self, request, obj=None):
        return has_admin_permission(request, obj)


    def save_model(self, request, obj: Bucket, form, change):
        # if os.environ.get("MANAGE_S3_BUCKETS", False):
        #     if obj.id is None:
        #         try:
        #             create_bucket(obj.name)
        #             if obj.public_folder:
        #                 create_public_folder_in_bucket(obj.name)
        #                 upload_file(obj.name, "public/MINIO_README.txt", "main/fixtures/MINIO_README.txt")
        #         except Exception as e:
        #             print('Cannot create Bucket: ' + str(e))
        #
        #     try:
        #         set_quota(obj.name, str(obj.quota) + obj.size_unit)
        #     except Exception as e:
        #         print('Cannot set Quota for bucket: ' + str(e))
        #
        #     if obj.geonetwork_group is None:
        #         try:
        #             geonetwork_api = GeonetworkApi()
        #             obj.geonetwork_group = geonetwork_api.create_group(obj.name)
        #         except Exception as e:
        #             print('Cannot create Geonetwork-group: ' + str(e))

        super().save_model(request, obj, form, change)


    def save_related(self, request, form, formset, change):
        super(BucketAdmin, self).save_related(request, form, formset, change)

        bucket = form.instance
        bucket_users = BucketUser.objects.filter(bucket=bucket).select_related('s3user')

        for bucket_user in bucket_users:
            try:
                if bucket_user.has_write_permission:
                    give_write_access_to_bucket(bucket.name, bucket_user.s3user)
                else:
                    remove_access_to_bucket(bucket.name, bucket_user.s3user)    # if changed from write to read, remove write access
                    give_read_access_to_bucket(bucket.name, bucket_user.s3user)

            except Exception as e:
                print('Cannot assign user {} to bucket {}: {}'.format(bucket_user.s3user.name, bucket.name, str(e)))


def has_admin_permission(request, bucket: Bucket):
    if request.user.is_superuser:
        return True
    try:
        bucket_user = BucketUser.objects.get(bucket=bucket, s3user__user=request.user)
        if bucket_user.is_bucket_admin:
            return True
    except:
        return False
