
from django.db import models # noqa
from django.contrib.postgres.fields import ArrayField    # noqa


OBJECT_TYPES = (
    ('Point', 'Point'),
    ('Polygon', 'Polygon'),
)

PROPERTY_TYPES = (
    ('text', 'free Text-Field'),
    ('integer', 'integer field'),
    ('decimal', 'decimal field'),
    ('dropdown', 'list of options'),
)

OPERAND_TYPES = (
    ('+', '+'),
    ('-', '-'),
    ('*', '*'),
    ('/', '/'),
    ('<', '<'),
    ('>', '>')
)

BRACKET_TYPES = (
    ('(', '('),
    (')', ')'),
)


class DrawingTool(models.Model):
    project = models.ForeignKey("main.Project", on_delete=models.CASCADE)
    name_de = models.CharField(max_length=100)
    name_en = models.CharField(max_length=100)
    max_num_per_user = models.PositiveIntegerField(default=5)
    group_label_de = models.CharField(max_length=40, default="Projekt")
    group_label_en = models.CharField(max_length=40, default="Project")
    submit_label_de = models.CharField(max_length=40, default="Bericht erstellen")
    submit_label_en = models.CharField(max_length=40, default="Request report")
    is_active = models.BooleanField(default=True)

    def __str__(self):
        return self.name_de

    class Meta:
        verbose_name = "Drawing-Tool"
        verbose_name_plural = "Drawing-Tools"

class Object(models.Model):
    drawing_tool = models.ForeignKey(DrawingTool, on_delete=models.CASCADE)
    name_de = models.CharField(max_length=100)
    name_en = models.CharField(max_length=100)
    image = models.URLField(max_length=4000, null=True, blank=True)
    max_num = models.PositiveIntegerField(default=20)
    type = models.CharField(max_length=20, choices=OBJECT_TYPES)
    min_dist = models.DecimalField(max_digits=12, decimal_places=7, null=True, blank=True)
    max_dist = models.DecimalField(max_digits=12, decimal_places=7, null=True, blank=True)
    min_area = models.DecimalField(max_digits=12, decimal_places=7, null=True, blank=True)
    max_area = models.DecimalField(max_digits=12, decimal_places=7, null=True, blank=True)

    def __str__(self):
        return self.name_de

class ObjectProperty(models.Model):
    object = models.ForeignKey(Object, on_delete=models.CASCADE)
    variable = models.CharField(max_length=1)
    name_de = models.CharField(max_length=100)
    name_en = models.CharField(max_length=100)
    property_type = models.CharField(max_length=20, choices=PROPERTY_TYPES)
    min_value = models.IntegerField(null=True, blank=True)
    max_value = models.IntegerField(null=True, blank=True)
    default_value = models.IntegerField(null=True, blank=True)
    options = ArrayField(models.CharField(max_length=100), blank=True, null=True, default=list)
    unit = models.CharField(max_length=5, null=True, blank=True)

    def __str__(self):
        return self.name_de

    class Meta:
        constraints = [
            models.UniqueConstraint(fields=['object', 'variable'], name='unique_object_var')
        ]
        verbose_name = "Object-Property"
        verbose_name_plural = "Object-Properties"


class ObjectProcess(models.Model):
    object = models.ForeignKey(Object, on_delete=models.CASCADE)
    label_de = models.CharField(max_length=1000)
    label_en = models.CharField(max_length=1000)
    process = models.ForeignKey("main.Process", on_delete=models.CASCADE)
    area_layer = models.ForeignKey("main.AreaLayer", on_delete=models.CASCADE)


class PropertyConstraint(models.Model):
    property = models.ForeignKey(ObjectProperty, on_delete=models.CASCADE)
    message_de = models.CharField(max_length=1000)
    message_en = models.CharField(max_length=1000)

    def __str__(self):
        result = ""
        for e in ConstraintElement.objects.filter(constraint=self).order_by('position'):
            result += str(e)
        return result

    class Meta:
        verbose_name = "Property-Constraint"
        verbose_name_plural = "Property-Constraints"


class ConstraintElement(models.Model):
    constraint = models.ForeignKey(PropertyConstraint, on_delete=models.CASCADE)
    position = models.PositiveIntegerField()
    num_value = models.DecimalField(max_digits=12, decimal_places=7, null=True, blank=True)
    obj_value = models.ForeignKey(ObjectProperty, on_delete=models.CASCADE, null=True, blank=True)
    operand = models.CharField(max_length=1, choices=OPERAND_TYPES, null=True, blank=True)
    bracket = models.CharField(max_length=1, choices=BRACKET_TYPES, null=True, blank=True)

    class Meta:
        constraints = [
            models.UniqueConstraint(fields=['constraint', 'position'], name='unique_constraint_pos')
        ]

    def __str__(self):
        if self.num_value:
            return str(self.num_value)
        if self.obj_value:
            return self.obj_value.variable
        if self.bracket:
            return str(self.bracket)
        if self.operand:
            return str(self.operand)
        return ''


# class DrawingToolRecord
# drawing_tool (=> Anlagenplanung)
# identifier (maybe "sXf3A")
# json_config (number of Objects and their properties)