from django.contrib import admin  # noqa
from django import forms # noqa

from drawing_tool.models import Object, ObjectProperty, PropertyConstraint, ConstraintElement  # noqa
from django.core.exceptions import ValidationError # noqa

from main.lib.backend import get_user_groups    # noqa


class ConstraintElementInlineForm(forms.ModelForm):

    def clean(self):
        cleaned_data = super().clean()

        if not cleaned_data.get('DELETE', False):
            values = []
            for field in ['num_value', 'obj_value', 'operand', 'bracket']:
                if cleaned_data.get(field) is not None:
                    values.append(str(cleaned_data[field]))

            if len(values) == 0:
                raise ValidationError("Please select one type per element.")

            if len(values) > 1:
                raise ValidationError("Please select only one thing per element.")

        return cleaned_data


class ConstraintElementInlineFormset(forms.BaseInlineFormSet):
    model = ConstraintElement

    def clean(self):

        super(ConstraintElementInlineFormset, self).clean()

        count_elements = 0
        count_comparisons = 0
        max_position = 0

        last_element = None
        previous_type = None

        count_opening_brackets = 0
        count_closing_brackets = 0

        for form in self.forms:
            cleaned_data = form.cleaned_data

            if cleaned_data == {}:
                continue    # form is empty and will not be saved. Skip validation.

            if not cleaned_data.get('DELETE', False):
                count_elements += 1

                if cleaned_data.get('operand') in ['>', '<']:
                    count_comparisons += 1

                if cleaned_data.get('position') == 1:
                    if cleaned_data.get('operand') is not None:
                        raise ValidationError("Cannot begin a constraint with an operand.")
                    if cleaned_data.get('bracket') is not None:
                        if cleaned_data.get('bracket') == ")":
                            raise ValidationError("Cannot begin a constraint with ')'.")

                current_type = None
                for field in ['num_value', 'obj_value', 'operand', 'bracket']:
                    if cleaned_data.get(field) is not None:
                        current_type = field

                if current_type == previous_type:
                    raise ValidationError("Cannot repeat same type of element in a constraint.")

                if cleaned_data.get('bracket') is not None:
                    if cleaned_data.get('bracket') == "(":
                        count_opening_brackets += 1
                    if cleaned_data.get('bracket') == ")":
                        count_closing_brackets += 1

                        if previous_type == 'operand':
                            raise ValidationError("Cannot continue after operand with ')'.")

                if cleaned_data.get('operand') is not None:
                    if previous_type == 'bracket':
                        raise ValidationError("Cannot continue with operand after a bracket.")

                previous_type = current_type

            if cleaned_data.get('position'):
                if cleaned_data.get('position') > max_position:
                    max_position = cleaned_data.get('position')
                    last_element = cleaned_data

        if count_opening_brackets != count_closing_brackets:
            raise ValidationError("The number of opening brackets and closing brackets must match.")

        if last_element:
            if last_element.get('operand') is not None:
                raise ValidationError("Cannot end a constraint with an operand.")

            if last_element.get('bracket') is not None:
                if last_element.get('bracket') == "(":
                    raise ValidationError("Cannot end a constraint with '('.")

        if count_elements < 3:
            raise ValidationError(f"Please create at least 3 elements, only {count_elements} were given.")

        if count_comparisons > 1:
            raise ValidationError(f"Please select only one comparison-operator.")

        if count_comparisons == 0:
            raise ValidationError(f"Please select one operand of type '>' or '<'.")


class ConstraintElementInline(admin.TabularInline):
    model = ConstraintElement
    form = ConstraintElementInlineForm
    formset = ConstraintElementInlineFormset
    min_num = 3
    extra = 0

    def formfield_for_foreignkey(self, db_field, request, **kwargs):

        if not request.user.is_superuser:
            # reduce fk-field to elements of the designated Object-Properties
            if db_field.name == "obj_value" and 'object_id' in request.resolver_match.kwargs:
                obj = Object.objects.get(id=request.resolver_match.kwargs['object_id'])
                kwargs["queryset"] = ObjectProperty.objects.filter(object__id=obj.id)

        return super().formfield_for_foreignkey(db_field, request, **kwargs)


class PropertyConstraintAdmin(admin.ModelAdmin):
    model = PropertyConstraint
    inlines = [ConstraintElementInline]
    list_display = ('property', '__str__')

    def formfield_for_foreignkey(self, db_field, request, **kwargs):

        if not request.user.is_superuser:
            # if not superuser, a user can only select his projects
            if db_field.name == "property":
                kwargs["queryset"] = ObjectProperty.objects.filter(
                    object__drawing_tool__project__bucket__group__id__in=get_user_groups(request)
                )

        return super().formfield_for_foreignkey(db_field, request, **kwargs)
