From Coq Require Import
  List
  PeanoNat
  Lia
  Peano_dec.
From advert.lib Require Import
  Maps
  Decision
  Semantics
  Tactics.
From advert.specs Require Import
  Config.
Import Notations ListNotations.

Class DAG_Config : Type := {
  dag_t : Type;
  dag_t_eq_dec : forall (x y : dag_t), {x = y} + {x <> y};
}.

Section ListJoin.
  Fixpoint join {T : Type} (l : list (list T)) :=
    match l with
    | [] => []
    | x :: xs => x ++ join xs
    end.

  Lemma join_mem {T : Type} (l : list (list T)) (x : T) : In x (join l) -> exists l', In l' l /\ In x l'.
  Proof. induction l.
         - cbn; contradiction.
         - cbn. intros Hin.
           apply in_app_or in Hin. destruct Hin as [Hin | Hin].
           + exists a; split; auto.
           + specialize (IHl Hin).
             destruct IHl as (l' & Hin1 & Hin2).
             exists l'; split; auto.
  Qed.

  Lemma join_in {T : Type} (l : list (list T)) (l' : list T) (x : T) : In l' l -> In x l' -> In x (join l).
  Proof. induction l as [| l''].
         - cbn; contradiction.
         - cbn; intros Hin1 Hin2.
           destruct Hin1 as [Hin1 | Hin1].
           + subst l''.
             apply in_or_app; auto.
           + specialize (IHl Hin1 Hin2).
             apply in_or_app; auto.
  Qed.

  Lemma join_filt {T : Type} (l : list (list T)) (P : T -> bool) : filter P (join l) = join (map (fun l' => filter P l') l).
  Proof. induction l.
         - cbn; auto.
         - cbn.
           rewrite <- IHl.
           apply filter_app.
  Qed.

  Lemma join_map {T U : Type} (l : list (list T)) (f : T -> U) : map f (join l) = join (map (fun l' => map f l') l).
  Proof. induction l.
         - cbn; auto.
         - cbn.
           rewrite <- IHl.
           apply map_app.
  Qed.

  Lemma join_app {T : Type} (l1 l2 : list (list T)) : join (l1 ++ l2) = join l1 ++ join l2.
  Proof. induction l1.
         - cbn; auto.
         - cbn.
           rewrite IHl1, app_assoc.
           auto.
  Qed.
End ListJoin.

Section Lemmas.
  Lemma filter_nil {T : Type} (l : list T) (f : T -> bool) : (forall x, In x l -> f x = false) -> filter f l = [].
  Proof. intros Hf.
         induction l.
         - cbn; auto.
         - specialize (IHl ltac:(intros x Hx; apply Hf; right; auto)).
           cbn; rewrite IHl.
           specialize (Hf _ ltac:(left; auto)).
           rewrite Hf; auto.
  Qed.

  Lemma join_eq_filter {T : Type} (l : list T) (f : T -> bool) : join (map (fun x => if f x then [x] else []) l) = filter f l.
  Proof. induction l.
         - cbn; auto.
         - cbn; rewrite IHl.
           destruct (f a); auto.
  Qed.
End Lemmas.

Section UDAG.
  Context `{dag_config : !DAG_Config}.

  #[export] Instance dag_t_eq_dec' (x y : dag_t) : Decision (x = y).
  Proof. destruct (dag_t_eq_dec x y); [left | right]; auto. Qed.

  Record UDAG_Vert : Type := { (* UDAG = Unauthenticated DAG *)
    udag_vert_round : nat;
    udag_vert_builder : nat;
    udag_vert_data : dag_t;
    udag_vert_preds : list nat; (* List of vert IDs *)
  }.

  Lemma udag_vert_eq_dec : forall (x y : UDAG_Vert), {x = y} + {x <> y}.
  Proof. intros x y.
         decide equality; apply decide; try typeclasses eauto.
  Qed.

  Record UDAG_State : Type := {
    udag_verts : NatMap UDAG_Vert; (* ID -> vertex *)
    udag_closure : NatMap (list nat);
  }.

  Definition udag_state_null : UDAG_State :=
  {| udag_verts := NatMap_empty _;
     udag_closure := NatMap_empty _;
  |}.

  Definition udag_get_closure (id : nat) (udag : UDAG_State) :=
  match NatMap_find id udag.(udag_closure) with
  | Some l => l
  | None => []
  end.

  Definition udag_compute_closure id preds udag := join (map (fun e => udag_get_closure e udag) preds ++ [[id]]).

  Definition udag_add_vert (id : nat) (vert : UDAG_Vert) (udag : UDAG_State) :=
  {| udag_verts := NatMap_add id vert udag.(udag_verts);
     udag_closure := NatMap_add id (udag_compute_closure id vert.(udag_vert_preds) udag) udag.(udag_closure);
  |}.

  Context `{participant : !BADO_Participant} `{node_assump : !BADO_NodeAssump} `{config : !BADO_Config}.

  Record udag_add_vert_pre (id : nat) (vert : UDAG_Vert) (udag : UDAG_State) : Prop := {
    udag_add_vert_pre_pos : vert.(udag_vert_round) > 0;
    udag_add_vert_pre_mem : In vert.(udag_vert_builder) bado_participant;
    udag_add_vert_pre_lt : Forall (fun pid => match NatMap_find pid udag.(udag_verts) with None => False | Some v => v.(udag_vert_round) < vert.(udag_vert_round) end) vert.(udag_vert_preds);
    udag_add_vert_pre_quorum : vert.(udag_vert_round) = 1 \/ exists quorum, is_quorum bado_comm quorum /\ Forall (fun nid => Exists (fun pid => match NatMap_find pid udag.(udag_verts) with None => True | Some v => v.(udag_vert_round) = vert.(udag_vert_round) - 1 /\ v.(udag_vert_builder) = nid end) vert.(udag_vert_preds)) quorum;
    udag_add_vert_pre_uniq : NatMap_find id udag.(udag_verts) = None;
    udag_add_vert_pre_honest_pred : bado_node_assump vert.(udag_vert_builder) <> Byzantine -> forall id', match NatMap_find id' udag.(udag_verts) with None => True | Some v => v.(udag_vert_builder) = vert.(udag_vert_builder) -> Exists (fun pid => match NatMap_find pid udag.(udag_verts) with None => True | Some v' => v'.(udag_vert_round) >= v.(udag_vert_round) /\ v'.(udag_vert_builder) = vert.(udag_vert_builder) end) vert.(udag_vert_preds) end;
  }.

  Inductive udag_step : UDAG_State -> UDAG_State -> Prop :=
  | udag_step_add_vert : forall id vert udag, udag_add_vert_pre id vert udag -> udag_step udag (udag_add_vert id vert udag).

  Definition udag_sem : Semantics := {|
    s_state := UDAG_State;
    s_init := udag_state_null;
    s_step := udag_step;
  |}.

End UDAG.

Notation udag_valid st_udag := (valid_state udag_sem st_udag).
Notation udag_reachable st_udag := (reachable udag_sem st_udag).

Ltac udag_unfold := unfold udag_add_vert, udag_get_closure.
Ltac udag_reduce := cbn beta delta [udag_verts udag_closure udag_vert_round udag_vert_builder udag_vert_data udag_vert_preds] iota.
Ltac udag_fold st_udag :=
  repeat (match goal with |- context[NatMap_find ?x st_udag.(udag_closure)] => fold (udag_get_closure x st_udag) end).

Section UDAGProps.
  Context `{participant : !BADO_Participant} `{node_assump : !BADO_NodeAssump} `{config : !BADO_Config} `{dag_config : !DAG_Config} `{assump : !BADO_Assump}.

  Lemma udag_mono :
    forall st_udag st_udag' id,
    udag_reachable st_udag st_udag' ->
    NatMap_find id st_udag.(udag_verts) <> None ->
    NatMap_find id st_udag'.(udag_verts) = NatMap_find id st_udag.(udag_verts).
  Proof. intros st_dag st_dag' id Hreach.
         induction Hreach as [| st st' st'' Hreach IH Hstep].
         - cbn; auto.
         - intros Hvert.
           specialize (IH Hvert).
           destruct Hstep as [id' vert' ? Hpre].
           udag_unfold; udag_reduce.
           rewrite <- IH.
           rewrite <- IH in Hvert.
           pose proof (udag_add_vert_pre_uniq _ _ _ Hpre) as Huniq.
           NatMap_case id id'; auto.
           contradiction.
  Qed.

  Lemma udag_closure_mono :
    forall st_udag st_udag' id,
    udag_reachable st_udag st_udag' ->
    NatMap_find id st_udag.(udag_verts) <> None ->
    udag_get_closure id st_udag' = udag_get_closure id st_udag.
  Proof.
    intros st_dag st_dag' id Hreach.
    induction Hreach as [| st st' st'' Hreach IH Hstep].
    - cbn; auto.
    - intros Hvert.
      specialize (IH Hvert).
      destruct Hstep as [id' vert' ? Hpre].
      rewrite <- IH.
      pose proof udag_mono _ _ _ Hreach Hvert as Hvert_same.
      udag_unfold; udag_reduce.
      NatMap_case id id'; auto.
      pose proof udag_add_vert_pre_uniq _ _ _ Hpre as Huniq.
      rewrite Hvert_same in Huniq.
      contradiction.
  Qed.

  Lemma udag_pos :
    forall st_udag id,
    udag_valid st_udag ->
    match NatMap_find id st_udag.(udag_verts) with
    | None => True
    | Some v => v.(udag_vert_round) > 0
    end.
  Proof.
    intros st_dag id Hval.
    induction Hval as [| st st' Hval IH Hstep].
    - cbn; auto.
    - destruct Hstep as [id' vert' ? Hpre].
      udag_unfold; udag_reduce.
      NatMap_case id id'; auto.
      eapply udag_add_vert_pre_pos; eauto.
  Qed.

  Lemma udag_mem :
    forall st_udag id,
    udag_valid st_udag ->
    match NatMap_find id st_udag.(udag_verts) with
    | None => True
    | Some v => In v.(udag_vert_builder) bado_participant
    end.
  Proof.
    intros st_dag id Hval.
    induction Hval as [| st st' Hval IH Hstep].
    - cbn; auto.
    - destruct Hstep as [id' vert' ? Hpre].
      udag_unfold; udag_reduce.
      NatMap_case id id'; auto.
      eapply udag_add_vert_pre_mem; eauto.
  Qed.

  Lemma udag_pred_valid :
    forall st_udag id,
    udag_valid st_udag ->
    match NatMap_find id st_udag.(udag_verts) with
    | None => True
    | Some v => Forall (fun pid => NatMap_find pid st_udag.(udag_verts) <> None) v.(udag_vert_preds)
    end.
  Proof.
    intros st_dag id Hval.
    induction Hval as [| st st' Hval IH Hstep].
    - cbn; auto.
    - destruct Hstep as [id' vert' ? Hpre].
      udag_unfold; udag_reduce.
      NatMap_case id id'; auto.
      + rewrite Forall_forall.
        intros x Hx.
        NatMap_case x id'; try discriminate.
        pose proof (udag_add_vert_pre_lt _ _ _ Hpre) as Hvalid.
        rewrite Forall_forall in Hvalid.
        specialize (Hvalid _ Hx).
        cond_case_auto Hvalid; try discriminate; contradiction.
      + cond_case_auto IH; auto.
        rewrite Forall_forall in IH; rewrite Forall_forall.
        intros x Hx; specialize (IH _ Hx).
        NatMap_case x id'; try discriminate; auto.
  Qed.

  Lemma udag_pred_lt :
    forall st_udag id,
    udag_valid st_udag ->
    match NatMap_find id st_udag.(udag_verts) with
    | None => True
    | Some v => Forall (fun pid => match NatMap_find pid st_udag.(udag_verts) with None => False | Some v' => v'.(udag_vert_round) < v.(udag_vert_round) end) v.(udag_vert_preds)
    end.
  Proof.
    intros st_dag id Hval.
    induction Hval as [| st st' Hval IH Hstep].
    - cbn; auto.
    - pose proof Hstep as Hstep'.
      destruct Hstep as [id' vert' ? Hpre].
      udag_unfold; udag_reduce.
      pose proof (udag_add_vert_pre_uniq _ _ _ Hpre) as Huniq.
      NatMap_case id id'; auto.
      + pose proof (udag_add_vert_pre_lt _ _ _ Hpre) as Hlt.
        rewrite Forall_forall in *.
        intros x Hx.
        specialize (Hlt _ Hx).
        NatMap_case x id'; auto.
        cond_case_auto Hlt; try contradiction; discriminate.
      + cond_case_auto IH; auto.
        rewrite Forall_forall in *.
        intros x Hx.
        specialize (IH _ Hx).
        NatMap_case x id'; auto.
        rewrite Huniq in IH; contradiction.
  Qed.

  Lemma udag_pred_closure_le :
    forall st_udag id,
    udag_valid st_udag ->
    match NatMap_find id st_udag.(udag_verts) with
    | None => True
    | Some v => Forall (fun pid => pid = id \/ match NatMap_find pid st_udag.(udag_verts) with None => False | Some v' => v'.(udag_vert_round) < v.(udag_vert_round) end) (udag_get_closure id st_udag)
    end.
  Proof.
    intros st_dag id Hval; revert id.
    induction Hval as [| st st' Hval IH Hstep].
    - cbn; auto.
    - intros id.
      destruct Hstep as [id' vert' ? Hpre].
      pose proof (udag_add_vert_pre_lt _ _ _ Hpre) as Hlt.
      pose proof (udag_add_vert_pre_uniq _ _ _ Hpre) as Huniq.
      rewrite Forall_forall in Hlt.
      udag_unfold; udag_reduce.
      NatMap_case id id'; auto.
      + rewrite Forall_forall.
        unfold udag_compute_closure.
        intros id'' Hin''.
        apply join_mem in Hin''.
        destruct Hin'' as (l' & Hin0'' & Hin1'').
        apply in_app_or in Hin0''.
        destruct Hin0'' as [Hin0''|Hin0''].
        * apply in_map_iff in Hin0''.
          destruct Hin0'' as (x & ? & Hinx); subst l'.
          specialize (Hlt _ Hinx).
          cond_case_auto Hlt; try contradiction.
          specialize (IH x).
          rewrite Ex in IH.
          rewrite Forall_forall in IH.
          specialize (IH _ Hin1'').
          right.
          destruct IH as [IH | IH].
          -- subst id''; NatMap_case x id'.
             ++ rewrite Huniq in Ex; discriminate.
             ++ rewrite Ex; auto.
          -- cond_case_auto IH; try contradiction.
             NatMap_case id'' id'.
             ++ rewrite Huniq in Ex0; discriminate.
             ++ rewrite Ex0; lia.
        * cbn in Hin0''.
          destruct Hin0'' as [Hin0''|?]; try contradiction.
          subst l'; cbn in Hin1''.
          destruct Hin1'' as [Hin1''|?]; try contradiction.
          left; auto.
      + specialize (IH id).
        cond_case_auto IH; auto.
        rewrite Forall_forall in *.
        udag_fold udag.
        intros x Hx.
        specialize (IH _ Hx).
        destruct IH as [IH | IH].
        * left; auto.
        * right; NatMap_case x id'.
          -- cond_case_auto IH; try contradiction; discriminate.
          -- apply IH.
  Qed.

  Lemma udag_closure_eq :
    forall st_udag id,
    udag_valid st_udag ->
    match NatMap_find id st_udag.(udag_verts) with
    | None => udag_get_closure id st_udag = []
    | Some v => udag_get_closure id st_udag = udag_compute_closure id v.(udag_vert_preds) st_udag
    end.
  Proof. 
    intros st_dag id Hval; revert id.
    induction Hval as [| st st' Hval IH Hstep].
    - cbn; auto.
    - intros id.
      destruct Hstep as [id' vert' ? Hpre].
      pose proof (udag_add_vert_pre_lt _ _ _ Hpre) as Hlt.
      pose proof (udag_add_vert_pre_uniq _ _ _ Hpre) as Huniq.
      rewrite Forall_forall in Hlt.
      udag_unfold; udag_reduce.
      NatMap_case id id'; auto.
      + unfold udag_compute_closure.
        match goal with |- join ?l1 = join ?l2 => assert (Heq_l : l1 = l2) end.
        2: rewrite Heq_l at 1; auto.
        match goal with |- ?l1 ++ _ = ?l2 ++ _ => assert (Heq_l : l1 = l2) end.
        2: rewrite Heq_l at 1; auto.
        apply map_ext_in.
        intros x Hx.
        unfold udag_get_closure; cbn [udag_closure]; udag_fold udag.
        NatMap_case x id'; auto.
        specialize (Hlt _ Hx).
        cond_case_auto Hlt; try contradiction; discriminate.
      + specialize (IH id).
        cond_case_auto IH; auto.
        udag_fold udag; rewrite IH.
        unfold udag_compute_closure; cbn [udag_closure].
        match goal with |- join ?l1 = join ?l2 => assert (Heq_l : l1 = l2) end.
        2: rewrite Heq_l at 1; auto.
        match goal with |- ?l1 ++ _ = ?l2 ++ _ => assert (Heq_l : l1 = l2) end.
        2: rewrite Heq_l at 1; auto.
        apply map_ext_in.
        intros x Hx.
        unfold udag_get_closure; cbn [udag_closure]; udag_fold udag.
        NatMap_case x id'; auto.
        pose proof (udag_pred_valid _ id Hval) as Hvalid.
        rewrite Ex in Hvalid.
        rewrite Forall_forall in Hvalid.
        specialize (Hvalid _ Hx).
        contradiction.
  Qed.

  Lemma udag_closure_valid :
    forall st_udag id id',
    udag_valid st_udag ->
    In id' (udag_get_closure id st_udag) ->
    NatMap_find id' st_udag.(udag_verts) <> None.
  Proof.
    intros udag id id' Hval Hin.
    pose proof (udag_closure_eq _ id Hval) as Hclos.
    cond_case_auto Hclos.
    2: rewrite Hclos in Hin; contradiction.
    clear Hclos.
    pose proof (udag_pred_closure_le _ id Hval) as Hle.
    rewrite Ex in Hle.
    rewrite Forall_forall in Hle.
    specialize (Hle _ Hin).
    destruct Hle as [Hle | Hle].
    1: subst id'; rewrite Ex; discriminate.
    cond_case_auto Hle; try contradiction; discriminate.
  Qed.

  Lemma udag_closure_self :
    forall st_udag id,
    udag_valid st_udag ->
    NatMap_find id st_udag.(udag_verts) <> None ->
    In id (udag_get_closure id st_udag).
  Proof.
    intros udag id Hval Hvert.
    pose proof (udag_closure_eq _ id Hval) as Hclos.
    cond_case_auto Hclos; try contradiction.
    rewrite Hclos; unfold udag_compute_closure.
    rewrite join_app; apply in_or_app.
    right; cbn; auto.
  Qed.

  Lemma udag_closure_pred :
    forall st_udag id v,
    udag_valid st_udag ->
    NatMap_find id st_udag.(udag_verts) = Some v ->
    incl v.(udag_vert_preds) (udag_get_closure id st_udag).
  Proof.
    intros udag id v Hval Hvert.
    intros x Hx.
    pose proof (udag_closure_eq _ id Hval) as Hclos.
    rewrite Hvert in Hclos.
    rewrite Hclos; unfold udag_compute_closure.
    rewrite join_app; apply in_or_app; left.
    eapply join_in.
    - rewrite in_map_iff; exists x; split; auto.
    - apply udag_closure_self; auto.
      clear Hclos.
      pose proof (udag_pred_valid _ id Hval) as Hvalid.
      rewrite Hvert in Hvalid.
      rewrite Forall_forall in Hvalid.
      apply Hvalid; auto.
  Qed.

  Lemma udag_closure_filt :
    forall st_udag id,
    udag_valid st_udag ->
    match NatMap_find id st_udag.(udag_verts) with
    | None => True
    | Some v => filter (fun pid => if decide (match NatMap_find pid st_udag.(udag_verts) with None => false | Some v' => v'.(udag_vert_round) =? v.(udag_vert_round) - 1 end = true) then true else false) (udag_get_closure id st_udag) = filter (fun pid => if decide (match NatMap_find pid st_udag.(udag_verts) with None => false | Some v' => v'.(udag_vert_round) =? v.(udag_vert_round) - 1 end = true) then true else false) v.(udag_vert_preds)
    end.
  Proof.
    intros udag id Hval.
    pose proof (udag_closure_eq _ id Hval) as Hclos.
    cond_case_auto Hclos; auto.
    rewrite Hclos.
    unfold udag_compute_closure.
    rewrite join_filt.
    rewrite map_app.
    rewrite join_app.
    cbn.
    rewrite Ex.
    rewrite decide_false_if.
    2: { rewrite Nat.eqb_eq.
         pose proof (udag_pos _ id Hval) as Hpos.
         rewrite Ex in Hpos.
         lia.
    }
    rewrite app_nil_r.
    rewrite Coqlib.list_map_compose.
    match goal with |- join (map ?f ?l) = _ => assert (Hmap_f_eq : forall x, In x l -> f x = if match NatMap_find x (udag_verts udag) with None => false | Some v' => v'.(udag_vert_round) =? u.(udag_vert_round) - 1 end then [x] else []) end.
    { intros x Hx.
      pose proof (udag_pred_lt _ id Hval) as Hlt.
      rewrite Ex in Hlt.
      rewrite Forall_forall in Hlt.
      specialize (Hlt _ Hx).
      cond_case_auto Hlt; try contradiction.
      pose proof (udag_closure_eq _ x Hval) as Hclos'.
      rewrite Ex0 in Hclos'.
      rewrite Hclos'.
      unfold udag_compute_closure.
      rewrite join_app.
      cbn.
      rewrite filter_app; cbn.
      rewrite Ex0.
      rewrite filter_nil.
      2: { intros y Hy.
           apply join_mem in Hy.
           destruct Hy as (l & Hl1 & Hl2).
           rewrite in_map_iff in Hl1.
           destruct Hl1 as (z & ? & Hz); subst l.
           pose proof (udag_pred_lt _ x Hval) as Hlt_x.
           rewrite Ex0 in Hlt_x.
           rewrite Forall_forall in Hlt_x.
           specialize (Hlt_x _ Hz).
           cond_case_auto Hlt_x; try contradiction.
           pose proof (udag_pred_closure_le _ z Hval) as Hlt_z.
           rewrite Ex1 in Hlt_z.
           rewrite Forall_forall in Hlt_z.
           specialize (Hlt_z _ Hl2).
           destruct Hlt_z as [Hy | Hy].
           - subst y.
             rewrite Ex1.
             rewrite decide_false_if; auto; rewrite Nat.eqb_eq; lia.
           - cond_case_auto Hy; try contradiction.
             rewrite decide_false_if; auto; rewrite Nat.eqb_eq; lia.
      }
      rewrite app_nil_l.
      destruct (udag_vert_round u0 =? udag_vert_round u - 1).
      - rewrite decide_true_if; auto.
      - rewrite decide_false_if; auto.
    }
    rewrite (map_ext_in _ _ _ Hmap_f_eq).
    clear Hmap_f_eq.
    rewrite join_eq_filter.
    apply filter_ext_in.
    intros x Hx.
    destruct (NatMap_find x (udag_verts udag)).
    - destruct (udag_vert_round u0 =? udag_vert_round u - 1).
      + rewrite decide_true_if; auto.
      + rewrite decide_false_if; auto.
    - rewrite decide_false_if; auto.
  Qed.

  Lemma udag_pred_quorum :
    forall st_udag id,
    udag_valid st_udag ->
    match NatMap_find id st_udag.(udag_verts) with
    | None => True
    | Some v => v.(udag_vert_round) = 1 \/ exists quorum, is_quorum bado_comm quorum /\ Forall (fun nid => Exists (fun pid => match NatMap_find pid st_udag.(udag_verts) with None => False | Some v' => v'.(udag_vert_round) = v.(udag_vert_round) - 1 /\ v'.(udag_vert_builder) = nid end) v.(udag_vert_preds)) quorum
    end.
  Proof.
    intros st_dag id Hval; revert id.
    induction Hval as [| st st' Hval IH Hstep].
    - cbn; auto.
    - intros id.
      destruct Hstep as [id' vert' ? Hpre].
      pose proof (udag_add_vert_pre_lt _ _ _ Hpre) as Hlt.
      pose proof (udag_add_vert_pre_uniq _ _ _ Hpre) as Huniq.
      pose proof (udag_add_vert_pre_quorum _ _ _ Hpre) as Hquorum.
      rewrite Forall_forall in Hlt.
      udag_unfold; udag_reduce.
      NatMap_case id id'; auto.
      + destruct Hquorum as [? | Hquorum].
        1: left; auto.
        right.
        destruct Hquorum as (quorum & Hquorum1 & Hquorum2).
        exists quorum; split; auto.
        rewrite Forall_forall in *.
        intros x Hx.
        specialize (Hquorum2 _ Hx).
        rewrite Exists_exists in *.
        destruct Hquorum2 as (y & Hy1 & Hy2).
        exists y; split; auto.
        specialize (Hlt _ Hy1).
        cond_case_auto Hlt; try contradiction.
        NatMap_case y id'.
        * rewrite Huniq in Ex; discriminate.
        * rewrite Ex; auto.
      + specialize (IH id).
        cond_case_auto IH; auto.
        destruct IH as [? | IH].
        1: left; auto.
        right.
        destruct IH as (quorum & Hquorum1 & Hquorum2).
        exists quorum; split; auto.
        rewrite Forall_forall in *.
        intros x Hx.
        specialize (Hquorum2 _ Hx).
        rewrite Exists_exists in *.
        destruct Hquorum2 as (y & Hy1 & Hy2).
        exists y; split; auto.
        pose proof (udag_pred_valid _ id Hval) as Hvalid.
        rewrite Ex in Hvalid.
        rewrite Forall_forall in Hvalid.
        specialize (Hvalid _ Hy1).
        cond_case_auto Hy2; try contradiction.
        NatMap_case y id'.
        * rewrite Huniq in Ex0; discriminate.
        * rewrite Ex0; auto.
  Qed.

  Lemma udag_closure_quorum :
    forall st_udag id r,
    udag_valid st_udag ->
    match NatMap_find id st_udag.(udag_verts) with
    | None => True
    | Some v => 1 <= r < v.(udag_vert_round) ->
      exists quorum,
        Forall (fun id' => match NatMap_find id' st_udag.(udag_verts) with None => False | Some v' => v'.(udag_vert_round) = r end) quorum /\
        is_quorum bado_comm (map (fun id' => match NatMap_find id' st_udag.(udag_verts) with None => 0 | Some v' => v'.(udag_vert_builder) end) quorum) /\
        incl quorum (udag_get_closure id st_udag)
    end.
  Proof.
    intros udag id r Hval.
    remember (match NatMap_find id udag.(udag_verts) with None => 0 | Some v => v.(udag_vert_round) end) as r_max.
    assert (Hle : r_max <= r_max) by auto.
    rewrite Heqr_max in Hle at 1.
    clear Heqr_max.
    revert r id Hle.
    induction r_max.
    - intros r id Hle.
      cond_case_auto Hle; auto.
      pose proof (udag_pos _ id Hval) as Hpos.
      rewrite Ex in Hpos.
      lia.
    - intros r id Hle.
      match type of Hle with ?x <= _ => assert (Heq : x <= r_max \/ x = S r_max) by lia end.
      destruct Heq as [Heq | Heq].
      1: apply (IHr_max r id Heq).
      clear Hle.
      cond_case_auto Heq; auto.
      rewrite Heq.
      pose proof (udag_pred_quorum _ id Hval) as Hquorum.
      rewrite Ex in Hquorum.
      destruct Hquorum as [Hquorum | (quorum & Hquorum1 & Hquorum2)].
      1: intros; lia.
      pose proof (udag_closure_eq _ id Hval) as Hclos.
      rewrite Ex in Hclos.
      intros Hr.
      assert (Hr' : r = r_max \/ 1 <= r < r_max) by lia; clear Hr.
      destruct Hr' as [Hr | Hr].
      + subst r_max.
        clear IHr_max.
        rewrite Forall_forall in Hquorum2.
        match type of Hquorum2 with forall x, In x quorum -> Exists ?f ?l' => assert (Hl : exists l, (forall x, In x quorum -> Exists f l) /\ (forall y, In y l -> exists x, In x quorum /\ f y) /\ incl l l') end.
        { clear Hquorum1 Hclos.
          induction quorum.
          - exists [].
            repeat split.
            + cbn; contradiction.
            + cbn; contradiction.
            + apply incl_nil_l.
          - specialize (IHquorum ltac:(intros x Hx; apply (Hquorum2 _ ltac:(right; apply Hx)))).
            destruct IHquorum as (l & Hl1 & Hl2 & Hl3).
            specialize (Hquorum2 _ ltac:(left; auto)).
            rewrite Exists_exists in Hquorum2.
            destruct Hquorum2 as (x & Hx).
            exists (x :: l); repeat split.
            + intros y Hy; destruct Hy as [Hy | Hy].
              * subst y; left; apply Hx.
              * right; apply Hl1; auto.
            + intros y Hy; destruct Hy as [Hy | Hy].
              * subst y; exists a; split; try left; auto; apply Hx.
              * specialize (Hl2 _ Hy); destruct Hl2 as (z & Hz1 & Hz2).
                exists z; split; try right; auto.
            + apply incl_cons; auto.
              apply Hx.
        }
        destruct Hl as (l & Hl1 & Hl2 & Hl3).
        exists l; repeat split.
        * rewrite Forall_forall; intros y Hy.
          specialize (Hl2 _ Hy).
          destruct Hl2 as (x & _ & Hx).
          cond_case_auto Hx; try contradiction.
          lia.
        * eapply is_quorum_superset.
          1: apply Hquorum1.
          intros x Hx.
          rewrite in_map_iff.
          specialize (Hl1 _ Hx).
          rewrite Exists_exists in Hl1.
          destruct Hl1 as (y & Hy1 & Hy2).
          exists y; split; auto.
          cond_case_auto Hy2; try contradiction; apply Hy2.
        * intros y Hy.
          specialize (Hl3 _ Hy).
          pose proof (udag_closure_pred _ id u Hval ltac:(auto)).
          auto.
      + pose proof (quorum_exists_honest bado_comm_safe _ Hquorum1) as (nid & Hnid & _).
        rewrite Forall_forall in Hquorum2.
        specialize (Hquorum2 _ Hnid).
        rewrite Exists_exists in Hquorum2.
        destruct Hquorum2 as (x & Hx1 & Hx2).
        cond_case_auto Hx2; try contradiction.
        rewrite Heq in Hx2; cbn in Hx2.
        replace (r_max - 0) with r_max in Hx2 by lia.
        clear quorum Hquorum1 Hnid.
        specialize (IHr_max r x).
        rewrite Ex0 in IHr_max.
        specialize (IHr_max ltac:(lia) ltac:(lia)).
        destruct IHr_max as (quorum & Hquorum1 & Hquorum2 & Hquorum3).
        exists quorum; repeat split; auto.
        rewrite Hclos; unfold udag_compute_closure.
        rewrite join_app.
        intros y Hy.
        specialize (Hquorum3 _ Hy).
        apply in_or_app; left.
        eapply join_in.
        2: apply Hquorum3.
        rewrite in_map_iff; exists x; split; auto.
  Qed.

  Lemma udag_honest_self_pred :
    forall st_udag id id',
    udag_valid st_udag ->
    match NatMap_find id st_udag.(udag_verts) with
    | None => True
    | Some v => bado_node_assump v.(udag_vert_builder) <> Byzantine ->
      match NatMap_find id' st_udag.(udag_verts) with
      | None => True
      | Some v' => v'.(udag_vert_builder) = v.(udag_vert_builder) -> v'.(udag_vert_round) < v.(udag_vert_round) -> Exists (fun pid => match NatMap_find pid st_udag.(udag_verts) with None => True | Some v'' => v''.(udag_vert_round) >= v'.(udag_vert_round) /\ v''.(udag_vert_builder) = v.(udag_vert_builder) end) v.(udag_vert_preds)
      end
    end.
  Proof.
    intros st_dag id id' Hval; revert id id'.
    induction Hval as [| st st' Hval IH Hstep].
    - cbn; auto.
    - intros id id'.
      destruct Hstep as [id'' vert' ? Hpre].
      pose proof (udag_add_vert_pre_lt _ _ _ Hpre) as Hlt.
      pose proof (udag_add_vert_pre_uniq _ _ _ Hpre) as Huniq.
      pose proof (udag_add_vert_pre_honest_pred _ _ _ Hpre) as Hhonest_pred.
      rewrite Forall_forall in Hlt.
      udag_unfold; udag_reduce.
      NatMap_case id id''; auto.
      + intros Hhonest.
        specialize (Hhonest_pred Hhonest id').
        NatMap_case id' id''.
        * intros; lia.
        * cond_case_auto Hhonest_pred; auto.
          intros Hbuilder _.
          specialize (Hhonest_pred Hbuilder).
          rewrite Exists_exists in *.
          destruct Hhonest_pred as (x & Hx1 & Hx2).
          exists x; split; auto.
          specialize (Hlt _ Hx1).
          cond_case_auto Hx2; try contradiction.
          NatMap_case x id''.
          -- rewrite Ex0 in Huniq; discriminate.
          -- rewrite Ex0; auto.
      + specialize (IH id id').
        cond_case_auto IH; auto.
        * intros Hhonest; specialize (IH Hhonest).
          NatMap_case id' id''.
          -- rewrite Ex0 in Huniq; discriminate.
          -- rewrite Ex0; intros Hid'1 Hid'2.
             specialize (IH Hid'1 Hid'2).
             rewrite Exists_exists in *.
             destruct IH as (x & Hx1 & Hx2).
             exists x; split; auto.
             pose proof (udag_pred_valid _ id Hval) as Hvalid.
             rewrite Ex in Hvalid.
             rewrite Forall_forall in Hvalid.
             specialize (Hvalid _ Hx1).
             NatMap_case x id''.
             ++ rewrite Huniq in Hvalid; contradiction.
             ++ apply Hx2.
        * intros Hhonest.
          NatMap_case id' id''.
          2: rewrite Ex0; auto.
          intros Hid'1 Hid'2.
          rewrite <- Hid'1 in Hhonest.
          specialize (Hhonest_pred Hhonest id).
          rewrite Ex in Hhonest_pred.
          specialize (Hhonest_pred ltac:(auto)).
          rewrite Exists_exists in Hhonest_pred.
          destruct Hhonest_pred as (x & Hx1 & Hx2).
          specialize (Hlt _ Hx1).
          cond_case_auto Hlt; try contradiction.
          lia.
  Qed.

  Lemma udag_honest_uniq :
    forall st_udag id id',
    udag_valid st_udag ->
    match NatMap_find id st_udag.(udag_verts) with
    | None => True
    | Some v => bado_node_assump v.(udag_vert_builder) <> Byzantine ->
      match NatMap_find id' st_udag.(udag_verts) with
      | None => True
      | Some v' => v'.(udag_vert_builder) = v.(udag_vert_builder) -> v'.(udag_vert_round) = v.(udag_vert_round) -> id = id'
      end
    end.
  Proof.
    intros st_dag id id' Hval; revert id id'.
    induction Hval as [| st st' Hval IH Hstep].
    - cbn; auto.
    - intros id id'.
      destruct Hstep as [id'' vert' ? Hpre].
      pose proof (udag_add_vert_pre_lt _ _ _ Hpre) as Hlt.
      pose proof (udag_add_vert_pre_uniq _ _ _ Hpre) as Huniq.
      pose proof (udag_add_vert_pre_honest_pred _ _ _ Hpre) as Hhonest_pred.
      rewrite Forall_forall in Hlt.
      udag_unfold; udag_reduce.
      NatMap_case id id''; auto.
      + intros Hhonest.
        specialize (Hhonest_pred Hhonest id').
        NatMap_case id' id''.
        * auto.
        * cond_case_auto Hhonest_pred; auto.
          intros Hid'1 Hid'2.
          specialize (Hhonest_pred Hid'1).
          rewrite Exists_exists in Hhonest_pred.
          destruct Hhonest_pred as (x & Hx1 & Hx2).
          specialize (Hlt _ Hx1).
          cond_case_auto Hlt; try contradiction.
          lia.
      + specialize (IH id id').
        cond_case_auto IH; auto.
        * intros Hhonest.
          NatMap_case id' id''.
          -- rewrite Ex0 in Huniq; discriminate.
          -- rewrite Ex0; auto.
        * intros Hhonest.
          NatMap_case id' id''.
          -- intros Hid'1 Hid'2.
             rewrite <- Hid'1 in Hhonest.
             specialize (Hhonest_pred Hhonest id).
             rewrite Ex in Hhonest_pred.
             specialize (Hhonest_pred ltac:(auto)).
             rewrite Exists_exists in Hhonest_pred.
             destruct Hhonest_pred as (x & Hx1 & Hx2).
             specialize (Hlt _ Hx1).
             cond_case_auto Hlt; try contradiction.
             lia.
          -- rewrite Ex0; auto.
  Qed.

    Definition udag_subdag (udag udag' : UDAG_State) :=
  udag_valid udag /\
  udag_valid udag' /\
  forall id, match NatMap_find id udag.(udag_verts) with
  | None => NatMap_find id udag'.(udag_verts) = None
  | Some v => match NatMap_find id udag'.(udag_verts) with
    | None => True
    | Some v' => v' = v
    end
  end.

  Lemma udag_reachable_subdag (udag udag' : UDAG_State) :
  udag_valid udag ->
  udag_reachable udag udag' ->
  udag_subdag udag' udag.
  Proof. intros Hval Hreach.
         induction Hreach as [| st st' st'' Hreach IH Hstep].
         - unfold udag_subdag.
           repeat split; auto.
           intros id; destruct (NatMap_find id (udag_verts st)); auto.
         - specialize (IH Hval).
           assert (Hval' : udag_valid st').
           { eapply valid_reach_valid; [apply Hval | apply Hreach]. }
           pose proof Hstep as Hstep'.
           destruct Hstep as [id v udag Hpre].
           unfold udag_subdag.
           repeat split; auto.
           1: eapply valid_reach_valid; try apply Hval'; eapply reachable_step; [apply reachable_self | apply Hstep'].
           intros id'.
           udag_unfold; udag_reduce.
           NatMap_case id' id.
           + pose proof (udag_add_vert_pre_uniq _ _ _ Hpre) as Huniq.
             destruct (NatMap_find id (udag_verts st)) eqn:Evert; auto.
             pose proof (udag_mono _ _ id Hreach) as Hmono.
             rewrite Evert in Hmono.
             specialize (Hmono ltac:(discriminate)).
             rewrite Hmono in Huniq; discriminate.
           + apply IH.
  Qed.

  Lemma udag_subdag_closure (udag udag' : UDAG_State) :
  udag_subdag udag udag' ->
  forall id, match NatMap_find id udag.(udag_verts) with
  | None => True
  | Some v => match NatMap_find id udag'.(udag_verts) with
    | None => True
    | Some v' => udag_get_closure id udag = udag_get_closure id udag'
    end
  end.
  Proof. intros Hsub.
         destruct Hsub as (Hval1 & Hval2 & Heq).
         pose (fun r id => match NatMap_find id (udag_verts udag) with None => True | Some v => v.(udag_vert_round) <= r end) as p.
         intros id.
         remember (match NatMap_find id (udag_verts udag) with None => 0 | Some v => v.(udag_vert_round) end) as r_max.
         assert (Hp : p r_max id).
         { unfold p; rewrite Heqr_max.
           destruct (NatMap_find id (udag_verts udag)); auto.
         }
         clear Heqr_max.
         revert id Hp.
         induction r_max.
         - intros id Hp.
           unfold p in Hp.
           cond_case_auto Hp; auto.
           pose proof (udag_pos _ id Hval1) as Hpos.
           rewrite Ex in Hpos.
           lia.
         - intros id Hp.
           unfold p in Hp.
           cond_case_auto Hp; auto.
           pose proof (udag_closure_eq _ id Hval1) as Hclos1.
           rewrite Ex in Hclos1.
           specialize (Heq id) as Heq'; rewrite Ex in Heq'.
           cond_case_auto Heq'; auto; subst u0.
           pose proof (udag_closure_eq _ id Hval2) as Hclos2.
           rewrite Ex0 in Hclos2.
           rewrite Hclos1, Hclos2.
           unfold udag_compute_closure.
           do 2 rewrite join_app; cbn.
           match goal with |- join ?l1 ++ _ = join ?l2 ++ _ => assert (Heq_l : l1 = l2) end.
           { apply map_ext_in.
             intros x Hx.
             pose proof (udag_pred_lt _ id Hval1) as Hlt.
             rewrite Ex in Hlt.
             rewrite Forall_forall in Hlt.
             specialize (Hlt _ Hx).
             cond_case_auto Hlt; try contradiction.
             assert (Hxp : p r_max x).
             { unfold p; rewrite Ex1; lia. }
             specialize (IHr_max _ Hxp).
             rewrite Ex1 in IHr_max.
             pose proof (udag_pred_lt _ id Hval2) as Hlt'.
             rewrite Ex0 in Hlt'.
             rewrite Forall_forall in Hlt'.
             specialize (Hlt' _ Hx).
             cond_case_auto Hlt'; try contradiction; clear Hlt'.
             auto.
           }
           rewrite Heq_l; auto.
  Qed.

  Lemma udag_subdag_closure_ex (udag udag' : UDAG_State) :
  udag_subdag udag udag' ->
  forall id,
    NatMap_find id udag'.(udag_verts) <> None ->
    forall id', In id' (udag_get_closure id udag) -> NatMap_find id' udag'.(udag_verts) <> None.
  Proof.
    intros Hsub id Hex.
    pose proof Hsub as Hsub'.
    destruct Hsub as (Hval1 & Hval2 & Heq).
    intros id' Hin.
    pose proof (udag_closure_eq _ id Hval1) as Hclos.
    cond_case_auto Hclos.
    2: rewrite Hclos in Hin; contradiction.
    clear Hclos.
    specialize (Heq id); rewrite Ex in Heq.
    cond_case_auto Heq; try contradiction; subst u0; clear Hex.
    pose proof (udag_subdag_closure _ _ Hsub' id) as Hclos.
    rewrite Ex, Ex0 in Hclos.
    pose proof (udag_pred_closure_le) as Hclos'.
    specialize (Hclos' _ id Hval2).
    rewrite Ex0 in Hclos'.
    rewrite Forall_forall in Hclos'.
    rewrite Hclos in Hin.
    specialize (Hclos' _ Hin).
    destruct Hclos' as [Hclos' | Hclos'].
    1 : subst id'; rewrite Ex0; discriminate.
    cond_case_auto Hclos'; try discriminate; contradiction.
  Qed.

End UDAGProps.
