From Coq Require Import
  List
  PeanoNat
  Lia
  Peano_dec.
From advert.lib Require Import
  Maps
  Semantics
  Coqlib.
From advert.specs Require Import
  Config
  UDAG
  LidoClient
  LidoCommand.
Import Notations ListNotations.

Section Lemmas.

  Lemma list_max_ex' : forall l, l = [] \/ In (list_max l) l.
  Proof. induction l.
         - left; auto.
         - right. cbn. destruct l.
           + cbn. left; lia.
           + fold (list_max (n :: l)). remember (list_max (n ::l)) as x. clear Heqx. destruct IHl as [? | IHl]. 1: discriminate.
             apply Nat.max_case. 1: left;auto. right; auto.
  Qed.

  Lemma list_max_ex : forall l, l <> [] -> In (list_max l) l.
  Proof. intros l Hex. pose proof (list_max_ex' l) as Hin. destruct Hin as [? | Hin]; try contradiction; auto.
  Qed.

End Lemmas.

Section LidoDAG.
  Context `{dag_config : !DAG_Config} `{participant : !BADO_Participant} `{leader : !BADO_Leader} `{node_assump : BADO_NodeAssump}.

  Record LidoDAG : Type := {
    lidodag_lido : ClientTree;
    lidodag_dag : UDAG_State;
  }.

  Definition lidodag_state_null : LidoDAG := {|
    lidodag_lido := client_tree_null;
    lidodag_dag := udag_state_null;
  |}.

  Definition lidodag_add_vert id vert lidodag := {|
    lidodag_lido := lidodag.(lidodag_lido);
    lidodag_dag := udag_add_vert id vert lidodag.(lidodag_dag);
  |}.

  Definition lidodag_add_ecache r pull_src lidodag := {|
    lidodag_lido := client_add_ecache r pull_src lidodag.(lidodag_lido);
    lidodag_dag := lidodag.(lidodag_dag);
  |}.

  Definition lidodag_add_mcache r ver data lidodag := {|
    lidodag_lido := client_add_mcache r ver data lidodag.(lidodag_lido);
    lidodag_dag := lidodag.(lidodag_dag);
  |}.

  Definition lidodag_add_ccache r ver lidodag := {|
    lidodag_lido := client_add_ccache r ver lidodag.(lidodag_lido);
    lidodag_dag := lidodag.(lidodag_dag);
  |}.

  Definition lidodag_add_tcache r lidodag := {|
    lidodag_lido := client_add_tcache r lidodag.(lidodag_lido);
    lidodag_dag := lidodag.(lidodag_dag);
  |}.

  Definition lidodag_issue_pull r lidodag := {|
    lidodag_lido := client_issue_pull r lidodag.(lidodag_lido);
    lidodag_dag := lidodag.(lidodag_dag);
  |}.

  Definition lidodag_issue_invoke r ver data lidodag := {|
    lidodag_lido := client_issue_invoke r ver data lidodag.(lidodag_lido);
    lidodag_dag := lidodag.(lidodag_dag);
  |}.

  Definition lidodag_issue_push r ver lidodag := {|
    lidodag_lido := client_issue_push r ver lidodag.(lidodag_lido);
    lidodag_dag := lidodag.(lidodag_dag);
  |}.

  Definition lidodag_drop_cmd r lidodag := {|
    lidodag_lido := client_drop_cmd r lidodag.(lidodag_lido);
    lidodag_dag := lidodag.(lidodag_dag);
  |}.

  Context `{config : !BADO_Config}.

  Definition max_nid := S (list_max bado_participant).

  Lemma max_nid_max : forall nid, In nid bado_participant -> (nid < max_nid)%nat.
  Proof. intros nid Hnid. unfold max_nid.
         assert (list_max bado_participant <= list_max bado_participant)%nat as Hle by auto.
         rewrite list_max_le in Hle.
         rewrite Forall_forall in Hle. specialize (Hle _ Hnid).
         lia.
  Qed.

  Definition lidodag_valid_data data lidodag := NatMap_find data lidodag.(lidodag_dag).(udag_verts) <> None.

  Inductive lidodag_step : LidoDAG -> LidoDAG -> Prop :=
  | lidodag_step_add_vert : forall id vert lidodag, udag_add_vert_pre id vert lidodag.(lidodag_dag) -> lidodag_step lidodag (lidodag_add_vert id vert lidodag)
  | lidodag_step_add_ecache : forall r pull_src lidodag, client_add_ecache_pre r pull_src lidodag.(lidodag_lido) -> lidodag_step lidodag (lidodag_add_ecache r pull_src lidodag)
  | lidodag_step_add_mcache : forall r ver data lidodag, client_add_mcache_pre r ver data lidodag.(lidodag_lido) -> lidodag_step lidodag (lidodag_add_mcache r ver data lidodag)
  | lidodag_step_add_ccache : forall r ver lidodag, client_add_ccache_pre r ver lidodag.(lidodag_lido) -> lidodag_step lidodag (lidodag_add_ccache r ver lidodag)
  | lidodag_step_add_tcache : forall r lidodag, lidodag_step lidodag (lidodag_add_tcache r lidodag)
  | lidodag_step_issue_pull : forall r lidodag, client_issue_cmd_pre (bado_leader_at r) (Pull r) lidodag.(lidodag_lido) -> lidodag_step lidodag (lidodag_issue_pull r lidodag)
  | lidodag_step_issue_invoke : forall r ver data lidodag, lidodag_valid_data data lidodag -> client_issue_cmd_pre (bado_leader_at r) (Invoke r ver data) lidodag.(lidodag_lido) -> lidodag_step lidodag (lidodag_issue_invoke r ver data lidodag)
  | lidodag_step_issue_push : forall r ver lidodag, client_issue_cmd_pre (bado_leader_at r) (Push r ver) lidodag.(lidodag_lido) -> lidodag_step lidodag (lidodag_issue_push r ver lidodag)
  | lidodag_step_drop_cmd : forall r lidodag, client_drop_cmd_pre r lidodag.(lidodag_lido) -> lidodag_step lidodag (lidodag_drop_cmd r lidodag).

  Definition lidodag_sem : Semantics := {|
    s_state := LidoDAG;
    s_init := lidodag_state_null;
    s_step := lidodag_step;
  |}.

End LidoDAG.

Notation lidodag_valid lidodag := (valid_state lidodag_sem lidodag).
Notation lidodag_reachable lidodag := (reachable lidodag_sem lidodag).
Ltac lidodag_step_case Hstep := inversion Hstep as [id vert lidodag Hpre | r pull_src lidodag Hpre | r ver data lidodag Hpre | r ver lidodag Hpre | r lidodag | r lidodag Hpre | r ver data lidodag Hpre1 Hpre2 | r ver lidodag Hpre | r lidodag].

Section LidoDAGProps.
  Context `{config : !BADO_Config} `{dag_config : !DAG_Config} `{participant : !BADO_Participant} `{leader : !BADO_Leader} `{node_assump : !BADO_NodeAssump} `{assump : !BADO_Assump}.

  Lemma lidodag_dag_valid lidodag : lidodag_valid lidodag -> udag_valid lidodag.(lidodag_dag).
  Proof. intros Hval.
         induction Hval as [|st st' Hval IH Hstep].
         - cbn. apply valid_state_init.
         - lidodag_step_case Hstep; subst st; subst st'; cbn; try apply IH.
           eapply valid_state_step. 1: apply IH.
           constructor; auto.
  Qed.

  Lemma lidodag_lido_valid lidodag : lidodag_valid lidodag -> client_tree_valid lidodag.(lidodag_lido).
  Proof. intros Hval.
         induction Hval as [|st st' Hval IH Hstep].
         - cbn. apply valid_state_init.
         - lidodag_step_case Hstep; subst st; subst st'; cbn; try apply IH.
           all: eapply valid_state_step; [apply IH | constructor; auto].
  Qed.

  Lemma lidodag_dag_reachable lidodag lidodag' : lidodag_reachable lidodag lidodag' -> udag_reachable lidodag.(lidodag_dag) lidodag'.(lidodag_dag).
  Proof. intros Hreach.
         induction Hreach as [|st st' st'' Hreach IH Hstep].
         - apply reachable_self.
         - lidodag_step_case Hstep; subst st' st''; cbn; try apply IH.
           eapply reachable_step. 1: apply IH.
           constructor; auto.
  Qed.

  Lemma lidodag_lido_reachable lidodag lidodag' : lidodag_reachable lidodag lidodag' -> client_reachable lidodag.(lidodag_lido) lidodag'.(lidodag_lido).
  Proof. intros Hreach.
         induction Hreach as [|st st' st'' Hreach IH Hstep].
         - apply reachable_self.
         - lidodag_step_case Hstep; subst st' st''; cbn; try apply IH.
           all: eapply reachable_step; [apply IH | constructor; auto].
  Qed.

  Lemma lidodag_vert_valid id lidodag : lidodag_valid lidodag -> NatMap_find id lidodag.(lidodag_dag).(udag_verts) <> None -> lidodag_valid_data id lidodag.
  Proof. intros Hval Hvert.
         auto.
  Qed.

  Lemma lidodag_invoke_valid nid lidodag : lidodag_valid lidodag -> match client_get_cmd nid (snd lidodag.(lidodag_lido)) with Invoke _ _ data => lidodag_valid_data data lidodag | _ => True end.
  Proof. intros Hval. revert nid.
         induction Hval as [|st st' Hval IH Hstep].
         - cbn; auto.
         - lidodag_step_case Hstep; subst st; subst st'.
           2,3,4,5,6,7,8,9: intros nid'; cbn; client_unfold; client_reduce; NatMap_cmp nid' (bado_leader_at r); auto; client_fold (lidodag_lido lidodag);
                specialize (IH nid'); match type of IH with match ?x with _ => _ end => destruct x end; auto.
           + intros nid'; cbn; unfold lidodag_add_vert; unfold lidodag_valid_data; udag_unfold; cbn [lidodag_dag]; udag_reduce.
             specialize (IH nid').
             match type of IH with match ?x with _ => _ end => destruct x end; auto.
             unfold lidodag_valid_data in IH.
             NatMap_cmp data id; auto; discriminate.
  Qed.

  Lemma lidodag_mcache_valid r ver lidodag : lidodag_valid lidodag -> match NatMap_find ver (client_get_round r lidodag.(lidodag_lido)).(client_round_mcaches) with None => True | Some n => lidodag_valid_data n lidodag end.
  Proof. intros Hval. revert r ver.
         induction Hval as [|st st' Hval IH Hstep].
         - cbn; auto.
         - lidodag_step_case Hstep; subst st; subst st'; cbn; try apply IH.
           2,4: intros r' ver'; specialize (IH r' ver'); client_unfold; client_reduce; NatMap_case r' r; client_reduce; client_fold (lidodag_lido lidodag);
                match type of IH with match ?x with _ => _ end => destruct x end; auto.
           + intros r' ver'; specialize (IH r' ver').
             match type of IH with match ?x with _ => _ end => destruct x end; auto.
             unfold lidodag_add_vert; unfold lidodag_valid_data; udag_unfold; cbn [lidodag_dag]; udag_reduce.
             unfold lidodag_valid_data in IH.
             NatMap_cmp n id; auto; discriminate.
           + intros r' ver'; specialize (IH r' ver').
             client_unfold; NatMap_case r' r; client_reduce; client_fold (lidodag_lido lidodag).
             * NatMap_case ver' ver; auto.
               pose proof (client_add_mcache_cmd _ _ _ _ Hpre) as Hcmd.
               pose proof (lidodag_invoke_valid (bado_leader_at r) _ Hval) as Hcmd'.
               rewrite Hcmd in Hcmd'.
               unfold lidodag_add_mcache; unfold lidodag_valid_data; cbn. apply Hcmd'.
             * auto.
  Qed.

  Lemma lidodag_safety_from_dag_safety lidodag dag' :
    lidodag_valid lidodag ->
    udag_reachable lidodag.(lidodag_dag) dag' ->
    lidodag_reachable lidodag {| lidodag_dag := dag'; lidodag_lido := lidodag.(lidodag_lido); |}.
  Proof. intros Hval Hreach.
         remember (lidodag_dag lidodag) as dag_init. revert Heqdag_init.
         induction Hreach as [|st st' st'' Hreach IH Hstep].
         - intros Heq. subst st.
           destruct lidodag; cbn; apply reachable_self.
         - intros Heq.
           specialize (IH Heq). subst st.
           destruct Hstep.
           eapply reachable_step.
           1: apply IH.
           cbn.
           match goal with |- lidodag_step ?x _ => replace udag with x.(lidodag_dag) at 2 end.
           2: cbn; auto.
           match goal with |- lidodag_step ?x _ => replace lidodag.(lidodag_lido) with x.(lidodag_lido) at 2 end.
           apply lidodag_step_add_vert; auto.
           cbn; auto.
  Qed.

End LidoDAGProps.
