(* The client view of cache tree *)
(* There is only one TCache per round *)
(* Every TCache is the outcome of some operation *)
(* Leaders can pull from any MCache, not just TCache *)

From Coq Require Import
  List
  PeanoNat
  Lia
  Peano_dec.
From advert.lib Require Import
  Maps
  Semantics.
From advert.specs Require Import
  Config
  LidoCommand.
Import Notations ListNotations.

Section Client.
  Record Client_RoundDesc : Type := {
    client_round_pull_src : option (nat * nat);
    client_round_mcaches : NatMap nat;
    client_round_max_ccache : option nat;
  }.

  Record Client_AuxData : Type := {
    client_mcache_log : NatMap (NatMap (list (nat * nat * nat)));
    client_cmd : NatMap ClientCmd;
    client_proposed_vals : list nat;
  }.

  Definition ClientTree := (NatMap Client_RoundDesc) * Client_AuxData.
  Definition client_data_null := {|
    client_mcache_log := NatMap_add 0 (NatMap_add 0 [] (NatMap_empty _)) (NatMap_empty _);
    client_cmd := NatMap_empty _;
    client_proposed_vals := [];
  |}.
  Definition client_tree_null := (NatMap_empty Client_RoundDesc, client_data_null).

  Definition client_round_null : Client_RoundDesc := {|
    client_round_pull_src := None;
    client_round_mcaches := NatMap_empty nat;
    client_round_max_ccache := None;
  |}.

  Definition client_get_round r (st_client : ClientTree) := match NatMap_find r (fst st_client) with
  | None => client_round_null
  | Some rd => rd
  end.

  Definition client_round_add_ecache (pull_src : nat * nat) (rd : Client_RoundDesc) := {|
    client_round_pull_src := Some pull_src;
    client_round_mcaches := rd.(client_round_mcaches);
    client_round_max_ccache := rd.(client_round_max_ccache);
  |}.

  Definition client_round_add_mcache (ver : nat) (data : nat) (rd : Client_RoundDesc) := {|
    client_round_pull_src := rd.(client_round_pull_src);
    client_round_mcaches := NatMap_add ver data rd.(client_round_mcaches);
    client_round_max_ccache := rd.(client_round_max_ccache);
  |}.

  Definition client_round_add_ccache (ver : nat) (rd : Client_RoundDesc) := {|
    client_round_pull_src := rd.(client_round_pull_src);
    client_round_mcaches := rd.(client_round_mcaches);
    client_round_max_ccache := match rd.(client_round_max_ccache) with None => Some ver | Some v => if v <=? ver then Some ver else Some v end;
  |}.

  Definition client_get_logs (st_client : ClientTree) := (snd st_client).(client_mcache_log).

  Definition client_round_get_logs (r : nat) (st_client : ClientTree) := match NatMap_find r (client_get_logs st_client) with
  | None => NatMap_empty _
  | Some logs => logs
  end.

  Definition client_get_log (r : nat) (ver : nat) (st_client : ClientTree) := NatMap_find ver (client_round_get_logs r st_client).

  Definition client_add_log' (r : nat) (ver : nat) (log : list (nat * nat * nat)) (st_client : ClientTree) := NatMap_add r (NatMap_add ver log (client_round_get_logs r st_client)) (client_get_logs st_client).

  Definition client_add_log (r : nat) (ver : nat) (data : nat) (st_client : ClientTree) :=
    let prev_log := match ver with
    | 0 => let ecache := match (client_get_round r st_client).(client_round_pull_src) with None => (0, 0) | Some pull_src => pull_src end in
           match client_get_log (fst ecache) (snd ecache) st_client with None => [] | Some log => log end
    | S v => match client_get_log r v st_client with None => [] | Some log => log end
    end in
    let new_log := prev_log ++ [((r, ver), data)] in
    ((fst st_client),
     {|
       client_mcache_log := client_add_log' r ver new_log st_client;
       client_cmd := (snd st_client).(client_cmd);
       client_proposed_vals := (snd st_client).(client_proposed_vals);
     |}).

  Definition client_get_cmd (nid : nat) (st_client : Client_AuxData) := match NatMap_find nid st_client.(client_cmd) with
  | None => Idle
  | Some cmd => cmd
  end.

  Definition client_set_cmd (nid : nat) (cmd : ClientCmd) (st_aux_client : Client_AuxData) := {|
    client_mcache_log := st_aux_client.(client_mcache_log);
    client_cmd := NatMap_add nid cmd st_aux_client.(client_cmd);
    client_proposed_vals := st_aux_client.(client_proposed_vals);
  |}.

  Definition client_add_proposed_val (m : nat) (st_aux_client : Client_AuxData) := {|
    client_mcache_log := st_aux_client.(client_mcache_log);
    client_cmd := st_aux_client.(client_cmd);
    client_proposed_vals := m :: st_aux_client.(client_proposed_vals);
  |}.

  Context `{participant : !BADO_Participant} `{leader : !BADO_Leader}.

  Definition client_add_ecache (r : nat) (pull_src : nat * nat) (st_client : ClientTree) := (NatMap_add r (client_round_add_ecache pull_src (client_get_round r st_client)) (fst st_client), client_set_cmd (bado_leader_at r) Idle (snd st_client)).

  Definition client_add_mcache (r : nat) (ver : nat) (data : nat) (st_client : ClientTree) := client_add_log r ver data (NatMap_add r (client_round_add_mcache ver data (client_get_round r st_client)) (fst st_client), client_set_cmd (bado_leader_at r) Idle (snd st_client)).

  Definition client_add_ccache (r : nat) (ver : nat) (st_client : ClientTree) := (NatMap_add r (client_round_add_ccache ver (client_get_round r st_client)) (fst st_client), client_set_cmd (bado_leader_at r) Idle (snd st_client)).

  Definition client_add_tcache (r : nat) (st_client : ClientTree) := (fst st_client, client_set_cmd (bado_leader_at r) Idle (snd st_client)).

  Definition client_issue_pull (r : nat) (st_client : ClientTree) := (fst st_client, client_set_cmd (bado_leader_at r) (Pull r) (snd st_client)).

  Definition client_issue_invoke (r : nat) (ver : nat) (data : nat) (st_client : ClientTree) := (fst st_client, client_add_proposed_val data (client_set_cmd (bado_leader_at r) (Invoke r ver data) (snd st_client))).

  Definition client_issue_push (r : nat) (ver : nat) (st_client : ClientTree) := (fst st_client, client_set_cmd (bado_leader_at r) (Push r ver) (snd st_client)).

  Definition client_drop_cmd (r : nat) (st_client : ClientTree) := (fst st_client, client_set_cmd (bado_leader_at r) Idle (snd st_client)).

  Record client_add_ecache_pre (r : nat) (pull_src : nat * nat) (st_client : ClientTree) : Prop := {
    client_add_ecache_pos : r > 0;
    client_add_ecache_nodup : (client_get_round r st_client).(client_round_pull_src) = None \/ (client_get_round r st_client).(client_round_pull_src) = Some pull_src;
    client_add_ecache_pull_lt : fst pull_src < r;
    client_add_ecache_pull_src_valid : (fst pull_src = 0 /\ snd pull_src = 0) \/ (fst pull_src > 0 /\ NatMap_find (snd pull_src) (client_get_round (fst pull_src) st_client).(client_round_mcaches) <> None);
    client_add_ecache_safe : forall r', r' < r -> let rd := client_get_round r' st_client in match rd.(client_round_max_ccache) with None => True | Some v => fst pull_src > r' \/ (fst pull_src = r' /\ snd pull_src >= v) end;
    client_add_ecache_cmd : client_get_cmd (bado_leader_at r) (snd st_client) = Pull r;
  }.

  Record client_add_mcache_pre (r : nat) (ver : nat) (data : nat) (st_client : ClientTree) := {
    client_add_mcache_pos : r > 0;
    client_add_mcache_nodup : NatMap_find ver (client_get_round r st_client).(client_round_mcaches) = None \/ NatMap_find ver (client_get_round r st_client).(client_round_mcaches) = Some data;
    client_add_mcache_valid : match ver with 0 => (client_get_round r st_client).(client_round_pull_src) <> None | S v => NatMap_find v (client_get_round r st_client).(client_round_mcaches) <> None end;
    client_add_mcache_cmd : client_get_cmd (bado_leader_at r) (snd st_client) = Invoke r ver data;
  }.

  Record client_add_ccache_pre (r : nat) (ver : nat) (st_client : ClientTree) := {
    client_add_ccache_pos : r > 0;
    client_add_ccache_has_mcache : NatMap_find ver (client_get_round r st_client).(client_round_mcaches) <> None;
    client_add_ccache_safe : forall r', r' > r -> let rd := client_get_round r' st_client in match rd.(client_round_pull_src) with None => True | Some src => fst src > r \/ (fst src = r /\ snd src >= ver) end;
    client_add_ccache_cmd : client_get_cmd (bado_leader_at r) (snd st_client) = Push r ver;
  }.

  Record client_issue_cmd_pre (nid : nat) (cmd : ClientCmd) (st_client : ClientTree) : Prop := {
    client_issue_cmd_idle : client_get_cmd nid (snd st_client) = Idle;
    client_issue_cmd_pos : cmd_time cmd > 0;
  }.

  Record client_drop_cmd_pre (r : nat) (st_client : ClientTree) : Prop := {
    client_drop_cmd_pos : r > 0;
    client_drop_cmd_cmd : cmd_time (client_get_cmd (bado_leader_at r) (snd st_client)) = r;
  }.

  Inductive client_step : ClientTree -> ClientTree -> Prop :=
  | client_step_add_ecache : forall r pull_src st_client, client_add_ecache_pre r pull_src st_client -> client_step st_client (client_add_ecache r pull_src st_client)
  | client_step_add_mcache : forall r ver data st_client, client_add_mcache_pre r ver data st_client -> client_step st_client (client_add_mcache r ver data st_client)
  | client_step_add_ccache : forall r ver st_client, client_add_ccache_pre r ver st_client -> client_step st_client (client_add_ccache r ver st_client)
  | client_step_add_tcache : forall r st_client, client_step st_client (client_add_tcache r st_client)
  | client_step_issue_pull : forall r st_client, client_issue_cmd_pre (bado_leader_at r) (Pull r) st_client -> client_step st_client (client_issue_pull r st_client)
  | client_step_issue_invoke : forall r ver data st_client, client_issue_cmd_pre (bado_leader_at r) (Invoke r ver data) st_client -> client_step st_client (client_issue_invoke r ver data st_client)
  | client_step_issue_push : forall r ver st_client, client_issue_cmd_pre (bado_leader_at r) (Push r ver) st_client -> client_step st_client (client_issue_push r ver st_client)
  | client_step_drop_cmd : forall r st_client, client_drop_cmd_pre r st_client -> client_step st_client (client_drop_cmd r st_client).

  Definition client_sem : Semantics := {|
    s_state := ClientTree;
    s_init := client_tree_null;
    s_step := client_step;
  |}.

End Client.

Notation client_tree_valid st_client := (valid_state client_sem st_client).
Notation client_reachable st_client := (reachable client_sem st_client).
Ltac client_step_case Hstep := inversion Hstep as [r pull_src st_client Hpre | r ver data st_client Hpre | r ver st_client Hpre | r st_client Hpre | r st_client Hpre | r ver data st_client Hpre | r ver st_client Hpre | r st_client Hpre].
Ltac client_unfold := unfold client_get_round, client_add_ecache, client_add_mcache, client_add_ccache, client_add_tcache, client_issue_pull, client_issue_invoke, client_issue_push, client_drop_cmd, client_set_cmd, client_round_add_ecache, client_round_add_mcache, client_round_add_ccache, client_add_log, client_add_log', client_get_log, client_round_get_logs, client_get_logs, client_get_cmd, client_add_proposed_val; cbn beta delta [fst snd] iota.
Ltac client_reduce := cbn beta delta [client_round_pull_src client_round_mcaches client_round_max_ccache client_cmd client_proposed_vals] iota.
Ltac client_fold st_client := repeat (match goal with |- context[NatMap_find ?x (fst st_client)] => fold (client_get_round x st_client) end); fold (client_get_logs st_client); repeat (match goal with |- context[NatMap_find ?x (client_get_logs st_client)] => fold (client_round_get_logs x st_client) end); repeat (match goal with |- context[NatMap_find ?y (client_round_get_logs ?x st_client)] => fold (client_get_log x y st_client) end); repeat (match goal with |- context[NatMap_find ?x (snd st_client).(client_cmd)] => fold (client_get_cmd x (snd st_client)) end).

Section ClientProps.
  Context `{participant : !BADO_Participant} `{leader : !BADO_Leader}.

  Lemma client_pull_committed : forall st_client r r', client_tree_valid st_client -> match (client_get_round r st_client).(client_round_max_ccache) with None => True | Some v => r' > r -> match (client_get_round r' st_client).(client_round_pull_src) with None => True | Some src => fst src > r \/ (fst src = r /\ snd src >= v) end end.
  Proof. intros st_client r' r'' Hval. revert r''. revert r'. induction Hval as [|st st' Hval IH Hstep]. 1: cbn; easy. client_step_case Hstep; subst st; subst st'.
         all: intros r' r''; client_unfold; NatMap_case r' r; NatMap_case r'' r; client_reduce; client_fold st_client; auto; try apply IH.
         - match goal with |- context[match ?x with _ => _ end] => destruct x eqn:Eccache end. 1: lia. easy.
         - match goal with |- context[match ?x with _ => _ end] => destruct x eqn:Eccache end. 2: easy. intro Hgt. destruct Hpre. specialize (client_add_ecache_safe0 _ Hgt). cbn in client_add_ecache_safe0. rewrite Eccache in client_add_ecache_safe0. auto.
         - match goal with |- context[match ?x with _ => _ end] => destruct x eqn:Enew_ccache end. 2: easy. lia.
         - match goal with |- context[match ?x with _ => _ end] => destruct x eqn:Enew_ccache end. 2: easy.
           intro Hgt. destruct Hpre. specialize (client_add_ccache_safe0 _ Hgt). cbn in client_add_ccache_safe0.
           match goal with |- context[match ?x with _ => _ end] => destruct x eqn:Eecache end. 2: easy.
           match type of Enew_ccache with context[match ?x with _ => _ end] => destruct x eqn:Eccache end.
           + match type of Enew_ccache with context[if ?p then _ else _] => destruct p eqn:Hle end.
             * inversion Enew_ccache; subst n0. auto.
             * inversion Enew_ccache; subst n0. specialize (IH r r''). rewrite Eccache in IH. specialize (IH Hgt). rewrite Eecache in IH. auto.
           + inversion Enew_ccache; subst n0. auto.
  Qed.

  Lemma client_pull_lt : forall st_client r, client_tree_valid st_client -> match (client_get_round r st_client).(client_round_pull_src) with None => True | Some src => fst src < r end.
  Proof. intros st_client r' Hval. revert r'. induction Hval as [|st st' Hval IH Hstep]. 1: cbn; easy. client_step_case Hstep; subst st; subst st'.
         all: intros r'; client_unfold; NatMap_case r' r; client_reduce; client_fold st_client; auto; try apply IH.
         - apply Hpre.
  Qed.

  Lemma client_pull_valid : forall st_client r r', client_tree_valid st_client -> match (client_get_round r st_client).(client_round_pull_src) with None => True | Some src => r' = fst src -> (fst src = 0 /\ snd src = 0) \/ (fst src > 0 /\ NatMap_find (snd src) (client_get_round r' st_client).(client_round_mcaches) <> None) end.
  Proof. intros st_client r' r'' Hval. revert r''. revert r'. induction Hval as [|st st' Hval IH Hstep]. 1: cbn; easy. client_step_case Hstep; subst st; subst st'.
         all: intros r' r''; client_unfold; NatMap_case r' r; NatMap_case r'' r; client_reduce; client_fold st_client; auto; try apply IH.
         - intro Heq; subst r; apply Hpre.
         - intro Heq; subst r''; apply Hpre.
         - match goal with |- context[match ?x with _ => _ end] => destruct x eqn:Eecache end. 2: easy.
           intro Heq. subst r. specialize (IH (fst p) (fst p)).  rewrite Eecache in IH. specialize (IH ltac:(easy)).
           destruct IH as [IH | IH]. 1: left; auto. right. split. 1: apply IH.
           NatMap_cmp (snd p) ver. 1: easy. apply IH.
         - match goal with |- context[match ?x with _ => _ end] => destruct x eqn:Eecache end. 2: easy.
           intro Heq. subst r. specialize (IH r' (fst p)). rewrite Eecache in IH. specialize (IH ltac:(easy)).
           destruct IH as [IH | IH]. 1: left; auto. right. split. 1: apply IH.
           NatMap_cmp (snd p) ver. 1: easy. apply IH.
  Qed.

  Lemma client_mcache_has_pred : forall st_client r v, client_tree_valid st_client -> NatMap_find v (client_get_round r st_client).(client_round_mcaches) <> None -> match v with 0 => (client_get_round r st_client).(client_round_pull_src) <> None | S v' => NatMap_find v' (client_get_round r st_client).(client_round_mcaches) <> None end.
  Proof. intros st_client r' v Hval. revert v. revert r'. induction Hval as [|st st' Hval IH Hstep]. 1: cbn; easy. client_step_case Hstep; subst st; subst st'.
         all: intros r'; client_unfold; NatMap_case r' r; client_reduce; client_fold st_client; auto; try apply IH.
         - intros v Hmcache. destruct v. 1: easy. specialize (IH r (S v) Hmcache). cbn in IH. auto.
         - intro v. NatMap_case v ver.
           + intros _. destruct ver eqn:Ever. 1: apply Hpre. assert (n <> S n) by lia. NatMap_rwo n (S n). apply Hpre.
           + intro Hmcache. specialize (IH _ _ Hmcache). destruct v. 1: apply IH. NatMap_case v ver. 1: easy. apply IH.
  Qed.

  Lemma client_mcache_has_pred' : forall st_client r v v', client_tree_valid st_client -> NatMap_find v' (client_get_round r st_client).(client_round_mcaches) <> None -> v <= v' -> NatMap_find v (client_get_round r st_client).(client_round_mcaches) <> None.
  Proof. intros st_client r v v' Hval. induction v'.
         - intros Hmcache Hle. assert (v = 0) by lia; subst v. auto.
         - intros Hmcache Hle. assert (Hle' : v <= v' \/ v = S v') by lia. destruct Hle' as [Hle' | Hle']. 2: subst v; auto.
           pose proof (client_mcache_has_pred _ _ _ Hval Hmcache) as Hmcache'. cbn in Hmcache'. specialize (IHv' Hmcache' Hle'). auto.
  Qed.

  Lemma client_commit_valid : forall st_client r v, client_tree_valid st_client -> (client_get_round r st_client).(client_round_max_ccache) = Some v -> NatMap_find v (client_get_round r st_client).(client_round_mcaches) <> None.
  Proof. intros st_client r' v Hval. revert v. revert r'. induction Hval as [|st st' Hval IH Hstep]. 1: cbn; easy. client_step_case Hstep; subst st; subst st'.
         all: intros r'; client_unfold; NatMap_case r' r; client_reduce; client_fold st_client; auto; try apply IH.
         - intros v Hccache. NatMap_cmp v ver. 1: easy. specialize (IH _ _ Hccache). auto.
         - intros v Hccache.
           match type of Hccache with context[match ?x with _ => _ end] => destruct x eqn:Eccache end.
           + match type of Hccache with context[if ?p then _ else _] => destruct p end; inversion Hccache; subst v. 1: apply Hpre. specialize (IH _ _ Eccache). auto.
           + inversion Hccache; subst v. apply Hpre.
  Qed.

  Lemma ecache_invariant : forall st_client st_client' r, client_reachable st_client st_client' -> match (client_get_round r st_client).(client_round_pull_src) with None => True | Some src => (client_get_round r st_client').(client_round_pull_src) = Some src end.
  Proof. intros st_client st_client' r Hreach. revert r. induction Hreach as [|st st' st'' Hreach IH Hstep].
         - intro r. match goal with |- context[match ?x with _ => _ end] => destruct x eqn:Eecache end; easy.
         - client_step_case Hstep; subst st'; subst st''.
           all: intros r'; client_unfold; NatMap_case r' r; client_reduce; client_fold st; auto; try apply IH.
           + destruct Hpre. specialize (IH r).
             match goal with |- context[match ?x with _ => _ end] => destruct x eqn:Eecache end. 2: easy.
             rewrite IH in client_add_ecache_nodup0. destruct client_add_ecache_nodup0; [discriminate | auto].
  Qed.

  Lemma mcache_invariant : forall st_client st_client' r v, client_reachable st_client st_client' -> match NatMap_find v (client_get_round r st_client).(client_round_mcaches) with None => True | Some d => NatMap_find v (client_get_round r st_client').(client_round_mcaches) = Some d end.
  Proof. intros st_client st_client' r v Hreach. revert v. revert r. induction Hreach as [|st st' st'' Hreach IH Hstep].
         - intros r v. match goal with |- context[match ?x with _ => _ end] => destruct x eqn:Emcache end; easy.
         - client_step_case Hstep; subst st'; subst st''.
           all: intros r'; client_unfold; NatMap_case r' r; client_reduce; client_fold st; auto; try apply IH.
           + intro v; specialize (IH r v).
             match goal with |- context[match ?x with _ => _ end] => destruct x eqn:Emcache end. 2: easy.
             destruct Hpre. NatMap_case v ver. 2: easy. rewrite IH in client_add_mcache_nodup0. destruct client_add_mcache_nodup0; [discriminate | auto].
  Qed.

  Lemma ccache_invariant : forall st_client st_client' r, client_reachable st_client st_client' -> match (client_get_round r st_client).(client_round_max_ccache) with None => True | Some v => match (client_get_round r st_client').(client_round_max_ccache) with None => False | Some v' => v' >= v end end.
  Proof. intros st_client st_client' r Hreach. revert r. induction Hreach as [|st st' st'' Hreach IH Hstep].
         - intro r. match goal with |- context[match ?x with _ => _ end] => destruct x eqn:Eccache end; auto.
         - client_step_case Hstep; subst st'; subst st''.
           all: intros r'; client_unfold; NatMap_case r' r; client_reduce; client_fold st; auto; try apply IH.
           + specialize (IH r).
             match goal with |- context[match ?x with _ => _ end] => destruct x eqn:Eccache end. 2: easy.
             match type of IH with context[match ?x with _ => _ end] => destruct x eqn:Eccache' end. 2: contradiction.
             match goal with |- context[if ?p then _ else _] => destruct p eqn:Hle end. 2: easy. apply Compare_dec.leb_complete in Hle. lia.
  Qed.

  Lemma client_round_0_no_ecache : forall st_client, client_tree_valid st_client -> (client_get_round 0 st_client).(client_round_pull_src) = None.
  Proof. intros st_client Hval. pose proof (client_pull_lt _ 0 Hval) as Hlt.
         match type of Hlt with context[match ?x with _ => _ end] => destruct x eqn:Eecache end; [lia | easy].
  Qed.

  Lemma client_round_0_no_mcache : forall st_client v, client_tree_valid st_client -> NatMap_find v (client_get_round 0 st_client).(client_round_mcaches) = None.
  Proof. intros st_client v Hval. induction v.
         - pose proof (client_mcache_has_pred _ 0 0 Hval) as Hpred. cbn in Hpred.
           pose proof (client_round_0_no_ecache _ Hval) as Hecache. rewrite Hecache in Hpred.
           match goal with |- context[?x = None] => destruct x eqn:Emcache end. 1: specialize (Hpred ltac:(easy)); contradiction. easy.
         - pose proof (client_mcache_has_pred _ 0 (S v) Hval) as Hpred. cbn in Hpred. rewrite IHv in Hpred.
           match goal with |- context[?x = None] => destruct x eqn:Emcache end. 1: specialize (Hpred ltac:(easy)); contradiction. easy.
  Qed.

  Lemma client_round_0_no_ccache : forall st_client, client_tree_valid st_client -> (client_get_round 0 st_client).(client_round_max_ccache) = None.
  Proof. intros st_client Hval.
         match goal with |- context[?x = None] => destruct x eqn:Eccache end. 2: easy.
         pose proof (client_commit_valid _ 0 n Hval Eccache) as Hmcache.
         pose proof (client_round_0_no_mcache _ n Hval) as Hno_mcache.
         contradiction.
  Qed.

End ClientProps.
