From Coq Require Import
  List.
From advert.lib Require Import
  Decision.

Section Participant.
  Class BADO_Participant : Type := {
    bado_participant : list nat;
  }.
End Participant.

Section Leader.
  Context `{participant : !BADO_Participant}.

  Class BADO_Leader : Type := {
    bado_leader_at : nat -> nat;
    bado_leader_valid : forall r, In (bado_leader_at r) bado_participant;
    bado_leader_fair : forall nid r, In nid bado_participant -> exists r', r' >= r /\ bado_leader_at r' = nid;
  }.
End Leader.

Section Quorum.
  Inductive NodeType : Set :=
  | Synchronous
  | Asynchronous
  | Byzantine.

  Class BADO_NodeAssump : Type := {
    bado_node_assump : nat -> NodeType;
  }.

  Record Committee_Data : Type := {
    committee_quorums : list (list nat);
  }.

  Context `{participant : !BADO_Participant} `{node_assump : !BADO_NodeAssump}.

  Record Committee (committee_data : Committee_Data) : Type := {
    committee_valid : forall quorum nid, In quorum committee_data.(committee_quorums) -> In nid quorum -> In nid bado_participant;
    committee_safe : forall quorum quorum', In quorum committee_data.(committee_quorums) -> In quorum' committee_data.(committee_quorums) -> exists nid, In nid quorum /\ In nid quorum' /\ bado_node_assump nid <> Byzantine;
  }.

  Definition committee_live {data : Committee_Data} (comm : Committee data) := exists quorum, In quorum data.(committee_quorums) /\ forall nid, In nid quorum -> bado_node_assump nid = Synchronous.

  Definition is_quorum (data : Committee_Data) (voters : list nat) := exists quorum, In quorum data.(committee_quorums) /\ incl quorum voters.

  #[export] Instance is_quorum_dec (data : Committee_Data) (voters : list nat) : Decision (is_quorum data voters).
  Proof. unfold is_quorum. pose proof (Exists_exists (fun q => incl q voters) data.(committee_quorums)) as Hex; cbn in Hex. apply (decide_rewrite _ _ Hex). typeclasses eauto. Qed.

  Lemma is_quorum_superset (data : Committee_Data) (voters voters' : list nat) : is_quorum data voters -> incl voters voters' -> is_quorum data voters'.
  Proof. intros Hquorum Hincl. unfold is_quorum in Hquorum. destruct Hquorum as (quorum & Hquorum1 & Hquorum2). unfold is_quorum. exists quorum. split. 1: auto. eapply incl_tran. 1: apply Hquorum2. apply Hincl. Qed.

  Lemma quorum_overlap_exists_honest {data : Committee_Data} (comm : Committee data) (quorum quorum' : list nat) : is_quorum data quorum -> is_quorum data quorum' -> exists nid, In nid quorum /\ In nid quorum' /\ bado_node_assump nid <> Byzantine.
  Proof. unfold is_quorum. intros Hqm1 Hqm2. destruct Hqm1 as (qm1 & Hqm1_1 & Hqm1_2). destruct Hqm2 as (qm2 & Hqm2_1 & Hqm2_2).
         pose proof (committee_safe data comm qm1 qm2 Hqm1_1 Hqm2_1) as Hnid. destruct Hnid as (nid & Hnid1 & Hnid2 & Hnid3). exists nid; repeat split; auto.
  Qed.

  Lemma quorum_exists_honest {data : Committee_Data} (comm : Committee data) (quorum : list nat) : is_quorum data quorum -> exists nid, In nid quorum /\ bado_node_assump nid <> Byzantine.
  Proof. intro Hquorum. pose proof (quorum_overlap_exists_honest comm _ _ Hquorum Hquorum) as Hnid. destruct Hnid as (nid & Hnid1 & _ & Hnid2). exists nid; split; auto.
  Qed.

  Context `{leader : !BADO_Leader}.

  Class BADO_Config : Type := {
    bado_comm : Committee_Data;    
    bado_m_comm : nat -> Committee_Data;
    bado_ver_lim : nat;
    bado_to_quorums : list (list nat);
  }.

  Context `{config : !BADO_Config}.

  Definition is_to_quorum (voters : list nat) := exists to_quorum, In to_quorum config.(bado_to_quorums) /\ incl to_quorum voters.

  #[export] Instance is_to_quorum_dec (voters : list nat) : Decision (is_to_quorum voters).
  Proof. unfold is_quorum. pose proof (Exists_exists (fun q => incl q voters) config.(bado_to_quorums)) as Hex; cbn in Hex. apply (decide_rewrite _ _ Hex). typeclasses eauto. Qed.

  Lemma is_to_quorum_superset (voters voters' : list nat) : is_to_quorum voters -> incl voters voters' -> is_to_quorum voters'.
  Proof. intros Hquorum Hincl. unfold is_to_quorum in Hquorum. destruct Hquorum as (quorum & Hquorum1 & Hquorum2). unfold is_to_quorum. exists quorum. split. 1: auto. eapply incl_tran. 1: apply Hquorum2. apply Hincl. Qed.

  Class BADO_Assump : Type := {
    bado_comm_safe : Committee bado_comm;
    bado_comm_live : committee_live bado_comm_safe;
    bado_m_comm_safe : forall r, Committee (bado_m_comm r);
    bado_m_comm_live : forall r, (* bado_node_assump (bado_leader_at r) = Synchronous -> *) committee_live (bado_m_comm_safe r);
    bado_to_quorum_valid : forall quorum nid, In quorum config.(bado_to_quorums) -> In nid quorum -> In nid bado_participant;
    bado_to_quorum_safe : forall quorum, In quorum config.(bado_to_quorums) -> exists nid, In nid quorum /\ bado_node_assump nid = Synchronous;
    bado_to_quorum_live : forall quorum, In quorum bado_comm.(committee_quorums) -> exists quorum', In quorum' config.(bado_to_quorums) /\ incl quorum' quorum /\ forall nid, In nid quorum' -> bado_node_assump nid = Synchronous;
    bado_to_quorum_overlap : forall quorum quorum', In quorum bado_comm.(committee_quorums) -> In quorum' config.(bado_to_quorums) -> exists nid, In nid quorum /\ In nid quorum';
  }.

  Context `{assump : !BADO_Assump}.

  Lemma to_quorum_exists_sync (to_quorum : list nat) : is_to_quorum to_quorum -> exists nid, In nid to_quorum /\ bado_node_assump nid = Synchronous.
  Proof. intro Hquorum. destruct Hquorum as (qm & Hqm1 & Hqm2). destruct (bado_to_quorum_safe _ Hqm1) as (nid & Hnid1 & Hnid2). exists nid. split; auto. Qed.

  Lemma quorum_exists_sync_to_quorum (quorum : list nat) : is_quorum bado_comm quorum -> exists to_quorum, is_to_quorum to_quorum /\ incl to_quorum quorum /\ forall nid, In nid to_quorum -> In nid bado_participant /\ bado_node_assump nid = Synchronous.
  Proof. intro Hquorum. destruct Hquorum as (qm & Hqm1 & Hqm2). destruct (bado_to_quorum_live _ Hqm1) as (tqm & Htqm1 & Htqm2 & Htqm3). exists tqm. split. 2: split.
         - exists tqm; split; auto; apply incl_refl.
         - eapply incl_tran; try apply Hqm2; auto.
         - intros nid Hnid; split. 2: apply Htqm3; auto.
           eapply bado_to_quorum_valid. 1: apply Htqm1. auto.
  Qed.

  Lemma quorum_to_quorum_overlap (quorum : list nat) (quorum' : list nat) : is_quorum bado_comm quorum -> is_to_quorum quorum' -> exists nid, In nid quorum /\ In nid quorum'.
  Proof. intros Hquorum Hquorum'.
         destruct Hquorum as (qm & Hqm1 & Hqm2).
         destruct Hquorum' as (qm' & Hqm'1 & Hqm'2).
         pose proof (bado_to_quorum_overlap _ _ Hqm1 Hqm'1) as (x & Hx1 & Hx2).
         exists x; split; auto.
  Qed.

End Quorum.
