(* Extend the safety result in Safety.v to safety of DAG consensus log *)

From Coq Require Import
  List
  Lia
  Peano_dec.
From advert.lib Require Import
  Semantics
  Decision
  Tactics
  Maps.
From advert.specs Require Import
  UDAG
  LidoClient
  LidoDAG
  Config.
From advert.proofs Require Import
  Safety.
Import Notations.
Import ListNotations.

Section DAGLog.
  (* For each vertex in UDAG, define the log up to that vertex *)
  Context `{dag_config : !DAG_Config} `{participant : !BADO_Participant} `{node_assump : !BADO_NodeAssump} `{config : !BADO_Config}.

  (* The deduplication function *)

  (* It is more convenient to first define a function that deduplicates a reversed list *)

  Fixpoint dedup_rev (l : list dag_t) :=
    match l with
    | [] => []
    | x :: xs => if (decide (In x xs)) then dedup_rev xs else x :: dedup_rev xs
    end.

  Definition dedup l := rev (dedup_rev (rev l)).

  (* If l is prefix of l', then dedup l is prefix of dedup l' *)

  Lemma dedup_prefix l1 l2 : exists l2', dedup (l1 ++ l2) = dedup l1 ++ l2'.
  Proof.
    revert l1.
    induction l2.
    - intros l1; exists []; do 2 rewrite app_nil_r; auto.
    - intros l1; specialize (IHl2 (l1 ++ [a])).
      rewrite <- app_assoc in IHl2.
      cbn in IHl2.
      destruct IHl2 as (l2' & IHl2).
      unfold dedup in IHl2 at 2.
      rewrite rev_app_distr in IHl2.
      cbn [rev app dedup_rev] in IHl2.
      destruct (decide (In a (rev l1))).
      + fold (dedup l1) in IHl2.
        exists l2'; auto.
      + cbn in IHl2.
        rewrite <- app_assoc in IHl2.
        exists ([a] ++ l2').
        auto.
  Qed.

  Context `{leader : !BADO_Leader}.

  (* For each MCache in the LiDO-DAG state, define the expanded consensus log *)
  Definition lidodag_leader_log r v lidodag :=
  match client_get_log r v lidodag.(lidodag_lido) with Some l => map snd l | _ => [] end.

  Lemma lidodag_leader_log_prefix r r' v v' lidodag :
  lidodag_valid lidodag ->
  r < r' ->
  match (client_get_round r lidodag.(lidodag_lido)).(client_round_max_ccache) with None => False | Some v'' => v'' >= v end ->
  NatMap_find v' (client_get_round r' lidodag.(lidodag_lido)).(client_round_mcaches) <> None ->
  exists l, lidodag_leader_log r' v' lidodag = lidodag_leader_log r v lidodag ++ l.
  Proof.
    intros Hval Hlt Hccache Hmcache.
    pose proof (lidodag_lido_valid _ Hval) as Hval_lido.
    pose proof (log_match _ Hval_lido) as Hlog.
    unfold log_match_prop in Hlog.
    cond_case_auto Hccache; try contradiction.
    assert (r > 0).
    { destruct r; [|lia].
      rewrite (client_round_0_no_ccache _ Hval_lido) in Ex; discriminate.
    }
    pose proof (client_commit_valid _ _ _ Hval_lido Ex) as Hmcache'.
    pose proof (client_mcache_has_pred' _ r _ _ Hval_lido Hmcache' Hccache) as Hmcache''.
    specialize (Hlog r _ ltac:(right; split; auto; apply Hmcache'')) as Hlog1.
    cond_case_auto Hlog1; try contradiction.
    specialize (Hlog r' _ ltac:(right; split; [lia | apply Hmcache])) as Hlog2.
    cond_case_auto Hlog2; try contradiction.
    pose proof (committed_mcache_in_log _ _ _ _ _ _ _ Hval_lido Hlog1 Hlog2 Hlt) as Hprefix.
    rewrite Ex in Hprefix.
    specialize (Hprefix ltac:(auto)) as (log'' & Hlog3).
    unfold lidodag_leader_log.
    rewrite Ex0, Ex1.
    subst l0.
    rewrite map_app; eexists; reflexivity.
  Qed.

  Definition lidodag_expanded_id_log r v lidodag :=
  join (map (fun id => udag_get_closure id lidodag.(lidodag_dag)) (lidodag_leader_log r v lidodag)).

  Lemma lidodag_expanded_id_log_valid r v lidodag :
  lidodag_valid lidodag ->
  forall id, In id (lidodag_expanded_id_log r v lidodag) ->
  NatMap_find id lidodag.(lidodag_dag).(udag_verts) <> None.
  Proof.
    intros Hval id Hid.
    unfold lidodag_expanded_id_log in Hid.
    apply join_mem in Hid.
    destruct Hid as (l & Hl1 & Hl2).
    rewrite in_map_iff in Hl1.
    destruct Hl1 as (id' & Hid'1 & Hid'2).
    subst l.
    pose proof (lidodag_dag_valid _ Hval) as Hval_dag.
    apply (udag_closure_valid _ _ _ Hval_dag Hl2).
  Qed.

  Definition lidodag_expanded_log (r : nat) (v : nat) (lidodag : LidoDAG) : lidodag_valid lidodag -> list dag_t.
  Proof.
    intros Hval.
    remember (lidodag_expanded_id_log r v lidodag) as id_log.
    pose proof (lidodag_expanded_id_log_valid r v lidodag Hval) as Hval'.
    rewrite <- Heqid_log in Hval'.
    clear Heqid_log.
    induction id_log.
    - exact [].
    - specialize (IHid_log ltac:(intros id Hid; apply Hval'; right; auto)).
      specialize (Hval' _ ltac:(left; auto)).
      destruct (NatMap_find a (udag_verts (lidodag_dag lidodag))) eqn:Evert.
      2: contradiction.
      exact (u.(udag_vert_data) :: IHid_log).
  Defined.

  Definition lidodag_expanded_deduped_log (r : nat) (v : nat) (lidodag : LidoDAG) (Hval : lidodag_valid lidodag) : list dag_t :=
  dedup (lidodag_expanded_log r v lidodag Hval).
End DAGLog.
