From Coq Require Import
  List
  Lia
  Peano_dec.
From advert.lib Require Import
  Semantics
  Maps.
From advert.specs Require Import
  LidoClient
  Config.
Import Notations.
Import ListNotations.

Section Log.
  (* In this section we define the log of each MCache. *)
  Inductive client_is_log (st_client : ClientTree) : (nat * nat) -> list (nat * nat * nat) -> Prop :=
  | client_is_log_root : client_is_log st_client (0, 0) []
  | client_is_log_pull : forall r src m log, (client_get_round r st_client).(client_round_pull_src) = Some src -> client_is_log st_client src log -> NatMap_find 0 (client_get_round r st_client).(client_round_mcaches) = Some m -> client_is_log st_client (r, 0) (log ++ [((r, 0), m)])
  | client_is_log_app : forall r v m log, client_is_log st_client (r, v) log -> NatMap_find (S v) (client_get_round r st_client).(client_round_mcaches) = Some m -> client_is_log st_client (r, (S v)) (log ++ [((r, (S v)), m)]).

  Context `{participant : !BADO_Participant} `{leader : !BADO_Leader}.

  Lemma log_invariant : forall st_client st_client' ts log1 log2, client_tree_valid st_client -> client_reachable st_client st_client' -> client_is_log st_client ts log1 -> client_is_log st_client' ts log2 -> log1 = log2.
  Proof. intros st_client st_client' ts log1 log2 Hval Hreach.
         assert (Hval' : client_tree_valid st_client'). { eapply valid_reach_valid. 1: apply Hval. apply Hreach. }
         revert log2. revert log1. destruct ts as [p q]. revert q. remember p as p'. assert (Hle : p' <= p) by lia. revert Hle. clear Heqp'. revert p'. induction p.
         - intros p' Hp'. assert (p' = 0) by lia; subst p'. induction q.
           + intros log1 log2 Hlog1 Hlog2. inversion Hlog1 as [| r src m log Hecache Hlog3 Hmcache | r v m log Hlog3 Hmcache].
             * inversion Hlog2 as [| r src m log Hecache Hlog3 Hmcache | r v m log Hlog3 Hmcache]. 1: easy.
               pose proof (client_round_0_no_mcache _ 0 Hval') as Hmcache'. rewrite Hmcache' in Hmcache. discriminate.
             * pose proof (client_round_0_no_mcache _ 0 Hval) as Hmcache'. rewrite Hmcache' in Hmcache. discriminate.
           + intros log1 log2 Hlog1 Hlog2. inversion Hlog1 as [| r src m log Hecache Hlog3 Hmcache | r v m log Hlog3 Hmcache].
             pose proof (client_round_0_no_mcache _ (S q) Hval) as Hmcache'. rewrite Hmcache' in Hmcache. discriminate.
         - intros p' Hp'. assert (Heq : p' <= p \/ p' = S p) by lia. destruct Heq as [Heq | Heq]. 1: apply IHp; auto. subst p'. induction q.
           + intros log1 log2 Hlog1 Hlog2. inversion Hlog1 as [| r src m log Hecache Hlog3 Hmcache | r v m log Hlog3 Hmcache].
             inversion Hlog2 as [| r' src' m' log' Hecache' Hlog4 Hmcache' | r' v' m' log' Hlog4 Hmcache'].
             pose proof (ecache_invariant _ _ (S p) Hreach) as Hinv. rewrite Hecache in Hinv. rewrite Hinv in Hecache'. inversion Hecache'; subst src'.
             pose proof (mcache_invariant _ _ (S p) 0 Hreach) as Hinv'. rewrite Hmcache in Hinv'. rewrite Hinv' in Hmcache'. inversion Hmcache'; subst m'.
             pose proof (client_pull_lt _ (S p) Hval) as Hlt. rewrite Hecache in Hlt. specialize (IHp (fst src) ltac:(lia) (snd src)).
             destruct src as [x y]; cbn in IHp. specialize (IHp _ _ Hlog3 Hlog4). subst log'. easy.
           + intros log1 log2 Hlog1 Hlog2. inversion Hlog1 as [| r src m log Hecache Hlog3 Hmcache | r v m log Hlog3 Hmcache].
             inversion Hlog2 as [| r' src' m' log' Hecache' Hlog4 Hmcache' | r' v' m' log' Hlog4 Hmcache'].
             specialize (IHq _ _ Hlog3 Hlog4). subst log'.
             pose proof (mcache_invariant _ _ (S p) (S q) Hreach) as Hinv. rewrite Hmcache in Hinv. rewrite Hinv in Hmcache'. inversion Hmcache'; subst m'.
             easy.
  Qed.

  Lemma log_uniq : forall st_client r v log1 log2, client_tree_valid st_client -> client_is_log st_client (r, v) log1 -> client_is_log st_client (r, v) log2 -> log1 = log2.
  Proof. intros st_client r v log1 log2 Hval Hlog1 Hlog2. assert (Hreach : client_reachable st_client st_client). { apply reachable_self. }
         eapply log_invariant; [apply Hval | apply Hreach | apply Hlog1 | apply Hlog2].
  Qed.

  Lemma log_exist : forall st_client r v, client_tree_valid st_client -> NatMap_find v (client_get_round r st_client).(client_round_mcaches) <> None -> exists log, client_is_log st_client (r, v) log.
  Proof. intros st_client r v Hval. revert v. remember r as r'. assert (Hle : r' <= r) by lia. clear Heqr'. revert Hle. revert r'. induction r.
         - intros r' Hle. assert (r' = 0) by lia; subst r'. intros v Hmcache.
           pose proof (client_round_0_no_mcache _ v Hval); contradiction.
         - intros r' Hle. assert (Heq : r' <= r \/ r' = S r) by lia. destruct Heq as [Heq | Heq]. 1: apply IHr; auto. subst r'. induction v.
           + intro Hmcache.
             pose proof (client_mcache_has_pred _ _ _ Hval Hmcache) as Hecache. cbn in Hecache.
             pose proof (client_pull_lt _ (S r) Hval) as Hecache'.
             match type of Hecache with context[?x = None] => destruct x eqn:Eecache end. 2: contradiction.
             specialize (IHr (fst p) ltac:(lia) (snd p)).
             pose proof (client_pull_valid _ (S r) (fst p) Hval) as Hmcache'. rewrite Eecache in Hmcache'. specialize (Hmcache' ltac:(easy)).
             match type of Hmcache with context[?x = None] => destruct x eqn:Emcache end. 2: contradiction.
             destruct p as [x y]. destruct Hmcache' as [Hmcache' | Hmcache']. 
             * eexists; eapply client_is_log_pull; try apply Eecache; try apply Emcache. cbn in Hmcache'; destruct Hmcache'; subst x; subst y. apply client_is_log_root.
             * destruct Hmcache' as (_ & Hmcache'). specialize (IHr Hmcache'). cbn in IHr. destruct IHr as (log & Hlog). eexists; eapply client_is_log_pull; try apply Eecache; try apply Emcache. apply Hlog.
           + intro Hmcache.
             pose proof (client_mcache_has_pred _ _ _ Hval Hmcache) as Hmcache'. cbn in Hmcache'. specialize (IHv Hmcache'). destruct IHv as (log & Hlog).
             match type of Hmcache with context[?x = None] => destruct x eqn:Emcache end. 2: contradiction.
             eexists. eapply client_is_log_app. 1: apply Hlog. apply Emcache.
  Qed.

  Lemma log_exist' : forall st_client r v, client_tree_valid st_client -> ((r = 0 /\ v = 0) \/ (r > 0 /\ NatMap_find v (client_get_round r st_client).(client_round_mcaches) <> None)) -> exists log, client_is_log st_client (r, v) log.
  Proof. intros st_client r v Hval Hmcache. destruct Hmcache as [Hmcache | Hmcache]. 1: destruct Hmcache; subst r v; exists []; apply client_is_log_root. destruct Hmcache; apply log_exist; auto.
  Qed.

  (* Every MCache in the tree is now associated with a log, and that log is unique and invariant. *)

  Lemma log_exist_mcache : forall st_client r v log, r > 0 -> client_is_log st_client (r, v) log -> NatMap_find v (client_get_round r st_client).(client_round_mcaches) <> None.
  Proof. intros st_client r v log Hpos Hlog. inversion Hlog as [| r' src m log' Hecache Hlog' Hmcache | r' v' m log' Hlog' Hmcache]. 1: lia.
         - rewrite Hmcache. easy.
         - rewrite Hmcache. easy.
  Qed.

  Lemma log_exist_mcache' : forall st_client r v log, client_tree_valid st_client -> client_is_log st_client (r, v) log -> (r = 0 /\ v = 0) \/ (r > 0 /\ NatMap_find v (client_get_round r st_client).(client_round_mcaches) <> None).
  Proof. intros st_client r v log Hval Hlog. inversion Hlog as [| r' src m log' Hecache Hlog' Hmcache | r' v' m log' Hlog' Hmcache]. 1: left; easy.
         all: subst r'; right; split; [|rewrite Hmcache; easy]; destruct (eq_nat_dec r 0); [|lia]; subst r; rewrite client_round_0_no_mcache in Hmcache; auto; discriminate.
  Qed.

  Lemma log_invariant' : forall st_client st_client' ts log, client_tree_valid st_client -> client_reachable st_client st_client' -> client_is_log st_client ts log -> client_is_log st_client' ts log.
  Proof. intros st_client st_client' ts log Hval Hreach Hlog. destruct ts as (r, v). apply log_exist_mcache' in Hlog as Hmcache; auto.
         assert (Hmcache' : (r = 0 /\ v = 0) \/ (r > 0 /\ NatMap_find v (client_round_mcaches (client_get_round r st_client')) <> None)).
         { destruct Hmcache as [Hmcache | (Hpos & Hmcache)]. 1: left; easy. right. split. 1: easy. pose proof (mcache_invariant _ _ r v Hreach) as Hmcache'. match type of Hmcache' with context[match ?x with _ => _ end] => destruct x eqn:Emcache end. 2: contradiction. rewrite Hmcache'. easy. }
         apply log_exist' in Hmcache'. 2: eapply valid_reach_valid; [apply Hval | apply Hreach]. destruct Hmcache' as (log' & Hlog').
         pose proof (log_invariant _ _ _ _ _ Hval Hreach Hlog Hlog'). subst log'. auto.
  Qed.

  Lemma same_round_mcache_in_log : forall st_client r v v' log log', client_tree_valid st_client -> client_is_log st_client (r, v) log -> client_is_log st_client (r, v') log' -> r > 0 -> v <= v' -> exists log'', log' = log ++ log''.
  Proof. intros st_client r v v' log log' Hval. revert log'. revert log. revert v'.
         induction v'.
         - intros log log' Hlog Hlog' Hgt Hle. inversion Hlog as [| r' src m log'' Hecache Hlog'' Hmcache | r' v' m log'' Hlog'' Hmcache]. 1: lia.
           + subst v. pose proof (log_uniq _ _ _ _ _ Hval Hlog Hlog'); subst log'; subst log. exists []. rewrite <- app_nil_end. easy.
           + subst v. lia.
         - intros log log' Hlog Hlog' Hgt Hle. assert (Heq : v <= v' \/ v = S v') by lia. destruct Heq as [Heq | Heq].
           + inversion Hlog' as [| r' src m log'' Hecache Hlog'' Hmcache | r' v'' m log'' Hlog'' Hmcache]. specialize (IHv' _ _ Hlog Hlog'' Hgt Heq). destruct IHv' as (log''' & Hlog'''). subst log''. rewrite <- app_assoc. eexists. reflexivity.
           + subst v. inversion Hlog' as [| r' src m log'' Hecache Hlog'' Hmcache | r' v'' m log'' Hlog'' Hmcache]. pose proof (log_uniq _ _ _ _ _ Hval Hlog Hlog'). subst log'. subst log. exists []. rewrite <- app_nil_end. easy.
  Qed.

  Lemma committed_mcache_in_log : forall st_client r r' v v' log log', client_tree_valid st_client -> client_is_log st_client (r, v) log -> client_is_log st_client (r', v') log' -> r < r' -> match (client_get_round r st_client).(client_round_max_ccache) with None => True | Some t => v <= t -> exists log'', log' = log ++ log'' end.
  Proof. intros st_client r r' v v' log log' Hval Hlog Hlog' Hlt. revert Hlog'. revert Hlog. revert log'. revert log. revert v'. revert v. remember r' as r_ub. rewrite Heqr_ub in Hlt. rewrite Heqr_ub. assert (Hle : r' <= r_ub) by lia. clear Heqr_ub. revert Hle. revert Hlt. revert r'. revert r. induction r_ub.
         - lia.
         - intros r r' Hlt Hle v v' log log' Hlog Hlog'. assert (Heq : r' <= r_ub \/ r' = S r_ub) by lia. destruct Heq as [Heq | Heq]. 1: apply (IHr_ub _ _ Hlt Heq _ _ _ _ Hlog Hlog'). subst r'.
           assert (Hmcache : NatMap_find 0 (client_get_round (S r_ub) st_client).(client_round_mcaches) <> None). { pose proof (log_exist_mcache _ (S r_ub) _ _ ltac:(lia) Hlog') as Hmcache. apply (client_mcache_has_pred' _ _ 0 _ Hval Hmcache ltac:(lia)). }
           pose proof (log_exist _ _ _ Hval Hmcache) as Hlog''. destruct Hlog'' as (log'' & Hlog'').
           pose proof (same_round_mcache_in_log _ _ _ _ _ _ Hval Hlog'' Hlog' ltac:(lia) ltac:(lia)) as Hlog_mid2. destruct Hlog_mid2 as (log_mid2 & Hlog_mid2). subst log'.
           inversion Hlog'' as [| r' src m log_mid1 Hecache Hlog_mid1 Hmcache' |]. destruct src as [x y].
           pose proof (client_pull_committed _ r (S r_ub) Hval) as Horder.
           pose proof (client_pull_lt _ (S r_ub) Hval) as Horder'. rewrite Hecache in Horder'.
           match goal with |- context[match ?x with _ => _ end] => destruct x eqn:Eccache end. 2: easy. rewrite Hecache in Horder. specialize (Horder ltac:(auto)). intro Hle'. cbn in Horder, Horder'.
           destruct Horder as [Horder | Horder].
           + specialize (IHr_ub r x ltac:(auto) ltac:(lia) _ _ _ _ Hlog Hlog_mid1). rewrite Eccache in IHr_ub. specialize (IHr_ub ltac:(auto)).
             destruct IHr_ub as (log_mid0 & Hlog_mid0). subst log_mid1. rewrite <- app_assoc. rewrite <- app_assoc. eexists. reflexivity.
           + destruct Horder as (Hx & Hy). subst x.
             assert (Hpos : r > 0). { assert (Hpos' : r = 0 \/ r > 0) by lia. destruct Hpos' as [Hpos' | Hpos']. 2: auto. subst r. exfalso. pose proof (client_round_0_no_ccache _ Hval) as Hccache. rewrite Hccache in Eccache. discriminate. }
             pose proof (same_round_mcache_in_log _ _ _ _ _ _ Hval Hlog Hlog_mid1 ltac:(auto) ltac:(lia)) as Hlog_mid0.
             destruct Hlog_mid0 as (log_mid0 & Hlog_mid0). subst log_mid1. rewrite <- app_assoc. rewrite <- app_assoc. eexists. reflexivity.
  Qed.

  (* Hence the log of every future MCache must extend from the log of existing CCaches. *)
  (* This also implies that all CCaches must form a linear chain *)

End Log.

Section LogMatch.
  (* In this section we show that the log defined above matches the log defined computationally in the cache tree *)
  Definition log_match_prop (st_client : ClientTree) := forall r v, ((r = 0 /\ v = 0) \/ (r > 0 /\ NatMap_find v (client_get_round r st_client).(client_round_mcaches) <> None)) -> match (client_get_log r v st_client) with None => False | Some log => client_is_log st_client (r, v) log end.

  Definition log_match_inv_prop (st_client : ClientTree) := forall r v, match (client_get_log r v st_client) with None => True | Some log => client_is_log st_client (r, v) log end.

  Lemma log_match_init : log_match_prop client_tree_null.
  Proof. unfold log_match_prop. intros r v Hmcache. destruct Hmcache as [Hmcache | Hmcache]. 2: cbn in Hmcache; destruct Hmcache; contradiction.
         destruct Hmcache; subst r v. cbn. apply client_is_log_root.
  Qed.

  Context `{participant : !BADO_Participant} `{leader : !BADO_Leader}.

  Lemma log_match_step : forall st_client st_client', client_tree_valid st_client -> log_match_prop st_client -> client_step st_client st_client' -> log_match_prop st_client'.
  Proof. intros st st' Hval Hmatch Hstep. assert (Hval' : client_tree_valid st'). { eapply valid_state_step; [apply Hval | apply Hstep]. } pose Hstep as Hstep'. client_step_case Hstep; clear Hstep'; subst st st'.
         all: unfold log_match_prop; intros r' v.
         5,6,7,8: intro Hmcache; unfold client_get_round, client_issue_pull, client_issue_invoke, client_issue_push, client_drop_cmd in Hmcache; cbn [fst] in Hmcache; fold (client_get_round r' st_client) in Hmcache; apply Hmatch in Hmcache; try (unfold client_issue_pull at 1); try (unfold client_issue_invoke at 1); try (unfold client_issue_push at 1); try (unfold client_drop_cmd at 1); try (unfold client_add_proposed_val at 1); unfold client_get_log, client_round_get_logs, client_get_logs, client_set_cmd; cbn [snd client_mcache_log]; client_fold st_client; destruct (client_get_log r' v st_client); try contradiction; eapply log_invariant'; [apply Hval | eapply reachable_step; [apply reachable_self | apply Hstep] | apply Hmcache].

         (* Add ECache *)
         - intro Hmcache. assert (Hmcache' : (r' = 0 /\ v = 0) \/ (r' > 0 /\ NatMap_find v (client_round_mcaches (client_get_round r' st_client)) <> None)).
           { destruct Hmcache as [Hmcache | Hmcache]. 1: left; auto. right.
             unfold client_get_round, client_add_ecache, client_round_add_ecache in Hmcache. cbn [fst snd] in Hmcache. destruct (eq_nat_dec r' r).
             - subst r'. rewrite (NatMap_gse _ r r) in Hmcache; auto.
             - rewrite (NatMap_gso _ r r') in Hmcache; auto.
           }
           unfold client_add_ecache at 1, client_set_cmd, client_get_log, client_round_get_logs, client_get_logs; cbn [snd client_mcache_log]; client_fold st_client.
           apply Hmatch in Hmcache' as Hlog. destruct (client_get_log r' v st_client). 2: contradiction. eapply log_invariant'; [apply Hval | eapply reachable_step; [apply reachable_self | apply Hstep] | apply Hlog].

         (* Add MCache *)
         - intro Hmcache. assert (Hmcache' : (((r' = 0 /\ v = 0) \/ (r' > 0 /\ NatMap_find v (client_round_mcaches (client_get_round r' st_client)) <> None)) /\ client_get_log r' v (client_add_mcache r ver data st_client) = client_get_log r' v st_client) \/ (r' = r /\ v = ver)).
           { destruct Hmcache as [Hmcache | Hmcache].
             - left. split. 1: left; auto. destruct Hmcache; subst r'; subst v. assert (Hneq : r <> 0) by (destruct Hpre; lia). unfold client_add_mcache, client_add_log, client_add_log', client_get_log, client_round_get_logs, client_get_logs. cbn [snd client_mcache_log]. rewrite (NatMap_gso _ r 0); auto.
             - destruct (eq_nat_dec r' r).
               + subst r'. destruct (eq_nat_dec v ver).
                 * subst v. right. split; easy.
                 * left. unfold client_get_round, client_add_mcache, client_add_log, client_round_add_mcache in Hmcache; cbn [fst] in Hmcache. rewrite (NatMap_gse _ r r) in Hmcache; auto. cbn [client_round_mcaches] in Hmcache. rewrite (NatMap_gso _ ver v) in Hmcache; auto. split. 1: right; auto.
                   unfold client_get_log, client_add_mcache, client_add_log, client_add_log', client_round_get_logs, client_get_logs; cbn [snd client_mcache_log]. rewrite (NatMap_gse _ r r); auto. rewrite (NatMap_gso _ ver v); auto.
               + left. unfold client_get_round, client_add_mcache, client_add_log, client_round_add_mcache in Hmcache; cbn [fst] in Hmcache. rewrite (NatMap_gso _ r r') in Hmcache; auto. fold (client_get_round r' st_client) in Hmcache. split. 1: right; auto.
                 unfold client_get_log, client_add_mcache, client_add_log, client_add_log', client_round_get_logs, client_get_logs; cbn [snd client_mcache_log]. rewrite (NatMap_gso _ r r'); auto.
           }
           destruct Hmcache' as [(Hmcache' & Hlog_eq) | Hmcache'].
           + rewrite Hlog_eq. apply Hmatch in Hmcache' as Hlog. destruct (client_get_log r' v st_client). 2: contradiction. eapply log_invariant'; [apply Hval | eapply reachable_step; [apply reachable_self | apply Hstep] | apply Hlog].
           + destruct Hmcache'; subst r' v. clear Hmcache. unfold client_add_mcache at 1, client_add_log, client_add_log', client_get_log, client_round_get_logs, client_get_logs; cbn [fst snd client_mcache_log]. rewrite (NatMap_gse _ r r); auto; rewrite (NatMap_gse _ ver ver); auto.
             destruct Hpre. destruct ver eqn:Ev.
             * destruct (client_round_pull_src (client_get_round r st_client)) eqn:Eecache. 2: contradiction. eapply client_is_log_pull.
               -- unfold client_get_round, client_add_mcache, client_add_log, client_round_add_mcache; cbn [fst]. rewrite (NatMap_gse _ r r); auto. cbn [client_round_pull_src]. apply Eecache.
               -- unfold client_get_round, client_set_cmd; cbn [fst client_mcache_log]. rewrite (NatMap_gse _ r r); auto. unfold client_round_add_mcache; cbn [client_round_pull_src]. fold (client_get_round r st_client). rewrite Eecache. client_fold st_client.
                  pose proof (client_pull_valid _ r (fst p) Hval) as Hpull. rewrite Eecache in Hpull. specialize (Hpull ltac:(easy)). apply Hmatch in Hpull as Hpull'.
                  destruct (client_get_log (fst p) (snd p) st_client). 2: contradiction. replace (fst p, snd p) with p in Hpull' by (destruct p; easy).
                  eapply log_invariant'; [apply Hval | eapply reachable_step; [apply reachable_self | apply Hstep] | apply Hpull'].
               -- unfold client_get_round, client_add_mcache, client_add_log; cbn [fst]. rewrite (NatMap_gse _ r r); auto.
             * destruct (NatMap_find n (client_round_mcaches (client_get_round r st_client))) eqn:Emcache. 2: contradiction. eapply client_is_log_app.
               -- unfold client_set_cmd; cbn [client_mcache_log]. client_fold st_client.
                  pose proof (Hmatch r n ltac:(right; split; [auto | rewrite Emcache; auto])) as Hlog. destruct (client_get_log r n st_client). 2: contradiction. eapply log_invariant'; [apply Hval | eapply reachable_step; [apply reachable_self | apply Hstep] | apply Hlog].
               -- unfold client_get_round, client_add_mcache, client_add_log; cbn [fst]. rewrite (NatMap_gse _ r r); auto. unfold client_round_add_mcache; cbn [client_round_mcaches]. rewrite (NatMap_gse _ (S n) (S n)); auto.

         (* Add CCache *)
         - intro Hmcache. assert (Hmcache' : (r' = 0 /\ v = 0) \/ (r' > 0 /\ NatMap_find v (client_round_mcaches (client_get_round r' st_client)) <> None)).
           { destruct Hmcache as [Hmcache | Hmcache]. 1: left; auto. right.
             unfold client_get_round, client_add_ccache, client_round_add_ccache in Hmcache. cbn [fst snd] in Hmcache. destruct (eq_nat_dec r' r).
             - subst r'. rewrite (NatMap_gse _ r r) in Hmcache; auto.
             - rewrite (NatMap_gso _ r r') in Hmcache; auto.
           }
           unfold client_add_ccache at 1, client_set_cmd, client_get_log, client_round_get_logs, client_get_logs; cbn [snd client_mcache_log]. client_fold st_client.
           apply Hmatch in Hmcache' as Hlog. destruct (client_get_log r' v st_client). 2: contradiction. eapply log_invariant'; [apply Hval | eapply reachable_step; [apply reachable_self | apply Hstep] | apply Hlog].

         (* Add TCache *)
         - intro Hmcache. assert (Hmcache' : (r' = 0 /\ v = 0) \/ (r' > 0 /\ NatMap_find v (client_round_mcaches (client_get_round r' st_client)) <> None)).
           { destruct Hmcache as [Hmcache | Hmcache]. 1: left; auto. right. apply Hmcache. }
           unfold client_add_tcache at 1, client_set_cmd, client_get_log, client_round_get_logs, client_get_logs; cbn [snd client_mcache_log]. client_fold st_client.
           apply Hmatch in Hmcache' as Hlog. destruct (client_get_log r' v st_client). 2: contradiction. eapply log_invariant'; [apply Hval | eapply reachable_step; [apply reachable_self | apply Hstep] | apply Hlog].
  Qed.

  Lemma log_match : forall (st_client : ClientTree), client_tree_valid st_client -> log_match_prop st_client.
  Proof. intros st_client Hval. induction Hval as [| st st' Hval IH Hstep].
         - apply log_match_init.
         - eapply log_match_step; [apply Hval | apply IH | apply Hstep].
  Qed.

  Lemma log_match_inv_init : log_match_inv_prop client_tree_null.
  Proof. unfold log_match_inv_prop, client_tree_null, client_data_null. intros r v. unfold client_get_log, client_round_get_logs, client_get_logs. cbn [snd client_mcache_log].
         destruct (eq_nat_dec 0 r).
         - rewrite (NatMap_gse _ 0 r); auto. subst r. NatMap_cmp v 0.
           + subst v. apply client_is_log_root.
           + cbn. easy.
         - rewrite (NatMap_gso _ 0 r); auto. cbn. easy.
  Qed.

  Lemma log_match_inv_step : forall st_client st_client', client_tree_valid st_client -> log_match_inv_prop st_client -> client_step st_client st_client' -> log_match_inv_prop st_client'.
  Proof. intros st st' Hval Hmatch Hstep. assert (Hval' : client_tree_valid st'). { eapply valid_state_step; [apply Hval | apply Hstep]. } pose Hstep as Hstep'. client_step_case Hstep; clear Hstep'; subst st st'.
         all: unfold log_match_inv_prop; intros r' v.

         5,6,7,8: try (unfold client_issue_pull at 1); try (unfold client_issue_invoke at 1); try (unfold client_issue_push at 1); try (unfold client_drop_cmd at 1); try (unfold client_add_proposed_val at 1); unfold client_set_cmd, client_get_log, client_round_get_logs, client_get_logs; cbn [snd client_mcache_log]; client_fold st_client; specialize (Hmatch r' v); destruct (client_get_log r' v st_client) eqn:Elog; try easy; eapply log_invariant'; [apply Hval | eapply reachable_step; [apply reachable_self | apply Hstep] | apply Hmatch].

         (* Add ECache *)
         - unfold client_add_ecache at 1; unfold client_get_log, client_set_cmd, client_round_get_logs, client_get_logs; cbn [snd client_mcache_log]. client_fold st_client.
           destruct (client_get_log r' v st_client) eqn: Elog. 2: easy. specialize (Hmatch r' v). rewrite Elog in Hmatch. eapply log_invariant'; [apply Hval | eapply reachable_step; [apply reachable_self | apply Hstep] | apply Hmatch].

         (* Add MCache *)
         - unfold client_add_mcache at 1; unfold client_get_log, client_set_cmd, client_round_get_logs, client_get_logs, client_add_log, client_add_log'. cbn [snd client_mcache_log]. destruct (eq_nat_dec r' r).
           + subst r'; rewrite (NatMap_gse _ r r); auto. NatMap_case v ver.
             * unfold client_get_round; cbn [fst]. rewrite (NatMap_gse _ r r); auto. fold (client_get_round r st_client). unfold client_round_add_mcache; destruct Hpre. destruct ver eqn:Ev.
               -- cbn [client_round_pull_src]. destruct (client_round_pull_src (client_get_round r st_client)) eqn:Eecache. 2: contradiction. unfold client_get_log, client_round_get_logs, client_get_logs; cbn [snd client_mcache_log]. client_fold st_client.
                  pose proof (client_pull_valid _ r (fst p) Hval) as Hpull. rewrite Eecache in Hpull. specialize (Hpull ltac:(easy)). apply log_match in Hpull; auto. destruct (client_get_log (fst p) (snd p) st_client). 2: contradiction.
                  eapply client_is_log_pull.
                  ++ unfold client_get_round, client_add_mcache, client_add_log, client_round_add_mcache; cbn [fst]; rewrite (NatMap_gse _ r r); auto; cbn [client_round_pull_src]. apply Eecache.
                  ++ replace (fst p, snd p) with p in Hpull by (destruct p; easy). eapply log_invariant'; [apply Hval | eapply reachable_step; [apply reachable_self | apply Hstep] | apply Hpull].
                  ++ unfold client_get_round, client_add_mcache, client_add_log, client_round_add_mcache; cbn [fst]; rewrite (NatMap_gse _ r r); auto.
               -- unfold client_get_log, client_round_get_logs, client_get_logs; cbn [snd client_mcache_log]. client_fold st_client.
                  pose proof (log_match _ Hval r n ltac:(right; split; auto)) as Hlog. destruct (client_get_log r n st_client). 2: contradiction.
                  eapply client_is_log_app.
                  ++ eapply log_invariant'; [apply Hval | eapply reachable_step; [apply reachable_self | apply Hstep] | apply Hlog].
                  ++ unfold client_get_round, client_add_mcache, client_add_log, client_round_add_mcache; cbn [fst]; rewrite (NatMap_gse _ r r); auto; cbn [client_round_mcaches]; rewrite (NatMap_gse _ (S n) (S n)); auto.
             * unfold client_round_get_logs, client_get_logs; cbn [snd client_mcache_log]. client_fold st_client.
               destruct (client_get_log r v st_client) eqn:Elog. 2: easy. specialize (Hmatch r v). rewrite Elog in Hmatch. eapply log_invariant'; [apply Hval | eapply reachable_step; [apply reachable_self | apply Hstep] | apply Hmatch].
           + rewrite (NatMap_gso _ r r'); auto. unfold client_round_get_logs, client_get_logs; cbn [snd client_mcache_log]. client_fold st_client.
             destruct (client_get_log r' v st_client) eqn:Elog. 2: easy. specialize (Hmatch r' v). rewrite Elog in Hmatch. eapply log_invariant'; [apply Hval | eapply reachable_step; [apply reachable_self | apply Hstep] | apply Hmatch].

         (* Add CCache *)
         - unfold client_add_ccache at 1; unfold client_get_log, client_set_cmd, client_round_get_logs, client_get_logs; cbn [snd client_mcache_log]. client_fold st_client.
           destruct (client_get_log r' v st_client) eqn: Elog. 2: easy. specialize (Hmatch r' v). rewrite Elog in Hmatch. eapply log_invariant'; [apply Hval | eapply reachable_step; [apply reachable_self | apply Hstep] | apply Hmatch].

         (* Add TCache *)
         - unfold client_add_tcache at 1; unfold client_get_log, client_set_cmd, client_round_get_logs, client_get_logs; cbn [snd client_mcache_log]. client_fold st_client.
           destruct (client_get_log r' v st_client) eqn: Elog. 2: easy. specialize (Hmatch r' v). rewrite Elog in Hmatch. eapply log_invariant'; [apply Hval | eapply reachable_step; [apply reachable_self | apply Hstep] | apply Hmatch].
  Qed.

  Lemma log_match_inv : forall (st_client : ClientTree), client_tree_valid st_client -> log_match_inv_prop st_client.
  Proof. intros st_client Hval. induction Hval as [| st st' Hval IH Hstep].
         - apply log_match_inv_init.
         - eapply log_match_inv_step; [apply Hval | apply IH | apply Hstep].
  Qed.

End LogMatch.
