(**********************************************************************)
(*                                                                    *)
(*                               Advert                               *)
(*                                                                    *)
(*                  Jieung Kim (jieung.kim@yale.edu)                  *)
(*                 Wolf Honore (wolf.honore@yale.edu)                 *)
(*                                                                    *)
(*                         Yale Flint Group                           *)
(*                                                                    *)
(**********************************************************************)

From Coq Require Import
  List
  ZArith.

#[local] Open Scope Z.
Set Default Goal Selector "!".

(** MathClasses-style "Decision" class. *)

Class Decision (P: Prop) := decide: {P} + {~P}.
Arguments decide P {_}.

Definition isTrue P `{Decision P} :=
  if decide P then true else false.

Definition isTrue_correct P `{Decision P}:
  isTrue P = true -> P.
Proof.
  unfold isTrue.
  destruct (decide P).
  - tauto.
  - discriminate.
Qed.

Ltac decision :=
  eapply isTrue_correct;
  reflexivity.

Ltac obviously H P :=
  assert (H: P) by decision.

Ltac ensure P :=
  let H := fresh "H" in
  obviously H P; clear H.

Lemma decision_decidable P `{Decision P} : Decidable.decidable P.
Proof. hnf; destruct (decide P); auto. Qed.

#[export] Hint Extern 10 (Decidable.decidable _) =>
  apply decision_decidable; typeclasses eauto : core.

(** * Instances *)

#[export] Instance decide_true : Decision True.
Proof. hnf; auto. Qed.

#[export] Instance decide_false : Decision False.
Proof. hnf; auto. Qed.

#[export] Instance decide_Zeq (m n: Z): Decision (m = n) := {decide := Z.eq_dec m n}.
#[export] Instance decide_Zle (m n: Z): Decision (m <= n) := {decide := Z_le_dec m n}.
#[export] Instance decide_Zlt (m n: Z): Decision (m < n) := {decide := Z_lt_dec m n}.
#[export] Instance decide_Zge (m n: Z): Decision (m >= n) := {decide := Z_ge_dec m n}.
#[export] Instance decide_Zgt (m n: Z): Decision (m > n) := {decide := Z_gt_dec m n}.
#[export] Instance decide_nateq m n: Decision (m = n) := { decide := eq_nat_dec m n }.
#[export] Instance decide_natle m n: Decision (m <= n)%nat := { decide := le_dec m n }.
#[export] Instance decide_natlt m n: Decision (m < n)%nat := { decide := lt_dec m n }.
#[export] Instance decide_poseq (m n: positive): Decision (m = n) := Pos.eq_dec m n.
#[export] Instance decide_booleq b1 b2: Decision (b1 = b2) := Bool.bool_dec b1 b2.
#[export] Instance decide_bool_lt b1 b2 : Decision (Bool.lt b1 b2).
Proof. destruct b1, b2; typeclasses eauto. Qed.

#[export] Instance decide_Neq (m n: N): Decision (m = n) := {decide := N.eq_dec m n}.

#[export] Instance decide_posle (m n: positive): Decision (Pos.le m n).
Proof.
  destruct (Pos.leb m n) eqn:leb.
  - left; apply Pos.leb_le, leb.
  - right; rewrite <- Pos.leb_le, leb; discriminate.
Defined.

#[export] Instance and_dec A B: Decision A -> Decision B -> Decision (A /\ B) := {
  decide :=
    match (decide A) with
      | left HA =>
          match (decide B) with
            | left HB => left (conj HA HB)
            | right HnB => right (fun H => HnB (proj2 H))
          end
      | right HnA => right (fun H => HnA (proj1 H))
    end
}.

#[export] Instance or_dec A B: Decision A -> Decision B -> Decision (A \/ B) := {
  decide :=
    match (decide A) with
      | left HA => left (or_introl HA)
      | right HnA =>
        match (decide B) with
          | left HB => left (or_intror HB)
          | right HnB => right (fun HAB => match HAB with
                                             | or_introl HA => HnA HA
                                             | or_intror HB => HnB HB
                                           end)
        end
    end
}.

#[refine, export] Instance impl_dec P Q `(Pdec: Decision P) `(Qdec: Decision Q): Decision (P->Q) :=
  {
    decide :=
      match Qdec with
        | left HQ => left (fun _ => HQ)
        | right HnQ =>
          match Pdec with
            | left HP => right _
            | right HnP => left _
          end
      end
  }.
Proof.
  * abstract tauto.
  * abstract tauto.
Defined.

#[refine, export] Instance not_dec P `(Pdec: Decision P): Decision (~P) :=
  {
    decide :=
      match Pdec with
        | left _ => right _
        | right _ => left _
      end
  }.
Proof.
  * abstract tauto.
  * abstract tauto.
Defined.

#[refine, export] Instance decide_none {A} (a: option A): Decision (a = None) := {
  decide :=
    match a with
      | Some _ => right _
      | None => left _
    end
}.
Proof.
  * abstract congruence.
  * abstract congruence.
Defined.

#[refine, export] Instance decide_option_eq {A}:
  (forall (x y : A), Decision (x = y)) ->
  (forall (x y : option A), Decision (x = y)) :=
  fun H x y =>
    match x, y with
      | Some x, Some y =>
        match decide (x = y) with
          | left H => left _
          | right H => right _
        end
      | None, None =>
        left eq_refl
      | _, _ =>
        right _
    end.
Proof.
  * f_equal; auto.
  * abstract (injection; eauto).
  * abstract discriminate.
  * abstract discriminate.
Defined.

Section DECIDE_PROD.
  Context A `{Adec: forall x y: A, Decision (x = y)}.
  Context B `{Bdec: forall x y: B, Decision (x = y)}.

  #[export] Instance decide_eq_pair: forall (x y: A * B), Decision (x = y).
  Proof.
    intros [x1 x2] [y1 y2].
    destruct (decide (x1 = y1)).
    - destruct (decide (x2 = y2)).
      + left; congruence.
      + right; intro H; inversion H; now auto.
    - right; intro H; inversion H; now auto.
  Defined.
End DECIDE_PROD.

(** * Decision procedures for lists *)

#[export] Instance decide_In {A}:
  (forall (x y: A), Decision (x = y)) ->
  (forall (a: A) (l: list A), Decision (In a l)) :=
    @In_dec A.

#[export] Instance decide_Forall {A} (P: A -> Prop):
  (forall a, Decision (P a)) ->
  (forall l, Decision (Forall P l)).
Proof.
  intros HP l.
  induction l.
  - left.
    constructor.
  - destruct (decide (Forall P l)) as [Hl | Hl].
    + destruct (decide (P a)) as [Ha | Ha].
      * left.
        constructor;
        assumption.
      * right.
        inversion 1.
        tauto.
    + right.
      inversion 1.
      tauto.
Defined.

#[export] Instance decide_Exists {A} (P: A -> Prop) `{forall x, Decision (P x)} xs :
  Decision (Exists P xs).
Proof. now apply Exists_dec; intros; apply decide. Qed.

#[export] Instance list_decide_eq {A} `{forall (x y: A), Decision (x = y)} :
  forall (xs ys: list A), Decision (xs = ys).
Proof. intros; apply list_eq_dec; auto. Qed.

#[export] Instance decide_NoDup {A} `{forall (x y: A), Decision (x = y)} (xs: list A) : Decision (NoDup xs).
Proof.
  induction xs as [| x xs]; [left; constructor |].
  destruct IHxs; [| right; now inversion 1].
  destruct (decide (In x xs)); [right; now inversion 1 | left; constructor; auto].
Qed.

#[export] Instance decide_incl {A} `{forall (x y: A), Decision (x = y)} (xs ys: list A) : Decision (incl xs ys).
Proof.
  induction xs as [| x xs]; [left; apply incl_nil_l |].
  destruct IHxs; [| now right; intros ?%List.incl_cons_inv].
  destruct (decide (In x ys));
    [left; auto using incl_cons | now right; intros ?%List.incl_cons_inv].
Qed.

(** * Decision procedures from [compare] *)

(** This takes care of many orders, which are defined as, say,
  [le x y := compare x y <> Gt]. *)

#[export] Instance comparison_eq_dec (x y: comparison): Decision (x = y).
Proof.
  red.
  decide equality.
Defined.

#[export] Program Instance comparison_ne_dec (x y: comparison): Decision (x <> y) :=
  match decide (x = y) with
    | left Hne => right _
    | right Hnne => left _
  end.

(** Decision and equivalence *)

#[refine, local] Instance decide_rewrite P Q (Heq: P <-> Q) `(Decision P): Decision Q :=
  match decide P with
    | left _ => left _
    | right _ => right _
  end.
Proof.
  all: abstract tauto.
Defined.

(** Decision and discriminable cases *)

Theorem decide_discr {A}
        (Q1 Q2 P: A -> Prop)
        (discr: forall i, {Q1 i} + {Q2 i})
        (dec_1: Decision (forall i, Q1 i -> P i))
        (dec_2: Decision (forall i, Q2 i -> P i)):
  Decision (forall i, P i).
Proof.
  unfold Decision in *.
  firstorder.
Defined.

(** * Facts *)
Section DecFacts.
  Context {A: Type} {P: Prop} `{Decision P}.

  Fact decide_true_if (t f: A) :
    P -> (if decide P then t else f) = t.
  Proof. intros; now destruct (decide _). Qed.

  Fact decide_false_if (t f: A) :
    ~P -> (if decide P then t else f) = f.
  Proof. intros; now destruct (decide _). Qed.

  Fact decide_impl_true (Q: Prop) (t f: A) :
    (Q -> P) -> Q -> (if decide P then t else f) = t.
  Proof. intros * HQP HQ; apply HQP in HQ; apply decide_true_if; auto. Qed.

  Fact decide_iff (Q: Prop) `{Decision Q} (t f: A) :
    (Q <-> P) -> (if decide P then t else f) = (if decide Q then t else f).
  Proof.
    intros * HQP.
    destruct (decide Q) as [HQ | HQ]; rewrite HQP in HQ;
      [apply decide_true_if | apply decide_false_if]; auto.
  Qed.

  Fact decide_neg (t f: A) :
    (if decide P then t else f) = (if decide (~P) then f else t).
  Proof. now destruct (decide P), (decide (~P)). Qed.
End DecFacts.

Section DecEq.
  Context {A B: Type}.
  Context `{forall (x y: A), Decision (x = y)}.

  Fact decide_refl (x: A) (t f: B) :
    (if decide (x = x) then t else f) = t.
  Proof. intros; apply decide_true_if; auto. Qed.

  Fact decide_sym (x y: A) (t f: B) :
    (if decide (x = y) then t else f) = (if decide (y = x) then t else f).
  Proof. now apply decide_iff. Qed.
End DecEq.

(* Convert a record to a conjunction *)
Ltac impls_to_conj H :=
  let H := eval cbn zeta in H in
  match H with
  | ?P -> ?R =>
    let X := impls_to_conj R in constr:(P /\ X)
  | ?R => True
  end.

Ltac decide_rec Rec :=
  match type of Rec with
  | ?P =>
    let X := impls_to_conj P in
    let H := fresh in
    enough (Decision X) by (destruct H; [left; constructor | right; intros []]; intuition auto);
      repeat apply and_dec
  end.
