From Coq Require Import
  List
  PeanoNat
  Lia
  Peano_dec.
From advert.lib Require Import
  Maps
  Decision
  Semantics
  Tactics.
From advert.specs Require Import
  UDAG
  LidoClient
  LidoDAG
  Config.
From advert.impl.mysticeti Require Import
  MysticetiDAG
  MysticetiCommit.
From advert.impl.mysticeti.proofs Require Import
  LivenessWeak.

Section Lemmas.

  Definition list_complete_upto (l : list nat) n := forall r, 1 <= r <= n -> In r l.

  Instance list_complete_upto_dec (l : list nat) n : Decision (list_complete_upto l n).
  Proof.
    induction n.
    - left; unfold list_complete_upto.
      intros r; lia.
    - destruct IHn as [IHn | IHn].
      + destruct (decide (In (S n) l)).
        * left; unfold list_complete_upto in *.
          intros r Hr.
          assert (Hr' : 1 <= r <= n \/ r = (S n)) by lia.
          destruct Hr' as [Hr' | Hr'].
          -- apply IHn; auto.
          -- subst r; auto.
        * right; unfold list_complete_upto.
          intros Hr; specialize (Hr (S n) ltac:(lia)).
          contradiction.
      + right; unfold list_complete_upto in *.
        intros Hr; apply IHn.
        intros r Hr'; apply Hr; lia.
  Qed.

  Lemma list_complete_upto_le_max : forall l n,
    list_complete_upto l n -> n <= list_max l.
  Proof.
    intros l n Hn.
    unfold list_complete_upto in Hn.
    assert (Hpos : n = 0 \/ n > 0) by lia.
    destruct Hpos as [Hpos | Hpos].
    1: subst n; lia.
    specialize (Hn n ltac:(lia)).
    pose proof list_max_le l (list_max l) as (Hmax & _).
    specialize (Hmax ltac:(auto)).
    rewrite Forall_forall in Hmax.
    specialize (Hmax _ Hn); auto.
  Qed.

  Lemma list_complete_upto_zero : forall l, list_complete_upto l 0.
  Proof.
    intros l; unfold list_complete_upto.
    intros r; lia.
  Qed.

  Lemma list_complete_upto_ex' : forall l n,
    ~ list_complete_upto l n ->
    exists n',
    list_complete_upto l n' /\ ~ list_complete_upto l (S n').
  Proof.
    intros l n; revert l.
    induction n.
    - intros l Hfalse.
      pose proof (list_complete_upto_zero l); contradiction.
    - intros l Hl.
      destruct (decide (list_complete_upto l n)) as [Hyes | Hno].
      + exists n; split; auto.
      + specialize (IHn l Hno).
        auto.
  Qed.

  Lemma list_complete_upto_ex : forall l,
    exists n,
    list_complete_upto l n /\ ~ list_complete_upto l (S n).
  Proof.
    intros l.
    apply (list_complete_upto_ex' l (S (list_max l))).
    intros Hfalse.
    pose proof (list_complete_upto_le_max _ _ Hfalse).
    lia.
  Qed.

End Lemmas.

Section MysticetiRefine.
  Context `{config : !BADO_Config} `{dag_config : !DAG_Config} `{participant : !BADO_Participant} `{leader : !BADO_Leader} `{node_assump : !BADO_NodeAssump} `{assump : !BADO_Assump}.

(* For each valid state of mcommit, there exists unique r such that
   rounds 1 <= r' <= r are committed, round r+1 is not committed *)

  Definition mcommit_high_round (mcommit : MDAG_Commit_State) (r : nat) :=
    (forall r', 1 <= r' <= r -> In r' (map fst (mcommit_commit mcommit)) \/ In r' (mcommit_nack mcommit)) /\
    ~ In (S r) (map fst (mcommit_commit mcommit)) /\
    ~ In (S r) (mcommit_nack mcommit).

  Lemma mcommit_high_round_uniq (mcommit : MDAG_Commit_State) r r' :
    mcommit_high_round mcommit r ->
    mcommit_high_round mcommit r' ->
    r = r'.
  Proof.
    intros Hr Hr'.
    unfold mcommit_high_round in Hr, Hr'.
    destruct Hr as (Hr1 & Hr2 & Hr3).
    destruct Hr' as (Hr'1 & Hr'2 & Hr'3).
    assert (Hcmp : r = r' \/ r < r' \/ r > r') by lia.
    destruct Hcmp as [Hcmp | [Hcmp | Hcmp]]; [auto | |].
    - specialize (Hr'1 (S r) ltac:(lia)).
      tauto.
    - specialize (Hr1 (S r') ltac:(lia)).
      tauto.
  Qed.

  Lemma mcommit_high_round_ex (mcommit : MDAG_Commit_State) :
    exists r, mcommit_high_round mcommit r.
  Proof.
    unfold mcommit_high_round.
    remember (map fst (mcommit_commit mcommit)) as l1; clear Heql1.
    remember (mcommit_nack mcommit) as l2; clear Heql2.
    pose proof (list_complete_upto_ex (l1 ++ l2)) as (r & Hr1 & Hr2).
    unfold list_complete_upto in *.
    assert (Hr3 : ~ In (S r) (l1 ++ l2)).
    { intros Hr3.
      apply Hr2.
      intros r' Hr'1; assert (Hr'2 : 1 <= r' <= r \/ r' = S r) by lia.
      destruct Hr'2 as [Hr'2 | Hr'2].
      - specialize (Hr1 _ Hr'2); auto.
      - subst r'; auto.
    }
    rewrite in_app_iff in Hr3.
    exists r.
    repeat split.
    2,3: tauto.
    intros r' Hr'; specialize (Hr1 _ Hr').
    rewrite in_app_iff in Hr1; auto.
  Qed.

(* If mcommit state X transitions to Y, then high_round can only increase *)

  Lemma mcommit_high_round_mono :
    forall mcommit mcommit' r r',
    mcommit_reachable mcommit mcommit' ->
    mcommit_high_round mcommit r ->
    mcommit_high_round mcommit' r' ->
    r <= r'.
  Proof.
    intros mcommit mcommit' r r' Hreach Hr Hr'.
    unfold mcommit_high_round in *.
    assert (Hgt : r <= r' \/ r > r') by lia.
    destruct Hgt as [Hgt | Hgt]; [auto | exfalso].
    destruct Hr as (Hr1 & Hr2 & Hr3).
    specialize (Hr1 (S r') ltac:(lia)).
    destruct Hr' as (_ & Hr'1 & Hr'2).
    destruct Hr1 as [Hr1 | Hr1].
    - rewrite in_map_iff in Hr1.
      destruct Hr1 as (x & Hx1 & Hx2).
      pose proof (mcommit_commit_mono _ _ Hreach _ Hx2) as Hin; destruct x; cbn in Hx1; subst n.
      apply Hr'1; rewrite in_map_iff; eexists; split; try apply Hin; auto.
    - pose proof (mcommit_nack_mono _ _ Hreach _ Hr1) as Hin.
      contradiction.
  Qed.

  Lemma mcommit_high_round_init :
    mcommit_high_round mcommit_state_null 0.
  Proof.
    unfold mcommit_high_round.
    split.
    1: lia.
    split; cbn; auto.
  Qed.

(* For each round 1 <= r <= high_round, there exists r' such that
   0 <= r' < r, r' = 0 \/ r' is committed, and every r'' with r' < r'' < r
   is not committed *)

  Definition mcommit_is_pull_src (mcommit : MDAG_Commit_State) r r' :=
    0 <= r' < r /\
    (r' = 0 \/ In r' (map fst (mcommit_commit mcommit))) /\
    forall r'', r' < r'' < r -> In r'' (mcommit_nack mcommit).

  Lemma mcommit_pull_src_uniq mcommit r r' r'' :
    mcommit_valid mcommit ->
    mcommit_is_pull_src mcommit r r' ->
    mcommit_is_pull_src mcommit r r'' ->
    r' = r''.
  Proof.
    intros Hval Hr1 Hr2.
    unfold mcommit_is_pull_src in *.
    destruct Hr1 as (Hr1_1 & Hr1_2 & Hr1_3).
    destruct Hr2 as (Hr2_1 & Hr2_2 & Hr2_3).
    assert (Hcmp : r' = r'' \/ r' < r'' \/ r' > r'') by lia.
    destruct Hcmp as [Hcmp | [Hcmp | Hcmp]].
    1: auto.
    all: exfalso.
    - specialize (Hr1_3 r'' ltac:(lia)).
      destruct Hr2_2 as [? | Hr2_2]; [lia|].
      rewrite in_map_iff in Hr2_2.
      destruct Hr2_2 as (x & Hx1 & Hx2); destruct x.
      cbn in Hx1; subst n.
      pose proof (mcommit_safety _ _ _ Hval Hx2); contradiction.
    - specialize (Hr2_3 r' ltac:(lia)).
      destruct Hr1_2 as [? | Hr1_2]; [lia|].
      rewrite in_map_iff in Hr1_2.
      destruct Hr1_2 as (x & Hx1 & Hx2); destruct x.
      cbn in Hx1; subst n.
      pose proof (mcommit_safety _ _ _ Hval Hx2); contradiction.
  Qed.

  Lemma mcommit_pull_src (mcommit : MDAG_Commit_State) high_r :
    mcommit_high_round mcommit high_r ->
    forall r, 1 <= r <= high_r ->
    exists r', mcommit_is_pull_src mcommit r r'.
  Proof.
    intros Hhigh r.
    induction r.
    - lia.
    - intros Hr.
      destruct r.
      + exists 0; unfold mcommit_is_pull_src.
        repeat split; auto.
        lia.
      + specialize (IHr ltac:(lia)).
        destruct (decide (In (S r) (map fst (mcommit_commit mcommit)))).
        * exists (S r); unfold mcommit_is_pull_src.
          repeat split; auto.
          all: lia.
        * destruct IHr as (r' & Hr').
          exists r'; unfold mcommit_is_pull_src in *.
          repeat split; auto.
          1,2: lia.
          1: apply Hr'.
          intros r'' Hr''1.
          assert (Hr''2 : r' < r'' < S r \/ r'' = S r) by lia.
          destruct Hr''2 as [Hr''2 | Hr''2].
          1: apply Hr'; auto.
          subst r''.
          unfold mcommit_high_round in Hhigh.
          destruct Hhigh as (Hhigh & _).
          specialize (Hhigh (S r) ltac:(lia)).
          tauto.
  Qed.

  Lemma mcommit_pull_src_mono mcommit mcommit' r r' :
    mcommit_reachable mcommit mcommit' ->
    mcommit_is_pull_src mcommit r r' ->
    mcommit_is_pull_src mcommit' r r'.
  Proof.
    intros Hreach Hr.
    unfold mcommit_is_pull_src in *.
    destruct Hr as (Hr1 & Hr2 & Hr3).
    repeat split.
    1,2: apply Hr1.
    - destruct Hr2 as [Hr2 | Hr2]; [auto|].
      rewrite in_map_iff in Hr2.
      destruct Hr2 as (x & Hx1 & Hx2).
      pose proof (mcommit_commit_mono _ _ Hreach _ Hx2) as Hmono.
      destruct x; cbn in Hx1; subst n.
      right; rewrite in_map_iff; eexists; split; try apply Hmono; auto.
    - intros r'' Hr''.
      specialize (Hr3 _ Hr'').
      pose proof (mcommit_nack_mono _ _ Hreach _ Hr3) as Hmono.
      auto.
  Qed.

  Lemma mcommit_pull_src_ge mcommit r r' r'' :
    mcommit_valid mcommit ->
    mcommit_is_pull_src mcommit r r' ->
    r'' < r ->
    In r'' (map fst (mcommit_commit mcommit)) ->
    r' >= r''.
  Proof.
    intros Hval Hpull Hlt Hcommit.
    unfold mcommit_is_pull_src in Hpull.
    destruct Hpull as (_ & _ & Hpull).
    assert (Hr' : r' >= r'' \/ r' < r'') by lia.
    destruct Hr' as [Hr' | Hr'].
    1: auto.
    specialize (Hpull r'' ltac:(split; auto)).
    rewrite in_map_iff in Hcommit.
    destruct Hcommit as (x & Hx1 & Hx2); destruct x; cbn in Hx1; subst n.
    pose proof (mcommit_safety _ _ _ Hval Hx2).
    contradiction.
  Qed.

(* Lidodag refinement relation *)

  Record R_mys_lidodag (mcommit : MDAG_Commit_State) (lidodag : LidoDAG) (r : nat) : Prop := {
    R_mys_lidodag_dag : mcommit.(mcommit_mdag).(mdag_udag) = lidodag.(lidodag_dag);
    R_mys_lidodag_ecache :
      forall r' id,
        1 <= r' <= r ->
        In (r', id) (mcommit_commit mcommit) ->
        exists r'',
          mcommit_is_pull_src mcommit r' r'' /\
          (client_get_round r' (lidodag.(lidodag_lido))).(client_round_pull_src) = Some (r'', 0);
    R_mys_lidodag_ecache_inv :
      forall r' ps,
        (client_get_round r' (lidodag.(lidodag_lido))).(client_round_pull_src) = Some ps ->
        r' <= r /\
        mcommit_is_pull_src mcommit r' (fst ps);
    R_mys_lidodag_mcache :
      forall r' id,
        1 <= r' <= r ->
        In (r', id) (mcommit_commit mcommit) ->
        NatMap_find 0 (client_get_round r' (lidodag.(lidodag_lido))).(client_round_mcaches) = Some id;
    R_mys_lidodag_mcache_inv :
      forall r' v id,
        NatMap_find v (client_get_round r' (lidodag.(lidodag_lido))).(client_round_mcaches) = Some id ->
        v = 0 /\ r' <= r /\ In (r', id) (mcommit_commit mcommit);
    R_mys_lidodag_ccache :
      forall r' id,
        1 <= r' <= r ->
        In (r', id) (mcommit_commit mcommit) ->
        (client_get_round r' (lidodag.(lidodag_lido))).(client_round_max_ccache) = Some 0;
    R_mys_lidodag_ccache_inv :
      forall r' v,
        1 <= r' <= r ->
        (client_get_round r' (lidodag.(lidodag_lido))).(client_round_max_ccache) = Some v ->
        v = 0;
    R_mys_lidodag_cmd :
      forall nid, client_get_cmd nid (snd lidodag.(lidodag_lido)) = LidoCommand.Idle;
  }.

  Lemma mcommit_refine_lidodag_init : R_mys_lidodag mcommit_state_null lidodag_state_null 0.
  Proof.
    constructor.
    - cbn; auto.
    - intros; lia.
    - cbn; discriminate.
    - intros; lia.
    - cbn; discriminate.
    - intros; lia.
    - cbn; discriminate.
    - cbn; auto.
  Qed.

  Lemma mcommit_refine_lidodag_step_same_round :
    forall mcommit mcommit' lidodag R high_R,
    mcommit_valid mcommit ->
    lidodag_valid lidodag ->
    mcommit_step mcommit mcommit' ->
    mcommit_high_round mcommit high_R ->
    R <= high_R ->
    R_mys_lidodag mcommit lidodag R ->
    exists lidodag',
      lidodag_reachable lidodag lidodag' /\
      R_mys_lidodag mcommit' lidodag' R.
  Proof.
    intros mcommit mcommit' lidodag R high_R Hval_mcommit Hval_lidodag Hstep Hhigh1 Hhigh2 HR.
    pose proof Hstep as Hstep_copy.
    assert (Hreach : mcommit_reachable mcommit mcommit') by (eapply reachable_step; [apply reachable_self | auto]).
    assert (Hval_mcommit' : mcommit_valid mcommit') by (eapply valid_reach_valid; try apply Hval_mcommit; auto).

    mcommit_step_case Hstep.
    - exists (lidodag_add_vert id vert lidodag).
      split.
      + eapply reachable_step; [apply reachable_self | constructor].
        rewrite <- (R_mys_lidodag_dag _ _ _ HR).
        apply Hpre2.
      + constructor.
        1: cbn; rewrite <- (R_mys_lidodag_dag _ _ _ HR); auto.
        all: cbn; apply HR.
    - exists (lidodag_add_vert id vert lidodag).
      split.
      + eapply reachable_step; [apply reachable_self | constructor].
        rewrite <- (R_mys_lidodag_dag _ _ _ HR).
        apply Hpre2.
      + constructor.
        1: cbn; rewrite <- (R_mys_lidodag_dag _ _ _ HR); auto.
        all: cbn; apply HR.
    - exists (lidodag_add_vert id vert lidodag).
      split.
      + eapply reachable_step; [apply reachable_self | constructor].
        rewrite <- (R_mys_lidodag_dag _ _ _ HR).
        apply Hpre2.
      + constructor.
        1: cbn; rewrite <- (R_mys_lidodag_dag _ _ _ HR); auto.
        all: cbn; apply HR.
    - exists (lidodag_add_vert id vert lidodag).
      split.
      + eapply reachable_step; [apply reachable_self | constructor].
        rewrite <- (R_mys_lidodag_dag _ _ _ HR).
        apply Hpre.
      + constructor.
        1: cbn; rewrite <- (R_mys_lidodag_dag _ _ _ HR); auto.
        all: cbn; apply HR.
    - exists lidodag.
      split.
      1: apply reachable_self.
      constructor; cbn; apply HR.
    - exists lidodag.
      split.
      1: apply reachable_self.
      constructor; cbn; apply HR.

    - exists lidodag.
      split.
      1: apply reachable_self.
      unfold mcommit_high_round in Hhigh1.
      destruct Hhigh1 as (Hhigh1 & _).
      constructor.
      + cbn; apply HR.
      + intros r' id' Hr'.
        specialize (Hhigh1 r' ltac:(lia)).
        intros Hin; cbn in Hin.
        destruct Hin as [Hin | Hin].
        * injection Hin; intros; subst r' id'; clear Hin.
          destruct Hhigh1 as [Hhigh1 | Hhigh1].
          -- rewrite in_map_iff in Hhigh1.
             destruct Hhigh1 as (x & Hx1 & Hx2); destruct x; cbn in Hx1; subst n.
             pose proof (mcommit_commit_mono _ _ Hreach _ Hx2) as Hmono.
             pose proof (mcommit_commit_eq _ _ _ _ Hval_mcommit' ltac:(cbn; left; auto) Hmono); subst n0.
             pose proof (R_mys_lidodag_ecache _ _ _ HR _ _ Hr' Hx2) as (r'' & Hr''1 & Hr''2).
             exists r''; split; auto.
             apply (mcommit_pull_src_mono _ _ _ _ Hreach Hr''1).
          -- pose proof (mcommit_nack_mono _ _ Hreach _ Hhigh1) as Hmono.
             pose proof (mcommit_safety _ _ _ Hval_mcommit' ltac:(cbn; left; auto)).
             contradiction.
        * pose proof (R_mys_lidodag_ecache _ _ _ HR _ _ Hr' Hin) as (r'' & Hr''1 & Hr''2).
          exists r''; split; auto.
          apply (mcommit_pull_src_mono _ _ _ _ Hreach Hr''1).
      + intros r' ps Hr'.
        pose proof R_mys_lidodag_ecache_inv _ _ _ HR _ _ Hr' as (Hr'0 & Hr'1).
        split; auto.
        eapply mcommit_pull_src_mono; eauto.
      + intros r' id' Hr'.
        specialize (Hhigh1 r' ltac:(lia)).
        intros Hin; cbn in Hin.
        destruct Hin as [Hin | Hin].
        * injection Hin; intros; subst r' id'; clear Hin.
          destruct Hhigh1 as [Hhigh1 | Hhigh1].
          -- rewrite in_map_iff in Hhigh1.
            destruct Hhigh1 as (x & Hx1 & Hx2); destruct x; cbn in Hx1; subst n.
            pose proof (mcommit_commit_mono _ _ Hreach _ Hx2) as Hmono.
            pose proof (mcommit_commit_eq _ _ _ _ Hval_mcommit' ltac:(cbn; left; auto) Hmono); subst n0.
            apply (R_mys_lidodag_mcache _ _ _ HR _ _ Hr' Hx2).
          -- pose proof (mcommit_nack_mono _ _ Hreach _ Hhigh1) as Hmono.
            pose proof (mcommit_safety _ _ _ Hval_mcommit' ltac:(cbn; left; auto)).
            contradiction.
        * apply (R_mys_lidodag_mcache _ _ _ HR _ _ Hr' Hin).
      + intros r' v id' Hr'.
        pose proof R_mys_lidodag_mcache_inv _ _ _ HR _ _ _ Hr' as (Hv & Hr'0 & Hr'1).
        repeat split; auto.
        cbn; right; auto.
      + intros r' id' Hr'.
        specialize (Hhigh1 r' ltac:(lia)).
        intros Hin; cbn in Hin.
        destruct Hin as [Hin | Hin].
        * injection Hin; intros; subst r' id'; clear Hin.
          destruct Hhigh1 as [Hhigh1 | Hhigh1].
          -- rewrite in_map_iff in Hhigh1.
            destruct Hhigh1 as (x & Hx1 & Hx2); destruct x; cbn in Hx1; subst n.
            pose proof (mcommit_commit_mono _ _ Hreach _ Hx2) as Hmono.
            pose proof (mcommit_commit_eq _ _ _ _ Hval_mcommit' ltac:(cbn; left; auto) Hmono); subst n0.
            apply (R_mys_lidodag_ccache _ _ _ HR _ _ Hr' Hx2).
          -- pose proof (mcommit_nack_mono _ _ Hreach _ Hhigh1) as Hmono.
            pose proof (mcommit_safety _ _ _ Hval_mcommit' ltac:(cbn; left; auto)).
            contradiction.
        * apply (R_mys_lidodag_ccache _ _ _ HR _ _ Hr' Hin).
      + intros r' v Hr' Hr'0.
        apply (R_mys_lidodag_ccache_inv _ _ _ HR _ _ Hr' Hr'0).
      + apply HR.

    - exists lidodag.
      split.
      1: apply reachable_self.
      unfold mcommit_high_round in Hhigh1.
      destruct Hhigh1 as (Hhigh1 & _).
      constructor.
      + cbn; apply HR.
      + intros r'' id' Hr''.
        specialize (Hhigh1 r'' ltac:(lia)).
        intros Hin; cbn in Hin.
        destruct Hin as [Hin | Hin].
        * injection Hin; intros; subst r'' id'; clear Hin.
          destruct Hhigh1 as [Hhigh1 | Hhigh1].
          -- rewrite in_map_iff in Hhigh1.
             destruct Hhigh1 as (x & Hx1 & Hx2); destruct x; cbn in Hx1; subst n.
             pose proof (mcommit_commit_mono _ _ Hreach _ Hx2) as Hmono.
             pose proof (mcommit_commit_eq _ _ _ _ Hval_mcommit' ltac:(cbn; left; auto) Hmono); subst n0.
             pose proof (R_mys_lidodag_ecache _ _ _ HR _ _ Hr'' Hx2) as (r''' & Hr'''1 & Hr'''2).
             exists r'''; split; auto.
             apply (mcommit_pull_src_mono _ _ _ _ Hreach Hr'''1).
          -- pose proof (mcommit_nack_mono _ _ Hreach _ Hhigh1) as Hmono.
             pose proof (mcommit_safety _ _ _ Hval_mcommit' ltac:(cbn; left; auto)).
             contradiction.
        * pose proof (R_mys_lidodag_ecache _ _ _ HR _ _ Hr'' Hin) as (r''' & Hr'''1 & Hr'''2).
          exists r'''; split; auto.
          apply (mcommit_pull_src_mono _ _ _ _ Hreach Hr'''1).
      + intros r'' ps Hr''.
        pose proof R_mys_lidodag_ecache_inv _ _ _ HR _ _ Hr'' as (Hr''0 & Hr''1).
        split; auto.
        eapply mcommit_pull_src_mono; eauto.
      + intros r'' id' Hr''.
        specialize (Hhigh1 r'' ltac:(lia)).
        intros Hin; cbn in Hin.
        destruct Hin as [Hin | Hin].
        * injection Hin; intros; subst r'' id'; clear Hin.
          destruct Hhigh1 as [Hhigh1 | Hhigh1].
          -- rewrite in_map_iff in Hhigh1.
            destruct Hhigh1 as (x & Hx1 & Hx2); destruct x; cbn in Hx1; subst n.
            pose proof (mcommit_commit_mono _ _ Hreach _ Hx2) as Hmono.
            pose proof (mcommit_commit_eq _ _ _ _ Hval_mcommit' ltac:(cbn; left; auto) Hmono); subst n0.
            apply (R_mys_lidodag_mcache _ _ _ HR _ _ Hr'' Hx2).
          -- pose proof (mcommit_nack_mono _ _ Hreach _ Hhigh1) as Hmono.
            pose proof (mcommit_safety _ _ _ Hval_mcommit' ltac:(cbn; left; auto)).
            contradiction.
        * apply (R_mys_lidodag_mcache _ _ _ HR _ _ Hr'' Hin).
      + intros r'' v id' Hr''.
        pose proof R_mys_lidodag_mcache_inv _ _ _ HR _ _ _ Hr'' as (Hv & Hr''0 & Hr''1).
        repeat split; auto.
        cbn; right; auto.
      + intros r'' id' Hr''.
        specialize (Hhigh1 r'' ltac:(lia)).
        intros Hin; cbn in Hin.
        destruct Hin as [Hin | Hin].
        * injection Hin; intros; subst r'' id'; clear Hin.
          destruct Hhigh1 as [Hhigh1 | Hhigh1].
          -- rewrite in_map_iff in Hhigh1.
            destruct Hhigh1 as (x & Hx1 & Hx2); destruct x; cbn in Hx1; subst n.
            pose proof (mcommit_commit_mono _ _ Hreach _ Hx2) as Hmono.
            pose proof (mcommit_commit_eq _ _ _ _ Hval_mcommit' ltac:(cbn; left; auto) Hmono); subst n0.
            apply (R_mys_lidodag_ccache _ _ _ HR _ _ Hr'' Hx2).
          -- pose proof (mcommit_nack_mono _ _ Hreach _ Hhigh1) as Hmono.
            pose proof (mcommit_safety _ _ _ Hval_mcommit' ltac:(cbn; left; auto)).
            contradiction.
        * apply (R_mys_lidodag_ccache _ _ _ HR _ _ Hr'' Hin).
      + intros r'' v Hr'' Hr''0.
        apply (R_mys_lidodag_ccache_inv _ _ _ HR _ _ Hr'' Hr''0).
      + apply HR.

    - exists lidodag.
      split.
      1: apply reachable_self.
      unfold mcommit_high_round in Hhigh1.
      destruct Hhigh1 as (Hhigh1 & _).
      constructor.
      + cbn; apply HR.
      + intros r'' id' Hr''.
        specialize (Hhigh1 r'' ltac:(lia)).
        intros Hin; cbn in Hin.
        pose proof (R_mys_lidodag_ecache _ _ _ HR _ _ Hr'' Hin) as (r''' & Hr'''1 & Hr'''2).
        exists r'''; split; auto.
        apply (mcommit_pull_src_mono _ _ _ _ Hreach Hr'''1).
      + intros r'' ps Hr''.
        pose proof R_mys_lidodag_ecache_inv _ _ _ HR _ _ Hr'' as (Hr''0 & Hr''1).
        split; auto.
        eapply mcommit_pull_src_mono; eauto.
      + intros r'' id' Hr''.
        specialize (Hhigh1 r'' ltac:(lia)).
        intros Hin; cbn in Hin.
        apply (R_mys_lidodag_mcache _ _ _ HR _ _ Hr'' Hin).
      + intros r'' v id' Hr''.
        pose proof R_mys_lidodag_mcache_inv _ _ _ HR _ _ _ Hr'' as (Hv & Hr''0 & Hr''1).
        repeat split; auto.
      + intros r'' id' Hr''.
        specialize (Hhigh1 r'' ltac:(lia)).
        intros Hin; cbn in Hin.
        apply (R_mys_lidodag_ccache _ _ _ HR _ _ Hr'' Hin).
      + intros r'' v Hr'' Hr''0.
        apply (R_mys_lidodag_ccache_inv _ _ _ HR _ _ Hr'' Hr''0).
      + apply HR.

    - exists lidodag.
      split.
      1: apply reachable_self.
      constructor.
      all: cbn; apply HR.

    - exists lidodag.
      split.
      1: apply reachable_self.
      constructor.
      all: cbn; apply HR.
  Qed.

  Lemma mcommit_refine_lidodag_step_next_round :
    forall mcommit lidodag R high_R,
    mcommit_valid mcommit ->
    lidodag_valid lidodag ->
    mcommit_high_round mcommit high_R ->
    R < high_R ->
    R_mys_lidodag mcommit lidodag R ->
    exists lidodag',
      lidodag_reachable lidodag lidodag' /\
      R_mys_lidodag mcommit lidodag' (S R).
  Proof.
    intros mcommit lidodag R high_R Hval_mcommit Hval_lidodag Hhigh1 Hhigh2 HR.
    pose proof mcommit_pull_src _ _ Hhigh1 (S R) ltac:(lia) as (R' & HR').

    unfold mcommit_high_round in Hhigh1.
    destruct Hhigh1 as (Hhigh1 & Hhigh3 & Hhigh4).
    pose proof Hhigh1 (S R) ltac:(lia) as HR1.
    destruct HR1 as [HR1|HR1].

    - (* commit *)
      apply in_map_iff in HR1.
      destruct HR1 as (ps & Hps1 & Hps).
      destruct ps as (r, id); cbn [fst] in Hps1; subst r.

      remember (lidodag_issue_pull (S R) lidodag) as lidodag1.
      assert (Hreach1 : lidodag_reachable lidodag lidodag1).
      { eapply reachable_step. 1: apply reachable_self.
        rewrite Heqlidodag1; constructor.
        constructor; cbn; [|lia].
        apply HR. }

      unfold mcommit_is_pull_src in HR'.
      destruct HR' as (HR'1 & HR'2 & HR'3).

      remember (lidodag_add_ecache (S R) (R', 0) lidodag1) as lidodag2.
      assert (Hreach2 : lidodag_reachable lidodag lidodag2).
      { eapply reachable_step. 1: apply Hreach1.
        rewrite Heqlidodag2; constructor. constructor; cbn [fst snd].
        - lia.
        - rewrite Heqlidodag1. cbn. client_unfold. client_fold (lidodag_lido lidodag).
          match goal with |- ?x = None \/ _ => remember x as f; destruct f; [symmetry in Heqf|left; auto] end.
          pose proof R_mys_lidodag_ecache_inv _ _ _ HR _ _ Heqf; lia.
        - lia.
        - assert (HR'4 : R' = 0 \/ R' > 0) by lia.
          destruct HR'4 as [HR'4|HR'4]; [left; auto|right; split; auto].
          destruct HR'2 as [?|HR'2]; [lia|].
          apply in_map_iff in HR'2; destruct HR'2 as (ps' & Hps'1 & Hps'2).
          destruct ps' as (r', id'); cbn [fst] in Hps'1; subst r'.
          pose proof R_mys_lidodag_mcache _ _ _ HR R' id' ltac:(lia) Hps'2 as Hfind.
          rewrite Heqlidodag1. cbn. client_unfold. client_fold (lidodag_lido lidodag).
          rewrite Hfind; discriminate.
        - intros r Hr.
          rewrite Heqlidodag1. cbn. client_unfold. client_fold (lidodag_lido lidodag).
          match goal with |- context[match ?x with _ => _ end] => remember x as f; destruct f; [symmetry in Heqf|auto] end.
          assert (Hr1 : r = 0 \/ r > 0) by lia.
          destruct Hr1 as [Hr1|Hr1]; [subst r|].
          + rewrite client_round_0_no_ccache in Heqf; [discriminate|].
            apply lidodag_lido_valid; auto.
          + pose proof R_mys_lidodag_ccache_inv _ _ _ HR r _ ltac:(lia) Heqf; subst n.
            assert (Hr2 : R' >= r \/ R' < r) by lia.
            destruct Hr2 as [Hr2|Hr2]; [lia|].
            specialize (HR'3 r ltac:(lia)).
            pose proof client_commit_valid _ _ _ (lidodag_lido_valid _ Hval_lidodag) Heqf as Hr3.
            match type of Hr3 with ?x <> None => remember x as f; destruct f; [symmetry in Heqf0; clear Hr3|contradiction] end.
            pose proof R_mys_lidodag_mcache_inv _ _ _ HR _ _ _ Heqf0 as (_ & _ & Hr3).
            pose proof mcommit_safety _ _ _ Hval_mcommit Hr3; contradiction.
        - rewrite Heqlidodag1. cbn. client_unfold. client_reduce.
          rewrite NatMap_gse; auto. }

      remember (lidodag_issue_invoke (S R) 0 id lidodag2) as lidodag3.
      assert (Hreach3 : lidodag_reachable lidodag lidodag3).
      { eapply reachable_step. 1: apply Hreach2.
        rewrite Heqlidodag3; constructor; [|constructor].
        - unfold lidodag_valid_data.
          rewrite Heqlidodag2; cbn.
          rewrite Heqlidodag1; cbn.
          erewrite <- R_mys_lidodag_dag; [|apply HR].
          pose proof mcommit_commit_valid _ _ _ Hval_mcommit Hps as Hid.
          match type of Hid with match ?x with _ => _ end => remember x as f; destruct f; [discriminate|contradiction] end.
        - rewrite Heqlidodag2; cbn. client_unfold. client_reduce.
          rewrite NatMap_gse; auto.
        - cbn; lia. }

      remember (lidodag_add_mcache (S R) 0 id lidodag3) as lidodag4.
      assert (Hreach4 : lidodag_reachable lidodag lidodag4).
      { eapply reachable_step. 1: apply Hreach3.
        rewrite Heqlidodag4; constructor. constructor.
        - lia.
        - left.
          rewrite Heqlidodag3. cbn. client_unfold. client_fold (lidodag_lido lidodag2).
          rewrite Heqlidodag2. cbn. client_unfold.
          rewrite NatMap_gse; [|auto]. client_reduce.
          rewrite Heqlidodag1. cbn. client_unfold. client_fold (lidodag_lido lidodag).
          match goal with |- ?x = None => remember x as f; destruct f; [symmetry in Heqf|auto] end.
          pose proof R_mys_lidodag_mcache_inv _ _ _ HR _ _ _ Heqf as (_ & ? & _); lia.
        - rewrite Heqlidodag3. cbn. client_unfold. client_fold (lidodag_lido lidodag2).
          rewrite Heqlidodag2. cbn. client_unfold.
          rewrite NatMap_gse; [|auto]. client_reduce. discriminate.
        - rewrite Heqlidodag3. cbn. client_unfold. client_reduce.
          rewrite NatMap_gse; auto. }

      remember (lidodag_issue_push (S R) 0 lidodag4) as lidodag5.
      assert (Hreach5 : lidodag_reachable lidodag lidodag5).
      { eapply reachable_step. 1: apply Hreach4.
        rewrite Heqlidodag5; constructor. constructor.
        - rewrite Heqlidodag4. cbn. client_unfold. client_reduce.
          rewrite NatMap_gse; auto.
        - cbn; lia. }

      remember (lidodag_add_ccache (S R) 0 lidodag5) as lidodag6.
      assert (Hreach6 : lidodag_reachable lidodag lidodag6).
      { eapply reachable_step. 1: apply Hreach5.
        rewrite Heqlidodag6; constructor. constructor.
        - lia.
        - rewrite Heqlidodag5. cbn. client_unfold. client_fold (lidodag_lido lidodag4).
          rewrite Heqlidodag4. cbn. client_unfold.
          rewrite NatMap_gse; auto. client_reduce.
          rewrite NatMap_gse; [discriminate|auto].
        - intros r Hr. cbn.
          rewrite Heqlidodag5. cbn. client_unfold. client_fold (lidodag_lido lidodag4).
          rewrite Heqlidodag4. cbn. client_unfold.
          rewrite NatMap_gso; [|lia]. client_fold (lidodag_lido lidodag3).
          rewrite Heqlidodag3. cbn. client_unfold. client_fold (lidodag_lido lidodag2).
          rewrite Heqlidodag2. cbn. client_unfold.
          rewrite NatMap_gso; [|lia]. client_fold (lidodag_lido lidodag1).
          rewrite Heqlidodag1. cbn. client_unfold. client_fold (lidodag_lido lidodag).
          match goal with |- match ?x with _ => _ end => remember x as f; destruct f; auto end.
          symmetry in Heqf.
          pose proof R_mys_lidodag_ecache_inv _ _ _ HR _ _ Heqf; lia.
        - rewrite Heqlidodag5. cbn. client_unfold. client_reduce.
          rewrite NatMap_gse; auto. }

      exists lidodag6; split; [|constructor].
      + auto.
      + rewrite Heqlidodag6. cbn.
        rewrite Heqlidodag5. cbn.
        rewrite Heqlidodag4. cbn.
        rewrite Heqlidodag3. cbn.
        rewrite Heqlidodag2. cbn.
        rewrite Heqlidodag1. cbn.
        apply HR.
      + intros r' id' Hr'1 Hr'2.
        assert (Hr'3 : 1 <= r' <= R \/ r' = S R) by lia.
        clear Hr'1; destruct Hr'3 as [Hr'1|Hr'1]; [|subst r'].
        * pose proof R_mys_lidodag_ecache _ _ _ HR _ _ Hr'1 Hr'2 as (r'' & Hr''1 & Hr''2).
          exists r''; split; auto.
          rewrite Heqlidodag6. cbn. client_unfold.
          rewrite NatMap_gso; [|lia]. client_fold (lidodag_lido lidodag5).
          rewrite Heqlidodag5. cbn. client_unfold. client_fold (lidodag_lido lidodag4).
          rewrite Heqlidodag4. cbn. client_unfold.
          rewrite NatMap_gso; [|lia]. client_fold (lidodag_lido lidodag3).
          rewrite Heqlidodag3. cbn. client_unfold. client_fold (lidodag_lido lidodag2).
          rewrite Heqlidodag2. cbn. client_unfold.
          rewrite NatMap_gso; [|lia]. client_fold (lidodag_lido lidodag1).
          rewrite Heqlidodag1. cbn. client_unfold. client_fold (lidodag_lido lidodag).
          auto.
        * exists R'; split; [unfold mcommit_is_pull_src; tauto|].
          rewrite Heqlidodag6. cbn. client_unfold.
          rewrite NatMap_gse; [|lia]. client_reduce.
          rewrite Heqlidodag5. cbn. client_unfold. client_fold (lidodag_lido lidodag4).
          rewrite Heqlidodag4. cbn. client_unfold.
          rewrite NatMap_gse; [|lia]. client_reduce.
          rewrite Heqlidodag3. cbn. client_unfold. client_fold (lidodag_lido lidodag2).
          rewrite Heqlidodag2. cbn. client_unfold.
          rewrite NatMap_gse; [|lia]. client_reduce. auto.
      + intros r' ps'.
        assert (Hr' : r' <> S R \/ r' = S R) by lia.
        destruct Hr' as [Hr'|Hr']; [|subst r'].
        * rewrite Heqlidodag6. cbn. client_unfold.
          rewrite NatMap_gso; [|lia]. client_fold (lidodag_lido lidodag5).
          rewrite Heqlidodag5. cbn. client_unfold. client_fold (lidodag_lido lidodag4).
          rewrite Heqlidodag4. cbn. client_unfold.
          rewrite NatMap_gso; [|lia]. client_fold (lidodag_lido lidodag3).
          rewrite Heqlidodag3. cbn. client_unfold. client_fold (lidodag_lido lidodag2).
          rewrite Heqlidodag2. cbn. client_unfold.
          rewrite NatMap_gso; [|lia]. client_fold (lidodag_lido lidodag1).
          rewrite Heqlidodag1. cbn. client_unfold. client_fold (lidodag_lido lidodag).
          intros Hps'.
          pose proof R_mys_lidodag_ecache_inv _ _ _ HR _ _ Hps' as (? & ?).
          split; auto.
        * rewrite Heqlidodag6. cbn. client_unfold.
          rewrite NatMap_gse; [|lia]. client_reduce.
          rewrite Heqlidodag5. cbn. client_unfold. client_fold (lidodag_lido lidodag4).
          rewrite Heqlidodag4. cbn. client_unfold.
          rewrite NatMap_gse; [|lia]. client_reduce.
          rewrite Heqlidodag3. cbn. client_unfold. client_fold (lidodag_lido lidodag2).
          rewrite Heqlidodag2. cbn. client_unfold.
          rewrite NatMap_gse; [|lia]. client_reduce.
          intros Hps'; inversion Hps'; subst ps'; clear Hps'.
          cbn [fst]; split; auto. unfold mcommit_is_pull_src; tauto.
      + intros r' id' Hr'1 Hr'2.
        assert (Hr'3 : 1 <= r' <= R \/ r' = S R) by lia.
        clear Hr'1; destruct Hr'3 as [Hr'1|Hr'1]; [|subst r'].
        * pose proof R_mys_lidodag_mcache _ _ _ HR _ _ Hr'1 Hr'2 as Hr'3.
          rewrite Heqlidodag6. cbn. client_unfold.
          rewrite NatMap_gso; [|lia]. client_fold (lidodag_lido lidodag5).
          rewrite Heqlidodag5. cbn. client_unfold. client_fold (lidodag_lido lidodag4).
          rewrite Heqlidodag4. cbn. client_unfold.
          rewrite NatMap_gso; [|lia]. client_fold (lidodag_lido lidodag3).
          rewrite Heqlidodag3. cbn. client_unfold. client_fold (lidodag_lido lidodag2).
          rewrite Heqlidodag2. cbn. client_unfold.
          rewrite NatMap_gso; [|lia]. client_fold (lidodag_lido lidodag1).
          rewrite Heqlidodag1. cbn. client_unfold. client_fold (lidodag_lido lidodag).
          auto.
        * rewrite Heqlidodag6. cbn. client_unfold.
          rewrite NatMap_gse; [|lia]. client_reduce.
          rewrite Heqlidodag5. cbn. client_unfold. client_fold (lidodag_lido lidodag4).
          rewrite Heqlidodag4. cbn. client_unfold.
          rewrite NatMap_gse; [|lia]. client_reduce.
          rewrite Heqlidodag3. cbn.
          pose proof mcommit_commit_eq _ _ _ _ Hval_mcommit Hps Hr'2. subst id. auto.
      + intros r' v id'.
        assert (Hr' : r' <> S R \/ r' = S R) by lia.
        destruct Hr' as [Hr'|Hr']; [|subst r'].
        * rewrite Heqlidodag6. cbn. client_unfold.
          rewrite NatMap_gso; [|lia]. client_fold (lidodag_lido lidodag5).
          rewrite Heqlidodag5. cbn. client_unfold. client_fold (lidodag_lido lidodag4).
          rewrite Heqlidodag4. cbn. client_unfold.
          rewrite NatMap_gso; [|lia]. client_fold (lidodag_lido lidodag3).
          rewrite Heqlidodag3. cbn. client_unfold. client_fold (lidodag_lido lidodag2).
          rewrite Heqlidodag2. cbn. client_unfold.
          rewrite NatMap_gso; [|lia]. client_fold (lidodag_lido lidodag1).
          rewrite Heqlidodag1. cbn. client_unfold. client_fold (lidodag_lido lidodag).
          intros Hr'1.
          pose proof R_mys_lidodag_mcache_inv _ _ _ HR _ _ _ Hr'1 as (? & ? & ?).
          repeat split; auto.
        * rewrite Heqlidodag6. cbn. client_unfold.
          rewrite NatMap_gse; [|lia]. client_reduce.
          rewrite Heqlidodag5. cbn. client_unfold. client_fold (lidodag_lido lidodag4).
          rewrite Heqlidodag4. cbn. client_unfold.
          rewrite NatMap_gse; [|lia]. client_reduce.
          assert (Hv : v <> 0 \/ v = 0) by lia.
          destruct Hv as [Hv|Hv]; [|subst v].
          --  rewrite NatMap_gso; [|lia].
              rewrite Heqlidodag3. cbn. client_unfold. client_fold (lidodag_lido lidodag2).
              rewrite Heqlidodag2. cbn. client_unfold.
              rewrite NatMap_gse; [|lia]. client_reduce.
              rewrite Heqlidodag1. cbn. client_unfold. client_fold (lidodag_lido lidodag).
              intros Hv1.
              pose proof R_mys_lidodag_mcache_inv _ _ _ HR _ _ _ Hv1; lia.
          --  rewrite NatMap_gse; [|lia].
              intros Hid; inversion Hid; subst id'; clear Hid. auto.
      + intros r' id' Hr'1 Hr'2.
        assert (Hr'3 : 1 <= r' <= R \/ r' = S R) by lia.
        clear Hr'1; destruct Hr'3 as [Hr'1|Hr'1]; [|subst r'].
        * pose proof R_mys_lidodag_ccache _ _ _ HR _ _ Hr'1 Hr'2 as Hr'3.
          rewrite Heqlidodag6. cbn. client_unfold.
          rewrite NatMap_gso; [|lia]. client_fold (lidodag_lido lidodag5).
          rewrite Heqlidodag5. cbn. client_unfold. client_fold (lidodag_lido lidodag4).
          rewrite Heqlidodag4. cbn. client_unfold.
          rewrite NatMap_gso; [|lia]. client_fold (lidodag_lido lidodag3).
          rewrite Heqlidodag3. cbn. client_unfold. client_fold (lidodag_lido lidodag2).
          rewrite Heqlidodag2. cbn. client_unfold.
          rewrite NatMap_gso; [|lia]. client_fold (lidodag_lido lidodag1).
          rewrite Heqlidodag1. cbn. client_unfold. client_fold (lidodag_lido lidodag).
          auto.
        * rewrite Heqlidodag6. cbn. client_unfold.
          rewrite NatMap_gse; [|lia]. client_reduce.
          rewrite Heqlidodag5. cbn. client_unfold. client_fold (lidodag_lido lidodag4).
          rewrite Heqlidodag4. cbn. client_unfold.
          rewrite NatMap_gse; [|lia]. client_reduce.
          rewrite Heqlidodag3. cbn. client_unfold. client_fold (lidodag_lido lidodag2).
          rewrite Heqlidodag2. cbn. client_unfold.
          rewrite NatMap_gse; [|lia]. client_reduce.
          rewrite Heqlidodag1. cbn. client_unfold. client_fold (lidodag_lido lidodag).
          match goal with |- context[match ?x with _ => _ end] => remember x as f; destruct f; auto end.
          symmetry in Heqf.
          pose proof client_commit_valid _ _ _ (lidodag_lido_valid _ Hval_lidodag) Heqf as Hfind.
          match type of Hfind with ?x <> None => remember x as f; destruct f; [symmetry in Heqf0; clear Hfind|contradiction] end.
          pose proof R_mys_lidodag_mcache_inv _ _ _ HR _ _ _ Heqf0 as (_ & ? & _). lia.
      + intros r' v Hr'.
        assert (Hr'1 : 1 <= r' <= R \/ r' = S R) by lia.
        clear Hr'; destruct Hr'1 as [Hr'|Hr']; [|subst r'].
        * rewrite Heqlidodag6. cbn. client_unfold.
          rewrite NatMap_gso; [|lia]. client_fold (lidodag_lido lidodag5).
          rewrite Heqlidodag5. cbn. client_unfold. client_fold (lidodag_lido lidodag4).
          rewrite Heqlidodag4. cbn. client_unfold.
          rewrite NatMap_gso; [|lia]. client_fold (lidodag_lido lidodag3).
          rewrite Heqlidodag3. cbn. client_unfold. client_fold (lidodag_lido lidodag2).
          rewrite Heqlidodag2. cbn. client_unfold.
          rewrite NatMap_gso; [|lia]. client_fold (lidodag_lido lidodag1).
          rewrite Heqlidodag1. cbn. client_unfold. client_fold (lidodag_lido lidodag).
          intros Hr'1.
          pose proof R_mys_lidodag_ccache_inv _ _ _ HR _ _ Hr' Hr'1; auto.
        * rewrite Heqlidodag6. cbn. client_unfold.
          rewrite NatMap_gse; [|lia]. client_reduce.
          rewrite Heqlidodag5. cbn. client_unfold. client_fold (lidodag_lido lidodag4).
          rewrite Heqlidodag4. cbn. client_unfold.
          rewrite NatMap_gse; [|lia]. client_reduce.
          rewrite Heqlidodag3. cbn. client_unfold. client_fold (lidodag_lido lidodag2).
          rewrite Heqlidodag2. cbn. client_unfold.
          rewrite NatMap_gse; [|lia]. client_reduce.
          rewrite Heqlidodag1. cbn. client_unfold. client_fold (lidodag_lido lidodag).
          match goal with |- context[match ?x with _ => _ end] => remember x as f; destruct f end.
          --  symmetry in Heqf.
              pose proof client_commit_valid _ _ _ (lidodag_lido_valid _ Hval_lidodag) Heqf as Hfind.
              match type of Hfind with ?x <> None => remember x as f; destruct f; [symmetry in Heqf0; clear Hfind|contradiction] end.
              pose proof R_mys_lidodag_mcache_inv _ _ _ HR _ _ _ Heqf0 as (_ & ? & _). lia.
          --  intros Hv; inversion Hv; auto.
      + intros nid.
        destruct (decide (nid = bado_leader_at (S R))).
        * subst nid.
          rewrite Heqlidodag6; cbn.
          client_unfold; client_reduce; rewrite NatMap_gse; auto.
        * rewrite Heqlidodag6; cbn.
          client_unfold; client_reduce; rewrite NatMap_gso; auto.
          client_fold (lidodag_lido lidodag5).
          rewrite Heqlidodag5; cbn.
          client_unfold; client_reduce; rewrite NatMap_gso; auto.
          client_fold (lidodag_lido lidodag4).
          rewrite Heqlidodag4; cbn.
          client_unfold; client_reduce; rewrite NatMap_gso; auto.
          client_fold (lidodag_lido lidodag3).
          rewrite Heqlidodag3; cbn.
          client_unfold; client_reduce; rewrite NatMap_gso; auto.
          client_fold (lidodag_lido lidodag2).
          rewrite Heqlidodag2; cbn.
          client_unfold; client_reduce; rewrite NatMap_gso; auto.
          client_fold (lidodag_lido lidodag1).
          rewrite Heqlidodag1; cbn.
          client_unfold; client_reduce; rewrite NatMap_gso; auto.
          apply HR.

    - (* nack *)
      exists lidodag; split; [|constructor].
      + apply reachable_self.
      + apply HR.
      + intros r id Hr1 Hr2.
        assert (Hr3 : 1 <= r <= R \/ r = S R) by lia.
        clear Hr1; destruct Hr3 as [Hr1|Hr1]; [|subst r].
        * eapply HR; eauto.
        * pose proof mcommit_safety _ _ _ Hval_mcommit Hr2; contradiction.
      + intros r ps Hps.
        pose proof R_mys_lidodag_ecache_inv _ _ _ HR _ _ Hps as (Hr1 & Hr2).
        split; auto.
      + intros r id Hr1 Hr2.
        assert (Hr3 : 1 <= r <= R \/ r = S R) by lia.
        clear Hr1; destruct Hr3 as [Hr1|Hr1]; [|subst r].
        * eapply HR; eauto.
        * pose proof mcommit_safety _ _ _ Hval_mcommit Hr2; contradiction.
      + intros r v id Hv.
        pose proof R_mys_lidodag_mcache_inv _ _ _ HR _ _ _ Hv as (? & ? & ?).
        repeat split; auto.
      + intros r id Hr1 Hr2.
        assert (Hr3 : 1 <= r <= R \/ r = S R) by lia.
        clear Hr1; destruct Hr3 as [Hr1|Hr1]; [|subst r].
        * eapply HR; eauto.
        * pose proof mcommit_safety _ _ _ Hval_mcommit Hr2; contradiction.
      + intros r v Hr1 Hr2.
        assert (Hr3 : 1 <= r <= R \/ r = S R) by lia.
        clear Hr1; destruct Hr3 as [Hr1|Hr1]; [|subst r].
        * eapply R_mys_lidodag_ccache_inv; eauto.
        * pose proof client_commit_valid _ _ _ (lidodag_lido_valid _ Hval_lidodag) Hr2 as Hfind.
          match type of Hfind with ?x <> None => remember x as f; destruct f; [clear Hfind; symmetry in Heqf|contradiction] end.
          pose proof R_mys_lidodag_mcache_inv _ _ _ HR _ _ _ Heqf as (? & _); auto.
      + apply HR.
  Qed.

  Lemma mcommit_refine_lidodag_step_next_round' :
    forall mcommit mcommit' lidodag R R' high_R high_R',
    mcommit_valid mcommit ->
    lidodag_valid lidodag ->
    mcommit_step mcommit mcommit' ->
    mcommit_high_round mcommit high_R ->
    mcommit_high_round mcommit' high_R' ->
    R <= high_R ->
    R_mys_lidodag mcommit lidodag R ->
    R <= R' <= high_R' ->
    exists lidodag',
      lidodag_reachable lidodag lidodag' /\
      R_mys_lidodag mcommit' lidodag' R'.
  Proof.
    intros mcommit mcommit' lidodag R R' high_R high_R'.
    intros Hval_mcommit Hval_lidodag Hstep Hhigh1 Hhigh2 Hle1 HR Hle2.
    induction R'.
    - assert (R = 0) by lia; subst R; clear Hle1 Hle2.
      pose proof (mcommit_refine_lidodag_step_same_round _ _ _ 0 _ Hval_mcommit Hval_lidodag Hstep Hhigh1 ltac:(lia) HR) as (lidodag' & Hreach & HR').
      exists lidodag'; split; auto.
    - assert (Hle3 : S R' = R \/ R <= R' <= high_R') by lia.
      destruct Hle3 as [Hle3 | Hle3].
      + subst R.
        pose proof (mcommit_refine_lidodag_step_same_round _ _ _ _ _ Hval_mcommit Hval_lidodag Hstep Hhigh1 Hle1 HR) as (lidodag' & Hreach & HR').
        exists lidodag'; split; auto.
      + specialize (IHR' Hle3) as (lidodag' & Hreach & HR').
        assert (Hval_mcommit' : mcommit_valid mcommit') by (eapply valid_reach_valid; try apply Hval_mcommit; eapply reachable_step; [apply reachable_self | auto]).
        assert (Hval_lidodag' : lidodag_valid lidodag') by (eapply valid_reach_valid; try apply Hval_lidodag; apply Hreach).
        pose proof (mcommit_refine_lidodag_step_next_round _ _ R' _ Hval_mcommit' Hval_lidodag' Hhigh2 ltac:(lia) HR') as (lidodag'' & Hreach' & HR'').
        exists lidodag''; split; auto.
        eapply reachable_trans.
        1: apply Hreach.
        apply Hreach'.
  Qed.

  Lemma mcommit_refine_lidodag_step :
    forall mcommit mcommit' lidodag high_R high_R',
    mcommit_valid mcommit ->
    lidodag_valid lidodag ->
    mcommit_step mcommit mcommit' ->
    mcommit_high_round mcommit high_R ->
    mcommit_high_round mcommit' high_R' ->
    R_mys_lidodag mcommit lidodag high_R ->
    exists lidodag',
      lidodag_reachable lidodag lidodag' /\
      R_mys_lidodag mcommit' lidodag' high_R'.
  Proof.
    intros mcommit mcommit' lidodag high_R high_R'.
    intros Hval_mcommit Hval_lidodag Hstep Hhigh1 Hhigh2 HR.
    assert (Hreach : mcommit_reachable mcommit mcommit') by (eapply reachable_step; [apply reachable_self | auto]).
    pose proof (mcommit_high_round_mono _ _ _ _ Hreach Hhigh1 Hhigh2) as Hle.
    pose proof (mcommit_refine_lidodag_step_next_round' _ _ _ high_R high_R' high_R high_R' Hval_mcommit Hval_lidodag Hstep Hhigh1 Hhigh2 ltac:(auto) HR ltac:(split; auto)) as (lidodag' & Hreach' & HR').
    exists lidodag'; split; auto.
  Qed.

  Lemma mcommit_refine_lidodag :
    forall mcommit high_R,
    mcommit_valid mcommit ->
    mcommit_high_round mcommit high_R ->
    exists lidodag,
      lidodag_valid lidodag /\
      R_mys_lidodag mcommit lidodag high_R.
  Proof.
    intros mcommit high_R Hval.
    revert high_R.
    induction Hval as [| st st' Hval IH Hstep].
    - intros high_R Hhigh.
      cbn in Hhigh.
      pose proof (mcommit_high_round_init) as Hhigh'.
      pose proof (mcommit_high_round_uniq _ _ _ Hhigh Hhigh'); subst high_R.
      exists lidodag_state_null; split.
      1: apply valid_state_init.
      apply mcommit_refine_lidodag_init.
    - pose proof (mcommit_high_round_ex st) as (high_R & Hhigh).
      specialize (IH _ Hhigh) as (lidodag & Hval_lidodag & HR).
      intros high_R' Hhigh'.
      pose proof (mcommit_refine_lidodag_step _ _ _ _ _ Hval Hval_lidodag Hstep Hhigh Hhigh' HR) as (lidodag' & Hreach & HR').
      exists lidodag'; split; auto.
      eapply valid_reach_valid; try apply Hval_lidodag; auto.
  Qed.

End MysticetiRefine.
