(* Liveness of Mysticeti will be proven over two refinement layers.

   In the first layer, we show that under a weak round-jumping policy,
   every honest leader vertex proposed after GST will get at least f+1
   certificates from honest nodes.

   In the second layer, we show that under a strong (more refined)
   round-jumping policy, every honest leader vertex will get 2f+1 certificates.
 *)

From Coq Require Import
  List
  PeanoNat
  Lia
  Peano_dec.
From advert.lib Require Import
  Maps
  Decision
  Semantics
  Tactics.
From advert.specs Require Import
  UDAG
  Config.
From advert.impl.mysticeti Require Import
  MysticetiDAG.
Import Notations ListNotations.

Section Lemmas.
  Lemma transition_point (f : nat -> nat) : forall n x, (f 0 < x)%nat -> (forall m, (m <= n)%nat -> (f m < x)%nat) \/ (exists m, (m < n)%nat /\ (f m < x)%nat /\ (f (S m) >= x)%nat).
  Proof. intros n x Hlt. induction n.
         - left. intros m Hm. destruct m eqn: Heqm; auto; lia.
         - destruct IHn as [IHn | IHn].
           + destruct ((x <=? f (S n))%nat) eqn:Hle.
             * right. rewrite Nat.leb_le in Hle. exists n. repeat split; auto.
             * left. rewrite Nat.leb_gt in Hle. intros m Hm. assert (Hm' : (m <= n)%nat \/ m = S n) by lia. destruct Hm'. 1: apply IHn; auto. subst m; auto.
           + destruct IHn as (m & Hm1 & Hm2 & Hm3). right; exists m; repeat split; auto.
  Qed.

  Fixpoint list_min l := match l with [] => 0%nat | x :: xs => match xs with [] => x | y :: ys => let t := list_min xs in if (x <=? t)%nat then x else t end end.

  Lemma list_min_ge : forall l n, l <> [] -> (list_min l >= n)%nat <-> Forall (fun k => (k >= n)%nat) l.
  Proof. intros l n Hex. induction l.
         - contradiction.
         - clear Hex. destruct l.
           + cbn. split. 1: intro; constructor; auto. intro Hge; rewrite Forall_forall in Hge; apply Hge; left; easy.
           + specialize (IHl ltac:(discriminate)). cbn. remember (list_min (n0 :: l)) as x. cbn in Heqx. rewrite <- Heqx. clear Heqx.
             destruct ((a <=? x)%nat) eqn:Hle; try rewrite Nat.leb_le in Hle; try rewrite Nat.leb_gt in Hle. all: split.
             * intro. constructor. 1: auto. apply IHl. lia.
             * intro Hge. rewrite Forall_forall in Hge. apply Hge; left; easy.
             * intro Hge. destruct IHl as (IHl & _). specialize (IHl Hge). constructor. 2: auto. lia.
             * intro Hge. apply Forall_inv_tail in Hge. destruct IHl as (_ & IHl). apply IHl; auto.
  Qed.

  Lemma list_max_ex' : forall l, l = [] \/ In (list_max l) l.
  Proof. induction l.
         - left; easy.
         - right. cbn. destruct l.
           + cbn. left; lia.
           + fold (list_max (n :: l)). remember (list_max (n ::l)) as x. clear Heqx. destruct IHl as [? | IHl]. 1: discriminate.
             apply Nat.max_case. 1: left; easy. right; auto.
  Qed.

  Lemma list_max_ex : forall l, l <> [] -> In (list_max l) l.
  Proof. intros l Hex. pose proof (list_max_ex' l) as Hin. destruct Hin as [? | Hin]; try contradiction; auto.
  Qed.

  Lemma list_min_ex' : forall l, l = [] \/ In (list_min l) l.
  Proof. induction l.
         - left; easy.
         - right. cbn. destruct l.
           + left; easy.
           + destruct IHl as [? | IHl]. 1: discriminate. remember (list_min (n :: l)) as x. clear Heqx.
             destruct ((a <=? x)%nat). 1: left; easy. right; auto.
  Qed.

  Lemma list_min_ex : forall l, l <> [] -> In (list_min l) l.
  Proof. intros l Hex. pose proof (list_min_ex' l) as Hin. destruct Hin as [? | Hin]; try contradiction; auto.
  Qed.

  Lemma list_min_eq : forall l x, In x l -> (forall y, In y l -> (y >= x)%nat) -> list_min l = x.
  Proof. intros l x. induction l.
         - intro. contradiction.
         - intro Hin. destruct Hin as [Hin | Hin].
           + subst a. intro Hge. cbn. destruct l. 1: easy.
             destruct ((x <=? list_min (n :: l))%nat) eqn:Hle; try rewrite Nat.leb_le in Hle; try rewrite Nat.leb_gt in Hle. 1: easy.
             pose proof (list_min_ex (n :: l) ltac:(discriminate)). specialize (Hge (list_min (n :: l)) ltac:(right; auto)). lia.
           + intro Hge. specialize (IHl Hin ltac:(intros y Hin'; apply (Hge y ltac:(right; auto)))). cbn. destruct l; try contradiction.
             rewrite IHl. specialize (Hge a ltac:(left; easy)). destruct ((a <=? x)%nat) eqn:Hle; try rewrite Nat.leb_le in Hle; try rewrite Nat.leb_gt in Hle; lia.
  Qed.

  Lemma list_min_le : forall l x, In x l -> (list_min l <= x)%nat.
  Proof. intros l x. induction l.
         - intro. contradiction.
         - intro Hin. destruct Hin as [Hin | Hin].
           + subst a. cbn. destruct l; auto.
             destruct ((x <=? list_min (n :: l))%nat) eqn:Hle; try rewrite Nat.leb_le in Hle; try rewrite Nat.leb_gt in Hle; lia.
           + specialize (IHl Hin). cbn. destruct l; try contradiction.
             destruct ((a <=? x)%nat) eqn:Hle; try rewrite Nat.leb_le in Hle; try rewrite Nat.leb_gt in Hle;
             destruct ((a <=? list_min (n :: l))%nat) eqn:Hre; try rewrite Nat.leb_le in Hre; try rewrite Nat.leb_gt in Hre; try lia.
  Qed.

  Lemma map_nonempty {T U : Type} : forall (l : list T) (f : T -> U), l <> [] -> map f l <> [].
  Proof. intros l f H.
         destruct l.
         - contradiction.
         - cbn; discriminate.
  Qed.

  Lemma all_true_or_one_false {T : Type} {P Q} (l : list T) : (forall x, In x l -> (P x \/ Q x)) -> (exists x, In x l) -> ((forall x, In x l -> Q x) \/ (exists x, In x l /\ P x)).
  Proof. intros Hor Hex. induction l.
         - destruct Hex; contradiction.
         - destruct l.
           + specialize (Hor a ltac:(left; easy)). destruct Hor as [Hm2 | Hm2].
             * right. exists a. split; auto. left; easy.
             * left; intros x Hin; destruct Hin as [Hin | Hin]; try contradiction; subst; auto.
           + specialize (IHl ltac:(intros; apply Hor; right; auto) ltac:(eexists; left; easy)).
             specialize (Hor a ltac:(left; easy)). destruct Hor as [Ha | Ha]. 1: right; exists a; split; auto; left; easy.
             destruct IHl as [IHl | IHl]. 2: right; destruct IHl as (x & Hx1 & Hx2); exists x; split; auto; right; auto.
             left. intros x Hx. destruct Hx as [Hx | Hx]. 1: subst; auto. apply IHl; auto.
  Qed.
End Lemmas.

Section WeakLiveness.
  Context `{config : !BADO_Config} `{dag_config : !DAG_Config} `{participant : !BADO_Participant} `{leader : !BADO_Leader} `{node_assump : !BADO_NodeAssump} `{assump : !BADO_Assump}.

  Definition hon_set := filter (fun nid => match bado_node_assump nid with Byzantine => false | _ => true end) bado_participant.
  Definition sync_set := filter (fun nid => match bado_node_assump nid with Synchronous => true | _ => false end) bado_participant.

  Lemma sync_set_correct : forall nid, In nid sync_set <-> In nid bado_participant /\ bado_node_assump nid = Synchronous.
  Proof. intro nid. split.
         - intro Hnid. unfold sync_set in Hnid. rewrite filter_In in Hnid. split. 1: apply Hnid. destruct Hnid as (_ & Hnid). destruct (bado_node_assump nid); try discriminate; easy.
         - intro Hnid. destruct Hnid as (Hnid1 & Hnid2). unfold sync_set. rewrite filter_In. split; auto. rewrite Hnid2. easy.
  Qed.

  Lemma sync_set_quorum : is_quorum bado_comm sync_set.
  Proof. pose proof bado_comm_live as Hlive. unfold committee_live in Hlive. destruct Hlive as (quorum & Hquorum1 & Hquorum2). pose proof ((committee_valid _ bado_comm_safe) quorum) as Hquorum3.
         eapply is_quorum_superset. 1: unfold is_quorum; exists quorum; split; auto; apply incl_refl.
         unfold incl. intros nid Hnid. specialize (Hquorum2 _ Hnid). specialize (Hquorum3 _ Hquorum1 Hnid).
         rewrite sync_set_correct. split; auto.
  Qed.

  Lemma sync_set_nonempty : sync_set <> [].
  Proof. intro Heq. pose proof (sync_set_quorum) as Hquorum. pose proof (quorum_exists_honest bado_comm_safe _ Hquorum) as Hnid. destruct Hnid as (nid & Hnid1 & _). rewrite Heq in Hnid1. cbn in Hnid1. auto.
  Qed.

  Lemma sync_set_nonempty' : exists x, In x sync_set.
  Proof. pose proof sync_set_nonempty; destruct sync_set; try contradiction.
         eexists; left; auto.
  Qed.
         

  Class MysticetiWeakOracle : Type := {
    mys_weak_oracle_dag : nat -> MDAG_State;
    mys_weak_oracle_local_ar : nat -> nat -> nat;
    mys_weak_oracle_local_rt : nat -> nat -> nat;

    (* Reachability *)
    mys_weak_oracle_init_valid : mdag_valid (mys_weak_oracle_dag 0);
    mys_weak_oracle_reachable : forall n, mdag_reachable (mys_weak_oracle_dag n) (mys_weak_oracle_dag (S n));

    (* Assumptions on round and timeouts *)
    mys_weak_oracle_ar_init_pos :
      forall nid,
      In nid bado_participant ->
      bado_node_assump nid = Synchronous ->
      mys_weak_oracle_local_ar 0 nid > 0;

    mys_weak_oracle_rt_init :
      forall nid,
      In nid bado_participant ->
      bado_node_assump nid = Synchronous ->
      mys_weak_oracle_local_rt 0 nid <= 3;

    mys_weak_oracle_ar_step :
      forall n nid,
      In nid bado_participant ->
      bado_node_assump nid = Synchronous ->
      let ar := mys_weak_oracle_local_ar n nid in
      let ar' := mys_weak_oracle_local_ar (S n) nid in
      let rt := mys_weak_oracle_local_rt n nid in
      let rt' := mys_weak_oracle_local_rt (S n) nid in
      ar' = ar /\ rt' = rt - 1 \/ ar' > ar /\ rt' = 3;

    mys_weak_oracle_ar_ge_vert :
      forall n nid id,
      bado_node_assump nid = Synchronous ->
      match NatMap_find id (mys_weak_oracle_dag n).(mdag_udag).(udag_verts) with
      | None => True
      | Some v => v.(udag_vert_builder) = nid ->
                  mys_weak_oracle_local_ar n nid >= v.(udag_vert_round) end;

    mys_weak_oracle_ar_has_vert :
      forall n nid,
      In nid bado_participant ->
      bado_node_assump nid = Synchronous ->
      mys_weak_oracle_local_ar n nid = 1 \/
      exists id,
        match NatMap_find id (mys_weak_oracle_dag n).(mdag_udag).(udag_verts) with
        | None => False
        | Some v => v.(udag_vert_builder) = nid /\ v.(udag_vert_round) = mys_weak_oracle_local_ar n nid - 1
        end;

    mys_weak_oracle_ar_has_quorum :
      forall n nid,
      In nid bado_participant ->
      bado_node_assump nid = Synchronous ->
      mys_weak_oracle_local_ar n nid = 1 \/
      exists quorum,
        is_quorum bado_comm quorum /\
        forall nid',
        In nid' quorum ->
        exists id,
          match NatMap_find id (mys_weak_oracle_dag n).(mdag_udag).(udag_verts) with
          | None => False
          | Some v => v.(udag_vert_builder) = nid' /\ v.(udag_vert_round) = mys_weak_oracle_local_ar n nid - 1
          end;

    mys_weak_oracle_timeout_init :
      forall nid r,
      bado_node_assump nid = Synchronous ->
      In (nid, r) (mys_weak_oracle_dag 0).(mdag_timeouts) ->
      r < mys_weak_oracle_local_ar 0 nid \/
      r = mys_weak_oracle_local_ar 0 nid /\
      mys_weak_oracle_local_rt 0 nid = 0;

    mys_weak_oracle_timeout :
      forall n nid,
      In nid bado_participant ->
      bado_node_assump nid = Synchronous ->
      mys_weak_oracle_local_rt n nid = 0 ->
      mys_weak_oracle_local_ar (S n) nid = mys_weak_oracle_local_ar n nid ->
      In (nid, mys_weak_oracle_local_ar n nid) (mys_weak_oracle_dag (S n)).(mdag_timeouts);

    mys_weak_oracle_timeout_inv :
      forall n nid r,
      In nid bado_participant ->
      bado_node_assump nid = Synchronous ->
      In (nid, r) (mys_weak_oracle_dag (S n)).(mdag_timeouts) ->
      In (nid, r) (mys_weak_oracle_dag n).(mdag_timeouts) \/
      r = mys_weak_oracle_local_ar n nid /\
      mys_weak_oracle_local_rt n nid = 0;

    (* Assumptions on progress *)
    mys_weak_oracle_catchup :
      forall n nid nid',
      In nid bado_participant ->
      bado_node_assump nid = Synchronous ->
      In nid' bado_participant ->
      bado_node_assump nid' = Synchronous ->
      mys_weak_oracle_local_ar (S n) nid' >= mys_weak_oracle_local_ar n nid;

    mys_weak_oracle_enter :
      forall n r,
      (forall nid,
       In nid bado_participant ->
       bado_node_assump nid = Synchronous ->
       exists id, match NatMap_find id (mys_weak_oracle_dag n).(mdag_udag).(udag_verts) with
                  | None => False
                  | Some v => v.(udag_vert_round) = r /\ v.(udag_vert_builder) = nid
                  end) ->
      forall nid,
      In nid bado_participant ->
      bado_node_assump nid = Synchronous ->
      mys_weak_oracle_local_ar (S n) nid > r;

    mys_weak_oracle_recv_leader_vert :
      forall n r,
      bado_node_assump (bado_leader_at r) = Synchronous ->
      (exists id, match NatMap_find id (mys_weak_oracle_dag n).(mdag_udag).(udag_verts) with
                  | None => False
                  | Some v => v.(udag_vert_round) = r /\ v.(udag_vert_builder) = bado_leader_at r
                  end) ->
      forall nid,
      In nid bado_participant ->
      bado_node_assump nid = Synchronous ->
      In (nid, r) (mys_weak_oracle_dag (S n)).(mdag_recv_leader_verts) \/
      (exists id, match NatMap_find id (mys_weak_oracle_dag (S n)).(mdag_udag).(udag_verts) with
                  | None => False
                  | Some v => v.(udag_vert_round) = S r /\ v.(udag_vert_builder) = nid
                  end);

    mys_weak_oracle_recv_cert :
      forall n r,
      r >= 1 ->
      bado_node_assump (bado_leader_at r) <> Byzantine ->
      (forall nid,
       In nid bado_participant ->
       bado_node_assump nid = Synchronous ->
       exists id, match NatMap_find id (mys_weak_oracle_dag n).(mdag_udag).(udag_verts) with
                  | None => False
                  | Some v => v.(udag_vert_round) = S r /\ v.(udag_vert_builder) = nid /\ mdag_vert_is_supporter v (mys_weak_oracle_dag n)
                  end) ->
      forall nid,
      In nid bado_participant ->
      bado_node_assump nid = Synchronous ->
      In (nid, r) (mys_weak_oracle_dag (S n)).(mdag_recv_certs) \/
      (exists id, match NatMap_find id (mys_weak_oracle_dag (S n)).(mdag_udag).(udag_verts) with
                  | None => False
                  | Some v => v.(udag_vert_round) = S (S r) /\ v.(udag_vert_builder) = nid
                  end);

    mys_weak_oracle_jump_no_vert :
      forall n nid r,
      bado_node_assump nid <> Byzantine ->
      In (nid, r) (mys_weak_oracle_dag n).(mdag_jumps) ->
      ~ exists id, match NatMap_find id (mys_weak_oracle_dag n).(mdag_udag).(udag_verts) with
                   | None => False
                   | Some v => v.(udag_vert_round) = r /\ v.(udag_vert_builder) = nid
                   end;
  }.

  Context `{weak_oracle : MysticetiWeakOracle}.

  Lemma mys_oracle_valid :
    forall n,
    mdag_valid (mys_weak_oracle_dag n).
  Proof. intros n.
         induction n.
         - apply mys_weak_oracle_init_valid.
         - eapply valid_reach_valid.
           + apply IHn.
           + apply mys_weak_oracle_reachable.
  Qed.

  Lemma mys_oracle_reachable :
    forall n m,
    n <= m ->
    mdag_reachable (mys_weak_oracle_dag n) (mys_weak_oracle_dag m).
  Proof. intros n m.
         induction m.
         - intros.
           replace n with 0 by lia.
           apply reachable_self.
         - intros Hle.
           assert (Hn : n <= m \/ n = S m) by lia.
           destruct Hn as [Hn | Hn].
           + specialize (IHm Hn).
             eapply reachable_trans.
             * apply IHm.
             * apply mys_weak_oracle_reachable.
           + subst n.
             apply reachable_self.
  Qed.

  Lemma mys_sync_timeout_ge :
    forall n r nid,
    In nid bado_participant ->
    bado_node_assump nid = Synchronous ->
    In (nid, r) (mys_weak_oracle_dag n).(mdag_timeouts) ->
    mys_weak_oracle_local_ar n nid > r \/
    mys_weak_oracle_local_ar n nid = r /\
    mys_weak_oracle_local_rt n nid = 0.
  Proof. intros n r nid Hpart Hsync.
         induction n.
         - intros Htimeout.
           pose proof (mys_weak_oracle_timeout_init _ _ Hsync Htimeout).
           lia.
         - intros Htimeout.
           pose proof (mys_weak_oracle_timeout_inv _ _ _ Hpart Hsync Htimeout) as Htimeout_inv.
           destruct Htimeout_inv as [Htimeout_inv | Htimeout_inv].
           + specialize (IHn Htimeout_inv).
             pose proof (mys_weak_oracle_ar_step n _ Hpart Hsync) as Hstep.
             cbn in Hstep.
             lia.
           + pose proof (mys_weak_oracle_ar_step n _ Hpart Hsync) as Hstep.
             cbn in Hstep.
             lia.
  Qed.

  Definition max_ar n nids := list_max (map (fun nid => mys_weak_oracle_local_ar n nid) nids).
  Definition min_rt n nids := list_min (map (fun nid => mys_weak_oracle_local_rt n nid) nids).
  Definition mys_ar n := max_ar n sync_set.
  Definition mys_rt n := min_rt n (filter (fun nid => mys_weak_oracle_local_ar n nid =? mys_ar n) sync_set).

  Lemma mys_ar_le :
    forall n nid,
    In nid bado_participant ->
    bado_node_assump nid = Synchronous ->
    mys_weak_oracle_local_ar n nid <= mys_ar n.
  Proof. intros n nid Hpart Hsync.
         unfold mys_ar, max_ar.
         match goal with |- context[list_max ?l] => pose proof (list_max_le l (list_max l)) as (Hle & _) end.
         specialize (Hle ltac:(auto)).
         rewrite Forall_forall in Hle.
         apply Hle.
         rewrite in_map_iff.
         exists nid; split; [auto | rewrite sync_set_correct; auto].
  Qed.

  Lemma mys_rt_set_nonempty : forall n, filter (fun nid => mys_weak_oracle_local_ar n nid =? mys_ar n) sync_set <> [].
  Proof. intros n.
         remember (mys_ar n) as r.
         unfold mys_ar, max_ar in Heqr.
         match type of Heqr with context[list_max ?l] => pose proof (list_max_ex l ltac:(apply map_nonempty; apply sync_set_nonempty)) as Hin end.
         rewrite <- Heqr in Hin.
         rewrite in_map_iff in Hin.
         destruct Hin as (nid & Hnid1 & Hnid2).
         match goal with |- ?l <> _ => assert (Hin : In nid l) end.
         2: intros Heq; rewrite Heq in Hin; contradiction.
         rewrite filter_In; split; auto.
         rewrite Nat.eqb_eq; auto.
  Qed.

  Lemma mys_rt_ex :
    forall n,
    exists nid, In nid sync_set /\ mys_weak_oracle_local_ar n nid = mys_ar n /\ mys_weak_oracle_local_rt n nid = mys_rt n.
  Proof. intros n.
         pose proof (mys_rt_set_nonempty n) as Hnonempty.
         remember (mys_rt n) as t.
         unfold mys_rt, min_rt in Heqt.
         match type of Heqt with context[list_min ?l] => pose proof (list_min_ex l ltac:(apply map_nonempty; apply Hnonempty)) as Hin end.
         rewrite <- Heqt in Hin.
         rewrite in_map_iff in Hin.
         destruct Hin as (nid & Hnid1 & Hnid2).
         rewrite filter_In in Hnid2.
         destruct Hnid2 as (Hnid2 & Hnid3).
         rewrite Nat.eqb_eq in Hnid3.
         exists nid; repeat split; auto.
  Qed.

  Lemma mys_rt_ge :
    forall n nid,
    In nid bado_participant ->
    bado_node_assump nid = Synchronous ->
    mys_weak_oracle_local_ar n nid = mys_ar n ->
    mys_weak_oracle_local_rt n nid >= mys_rt n.
  Proof. intros n nid Hpart Hsync Har.
         unfold mys_rt, min_rt.
         match goal with |- context[list_min ?l] => pose proof (list_min_ge l (list_min l) ltac:(apply map_nonempty; apply mys_rt_set_nonempty)) as (Hge & _) end.
         specialize (Hge ltac:(auto)).
         rewrite Forall_forall in Hge.
         apply Hge.
         rewrite in_map_iff.
         exists nid; split; auto.
         rewrite filter_In; split.
         1: rewrite sync_set_correct; auto.
         rewrite Nat.eqb_eq; auto.
  Qed.

  Lemma mys_local_rt_lim :
    forall n nid,
    In nid bado_participant ->
    bado_node_assump nid = Synchronous ->
    mys_weak_oracle_local_rt n nid <= 3.
  Proof. intros n nid Hpart Hsync.
         induction n.
         - apply mys_weak_oracle_rt_init; auto.
         - pose proof (mys_weak_oracle_ar_step n nid Hpart Hsync) as Hstep.
           cbn in Hstep.
           lia.
  Qed.

  Lemma mys_rt_lim :
    forall n,
    mys_rt n <= 3.
  Proof. intros n.
         pose proof (mys_rt_ex n) as (nid & Hnid1 & _ & Hnid2).
         rewrite sync_set_correct in Hnid1.
         pose proof (mys_local_rt_lim n nid ltac:(apply Hnid1) ltac:(apply Hnid1)).
         lia.
  Qed.

  Lemma mys_ar_step :
    forall n,
    let ar := mys_ar n in
    let ar' := mys_ar (S n) in
    let rt := mys_rt n in
    let rt' := mys_rt (S n) in
    ar' = ar /\ rt' = rt - 1 \/ ar' > ar /\ rt' = 3.
  Proof. intros n; cbn.
         pose proof (mys_rt_ex n) as (nid & Hnid1 & Hnid2 & Hnid3).
         pose proof Hnid1 as Hnid4.
         rewrite sync_set_correct in Hnid4.
         pose proof (mys_ar_le (S n) nid ltac:(apply Hnid4) ltac:(apply Hnid4)) as Hle.
         pose proof (mys_weak_oracle_ar_step n nid ltac:(apply Hnid4) ltac:(apply Hnid4)) as Hstep.
         cbn in Hstep.
         assert (Har : mys_ar (S n) = mys_ar n \/ mys_ar (S n) > mys_ar n) by lia.
         destruct Har as [Har | Har].
         - left; split; auto.
           destruct Hstep as [Hstep | Hstep].
           2: pose proof (mys_ar_le (S n) nid ltac:(apply Hnid4) ltac:(apply Hnid4)); lia.
           pose proof (mys_rt_ex (S n)) as (nid' & Hnid'1 & Hnid'2 & Hnid'3).
           pose proof Hnid'1 as Hnid'4.
           rewrite sync_set_correct in Hnid'4.
           pose proof (mys_weak_oracle_ar_step n nid' ltac:(apply Hnid'4) ltac:(apply Hnid'4)) as Hstep'.
           cbn in Hstep'.
           destruct Hstep' as [Hstep' | Hstep'].
           + pose proof (mys_rt_ge (S n) nid ltac:(apply Hnid4) ltac:(apply Hnid4) ltac:(lia)).
             pose proof (mys_rt_ge n nid' ltac:(apply Hnid'4) ltac:(apply Hnid'4) ltac:(lia)).
             lia.
           + pose proof (mys_rt_ge (S n) nid ltac:(apply Hnid4) ltac:(apply Hnid4) ltac:(lia)).
             pose proof (mys_local_rt_lim n nid ltac:(apply Hnid4) ltac:(apply Hnid4)).
             lia.
         - right; split; auto.
           clear nid Hnid1 Hnid2 Hnid3 Hnid4 Hle Hstep.
           pose proof (mys_rt_ex (S n)) as (nid & Hnid1 & Hnid2 & Hnid3).
           pose proof Hnid1 as Hnid4.
           rewrite sync_set_correct in Hnid4.
           pose proof (mys_ar_le n nid ltac:(apply Hnid4) ltac:(apply Hnid4)) as Hle.
           pose proof (mys_weak_oracle_ar_step n nid ltac:(apply Hnid4) ltac:(apply Hnid4)) as Hstep.
           cbn in Hstep; lia.
  Qed.

  Lemma mys_ar_mono :
    forall n m,
    n <= m -> mys_ar n <= mys_ar m.
  Proof. intros n m.
         induction m.
         - intros; replace n with 0 by lia.
           auto.
         - intros Hn'.
           assert (Hn : n <= m \/ n = S m) by lia; clear Hn'.
           destruct Hn as [Hn | Hn].
           + specialize (IHm Hn).
             pose proof (mys_ar_step m) as Hstep; cbn in Hstep.
             lia.
           + subst n; clear IHm; auto.
  Qed.

  Lemma mys_sync_elapse :
    forall n k nid,
    In nid bado_participant ->
    bado_node_assump nid = Synchronous ->
    mys_weak_oracle_local_ar (n + k) nid > mys_weak_oracle_local_ar n nid \/
    mys_weak_oracle_local_ar (n + k) nid = mys_weak_oracle_local_ar n nid /\
    mys_weak_oracle_local_rt (n + k) nid = mys_weak_oracle_local_rt n nid - k.
  Proof. intros n k nid Hpart Hsync.
         induction k.
         - replace (n + 0)%nat with n by lia.
           right; lia.
         - pose proof (mys_weak_oracle_ar_step (n + k) nid ltac:(auto) ltac:(auto)) as Hstep.
           cbn in Hstep.
           replace (n + S k)%nat with (S (n + k)) by lia.
           lia.
  Qed.

  Lemma mys_rt_pos_no_timeout' :
    forall n r,
    mys_ar n < r \/ mys_ar n = r /\ mys_rt n > 0 ->
    Forall (fun nid => ~ In (nid, r) (mys_weak_oracle_dag (S n)).(mdag_timeouts)) sync_set.
  Proof. intros n r Hpos.
         rewrite Forall_forall.
         intros nid Hnid1.
         pose proof Hnid1 as Hnid2.
         rewrite sync_set_correct in Hnid1.
         pose proof (mys_ar_le n nid ltac:(apply Hnid1) ltac:(apply Hnid1)) as Hle.
         assert (Hno_timeout : ~ In (nid, r) (mdag_timeouts (mys_weak_oracle_dag n))).
         { intros Htimeout.
           pose proof (mys_sync_timeout_ge _ _ _ ltac:(apply Hnid1) ltac:(apply Hnid1) Htimeout) as Hge.
           destruct Hge as [? | Hge]; [lia|].
           destruct Hpos as [? | Hpos]; [lia|].
           pose proof (mys_rt_ge n _ ltac:(apply Hnid1) ltac:(apply Hnid1) ltac:(lia)).
           lia.
         }
         intros Htimeout.
         pose proof (mys_weak_oracle_timeout_inv _ _ _ ltac:(apply Hnid1) ltac:(apply Hnid1) Htimeout) as [? | Htimeout_inv]; [contradiction|].
         destruct Hpos as [? | Hpos]; [lia|].
         pose proof (mys_rt_ge n _ ltac:(apply Hnid1) ltac:(apply Hnid1) ltac:(destruct Htimeout_inv; destruct Hpos; congruence)).
         lia.
  Qed.

  Lemma mys_rt_pos_no_timeout :
    forall n,
    mys_rt n > 0 ->
    Forall (fun nid => ~ In (nid, mys_ar n) (mys_weak_oracle_dag (S n)).(mdag_timeouts)) sync_set.
  Proof. intros; apply mys_rt_pos_no_timeout'; right; split; auto. Qed.

  Lemma mys_sync_progress_or_timeout :
    forall n nid,
    In nid bado_participant ->
    bado_node_assump nid = Synchronous ->
    mys_weak_oracle_local_ar (n + 4) nid > mys_weak_oracle_local_ar n nid \/
    mys_weak_oracle_local_ar (n + 4) nid = mys_weak_oracle_local_ar n nid /\
    In (nid, mys_weak_oracle_local_ar n nid) (mys_weak_oracle_dag (n + 4)).(mdag_timeouts).
  Proof. intros n nid Hpart Hsync.
         pose proof (mys_local_rt_lim n _ Hpart Hsync) as Hlim.
         pose proof (mys_sync_elapse n 3 _ Hpart Hsync) as Helapse.
         destruct Helapse as [Helapse | Helapse].
         - left.
           pose proof (mys_weak_oracle_ar_step (n + 3) _ Hpart Hsync) as Hstep.
           cbn in Hstep.
           replace (n + 4)%nat with (S (n + 3)) by lia.
           lia.
         - assert (Hrt : mys_weak_oracle_local_rt (n + 3) nid = 0) by lia.
           pose proof (mys_weak_oracle_ar_step (n + 3) _ Hpart Hsync) as Hstep.
           cbn in Hstep.
           replace (n + 4)%nat with (S (n + 3)) by lia.
           destruct Hstep as [Hstep | Hstep].
           + right; split.
             1: lia.
             pose proof (mys_weak_oracle_timeout (n + 3) _ Hpart Hsync Hrt ltac:(apply Hstep)) as Htimeout.
             destruct Helapse as (Helapse & ?).
             rewrite <- Helapse.
             apply Htimeout.
           + left.
             replace (n + 4)%nat with (S (n + 3)) by lia.
             lia.
  Qed.

  Lemma mys_progress' :
    forall n,
    mys_ar n < mys_ar (n + 6).
  Proof. intros n.
         assert (Henter : forall nid, In nid sync_set -> mys_weak_oracle_local_ar (n + 1) nid > mys_ar n \/ mys_weak_oracle_local_ar (n + 1) nid = mys_ar n).
         { pose proof (mys_rt_ex n) as (nid' & Hnid'1 & Hnid'2 & _).
           rewrite sync_set_correct in Hnid'1.
           intros nid Hnid.
           rewrite sync_set_correct in Hnid.
           pose proof (mys_weak_oracle_catchup n nid' nid ltac:(apply Hnid'1) ltac:(apply Hnid'1) ltac:(apply Hnid) ltac:(apply Hnid)).
           replace (n + 1)%nat with (S n) by lia.
           lia.
         }
         pose proof (all_true_or_one_false _ Henter sync_set_nonempty') as Har.
         destruct Har as [Har | Har].
         - pose proof (mys_sync_progress_or_timeout (n + 1)) as Hprog.
           match type of Hprog with forall nid, _ -> _ -> ?P => assert (Hprog' : forall nid, In nid sync_set -> P) end.
           { intros x Hx; rewrite sync_set_correct in Hx; apply Hprog; apply Hx. }
           clear Hprog.
           pose proof (all_true_or_one_false _ Hprog' sync_set_nonempty') as Har'; clear Hprog'.
           replace (n + 1 + 4)%nat with (n + 5)%nat in Har' by lia.
           destruct Har' as [Har' | Har'].
           + pose proof (mys_weak_oracle_enter (n + 5) (mys_ar n)) as Henter'.
             match type of Henter' with ?P -> _ => assert (Htimeout : P) end.
             { clear Henter'.
               intros nid Hnid1 Hnid2.
               specialize (Har' nid ltac:(rewrite sync_set_correct; split; auto)).
               specialize (Har nid ltac:(rewrite sync_set_correct; split; auto)).
               pose proof (mdag_timeout_exist_vert _ _ _ (mys_oracle_valid (n + 5)) ltac:(apply Har')) as (id & Hid).
               exists id.
               cond_case_auto Hid; try contradiction.
               rewrite Har in Hid.
               split; apply Hid.
             }
             specialize (Henter' Htimeout); clear Htimeout.
             replace (S (n + 5)) with (n + 6)%nat in Henter' by lia.
             pose proof (sync_set_nonempty') as (x & Hx).
             rewrite sync_set_correct in Hx.
             specialize (Henter' x ltac:(apply Hx) ltac:(apply Hx)).
             pose proof (mys_ar_le (n + 6) x ltac:(apply Hx) ltac:(apply Hx)).
             lia.
           + destruct Har' as (nid & Hnid1 & Hnid2).
             specialize (Har _ Hnid1).
             rewrite sync_set_correct in Hnid1.
             pose proof (mys_weak_oracle_ar_step (n + 5) nid ltac:(apply Hnid1) ltac:(apply Hnid1)) as Hstep.
             cbn in Hstep.
             replace (S (n + 5)) with (n + 6)%nat in Hstep by lia.
             pose proof (mys_ar_le (n + 6) nid ltac:(apply Hnid1) ltac:(apply Hnid1)).
             lia.
         - destruct Har as (nid & Hnid1 & Hnid2).
           rewrite sync_set_correct in Hnid1.
           pose proof (mys_sync_elapse (n + 1) 5 nid ltac:(apply Hnid1) ltac:(apply Hnid1)) as Helapse.
           replace (n + 1 + 5)%nat with (n + 6)%nat in Helapse by lia.
           pose proof (mys_ar_le (n + 6) nid ltac:(apply Hnid1) ltac:(apply Hnid1)).
           lia.
  Qed.

  Lemma mys_progress :
    forall r,
    r >= mys_ar 0 ->
    exists n, mys_ar n >= r.
  Proof. intros r Hgt.
         induction r.
         - exists 0; lia.
         - assert (Hr : S r = mys_ar 0 \/ r >= mys_ar 0) by lia.
           destruct Hr as [Hr | Hr].
           + exists 0; lia.
           + specialize (IHr Hr) as (n & Hn).
             exists (n + 6)%nat.
             pose proof (mys_progress' n).
             lia.
  Qed.

  Lemma mys_weak_oracle_ar_ge_jump :
    forall n r nid,
    bado_node_assump nid = Synchronous ->
    In (nid, r) (mys_weak_oracle_dag n).(mdag_jumps) ->
    mys_ar n >= r.
  Proof. intros n r nid Hsync Hjump.
         pose proof (mdag_jump_has_quorum _ _ _ (mys_oracle_valid n) Hjump) as (quorum & Hquorum1 & Hquorum2).
         pose proof (quorum_overlap_exists_honest bado_comm_safe _ _ Hquorum2 sync_set_quorum) as (nid' & Hnid'1 & Hnid'2 & _).
         clear Hquorum2.
         rewrite Forall_forall in Hquorum1.
         rewrite in_map_iff in Hnid'1.
         destruct Hnid'1 as (x & Hx1 & Hx2).
         specialize (Hquorum1 _ Hx2).
         cond_case_auto Hquorum1; try contradiction.
         subst nid' r.
         rewrite sync_set_correct in Hnid'2.
         pose proof (mys_weak_oracle_ar_ge_vert n _ x ltac:(apply Hnid'2)) as Hge.
         rewrite Ex in Hge.
         specialize (Hge ltac:(auto)).
         pose proof (mys_ar_le n _ ltac:(apply Hnid'2) ltac:(apply Hnid'2)).
         lia.
  Qed.

  Lemma mys_history :
    forall n r nid,
    In nid bado_participant ->
    bado_node_assump nid = Synchronous ->
    1 <= r < mys_weak_oracle_local_ar n nid ->
    (exists id, match NatMap_find id (mys_weak_oracle_dag n).(mdag_udag).(udag_verts) with
                | None => False
                | Some v => v.(udag_vert_round) = r /\ v.(udag_vert_builder) = nid
              end) \/
    In (nid, r) (mys_weak_oracle_dag n).(mdag_jumps).
  Proof. intros n r nid Hpart Hsync Hr.
         pose proof (mys_weak_oracle_ar_has_vert n _ Hpart Hsync) as Hvert.
         destruct Hvert as [? | (id & Hvert)]; [lia|].
         cond_case_auto Hvert; try contradiction.
         pose proof (mdag_vert_jump_history _ nid (mys_weak_oracle_local_ar n nid - 1) (mys_oracle_valid n) ltac:(congruence)) as Hhist.
         specialize (Hhist ltac:(left; exists id; rewrite Ex; split; apply Hvert)).
         apply (Hhist r ltac:(lia)).
  Qed.

  Lemma mys_ar_has_quorum :
    forall n r,
    1 <= r < mys_ar n ->
    exists quorum,
      is_quorum bado_comm quorum /\
      forall nid,
      In nid quorum ->
      exists id,
        match NatMap_find id (mys_weak_oracle_dag n).(mdag_udag).(udag_verts) with
        | None => False
        | Some v => v.(udag_vert_builder) = nid /\ v.(udag_vert_round) = r
        end.
  Proof. intros n.
         pose proof (mys_rt_ex n) as (nid & Hnid1 & Hnid2 & _).
         pose proof Hnid1 as Hnid3.
         rewrite sync_set_correct in Hnid3.
         pose proof (mys_weak_oracle_ar_has_quorum n _ ltac:(apply Hnid3) ltac:(apply Hnid3)) as Har.
         destruct Har as [Har | Hquorum].
         1: intros; lia.
         rewrite Hnid2 in Hquorum.
         clear nid Hnid1 Hnid2 Hnid3.
         intros r Hr.
         assert (Hr' : r = mys_ar n - 1 \/ 1 <= r < mys_ar n - 1) by lia.
         destruct Hquorum as (quorum & Hquorum1 & Hquorum2).
         destruct Hr' as [Hr' | Hr'].
         - subst r; clear Hr.
           exists quorum; split; auto.
         - pose proof (quorum_exists_honest bado_comm_safe _ Hquorum1) as (nid' & Hnid' & _).
           specialize (Hquorum2 _ Hnid').
           clear Hr quorum Hquorum1 Hnid'.
           destruct Hquorum2 as (id & Hid).
           cond_case_auto Hid; try contradiction.
           pose proof (udag_closure_quorum _ id r ltac:(apply mdag_udag_valid; apply (mys_oracle_valid n))) as Hquorum.
           rewrite Ex in Hquorum.
           specialize (Hquorum ltac:(destruct Hid; congruence)).
           clear u nid' Ex Hid.
           destruct Hquorum as (quorum & Hquorum1 & Hquorum2).
           eexists; split; [apply Hquorum2|].
           intros nid' Hnid'.
           rewrite in_map_iff in Hnid'.
           destruct Hnid' as (x & Hx1 & Hx2).
           rewrite Forall_forall in Hquorum1.
           specialize (Hquorum1 _ Hx2).
           cond_case_auto Hquorum1; try contradiction.
           exists x; rewrite Ex; auto.
  Qed.

  Lemma mys_catchup :
    forall n nid,
    In nid bado_participant ->
    bado_node_assump nid = Synchronous ->
    mys_weak_oracle_local_ar (S n) nid >= mys_ar n.
  Proof.
    intros n nid Hpart Hsync.
    pose proof (mys_rt_ex n) as (nid' & Hnid'1 & Hnid'2 & Hnid'3).
    rewrite sync_set_correct in Hnid'1.
    pose proof (mys_weak_oracle_catchup n _ _ ltac:(apply Hnid'1) ltac:(apply Hnid'1) Hpart Hsync).
    lia.
  Qed.

  Lemma mys_all_vert_supporter n r :
    (mys_ar n < S r \/ mys_ar n = S r /\ mys_rt n >= 1) ->
    forall id, match NatMap_find id (mys_weak_oracle_dag (S n)).(mdag_udag).(udag_verts) with
               | None => True
               | Some v => v.(udag_vert_round) = S r -> bado_node_assump v.(udag_vert_builder) = Synchronous -> mdag_vert_is_supporter v (mys_weak_oracle_dag (S n))
               end.
  Proof. intros Har id.
         match goal with |- match ?x with _ => _ end => destruct x eqn:Evert end; auto.
         intros Hround Hsync.
         pose proof (udag_mem _ id (mdag_udag_valid _ (mys_oracle_valid (S n)))) as Hpart.
         rewrite Evert in Hpart.
         pose proof (mdag_honest_vert_no_timeout_perfect _ id (mys_oracle_valid (S n))) as Hperfect.
         rewrite Evert in Hperfect.
         apply Hperfect; auto.
         rewrite Forall_forall.
         intros x Hx1 Hx2 Heq.
         destruct x as [nid r']; cbn in *.
         subst r'.
         pose proof (mdag_timeout_mem _ _ _ (mys_oracle_valid (S n)) Hx1) as Hx3.
         pose proof (mys_rt_pos_no_timeout' n (S r) Har) as Hno_timeout.
         rewrite Forall_forall in Hno_timeout.
         eapply Hno_timeout.
         2: rewrite Hround in Hx1; apply Hx1.
         rewrite sync_set_correct; auto.
  Qed.

  Lemma mys_all_vert_certificate n r :
    (mys_ar n < S (S r) \/ mys_ar n = S (S r) /\ mys_rt n >= 1) ->
    forall id, match NatMap_find id (mys_weak_oracle_dag (S n)).(mdag_udag).(udag_verts) with
               | None => True
               | Some v => v.(udag_vert_round) = S (S r) -> bado_node_assump v.(udag_vert_builder) = Synchronous -> mdag_vert_is_certificate v (mys_weak_oracle_dag (S n))
               end.
  Proof. intros Har id.
         match goal with |- match ?x with _ => _ end => destruct x eqn:Evert end; auto.
         intros Hround Hsync.
         pose proof (udag_mem _ id (mdag_udag_valid _ (mys_oracle_valid (S n)))) as Hpart.
         rewrite Evert in Hpart.
         pose proof (mdag_honest_vert_no_timeout_perfect _ id (mys_oracle_valid (S n))) as Hperfect.
         rewrite Evert in Hperfect.
         apply Hperfect; auto.
         rewrite Forall_forall.
         intros x Hx1 Hx2 Heq.
         destruct x as [nid r']; cbn in *.
         subst r'.
         pose proof (mdag_timeout_mem _ _ _ (mys_oracle_valid (S n)) Hx1) as Hx3.
         pose proof (mys_rt_pos_no_timeout' n (S (S r)) Har) as Hno_timeout.
         rewrite Forall_forall in Hno_timeout.
         eapply Hno_timeout.
         2: rewrite Hround in Hx1; apply Hx1.
         rewrite sync_set_correct; auto.
  Qed.

  Lemma mys_local_ar_pos : forall n nid,
    In nid bado_participant ->
    bado_node_assump nid = Synchronous ->
    mys_weak_oracle_local_ar n nid >= 1.
  Proof.
    intros n nid Hnid1 Hnid2.
    induction n.
    - pose proof mys_weak_oracle_ar_init_pos nid Hnid1 Hnid2; auto.
    - pose proof mys_weak_oracle_ar_step n nid Hnid1 Hnid2 as Hstep; cbn in Hstep.
      lia.
  Qed.

  Lemma mys_ar_pos : forall n, mys_ar n >= 1.
  Proof.
    intros n.
    pose proof (mys_ar_le n) as Hle.
    pose proof (quorum_exists_honest bado_comm_safe _ sync_set_quorum) as (nid & Hnid1 & _).
    rewrite sync_set_correct in Hnid1.
    destruct Hnid1 as (Hnid1 & Hnid2).
    specialize (Hle _ Hnid1 Hnid2).
    pose proof (mys_local_ar_pos n nid Hnid1 Hnid2).
    lia.
  Qed.

  (* We now define several "checkpoints": states that the DAG state should progress through to achieve liveness *)
  Definition mys_ckp1 n r :=
    mys_ar n = S r /\ mys_rt n >= 2.

  Definition mys_ckp2 n r :=
    mys_ar n = S r /\
    mys_rt n >= 1 /\
    exists id, match NatMap_find id (mys_weak_oracle_dag n).(mdag_udag).(udag_verts) with
               | None => False
               | Some v => v.(udag_vert_round) = r /\ v.(udag_vert_builder) = bado_leader_at r end.

  Definition mys_ckp3 n r :=
    mys_ar n = S r /\
    Forall (fun nid => In (nid, r) (mys_weak_oracle_dag n).(mdag_recv_leader_verts) \/
                       exists id, match NatMap_find id (mys_weak_oracle_dag n).(mdag_udag).(udag_verts) with
                                  | None => False
                                  | Some v => v.(udag_vert_round) = S r /\ v.(udag_vert_builder) = nid /\ mdag_vert_is_supporter v (mys_weak_oracle_dag n) end)
           sync_set.

  Definition mys_ckp4 n r :=
    mys_ar n = S (S r) /\
    mys_rt n >= 2 /\
    Forall (fun nid => In (nid, r) (mys_weak_oracle_dag n).(mdag_recv_leader_verts) \/
                       mys_weak_oracle_local_ar n nid = S r /\ mys_weak_oracle_local_rt n nid >= 1 \/
                       mys_weak_oracle_local_ar n nid < S r \/
                       exists id, match NatMap_find id (mys_weak_oracle_dag n).(mdag_udag).(udag_verts) with
                                  | None => False
                                  | Some v => v.(udag_vert_round) = S r /\ v.(udag_vert_builder) = nid /\ mdag_vert_is_supporter v (mys_weak_oracle_dag n) end)
           sync_set /\
    exists quorum, is_to_quorum quorum /\
    Forall (fun nid => bado_node_assump nid = Synchronous /\
                       exists id, match NatMap_find id (mys_weak_oracle_dag n).(mdag_udag).(udag_verts) with
                                  | None => False
                                  | Some v => v.(udag_vert_round) = S r /\ v.(udag_vert_builder) = nid /\ mdag_vert_is_supporter v (mys_weak_oracle_dag n) end)
           quorum.

  Definition mys_ckp5 n r :=
    mys_ar n = S (S r) /\
    mys_rt n >= 1 /\
    Forall (fun nid => exists id, match NatMap_find id (mys_weak_oracle_dag n).(mdag_udag).(udag_verts) with
                                  | None => False
                                  | Some v => v.(udag_vert_round) = S r /\ v.(udag_vert_builder) = nid /\ mdag_vert_is_supporter v (mys_weak_oracle_dag n) end)
           sync_set.

  Definition mys_ckp6 n r :=
    mys_ar n >= S (S (S r)) /\
    Forall (fun nid => In (nid, S (S r)) (mys_weak_oracle_dag n).(mdag_jumps) \/
                       mys_weak_oracle_local_ar n nid = S (S r) /\ mys_weak_oracle_local_rt n nid >= 1 \/
                       mys_weak_oracle_local_ar n nid < S (S r) \/
                       exists id, match NatMap_find id (mys_weak_oracle_dag n).(mdag_udag).(udag_verts) with
                                  | None => False
                                  | Some v => v.(udag_vert_round) = S (S r) /\ v.(udag_vert_builder) = nid /\ mdag_vert_is_certificate v (mys_weak_oracle_dag n) end)
           sync_set /\
    exists quorum, is_to_quorum quorum /\
    Forall (fun nid => bado_node_assump nid = Synchronous /\
                       exists id, match NatMap_find id (mys_weak_oracle_dag n).(mdag_udag).(udag_verts) with
                                  | None => False
                                  | Some v => v.(udag_vert_round) = S (S r) /\ v.(udag_vert_builder) = nid /\ mdag_vert_is_certificate v (mys_weak_oracle_dag n) end)
           quorum.

  Definition mys_ckp7 n r :=
    mys_ar n >= S (S r) /\
    Forall (fun nid => In (nid, r) (mys_weak_oracle_dag n).(mdag_recv_certs) \/
                       exists id, match NatMap_find id (mys_weak_oracle_dag n).(mdag_udag).(udag_verts) with
                                  | None => False
                                  | Some v => v.(udag_vert_round) = S (S r) /\ v.(udag_vert_builder) = nid /\ mdag_vert_is_certificate v (mys_weak_oracle_dag n) end)
           sync_set.

  Definition mys_ckp8 n r :=
    Forall (fun nid => exists id, match NatMap_find id (mys_weak_oracle_dag n).(mdag_udag).(udag_verts) with
                                  | None => False
                                  | Some v => v.(udag_vert_round) = S (S r) /\ v.(udag_vert_builder) = nid /\ mdag_vert_is_certificate v (mys_weak_oracle_dag n) end)
           sync_set.

  (* Definition of mys_ckp9 is in LivenessStrong.v, because it refers to mcommit_commit which is not defined here. *)

  (* 7 -> 8 and 6 -> 8 can only be proved under stronger assumptions.
     In this layer, we establish 5 -> 7, 4 -> 5 or 6, 3 -> 4 or 6,
     2 -> 3 or 4 or 6, 1 -> 2 or 4 or 6
   *)

  Lemma mys_live1 n r :
    r >= 1 -> bado_node_assump (bado_leader_at r) <> Byzantine ->
    mys_ckp5 n r -> mys_ckp7 (S n) r.
  Proof. intros Hpos Hleader_honest Hckp.
         unfold mys_ckp5 in Hckp.
         unfold mys_ckp7.
         split.
         1: pose proof (mys_ar_step n) as Hstep; cbn in Hstep; lia.
         destruct Hckp as (Har & Hrt & Hsupp).
         pose proof (mys_weak_oracle_recv_cert n r Hpos Hleader_honest) as Hrecv_cert.
         match type of Hrecv_cert with ?P -> _ => assert (H : P) end.
         2: specialize (Hrecv_cert H); clear H.
         1: { clear Hrecv_cert.
              intros nid Hpart Hsync.
              rewrite Forall_forall in Hsupp.
              apply Hsupp.
              rewrite sync_set_correct; auto.
         }
         clear Hsupp.
         rewrite Forall_forall.
         intros nid Hnid.
         rewrite sync_set_correct in Hnid.
         specialize (Hrecv_cert _ ltac:(apply Hnid) ltac:(apply Hnid)).
         destruct Hrecv_cert as [Hrecv_cert | Hvert].
         - left; auto.
         - right.
           destruct Hvert as (id & Hid).
           cond_case_auto Hid; try contradiction.
           exists id.
           rewrite Ex.
           repeat split; try apply Hid.
           pose proof (mys_rt_pos_no_timeout n Hrt) as Hno_timeout.
           pose proof (mdag_honest_vert_no_timeout_perfect _ id (mys_oracle_valid (S n))) as Hperfect.
           rewrite Ex in Hperfect.
           apply Hperfect; [destruct Hid; destruct Hnid; congruence|].
           clear Hperfect.
           rewrite Forall_forall.
           intros to Hto Hsync.
           destruct to; cbn in *.
           pose proof (mdag_timeout_mem _ _ _ (mys_oracle_valid _) Hto) as Hpart.
           intros Hround; subst n1.
           rewrite Forall_forall in Hno_timeout.
           eapply Hno_timeout; [rewrite sync_set_correct; split; try apply Hsync; try apply Hpart|].
           destruct Hid as (Hid & _); rewrite Har; rewrite <- Hid; auto.
  Qed.

  Lemma mys_live2 n r :
    r >= 1 ->
    mys_ckp4 n r -> mys_ckp5 (S n) r \/ mys_ckp6 (S n) r.
  Proof. intros Hpos Hckp.
         destruct Hckp as (Har & Hrt & Hverts & Hquorum).

         assert (Hsupp : forall id, match NatMap_find id (mys_weak_oracle_dag (S n)).(mdag_udag).(udag_verts) with
                                    | None => True
                                    | Some v => v.(udag_vert_round) = S r -> bado_node_assump v.(udag_vert_builder) = Synchronous -> mdag_vert_is_supporter v (mys_weak_oracle_dag (S n))
                                    end).
         { intros id.
           match goal with |- match ?x with _ => _ end => destruct x eqn:Evert end; auto.
           intros Hround Hsync.
           pose proof (udag_mem _ id (mdag_udag_valid _ (mys_oracle_valid (S n)))) as Hpart.
           rewrite Evert in Hpart.
           pose proof (mdag_honest_vert _ id (mys_oracle_valid (S n))) as Hvert.
           rewrite Evert in Hvert.
           specialize (Hvert ltac:(congruence)).
           destruct Hvert as [Hvert | [Hvert | Hvert]].
           - apply Hvert.
           - rewrite Forall_forall in Hverts.
             specialize (Hverts _ ltac:(rewrite sync_set_correct; split; try apply Hsync; try apply Hpart)).
             destruct Hverts as [Hverts | [Hverts | [Hverts | Hverts]]].
             + destruct Hvert as (_ & Hvert & _); apply Hvert.
               rewrite Hround; replace (S r - 1) with r by lia.
               apply (mdag_recv_leader_vert_mono _ _ (mys_oracle_reachable n (S n) ltac:(lia))); auto.
             + destruct Hvert as (Htimeout & _).
               rewrite Hround in Htimeout.
               pose proof (mys_weak_oracle_ar_step n _ Hpart Hsync) as Hstep; cbn in Hstep.
               pose proof (mys_sync_timeout_ge n (S r) _ Hpart Hsync) as Hge.
               pose proof (mys_weak_oracle_timeout_inv n _ (S r) Hpart Hsync Htimeout) as [Htimeout' | ?]; [|lia].
               specialize (Hge Htimeout'); lia.
             + destruct Hvert as (Htimeout & _).
               rewrite Hround in Htimeout.
               pose proof (mys_weak_oracle_ar_step n _ Hpart Hsync) as Hstep; cbn in Hstep.
               pose proof (mys_sync_timeout_ge n (S r) _ Hpart Hsync) as Hge.
               pose proof (mys_weak_oracle_timeout_inv n _ (S r) Hpart Hsync Htimeout) as [Htimeout' | ?]; [|lia].
               specialize (Hge Htimeout'); lia.
             + clear Hvert.
               destruct Hverts as (id' & Hid').
               cond_case_auto Hid'; try contradiction.
               destruct Hid' as (? & ? & Hsupp).
               pose proof (udag_mono _ _ id' (mdag_udag_reachable _ _ (mys_oracle_reachable n (S n) ltac:(lia))) ltac:(congruence)) as Hmono.
               rewrite <- Hmono in Ex.
               pose proof (udag_honest_uniq _ id id' (mdag_udag_valid _ (mys_oracle_valid (S n)))) as Huniq.
               rewrite Evert, Ex in Huniq.
               specialize (Huniq ltac:(congruence) ltac:(congruence) ltac:(congruence)); subst id'.
               rewrite Evert in Ex; inversion Ex; subst u0; clear Ex.
               eapply mdag_vert_is_supporter_reachable.
               3: apply Hsupp.
               1: apply mys_oracle_valid.
               1: apply mys_oracle_reachable; auto.
           - destruct Hquorum as (quorum & Hquorum1 & Hquorum2).
             destruct Hvert as (quorum' & Hquorum'1 & Hquorum'2 & _ & _ & Hquorum'3 & _).
             clear Hverts.
             rewrite Forall_forall in *.
             pose proof (quorum_to_quorum_overlap _ _ Hquorum'2 Hquorum1) as (x & Hx1 & Hx2).
             clear Hquorum'2.
             specialize (Hquorum2 _ Hx2) as (Hx3 & id' & Hid'); clear Hx2.
             rewrite in_map_iff in Hx1.
             destruct Hx1 as (y & Hy1 & Hy2).
             specialize (Hquorum'1 _ Hy2); specialize (Hquorum'3 _ Hy2).
             cond_case_auto Hquorum'1; try contradiction.
             cond_case_auto Hid'; try contradiction.
             pose proof (udag_mono _ _ id' (mdag_udag_reachable _ _ (mys_oracle_reachable n (S n) ltac:(lia))) ltac:(congruence)) as Hmono.
             rewrite <- Hmono in Ex0.
             destruct Hid' as (? & ? & Hsupp).
             pose proof (udag_honest_uniq _ y id' (mdag_udag_valid _ (mys_oracle_valid (S n)))) as Huniq.
             rewrite Ex, Ex0 in Huniq.
             specialize (Huniq ltac:(congruence) ltac:(congruence) ltac:(congruence)); subst id'.
             rewrite Ex0 in Ex; inversion Ex; subst u1; clear Ex.
             apply Hquorum'3.
             eapply mdag_vert_is_supporter_reachable.
             3: apply Hsupp.
             1: apply mys_oracle_valid.
             1: apply mys_oracle_reachable; auto.
         }

         pose proof (mys_all_vert_certificate n r ltac:(lia)) as Hcert.
         clear Hquorum.

         pose proof (mys_ar_step n) as Hstep.
         cbn in Hstep.
         assert (Har' : mys_ar (S n) = S (S r) \/ mys_ar (S n) >= S (S (S r))) by lia.
         destruct Har' as [Har' | Har'].
         - left; unfold mys_ckp5.
           repeat split; try lia.
           clear Hstep.
           rewrite Forall_forall.
           intros nid Hnid1.
           pose proof Hnid1 as Hnid2.
           rewrite sync_set_correct in Hnid2.
           pose proof (mys_ar_le (S n) _ ltac:(apply Hnid2) ltac:(apply Hnid2)) as Hnid3.
           pose proof (mys_catchup n _ ltac:(apply Hnid2) ltac:(apply Hnid2)) as Hnid4.
           assert (Hnid5 : mys_weak_oracle_local_ar (S n) nid = S (S r)) by lia.
           clear Hnid3 Hnid4.
           pose proof (mys_weak_oracle_ar_has_vert (S n) _ ltac:(apply Hnid2) ltac:(apply Hnid2)) as [? | Hnid3]; [lia|].
           destruct Hnid3 as (id & Hid).
           exists id.
           cond_case_auto Hid; try contradiction.
           repeat split; try lia.
           specialize (Hsupp id).
           rewrite Ex in Hsupp.
           apply (Hsupp ltac:(lia) ltac:(destruct Hnid2; destruct Hid; congruence)).
         - right; unfold mys_ckp6.
           repeat split; try lia.
           + clear Hstep.
             rewrite Forall_forall.
             intros nid Hnid1.
             pose proof Hnid1 as Hnid2.
             rewrite sync_set_correct in Hnid2.
             pose proof (mys_catchup n _ ltac:(apply Hnid2) ltac:(apply Hnid2)) as Hnid3.
             assert (Hnid4 : mys_weak_oracle_local_ar (S n) nid = S (S r) \/ mys_weak_oracle_local_ar (S n) nid >= S (S (S r))) by lia.
             clear Hnid3.
             destruct Hnid4 as [Hnid4 | Hnid4].
             * right; left; split; auto.
               pose proof (mys_rt_ge n _ ltac:(apply Hnid2) ltac:(apply Hnid2)).
               pose proof (mys_weak_oracle_ar_step n _ ltac:(apply Hnid2) ltac:(apply Hnid2)) as Hstep.
               cbn in Hstep.
               lia.
             * pose proof (mys_history (S n) (S (S r)) _ ltac:(apply Hnid2) ltac:(apply Hnid2) ltac:(lia)) as Hhist.
               destruct Hhist as [Hhist | Hhist]; [|left; auto].
               right; right; right.
               destruct Hhist as (id & Hvert).
               exists id.
               cond_case_auto Hvert; try contradiction.
               repeat split; try apply Hvert.
               specialize (Hcert id).
               rewrite Ex in Hcert.
               apply (Hcert ltac:(apply Hvert) ltac:(destruct Hvert; destruct Hnid2; congruence)).
           + clear Hstep.
             pose proof (mys_ar_has_quorum (S n) (S (S r)) ltac:(lia)) as (quorum & Hquorum1 & Hquorum2).
             pose proof (quorum_exists_sync_to_quorum _ Hquorum1) as (toq & Htoq1 & Htoq2 & Htoq3).
             exists toq.
             split; auto.
             rewrite Forall_forall.
             intros x Hx1.
             specialize (Htoq3 _ Hx1) as (Hx2 & Hx3).
             split; auto.
             specialize (Htoq2 _ Hx1) as Hx4.
             specialize (Hquorum2 _ Hx4) as (id & Hid).
             cond_case_auto Hid; try contradiction.
             exists id; rewrite Ex.
             destruct Hid; repeat split; auto.
             specialize (Hcert id); rewrite Ex in Hcert.
             apply (Hcert ltac:(auto) ltac:(congruence)).
  Qed.

  Lemma mys_live3 n r :
    r >= 1 ->
    mys_ckp3 n r -> exists n', mys_ckp4 n' r \/ mys_ckp6 n' r.
  Proof. intros Hpos Hckp.
         unfold mys_ckp3 in Hckp.
         destruct Hckp as (Har & Hsupp).
         pose proof (mys_ar_mono 0 n ltac:(lia)) as Hge.
         pose proof (mys_progress (S (S r)) ltac:(lia)) as (n'' & Hn'').
         
         pose proof (transition_point mys_ar n'' (S (S r)) ltac:(lia)) as Henter.
         destruct Henter as [Henter | Henter].
         1: specialize (Henter n'' ltac:(auto)); lia.
         destruct Henter as (n' & _ & Har'' & Har').
         exists (S n'); clear Hge n'' Hn''.
         assert (Hgt : S n' > n) by (pose proof (mys_ar_mono (S n') n); lia).
         assert (Har''' : mys_ar n' = S r) by (pose proof (mys_ar_mono n n' ltac:(lia)); lia).
         clear Har''.

         assert (Hsupp' : forall id, match NatMap_find id (mys_weak_oracle_dag (S n')).(mdag_udag).(udag_verts) with
                                     | None => True
                                     | Some v => v.(udag_vert_round) = S r -> bado_node_assump v.(udag_vert_builder) = Synchronous -> mdag_vert_is_supporter v (mys_weak_oracle_dag (S n'))
                                    end).
         { intros id.
           match goal with |- match ?x with _ => _ end => destruct x eqn:Evert end; auto.
           intros Hround Hsync.
           pose proof (udag_mem _ id (mdag_udag_valid _ (mys_oracle_valid (S n')))) as Hpart.
           rewrite Evert in Hpart.
           rewrite Forall_forall in Hsupp.
           specialize (Hsupp _ ltac:(rewrite sync_set_correct; split; try apply Hpart; try apply Hsync)).
           destruct Hsupp as [Hsupp | Hsupp].
           - pose proof (mdag_recv_leader_vert_mono _ _ (mys_oracle_reachable n (S n') ltac:(lia)) _ Hsupp) as Hsupp'.
             pose proof (mdag_honest_recv_leader_vert_supporter _ id (mys_oracle_valid (S n'))) as Hvert.
             rewrite Evert in Hvert.
             apply Hvert.
             1: congruence.
             rewrite Hround.
             replace (S r - 1) with r by lia.
             auto.
           - destruct Hsupp as (id' & Hid').
             cond_case_auto Hid'; try contradiction.
             pose proof (udag_mono _ _ id' (mdag_udag_reachable _ _ (mys_oracle_reachable n (S n') ltac:(lia))) ltac:(congruence)) as Hmono.
             rewrite <- Hmono in Ex.
             destruct Hid' as (? & ? & Hsupp).
             pose proof (udag_honest_uniq _ id id' (mdag_udag_valid _ (mys_oracle_valid (S n')))) as Huniq.
             rewrite Evert, Ex in Huniq.
             specialize (Huniq ltac:(congruence) ltac:(congruence) ltac:(congruence)); subst id'.
             rewrite Evert in Ex; inversion Ex; subst u0; clear Ex.
             eapply mdag_vert_is_supporter_reachable.
             3: apply Hsupp.
             1: apply mys_oracle_valid.
             1: apply mys_oracle_reachable; lia.
         }

         pose proof (mys_all_vert_certificate n' r ltac:(lia)) as Hcert'.

         assert (Har'' : mys_ar (S n') = S (S r) \/ mys_ar (S n') >= S (S (S r))) by lia; clear Har'.
         destruct Har'' as [Har' | Har'].
         - left; unfold mys_ckp4.
           repeat split; auto.
           1: pose proof (mys_ar_step n') as Hstep; cbn in Hstep; lia; clear Hstep.
           + rewrite Forall_forall.
             intros nid Hnid1.
             pose proof Hnid1 as Hnid2.
             rewrite sync_set_correct in Hnid2.
             rewrite Forall_forall in Hsupp.
             specialize (Hsupp _ Hnid1) as [Hnid3 | Hnid3].
             * left.
               apply (mdag_recv_leader_vert_mono _ _ ltac:(apply (mys_oracle_reachable n (S n')); lia) _ Hnid3).
             * right; right; right.
               destruct Hnid3 as (id & Hid).
               cond_case_auto Hid; try contradiction.
               exists id.
               pose proof (udag_mono _ _ id ltac:(apply mdag_udag_reachable; apply (mys_oracle_reachable n (S n') ltac:(lia))) ltac:(congruence)) as Hmono.
               rewrite Hmono.
               rewrite Ex; repeat split; try apply Hid.
               eapply mdag_vert_is_supporter_reachable.
               3: apply Hid.
               1: apply mys_oracle_valid.
               1: apply mys_oracle_reachable; lia.
           + clear Hsupp.
             pose proof (mys_ar_has_quorum (S n') (S r) ltac:(lia)) as (quorum & Hquorum1 & Hquorum2).
             pose proof (quorum_exists_sync_to_quorum _ Hquorum1) as (toq & Htoq1 & Htoq2 & Htoq3).
             exists toq; split; auto.
             rewrite Forall_forall.
             intros x Hx1.
             specialize (Htoq3 _ Hx1) as (Hx2 & Hx3).
             split; auto.
             specialize (Htoq2 _ Hx1) as Hx4.
             specialize (Hquorum2 _ Hx4) as (id & Hid).
             exists id.
             cond_case_auto Hid; try contradiction.
             repeat split; try apply Hid.
             specialize (Hsupp' id).
             rewrite Ex in Hsupp'.
             apply (Hsupp' ltac:(destruct Hid; congruence) ltac:(destruct Hid; congruence)).
         - right; unfold mys_ckp6.
           repeat split; auto.
           + rewrite Forall_forall.
             intros x Hx1.
             pose proof Hx1 as Hx2.
             rewrite sync_set_correct in Hx2.
             pose proof (mys_ar_le n' _ ltac:(apply Hx2) ltac:(apply Hx2)) as Hle.
             pose proof (mys_weak_oracle_ar_step n' _ ltac:(apply Hx2) ltac:(apply Hx2)) as Hstep.
             cbn in Hstep.
             assert (Har'' : let ar := mys_weak_oracle_local_ar (S n') x in ar < S (S r) \/ ar = S (S r) \/ ar >= S (S (S r))) by lia.
             cbn in Har''.
             destruct Har'' as [Har'' | [Har'' | Har'']].
             * right; right; left; auto.
             * right; left; split; auto; lia.
             * clear Hle Hstep.
               pose proof (mys_history (S n') (S (S r)) _ ltac:(apply Hx2) ltac:(apply Hx2) ltac:(lia)) as [Hhist | Hhist].
               -- right; right; right.
                  destruct Hhist as (id & Hid).
                  exists id; cond_case_auto Hid; try contradiction.
                  repeat split; try apply Hid.
                  specialize (Hcert' id); rewrite Ex in Hcert'.
                  apply (Hcert' ltac:(apply Hid) ltac:(destruct Hid; destruct Hx2; congruence)).
               -- left; auto.
           + pose proof (mys_ar_has_quorum (S n') (S (S r)) ltac:(lia)) as (quorum & Hquorum1 & Hquorum2).
             pose proof (quorum_exists_sync_to_quorum _ Hquorum1) as (toq & Htoq1 & Htoq2 & Htoq3).
             exists toq.
             split; auto.
             rewrite Forall_forall.
             intros x Hx1.
             specialize (Htoq3 _ Hx1) as (Hx2 & Hx3).
             split; auto.
             specialize (Htoq2 _ Hx1) as Hx4.
             specialize (Hquorum2 _ Hx4) as (id & Hid).
             cond_case_auto Hid; try contradiction.
             exists id; rewrite Ex.
             destruct Hid; repeat split; auto.
             specialize (Hcert' id); rewrite Ex in Hcert'.
             apply (Hcert' ltac:(auto) ltac:(congruence)).
  Qed.

  Lemma mys_live4 n r :
    r >= 1 -> bado_node_assump (bado_leader_at r) = Synchronous ->
    mys_ckp2 n r -> mys_ckp3 (S n) r \/ mys_ckp4 (S n) r \/ mys_ckp6 (S n) r.
  Proof. intros Hpos Hleader_sync Hckp.
         destruct Hckp as (Har & Hrt & id & Hvert).
         cond_case_auto Hvert; try contradiction.

         pose proof (mys_all_vert_supporter n r ltac:(lia)) as Hsupp.
         pose proof (mys_all_vert_certificate n r ltac:(lia)) as Hcert.

         pose proof (mys_ar_step n) as Hstep; cbn in Hstep.
         assert (Har' : mys_ar (S n) = S r \/ mys_ar (S n) = S (S r) \/ mys_ar (S n) >= S (S (S r))) by lia.
         destruct Har' as [Har' | [Har' | Har']].
         - left; unfold mys_ckp3; split; auto; clear Hstep.
           rewrite Forall_forall.
           intros x Hx1.
           pose proof Hx1 as Hx2.
           rewrite sync_set_correct in Hx2.
           pose proof (mys_weak_oracle_recv_leader_vert n r ltac:(auto) ltac:(exists id; rewrite Ex; auto) _ ltac:(apply Hx2) ltac:(apply Hx2)) as Hrecv.
           destruct Hrecv as [Hrecv | Hrecv].
           + left; auto.
           + right; destruct Hrecv as (id' & Hid').
             exists id'; cond_case_auto Hid'; try contradiction.
             repeat split; try apply Hid'.
             specialize (Hsupp id'); rewrite Ex0 in Hsupp.
             apply (Hsupp ltac:(apply Hid') ltac:(destruct Hid'; destruct Hx2; congruence)).
         - right; left; unfold mys_ckp4; repeat split; auto; try lia; clear Hstep.
           + rewrite Forall_forall.
             intros x Hx1.
             pose proof Hx1 as Hx2.
             rewrite sync_set_correct in Hx2.
             pose proof (mys_weak_oracle_recv_leader_vert n r ltac:(auto) ltac:(exists id; rewrite Ex; auto) _ ltac:(apply Hx2) ltac:(apply Hx2)) as Hrecv.
             destruct Hrecv as [Hrecv | Hrecv].
             * left; auto.
             * right; right; right.
               destruct Hrecv as (id' & Hid').
               exists id'; cond_case_auto Hid'; try contradiction.
               repeat split; try apply Hid'.
               specialize (Hsupp id'); rewrite Ex0 in Hsupp.
               apply (Hsupp ltac:(apply Hid') ltac:(destruct Hid'; destruct Hx2; congruence)).
           + pose proof (mys_ar_has_quorum (S n) (S r) ltac:(lia)) as (quorum & Hquorum1 & Hquorum2).
             pose proof (quorum_exists_sync_to_quorum _ Hquorum1) as (toq & Htoq1 & Htoq2 & Htoq3).
             exists toq; split; auto.
             rewrite Forall_forall.
             intros x Hx1.
             specialize (Htoq3 _ Hx1) as (Hx2 & Hx3).
             split; auto.
             specialize (Htoq2 _ Hx1) as Hx4.
             specialize (Hquorum2 _ Hx4) as (id' & Hid').
             exists id'.
             cond_case_auto Hid'; try contradiction.
             repeat split; try apply Hid'.
             specialize (Hsupp id').
             rewrite Ex0 in Hsupp.
             apply (Hsupp ltac:(destruct Hid'; congruence) ltac:(destruct Hid'; congruence)).
         - right; right; unfold mys_ckp6; repeat split; auto; clear Hstep.
           + rewrite Forall_forall.
             intros x Hx1.
             pose proof Hx1 as Hx2.
             rewrite sync_set_correct in Hx2.
             pose proof (mys_ar_le n _ ltac:(apply Hx2) ltac:(apply Hx2)) as Hle.
             pose proof (mys_weak_oracle_ar_step n _ ltac:(apply Hx2) ltac:(apply Hx2)) as Hstep.
             cbn in Hstep.
             assert (Har'' : let ar := mys_weak_oracle_local_ar (S n) x in ar < S (S r) \/ ar = S (S r) \/ ar >= S (S (S r))) by lia.
             cbn in Har''.
             destruct Har'' as [Har'' | [Har'' | Har'']].
             * right; right; left; auto.
             * right; left; split; auto; lia.
             * clear Hle Hstep.
               pose proof (mys_history (S n) (S (S r)) _ ltac:(apply Hx2) ltac:(apply Hx2) ltac:(lia)) as [Hhist | Hhist].
               -- right; right; right.
                  destruct Hhist as (id' & Hid').
                  exists id'; cond_case_auto Hid'; try contradiction.
                  repeat split; try apply Hid'.
                  specialize (Hcert id'); rewrite Ex0 in Hcert.
                  apply (Hcert ltac:(apply Hid') ltac:(destruct Hid'; destruct Hx2; congruence)).
               -- left; auto.
           + pose proof (mys_ar_has_quorum (S n) (S (S r)) ltac:(lia)) as (quorum & Hquorum1 & Hquorum2).
             pose proof (quorum_exists_sync_to_quorum _ Hquorum1) as (toq & Htoq1 & Htoq2 & Htoq3).
             exists toq.
             split; auto.
             rewrite Forall_forall.
             intros x Hx1.
             specialize (Htoq3 _ Hx1) as (Hx2 & Hx3).
             split; auto.
             specialize (Htoq2 _ Hx1) as Hx4.
             specialize (Hquorum2 _ Hx4) as (id' & Hid').
             cond_case_auto Hid'; try contradiction.
             exists id'; rewrite Ex0.
             destruct Hid'; repeat split; auto.
             specialize (Hcert id'); rewrite Ex0 in Hcert.
             apply (Hcert ltac:(auto) ltac:(congruence)).
  Qed.

  Lemma mys_live5 n r :
    r >= 1 -> bado_node_assump (bado_leader_at r) = Synchronous ->
    mys_ckp1 n r -> mys_ckp2 (S n) r \/ mys_ckp4 (S n) r \/ mys_ckp6 (S n) r.
  Proof. intros Hpos Hleader_sync Hckp.
         destruct Hckp as (Har & Hrt).

         pose proof (mys_all_vert_supporter n r ltac:(lia)) as Hsupp.
         pose proof (mys_all_vert_certificate n r ltac:(lia)) as Hcert.

         pose proof (mys_ar_step n) as Hstep; cbn in Hstep.
         assert (Har' : mys_ar (S n) = S r \/ mys_ar (S n) = S (S r) \/ mys_ar (S n) >= S (S (S r))) by lia.
         destruct Har' as [Har' | [Har' | Har']].
         - left; unfold mys_ckp2; repeat split; auto; try lia.
           pose proof (mys_ar_le (S n) _ (bado_leader_valid r) Hleader_sync) as Hle.
           pose proof (mys_catchup n _ (bado_leader_valid r) Hleader_sync) as Hge.
           assert (Henter : mys_weak_oracle_local_ar (S n) (bado_leader_at r) = S r) by lia.
           clear Hle Hge.
           pose proof (mys_weak_oracle_ar_has_vert (S n) _ (bado_leader_valid r) Hleader_sync) as [? | Hvert]; [lia|].
           rewrite Henter in Hvert; replace (S r - 1) with r in Hvert by lia.
           destruct Hvert as (id & Hvert); exists id; cond_case_auto Hvert; try contradiction.
           split; apply Hvert.
         - right; left; unfold mys_ckp4; repeat split; auto; try lia; clear Hstep.
           + rewrite Forall_forall.
             intros x Hx1.
             pose proof Hx1 as Hx2.
             rewrite sync_set_correct in Hx2.
             pose proof (mys_ar_le (S n) x ltac:(apply Hx2) ltac:(apply Hx2)) as Hle.
             pose proof (mys_catchup n _ ltac:(apply Hx2) ltac:(apply Hx2)) as Hge.
             assert (Har'' : let ar := mys_weak_oracle_local_ar (S n) x in ar = S r \/ ar = S (S r)) by lia; cbn in Har''.
             destruct Har'' as [Har'' | Har''].
             * right; left; split; auto.
               pose proof (mys_weak_oracle_ar_step n _ ltac:(apply Hx2) ltac:(apply Hx2)) as Hstep.
               cbn in Hstep.
               pose proof (mys_rt_ge n _ ltac:(apply Hx2) ltac:(apply Hx2)).
               lia.
             * right; right; right.
               pose proof (mys_weak_oracle_ar_has_vert (S n) _ ltac:(apply Hx2) ltac:(apply Hx2)) as [? | Hvert]; [lia|].
               destruct Hvert as (id & Hvert); exists id; cond_case_auto Hvert; try contradiction.
               rewrite Har'' in Hvert.
               cbn in Hvert.
               repeat split; try apply Hvert.
               specialize (Hsupp id).
               rewrite Ex in Hsupp.
               apply (Hsupp ltac:(apply Hvert) ltac:(destruct Hx2; destruct Hvert; congruence)).
           + pose proof (mys_ar_has_quorum (S n) (S r) ltac:(lia)) as (quorum & Hquorum1 & Hquorum2).
             pose proof (quorum_exists_sync_to_quorum _ Hquorum1) as (toq & Htoq1 & Htoq2 & Htoq3).
             exists toq; split; auto.
             rewrite Forall_forall.
             intros x Hx1.
             specialize (Htoq3 _ Hx1) as (Hx2 & Hx3).
             split; auto.
             specialize (Htoq2 _ Hx1) as Hx4.
             specialize (Hquorum2 _ Hx4) as (id & Hid).
             exists id.
             cond_case_auto Hid; try contradiction.
             repeat split; try apply Hid.
             specialize (Hsupp id).
             rewrite Ex in Hsupp.
             apply (Hsupp ltac:(destruct Hid; congruence) ltac:(destruct Hid; congruence)).
         - right; right; unfold mys_ckp6; repeat split; auto; clear Hstep.
           + rewrite Forall_forall.
             intros x Hx1.
             pose proof Hx1 as Hx2.
             rewrite sync_set_correct in Hx2.
             pose proof (mys_ar_le n _ ltac:(apply Hx2) ltac:(apply Hx2)) as Hle.
             pose proof (mys_weak_oracle_ar_step n _ ltac:(apply Hx2) ltac:(apply Hx2)) as Hstep.
             cbn in Hstep.
             assert (Har'' : let ar := mys_weak_oracle_local_ar (S n) x in ar < S (S r) \/ ar = S (S r) \/ ar >= S (S (S r))) by lia.
             cbn in Har''.
             destruct Har'' as [Har'' | [Har'' | Har'']].
             * right; right; left; auto.
             * right; left; split; auto; lia.
             * clear Hle Hstep.
               pose proof (mys_history (S n) (S (S r)) _ ltac:(apply Hx2) ltac:(apply Hx2) ltac:(lia)) as [Hhist | Hhist].
               -- right; right; right.
                  destruct Hhist as (id & Hid).
                  exists id; cond_case_auto Hid; try contradiction.
                  repeat split; try apply Hid.
                  specialize (Hcert id); rewrite Ex in Hcert.
                  apply (Hcert ltac:(apply Hid) ltac:(destruct Hid; destruct Hx2; congruence)).
               -- left; auto.
           + pose proof (mys_ar_has_quorum (S n) (S (S r)) ltac:(lia)) as (quorum & Hquorum1 & Hquorum2).
             pose proof (quorum_exists_sync_to_quorum _ Hquorum1) as (toq & Htoq1 & Htoq2 & Htoq3).
             exists toq.
             split; auto.
             rewrite Forall_forall.
             intros x Hx1.
             specialize (Htoq3 _ Hx1) as (Hx2 & Hx3).
             split; auto.
             specialize (Htoq2 _ Hx1) as Hx4.
             specialize (Hquorum2 _ Hx4) as (id & Hid).
             cond_case_auto Hid; try contradiction.
             exists id; rewrite Ex.
             destruct Hid; repeat split; auto.
             specialize (Hcert id); rewrite Ex in Hcert.
             apply (Hcert ltac:(auto) ltac:(congruence)).
  Qed.

  Lemma mys_live6 r :
    r >= mys_ar 0 ->
    exists n, mys_ckp1 n r \/ mys_ckp4 n r \/ mys_ckp6 n r.
  Proof. intros Hr.
         pose proof (mys_progress (S r) ltac:(lia)) as (n' & Hn').
         pose proof (transition_point mys_ar n' (S r) ltac:(lia)) as Henter.
         destruct Henter as [Henter | Henter].
         1: specialize (Henter n' ltac:(auto)); lia.
         destruct Henter as (n & _ & Hn1 & Hn2).
         clear n' Hn'; exists (S n).
         pose proof (mys_ar_step n) as Hstep; cbn in Hstep.
         destruct Hstep as [? | Hstep]; [lia|].

         pose proof (mys_all_vert_supporter n r ltac:(lia)) as Hsupp.
         pose proof (mys_all_vert_certificate n r ltac:(lia)) as Hcert.

         assert (Har : mys_ar (S n) = S r \/ mys_ar (S n) = S (S r) \/ mys_ar (S n) >= S (S (S r))) by lia.
         destruct Har as [Har | [Har | Har]].
         - left; split; lia.
         - right; left; unfold mys_ckp4; repeat split; auto; try lia; clear Hstep.
           + rewrite Forall_forall.
             intros x Hx1.
             pose proof Hx1 as Hx2.
             rewrite sync_set_correct in Hx2.
             pose proof (mys_ar_le (S n) x ltac:(apply Hx2) ltac:(apply Hx2)) as Hle.
             pose proof (mys_ar_le n x ltac:(apply Hx2) ltac:(apply Hx2)) as Hle'.
             assert (Har' : let ar := mys_weak_oracle_local_ar (S n) x in ar < S r \/ ar = S r \/ ar = S (S r)) by lia; cbn in Har'.
             destruct Har' as [Har' | [Har' | Har']].
             * right; right; left; auto.
             * right; left; split; auto.
               pose proof (mys_weak_oracle_ar_step n _ ltac:(apply Hx2) ltac:(apply Hx2)) as Hstep.
               cbn in Hstep.
               lia.
             * right; right; right.
               pose proof (mys_weak_oracle_ar_has_vert (S n) _ ltac:(apply Hx2) ltac:(apply Hx2)) as [? | Hvert]; [lia|].
               destruct Hvert as (id & Hvert); exists id; cond_case_auto Hvert; try contradiction.
               rewrite Har' in Hvert.
               cbn in Hvert.
               repeat split; try apply Hvert.
               specialize (Hsupp id).
               rewrite Ex in Hsupp.
               apply (Hsupp ltac:(apply Hvert) ltac:(destruct Hx2; destruct Hvert; congruence)).
           + pose proof (mys_ar_has_quorum (S n) (S r) ltac:(lia)) as (quorum & Hquorum1 & Hquorum2).
             pose proof (quorum_exists_sync_to_quorum _ Hquorum1) as (toq & Htoq1 & Htoq2 & Htoq3).
             exists toq; split; auto.
             rewrite Forall_forall.
             intros x Hx1.
             specialize (Htoq3 _ Hx1) as (Hx2 & Hx3).
             split; auto.
             specialize (Htoq2 _ Hx1) as Hx4.
             specialize (Hquorum2 _ Hx4) as (id & Hid).
             exists id.
             cond_case_auto Hid; try contradiction.
             repeat split; try apply Hid.
             specialize (Hsupp id).
             rewrite Ex in Hsupp.
             apply (Hsupp ltac:(destruct Hid; congruence) ltac:(destruct Hid; congruence)).
         - right; right; unfold mys_ckp6; repeat split; auto; clear Hstep.
           + rewrite Forall_forall.
             intros x Hx1.
             pose proof Hx1 as Hx2.
             rewrite sync_set_correct in Hx2.
             pose proof (mys_ar_le n _ ltac:(apply Hx2) ltac:(apply Hx2)) as Hle.
             pose proof (mys_weak_oracle_ar_step n _ ltac:(apply Hx2) ltac:(apply Hx2)) as Hstep.
             cbn in Hstep.
             assert (Har'' : let ar := mys_weak_oracle_local_ar (S n) x in ar < S (S r) \/ ar = S (S r) \/ ar >= S (S (S r))) by lia.
             cbn in Har''.
             destruct Har'' as [Har'' | [Har'' | Har'']].
             * right; right; left; auto.
             * right; left; split; auto; lia.
             * clear Hle Hstep.
               pose proof (mys_history (S n) (S (S r)) _ ltac:(apply Hx2) ltac:(apply Hx2) ltac:(lia)) as [Hhist | Hhist].
               -- right; right; right.
                  destruct Hhist as (id & Hid).
                  exists id; cond_case_auto Hid; try contradiction.
                  repeat split; try apply Hid.
                  specialize (Hcert id); rewrite Ex in Hcert.
                  apply (Hcert ltac:(apply Hid) ltac:(destruct Hid; destruct Hx2; congruence)).
               -- left; auto.
           + pose proof (mys_ar_has_quorum (S n) (S (S r)) ltac:(lia)) as (quorum & Hquorum1 & Hquorum2).
             pose proof (quorum_exists_sync_to_quorum _ Hquorum1) as (toq & Htoq1 & Htoq2 & Htoq3).
             exists toq.
             split; auto.
             rewrite Forall_forall.
             intros x Hx1.
             specialize (Htoq3 _ Hx1) as (Hx2 & Hx3).
             split; auto.
             specialize (Htoq2 _ Hx1) as Hx4.
             specialize (Hquorum2 _ Hx4) as (id & Hid).
             cond_case_auto Hid; try contradiction.
             exists id; rewrite Ex.
             destruct Hid; repeat split; auto.
             specialize (Hcert id); rewrite Ex in Hcert.
             apply (Hcert ltac:(auto) ltac:(congruence)).
  Qed.

  Lemma mys_liveness_weak' r :
    r >= mys_ar 0 ->
    bado_node_assump (bado_leader_at r) = Synchronous ->
    exists n, mys_ckp6 n r \/ mys_ckp7 n r.
  Proof.
    intros Hr Hsync.
    pose proof (mys_ar_pos 0).

    pose proof (mys_live6 r Hr) as (n1 & Hlive1).
    rewrite <- or_assoc in Hlive1.
    destruct Hlive1 as [Hlive1 | Hlive1].
    2: exists n1; auto.

    assert (Hlive2: exists n2, (mys_ckp2 n2 r \/ mys_ckp4 n2 r) \/ mys_ckp6 n2 r).
    { destruct Hlive1 as [Hlive1 | Hlive1].
      2: exists n1; auto.
      pose proof (mys_live5 n1 r) as Hlive2.
      specialize (Hlive2 ltac:(lia) Hsync Hlive1).
      exists (S n1); rewrite or_assoc; auto.
    }
    destruct Hlive2 as (n2 & [Hlive2 | Hlive2]).
    2: exists n2; auto.

    assert (Hlive3: exists n3, (mys_ckp3 n3 r \/ mys_ckp4 n3 r) \/ mys_ckp6 n3 r).
    { destruct Hlive2 as [Hlive2 | Hlive2].
      2: exists n2; auto.
      pose proof (mys_live4 n2 r ltac:(lia) Hsync Hlive2).
      exists (S n2); rewrite or_assoc; auto.
    }
    destruct Hlive3 as (n3 & [Hlive3 | Hlive3]).
    2: exists n3; auto.

    assert (Hlive4 : exists n4, mys_ckp4 n4 r \/ mys_ckp6 n4 r).
    { destruct Hlive3 as [Hlive3 | Hlive3].
      2: exists n3; auto.
      pose proof (mys_live3 n3 r ltac:(lia) Hlive3) as (n4 & Hlive4).
      exists n4; auto.
    }
    destruct Hlive4 as (n4 & [Hlive4 | Hlive4]).
    2: exists n4; auto.

    pose proof (mys_live2 n4 r ltac:(lia) Hlive4) as [Hlive5 | Hlive5].
    2: exists (S n4); auto.

    pose proof (mys_live1 (S n4) r ltac:(lia) ltac:(rewrite Hsync; discriminate) Hlive5) as Hlive6.
    exists (S (S n4)); auto.
  Qed.

  Lemma mys_liveness_weak r :
    r >= mys_ar 0 ->
    bado_node_assump (bado_leader_at r) = Synchronous ->
    exists n quorum,
    is_to_quorum quorum /\
    Forall (fun nid => bado_node_assump nid = Synchronous /\
                       exists id, match NatMap_find id (mys_weak_oracle_dag n).(mdag_udag).(udag_verts) with
                                  | None => False
                                  | Some v => v.(udag_vert_round) = S (S r) /\ v.(udag_vert_builder) = nid /\ mdag_vert_is_certificate v (mys_weak_oracle_dag n) end)
           quorum.
  Proof.
    intros Hr1 Hr2.
    pose proof (mys_liveness_weak' r Hr1 Hr2) as (n & [Hn | Hn]).
    - unfold mys_ckp6 in Hn.
      exists n; apply Hn.
    - unfold mys_ckp7 in Hn.
      pose proof (mys_progress' n).
      pose proof (mys_ar_has_quorum (n + 6) (S (S r)) ltac:(lia)) as (quorum & Hquorum1 & Hquorum2).
      pose proof (quorum_exists_sync_to_quorum _ Hquorum1) as (toq & Htoq1 & Htoq2 & Htoq3).
      exists (n + 6)%nat; exists toq; split; auto.
      rewrite Forall_forall.
      intros nid Hnid1.
      apply Htoq3 in Hnid1 as Hnid2.
      destruct Hnid2 as (Hnid2 & Hnid3).
      pose proof (sync_set_correct nid) as (_ & Hnid4).
      specialize (Hnid4 ltac:(split; auto)).
      destruct Hn as (_ & Hn).
      rewrite Forall_forall in Hn.
      specialize (Hn _ Hnid4).
      split; auto.
      destruct Hn as [Hn | Hn].
      + specialize (Hquorum2 _ ltac:(eapply Htoq2; apply Hnid1)).
        destruct Hquorum2 as (id & Hid).
        exists id.
        cond_case_auto Hid; auto.
        repeat split; try apply Hid.
        pose proof (mys_oracle_reachable n (n + 6) ltac:(lia)) as Hreach.
        pose proof (mdag_recv_cert_mono _ _ Hreach _ Hn) as Hcert.
        pose proof (mdag_honest_recv_leader_vert_certificate _ id (mys_oracle_valid (n + 6))) as Hcert'.
        rewrite Ex in Hcert'.
        destruct Hid as (Hid1 & Hid2).
        rewrite Hid1, Hid2 in Hcert'.
        specialize (Hcert' ltac:(rewrite Hnid3; discriminate)).
        replace (S (S r) - 2) with r in Hcert' by lia.
        specialize (Hcert' Hcert); auto.
      + clear quorum Hquorum1 Hquorum2 toq Htoq1 Htoq2 Htoq3 Hnid1.
        destruct Hn as (id & Hid).
        exists id.
        cond_case_auto Hid.
        2: contradiction.
        destruct Hid as (Hid1 & Hid2 & Hid3).
        pose proof (mys_oracle_reachable n (n + 6) ltac:(lia)) as Hreach1.
        pose proof (mdag_udag_reachable _ _ Hreach1) as Hreach2.
        pose proof (mdag_udag_valid _ (mys_oracle_valid n)) as Hval.
        pose proof (udag_reachable_subdag _ _ Hval Hreach2) as Hsub.
        unfold udag_subdag in Hsub.
        destruct Hsub as (_ & _ & Hsub).
        specialize (Hsub id); rewrite Ex in Hsub.
        cond_case_auto Hsub; try discriminate; subst u0.
        repeat split; auto.
        apply (mdag_vert_is_certificate_reachable _ _ u (mys_oracle_valid n) Hreach1 Hid3).
  Qed.

  (* Lemma 6 in the paper: If every process is synchronous, then every vertex is a certificate *)

  Lemma every_vert_certificate :
  (forall nid, In nid bado_participant -> bado_node_assump nid = Synchronous) ->
  forall n id, match NatMap_find id (mys_weak_oracle_dag n).(mdag_udag).(udag_verts) with
               | None => True
               | Some v => v.(udag_vert_round) >= mys_ar 0 + 2 -> mdag_vert_is_certificate v (mys_weak_oracle_dag n)
               end.
  Proof.
    intros Hall_sync n id.
    destruct (NatMap_find id (udag_verts (mdag_udag (mys_weak_oracle_dag n)))) eqn:Evert; auto.
    intros Hvert1.
    assert (Hvert2 : In (udag_vert_builder u) bado_participant).
    { pose proof (mdag_udag_valid _ (mys_oracle_valid n)) as Hval.
      pose proof (udag_mem _ id Hval) as Hmem; rewrite Evert in Hmem.
      auto.
    }
    specialize (Hall_sync _ Hvert2) as Hvert3.
    pose proof (udag_pos _ id (mdag_udag_valid _ (mys_oracle_valid n))) as Hvert4.
    rewrite Evert in Hvert4.
    pose proof (mys_liveness_weak' (udag_vert_round u - 2) ltac:(lia) ltac:(apply Hall_sync; apply bado_leader_valid)) as (n' & Hn').
    remember (Nat.max n n') as n''.
    assert (Hn''1 : n'' >= n) by lia.
    assert (Hn''2 : n'' >= n') by lia.
    clear Heqn''.
    pose proof (mdag_udag_reachable _ _ (mys_oracle_reachable _ _ Hn''1)) as Hreach1.
    pose proof (mdag_udag_reachable _ _ (mys_oracle_reachable _ _ Hn''2)) as Hreach2.

    destruct Hn' as [Hn' | Hn'].
    (* Handle the easy case first *)
    2: { unfold mys_ckp7 in Hn'.
         destruct Hn' as (_ & Hn').
         rewrite Forall_forall in Hn'.
         specialize (Hn' (udag_vert_builder u) ltac:(rewrite sync_set_correct; split; auto)).
         destruct Hn' as [Hn' | Hn'].
         - pose proof (mdag_recv_cert_mono _ _ (mys_oracle_reachable _ _ Hn''2) _ Hn') as Hmono1.
           pose proof (udag_mono _ _ id Hreach1) as Hmono2.
           rewrite Evert in Hmono2.
           specialize (Hmono2 ltac:(discriminate)).
           pose proof (mdag_honest_recv_leader_vert_certificate _ id (mys_oracle_valid n'')) as Hcert.
           rewrite Hmono2 in Hcert.
           specialize (Hcert ltac:(rewrite Hvert3; discriminate) ltac:(auto)).
           apply (mdag_vert_is_certificate_reachable_inv _ _ id u (mys_oracle_valid n) (mys_oracle_reachable _ _ Hn''1) Evert ltac:(auto)).
         - destruct Hn' as (id' & Hid').
           cond_case_auto Hid'; try contradiction.
           replace (S (S (udag_vert_round u - 2))) with (udag_vert_round u) in Hid' by lia.
           destruct Hid' as (Hid'1 & Hid'2 & Hid'3).
           pose proof (udag_mono _ _ id Hreach1 ltac:(rewrite Evert; discriminate)) as Hmono1.
           rewrite Evert in Hmono1.
           pose proof (udag_mono _ _ id' Hreach2 ltac:(rewrite Ex; discriminate)) as Hmono2.
           rewrite Ex in Hmono2.
           pose proof (udag_honest_uniq _ id id' (mdag_udag_valid _ (mys_oracle_valid n''))) as Huniq.
           rewrite Hmono1, Hmono2 in Huniq.
           specialize (Huniq ltac:(rewrite Hvert3; discriminate) ltac:(auto) ltac:(auto)).
           subst id'.
           rewrite Hmono1 in Hmono2; injection Hmono2; intros; subst u0; clear Hmono2.
           pose proof (mdag_vert_is_certificate_reachable _ _ u (mys_oracle_valid _) (mys_oracle_reachable _ _ Hn''2) Hid'3) as Hcert.
           apply (mdag_vert_is_certificate_reachable_inv _ _ id u (mys_oracle_valid _) (mys_oracle_reachable _ _ Hn''1) Evert Hcert).
    }

    (* Now the hard part *)
    unfold mys_ckp6 in Hn'.
    destruct Hn' as (Har & Hn' & Htoq).
    replace (S (S (udag_vert_round u - 2))) with (udag_vert_round u) in Har, Hn', Htoq by lia.

    rewrite Forall_forall in Hn'.
    specialize (Hn' (udag_vert_builder u) ltac:(rewrite sync_set_correct; split; auto)).
    match type of Hn' with ?A \/ ?B \/ ?C \/ ?D => rewrite <- (or_assoc B C D) in Hn'; rewrite (or_comm _ D) in Hn' end.

    destruct Hn' as [Hn' | [Hn' | Hn']].
    - exfalso.
      pose proof (udag_mono _ _ id Hreach1 ltac:(rewrite Evert; discriminate)) as Hmono1.
      rewrite Evert in Hmono1.
      pose proof (mdag_jump_mono _ _ (mys_oracle_reachable _ _ Hn''2) _ Hn') as Hmono2.
      apply (mys_weak_oracle_jump_no_vert _ _ _ ltac:(rewrite Hvert3; discriminate) Hmono2).
      exists id; rewrite Hmono1; split; auto.

    - destruct Hn' as (id' & Hid').
      cond_case_auto Hid'; try contradiction.
      destruct Hid' as (Hid'1 & Hid'2 & Hid'3).
      pose proof (udag_mono _ _ id Hreach1 ltac:(rewrite Evert; discriminate)) as Hmono1.
      rewrite Evert in Hmono1.
      pose proof (udag_mono _ _ id' Hreach2 ltac:(rewrite Ex; discriminate)) as Hmono2.
      rewrite Ex in Hmono2.
      pose proof (udag_honest_uniq _ id id' (mdag_udag_valid _ (mys_oracle_valid n''))) as Huniq.
      rewrite Hmono1, Hmono2 in Huniq.
      specialize (Huniq ltac:(rewrite Hvert3; discriminate) ltac:(auto) ltac:(auto)).
      subst id'.
      rewrite Hmono1 in Hmono2; injection Hmono2; intros; subst u0; clear Hmono2.
      pose proof (mdag_vert_is_certificate_reachable _ _ u (mys_oracle_valid _) (mys_oracle_reachable _ _ Hn''2) Hid'3) as Hcert.
      apply (mdag_vert_is_certificate_reachable_inv _ _ id u (mys_oracle_valid _) (mys_oracle_reachable _ _ Hn''1) Evert Hcert).

    - clear n'' Hn''1 Hn''2 Hreach1 Hreach2.
      remember (Nat.max n (S n')) as n''.
      assert (Hn''1 : n'' >= n) by lia.
      assert (Hn''2 : n'' >= S n') by lia.
      clear Heqn''.
      pose proof (mdag_udag_reachable _ _ (mys_oracle_reachable _ _ Hn''1)) as Hreach1.
      pose proof (mdag_udag_reachable _ _ (mys_oracle_reachable _ _ Hn''2)) as Hreach2.

      pose proof (mys_catchup n' (udag_vert_builder u) ltac:(auto) ltac:(auto)).
      pose proof (mys_history (S n') (udag_vert_round u) (udag_vert_builder u) ltac:(auto) ltac:(auto) ltac:(lia)) as [(id' & Hid') | Hjump].
      2: { exfalso.
           pose proof (udag_mono _ _ id Hreach1 ltac:(rewrite Evert; discriminate)) as Hmono1.
           rewrite Evert in Hmono1.
           pose proof (mdag_jump_mono _ _ (mys_oracle_reachable _ _ Hn''2) _ Hjump) as Hmono2.
           apply (mys_weak_oracle_jump_no_vert _ _ _ ltac:(rewrite Hvert3; discriminate) Hmono2).
           exists id; rewrite Hmono1; split; auto.
      }
      cond_case_auto Hid'; try contradiction.
      destruct Hid' as (Hid'1 & Hid'2).
      pose proof (udag_mono _ _ id Hreach1 ltac:(rewrite Evert; discriminate)) as Hmono1.
      rewrite Evert in Hmono1.
      pose proof (udag_mono _ _ id' Hreach2 ltac:(rewrite Ex; discriminate)) as Hmono2.
      rewrite Ex in Hmono2.
      pose proof (udag_honest_uniq _ id id' (mdag_udag_valid _ (mys_oracle_valid n''))) as Huniq.
      rewrite Hmono1, Hmono2 in Huniq.
      specialize (Huniq ltac:(rewrite Hvert3; discriminate) ltac:(auto) ltac:(auto)).
      subst id'.
      rewrite Hmono1 in Hmono2; injection Hmono2; intros; subst u0; clear Hmono2.
      clear Hid'1 Hid'2.

      pose proof (mdag_honest_vert _ id (mys_oracle_valid (S n'))) as Hvert5.
      rewrite Ex in Hvert5.
      specialize (Hvert5 ltac:(rewrite Hvert3; discriminate)).
      destruct Hvert5 as [(_ & Hvert5) | [(Hvert5 & _) | (quorum & Hquorum1 & Hquorum2 & _ & _ & _ & Hquorum3)]].
      + pose proof (mdag_vert_is_certificate_reachable _ _ u (mys_oracle_valid _) (mys_oracle_reachable _ _ Hn''2) Hvert5) as Hcert.
      apply (mdag_vert_is_certificate_reachable_inv _ _ id u (mys_oracle_valid _) (mys_oracle_reachable _ _ Hn''1) Evert Hcert).
      + pose proof (mys_weak_oracle_timeout_inv n' (udag_vert_builder u) _ ltac:(auto) ltac:(auto) Hvert5) as [Htimeout | Htimeout2].
        2: lia.
        pose proof (mys_sync_timeout_ge _ _ (udag_vert_builder u) ltac:(auto) ltac:(auto) Htimeout); lia.
      + destruct Htoq as (toq & Htoq1 & Htoq2).
        pose proof (quorum_to_quorum_overlap _ _ Hquorum2 Htoq1) as (x & Hx1 & Hx2).
        rewrite in_map_iff in Hx1.
        destruct Hx1 as (y & Hy1 & Hy2).
        rewrite Forall_forall in Hquorum1.
        specialize (Hquorum1 _ Hy2).
        cond_case_auto Hquorum1; try contradiction.
        subst x.
        clear Hquorum2 Htoq1.
        rewrite Forall_forall in Htoq2.
        specialize (Htoq2 _ Hx2).
        destruct Htoq2 as (Hx3 & (id' & Hx4)).
        cond_case_auto Hx4; try contradiction.
        destruct Hx4 as (Hx4 & Hx5 & Hx6).
        pose proof (mdag_udag_reachable _ _ (mys_oracle_reachable n' (S n') ltac:(auto))) as Hreach3.
        pose proof (udag_mono _ _ id' Hreach3 ltac:(rewrite Ex1; discriminate)) as Hmono3.
        rewrite Ex1 in Hmono3.
        pose proof (udag_honest_uniq _ y id' (mdag_udag_valid _ (mys_oracle_valid (S n')))) as Huniq.
        rewrite Ex0, Hmono3 in Huniq.
        specialize (Huniq ltac:(rewrite Hx3; discriminate) ltac:(auto) ltac:(congruence)).
        subst y.
        rewrite Ex0 in Hmono3; injection Hmono3; intros; subst u1; clear Hmono3.

        rewrite Forall_forall in Hquorum3.
        specialize (Hquorum3 _ Hy2).
        rewrite Ex0 in Hquorum3.
        pose proof (mdag_vert_is_certificate_reachable _ _ u0 (mys_oracle_valid _) (mys_oracle_reachable n' (S n') ltac:(auto)) ltac:(auto)) as Hcert.
        specialize (Hquorum3 ltac:(auto)).
        pose proof (mdag_vert_is_certificate_reachable _ _ u (mys_oracle_valid _) (mys_oracle_reachable _ _ Hn''2) ltac:(auto)) as Hcert'.
        apply (mdag_vert_is_certificate_reachable_inv _ _ id u (mys_oracle_valid _) (mys_oracle_reachable _ _ Hn''1) ltac:(auto) ltac:(auto)).
  Qed.

End WeakLiveness.
