From Coq Require Import
  List
  PeanoNat
  Lia
  Peano_dec.
From advert.lib Require Import
  Maps
  Decision
  Semantics
  Tactics.
From advert.specs Require Import
  UDAG
  Config.
From advert.impl.mysticeti Require Import
  MysticetiDAG
  MysticetiCommit.
From advert.impl.mysticeti.proofs Require Import
  LivenessWeak.
Import Notations ListNotations.

Section LivenessStrong.
  Context `{config : !BADO_Config} `{dag_config : !DAG_Config} `{participant : !BADO_Participant} `{leader : !BADO_Leader} `{node_assump : !BADO_NodeAssump} `{assump : !BADO_Assump}.

  Class MysticetiStrongOracle : Type := {
    mys_strong_oracle_dag : nat -> MDAG_Commit_State;
    mys_strong_oracle_local_ar : nat -> nat -> nat;
    mys_strong_oracle_local_rt : nat -> nat -> nat;

    mys_strong_oracle_init_valid : mcommit_valid (mys_strong_oracle_dag 0);
    mys_strong_oracle_reachable : forall n, mcommit_reachable (mys_strong_oracle_dag n) (mys_strong_oracle_dag (S n));

    mys_strong_oracle_gct : nat; (* GCT = Global Catchup Time *)
    mys_strong_oracle_gct_enter : (mys_strong_oracle_dag mys_strong_oracle_gct).(mcommit_gct) = true;

    mys_strong_oracle_ar_init_pos :
      forall nid,
      In nid bado_participant ->
      bado_node_assump nid = Synchronous ->
      mys_strong_oracle_local_ar 0 nid > 0;

    mys_strong_oracle_rt_init :
      forall nid,
      In nid bado_participant ->
      bado_node_assump nid = Synchronous ->
      mys_strong_oracle_local_rt 0 nid <= 3;

    mys_strong_oracle_ar_step :
      forall n nid,
      In nid bado_participant ->
      bado_node_assump nid = Synchronous ->
      let ar := mys_strong_oracle_local_ar n nid in
      let ar' := mys_strong_oracle_local_ar (S n) nid in
      let rt := mys_strong_oracle_local_rt n nid in
      let rt' := mys_strong_oracle_local_rt (S n) nid in
      ar' = ar /\ rt' = rt - 1 \/ ar' > ar /\ rt' = 3;

    mys_strong_oracle_ar_ge_vert :
      forall n nid id,
      bado_node_assump nid = Synchronous ->
      match NatMap_find id (mys_strong_oracle_dag n).(mcommit_mdag).(mdag_udag).(udag_verts) with
      | None => True
      | Some v => v.(udag_vert_builder) = nid ->
                  mys_strong_oracle_local_ar n nid >= v.(udag_vert_round) end;

    mys_strong_oracle_ar_has_vert :
      forall n nid,
      In nid bado_participant ->
      bado_node_assump nid = Synchronous ->
      mys_strong_oracle_local_ar n nid = 1 \/
      exists id,
        match NatMap_find id (mys_strong_oracle_dag n).(mcommit_mdag).(mdag_udag).(udag_verts) with
        | None => False
        | Some v => v.(udag_vert_builder) = nid /\ v.(udag_vert_round) = mys_strong_oracle_local_ar n nid - 1
        end;

    mys_strong_oracle_ar_has_quorum :
      forall n nid,
      In nid bado_participant ->
      bado_node_assump nid = Synchronous ->
      mys_strong_oracle_local_ar n nid = 1 \/
      exists quorum,
        is_quorum bado_comm quorum /\
        forall nid',
        In nid' quorum ->
        exists id,
          match NatMap_find id (mys_strong_oracle_dag n).(mcommit_mdag).(mdag_udag).(udag_verts) with
          | None => False
          | Some v => v.(udag_vert_builder) = nid' /\ v.(udag_vert_round) = mys_strong_oracle_local_ar n nid - 1
          end;

    mys_strong_oracle_timeout_init :
      forall nid r,
      bado_node_assump nid = Synchronous ->
      In (nid, r) (mys_strong_oracle_dag 0).(mcommit_mdag).(mdag_timeouts) ->
      r < mys_strong_oracle_local_ar 0 nid \/
      r = mys_strong_oracle_local_ar 0 nid /\
      mys_strong_oracle_local_rt 0 nid = 0;

    mys_strong_oracle_timeout :
      forall n nid,
      In nid bado_participant ->
      bado_node_assump nid = Synchronous ->
      mys_strong_oracle_local_rt n nid = 0 ->
      mys_strong_oracle_local_ar (S n) nid = mys_strong_oracle_local_ar n nid ->
      In (nid, mys_strong_oracle_local_ar n nid) (mys_strong_oracle_dag (S n)).(mcommit_mdag).(mdag_timeouts);

    mys_strong_oracle_timeout_inv :
      forall n nid r,
      In nid bado_participant ->
      bado_node_assump nid = Synchronous ->
      In (nid, r) (mys_strong_oracle_dag (S n)).(mcommit_mdag).(mdag_timeouts) ->
      In (nid, r) (mys_strong_oracle_dag n).(mcommit_mdag).(mdag_timeouts) \/
      r = mys_strong_oracle_local_ar n nid /\
      mys_strong_oracle_local_rt n nid = 0;

    mys_strong_oracle_catchup :
      forall n nid nid',
      In nid bado_participant ->
      bado_node_assump nid = Synchronous ->
      In nid' bado_participant ->
      bado_node_assump nid' = Synchronous ->
      mys_strong_oracle_local_ar (S n) nid' >= mys_strong_oracle_local_ar n nid;

    mys_strong_oracle_enter :
      forall n r,
      (forall nid,
       In nid bado_participant ->
       bado_node_assump nid = Synchronous ->
       exists id, match NatMap_find id (mys_strong_oracle_dag n).(mcommit_mdag).(mdag_udag).(udag_verts) with
                  | None => False
                  | Some v => v.(udag_vert_round) = r /\ v.(udag_vert_builder) = nid
                  end) ->
      forall nid,
      In nid bado_participant ->
      bado_node_assump nid = Synchronous ->
      mys_strong_oracle_local_ar (S n) nid > r;

    mys_strong_oracle_recv_leader_vert :
      forall n r,
      bado_node_assump (bado_leader_at r) = Synchronous ->
      (exists id, match NatMap_find id (mys_strong_oracle_dag n).(mcommit_mdag).(mdag_udag).(udag_verts) with
                  | None => False
                  | Some v => v.(udag_vert_round) = r /\ v.(udag_vert_builder) = bado_leader_at r
                  end) ->
      forall nid,
      In nid bado_participant ->
      bado_node_assump nid = Synchronous ->
      In (nid, r) (mys_strong_oracle_dag (S n)).(mcommit_mdag).(mdag_recv_leader_verts) \/
      (exists id, match NatMap_find id (mys_strong_oracle_dag (S n)).(mcommit_mdag).(mdag_udag).(udag_verts) with
                  | None => False
                  | Some v => v.(udag_vert_round) = S r /\ v.(udag_vert_builder) = nid
                  end);

    mys_strong_oracle_recv_cert :
      forall n r,
      r >= 1 ->
      bado_node_assump (bado_leader_at r) <> Byzantine ->
      (forall nid,
       In nid bado_participant ->
       bado_node_assump nid = Synchronous ->
       exists id, match NatMap_find id (mys_strong_oracle_dag n).(mcommit_mdag).(mdag_udag).(udag_verts) with
                  | None => False
                  | Some v => v.(udag_vert_round) = S r /\ v.(udag_vert_builder) = nid /\ mdag_vert_is_supporter v (mys_strong_oracle_dag n).(mcommit_mdag)
                  end) ->
      forall nid,
      In nid bado_participant ->
      bado_node_assump nid = Synchronous ->
      In (nid, r) (mys_strong_oracle_dag (S n)).(mcommit_mdag).(mdag_recv_certs) \/
      (exists id, match NatMap_find id (mys_strong_oracle_dag (S n)).(mcommit_mdag).(mdag_udag).(udag_verts) with
                  | None => False
                  | Some v => v.(udag_vert_round) = S (S r) /\ v.(udag_vert_builder) = nid
                  end);

    mys_strong_oracle_commit :
      forall n r,
      r >= 1 ->
      (forall nid,
       In nid bado_participant ->
       bado_node_assump nid = Synchronous ->
       exists id, match NatMap_find id (mys_strong_oracle_dag n).(mcommit_mdag).(mdag_udag).(udag_verts) with
                  | None => False
                  | Some v => v.(udag_vert_round) = S (S r) /\ v.(udag_vert_builder) = nid /\ mdag_vert_is_certificate v (mys_strong_oracle_dag n).(mcommit_mdag)
                  end) ->
      exists id, In (r, id) (mys_strong_oracle_dag (S n)).(mcommit_commit);

    mys_strong_oracle_jump_no_vert :
      forall n nid r,
      bado_node_assump nid <> Byzantine ->
      In (nid, r) (mys_strong_oracle_dag n).(mcommit_mdag).(mdag_jumps) ->
      ~ exists id, match NatMap_find id (mys_strong_oracle_dag n).(mcommit_mdag).(mdag_udag).(udag_verts) with
                   | None => False
                   | Some v => v.(udag_vert_round) = r /\ v.(udag_vert_builder) = nid
                   end;
  }.

  Context `{strong_oracle : MysticetiStrongOracle}.

  Definition mys_ckp9 n r := exists id, In (r, id) (mys_strong_oracle_dag (S n)).(mcommit_commit).

  Instance weak_oracle : MysticetiWeakOracle.
  Proof. econstructor.
         Unshelve.
         17: exact (fun n => (mys_strong_oracle_dag n).(mcommit_mdag)).
         17: exact mys_strong_oracle_local_ar.
         17: exact mys_strong_oracle_local_rt.
         all: try (cbn; apply strong_oracle).
         - cbn; apply mcommit_mdag_valid; apply strong_oracle.
         - cbn; intros; apply mcommit_mdag_reachable; apply strong_oracle.
  Defined.

  Lemma mcommit_oracle_valid :
    forall n,
    mcommit_valid (mys_strong_oracle_dag n).
  Proof. intros n.
         induction n.
         - apply mys_strong_oracle_init_valid.
         - eapply valid_reach_valid.
           + apply IHn.
           + apply mys_strong_oracle_reachable.
  Qed.

  Lemma mcommit_oracle_reachable :
    forall n m,
    n <= m ->
    mcommit_reachable (mys_strong_oracle_dag n) (mys_strong_oracle_dag m).
  Proof. intros n m.
         induction m.
         - intros.
           replace n with 0 by lia.
           apply reachable_self.
         - intros Hle.
           assert (Hn : n <= m \/ n = S m) by lia.
           destruct Hn as [Hn | Hn].
           + specialize (IHm Hn).
             eapply reachable_trans.
             * apply IHm.
             * apply mys_strong_oracle_reachable.
           + subst n.
             apply reachable_self.
  Qed.

  Lemma mys_gct_jumps : forall nid r,
    bado_node_assump nid = Synchronous ->
    In (nid, r) (mys_strong_oracle_dag mys_strong_oracle_gct).(mcommit_mdag).(mdag_jumps) ->
    r <= mys_ar mys_strong_oracle_gct.
  Proof. intros nid r Hsync Hin.
         pose proof (mys_weak_oracle_ar_ge_jump mys_strong_oracle_gct r nid Hsync Hin) as H.
         apply H.
  Qed.

  Lemma mys_strong_jump_has_decision : forall nid r n,
    bado_node_assump nid = Synchronous ->
    r > mys_ar mys_strong_oracle_gct ->
    In (nid, r) (mys_strong_oracle_dag n).(mcommit_mdag).(mdag_jumps) ->
    r <= 2 \/ (exists id, In (r - 2, id) (mys_strong_oracle_dag n).(mcommit_commit)) \/ In (r - 2) (mys_strong_oracle_dag n).(mcommit_nack).
  Proof. intros nid r n Hsync Hgt Hin.
         assert (Hnot_in : ~ In (nid, r) (mys_strong_oracle_dag mys_strong_oracle_gct).(mcommit_mdag).(mdag_jumps)).
         { intros Hin'.
           pose proof (mys_gct_jumps _ _ Hsync Hin').
           lia.
         }
         assert (Hn : n <= mys_strong_oracle_gct \/ n > mys_strong_oracle_gct) by lia.
         destruct Hn as [Hn | Hn].
         1: { pose proof (mcommit_oracle_reachable _ _ Hn) as Hreach.
              pose proof (mdag_jump_mono _ _ (mcommit_mdag_reachable _ _ Hreach) _ Hin).
              contradiction.
         }
         pose proof (mcommit_gct_jump_has_decision nid r (mys_strong_oracle_dag mys_strong_oracle_gct) (mys_strong_oracle_dag n) ltac:(apply mcommit_oracle_reachable; lia) mys_strong_oracle_gct_enter Hin) as Hdec.
         destruct Hdec as [? | Hdec]; [contradiction | auto].
  Qed.

  Lemma mys_ckp6_no_skip n n' r :
    mys_ckp6 n r -> ~ In r (mys_strong_oracle_dag n').(mcommit_nack).
  Proof. intros Hckp Hnack.
         pose proof (mcommit_nack_invariant _ _ (mcommit_oracle_valid n') Hnack) as (r' & id' & Hpos & Hge & Hcommit & _ & Hno_cert).
         destruct Hckp as (_ & _ & toq & Htoq1 & Htoq2).
         pose proof (udag_closure_quorum _ id' (S (S r)) (mdag_udag_valid _ (mcommit_mdag_valid _ (mcommit_oracle_valid n')))) as Hquorum.
         pose proof (mcommit_commit_valid _ _ _ (mcommit_oracle_valid n') Hcommit) as Hvert; clear Hcommit.
         cond_case_auto Hvert; try contradiction.
         specialize (Hquorum ltac:(lia)).
         destruct Hquorum as (quorum & Hquorum1 & Hquorum2 & Hquorum3).
         clear u Ex Hvert.
         rewrite Forall_forall in *.
         pose proof (quorum_to_quorum_overlap _ _ Hquorum2 Htoq1) as (nid & Hnid1 & Hnid2).
         clear Htoq1 Hquorum2.
         specialize (Htoq2 _ Hnid2); clear Hnid2.
         destruct Htoq2 as (Hnid2 & id & Hid).
         rewrite in_map_iff in Hnid1.
         destruct Hnid1 as (x & Hx1 & Hx2).
         specialize (Hquorum1 _ Hx2).
         cond_case_auto Hquorum1; try contradiction.
         specialize (Hquorum3 _ Hx2); clear Hx2.
         specialize (Hno_cert _ Hquorum3); rewrite Ex in Hno_cert.
         specialize (Hno_cert ltac:(replace (r + 2)%nat with (S (S r)) by lia; auto)).
         cond_case_auto Hid; try contradiction.
         destruct Hid as (Hid1 & Hid2 & Hid3).
         cbn in Ex0.

         assert (Heq : id = x /\ u = u0).
         { destruct (decide (n <= n')).
           - pose proof (mcommit_oracle_reachable n n' l) as Hreach.
             pose proof (udag_mono _ _ id (mdag_udag_reachable _ _ (mcommit_mdag_reachable _ _ Hreach))) as Hmono.
             specialize (Hmono ltac:(rewrite Ex0; discriminate)).
             pose proof (udag_honest_uniq _ id x (mdag_udag_valid _ (mcommit_mdag_valid _ (mcommit_oracle_valid n')))) as Huniq.
             rewrite Hmono, Ex0, Ex in Huniq.
             specialize (Huniq ltac:(congruence) ltac:(congruence) ltac:(congruence)).
             split; auto; congruence.
           - pose proof (mcommit_oracle_reachable n' n ltac:(lia)) as Hreach.
             pose proof (udag_mono _ _ x (mdag_udag_reachable _ _ (mcommit_mdag_reachable _ _ Hreach))) as Hmono.
             specialize (Hmono ltac:(rewrite Ex; discriminate)).
             pose proof (udag_honest_uniq _ id x (mdag_udag_valid _ (mcommit_mdag_valid _ (mcommit_oracle_valid n)))) as Huniq.
             rewrite Hmono, Ex0, Ex in Huniq.
             specialize (Huniq ltac:(congruence) ltac:(congruence) ltac:(congruence)).
             split; auto; congruence.
         }

         destruct Heq; subst x u0.
         clear toq Hnid2 Hnack r' id' Hpos Hge quorum Hquorum1 Hquorum3 Hx1.
         cbn in Hid3.
         destruct (decide (n <= n')).
         - pose proof (mcommit_mdag_reachable _ _ (mcommit_oracle_reachable n n' l)) as Hreach.
           pose proof (mdag_vert_is_certificate_reachable _ _ _ (mcommit_mdag_valid _ (mcommit_oracle_valid n)) Hreach Hid3).
           contradiction.
         - pose proof (mcommit_mdag_reachable _ _ (mcommit_oracle_reachable n' n ltac:(lia))) as Hreach.
           pose proof (mdag_vert_is_certificate_reachable_inv _ _ _ _ (mcommit_mdag_valid _ (mcommit_oracle_valid n')) Hreach Ex Hid3).
           contradiction.
  Qed.

  Lemma mys_ckp7_no_skip n n' r :
    mys_ckp7 n r -> ~ In r (mys_strong_oracle_dag n').(mcommit_nack).
  Proof. intros Hckp Hnack.
         destruct Hckp as (_ & Hcommit_sync).
         pose proof (mcommit_nack_invariant _ _ (mcommit_oracle_valid n') Hnack) as (r' & id' & Hpos & Hge & Hcommit & _ & Hno_cert).
         pose proof (mcommit_commit_valid _ _ _ (mcommit_oracle_valid n') Hcommit) as Hvert; clear Hcommit.
         cond_case_auto Hvert; try contradiction.
         pose proof (udag_closure_quorum _ id' (S (S r)) (mdag_udag_valid _ (mcommit_mdag_valid _ (mcommit_oracle_valid n')))) as Hquorum.
         rewrite Ex in Hquorum.
         specialize (Hquorum ltac:(lia)).
         destruct Hquorum as (quorum & Hquorum1 & Hquorum2 & Hquorum3).
         clear u Ex Hvert.
         rewrite Forall_forall in *.
         pose proof (quorum_overlap_exists_honest bado_comm_safe _ _ sync_set_quorum Hquorum2) as (nid & Hnid1 & Hnid2 & _).
         pose proof Hnid1 as Hnid3; rewrite sync_set_correct in Hnid3.
         clear Hquorum2.
         rewrite in_map_iff in Hnid2.
         destruct Hnid2 as (x & Hx1 & Hx2).
         specialize (Hquorum1 _ Hx2).
         cond_case_auto Hquorum1; try contradiction.
         specialize (Hquorum3 _ Hx2); clear Hx2.
         specialize (Hno_cert _ Hquorum3).
         rewrite Ex in Hno_cert.
         specialize (Hno_cert ltac:(replace (r + 2)%nat with (S (S r)) by lia; auto)).
         specialize (Hcommit_sync _ Hnid1) as [Hcommit_sync | Hcommit_sync]; cbn in Hcommit_sync.

         - (* recv_cert created *)
           destruct (decide (n <= n')).
           + pose proof (mcommit_oracle_reachable n n' l) as Hreach.
             pose proof (mdag_recv_cert_mono _ _ (mcommit_mdag_reachable _ _ Hreach) _ Hcommit_sync) as Hcommit_sync'.
             pose proof (mdag_honest_recv_leader_vert_certificate _ x (mcommit_mdag_valid _ (mcommit_oracle_valid n'))) as Hcert.
             rewrite Ex in Hcert.
             rewrite Hquorum1 in Hcert.
             replace (S (S r) - 2) with r in Hcert by lia.
             specialize (Hcert ltac:(destruct Hnid3; congruence) ltac:(congruence)).
             contradiction.
           + clear Hnack r' id' Hpos Hge quorum Hquorum3.
             pose proof (mcommit_oracle_reachable n' n ltac:(lia)) as Hreach.
             pose proof (udag_mono _ _ x (mdag_udag_reachable _ _ (mcommit_mdag_reachable _ _ Hreach)) ltac:(rewrite Ex; discriminate)) as Hmono.
             pose proof (mdag_honest_recv_leader_vert_certificate _ x (mcommit_mdag_valid _ (mcommit_oracle_valid n))) as Hcert.
             rewrite Hmono, Ex in Hcert.
             rewrite Hquorum1 in Hcert; replace (S (S r) - 2) with r in Hcert by lia.
             specialize (Hcert ltac:(destruct Hnid3; congruence) ltac:(congruence)).
             clear Hnid1 Hnid3 Hcommit_sync n0 Hmono.
             pose proof (mdag_vert_is_certificate_reachable_inv _ _ _ _ (mcommit_mdag_valid _ (mcommit_oracle_valid n')) (mcommit_mdag_reachable _ _ Hreach) Ex Hcert).
             contradiction.

         - (* certificate exists *)
           destruct Hcommit_sync as (id & Hid).
           cond_case_auto Hid; try contradiction.
           destruct Hid as (Hid1 & Hid2 & Hid3).

           assert (Heq : id = x /\ u = u0).
           { destruct Hnid3.
             destruct (decide (n <= n')).
             - pose proof (mcommit_oracle_reachable n n' l) as Hreach.
               pose proof (udag_mono _ _ id (mdag_udag_reachable _ _ (mcommit_mdag_reachable _ _ Hreach))) as Hmono.
               specialize (Hmono ltac:(rewrite Ex0; discriminate)).
               pose proof (udag_honest_uniq _ id x (mdag_udag_valid _ (mcommit_mdag_valid _ (mcommit_oracle_valid n')))) as Huniq.
               rewrite Hmono, Ex0, Ex in Huniq.
               specialize (Huniq ltac:(congruence) ltac:(congruence) ltac:(congruence)).
               split; auto; congruence.
             - pose proof (mcommit_oracle_reachable n' n ltac:(lia)) as Hreach.
               pose proof (udag_mono _ _ x (mdag_udag_reachable _ _ (mcommit_mdag_reachable _ _ Hreach))) as Hmono.
               specialize (Hmono ltac:(rewrite Ex; discriminate)).
               pose proof (udag_honest_uniq _ id x (mdag_udag_valid _ (mcommit_mdag_valid _ (mcommit_oracle_valid n)))) as Huniq.
               rewrite Hmono, Ex0, Ex in Huniq.
               specialize (Huniq ltac:(congruence) ltac:(congruence) ltac:(congruence)).
               split; auto; congruence.
         }

         destruct Heq; subst x u0.
         clear Hnack r' id' Hpos Hge quorum Hquorum1 Hquorum3 Hnid1 Hnid3 Hx1.
         destruct (decide (n <= n')).
         + pose proof (mcommit_mdag_reachable _ _ (mcommit_oracle_reachable n n' l)) as Hreach.
           pose proof (mdag_vert_is_certificate_reachable _ _ _ (mcommit_mdag_valid _ (mcommit_oracle_valid n)) Hreach Hid3).
           contradiction.
         + pose proof (mcommit_mdag_reachable _ _ (mcommit_oracle_reachable n' n ltac:(lia))) as Hreach.
           pose proof (mdag_vert_is_certificate_reachable_inv _ _ _ _ (mcommit_mdag_valid _ (mcommit_oracle_valid n')) Hreach Ex Hid3).
           contradiction.
  Qed.

  Lemma mys_live7 n r :
    r >= 1 ->
    S r >= mys_ar (mys_strong_oracle_gct) ->
    mys_ckp7 n r -> exists n', mys_ckp8 n' r \/ exists id, In (r, id) (mys_strong_oracle_dag n').(mcommit_commit).
  Proof. intros Hge1 Hge2 Hckp.
         pose proof Hckp as Hckp'.
         destruct Hckp as (Har & Hckp).
         pose proof (mys_ar_mono 0 mys_strong_oracle_gct ltac:(lia)) as Hge'.
         pose proof (mys_progress (S (S (S r))) ltac:(lia)) as (n' & Hn').
         assert (Hn'' : exists n'', n'' >= n /\ mys_ar n'' >= S (S (S r))).
         { destruct (decide (n <= n')).
           1: exists n'; split; auto.
           exists n; split; auto.
           pose proof (mys_ar_mono n' n ltac:(lia)); lia.
         }
         clear n' Hn'.
         destruct Hn'' as (n' & Hn'1 & Hn'2).
         cbn in Hckp.

         exists (S n').
         assert (Hcert : Forall
                           (fun nid => (exists id, In (r, id) (mys_strong_oracle_dag (S n')).(mcommit_commit)) \/
                                       (exists id, match NatMap_find id (mys_strong_oracle_dag (S n')).(mcommit_mdag).(mdag_udag).(udag_verts) with
                                                   | None => False
                                                   | Some v => v.(udag_vert_round) = S (S r) /\ v.(udag_vert_builder) = nid /\ mdag_vert_is_certificate v (mys_strong_oracle_dag (S n')).(mcommit_mdag)
                                                   end))
                           sync_set).
         { rewrite Forall_forall in *.
           intros x Hx1.
           pose proof Hx1 as Hx2; rewrite sync_set_correct in Hx2.
           specialize (Hckp _ Hx1) as [Hckp | Hckp].
           - pose proof (mdag_recv_cert_mono _ _ (mcommit_mdag_reachable _ _ (mcommit_oracle_reachable n (S n') ltac:(lia))) _ Hckp) as Hcert.
             clear Hckp.
             pose proof (mys_catchup n' x ltac:(apply Hx2) ltac:(apply Hx2)) as Henter; cbn in Henter.
             pose proof (mys_history (S n') (S (S r)) x ltac:(apply Hx2) ltac:(apply Hx2) ltac:(cbn; lia)) as [Hvert | Hjump].
             + destruct Hvert as (id & Hid).
               cond_case_auto Hid; try contradiction; cbn in Ex.
               pose proof (mdag_honest_recv_leader_vert_certificate _ id (mcommit_mdag_valid _ (mcommit_oracle_valid (S n')))) as Hcert'.
               rewrite Ex in Hcert'.
               destruct Hx2; destruct Hid as (Hid1 & Hid2).
               rewrite Hid1 in Hcert'.
               replace (S (S r) - 2) with r in Hcert' by lia.
               specialize (Hcert' ltac:(congruence) ltac:(congruence)).
               right; exists id; rewrite Ex; repeat split; auto.
             + pose proof (mys_strong_jump_has_decision x (S (S r)) _ ltac:(apply Hx2) ltac:(lia) Hjump) as [Hdec | [Hdec | Hdec]].
               * lia.
               * replace (S (S r) - 2) with r in Hdec by lia.
                 left; apply Hdec.
               * replace (S (S r) - 2) with r in Hdec by lia.
                 pose proof (mys_ckp7_no_skip n (S n') r Hckp').
                 contradiction.
           - destruct Hckp as (id & Hid).
             right; exists id.
             cond_case_auto Hid; try contradiction.
             pose proof (mcommit_oracle_reachable n (S n') ltac:(lia)) as Hreach.
             pose proof (udag_mono _ _ id (mdag_udag_reachable _ _ (mcommit_mdag_reachable _ _ Hreach)) ltac:(congruence)) as Hmono.
             rewrite Hmono; rewrite Ex.
             repeat split; try apply Hid.
             apply (mdag_vert_is_certificate_reachable _ _ _ (mcommit_mdag_valid _ (mcommit_oracle_valid n)) (mcommit_mdag_reachable _ _ Hreach) ltac:(apply Hid)).
         }

         rewrite Forall_forall in Hcert.
         pose proof (all_true_or_one_false sync_set Hcert sync_set_nonempty') as [Hcert' | Hcommit].
         - left; unfold mys_ckp8.
           rewrite Forall_forall; auto.
         - destruct Hcommit as (_ & _ & Hcommit).
           right; auto.
  Qed.

  Lemma mys_live8 n r :
    r >= 1 ->
    S r >= mys_ar (mys_strong_oracle_gct) ->
    mys_ckp6 n r -> mys_ckp8 (S n) r \/ mys_ckp9 n r.
  Proof. unfold mys_ckp9.
         intros Hge1 Hge2 Hckp.
         pose proof Hckp as Hckp'.
         destruct Hckp as (Har & Hcert & toq & Htoq1 & Htoq2).
         cbn in Hcert, Htoq2.
         assert (Hcert' : Forall
                           (fun nid => (exists id, In (r, id) (mys_strong_oracle_dag (S n)).(mcommit_commit)) \/
                                       (exists id, match NatMap_find id (mys_strong_oracle_dag (S n)).(mcommit_mdag).(mdag_udag).(udag_verts) with
                                                   | None => False
                                                   | Some v => v.(udag_vert_round) = S (S r) /\ v.(udag_vert_builder) = nid /\ mdag_vert_is_certificate v (mys_strong_oracle_dag (S n)).(mcommit_mdag)
                                                   end))
                           sync_set).
         { pose proof (mcommit_oracle_reachable n (S n) ltac:(auto)) as Hreach.
           rewrite Forall_forall in *.
           intros x Hx1.
           pose proof Hx1 as Hx2; rewrite sync_set_correct in Hx2.
           pose proof (mys_catchup n x ltac:(apply Hx2) ltac:(apply Hx2)) as Henter; cbn in Henter.
           specialize (Hcert _ Hx1).
           match type of Hcert with ?A \/ ?B \/ ?C \/ ?D => assert (Hcert' : A \/ D \/ (B \/ C)) end.
           { destruct Hcert as [H | [H | [H | H]]]; auto. }
           clear Hcert.
           destruct Hcert' as [Hjump | [Hcert | Htime]].
           1: pose proof (mdag_jump_mono _ _ (mcommit_mdag_reachable _ _ Hreach) _ Hjump) as Hjump';
              pose proof (mys_strong_jump_has_decision _ (S (S r)) (S n) ltac:(apply Hx2) ltac:(lia) Hjump') as Hdec;
              replace (S (S r) - 2) with r in Hdec by lia;
              destruct Hdec as [Hdec | [Hdec | Hdec]]; [lia | left; auto | pose proof (mys_ckp6_no_skip _ (S n) _ Hckp'); contradiction].
           1: destruct Hcert as (id & Hid);
              cond_case_auto Hid; try contradiction;
              right; exists id;
              pose proof (udag_mono _ _ id (mdag_udag_reachable _ _ (mcommit_mdag_reachable _ _ Hreach)) ltac:(congruence)) as Hmono;
              rewrite Hmono, Ex;
              repeat split; try apply Hid;
              apply (mdag_vert_is_certificate_reachable _ _ _ (mcommit_mdag_valid _ (mcommit_oracle_valid n)) (mcommit_mdag_reachable _ _ Hreach) ltac:(apply Hid)).
           assert (Hno_timeout : ~ In (x, (S (S r))) (mys_strong_oracle_dag n).(mcommit_mdag).(mdag_timeouts)).
           { intros Htimeout; pose proof (mys_sync_timeout_ge _ _ _ ltac:(apply Hx2) ltac:(apply Hx2) Htimeout); cbn in *; lia. }
           assert (Hno_timeout' : ~ In (x, (S (S r))) (mys_strong_oracle_dag (S n)).(mcommit_mdag).(mdag_timeouts)).
           { intros Htimeout; pose proof (mys_weak_oracle_timeout_inv _ _ _ ltac:(apply Hx2) ltac:(apply Hx2) Htimeout) as [? | Htime']; [contradiction | cbn in Htime'; lia]. }
           clear Htime Hno_timeout.
           pose proof (mys_history (S n) (S (S r)) x ltac:(apply Hx2) ltac:(apply Hx2) ltac:(cbn; lia)) as [Hvert | Hjump].
           2: pose proof (mys_strong_jump_has_decision _ (S (S r)) (S n) ltac:(apply Hx2) ltac:(lia) Hjump) as Hdec;
              replace (S (S r) - 2) with r in Hdec by lia;
              destruct Hdec as [Hdec | [Hdec | Hdec]]; [lia | left; auto | pose proof (mys_ckp6_no_skip _ (S n) _ Hckp'); contradiction].
           destruct Hvert as (id & Hid1); cbn in Hid1.
           right; exists id.
           cond_case_auto Hid1; try contradiction.
           repeat split; try apply Hid1.
           pose proof (mdag_honest_vert _ id (mcommit_mdag_valid _ (mcommit_oracle_valid (S n)))) as Hid2.
           rewrite Ex in Hid2.
           destruct Hid1; destruct Hx2.
           specialize (Hid2 ltac:(congruence)).
           destruct Hid2 as [Hid2 | [Hid2 | Hid2]].
           - apply Hid2.
           - destruct Hid2 as (Hid2 & _); exfalso; apply Hno_timeout'; congruence.
           - clear Hno_timeout'.
             destruct Hid2 as (quorum & Hquorum1 & Hquorum2 & _ & _ & _ & Hquorum3).
             rewrite Forall_forall in *.
             pose proof (quorum_to_quorum_overlap _ _ Hquorum2 Htoq1) as (nid & Hnid1 & Hnid2).
             clear Hquorum2 Htoq1.
             specialize (Htoq2 _ Hnid2) as (Hnid3 & Hvert).
             clear Hnid2.
             rewrite in_map_iff in Hnid1.
             destruct Hnid1 as (y & Hy1 & Hy2).
             specialize (Hquorum1 _ Hy2).
             cond_case_auto Hquorum1; try contradiction.
             specialize (Hquorum3 _ Hy2).
             rewrite Ex0 in Hquorum3.
             clear Hy2.
             destruct Hvert as (id' & Hid').
             cond_case_auto Hid'; try contradiction.
             clear toq quorum.
             apply Hquorum3.
             destruct Hid' as (? & ? & Hcert).
             pose proof (udag_mono _ _ id' (mdag_udag_reachable _ _ (mcommit_mdag_reachable _ _ Hreach)) ltac:(congruence)) as Hmono.
             pose proof (udag_honest_uniq _ y id' (mdag_udag_valid _ (mcommit_mdag_valid _ (mcommit_oracle_valid (S n))))) as Huniq.
             rewrite Hmono, Ex0, Ex1 in Huniq.
             specialize (Huniq ltac:(congruence) ltac:(congruence) ltac:(congruence)).
             subst y.
             rewrite <- Hmono in Ex1; rewrite Ex1 in Ex0; inversion Ex0; subst u1; clear Ex0.
             apply (mdag_vert_is_certificate_reachable _ _ u0 (mcommit_mdag_valid _ (mcommit_oracle_valid n)) (mcommit_mdag_reachable _ _ Hreach) Hcert).
         }

         rewrite Forall_forall in Hcert'.
         pose proof (all_true_or_one_false sync_set Hcert' sync_set_nonempty') as [Hcert'' | Hcommit].
         - left; unfold mys_ckp8.
           rewrite Forall_forall; auto.
         - destruct Hcommit as (_ & _ & Hcommit).
           right; auto.
  Qed.

  Lemma mys_live9 n r :
    r >= 1 ->
    mys_ckp8 n r -> mys_ckp9 n r.
  Proof. unfold mys_ckp9.
         intros Hge Hckp.
         unfold mys_ckp8 in Hckp.
         rewrite Forall_forall in Hckp.
         apply mys_strong_oracle_commit; auto.
         intros; apply Hckp.
         rewrite sync_set_correct; split; auto.
  Qed.

  Lemma mys_liveness_strong r :
    r >= mys_ar 0 ->
    S r >= mys_ar (mys_strong_oracle_gct) ->
    bado_node_assump (bado_leader_at r) = Synchronous ->
    exists n, exists id, In (r, id) (mys_strong_oracle_dag n).(mcommit_commit).
  Proof.
    intros Hr1 Hr2 Hsync.
    pose proof (mys_liveness_weak' r Hr1 Hsync) as (n1 & Hlive1).
    pose proof (mys_ar_pos 0).

    assert (Hlive2 : exists n2, mys_ckp8 n2 r \/ exists id, In (r, id) (mcommit_commit (mys_strong_oracle_dag n2))).
    { destruct Hlive1 as [Hlive1 | Hlive1].
      - pose proof (mys_live8 n1 r ltac:(lia) Hr2 Hlive1) as Hlive2.
        exists (S n1); auto.
      - pose proof (mys_live7 n1 r ltac:(lia) Hr2 Hlive1) as (n2 & Hlive2).
        exists n2; auto.
    }
    destruct Hlive2 as (n2 & [Hlive2 | Hlive2]).
    2: exists n2; auto.

    pose proof (mys_live9 n2 r ltac:(lia) Hlive2).
    exists (S n2); auto.
  Qed.

End LivenessStrong.
