From Coq Require Import
  List
  PeanoNat
  Lia
  Peano_dec.
From advert.lib Require Import
  Maps
  Decision
  Semantics
  Tactics.
From advert.specs Require Import
  UDAG
  Config.
Import Notations ListNotations.
Open Scope bool_scope.

Section First_Elem.
  (* Mysticeti's concept of "support" crucially relies on "the first element in an ordered list satisfying a condition" *)
  Fixpoint first_elem {T : Type} (l : list T) (P : T -> Prop) {P_dec : forall x, Decision (P x)} :=
  match l with
  | [] => None
  | x :: xs => if decide (P x) then Some x else first_elem xs P
  end.

  Lemma first_elem_correct {T : Type} (l : list T) (P : T -> Prop) {P_dec : forall x, Decision (P x)} (x : T) :
  first_elem l P = Some x ->
  In x l /\ P x.
  Proof. revert x.
         induction l.
         - cbn; intros; discriminate.
         - intros x Heq.
           cbn in Heq.
           destruct (decide (P a)).
           2: specialize (IHl _ Heq); split; try right; apply IHl.
           inversion Heq; subst a; clear Heq.
           split; try left; auto.
  Qed.

  Lemma first_elem_ex {T : Type} (l : list T) (P : T -> Prop) {P_dec : forall x, Decision (P x)} :
  Exists P l -> first_elem l P <> None.
  Proof. intros Hex.
         induction Hex.
         - cbn.
           rewrite decide_true_if; auto; discriminate.
         - cbn.
           destruct (decide (P x)).
           + discriminate.
           + auto.
  Qed.

  Lemma first_elem_filt {T : Type} (l : list T) (P : T -> Prop) {P_dec : forall x, Decision (P x)} (Q : T -> Prop) {Q_dec : forall x, Decision (Q x)} :
  (forall x, In x l -> P x -> Q x) ->
  first_elem l P = first_elem (filter (fun x => if decide (Q x) then true else false) l) P.
  Proof. intros Himp.
         induction l.
         - cbn; auto.
         - specialize (IHl ltac:(intros x Hx; specialize (Himp x ltac:(right; auto)); auto)).
           cbn.
           destruct (decide (P a)).
           + specialize (Himp _ ltac:(left; auto) p).
             rewrite decide_true_if; auto; cbn.
             rewrite decide_true_if; auto.
           + destruct (decide (Q a)); auto.
             cbn; rewrite decide_false_if; auto.
  Qed.

  Lemma first_elem_equiv {T : Type} (l : list T) (P : T -> Prop) {P_dec : forall x, Decision (P x)} (Q : T -> Prop) {Q_dec : forall x, Decision (Q x)} :
  (forall x, In x l -> P x <-> Q x) ->
  first_elem l P = first_elem l Q.
  Proof. intros Heq.
         induction l.
         - cbn; auto.
         - specialize (IHl ltac:(intros x Hx; specialize (Heq x ltac:(right; auto)); auto)).
           cbn.
           specialize (Heq _ ltac:(left; auto)).
           destruct (decide (P a)).
           + rewrite Heq in p; rewrite decide_true_if; auto.
           + rewrite Heq in n; rewrite decide_false_if; auto.
  Qed.
End First_Elem.

Section MysticetiDAG.
(* Mysticeti implements consensus upon UDAG in the following way.
   Initially, each process is in round 1.
   A process creates a vertex in round r in one of the following three situations:
   1. The process is in round r, and the following two conditions are satisfied simultaneously:
      A. r = 1 or the process has received at least one leader vertex of round r-1 (along with its entire closure);
      B. r <= 2 or the process has received 2f+1 vertices in round r-1 from different processes supporting a single leader vertex of round r-2.
      In this case, it should create a vertex that is simultaneously a supporter of a leader vertex of round r-1,
      and a certificate for a leader vertex of round r-2.
   2. The process is in round r, and its local timer has timed-out.
      In this case, if it has previously received at least one leader vertex of round r-1,
      then the vertex it creates should also reference at least one leader vertex of round r-1.
      If it has previously received 2f+1 vertices in round r-1 from different processes supporting a single leader vertex of round r-2,
      then it should create a vertex that is a certificate of that leader vertex.
   3. The process has received 2f+1 vertices in some round r' >= r (along with its entire closure).
      In this case it must have also received at least 2f+1 vertices in round r.
      If any vertex within this set supports some leader vertex of round r-1,
      then it should also support at least one leader vertex of round r-1.
      If any vertex in this set is a certificate of round r-2, then
      it should also be a certificate of round r-2.
   A process enters round r when it observes 2f+1 vertices of round r-1.
 *)

  Context `{dag_config : !DAG_Config}.

  Record MDAG_State : Type := {
    mdag_udag : UDAG_State;
    mdag_recv_leader_verts : list (nat * nat); (* When process p receives leader vertex of round r and has never timed-out in any round r' > r,
                                                  it sends a virtual msg recv_leader_vert (p, r) *)
    mdag_recv_certs : list (nat * nat); (* When process p receives 2f+1 vertices in round r+1 supporting a single leader vertex of round r,
                                           and has never timed-out in any round r' > r+1,
                                           it sends a virtual msg recv_cert (p, r) *)
    mdag_timeouts : list (nat * nat); (* When process p times-out in round r, it sends a virtual msg timeout (p, r) *)
    mdag_jumps : list (nat * nat); (* When process p decides to jump over round r, it sends a virtual msg jump (p, r) *)
  }.

  Definition mdag_state_null : MDAG_State :=
  {| mdag_udag := udag_state_null;
     mdag_recv_leader_verts := [];
     mdag_recv_certs := [];
     mdag_timeouts := [];
     mdag_jumps := [];
  |}.

  Definition first_in_closure (r nid : nat) (closure : list nat) (udag : UDAG_State) :=
  first_elem closure 
             (fun id => match NatMap_find id udag.(udag_verts) with
                        | None => false
                        | Some v => (v.(udag_vert_round) =? r) && (v.(udag_vert_builder) =? nid)
                        end = true).

  (* The definition of "support" according to the paper *)

  Definition udag_supports (id id' : nat) (udag : UDAG_State) :=
  match NatMap_find id' udag.(udag_verts) with
  | None => False
  | Some v =>
    first_in_closure v.(udag_vert_round) v.(udag_vert_builder) (udag_get_closure id udag) udag = Some id'
  end.

  (* In practice, however, we almost only care about supporters
     in the immediate next round. In this case the definition of
     "support" can be simplified
   *)

  Definition udag_supports' (id id' : nat) (udag : UDAG_State) :=
  match NatMap_find id udag.(udag_verts) with
  | None => False
  | Some v =>
    match NatMap_find id' udag.(udag_verts) with
    | None => False
    | Some v' => first_in_closure v'.(udag_vert_round) v'.(udag_vert_builder) v.(udag_vert_preds) udag = Some id'
    end
  end.

  Instance udag_supports_dec (id id' : nat) (udag : UDAG_State) : Decision (udag_supports id id' udag).
  Proof. unfold udag_supports.
         match goal with |- context[match ?x with _ => _ end] => destruct x end.
         all: typeclasses eauto.
  Qed.

  Instance udag_supports'_dec (id id' : nat) (udag : UDAG_State) : Decision (udag_supports' id id' udag).
  Proof. unfold udag_supports'.
         match goal with |- context[match ?x with _ => _ end] => destruct x end.
         2: typeclasses eauto.
         match goal with |- context[match ?x with _ => _ end] => destruct x end.
         all: typeclasses eauto.
  Qed.

  Definition mdag_add_vert (id : nat) (vert : UDAG_Vert) (mdag : MDAG_State) :=
  {| mdag_udag := udag_add_vert id vert mdag.(mdag_udag);
     mdag_recv_leader_verts := mdag.(mdag_recv_leader_verts);
     mdag_recv_certs := mdag.(mdag_recv_certs);
     mdag_timeouts := mdag.(mdag_timeouts);
     mdag_jumps := mdag.(mdag_jumps);
  |}.

  Definition mdag_add_vert_and_timeout (id : nat) (vert : UDAG_Vert) (mdag : MDAG_State) :=
  {| mdag_udag := udag_add_vert id vert mdag.(mdag_udag);
     mdag_recv_leader_verts := mdag.(mdag_recv_leader_verts);
     mdag_recv_certs := mdag.(mdag_recv_certs);
     mdag_timeouts := (vert.(udag_vert_builder), vert.(udag_vert_round)) :: mdag.(mdag_timeouts);
     mdag_jumps := mdag.(mdag_jumps);
  |}.

  Definition mdag_add_recv_leader_vert (nid r : nat) (mdag : MDAG_State) :=
  {| mdag_udag := mdag.(mdag_udag);
     mdag_recv_leader_verts := (nid, r) :: mdag.(mdag_recv_leader_verts);
     mdag_recv_certs := mdag.(mdag_recv_certs);
     mdag_timeouts := mdag.(mdag_timeouts);
     mdag_jumps := mdag.(mdag_jumps);
  |}.

  Definition mdag_add_recv_cert (nid r : nat) (mdag : MDAG_State) :=
  {| mdag_udag := mdag.(mdag_udag);
     mdag_recv_leader_verts := mdag.(mdag_recv_leader_verts);
     mdag_recv_certs := (nid, r) :: mdag.(mdag_recv_certs);
     mdag_timeouts := mdag.(mdag_timeouts);
     mdag_jumps := mdag.(mdag_jumps);
  |}.

  Definition mdag_add_jump (nid r : nat) (mdag : MDAG_State) :=
  {| mdag_udag := mdag.(mdag_udag);
     mdag_recv_leader_verts := mdag.(mdag_recv_leader_verts);
     mdag_recv_certs := mdag.(mdag_recv_certs);
     mdag_timeouts := mdag.(mdag_timeouts);
     mdag_jumps := (nid, r) :: mdag.(mdag_jumps);
  |}.

  Context `{participant : !BADO_Participant} `{leader : !BADO_Leader} `{node_assump : !BADO_NodeAssump} `{config : !BADO_Config}.

  Lemma udag_supports_equiv (id id' : nat) (udag : UDAG_State) :
  udag_valid udag ->
  match NatMap_find id udag.(udag_verts) with
  | None => True
  | Some v =>
    match NatMap_find id' udag.(udag_verts) with
    | None => True
    | Some v' => v.(udag_vert_round) = (v'.(udag_vert_round) + 1)%nat -> udag_supports id id' udag <-> udag_supports' id id' udag
    end
  end.
  Proof. intros Hval.
         destruct (NatMap_find id (udag_verts udag)) eqn:Evert; auto.
         destruct (NatMap_find id' (udag_verts udag)) eqn:Evert'; auto.
         intros Hr.
         unfold udag_supports, udag_supports'.
         rewrite Evert, Evert'.
         unfold first_in_closure.
         match goal with |- context[first_elem _ ?p] => assert (Himp : forall l x, In x l -> p x -> match (NatMap_find x (udag_verts udag)) with Some v => v.(udag_vert_round) =? u0.(udag_vert_round) | None => false end = true) end.
         { intros l x Hx.
           destruct (NatMap_find x (udag_verts udag)); auto.
           intros Htrue; arith_simpl Htrue.
           rewrite Nat.eqb_eq; apply Htrue.
         }
         rewrite (first_elem_filt (udag_get_closure id udag) _ _ (Himp (udag_get_closure id udag))).
         rewrite (first_elem_filt (udag_vert_preds u) _ _ (Himp (udag_vert_preds u))).
         clear Himp.
         match goal with |- context[filter ?f _] => assert (Heq_l : filter f (udag_get_closure id udag) = filter f (udag_vert_preds u)) end.
         2: rewrite Heq_l; split; auto.
         pose proof (udag_closure_filt _ id Hval) as Hfilt.
         rewrite Evert in Hfilt.
         replace (udag_vert_round u - 1) with (udag_vert_round u0) in Hfilt by lia.
         apply Hfilt.
  Qed.

  Lemma udag_subdag_first_in_closure_eq (udag udag' : UDAG_State) (id : nat) (l : list nat) (r nid : nat) :
  udag_subdag udag udag' ->
  NatMap_find id udag'.(udag_verts) <> None ->
  incl l (udag_get_closure id udag) ->
  first_in_closure r nid l udag' = first_in_closure r nid l udag.
  Proof. intros Hsub Hex Hincl.
         pose proof Hsub as Hsub'.
         destruct Hsub as (Hval1 & Hval2 & Heq).
         specialize (Heq id) as Heq1.
         destruct (NatMap_find id (udag_verts udag)) eqn:Evert; try contradiction.
         destruct (NatMap_find id (udag_verts udag')) eqn:Evert'; try contradiction.
         subst u0; clear Hex.
         unfold first_in_closure.
         apply first_elem_equiv.
         intros x Hx.
         specialize (Hincl _ Hx).
         destruct (NatMap_find x (udag_verts udag)) eqn:Evert''.
         2: specialize (Heq x); rewrite Evert'' in Heq; rewrite Heq; split; auto.
         destruct (NatMap_find x (udag_verts udag')) eqn:Evert'''.
         2: pose proof (udag_subdag_closure_ex _ _ Hsub' _ ltac:(rewrite Evert'; discriminate) _ Hincl); contradiction.
         specialize (Heq x); rewrite Evert'', Evert''' in Heq; subst u1.
         split; auto.
  Qed.

  Lemma udag_subdag_supports (id id' : nat) (udag udag' : UDAG_State) :
  udag_subdag udag udag' ->
  NatMap_find id udag'.(udag_verts) <> None ->
  udag_supports id id' udag <-> udag_supports id id' udag'.
  Proof. intros Hsub Hex.
         pose proof Hsub as Hsub'.
         unfold udag_supports.
         pose proof (udag_subdag_closure _ _ Hsub id) as Hclos_eq.
         unfold udag_subdag in Hsub.
         destruct Hsub as (Hval1 & Hval2 & Heq).
         specialize (Heq id) as Heq'.
         destruct (NatMap_find id (udag_verts udag)); try contradiction.
         destruct (NatMap_find id (udag_verts udag')) eqn:Evert; try contradiction.
         subst u0; clear Hex.
         rewrite <- Hclos_eq.
         destruct (decide (In id' (udag_get_closure id udag))).
         - pose proof (udag_subdag_closure_ex _ _ Hsub' _ ltac:(rewrite Evert; discriminate) _ i) as Hex.
           specialize (Heq id').
           destruct (NatMap_find id' (udag_verts udag)); try contradiction.
           destruct (NatMap_find id' (udag_verts udag')); try contradiction; subst u1.
           symmetry; erewrite udag_subdag_first_in_closure_eq.
           2: apply Hsub'.
           2: rewrite Evert; discriminate.
           2: apply incl_refl.
           split; auto.
         - (* If id' is not in closure, then first_in_closure surely cannot return Some id' *)
           unfold first_in_closure.
           assert (Hneq : forall f {f_dec : forall x, Decision (f x)}, first_elem (udag_get_closure id udag) f <> Some id').
           { intros f f_dec.
             pose proof (first_elem_correct (udag_get_closure id udag) f) as Hcorrect.
             intros Hfirst_eq.
             specialize (Hcorrect _ Hfirst_eq).
             destruct Hcorrect; contradiction.
           }
           destruct (NatMap_find id' (udag_verts udag)); destruct (NatMap_find id' (udag_verts udag')).
           all: split; try contradiction.
           all: intros Hfalse; specialize (Hneq _ _ Hfalse); contradiction.
  Qed.

  (* Two versions of udag_certifies. The first one uses udag_supports, the second one uses udag_supports'. *)

  Definition udag_certifies (id id' : nat) (udag : UDAG_State) :=
  match NatMap_find id' udag.(udag_verts) with
  | None => False
  | Some v =>
    match NatMap_find id udag.(udag_verts) with
    | None => False
    | Some v' =>
      v'.(udag_vert_round) = (v.(udag_vert_round) + 2)%nat /\
      is_quorum bado_comm
      (map (fun id'' => match NatMap_find id'' udag.(udag_verts) with None => 0 | Some v'' => v''.(udag_vert_builder) end)
           (filter (fun id'' => match NatMap_find id'' udag.(udag_verts) with None => false | Some v'' => (v''.(udag_vert_round) =? v.(udag_vert_round) + 1)%nat && if decide (udag_supports id'' id' udag) then true else false end)
                   v'.(udag_vert_preds)))
    end
  end.

  Definition udag_certifies' (id id' : nat) (udag : UDAG_State) :=
  match NatMap_find id' udag.(udag_verts) with
  | None => False
  | Some v =>
    match NatMap_find id udag.(udag_verts) with
    | None => False
    | Some v' =>
      v'.(udag_vert_round) = (v.(udag_vert_round) + 2)%nat /\
      is_quorum bado_comm
      (map (fun id'' => match NatMap_find id'' udag.(udag_verts) with None => 0 | Some v'' => v''.(udag_vert_builder) end)
           (filter (fun id'' => match NatMap_find id'' udag.(udag_verts) with None => false | Some v'' => (v''.(udag_vert_round) =? v.(udag_vert_round) + 1)%nat && if decide (udag_supports' id'' id' udag) then true else false end)
                   v'.(udag_vert_preds)))
    end
  end.

  Lemma udag_certifies_equiv (id id' : nat) (udag : UDAG_State) :
  udag_valid udag ->
  udag_certifies id id' udag <-> udag_certifies' id id' udag.
  Proof. intros Hval.
         unfold udag_certifies, udag_certifies'.
         destruct (NatMap_find id' (udag_verts udag)) eqn:Evert.
         2: split; auto.
         destruct (NatMap_find id (udag_verts udag)).
         2: split; auto.
         match goal with |- (_ /\ is_quorum _ (map _ ?l1)) <-> (_ /\ is_quorum _ (map _ ?l2)) => assert (Heq : l1 = l2) end.
         2: rewrite Heq; split; auto.
         apply filter_ext_in.
         intros x _.
         destruct (NatMap_find x (udag_verts udag)) eqn:Ex; auto.
         destruct (udag_vert_round u1 =? udag_vert_round u + 1) eqn:Hround.
         all: cbn; auto.
         arith_simpl Hround.
         pose proof (udag_supports_equiv x id' udag Hval) as Hequiv.
         rewrite Evert, Ex in Hequiv.
         specialize (Hequiv Hround).
         destruct (decide (udag_supports x id' udag)).
         - rewrite Hequiv in u2.
           rewrite decide_true_if; auto.
         - rewrite Hequiv in n.
           rewrite decide_false_if; auto.
  Qed.

  Instance udag_certifies'_dec (id id' : nat) (udag : UDAG_State) : Decision (udag_certifies' id id' udag).
  Proof. unfold udag_certifies'.
         match goal with |- context[match ?x with _ => _ end] => destruct x end.
         2: typeclasses eauto.
         match goal with |- context[match ?x with _ => _ end] => destruct x end.
         all: typeclasses eauto.
  Qed.

  Definition mdag_vert_is_supporter (vert : UDAG_Vert) (mdag : MDAG_State) :=
  vert.(udag_vert_round) = 1 \/
  vert.(udag_vert_round) > 1 /\
  first_in_closure (vert.(udag_vert_round) - 1) (bado_leader_at (vert.(udag_vert_round) - 1)) vert.(udag_vert_preds) mdag.(mdag_udag) <> None.

  Definition mdag_vert_is_certificate (vert : UDAG_Vert) (mdag : MDAG_State) :=
  vert.(udag_vert_round) <= 2 \/
  vert.(udag_vert_round) > 2 /\
  Exists (fun pid =>
    match NatMap_find pid mdag.(mdag_udag).(udag_verts) with
    | None => False
    | Some v =>
      match first_in_closure (vert.(udag_vert_round) - 2) (bado_leader_at (vert.(udag_vert_round) - 2)) v.(udag_vert_preds) mdag.(mdag_udag) with
      | None => False
      | Some id =>
        is_quorum bado_comm (map (fun pid => match NatMap_find pid mdag.(mdag_udag).(udag_verts) with
                                             | None => 0
                                             | Some v' => v'.(udag_vert_builder)
                                             end)
                                 (filter (fun pid => (if decide (udag_supports' pid id mdag.(mdag_udag)) then true else false) && (match NatMap_find pid mdag.(mdag_udag).(udag_verts) with None => false | Some v' => v'.(udag_vert_round) =? vert.(udag_vert_round) - 1 end))
                                         vert.(udag_vert_preds)))
      end
    end
  ) vert.(udag_vert_preds).

  Definition mdag_vert_is_certificate' (vert : UDAG_Vert) (mdag : MDAG_State) :=
  vert.(udag_vert_round) <= 2 \/
  vert.(udag_vert_round) > 2 /\
  exists id quorum,
    incl quorum vert.(udag_vert_preds) /\
    match NatMap_find id mdag.(mdag_udag).(udag_verts) with None => False | Some v => v.(udag_vert_round) = vert.(udag_vert_round) - 2 /\ v.(udag_vert_builder) = bado_leader_at (vert.(udag_vert_round) - 2) end /\
    Forall (fun id' => match NatMap_find id' mdag.(mdag_udag).(udag_verts) with None => False | Some v => v.(udag_vert_round) = vert.(udag_vert_round) - 1 end) quorum /\
    is_quorum bado_comm (map (fun id' => match NatMap_find id' mdag.(mdag_udag).(udag_verts) with None => 0 | Some v => v.(udag_vert_builder) end) quorum) /\
    Forall (fun id' => udag_supports' id' id mdag.(mdag_udag)) quorum.

  Definition mdag_add_recv_leader_vert_pre (nid r : nat) (mdag : MDAG_State) :=
  (forall id, match NatMap_find id mdag.(mdag_udag).(udag_verts) with None => True | Some v => v.(udag_vert_builder) = nid -> v.(udag_vert_round) <> (r + 1)%nat end) /\
  exists id, match NatMap_find id mdag.(mdag_udag).(udag_verts) with None => False | Some v => v.(udag_vert_round) = r /\ v.(udag_vert_builder) = bado_leader_at r end.

  Definition mdag_add_recv_cert_pre (nid r : nat) (mdag : MDAG_State) :=
  (forall id, match NatMap_find id mdag.(mdag_udag).(udag_verts) with None => True | Some v => v.(udag_vert_builder) = nid -> v.(udag_vert_round) <> (r + 2)%nat end) /\
  exists id quorum,
    match NatMap_find id mdag.(mdag_udag).(udag_verts) with None => False | Some v => v.(udag_vert_round) = r /\ v.(udag_vert_builder) = bado_leader_at r end /\
    Forall (fun id' => match NatMap_find id' mdag.(mdag_udag).(udag_verts) with None => False | Some v => v.(udag_vert_round) = (r + 1)%nat end) quorum /\
    is_quorum bado_comm (map (fun id' => match NatMap_find id' mdag.(mdag_udag).(udag_verts) with None => 0 | Some v => v.(udag_vert_builder) end) quorum) /\
    Forall (fun id' => udag_supports' id' id mdag.(mdag_udag)) quorum.

  (* We shall prove liveness of Mysticeti in two steps.
     In the first step, we assume arbitrary round-jumping behavior.
     We prove that every honest leader vertex eventually gets f+1 honest certificates.
     Then we assume round-jumping occurs only under restricted conditions.
     We prove that every honest leader vertex is eventually committed.
   *)

  Definition mdag_can_add_vert (r : nat) (nid : nat) (mdag : MDAG_State) :=
  r = 1 \/ (exists id, match NatMap_find id mdag.(mdag_udag).(udag_verts) with None => False | Some v => v.(udag_vert_round) = r - 1 /\ v.(udag_vert_builder) = nid end) \/ In (nid, r - 1) mdag.(mdag_jumps).

  Definition mdag_jump_pre (r : nat) (mdag : MDAG_State) :=
  exists quorum,
  Forall (fun id => match NatMap_find id mdag.(mdag_udag).(udag_verts) with None => False | Some v => v.(udag_vert_round) = r end) quorum /\
  is_quorum bado_comm (map (fun id => match NatMap_find id mdag.(mdag_udag).(udag_verts) with None => 0 | Some v => v.(udag_vert_builder) end) quorum).

  (* First situation: the vertex added is a "perfect vertex", i.e.
     the vertex is simultaneously a supporter of the previous leader vertex,
     and a certificate for the second previous leader vertex.
   *)

  Definition mdag_add_vert_perfect_pre (id : nat) (vert : UDAG_Vert) (mdag : MDAG_State) :=
  udag_add_vert_pre id vert mdag.(mdag_udag) /\
  mdag_vert_is_supporter vert mdag /\
  mdag_vert_is_certificate vert mdag.

  (* Second situation: the vertex is added upon timeout. *)

  Definition mdag_add_vert_timeout_pre (id : nat) (vert : UDAG_Vert) (mdag : MDAG_State) :=
  udag_add_vert_pre id vert mdag.(mdag_udag) /\
  (In (vert.(udag_vert_builder), vert.(udag_vert_round) - 1) mdag.(mdag_recv_leader_verts) -> mdag_vert_is_supporter vert mdag) /\
  (In (vert.(udag_vert_builder), vert.(udag_vert_round) - 2) mdag.(mdag_recv_certs) -> mdag_vert_is_certificate vert mdag).

  (* Third situation: the vertex is added upon observing a quorum of vertices *)

  Definition mdag_add_vert_quorum_pre (id : nat) (vert : UDAG_Vert) (quorum : list nat) (mdag : MDAG_State) :=
  udag_add_vert_pre id vert mdag.(mdag_udag) /\
  Forall (fun id' => match NatMap_find id' mdag.(mdag_udag).(udag_verts) with None => False | Some v => v.(udag_vert_round) = vert.(udag_vert_round) end) quorum /\
  is_quorum bado_comm (map (fun id' => match NatMap_find id' mdag.(mdag_udag).(udag_verts) with None => 0 | Some v => v.(udag_vert_builder) end) quorum) /\
  (In (vert.(udag_vert_builder), vert.(udag_vert_round) - 1) mdag.(mdag_recv_leader_verts) -> mdag_vert_is_supporter vert mdag) /\
  (In (vert.(udag_vert_builder), vert.(udag_vert_round) - 2) mdag.(mdag_recv_certs) -> mdag_vert_is_certificate vert mdag) /\
  Forall (fun id' => match NatMap_find id' mdag.(mdag_udag).(udag_verts) with None => True | Some v => mdag_vert_is_supporter v mdag -> mdag_vert_is_supporter vert mdag end) quorum /\
  Forall (fun id' => match NatMap_find id' mdag.(mdag_udag).(udag_verts) with None => True | Some v => mdag_vert_is_certificate v mdag -> mdag_vert_is_certificate vert mdag end) quorum.

  (* Byzantine processes can add any valid vertex *)

  Definition mdag_add_vert_byz_pre (id : nat) (vert : UDAG_Vert) (mdag : MDAG_State) :=
  udag_add_vert_pre id vert mdag.(mdag_udag) /\
  bado_node_assump vert.(udag_vert_builder) = Byzantine.

  (* mdag_step' differs from mdag_step in that it does not allow round-jumping.
     This gives lower refinement layers better control over round-jumping.
   *)

  Inductive mdag_step : MDAG_State -> MDAG_State -> Prop :=
  | mdag_step_add_vert_perfect : forall id vert mdag, mdag_can_add_vert vert.(udag_vert_round) vert.(udag_vert_builder) mdag -> mdag_add_vert_perfect_pre id vert mdag -> mdag_step mdag (mdag_add_vert id vert mdag)
  | mdag_step_add_vert_timeout : forall id vert mdag, mdag_can_add_vert vert.(udag_vert_round) vert.(udag_vert_builder) mdag -> mdag_add_vert_timeout_pre id vert mdag -> mdag_step mdag (mdag_add_vert_and_timeout id vert mdag)
  | mdag_step_add_vert_quorum : forall id vert quorum mdag, mdag_can_add_vert vert.(udag_vert_round) vert.(udag_vert_builder) mdag -> mdag_add_vert_quorum_pre id vert quorum mdag -> mdag_step mdag (mdag_add_vert id vert mdag)
  | mdag_step_add_vert_byz : forall id vert mdag, mdag_add_vert_byz_pre id vert mdag -> mdag_step mdag (mdag_add_vert id vert mdag)
  | mdag_step_add_recv_leader_vert : forall nid r mdag, mdag_add_recv_leader_vert_pre nid r mdag -> mdag_step mdag (mdag_add_recv_leader_vert nid r mdag)
  | mdag_step_add_recv_cert : forall nid r mdag, mdag_add_recv_cert_pre nid r mdag -> mdag_step mdag (mdag_add_recv_cert nid r mdag)
  | mdag_step_add_jump : forall nid r mdag, mdag_can_add_vert r nid mdag -> mdag_jump_pre r mdag -> mdag_step mdag (mdag_add_jump nid r mdag).

  Inductive mdag_step' : MDAG_State -> MDAG_State -> Prop :=
  | mdag_step_add_vert_perfect' : forall id vert mdag, mdag_can_add_vert vert.(udag_vert_round) vert.(udag_vert_builder) mdag -> mdag_add_vert_perfect_pre id vert mdag -> mdag_step' mdag (mdag_add_vert id vert mdag)
  | mdag_step_add_vert_timeout' : forall id vert mdag, mdag_can_add_vert vert.(udag_vert_round) vert.(udag_vert_builder) mdag -> mdag_add_vert_timeout_pre id vert mdag -> mdag_step' mdag (mdag_add_vert_and_timeout id vert mdag)
  | mdag_step_add_vert_quorum' : forall id vert quorum mdag, mdag_can_add_vert vert.(udag_vert_round) vert.(udag_vert_builder) mdag -> mdag_add_vert_quorum_pre id vert quorum mdag -> mdag_step' mdag (mdag_add_vert id vert mdag)
  | mdag_step_add_vert_byz' : forall id vert mdag, mdag_add_vert_byz_pre id vert mdag -> mdag_step' mdag (mdag_add_vert id vert mdag)
  | mdag_step_add_recv_leader_vert' : forall nid r mdag, mdag_add_recv_leader_vert_pre nid r mdag -> mdag_step' mdag (mdag_add_recv_leader_vert nid r mdag)
  | mdag_step_add_recv_cert' : forall nid r mdag, mdag_add_recv_cert_pre nid r mdag -> mdag_step' mdag (mdag_add_recv_cert nid r mdag).

  Lemma mdag_step_imp (mdag mdag' : MDAG_State) :
  mdag_step' mdag mdag' -> mdag_step mdag mdag'.
  Proof. intros Hstep'.
         destruct Hstep' as [
           id vert mdag Hpre1 Hpre2 |
           id vert mdag Hpre1 Hpre2 |
           id vert quorum mdag Hpre1 Hpre2 |
           id vert mdag Hpre |
           nid r mdag Hpre |
           nid r mdag Hpre
         ].
         - apply mdag_step_add_vert_perfect; auto.
         - apply mdag_step_add_vert_timeout; auto.
         - eapply mdag_step_add_vert_quorum; auto; apply Hpre2.
         - apply mdag_step_add_vert_byz; auto.
         - apply mdag_step_add_recv_leader_vert; auto.
         - apply mdag_step_add_recv_cert; auto.
  Qed.

  Definition mdag_sem : Semantics := {|
    s_state := MDAG_State;
    s_init := mdag_state_null;
    s_step := mdag_step;
  |}.

End MysticetiDAG.

Notation mdag_valid st_mdag := (valid_state mdag_sem st_mdag).
Notation mdag_reachable st_mdag := (reachable mdag_sem st_mdag).
Ltac mdag_unfold := unfold mdag_add_vert, mdag_add_vert_and_timeout, mdag_add_recv_leader_vert, mdag_add_recv_cert, mdag_add_jump.
Ltac mdag_reduce := cbn beta delta [mdag_udag mdag_recv_leader_verts mdag_recv_certs mdag_timeouts mdag_jumps] iota.

Ltac mdag_step_case Hstep :=
  destruct Hstep as [
    id vert mdag Hpre1 Hpre2 |
    id vert mdag Hpre1 Hpre2 |
    id vert quorum mdag Hpre1 Hpre2 |
    id vert mdag Hpre |
    nid r mdag Hpre |
    nid r mdag Hpre |
    nid r mdag Hpre1 Hpre2
  ].

Ltac mdag_step_case' Hstep :=
  destruct Hstep as [
    id vert mdag Hpre1 Hpre2 |
    id vert mdag Hpre1 Hpre2 |
    id vert quorum mdag Hpre1 Hpre2 |
    id vert mdag Hpre |
    nid r mdag Hpre |
    nid r mdag Hpre
  ].

Section MDAGProps.
  Context `{config : !BADO_Config} `{dag_config : !DAG_Config} `{participant : !BADO_Participant} `{leader : !BADO_Leader} `{node_assump : !BADO_NodeAssump} `{assump : !BADO_Assump}.

  Lemma mdag_vert_is_certificate_equiv (vert : UDAG_Vert) (mdag : MDAG_State) :
  mdag_vert_is_certificate vert mdag <-> mdag_vert_is_certificate' vert mdag.
  Proof. split.
         - intros Hcert.
           unfold mdag_vert_is_certificate in Hcert.
           unfold mdag_vert_is_certificate'.
           destruct Hcert as [Hcert | (Hge & Hcert)].
           1: left; auto.
           right; split; auto.
           rewrite Exists_exists in Hcert.
           destruct Hcert as (x & Hx1 & Hx2).
           cond_case_auto Hx2; try contradiction.
           match type of Hx2 with is_quorum _ (map _ ?l) => remember l as quorum end.
           unfold first_in_closure in Ex0.
           apply first_elem_correct in Ex0.
           destruct Ex0 as (Hn1 & Hn2).
           cond_case_auto Hn2; try discriminate.
           exists n; exists quorum.
           repeat split.
           + subst quorum; intros y Hy; rewrite filter_In in Hy; apply Hy.
           + rewrite Ex0; auto.
           + rewrite Forall_forall.
             intros y Hy.
             rewrite Heqquorum in Hy.
             rewrite filter_In in Hy.
             destruct Hy as (_ & Hy).
             cond_case_auto Hy; try (destruct Hy; discriminate).
             apply Hy.
           + apply Hx2.
           + rewrite Forall_forall.
             intros y Hy.
             rewrite Heqquorum in Hy.
             rewrite filter_In in Hy.
             destruct Hy as (_ & Hy).
             cond_case_auto Hy; try (destruct Hy; discriminate).
             auto.

         - intros Hcert.
           unfold mdag_vert_is_certificate' in Hcert.
           unfold mdag_vert_is_certificate.
           destruct Hcert as [Hcert | (Hge & Hcert)].
           1: left; auto.
           right; split; auto.
           destruct Hcert as (id & quorum & Hquorum1 & Hquorum2 & Hquorum3 & Hquorum4 & Hquorum5).
           cond_case_auto Hquorum2; try contradiction.
           pose proof (quorum_exists_honest bado_comm_safe _ Hquorum4) as (nid & Hnid & _).
           rewrite Forall_forall in *.
           rewrite in_map_iff in Hnid.
           destruct Hnid as (x & Hx1 & Hx2).
           specialize (Hquorum1 _ Hx2) as Hx3.
           pose proof (Hquorum3 _ Hx2) as Hx4.
           cond_case_auto Hx4; try contradiction; subst nid.
           pose proof (Hquorum5 _ Hx2) as Hx6.
           unfold udag_supports' in Hx6.
           rewrite Ex, Ex0 in Hx6.
           rewrite Exists_exists.
           exists x; split; auto.
           rewrite Ex0.
           destruct Hquorum2 as (Hquorum2 & Hquorum6).
           rewrite <- Hquorum2 in Hquorum6.
           rewrite <- Hquorum2; rewrite <- Hquorum6; rewrite Hx6.
           clear x u0 Ex0 Hx2 Hx3 Hx4 Hx6.
           eapply is_quorum_superset.
           1: apply Hquorum4.
           intros x Hx.
           rewrite in_map_iff in Hx.
           destruct Hx as (y & Hy1 & Hy2).
           specialize (Hquorum3 _ Hy2); cond_case_auto Hquorum3; try contradiction; subst x.
           rewrite in_map_iff; exists y; split.
           1: rewrite Ex0; auto.
           rewrite filter_In; split; auto.
           specialize (Hquorum5 _ Hy2).
           rewrite decide_true_if; auto.
           rewrite Ex0.
           rewrite <- Nat.eqb_eq in Hquorum3.
           rewrite Hquorum3; auto.
  Qed.

  Lemma mdag_certificate_certifies (id : nat) (vert : UDAG_Vert) (mdag : MDAG_State) :
  NatMap_find id mdag.(mdag_udag).(udag_verts) = Some vert ->
  mdag_vert_is_certificate vert mdag <-> (vert.(udag_vert_round) <= 2 \/ vert.(udag_vert_round) > 2 /\ exists id', match NatMap_find id' mdag.(mdag_udag).(udag_verts) with None => False | Some v => v.(udag_vert_round) = vert.(udag_vert_round) - 2 /\ v.(udag_vert_builder) = bado_leader_at (vert.(udag_vert_round) - 2) end /\ udag_certifies' id id' mdag.(mdag_udag)).
  Proof. intros Hvert; split.
         - intros Hcert.
           rewrite mdag_vert_is_certificate_equiv in Hcert.
           unfold mdag_vert_is_certificate' in Hcert.
           destruct Hcert as [Hcert | (Hge & id' & quorum & Hcert)].
           1: left; auto.
           right; split; auto.
           exists id'.
           destruct Hcert as (Hcert1 & Hcert2 & Hcert3 & Hcert4 & Hcert5).
           split; auto.
           unfold udag_certifies'.
           cond_case_auto Hcert2; try contradiction.
           rewrite Hvert; split.
           1: lia.
           eapply is_quorum_superset; try apply Hcert4.
           intros x Hx.
           rewrite in_map_iff in *.
           destruct Hx as (y & Hy1 & Hy2).
           exists y; split; auto.
           rewrite Forall_forall in Hcert3.
           specialize (Hcert3 _ Hy2).
           cond_case_auto Hcert3; try contradiction.
           subst x.
           rewrite filter_In; split; auto.
           rewrite Ex0.
           rewrite Forall_forall in Hcert5.
           specialize (Hcert5 _ Hy2).
           rewrite decide_true_if; auto.
           match goal with |- ?x && _ = _ => replace x with true; auto end.
           symmetry; rewrite Nat.eqb_eq; lia.

         - intros Hcert.
           unfold mdag_vert_is_certificate.
           destruct Hcert as [Hcert | (Hge & Hcert)].
           1: left; auto.
           right; split; auto.
           destruct Hcert as (id' & Hcert1 & Hcert2).
           unfold udag_certifies' in Hcert2.
           rewrite Hvert in Hcert2; cond_case_auto Hcert1; try contradiction.
           destruct Hcert2 as (Hcert2 & Hcert3).
           pose proof (quorum_exists_honest bado_comm_safe _ Hcert3) as (nid & Hnid & _).
           rewrite in_map_iff in Hnid.
           destruct Hnid as (x & Hx1 & Hx2).
           rewrite filter_In in Hx2.
           destruct Hx2 as (Hx2 & Hx3).
           cond_case_auto Hx3; try discriminate.
           2: destruct Hx3; discriminate.
           clear p; subst nid.
           destruct Hx3 as (Hx3 & _).
           unfold udag_supports' in u0.
           rewrite Ex, Ex0 in u0.
           destruct Hcert1 as (Hcert1 & Hcert4).
           rewrite <- Hcert1 in Hcert4; rewrite <- Hcert1, <- Hcert4.
           rewrite Exists_exists; exists x; split; auto.
           rewrite Ex0, u0.
           eapply is_quorum_superset; try apply Hcert3.
           intros y Hy.
           rewrite in_map_iff in *.
           destruct Hy as (z & Hz1 & Hz2).
           exists z; split; auto.
           rewrite filter_In in *.
           destruct Hz2 as (Hz2 & Hz3).
           split; auto.
           cond_case_auto Hz3; try discriminate.
           2: destruct Hz3; discriminate.
           destruct Hz3 as (Hz3 & _).
           match goal with |- _ && ?x = _ => replace x with true; auto end.
           symmetry; rewrite Nat.eqb_eq; lia.
  Qed.

  Lemma udag_certifies_in_closure (id id' : nat) (udag : UDAG_State) :
  udag_valid udag ->
  udag_certifies id id' udag ->
  In id' (udag_get_closure id udag).
  Proof. intros Hval Hcert.
         unfold udag_certifies in Hcert.
         destruct (NatMap_find id' (udag_verts udag)); try contradiction.
         destruct (NatMap_find id (udag_verts udag)) eqn:Evert; try contradiction.
         destruct Hcert as (_ & Hquorum).
         pose proof (quorum_exists_honest bado_comm_safe _ Hquorum) as (nid & Hnid & _).
         rewrite in_map_iff in Hnid.
         destruct Hnid as (x & _ & Hx); clear nid.
         rewrite filter_In in Hx.
         destruct Hx as (Hx1 & Hx2).
         cond_case_auto Hx2; try discriminate.
         2: destruct Hx2; discriminate.
         destruct Hx2 as (Hx2 & _); clear p.
         unfold udag_supports in u1.
         cond_case_auto u1; try contradiction.
         unfold first_in_closure in u1.
         apply first_elem_correct in u1.
         destruct u1 as (u1 & _).
         pose proof (udag_closure_eq _ id Hval) as Hclos.
         rewrite Evert in Hclos.
         rewrite Hclos.
         unfold udag_compute_closure.
         rewrite join_app.
         apply in_or_app; left.
         eapply join_in.
         2: apply u1.
         rewrite in_map_iff.
         exists x; split; auto.
  Qed.

  Lemma udag_subdag_certifies (id id' : nat) (udag udag' : UDAG_State) :
  udag_subdag udag udag' ->
  NatMap_find id udag'.(udag_verts) <> None ->
  udag_certifies id id' udag <-> udag_certifies id id' udag'.
  Proof. intros Hsub Hex.
         pose proof Hsub as Hsub'.
         destruct Hsub as (Hval1 & Hval2 & Heq).
         specialize (Heq id) as Heq'.
         destruct (NatMap_find id (udag_verts udag')) eqn:Evert; try contradiction.
         destruct (NatMap_find id (udag_verts udag)) eqn:Evert'; try discriminate.
         subst u0; clear Hex.
         destruct (decide (In id' (udag_get_closure id udag))).
         2: { (* If id' is not in closure, then surely it cannot be certified *)
              pose proof (udag_subdag_closure _ _ Hsub' id) as Hclos_eq.
              rewrite Evert, Evert' in Hclos_eq.
              pose proof n as n'; rewrite Hclos_eq in n'; clear Hclos_eq.
              split; intros Hcert; pose proof (udag_certifies_in_closure id id') as Hin.
              - specialize (Hin _ Hval1 Hcert); contradiction.
              - specialize (Hin _ Hval2 Hcert); contradiction.
         }
         pose proof (udag_subdag_closure_ex _ _ Hsub' _ ltac:(rewrite Evert; discriminate) _ i) as Hex.
         specialize (Heq id') as Heq'.
         destruct (NatMap_find id' (udag_verts udag)) eqn:Evert''; try contradiction.
         destruct (NatMap_find id' (udag_verts udag')) eqn:Evert'''; try contradiction.
         subst u1; clear Hex.
         unfold udag_certifies.
         rewrite Evert'', Evert', Evert''', Evert.
         match goal with |- _ /\ is_quorum _ ?l1 <-> _ /\ is_quorum _ ?l2 => assert (Heq_l : l1 = l2) end.
         2: rewrite Heq_l; split; auto.
         pose proof (udag_pred_valid _ id Hval2) as Hpred_ex.
         rewrite Evert in Hpred_ex.
         assert (Hpred_eq : Forall (fun pid => NatMap_find pid (udag_verts udag') = NatMap_find pid (udag_verts udag)) (udag_vert_preds u)).
         { rewrite Forall_forall; intros x Hx.
           specialize (Heq x).
           rewrite Forall_forall in Hpred_ex; specialize (Hpred_ex x Hx).
           destruct (NatMap_find x (udag_verts udag)); try contradiction.
           destruct (NatMap_find x (udag_verts udag')); try contradiction.
           subst u2; auto.
         }
         match goal with |- map _ ?l1 = map _ ?l2 => assert (Heq_l : l1 = l2) end.
         { apply filter_ext_in.
           intros x Hx.
           rewrite Forall_forall in Hpred_eq; specialize (Hpred_eq _ Hx).
           destruct (NatMap_find x (udag_verts udag)); rewrite Hpred_eq; auto.
           pose proof (udag_subdag_supports x id' _ _ Hsub' ltac:(rewrite Hpred_eq; discriminate)) as Hequiv.
           destruct (decide (udag_supports x id' udag)).
           - rewrite Hequiv in u2; rewrite decide_true_if; auto.
           - rewrite Hequiv in n; rewrite decide_false_if; auto.
         }
         rewrite Heq_l; clear Heq_l.
         apply map_ext_in.
         intros x Hx.
         apply filter_In in Hx.
         destruct Hx as (Hx & _).
         rewrite Forall_forall in Hpred_eq.
         specialize (Hpred_eq _ Hx).
         destruct (NatMap_find x (udag_verts udag)); rewrite Hpred_eq; auto.
  Qed.

  Lemma udag_certifies_uniq (id id' : nat) (udag : UDAG_State) :
  udag_valid udag ->
  udag_certifies' id id' udag ->
  match NatMap_find id' udag.(udag_verts) with
  | None => False
  | Some v =>
    forall id'' id''', udag_certifies' id'' id''' udag ->
    match NatMap_find id''' udag.(udag_verts) with
    | None => False
    | Some v' =>
      v'.(udag_vert_round) = v.(udag_vert_round) -> v'.(udag_vert_builder) = v.(udag_vert_builder) -> id' = id'''
    end
  end.
  Proof. intros Hval Hcert.
         unfold udag_certifies' in Hcert.
         cond_case_auto Hcert; try contradiction.
         intros cid cid' Hcert'.
         unfold udag_certifies' in Hcert'.
         cond_case_auto Hcert'; try contradiction.
         intros Heq1 Heq2.
         destruct Hcert as (_ & Hquorum1).
         destruct Hcert' as (_ & Hquorum2).
         pose proof (quorum_overlap_exists_honest bado_comm_safe _ _ Hquorum1 Hquorum2) as (nid & Hnid1 & Hnid2 & Hnid3).
         clear Hquorum1 Hquorum2.
         rewrite in_map_iff in Hnid1, Hnid2.
         destruct Hnid1 as (x1 & Hx1_1 & Hx1_2).
         destruct Hnid2 as (x2 & Hx2_1 & Hx2_2).
         rewrite filter_In in Hx1_2, Hx2_2.
         destruct Hx1_2 as (Hx1_2 & Hx1_3).
         destruct Hx2_2 as (Hx2_2 & Hx2_3).
         cond_case_auto Hx1_3; try discriminate.
         2: destruct Hx1_3; discriminate.
         cond_case_auto Hx2_3; try discriminate.
         2: destruct Hx2_3; discriminate.
         clear p p0.
         pose proof (udag_honest_uniq _ x1 x2 Hval) as Huniq.
         rewrite Ex3, Ex4 in Huniq.
         specialize (Huniq ltac:(rewrite Hx1_1; auto) ltac:(lia) ltac:(lia)).
         subst x2.
         unfold udag_supports' in u3, u5.
         rewrite Ex, Ex3 in u3.
         rewrite Ex3, Ex1 in u5.
         rewrite Heq1, Heq2 in u5.
         rewrite u3 in u5.
         inversion u5; auto.
  Qed.

  Lemma mdag_udag_valid : forall mdag,
  mdag_valid mdag ->
  udag_valid mdag.(mdag_udag).
  Proof. intros mdag Hval.
         induction Hval as [| st st' Hval IH Hstep].
         - cbn; apply valid_state_init.
         - mdag_step_case Hstep.
           5,6,7: cbn; auto.
           all: cbn; eapply valid_reach_valid; try apply IH.
           all: eapply reachable_step; try apply reachable_self.
           all: constructor; try apply Hpre2; try apply Hpre.
  Qed.

  Lemma mdag_udag_reachable : forall mdag mdag',
  mdag_reachable mdag mdag' ->
  udag_reachable mdag.(mdag_udag) mdag'.(mdag_udag).
  Proof.
    intros mdag mdag' Hreach.
    induction Hreach as [| st st' st'' Hreach IH Hstep].
    - apply reachable_self.
    - eapply reachable_trans.
      1: apply IH.
      mdag_step_case Hstep; cbn; try apply reachable_self.
      all: eapply reachable_step; try apply reachable_self.
      all: constructor; try apply Hpre2; try apply Hpre.
  Qed.

  Lemma mdag_recv_leader_vert_mono : forall mdag mdag',
  mdag_reachable mdag mdag' ->
  incl mdag.(mdag_recv_leader_verts) mdag'.(mdag_recv_leader_verts).
  Proof. intros mdag mdag' Hreach.
         induction Hreach as [| st st' st'' Hreach IH Hstep].
         - apply incl_refl.
         - mdag_step_case Hstep.
           all: cbn; eapply incl_tran; [apply IH | try apply incl_tl; apply incl_refl].
  Qed.

  Lemma mdag_recv_cert_mono : forall mdag mdag',
  mdag_reachable mdag mdag' ->
  incl mdag.(mdag_recv_certs) mdag'.(mdag_recv_certs).
  Proof. intros mdag mdag' Hreach.
         induction Hreach as [| st st' st'' Hreach IH Hstep].
         - apply incl_refl.
         - mdag_step_case Hstep.
           all: cbn; eapply incl_tran; [apply IH | try apply incl_tl; apply incl_refl].
  Qed.

  Lemma mdag_timeout_mono : forall mdag mdag',
  mdag_reachable mdag mdag' ->
  incl mdag.(mdag_timeouts) mdag'.(mdag_timeouts).
  Proof. intros mdag mdag' Hreach.
         induction Hreach as [| st st' st'' Hreach IH Hstep].
         - apply incl_refl.
         - mdag_step_case Hstep.
           all: cbn; eapply incl_tran; [apply IH | try apply incl_tl; apply incl_refl].
  Qed.

  Lemma mdag_jump_mono : forall mdag mdag',
  mdag_reachable mdag mdag' ->
  incl mdag.(mdag_jumps) mdag'.(mdag_jumps).
  Proof. intros mdag mdag' Hreach.
         induction Hreach as [| st st' st'' Hreach IH Hstep].
         - apply incl_refl.
         - mdag_step_case Hstep.
           all: cbn; eapply incl_tran; [apply IH | try apply incl_tl; apply incl_refl].
  Qed.

  Lemma mdag_timeout_mem : forall mdag nid r,
  mdag_valid mdag ->
  In (nid, r) mdag.(mdag_timeouts) -> In nid bado_participant.
  Proof. intros mdag nid' r' Hval.
         induction Hval as [| st st' Hval IH Hstep].
         - cbn; contradiction.
         - mdag_step_case Hstep; try (cbn; apply IH).
           cbn; intros Htimeout; destruct Htimeout as [Hto | Hto].
           + inversion Hto; apply Hpre2.
           + apply IH; auto.
  Qed.

  Lemma mdag_vert_has_pred_or_jump : forall mdag id,
  mdag_valid mdag ->
  match NatMap_find id mdag.(mdag_udag).(udag_verts) with
  | None => True
  | Some v => bado_node_assump v.(udag_vert_builder) <> Byzantine ->
    v.(udag_vert_round) = 1 \/ v.(udag_vert_round) > 1 /\ ((exists id', match NatMap_find id' mdag.(mdag_udag).(udag_verts) with None => False | Some v' => v'.(udag_vert_builder) = v.(udag_vert_builder) /\ v'.(udag_vert_round) = v.(udag_vert_round) - 1 end) \/ In (v.(udag_vert_builder), v.(udag_vert_round) - 1) mdag.(mdag_jumps))
  end.
  Proof. intros mdag id Hval.
         revert id.
         induction Hval as [| st st' Hval IH Hstep].
         - cbn; auto.
         - mdag_step_case Hstep.
           5,6: cbn; apply IH.
           5: cbn; intros id; specialize (IH id); cond_case_auto IH; auto;
              intros Hhonest; specialize (IH Hhonest); destruct IH as [IH | (IH1 & [IH2 | IH2])];
              [left; auto | right; split; auto | right; split; auto].
           all: pose proof (udag_add_vert_pre_uniq _ _ _ ltac:(try apply Hpre2; try apply Hpre)) as Huniq.
           1,2,3: pose proof (udag_add_vert_pre_pos _ _ _ ltac:(apply Hpre2)) as Hpos.
           1,2,3: clear Hpre2; unfold mdag_can_add_vert in Hpre1.
           all: intros id'; mdag_unfold; mdag_reduce; udag_unfold; udag_reduce.
           all: NatMap_case id' id.
           2,4,6,8: specialize (IH id'); cond_case_auto IH; auto;
                    intros Hhonest; specialize (IH Hhonest);
                    destruct IH as [IH | (IH1 & [IH2 | IH2])]; [left; auto | right; split; auto; left | right; split; auto; right];
                    try (destruct IH2 as (id'' & Hid''); exists id''; cond_case_auto Hid''; try contradiction;
                    NatMap_case id'' id; [rewrite Huniq in Ex0; discriminate | rewrite Ex0; auto]).
           4: intros Hhonest; destruct Hpre; contradiction.
           all: intros Hhonest;
                destruct (decide (udag_vert_round vert > 1)).
           1,3,5: right; split; auto;
                  destruct Hpre1 as [? | Hpre1]; [lia|];
                  destruct Hpre1 as [(x & Hx) | Hin]; [left | right; auto]; exists x; cond_case_auto Hx; try contradiction;
                  NatMap_case x id; [rewrite Huniq in Ex; discriminate | rewrite Ex; lia].
           all: left; lia.
  Qed.

  Lemma mdag_jump_has_pred_or_jump : forall mdag nid r,
  mdag_valid mdag ->
  In (nid, r) mdag.(mdag_jumps) ->
  r = 1 \/ r > 1 /\ ((exists id', match NatMap_find id' mdag.(mdag_udag).(udag_verts) with None => False | Some v => v.(udag_vert_builder) = nid /\ v.(udag_vert_round) = r - 1 end) \/ In (nid, r - 1) mdag.(mdag_jumps)).
  Proof. intros mdag nid r Hval.
         revert nid r.
         induction Hval as [| st st' Hval IH Hstep].
         - cbn; intros; contradiction.
         - mdag_step_case Hstep.
           5,6: cbn; auto.
           1,2,3,4: cbn [mdag_add_vert mdag_add_vert_and_timeout mdag_jumps mdag_udag udag_add_vert udag_verts];
                    intros nid r Hin; specialize (IH _ _ Hin); destruct IH as [IH | (IH1 & [IH2 | IH2])];
                    [left; auto | right; split; auto; left | right; split; auto; right];
                    destruct IH2 as (x & Hx); cond_case_auto Hx; try contradiction; exists x;
                    pose proof (udag_add_vert_pre_uniq _ _ _ ltac:(try apply Hpre2; try apply Hpre)) as Huniq;
                    NatMap_case x id; try (rewrite Ex; auto); rewrite Ex in Huniq; discriminate.

           cbn.
           intros nid' r' Hin.
           destruct Hin as [Hin | Hin].
           2: specialize (IH _ _ Hin); destruct IH as [IH | (IH1 & [IH2 | IH2])]; [left; auto | right; split; auto | right; split; auto].
           inversion Hin; subst nid' r'; clear Hin.
           unfold mdag_can_add_vert in Hpre1.
           destruct Hpre1 as [Hpre1 | [Hpre1 | Hpre1]].
           + left; auto.
           + destruct Hpre1 as (x & Hx).
             cond_case_auto Hx; try contradiction.
             pose proof (udag_pos _ x (mdag_udag_valid _ Hval)) as Hpos.
             rewrite Ex in Hpos.
             right; split; [lia | left].
             exists x; rewrite Ex; split; apply Hx.
           + specialize (IH _ _ Hpre1).
             right; split; [lia | right; right; auto].
  Qed.

  Lemma mdag_vert_jump_history : forall mdag nid r,
  mdag_valid mdag ->
  bado_node_assump nid <> Byzantine ->
  ((exists id, match NatMap_find id mdag.(mdag_udag).(udag_verts) with
               | None => False
               | Some v => v.(udag_vert_round) = r /\ v.(udag_vert_builder) = nid
               end) \/
   In (nid, r) mdag.(mdag_jumps)) ->
  forall r', 1 <= r' <= r ->
  (exists id, match NatMap_find id mdag.(mdag_udag).(udag_verts) with
              | None => False
              | Some v => v.(udag_vert_round) = r' /\ v.(udag_vert_builder) = nid
              end) \/
  In (nid, r') mdag.(mdag_jumps).
  Proof. intros mdag nid r Hval Hhonest.
         induction r.
         - intros; lia.
         - intros Hvert r' Hr'.
           assert (Hr'' : r' = S r \/ 1 <= r' <= r) by lia.
           clear Hr'.
           destruct Hr'' as [Hr'' | Hr''].
           + subst r'; auto.
           + apply IHr; auto.
             destruct Hvert as [Hvert | Hjump].
             * destruct Hvert as (id & Hvert).
               cond_case_auto Hvert; try contradiction.
               pose proof (mdag_vert_has_pred_or_jump _ id Hval) as Hpred.
               rewrite Ex in Hpred.
               specialize (Hpred ltac:(destruct Hvert; congruence)).
               destruct Hpred as [? | (_ & Hpred)]; [lia|].
               replace (udag_vert_round u - 1) with r in Hpred by lia.
               destruct Hpred as [(id' & Hid') | Hjump].
               -- left; exists id'.
                  cond_case_auto Hid'; try contradiction.
                  lia.
               -- right; destruct Hvert; congruence.
             * pose proof (mdag_jump_has_pred_or_jump _ _ _ Hval Hjump) as Hpred.
               destruct Hpred as [? | (_ & Hpred)]; [lia|].
               replace (S r - 1) with r in Hpred by lia.
               destruct Hpred as [(id' & Hid') | Hjump'].
               -- left; exists id'.
                  cond_case_auto Hid'; try contradiction.
                  lia.
               -- right; auto.
  Qed.

  Lemma mdag_jump_has_quorum : forall mdag nid r,
  mdag_valid mdag ->
  In (nid, r) mdag.(mdag_jumps) ->
  exists quorum,
  Forall (fun id => match NatMap_find id mdag.(mdag_udag).(udag_verts) with None => False | Some v => v.(udag_vert_round) = r end) quorum /\
  is_quorum bado_comm (map (fun id => match NatMap_find id mdag.(mdag_udag).(udag_verts) with None => 0 | Some v => v.(udag_vert_builder) end) quorum).
  Proof. intros mdag nid r Hval.
         revert nid r.
         induction Hval as [| st st' Hval IH Hstep].
         - cbn; intros; contradiction.
         - pose proof Hstep as Hstep'.
           mdag_step_case Hstep.
           5,6: cbn; auto.
           5: cbn; intros nid' r' Hin;
              destruct Hin as [Hin | Hin]; [|apply (IH _ _ Hin)];
              inversion Hin; subst nid' r'; clear Hin; apply Hpre2.
           all: cbn [mdag_add_vert mdag_add_vert_and_timeout mdag_udag udag_add_vert udag_verts mdag_jumps].
           all: intros nid r Hin; specialize (IH _ _ Hin).
           all: pose proof (udag_add_vert_pre_uniq _ _ _ ltac:(try apply Hpre2; try apply Hpre)) as Huniq.
           all: destruct IH as (quorum' & IH1 & IH2); exists quorum'; split; rewrite Forall_forall in *.
           1,3,5,7: intros x Hx; specialize (IH1 _ Hx); cond_case_auto IH1; try contradiction;
                    NatMap_case x id; [|rewrite Ex; auto]; rewrite Ex in Huniq; discriminate.
           all: eapply is_quorum_superset; try apply IH2.
           all: intros x Hx; rewrite in_map_iff in Hx; destruct Hx as (y & Hy1 & Hy2).
           all: specialize (IH1 _ Hy2); cond_case_auto IH1; try contradiction; subst x.
           all: rewrite in_map_iff; exists y; split; auto.
           all: NatMap_case y id; [|rewrite Ex; auto]; rewrite Ex in Huniq; discriminate.
  Qed.

  Lemma mdag_vert_is_supporter_reachable : forall mdag mdag' v,
  mdag_valid mdag ->
  mdag_reachable mdag mdag' ->
  mdag_vert_is_supporter v mdag ->
  mdag_vert_is_supporter v mdag'.
  Proof.
    intros mdag mdag' v Hval Hreach Hspt.
    pose proof (mdag_udag_reachable _ _ Hreach) as Hreach'.

    unfold mdag_vert_is_supporter in *.
    destruct Hspt as [Hspt|[Hspt0 Hspt1]]; [left; auto|right; split; auto].
    unfold first_in_closure in *.
    match type of Hspt1 with ?x <> None => remember x as Hx; symmetry in HeqHx; destruct Hx; try contradiction; clear Hspt1 end.
    apply first_elem_correct in HeqHx; destruct HeqHx as (Hspt1 & Hspt2).
    apply first_elem_ex; rewrite Exists_exists.
    exists n; split; auto.
    cond_case_auto Hspt2; try discriminate.
    pose proof (udag_mono _ _ n Hreach' ltac:(rewrite Ex; discriminate)) as Heq.
    rewrite Heq; rewrite Ex.
    apply andb_true_intro; split; rewrite Nat.eqb_eq; apply Hspt2.
  Qed.

  Lemma mdag_vert_is_supporter_step : forall mdag mdag' v,
  mdag_valid mdag ->
  mdag_step mdag mdag' ->
  mdag_vert_is_supporter v mdag ->
  mdag_vert_is_supporter v mdag'.
  Proof. intros mdag mdag' v Hval Hstep.
         apply mdag_vert_is_supporter_reachable; auto.
         eapply reachable_step; [apply reachable_self | auto].
  Qed.

  Lemma mdag_vert_is_certificate_reachable : forall mdag mdag' v,
  mdag_valid mdag ->
  mdag_reachable mdag mdag' ->
  mdag_vert_is_certificate v mdag ->
  mdag_vert_is_certificate v mdag'.
  Proof.
    intros mdag mdag' v Hval Hreach Hctf.
    apply mdag_vert_is_certificate_equiv.
    apply mdag_vert_is_certificate_equiv in Hctf.
    unfold mdag_vert_is_certificate' in *.
    destruct Hctf as [Hctf|[Hctf0 [id' [quorum' [Hctf1 [Hctf2 [Hctf3 [Hctf4 Hctf5]]]]]]]]; [left; auto|right; split; auto].
    exists id', quorum'; repeat rewrite Forall_forall in *.
    assert (Hval' : mdag_valid mdag').
    { eapply valid_reach_valid; try apply Hreach; auto. }
    pose proof (mdag_udag_reachable _ _ Hreach) as Hreach'.
    pose proof (udag_mono (mdag_udag mdag) (mdag_udag mdag')) as Hmono.
    specialize (Hmono id' Hreach') as Hmono'.
    cond_case_auto Hctf2; try contradiction.
    specialize (Hmono' ltac:(discriminate)).
    repeat split.
    - auto.
    - rewrite Hmono'; auto.
    - intros x Hx; specialize (Hctf3 _ Hx).
      cond_case_auto Hctf3; try contradiction.
      specialize (Hmono x Hreach'); rewrite Ex0 in Hmono.
      specialize (Hmono ltac:(discriminate)).
      rewrite Hmono; auto.
    - eapply is_quorum_superset.
      1: apply Hctf4.
      intros x Hx.
      rewrite in_map_iff in *.
      destruct Hx as (y & Hy1 & Hy2); exists y; split; auto.
      specialize (Hctf3 _ Hy2).
      cond_case_auto Hctf3; try contradiction; subst x.
      specialize (Hmono y Hreach').
      rewrite Ex0 in Hmono.
      specialize (Hmono ltac:(discriminate)).
      rewrite Hmono; auto.
    - intros x Hx.
      specialize (Hctf5 _ Hx).
      unfold udag_supports' in *.
      cond_case_auto Hctf5; try contradiction.
      rewrite Hmono'.
      specialize (Hmono id' Hreach') as Hmono''.
      rewrite Ex1 in Hmono''.
      specialize (Hmono'' ltac:(discriminate)).
      inversion Ex; subst u1; clear Ex Hmono''.
      specialize (Hmono x Hreach').
      rewrite Ex0 in Hmono.
      specialize (Hmono ltac:(discriminate)).
      rewrite Hmono.
      pose proof (udag_reachable_subdag _ _ (mdag_udag_valid _ Hval) Hreach') as Hsub.
      pose proof (udag_subdag_first_in_closure_eq _ _ x (udag_vert_preds u0) (udag_vert_round u) (udag_vert_builder u) Hsub ltac:(rewrite Ex0; discriminate)) as Heq.
      rewrite <- Heq; auto.
      apply udag_closure_pred; auto; apply mdag_udag_valid; auto.
  Qed.

  Lemma mdag_vert_is_certificate_step : forall mdag mdag' v,
  mdag_valid mdag ->
  mdag_step mdag mdag' ->
  mdag_vert_is_certificate v mdag ->
  mdag_vert_is_certificate v mdag'.
  Proof. intros mdag mdag' v Hval Hstep.
         apply mdag_vert_is_certificate_reachable; auto.
         eapply reachable_step; [apply reachable_self | auto].
  Qed.

  Lemma mdag_vert_is_supporter_step_inv : forall mdag mdag' id v,
  mdag_valid mdag ->
  mdag_step mdag mdag' ->
  NatMap_find id mdag.(mdag_udag).(udag_verts) = Some v ->
  mdag_vert_is_supporter v mdag' ->
  mdag_vert_is_supporter v mdag.
  Proof.
    intros mdag mdag' id' v Hval Hstep Hid Hspt.
    assert (Hreach : mdag_reachable mdag mdag').
    { eapply reachable_step; [apply reachable_self | apply Hstep]. }
    pose proof (mdag_udag_reachable _ _ Hreach) as Hreach'.

    unfold mdag_vert_is_supporter in *.
    destruct Hspt as [Hspt|[Hspt0 Hspt1]]; [left; auto|right; split; auto].
    unfold first_in_closure in *.
    match type of Hspt1 with ?x <> None => remember x as Hx; symmetry in HeqHx; destruct Hx; try contradiction; clear Hspt1 end.
    apply first_elem_correct in HeqHx; destruct HeqHx as (Hspt1 & Hspt2).
    apply first_elem_ex; rewrite Exists_exists.
    exists n; split; auto.
    cond_case_auto Hspt2; try discriminate.
    pose proof (udag_pred_valid _ id' (mdag_udag_valid _ Hval)) as Hvalid.
    rewrite Hid in Hvalid.
    rewrite Forall_forall in Hvalid.
    specialize (Hvalid _ Hspt1).
    pose proof (udag_mono _ _ n Hreach' Hvalid) as Heq.
    rewrite <- Heq; rewrite Ex.
    apply andb_true_intro; split; rewrite Nat.eqb_eq; apply Hspt2.
  Qed.

  Lemma mdag_vert_is_certificate_reachable_inv : forall mdag mdag' id v,
  mdag_valid mdag ->
  mdag_reachable mdag mdag' ->
  NatMap_find id mdag.(mdag_udag).(udag_verts) = Some v ->
  mdag_vert_is_certificate v mdag' ->
  mdag_vert_is_certificate v mdag.
  Proof.
    intros mdag mdag' id v Hval Hreach Hid Hctf.
    assert (Hval' : mdag_valid mdag').
    { eapply valid_reach_valid; try apply Hval; auto. }
    assert (Hreach' : udag_reachable (mdag_udag mdag) (mdag_udag mdag')).
    { apply mdag_udag_reachable; auto. }

    unfold mdag_vert_is_certificate in *.
    destruct Hctf as [? | (? & Hctf)]; [left; auto | right; split; auto].
    rewrite Exists_exists in *.
    destruct Hctf as (x & Hx1 & Hx2).
    exists x; split; auto.
    pose proof (udag_pred_valid _ id (mdag_udag_valid _ Hval)) as Hvalid.
    rewrite Hid in Hvalid.
    rewrite Forall_forall in Hvalid.
    specialize (Hvalid _ Hx1) as Hvalid'.
    pose proof (udag_mono _ _ _ Hreach' Hvalid') as Heq.
    cond_case_auto Hx2; try contradiction.
    symmetry in Heq; rewrite Heq.
    pose proof (udag_reachable_subdag _ _ (mdag_udag_valid _ Hval) Hreach') as Hsub.
    rewrite <- (udag_subdag_first_in_closure_eq _ _ x (udag_vert_preds u) (udag_vert_round v - 2) (bado_leader_at (udag_vert_round v - 2)) Hsub Hvalid') in Ex0.
    2: apply udag_closure_pred; auto; apply mdag_udag_valid; auto.
    pose proof (first_elem_correct _ _ _ Ex0) as (Hn & _).
    pose proof (udag_pred_valid _ x (mdag_udag_valid _ Hval)) as Hvalid_n.
    rewrite Heq in Hvalid_n.
    rewrite Forall_forall in Hvalid_n.
    specialize (Hvalid_n _ Hn).
    pose proof (udag_mono _ _ _ Hreach' Hvalid_n) as Heq_n.

    rewrite Ex0.
    eapply is_quorum_superset; try apply Hx2.
    intros y Hy.
    rewrite in_map_iff in *.
    destruct Hy as (z & Hz1 & Hz2).
    rewrite filter_In in Hz2.
    destruct Hz2 as (Hz2 & Hz3).
    cond_case_auto Hz3; try (destruct Hz3; discriminate).
    clear p.
    specialize (Hvalid _ Hz2) as Hvalid''.
    pose proof (udag_mono _ _ _ Hreach' Hvalid'') as Heq'.
    rewrite Ex1 in Heq'; symmetry in Heq'.
    exists z; rewrite Heq'; split; auto.
    rewrite filter_In; split; auto.
    rewrite Heq'.
    rewrite decide_true_if.
    2: { unfold udag_supports' in *.
         rewrite Ex1 in u0; rewrite Heq'.
         rewrite Heq_n in u0.
         cond_case_auto u0; try contradiction; clear Hvalid_n.
         rewrite <- (udag_subdag_first_in_closure_eq _ _ z (udag_vert_preds u1) (udag_vert_round u2) (udag_vert_builder u2) Hsub ltac:(rewrite Heq'; discriminate)) in u0.
         2: apply udag_closure_pred; auto; apply mdag_udag_valid; auto.
         auto.
    }
    match goal with |- _ && ?x = true => replace x with true; auto end.
    symmetry; rewrite Nat.eqb_eq; apply Hz3.
  Qed.

  Lemma mdag_vert_is_certificate_step_inv : forall mdag mdag' id v,
  mdag_valid mdag ->
  mdag_step mdag mdag' ->
  NatMap_find id mdag.(mdag_udag).(udag_verts) = Some v ->
  mdag_vert_is_certificate v mdag' ->
  mdag_vert_is_certificate v mdag.
  Proof. intros mdag mdag' id v Hval Hstep.
         apply mdag_vert_is_certificate_reachable_inv; auto.
         eapply reachable_step; [apply reachable_self | auto].
  Qed.

  Lemma mdag_honest_vert : forall mdag id,
  mdag_valid mdag ->
  match NatMap_find id mdag.(mdag_udag).(udag_verts) with
  | None => True
  | Some v => bado_node_assump v.(udag_vert_builder) <> Byzantine ->
    (* Case 1 *)
    mdag_vert_is_supporter v mdag /\
    mdag_vert_is_certificate v mdag \/
    (* Case 2 *)
    In (v.(udag_vert_builder), v.(udag_vert_round)) mdag.(mdag_timeouts) /\
    (In (v.(udag_vert_builder), v.(udag_vert_round) - 1) mdag.(mdag_recv_leader_verts) -> mdag_vert_is_supporter v mdag) /\
    (In (v.(udag_vert_builder), v.(udag_vert_round) - 2) mdag.(mdag_recv_certs) -> mdag_vert_is_certificate v mdag) \/
    (* Case 3 *)
    exists quorum,
    Forall (fun id' => match NatMap_find id' mdag.(mdag_udag).(udag_verts) with None => False | Some v' => v'.(udag_vert_round) = v.(udag_vert_round) end) quorum /\
    is_quorum bado_comm (map (fun id' => match NatMap_find id' mdag.(mdag_udag).(udag_verts) with None => 0 | Some v => v.(udag_vert_builder) end) quorum) /\
    (In (v.(udag_vert_builder), v.(udag_vert_round) - 1) mdag.(mdag_recv_leader_verts) -> mdag_vert_is_supporter v mdag) /\
    (In (v.(udag_vert_builder), v.(udag_vert_round) - 2) mdag.(mdag_recv_certs) -> mdag_vert_is_certificate v mdag) /\
    Forall (fun id' => match NatMap_find id' mdag.(mdag_udag).(udag_verts) with None => True | Some v' => mdag_vert_is_supporter v' mdag -> mdag_vert_is_supporter v mdag end) quorum /\
    Forall (fun id' => match NatMap_find id' mdag.(mdag_udag).(udag_verts) with None => True | Some v' => mdag_vert_is_certificate v' mdag -> mdag_vert_is_certificate v mdag end) quorum
  end.
  Proof. intros mdag id Hval.
         revert id.
         induction Hval as [| st st' Hval IH Hstep].
         - cbn; auto.
         - pose proof Hstep as Hstep'; cbn in Hstep'.
           mdag_step_case Hstep.
           + intros id'; unfold mdag_add_vert at 1; cbn [mdag_udag]; unfold udag_add_vert at 1; cbn [udag_verts].
             NatMap_case id' id.
             * intros _; left; clear IH.
               unfold mdag_add_vert_perfect_pre in Hpre2.
               destruct Hpre2 as (_ & Hpre2 & Hpre3).
               split.
               -- eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
               -- eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
             * specialize (IH id'); cond_case_auto IH; auto.
               intros Hhonest; specialize (IH Hhonest).
               destruct IH as [IH | [IH | IH]].
               -- left.
                  destruct IH as (IH1 & IH2).
                  split.
                  ++ eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
                  ++ eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
               -- right; left; cbn; split.
                  1: apply IH.
                  destruct IH as (_ & IH1 & IH2).
                  split.
                  ++ intros; specialize (IH1 ltac:(auto)); eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
                  ++ intros; specialize (IH2 ltac:(auto)); eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
               -- right; right; cbn [mdag_add_vert mdag_recv_leader_verts mdag_recv_certs mdag_udag udag_verts udag_add_vert].
                  destruct IH as (quorum & Hquorum1 & Hquorum2 & Hvert1 & Hvert2 & Hquorum3 & Hquorum4).
                  pose proof (udag_add_vert_pre_uniq _ _ _ ltac:(apply Hpre2)) as Huniq.
                  exists quorum; repeat split.
                  ++ clear Hquorum2 Hvert1 Hvert2 Hquorum3 Hquorum4.
                     rewrite Forall_forall in *.
                     intros x Hx.
                     specialize (Hquorum1 _ Hx).
                     cond_case_auto Hquorum1; try contradiction.
                     NatMap_case x id.
                     1: rewrite Ex0 in Huniq; discriminate.
                     rewrite Ex0; auto.
                  ++ clear Hvert1 Hvert2 Hquorum3 Hquorum4.
                     match goal with Hl : is_quorum _ ?l1 |- is_quorum _ ?l2 => assert (Heq_l : l1 = l2) end.
                     2: rewrite <- Heq_l; auto.
                     clear Hquorum2.
                     apply map_ext_in.
                     rewrite Forall_forall in Hquorum1.
                     intros x Hx.
                     specialize (Hquorum1 _ Hx).
                     cond_case_auto Hquorum1; try contradiction.
                     NatMap_case x id.
                     1: rewrite Ex0 in Huniq; discriminate.
                     rewrite Ex0; auto.
                  ++ clear Hquorum1 Hquorum2 Hvert2 Hquorum3 Hquorum4.
                     intros Hin; specialize (Hvert1 Hin).
                     eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
                  ++ clear Hquorum1 Hquorum2 Hvert1 Hquorum3 Hquorum4.
                     intros Hin; specialize (Hvert2 Hin).
                     eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
                  ++ clear Hquorum2 Hvert1 Hvert2 Hquorum4.
                     rewrite Forall_forall in *.
                     intros x Hx.
                     specialize (Hquorum1 _ Hx).
                     specialize (Hquorum3 _ Hx).
                     cond_case_auto Hquorum1; try contradiction; clear Hquorum1.
                     NatMap_case x id.
                     1: rewrite Ex0 in Huniq; discriminate.
                     rewrite Ex0; intros Hsupp.
                     pose proof (mdag_vert_is_supporter_step_inv _ _ _ _ Hval Hstep' Ex0 Hsupp).
                     eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
                  ++ clear Hquorum2 Hvert1 Hvert2 Hquorum3.
                     rewrite Forall_forall in *.
                     intros x Hx.
                     specialize (Hquorum1 _ Hx).
                     specialize (Hquorum4 _ Hx).
                     cond_case_auto Hquorum1; try contradiction; clear Hquorum1.
                     NatMap_case x id.
                     1: rewrite Ex0 in Huniq; discriminate.
                     rewrite Ex0; intros Hcert.
                     pose proof (mdag_vert_is_certificate_step_inv _ _ _ _ Hval Hstep' Ex0 Hcert).
                     eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.

           + intros id'; unfold mdag_add_vert_and_timeout at 1; cbn [mdag_udag]; unfold udag_add_vert at 1; cbn [udag_verts].
             NatMap_case id' id.
             * intros _; right; left; cbn; split.
               1: left; auto.
               clear IH.
               destruct Hpre2 as (_ & Hpre2 & Hpre3).
               split.
               -- intros Hin; specialize (Hpre2 Hin); eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
               -- intros Hin; specialize (Hpre3 Hin); eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
             * specialize (IH id'); cond_case_auto IH; auto.
               intros Hhonest; specialize (IH Hhonest).
               destruct IH as [IH | [IH | IH]].
               -- left.
                  destruct IH as (IH1 & IH2).
                  split.
                  ++ eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
                  ++ eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
               -- right; left; cbn; split.
                  1: right; apply IH.
                  destruct IH as (_ & IH1 & IH2).
                  split.
                  ++ intros; specialize (IH1 ltac:(auto)); eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
                  ++ intros; specialize (IH2 ltac:(auto)); eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
               -- right; right; cbn [mdag_add_vert_and_timeout mdag_recv_leader_verts mdag_recv_certs mdag_udag udag_verts udag_add_vert].
                  destruct IH as (quorum & Hquorum1 & Hquorum2 & Hvert1 & Hvert2 & Hquorum3 & Hquorum4).
                  pose proof (udag_add_vert_pre_uniq _ _ _ ltac:(apply Hpre2)) as Huniq.
                  exists quorum; repeat split.
                  ++ clear Hquorum2 Hvert1 Hvert2 Hquorum3 Hquorum4.
                     rewrite Forall_forall in *.
                     intros x Hx.
                     specialize (Hquorum1 _ Hx).
                     cond_case_auto Hquorum1; try contradiction.
                     NatMap_case x id.
                     1: rewrite Ex0 in Huniq; discriminate.
                     rewrite Ex0; auto.
                  ++ clear Hvert1 Hvert2 Hquorum3 Hquorum4.
                     match goal with Hl : is_quorum _ ?l1 |- is_quorum _ ?l2 => assert (Heq_l : l1 = l2) end.
                     2: rewrite <- Heq_l; auto.
                     clear Hquorum2.
                     apply map_ext_in.
                     rewrite Forall_forall in Hquorum1.
                     intros x Hx.
                     specialize (Hquorum1 _ Hx).
                     cond_case_auto Hquorum1; try contradiction.
                     NatMap_case x id.
                     1: rewrite Ex0 in Huniq; discriminate.
                     rewrite Ex0; auto.
                  ++ clear Hquorum1 Hquorum2 Hvert2 Hquorum3 Hquorum4.
                     intros Hin; specialize (Hvert1 Hin).
                     eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
                  ++ clear Hquorum1 Hquorum2 Hvert1 Hquorum3 Hquorum4.
                     intros Hin; specialize (Hvert2 Hin).
                     eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
                  ++ clear Hquorum2 Hvert1 Hvert2 Hquorum4.
                     rewrite Forall_forall in *.
                     intros x Hx.
                     specialize (Hquorum1 _ Hx).
                     specialize (Hquorum3 _ Hx).
                     cond_case_auto Hquorum1; try contradiction; clear Hquorum1.
                     NatMap_case x id.
                     1: rewrite Ex0 in Huniq; discriminate.
                     rewrite Ex0; intros Hsupp.
                     pose proof (mdag_vert_is_supporter_step_inv _ _ _ _ Hval Hstep' Ex0 Hsupp).
                     eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
                  ++ clear Hquorum2 Hvert1 Hvert2 Hquorum3.
                     rewrite Forall_forall in *.
                     intros x Hx.
                     specialize (Hquorum1 _ Hx).
                     specialize (Hquorum4 _ Hx).
                     cond_case_auto Hquorum1; try contradiction; clear Hquorum1.
                     NatMap_case x id.
                     1: rewrite Ex0 in Huniq; discriminate.
                     rewrite Ex0; intros Hcert.
                     pose proof (mdag_vert_is_certificate_step_inv _ _ _ _ Hval Hstep' Ex0 Hcert).
                     eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.

           + intros id'; unfold mdag_add_vert at 1; cbn [mdag_udag]; unfold udag_add_vert at 1; cbn [udag_verts].
             NatMap_case id' id.
             * intros _; right; right; cbn [mdag_add_vert mdag_recv_leader_verts mdag_recv_certs mdag_udag udag_verts udag_add_vert].
               clear IH.
               destruct Hpre2 as (Hpre2 & Hquorum1 & Hquorum2 & Hvert1 & Hvert2 & Hquorum3 & Hquorum4).
               pose proof (udag_add_vert_pre_uniq _ _ _ ltac:(apply Hpre2)) as Huniq.
               exists quorum; repeat split.
               -- clear Hquorum2 Hvert1 Hvert2 Hquorum3 Hquorum4.
                  rewrite Forall_forall in *.
                  intros x Hx.
                  specialize (Hquorum1 _ Hx).
                  cond_case_auto Hquorum1; try contradiction.
                  NatMap_case x id.
                  1: rewrite Ex in Huniq; discriminate.
                  rewrite Ex; auto.
               -- clear Hvert1 Hvert2 Hquorum3 Hquorum4.
                  match goal with Hl : is_quorum _ ?l1 |- is_quorum _ ?l2 => assert (Heq_l : l1 = l2) end.
                  2: rewrite <- Heq_l; auto.
                  clear Hquorum2.
                  apply map_ext_in.
                  rewrite Forall_forall in Hquorum1.
                  intros x Hx.
                  specialize (Hquorum1 _ Hx).
                  cond_case_auto Hquorum1; try contradiction.
                  NatMap_case x id.
                  1: rewrite Ex in Huniq; discriminate.
                  rewrite Ex; auto.
               -- clear Hquorum1 Hquorum2 Hvert2 Hquorum3 Hquorum4.
                  intros Hin; specialize (Hvert1 Hin).
                  eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
               -- clear Hquorum1 Hquorum2 Hvert1 Hquorum3 Hquorum4.
                  intros Hin; specialize (Hvert2 Hin).
                  eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
               -- clear Hquorum2 Hvert1 Hvert2 Hquorum4.
                  rewrite Forall_forall in *.
                  intros x Hx.
                  specialize (Hquorum1 _ Hx).
                  specialize (Hquorum3 _ Hx).
                  cond_case_auto Hquorum1; try contradiction; clear Hquorum1.
                  NatMap_case x id.
                  1: rewrite Ex in Huniq; discriminate.
                  rewrite Ex; intros Hsupp.
                  pose proof (mdag_vert_is_supporter_step_inv _ _ _ _ Hval Hstep' Ex Hsupp).
                  eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
               -- clear Hquorum2 Hvert1 Hvert2 Hquorum3.
                     rewrite Forall_forall in *.
                     intros x Hx.
                     specialize (Hquorum1 _ Hx).
                     specialize (Hquorum4 _ Hx).
                     cond_case_auto Hquorum1; try contradiction; clear Hquorum1.
                     NatMap_case x id.
                     1: rewrite Ex in Huniq; discriminate.
                     rewrite Ex; intros Hcert.
                     pose proof (mdag_vert_is_certificate_step_inv _ _ _ _ Hval Hstep' Ex Hcert).
                     eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
             * specialize (IH id'); cond_case_auto IH; auto.
               intros Hhonest; specialize (IH Hhonest).
               destruct IH as [IH | [IH | IH]].
               -- left.
                  destruct IH as (IH1 & IH2).
                  split.
                  ++ eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
                  ++ eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
               -- right; left; cbn; split.
                  1: apply IH.
                  destruct IH as (_ & IH1 & IH2).
                  split.
                  ++ intros; specialize (IH1 ltac:(auto)); eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
                  ++ intros; specialize (IH2 ltac:(auto)); eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
               -- pose proof (udag_add_vert_pre_uniq _ _ _ ltac:(apply Hpre2)) as Huniq.
                  clear quorum Hpre2.
                  right; right; cbn [mdag_add_vert mdag_recv_leader_verts mdag_recv_certs mdag_udag udag_verts udag_add_vert].
                  destruct IH as (quorum & Hquorum1 & Hquorum2 & Hvert1 & Hvert2 & Hquorum3 & Hquorum4).
                  exists quorum; repeat split.
                  ++ clear Hquorum2 Hvert1 Hvert2 Hquorum3 Hquorum4.
                     rewrite Forall_forall in *.
                     intros x Hx.
                     specialize (Hquorum1 _ Hx).
                     cond_case_auto Hquorum1; try contradiction.
                     NatMap_case x id.
                     1: rewrite Ex0 in Huniq; discriminate.
                     rewrite Ex0; auto.
                  ++ clear Hvert1 Hvert2 Hquorum3 Hquorum4.
                     match goal with Hl : is_quorum _ ?l1 |- is_quorum _ ?l2 => assert (Heq_l : l1 = l2) end.
                     2: rewrite <- Heq_l; auto.
                     clear Hquorum2.
                     apply map_ext_in.
                     rewrite Forall_forall in Hquorum1.
                     intros x Hx.
                     specialize (Hquorum1 _ Hx).
                     cond_case_auto Hquorum1; try contradiction.
                     NatMap_case x id.
                     1: rewrite Ex0 in Huniq; discriminate.
                     rewrite Ex0; auto.
                  ++ clear Hquorum1 Hquorum2 Hvert2 Hquorum3 Hquorum4.
                     intros Hin; specialize (Hvert1 Hin).
                     eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
                  ++ clear Hquorum1 Hquorum2 Hvert1 Hquorum3 Hquorum4.
                     intros Hin; specialize (Hvert2 Hin).
                     eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
                  ++ clear Hquorum2 Hvert1 Hvert2 Hquorum4.
                     rewrite Forall_forall in *.
                     intros x Hx.
                     specialize (Hquorum1 _ Hx).
                     specialize (Hquorum3 _ Hx).
                     cond_case_auto Hquorum1; try contradiction; clear Hquorum1.
                     NatMap_case x id.
                     1: rewrite Ex0 in Huniq; discriminate.
                     rewrite Ex0; intros Hsupp.
                     pose proof (mdag_vert_is_supporter_step_inv _ _ _ _ Hval Hstep' Ex0 Hsupp).
                     eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
                  ++ clear Hquorum2 Hvert1 Hvert2 Hquorum3.
                     rewrite Forall_forall in *.
                     intros x Hx.
                     specialize (Hquorum1 _ Hx).
                     specialize (Hquorum4 _ Hx).
                     cond_case_auto Hquorum1; try contradiction; clear Hquorum1.
                     NatMap_case x id.
                     1: rewrite Ex0 in Huniq; discriminate.
                     rewrite Ex0; intros Hcert.
                     pose proof (mdag_vert_is_certificate_step_inv _ _ _ _ Hval Hstep' Ex0 Hcert).
                     eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.

           + intros id'; unfold mdag_add_vert at 1; cbn [mdag_udag]; unfold udag_add_vert at 1; cbn [udag_verts].
             NatMap_case id' id.
             * intros Hhonest.
               unfold mdag_add_vert_byz_pre in Hpre.
               destruct Hpre; contradiction.
             * specialize (IH id'); cond_case_auto IH; auto.
               intros Hhonest; specialize (IH Hhonest).
               destruct IH as [IH | [IH | IH]].
               -- left.
                  destruct IH as (IH1 & IH2).
                  split.
                  ++ eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
                  ++ eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
               -- right; left; cbn; split.
                  1: apply IH.
                  destruct IH as (_ & IH1 & IH2).
                  split.
                  ++ intros; specialize (IH1 ltac:(auto)); eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
                  ++ intros; specialize (IH2 ltac:(auto)); eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
               -- right; right; cbn [mdag_add_vert mdag_recv_leader_verts mdag_recv_certs mdag_udag udag_verts udag_add_vert].
                  destruct IH as (quorum & Hquorum1 & Hquorum2 & Hvert1 & Hvert2 & Hquorum3 & Hquorum4).
                  pose proof (udag_add_vert_pre_uniq _ _ _ ltac:(apply Hpre)) as Huniq.
                  exists quorum; repeat split.
                  ++ clear Hquorum2 Hvert1 Hvert2 Hquorum3 Hquorum4.
                     rewrite Forall_forall in *.
                     intros x Hx.
                     specialize (Hquorum1 _ Hx).
                     cond_case_auto Hquorum1; try contradiction.
                     NatMap_case x id.
                     1: rewrite Ex0 in Huniq; discriminate.
                     rewrite Ex0; auto.
                  ++ clear Hvert1 Hvert2 Hquorum3 Hquorum4.
                     match goal with Hl : is_quorum _ ?l1 |- is_quorum _ ?l2 => assert (Heq_l : l1 = l2) end.
                     2: rewrite <- Heq_l; auto.
                     clear Hquorum2.
                     apply map_ext_in.
                     rewrite Forall_forall in Hquorum1.
                     intros x Hx.
                     specialize (Hquorum1 _ Hx).
                     cond_case_auto Hquorum1; try contradiction.
                     NatMap_case x id.
                     1: rewrite Ex0 in Huniq; discriminate.
                     rewrite Ex0; auto.
                  ++ clear Hquorum1 Hquorum2 Hvert2 Hquorum3 Hquorum4.
                     intros Hin; specialize (Hvert1 Hin).
                     eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
                  ++ clear Hquorum1 Hquorum2 Hvert1 Hquorum3 Hquorum4.
                     intros Hin; specialize (Hvert2 Hin).
                     eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
                  ++ clear Hquorum2 Hvert1 Hvert2 Hquorum4.
                     rewrite Forall_forall in *.
                     intros x Hx.
                     specialize (Hquorum1 _ Hx).
                     specialize (Hquorum3 _ Hx).
                     cond_case_auto Hquorum1; try contradiction; clear Hquorum1.
                     NatMap_case x id.
                     1: rewrite Ex0 in Huniq; discriminate.
                     rewrite Ex0; intros Hsupp.
                     pose proof (mdag_vert_is_supporter_step_inv _ _ _ _ Hval Hstep' Ex0 Hsupp).
                     eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
                  ++ clear Hquorum2 Hvert1 Hvert2 Hquorum3.
                     rewrite Forall_forall in *.
                     intros x Hx.
                     specialize (Hquorum1 _ Hx).
                     specialize (Hquorum4 _ Hx).
                     cond_case_auto Hquorum1; try contradiction; clear Hquorum1.
                     NatMap_case x id.
                     1: rewrite Ex0 in Huniq; discriminate.
                     rewrite Ex0; intros Hcert.
                     pose proof (mdag_vert_is_certificate_step_inv _ _ _ _ Hval Hstep' Ex0 Hcert).
                     eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.

           + cbn.
             intros id.
             destruct (NatMap_find id mdag.(mdag_udag).(udag_verts)) eqn:Evert; auto.
             assert (Hu : (nid, r) <> (udag_vert_builder u, udag_vert_round u - 1)).
             { clear IH.
               destruct Hpre as (Hpre1 & Hpre2).
               specialize (Hpre1 id).
               rewrite Evert in Hpre1.
               intros Heq; inversion Heq; subst nid r; clear Heq.
               specialize (Hpre1 ltac:(auto)).
               destruct Hpre2 as (x & Hx).
               cond_case_auto Hx; try contradiction.
               pose proof (udag_pos _ x ltac:(apply mdag_udag_valid; apply Hval)) as Hpos.
               rewrite Ex in Hpos.
               lia.
             }
             specialize (IH id); rewrite Evert in IH.
             intros Hhonest; specialize (IH Hhonest).
             destruct IH as [IH | [IH | IH]].
             * left.
               destruct IH as (IH1 & IH2).
               split.
               -- eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
               -- eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
             * right; left; cbn; split.
               1: apply IH.
               destruct IH as (_ & IH1 & IH2).
               split.
               -- intros Hin; destruct Hin as [Hin | Hin]; try contradiction.
                  specialize (IH1 ltac:(auto)); eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
               -- intros; specialize (IH2 ltac:(auto)); eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
             * right; right; cbn [mdag_add_vert mdag_recv_leader_verts mdag_recv_certs mdag_udag udag_verts udag_add_vert].
               destruct IH as (quorum & Hquorum1 & Hquorum2 & Hvert1 & Hvert2 & Hquorum3 & Hquorum4).
               exists quorum; repeat split.
               -- clear Hquorum2 Hvert1 Hvert2 Hquorum3 Hquorum4.
                  rewrite Forall_forall in *.
                  intros x Hx.
                  specialize (Hquorum1 _ Hx).
                  cond_case_auto Hquorum1; try contradiction.
                  auto.
               -- clear Hvert1 Hvert2 Hquorum3 Hquorum4.
                  auto.
               -- clear Hquorum1 Hquorum2 Hvert2 Hquorum3 Hquorum4.
                  intros Hin; destruct Hin as [Hin | Hin]; try contradiction; specialize (Hvert1 Hin).
                  eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
               -- clear Hquorum1 Hquorum2 Hvert1 Hquorum3 Hquorum4.
                  intros Hin; specialize (Hvert2 Hin).
                  eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
               -- clear Hquorum2 Hvert1 Hvert2 Hquorum4.
                  rewrite Forall_forall in *.
                  intros x Hx.
                  specialize (Hquorum1 _ Hx).
                  specialize (Hquorum3 _ Hx).
                  cond_case_auto Hquorum1; try contradiction; clear Hquorum1.
                  intros Hsupp.
                  pose proof (mdag_vert_is_supporter_step_inv _ _ _ _ Hval Hstep' Ex Hsupp).
                  eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
               -- clear Hquorum2 Hvert1 Hvert2 Hquorum3.
                  rewrite Forall_forall in *.
                  intros x Hx.
                  specialize (Hquorum1 _ Hx).
                  specialize (Hquorum4 _ Hx).
                  cond_case_auto Hquorum1; try contradiction; clear Hquorum1.
                  intros Hcert.
                  pose proof (mdag_vert_is_certificate_step_inv _ _ _ _ Hval Hstep' Ex Hcert).
                  eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.

           + cbn.
             intros id.
             destruct (NatMap_find id mdag.(mdag_udag).(udag_verts)) eqn:Evert; auto.
             assert (Hu : (nid, r) <> (udag_vert_builder u, udag_vert_round u - 2)).
             { clear IH.
               destruct Hpre as (Hpre1 & Hpre2).
               specialize (Hpre1 id).
               rewrite Evert in Hpre1.
               intros Heq; inversion Heq; subst nid r; clear Heq.
               specialize (Hpre1 ltac:(auto)).
               destruct Hpre2 as (x & _ & Hx & _).
               cond_case_auto Hx; try contradiction.
               pose proof (udag_pos _ x ltac:(apply mdag_udag_valid; apply Hval)) as Hpos.
               rewrite Ex in Hpos.
               lia.
             }
             specialize (IH id); rewrite Evert in IH.
             intros Hhonest; specialize (IH Hhonest).
             destruct IH as [IH | [IH | IH]].
             * left.
               destruct IH as (IH1 & IH2).
               split.
               -- eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
               -- eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
             * right; left; cbn; split.
               1: apply IH.
               destruct IH as (_ & IH1 & IH2).
               split.
               -- intros; specialize (IH1 ltac:(auto)); eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
               -- intros Hin; destruct Hin as [Hin | Hin]; try contradiction.
                  specialize (IH2 ltac:(auto)); eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
             * right; right; cbn [mdag_add_vert mdag_recv_leader_verts mdag_recv_certs mdag_udag udag_verts udag_add_vert].
               destruct IH as (quorum & Hquorum1 & Hquorum2 & Hvert1 & Hvert2 & Hquorum3 & Hquorum4).
               exists quorum; repeat split.
               -- clear Hquorum2 Hvert1 Hvert2 Hquorum3 Hquorum4.
                  rewrite Forall_forall in *.
                  intros x Hx.
                  specialize (Hquorum1 _ Hx).
                  cond_case_auto Hquorum1; try contradiction.
                  auto.
               -- clear Hvert1 Hvert2 Hquorum3 Hquorum4.
                  auto.
               -- clear Hquorum1 Hquorum2 Hvert2 Hquorum3 Hquorum4.
                  intros Hin; specialize (Hvert1 Hin).
                  eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
               -- clear Hquorum1 Hquorum2 Hvert1 Hquorum3 Hquorum4.
                  intros Hin; destruct Hin as [Hin | Hin]; try contradiction; specialize (Hvert2 Hin).
                  eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
               -- clear Hquorum2 Hvert1 Hvert2 Hquorum4.
                  rewrite Forall_forall in *.
                  intros x Hx.
                  specialize (Hquorum1 _ Hx).
                  specialize (Hquorum3 _ Hx).
                  cond_case_auto Hquorum1; try contradiction; clear Hquorum1.
                  intros Hsupp.
                  pose proof (mdag_vert_is_supporter_step_inv _ _ _ _ Hval Hstep' Ex Hsupp).
                  eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
               -- clear Hquorum2 Hvert1 Hvert2 Hquorum3.
                  rewrite Forall_forall in *.
                  intros x Hx.
                  specialize (Hquorum1 _ Hx).
                  specialize (Hquorum4 _ Hx).
                  cond_case_auto Hquorum1; try contradiction; clear Hquorum1.
                  intros Hcert.
                  pose proof (mdag_vert_is_certificate_step_inv _ _ _ _ Hval Hstep' Ex Hcert).
                  eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.

           + cbn.
             intros id.
             destruct (NatMap_find id mdag.(mdag_udag).(udag_verts)) eqn:Evert; auto.
             specialize (IH id); rewrite Evert in IH.
             intros Hhonest; specialize (IH Hhonest).
             destruct IH as [IH | [IH | IH]].
             * left.
               destruct IH as (IH1 & IH2).
               split.
               -- eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
               -- eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
             * right; left; cbn; split.
               1: apply IH.
               destruct IH as (_ & IH1 & IH2).
               split.
               -- intros; specialize (IH1 ltac:(auto)); eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
               -- intros; specialize (IH2 ltac:(auto)); eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
             * right; right; cbn [mdag_add_vert mdag_recv_leader_verts mdag_recv_certs mdag_udag udag_verts udag_add_vert].
               destruct IH as (quorum & Hquorum1 & Hquorum2 & Hvert1 & Hvert2 & Hquorum3 & Hquorum4).
               exists quorum; repeat split.
               -- clear Hquorum2 Hvert1 Hvert2 Hquorum3 Hquorum4.
                  rewrite Forall_forall in *.
                  intros x Hx.
                  specialize (Hquorum1 _ Hx).
                  cond_case_auto Hquorum1; try contradiction.
                  auto.
               -- clear Hvert1 Hvert2 Hquorum3 Hquorum4.
                  auto.
               -- clear Hquorum1 Hquorum2 Hvert2 Hquorum3 Hquorum4.
                  intros Hin; specialize (Hvert1 Hin).
                  eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
               -- clear Hquorum1 Hquorum2 Hvert1 Hquorum3 Hquorum4.
                  intros Hin; specialize (Hvert2 Hin).
                  eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
               -- clear Hquorum2 Hvert1 Hvert2 Hquorum4.
                  rewrite Forall_forall in *.
                  intros x Hx.
                  specialize (Hquorum1 _ Hx).
                  specialize (Hquorum3 _ Hx).
                  cond_case_auto Hquorum1; try contradiction; clear Hquorum1.
                  intros Hsupp.
                  pose proof (mdag_vert_is_supporter_step_inv _ _ _ _ Hval Hstep' Ex Hsupp).
                  eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
               -- clear Hquorum2 Hvert1 Hvert2 Hquorum3.
                  rewrite Forall_forall in *.
                  intros x Hx.
                  specialize (Hquorum1 _ Hx).
                  specialize (Hquorum4 _ Hx).
                  cond_case_auto Hquorum1; try contradiction; clear Hquorum1.
                  intros Hcert.
                  pose proof (mdag_vert_is_certificate_step_inv _ _ _ _ Hval Hstep' Ex Hcert).
                  eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
  Qed.

  Lemma mdag_honest_vert_no_timeout_perfect :
  forall mdag id,
  mdag_valid mdag ->
  match NatMap_find id mdag.(mdag_udag).(udag_verts) with
  | None => True
  | Some v =>
    bado_node_assump v.(udag_vert_builder) = Synchronous ->
    Forall (fun timeout => bado_node_assump (fst timeout) = Synchronous -> snd timeout <> v.(udag_vert_round)) mdag.(mdag_timeouts) ->
    mdag_vert_is_supporter v mdag /\ mdag_vert_is_certificate v mdag
  end.
  Proof. intros mdag id Hval.
         revert id.
         induction Hval as [| st st' Hval IH Hstep].
         - cbn; auto.
         - pose proof Hstep as Hstep'; cbn in Hstep'.
           mdag_step_case Hstep.
           + intros id'; unfold mdag_add_vert at 1; cbn [mdag_udag]; unfold udag_add_vert at 1; cbn [udag_verts].
             NatMap_case id' id.
             * intros _ _.
               clear IH.
               destruct Hpre2 as (_ & Hpre2 & Hpre3).
               split.
               -- eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
               -- eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
             * specialize (IH id').
               cond_case_auto IH; auto; cbn.
               intros H1 H2.
               specialize (IH H1 H2).
               destruct IH; split.
               -- eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
               -- eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
           + intros id'; unfold mdag_add_vert_and_timeout at 1; cbn [mdag_udag]; unfold udag_add_vert at 1; cbn [udag_verts].
             NatMap_case id' id.
             * cbn.
               intros H1 H2.
               rewrite Forall_forall in H2.
               specialize (H2 _ ltac:(left; auto)).
               cbn in H2.
               specialize (H2 ltac:(auto)).
               contradiction.
             * specialize (IH id').
               cond_case_auto IH; auto; cbn.
               intros H1 H2.
               apply Forall_inv_tail in H2.
               specialize (IH H1 H2).
               destruct IH; split.
               -- eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
               -- eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
           + intros id'; unfold mdag_add_vert at 1; cbn [mdag_udag]; unfold udag_add_vert at 1; cbn [udag_verts].
             NatMap_case id' id.
             * intros Hsync Htimeouts.
               cbn in Htimeouts.
               destruct Hpre2 as (_ & Hquorum1 & Hquorum2 & _ & _ & Hquorum3 & Hquorum4).
               pose proof (bado_comm_live) as (squorum & Hsquorum1 & Hsquorum2).
               assert (Hsquorum3 : is_quorum bado_comm squorum).
               { unfold is_quorum; exists squorum; split; auto; apply incl_refl. }
               pose proof (quorum_overlap_exists_honest bado_comm_safe _ _ Hquorum2 Hsquorum3) as (nid & Hnid1 & Hnid2 & _).
               clear Hquorum2 Hsquorum1 Hsquorum3.
               specialize (Hsquorum2 _ Hnid2); clear squorum Hnid2.
               rewrite in_map_iff in Hnid1.
               destruct Hnid1 as (x & Hx1 & Hx2).
               rewrite Forall_forall in Hquorum1, Hquorum3, Hquorum4.
               specialize (Hquorum1 _ Hx2).
               specialize (Hquorum3 _ Hx2).
               specialize (Hquorum4 _ Hx2).
               cond_case_auto Hquorum1; try contradiction; subst nid.
               specialize (IH x).
               rewrite Ex in IH.
               rewrite Hquorum1 in IH.
               specialize (IH Hsquorum2 Htimeouts).
               clear Hquorum1 Hsync Htimeouts Hsquorum2 Hx2.
               specialize (Hquorum3 ltac:(apply IH)).
               specialize (Hquorum4 ltac:(apply IH)).
               split.
               -- eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
               -- eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
             * specialize (IH id').
               cond_case_auto IH; auto; cbn.
               intros H1 H2.
               specialize (IH H1 H2).
               destruct IH; split.
               -- eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
               -- eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
           + cbn [mdag_add_vert mdag_udag udag_add_vert udag_verts mdag_timeouts].
             intros id'; NatMap_case id' id.
             * intros Hsync.
               destruct Hpre as (_ & Hbyz).
               rewrite Hbyz in Hsync; discriminate.
             * specialize (IH id').
               cond_case_auto IH; auto; cbn.
               intros H1 H2.
               specialize (IH H1 H2).
               destruct IH; split.
               -- eapply mdag_vert_is_supporter_step; try apply Hstep'; auto.
               -- eapply mdag_vert_is_certificate_step; try apply Hstep'; auto.
           + cbn [mdag_add_recv_leader_vert mdag_udag mdag_timeouts].
             intros id; specialize (IH id).
             cond_case_auto IH; auto; cbn.
           + cbn [mdag_add_recv_cert mdag_udag mdag_timeouts].
             intros id; specialize (IH id).
             cond_case_auto IH; auto; cbn.
           + cbn [mdag_add_jump mdag_udag mdag_timeouts].
             intros id; specialize (IH id).
             cond_case_auto IH; auto; cbn.
  Qed.

  Lemma mdag_honest_recv_leader_vert_supporter :
  forall mdag id,
  mdag_valid mdag ->
  match NatMap_find id mdag.(mdag_udag).(udag_verts) with
  | None => True
  | Some v =>
    bado_node_assump v.(udag_vert_builder) <> Byzantine ->
    In (v.(udag_vert_builder), v.(udag_vert_round) - 1) mdag.(mdag_recv_leader_verts) ->
    mdag_vert_is_supporter v mdag
  end.
  Proof. intros mdag id Hval.
         pose proof (mdag_honest_vert _ id Hval) as Hvert.
         cond_case_auto Hvert; auto.
         intros Hhonest.
         specialize (Hvert Hhonest).
         destruct Hvert as [Hvert | [Hvert | Hvert]].
         - intros _; apply Hvert.
         - apply Hvert.
         - destruct Hvert as (_ & _ & _ & Hvert & _); auto.
  Qed.

  Lemma mdag_honest_recv_leader_vert_certificate :
  forall mdag id,
  mdag_valid mdag ->
  match NatMap_find id mdag.(mdag_udag).(udag_verts) with
  | None => True
  | Some v =>
    bado_node_assump v.(udag_vert_builder) <> Byzantine ->
    In (v.(udag_vert_builder), v.(udag_vert_round) - 2) mdag.(mdag_recv_certs) ->
    mdag_vert_is_certificate v mdag
  end.
  Proof. intros mdag id Hval.
         pose proof (mdag_honest_vert _ id Hval) as Hvert.
         cond_case_auto Hvert; auto.
         intros Hhonest.
         specialize (Hvert Hhonest).
         destruct Hvert as [Hvert | [Hvert | Hvert]].
         - intros _; apply Hvert.
         - apply Hvert.
         - destruct Hvert as (_ & _ & _ & _ & Hvert & _); auto.
  Qed.

  Lemma mdag_timeout_exist_vert :
  forall mdag nid r,
  mdag_valid mdag ->
  In (nid, r) mdag.(mdag_timeouts) ->
  exists id, match NatMap_find id mdag.(mdag_udag).(udag_verts) with
  | None => False
  | Some v => v.(udag_vert_builder) = nid /\ v.(udag_vert_round) = r
  end.
  Proof. intros mdag nid r Hval.
         revert nid r.
         induction Hval as [| st st' Hval IH Hstep].
         - cbn; contradiction.
         - mdag_step_case Hstep.
           5,6,7: cbn; auto.
           1,3,4: mdag_unfold; mdag_reduce; udag_unfold; udag_reduce;
                  intros nid r Htimeout;
                  specialize (IH _ _ Htimeout);
                  destruct IH as (x & Hx);
                  cond_case_auto Hx; try contradiction;
                  pose proof (udag_add_vert_pre_uniq _ _ _ ltac:(try apply Hpre2; try apply Hpre)) as Huniq;
                  exists x; NatMap_case x id;
                  [rewrite Ex in Huniq; discriminate | rewrite Ex; auto].

           mdag_unfold; mdag_reduce; udag_unfold; udag_reduce.
           intros nid r Htimeout.
           destruct Htimeout as [Htimeout | Htimeout].
           + inversion Htimeout; subst nid r.
             exists id.
             rewrite NatMap_gse; auto.
           + specialize (IH _ _ Htimeout).
             destruct IH as (x & Hx).
             cond_case_auto Hx; try contradiction.
             pose proof (udag_add_vert_pre_uniq _ _ _ ltac:(try apply Hpre2; try apply Hpre)) as Huniq.
             exists x; NatMap_case x id.
             1: rewrite Ex in Huniq; discriminate.
             rewrite Ex; auto.
  Qed.

End MDAGProps.
