From Coq Require Import
  List
  PeanoNat
  Lia
  Peano_dec.
From advert.lib Require Import
  Maps
  Decision
  Semantics
  Tactics.
From advert.specs Require Import
  UDAG
  Config.
From advert.impl.mysticeti Require Import
  MysticetiDAG.
Import Notations ListNotations.
Open Scope bool_scope.

Section MCommit.
  Context `{config : !BADO_Config} `{dag_config : !DAG_Config} `{participant : !BADO_Participant} `{leader : !BADO_Leader}.

  Record MDAG_Commit_State : Type := {
    mcommit_mdag : MDAG_State;
    mcommit_commit : list (nat * nat); (* round * id *)
    mcommit_nack : list nat; (* round *)
    mcommit_gct : bool; (* gct = false initially, and round-jumping may occur arbitrarily *)
  }.

  Definition mcommit_state_null := {|
    mcommit_mdag := mdag_state_null;
    mcommit_commit := [];
    mcommit_nack := [];
    mcommit_gct := false;
  |}.

  Definition mcommit_mdag_step (mdag : MDAG_State) (mcommit : MDAG_Commit_State) :=
  {| mcommit_mdag := mdag;
     mcommit_commit := mcommit.(mcommit_commit);
     mcommit_nack := mcommit.(mcommit_nack);
     mcommit_gct := mcommit.(mcommit_gct);
  |}.

  Definition mcommit_add_commit (r id : nat) (mcommit : MDAG_Commit_State) :=
  {| mcommit_mdag := mcommit.(mcommit_mdag);
     mcommit_commit := (r, id) :: mcommit.(mcommit_commit);
     mcommit_nack := mcommit.(mcommit_nack);
     mcommit_gct := mcommit.(mcommit_gct);
  |}.

  Definition mcommit_add_nack (r : nat) (mcommit : MDAG_Commit_State) :=
  {| mcommit_mdag := mcommit.(mcommit_mdag);
     mcommit_commit := mcommit.(mcommit_commit);
     mcommit_nack := r :: mcommit.(mcommit_nack);
     mcommit_gct := mcommit.(mcommit_gct);
  |}.

  Definition mcommit_add_jump (nid r : nat) (mcommit : MDAG_Commit_State) :=
  {| mcommit_mdag := mdag_add_jump nid r mcommit.(mcommit_mdag);
     mcommit_commit := mcommit.(mcommit_commit);
     mcommit_nack := mcommit.(mcommit_nack);
     mcommit_gct := mcommit.(mcommit_gct);
  |}.

  Definition mcommit_start_gct (mcommit : MDAG_Commit_State) :=
  {| mcommit_mdag := mcommit.(mcommit_mdag);
     mcommit_commit := mcommit.(mcommit_commit);
     mcommit_nack := mcommit.(mcommit_nack);
     mcommit_gct := true;
  |}.

  Definition mcommit_add_commit_primary_pre (r id : nat) (mcommit : MDAG_Commit_State) :=
  match NatMap_find id mcommit.(mcommit_mdag).(mdag_udag).(udag_verts) with None => False | Some v => v.(udag_vert_round) = r /\ v.(udag_vert_builder) = bado_leader_at r end /\
  exists quorum,
  Forall (fun id' => udag_certifies' id' id mcommit.(mcommit_mdag).(mdag_udag)) quorum /\
  is_quorum bado_comm (map (fun id' => match NatMap_find id' mcommit.(mcommit_mdag).(mdag_udag).(udag_verts) with None => 0 | Some v => v.(udag_vert_builder) end) quorum).

  Definition mcommit_add_commit_secondary_pre (r r' id : nat) (mcommit : MDAG_Commit_State) :=
  match NatMap_find id mcommit.(mcommit_mdag).(mdag_udag).(udag_verts) with None => False | Some v => v.(udag_vert_round) = r /\ v.(udag_vert_builder) = bado_leader_at r end /\
  r' >= r + 3 /\
  exists id',
  In (r', id') mcommit.(mcommit_commit) /\
  (forall r'', r + 3 <= r'' < r' -> In r'' mcommit.(mcommit_nack)) /\
  Exists (fun id'' => udag_certifies' id'' id mcommit.(mcommit_mdag).(mdag_udag)) (udag_get_closure id' mcommit.(mcommit_mdag).(mdag_udag)).

  Definition mcommit_add_nack_pre (r r' : nat) (mcommit : MDAG_Commit_State) :=
  r >= 1 /\
  r' >= r + 3 /\
  exists id',
  In (r', id') mcommit.(mcommit_commit) /\
  (forall r'', r + 3 <= r'' < r' -> In r'' mcommit.(mcommit_nack)) /\
  Forall (fun id'' => match NatMap_find id'' mcommit.(mcommit_mdag).(mdag_udag).(udag_verts) with None => True | Some v => v.(udag_vert_round) = (r + 2)%nat -> ~ mdag_vert_is_certificate v mcommit.(mcommit_mdag) end) (udag_get_closure id' mcommit.(mcommit_mdag).(mdag_udag)).

  Definition mcommit_add_jump_pre (nid r : nat) (mcommit : MDAG_Commit_State) :=
  mdag_can_add_vert r nid mcommit.(mcommit_mdag) /\
  mdag_jump_pre r mcommit.(mcommit_mdag) /\
  (mcommit.(mcommit_gct) = false \/ r <= 2 \/ (exists id, In ((r - 2), id) mcommit.(mcommit_commit)) \/ In (r - 2) mcommit.(mcommit_nack)).

  Definition mcommit_start_gct_pre (mcommit : MDAG_Commit_State) :=
  mcommit.(mcommit_gct) = false.

  Context `{node_assump : !BADO_NodeAssump}.

  Inductive mcommit_step : MDAG_Commit_State -> MDAG_Commit_State -> Prop :=
  | mcommit_step_mdag_step : forall mdag mcommit, mdag_step' mcommit.(mcommit_mdag) mdag -> mcommit_step mcommit (mcommit_mdag_step mdag mcommit)
  | mcommit_step_add_commit_primary : forall r id mcommit, mcommit_add_commit_primary_pre r id mcommit -> mcommit_step mcommit (mcommit_add_commit r id mcommit)
  | mcommit_step_add_commit_secondary : forall r r' id mcommit, mcommit_add_commit_secondary_pre r r' id mcommit -> mcommit_step mcommit (mcommit_add_commit r id mcommit)
  | mcommit_step_add_nack : forall r r' mcommit, mcommit_add_nack_pre r r' mcommit -> mcommit_step mcommit (mcommit_add_nack r mcommit)
  | mcommit_step_add_jump : forall nid r mcommit, mcommit_add_jump_pre nid r mcommit -> mcommit_step mcommit (mcommit_add_jump nid r mcommit)
  | mcommit_step_start_gct : forall mcommit, mcommit_start_gct_pre mcommit -> mcommit_step mcommit (mcommit_start_gct mcommit).

  Definition mcommit_sem : Semantics := {|
    s_state := MDAG_Commit_State;
    s_init := mcommit_state_null;
    s_step := mcommit_step;
  |}.

End MCommit.

Notation mcommit_valid st_mcommit := (valid_state mcommit_sem st_mcommit).
Notation mcommit_reachable st_mcommit := (reachable mcommit_sem st_mcommit).
Ltac mcommit_unfold := unfold mcommit_mdag_step, mcommit_add_commit, mcommit_add_nack.
Ltac mcommit_reduce := cbn beta delta [mcommit_mdag mcommit_commit mcommit_nack] iota.

Ltac mcommit_step_case Hstep :=
  destruct Hstep as [
    mdag mcommit Hstep |
    r id mcommit Hpre |
    r r' id mcommit Hpre |
    r r' mcommit Hpre |
    nid r mcommit Hpre |
    mcommit Hpre
  ];
  [pose proof (mdag_step_imp _ _ Hstep) as Hstep'; remember mcommit.(mcommit_mdag) as mdag'; mdag_step_case' Hstep; subst mdag | | | | | ].

Section MCommitProps.
  Context `{config : !BADO_Config} `{dag_config : !DAG_Config} `{participant : !BADO_Participant} `{leader : !BADO_Leader} `{node_assump : !BADO_NodeAssump} `{assump : !BADO_Assump}.

  Lemma mcommit_mdag_valid : forall mcommit,
  mcommit_valid mcommit ->
  mdag_valid mcommit.(mcommit_mdag).
  Proof. intros mcommit Hval.
         induction Hval as [| st st' Hval IH Hstep].
         - cbn; apply valid_state_init.
         - mcommit_step_case Hstep.
           7,8,9,11: cbn; auto.
           7: cbn; eapply valid_state_step; [apply IH | constructor; apply Hpre].
           all: cbn; eapply valid_state_step; [apply IH | apply Hstep'].
  Qed.

  Lemma mcommit_mdag_reachable : forall mcommit mcommit',
  mcommit_reachable mcommit mcommit' ->
  mdag_reachable mcommit.(mcommit_mdag) mcommit'.(mcommit_mdag).
  Proof. intros mcommit mcommit' Hreach.
         induction Hreach as [| st st' st'' Hreach IH Hstep].
         - apply reachable_self.
         - eapply reachable_trans.
           1: apply IH.
           mcommit_step_case Hstep; cbn; try apply reachable_self; try (eapply reachable_step; [apply reachable_self | apply Hstep']).
           eapply reachable_step; [apply reachable_self|].
           constructor; apply Hpre.
  Qed.

  Lemma mcommit_commit_valid : forall r id mcommit,
  mcommit_valid mcommit ->
  In (r, id) mcommit.(mcommit_commit) ->
  match NatMap_find id mcommit.(mcommit_mdag).(mdag_udag).(udag_verts) with
  | None => False
  | Some v => v.(udag_vert_round) = r /\ v.(udag_vert_builder) = bado_leader_at r
  end.
  Proof. intros r'' id'' mcommit Hval.
         induction Hval as [| st st' Hval IH Hstep].
         - cbn; auto.
         - mcommit_step_case Hstep.
           5,6,9,10,11: cbn; auto.
           1,2,3,4: pose proof (udag_add_vert_pre_uniq _ _ _ ltac:(try apply Hpre2; try apply Hpre)) as Huniq;
                    mcommit_unfold; mcommit_reduce; mdag_unfold; mdag_reduce; udag_unfold; udag_reduce;
                    NatMap_case id'' id;
                    intros Hin;
                    specialize (IH Hin);
                    [cond_case_auto IH; try contradiction; discriminate | auto].
           all: cbn;
                intros Hin;
                destruct Hin as [Hin | Hin];
                [|apply (IH Hin)];
                inversion Hin; subst r'' id''; clear Hin IH;
                apply Hpre.
  Qed.

  Lemma mcommit_commit_primary_or_secondary : forall r id mcommit,
  mcommit_valid mcommit ->
  In (r, id) mcommit.(mcommit_commit) ->
  (* Primary *)
  (exists quorum,
     Forall (fun id' => udag_certifies' id' id mcommit.(mcommit_mdag).(mdag_udag)) quorum /\
     is_quorum bado_comm (map (fun id' => match NatMap_find id' mcommit.(mcommit_mdag).(mdag_udag).(udag_verts) with None => 0 | Some v => v.(udag_vert_builder) end) quorum)) \/
  (* Secondary *)
  (exists r' id',
     r' >= r + 3 /\
     In (r', id') mcommit.(mcommit_commit) /\
     (forall r'', r + 3 <= r'' < r' -> In r'' mcommit.(mcommit_nack)) /\
     Exists (fun id'' => udag_certifies' id'' id mcommit.(mcommit_mdag).(mdag_udag)) (udag_get_closure id' mcommit.(mcommit_mdag).(mdag_udag))).
  Proof. intros r'' id'' mcommit Hval; revert r'' id''.
         induction Hval as [| st st' Hval IH Hstep].
         - cbn; contradiction.
         - mcommit_step_case Hstep.
           5,6,10,11: cbn; auto.
           1,2,3,4: mcommit_unfold; mcommit_reduce; mdag_unfold; mdag_reduce;
                    pose proof (mdag_udag_reachable _ _ ltac:(eapply reachable_step; [apply reachable_self | apply Hstep'])) as Hstep'';
                    pose proof (mdag_udag_valid _ (mcommit_mdag_valid _ Hval)) as Hval';
                    pose proof (valid_reach_valid _ _ _ Hval' Hstep'') as Hval'';
                    pose proof (udag_reachable_subdag _ _ Hval' Hstep'') as Hsub;
                    intros r'' id'' Hin; specialize (IH _ _ Hin);
                    destruct IH as [(quorum' & IH1 & IH2) | (r' & id' & IH1 & IH2 & IH3 & IH4)];
                    [left; exists quorum'; split | right; exists r'; exists id'; repeat split; auto];
                    [ rewrite Forall_forall in *;
                      intros x Hx; specialize (IH1 _ Hx);
                      pose proof IH1 as IH3; unfold udag_certifies' in IH3;
                      cond_case_auto IH3; try contradiction;
                      pose proof (udag_subdag_certifies _ id'' _ _ Hsub ltac:(rewrite Ex0; discriminate)) as Hcert;
                      do 2 rewrite udag_certifies_equiv in Hcert; auto; rewrite Hcert; auto |
                      eapply is_quorum_superset; try apply IH2;
                      udag_unfold; udag_reduce;
                      intros x Hx; rewrite in_map_iff in *;
                      destruct Hx as (y & Hy1 & Hy2); exists y; split; auto;
                      rewrite Forall_forall in IH1; specialize (IH1 _ Hy2);
                      unfold udag_certifies' in IH1;
                      cond_case_auto IH1; try contradiction; subst x;
                      pose proof (udag_add_vert_pre_uniq _ _ _ ltac:(try apply Hpre2; try apply Hpre)) as Huniq;
                      NatMap_case y id; [rewrite Ex0 in Huniq; discriminate | rewrite Ex0; auto] |
                      rewrite Exists_exists in *;
                      destruct IH4 as (x & Hx1 & Hx2);
                      exists x;
                      pose proof (mcommit_commit_valid _ _ _ Hval IH2) as Hvalid;
                      cond_case_auto Hvalid; try contradiction; clear Hvalid;
                      pose proof (udag_closure_mono _ _ _ Hstep'' ltac:(rewrite Ex; discriminate)) as Heq;
                      cbn in Heq; rewrite Heq; split; auto;
                      pose proof Hx2 as Hx3; unfold udag_certifies' in Hx3;
                      cond_case_auto Hx3; try contradiction;
                      pose proof (udag_subdag_certifies _ id'' _ _ Hsub ltac:(rewrite Ex1; discriminate)) as Hcert;
                      do 2 rewrite udag_certifies_equiv in Hcert; auto;
                      rewrite Hcert; auto
                    ].

           + cbn.
             intros r'' id'' Hin; destruct Hin as [Hin | Hin].
             1: inversion Hin; subst r'' id''; clear Hin IH; left; apply Hpre.
             specialize (IH _ _ Hin).
             destruct IH as [IH | (r' & id' & IH1 & IH2 & IH3 & IH4)]; [left; auto | right; exists r'; exists id'; repeat split; auto].

           + cbn.
             intros r'' id'' Hin; destruct Hin as [Hin | Hin].
             1: inversion Hin; subst r'' id''; clear Hin IH; right; destruct Hpre as (_ & Hpre1 & id' & Hpre2 & Hpre3 & Hpre4); exists r'; exists id'; repeat split; auto.
             specialize (IH _ _ Hin).
             destruct IH as [IH | (r''' & id' & IH1 & IH2 & IH3 & IH4)]; [left; auto | right; exists r'''; exists id'; repeat split; auto].

           + cbn.
             intros r'' id'' Hin; specialize (IH _ _ Hin).
             destruct IH as [IH | (r''' & id' & IH1 & IH2 & IH3 & IH4)]; [left; auto | right; exists r'''; exists id'; repeat split; auto].
  Qed.

  Lemma mcommit_commit_certified : forall r id mcommit,
  mcommit_valid mcommit ->
  In (r, id) mcommit.(mcommit_commit) ->
  exists id', udag_certifies' id' id mcommit.(mcommit_mdag).(mdag_udag).
  Proof. intros r id mcommit Hval Hin.
         pose proof (mcommit_commit_primary_or_secondary _ _ _ Hval Hin) as Hcommit.
         destruct Hcommit as [Hcommit | Hcommit].
         - destruct Hcommit as (quorum & Hquorum1 & Hquorum2).
           pose proof (quorum_exists_honest bado_comm_safe _ Hquorum2) as (x & Hx & _).
           rewrite in_map_iff in Hx.
           destruct Hx as (y & _ & Hy).
           rewrite Forall_forall in Hquorum1.
           specialize (Hquorum1 _ Hy).
           exists y; auto.
         - destruct Hcommit as (_ & id' & _ & _ & _ & Hcommit).
           rewrite Exists_exists in Hcommit.
           destruct Hcommit as (x & _ & Hx).
           exists x; auto.
  Qed.

  Lemma mcommit_commit_eq : forall r id id' mcommit,
  mcommit_valid mcommit ->
  In (r, id) mcommit.(mcommit_commit) ->
  In (r, id') mcommit.(mcommit_commit) ->
  id = id'.
  Proof. intros r id id' mcommit Hval Hin1 Hin2.
         pose proof (mcommit_commit_valid _ _ _ Hval Hin1) as Hleader1.
         pose proof (mcommit_commit_valid _ _ _ Hval Hin2) as Hleader2.
         cond_case_auto Hleader1; try contradiction.
         cond_case_auto Hleader2; try contradiction.
         pose proof (mcommit_commit_certified _ _ _ Hval Hin1) as (x1 & Hcert1).
         pose proof (mcommit_commit_certified _ _ _ Hval Hin2) as (x2 & Hcert2).
         pose proof (mdag_udag_valid _ (mcommit_mdag_valid _ Hval)) as Hval'.
         pose proof (udag_certifies_uniq _ _ _ Hval' Hcert1) as Huniq.
         rewrite Ex in Huniq.
         specialize (Huniq _ _ Hcert2).
         rewrite Ex0 in Huniq.
         apply Huniq; lia.
  Qed.

  Lemma mcommit_nack_invariant : forall r mcommit,
  mcommit_valid mcommit ->
  In r mcommit.(mcommit_nack) ->
  exists r' id',
    r >= 1 /\
    r' >= r + 3 /\
    In (r', id') mcommit.(mcommit_commit) /\
    (forall r'', r + 3 <= r'' < r' -> In r'' mcommit.(mcommit_nack)) /\
    Forall (fun id'' => match NatMap_find id'' mcommit.(mcommit_mdag).(mdag_udag).(udag_verts) with None => True | Some v => v.(udag_vert_round) = (r + 2)%nat -> ~ mdag_vert_is_certificate v mcommit.(mcommit_mdag) end) (udag_get_closure id' mcommit.(mcommit_mdag).(mdag_udag)).
  Proof. intros r'' mcommit Hval; revert r''.
         induction Hval as [| st st' Hval IH Hstep].
         - cbn; contradiction.
         - mcommit_step_case Hstep.
           5,6,10,11: cbn; auto.
           1,2,3,4: pose proof (mdag_udag_reachable _ _ ltac:(eapply reachable_step; [apply reachable_self | apply Hstep'])) as Hstep'';
                    cbn [mcommit_mdag_step mcommit_mdag mcommit_commit mcommit_nack mdag_add_vert mdag_add_vert_and_timeout mdag_udag udag_add_vert udag_verts];
                    intros r'' Hin; specialize (IH _ Hin);
                    destruct IH as (r''' & id' & IH1 & IH2 & IH3 & IH4 & IH5); exists r'''; exists id'; repeat split; auto;
                    rewrite Forall_forall in *;
                    pose proof (mcommit_commit_valid _ _ _ Hval IH3) as Hvalid;
                    cond_case_auto Hvalid; try contradiction; clear Hvalid;
                    pose proof (udag_closure_mono _ _ _ Hstep'' ltac:(rewrite Ex; discriminate)) as Heq;
                    cbn [mdag_udag mdag_add_vert mdag_add_vert_and_timeout] in Heq; rewrite Heq;
                    intros x Hx; specialize (IH5 _ Hx);
                    pose proof (udag_add_vert_pre_uniq _ _ _ ltac:(try apply Hpre2; try apply Hpre)) as Huniq;
                    pose proof (udag_closure_valid _ _ _ (mdag_udag_valid _ (mcommit_mdag_valid _ Hval)) Hx) as H;
                    NatMap_case x id; [contradiction | clear H];
                    cond_case_auto IH5; auto;
                    intros Hr; specialize (IH5 Hr);
                    intros Hcert; apply IH5;
                    apply (mdag_vert_is_certificate_step_inv _ _ _ _ (mcommit_mdag_valid _ Hval) Hstep' Ex0 Hcert).

           + cbn.
             intros r'' Hin; specialize (IH _ Hin).
             destruct IH as (r''' & id' & IH1 & IH2 & IH3 & IH4 & IH5).
             exists r'''; exists id'; repeat split; auto.

           + cbn.
             intros r'' Hin; specialize (IH _ Hin).
             destruct IH as (r''' & id' & IH1 & IH2 & IH3 & IH4 & IH5).
             exists r'''; exists id'; repeat split; auto.

           + cbn.
             intros r'' Hin; destruct Hin as [Hin | Hin].
             2: specialize (IH _ Hin);
                destruct IH as (r''' & id' & IH1 & IH2 & IH3 & IH4 & IH5);
                exists r'''; exists id'; repeat split; auto.
             subst r''.
             exists r'; destruct Hpre as (Hpre1 & Hpre2 & id' & Hpre3 & Hpre4 & Hpre5).
             exists id'; repeat split; auto.
  Qed.

  Lemma mcommit_safety : forall r id mcommit,
  mcommit_valid mcommit ->
  In (r, id) mcommit.(mcommit_commit) ->
  ~ In r mcommit.(mcommit_nack).
  Proof. intros r'' id'' mcommit Hval.
         revert r'' id''.
         induction Hval as [| st st' Hval IH Hstep].
         - cbn; auto.
         - intros r'' id''.
           mcommit_step_case Hstep.
           1,2,3,4,5,6,10,11: cbn; apply IH.
           + cbn.
             intros Hin.
             destruct Hin as [Hin | Hin].
             2: eapply IH; apply Hin.
             inversion Hin; subst r'' id''; clear Hin IH.
             unfold mcommit_add_commit_primary_pre in Hpre.
             intros Hin.
             pose proof (mcommit_nack_invariant _ _ Hval Hin) as Hnack.
             destruct Hnack as (r' & id' & Hpos & Hge & Hcommit & _ & Hno_cert).
             destruct Hpre as (Hvert & Hquorum).
             cond_case_auto Hvert; try contradiction.
             destruct Hquorum as (quorum & Hquorum1 & Hquorum2).
             pose proof (mcommit_commit_valid _ _ _ Hval Hcommit) as Hvalid.
             cond_case_auto Hvalid; try contradiction.
             pose proof (udag_closure_quorum _ id' (r + 2) ltac:(eapply mdag_udag_valid; eapply mcommit_mdag_valid; apply Hval)) as Hquorum'.
             rewrite Ex0 in Hquorum'.
             specialize (Hquorum' ltac:(lia)).
             destruct Hquorum' as (quorum' & Hquorum'1 & Hquorum'2 & Hquorum'3).
             pose proof (quorum_overlap_exists_honest bado_comm_safe _ _ Hquorum2 Hquorum'2) as (nid & Hnid1 & Hnid2 & Hnid3).
             clear Hquorum2 Hquorum'2.
             rewrite in_map_iff in Hnid1, Hnid2.
             destruct Hnid1 as (x & Hx1 & Hx2).
             destruct Hnid2 as (y & Hy1 & Hy2).
             rewrite Forall_forall in *.
             specialize (Hquorum1 _ Hx2).
             pose proof Hquorum1 as Hx3.
             unfold udag_certifies' in Hquorum1.
             cond_case_auto Hquorum1; try contradiction.
             destruct Hquorum1 as (Hquorum1 & _).
             specialize (Hquorum'1 _ Hy2).
             cond_case_auto Hquorum'1; try contradiction.
             inversion Ex; subst u1; clear Ex.
             pose proof (udag_honest_uniq _ x y ltac:(eapply mdag_udag_valid; eapply mcommit_mdag_valid; apply Hval)) as Heq.
             rewrite Ex2, Ex3 in Heq.
             specialize (Heq ltac:(rewrite Hx1; auto) ltac:(lia) ltac:(lia)).
             subst y.
             rewrite Ex2 in Ex3; inversion Ex3; subst u3; clear Ex3 Hy1 Hx2.
             specialize (Hquorum'3 _ Hy2); clear Hy2.
             specialize (Hno_cert _ Hquorum'3).
             rewrite Ex2 in Hno_cert.
             specialize (Hno_cert ltac:(auto)).
             apply Hno_cert.
             rewrite <- udag_certifies_equiv in Hx3.
             2: apply mdag_udag_valid; apply mcommit_mdag_valid; auto.
             rewrite mdag_certificate_certifies.
             2: apply Ex2.
             right; split; [lia|].
             exists id; split.
             2: rewrite <- udag_certifies_equiv; auto.
             2: apply mdag_udag_valid; apply mcommit_mdag_valid; auto.
             rewrite Ex1; rewrite Hquorum1.
             replace (udag_vert_round u + 2 - 2) with (udag_vert_round u) by lia; split; auto.
             destruct Hvert as (Hvert & ?); rewrite Hvert; auto.

           + cbn.
             intros Hin.
             destruct Hin as [Hin | Hin].
             2: eapply IH; apply Hin.
             inversion Hin; subst r'' id''; clear Hin.
             unfold mcommit_add_commit_secondary_pre in Hpre.
             intros Hin.
             pose proof (mcommit_nack_invariant _ _ Hval Hin) as Hnack.
             destruct Hnack as (r'' & id'' & Hpos & Hge & Hcommit & Hnacks & Hno_cert).
             destruct Hpre as (Hpre1 & Hpre2).
             cond_case_auto Hpre1; try contradiction.
             destruct Hpre2 as (Hge' & id' & Hcommit' & Hnacks' & Hcert).
             assert (Heq : r' = r'').
             { clear Hcert Hno_cert.
               destruct (decide (r'' < r')).
               { specialize (Hnacks' r'' ltac:(split; auto)).
                 specialize (IH _ _ Hcommit Hnacks').
                 contradiction.
               }
               destruct (decide (r' < r'')).
               { specialize (Hnacks r' ltac:(split; auto)).
                 specialize (IH _ _ Hcommit' Hnacks).
                 contradiction.
               }
               lia.
             }
             subst r''.
             pose proof (mcommit_commit_eq _ _ _ _ Hval Hcommit Hcommit').
             subst id''.
             clear Hge' Hcommit' Hnacks' Hnacks Hin Hge IH.
             rewrite Exists_exists in Hcert.
             destruct Hcert as (x & Hx1 & Hx2).
             rewrite Forall_forall in Hno_cert.
             specialize (Hno_cert _ Hx1).
             pose proof (mcommit_commit_valid _ _ _ Hval Hcommit) as Hvalid.
             cond_case_auto Hvalid; try contradiction.
             clear Hvalid.
             pose proof Hx2 as Hx3.
             unfold udag_certifies' in Hx3.
             cond_case_auto Hx3; try contradiction.
             inversion Ex; subst u1; clear Ex.
             destruct Hx3 as (Hx3 & _).
             specialize (Hno_cert ltac:(lia)).
             apply Hno_cert.
             rewrite mdag_certificate_certifies.
             2: apply Ex2.
             right; split; [lia|].
             exists id; split; auto.
             rewrite Ex1; rewrite Hx3.
             replace (udag_vert_round u + 2 - 2) with (udag_vert_round u) by lia.
             split; auto.
             destruct Hpre1 as (Hpre1 & ?); rewrite Hpre1; auto.

           + cbn.
             intros Hcommit Hnack.
             destruct Hnack as [Hnack | Hnack].
             2: apply (IH _ _ Hcommit Hnack).
             subst r''.
             rename id'' into id.
             unfold mcommit_add_nack_pre in Hpre.
             rename Hcommit into Hcommit'.
             destruct Hpre as (Hpos & Hge & id' & Hcommit & Hnacks & Hno_cert).
             pose proof (mcommit_commit_valid _ _ _ Hval Hcommit') as Hvert.
             cond_case_auto Hvert; try contradiction.
             pose proof (mcommit_commit_primary_or_secondary _ _ _ Hval Hcommit') as Hcommit''.
             clear Hcommit'.
             destruct Hcommit'' as [Hcommit' | Hcommit'].
             * destruct Hcommit' as (quorum & Hquorum1 & Hquorum2).
               pose proof (mcommit_commit_valid _ _ _ Hval Hcommit) as Hvalid.
               cond_case_auto Hvalid; try contradiction.
               pose proof (udag_closure_quorum _ id' (r + 2) ltac:(eapply mdag_udag_valid; eapply mcommit_mdag_valid; apply Hval)) as Hquorum'.
               rewrite Ex0 in Hquorum'.
               specialize (Hquorum' ltac:(lia)).
               destruct Hquorum' as (quorum' & Hquorum'1 & Hquorum'2 & Hquorum'3).
               pose proof (quorum_overlap_exists_honest bado_comm_safe _ _ Hquorum2 Hquorum'2) as (nid & Hnid1 & Hnid2 & Hnid3).
               clear Hquorum2 Hquorum'2.
               rewrite in_map_iff in Hnid1, Hnid2.
               destruct Hnid1 as (x & Hx1 & Hx2).
               destruct Hnid2 as (y & Hy1 & Hy2).
               rewrite Forall_forall in *.
               specialize (Hquorum1 _ Hx2).
               pose proof Hquorum1 as Hx3.
               unfold udag_certifies' in Hquorum1.
               cond_case_auto Hquorum1; try contradiction.
               destruct Hquorum1 as (Hquorum1 & _).
               specialize (Hquorum'1 _ Hy2).
               cond_case_auto Hquorum'1; try contradiction.
               inversion Ex; subst u1; clear Ex.
               pose proof (udag_honest_uniq _ x y ltac:(eapply mdag_udag_valid; eapply mcommit_mdag_valid; apply Hval)) as Heq.
               rewrite Ex2, Ex3 in Heq.
               specialize (Heq ltac:(rewrite Hx1; auto) ltac:(lia) ltac:(lia)).
               subst y.
               rewrite Ex2 in Ex3; inversion Ex3; subst u3; clear Ex3 Hy1 Hx2.
               specialize (Hquorum'3 _ Hy2); clear Hy2.
               specialize (Hno_cert _ Hquorum'3).
               rewrite Ex2 in Hno_cert.
               specialize (Hno_cert ltac:(auto)).
               apply Hno_cert.
               rewrite <- udag_certifies_equiv in Hx3.
               2: apply mdag_udag_valid; apply mcommit_mdag_valid; auto.
               rewrite mdag_certificate_certifies.
               2: apply Ex2.
               right; split; [lia|].
               exists id; split.
               2: rewrite <- udag_certifies_equiv; auto.
               2: apply mdag_udag_valid; apply mcommit_mdag_valid; auto.
               rewrite Ex1; rewrite Hquorum1.
               replace (udag_vert_round u + 2 - 2) with (udag_vert_round u) by lia; split; auto.
               destruct Hvert as (Hvert & ?); rewrite Hvert; auto.

             * destruct Hcommit' as (r'' & id'' & Hge' & Hcommit' & Hnacks' & Hcert).
               assert (Heq : r' = r'').
               { clear Hcert Hno_cert.
                 destruct (decide (r'' < r')).
                 { specialize (Hnacks r'' ltac:(split; auto)).
                   specialize (IH _ _ Hcommit' Hnacks).
                   contradiction.
                 }
                 destruct (decide (r' < r'')).
                 { specialize (Hnacks' r' ltac:(split; auto)).
                   specialize (IH _ _ Hcommit Hnacks').
                   contradiction.
                 }
                 lia.
               }
               subst r''.
               pose proof (mcommit_commit_eq _ _ _ _ Hval Hcommit Hcommit').
               subst id''.
               clear Hge' Hcommit' Hnacks' Hnacks Hge IH.
               rewrite Exists_exists in Hcert.
               destruct Hcert as (x & Hx1 & Hx2).
               rewrite Forall_forall in Hno_cert.
               specialize (Hno_cert _ Hx1).
               pose proof (mcommit_commit_valid _ _ _ Hval Hcommit) as Hvalid.
               cond_case_auto Hvalid; try contradiction.
               clear Hvalid.
               pose proof Hx2 as Hx3.
               unfold udag_certifies' in Hx3.
               cond_case_auto Hx3; try contradiction.
               inversion Ex; subst u1; clear Ex.
               destruct Hx3 as (Hx3 & _).
               specialize (Hno_cert ltac:(lia)).
               apply Hno_cert.
               rewrite mdag_certificate_certifies.
               2: apply Ex2.
               right; split; [lia|].
               exists id; split; auto.
               rewrite Ex1; rewrite Hx3.
               replace (udag_vert_round u + 2 - 2) with (udag_vert_round u) by lia.
               split; auto.
               destruct Hvert as (Hvert & ?); rewrite Hvert; auto.
  Qed.

  Lemma mcommit_commit_mono : forall mcommit mcommit',
  mcommit_reachable mcommit mcommit' ->
  incl mcommit.(mcommit_commit) mcommit'.(mcommit_commit).
  Proof. intros mcommit mcommit' Hreach.
         induction Hreach as [| st st' st'' Hreach IH Hstep].
         - apply incl_refl.
         - mcommit_step_case Hstep.
           all: cbn; eapply incl_tran; [apply IH | try apply incl_tl; apply incl_refl].
  Qed.

  Lemma mcommit_nack_mono : forall mcommit mcommit',
  mcommit_reachable mcommit mcommit' ->
  incl mcommit.(mcommit_nack) mcommit'.(mcommit_nack).
  Proof. intros mcommit mcommit' Hreach.
         induction Hreach as [| st st' st'' Hreach IH Hstep].
         - apply incl_refl.
         - mcommit_step_case Hstep.
           all: cbn; eapply incl_tran; [apply IH | try apply incl_tl; apply incl_refl].
  Qed.

  Lemma mcommit_gct_persist : forall mcommit mcommit',
  mcommit_reachable mcommit mcommit' ->
  mcommit.(mcommit_gct) = true ->
  mcommit'.(mcommit_gct) = true.
  Proof. intros mcommit mcommit' Hreach.
         induction Hreach as [| st st' st'' Hreach IH Hstep].
         - auto.
         - mcommit_step_case Hstep.
           all: cbn; auto.
  Qed.

  Lemma mcommit_gct_jump_has_decision : forall nid r mcommit mcommit',
  mcommit_reachable mcommit mcommit' ->
  mcommit.(mcommit_gct) = true ->
  In (nid, r) mcommit'.(mcommit_mdag).(mdag_jumps) ->
  In (nid, r) mcommit.(mcommit_mdag).(mdag_jumps) \/
  r <= 2 \/ (exists id, In (r - 2, id) mcommit'.(mcommit_commit)) \/ In (r - 2) mcommit'.(mcommit_nack).
  Proof. intros nid'' r'' mcommit mcommit' Hreach.
         revert nid'' r''.
         induction Hreach as [| st st' st'' Hreach IH Hstep].
         - intros; left; auto.
         - intros nid'' r'' Hgct.
           mcommit_step_case Hstep.
           all: try (cbn; revert Hgct; apply IH).
           1,2,3: cbn;
                  intros Hin; specialize (IH _ _ Hgct Hin);
                  destruct IH as [IH | [IH | [IH | IH]]]; [left | right; left | right; right; left | right; right; right]; auto;
                  destruct IH as (x & Hx); exists x; auto.
           cbn.
           intros Hin; destruct Hin as [Hin | Hin].
           2: apply IH; auto.
           inversion Hin; subst nid'' r''; clear Hin.
           right.
           destruct Hpre as (_ & _ & [Hpre | [Hpre | [Hpre | Hpre]]]).
           + pose proof (mcommit_gct_persist _ _ Hreach Hgct) as Hpersist.
             rewrite Hpersist in Hpre; discriminate.
           + left; auto.
           + right; left; auto.
           + right; right; auto.
  Qed.

  (* If three consecutive rounds are committed, then all previous rounds can be decided *)

  (* Some preparatory lemmas *)
  Definition mdag_id_is_certificate_with_round (r : nat) (id : nat) (mdag : MDAG_State) :=
    match NatMap_find id (udag_verts (mdag_udag mdag)) with
    | None => False
    | Some vert => vert.(udag_vert_round) = r /\ mdag_vert_is_certificate vert mdag
    end.

  Instance mdag_vert_is_certificate_with_round_dec r id mdag : Decision (mdag_id_is_certificate_with_round r id mdag).
  Proof.
    unfold mdag_id_is_certificate_with_round.
    destruct (NatMap_find id (udag_verts (mdag_udag mdag))).
    2: typeclasses eauto.
    apply and_dec.
    1: typeclasses eauto.
    unfold mdag_vert_is_certificate.
    apply or_dec; [typeclasses eauto|].
    apply and_dec; [typeclasses eauto|].
    apply decide_Exists.
    intros x; destruct (NatMap_find x (udag_verts (mdag_udag mdag))).
    - match goal with |- context[match ?x with _ => _ end] => destruct x end.
      all: typeclasses eauto.
    - typeclasses eauto.
  Qed.

  Lemma mcommit_can_decide_minus_three : forall r r' id mcommit,
  mcommit_valid mcommit ->
  r' > 0 ->
  r >= r' + 3 ->
  In (r, id) mcommit.(mcommit_commit) ->
  (forall r'', r' + 3 <= r'' < r -> In r'' mcommit.(mcommit_nack)) ->
  mcommit_step mcommit (mcommit_add_nack r' mcommit) \/ exists id', mcommit_step mcommit (mcommit_add_commit r' id' mcommit).
  Proof.
    intros r r' id mcommit Hval Hpos1 Hpos2 Hcommit.
    destruct (decide (Exists (fun id' => mdag_id_is_certificate_with_round (r' + 2)%nat id' (mcommit_mdag mcommit)) (udag_get_closure id (mdag_udag (mcommit_mdag mcommit))))).
    - rewrite Exists_exists in e.
      destruct e as (cert_id & Hcert1 & Hcert2).
      unfold mdag_id_is_certificate_with_round in Hcert2.
      match type of Hcert2 with match ?x with _ => _ end => destruct x eqn:Ecert end.
      2: contradiction.
      destruct Hcert2 as (Hcert2 & Hcert3).
      rewrite mdag_certificate_certifies in Hcert3.
      2: apply Ecert.
      destruct Hcert3 as [? | (_ & Hcert3)]; [lia|].
      destruct Hcert3 as (id' & Hid'1 & Hid'2).
      match type of Hid'1 with match ?x with _ => _ end => destruct x eqn:Evert' end.
      2: contradiction.
      right; exists id'.
      eapply mcommit_step_add_commit_secondary.
      unfold mcommit_add_commit_secondary_pre.
      split.
      1: rewrite Evert'; replace r' with (udag_vert_round u - 2) by lia; auto.
      split.
      2: exists id; split.
      2: apply Hcommit.
      1: lia.
      split; auto.
      rewrite Exists_exists; exists cert_id; split; auto.
    - intros Hnack; left.
      eapply mcommit_step_add_nack.
      unfold mcommit_add_nack_pre.
      repeat split.
      1: lia.
      2: exists id.
      2: split.
      2: apply Hcommit.
      1: lia.
      split; auto.
      rewrite Forall_forall; rewrite Exists_exists in n.
      intros x Hx.
      match goal with |- match ?x with _ => _ end => destruct x eqn:Ex end.
      2: auto.
      replace (r - 3 + 2)%nat with (r - 1) by lia.
      intros Hx1 Hx2.
      apply n; exists x; split; auto.
      unfold mdag_id_is_certificate_with_round.
      rewrite Ex; split; auto.
  Qed.

  Lemma mcommit_can_decide_secondary : forall r r' id1 id2 id3 mcommit,
  mcommit_valid mcommit ->
  In (r, id1) mcommit.(mcommit_commit) ->
  In (S r, id2) mcommit.(mcommit_commit) ->
  In (S (S r), id3) mcommit.(mcommit_commit) ->
  r' < r ->
  (forall r'', r' < r'' < r -> In r'' mcommit.(mcommit_nack) \/ exists id', In (r'', id') mcommit.(mcommit_commit)) ->
  exists r'' id'',
    r'' >= r' + 3 /\
    In (r'', id'') mcommit.(mcommit_commit) /\
    forall r''', r' + 3 <= r''' < r'' -> In r''' mcommit.(mcommit_nack).
  Proof.
    intros r r' id1 id2 id3 mcommit Hval.
    intros Hcommit1 Hcommit2 Hcommit3.
    remember (r - r') as d.
    revert r' Heqd.
    induction d.
    - lia.
    - intros r' Heqr'.
      destruct d as [|d].
      + (* d = 1 *)
        intros _ _.
        exists (S (S r)); exists id3.
        split; [lia|].
        split; auto.
        lia.
      + destruct d as [|d].
        * (* d = 2 *)
          intros _.
          exists (S r); exists id2.
          split; [lia|].
          split; auto.
          lia.
        * destruct d as [|d].
          (* d = 3 *)
          -- intros _.
             exists r; exists id1.
             split; [lia|].
             split; auto.
             lia.
          -- (* d >= 4 *)
             intros _.
             destruct (decide (In (r' + 3)%nat (map fst (mcommit_commit mcommit)))).
             ++ rewrite in_map_iff in i.
                destruct i as (p & Hp1 & Hp2).
                destruct p; cbn in Hp1; subst n.
                intros _; exists (r' + 3)%nat; exists n0.
                repeat split; auto.
                lia.
             ++ specialize (IHd (S r') ltac:(lia) ltac:(lia)).
                intros Hdecision.
                specialize (IHd ltac:(intros; apply Hdecision; lia)).
                destruct IHd as (r'' & id'' & IHd1 & IHd2 & IHd3).
                exists r''; exists id''; repeat split; auto.
                1: lia.
                intros r''' Hr'''.
                assert (Heq : (S r' + 3)%nat <= r''' < r'' \/ r''' = (r' + 3)%nat) by lia.
                destruct Heq as [? | Heq].
                1: apply IHd3; auto.
                subst r'''.
                specialize (Hdecision (r' + 3)%nat ltac:(lia)).
                destruct Hdecision as [Hdecision | Hdecision].
                1: auto.
                destruct Hdecision as (id' & Hid').
                exfalso; apply n; rewrite in_map_iff; exists ((r' + 3)%nat, id'); split; auto.
  Qed.

  (* The lemma *)

  Lemma mcommit_can_decide : forall r r' id1 id2 id3 mcommit,
  mcommit_valid mcommit ->
  In (r, id1) mcommit.(mcommit_commit) ->
  In (S r, id2) mcommit.(mcommit_commit) ->
  In (S (S r), id3) mcommit.(mcommit_commit) ->
  0 < r' < r ->
  (forall r'', r' < r'' < r -> In r'' mcommit.(mcommit_nack) \/ exists id', In (r'', id') mcommit.(mcommit_commit)) ->
  mcommit_step mcommit (mcommit_add_nack r' mcommit) \/ exists id', mcommit_step mcommit (mcommit_add_commit r' id' mcommit).
  Proof.
    intros r r' id1 id2 id3 mcommit Hval.
    intros Hcommit1 Hcommit2 Hcommit3 Hlt Hdecision.
    pose proof (mcommit_can_decide_secondary r r' id1 id2 id3 _ Hval Hcommit1 Hcommit2 Hcommit3 ltac:(apply Hlt) Hdecision) as Hsecondary.
    destruct Hsecondary as (r'' & id'' & Hr''1 & Hr''2 & Hr''3).
    apply (mcommit_can_decide_minus_three r'' r' id'' _ Hval ltac:(apply Hlt) Hr''1 Hr''2 Hr''3).
  Qed.

End MCommitProps.
