# Introduction

This is the artifact for S&P 2026 paper #131 "Mechanized Safety and Liveness Proofs for the Mysticeti Consensus Protocol under the LiDO-DAG Framework."
This artifact contains two parts.
The first part (`formal_proof`) contains a formal model of the Mysticeti consensus protocol, and machine-checked safety & liveness proofs of it.
They are developed in the Rocq (formerly Coq) proof assistent. We use Rocq 8.15.

The second part (`sui_testcase`) contains a testcase for the implementation of Mysticeti used by the Sui blockchain.
It is used to show that the current implementation of Mysticeti is incorrect and may lead to liveness attacks (see Section 6 of the paper).

The general organization of files in this artifact follows the structure specified at https://sp2026.ieee-security.org/artifact_instructions.html.

# Installation

For each part of the artifact we use a Dockerfile for setting up the build environment.
Therefore, to install this artifact one must have either Docker or Podman installed.
The provided install script uses Podman because it seems to be the better supported option on Debian.

For the purpose of artifact evaluation we tested the artifact on the SPHERE testbed (https://launch.sphere-testbed.net/).
There are special difficulties in installing Podman on the SPHERE testbed:
1. The default VM provisioned by SPHERE does not have `dbus-user-session` installed.
It is a crucial system service for running Podman.
2. The default storage provisioned by SPHERE is not sufficient to build our artifacts,
and we need to configure Podman to use the extra storage provided in `/dev/vdb`.

For these reasons, the install script is split into several steps,
and we expect them to be used as follows:
1. (Do this step **ONLY** if testing the artifact on the SPHERE testbed)
Run `infrastructure/allocation` to setup the extra disk storage provisioned by SPHERE.
2. Run `./install_step1.sh` to install Podman and other packages needed by Podman.
3. Run `sudo reboot`, and restart the terminal connection.
This is necessary to get `dbus-user-session` running properly.
4. (Do this step **ONLY** if testing the artifact on the SPHERE testbed)
Run `./install_configure_podman.sh` to configure Podman storage locations.
5. Run `./install_step2.sh` to build the Docker images.
6. Run `./claims/claim1/run.sh` and `./claims/claim2/run.sh` to test each part of the artifact.

We expect that `./install_step2.sh` can be used on any x86-64 system with Podman installed to build the Docker images.
However, AArch64 machines (including Apple Silicon) are not supported because one of the base images we used (`coqorg/coq`) does not support AArch64.

# Documentation for the Formal Proof

We first describe the formal network model of the Mysticeti protocol.
Then we introduce the LiDO-DAG abstract model.
Finally we describe the refinement proof between the network model and the abstract model.

## Included Files

* `specs/`: Files that define the LiDO-DAG abstract model.
* `proofs/`: Safety proofs for the LiDO-DAG model.
* `impl/mysticeti/`: Files that define the Mysticeti protocol and prove its network-level safety property.
* `impl/mysticeti/proofs`: Files that formalize the liveness proof of Mysticeti.
* `impl/mysticeti/refine`: Files that prove refinement between Mysticeti and LiDO-DAG.
* `lib/`: Some supporting files.

## The Consensus Configuration

The standard setting for byzantine fault-tolerant consensus is that there are a total of `3f+1` network processes,
of which `2f+1` are non-faulty, and `f` may have byzantine faults.
A "quorum" set is any set of at least `2f+1` network processes.
By the pigeonhole principle, two quorum sets must intersect on at least `f+1` network processes.
Since at most `f` processes may be faulty, they intersect on at least one non-faulty process.
This is a key property of quorum sets used by many different consensus protocols.

The LiDO-DAG model uses an abstract notion of "quorum" sets, formalized in `specs/Config.v`.
First, we assume there is a fixed set of participanting processes.
Each process has a unique ID number, and we usually use `nid` to represent this number.
We assume the list of participating processes is given by `bado_participant`.

We also assume the execution of consensus is divided into "views,"
and each view has a predetermined leader.
For Mysticeti, each view exactly corresponds to one round in the DAG graph,
and we shall not distinguish between "view" and "round" below.
We assume the leader of round `r` is given by `bado_leader_at r`.
We require the leader schedule to be valid (all leaders are in the list of participants) and fair (every participant eventually becomes the leader).
Liveness of Mysticeti additionally requires that there are eventually three consecutive non-faulty leaders (not specified in `Config.v`).

The participating processes are divided into three types, called `Synchronous`, `Asynchronous`, and `Byzantine`.
The synchronous processes are the "non-faulty" processes described above.
The asynchronous processes are the processes that follow the protocol honestly, but may not be responsive to messages (e.g. crash fault nodes).
In the standard setting there are no asynchronous processes, and this aspect of the LiDO-DAG model is unused in this work.
The byzantine processes are the processes that may send messages arbitrarily, subject only to cryptographic constraints.
We say a process is "honest" if it is either synchronous or asynchronous.
We assume the type of process `nid` is given by `bado_process_assump nid`.

We now introduce the notion of a "committee."
A committee C is simply a list of sets of participants.
Each set in this list is considered a "minimal quorum" of this committee.
A set S of participants is considered a quorum of a committee C, if S is a superset of any minimal quorum of C.
The basic requirements we impose on a committee are:
* `committee_valid`: Each member in a minimal quorum must be a consensus participant.
* `committee_safe`: Two minimal quorums must intersect on at least one honest (synchronous or asynchronous) process.
* `committee_live`: There is at least one minimal quorum consisting of only synchronous processes.

In the Mysticeti protocol there is only one committee.
We denote this committee by `bado_comm` (see below).
The minimal quorums are sets that contain exactly `2f+1` participants.
Since there are `2f+1` synchronous processes, the `committee_live` condition is satisfies.
We use `is_quorum bado_comm l` to represent the condition that the set of participants `l` is a quorum set.

We also need a notion of "weak quorums" (called "timeout quorums" in the LiDO-DAG model due to its relation to timeout messages).
A weak quorum is a set of participants, such that every participant agrees contains at least one synchronous process.
In the standard setting a weak quorum is any set that contains at least `f+1` participants.

For the safety and liveness proofs of Mysticeti we need the following properties of weak quorums:
* `bado_to_quorum_valid`: Each member of a weak quorum is a participant of consensus.
* `bado_to_quorum_safe`: Each weak quorum contains at least one synchronous member.
* `bado_to_quorum_live`: Each quorum set contains a subset that 1) contains only synchronous processes; 2) forms a weak quorum.
* `bado_to_quorum_overlap`: A quorum set and a weak quorum set always intersect on at least one member (not necessarily honest).

A consensus configuration in the LiDO-DAG model is thus given by the following data:
```coq
Class BADO_Config : Type := {
  bado_comm : Committee_Data; (* The global committee *)
  bado_m_comm : nat -> Committee_Data; (* Round-local committees, not used by Mysticeti *)
  bado_ver_lim : nat; (* Not used by Mysticeti *)
  bado_to_quorums : list (list nat); (* Minimal weak quorums, "to" stands for "timeout" *)
}.
```

The assumptions we make on the configuration data are formalized in the structure `BADO_Assump`.

## The Formal Model of Mysticeti

### The Unauthenticated DAG Model

Since Mysticeti is an unauthenticated DAG-based consensus protocol, we first formalized a model of unauthenticated DAG in `specs/UDAG.v`.
We expect this model can be reused for different protocols based on unauthenticated DAG.
Authenticated DAGs can be considered as special cases of unauthenticated DAGs.
However, the artifact of LiDO-DAG provides a model that is more specialized to that setting.

We use the type `dag_t` to represent the type of data embedded in each vertex of DAG.
We assume equality of `dag_t` is decidable (`dag_t_eq_dec`).
This is because later we will have to deduplicate lists of data blocks.
We do not otherwise make assumptions on `dag_t`.

Each vertex of the DAG is represented by the following structure:
```coq
Record UDAG_Vert : Type := { (* UDAG = Unauthenticated DAG *)
  udag_vert_round : nat; (* Round number that this vertex belongs to *)
  udag_vert_builder : nat; (* ID of process that built this vertex *)
  udag_vert_data : dag_t; (* Data embedded in this vertex *)
  udag_vert_preds : list nat; (* List of predecessor vertex IDs *)
}.
```
This corresponds to the definition of `Vertex` on page 3 of the paper.

The state of the DAG is represented by the following structure:
```coq
Record UDAG_State : Type := {
  udag_verts : NatMap UDAG_Vert; (* ID -> vertex *)
  udag_closure : NatMap (list nat); (* ID -> closure *)
  }.
```

The field `udag_verts` is a finite map from vertex ID to the vertex structure.
The ID of a vertex is assigned non-deterministically when it is added to the DAG. See below.
The field `udag_closure` records the closure of a vertex in the graph.
The closure of a vertex is defined on page 5 of the paper.
It is computed automatically when the vertex is added.

The UDAG model is a transition system with only one kind of transition step, called `udag_step_add_vert`,
which adds a single vertex to the DAG.
The preconditions of this step are specified in `udag_add_vert_pre`.
* `udag_add_vert_pre_pos`: We assume the round numbers of DAG start from 1. Hence `udag_vert_round > 0`.
* `udag_add_vert_pre_mem`: The builder of the vertex must be a consensus participant.
We implicitly assume that all vertices are cryptographically signed.
Thus it is always possible to uniquely identify the builder of a vertex.
Also, byzantine processes cannot forge vertices from honest processes.
(How we model byzantine processes is described later.)
* `udag_add_vert_pre_lt`: Each predecessor ID in the `udag_vert_preds` field corresponds to an existing vertex,
and it belongs to an earlier round than the vertex to be added.
* `udag_add_vert_pre_quorum`: Either the new vertex is a genesis vertex (`udag_vert_round = 1`),
or its `udag_vert_preds` field contains vertices in the immediate previous round built by a quorum (`2f+1` participants) of processes.
* `udag_add_vert_pre_uniq`: The ID to be assigned to the new vertex is `id`, and `id` is not currently assigned to any vertex.
In most DAG-based protocols, the ID of a vertex is computed via a cryptographic hash function.
If we model the hash function as a random oracle, then the probability that two vertices share the same ID is negligible.
In our model, we assume the ID is non-deterministically chosen, subject to the restriction that two vertices cannot use the same ID.
See the discusson on page 4 of the paper for this point.
* `udag_add_vert_pre_honest_pred`: When an honest process `nid` adds a vertex `v` into the DAG,
the predecessor list of `v` should contain the most recent vertex previously created by `nid` (unless `nid` has never created any vertex before).
Here the phrase "most recent" means "having the highest round number."
This requirement is important for the liveness proof.
This requirement also implies that an honest process can only create one vertex per round.
If it has already created a vertex in round `r`, and attempts to create a second vertex in round `r`,
then the second vertex would be required to list the first vertex (or a later vertex) as a predecessor, which would violate `udag_add_vert_pre_lt`.

When a new vertex `vert` is assigned an ID number (`id`) and added to the DAG,
we update the finite map `udag_verts` so that `id` now maps to `vert`.
We also update the finite map `udag_closure` so that `id` now maps to the closure of `vert`, computed by the function `udag_compute_closure`.
See the function `udag_add_vert`.

**Relation to Algorithm 1 in the paper**: Algorithm 1 listed in the paper (page 4) maintains two sets called "used IDs" and "invalid IDs."
In the UDAG model, used IDs are simply the IDs that have already been assigned to a vertex (`NatMap_find id udag.(udag_verts) <> None`).
The set of invalid IDs is not explicitly modeled.
It is implicitly assumed, as the vertices which violate the requirements in `udag_add_vert_pre` cannot be added to the DAG.
The condition `udag_add_vert_pre_lt` requires that the IDs listed in `udag_vert_preds` refer to existing vertices in the DAG,
which excludes the IDs in the invalid ID set.
The statement "discard record and return" corresponds to the cases where the requirements listed in `udag_add_vert_pre` fail,
and so the vertex is not added to the DAG.

In `UDAG.v` we proved many lemmas about the UDAG model which are generically useful to unauthenticated DAG-based protocols.
For example, `udag_closure_quorum` states: if `v` is a vertex in round `r` of the DAG,
then for every round `r'` with `1 <= r' < r`, the closure of `v` contains vertices in round `r'` of the DAG built by a quorum of processes.

### The MysticetiDAG Model

The MysticetiDAG model, implemented in `impl/mysticeti/MysticetiDAG.v`, formalizes the "supporter" and "certificate" patterns of Mysticeti
(Definition 1 and Definition 2 in the paper),
as well as the vertex creation rules of Mysticeti (Algorithm 3).
The modifications described in Algorithm 4 is implemented as a refinement layer on top of MysticetiDAG (see MysticetiCommit below).

The supporter relation (Definition 1 in the paper) is defined twice in MysticetiDAG.
The first definition (`Definition udag_supports`) is a direct translation of the definition in the original Mysticeti paper
(see page 3 of "Mysticeti: Reaching the Latency Limits with Uncertified DAGs").
With this definition, `udag_supports id id' udag` means vertex `id` is a supporter of vertex `id'` in the DAG `udag`.

In all situations that we use the supporter relation, `id` and `id'` are vertices in consecutive rounds.
In this case the definition can be simplified to `Definition udag_supports'`.
The two definitions are equivalent (see `Lemma udag_supports_equiv`).
Similarly, the certificate pattern (Definition 2 in the paper) is defined twice
(see `Definition udag_certifies`, `Definition udag_certifies'`, and `Lemma udag_certifies_equiv`).

We also defined two predicates called `mdag_vert_is_supporter` and `mdag_vert_is_certificate`.
The meaning of these definitions are as follows:
* `mdag_vert_is_supporter vert mdag`:
If the vertex `vert` is added to `mdag`, then either it is a genesis vertex, or it is supporter of some leader vertex of round `r-1`
where `r` is the round number of `vert`.
(Note that `vert` does not need to be currently existing in `mdag`.)
* `mdag_vert_is_certificate vert mdag`:
If the vertex `vert` is added to `mdag`, then either it is a vertex in round 1 or 2, or it is a certificate for some leader vertex of round `r-2`
where `r` is the round number of `vert`.
(Note that `vert` does not need to be currently existing in `mdag`.)

The state variables of the MysticetiDAG model are:
```coq
Record MDAG_State : Type := {
  mdag_udag : UDAG_State;
  mdag_recv_leader_verts : list (nat * nat); (* When process p receives leader vertex of round r and has never timed-out in any round r' > r,
                                                it sends a virtual msg recv_leader_vert (p, r) *)
  mdag_recv_certs : list (nat * nat); (* When process p receives 2f+1 vertices in round r+1 supporting a single leader vertex of round r,
                                         and has never timed-out in any round r' > r+1,
                                         it sends a virtual msg recv_cert (p, r) *)
  mdag_timeouts : list (nat * nat); (* When process p times-out in round r, it sends a virtual msg timeout (p, r) *)
  mdag_jumps : list (nat * nat); (* When process p decides to jump over round r, it sends a virtual msg jump (p, r) *)
}.
```

Note that this model does not capture the local timer of each process.
They will be introduced in the liveness proof.
For the safety proof, we simply assume timeout may happen arbitrarily.

According to Algorithm 3, there are three cases under which an honest process should create a vertex in round `r` (aside from creating the genesis vertex):
* The process has observed both a leader vertex of round `r-1` and `2f+1` supporters for some leader vertex of round `r-2`.
* The process is currently in round `r`, and its local timer has expired.
* The process has observed `2f+1` vertices in round `r`, and `r > curr_round`.

The creation of the genesis vertices can be considered as a special case of the first case.
Note that both `mdag_vert_is_supporter` and `mdag_vert_is_certificate` allows `vert` to be a genesis vertex.
Additionally, byzantine processes can create vertices arbitrarily, subject only to the constraints in the UDAG model.

As such, the transition relation of MysticetiDAG (see `Inductive mdag_step`) contains four kinds of steps for creating vertices:
* `mdag_step_add_vert_perfect`:
Add a "perfect" vertex, which is a vertex that both supports a leader vertex in round `r-1` and certifies a leader vertex in round `r-2`.
* `mdag_step_add_vert_timeout`:
Add a vertex upon a timeout event.
* `mdag_step_add_vert_quorum`:
Add a vertex in round `r` upon observing `2f+1` vertices in round `r`.
* `mdag_step_add_vert_byz`:
Add a vertex created by a byzantine process.

The other transition steps in `mdag_step` are related to the "virtual msgs" (consider them as history variables) in `MDAG_State`.
They are used in the liveness proof to keep track of which processes have received which vertices in the DAG,
and which processes have decided to jump over a certain round.

The difference between `mdag_step` and `mdag_step'` is that `mdag_step'` omits the `mdag_step_add_jump` transition in `mdag_step`.
This is because we will introduce restrictions on the round-jumping behavior in the MysticetiCommit model (see Algorithm 4 in the paper).

### The MysticetiCommit Model

The MysticetiCommit model, formalized in `impl/mysticeti/MysticetiCommit.v`,
builds upon the MysticetiDAG model and formalizes the vertex ordering algorithm of Mysticeti (Algorithm 2 in the paper).

The state variables of `MysticetiCommit` are:
```coq
Record MDAG_Commit_State : Type := {
  mcommit_mdag : MDAG_State;
  mcommit_commit : list (nat * nat); (* round * id. At least one honest process has marked round `r` as `Committed(id)`. *)
  mcommit_nack : list nat; (* round. At least one honest processes has marked round `r` as `Skipped`. *)
  mcommit_gct : bool; (* GCT, see below *)
}.
```

The vertex ordering algorithm of Mysticeti proceeds by having each honest process interpret its locally observed DAG,
and mark each round as either "Undecided," "Committed(id)," or "Skipped."
The variables `mcommit_commit` and `mcommit_nack` record the history of honest processes making these decisions.

As discussed in the paper, to maintain liveness of Mysticeti the round-jumping behavior of honest processes must be restricted.
For this reason we introduced a timepoint called the "global catchup time" (GCT),
which represents the timepoint every participant has upgraded to follow the new rules.
Before GCT, round-jumping may occur arbitrarily as in the current implementation.
After GCT, round-jumping must follow the rules spelled out in lines 4--7 in Algorithm 4 of the paper.
The MysticetiCommit model captures this rule with a boolean variable `mcommit_gct`.

The transition steps of the MysticetiCommit model are:
* `mcommit_step_mdag_step`: A step the occurred in the MysticetiDAG layer, except round-jumping which is subject to additional constraints.
* `mcommit_step_add_commit_primary`, `mcommit_step_add_commit_secondary`, `mcommit_step_add_nack`:
The direct/indirect commit/skip decision steps, as specified in Algorithm 2 in the paper.
* `mcommit_step_add_jump`: A round-jump step. Before GCT, there is no constraint.
After GCT, the constraint is that the process must have made a decision for round `r-2` when jumping over round `r`.
* `mcommit_step_start_gct`: Indicates that GCT is in effect.

### The Network-level Safety & Liveness Proofs

The safety theorem of Mysticeti states that no two honest processes make conflicting decisions for each round.
This is `Lemma mcommit_safety` and `Lemma mcommit_commit_eq` in `MysticetiCommit.v`.
The first lemma states that if some honest process has marked a round as `Committed(id)`, then no other process marks it as `Skipped`, and vice versa.
The second lemma states that only a single leader vertex from each round may be committed.

The liveness theorem of Mysticeti states that every leader vertex created by a synchronous process after GST
will eventually get `2f+1` certificates and become committed.
As discussed in the paper, the original proof of this proposition is wrong.
The new liveness theorem is split into two parts:
* The "weak liveness" theorem states that after GST, every leader vertex from synchronous processes
will get at least `f+1` certificates from synchronous processes.
This is not enough to commit the leader vertex, but is sufficient to prevent other processes from skipping the leader vertex.
* The "strong liveness" theorem states that after `max{GST, GCT}`, every leader vertex from synchronous processes will be marked as committed.

See pages 10--12 of the paper for an overview of the liveness proof.
The proof proceeds by dividing the timed-traces of Mysticeti into segments of Delta, and analyzing how the system state evolves over each period of Delta.
In `impl/mysticeti/proofs/LivenessWeak.v`, we define segmented traces of the MysticetiDAG model, and prove the weak liveness result.
In `impl/mysticeti/proofs/LivenessStrong.v`, we define segmented traces of the MysticetiCommit model (which are also segmented traces of MysticetiDAG),
and prove the strong liveness result.

Segmented traces of MysticetiDAG are captured by the `MysticetiWeakOracle` structure, which was briefly introduced in the paper (see Figure 11).
The liveness assumptions are listed in Appendix B.1, and are formalized as safety properties on how the system proceeds over each period of Delta.

For each reachable state of MysticetiDAG we define two variables called the "global active view" (GAV) and the "global remaining time" (GRT).
See page 14 of the paper.
These correspond to `Definition mys_ar n` and `Definition mys_rt n` where `n` is the number of periods of Delta that have elapsed since GST.

For each round `r` we defined nine "liveness checkpoints" (see Appendix B.2 of the paper).
The first eight checkpoints are formalized as predicates `mys_ckp1` through `mys_ckp8` in `LivenessWeak.v`.
The last checkpoint is `mys_ckp9` which is defined in `LivenessStrong.v` because it relies on the MysticetiCommit model.

The weak liveness theorem is `Lemma mys_liveness_weak` in `LivenessWeak.v`.
It says every round `r` after GST with a synchronous leader will get `f+1` certificates from synchronous processes.

The strong liveness theorem is `Lemma mys_liveness_strong` in `LivenessStrong.v`.
It says every round after `max{GST, GCT}`, every round `r` with a synchronous leader will reach checkpoint 9, which is that a leader vertex is committed.

### Relation to Propositions in the Paper

* Proposition 1 (Local DAGs are subgraphs of the global DAG):
this proposition is assumed implicitly in the model as the formal model does not yet capture the local DAG of each process.
We simply assume each process can see all vertices in the global DAG,
while this theorem says the vertices seen by each process is a subset of the global DAG.
* Proposition 2:
this proposition is encoded as a liveness assumption. See Appendix B.1.
* Theorem 1 (Safety of Mysticeti): `Lemma mcommit_safety` and `Lemma mcommit_commit_eq` in `MysticetiCommit.v`.
* Theorem 2: `Lemma mcommit_can_decide` in `MysticetiCommit.v`.
* Theorem 3 (Weak Liveness of Mysticeti): `Lemma mys_liveness_weak` in `LivenessWeak.v`.
* Theorem 4 (Strong Liveness of Mysticeti): `Lemma mys_liveness_strong` in `LivenessStrong.v`.
* Theorem 5: `Lemma every_vert_certificate` in `LivenessWeak.v`.
* Lemmas 1--3: Contained in the proof of Theorem 1.
* Lemma 4: `Lemma mdag_honest_vert_no_timeout_perfect` in `MysticetiDAG.v`.
* Figure 13: See `Lemma mys_live1` through `Lemma mys_live6` in `LivenessWeak.v`, and `Lemma mys_live7` through `Lemma mys_live9` in `LivenessStrong.v`.
For example, in Figure 13 there are arrows from Ckp2 to Ckp3, Ckp4, and Ckp6.
This corresponds to `Lemma mys_live4` where we show if round `r` has reached checkpoint 2,
then within Delta it reaches either checkpoint 3, checkpoint 4, or checkpoint 6.

## The LiDO-DAG Model

The "safety" and "liveness" theorems of the network model are not quite the same as what we usually consider as "safety" and "liveness" of consensus.
The usual definition is that, the local consensus logs observed by each node are consistent with each other,
and each vertex proposed by a non-faulty process is eventually appended to the consensus log.
The goal of the LiDO-DAG model is to provide a simple and reusable way to bridge the gap between the two definitions.

Appendix A of the manuscript contains an introduction to the LiDO-DAG model.
It is an abstract transition system that models how the consensus log grows during execution of consensus.
Specifically, we assume the execution of consensus is divided into "logical views."
In each logical view there is a predetermined leader, and the leader attempts to propose and commit a single leader vertex.
To do so, it must perform three abstract actions called "pull," "invoke," and "push."
* Pull: The leader determines the "parent" of the current view, which is the branch of consensus log that it plans to extend.
* Invoke: The leader extends the branch that was chosen in the Pull step with one leader vertex.
The LiDO-DAG model was originally designed to allow multiple Invoke operations in each round.
However, insofar as almost every DAG-based consensus protocol allows a leader to create only one leader vertex per round,
this aspect of LiDO-DAG is currently unused.
* Push: The leader commits the log extension proposed in the Invoke step.

We use a "cache-tree" structure to represent the outcome of each operation in each round.
A successful Pull, Invoke, or Push results in a tree node called ECache (leader Election), MCache (Method invocation), and CCache (Commit), respectively.
They are chained together to form a tree structure. See Figure 7 in the paper.

Safety of consensus can thus be formulated as: all committed log extensions (CCaches in the cache-tree) must be on the same branch of the tree.
Liveness of consensus is simply that eventually new leader vertices will be committed.

Formally, the LiDO-DAG model is specified in `specs/LidoDAG.v`.
Its state consists of two parts: an unauthenticated DAG (UDAG as introduced above) and a cache-tree structure called `ClientTree` in the codebase.
The `ClientTree` structure is specified in `specs/LidoClient.v`.
The state of `ClientTree` consists of the following data:
* For each logical view, there is a "round descriptor," called `Client_RoundDesc` in the codebase.
(Currently, "logical views" are just called "rounds" in the LiDO-DAG codebase, and we shall just use "round" below.)
* For each MCache existing in the cache tree, there is an associated consensus log `client_mcache_log`,
which represents the log of proposed data blocks in the branch of cache-tree up to that MCache.
* For each participating process there is a field `client_cmd`, which is used to keep track of which operation each process is attempting.
This is useful in consensus protocols where multiple leaders might be proposing blocks concurrently,
but for Mysticeti this is mostly unused.
* A field `client_proposed_vals` which keeps track of Invoke attempt history.
This is unused for Mysticeti.

The "round descriptor" consists of the following data:
* `client_round_pull_src`: The parent of the current logical view.
It is specified as `(round * version)`.
As discussed above, LiDO-DAG was originally designed to allow multiple Invoke steps in each round.
The version number is used to distinguish between multiple MCaches in each round.
Since only one leader vertex may be committed in each round in Mysticeti, the version number is always 0.
* `client_round_mcaches`: A finite map from version numbers to MCaches.
For LiDO-DAG, each MCache contains the ID of a leader vertex.
* `client_round_max_ccache`: The version number of the latest committed MCache in this round.
For Mysticeti, this field is `None` if no leader vertex has been committed, and `Some 0` if a leader vertex has been committed.

The safety proof of LiDO-DAG is formalized in `proofs/Safety.v` and `proofs/SafetyDAG.v`.
In `Safety.v`, we proved that all committed MCaches are on a single branch of the cache-tree.
Formally, this means if two MCaches `X` and `Y` are committed, then either `log(Y)` extends from `log(X)`, or `log(X)` extends from `log(Y)`.
This corresponds to the "leader vertex consensus log" discussed in the paper.
In `SafetyDAG.v`, we define how to expand the leader vertex consensus log into the full consensus log, and prove a similar safety proposition.
Together with the refinement proofs between LiDO-DAG and Mysticeti, this finishes the safety proof of Mysticeti.

Liveness of Mysticeti simply requires new CCaches are eventually added to the cache tree.
This is immediate from the network-level liveness proofs and the refinement relation described below.

## The Refinement Proof

The refinement proof between Mysticeti and LiDO-DAG is formalized in `impl/mysticeti/refine/Refine.v`.

As discussed in the paper, for Mysticeti there is a one-to-one correspondence between "logical views" and DAG rounds.
Indeed, in each DAG round there is a leader who is supposed to create at least one leader vertex,
and the other processes vote to support and certify this leader vertex.

For each reachable state of MysticetiCommit,
there is a highest round `r` such that all rounds up to `r` (inclusive) have been marked as Committed or Skipped by at least one synchronous process.
We call this the highest completed round, represented by `Definition mcommit_high_round`.
We proved that the highest completed round is well-defined (`Lemma mcommit_high_round_uniq` and `Lemma mcommit_high_round_ex`).
Also, it increases monotonically as network progresses (`Lemma mcommit_high_round_mono`).

We construct the LiDO-DAG cache tree up to the highest completed round.
We defined a refinement relation `R_mys_lidodag` between MysticetiCommit states and LiDO-DAG states.
This relation takes an additional parameter `r` which means the relation holds only up to round `r`.
The refinement result is `Lemma mcommit_refine_lidodag` which proves the existence of a LiDO-DAG cache tree
that satisfies the refinement relation up to the highest completed round.

# Documentation for the Sui Blockchain Testcase

In this part of the artifact we provide a testcase for the Sui blockchain that simulates the vertex delivery order shown in Figure 3 of the paper.
It is intended to show that the current implementation of Mysticeti handles round-jumping incorrectly.

We provide the testcase as a Git patch over the Sui codebase, specifically commit 2f52a72 (tag `testnet-v1.43.1`),
which was the latest version of Sui when the code audit was performed.
The testcase can be executed with the following steps:
1. Install the Rust toolchain version 1.81.
This can be done by installing the Docker base image `rust:1.81`.
2. Install `libclang-dev`: `apt-get update && apt-get install libclang-dev`.
3. Clone the Sui blockchain codebase: `git clone https://github.com/MystenLabs/sui --depth=1 --branch testnet-v1.43.1`.
4. Apply the patch: `git apply sui_testcase.patch`.
5. Install nextest: `cargo install cargo-nextest --locked --version 0.9.92`.
6. Build and run the testcase. The command below is adapted from the `scripts/simtest/cargo-simtest` file in the Sui codebase,
and provided in the `build_and_run.sh` file:
```bash
SIMTEST_STATIC_INIT_MOVE="$(pwd)/examples/move/basics" \
cargo nextest run test_mysticeti_liveness \
--package path+file://$(pwd)/consensus/simtests#consensus-simtests@0.1.0 \
--cargo-profile simulator \
--config 'build.rustflags = ["--cfg","msim"]' \
--config 'patch.crates-io.tokio.git = "https://github.com/MystenLabs/mysten-sim.git"' \
--config 'patch.crates-io.tokio.rev = "e06fc6fa9aeb978ffc621f8b27c06a404042279f"' \
--config 'patch.crates-io.futures-timer.git = "https://github.com/MystenLabs/mysten-sim.git"' \
--config 'patch.crates-io.futures-timer.rev = "e06fc6fa9aeb978ffc621f8b27c06a404042279f"' \
--config 'patch.crates-io.blst.git = "https://github.com/MystenLabs/mock-blst.git"' \
--config 'patch.crates-io.blst.rev = "630ca4d55de8e199e62c5b6a695c702d95fe6498"' \
--nocapture
```
Ignore all warnings.

The final step (building and running the testcase) may take more than an hour to complete.
For this reason, the provided Dockerfile only completes steps 1--5 above.
Step 6 is completed in `claims/claim2/run.sh`.

If you encounter the error
```text
error[E0308]: mismatched types
   --> crates/sui-storage/tests/key_value_tests.rs:421:5
    |
421 |     #[sim_test(config = "constant_latency_ms(250)")]
    |     ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
    |     |
    |     expected `Receiver<()>`, found `Receiver<_>`
    |     arguments to this function are incorrect
    |
    = note: `tokio::sync::oneshot::Receiver<_>` and `real_tokio::sync::oneshot::Receiver<()>` have similar names, but are actually distinct types
```
please execute `git restore Cargo.lock` and execute the build command again.

We have previously shared the testcase file with researchers at Mysten Labs and they have confirmed it represents a bug in the current implementation.

## Expected Outcome

In the output log of the testcase you should see the following two lines:
```text
2022-02-05T10:43:51.000000Z  INFO node{id=1 name="client"}: consensus_core::core: consensus/core/src/core.rs:604: Created block B1([0],KqNdoHdpLhNAznoEid/n1YfisvbaXpypscGFdlHal7Q=)(1644057831000ms;[B0([0],rquzEUtnQL1bMLK006vVHk8UqMYcBiQV+vinZrvOCgM=), B0([1],eDMXObWWmFR7PM9GMWum85qboMAKKzwONnU5npeqDeQ=), B0([2],nBr/taD6OGhSuh+qyp5Tw0ZBvgLbORyHcrqR0vbLjec=), B0([3],y605Di0bkb8yqDZj62ZaZhsAiLEsg88kQMCvuh/FbwE=)];0t;0c) for round 1

(some other events in between these two lines)

2022-02-05T10:43:51.000000Z  INFO node{id=1 name="client"}: consensus_simtests::my_simtests::test: consensus/simtests/src/tests/my_simtests.rs:302: New round has been received 4, resetting timer
```
There should be no other lines that contain the text "Created block."
This indicates that the implementation created a vertex in round 1,
and then jumped to round 4 without creating a vertex in round 2 or 3.
This violates the round-jumping rule we specified in Algorithm 4.

## Detailed explanation of the testcase

The patch files does two things:
1. It modifies the visibility of some crates, which makes writing the testcase easier.
2. It adds a file called `my_simtests.rs` under `consensus/simtests/src/tests/`.
This file contains a testcase function called `test_mysticeti_liveness`.

The first part of the function (up to line 157 of the file) is mostly copy-pasted from the files
`consensus/simtests/src/simtests.rs` and `consensus/core/src/authority_node.rs`.
The goal is to instantiate `struct Core` which is the state machine of the Mysticeti protocol.

The interesting part of `test_mysticeti_liveness` begins from line 161 of the file.
Here we simulate the vertex creation and delivery order shown in Figure 3 of the paper.
In Figure 3 the processes are numbered from 1 to 4, but in the Sui source code they are numbered from 0 to 3.
We follow the Sui convention here.
We use `block_X_Y` to denote the block in round `X` created by process `Y`.
We call the function `create_block_for_test()` to simulate vertex creation by processes 1 through 3.
We call the function `core.add_blocks()` to simulate delivering vertices to process 0.
The source code should be straightforward to read.

As discussed in the paper, the difference between the correct and the incorrect behavior
is whether the state machine creates a vertex in round 2 before jumping to round 3.
Therefore, we monitor the log output of the state machine for whether it creates blocks in round 2.
The experiment outcome we observed was that the state machine did not create a block in round 2.
This is sufficient to show that the current Sui implementation is incorrect and may lead to liveness attacks.
