#@args [session directory]
#@desc Build a graph representing interregional connectivity from the specified data source. Diffusion tractography or functional data may be used. In the former case the source is specified using the TractName option, and corresponding .trk and .trkl files must exist. (These are generated by "track".) Estimation of functional connectivity can be regularised by use of a shrinkage approach, which is particularly recommended when the number of time points is not appreciably larger than the number of target regions. Target regions may be specified individually or by type, using names from the parcellation's lookup table. The ParcellationConfidence variable controls the inclusiveness of the transformed parcellation: the closer to 0, the more inclusive. Self-connections are included by default.
#@group Graph and network analysis

library(tractor.reg)
library(tractor.session)
library(tractor.graph)

runExperiment <- function ()
{
    targetRegions <- getConfigVariable("TargetRegions", "cerebral_cortex")
    parcellationConfidence <- getConfigVariable("ParcellationConfidence", 0.2)
    graphName <- getConfigVariable("GraphName", "graph")
    selfConnections <- getConfigVariable("SelfConnections", TRUE)
    type <- getConfigVariable("Type", "diffusion", validValues=c("diffusion","functional"))
    tractName <- getConfigVariable("TractName", NULL, "character")
    regionTimeSeries <- getConfigVariable("RegionTimeSeries", "mean", validValues=c("mean","pc"))
    useShrinkage <- getConfigVariable("UseShrinkage", FALSE)
    varianceLambda <- getConfigVariable("VarianceShrinkageIntensity", NULL, "numeric")
    correlationLambda <- getConfigVariable("CorrelationShrinkageIntensity", NULL, "numeric")
    nuisanceRegressorFile <- getConfigVariable("NuisanceRegressorFile", NULL, "character")
    
    targetRegions <- splitAndConvertString(targetRegions, ",", fixed=TRUE)
    
    session <- attachMriSession(ifelse(nArguments()==0, ".", Arguments[1]))
    targets <- resolveRegions(targetRegions, session, space=type, parcellationConfidence)
    nRegions <- length(targets$labels)
    report(OL$Info, "Using #{nRegions} matched target regions")
    
    if (type == "diffusion")
    {
        library(tractor.nt)
        library(tractor.track)
        
        streamSource <- readStreamlines(tractName)
        
        # Find the indices of streamlines passing through each region
        # If some of the targets come from non-parcellation files, this involves a relabelling pass
        if (all(targets$fromParcellation))
            matchingIndices <- streamSource$matchLabels(targets$indices, combine="none")
        else
            matchingIndices <- streamSource$matchLabels(targets$indices, targets$image, combine="none")
        
        report(OL$Info, "Finding streamlines passing through each region")
        regionLocations <- matrix(NA, nrow=nRegions, ncol=3)    # Physical location of each region's spatial median, in mm
        voxelCount      <- integer(nRegions)                    # Number of voxels
        volume          <- numeric(nRegions)                    # Volume in mm^3
        for (i in seq_len(nRegions))
        {
            report(OL$Verbose, "Matching region \"#{targets$labels[i]}\"")
            regionVoxels <- targets$image$find(targets$indices[i])
            regionLocations[i,] <- RNifti::voxelToWorld(apply(regionVoxels,2,median), targets$image, simple=TRUE)
            voxelCount[i] <- nrow(regionVoxels)
            volume[i] <- voxelCount[i] * abs(prod(targets$image$getVoxelDimensions()))
        }
        
        fa <- session$getImageByType("FA", "diffusion")
        md <- session$getImageByType("MD", "diffusion")
        
        report(OL$Info, "Creating connectivity matrix")
        edgeList         <- matrix(NA, nrow=0, ncol=2)      # Edge list, one edge per row
        nStreamlines     <- numeric(0)                      # Number of streamlines forming the connection
        binaryFA         <- numeric(0)                      # Mean FA, counting voxels only once
        weightedFA       <- numeric(0)                      # Mean FA, counting voxels each time they are visited
        binaryMD         <- numeric(0)                      # Mean MD, counting voxels only once
        weightedMD       <- numeric(0)                      # Mean MD, counting voxels each time they are visited
        streamlineLength <- numeric(0)                      # Average length of streamlines connecting each pair of regions
        uniqueVoxels     <- numeric(0)                      # Number of unique voxels visited
        voxelVisits      <- numeric(0)                      # Number of voxel visits across all streamlines
        
        for (i in seq_len(nRegions))
        {
            report(OL$Verbose, "Finding connections for region \"#{targets$labels[i]}\"")
            
            for (j in seq_len(ifelse(selfConnections,i,i-1)))
            {
                # The subset of streamline indices connecting these two regions
                currentStreamlineIndices <- intersect(matchingIndices[[i]], matchingIndices[[j]])
                nCurrentStreamlines <- length(currentStreamlineIndices)
                if (nCurrentStreamlines == 0)
                    next
                
                edgeList <- rbind(edgeList, c(j,i))
                nStreamlines <- c(nStreamlines, nCurrentStreamlines)
                
                info <- streamSource$select(currentStreamlineIndices)$process(requireStreamlines=FALSE, requireMap=TRUE, requireLengths=TRUE)
                visitedLocations <- info$map$find()
                
                binaryFA <- c(binaryFA, mean(fa[visitedLocations],na.rm=TRUE))
                weightedFA <- c(weightedFA, weighted.mean(fa[visitedLocations],info$map[visitedLocations],na.rm=TRUE))
                binaryMD <- c(binaryMD, mean(md[visitedLocations],na.rm=TRUE))
                weightedMD <- c(weightedMD, weighted.mean(md[visitedLocations],info$map[visitedLocations],na.rm=TRUE))
                streamlineLength <- c(streamlineLength, mean(info$lengths))
                uniqueVoxels <- c(uniqueVoxels, nrow(visitedLocations))
                voxelVisits <- c(voxelVisits, sum(info$map[visitedLocations],na.rm=TRUE))
            }
        }
        
        report(OL$Info, "Creating and writing graph")
        graph <- asGraph(edgeList, edgeList=TRUE, directed=FALSE, selfConnections=selfConnections, nVertices=nRegions)
        graph$setVertexAttributes(name=targets$labels, voxelCount=voxelCount, volume=volume)
        graph$setVertexLocations(regionLocations, "mm", paste(session$getDirectory(),"diffusion",sep=":"))
        graph$setEdgeAttributes(nStreamlines=nStreamlines, binaryFA=binaryFA, weightedFA=weightedFA, binaryMD=binaryMD, weightedMD=weightedMD, streamlineLength=streamlineLength, uniqueVoxels=uniqueVoxels, voxelVisits=voxelVisits)
        graph$serialise(graphName)
    }
    else if (type == "functional")
    {
        library(corpcor)
        
        report(OL$Info, "Reading data")
        data <- session$getImageByType("data", "functional")
        nVolumes <- dim(data)[4]
        
        report(OL$Info, "Calculating representative time series for each region")
        timeSeries      <- matrix(NA, nrow=nVolumes, ncol=nRegions)
        regionLocations <- matrix(NA, nrow=nRegions, ncol=3)    # Physical location of each region's spatial median, in mm
        voxelCount      <- integer(nRegions)                    # Number of voxels
        volume          <- numeric(nRegions)                    # Volume in mm^3
        for (i in seq_len(nRegions))
        {
            report(OL$Verbose, "Extracting region \"#{targets$labels[i]}\"")
            index <- targets$indices[i]
            regionImage <- targets$image$copy()$map(function(x) ifelse(x==index,1L,0L), sparse=TRUE)
            allLocations <- regionImage$getNonzeroIndices(array=TRUE)
            regionLocations[i,] <- apply(allLocations, 2, median)
            regionLocations[i,] <- RNifti::voxelToWorld(regionLocations[i,], regionImage, simple=TRUE)
            voxelCount[i] <- length(regionImage$getNonzeroIndices(array=FALSE))
            volume[i] <- voxelCount[i] * abs(prod(regionImage$getVoxelDimensions()))
            
            allTimeSeries <- t(data$apply(4, "[", allLocations))
            if (regionTimeSeries == "mean")
            {
                allTimeSeries <- scale(allTimeSeries)
                timeSeries[,i] <- rowMeans(allTimeSeries, na.rm=TRUE)
                report(OL$Verbose, "The mean time series captures #{var(timeSeries[,i])/sum(diag(var(allTimeSeries)))*100}% of the variance", round=2)
            }
            else if (regionTimeSeries == "pc")
            {
                zeroVariance <- (colSums(cov(allTimeSeries)) == 0)
                pca <- prcomp(allTimeSeries[,!zeroVariance], scale.=TRUE)
                timeSeries[,i] <- pca$x[,1]
                variances <- pca$sdev^2
                report(OL$Verbose, "The first PC captures #{variances[1]/sum(variances)*100}% of the variance", round=2)
            }
        }
        
        if (is.null(nuisanceRegressorFile))
            nNuisanceRegressors <- 0L
        else
        {
            nuisanceRegressors <- as.matrix(read.table(nuisanceRegressorFile))
            if (nrow(nuisanceRegressors) != nVolumes)
            {
                if (ncol(nuisanceRegressors) == nVolumes)
                    nuisanceRegressors <- t(nuisanceRegressors)
                else
                    report(OL$Error, "Nuisance regressor file doesn't match data dimensions")
            }
            
            nNuisanceRegressors <- ncol(nuisanceRegressors)
            report(OL$Info, "Incorporating #{nNuisanceRegressors} nuisance regressor(s)")
            timeSeries <- cbind(timeSeries, nuisanceRegressors)
        }
        
        report(OL$Info, "Calculating interregional correlations")
        if (useShrinkage)
        {
            dropNull <- function(x) Filter(Negate(is.null), x)
            covariance <- do.call("cov.shrink", dropNull(list(timeSeries,verbose=FALSE,lambda=correlationLambda,lambda.var=varianceLambda)))
            correlation <- do.call("cor.shrink", dropNull(list(timeSeries,verbose=FALSE,lambda=correlationLambda)))
            precision <- do.call("invcov.shrink", dropNull(list(timeSeries,verbose=FALSE,lambda=correlationLambda,lambda.var=varianceLambda)))
            partialCorrelation <- do.call("pcor.shrink", dropNull(list(timeSeries,verbose=FALSE,lambda=correlationLambda)))
        }
        else
        {
            covariance <- cov(timeSeries)
            correlation <- cor(timeSeries)
            precision <- pseudoinverse(covariance)
            partialCorrelation <- cor2pcor(correlation)
        }
        
        isFiniteAndNonzero <- function(x) is.finite(x) & (x != 0)
        adjacencyMatrix <- as.integer(isFiniteAndNonzero(covariance) & isFiniteAndNonzero(correlation) & isFiniteAndNonzero(precision) & isFiniteAndNonzero(partialCorrelation))
        dim(adjacencyMatrix) <- rep(nRegions+nNuisanceRegressors, 2)
        adjacencyMatrix <- adjacencyMatrix[seq_len(nRegions),seq_len(nRegions)]
        dimnames(adjacencyMatrix) <- list(targets$labels, targets$labels)
        
        report(OL$Info, "Creating and writing graph")
        graph <- asGraph(adjacencyMatrix, edgeList=FALSE, directed=FALSE, selfConnections=selfConnections)
        edges <- graph$getEdges()
        graph$setVertexAttributes(name=targets$labels, voxelCount=voxelCount, volume=volume)
        graph$setVertexLocations(regionLocations, "mm", paste(session$getDirectory(),"functional",sep=":"))
        graph$setEdgeAttributes(covariance=covariance[edges], correlation=correlation[edges], precision=precision[edges], partialCorrelation=partialCorrelation[edges])
        graph$serialise(graphName)
    }
}
