"""`eds.congestive_heart_failure` pipeline"""

from typing import Optional

from edsnlp.core import PipelineProtocol
from edsnlp.pipes.base import SpanSetterArg
from edsnlp.pipes.core.contextual_matcher.models import FullConfig

from ..base import DisorderMatcher
from .patterns import default_patterns


class CongestiveHeartFailureMatcher(DisorderMatcher):
    """
    The `eds.congestive_heart_failure` pipeline component extracts mentions of
    congestive heart failure. It will notably match:

    - Mentions of various diseases (see below)
    - Heart transplantation
    - AF (Atrial Fibrillation)
    - Pacemaker

    ??? info "Details of the used patterns"
        ```{ .python .no-check }
        # fmt: off
        --8<-- "edsnlp/pipes/ner/disorders/congestive_heart_failure/patterns.py"
        # fmt: on
        ```

    Usage
    -----
    ```python
    import edsnlp, edsnlp.pipes as eds

    nlp = edsnlp.blank("eds")
    nlp.add_pipe(eds.sentences())
    nlp.add_pipe(
        eds.normalizer(
            accents=True,
            lowercase=True,
            quotes=True,
            spaces=True,
            pollution=dict(
                information=True,
                bars=True,
                biology=True,
                doctors=True,
                web=True,
                coding=True,
                footer=True,
            ),
        ),
    )
    nlp.add_pipe(eds.congestive_heart_failure())
    ```

    Below are a few examples:

    --8<-- "docs/assets/fragments/congestive-heart-failure-examples.md"

    Parameters
    ----------
    nlp : Optional[PipelineProtocol]
        The pipeline object
    name : str,
        The name of the component
    patterns : FullConfig
        The patterns to use for matching
    label : str
        The label to use for the `Span` object and the extension
    span_setter : SpanSetterArg
        How to set matches on the doc

    Authors and citation
    --------------------
    The `eds.congestive_heart_failure` component was developed by AP-HP's Data Science team with a
    team of medical experts, following the insights of the algorithm proposed
    by [@petitjean_2024].
    """  # noqa: E501

    def __init__(
        self,
        nlp: Optional[PipelineProtocol],
        name: str = "congestive_heart_failure",
        *,
        patterns: FullConfig = default_patterns,
        label: str = "congestive_heart_failure",
        span_setter: SpanSetterArg = {"ents": True, "congestive_heart_failure": True},
    ):
        super().__init__(
            nlp=nlp,
            name=name,
            label=label,
            patterns=patterns,
            span_setter=span_setter,
        )
