"""`eds.cerebrovascular_accident` pipeline"""

from typing import List, Optional

from spacy.tokens import Doc, Span

from edsnlp.core import PipelineProtocol
from edsnlp.pipes.base import SpanSetterArg
from edsnlp.pipes.core.contextual_matcher.models import FullConfig

from ..base import DisorderMatcher
from .patterns import default_patterns


class CerebrovascularAccidentMatcher(DisorderMatcher):
    """
    The `eds.cerebrovascular_accident` pipeline component extracts mentions of
    cerebrovascular accident. It will notably match:

    - Mentions of AVC/AIT
    - Mentions of bleeding, hemorrhage, thrombus, ischemia, etc., localized in the brain

    ??? info "Details of the used patterns"
        ```{ .python .no-check }
        # fmt: off
        --8<-- "edsnlp/pipes/ner/disorders/cerebrovascular_accident/patterns.py"
        # fmt: on
        ```

    Extensions
    ----------
    On each span `span` that match, the following attributes are available:

    - `span._.detailed_status`: set to None

    Usage
    -----
    ```python
    import edsnlp, edsnlp.pipes as eds

    nlp = edsnlp.blank("eds")
    nlp.add_pipe(eds.sentences())
    nlp.add_pipe(
        eds.normalizer(
            accents=True,
            lowercase=True,
            quotes=True,
            spaces=True,
            pollution=dict(
                information=True,
                bars=True,
                biology=True,
                doctors=True,
                web=True,
                coding=True,
                footer=True,
            ),
        ),
    )
    nlp.add_pipe(eds.cerebrovascular_accident())
    ```

    Below are a few examples:

    --8<-- "docs/assets/fragments/cerebrovascular-accident-examples.md"

    Parameters
    ----------
    nlp : Optional[PipelineProtocol]
        The pipeline
    name : Optional[str]
        The name of the component
    patterns: FullConfig
        The patterns to use for matching
    label : str
        The label to use for the `Span` object and the extension
    span_setter : SpanSetterArg
        How to set matches on the doc

    Authors and citation
    --------------------
    The `eds.cerebrovascular_accident` component was developed by AP-HP's Data Science team with a
    team of medical experts, following the insights of the algorithm proposed
    by [@petitjean_2024].
    """  # noqa: E501

    def __init__(
        self,
        nlp: Optional[PipelineProtocol],
        name: str = "cerebrovascular_accident",
        *,
        patterns: FullConfig = default_patterns,
        label: str = "cerebrovascular_accident",
        span_setter: SpanSetterArg = {"ents": True, "cerebrovascular_accident": True},
    ):
        super().__init__(
            nlp=nlp,
            name=name,
            label=label,
            patterns=patterns,
            include_assigned=False,
            span_setter=span_setter,
        )

    def process(self, doc: Doc) -> List[Span]:
        for span in super().process(doc):
            if (span._.source == "with_localization") and (
                "brain_localized" not in span._.assigned.keys()
            ):
                continue

            if span._.source == "ischemia":
                if "brain" not in span._.assigned.keys():
                    continue

            yield span
