#ifndef NEUTRINO_H
#define NEUTRINO_H

// Define available neutrino cooling methods
#define NEUTRINO_METHOD_kipp 1
#define NEUTRINO_METHOD_sneut5 2

#include <AMReX_REAL.H>
#include <AMReX_Array.H>
#include <microphysics_autodiff.H>
#include <fundamental_constants.H>

#include <kipp.H>
#include <sneut5.H>

using namespace amrex::literals;

#if NEUTRINO_METHOD == NEUTRINO_METHOD_kipp
const std::string neutrino_name = "kipp";
#elif NEUTRINO_METHOD == NEUTRINO_METHOD_sneut5
const std::string neutrino_name = "sneut5";
#endif

template <int do_derivatives, typename number_t>
AMREX_GPU_HOST_DEVICE AMREX_INLINE
void neutrino_cooling(const number_t& temp, const amrex::Real dens,
                          const number_t& abar, const number_t& zbar,
                          number_t& sneut, number_t& dsneutdt,
                          number_t& dsneutdd, number_t& dsnuda, number_t& dsnudz,
                          amrex::Real& pair, amrex::Real& phot,
                          amrex::Real& plas, amrex::Real& brem)
{
#if NEUTRINO_METHOD == NEUTRINO_METHOD_kipp
    kipp<do_derivatives>(temp, dens, abar, zbar,
                         sneut, dsneutdt, dsneutdd, dsnuda, dsnudz,
                         pair, phot, plas, brem);
#elif NEUTRINO_METHOD == NEUTRINO_METHOD_sneut5
    sneut5<do_derivatives>(temp, dens, abar, zbar,
                           sneut, dsneutdt, dsneutdd, dsnuda, dsnudz,
                           pair, phot, plas, brem);
#endif
}

// simpler interface for when we do not care about the individual contributions
template <int do_derivatives, typename number_t>
AMREX_GPU_HOST_DEVICE AMREX_INLINE
void neutrino_cooling(const number_t& temp, const amrex::Real dens,
                      const number_t& abar, const number_t& zbar,
                      number_t& sneut, number_t& dsneutdt,
                      number_t& dsneutdd, number_t& dsnuda, number_t& dsnudz)
{
    amrex::Real pair, phot, plas, brem;

#if NEUTRINO_METHOD == NEUTRINO_METHOD_kipp
    kipp<do_derivatives>(temp, dens, abar, zbar,
                         sneut, dsneutdt, dsneutdd, dsnuda, dsnudz,
                         pair, phot, plas, brem);
#elif NEUTRINO_METHOD == NEUTRINO_METHOD_sneut5
    sneut5<do_derivatives>(temp, dens, abar, zbar,
                           sneut, dsneutdt, dsneutdd, dsnuda, dsnudz,
                           pair, phot, plas, brem);
#endif
}

#endif // NEUTRINO_H
