#ifndef actual_rhs_H
#define actual_rhs_H

#include <AMReX_REAL.H>
#include <AMReX_Array.H>

#include <extern_parameters.H>
#include <actual_network.H>
#include <burn_type.H>
#include <jacobian_utilities.H>
#ifdef SCREENING
#include <screen.H>
#endif
#include <microphysics_autodiff.H>
#ifdef NEUTRINOS
#include <neutrino.H>
#endif
#include <reaclib_rates.H>
#include <table_rates.H>

using namespace amrex::literals;
using namespace ArrayUtil;

using namespace Species;
using namespace Rates;

using namespace rate_tables;


template<class T>
AMREX_GPU_HOST_DEVICE AMREX_INLINE
void ener_gener_rate(T const& dydt, amrex::Real& enuc)
{

    // Computes the instantaneous energy generation rate (from the nuclei)

    // This is basically e = m c**2

    enuc = 0.0_rt;

    for (int n = 1; n <= NumSpec; ++n) {
        enuc += dydt(n) * network::mion(n);
    }

    enuc *= C::enuc_conv2;
}


template <int do_T_derivatives, typename T>
AMREX_GPU_HOST_DEVICE AMREX_INLINE
void evaluate_rates(const burn_t& state, T& rate_eval) {


    // create molar fractions

    amrex::Array1D<amrex::Real, 1, NumSpec> Y;
    for (int n = 1; n <= NumSpec; ++n) {
        Y(n) = state.xn[n-1] * aion_inv[n-1];
    }

    [[maybe_unused]] amrex::Real rhoy = state.rho * state.y_e;

    // Calculate Reaclib rates

    using number_t = std::conditional_t<do_T_derivatives, autodiff::dual, amrex::Real>;
    number_t temp = state.T;
    if constexpr (do_T_derivatives) {
        // seed the dual number for temperature before calculating anything with it
        autodiff::seed(temp);
    }

#ifdef SCREENING
    plasma_state_t<number_t> pstate{};
    fill_plasma_state(pstate, temp, state.rho, Y);
#endif

    tf_t tfactors = evaluate_tfactors(state.T);

    fill_reaclib_rates<do_T_derivatives, T>(tfactors, rate_eval);


#ifdef SCREENING
    // Evaluate screening factors

    amrex::Real ratraw, dratraw_dT;
    amrex::Real scor, dscor_dt;
    [[maybe_unused]] amrex::Real scor2, dscor2_dt;


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 1.0_rt, 1.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_p_to_d_weak_bet_pos_);
    rate_eval.screened_rates(k_p_p_to_d_weak_bet_pos_) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_p_to_d_weak_bet_pos_);
        rate_eval.dscreened_rates_dT(k_p_p_to_d_weak_bet_pos_) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_p_to_d_weak_electron_capture);
    rate_eval.screened_rates(k_p_p_to_d_weak_electron_capture) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_p_to_d_weak_electron_capture);
        rate_eval.dscreened_rates_dT(k_p_p_to_d_weak_electron_capture) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_n_p_p_to_p_d);
    rate_eval.screened_rates(k_n_p_p_to_p_d) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_n_p_p_to_p_d);
        rate_eval.dscreened_rates_dT(k_n_p_p_to_p_d) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_p_He4_He4_to_He3_Be7);
    rate_eval.screened_rates(k_p_p_He4_He4_to_He3_Be7) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_p_He4_He4_to_He3_Be7);
        rate_eval.dscreened_rates_dT(k_p_p_He4_He4_to_He3_Be7) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 1.0_rt, 2.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_d_to_He3);
    rate_eval.screened_rates(k_p_d_to_He3) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_d_to_He3);
        rate_eval.dscreened_rates_dT(k_p_d_to_He3) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_d_to_n_p_p);
    rate_eval.screened_rates(k_p_d_to_n_p_p) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_d_to_n_p_p);
        rate_eval.dscreened_rates_dT(k_p_d_to_n_p_p) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 2.0_rt, 1.0_rt, 2.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_d_d_to_He4);
    rate_eval.screened_rates(k_d_d_to_He4) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_d_d_to_He4);
        rate_eval.dscreened_rates_dT(k_d_d_to_He4) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_d_d_to_n_He3);
    rate_eval.screened_rates(k_d_d_to_n_He3) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_d_d_to_n_He3);
        rate_eval.dscreened_rates_dT(k_d_d_to_n_He3) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 2.0_rt, 2.0_rt, 4.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_d_to_Li6);
    rate_eval.screened_rates(k_He4_d_to_Li6) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_d_to_Li6);
        rate_eval.dscreened_rates_dT(k_He4_d_to_Li6) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 2.0_rt, 3.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_He3_to_He4_weak_bet_pos_);
    rate_eval.screened_rates(k_p_He3_to_He4_weak_bet_pos_) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_He3_to_He4_weak_bet_pos_);
        rate_eval.dscreened_rates_dT(k_p_He3_to_He4_weak_bet_pos_) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 2.0_rt, 3.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_He3_to_Be7);
    rate_eval.screened_rates(k_He4_He3_to_Be7) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_He3_to_Be7);
        rate_eval.dscreened_rates_dT(k_He4_He3_to_Be7) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_He3_to_p_Li6);
    rate_eval.screened_rates(k_He4_He3_to_p_Li6) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_He3_to_p_Li6);
        rate_eval.dscreened_rates_dT(k_He4_He3_to_p_Li6) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 3.0_rt, 6.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Li6_to_Be7);
    rate_eval.screened_rates(k_p_Li6_to_Be7) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Li6_to_Be7);
        rate_eval.dscreened_rates_dT(k_p_Li6_to_Be7) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Li6_to_He4_He3);
    rate_eval.screened_rates(k_p_Li6_to_He4_He3) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Li6_to_He4_He3);
        rate_eval.dscreened_rates_dT(k_p_Li6_to_He4_He3) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 3.0_rt, 6.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Li6_to_B10);
    rate_eval.screened_rates(k_He4_Li6_to_B10) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Li6_to_B10);
        rate_eval.dscreened_rates_dT(k_He4_Li6_to_B10) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Li6_to_p_Be9);
    rate_eval.screened_rates(k_He4_Li6_to_p_Be9) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Li6_to_p_Be9);
        rate_eval.dscreened_rates_dT(k_He4_Li6_to_p_Be9) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 3.0_rt, 7.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Li7_to_B11);
    rate_eval.screened_rates(k_He4_Li7_to_B11) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Li7_to_B11);
        rate_eval.dscreened_rates_dT(k_He4_Li7_to_B11) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Li7_to_n_B10);
    rate_eval.screened_rates(k_He4_Li7_to_n_B10) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Li7_to_n_B10);
        rate_eval.dscreened_rates_dT(k_He4_Li7_to_n_B10) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 4.0_rt, 7.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Be7_to_B8);
    rate_eval.screened_rates(k_p_Be7_to_B8) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Be7_to_B8);
        rate_eval.dscreened_rates_dT(k_p_Be7_to_B8) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 4.0_rt, 9.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Be9_to_B10);
    rate_eval.screened_rates(k_p_Be9_to_B10) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Be9_to_B10);
        rate_eval.dscreened_rates_dT(k_p_Be9_to_B10) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Be9_to_He4_Li6);
    rate_eval.screened_rates(k_p_Be9_to_He4_Li6) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Be9_to_He4_Li6);
        rate_eval.dscreened_rates_dT(k_p_Be9_to_He4_Li6) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Be9_to_d_He4_He4);
    rate_eval.screened_rates(k_p_Be9_to_d_He4_He4) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Be9_to_d_He4_He4);
        rate_eval.dscreened_rates_dT(k_p_Be9_to_d_He4_He4) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Be9_to_n_p_He4_He4);
    rate_eval.screened_rates(k_p_Be9_to_n_p_He4_He4) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Be9_to_n_p_He4_He4);
        rate_eval.dscreened_rates_dT(k_p_Be9_to_n_p_He4_He4) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 5.0_rt, 11.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_B11_to_C12);
    rate_eval.screened_rates(k_p_B11_to_C12) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_B11_to_C12);
        rate_eval.dscreened_rates_dT(k_p_B11_to_C12) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_B11_to_He4_He4_He4);
    rate_eval.screened_rates(k_p_B11_to_He4_He4_He4) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_B11_to_He4_He4_He4);
        rate_eval.dscreened_rates_dT(k_p_B11_to_He4_He4_He4) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 6.0_rt, 12.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_C12_to_N13);
    rate_eval.screened_rates(k_p_C12_to_N13) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_C12_to_N13);
        rate_eval.dscreened_rates_dT(k_p_C12_to_N13) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 6.0_rt, 12.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_C12_to_O16);
    rate_eval.screened_rates(k_He4_C12_to_O16) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_C12_to_O16);
        rate_eval.dscreened_rates_dT(k_He4_C12_to_O16) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_C12_to_n_O15);
    rate_eval.screened_rates(k_He4_C12_to_n_O15) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_C12_to_n_O15);
        rate_eval.dscreened_rates_dT(k_He4_C12_to_n_O15) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_C12_to_p_N15);
    rate_eval.screened_rates(k_He4_C12_to_p_N15) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_C12_to_p_N15);
        rate_eval.dscreened_rates_dT(k_He4_C12_to_p_N15) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 6.0_rt, 13.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_C13_to_N14);
    rate_eval.screened_rates(k_p_C13_to_N14) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_C13_to_N14);
        rate_eval.dscreened_rates_dT(k_p_C13_to_N14) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_C13_to_n_N13);
    rate_eval.screened_rates(k_p_C13_to_n_N13) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_C13_to_n_N13);
        rate_eval.dscreened_rates_dT(k_p_C13_to_n_N13) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_C13_to_He4_B10);
    rate_eval.screened_rates(k_p_C13_to_He4_B10) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_C13_to_He4_B10);
        rate_eval.dscreened_rates_dT(k_p_C13_to_He4_B10) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 6.0_rt, 14.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_C14_to_N15);
    rate_eval.screened_rates(k_p_C14_to_N15) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_C14_to_N15);
        rate_eval.dscreened_rates_dT(k_p_C14_to_N15) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_C14_to_n_N14);
    rate_eval.screened_rates(k_p_C14_to_n_N14) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_C14_to_n_N14);
        rate_eval.dscreened_rates_dT(k_p_C14_to_n_N14) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_C14_to_He4_B11);
    rate_eval.screened_rates(k_p_C14_to_He4_B11) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_C14_to_He4_B11);
        rate_eval.dscreened_rates_dT(k_p_C14_to_He4_B11) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 6.0_rt, 14.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_C14_to_O18);
    rate_eval.screened_rates(k_He4_C14_to_O18) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_C14_to_O18);
        rate_eval.dscreened_rates_dT(k_He4_C14_to_O18) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_C14_to_n_O17);
    rate_eval.screened_rates(k_He4_C14_to_n_O17) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_C14_to_n_O17);
        rate_eval.dscreened_rates_dT(k_He4_C14_to_n_O17) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 7.0_rt, 13.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_N13_to_O14);
    rate_eval.screened_rates(k_p_N13_to_O14) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_N13_to_O14);
        rate_eval.dscreened_rates_dT(k_p_N13_to_O14) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 7.0_rt, 14.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_N14_to_O15);
    rate_eval.screened_rates(k_p_N14_to_O15) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_N14_to_O15);
        rate_eval.dscreened_rates_dT(k_p_N14_to_O15) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_N14_to_n_O14);
    rate_eval.screened_rates(k_p_N14_to_n_O14) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_N14_to_n_O14);
        rate_eval.dscreened_rates_dT(k_p_N14_to_n_O14) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 7.0_rt, 14.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_N14_to_F18);
    rate_eval.screened_rates(k_He4_N14_to_F18) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_N14_to_F18);
        rate_eval.dscreened_rates_dT(k_He4_N14_to_F18) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_N14_to_n_F17);
    rate_eval.screened_rates(k_He4_N14_to_n_F17) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_N14_to_n_F17);
        rate_eval.dscreened_rates_dT(k_He4_N14_to_n_F17) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_N14_to_p_O17);
    rate_eval.screened_rates(k_He4_N14_to_p_O17) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_N14_to_p_O17);
        rate_eval.dscreened_rates_dT(k_He4_N14_to_p_O17) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 7.0_rt, 15.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_N15_to_O16);
    rate_eval.screened_rates(k_p_N15_to_O16) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_N15_to_O16);
        rate_eval.dscreened_rates_dT(k_p_N15_to_O16) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_N15_to_n_O15);
    rate_eval.screened_rates(k_p_N15_to_n_O15) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_N15_to_n_O15);
        rate_eval.dscreened_rates_dT(k_p_N15_to_n_O15) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_N15_to_He4_C12);
    rate_eval.screened_rates(k_p_N15_to_He4_C12) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_N15_to_He4_C12);
        rate_eval.dscreened_rates_dT(k_p_N15_to_He4_C12) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 7.0_rt, 15.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_N15_to_F19);
    rate_eval.screened_rates(k_He4_N15_to_F19) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_N15_to_F19);
        rate_eval.dscreened_rates_dT(k_He4_N15_to_F19) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_N15_to_n_F18);
    rate_eval.screened_rates(k_He4_N15_to_n_F18) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_N15_to_n_F18);
        rate_eval.dscreened_rates_dT(k_He4_N15_to_n_F18) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_N15_to_p_O18);
    rate_eval.screened_rates(k_He4_N15_to_p_O18) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_N15_to_p_O18);
        rate_eval.dscreened_rates_dT(k_He4_N15_to_p_O18) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 8.0_rt, 14.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_O14_to_Ne18);
    rate_eval.screened_rates(k_He4_O14_to_Ne18) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_O14_to_Ne18);
        rate_eval.dscreened_rates_dT(k_He4_O14_to_Ne18) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_O14_to_p_F17);
    rate_eval.screened_rates(k_He4_O14_to_p_F17) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_O14_to_p_F17);
        rate_eval.dscreened_rates_dT(k_He4_O14_to_p_F17) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 8.0_rt, 15.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_O15_to_Ne19);
    rate_eval.screened_rates(k_He4_O15_to_Ne19) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_O15_to_Ne19);
        rate_eval.dscreened_rates_dT(k_He4_O15_to_Ne19) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_O15_to_n_Ne18);
    rate_eval.screened_rates(k_He4_O15_to_n_Ne18) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_O15_to_n_Ne18);
        rate_eval.dscreened_rates_dT(k_He4_O15_to_n_Ne18) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_O15_to_p_F18);
    rate_eval.screened_rates(k_He4_O15_to_p_F18) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_O15_to_p_F18);
        rate_eval.dscreened_rates_dT(k_He4_O15_to_p_F18) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 8.0_rt, 16.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_O16_to_F17);
    rate_eval.screened_rates(k_p_O16_to_F17) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_O16_to_F17);
        rate_eval.dscreened_rates_dT(k_p_O16_to_F17) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_O16_to_He4_N13);
    rate_eval.screened_rates(k_p_O16_to_He4_N13) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_O16_to_He4_N13);
        rate_eval.dscreened_rates_dT(k_p_O16_to_He4_N13) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 8.0_rt, 16.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_O16_to_Ne20);
    rate_eval.screened_rates(k_He4_O16_to_Ne20) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_O16_to_Ne20);
        rate_eval.dscreened_rates_dT(k_He4_O16_to_Ne20) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_O16_to_n_Ne19);
    rate_eval.screened_rates(k_He4_O16_to_n_Ne19) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_O16_to_n_Ne19);
        rate_eval.dscreened_rates_dT(k_He4_O16_to_n_Ne19) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_O16_to_p_F19);
    rate_eval.screened_rates(k_He4_O16_to_p_F19) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_O16_to_p_F19);
        rate_eval.dscreened_rates_dT(k_He4_O16_to_p_F19) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 8.0_rt, 17.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_O17_to_F18);
    rate_eval.screened_rates(k_p_O17_to_F18) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_O17_to_F18);
        rate_eval.dscreened_rates_dT(k_p_O17_to_F18) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_O17_to_n_F17);
    rate_eval.screened_rates(k_p_O17_to_n_F17) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_O17_to_n_F17);
        rate_eval.dscreened_rates_dT(k_p_O17_to_n_F17) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_O17_to_He4_N14);
    rate_eval.screened_rates(k_p_O17_to_He4_N14) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_O17_to_He4_N14);
        rate_eval.dscreened_rates_dT(k_p_O17_to_He4_N14) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 8.0_rt, 17.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_O17_to_Ne21);
    rate_eval.screened_rates(k_He4_O17_to_Ne21) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_O17_to_Ne21);
        rate_eval.dscreened_rates_dT(k_He4_O17_to_Ne21) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_O17_to_n_Ne20);
    rate_eval.screened_rates(k_He4_O17_to_n_Ne20) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_O17_to_n_Ne20);
        rate_eval.dscreened_rates_dT(k_He4_O17_to_n_Ne20) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 8.0_rt, 18.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_O18_to_F19);
    rate_eval.screened_rates(k_p_O18_to_F19) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_O18_to_F19);
        rate_eval.dscreened_rates_dT(k_p_O18_to_F19) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_O18_to_n_F18);
    rate_eval.screened_rates(k_p_O18_to_n_F18) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_O18_to_n_F18);
        rate_eval.dscreened_rates_dT(k_p_O18_to_n_F18) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_O18_to_He4_N15);
    rate_eval.screened_rates(k_p_O18_to_He4_N15) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_O18_to_He4_N15);
        rate_eval.dscreened_rates_dT(k_p_O18_to_He4_N15) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 8.0_rt, 18.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_O18_to_Ne22);
    rate_eval.screened_rates(k_He4_O18_to_Ne22) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_O18_to_Ne22);
        rate_eval.dscreened_rates_dT(k_He4_O18_to_Ne22) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_O18_to_n_Ne21);
    rate_eval.screened_rates(k_He4_O18_to_n_Ne21) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_O18_to_n_Ne21);
        rate_eval.dscreened_rates_dT(k_He4_O18_to_n_Ne21) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 9.0_rt, 17.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_F17_to_Ne18);
    rate_eval.screened_rates(k_p_F17_to_Ne18) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_F17_to_Ne18);
        rate_eval.dscreened_rates_dT(k_p_F17_to_Ne18) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_F17_to_He4_O14);
    rate_eval.screened_rates(k_p_F17_to_He4_O14) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_F17_to_He4_O14);
        rate_eval.dscreened_rates_dT(k_p_F17_to_He4_O14) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 9.0_rt, 17.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_F17_to_Na21);
    rate_eval.screened_rates(k_He4_F17_to_Na21) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_F17_to_Na21);
        rate_eval.dscreened_rates_dT(k_He4_F17_to_Na21) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_F17_to_p_Ne20);
    rate_eval.screened_rates(k_He4_F17_to_p_Ne20) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_F17_to_p_Ne20);
        rate_eval.dscreened_rates_dT(k_He4_F17_to_p_Ne20) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 9.0_rt, 18.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_F18_to_Ne19);
    rate_eval.screened_rates(k_p_F18_to_Ne19) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_F18_to_Ne19);
        rate_eval.dscreened_rates_dT(k_p_F18_to_Ne19) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_F18_to_n_Ne18);
    rate_eval.screened_rates(k_p_F18_to_n_Ne18) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_F18_to_n_Ne18);
        rate_eval.dscreened_rates_dT(k_p_F18_to_n_Ne18) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_F18_to_He4_O15);
    rate_eval.screened_rates(k_p_F18_to_He4_O15) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_F18_to_He4_O15);
        rate_eval.dscreened_rates_dT(k_p_F18_to_He4_O15) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 9.0_rt, 18.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_F18_to_Na22);
    rate_eval.screened_rates(k_He4_F18_to_Na22) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_F18_to_Na22);
        rate_eval.dscreened_rates_dT(k_He4_F18_to_Na22) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_F18_to_n_Na21);
    rate_eval.screened_rates(k_He4_F18_to_n_Na21) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_F18_to_n_Na21);
        rate_eval.dscreened_rates_dT(k_He4_F18_to_n_Na21) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_F18_to_p_Ne21);
    rate_eval.screened_rates(k_He4_F18_to_p_Ne21) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_F18_to_p_Ne21);
        rate_eval.dscreened_rates_dT(k_He4_F18_to_p_Ne21) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 9.0_rt, 19.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_F19_to_Ne20);
    rate_eval.screened_rates(k_p_F19_to_Ne20) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_F19_to_Ne20);
        rate_eval.dscreened_rates_dT(k_p_F19_to_Ne20) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_F19_to_n_Ne19);
    rate_eval.screened_rates(k_p_F19_to_n_Ne19) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_F19_to_n_Ne19);
        rate_eval.dscreened_rates_dT(k_p_F19_to_n_Ne19) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_F19_to_He4_O16);
    rate_eval.screened_rates(k_p_F19_to_He4_O16) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_F19_to_He4_O16);
        rate_eval.dscreened_rates_dT(k_p_F19_to_He4_O16) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 9.0_rt, 19.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_F19_to_Na23);
    rate_eval.screened_rates(k_He4_F19_to_Na23) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_F19_to_Na23);
        rate_eval.dscreened_rates_dT(k_He4_F19_to_Na23) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_F19_to_n_Na22);
    rate_eval.screened_rates(k_He4_F19_to_n_Na22) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_F19_to_n_Na22);
        rate_eval.dscreened_rates_dT(k_He4_F19_to_n_Na22) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_F19_to_p_Ne22);
    rate_eval.screened_rates(k_He4_F19_to_p_Ne22) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_F19_to_p_Ne22);
        rate_eval.dscreened_rates_dT(k_He4_F19_to_p_Ne22) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 10.0_rt, 19.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ne19_to_Mg23);
    rate_eval.screened_rates(k_He4_Ne19_to_Mg23) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ne19_to_Mg23);
        rate_eval.dscreened_rates_dT(k_He4_Ne19_to_Mg23) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ne19_to_p_Na22);
    rate_eval.screened_rates(k_He4_Ne19_to_p_Na22) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ne19_to_p_Na22);
        rate_eval.dscreened_rates_dT(k_He4_Ne19_to_p_Na22) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 10.0_rt, 20.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ne20_to_Na21);
    rate_eval.screened_rates(k_p_Ne20_to_Na21) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ne20_to_Na21);
        rate_eval.dscreened_rates_dT(k_p_Ne20_to_Na21) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ne20_to_He4_F17);
    rate_eval.screened_rates(k_p_Ne20_to_He4_F17) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ne20_to_He4_F17);
        rate_eval.dscreened_rates_dT(k_p_Ne20_to_He4_F17) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 10.0_rt, 20.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ne20_to_Mg24);
    rate_eval.screened_rates(k_He4_Ne20_to_Mg24) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ne20_to_Mg24);
        rate_eval.dscreened_rates_dT(k_He4_Ne20_to_Mg24) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ne20_to_n_Mg23);
    rate_eval.screened_rates(k_He4_Ne20_to_n_Mg23) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ne20_to_n_Mg23);
        rate_eval.dscreened_rates_dT(k_He4_Ne20_to_n_Mg23) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ne20_to_p_Na23);
    rate_eval.screened_rates(k_He4_Ne20_to_p_Na23) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ne20_to_p_Na23);
        rate_eval.dscreened_rates_dT(k_He4_Ne20_to_p_Na23) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ne20_to_C12_C12);
    rate_eval.screened_rates(k_He4_Ne20_to_C12_C12) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ne20_to_C12_C12);
        rate_eval.dscreened_rates_dT(k_He4_Ne20_to_C12_C12) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 10.0_rt, 21.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ne21_to_Na22);
    rate_eval.screened_rates(k_p_Ne21_to_Na22) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ne21_to_Na22);
        rate_eval.dscreened_rates_dT(k_p_Ne21_to_Na22) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ne21_to_n_Na21);
    rate_eval.screened_rates(k_p_Ne21_to_n_Na21) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ne21_to_n_Na21);
        rate_eval.dscreened_rates_dT(k_p_Ne21_to_n_Na21) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ne21_to_He4_F18);
    rate_eval.screened_rates(k_p_Ne21_to_He4_F18) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ne21_to_He4_F18);
        rate_eval.dscreened_rates_dT(k_p_Ne21_to_He4_F18) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 10.0_rt, 21.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ne21_to_Mg25);
    rate_eval.screened_rates(k_He4_Ne21_to_Mg25) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ne21_to_Mg25);
        rate_eval.dscreened_rates_dT(k_He4_Ne21_to_Mg25) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ne21_to_n_Mg24);
    rate_eval.screened_rates(k_He4_Ne21_to_n_Mg24) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ne21_to_n_Mg24);
        rate_eval.dscreened_rates_dT(k_He4_Ne21_to_n_Mg24) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 10.0_rt, 22.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ne22_to_Na23);
    rate_eval.screened_rates(k_p_Ne22_to_Na23) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ne22_to_Na23);
        rate_eval.dscreened_rates_dT(k_p_Ne22_to_Na23) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ne22_to_n_Na22);
    rate_eval.screened_rates(k_p_Ne22_to_n_Na22) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ne22_to_n_Na22);
        rate_eval.dscreened_rates_dT(k_p_Ne22_to_n_Na22) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ne22_to_He4_F19);
    rate_eval.screened_rates(k_p_Ne22_to_He4_F19) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ne22_to_He4_F19);
        rate_eval.dscreened_rates_dT(k_p_Ne22_to_He4_F19) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 10.0_rt, 22.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ne22_to_Mg26);
    rate_eval.screened_rates(k_He4_Ne22_to_Mg26) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ne22_to_Mg26);
        rate_eval.dscreened_rates_dT(k_He4_Ne22_to_Mg26) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ne22_to_n_Mg25);
    rate_eval.screened_rates(k_He4_Ne22_to_n_Mg25) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ne22_to_n_Mg25);
        rate_eval.dscreened_rates_dT(k_He4_Ne22_to_n_Mg25) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 11.0_rt, 21.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Na21_to_Al25);
    rate_eval.screened_rates(k_He4_Na21_to_Al25) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Na21_to_Al25);
        rate_eval.dscreened_rates_dT(k_He4_Na21_to_Al25) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Na21_to_p_Mg24);
    rate_eval.screened_rates(k_He4_Na21_to_p_Mg24) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Na21_to_p_Mg24);
        rate_eval.dscreened_rates_dT(k_He4_Na21_to_p_Mg24) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 11.0_rt, 22.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Na22_to_Mg23);
    rate_eval.screened_rates(k_p_Na22_to_Mg23) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Na22_to_Mg23);
        rate_eval.dscreened_rates_dT(k_p_Na22_to_Mg23) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Na22_to_He4_Ne19);
    rate_eval.screened_rates(k_p_Na22_to_He4_Ne19) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Na22_to_He4_Ne19);
        rate_eval.dscreened_rates_dT(k_p_Na22_to_He4_Ne19) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 11.0_rt, 22.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Na22_to_Al26);
    rate_eval.screened_rates(k_He4_Na22_to_Al26) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Na22_to_Al26);
        rate_eval.dscreened_rates_dT(k_He4_Na22_to_Al26) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Na22_to_n_Al25);
    rate_eval.screened_rates(k_He4_Na22_to_n_Al25) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Na22_to_n_Al25);
        rate_eval.dscreened_rates_dT(k_He4_Na22_to_n_Al25) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Na22_to_p_Mg25);
    rate_eval.screened_rates(k_He4_Na22_to_p_Mg25) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Na22_to_p_Mg25);
        rate_eval.dscreened_rates_dT(k_He4_Na22_to_p_Mg25) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 11.0_rt, 23.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Na23_to_Mg24);
    rate_eval.screened_rates(k_p_Na23_to_Mg24) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Na23_to_Mg24);
        rate_eval.dscreened_rates_dT(k_p_Na23_to_Mg24) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Na23_to_n_Mg23);
    rate_eval.screened_rates(k_p_Na23_to_n_Mg23) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Na23_to_n_Mg23);
        rate_eval.dscreened_rates_dT(k_p_Na23_to_n_Mg23) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Na23_to_He4_Ne20);
    rate_eval.screened_rates(k_p_Na23_to_He4_Ne20) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Na23_to_He4_Ne20);
        rate_eval.dscreened_rates_dT(k_p_Na23_to_He4_Ne20) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Na23_to_C12_C12);
    rate_eval.screened_rates(k_p_Na23_to_C12_C12) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Na23_to_C12_C12);
        rate_eval.dscreened_rates_dT(k_p_Na23_to_C12_C12) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 11.0_rt, 23.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Na23_to_Al27);
    rate_eval.screened_rates(k_He4_Na23_to_Al27) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Na23_to_Al27);
        rate_eval.dscreened_rates_dT(k_He4_Na23_to_Al27) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Na23_to_n_Al26);
    rate_eval.screened_rates(k_He4_Na23_to_n_Al26) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Na23_to_n_Al26);
        rate_eval.dscreened_rates_dT(k_He4_Na23_to_n_Al26) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Na23_to_p_Mg26);
    rate_eval.screened_rates(k_He4_Na23_to_p_Mg26) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Na23_to_p_Mg26);
        rate_eval.dscreened_rates_dT(k_He4_Na23_to_p_Mg26) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 12.0_rt, 24.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Mg24_to_Al25);
    rate_eval.screened_rates(k_p_Mg24_to_Al25) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Mg24_to_Al25);
        rate_eval.dscreened_rates_dT(k_p_Mg24_to_Al25) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Mg24_to_He4_Na21);
    rate_eval.screened_rates(k_p_Mg24_to_He4_Na21) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Mg24_to_He4_Na21);
        rate_eval.dscreened_rates_dT(k_p_Mg24_to_He4_Na21) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 12.0_rt, 24.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Mg24_to_Si28);
    rate_eval.screened_rates(k_He4_Mg24_to_Si28) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mg24_to_Si28);
        rate_eval.dscreened_rates_dT(k_He4_Mg24_to_Si28) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Mg24_to_p_Al27);
    rate_eval.screened_rates(k_He4_Mg24_to_p_Al27) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mg24_to_p_Al27);
        rate_eval.dscreened_rates_dT(k_He4_Mg24_to_p_Al27) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Mg24_to_C12_O16);
    rate_eval.screened_rates(k_He4_Mg24_to_C12_O16) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mg24_to_C12_O16);
        rate_eval.dscreened_rates_dT(k_He4_Mg24_to_C12_O16) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 12.0_rt, 25.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Mg25_to_Al26);
    rate_eval.screened_rates(k_p_Mg25_to_Al26) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Mg25_to_Al26);
        rate_eval.dscreened_rates_dT(k_p_Mg25_to_Al26) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Mg25_to_n_Al25);
    rate_eval.screened_rates(k_p_Mg25_to_n_Al25) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Mg25_to_n_Al25);
        rate_eval.dscreened_rates_dT(k_p_Mg25_to_n_Al25) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Mg25_to_He4_Na22);
    rate_eval.screened_rates(k_p_Mg25_to_He4_Na22) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Mg25_to_He4_Na22);
        rate_eval.dscreened_rates_dT(k_p_Mg25_to_He4_Na22) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 12.0_rt, 25.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Mg25_to_Si29);
    rate_eval.screened_rates(k_He4_Mg25_to_Si29) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mg25_to_Si29);
        rate_eval.dscreened_rates_dT(k_He4_Mg25_to_Si29) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Mg25_to_n_Si28);
    rate_eval.screened_rates(k_He4_Mg25_to_n_Si28) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mg25_to_n_Si28);
        rate_eval.dscreened_rates_dT(k_He4_Mg25_to_n_Si28) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 12.0_rt, 26.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Mg26_to_Al27);
    rate_eval.screened_rates(k_p_Mg26_to_Al27) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Mg26_to_Al27);
        rate_eval.dscreened_rates_dT(k_p_Mg26_to_Al27) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Mg26_to_n_Al26);
    rate_eval.screened_rates(k_p_Mg26_to_n_Al26) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Mg26_to_n_Al26);
        rate_eval.dscreened_rates_dT(k_p_Mg26_to_n_Al26) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Mg26_to_He4_Na23);
    rate_eval.screened_rates(k_p_Mg26_to_He4_Na23) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Mg26_to_He4_Na23);
        rate_eval.dscreened_rates_dT(k_p_Mg26_to_He4_Na23) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 12.0_rt, 26.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Mg26_to_Si30);
    rate_eval.screened_rates(k_He4_Mg26_to_Si30) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mg26_to_Si30);
        rate_eval.dscreened_rates_dT(k_He4_Mg26_to_Si30) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Mg26_to_n_Si29);
    rate_eval.screened_rates(k_He4_Mg26_to_n_Si29) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mg26_to_n_Si29);
        rate_eval.dscreened_rates_dT(k_He4_Mg26_to_n_Si29) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 13.0_rt, 25.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Al25_to_P29);
    rate_eval.screened_rates(k_He4_Al25_to_P29) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Al25_to_P29);
        rate_eval.dscreened_rates_dT(k_He4_Al25_to_P29) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Al25_to_p_Si28);
    rate_eval.screened_rates(k_He4_Al25_to_p_Si28) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Al25_to_p_Si28);
        rate_eval.dscreened_rates_dT(k_He4_Al25_to_p_Si28) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 13.0_rt, 26.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Al26_to_P30);
    rate_eval.screened_rates(k_He4_Al26_to_P30) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Al26_to_P30);
        rate_eval.dscreened_rates_dT(k_He4_Al26_to_P30) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Al26_to_n_P29);
    rate_eval.screened_rates(k_He4_Al26_to_n_P29) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Al26_to_n_P29);
        rate_eval.dscreened_rates_dT(k_He4_Al26_to_n_P29) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Al26_to_p_Si29);
    rate_eval.screened_rates(k_He4_Al26_to_p_Si29) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Al26_to_p_Si29);
        rate_eval.dscreened_rates_dT(k_He4_Al26_to_p_Si29) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 13.0_rt, 27.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Al27_to_Si28);
    rate_eval.screened_rates(k_p_Al27_to_Si28) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Al27_to_Si28);
        rate_eval.dscreened_rates_dT(k_p_Al27_to_Si28) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Al27_to_He4_Mg24);
    rate_eval.screened_rates(k_p_Al27_to_He4_Mg24) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Al27_to_He4_Mg24);
        rate_eval.dscreened_rates_dT(k_p_Al27_to_He4_Mg24) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Al27_to_C12_O16);
    rate_eval.screened_rates(k_p_Al27_to_C12_O16) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Al27_to_C12_O16);
        rate_eval.dscreened_rates_dT(k_p_Al27_to_C12_O16) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 13.0_rt, 27.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Al27_to_P31);
    rate_eval.screened_rates(k_He4_Al27_to_P31) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Al27_to_P31);
        rate_eval.dscreened_rates_dT(k_He4_Al27_to_P31) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Al27_to_n_P30);
    rate_eval.screened_rates(k_He4_Al27_to_n_P30) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Al27_to_n_P30);
        rate_eval.dscreened_rates_dT(k_He4_Al27_to_n_P30) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Al27_to_p_Si30);
    rate_eval.screened_rates(k_He4_Al27_to_p_Si30) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Al27_to_p_Si30);
        rate_eval.dscreened_rates_dT(k_He4_Al27_to_p_Si30) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 14.0_rt, 28.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Si28_to_P29);
    rate_eval.screened_rates(k_p_Si28_to_P29) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Si28_to_P29);
        rate_eval.dscreened_rates_dT(k_p_Si28_to_P29) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Si28_to_He4_Al25);
    rate_eval.screened_rates(k_p_Si28_to_He4_Al25) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Si28_to_He4_Al25);
        rate_eval.dscreened_rates_dT(k_p_Si28_to_He4_Al25) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 14.0_rt, 28.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Si28_to_S32);
    rate_eval.screened_rates(k_He4_Si28_to_S32) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Si28_to_S32);
        rate_eval.dscreened_rates_dT(k_He4_Si28_to_S32) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Si28_to_p_P31);
    rate_eval.screened_rates(k_He4_Si28_to_p_P31) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Si28_to_p_P31);
        rate_eval.dscreened_rates_dT(k_He4_Si28_to_p_P31) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Si28_to_C12_Ne20);
    rate_eval.screened_rates(k_He4_Si28_to_C12_Ne20) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Si28_to_C12_Ne20);
        rate_eval.dscreened_rates_dT(k_He4_Si28_to_C12_Ne20) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Si28_to_O16_O16);
    rate_eval.screened_rates(k_He4_Si28_to_O16_O16) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Si28_to_O16_O16);
        rate_eval.dscreened_rates_dT(k_He4_Si28_to_O16_O16) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 14.0_rt, 29.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Si29_to_P30);
    rate_eval.screened_rates(k_p_Si29_to_P30) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Si29_to_P30);
        rate_eval.dscreened_rates_dT(k_p_Si29_to_P30) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Si29_to_n_P29);
    rate_eval.screened_rates(k_p_Si29_to_n_P29) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Si29_to_n_P29);
        rate_eval.dscreened_rates_dT(k_p_Si29_to_n_P29) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Si29_to_He4_Al26);
    rate_eval.screened_rates(k_p_Si29_to_He4_Al26) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Si29_to_He4_Al26);
        rate_eval.dscreened_rates_dT(k_p_Si29_to_He4_Al26) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 14.0_rt, 29.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Si29_to_S33);
    rate_eval.screened_rates(k_He4_Si29_to_S33) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Si29_to_S33);
        rate_eval.dscreened_rates_dT(k_He4_Si29_to_S33) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Si29_to_n_S32);
    rate_eval.screened_rates(k_He4_Si29_to_n_S32) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Si29_to_n_S32);
        rate_eval.dscreened_rates_dT(k_He4_Si29_to_n_S32) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Si29_to_p_P32);
    rate_eval.screened_rates(k_He4_Si29_to_p_P32) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Si29_to_p_P32);
        rate_eval.dscreened_rates_dT(k_He4_Si29_to_p_P32) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 14.0_rt, 30.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Si30_to_P31);
    rate_eval.screened_rates(k_p_Si30_to_P31) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Si30_to_P31);
        rate_eval.dscreened_rates_dT(k_p_Si30_to_P31) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Si30_to_n_P30);
    rate_eval.screened_rates(k_p_Si30_to_n_P30) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Si30_to_n_P30);
        rate_eval.dscreened_rates_dT(k_p_Si30_to_n_P30) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Si30_to_He4_Al27);
    rate_eval.screened_rates(k_p_Si30_to_He4_Al27) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Si30_to_He4_Al27);
        rate_eval.dscreened_rates_dT(k_p_Si30_to_He4_Al27) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 14.0_rt, 30.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Si30_to_S34);
    rate_eval.screened_rates(k_He4_Si30_to_S34) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Si30_to_S34);
        rate_eval.dscreened_rates_dT(k_He4_Si30_to_S34) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Si30_to_n_S33);
    rate_eval.screened_rates(k_He4_Si30_to_n_S33) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Si30_to_n_S33);
        rate_eval.dscreened_rates_dT(k_He4_Si30_to_n_S33) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Si30_to_p_P33);
    rate_eval.screened_rates(k_He4_Si30_to_p_P33) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Si30_to_p_P33);
        rate_eval.dscreened_rates_dT(k_He4_Si30_to_p_P33) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 14.0_rt, 31.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Si31_to_P32);
    rate_eval.screened_rates(k_p_Si31_to_P32) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Si31_to_P32);
        rate_eval.dscreened_rates_dT(k_p_Si31_to_P32) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Si31_to_n_P31);
    rate_eval.screened_rates(k_p_Si31_to_n_P31) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Si31_to_n_P31);
        rate_eval.dscreened_rates_dT(k_p_Si31_to_n_P31) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 14.0_rt, 31.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Si31_to_S35);
    rate_eval.screened_rates(k_He4_Si31_to_S35) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Si31_to_S35);
        rate_eval.dscreened_rates_dT(k_He4_Si31_to_S35) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Si31_to_n_S34);
    rate_eval.screened_rates(k_He4_Si31_to_n_S34) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Si31_to_n_S34);
        rate_eval.dscreened_rates_dT(k_He4_Si31_to_n_S34) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 14.0_rt, 32.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Si32_to_P33);
    rate_eval.screened_rates(k_p_Si32_to_P33) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Si32_to_P33);
        rate_eval.dscreened_rates_dT(k_p_Si32_to_P33) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Si32_to_n_P32);
    rate_eval.screened_rates(k_p_Si32_to_n_P32) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Si32_to_n_P32);
        rate_eval.dscreened_rates_dT(k_p_Si32_to_n_P32) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 14.0_rt, 32.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Si32_to_S36);
    rate_eval.screened_rates(k_He4_Si32_to_S36) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Si32_to_S36);
        rate_eval.dscreened_rates_dT(k_He4_Si32_to_S36) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Si32_to_n_S35);
    rate_eval.screened_rates(k_He4_Si32_to_n_S35) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Si32_to_n_S35);
        rate_eval.dscreened_rates_dT(k_He4_Si32_to_n_S35) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 15.0_rt, 29.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_P29_to_Cl33);
    rate_eval.screened_rates(k_He4_P29_to_Cl33) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_P29_to_Cl33);
        rate_eval.dscreened_rates_dT(k_He4_P29_to_Cl33) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_P29_to_p_S32);
    rate_eval.screened_rates(k_He4_P29_to_p_S32) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_P29_to_p_S32);
        rate_eval.dscreened_rates_dT(k_He4_P29_to_p_S32) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 15.0_rt, 30.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_P30_to_Cl34);
    rate_eval.screened_rates(k_He4_P30_to_Cl34) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_P30_to_Cl34);
        rate_eval.dscreened_rates_dT(k_He4_P30_to_Cl34) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_P30_to_n_Cl33);
    rate_eval.screened_rates(k_He4_P30_to_n_Cl33) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_P30_to_n_Cl33);
        rate_eval.dscreened_rates_dT(k_He4_P30_to_n_Cl33) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_P30_to_p_S33);
    rate_eval.screened_rates(k_He4_P30_to_p_S33) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_P30_to_p_S33);
        rate_eval.dscreened_rates_dT(k_He4_P30_to_p_S33) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 15.0_rt, 31.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_P31_to_S32);
    rate_eval.screened_rates(k_p_P31_to_S32) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_P31_to_S32);
        rate_eval.dscreened_rates_dT(k_p_P31_to_S32) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_P31_to_He4_Si28);
    rate_eval.screened_rates(k_p_P31_to_He4_Si28) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_P31_to_He4_Si28);
        rate_eval.dscreened_rates_dT(k_p_P31_to_He4_Si28) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_P31_to_C12_Ne20);
    rate_eval.screened_rates(k_p_P31_to_C12_Ne20) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_P31_to_C12_Ne20);
        rate_eval.dscreened_rates_dT(k_p_P31_to_C12_Ne20) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_P31_to_O16_O16);
    rate_eval.screened_rates(k_p_P31_to_O16_O16) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_P31_to_O16_O16);
        rate_eval.dscreened_rates_dT(k_p_P31_to_O16_O16) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 15.0_rt, 31.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_P31_to_Cl35);
    rate_eval.screened_rates(k_He4_P31_to_Cl35) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_P31_to_Cl35);
        rate_eval.dscreened_rates_dT(k_He4_P31_to_Cl35) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_P31_to_n_Cl34);
    rate_eval.screened_rates(k_He4_P31_to_n_Cl34) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_P31_to_n_Cl34);
        rate_eval.dscreened_rates_dT(k_He4_P31_to_n_Cl34) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_P31_to_p_S34);
    rate_eval.screened_rates(k_He4_P31_to_p_S34) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_P31_to_p_S34);
        rate_eval.dscreened_rates_dT(k_He4_P31_to_p_S34) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 15.0_rt, 32.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_P32_to_S33);
    rate_eval.screened_rates(k_p_P32_to_S33) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_P32_to_S33);
        rate_eval.dscreened_rates_dT(k_p_P32_to_S33) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_P32_to_n_S32);
    rate_eval.screened_rates(k_p_P32_to_n_S32) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_P32_to_n_S32);
        rate_eval.dscreened_rates_dT(k_p_P32_to_n_S32) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_P32_to_He4_Si29);
    rate_eval.screened_rates(k_p_P32_to_He4_Si29) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_P32_to_He4_Si29);
        rate_eval.dscreened_rates_dT(k_p_P32_to_He4_Si29) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 15.0_rt, 32.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_P32_to_Cl36);
    rate_eval.screened_rates(k_He4_P32_to_Cl36) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_P32_to_Cl36);
        rate_eval.dscreened_rates_dT(k_He4_P32_to_Cl36) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_P32_to_n_Cl35);
    rate_eval.screened_rates(k_He4_P32_to_n_Cl35) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_P32_to_n_Cl35);
        rate_eval.dscreened_rates_dT(k_He4_P32_to_n_Cl35) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_P32_to_p_S35);
    rate_eval.screened_rates(k_He4_P32_to_p_S35) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_P32_to_p_S35);
        rate_eval.dscreened_rates_dT(k_He4_P32_to_p_S35) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 15.0_rt, 33.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_P33_to_S34);
    rate_eval.screened_rates(k_p_P33_to_S34) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_P33_to_S34);
        rate_eval.dscreened_rates_dT(k_p_P33_to_S34) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_P33_to_n_S33);
    rate_eval.screened_rates(k_p_P33_to_n_S33) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_P33_to_n_S33);
        rate_eval.dscreened_rates_dT(k_p_P33_to_n_S33) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_P33_to_He4_Si30);
    rate_eval.screened_rates(k_p_P33_to_He4_Si30) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_P33_to_He4_Si30);
        rate_eval.dscreened_rates_dT(k_p_P33_to_He4_Si30) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 15.0_rt, 33.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_P33_to_Cl37);
    rate_eval.screened_rates(k_He4_P33_to_Cl37) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_P33_to_Cl37);
        rate_eval.dscreened_rates_dT(k_He4_P33_to_Cl37) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_P33_to_n_Cl36);
    rate_eval.screened_rates(k_He4_P33_to_n_Cl36) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_P33_to_n_Cl36);
        rate_eval.dscreened_rates_dT(k_He4_P33_to_n_Cl36) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_P33_to_p_S36);
    rate_eval.screened_rates(k_He4_P33_to_p_S36) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_P33_to_p_S36);
        rate_eval.dscreened_rates_dT(k_He4_P33_to_p_S36) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 16.0_rt, 32.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_S32_to_Cl33);
    rate_eval.screened_rates(k_p_S32_to_Cl33) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_S32_to_Cl33);
        rate_eval.dscreened_rates_dT(k_p_S32_to_Cl33) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_S32_to_He4_P29);
    rate_eval.screened_rates(k_p_S32_to_He4_P29) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_S32_to_He4_P29);
        rate_eval.dscreened_rates_dT(k_p_S32_to_He4_P29) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 16.0_rt, 32.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_S32_to_Ar36);
    rate_eval.screened_rates(k_He4_S32_to_Ar36) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_S32_to_Ar36);
        rate_eval.dscreened_rates_dT(k_He4_S32_to_Ar36) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_S32_to_p_Cl35);
    rate_eval.screened_rates(k_He4_S32_to_p_Cl35) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_S32_to_p_Cl35);
        rate_eval.dscreened_rates_dT(k_He4_S32_to_p_Cl35) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 16.0_rt, 33.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_S33_to_Cl34);
    rate_eval.screened_rates(k_p_S33_to_Cl34) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_S33_to_Cl34);
        rate_eval.dscreened_rates_dT(k_p_S33_to_Cl34) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_S33_to_n_Cl33);
    rate_eval.screened_rates(k_p_S33_to_n_Cl33) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_S33_to_n_Cl33);
        rate_eval.dscreened_rates_dT(k_p_S33_to_n_Cl33) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_S33_to_He4_P30);
    rate_eval.screened_rates(k_p_S33_to_He4_P30) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_S33_to_He4_P30);
        rate_eval.dscreened_rates_dT(k_p_S33_to_He4_P30) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 16.0_rt, 33.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_S33_to_Ar37);
    rate_eval.screened_rates(k_He4_S33_to_Ar37) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_S33_to_Ar37);
        rate_eval.dscreened_rates_dT(k_He4_S33_to_Ar37) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_S33_to_n_Ar36);
    rate_eval.screened_rates(k_He4_S33_to_n_Ar36) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_S33_to_n_Ar36);
        rate_eval.dscreened_rates_dT(k_He4_S33_to_n_Ar36) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_S33_to_p_Cl36);
    rate_eval.screened_rates(k_He4_S33_to_p_Cl36) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_S33_to_p_Cl36);
        rate_eval.dscreened_rates_dT(k_He4_S33_to_p_Cl36) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 16.0_rt, 34.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_S34_to_Cl35);
    rate_eval.screened_rates(k_p_S34_to_Cl35) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_S34_to_Cl35);
        rate_eval.dscreened_rates_dT(k_p_S34_to_Cl35) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_S34_to_n_Cl34);
    rate_eval.screened_rates(k_p_S34_to_n_Cl34) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_S34_to_n_Cl34);
        rate_eval.dscreened_rates_dT(k_p_S34_to_n_Cl34) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_S34_to_He4_P31);
    rate_eval.screened_rates(k_p_S34_to_He4_P31) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_S34_to_He4_P31);
        rate_eval.dscreened_rates_dT(k_p_S34_to_He4_P31) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 16.0_rt, 34.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_S34_to_Ar38);
    rate_eval.screened_rates(k_He4_S34_to_Ar38) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_S34_to_Ar38);
        rate_eval.dscreened_rates_dT(k_He4_S34_to_Ar38) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_S34_to_n_Ar37);
    rate_eval.screened_rates(k_He4_S34_to_n_Ar37) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_S34_to_n_Ar37);
        rate_eval.dscreened_rates_dT(k_He4_S34_to_n_Ar37) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_S34_to_p_Cl37);
    rate_eval.screened_rates(k_He4_S34_to_p_Cl37) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_S34_to_p_Cl37);
        rate_eval.dscreened_rates_dT(k_He4_S34_to_p_Cl37) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 16.0_rt, 35.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_S35_to_Cl36);
    rate_eval.screened_rates(k_p_S35_to_Cl36) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_S35_to_Cl36);
        rate_eval.dscreened_rates_dT(k_p_S35_to_Cl36) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_S35_to_n_Cl35);
    rate_eval.screened_rates(k_p_S35_to_n_Cl35) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_S35_to_n_Cl35);
        rate_eval.dscreened_rates_dT(k_p_S35_to_n_Cl35) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_S35_to_He4_P32);
    rate_eval.screened_rates(k_p_S35_to_He4_P32) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_S35_to_He4_P32);
        rate_eval.dscreened_rates_dT(k_p_S35_to_He4_P32) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 16.0_rt, 35.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_S35_to_Ar39);
    rate_eval.screened_rates(k_He4_S35_to_Ar39) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_S35_to_Ar39);
        rate_eval.dscreened_rates_dT(k_He4_S35_to_Ar39) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_S35_to_n_Ar38);
    rate_eval.screened_rates(k_He4_S35_to_n_Ar38) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_S35_to_n_Ar38);
        rate_eval.dscreened_rates_dT(k_He4_S35_to_n_Ar38) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 16.0_rt, 36.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_S36_to_Cl37);
    rate_eval.screened_rates(k_p_S36_to_Cl37) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_S36_to_Cl37);
        rate_eval.dscreened_rates_dT(k_p_S36_to_Cl37) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_S36_to_n_Cl36);
    rate_eval.screened_rates(k_p_S36_to_n_Cl36) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_S36_to_n_Cl36);
        rate_eval.dscreened_rates_dT(k_p_S36_to_n_Cl36) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_S36_to_He4_P33);
    rate_eval.screened_rates(k_p_S36_to_He4_P33) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_S36_to_He4_P33);
        rate_eval.dscreened_rates_dT(k_p_S36_to_He4_P33) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 16.0_rt, 36.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_S36_to_Ar40);
    rate_eval.screened_rates(k_He4_S36_to_Ar40) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_S36_to_Ar40);
        rate_eval.dscreened_rates_dT(k_He4_S36_to_Ar40) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_S36_to_n_Ar39);
    rate_eval.screened_rates(k_He4_S36_to_n_Ar39) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_S36_to_n_Ar39);
        rate_eval.dscreened_rates_dT(k_He4_S36_to_n_Ar39) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 17.0_rt, 33.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Cl33_to_K37);
    rate_eval.screened_rates(k_He4_Cl33_to_K37) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cl33_to_K37);
        rate_eval.dscreened_rates_dT(k_He4_Cl33_to_K37) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cl33_to_p_Ar36);
    rate_eval.screened_rates(k_He4_Cl33_to_p_Ar36) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cl33_to_p_Ar36);
        rate_eval.dscreened_rates_dT(k_He4_Cl33_to_p_Ar36) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 17.0_rt, 34.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Cl34_to_K38);
    rate_eval.screened_rates(k_He4_Cl34_to_K38) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cl34_to_K38);
        rate_eval.dscreened_rates_dT(k_He4_Cl34_to_K38) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cl34_to_n_K37);
    rate_eval.screened_rates(k_He4_Cl34_to_n_K37) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cl34_to_n_K37);
        rate_eval.dscreened_rates_dT(k_He4_Cl34_to_n_K37) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cl34_to_p_Ar37);
    rate_eval.screened_rates(k_He4_Cl34_to_p_Ar37) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cl34_to_p_Ar37);
        rate_eval.dscreened_rates_dT(k_He4_Cl34_to_p_Ar37) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 17.0_rt, 35.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Cl35_to_Ar36);
    rate_eval.screened_rates(k_p_Cl35_to_Ar36) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cl35_to_Ar36);
        rate_eval.dscreened_rates_dT(k_p_Cl35_to_Ar36) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cl35_to_He4_S32);
    rate_eval.screened_rates(k_p_Cl35_to_He4_S32) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cl35_to_He4_S32);
        rate_eval.dscreened_rates_dT(k_p_Cl35_to_He4_S32) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 17.0_rt, 35.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Cl35_to_K39);
    rate_eval.screened_rates(k_He4_Cl35_to_K39) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cl35_to_K39);
        rate_eval.dscreened_rates_dT(k_He4_Cl35_to_K39) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cl35_to_n_K38);
    rate_eval.screened_rates(k_He4_Cl35_to_n_K38) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cl35_to_n_K38);
        rate_eval.dscreened_rates_dT(k_He4_Cl35_to_n_K38) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cl35_to_p_Ar38);
    rate_eval.screened_rates(k_He4_Cl35_to_p_Ar38) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cl35_to_p_Ar38);
        rate_eval.dscreened_rates_dT(k_He4_Cl35_to_p_Ar38) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 17.0_rt, 36.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Cl36_to_Ar37);
    rate_eval.screened_rates(k_p_Cl36_to_Ar37) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cl36_to_Ar37);
        rate_eval.dscreened_rates_dT(k_p_Cl36_to_Ar37) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cl36_to_n_Ar36);
    rate_eval.screened_rates(k_p_Cl36_to_n_Ar36) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cl36_to_n_Ar36);
        rate_eval.dscreened_rates_dT(k_p_Cl36_to_n_Ar36) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cl36_to_He4_S33);
    rate_eval.screened_rates(k_p_Cl36_to_He4_S33) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cl36_to_He4_S33);
        rate_eval.dscreened_rates_dT(k_p_Cl36_to_He4_S33) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 17.0_rt, 36.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Cl36_to_K40);
    rate_eval.screened_rates(k_He4_Cl36_to_K40) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cl36_to_K40);
        rate_eval.dscreened_rates_dT(k_He4_Cl36_to_K40) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cl36_to_n_K39);
    rate_eval.screened_rates(k_He4_Cl36_to_n_K39) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cl36_to_n_K39);
        rate_eval.dscreened_rates_dT(k_He4_Cl36_to_n_K39) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cl36_to_p_Ar39);
    rate_eval.screened_rates(k_He4_Cl36_to_p_Ar39) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cl36_to_p_Ar39);
        rate_eval.dscreened_rates_dT(k_He4_Cl36_to_p_Ar39) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 17.0_rt, 37.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Cl37_to_Ar38);
    rate_eval.screened_rates(k_p_Cl37_to_Ar38) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cl37_to_Ar38);
        rate_eval.dscreened_rates_dT(k_p_Cl37_to_Ar38) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cl37_to_n_Ar37);
    rate_eval.screened_rates(k_p_Cl37_to_n_Ar37) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cl37_to_n_Ar37);
        rate_eval.dscreened_rates_dT(k_p_Cl37_to_n_Ar37) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cl37_to_He4_S34);
    rate_eval.screened_rates(k_p_Cl37_to_He4_S34) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cl37_to_He4_S34);
        rate_eval.dscreened_rates_dT(k_p_Cl37_to_He4_S34) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 17.0_rt, 37.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Cl37_to_K41);
    rate_eval.screened_rates(k_He4_Cl37_to_K41) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cl37_to_K41);
        rate_eval.dscreened_rates_dT(k_He4_Cl37_to_K41) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cl37_to_n_K40);
    rate_eval.screened_rates(k_He4_Cl37_to_n_K40) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cl37_to_n_K40);
        rate_eval.dscreened_rates_dT(k_He4_Cl37_to_n_K40) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cl37_to_p_Ar40);
    rate_eval.screened_rates(k_He4_Cl37_to_p_Ar40) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cl37_to_p_Ar40);
        rate_eval.dscreened_rates_dT(k_He4_Cl37_to_p_Ar40) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 18.0_rt, 36.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ar36_to_K37);
    rate_eval.screened_rates(k_p_Ar36_to_K37) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ar36_to_K37);
        rate_eval.dscreened_rates_dT(k_p_Ar36_to_K37) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ar36_to_He4_Cl33);
    rate_eval.screened_rates(k_p_Ar36_to_He4_Cl33) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ar36_to_He4_Cl33);
        rate_eval.dscreened_rates_dT(k_p_Ar36_to_He4_Cl33) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 18.0_rt, 36.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ar36_to_Ca40);
    rate_eval.screened_rates(k_He4_Ar36_to_Ca40) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ar36_to_Ca40);
        rate_eval.dscreened_rates_dT(k_He4_Ar36_to_Ca40) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ar36_to_p_K39);
    rate_eval.screened_rates(k_He4_Ar36_to_p_K39) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ar36_to_p_K39);
        rate_eval.dscreened_rates_dT(k_He4_Ar36_to_p_K39) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 18.0_rt, 37.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ar37_to_K38);
    rate_eval.screened_rates(k_p_Ar37_to_K38) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ar37_to_K38);
        rate_eval.dscreened_rates_dT(k_p_Ar37_to_K38) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ar37_to_n_K37);
    rate_eval.screened_rates(k_p_Ar37_to_n_K37) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ar37_to_n_K37);
        rate_eval.dscreened_rates_dT(k_p_Ar37_to_n_K37) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ar37_to_He4_Cl34);
    rate_eval.screened_rates(k_p_Ar37_to_He4_Cl34) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ar37_to_He4_Cl34);
        rate_eval.dscreened_rates_dT(k_p_Ar37_to_He4_Cl34) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 18.0_rt, 37.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ar37_to_Ca41);
    rate_eval.screened_rates(k_He4_Ar37_to_Ca41) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ar37_to_Ca41);
        rate_eval.dscreened_rates_dT(k_He4_Ar37_to_Ca41) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ar37_to_n_Ca40);
    rate_eval.screened_rates(k_He4_Ar37_to_n_Ca40) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ar37_to_n_Ca40);
        rate_eval.dscreened_rates_dT(k_He4_Ar37_to_n_Ca40) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ar37_to_p_K40);
    rate_eval.screened_rates(k_He4_Ar37_to_p_K40) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ar37_to_p_K40);
        rate_eval.dscreened_rates_dT(k_He4_Ar37_to_p_K40) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 18.0_rt, 38.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ar38_to_K39);
    rate_eval.screened_rates(k_p_Ar38_to_K39) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ar38_to_K39);
        rate_eval.dscreened_rates_dT(k_p_Ar38_to_K39) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ar38_to_n_K38);
    rate_eval.screened_rates(k_p_Ar38_to_n_K38) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ar38_to_n_K38);
        rate_eval.dscreened_rates_dT(k_p_Ar38_to_n_K38) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ar38_to_He4_Cl35);
    rate_eval.screened_rates(k_p_Ar38_to_He4_Cl35) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ar38_to_He4_Cl35);
        rate_eval.dscreened_rates_dT(k_p_Ar38_to_He4_Cl35) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 18.0_rt, 38.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ar38_to_Ca42);
    rate_eval.screened_rates(k_He4_Ar38_to_Ca42) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ar38_to_Ca42);
        rate_eval.dscreened_rates_dT(k_He4_Ar38_to_Ca42) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ar38_to_n_Ca41);
    rate_eval.screened_rates(k_He4_Ar38_to_n_Ca41) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ar38_to_n_Ca41);
        rate_eval.dscreened_rates_dT(k_He4_Ar38_to_n_Ca41) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ar38_to_p_K41);
    rate_eval.screened_rates(k_He4_Ar38_to_p_K41) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ar38_to_p_K41);
        rate_eval.dscreened_rates_dT(k_He4_Ar38_to_p_K41) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 18.0_rt, 39.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ar39_to_K40);
    rate_eval.screened_rates(k_p_Ar39_to_K40) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ar39_to_K40);
        rate_eval.dscreened_rates_dT(k_p_Ar39_to_K40) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ar39_to_n_K39);
    rate_eval.screened_rates(k_p_Ar39_to_n_K39) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ar39_to_n_K39);
        rate_eval.dscreened_rates_dT(k_p_Ar39_to_n_K39) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ar39_to_He4_Cl36);
    rate_eval.screened_rates(k_p_Ar39_to_He4_Cl36) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ar39_to_He4_Cl36);
        rate_eval.dscreened_rates_dT(k_p_Ar39_to_He4_Cl36) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 18.0_rt, 39.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ar39_to_Ca43);
    rate_eval.screened_rates(k_He4_Ar39_to_Ca43) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ar39_to_Ca43);
        rate_eval.dscreened_rates_dT(k_He4_Ar39_to_Ca43) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ar39_to_n_Ca42);
    rate_eval.screened_rates(k_He4_Ar39_to_n_Ca42) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ar39_to_n_Ca42);
        rate_eval.dscreened_rates_dT(k_He4_Ar39_to_n_Ca42) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 18.0_rt, 40.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ar40_to_K41);
    rate_eval.screened_rates(k_p_Ar40_to_K41) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ar40_to_K41);
        rate_eval.dscreened_rates_dT(k_p_Ar40_to_K41) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ar40_to_n_K40);
    rate_eval.screened_rates(k_p_Ar40_to_n_K40) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ar40_to_n_K40);
        rate_eval.dscreened_rates_dT(k_p_Ar40_to_n_K40) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ar40_to_He4_Cl37);
    rate_eval.screened_rates(k_p_Ar40_to_He4_Cl37) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ar40_to_He4_Cl37);
        rate_eval.dscreened_rates_dT(k_p_Ar40_to_He4_Cl37) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 18.0_rt, 40.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ar40_to_Ca44);
    rate_eval.screened_rates(k_He4_Ar40_to_Ca44) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ar40_to_Ca44);
        rate_eval.dscreened_rates_dT(k_He4_Ar40_to_Ca44) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ar40_to_n_Ca43);
    rate_eval.screened_rates(k_He4_Ar40_to_n_Ca43) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ar40_to_n_Ca43);
        rate_eval.dscreened_rates_dT(k_He4_Ar40_to_n_Ca43) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 19.0_rt, 39.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_K39_to_Ca40);
    rate_eval.screened_rates(k_p_K39_to_Ca40) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_K39_to_Ca40);
        rate_eval.dscreened_rates_dT(k_p_K39_to_Ca40) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_K39_to_He4_Ar36);
    rate_eval.screened_rates(k_p_K39_to_He4_Ar36) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_K39_to_He4_Ar36);
        rate_eval.dscreened_rates_dT(k_p_K39_to_He4_Ar36) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 19.0_rt, 39.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_K39_to_Sc43);
    rate_eval.screened_rates(k_He4_K39_to_Sc43) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_K39_to_Sc43);
        rate_eval.dscreened_rates_dT(k_He4_K39_to_Sc43) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_K39_to_p_Ca42);
    rate_eval.screened_rates(k_He4_K39_to_p_Ca42) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_K39_to_p_Ca42);
        rate_eval.dscreened_rates_dT(k_He4_K39_to_p_Ca42) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 19.0_rt, 40.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_K40_to_Ca41);
    rate_eval.screened_rates(k_p_K40_to_Ca41) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_K40_to_Ca41);
        rate_eval.dscreened_rates_dT(k_p_K40_to_Ca41) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_K40_to_n_Ca40);
    rate_eval.screened_rates(k_p_K40_to_n_Ca40) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_K40_to_n_Ca40);
        rate_eval.dscreened_rates_dT(k_p_K40_to_n_Ca40) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_K40_to_He4_Ar37);
    rate_eval.screened_rates(k_p_K40_to_He4_Ar37) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_K40_to_He4_Ar37);
        rate_eval.dscreened_rates_dT(k_p_K40_to_He4_Ar37) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 19.0_rt, 40.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_K40_to_Sc44);
    rate_eval.screened_rates(k_He4_K40_to_Sc44) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_K40_to_Sc44);
        rate_eval.dscreened_rates_dT(k_He4_K40_to_Sc44) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_K40_to_n_Sc43);
    rate_eval.screened_rates(k_He4_K40_to_n_Sc43) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_K40_to_n_Sc43);
        rate_eval.dscreened_rates_dT(k_He4_K40_to_n_Sc43) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_K40_to_p_Ca43);
    rate_eval.screened_rates(k_He4_K40_to_p_Ca43) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_K40_to_p_Ca43);
        rate_eval.dscreened_rates_dT(k_He4_K40_to_p_Ca43) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 19.0_rt, 41.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_K41_to_Ca42);
    rate_eval.screened_rates(k_p_K41_to_Ca42) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_K41_to_Ca42);
        rate_eval.dscreened_rates_dT(k_p_K41_to_Ca42) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_K41_to_n_Ca41);
    rate_eval.screened_rates(k_p_K41_to_n_Ca41) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_K41_to_n_Ca41);
        rate_eval.dscreened_rates_dT(k_p_K41_to_n_Ca41) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_K41_to_He4_Ar38);
    rate_eval.screened_rates(k_p_K41_to_He4_Ar38) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_K41_to_He4_Ar38);
        rate_eval.dscreened_rates_dT(k_p_K41_to_He4_Ar38) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 19.0_rt, 41.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_K41_to_Sc45);
    rate_eval.screened_rates(k_He4_K41_to_Sc45) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_K41_to_Sc45);
        rate_eval.dscreened_rates_dT(k_He4_K41_to_Sc45) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_K41_to_n_Sc44);
    rate_eval.screened_rates(k_He4_K41_to_n_Sc44) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_K41_to_n_Sc44);
        rate_eval.dscreened_rates_dT(k_He4_K41_to_n_Sc44) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_K41_to_p_Ca44);
    rate_eval.screened_rates(k_He4_K41_to_p_Ca44) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_K41_to_p_Ca44);
        rate_eval.dscreened_rates_dT(k_He4_K41_to_p_Ca44) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 20.0_rt, 40.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ca40_to_Ti44);
    rate_eval.screened_rates(k_He4_Ca40_to_Ti44) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ca40_to_Ti44);
        rate_eval.dscreened_rates_dT(k_He4_Ca40_to_Ti44) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ca40_to_p_Sc43);
    rate_eval.screened_rates(k_He4_Ca40_to_p_Sc43) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ca40_to_p_Sc43);
        rate_eval.dscreened_rates_dT(k_He4_Ca40_to_p_Sc43) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 20.0_rt, 41.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ca41_to_Ti45);
    rate_eval.screened_rates(k_He4_Ca41_to_Ti45) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ca41_to_Ti45);
        rate_eval.dscreened_rates_dT(k_He4_Ca41_to_Ti45) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ca41_to_n_Ti44);
    rate_eval.screened_rates(k_He4_Ca41_to_n_Ti44) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ca41_to_n_Ti44);
        rate_eval.dscreened_rates_dT(k_He4_Ca41_to_n_Ti44) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ca41_to_p_Sc44);
    rate_eval.screened_rates(k_He4_Ca41_to_p_Sc44) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ca41_to_p_Sc44);
        rate_eval.dscreened_rates_dT(k_He4_Ca41_to_p_Sc44) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 20.0_rt, 42.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ca42_to_Sc43);
    rate_eval.screened_rates(k_p_Ca42_to_Sc43) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ca42_to_Sc43);
        rate_eval.dscreened_rates_dT(k_p_Ca42_to_Sc43) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ca42_to_He4_K39);
    rate_eval.screened_rates(k_p_Ca42_to_He4_K39) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ca42_to_He4_K39);
        rate_eval.dscreened_rates_dT(k_p_Ca42_to_He4_K39) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 20.0_rt, 42.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ca42_to_Ti46);
    rate_eval.screened_rates(k_He4_Ca42_to_Ti46) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ca42_to_Ti46);
        rate_eval.dscreened_rates_dT(k_He4_Ca42_to_Ti46) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ca42_to_n_Ti45);
    rate_eval.screened_rates(k_He4_Ca42_to_n_Ti45) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ca42_to_n_Ti45);
        rate_eval.dscreened_rates_dT(k_He4_Ca42_to_n_Ti45) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ca42_to_p_Sc45);
    rate_eval.screened_rates(k_He4_Ca42_to_p_Sc45) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ca42_to_p_Sc45);
        rate_eval.dscreened_rates_dT(k_He4_Ca42_to_p_Sc45) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 20.0_rt, 43.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ca43_to_Sc44);
    rate_eval.screened_rates(k_p_Ca43_to_Sc44) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ca43_to_Sc44);
        rate_eval.dscreened_rates_dT(k_p_Ca43_to_Sc44) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ca43_to_n_Sc43);
    rate_eval.screened_rates(k_p_Ca43_to_n_Sc43) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ca43_to_n_Sc43);
        rate_eval.dscreened_rates_dT(k_p_Ca43_to_n_Sc43) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ca43_to_He4_K40);
    rate_eval.screened_rates(k_p_Ca43_to_He4_K40) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ca43_to_He4_K40);
        rate_eval.dscreened_rates_dT(k_p_Ca43_to_He4_K40) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 20.0_rt, 43.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ca43_to_Ti47);
    rate_eval.screened_rates(k_He4_Ca43_to_Ti47) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ca43_to_Ti47);
        rate_eval.dscreened_rates_dT(k_He4_Ca43_to_Ti47) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ca43_to_n_Ti46);
    rate_eval.screened_rates(k_He4_Ca43_to_n_Ti46) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ca43_to_n_Ti46);
        rate_eval.dscreened_rates_dT(k_He4_Ca43_to_n_Ti46) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ca43_to_p_Sc46);
    rate_eval.screened_rates(k_He4_Ca43_to_p_Sc46) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ca43_to_p_Sc46);
        rate_eval.dscreened_rates_dT(k_He4_Ca43_to_p_Sc46) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 20.0_rt, 44.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ca44_to_Sc45);
    rate_eval.screened_rates(k_p_Ca44_to_Sc45) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ca44_to_Sc45);
        rate_eval.dscreened_rates_dT(k_p_Ca44_to_Sc45) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ca44_to_n_Sc44);
    rate_eval.screened_rates(k_p_Ca44_to_n_Sc44) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ca44_to_n_Sc44);
        rate_eval.dscreened_rates_dT(k_p_Ca44_to_n_Sc44) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ca44_to_He4_K41);
    rate_eval.screened_rates(k_p_Ca44_to_He4_K41) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ca44_to_He4_K41);
        rate_eval.dscreened_rates_dT(k_p_Ca44_to_He4_K41) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 20.0_rt, 44.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ca44_to_Ti48);
    rate_eval.screened_rates(k_He4_Ca44_to_Ti48) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ca44_to_Ti48);
        rate_eval.dscreened_rates_dT(k_He4_Ca44_to_Ti48) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ca44_to_n_Ti47);
    rate_eval.screened_rates(k_He4_Ca44_to_n_Ti47) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ca44_to_n_Ti47);
        rate_eval.dscreened_rates_dT(k_He4_Ca44_to_n_Ti47) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ca44_to_p_Sc47);
    rate_eval.screened_rates(k_He4_Ca44_to_p_Sc47) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ca44_to_p_Sc47);
        rate_eval.dscreened_rates_dT(k_He4_Ca44_to_p_Sc47) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 20.0_rt, 45.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ca45_to_Sc46);
    rate_eval.screened_rates(k_p_Ca45_to_Sc46) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ca45_to_Sc46);
        rate_eval.dscreened_rates_dT(k_p_Ca45_to_Sc46) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ca45_to_n_Sc45);
    rate_eval.screened_rates(k_p_Ca45_to_n_Sc45) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ca45_to_n_Sc45);
        rate_eval.dscreened_rates_dT(k_p_Ca45_to_n_Sc45) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 20.0_rt, 45.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ca45_to_Ti49);
    rate_eval.screened_rates(k_He4_Ca45_to_Ti49) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ca45_to_Ti49);
        rate_eval.dscreened_rates_dT(k_He4_Ca45_to_Ti49) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ca45_to_n_Ti48);
    rate_eval.screened_rates(k_He4_Ca45_to_n_Ti48) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ca45_to_n_Ti48);
        rate_eval.dscreened_rates_dT(k_He4_Ca45_to_n_Ti48) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ca45_to_p_Sc48);
    rate_eval.screened_rates(k_He4_Ca45_to_p_Sc48) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ca45_to_p_Sc48);
        rate_eval.dscreened_rates_dT(k_He4_Ca45_to_p_Sc48) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 20.0_rt, 46.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ca46_to_Sc47);
    rate_eval.screened_rates(k_p_Ca46_to_Sc47) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ca46_to_Sc47);
        rate_eval.dscreened_rates_dT(k_p_Ca46_to_Sc47) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ca46_to_n_Sc46);
    rate_eval.screened_rates(k_p_Ca46_to_n_Sc46) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ca46_to_n_Sc46);
        rate_eval.dscreened_rates_dT(k_p_Ca46_to_n_Sc46) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 20.0_rt, 46.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ca46_to_Ti50);
    rate_eval.screened_rates(k_He4_Ca46_to_Ti50) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ca46_to_Ti50);
        rate_eval.dscreened_rates_dT(k_He4_Ca46_to_Ti50) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ca46_to_n_Ti49);
    rate_eval.screened_rates(k_He4_Ca46_to_n_Ti49) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ca46_to_n_Ti49);
        rate_eval.dscreened_rates_dT(k_He4_Ca46_to_n_Ti49) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ca46_to_p_Sc49);
    rate_eval.screened_rates(k_He4_Ca46_to_p_Sc49) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ca46_to_p_Sc49);
        rate_eval.dscreened_rates_dT(k_He4_Ca46_to_p_Sc49) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 20.0_rt, 47.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ca47_to_Sc48);
    rate_eval.screened_rates(k_p_Ca47_to_Sc48) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ca47_to_Sc48);
        rate_eval.dscreened_rates_dT(k_p_Ca47_to_Sc48) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ca47_to_n_Sc47);
    rate_eval.screened_rates(k_p_Ca47_to_n_Sc47) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ca47_to_n_Sc47);
        rate_eval.dscreened_rates_dT(k_p_Ca47_to_n_Sc47) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 20.0_rt, 47.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ca47_to_Ti51);
    rate_eval.screened_rates(k_He4_Ca47_to_Ti51) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ca47_to_Ti51);
        rate_eval.dscreened_rates_dT(k_He4_Ca47_to_Ti51) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ca47_to_n_Ti50);
    rate_eval.screened_rates(k_He4_Ca47_to_n_Ti50) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ca47_to_n_Ti50);
        rate_eval.dscreened_rates_dT(k_He4_Ca47_to_n_Ti50) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 20.0_rt, 48.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ca48_to_Sc49);
    rate_eval.screened_rates(k_p_Ca48_to_Sc49) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ca48_to_Sc49);
        rate_eval.dscreened_rates_dT(k_p_Ca48_to_Sc49) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ca48_to_n_Sc48);
    rate_eval.screened_rates(k_p_Ca48_to_n_Sc48) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ca48_to_n_Sc48);
        rate_eval.dscreened_rates_dT(k_p_Ca48_to_n_Sc48) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 21.0_rt, 43.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Sc43_to_Ti44);
    rate_eval.screened_rates(k_p_Sc43_to_Ti44) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Sc43_to_Ti44);
        rate_eval.dscreened_rates_dT(k_p_Sc43_to_Ti44) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Sc43_to_He4_Ca40);
    rate_eval.screened_rates(k_p_Sc43_to_He4_Ca40) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Sc43_to_He4_Ca40);
        rate_eval.dscreened_rates_dT(k_p_Sc43_to_He4_Ca40) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 21.0_rt, 43.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Sc43_to_V47);
    rate_eval.screened_rates(k_He4_Sc43_to_V47) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Sc43_to_V47);
        rate_eval.dscreened_rates_dT(k_He4_Sc43_to_V47) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Sc43_to_n_V46);
    rate_eval.screened_rates(k_He4_Sc43_to_n_V46) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Sc43_to_n_V46);
        rate_eval.dscreened_rates_dT(k_He4_Sc43_to_n_V46) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Sc43_to_p_Ti46);
    rate_eval.screened_rates(k_He4_Sc43_to_p_Ti46) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Sc43_to_p_Ti46);
        rate_eval.dscreened_rates_dT(k_He4_Sc43_to_p_Ti46) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 21.0_rt, 44.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Sc44_to_Ti45);
    rate_eval.screened_rates(k_p_Sc44_to_Ti45) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Sc44_to_Ti45);
        rate_eval.dscreened_rates_dT(k_p_Sc44_to_Ti45) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Sc44_to_n_Ti44);
    rate_eval.screened_rates(k_p_Sc44_to_n_Ti44) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Sc44_to_n_Ti44);
        rate_eval.dscreened_rates_dT(k_p_Sc44_to_n_Ti44) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Sc44_to_He4_Ca41);
    rate_eval.screened_rates(k_p_Sc44_to_He4_Ca41) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Sc44_to_He4_Ca41);
        rate_eval.dscreened_rates_dT(k_p_Sc44_to_He4_Ca41) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 21.0_rt, 44.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Sc44_to_V48);
    rate_eval.screened_rates(k_He4_Sc44_to_V48) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Sc44_to_V48);
        rate_eval.dscreened_rates_dT(k_He4_Sc44_to_V48) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Sc44_to_n_V47);
    rate_eval.screened_rates(k_He4_Sc44_to_n_V47) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Sc44_to_n_V47);
        rate_eval.dscreened_rates_dT(k_He4_Sc44_to_n_V47) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Sc44_to_p_Ti47);
    rate_eval.screened_rates(k_He4_Sc44_to_p_Ti47) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Sc44_to_p_Ti47);
        rate_eval.dscreened_rates_dT(k_He4_Sc44_to_p_Ti47) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 21.0_rt, 45.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Sc45_to_Ti46);
    rate_eval.screened_rates(k_p_Sc45_to_Ti46) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Sc45_to_Ti46);
        rate_eval.dscreened_rates_dT(k_p_Sc45_to_Ti46) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Sc45_to_n_Ti45);
    rate_eval.screened_rates(k_p_Sc45_to_n_Ti45) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Sc45_to_n_Ti45);
        rate_eval.dscreened_rates_dT(k_p_Sc45_to_n_Ti45) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Sc45_to_He4_Ca42);
    rate_eval.screened_rates(k_p_Sc45_to_He4_Ca42) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Sc45_to_He4_Ca42);
        rate_eval.dscreened_rates_dT(k_p_Sc45_to_He4_Ca42) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 21.0_rt, 45.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Sc45_to_V49);
    rate_eval.screened_rates(k_He4_Sc45_to_V49) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Sc45_to_V49);
        rate_eval.dscreened_rates_dT(k_He4_Sc45_to_V49) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Sc45_to_n_V48);
    rate_eval.screened_rates(k_He4_Sc45_to_n_V48) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Sc45_to_n_V48);
        rate_eval.dscreened_rates_dT(k_He4_Sc45_to_n_V48) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Sc45_to_p_Ti48);
    rate_eval.screened_rates(k_He4_Sc45_to_p_Ti48) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Sc45_to_p_Ti48);
        rate_eval.dscreened_rates_dT(k_He4_Sc45_to_p_Ti48) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 21.0_rt, 46.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Sc46_to_Ti47);
    rate_eval.screened_rates(k_p_Sc46_to_Ti47) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Sc46_to_Ti47);
        rate_eval.dscreened_rates_dT(k_p_Sc46_to_Ti47) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Sc46_to_n_Ti46);
    rate_eval.screened_rates(k_p_Sc46_to_n_Ti46) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Sc46_to_n_Ti46);
        rate_eval.dscreened_rates_dT(k_p_Sc46_to_n_Ti46) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Sc46_to_He4_Ca43);
    rate_eval.screened_rates(k_p_Sc46_to_He4_Ca43) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Sc46_to_He4_Ca43);
        rate_eval.dscreened_rates_dT(k_p_Sc46_to_He4_Ca43) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 21.0_rt, 46.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Sc46_to_V50);
    rate_eval.screened_rates(k_He4_Sc46_to_V50) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Sc46_to_V50);
        rate_eval.dscreened_rates_dT(k_He4_Sc46_to_V50) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Sc46_to_n_V49);
    rate_eval.screened_rates(k_He4_Sc46_to_n_V49) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Sc46_to_n_V49);
        rate_eval.dscreened_rates_dT(k_He4_Sc46_to_n_V49) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Sc46_to_p_Ti49);
    rate_eval.screened_rates(k_He4_Sc46_to_p_Ti49) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Sc46_to_p_Ti49);
        rate_eval.dscreened_rates_dT(k_He4_Sc46_to_p_Ti49) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 21.0_rt, 47.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Sc47_to_Ti48);
    rate_eval.screened_rates(k_p_Sc47_to_Ti48) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Sc47_to_Ti48);
        rate_eval.dscreened_rates_dT(k_p_Sc47_to_Ti48) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Sc47_to_n_Ti47);
    rate_eval.screened_rates(k_p_Sc47_to_n_Ti47) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Sc47_to_n_Ti47);
        rate_eval.dscreened_rates_dT(k_p_Sc47_to_n_Ti47) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Sc47_to_He4_Ca44);
    rate_eval.screened_rates(k_p_Sc47_to_He4_Ca44) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Sc47_to_He4_Ca44);
        rate_eval.dscreened_rates_dT(k_p_Sc47_to_He4_Ca44) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 21.0_rt, 47.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Sc47_to_V51);
    rate_eval.screened_rates(k_He4_Sc47_to_V51) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Sc47_to_V51);
        rate_eval.dscreened_rates_dT(k_He4_Sc47_to_V51) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Sc47_to_n_V50);
    rate_eval.screened_rates(k_He4_Sc47_to_n_V50) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Sc47_to_n_V50);
        rate_eval.dscreened_rates_dT(k_He4_Sc47_to_n_V50) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Sc47_to_p_Ti50);
    rate_eval.screened_rates(k_He4_Sc47_to_p_Ti50) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Sc47_to_p_Ti50);
        rate_eval.dscreened_rates_dT(k_He4_Sc47_to_p_Ti50) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 21.0_rt, 48.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Sc48_to_Ti49);
    rate_eval.screened_rates(k_p_Sc48_to_Ti49) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Sc48_to_Ti49);
        rate_eval.dscreened_rates_dT(k_p_Sc48_to_Ti49) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Sc48_to_n_Ti48);
    rate_eval.screened_rates(k_p_Sc48_to_n_Ti48) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Sc48_to_n_Ti48);
        rate_eval.dscreened_rates_dT(k_p_Sc48_to_n_Ti48) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Sc48_to_He4_Ca45);
    rate_eval.screened_rates(k_p_Sc48_to_He4_Ca45) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Sc48_to_He4_Ca45);
        rate_eval.dscreened_rates_dT(k_p_Sc48_to_He4_Ca45) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 21.0_rt, 48.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Sc48_to_V52);
    rate_eval.screened_rates(k_He4_Sc48_to_V52) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Sc48_to_V52);
        rate_eval.dscreened_rates_dT(k_He4_Sc48_to_V52) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Sc48_to_n_V51);
    rate_eval.screened_rates(k_He4_Sc48_to_n_V51) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Sc48_to_n_V51);
        rate_eval.dscreened_rates_dT(k_He4_Sc48_to_n_V51) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Sc48_to_p_Ti51);
    rate_eval.screened_rates(k_He4_Sc48_to_p_Ti51) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Sc48_to_p_Ti51);
        rate_eval.dscreened_rates_dT(k_He4_Sc48_to_p_Ti51) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 21.0_rt, 49.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Sc49_to_Ti50);
    rate_eval.screened_rates(k_p_Sc49_to_Ti50) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Sc49_to_Ti50);
        rate_eval.dscreened_rates_dT(k_p_Sc49_to_Ti50) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Sc49_to_n_Ti49);
    rate_eval.screened_rates(k_p_Sc49_to_n_Ti49) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Sc49_to_n_Ti49);
        rate_eval.dscreened_rates_dT(k_p_Sc49_to_n_Ti49) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Sc49_to_He4_Ca46);
    rate_eval.screened_rates(k_p_Sc49_to_He4_Ca46) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Sc49_to_He4_Ca46);
        rate_eval.dscreened_rates_dT(k_p_Sc49_to_He4_Ca46) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 22.0_rt, 44.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ti44_to_Cr48);
    rate_eval.screened_rates(k_He4_Ti44_to_Cr48) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ti44_to_Cr48);
        rate_eval.dscreened_rates_dT(k_He4_Ti44_to_Cr48) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ti44_to_p_V47);
    rate_eval.screened_rates(k_He4_Ti44_to_p_V47) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ti44_to_p_V47);
        rate_eval.dscreened_rates_dT(k_He4_Ti44_to_p_V47) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 22.0_rt, 45.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ti45_to_V46);
    rate_eval.screened_rates(k_p_Ti45_to_V46) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ti45_to_V46);
        rate_eval.dscreened_rates_dT(k_p_Ti45_to_V46) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 22.0_rt, 45.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ti45_to_Cr49);
    rate_eval.screened_rates(k_He4_Ti45_to_Cr49) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ti45_to_Cr49);
        rate_eval.dscreened_rates_dT(k_He4_Ti45_to_Cr49) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ti45_to_n_Cr48);
    rate_eval.screened_rates(k_He4_Ti45_to_n_Cr48) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ti45_to_n_Cr48);
        rate_eval.dscreened_rates_dT(k_He4_Ti45_to_n_Cr48) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ti45_to_p_V48);
    rate_eval.screened_rates(k_He4_Ti45_to_p_V48) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ti45_to_p_V48);
        rate_eval.dscreened_rates_dT(k_He4_Ti45_to_p_V48) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 22.0_rt, 46.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ti46_to_V47);
    rate_eval.screened_rates(k_p_Ti46_to_V47) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ti46_to_V47);
        rate_eval.dscreened_rates_dT(k_p_Ti46_to_V47) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ti46_to_n_V46);
    rate_eval.screened_rates(k_p_Ti46_to_n_V46) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ti46_to_n_V46);
        rate_eval.dscreened_rates_dT(k_p_Ti46_to_n_V46) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ti46_to_He4_Sc43);
    rate_eval.screened_rates(k_p_Ti46_to_He4_Sc43) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ti46_to_He4_Sc43);
        rate_eval.dscreened_rates_dT(k_p_Ti46_to_He4_Sc43) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 22.0_rt, 46.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ti46_to_Cr50);
    rate_eval.screened_rates(k_He4_Ti46_to_Cr50) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ti46_to_Cr50);
        rate_eval.dscreened_rates_dT(k_He4_Ti46_to_Cr50) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ti46_to_n_Cr49);
    rate_eval.screened_rates(k_He4_Ti46_to_n_Cr49) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ti46_to_n_Cr49);
        rate_eval.dscreened_rates_dT(k_He4_Ti46_to_n_Cr49) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ti46_to_p_V49);
    rate_eval.screened_rates(k_He4_Ti46_to_p_V49) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ti46_to_p_V49);
        rate_eval.dscreened_rates_dT(k_He4_Ti46_to_p_V49) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 22.0_rt, 47.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ti47_to_V48);
    rate_eval.screened_rates(k_p_Ti47_to_V48) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ti47_to_V48);
        rate_eval.dscreened_rates_dT(k_p_Ti47_to_V48) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ti47_to_n_V47);
    rate_eval.screened_rates(k_p_Ti47_to_n_V47) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ti47_to_n_V47);
        rate_eval.dscreened_rates_dT(k_p_Ti47_to_n_V47) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ti47_to_He4_Sc44);
    rate_eval.screened_rates(k_p_Ti47_to_He4_Sc44) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ti47_to_He4_Sc44);
        rate_eval.dscreened_rates_dT(k_p_Ti47_to_He4_Sc44) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 22.0_rt, 47.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ti47_to_Cr51);
    rate_eval.screened_rates(k_He4_Ti47_to_Cr51) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ti47_to_Cr51);
        rate_eval.dscreened_rates_dT(k_He4_Ti47_to_Cr51) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ti47_to_n_Cr50);
    rate_eval.screened_rates(k_He4_Ti47_to_n_Cr50) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ti47_to_n_Cr50);
        rate_eval.dscreened_rates_dT(k_He4_Ti47_to_n_Cr50) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ti47_to_p_V50);
    rate_eval.screened_rates(k_He4_Ti47_to_p_V50) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ti47_to_p_V50);
        rate_eval.dscreened_rates_dT(k_He4_Ti47_to_p_V50) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 22.0_rt, 48.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ti48_to_V49);
    rate_eval.screened_rates(k_p_Ti48_to_V49) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ti48_to_V49);
        rate_eval.dscreened_rates_dT(k_p_Ti48_to_V49) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ti48_to_n_V48);
    rate_eval.screened_rates(k_p_Ti48_to_n_V48) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ti48_to_n_V48);
        rate_eval.dscreened_rates_dT(k_p_Ti48_to_n_V48) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ti48_to_He4_Sc45);
    rate_eval.screened_rates(k_p_Ti48_to_He4_Sc45) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ti48_to_He4_Sc45);
        rate_eval.dscreened_rates_dT(k_p_Ti48_to_He4_Sc45) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 22.0_rt, 48.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ti48_to_Cr52);
    rate_eval.screened_rates(k_He4_Ti48_to_Cr52) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ti48_to_Cr52);
        rate_eval.dscreened_rates_dT(k_He4_Ti48_to_Cr52) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ti48_to_n_Cr51);
    rate_eval.screened_rates(k_He4_Ti48_to_n_Cr51) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ti48_to_n_Cr51);
        rate_eval.dscreened_rates_dT(k_He4_Ti48_to_n_Cr51) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ti48_to_p_V51);
    rate_eval.screened_rates(k_He4_Ti48_to_p_V51) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ti48_to_p_V51);
        rate_eval.dscreened_rates_dT(k_He4_Ti48_to_p_V51) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 22.0_rt, 49.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ti49_to_V50);
    rate_eval.screened_rates(k_p_Ti49_to_V50) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ti49_to_V50);
        rate_eval.dscreened_rates_dT(k_p_Ti49_to_V50) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ti49_to_n_V49);
    rate_eval.screened_rates(k_p_Ti49_to_n_V49) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ti49_to_n_V49);
        rate_eval.dscreened_rates_dT(k_p_Ti49_to_n_V49) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ti49_to_He4_Sc46);
    rate_eval.screened_rates(k_p_Ti49_to_He4_Sc46) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ti49_to_He4_Sc46);
        rate_eval.dscreened_rates_dT(k_p_Ti49_to_He4_Sc46) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 22.0_rt, 49.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ti49_to_Cr53);
    rate_eval.screened_rates(k_He4_Ti49_to_Cr53) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ti49_to_Cr53);
        rate_eval.dscreened_rates_dT(k_He4_Ti49_to_Cr53) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ti49_to_n_Cr52);
    rate_eval.screened_rates(k_He4_Ti49_to_n_Cr52) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ti49_to_n_Cr52);
        rate_eval.dscreened_rates_dT(k_He4_Ti49_to_n_Cr52) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ti49_to_p_V52);
    rate_eval.screened_rates(k_He4_Ti49_to_p_V52) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ti49_to_p_V52);
        rate_eval.dscreened_rates_dT(k_He4_Ti49_to_p_V52) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 22.0_rt, 50.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ti50_to_V51);
    rate_eval.screened_rates(k_p_Ti50_to_V51) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ti50_to_V51);
        rate_eval.dscreened_rates_dT(k_p_Ti50_to_V51) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ti50_to_n_V50);
    rate_eval.screened_rates(k_p_Ti50_to_n_V50) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ti50_to_n_V50);
        rate_eval.dscreened_rates_dT(k_p_Ti50_to_n_V50) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ti50_to_He4_Sc47);
    rate_eval.screened_rates(k_p_Ti50_to_He4_Sc47) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ti50_to_He4_Sc47);
        rate_eval.dscreened_rates_dT(k_p_Ti50_to_He4_Sc47) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 22.0_rt, 50.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ti50_to_Cr54);
    rate_eval.screened_rates(k_He4_Ti50_to_Cr54) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ti50_to_Cr54);
        rate_eval.dscreened_rates_dT(k_He4_Ti50_to_Cr54) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ti50_to_n_Cr53);
    rate_eval.screened_rates(k_He4_Ti50_to_n_Cr53) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ti50_to_n_Cr53);
        rate_eval.dscreened_rates_dT(k_He4_Ti50_to_n_Cr53) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 22.0_rt, 51.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ti51_to_V52);
    rate_eval.screened_rates(k_p_Ti51_to_V52) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ti51_to_V52);
        rate_eval.dscreened_rates_dT(k_p_Ti51_to_V52) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ti51_to_n_V51);
    rate_eval.screened_rates(k_p_Ti51_to_n_V51) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ti51_to_n_V51);
        rate_eval.dscreened_rates_dT(k_p_Ti51_to_n_V51) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ti51_to_He4_Sc48);
    rate_eval.screened_rates(k_p_Ti51_to_He4_Sc48) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ti51_to_He4_Sc48);
        rate_eval.dscreened_rates_dT(k_p_Ti51_to_He4_Sc48) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 23.0_rt, 46.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_V46_to_Mn50);
    rate_eval.screened_rates(k_He4_V46_to_Mn50) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_V46_to_Mn50);
        rate_eval.dscreened_rates_dT(k_He4_V46_to_Mn50) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_V46_to_p_Cr49);
    rate_eval.screened_rates(k_He4_V46_to_p_Cr49) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_V46_to_p_Cr49);
        rate_eval.dscreened_rates_dT(k_He4_V46_to_p_Cr49) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 23.0_rt, 47.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_V47_to_Cr48);
    rate_eval.screened_rates(k_p_V47_to_Cr48) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_V47_to_Cr48);
        rate_eval.dscreened_rates_dT(k_p_V47_to_Cr48) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_V47_to_He4_Ti44);
    rate_eval.screened_rates(k_p_V47_to_He4_Ti44) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_V47_to_He4_Ti44);
        rate_eval.dscreened_rates_dT(k_p_V47_to_He4_Ti44) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 23.0_rt, 47.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_V47_to_Mn51);
    rate_eval.screened_rates(k_He4_V47_to_Mn51) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_V47_to_Mn51);
        rate_eval.dscreened_rates_dT(k_He4_V47_to_Mn51) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_V47_to_n_Mn50);
    rate_eval.screened_rates(k_He4_V47_to_n_Mn50) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_V47_to_n_Mn50);
        rate_eval.dscreened_rates_dT(k_He4_V47_to_n_Mn50) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_V47_to_p_Cr50);
    rate_eval.screened_rates(k_He4_V47_to_p_Cr50) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_V47_to_p_Cr50);
        rate_eval.dscreened_rates_dT(k_He4_V47_to_p_Cr50) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 23.0_rt, 48.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_V48_to_Cr49);
    rate_eval.screened_rates(k_p_V48_to_Cr49) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_V48_to_Cr49);
        rate_eval.dscreened_rates_dT(k_p_V48_to_Cr49) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_V48_to_n_Cr48);
    rate_eval.screened_rates(k_p_V48_to_n_Cr48) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_V48_to_n_Cr48);
        rate_eval.dscreened_rates_dT(k_p_V48_to_n_Cr48) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_V48_to_He4_Ti45);
    rate_eval.screened_rates(k_p_V48_to_He4_Ti45) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_V48_to_He4_Ti45);
        rate_eval.dscreened_rates_dT(k_p_V48_to_He4_Ti45) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 23.0_rt, 48.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_V48_to_Mn52);
    rate_eval.screened_rates(k_He4_V48_to_Mn52) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_V48_to_Mn52);
        rate_eval.dscreened_rates_dT(k_He4_V48_to_Mn52) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_V48_to_n_Mn51);
    rate_eval.screened_rates(k_He4_V48_to_n_Mn51) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_V48_to_n_Mn51);
        rate_eval.dscreened_rates_dT(k_He4_V48_to_n_Mn51) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_V48_to_p_Cr51);
    rate_eval.screened_rates(k_He4_V48_to_p_Cr51) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_V48_to_p_Cr51);
        rate_eval.dscreened_rates_dT(k_He4_V48_to_p_Cr51) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 23.0_rt, 49.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_V49_to_Cr50);
    rate_eval.screened_rates(k_p_V49_to_Cr50) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_V49_to_Cr50);
        rate_eval.dscreened_rates_dT(k_p_V49_to_Cr50) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_V49_to_n_Cr49);
    rate_eval.screened_rates(k_p_V49_to_n_Cr49) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_V49_to_n_Cr49);
        rate_eval.dscreened_rates_dT(k_p_V49_to_n_Cr49) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_V49_to_He4_Ti46);
    rate_eval.screened_rates(k_p_V49_to_He4_Ti46) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_V49_to_He4_Ti46);
        rate_eval.dscreened_rates_dT(k_p_V49_to_He4_Ti46) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 23.0_rt, 49.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_V49_to_Mn53);
    rate_eval.screened_rates(k_He4_V49_to_Mn53) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_V49_to_Mn53);
        rate_eval.dscreened_rates_dT(k_He4_V49_to_Mn53) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_V49_to_n_Mn52);
    rate_eval.screened_rates(k_He4_V49_to_n_Mn52) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_V49_to_n_Mn52);
        rate_eval.dscreened_rates_dT(k_He4_V49_to_n_Mn52) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_V49_to_p_Cr52);
    rate_eval.screened_rates(k_He4_V49_to_p_Cr52) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_V49_to_p_Cr52);
        rate_eval.dscreened_rates_dT(k_He4_V49_to_p_Cr52) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 23.0_rt, 50.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_V50_to_Cr51);
    rate_eval.screened_rates(k_p_V50_to_Cr51) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_V50_to_Cr51);
        rate_eval.dscreened_rates_dT(k_p_V50_to_Cr51) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_V50_to_n_Cr50);
    rate_eval.screened_rates(k_p_V50_to_n_Cr50) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_V50_to_n_Cr50);
        rate_eval.dscreened_rates_dT(k_p_V50_to_n_Cr50) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_V50_to_He4_Ti47);
    rate_eval.screened_rates(k_p_V50_to_He4_Ti47) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_V50_to_He4_Ti47);
        rate_eval.dscreened_rates_dT(k_p_V50_to_He4_Ti47) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 23.0_rt, 50.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_V50_to_Mn54);
    rate_eval.screened_rates(k_He4_V50_to_Mn54) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_V50_to_Mn54);
        rate_eval.dscreened_rates_dT(k_He4_V50_to_Mn54) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_V50_to_n_Mn53);
    rate_eval.screened_rates(k_He4_V50_to_n_Mn53) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_V50_to_n_Mn53);
        rate_eval.dscreened_rates_dT(k_He4_V50_to_n_Mn53) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_V50_to_p_Cr53);
    rate_eval.screened_rates(k_He4_V50_to_p_Cr53) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_V50_to_p_Cr53);
        rate_eval.dscreened_rates_dT(k_He4_V50_to_p_Cr53) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 23.0_rt, 51.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_V51_to_Cr52);
    rate_eval.screened_rates(k_p_V51_to_Cr52) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_V51_to_Cr52);
        rate_eval.dscreened_rates_dT(k_p_V51_to_Cr52) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_V51_to_n_Cr51);
    rate_eval.screened_rates(k_p_V51_to_n_Cr51) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_V51_to_n_Cr51);
        rate_eval.dscreened_rates_dT(k_p_V51_to_n_Cr51) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_V51_to_He4_Ti48);
    rate_eval.screened_rates(k_p_V51_to_He4_Ti48) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_V51_to_He4_Ti48);
        rate_eval.dscreened_rates_dT(k_p_V51_to_He4_Ti48) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 23.0_rt, 51.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_V51_to_Mn55);
    rate_eval.screened_rates(k_He4_V51_to_Mn55) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_V51_to_Mn55);
        rate_eval.dscreened_rates_dT(k_He4_V51_to_Mn55) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_V51_to_n_Mn54);
    rate_eval.screened_rates(k_He4_V51_to_n_Mn54) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_V51_to_n_Mn54);
        rate_eval.dscreened_rates_dT(k_He4_V51_to_n_Mn54) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_V51_to_p_Cr54);
    rate_eval.screened_rates(k_He4_V51_to_p_Cr54) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_V51_to_p_Cr54);
        rate_eval.dscreened_rates_dT(k_He4_V51_to_p_Cr54) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 23.0_rt, 52.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_V52_to_Cr53);
    rate_eval.screened_rates(k_p_V52_to_Cr53) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_V52_to_Cr53);
        rate_eval.dscreened_rates_dT(k_p_V52_to_Cr53) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_V52_to_n_Cr52);
    rate_eval.screened_rates(k_p_V52_to_n_Cr52) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_V52_to_n_Cr52);
        rate_eval.dscreened_rates_dT(k_p_V52_to_n_Cr52) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_V52_to_He4_Ti49);
    rate_eval.screened_rates(k_p_V52_to_He4_Ti49) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_V52_to_He4_Ti49);
        rate_eval.dscreened_rates_dT(k_p_V52_to_He4_Ti49) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 24.0_rt, 48.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Cr48_to_Fe52);
    rate_eval.screened_rates(k_He4_Cr48_to_Fe52) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cr48_to_Fe52);
        rate_eval.dscreened_rates_dT(k_He4_Cr48_to_Fe52) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cr48_to_p_Mn51);
    rate_eval.screened_rates(k_He4_Cr48_to_p_Mn51) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cr48_to_p_Mn51);
        rate_eval.dscreened_rates_dT(k_He4_Cr48_to_p_Mn51) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 24.0_rt, 49.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Cr49_to_Mn50);
    rate_eval.screened_rates(k_p_Cr49_to_Mn50) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cr49_to_Mn50);
        rate_eval.dscreened_rates_dT(k_p_Cr49_to_Mn50) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cr49_to_He4_V46);
    rate_eval.screened_rates(k_p_Cr49_to_He4_V46) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cr49_to_He4_V46);
        rate_eval.dscreened_rates_dT(k_p_Cr49_to_He4_V46) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 24.0_rt, 49.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Cr49_to_Fe53);
    rate_eval.screened_rates(k_He4_Cr49_to_Fe53) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cr49_to_Fe53);
        rate_eval.dscreened_rates_dT(k_He4_Cr49_to_Fe53) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cr49_to_n_Fe52);
    rate_eval.screened_rates(k_He4_Cr49_to_n_Fe52) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cr49_to_n_Fe52);
        rate_eval.dscreened_rates_dT(k_He4_Cr49_to_n_Fe52) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cr49_to_p_Mn52);
    rate_eval.screened_rates(k_He4_Cr49_to_p_Mn52) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cr49_to_p_Mn52);
        rate_eval.dscreened_rates_dT(k_He4_Cr49_to_p_Mn52) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 24.0_rt, 50.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Cr50_to_Mn51);
    rate_eval.screened_rates(k_p_Cr50_to_Mn51) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cr50_to_Mn51);
        rate_eval.dscreened_rates_dT(k_p_Cr50_to_Mn51) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cr50_to_n_Mn50);
    rate_eval.screened_rates(k_p_Cr50_to_n_Mn50) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cr50_to_n_Mn50);
        rate_eval.dscreened_rates_dT(k_p_Cr50_to_n_Mn50) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cr50_to_He4_V47);
    rate_eval.screened_rates(k_p_Cr50_to_He4_V47) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cr50_to_He4_V47);
        rate_eval.dscreened_rates_dT(k_p_Cr50_to_He4_V47) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 24.0_rt, 50.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Cr50_to_Fe54);
    rate_eval.screened_rates(k_He4_Cr50_to_Fe54) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cr50_to_Fe54);
        rate_eval.dscreened_rates_dT(k_He4_Cr50_to_Fe54) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cr50_to_n_Fe53);
    rate_eval.screened_rates(k_He4_Cr50_to_n_Fe53) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cr50_to_n_Fe53);
        rate_eval.dscreened_rates_dT(k_He4_Cr50_to_n_Fe53) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cr50_to_p_Mn53);
    rate_eval.screened_rates(k_He4_Cr50_to_p_Mn53) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cr50_to_p_Mn53);
        rate_eval.dscreened_rates_dT(k_He4_Cr50_to_p_Mn53) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 24.0_rt, 51.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Cr51_to_Mn52);
    rate_eval.screened_rates(k_p_Cr51_to_Mn52) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cr51_to_Mn52);
        rate_eval.dscreened_rates_dT(k_p_Cr51_to_Mn52) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cr51_to_n_Mn51);
    rate_eval.screened_rates(k_p_Cr51_to_n_Mn51) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cr51_to_n_Mn51);
        rate_eval.dscreened_rates_dT(k_p_Cr51_to_n_Mn51) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cr51_to_He4_V48);
    rate_eval.screened_rates(k_p_Cr51_to_He4_V48) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cr51_to_He4_V48);
        rate_eval.dscreened_rates_dT(k_p_Cr51_to_He4_V48) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 24.0_rt, 51.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Cr51_to_Fe55);
    rate_eval.screened_rates(k_He4_Cr51_to_Fe55) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cr51_to_Fe55);
        rate_eval.dscreened_rates_dT(k_He4_Cr51_to_Fe55) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cr51_to_n_Fe54);
    rate_eval.screened_rates(k_He4_Cr51_to_n_Fe54) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cr51_to_n_Fe54);
        rate_eval.dscreened_rates_dT(k_He4_Cr51_to_n_Fe54) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cr51_to_p_Mn54);
    rate_eval.screened_rates(k_He4_Cr51_to_p_Mn54) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cr51_to_p_Mn54);
        rate_eval.dscreened_rates_dT(k_He4_Cr51_to_p_Mn54) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 24.0_rt, 52.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Cr52_to_Mn53);
    rate_eval.screened_rates(k_p_Cr52_to_Mn53) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cr52_to_Mn53);
        rate_eval.dscreened_rates_dT(k_p_Cr52_to_Mn53) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cr52_to_n_Mn52);
    rate_eval.screened_rates(k_p_Cr52_to_n_Mn52) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cr52_to_n_Mn52);
        rate_eval.dscreened_rates_dT(k_p_Cr52_to_n_Mn52) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cr52_to_He4_V49);
    rate_eval.screened_rates(k_p_Cr52_to_He4_V49) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cr52_to_He4_V49);
        rate_eval.dscreened_rates_dT(k_p_Cr52_to_He4_V49) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 24.0_rt, 52.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Cr52_to_Fe56);
    rate_eval.screened_rates(k_He4_Cr52_to_Fe56) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cr52_to_Fe56);
        rate_eval.dscreened_rates_dT(k_He4_Cr52_to_Fe56) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cr52_to_n_Fe55);
    rate_eval.screened_rates(k_He4_Cr52_to_n_Fe55) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cr52_to_n_Fe55);
        rate_eval.dscreened_rates_dT(k_He4_Cr52_to_n_Fe55) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cr52_to_p_Mn55);
    rate_eval.screened_rates(k_He4_Cr52_to_p_Mn55) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cr52_to_p_Mn55);
        rate_eval.dscreened_rates_dT(k_He4_Cr52_to_p_Mn55) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 24.0_rt, 53.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Cr53_to_Mn54);
    rate_eval.screened_rates(k_p_Cr53_to_Mn54) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cr53_to_Mn54);
        rate_eval.dscreened_rates_dT(k_p_Cr53_to_Mn54) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cr53_to_n_Mn53);
    rate_eval.screened_rates(k_p_Cr53_to_n_Mn53) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cr53_to_n_Mn53);
        rate_eval.dscreened_rates_dT(k_p_Cr53_to_n_Mn53) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cr53_to_He4_V50);
    rate_eval.screened_rates(k_p_Cr53_to_He4_V50) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cr53_to_He4_V50);
        rate_eval.dscreened_rates_dT(k_p_Cr53_to_He4_V50) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 24.0_rt, 53.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Cr53_to_Fe57);
    rate_eval.screened_rates(k_He4_Cr53_to_Fe57) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cr53_to_Fe57);
        rate_eval.dscreened_rates_dT(k_He4_Cr53_to_Fe57) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cr53_to_n_Fe56);
    rate_eval.screened_rates(k_He4_Cr53_to_n_Fe56) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cr53_to_n_Fe56);
        rate_eval.dscreened_rates_dT(k_He4_Cr53_to_n_Fe56) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 24.0_rt, 54.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Cr54_to_Mn55);
    rate_eval.screened_rates(k_p_Cr54_to_Mn55) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cr54_to_Mn55);
        rate_eval.dscreened_rates_dT(k_p_Cr54_to_Mn55) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cr54_to_n_Mn54);
    rate_eval.screened_rates(k_p_Cr54_to_n_Mn54) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cr54_to_n_Mn54);
        rate_eval.dscreened_rates_dT(k_p_Cr54_to_n_Mn54) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cr54_to_He4_V51);
    rate_eval.screened_rates(k_p_Cr54_to_He4_V51) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cr54_to_He4_V51);
        rate_eval.dscreened_rates_dT(k_p_Cr54_to_He4_V51) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 24.0_rt, 54.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Cr54_to_Fe58);
    rate_eval.screened_rates(k_He4_Cr54_to_Fe58) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cr54_to_Fe58);
        rate_eval.dscreened_rates_dT(k_He4_Cr54_to_Fe58) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cr54_to_n_Fe57);
    rate_eval.screened_rates(k_He4_Cr54_to_n_Fe57) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cr54_to_n_Fe57);
        rate_eval.dscreened_rates_dT(k_He4_Cr54_to_n_Fe57) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 25.0_rt, 50.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Mn50_to_Co54);
    rate_eval.screened_rates(k_He4_Mn50_to_Co54) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mn50_to_Co54);
        rate_eval.dscreened_rates_dT(k_He4_Mn50_to_Co54) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Mn50_to_n_Co53);
    rate_eval.screened_rates(k_He4_Mn50_to_n_Co53) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mn50_to_n_Co53);
        rate_eval.dscreened_rates_dT(k_He4_Mn50_to_n_Co53) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Mn50_to_p_Fe53);
    rate_eval.screened_rates(k_He4_Mn50_to_p_Fe53) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mn50_to_p_Fe53);
        rate_eval.dscreened_rates_dT(k_He4_Mn50_to_p_Fe53) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 25.0_rt, 51.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Mn51_to_Fe52);
    rate_eval.screened_rates(k_p_Mn51_to_Fe52) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Mn51_to_Fe52);
        rate_eval.dscreened_rates_dT(k_p_Mn51_to_Fe52) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Mn51_to_He4_Cr48);
    rate_eval.screened_rates(k_p_Mn51_to_He4_Cr48) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Mn51_to_He4_Cr48);
        rate_eval.dscreened_rates_dT(k_p_Mn51_to_He4_Cr48) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 25.0_rt, 51.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Mn51_to_Co55);
    rate_eval.screened_rates(k_He4_Mn51_to_Co55) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mn51_to_Co55);
        rate_eval.dscreened_rates_dT(k_He4_Mn51_to_Co55) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Mn51_to_n_Co54);
    rate_eval.screened_rates(k_He4_Mn51_to_n_Co54) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mn51_to_n_Co54);
        rate_eval.dscreened_rates_dT(k_He4_Mn51_to_n_Co54) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Mn51_to_p_Fe54);
    rate_eval.screened_rates(k_He4_Mn51_to_p_Fe54) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mn51_to_p_Fe54);
        rate_eval.dscreened_rates_dT(k_He4_Mn51_to_p_Fe54) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 25.0_rt, 52.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Mn52_to_Fe53);
    rate_eval.screened_rates(k_p_Mn52_to_Fe53) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Mn52_to_Fe53);
        rate_eval.dscreened_rates_dT(k_p_Mn52_to_Fe53) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Mn52_to_n_Fe52);
    rate_eval.screened_rates(k_p_Mn52_to_n_Fe52) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Mn52_to_n_Fe52);
        rate_eval.dscreened_rates_dT(k_p_Mn52_to_n_Fe52) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Mn52_to_He4_Cr49);
    rate_eval.screened_rates(k_p_Mn52_to_He4_Cr49) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Mn52_to_He4_Cr49);
        rate_eval.dscreened_rates_dT(k_p_Mn52_to_He4_Cr49) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 25.0_rt, 52.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Mn52_to_Co56);
    rate_eval.screened_rates(k_He4_Mn52_to_Co56) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mn52_to_Co56);
        rate_eval.dscreened_rates_dT(k_He4_Mn52_to_Co56) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Mn52_to_n_Co55);
    rate_eval.screened_rates(k_He4_Mn52_to_n_Co55) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mn52_to_n_Co55);
        rate_eval.dscreened_rates_dT(k_He4_Mn52_to_n_Co55) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Mn52_to_p_Fe55);
    rate_eval.screened_rates(k_He4_Mn52_to_p_Fe55) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mn52_to_p_Fe55);
        rate_eval.dscreened_rates_dT(k_He4_Mn52_to_p_Fe55) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 25.0_rt, 53.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Mn53_to_Fe54);
    rate_eval.screened_rates(k_p_Mn53_to_Fe54) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Mn53_to_Fe54);
        rate_eval.dscreened_rates_dT(k_p_Mn53_to_Fe54) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Mn53_to_n_Fe53);
    rate_eval.screened_rates(k_p_Mn53_to_n_Fe53) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Mn53_to_n_Fe53);
        rate_eval.dscreened_rates_dT(k_p_Mn53_to_n_Fe53) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Mn53_to_He4_Cr50);
    rate_eval.screened_rates(k_p_Mn53_to_He4_Cr50) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Mn53_to_He4_Cr50);
        rate_eval.dscreened_rates_dT(k_p_Mn53_to_He4_Cr50) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 25.0_rt, 53.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Mn53_to_Co57);
    rate_eval.screened_rates(k_He4_Mn53_to_Co57) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mn53_to_Co57);
        rate_eval.dscreened_rates_dT(k_He4_Mn53_to_Co57) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Mn53_to_n_Co56);
    rate_eval.screened_rates(k_He4_Mn53_to_n_Co56) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mn53_to_n_Co56);
        rate_eval.dscreened_rates_dT(k_He4_Mn53_to_n_Co56) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Mn53_to_p_Fe56);
    rate_eval.screened_rates(k_He4_Mn53_to_p_Fe56) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mn53_to_p_Fe56);
        rate_eval.dscreened_rates_dT(k_He4_Mn53_to_p_Fe56) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 25.0_rt, 54.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Mn54_to_Fe55);
    rate_eval.screened_rates(k_p_Mn54_to_Fe55) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Mn54_to_Fe55);
        rate_eval.dscreened_rates_dT(k_p_Mn54_to_Fe55) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Mn54_to_n_Fe54);
    rate_eval.screened_rates(k_p_Mn54_to_n_Fe54) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Mn54_to_n_Fe54);
        rate_eval.dscreened_rates_dT(k_p_Mn54_to_n_Fe54) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Mn54_to_He4_Cr51);
    rate_eval.screened_rates(k_p_Mn54_to_He4_Cr51) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Mn54_to_He4_Cr51);
        rate_eval.dscreened_rates_dT(k_p_Mn54_to_He4_Cr51) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 25.0_rt, 54.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Mn54_to_Co58);
    rate_eval.screened_rates(k_He4_Mn54_to_Co58) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mn54_to_Co58);
        rate_eval.dscreened_rates_dT(k_He4_Mn54_to_Co58) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Mn54_to_n_Co57);
    rate_eval.screened_rates(k_He4_Mn54_to_n_Co57) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mn54_to_n_Co57);
        rate_eval.dscreened_rates_dT(k_He4_Mn54_to_n_Co57) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Mn54_to_p_Fe57);
    rate_eval.screened_rates(k_He4_Mn54_to_p_Fe57) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mn54_to_p_Fe57);
        rate_eval.dscreened_rates_dT(k_He4_Mn54_to_p_Fe57) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 25.0_rt, 55.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Mn55_to_Fe56);
    rate_eval.screened_rates(k_p_Mn55_to_Fe56) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Mn55_to_Fe56);
        rate_eval.dscreened_rates_dT(k_p_Mn55_to_Fe56) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Mn55_to_n_Fe55);
    rate_eval.screened_rates(k_p_Mn55_to_n_Fe55) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Mn55_to_n_Fe55);
        rate_eval.dscreened_rates_dT(k_p_Mn55_to_n_Fe55) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Mn55_to_He4_Cr52);
    rate_eval.screened_rates(k_p_Mn55_to_He4_Cr52) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Mn55_to_He4_Cr52);
        rate_eval.dscreened_rates_dT(k_p_Mn55_to_He4_Cr52) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 25.0_rt, 55.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Mn55_to_Co59);
    rate_eval.screened_rates(k_He4_Mn55_to_Co59) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mn55_to_Co59);
        rate_eval.dscreened_rates_dT(k_He4_Mn55_to_Co59) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Mn55_to_n_Co58);
    rate_eval.screened_rates(k_He4_Mn55_to_n_Co58) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mn55_to_n_Co58);
        rate_eval.dscreened_rates_dT(k_He4_Mn55_to_n_Co58) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Mn55_to_p_Fe58);
    rate_eval.screened_rates(k_He4_Mn55_to_p_Fe58) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mn55_to_p_Fe58);
        rate_eval.dscreened_rates_dT(k_He4_Mn55_to_p_Fe58) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 26.0_rt, 52.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Fe52_to_Co53);
    rate_eval.screened_rates(k_p_Fe52_to_Co53) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Fe52_to_Co53);
        rate_eval.dscreened_rates_dT(k_p_Fe52_to_Co53) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 26.0_rt, 52.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Fe52_to_Ni56);
    rate_eval.screened_rates(k_He4_Fe52_to_Ni56) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Fe52_to_Ni56);
        rate_eval.dscreened_rates_dT(k_He4_Fe52_to_Ni56) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Fe52_to_p_Co55);
    rate_eval.screened_rates(k_He4_Fe52_to_p_Co55) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Fe52_to_p_Co55);
        rate_eval.dscreened_rates_dT(k_He4_Fe52_to_p_Co55) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 26.0_rt, 53.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Fe53_to_Co54);
    rate_eval.screened_rates(k_p_Fe53_to_Co54) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Fe53_to_Co54);
        rate_eval.dscreened_rates_dT(k_p_Fe53_to_Co54) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Fe53_to_n_Co53);
    rate_eval.screened_rates(k_p_Fe53_to_n_Co53) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Fe53_to_n_Co53);
        rate_eval.dscreened_rates_dT(k_p_Fe53_to_n_Co53) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Fe53_to_He4_Mn50);
    rate_eval.screened_rates(k_p_Fe53_to_He4_Mn50) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Fe53_to_He4_Mn50);
        rate_eval.dscreened_rates_dT(k_p_Fe53_to_He4_Mn50) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 26.0_rt, 53.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Fe53_to_Ni57);
    rate_eval.screened_rates(k_He4_Fe53_to_Ni57) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Fe53_to_Ni57);
        rate_eval.dscreened_rates_dT(k_He4_Fe53_to_Ni57) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Fe53_to_n_Ni56);
    rate_eval.screened_rates(k_He4_Fe53_to_n_Ni56) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Fe53_to_n_Ni56);
        rate_eval.dscreened_rates_dT(k_He4_Fe53_to_n_Ni56) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Fe53_to_p_Co56);
    rate_eval.screened_rates(k_He4_Fe53_to_p_Co56) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Fe53_to_p_Co56);
        rate_eval.dscreened_rates_dT(k_He4_Fe53_to_p_Co56) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 26.0_rt, 54.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Fe54_to_Co55);
    rate_eval.screened_rates(k_p_Fe54_to_Co55) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Fe54_to_Co55);
        rate_eval.dscreened_rates_dT(k_p_Fe54_to_Co55) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Fe54_to_n_Co54);
    rate_eval.screened_rates(k_p_Fe54_to_n_Co54) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Fe54_to_n_Co54);
        rate_eval.dscreened_rates_dT(k_p_Fe54_to_n_Co54) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Fe54_to_He4_Mn51);
    rate_eval.screened_rates(k_p_Fe54_to_He4_Mn51) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Fe54_to_He4_Mn51);
        rate_eval.dscreened_rates_dT(k_p_Fe54_to_He4_Mn51) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 26.0_rt, 54.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Fe54_to_Ni58);
    rate_eval.screened_rates(k_He4_Fe54_to_Ni58) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Fe54_to_Ni58);
        rate_eval.dscreened_rates_dT(k_He4_Fe54_to_Ni58) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Fe54_to_n_Ni57);
    rate_eval.screened_rates(k_He4_Fe54_to_n_Ni57) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Fe54_to_n_Ni57);
        rate_eval.dscreened_rates_dT(k_He4_Fe54_to_n_Ni57) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Fe54_to_p_Co57);
    rate_eval.screened_rates(k_He4_Fe54_to_p_Co57) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Fe54_to_p_Co57);
        rate_eval.dscreened_rates_dT(k_He4_Fe54_to_p_Co57) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 26.0_rt, 55.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Fe55_to_Co56);
    rate_eval.screened_rates(k_p_Fe55_to_Co56) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Fe55_to_Co56);
        rate_eval.dscreened_rates_dT(k_p_Fe55_to_Co56) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Fe55_to_n_Co55);
    rate_eval.screened_rates(k_p_Fe55_to_n_Co55) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Fe55_to_n_Co55);
        rate_eval.dscreened_rates_dT(k_p_Fe55_to_n_Co55) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Fe55_to_He4_Mn52);
    rate_eval.screened_rates(k_p_Fe55_to_He4_Mn52) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Fe55_to_He4_Mn52);
        rate_eval.dscreened_rates_dT(k_p_Fe55_to_He4_Mn52) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 26.0_rt, 55.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Fe55_to_Ni59);
    rate_eval.screened_rates(k_He4_Fe55_to_Ni59) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Fe55_to_Ni59);
        rate_eval.dscreened_rates_dT(k_He4_Fe55_to_Ni59) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Fe55_to_n_Ni58);
    rate_eval.screened_rates(k_He4_Fe55_to_n_Ni58) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Fe55_to_n_Ni58);
        rate_eval.dscreened_rates_dT(k_He4_Fe55_to_n_Ni58) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Fe55_to_p_Co58);
    rate_eval.screened_rates(k_He4_Fe55_to_p_Co58) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Fe55_to_p_Co58);
        rate_eval.dscreened_rates_dT(k_He4_Fe55_to_p_Co58) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 26.0_rt, 56.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Fe56_to_Co57);
    rate_eval.screened_rates(k_p_Fe56_to_Co57) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Fe56_to_Co57);
        rate_eval.dscreened_rates_dT(k_p_Fe56_to_Co57) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Fe56_to_n_Co56);
    rate_eval.screened_rates(k_p_Fe56_to_n_Co56) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Fe56_to_n_Co56);
        rate_eval.dscreened_rates_dT(k_p_Fe56_to_n_Co56) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Fe56_to_He4_Mn53);
    rate_eval.screened_rates(k_p_Fe56_to_He4_Mn53) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Fe56_to_He4_Mn53);
        rate_eval.dscreened_rates_dT(k_p_Fe56_to_He4_Mn53) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 26.0_rt, 56.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Fe56_to_Ni60);
    rate_eval.screened_rates(k_He4_Fe56_to_Ni60) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Fe56_to_Ni60);
        rate_eval.dscreened_rates_dT(k_He4_Fe56_to_Ni60) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Fe56_to_n_Ni59);
    rate_eval.screened_rates(k_He4_Fe56_to_n_Ni59) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Fe56_to_n_Ni59);
        rate_eval.dscreened_rates_dT(k_He4_Fe56_to_n_Ni59) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Fe56_to_p_Co59);
    rate_eval.screened_rates(k_He4_Fe56_to_p_Co59) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Fe56_to_p_Co59);
        rate_eval.dscreened_rates_dT(k_He4_Fe56_to_p_Co59) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 26.0_rt, 57.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Fe57_to_Co58);
    rate_eval.screened_rates(k_p_Fe57_to_Co58) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Fe57_to_Co58);
        rate_eval.dscreened_rates_dT(k_p_Fe57_to_Co58) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Fe57_to_n_Co57);
    rate_eval.screened_rates(k_p_Fe57_to_n_Co57) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Fe57_to_n_Co57);
        rate_eval.dscreened_rates_dT(k_p_Fe57_to_n_Co57) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Fe57_to_He4_Mn54);
    rate_eval.screened_rates(k_p_Fe57_to_He4_Mn54) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Fe57_to_He4_Mn54);
        rate_eval.dscreened_rates_dT(k_p_Fe57_to_He4_Mn54) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 26.0_rt, 57.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Fe57_to_Ni61);
    rate_eval.screened_rates(k_He4_Fe57_to_Ni61) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Fe57_to_Ni61);
        rate_eval.dscreened_rates_dT(k_He4_Fe57_to_Ni61) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Fe57_to_n_Ni60);
    rate_eval.screened_rates(k_He4_Fe57_to_n_Ni60) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Fe57_to_n_Ni60);
        rate_eval.dscreened_rates_dT(k_He4_Fe57_to_n_Ni60) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 26.0_rt, 58.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Fe58_to_Co59);
    rate_eval.screened_rates(k_p_Fe58_to_Co59) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Fe58_to_Co59);
        rate_eval.dscreened_rates_dT(k_p_Fe58_to_Co59) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Fe58_to_n_Co58);
    rate_eval.screened_rates(k_p_Fe58_to_n_Co58) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Fe58_to_n_Co58);
        rate_eval.dscreened_rates_dT(k_p_Fe58_to_n_Co58) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Fe58_to_He4_Mn55);
    rate_eval.screened_rates(k_p_Fe58_to_He4_Mn55) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Fe58_to_He4_Mn55);
        rate_eval.dscreened_rates_dT(k_p_Fe58_to_He4_Mn55) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 26.0_rt, 58.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Fe58_to_Ni62);
    rate_eval.screened_rates(k_He4_Fe58_to_Ni62) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Fe58_to_Ni62);
        rate_eval.dscreened_rates_dT(k_He4_Fe58_to_Ni62) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Fe58_to_n_Ni61);
    rate_eval.screened_rates(k_He4_Fe58_to_n_Ni61) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Fe58_to_n_Ni61);
        rate_eval.dscreened_rates_dT(k_He4_Fe58_to_n_Ni61) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 27.0_rt, 53.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Co53_to_Cu57);
    rate_eval.screened_rates(k_He4_Co53_to_Cu57) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Co53_to_Cu57);
        rate_eval.dscreened_rates_dT(k_He4_Co53_to_Cu57) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Co53_to_p_Ni56);
    rate_eval.screened_rates(k_He4_Co53_to_p_Ni56) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Co53_to_p_Ni56);
        rate_eval.dscreened_rates_dT(k_He4_Co53_to_p_Ni56) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 27.0_rt, 54.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Co54_to_Cu58);
    rate_eval.screened_rates(k_He4_Co54_to_Cu58) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Co54_to_Cu58);
        rate_eval.dscreened_rates_dT(k_He4_Co54_to_Cu58) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Co54_to_n_Cu57);
    rate_eval.screened_rates(k_He4_Co54_to_n_Cu57) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Co54_to_n_Cu57);
        rate_eval.dscreened_rates_dT(k_He4_Co54_to_n_Cu57) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Co54_to_p_Ni57);
    rate_eval.screened_rates(k_He4_Co54_to_p_Ni57) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Co54_to_p_Ni57);
        rate_eval.dscreened_rates_dT(k_He4_Co54_to_p_Ni57) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 27.0_rt, 55.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Co55_to_Ni56);
    rate_eval.screened_rates(k_p_Co55_to_Ni56) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Co55_to_Ni56);
        rate_eval.dscreened_rates_dT(k_p_Co55_to_Ni56) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Co55_to_He4_Fe52);
    rate_eval.screened_rates(k_p_Co55_to_He4_Fe52) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Co55_to_He4_Fe52);
        rate_eval.dscreened_rates_dT(k_p_Co55_to_He4_Fe52) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 27.0_rt, 55.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Co55_to_Cu59);
    rate_eval.screened_rates(k_He4_Co55_to_Cu59) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Co55_to_Cu59);
        rate_eval.dscreened_rates_dT(k_He4_Co55_to_Cu59) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Co55_to_n_Cu58);
    rate_eval.screened_rates(k_He4_Co55_to_n_Cu58) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Co55_to_n_Cu58);
        rate_eval.dscreened_rates_dT(k_He4_Co55_to_n_Cu58) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Co55_to_p_Ni58);
    rate_eval.screened_rates(k_He4_Co55_to_p_Ni58) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Co55_to_p_Ni58);
        rate_eval.dscreened_rates_dT(k_He4_Co55_to_p_Ni58) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 27.0_rt, 56.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Co56_to_Ni57);
    rate_eval.screened_rates(k_p_Co56_to_Ni57) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Co56_to_Ni57);
        rate_eval.dscreened_rates_dT(k_p_Co56_to_Ni57) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Co56_to_n_Ni56);
    rate_eval.screened_rates(k_p_Co56_to_n_Ni56) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Co56_to_n_Ni56);
        rate_eval.dscreened_rates_dT(k_p_Co56_to_n_Ni56) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Co56_to_He4_Fe53);
    rate_eval.screened_rates(k_p_Co56_to_He4_Fe53) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Co56_to_He4_Fe53);
        rate_eval.dscreened_rates_dT(k_p_Co56_to_He4_Fe53) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 27.0_rt, 56.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Co56_to_Cu60);
    rate_eval.screened_rates(k_He4_Co56_to_Cu60) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Co56_to_Cu60);
        rate_eval.dscreened_rates_dT(k_He4_Co56_to_Cu60) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Co56_to_n_Cu59);
    rate_eval.screened_rates(k_He4_Co56_to_n_Cu59) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Co56_to_n_Cu59);
        rate_eval.dscreened_rates_dT(k_He4_Co56_to_n_Cu59) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Co56_to_p_Ni59);
    rate_eval.screened_rates(k_He4_Co56_to_p_Ni59) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Co56_to_p_Ni59);
        rate_eval.dscreened_rates_dT(k_He4_Co56_to_p_Ni59) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 27.0_rt, 57.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Co57_to_Ni58);
    rate_eval.screened_rates(k_p_Co57_to_Ni58) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Co57_to_Ni58);
        rate_eval.dscreened_rates_dT(k_p_Co57_to_Ni58) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Co57_to_n_Ni57);
    rate_eval.screened_rates(k_p_Co57_to_n_Ni57) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Co57_to_n_Ni57);
        rate_eval.dscreened_rates_dT(k_p_Co57_to_n_Ni57) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Co57_to_He4_Fe54);
    rate_eval.screened_rates(k_p_Co57_to_He4_Fe54) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Co57_to_He4_Fe54);
        rate_eval.dscreened_rates_dT(k_p_Co57_to_He4_Fe54) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 27.0_rt, 57.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Co57_to_Cu61);
    rate_eval.screened_rates(k_He4_Co57_to_Cu61) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Co57_to_Cu61);
        rate_eval.dscreened_rates_dT(k_He4_Co57_to_Cu61) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Co57_to_n_Cu60);
    rate_eval.screened_rates(k_He4_Co57_to_n_Cu60) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Co57_to_n_Cu60);
        rate_eval.dscreened_rates_dT(k_He4_Co57_to_n_Cu60) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Co57_to_p_Ni60);
    rate_eval.screened_rates(k_He4_Co57_to_p_Ni60) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Co57_to_p_Ni60);
        rate_eval.dscreened_rates_dT(k_He4_Co57_to_p_Ni60) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 27.0_rt, 58.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Co58_to_Ni59);
    rate_eval.screened_rates(k_p_Co58_to_Ni59) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Co58_to_Ni59);
        rate_eval.dscreened_rates_dT(k_p_Co58_to_Ni59) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Co58_to_n_Ni58);
    rate_eval.screened_rates(k_p_Co58_to_n_Ni58) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Co58_to_n_Ni58);
        rate_eval.dscreened_rates_dT(k_p_Co58_to_n_Ni58) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Co58_to_He4_Fe55);
    rate_eval.screened_rates(k_p_Co58_to_He4_Fe55) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Co58_to_He4_Fe55);
        rate_eval.dscreened_rates_dT(k_p_Co58_to_He4_Fe55) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 27.0_rt, 58.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Co58_to_Cu62);
    rate_eval.screened_rates(k_He4_Co58_to_Cu62) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Co58_to_Cu62);
        rate_eval.dscreened_rates_dT(k_He4_Co58_to_Cu62) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Co58_to_n_Cu61);
    rate_eval.screened_rates(k_He4_Co58_to_n_Cu61) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Co58_to_n_Cu61);
        rate_eval.dscreened_rates_dT(k_He4_Co58_to_n_Cu61) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Co58_to_p_Ni61);
    rate_eval.screened_rates(k_He4_Co58_to_p_Ni61) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Co58_to_p_Ni61);
        rate_eval.dscreened_rates_dT(k_He4_Co58_to_p_Ni61) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 27.0_rt, 59.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Co59_to_Ni60);
    rate_eval.screened_rates(k_p_Co59_to_Ni60) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Co59_to_Ni60);
        rate_eval.dscreened_rates_dT(k_p_Co59_to_Ni60) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Co59_to_n_Ni59);
    rate_eval.screened_rates(k_p_Co59_to_n_Ni59) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Co59_to_n_Ni59);
        rate_eval.dscreened_rates_dT(k_p_Co59_to_n_Ni59) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Co59_to_He4_Fe56);
    rate_eval.screened_rates(k_p_Co59_to_He4_Fe56) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Co59_to_He4_Fe56);
        rate_eval.dscreened_rates_dT(k_p_Co59_to_He4_Fe56) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 27.0_rt, 59.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Co59_to_Cu63);
    rate_eval.screened_rates(k_He4_Co59_to_Cu63) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Co59_to_Cu63);
        rate_eval.dscreened_rates_dT(k_He4_Co59_to_Cu63) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Co59_to_n_Cu62);
    rate_eval.screened_rates(k_He4_Co59_to_n_Cu62) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Co59_to_n_Cu62);
        rate_eval.dscreened_rates_dT(k_He4_Co59_to_n_Cu62) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Co59_to_p_Ni62);
    rate_eval.screened_rates(k_He4_Co59_to_p_Ni62) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Co59_to_p_Ni62);
        rate_eval.dscreened_rates_dT(k_He4_Co59_to_p_Ni62) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 28.0_rt, 56.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ni56_to_Cu57);
    rate_eval.screened_rates(k_p_Ni56_to_Cu57) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni56_to_Cu57);
        rate_eval.dscreened_rates_dT(k_p_Ni56_to_Cu57) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ni56_to_He4_Co53);
    rate_eval.screened_rates(k_p_Ni56_to_He4_Co53) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni56_to_He4_Co53);
        rate_eval.dscreened_rates_dT(k_p_Ni56_to_He4_Co53) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 28.0_rt, 56.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ni56_to_Zn60);
    rate_eval.screened_rates(k_He4_Ni56_to_Zn60) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ni56_to_Zn60);
        rate_eval.dscreened_rates_dT(k_He4_Ni56_to_Zn60) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ni56_to_n_Zn59);
    rate_eval.screened_rates(k_He4_Ni56_to_n_Zn59) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ni56_to_n_Zn59);
        rate_eval.dscreened_rates_dT(k_He4_Ni56_to_n_Zn59) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ni56_to_p_Cu59);
    rate_eval.screened_rates(k_He4_Ni56_to_p_Cu59) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ni56_to_p_Cu59);
        rate_eval.dscreened_rates_dT(k_He4_Ni56_to_p_Cu59) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 28.0_rt, 57.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ni57_to_Cu58);
    rate_eval.screened_rates(k_p_Ni57_to_Cu58) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni57_to_Cu58);
        rate_eval.dscreened_rates_dT(k_p_Ni57_to_Cu58) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ni57_to_n_Cu57);
    rate_eval.screened_rates(k_p_Ni57_to_n_Cu57) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni57_to_n_Cu57);
        rate_eval.dscreened_rates_dT(k_p_Ni57_to_n_Cu57) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ni57_to_He4_Co54);
    rate_eval.screened_rates(k_p_Ni57_to_He4_Co54) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni57_to_He4_Co54);
        rate_eval.dscreened_rates_dT(k_p_Ni57_to_He4_Co54) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 28.0_rt, 57.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ni57_to_Zn61);
    rate_eval.screened_rates(k_He4_Ni57_to_Zn61) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ni57_to_Zn61);
        rate_eval.dscreened_rates_dT(k_He4_Ni57_to_Zn61) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ni57_to_n_Zn60);
    rate_eval.screened_rates(k_He4_Ni57_to_n_Zn60) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ni57_to_n_Zn60);
        rate_eval.dscreened_rates_dT(k_He4_Ni57_to_n_Zn60) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ni57_to_p_Cu60);
    rate_eval.screened_rates(k_He4_Ni57_to_p_Cu60) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ni57_to_p_Cu60);
        rate_eval.dscreened_rates_dT(k_He4_Ni57_to_p_Cu60) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 28.0_rt, 58.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ni58_to_Cu59);
    rate_eval.screened_rates(k_p_Ni58_to_Cu59) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni58_to_Cu59);
        rate_eval.dscreened_rates_dT(k_p_Ni58_to_Cu59) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ni58_to_n_Cu58);
    rate_eval.screened_rates(k_p_Ni58_to_n_Cu58) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni58_to_n_Cu58);
        rate_eval.dscreened_rates_dT(k_p_Ni58_to_n_Cu58) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ni58_to_He4_Co55);
    rate_eval.screened_rates(k_p_Ni58_to_He4_Co55) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni58_to_He4_Co55);
        rate_eval.dscreened_rates_dT(k_p_Ni58_to_He4_Co55) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 28.0_rt, 58.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ni58_to_Zn62);
    rate_eval.screened_rates(k_He4_Ni58_to_Zn62) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ni58_to_Zn62);
        rate_eval.dscreened_rates_dT(k_He4_Ni58_to_Zn62) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ni58_to_n_Zn61);
    rate_eval.screened_rates(k_He4_Ni58_to_n_Zn61) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ni58_to_n_Zn61);
        rate_eval.dscreened_rates_dT(k_He4_Ni58_to_n_Zn61) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ni58_to_p_Cu61);
    rate_eval.screened_rates(k_He4_Ni58_to_p_Cu61) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ni58_to_p_Cu61);
        rate_eval.dscreened_rates_dT(k_He4_Ni58_to_p_Cu61) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 28.0_rt, 59.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ni59_to_Cu60);
    rate_eval.screened_rates(k_p_Ni59_to_Cu60) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni59_to_Cu60);
        rate_eval.dscreened_rates_dT(k_p_Ni59_to_Cu60) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ni59_to_n_Cu59);
    rate_eval.screened_rates(k_p_Ni59_to_n_Cu59) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni59_to_n_Cu59);
        rate_eval.dscreened_rates_dT(k_p_Ni59_to_n_Cu59) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ni59_to_He4_Co56);
    rate_eval.screened_rates(k_p_Ni59_to_He4_Co56) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni59_to_He4_Co56);
        rate_eval.dscreened_rates_dT(k_p_Ni59_to_He4_Co56) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 28.0_rt, 59.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ni59_to_Zn63);
    rate_eval.screened_rates(k_He4_Ni59_to_Zn63) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ni59_to_Zn63);
        rate_eval.dscreened_rates_dT(k_He4_Ni59_to_Zn63) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ni59_to_n_Zn62);
    rate_eval.screened_rates(k_He4_Ni59_to_n_Zn62) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ni59_to_n_Zn62);
        rate_eval.dscreened_rates_dT(k_He4_Ni59_to_n_Zn62) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ni59_to_p_Cu62);
    rate_eval.screened_rates(k_He4_Ni59_to_p_Cu62) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ni59_to_p_Cu62);
        rate_eval.dscreened_rates_dT(k_He4_Ni59_to_p_Cu62) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 28.0_rt, 60.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ni60_to_Cu61);
    rate_eval.screened_rates(k_p_Ni60_to_Cu61) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni60_to_Cu61);
        rate_eval.dscreened_rates_dT(k_p_Ni60_to_Cu61) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ni60_to_n_Cu60);
    rate_eval.screened_rates(k_p_Ni60_to_n_Cu60) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni60_to_n_Cu60);
        rate_eval.dscreened_rates_dT(k_p_Ni60_to_n_Cu60) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ni60_to_He4_Co57);
    rate_eval.screened_rates(k_p_Ni60_to_He4_Co57) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni60_to_He4_Co57);
        rate_eval.dscreened_rates_dT(k_p_Ni60_to_He4_Co57) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 28.0_rt, 60.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ni60_to_Zn64);
    rate_eval.screened_rates(k_He4_Ni60_to_Zn64) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ni60_to_Zn64);
        rate_eval.dscreened_rates_dT(k_He4_Ni60_to_Zn64) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ni60_to_n_Zn63);
    rate_eval.screened_rates(k_He4_Ni60_to_n_Zn63) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ni60_to_n_Zn63);
        rate_eval.dscreened_rates_dT(k_He4_Ni60_to_n_Zn63) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ni60_to_p_Cu63);
    rate_eval.screened_rates(k_He4_Ni60_to_p_Cu63) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ni60_to_p_Cu63);
        rate_eval.dscreened_rates_dT(k_He4_Ni60_to_p_Cu63) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 28.0_rt, 61.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ni61_to_Cu62);
    rate_eval.screened_rates(k_p_Ni61_to_Cu62) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni61_to_Cu62);
        rate_eval.dscreened_rates_dT(k_p_Ni61_to_Cu62) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ni61_to_n_Cu61);
    rate_eval.screened_rates(k_p_Ni61_to_n_Cu61) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni61_to_n_Cu61);
        rate_eval.dscreened_rates_dT(k_p_Ni61_to_n_Cu61) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ni61_to_He4_Co58);
    rate_eval.screened_rates(k_p_Ni61_to_He4_Co58) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni61_to_He4_Co58);
        rate_eval.dscreened_rates_dT(k_p_Ni61_to_He4_Co58) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 28.0_rt, 61.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ni61_to_Zn65);
    rate_eval.screened_rates(k_He4_Ni61_to_Zn65) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ni61_to_Zn65);
        rate_eval.dscreened_rates_dT(k_He4_Ni61_to_Zn65) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ni61_to_n_Zn64);
    rate_eval.screened_rates(k_He4_Ni61_to_n_Zn64) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ni61_to_n_Zn64);
        rate_eval.dscreened_rates_dT(k_He4_Ni61_to_n_Zn64) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ni61_to_p_Cu64);
    rate_eval.screened_rates(k_He4_Ni61_to_p_Cu64) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ni61_to_p_Cu64);
        rate_eval.dscreened_rates_dT(k_He4_Ni61_to_p_Cu64) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 28.0_rt, 62.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ni62_to_Cu63);
    rate_eval.screened_rates(k_p_Ni62_to_Cu63) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni62_to_Cu63);
        rate_eval.dscreened_rates_dT(k_p_Ni62_to_Cu63) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ni62_to_n_Cu62);
    rate_eval.screened_rates(k_p_Ni62_to_n_Cu62) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni62_to_n_Cu62);
        rate_eval.dscreened_rates_dT(k_p_Ni62_to_n_Cu62) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ni62_to_He4_Co59);
    rate_eval.screened_rates(k_p_Ni62_to_He4_Co59) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni62_to_He4_Co59);
        rate_eval.dscreened_rates_dT(k_p_Ni62_to_He4_Co59) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 28.0_rt, 62.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ni62_to_Zn66);
    rate_eval.screened_rates(k_He4_Ni62_to_Zn66) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ni62_to_Zn66);
        rate_eval.dscreened_rates_dT(k_He4_Ni62_to_Zn66) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ni62_to_n_Zn65);
    rate_eval.screened_rates(k_He4_Ni62_to_n_Zn65) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ni62_to_n_Zn65);
        rate_eval.dscreened_rates_dT(k_He4_Ni62_to_n_Zn65) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Ni62_to_p_Cu65);
    rate_eval.screened_rates(k_He4_Ni62_to_p_Cu65) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ni62_to_p_Cu65);
        rate_eval.dscreened_rates_dT(k_He4_Ni62_to_p_Cu65) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 28.0_rt, 63.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ni63_to_Cu64);
    rate_eval.screened_rates(k_p_Ni63_to_Cu64) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni63_to_Cu64);
        rate_eval.dscreened_rates_dT(k_p_Ni63_to_Cu64) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ni63_to_n_Cu63);
    rate_eval.screened_rates(k_p_Ni63_to_n_Cu63) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni63_to_n_Cu63);
        rate_eval.dscreened_rates_dT(k_p_Ni63_to_n_Cu63) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 28.0_rt, 64.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ni64_to_Cu65);
    rate_eval.screened_rates(k_p_Ni64_to_Cu65) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni64_to_Cu65);
        rate_eval.dscreened_rates_dT(k_p_Ni64_to_Cu65) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ni64_to_n_Cu64);
    rate_eval.screened_rates(k_p_Ni64_to_n_Cu64) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ni64_to_n_Cu64);
        rate_eval.dscreened_rates_dT(k_p_Ni64_to_n_Cu64) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 29.0_rt, 58.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Cu58_to_Zn59);
    rate_eval.screened_rates(k_p_Cu58_to_Zn59) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cu58_to_Zn59);
        rate_eval.dscreened_rates_dT(k_p_Cu58_to_Zn59) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 29.0_rt, 58.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Cu58_to_Ga62);
    rate_eval.screened_rates(k_He4_Cu58_to_Ga62) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cu58_to_Ga62);
        rate_eval.dscreened_rates_dT(k_He4_Cu58_to_Ga62) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cu58_to_p_Zn61);
    rate_eval.screened_rates(k_He4_Cu58_to_p_Zn61) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cu58_to_p_Zn61);
        rate_eval.dscreened_rates_dT(k_He4_Cu58_to_p_Zn61) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 29.0_rt, 59.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Cu59_to_Zn60);
    rate_eval.screened_rates(k_p_Cu59_to_Zn60) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cu59_to_Zn60);
        rate_eval.dscreened_rates_dT(k_p_Cu59_to_Zn60) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cu59_to_n_Zn59);
    rate_eval.screened_rates(k_p_Cu59_to_n_Zn59) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cu59_to_n_Zn59);
        rate_eval.dscreened_rates_dT(k_p_Cu59_to_n_Zn59) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cu59_to_He4_Ni56);
    rate_eval.screened_rates(k_p_Cu59_to_He4_Ni56) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cu59_to_He4_Ni56);
        rate_eval.dscreened_rates_dT(k_p_Cu59_to_He4_Ni56) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 29.0_rt, 59.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Cu59_to_Ga63);
    rate_eval.screened_rates(k_He4_Cu59_to_Ga63) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cu59_to_Ga63);
        rate_eval.dscreened_rates_dT(k_He4_Cu59_to_Ga63) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cu59_to_n_Ga62);
    rate_eval.screened_rates(k_He4_Cu59_to_n_Ga62) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cu59_to_n_Ga62);
        rate_eval.dscreened_rates_dT(k_He4_Cu59_to_n_Ga62) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cu59_to_p_Zn62);
    rate_eval.screened_rates(k_He4_Cu59_to_p_Zn62) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cu59_to_p_Zn62);
        rate_eval.dscreened_rates_dT(k_He4_Cu59_to_p_Zn62) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 29.0_rt, 60.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Cu60_to_Zn61);
    rate_eval.screened_rates(k_p_Cu60_to_Zn61) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cu60_to_Zn61);
        rate_eval.dscreened_rates_dT(k_p_Cu60_to_Zn61) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cu60_to_n_Zn60);
    rate_eval.screened_rates(k_p_Cu60_to_n_Zn60) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cu60_to_n_Zn60);
        rate_eval.dscreened_rates_dT(k_p_Cu60_to_n_Zn60) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cu60_to_He4_Ni57);
    rate_eval.screened_rates(k_p_Cu60_to_He4_Ni57) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cu60_to_He4_Ni57);
        rate_eval.dscreened_rates_dT(k_p_Cu60_to_He4_Ni57) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 29.0_rt, 60.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Cu60_to_Ga64);
    rate_eval.screened_rates(k_He4_Cu60_to_Ga64) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cu60_to_Ga64);
        rate_eval.dscreened_rates_dT(k_He4_Cu60_to_Ga64) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cu60_to_n_Ga63);
    rate_eval.screened_rates(k_He4_Cu60_to_n_Ga63) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cu60_to_n_Ga63);
        rate_eval.dscreened_rates_dT(k_He4_Cu60_to_n_Ga63) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cu60_to_p_Zn63);
    rate_eval.screened_rates(k_He4_Cu60_to_p_Zn63) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cu60_to_p_Zn63);
        rate_eval.dscreened_rates_dT(k_He4_Cu60_to_p_Zn63) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 29.0_rt, 61.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Cu61_to_Zn62);
    rate_eval.screened_rates(k_p_Cu61_to_Zn62) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cu61_to_Zn62);
        rate_eval.dscreened_rates_dT(k_p_Cu61_to_Zn62) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cu61_to_n_Zn61);
    rate_eval.screened_rates(k_p_Cu61_to_n_Zn61) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cu61_to_n_Zn61);
        rate_eval.dscreened_rates_dT(k_p_Cu61_to_n_Zn61) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cu61_to_He4_Ni58);
    rate_eval.screened_rates(k_p_Cu61_to_He4_Ni58) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cu61_to_He4_Ni58);
        rate_eval.dscreened_rates_dT(k_p_Cu61_to_He4_Ni58) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 29.0_rt, 62.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Cu62_to_Zn63);
    rate_eval.screened_rates(k_p_Cu62_to_Zn63) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cu62_to_Zn63);
        rate_eval.dscreened_rates_dT(k_p_Cu62_to_Zn63) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cu62_to_n_Zn62);
    rate_eval.screened_rates(k_p_Cu62_to_n_Zn62) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cu62_to_n_Zn62);
        rate_eval.dscreened_rates_dT(k_p_Cu62_to_n_Zn62) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cu62_to_He4_Ni59);
    rate_eval.screened_rates(k_p_Cu62_to_He4_Ni59) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cu62_to_He4_Ni59);
        rate_eval.dscreened_rates_dT(k_p_Cu62_to_He4_Ni59) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 29.0_rt, 63.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Cu63_to_Zn64);
    rate_eval.screened_rates(k_p_Cu63_to_Zn64) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cu63_to_Zn64);
        rate_eval.dscreened_rates_dT(k_p_Cu63_to_Zn64) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cu63_to_n_Zn63);
    rate_eval.screened_rates(k_p_Cu63_to_n_Zn63) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cu63_to_n_Zn63);
        rate_eval.dscreened_rates_dT(k_p_Cu63_to_n_Zn63) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cu63_to_He4_Ni60);
    rate_eval.screened_rates(k_p_Cu63_to_He4_Ni60) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cu63_to_He4_Ni60);
        rate_eval.dscreened_rates_dT(k_p_Cu63_to_He4_Ni60) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 29.0_rt, 64.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Cu64_to_Zn65);
    rate_eval.screened_rates(k_p_Cu64_to_Zn65) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cu64_to_Zn65);
        rate_eval.dscreened_rates_dT(k_p_Cu64_to_Zn65) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cu64_to_n_Zn64);
    rate_eval.screened_rates(k_p_Cu64_to_n_Zn64) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cu64_to_n_Zn64);
        rate_eval.dscreened_rates_dT(k_p_Cu64_to_n_Zn64) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cu64_to_He4_Ni61);
    rate_eval.screened_rates(k_p_Cu64_to_He4_Ni61) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cu64_to_He4_Ni61);
        rate_eval.dscreened_rates_dT(k_p_Cu64_to_He4_Ni61) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 29.0_rt, 65.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Cu65_to_Zn66);
    rate_eval.screened_rates(k_p_Cu65_to_Zn66) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cu65_to_Zn66);
        rate_eval.dscreened_rates_dT(k_p_Cu65_to_Zn66) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cu65_to_n_Zn65);
    rate_eval.screened_rates(k_p_Cu65_to_n_Zn65) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cu65_to_n_Zn65);
        rate_eval.dscreened_rates_dT(k_p_Cu65_to_n_Zn65) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Cu65_to_He4_Ni62);
    rate_eval.screened_rates(k_p_Cu65_to_He4_Ni62) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Cu65_to_He4_Ni62);
        rate_eval.dscreened_rates_dT(k_p_Cu65_to_He4_Ni62) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 30.0_rt, 59.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Zn59_to_Ge63);
    rate_eval.screened_rates(k_He4_Zn59_to_Ge63) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Zn59_to_Ge63);
        rate_eval.dscreened_rates_dT(k_He4_Zn59_to_Ge63) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Zn59_to_p_Ga62);
    rate_eval.screened_rates(k_He4_Zn59_to_p_Ga62) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Zn59_to_p_Ga62);
        rate_eval.dscreened_rates_dT(k_He4_Zn59_to_p_Ga62) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 30.0_rt, 60.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Zn60_to_Ge64);
    rate_eval.screened_rates(k_He4_Zn60_to_Ge64) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Zn60_to_Ge64);
        rate_eval.dscreened_rates_dT(k_He4_Zn60_to_Ge64) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Zn60_to_n_Ge63);
    rate_eval.screened_rates(k_He4_Zn60_to_n_Ge63) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Zn60_to_n_Ge63);
        rate_eval.dscreened_rates_dT(k_He4_Zn60_to_n_Ge63) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Zn60_to_p_Ga63);
    rate_eval.screened_rates(k_He4_Zn60_to_p_Ga63) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Zn60_to_p_Ga63);
        rate_eval.dscreened_rates_dT(k_He4_Zn60_to_p_Ga63) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 30.0_rt, 61.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Zn61_to_Ga62);
    rate_eval.screened_rates(k_p_Zn61_to_Ga62) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Zn61_to_Ga62);
        rate_eval.dscreened_rates_dT(k_p_Zn61_to_Ga62) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Zn61_to_He4_Cu58);
    rate_eval.screened_rates(k_p_Zn61_to_He4_Cu58) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Zn61_to_He4_Cu58);
        rate_eval.dscreened_rates_dT(k_p_Zn61_to_He4_Cu58) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 30.0_rt, 62.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Zn62_to_Ga63);
    rate_eval.screened_rates(k_p_Zn62_to_Ga63) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Zn62_to_Ga63);
        rate_eval.dscreened_rates_dT(k_p_Zn62_to_Ga63) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Zn62_to_n_Ga62);
    rate_eval.screened_rates(k_p_Zn62_to_n_Ga62) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Zn62_to_n_Ga62);
        rate_eval.dscreened_rates_dT(k_p_Zn62_to_n_Ga62) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Zn62_to_He4_Cu59);
    rate_eval.screened_rates(k_p_Zn62_to_He4_Cu59) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Zn62_to_He4_Cu59);
        rate_eval.dscreened_rates_dT(k_p_Zn62_to_He4_Cu59) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 30.0_rt, 63.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Zn63_to_Ga64);
    rate_eval.screened_rates(k_p_Zn63_to_Ga64) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Zn63_to_Ga64);
        rate_eval.dscreened_rates_dT(k_p_Zn63_to_Ga64) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Zn63_to_n_Ga63);
    rate_eval.screened_rates(k_p_Zn63_to_n_Ga63) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Zn63_to_n_Ga63);
        rate_eval.dscreened_rates_dT(k_p_Zn63_to_n_Ga63) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Zn63_to_He4_Cu60);
    rate_eval.screened_rates(k_p_Zn63_to_He4_Cu60) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Zn63_to_He4_Cu60);
        rate_eval.dscreened_rates_dT(k_p_Zn63_to_He4_Cu60) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 31.0_rt, 62.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ga62_to_Ge63);
    rate_eval.screened_rates(k_p_Ga62_to_Ge63) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ga62_to_Ge63);
        rate_eval.dscreened_rates_dT(k_p_Ga62_to_Ge63) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ga62_to_He4_Zn59);
    rate_eval.screened_rates(k_p_Ga62_to_He4_Zn59) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ga62_to_He4_Zn59);
        rate_eval.dscreened_rates_dT(k_p_Ga62_to_He4_Zn59) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 31.0_rt, 63.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ga63_to_Ge64);
    rate_eval.screened_rates(k_p_Ga63_to_Ge64) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ga63_to_Ge64);
        rate_eval.dscreened_rates_dT(k_p_Ga63_to_Ge64) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ga63_to_n_Ge63);
    rate_eval.screened_rates(k_p_Ga63_to_n_Ge63) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ga63_to_n_Ge63);
        rate_eval.dscreened_rates_dT(k_p_Ga63_to_n_Ge63) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ga63_to_He4_Zn60);
    rate_eval.screened_rates(k_p_Ga63_to_He4_Zn60) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ga63_to_He4_Zn60);
        rate_eval.dscreened_rates_dT(k_p_Ga63_to_He4_Zn60) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 2.0_rt, 2.0_rt, 3.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_d_He3_to_p_He4);
    rate_eval.screened_rates(k_d_He3_to_p_He4) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_d_He3_to_p_He4);
        rate_eval.dscreened_rates_dT(k_d_He3_to_p_He4) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 2.0_rt, 4.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_He4_to_d_He3);
    rate_eval.screened_rates(k_p_He4_to_d_He3) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_He4_to_d_He3);
        rate_eval.dscreened_rates_dT(k_p_He4_to_d_He3) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_n_p_He4_to_Li6);
    rate_eval.screened_rates(k_n_p_He4_to_Li6) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_n_p_He4_to_Li6);
        rate_eval.dscreened_rates_dT(k_n_p_He4_to_Li6) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 2.0_rt, 4.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_He4_to_n_Be7);
    rate_eval.screened_rates(k_He4_He4_to_n_Be7) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_He4_to_n_Be7);
        rate_eval.dscreened_rates_dT(k_He4_He4_to_n_Be7) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_He4_to_p_Li7);
    rate_eval.screened_rates(k_He4_He4_to_p_Li7) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_He4_to_p_Li7);
        rate_eval.dscreened_rates_dT(k_He4_He4_to_p_Li7) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_n_He4_He4_to_Be9);
    rate_eval.screened_rates(k_n_He4_He4_to_Be9) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_n_He4_He4_to_Be9);
        rate_eval.dscreened_rates_dT(k_n_He4_He4_to_Be9) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_n_He4_He4_to_d_Li7);
    rate_eval.screened_rates(k_n_He4_He4_to_d_Li7) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_n_He4_He4_to_d_Li7);
        rate_eval.dscreened_rates_dT(k_n_He4_He4_to_d_Li7) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 2.0_rt, 3.0_rt, 6.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_d_Li6_to_n_Be7);
    rate_eval.screened_rates(k_d_Li6_to_n_Be7) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_d_Li6_to_n_Be7);
        rate_eval.dscreened_rates_dT(k_d_Li6_to_n_Be7) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_d_Li6_to_p_Li7);
    rate_eval.screened_rates(k_d_Li6_to_p_Li7) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_d_Li6_to_p_Li7);
        rate_eval.dscreened_rates_dT(k_d_Li6_to_p_Li7) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 3.0_rt, 7.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Li7_to_n_Be7);
    rate_eval.screened_rates(k_p_Li7_to_n_Be7) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Li7_to_n_Be7);
        rate_eval.dscreened_rates_dT(k_p_Li7_to_n_Be7) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Li7_to_d_Li6);
    rate_eval.screened_rates(k_p_Li7_to_d_Li6) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Li7_to_d_Li6);
        rate_eval.dscreened_rates_dT(k_p_Li7_to_d_Li6) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Li7_to_He4_He4);
    rate_eval.screened_rates(k_p_Li7_to_He4_He4) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Li7_to_He4_He4);
        rate_eval.dscreened_rates_dT(k_p_Li7_to_He4_He4) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 4.0_rt, 7.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Be7_to_p_B10);
    rate_eval.screened_rates(k_He4_Be7_to_p_B10) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Be7_to_p_B10);
        rate_eval.dscreened_rates_dT(k_He4_Be7_to_p_B10) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 4.0_rt, 9.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Be9_to_n_C12);
    rate_eval.screened_rates(k_He4_Be9_to_n_C12) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Be9_to_n_C12);
        rate_eval.dscreened_rates_dT(k_He4_Be9_to_n_C12) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 5.0_rt, 10.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_B10_to_He4_Be7);
    rate_eval.screened_rates(k_p_B10_to_He4_Be7) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_B10_to_He4_Be7);
        rate_eval.dscreened_rates_dT(k_p_B10_to_He4_Be7) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 5.0_rt, 10.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_B10_to_n_N13);
    rate_eval.screened_rates(k_He4_B10_to_n_N13) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_B10_to_n_N13);
        rate_eval.dscreened_rates_dT(k_He4_B10_to_n_N13) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_B10_to_p_C13);
    rate_eval.screened_rates(k_He4_B10_to_p_C13) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_B10_to_p_C13);
        rate_eval.dscreened_rates_dT(k_He4_B10_to_p_C13) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 5.0_rt, 11.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_B11_to_n_N14);
    rate_eval.screened_rates(k_He4_B11_to_n_N14) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_B11_to_n_N14);
        rate_eval.dscreened_rates_dT(k_He4_B11_to_n_N14) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_B11_to_p_C14);
    rate_eval.screened_rates(k_He4_B11_to_p_C14) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_B11_to_p_C14);
        rate_eval.dscreened_rates_dT(k_He4_B11_to_p_C14) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(6.0_rt, 12.0_rt, 6.0_rt, 12.0_rt);


        static_assert(scn_fac.z1 == 6.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_C12_C12_to_n_Mg23);
    rate_eval.screened_rates(k_C12_C12_to_n_Mg23) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_C12_C12_to_n_Mg23);
        rate_eval.dscreened_rates_dT(k_C12_C12_to_n_Mg23) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_C12_C12_to_p_Na23);
    rate_eval.screened_rates(k_C12_C12_to_p_Na23) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_C12_C12_to_p_Na23);
        rate_eval.dscreened_rates_dT(k_C12_C12_to_p_Na23) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_C12_C12_to_He4_Ne20);
    rate_eval.screened_rates(k_C12_C12_to_He4_Ne20) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_C12_C12_to_He4_Ne20);
        rate_eval.dscreened_rates_dT(k_C12_C12_to_He4_Ne20) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 2.0_rt, 6.0_rt, 13.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_d_C13_to_n_N14);
    rate_eval.screened_rates(k_d_C13_to_n_N14) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_d_C13_to_n_N14);
        rate_eval.dscreened_rates_dT(k_d_C13_to_n_N14) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 6.0_rt, 13.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_C13_to_n_O16);
    rate_eval.screened_rates(k_He4_C13_to_n_O16) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_C13_to_n_O16);
        rate_eval.dscreened_rates_dT(k_He4_C13_to_n_O16) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 2.0_rt, 6.0_rt, 14.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_d_C14_to_n_N15);
    rate_eval.screened_rates(k_d_C14_to_n_N15) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_d_C14_to_n_N15);
        rate_eval.dscreened_rates_dT(k_d_C14_to_n_N15) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 7.0_rt, 13.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_N13_to_p_O16);
    rate_eval.screened_rates(k_He4_N13_to_p_O16) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_N13_to_p_O16);
        rate_eval.dscreened_rates_dT(k_He4_N13_to_p_O16) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(6.0_rt, 12.0_rt, 8.0_rt, 16.0_rt);


        static_assert(scn_fac.z1 == 6.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_C12_O16_to_p_Al27);
    rate_eval.screened_rates(k_C12_O16_to_p_Al27) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_C12_O16_to_p_Al27);
        rate_eval.dscreened_rates_dT(k_C12_O16_to_p_Al27) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_C12_O16_to_He4_Mg24);
    rate_eval.screened_rates(k_C12_O16_to_He4_Mg24) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_C12_O16_to_He4_Mg24);
        rate_eval.dscreened_rates_dT(k_C12_O16_to_He4_Mg24) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(8.0_rt, 16.0_rt, 8.0_rt, 16.0_rt);


        static_assert(scn_fac.z1 == 8.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_O16_O16_to_p_P31);
    rate_eval.screened_rates(k_O16_O16_to_p_P31) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_O16_O16_to_p_P31);
        rate_eval.dscreened_rates_dT(k_O16_O16_to_p_P31) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_O16_O16_to_He4_Si28);
    rate_eval.screened_rates(k_O16_O16_to_He4_Si28) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_O16_O16_to_He4_Si28);
        rate_eval.dscreened_rates_dT(k_O16_O16_to_He4_Si28) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 10.0_rt, 18.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ne18_to_p_Na21);
    rate_eval.screened_rates(k_He4_Ne18_to_p_Na21) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ne18_to_p_Na21);
        rate_eval.dscreened_rates_dT(k_He4_Ne18_to_p_Na21) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(6.0_rt, 12.0_rt, 10.0_rt, 20.0_rt);


        static_assert(scn_fac.z1 == 6.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_C12_Ne20_to_p_P31);
    rate_eval.screened_rates(k_C12_Ne20_to_p_P31) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_C12_Ne20_to_p_P31);
        rate_eval.dscreened_rates_dT(k_C12_Ne20_to_p_P31) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_C12_Ne20_to_He4_Si28);
    rate_eval.screened_rates(k_C12_Ne20_to_He4_Si28) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_C12_Ne20_to_He4_Si28);
        rate_eval.dscreened_rates_dT(k_C12_Ne20_to_He4_Si28) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 11.0_rt, 21.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Na21_to_He4_Ne18);
    rate_eval.screened_rates(k_p_Na21_to_He4_Ne18) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Na21_to_He4_Ne18);
        rate_eval.dscreened_rates_dT(k_p_Na21_to_He4_Ne18) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 12.0_rt, 23.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Mg23_to_p_Al26);
    rate_eval.screened_rates(k_He4_Mg23_to_p_Al26) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Mg23_to_p_Al26);
        rate_eval.dscreened_rates_dT(k_He4_Mg23_to_p_Al26) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 13.0_rt, 26.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Al26_to_He4_Mg23);
    rate_eval.screened_rates(k_p_Al26_to_He4_Mg23) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Al26_to_He4_Mg23);
        rate_eval.dscreened_rates_dT(k_p_Al26_to_He4_Mg23) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 19.0_rt, 37.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_K37_to_p_Ca40);
    rate_eval.screened_rates(k_He4_K37_to_p_Ca40) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_K37_to_p_Ca40);
        rate_eval.dscreened_rates_dT(k_He4_K37_to_p_Ca40) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 19.0_rt, 38.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_K38_to_p_Ca41);
    rate_eval.screened_rates(k_He4_K38_to_p_Ca41) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_K38_to_p_Ca41);
        rate_eval.dscreened_rates_dT(k_He4_K38_to_p_Ca41) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 20.0_rt, 40.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ca40_to_He4_K37);
    rate_eval.screened_rates(k_p_Ca40_to_He4_K37) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ca40_to_He4_K37);
        rate_eval.dscreened_rates_dT(k_p_Ca40_to_He4_K37) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 20.0_rt, 41.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ca41_to_He4_K38);
    rate_eval.screened_rates(k_p_Ca41_to_He4_K38) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ca41_to_He4_K38);
        rate_eval.dscreened_rates_dT(k_p_Ca41_to_He4_K38) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 20.0_rt, 48.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ca48_to_n_Ti51);
    rate_eval.screened_rates(k_He4_Ca48_to_n_Ti51) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ca48_to_n_Ti51);
        rate_eval.dscreened_rates_dT(k_He4_Ca48_to_n_Ti51) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 21.0_rt, 49.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Sc49_to_n_V52);
    rate_eval.screened_rates(k_He4_Sc49_to_n_V52) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Sc49_to_n_V52);
        rate_eval.dscreened_rates_dT(k_He4_Sc49_to_n_V52) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 22.0_rt, 51.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ti51_to_n_Cr54);
    rate_eval.screened_rates(k_He4_Ti51_to_n_Cr54) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ti51_to_n_Cr54);
        rate_eval.dscreened_rates_dT(k_He4_Ti51_to_n_Cr54) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 23.0_rt, 52.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_V52_to_n_Mn55);
    rate_eval.screened_rates(k_He4_V52_to_n_Mn55) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_V52_to_n_Mn55);
        rate_eval.dscreened_rates_dT(k_He4_V52_to_n_Mn55) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 28.0_rt, 63.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Ni63_to_n_Zn66);
    rate_eval.screened_rates(k_He4_Ni63_to_n_Zn66) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Ni63_to_n_Zn66);
        rate_eval.dscreened_rates_dT(k_He4_Ni63_to_n_Zn66) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 29.0_rt, 57.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Cu57_to_p_Zn60);
    rate_eval.screened_rates(k_He4_Cu57_to_p_Zn60) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cu57_to_p_Zn60);
        rate_eval.dscreened_rates_dT(k_He4_Cu57_to_p_Zn60) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 29.0_rt, 61.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Cu61_to_n_Ga64);
    rate_eval.screened_rates(k_He4_Cu61_to_n_Ga64) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cu61_to_n_Ga64);
        rate_eval.dscreened_rates_dT(k_He4_Cu61_to_n_Ga64) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Cu61_to_p_Zn64);
    rate_eval.screened_rates(k_He4_Cu61_to_p_Zn64) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cu61_to_p_Zn64);
        rate_eval.dscreened_rates_dT(k_He4_Cu61_to_p_Zn64) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 29.0_rt, 62.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Cu62_to_p_Zn65);
    rate_eval.screened_rates(k_He4_Cu62_to_p_Zn65) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cu62_to_p_Zn65);
        rate_eval.dscreened_rates_dT(k_He4_Cu62_to_p_Zn65) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 29.0_rt, 63.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Cu63_to_p_Zn66);
    rate_eval.screened_rates(k_He4_Cu63_to_p_Zn66) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Cu63_to_p_Zn66);
        rate_eval.dscreened_rates_dT(k_He4_Cu63_to_p_Zn66) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 30.0_rt, 60.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Zn60_to_He4_Cu57);
    rate_eval.screened_rates(k_p_Zn60_to_He4_Cu57) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Zn60_to_He4_Cu57);
        rate_eval.dscreened_rates_dT(k_p_Zn60_to_He4_Cu57) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 30.0_rt, 61.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He4_Zn61_to_n_Ge64);
    rate_eval.screened_rates(k_He4_Zn61_to_n_Ge64) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Zn61_to_n_Ge64);
        rate_eval.dscreened_rates_dT(k_He4_Zn61_to_n_Ge64) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_He4_Zn61_to_p_Ga64);
    rate_eval.screened_rates(k_He4_Zn61_to_p_Ga64) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_Zn61_to_p_Ga64);
        rate_eval.dscreened_rates_dT(k_He4_Zn61_to_p_Ga64) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 30.0_rt, 64.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Zn64_to_n_Ga64);
    rate_eval.screened_rates(k_p_Zn64_to_n_Ga64) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Zn64_to_n_Ga64);
        rate_eval.dscreened_rates_dT(k_p_Zn64_to_n_Ga64) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Zn64_to_He4_Cu61);
    rate_eval.screened_rates(k_p_Zn64_to_He4_Cu61) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Zn64_to_He4_Cu61);
        rate_eval.dscreened_rates_dT(k_p_Zn64_to_He4_Cu61) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 30.0_rt, 65.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Zn65_to_He4_Cu62);
    rate_eval.screened_rates(k_p_Zn65_to_He4_Cu62) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Zn65_to_He4_Cu62);
        rate_eval.dscreened_rates_dT(k_p_Zn65_to_He4_Cu62) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 30.0_rt, 66.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Zn66_to_He4_Cu63);
    rate_eval.screened_rates(k_p_Zn66_to_He4_Cu63) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Zn66_to_He4_Cu63);
        rate_eval.dscreened_rates_dT(k_p_Zn66_to_He4_Cu63) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 31.0_rt, 64.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_Ga64_to_n_Ge64);
    rate_eval.screened_rates(k_p_Ga64_to_n_Ge64) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ga64_to_n_Ge64);
        rate_eval.dscreened_rates_dT(k_p_Ga64_to_n_Ge64) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_Ga64_to_He4_Zn61);
    rate_eval.screened_rates(k_p_Ga64_to_He4_Zn61) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_Ga64_to_He4_Zn61);
        rate_eval.dscreened_rates_dT(k_p_Ga64_to_He4_Zn61) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 3.0_rt, 2.0_rt, 3.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He3_He3_to_p_p_He4);
    rate_eval.screened_rates(k_He3_He3_to_p_p_He4) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He3_He3_to_p_p_He4);
        rate_eval.dscreened_rates_dT(k_He3_He3_to_p_p_He4) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 2.0_rt, 3.0_rt, 7.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_d_Li7_to_n_He4_He4);
    rate_eval.screened_rates(k_d_Li7_to_n_He4_He4) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_d_Li7_to_n_He4_He4);
        rate_eval.dscreened_rates_dT(k_d_Li7_to_n_He4_He4) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 2.0_rt, 4.0_rt, 7.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_d_Be7_to_p_He4_He4);
    rate_eval.screened_rates(k_d_Be7_to_p_He4_He4) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_d_Be7_to_p_He4_He4);
        rate_eval.dscreened_rates_dT(k_d_Be7_to_p_He4_He4) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 3.0_rt, 3.0_rt, 7.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He3_Li7_to_n_p_He4_He4);
    rate_eval.screened_rates(k_He3_Li7_to_n_p_He4_He4) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He3_Li7_to_n_p_He4_He4);
        rate_eval.dscreened_rates_dT(k_He3_Li7_to_n_p_He4_He4) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 3.0_rt, 4.0_rt, 7.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_He3_Be7_to_p_p_He4_He4);
    rate_eval.screened_rates(k_He3_Be7_to_p_p_He4_He4) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He3_Be7_to_p_p_He4_He4);
        rate_eval.dscreened_rates_dT(k_He3_Be7_to_p_p_He4_He4) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 2.0_rt, 4.0_rt);


        static_assert(scn_fac.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }



    {
        constexpr auto scn_fac2 = scrn::calculate_screen_factor(2.0_rt, 4.0_rt, 4.0_rt, 8.0_rt);


        static_assert(scn_fac2.z1 == 2.0_rt);


        actual_screen(pstate, scn_fac2, scor2, dscor2_dt);

    }


    ratraw = rate_eval.screened_rates(k_He4_He4_He4_to_C12);
    rate_eval.screened_rates(k_He4_He4_He4_to_C12) *= scor * scor2;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_He4_He4_to_C12);
        rate_eval.dscreened_rates_dT(k_He4_He4_He4_to_C12) = ratraw * (scor * dscor2_dt + dscor_dt * scor2) + dratraw_dT * scor * scor2;
    }

    ratraw = rate_eval.screened_rates(k_He4_He4_He4_to_p_B11);
    rate_eval.screened_rates(k_He4_He4_He4_to_p_B11) *= scor * scor2;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_He4_He4_He4_to_p_B11);
        rate_eval.dscreened_rates_dT(k_He4_He4_He4_to_p_B11) = ratraw * (scor * dscor2_dt + dscor_dt * scor2) + dratraw_dT * scor * scor2;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 1.0_rt, 1.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_p_He4_to_He3_He3);
    rate_eval.screened_rates(k_p_p_He4_to_He3_He3) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_p_He4_to_He3_He3);
        rate_eval.dscreened_rates_dT(k_p_p_He4_to_He3_He3) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 1.0_rt, 2.0_rt, 4.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_p_He4_He4_to_n_B8);
    rate_eval.screened_rates(k_p_He4_He4_to_n_B8) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_He4_He4_to_n_B8);
        rate_eval.dscreened_rates_dT(k_p_He4_He4_to_n_B8) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_p_He4_He4_to_d_Be7);
    rate_eval.screened_rates(k_p_He4_He4_to_d_Be7) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_p_He4_He4_to_d_Be7);
        rate_eval.dscreened_rates_dT(k_p_He4_He4_to_d_Be7) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_n_p_He4_He4_to_He3_Li7);
    rate_eval.screened_rates(k_n_p_He4_He4_to_He3_Li7) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_n_p_He4_He4_to_He3_Li7);
        rate_eval.dscreened_rates_dT(k_n_p_He4_He4_to_He3_Li7) = ratraw * dscor_dt + dratraw_dT * scor;
    }

    ratraw = rate_eval.screened_rates(k_n_p_He4_He4_to_p_Be9);
    rate_eval.screened_rates(k_n_p_He4_He4_to_p_Be9) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_n_p_He4_He4_to_p_Be9);
        rate_eval.dscreened_rates_dT(k_n_p_He4_He4_to_p_Be9) = ratraw * dscor_dt + dratraw_dT * scor;
    }


    {
        constexpr auto scn_fac = scrn::calculate_screen_factor(1.0_rt, 2.0_rt, 2.0_rt, 4.0_rt);


        static_assert(scn_fac.z1 == 1.0_rt);


        actual_screen(pstate, scn_fac, scor, dscor_dt);
    }


    ratraw = rate_eval.screened_rates(k_d_He4_He4_to_p_Be9);
    rate_eval.screened_rates(k_d_He4_He4_to_p_Be9) *= scor;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        dratraw_dT = rate_eval.dscreened_rates_dT(k_d_He4_He4_to_p_Be9);
        rate_eval.dscreened_rates_dT(k_d_He4_He4_to_p_Be9) = ratraw * dscor_dt + dratraw_dT * scor;
    }

#endif

    // Fill approximate rates

    fill_approx_rates<do_T_derivatives, T>(tfactors, state.rho, Y, rate_eval);

    // Calculate tabular rates

    [[maybe_unused]] amrex::Real rate, drate_dt, edot_nu, edot_gamma;

    rate_eval.enuc_weak = 0.0_rt;

    tabular_evaluate(j_Na21_Ne21_meta, j_Na21_Ne21_rhoy, j_Na21_Ne21_temp, j_Na21_Ne21_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Na21_to_Ne21) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Na21_to_Ne21) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Na21) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ne21_Na21_meta, j_Ne21_Na21_rhoy, j_Ne21_Na21_temp, j_Ne21_Na21_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ne21_to_Na21) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ne21_to_Na21) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ne21) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Na22_Ne22_meta, j_Na22_Ne22_rhoy, j_Na22_Ne22_temp, j_Na22_Ne22_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Na22_to_Ne22) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Na22_to_Ne22) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Na22) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ne22_Na22_meta, j_Ne22_Na22_rhoy, j_Ne22_Na22_temp, j_Ne22_Na22_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ne22_to_Na22) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ne22_to_Na22) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ne22) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mg23_Na23_meta, j_Mg23_Na23_rhoy, j_Mg23_Na23_temp, j_Mg23_Na23_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mg23_to_Na23) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Mg23_to_Na23) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Mg23) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Na23_Mg23_meta, j_Na23_Mg23_rhoy, j_Na23_Mg23_temp, j_Na23_Mg23_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Na23_to_Mg23) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Na23_to_Mg23) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Na23) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Al25_Mg25_meta, j_Al25_Mg25_rhoy, j_Al25_Mg25_temp, j_Al25_Mg25_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Al25_to_Mg25) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Al25_to_Mg25) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Al25) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mg25_Al25_meta, j_Mg25_Al25_rhoy, j_Mg25_Al25_temp, j_Mg25_Al25_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mg25_to_Al25) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Mg25_to_Al25) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Mg25) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Al26_Mg26_meta, j_Al26_Mg26_rhoy, j_Al26_Mg26_temp, j_Al26_Mg26_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Al26_to_Mg26) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Al26_to_Mg26) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Al26) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mg26_Al26_meta, j_Mg26_Al26_rhoy, j_Mg26_Al26_temp, j_Mg26_Al26_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mg26_to_Al26) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Mg26_to_Al26) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Mg26) * (edot_nu + edot_gamma);

    tabular_evaluate(j_P29_Si29_meta, j_P29_Si29_rhoy, j_P29_Si29_temp, j_P29_Si29_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_P29_to_Si29) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_P29_to_Si29) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(P29) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Si29_P29_meta, j_Si29_P29_rhoy, j_Si29_P29_temp, j_Si29_P29_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Si29_to_P29) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Si29_to_P29) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Si29) * (edot_nu + edot_gamma);

    tabular_evaluate(j_P30_Si30_meta, j_P30_Si30_rhoy, j_P30_Si30_temp, j_P30_Si30_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_P30_to_Si30) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_P30_to_Si30) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(P30) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Si30_P30_meta, j_Si30_P30_rhoy, j_Si30_P30_temp, j_Si30_P30_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Si30_to_P30) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Si30_to_P30) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Si30) * (edot_nu + edot_gamma);

    tabular_evaluate(j_P31_Si31_meta, j_P31_Si31_rhoy, j_P31_Si31_temp, j_P31_Si31_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_P31_to_Si31) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_P31_to_Si31) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(P31) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Si31_P31_meta, j_Si31_P31_rhoy, j_Si31_P31_temp, j_Si31_P31_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Si31_to_P31) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Si31_to_P31) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Si31) * (edot_nu + edot_gamma);

    tabular_evaluate(j_P32_S32_meta, j_P32_S32_rhoy, j_P32_S32_temp, j_P32_S32_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_P32_to_S32) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_P32_to_S32) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(P32) * (edot_nu + edot_gamma);

    tabular_evaluate(j_P32_Si32_meta, j_P32_Si32_rhoy, j_P32_Si32_temp, j_P32_Si32_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_P32_to_Si32) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_P32_to_Si32) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(P32) * (edot_nu + edot_gamma);

    tabular_evaluate(j_S32_P32_meta, j_S32_P32_rhoy, j_S32_P32_temp, j_S32_P32_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_S32_to_P32) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_S32_to_P32) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(S32) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Si32_P32_meta, j_Si32_P32_rhoy, j_Si32_P32_temp, j_Si32_P32_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Si32_to_P32) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Si32_to_P32) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Si32) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cl33_S33_meta, j_Cl33_S33_rhoy, j_Cl33_S33_temp, j_Cl33_S33_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cl33_to_S33) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cl33_to_S33) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cl33) * (edot_nu + edot_gamma);

    tabular_evaluate(j_P33_S33_meta, j_P33_S33_rhoy, j_P33_S33_temp, j_P33_S33_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_P33_to_S33) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_P33_to_S33) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(P33) * (edot_nu + edot_gamma);

    tabular_evaluate(j_S33_Cl33_meta, j_S33_Cl33_rhoy, j_S33_Cl33_temp, j_S33_Cl33_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_S33_to_Cl33) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_S33_to_Cl33) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(S33) * (edot_nu + edot_gamma);

    tabular_evaluate(j_S33_P33_meta, j_S33_P33_rhoy, j_S33_P33_temp, j_S33_P33_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_S33_to_P33) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_S33_to_P33) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(S33) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cl34_S34_meta, j_Cl34_S34_rhoy, j_Cl34_S34_temp, j_Cl34_S34_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cl34_to_S34) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cl34_to_S34) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cl34) * (edot_nu + edot_gamma);

    tabular_evaluate(j_S34_Cl34_meta, j_S34_Cl34_rhoy, j_S34_Cl34_temp, j_S34_Cl34_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_S34_to_Cl34) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_S34_to_Cl34) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(S34) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cl35_S35_meta, j_Cl35_S35_rhoy, j_Cl35_S35_temp, j_Cl35_S35_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cl35_to_S35) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cl35_to_S35) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cl35) * (edot_nu + edot_gamma);

    tabular_evaluate(j_S35_Cl35_meta, j_S35_Cl35_rhoy, j_S35_Cl35_temp, j_S35_Cl35_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_S35_to_Cl35) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_S35_to_Cl35) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(S35) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ar36_Cl36_meta, j_Ar36_Cl36_rhoy, j_Ar36_Cl36_temp, j_Ar36_Cl36_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ar36_to_Cl36) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ar36_to_Cl36) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ar36) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cl36_Ar36_meta, j_Cl36_Ar36_rhoy, j_Cl36_Ar36_temp, j_Cl36_Ar36_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cl36_to_Ar36) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cl36_to_Ar36) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cl36) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cl36_S36_meta, j_Cl36_S36_rhoy, j_Cl36_S36_temp, j_Cl36_S36_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cl36_to_S36) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cl36_to_S36) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cl36) * (edot_nu + edot_gamma);

    tabular_evaluate(j_S36_Cl36_meta, j_S36_Cl36_rhoy, j_S36_Cl36_temp, j_S36_Cl36_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_S36_to_Cl36) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_S36_to_Cl36) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(S36) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ar37_Cl37_meta, j_Ar37_Cl37_rhoy, j_Ar37_Cl37_temp, j_Ar37_Cl37_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ar37_to_Cl37) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ar37_to_Cl37) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ar37) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ar37_K37_meta, j_Ar37_K37_rhoy, j_Ar37_K37_temp, j_Ar37_K37_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ar37_to_K37) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ar37_to_K37) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ar37) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cl37_Ar37_meta, j_Cl37_Ar37_rhoy, j_Cl37_Ar37_temp, j_Cl37_Ar37_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cl37_to_Ar37) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cl37_to_Ar37) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cl37) * (edot_nu + edot_gamma);

    tabular_evaluate(j_K37_Ar37_meta, j_K37_Ar37_rhoy, j_K37_Ar37_temp, j_K37_Ar37_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_K37_to_Ar37) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_K37_to_Ar37) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(K37) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ar38_K38_meta, j_Ar38_K38_rhoy, j_Ar38_K38_temp, j_Ar38_K38_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ar38_to_K38) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ar38_to_K38) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ar38) * (edot_nu + edot_gamma);

    tabular_evaluate(j_K38_Ar38_meta, j_K38_Ar38_rhoy, j_K38_Ar38_temp, j_K38_Ar38_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_K38_to_Ar38) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_K38_to_Ar38) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(K38) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ar39_K39_meta, j_Ar39_K39_rhoy, j_Ar39_K39_temp, j_Ar39_K39_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ar39_to_K39) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ar39_to_K39) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ar39) * (edot_nu + edot_gamma);

    tabular_evaluate(j_K39_Ar39_meta, j_K39_Ar39_rhoy, j_K39_Ar39_temp, j_K39_Ar39_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_K39_to_Ar39) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_K39_to_Ar39) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(K39) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ar40_K40_meta, j_Ar40_K40_rhoy, j_Ar40_K40_temp, j_Ar40_K40_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ar40_to_K40) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ar40_to_K40) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ar40) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ca40_K40_meta, j_Ca40_K40_rhoy, j_Ca40_K40_temp, j_Ca40_K40_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ca40_to_K40) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ca40_to_K40) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ca40) * (edot_nu + edot_gamma);

    tabular_evaluate(j_K40_Ar40_meta, j_K40_Ar40_rhoy, j_K40_Ar40_temp, j_K40_Ar40_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_K40_to_Ar40) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_K40_to_Ar40) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(K40) * (edot_nu + edot_gamma);

    tabular_evaluate(j_K40_Ca40_meta, j_K40_Ca40_rhoy, j_K40_Ca40_temp, j_K40_Ca40_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_K40_to_Ca40) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_K40_to_Ca40) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(K40) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ca41_K41_meta, j_Ca41_K41_rhoy, j_Ca41_K41_temp, j_Ca41_K41_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ca41_to_K41) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ca41_to_K41) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ca41) * (edot_nu + edot_gamma);

    tabular_evaluate(j_K41_Ca41_meta, j_K41_Ca41_rhoy, j_K41_Ca41_temp, j_K41_Ca41_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_K41_to_Ca41) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_K41_to_Ca41) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(K41) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ca43_Sc43_meta, j_Ca43_Sc43_rhoy, j_Ca43_Sc43_temp, j_Ca43_Sc43_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ca43_to_Sc43) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ca43_to_Sc43) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ca43) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc43_Ca43_meta, j_Sc43_Ca43_rhoy, j_Sc43_Ca43_temp, j_Sc43_Ca43_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc43_to_Ca43) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Sc43_to_Ca43) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Sc43) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ca44_Sc44_meta, j_Ca44_Sc44_rhoy, j_Ca44_Sc44_temp, j_Ca44_Sc44_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ca44_to_Sc44) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ca44_to_Sc44) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ca44) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc44_Ca44_meta, j_Sc44_Ca44_rhoy, j_Sc44_Ca44_temp, j_Sc44_Ca44_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc44_to_Ca44) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Sc44_to_Ca44) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Sc44) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc44_Ti44_meta, j_Sc44_Ti44_rhoy, j_Sc44_Ti44_temp, j_Sc44_Ti44_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc44_to_Ti44) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Sc44_to_Ti44) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Sc44) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti44_Sc44_meta, j_Ti44_Sc44_rhoy, j_Ti44_Sc44_temp, j_Ti44_Sc44_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti44_to_Sc44) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ti44_to_Sc44) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ti44) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Co53_Fe53_meta, j_Co53_Fe53_rhoy, j_Co53_Fe53_temp, j_Co53_Fe53_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Co53_to_Fe53) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Co53_to_Fe53) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Co53) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Fe53_Co53_meta, j_Fe53_Co53_rhoy, j_Fe53_Co53_temp, j_Fe53_Co53_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Fe53_to_Co53) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Fe53_to_Co53) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Fe53) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu57_Ni57_meta, j_Cu57_Ni57_rhoy, j_Cu57_Ni57_temp, j_Cu57_Ni57_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu57_to_Ni57) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cu57_to_Ni57) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cu57) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni57_Cu57_meta, j_Ni57_Cu57_rhoy, j_Ni57_Cu57_temp, j_Ni57_Cu57_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni57_to_Cu57) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ni57_to_Cu57) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ni57) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ca45_Sc45_meta, j_Ca45_Sc45_rhoy, j_Ca45_Sc45_temp, j_Ca45_Sc45_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ca45_to_Sc45) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ca45_to_Sc45) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ca45) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc45_Ca45_meta, j_Sc45_Ca45_rhoy, j_Sc45_Ca45_temp, j_Sc45_Ca45_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc45_to_Ca45) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Sc45_to_Ca45) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Sc45) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc45_Ti45_meta, j_Sc45_Ti45_rhoy, j_Sc45_Ti45_temp, j_Sc45_Ti45_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc45_to_Ti45) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Sc45_to_Ti45) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Sc45) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti45_Sc45_meta, j_Ti45_Sc45_rhoy, j_Ti45_Sc45_temp, j_Ti45_Sc45_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti45_to_Sc45) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ti45_to_Sc45) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ti45) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ca46_Sc46_meta, j_Ca46_Sc46_rhoy, j_Ca46_Sc46_temp, j_Ca46_Sc46_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ca46_to_Sc46) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ca46_to_Sc46) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ca46) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc46_Ca46_meta, j_Sc46_Ca46_rhoy, j_Sc46_Ca46_temp, j_Sc46_Ca46_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc46_to_Ca46) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Sc46_to_Ca46) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Sc46) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc46_Ti46_meta, j_Sc46_Ti46_rhoy, j_Sc46_Ti46_temp, j_Sc46_Ti46_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc46_to_Ti46) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Sc46_to_Ti46) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Sc46) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti46_Sc46_meta, j_Ti46_Sc46_rhoy, j_Ti46_Sc46_temp, j_Ti46_Sc46_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti46_to_Sc46) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ti46_to_Sc46) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ti46) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti46_V46_meta, j_Ti46_V46_rhoy, j_Ti46_V46_temp, j_Ti46_V46_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti46_to_V46) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ti46_to_V46) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ti46) * (edot_nu + edot_gamma);

    tabular_evaluate(j_V46_Ti46_meta, j_V46_Ti46_rhoy, j_V46_Ti46_temp, j_V46_Ti46_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_V46_to_Ti46) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_V46_to_Ti46) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(V46) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ca47_Sc47_meta, j_Ca47_Sc47_rhoy, j_Ca47_Sc47_temp, j_Ca47_Sc47_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ca47_to_Sc47) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ca47_to_Sc47) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ca47) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc47_Ca47_meta, j_Sc47_Ca47_rhoy, j_Sc47_Ca47_temp, j_Sc47_Ca47_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc47_to_Ca47) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Sc47_to_Ca47) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Sc47) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc47_Ti47_meta, j_Sc47_Ti47_rhoy, j_Sc47_Ti47_temp, j_Sc47_Ti47_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc47_to_Ti47) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Sc47_to_Ti47) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Sc47) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti47_Sc47_meta, j_Ti47_Sc47_rhoy, j_Ti47_Sc47_temp, j_Ti47_Sc47_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti47_to_Sc47) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ti47_to_Sc47) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ti47) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti47_V47_meta, j_Ti47_V47_rhoy, j_Ti47_V47_temp, j_Ti47_V47_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti47_to_V47) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ti47_to_V47) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ti47) * (edot_nu + edot_gamma);

    tabular_evaluate(j_V47_Ti47_meta, j_V47_Ti47_rhoy, j_V47_Ti47_temp, j_V47_Ti47_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_V47_to_Ti47) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_V47_to_Ti47) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(V47) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ca48_Sc48_meta, j_Ca48_Sc48_rhoy, j_Ca48_Sc48_temp, j_Ca48_Sc48_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ca48_to_Sc48) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ca48_to_Sc48) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ca48) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cr48_V48_meta, j_Cr48_V48_rhoy, j_Cr48_V48_temp, j_Cr48_V48_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cr48_to_V48) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cr48_to_V48) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cr48) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc48_Ca48_meta, j_Sc48_Ca48_rhoy, j_Sc48_Ca48_temp, j_Sc48_Ca48_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc48_to_Ca48) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Sc48_to_Ca48) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Sc48) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc48_Ti48_meta, j_Sc48_Ti48_rhoy, j_Sc48_Ti48_temp, j_Sc48_Ti48_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc48_to_Ti48) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Sc48_to_Ti48) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Sc48) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti48_Sc48_meta, j_Ti48_Sc48_rhoy, j_Ti48_Sc48_temp, j_Ti48_Sc48_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti48_to_Sc48) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ti48_to_Sc48) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ti48) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti48_V48_meta, j_Ti48_V48_rhoy, j_Ti48_V48_temp, j_Ti48_V48_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti48_to_V48) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ti48_to_V48) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ti48) * (edot_nu + edot_gamma);

    tabular_evaluate(j_V48_Cr48_meta, j_V48_Cr48_rhoy, j_V48_Cr48_temp, j_V48_Cr48_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_V48_to_Cr48) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_V48_to_Cr48) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(V48) * (edot_nu + edot_gamma);

    tabular_evaluate(j_V48_Ti48_meta, j_V48_Ti48_rhoy, j_V48_Ti48_temp, j_V48_Ti48_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_V48_to_Ti48) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_V48_to_Ti48) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(V48) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cr49_V49_meta, j_Cr49_V49_rhoy, j_Cr49_V49_temp, j_Cr49_V49_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cr49_to_V49) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cr49_to_V49) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cr49) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc49_Ti49_meta, j_Sc49_Ti49_rhoy, j_Sc49_Ti49_temp, j_Sc49_Ti49_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc49_to_Ti49) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Sc49_to_Ti49) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Sc49) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti49_Sc49_meta, j_Ti49_Sc49_rhoy, j_Ti49_Sc49_temp, j_Ti49_Sc49_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti49_to_Sc49) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ti49_to_Sc49) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ti49) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti49_V49_meta, j_Ti49_V49_rhoy, j_Ti49_V49_temp, j_Ti49_V49_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti49_to_V49) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ti49_to_V49) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ti49) * (edot_nu + edot_gamma);

    tabular_evaluate(j_V49_Cr49_meta, j_V49_Cr49_rhoy, j_V49_Cr49_temp, j_V49_Cr49_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_V49_to_Cr49) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_V49_to_Cr49) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(V49) * (edot_nu + edot_gamma);

    tabular_evaluate(j_V49_Ti49_meta, j_V49_Ti49_rhoy, j_V49_Ti49_temp, j_V49_Ti49_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_V49_to_Ti49) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_V49_to_Ti49) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(V49) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cr50_Mn50_meta, j_Cr50_Mn50_rhoy, j_Cr50_Mn50_temp, j_Cr50_Mn50_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cr50_to_Mn50) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cr50_to_Mn50) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cr50) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cr50_V50_meta, j_Cr50_V50_rhoy, j_Cr50_V50_temp, j_Cr50_V50_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cr50_to_V50) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cr50_to_V50) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cr50) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mn50_Cr50_meta, j_Mn50_Cr50_rhoy, j_Mn50_Cr50_temp, j_Mn50_Cr50_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mn50_to_Cr50) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Mn50_to_Cr50) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Mn50) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti50_V50_meta, j_Ti50_V50_rhoy, j_Ti50_V50_temp, j_Ti50_V50_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti50_to_V50) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ti50_to_V50) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ti50) * (edot_nu + edot_gamma);

    tabular_evaluate(j_V50_Cr50_meta, j_V50_Cr50_rhoy, j_V50_Cr50_temp, j_V50_Cr50_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_V50_to_Cr50) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_V50_to_Cr50) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(V50) * (edot_nu + edot_gamma);

    tabular_evaluate(j_V50_Ti50_meta, j_V50_Ti50_rhoy, j_V50_Ti50_temp, j_V50_Ti50_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_V50_to_Ti50) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_V50_to_Ti50) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(V50) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cr51_Mn51_meta, j_Cr51_Mn51_rhoy, j_Cr51_Mn51_temp, j_Cr51_Mn51_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cr51_to_Mn51) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cr51_to_Mn51) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cr51) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cr51_V51_meta, j_Cr51_V51_rhoy, j_Cr51_V51_temp, j_Cr51_V51_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cr51_to_V51) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cr51_to_V51) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cr51) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mn51_Cr51_meta, j_Mn51_Cr51_rhoy, j_Mn51_Cr51_temp, j_Mn51_Cr51_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mn51_to_Cr51) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Mn51_to_Cr51) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Mn51) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti51_V51_meta, j_Ti51_V51_rhoy, j_Ti51_V51_temp, j_Ti51_V51_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti51_to_V51) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ti51_to_V51) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ti51) * (edot_nu + edot_gamma);

    tabular_evaluate(j_V51_Cr51_meta, j_V51_Cr51_rhoy, j_V51_Cr51_temp, j_V51_Cr51_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_V51_to_Cr51) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_V51_to_Cr51) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(V51) * (edot_nu + edot_gamma);

    tabular_evaluate(j_V51_Ti51_meta, j_V51_Ti51_rhoy, j_V51_Ti51_temp, j_V51_Ti51_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_V51_to_Ti51) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_V51_to_Ti51) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(V51) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cr52_Mn52_meta, j_Cr52_Mn52_rhoy, j_Cr52_Mn52_temp, j_Cr52_Mn52_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cr52_to_Mn52) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cr52_to_Mn52) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cr52) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cr52_V52_meta, j_Cr52_V52_rhoy, j_Cr52_V52_temp, j_Cr52_V52_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cr52_to_V52) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cr52_to_V52) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cr52) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Fe52_Mn52_meta, j_Fe52_Mn52_rhoy, j_Fe52_Mn52_temp, j_Fe52_Mn52_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Fe52_to_Mn52) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Fe52_to_Mn52) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Fe52) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mn52_Cr52_meta, j_Mn52_Cr52_rhoy, j_Mn52_Cr52_temp, j_Mn52_Cr52_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mn52_to_Cr52) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Mn52_to_Cr52) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Mn52) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mn52_Fe52_meta, j_Mn52_Fe52_rhoy, j_Mn52_Fe52_temp, j_Mn52_Fe52_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mn52_to_Fe52) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Mn52_to_Fe52) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Mn52) * (edot_nu + edot_gamma);

    tabular_evaluate(j_V52_Cr52_meta, j_V52_Cr52_rhoy, j_V52_Cr52_temp, j_V52_Cr52_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_V52_to_Cr52) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_V52_to_Cr52) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(V52) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cr53_Mn53_meta, j_Cr53_Mn53_rhoy, j_Cr53_Mn53_temp, j_Cr53_Mn53_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cr53_to_Mn53) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cr53_to_Mn53) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cr53) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Fe53_Mn53_meta, j_Fe53_Mn53_rhoy, j_Fe53_Mn53_temp, j_Fe53_Mn53_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Fe53_to_Mn53) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Fe53_to_Mn53) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Fe53) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mn53_Cr53_meta, j_Mn53_Cr53_rhoy, j_Mn53_Cr53_temp, j_Mn53_Cr53_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mn53_to_Cr53) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Mn53_to_Cr53) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Mn53) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mn53_Fe53_meta, j_Mn53_Fe53_rhoy, j_Mn53_Fe53_temp, j_Mn53_Fe53_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mn53_to_Fe53) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Mn53_to_Fe53) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Mn53) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Co54_Fe54_meta, j_Co54_Fe54_rhoy, j_Co54_Fe54_temp, j_Co54_Fe54_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Co54_to_Fe54) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Co54_to_Fe54) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Co54) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cr54_Mn54_meta, j_Cr54_Mn54_rhoy, j_Cr54_Mn54_temp, j_Cr54_Mn54_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cr54_to_Mn54) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cr54_to_Mn54) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cr54) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Fe54_Co54_meta, j_Fe54_Co54_rhoy, j_Fe54_Co54_temp, j_Fe54_Co54_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Fe54_to_Co54) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Fe54_to_Co54) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Fe54) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Fe54_Mn54_meta, j_Fe54_Mn54_rhoy, j_Fe54_Mn54_temp, j_Fe54_Mn54_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Fe54_to_Mn54) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Fe54_to_Mn54) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Fe54) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mn54_Cr54_meta, j_Mn54_Cr54_rhoy, j_Mn54_Cr54_temp, j_Mn54_Cr54_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mn54_to_Cr54) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Mn54_to_Cr54) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Mn54) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mn54_Fe54_meta, j_Mn54_Fe54_rhoy, j_Mn54_Fe54_temp, j_Mn54_Fe54_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mn54_to_Fe54) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Mn54_to_Fe54) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Mn54) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Co55_Fe55_meta, j_Co55_Fe55_rhoy, j_Co55_Fe55_temp, j_Co55_Fe55_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Co55_to_Fe55) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Co55_to_Fe55) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Co55) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Fe55_Co55_meta, j_Fe55_Co55_rhoy, j_Fe55_Co55_temp, j_Fe55_Co55_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Fe55_to_Co55) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Fe55_to_Co55) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Fe55) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Fe55_Mn55_meta, j_Fe55_Mn55_rhoy, j_Fe55_Mn55_temp, j_Fe55_Mn55_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Fe55_to_Mn55) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Fe55_to_Mn55) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Fe55) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mn55_Fe55_meta, j_Mn55_Fe55_rhoy, j_Mn55_Fe55_temp, j_Mn55_Fe55_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mn55_to_Fe55) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Mn55_to_Fe55) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Mn55) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Co56_Fe56_meta, j_Co56_Fe56_rhoy, j_Co56_Fe56_temp, j_Co56_Fe56_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Co56_to_Fe56) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Co56_to_Fe56) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Co56) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Co56_Ni56_meta, j_Co56_Ni56_rhoy, j_Co56_Ni56_temp, j_Co56_Ni56_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Co56_to_Ni56) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Co56_to_Ni56) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Co56) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Fe56_Co56_meta, j_Fe56_Co56_rhoy, j_Fe56_Co56_temp, j_Fe56_Co56_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Fe56_to_Co56) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Fe56_to_Co56) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Fe56) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni56_Co56_meta, j_Ni56_Co56_rhoy, j_Ni56_Co56_temp, j_Ni56_Co56_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni56_to_Co56) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ni56_to_Co56) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ni56) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Co57_Fe57_meta, j_Co57_Fe57_rhoy, j_Co57_Fe57_temp, j_Co57_Fe57_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Co57_to_Fe57) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Co57_to_Fe57) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Co57) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Co57_Ni57_meta, j_Co57_Ni57_rhoy, j_Co57_Ni57_temp, j_Co57_Ni57_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Co57_to_Ni57) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Co57_to_Ni57) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Co57) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Fe57_Co57_meta, j_Fe57_Co57_rhoy, j_Fe57_Co57_temp, j_Fe57_Co57_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Fe57_to_Co57) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Fe57_to_Co57) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Fe57) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni57_Co57_meta, j_Ni57_Co57_rhoy, j_Ni57_Co57_temp, j_Ni57_Co57_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni57_to_Co57) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ni57_to_Co57) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ni57) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Co58_Fe58_meta, j_Co58_Fe58_rhoy, j_Co58_Fe58_temp, j_Co58_Fe58_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Co58_to_Fe58) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Co58_to_Fe58) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Co58) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Co58_Ni58_meta, j_Co58_Ni58_rhoy, j_Co58_Ni58_temp, j_Co58_Ni58_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Co58_to_Ni58) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Co58_to_Ni58) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Co58) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu58_Ni58_meta, j_Cu58_Ni58_rhoy, j_Cu58_Ni58_temp, j_Cu58_Ni58_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu58_to_Ni58) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cu58_to_Ni58) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cu58) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Fe58_Co58_meta, j_Fe58_Co58_rhoy, j_Fe58_Co58_temp, j_Fe58_Co58_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Fe58_to_Co58) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Fe58_to_Co58) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Fe58) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni58_Co58_meta, j_Ni58_Co58_rhoy, j_Ni58_Co58_temp, j_Ni58_Co58_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni58_to_Co58) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ni58_to_Co58) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ni58) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni58_Cu58_meta, j_Ni58_Cu58_rhoy, j_Ni58_Cu58_temp, j_Ni58_Cu58_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni58_to_Cu58) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ni58_to_Cu58) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ni58) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Co59_Ni59_meta, j_Co59_Ni59_rhoy, j_Co59_Ni59_temp, j_Co59_Ni59_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Co59_to_Ni59) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Co59_to_Ni59) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Co59) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu59_Ni59_meta, j_Cu59_Ni59_rhoy, j_Cu59_Ni59_temp, j_Cu59_Ni59_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu59_to_Ni59) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cu59_to_Ni59) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cu59) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni59_Co59_meta, j_Ni59_Co59_rhoy, j_Ni59_Co59_temp, j_Ni59_Co59_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni59_to_Co59) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ni59_to_Co59) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ni59) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni59_Cu59_meta, j_Ni59_Cu59_rhoy, j_Ni59_Cu59_temp, j_Ni59_Cu59_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni59_to_Cu59) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ni59_to_Cu59) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ni59) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu60_Ni60_meta, j_Cu60_Ni60_rhoy, j_Cu60_Ni60_temp, j_Cu60_Ni60_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu60_to_Ni60) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cu60_to_Ni60) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cu60) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu60_Zn60_meta, j_Cu60_Zn60_rhoy, j_Cu60_Zn60_temp, j_Cu60_Zn60_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu60_to_Zn60) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cu60_to_Zn60) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cu60) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni60_Cu60_meta, j_Ni60_Cu60_rhoy, j_Ni60_Cu60_temp, j_Ni60_Cu60_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni60_to_Cu60) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ni60_to_Cu60) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ni60) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Zn60_Cu60_meta, j_Zn60_Cu60_rhoy, j_Zn60_Cu60_temp, j_Zn60_Cu60_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Zn60_to_Cu60) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Zn60_to_Cu60) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Zn60) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu61_Ni61_meta, j_Cu61_Ni61_rhoy, j_Cu61_Ni61_temp, j_Cu61_Ni61_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu61_to_Ni61) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cu61_to_Ni61) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cu61) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu61_Zn61_meta, j_Cu61_Zn61_rhoy, j_Cu61_Zn61_temp, j_Cu61_Zn61_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu61_to_Zn61) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cu61_to_Zn61) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cu61) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni61_Cu61_meta, j_Ni61_Cu61_rhoy, j_Ni61_Cu61_temp, j_Ni61_Cu61_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni61_to_Cu61) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ni61_to_Cu61) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ni61) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Zn61_Cu61_meta, j_Zn61_Cu61_rhoy, j_Zn61_Cu61_temp, j_Zn61_Cu61_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Zn61_to_Cu61) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Zn61_to_Cu61) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Zn61) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu62_Ni62_meta, j_Cu62_Ni62_rhoy, j_Cu62_Ni62_temp, j_Cu62_Ni62_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu62_to_Ni62) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cu62_to_Ni62) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cu62) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu62_Zn62_meta, j_Cu62_Zn62_rhoy, j_Cu62_Zn62_temp, j_Cu62_Zn62_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu62_to_Zn62) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cu62_to_Zn62) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cu62) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ga62_Zn62_meta, j_Ga62_Zn62_rhoy, j_Ga62_Zn62_temp, j_Ga62_Zn62_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ga62_to_Zn62) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ga62_to_Zn62) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ga62) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni62_Cu62_meta, j_Ni62_Cu62_rhoy, j_Ni62_Cu62_temp, j_Ni62_Cu62_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni62_to_Cu62) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ni62_to_Cu62) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ni62) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Zn62_Cu62_meta, j_Zn62_Cu62_rhoy, j_Zn62_Cu62_temp, j_Zn62_Cu62_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Zn62_to_Cu62) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Zn62_to_Cu62) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Zn62) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Zn62_Ga62_meta, j_Zn62_Ga62_rhoy, j_Zn62_Ga62_temp, j_Zn62_Ga62_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Zn62_to_Ga62) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Zn62_to_Ga62) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Zn62) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu63_Ni63_meta, j_Cu63_Ni63_rhoy, j_Cu63_Ni63_temp, j_Cu63_Ni63_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu63_to_Ni63) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cu63_to_Ni63) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cu63) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu63_Zn63_meta, j_Cu63_Zn63_rhoy, j_Cu63_Zn63_temp, j_Cu63_Zn63_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu63_to_Zn63) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cu63_to_Zn63) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cu63) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ga63_Zn63_meta, j_Ga63_Zn63_rhoy, j_Ga63_Zn63_temp, j_Ga63_Zn63_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ga63_to_Zn63) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ga63_to_Zn63) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ga63) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni63_Cu63_meta, j_Ni63_Cu63_rhoy, j_Ni63_Cu63_temp, j_Ni63_Cu63_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni63_to_Cu63) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ni63_to_Cu63) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ni63) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Zn63_Cu63_meta, j_Zn63_Cu63_rhoy, j_Zn63_Cu63_temp, j_Zn63_Cu63_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Zn63_to_Cu63) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Zn63_to_Cu63) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Zn63) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Zn63_Ga63_meta, j_Zn63_Ga63_rhoy, j_Zn63_Ga63_temp, j_Zn63_Ga63_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Zn63_to_Ga63) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Zn63_to_Ga63) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Zn63) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu64_Ni64_meta, j_Cu64_Ni64_rhoy, j_Cu64_Ni64_temp, j_Cu64_Ni64_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu64_to_Ni64) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cu64_to_Ni64) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cu64) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu64_Zn64_meta, j_Cu64_Zn64_rhoy, j_Cu64_Zn64_temp, j_Cu64_Zn64_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu64_to_Zn64) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cu64_to_Zn64) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cu64) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ga64_Ge64_meta, j_Ga64_Ge64_rhoy, j_Ga64_Ge64_temp, j_Ga64_Ge64_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ga64_to_Ge64) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ga64_to_Ge64) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ga64) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ga64_Zn64_meta, j_Ga64_Zn64_rhoy, j_Ga64_Zn64_temp, j_Ga64_Zn64_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ga64_to_Zn64) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ga64_to_Zn64) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ga64) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ge64_Ga64_meta, j_Ge64_Ga64_rhoy, j_Ge64_Ga64_temp, j_Ge64_Ga64_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ge64_to_Ga64) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ge64_to_Ga64) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ge64) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni64_Cu64_meta, j_Ni64_Cu64_rhoy, j_Ni64_Cu64_temp, j_Ni64_Cu64_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni64_to_Cu64) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Ni64_to_Cu64) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Ni64) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Zn64_Cu64_meta, j_Zn64_Cu64_rhoy, j_Zn64_Cu64_temp, j_Zn64_Cu64_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Zn64_to_Cu64) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Zn64_to_Cu64) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Zn64) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Zn64_Ga64_meta, j_Zn64_Ga64_rhoy, j_Zn64_Ga64_temp, j_Zn64_Ga64_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Zn64_to_Ga64) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Zn64_to_Ga64) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Zn64) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu65_Zn65_meta, j_Cu65_Zn65_rhoy, j_Cu65_Zn65_temp, j_Cu65_Zn65_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu65_to_Zn65) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Cu65_to_Zn65) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Cu65) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Zn65_Cu65_meta, j_Zn65_Cu65_rhoy, j_Zn65_Cu65_temp, j_Zn65_Cu65_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Zn65_to_Cu65) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_Zn65_to_Cu65) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(Zn65) * (edot_nu + edot_gamma);

    tabular_evaluate(j_n_p_meta, j_n_p_rhoy, j_n_p_temp, j_n_p_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_n_to_p) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_n_to_p) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(N) * (edot_nu + edot_gamma);

    tabular_evaluate(j_p_n_meta, j_p_n_rhoy, j_p_n_temp, j_p_n_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_p_to_n) = rate;
    if constexpr (std::is_same_v<T, rate_derivs_t>) {
        rate_eval.dscreened_rates_dT(k_p_to_n) = drate_dt;
    }
    rate_eval.enuc_weak += C::n_A * Y(H1) * (edot_nu + edot_gamma);


}

#ifdef NSE_NET
AMREX_GPU_HOST_DEVICE AMREX_INLINE
void get_ydot_weak(const burn_t& state,
             amrex::Array1D<amrex::Real, 1, neqs>& ydot_nuc,
             amrex::Real& enuc_weak,
             [[maybe_unused]] const amrex::Array1D<amrex::Real, 1, NumSpec>& Y) {
    ///
    /// Calculate Ydots contribute only from weak reactions.
    /// This is used to calculate dyedt and energy generation from
    /// weak reactions for self-consistent NSE
    ///


    // initialize ydot_nuc to 0

    for (int i = 1; i <= neqs; ++i) {
        ydot_nuc(i) = 0.0_rt;
    }

    rate_t rate_eval;

    [[maybe_unused]] amrex::Real rate, drate_dt, edot_nu, edot_gamma;
    [[maybe_unused]] amrex::Real rhoy = state.rho * state.y_e;

    rate_eval.enuc_weak = 0.0_rt;

    // Calculate tabular rates and get ydot_weak

    tabular_evaluate(j_Na21_Ne21_meta, j_Na21_Ne21_rhoy, j_Na21_Ne21_temp, j_Na21_Ne21_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Na21_to_Ne21) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Na21) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ne21_Na21_meta, j_Ne21_Na21_rhoy, j_Ne21_Na21_temp, j_Ne21_Na21_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ne21_to_Na21) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ne21) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Na22_Ne22_meta, j_Na22_Ne22_rhoy, j_Na22_Ne22_temp, j_Na22_Ne22_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Na22_to_Ne22) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Na22) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ne22_Na22_meta, j_Ne22_Na22_rhoy, j_Ne22_Na22_temp, j_Ne22_Na22_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ne22_to_Na22) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ne22) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mg23_Na23_meta, j_Mg23_Na23_rhoy, j_Mg23_Na23_temp, j_Mg23_Na23_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mg23_to_Na23) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Mg23) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Na23_Mg23_meta, j_Na23_Mg23_rhoy, j_Na23_Mg23_temp, j_Na23_Mg23_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Na23_to_Mg23) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Na23) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Al25_Mg25_meta, j_Al25_Mg25_rhoy, j_Al25_Mg25_temp, j_Al25_Mg25_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Al25_to_Mg25) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Al25) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mg25_Al25_meta, j_Mg25_Al25_rhoy, j_Mg25_Al25_temp, j_Mg25_Al25_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mg25_to_Al25) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Mg25) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Al26_Mg26_meta, j_Al26_Mg26_rhoy, j_Al26_Mg26_temp, j_Al26_Mg26_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Al26_to_Mg26) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Al26) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mg26_Al26_meta, j_Mg26_Al26_rhoy, j_Mg26_Al26_temp, j_Mg26_Al26_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mg26_to_Al26) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Mg26) * (edot_nu + edot_gamma);

    tabular_evaluate(j_P29_Si29_meta, j_P29_Si29_rhoy, j_P29_Si29_temp, j_P29_Si29_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_P29_to_Si29) = rate;
    rate_eval.enuc_weak += C::n_A * Y(P29) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Si29_P29_meta, j_Si29_P29_rhoy, j_Si29_P29_temp, j_Si29_P29_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Si29_to_P29) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Si29) * (edot_nu + edot_gamma);

    tabular_evaluate(j_P30_Si30_meta, j_P30_Si30_rhoy, j_P30_Si30_temp, j_P30_Si30_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_P30_to_Si30) = rate;
    rate_eval.enuc_weak += C::n_A * Y(P30) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Si30_P30_meta, j_Si30_P30_rhoy, j_Si30_P30_temp, j_Si30_P30_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Si30_to_P30) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Si30) * (edot_nu + edot_gamma);

    tabular_evaluate(j_P31_Si31_meta, j_P31_Si31_rhoy, j_P31_Si31_temp, j_P31_Si31_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_P31_to_Si31) = rate;
    rate_eval.enuc_weak += C::n_A * Y(P31) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Si31_P31_meta, j_Si31_P31_rhoy, j_Si31_P31_temp, j_Si31_P31_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Si31_to_P31) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Si31) * (edot_nu + edot_gamma);

    tabular_evaluate(j_P32_S32_meta, j_P32_S32_rhoy, j_P32_S32_temp, j_P32_S32_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_P32_to_S32) = rate;
    rate_eval.enuc_weak += C::n_A * Y(P32) * (edot_nu + edot_gamma);

    tabular_evaluate(j_P32_Si32_meta, j_P32_Si32_rhoy, j_P32_Si32_temp, j_P32_Si32_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_P32_to_Si32) = rate;
    rate_eval.enuc_weak += C::n_A * Y(P32) * (edot_nu + edot_gamma);

    tabular_evaluate(j_S32_P32_meta, j_S32_P32_rhoy, j_S32_P32_temp, j_S32_P32_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_S32_to_P32) = rate;
    rate_eval.enuc_weak += C::n_A * Y(S32) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Si32_P32_meta, j_Si32_P32_rhoy, j_Si32_P32_temp, j_Si32_P32_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Si32_to_P32) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Si32) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cl33_S33_meta, j_Cl33_S33_rhoy, j_Cl33_S33_temp, j_Cl33_S33_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cl33_to_S33) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cl33) * (edot_nu + edot_gamma);

    tabular_evaluate(j_P33_S33_meta, j_P33_S33_rhoy, j_P33_S33_temp, j_P33_S33_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_P33_to_S33) = rate;
    rate_eval.enuc_weak += C::n_A * Y(P33) * (edot_nu + edot_gamma);

    tabular_evaluate(j_S33_Cl33_meta, j_S33_Cl33_rhoy, j_S33_Cl33_temp, j_S33_Cl33_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_S33_to_Cl33) = rate;
    rate_eval.enuc_weak += C::n_A * Y(S33) * (edot_nu + edot_gamma);

    tabular_evaluate(j_S33_P33_meta, j_S33_P33_rhoy, j_S33_P33_temp, j_S33_P33_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_S33_to_P33) = rate;
    rate_eval.enuc_weak += C::n_A * Y(S33) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cl34_S34_meta, j_Cl34_S34_rhoy, j_Cl34_S34_temp, j_Cl34_S34_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cl34_to_S34) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cl34) * (edot_nu + edot_gamma);

    tabular_evaluate(j_S34_Cl34_meta, j_S34_Cl34_rhoy, j_S34_Cl34_temp, j_S34_Cl34_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_S34_to_Cl34) = rate;
    rate_eval.enuc_weak += C::n_A * Y(S34) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cl35_S35_meta, j_Cl35_S35_rhoy, j_Cl35_S35_temp, j_Cl35_S35_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cl35_to_S35) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cl35) * (edot_nu + edot_gamma);

    tabular_evaluate(j_S35_Cl35_meta, j_S35_Cl35_rhoy, j_S35_Cl35_temp, j_S35_Cl35_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_S35_to_Cl35) = rate;
    rate_eval.enuc_weak += C::n_A * Y(S35) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ar36_Cl36_meta, j_Ar36_Cl36_rhoy, j_Ar36_Cl36_temp, j_Ar36_Cl36_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ar36_to_Cl36) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ar36) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cl36_Ar36_meta, j_Cl36_Ar36_rhoy, j_Cl36_Ar36_temp, j_Cl36_Ar36_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cl36_to_Ar36) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cl36) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cl36_S36_meta, j_Cl36_S36_rhoy, j_Cl36_S36_temp, j_Cl36_S36_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cl36_to_S36) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cl36) * (edot_nu + edot_gamma);

    tabular_evaluate(j_S36_Cl36_meta, j_S36_Cl36_rhoy, j_S36_Cl36_temp, j_S36_Cl36_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_S36_to_Cl36) = rate;
    rate_eval.enuc_weak += C::n_A * Y(S36) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ar37_Cl37_meta, j_Ar37_Cl37_rhoy, j_Ar37_Cl37_temp, j_Ar37_Cl37_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ar37_to_Cl37) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ar37) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ar37_K37_meta, j_Ar37_K37_rhoy, j_Ar37_K37_temp, j_Ar37_K37_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ar37_to_K37) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ar37) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cl37_Ar37_meta, j_Cl37_Ar37_rhoy, j_Cl37_Ar37_temp, j_Cl37_Ar37_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cl37_to_Ar37) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cl37) * (edot_nu + edot_gamma);

    tabular_evaluate(j_K37_Ar37_meta, j_K37_Ar37_rhoy, j_K37_Ar37_temp, j_K37_Ar37_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_K37_to_Ar37) = rate;
    rate_eval.enuc_weak += C::n_A * Y(K37) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ar38_K38_meta, j_Ar38_K38_rhoy, j_Ar38_K38_temp, j_Ar38_K38_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ar38_to_K38) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ar38) * (edot_nu + edot_gamma);

    tabular_evaluate(j_K38_Ar38_meta, j_K38_Ar38_rhoy, j_K38_Ar38_temp, j_K38_Ar38_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_K38_to_Ar38) = rate;
    rate_eval.enuc_weak += C::n_A * Y(K38) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ar39_K39_meta, j_Ar39_K39_rhoy, j_Ar39_K39_temp, j_Ar39_K39_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ar39_to_K39) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ar39) * (edot_nu + edot_gamma);

    tabular_evaluate(j_K39_Ar39_meta, j_K39_Ar39_rhoy, j_K39_Ar39_temp, j_K39_Ar39_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_K39_to_Ar39) = rate;
    rate_eval.enuc_weak += C::n_A * Y(K39) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ar40_K40_meta, j_Ar40_K40_rhoy, j_Ar40_K40_temp, j_Ar40_K40_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ar40_to_K40) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ar40) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ca40_K40_meta, j_Ca40_K40_rhoy, j_Ca40_K40_temp, j_Ca40_K40_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ca40_to_K40) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ca40) * (edot_nu + edot_gamma);

    tabular_evaluate(j_K40_Ar40_meta, j_K40_Ar40_rhoy, j_K40_Ar40_temp, j_K40_Ar40_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_K40_to_Ar40) = rate;
    rate_eval.enuc_weak += C::n_A * Y(K40) * (edot_nu + edot_gamma);

    tabular_evaluate(j_K40_Ca40_meta, j_K40_Ca40_rhoy, j_K40_Ca40_temp, j_K40_Ca40_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_K40_to_Ca40) = rate;
    rate_eval.enuc_weak += C::n_A * Y(K40) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ca41_K41_meta, j_Ca41_K41_rhoy, j_Ca41_K41_temp, j_Ca41_K41_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ca41_to_K41) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ca41) * (edot_nu + edot_gamma);

    tabular_evaluate(j_K41_Ca41_meta, j_K41_Ca41_rhoy, j_K41_Ca41_temp, j_K41_Ca41_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_K41_to_Ca41) = rate;
    rate_eval.enuc_weak += C::n_A * Y(K41) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ca43_Sc43_meta, j_Ca43_Sc43_rhoy, j_Ca43_Sc43_temp, j_Ca43_Sc43_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ca43_to_Sc43) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ca43) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc43_Ca43_meta, j_Sc43_Ca43_rhoy, j_Sc43_Ca43_temp, j_Sc43_Ca43_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc43_to_Ca43) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Sc43) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ca44_Sc44_meta, j_Ca44_Sc44_rhoy, j_Ca44_Sc44_temp, j_Ca44_Sc44_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ca44_to_Sc44) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ca44) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc44_Ca44_meta, j_Sc44_Ca44_rhoy, j_Sc44_Ca44_temp, j_Sc44_Ca44_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc44_to_Ca44) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Sc44) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc44_Ti44_meta, j_Sc44_Ti44_rhoy, j_Sc44_Ti44_temp, j_Sc44_Ti44_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc44_to_Ti44) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Sc44) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti44_Sc44_meta, j_Ti44_Sc44_rhoy, j_Ti44_Sc44_temp, j_Ti44_Sc44_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti44_to_Sc44) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ti44) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Co53_Fe53_meta, j_Co53_Fe53_rhoy, j_Co53_Fe53_temp, j_Co53_Fe53_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Co53_to_Fe53) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Co53) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Fe53_Co53_meta, j_Fe53_Co53_rhoy, j_Fe53_Co53_temp, j_Fe53_Co53_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Fe53_to_Co53) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Fe53) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu57_Ni57_meta, j_Cu57_Ni57_rhoy, j_Cu57_Ni57_temp, j_Cu57_Ni57_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu57_to_Ni57) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cu57) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni57_Cu57_meta, j_Ni57_Cu57_rhoy, j_Ni57_Cu57_temp, j_Ni57_Cu57_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni57_to_Cu57) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ni57) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ca45_Sc45_meta, j_Ca45_Sc45_rhoy, j_Ca45_Sc45_temp, j_Ca45_Sc45_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ca45_to_Sc45) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ca45) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc45_Ca45_meta, j_Sc45_Ca45_rhoy, j_Sc45_Ca45_temp, j_Sc45_Ca45_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc45_to_Ca45) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Sc45) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc45_Ti45_meta, j_Sc45_Ti45_rhoy, j_Sc45_Ti45_temp, j_Sc45_Ti45_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc45_to_Ti45) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Sc45) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti45_Sc45_meta, j_Ti45_Sc45_rhoy, j_Ti45_Sc45_temp, j_Ti45_Sc45_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti45_to_Sc45) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ti45) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ca46_Sc46_meta, j_Ca46_Sc46_rhoy, j_Ca46_Sc46_temp, j_Ca46_Sc46_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ca46_to_Sc46) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ca46) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc46_Ca46_meta, j_Sc46_Ca46_rhoy, j_Sc46_Ca46_temp, j_Sc46_Ca46_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc46_to_Ca46) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Sc46) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc46_Ti46_meta, j_Sc46_Ti46_rhoy, j_Sc46_Ti46_temp, j_Sc46_Ti46_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc46_to_Ti46) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Sc46) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti46_Sc46_meta, j_Ti46_Sc46_rhoy, j_Ti46_Sc46_temp, j_Ti46_Sc46_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti46_to_Sc46) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ti46) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti46_V46_meta, j_Ti46_V46_rhoy, j_Ti46_V46_temp, j_Ti46_V46_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti46_to_V46) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ti46) * (edot_nu + edot_gamma);

    tabular_evaluate(j_V46_Ti46_meta, j_V46_Ti46_rhoy, j_V46_Ti46_temp, j_V46_Ti46_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_V46_to_Ti46) = rate;
    rate_eval.enuc_weak += C::n_A * Y(V46) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ca47_Sc47_meta, j_Ca47_Sc47_rhoy, j_Ca47_Sc47_temp, j_Ca47_Sc47_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ca47_to_Sc47) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ca47) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc47_Ca47_meta, j_Sc47_Ca47_rhoy, j_Sc47_Ca47_temp, j_Sc47_Ca47_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc47_to_Ca47) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Sc47) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc47_Ti47_meta, j_Sc47_Ti47_rhoy, j_Sc47_Ti47_temp, j_Sc47_Ti47_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc47_to_Ti47) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Sc47) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti47_Sc47_meta, j_Ti47_Sc47_rhoy, j_Ti47_Sc47_temp, j_Ti47_Sc47_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti47_to_Sc47) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ti47) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti47_V47_meta, j_Ti47_V47_rhoy, j_Ti47_V47_temp, j_Ti47_V47_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti47_to_V47) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ti47) * (edot_nu + edot_gamma);

    tabular_evaluate(j_V47_Ti47_meta, j_V47_Ti47_rhoy, j_V47_Ti47_temp, j_V47_Ti47_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_V47_to_Ti47) = rate;
    rate_eval.enuc_weak += C::n_A * Y(V47) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ca48_Sc48_meta, j_Ca48_Sc48_rhoy, j_Ca48_Sc48_temp, j_Ca48_Sc48_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ca48_to_Sc48) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ca48) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cr48_V48_meta, j_Cr48_V48_rhoy, j_Cr48_V48_temp, j_Cr48_V48_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cr48_to_V48) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cr48) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc48_Ca48_meta, j_Sc48_Ca48_rhoy, j_Sc48_Ca48_temp, j_Sc48_Ca48_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc48_to_Ca48) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Sc48) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc48_Ti48_meta, j_Sc48_Ti48_rhoy, j_Sc48_Ti48_temp, j_Sc48_Ti48_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc48_to_Ti48) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Sc48) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti48_Sc48_meta, j_Ti48_Sc48_rhoy, j_Ti48_Sc48_temp, j_Ti48_Sc48_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti48_to_Sc48) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ti48) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti48_V48_meta, j_Ti48_V48_rhoy, j_Ti48_V48_temp, j_Ti48_V48_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti48_to_V48) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ti48) * (edot_nu + edot_gamma);

    tabular_evaluate(j_V48_Cr48_meta, j_V48_Cr48_rhoy, j_V48_Cr48_temp, j_V48_Cr48_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_V48_to_Cr48) = rate;
    rate_eval.enuc_weak += C::n_A * Y(V48) * (edot_nu + edot_gamma);

    tabular_evaluate(j_V48_Ti48_meta, j_V48_Ti48_rhoy, j_V48_Ti48_temp, j_V48_Ti48_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_V48_to_Ti48) = rate;
    rate_eval.enuc_weak += C::n_A * Y(V48) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cr49_V49_meta, j_Cr49_V49_rhoy, j_Cr49_V49_temp, j_Cr49_V49_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cr49_to_V49) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cr49) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Sc49_Ti49_meta, j_Sc49_Ti49_rhoy, j_Sc49_Ti49_temp, j_Sc49_Ti49_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Sc49_to_Ti49) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Sc49) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti49_Sc49_meta, j_Ti49_Sc49_rhoy, j_Ti49_Sc49_temp, j_Ti49_Sc49_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti49_to_Sc49) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ti49) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti49_V49_meta, j_Ti49_V49_rhoy, j_Ti49_V49_temp, j_Ti49_V49_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti49_to_V49) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ti49) * (edot_nu + edot_gamma);

    tabular_evaluate(j_V49_Cr49_meta, j_V49_Cr49_rhoy, j_V49_Cr49_temp, j_V49_Cr49_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_V49_to_Cr49) = rate;
    rate_eval.enuc_weak += C::n_A * Y(V49) * (edot_nu + edot_gamma);

    tabular_evaluate(j_V49_Ti49_meta, j_V49_Ti49_rhoy, j_V49_Ti49_temp, j_V49_Ti49_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_V49_to_Ti49) = rate;
    rate_eval.enuc_weak += C::n_A * Y(V49) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cr50_Mn50_meta, j_Cr50_Mn50_rhoy, j_Cr50_Mn50_temp, j_Cr50_Mn50_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cr50_to_Mn50) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cr50) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cr50_V50_meta, j_Cr50_V50_rhoy, j_Cr50_V50_temp, j_Cr50_V50_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cr50_to_V50) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cr50) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mn50_Cr50_meta, j_Mn50_Cr50_rhoy, j_Mn50_Cr50_temp, j_Mn50_Cr50_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mn50_to_Cr50) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Mn50) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti50_V50_meta, j_Ti50_V50_rhoy, j_Ti50_V50_temp, j_Ti50_V50_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti50_to_V50) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ti50) * (edot_nu + edot_gamma);

    tabular_evaluate(j_V50_Cr50_meta, j_V50_Cr50_rhoy, j_V50_Cr50_temp, j_V50_Cr50_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_V50_to_Cr50) = rate;
    rate_eval.enuc_weak += C::n_A * Y(V50) * (edot_nu + edot_gamma);

    tabular_evaluate(j_V50_Ti50_meta, j_V50_Ti50_rhoy, j_V50_Ti50_temp, j_V50_Ti50_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_V50_to_Ti50) = rate;
    rate_eval.enuc_weak += C::n_A * Y(V50) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cr51_Mn51_meta, j_Cr51_Mn51_rhoy, j_Cr51_Mn51_temp, j_Cr51_Mn51_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cr51_to_Mn51) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cr51) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cr51_V51_meta, j_Cr51_V51_rhoy, j_Cr51_V51_temp, j_Cr51_V51_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cr51_to_V51) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cr51) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mn51_Cr51_meta, j_Mn51_Cr51_rhoy, j_Mn51_Cr51_temp, j_Mn51_Cr51_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mn51_to_Cr51) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Mn51) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ti51_V51_meta, j_Ti51_V51_rhoy, j_Ti51_V51_temp, j_Ti51_V51_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ti51_to_V51) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ti51) * (edot_nu + edot_gamma);

    tabular_evaluate(j_V51_Cr51_meta, j_V51_Cr51_rhoy, j_V51_Cr51_temp, j_V51_Cr51_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_V51_to_Cr51) = rate;
    rate_eval.enuc_weak += C::n_A * Y(V51) * (edot_nu + edot_gamma);

    tabular_evaluate(j_V51_Ti51_meta, j_V51_Ti51_rhoy, j_V51_Ti51_temp, j_V51_Ti51_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_V51_to_Ti51) = rate;
    rate_eval.enuc_weak += C::n_A * Y(V51) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cr52_Mn52_meta, j_Cr52_Mn52_rhoy, j_Cr52_Mn52_temp, j_Cr52_Mn52_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cr52_to_Mn52) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cr52) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cr52_V52_meta, j_Cr52_V52_rhoy, j_Cr52_V52_temp, j_Cr52_V52_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cr52_to_V52) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cr52) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Fe52_Mn52_meta, j_Fe52_Mn52_rhoy, j_Fe52_Mn52_temp, j_Fe52_Mn52_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Fe52_to_Mn52) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Fe52) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mn52_Cr52_meta, j_Mn52_Cr52_rhoy, j_Mn52_Cr52_temp, j_Mn52_Cr52_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mn52_to_Cr52) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Mn52) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mn52_Fe52_meta, j_Mn52_Fe52_rhoy, j_Mn52_Fe52_temp, j_Mn52_Fe52_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mn52_to_Fe52) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Mn52) * (edot_nu + edot_gamma);

    tabular_evaluate(j_V52_Cr52_meta, j_V52_Cr52_rhoy, j_V52_Cr52_temp, j_V52_Cr52_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_V52_to_Cr52) = rate;
    rate_eval.enuc_weak += C::n_A * Y(V52) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cr53_Mn53_meta, j_Cr53_Mn53_rhoy, j_Cr53_Mn53_temp, j_Cr53_Mn53_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cr53_to_Mn53) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cr53) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Fe53_Mn53_meta, j_Fe53_Mn53_rhoy, j_Fe53_Mn53_temp, j_Fe53_Mn53_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Fe53_to_Mn53) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Fe53) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mn53_Cr53_meta, j_Mn53_Cr53_rhoy, j_Mn53_Cr53_temp, j_Mn53_Cr53_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mn53_to_Cr53) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Mn53) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mn53_Fe53_meta, j_Mn53_Fe53_rhoy, j_Mn53_Fe53_temp, j_Mn53_Fe53_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mn53_to_Fe53) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Mn53) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Co54_Fe54_meta, j_Co54_Fe54_rhoy, j_Co54_Fe54_temp, j_Co54_Fe54_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Co54_to_Fe54) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Co54) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cr54_Mn54_meta, j_Cr54_Mn54_rhoy, j_Cr54_Mn54_temp, j_Cr54_Mn54_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cr54_to_Mn54) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cr54) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Fe54_Co54_meta, j_Fe54_Co54_rhoy, j_Fe54_Co54_temp, j_Fe54_Co54_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Fe54_to_Co54) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Fe54) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Fe54_Mn54_meta, j_Fe54_Mn54_rhoy, j_Fe54_Mn54_temp, j_Fe54_Mn54_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Fe54_to_Mn54) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Fe54) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mn54_Cr54_meta, j_Mn54_Cr54_rhoy, j_Mn54_Cr54_temp, j_Mn54_Cr54_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mn54_to_Cr54) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Mn54) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mn54_Fe54_meta, j_Mn54_Fe54_rhoy, j_Mn54_Fe54_temp, j_Mn54_Fe54_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mn54_to_Fe54) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Mn54) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Co55_Fe55_meta, j_Co55_Fe55_rhoy, j_Co55_Fe55_temp, j_Co55_Fe55_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Co55_to_Fe55) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Co55) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Fe55_Co55_meta, j_Fe55_Co55_rhoy, j_Fe55_Co55_temp, j_Fe55_Co55_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Fe55_to_Co55) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Fe55) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Fe55_Mn55_meta, j_Fe55_Mn55_rhoy, j_Fe55_Mn55_temp, j_Fe55_Mn55_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Fe55_to_Mn55) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Fe55) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Mn55_Fe55_meta, j_Mn55_Fe55_rhoy, j_Mn55_Fe55_temp, j_Mn55_Fe55_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Mn55_to_Fe55) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Mn55) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Co56_Fe56_meta, j_Co56_Fe56_rhoy, j_Co56_Fe56_temp, j_Co56_Fe56_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Co56_to_Fe56) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Co56) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Co56_Ni56_meta, j_Co56_Ni56_rhoy, j_Co56_Ni56_temp, j_Co56_Ni56_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Co56_to_Ni56) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Co56) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Fe56_Co56_meta, j_Fe56_Co56_rhoy, j_Fe56_Co56_temp, j_Fe56_Co56_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Fe56_to_Co56) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Fe56) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni56_Co56_meta, j_Ni56_Co56_rhoy, j_Ni56_Co56_temp, j_Ni56_Co56_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni56_to_Co56) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ni56) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Co57_Fe57_meta, j_Co57_Fe57_rhoy, j_Co57_Fe57_temp, j_Co57_Fe57_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Co57_to_Fe57) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Co57) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Co57_Ni57_meta, j_Co57_Ni57_rhoy, j_Co57_Ni57_temp, j_Co57_Ni57_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Co57_to_Ni57) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Co57) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Fe57_Co57_meta, j_Fe57_Co57_rhoy, j_Fe57_Co57_temp, j_Fe57_Co57_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Fe57_to_Co57) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Fe57) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni57_Co57_meta, j_Ni57_Co57_rhoy, j_Ni57_Co57_temp, j_Ni57_Co57_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni57_to_Co57) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ni57) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Co58_Fe58_meta, j_Co58_Fe58_rhoy, j_Co58_Fe58_temp, j_Co58_Fe58_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Co58_to_Fe58) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Co58) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Co58_Ni58_meta, j_Co58_Ni58_rhoy, j_Co58_Ni58_temp, j_Co58_Ni58_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Co58_to_Ni58) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Co58) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu58_Ni58_meta, j_Cu58_Ni58_rhoy, j_Cu58_Ni58_temp, j_Cu58_Ni58_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu58_to_Ni58) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cu58) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Fe58_Co58_meta, j_Fe58_Co58_rhoy, j_Fe58_Co58_temp, j_Fe58_Co58_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Fe58_to_Co58) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Fe58) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni58_Co58_meta, j_Ni58_Co58_rhoy, j_Ni58_Co58_temp, j_Ni58_Co58_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni58_to_Co58) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ni58) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni58_Cu58_meta, j_Ni58_Cu58_rhoy, j_Ni58_Cu58_temp, j_Ni58_Cu58_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni58_to_Cu58) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ni58) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Co59_Ni59_meta, j_Co59_Ni59_rhoy, j_Co59_Ni59_temp, j_Co59_Ni59_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Co59_to_Ni59) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Co59) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu59_Ni59_meta, j_Cu59_Ni59_rhoy, j_Cu59_Ni59_temp, j_Cu59_Ni59_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu59_to_Ni59) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cu59) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni59_Co59_meta, j_Ni59_Co59_rhoy, j_Ni59_Co59_temp, j_Ni59_Co59_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni59_to_Co59) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ni59) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni59_Cu59_meta, j_Ni59_Cu59_rhoy, j_Ni59_Cu59_temp, j_Ni59_Cu59_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni59_to_Cu59) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ni59) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu60_Ni60_meta, j_Cu60_Ni60_rhoy, j_Cu60_Ni60_temp, j_Cu60_Ni60_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu60_to_Ni60) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cu60) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu60_Zn60_meta, j_Cu60_Zn60_rhoy, j_Cu60_Zn60_temp, j_Cu60_Zn60_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu60_to_Zn60) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cu60) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni60_Cu60_meta, j_Ni60_Cu60_rhoy, j_Ni60_Cu60_temp, j_Ni60_Cu60_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni60_to_Cu60) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ni60) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Zn60_Cu60_meta, j_Zn60_Cu60_rhoy, j_Zn60_Cu60_temp, j_Zn60_Cu60_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Zn60_to_Cu60) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Zn60) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu61_Ni61_meta, j_Cu61_Ni61_rhoy, j_Cu61_Ni61_temp, j_Cu61_Ni61_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu61_to_Ni61) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cu61) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu61_Zn61_meta, j_Cu61_Zn61_rhoy, j_Cu61_Zn61_temp, j_Cu61_Zn61_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu61_to_Zn61) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cu61) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni61_Cu61_meta, j_Ni61_Cu61_rhoy, j_Ni61_Cu61_temp, j_Ni61_Cu61_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni61_to_Cu61) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ni61) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Zn61_Cu61_meta, j_Zn61_Cu61_rhoy, j_Zn61_Cu61_temp, j_Zn61_Cu61_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Zn61_to_Cu61) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Zn61) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu62_Ni62_meta, j_Cu62_Ni62_rhoy, j_Cu62_Ni62_temp, j_Cu62_Ni62_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu62_to_Ni62) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cu62) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu62_Zn62_meta, j_Cu62_Zn62_rhoy, j_Cu62_Zn62_temp, j_Cu62_Zn62_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu62_to_Zn62) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cu62) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ga62_Zn62_meta, j_Ga62_Zn62_rhoy, j_Ga62_Zn62_temp, j_Ga62_Zn62_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ga62_to_Zn62) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ga62) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni62_Cu62_meta, j_Ni62_Cu62_rhoy, j_Ni62_Cu62_temp, j_Ni62_Cu62_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni62_to_Cu62) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ni62) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Zn62_Cu62_meta, j_Zn62_Cu62_rhoy, j_Zn62_Cu62_temp, j_Zn62_Cu62_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Zn62_to_Cu62) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Zn62) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Zn62_Ga62_meta, j_Zn62_Ga62_rhoy, j_Zn62_Ga62_temp, j_Zn62_Ga62_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Zn62_to_Ga62) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Zn62) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu63_Ni63_meta, j_Cu63_Ni63_rhoy, j_Cu63_Ni63_temp, j_Cu63_Ni63_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu63_to_Ni63) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cu63) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu63_Zn63_meta, j_Cu63_Zn63_rhoy, j_Cu63_Zn63_temp, j_Cu63_Zn63_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu63_to_Zn63) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cu63) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ga63_Zn63_meta, j_Ga63_Zn63_rhoy, j_Ga63_Zn63_temp, j_Ga63_Zn63_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ga63_to_Zn63) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ga63) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni63_Cu63_meta, j_Ni63_Cu63_rhoy, j_Ni63_Cu63_temp, j_Ni63_Cu63_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni63_to_Cu63) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ni63) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Zn63_Cu63_meta, j_Zn63_Cu63_rhoy, j_Zn63_Cu63_temp, j_Zn63_Cu63_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Zn63_to_Cu63) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Zn63) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Zn63_Ga63_meta, j_Zn63_Ga63_rhoy, j_Zn63_Ga63_temp, j_Zn63_Ga63_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Zn63_to_Ga63) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Zn63) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu64_Ni64_meta, j_Cu64_Ni64_rhoy, j_Cu64_Ni64_temp, j_Cu64_Ni64_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu64_to_Ni64) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cu64) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu64_Zn64_meta, j_Cu64_Zn64_rhoy, j_Cu64_Zn64_temp, j_Cu64_Zn64_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu64_to_Zn64) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cu64) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ga64_Ge64_meta, j_Ga64_Ge64_rhoy, j_Ga64_Ge64_temp, j_Ga64_Ge64_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ga64_to_Ge64) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ga64) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ga64_Zn64_meta, j_Ga64_Zn64_rhoy, j_Ga64_Zn64_temp, j_Ga64_Zn64_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ga64_to_Zn64) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ga64) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ge64_Ga64_meta, j_Ge64_Ga64_rhoy, j_Ge64_Ga64_temp, j_Ge64_Ga64_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ge64_to_Ga64) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ge64) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Ni64_Cu64_meta, j_Ni64_Cu64_rhoy, j_Ni64_Cu64_temp, j_Ni64_Cu64_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Ni64_to_Cu64) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Ni64) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Zn64_Cu64_meta, j_Zn64_Cu64_rhoy, j_Zn64_Cu64_temp, j_Zn64_Cu64_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Zn64_to_Cu64) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Zn64) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Zn64_Ga64_meta, j_Zn64_Ga64_rhoy, j_Zn64_Ga64_temp, j_Zn64_Ga64_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Zn64_to_Ga64) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Zn64) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Cu65_Zn65_meta, j_Cu65_Zn65_rhoy, j_Cu65_Zn65_temp, j_Cu65_Zn65_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Cu65_to_Zn65) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Cu65) * (edot_nu + edot_gamma);

    tabular_evaluate(j_Zn65_Cu65_meta, j_Zn65_Cu65_rhoy, j_Zn65_Cu65_temp, j_Zn65_Cu65_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_Zn65_to_Cu65) = rate;
    rate_eval.enuc_weak += C::n_A * Y(Zn65) * (edot_nu + edot_gamma);

    tabular_evaluate(j_n_p_meta, j_n_p_rhoy, j_n_p_temp, j_n_p_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_n_to_p) = rate;
    rate_eval.enuc_weak += C::n_A * Y(N) * (edot_nu + edot_gamma);

    tabular_evaluate(j_p_n_meta, j_p_n_rhoy, j_p_n_temp, j_p_n_data,
                     rhoy, state.T, rate, drate_dt, edot_nu, edot_gamma);
    rate_eval.screened_rates(k_p_to_n) = rate;
    rate_eval.enuc_weak += C::n_A * Y(H1) * (edot_nu + edot_gamma);

    auto screened_rates = rate_eval.screened_rates;

    ydot_nuc(N) =
        (-screened_rates(k_n_to_p)*Y(N) + screened_rates(k_p_to_n)*Y(H1));

    ydot_nuc(H1) =
        (screened_rates(k_n_to_p)*Y(N) + -screened_rates(k_p_to_n)*Y(H1));

    ydot_nuc(H2) = 0.0_rt;

    ydot_nuc(He3) = 0.0_rt;

    ydot_nuc(He4) = 0.0_rt;

    ydot_nuc(Li6) = 0.0_rt;

    ydot_nuc(Li7) = 0.0_rt;

    ydot_nuc(Be7) = 0.0_rt;

    ydot_nuc(Be9) = 0.0_rt;

    ydot_nuc(B8) = 0.0_rt;

    ydot_nuc(B10) = 0.0_rt;

    ydot_nuc(B11) = 0.0_rt;

    ydot_nuc(C12) = 0.0_rt;

    ydot_nuc(C13) = 0.0_rt;

    ydot_nuc(C14) = 0.0_rt;

    ydot_nuc(N13) = 0.0_rt;

    ydot_nuc(N14) = 0.0_rt;

    ydot_nuc(N15) = 0.0_rt;

    ydot_nuc(O14) = 0.0_rt;

    ydot_nuc(O15) = 0.0_rt;

    ydot_nuc(O16) = 0.0_rt;

    ydot_nuc(O17) = 0.0_rt;

    ydot_nuc(O18) = 0.0_rt;

    ydot_nuc(F17) = 0.0_rt;

    ydot_nuc(F18) = 0.0_rt;

    ydot_nuc(F19) = 0.0_rt;

    ydot_nuc(Ne18) = 0.0_rt;

    ydot_nuc(Ne19) = 0.0_rt;

    ydot_nuc(Ne20) = 0.0_rt;

    ydot_nuc(Ne21) =
        (screened_rates(k_Na21_to_Ne21)*Y(Na21) + -screened_rates(k_Ne21_to_Na21)*Y(Ne21));

    ydot_nuc(Ne22) =
        (screened_rates(k_Na22_to_Ne22)*Y(Na22) + -screened_rates(k_Ne22_to_Na22)*Y(Ne22));

    ydot_nuc(Na21) =
        (-screened_rates(k_Na21_to_Ne21)*Y(Na21) + screened_rates(k_Ne21_to_Na21)*Y(Ne21));

    ydot_nuc(Na22) =
        (-screened_rates(k_Na22_to_Ne22)*Y(Na22) + screened_rates(k_Ne22_to_Na22)*Y(Ne22));

    ydot_nuc(Na23) =
        (screened_rates(k_Mg23_to_Na23)*Y(Mg23) + -screened_rates(k_Na23_to_Mg23)*Y(Na23));

    ydot_nuc(Mg23) =
        (-screened_rates(k_Mg23_to_Na23)*Y(Mg23) + screened_rates(k_Na23_to_Mg23)*Y(Na23));

    ydot_nuc(Mg24) = 0.0_rt;

    ydot_nuc(Mg25) =
        (screened_rates(k_Al25_to_Mg25)*Y(Al25) + -screened_rates(k_Mg25_to_Al25)*Y(Mg25));

    ydot_nuc(Mg26) =
        (screened_rates(k_Al26_to_Mg26)*Y(Al26) + -screened_rates(k_Mg26_to_Al26)*Y(Mg26));

    ydot_nuc(Al25) =
        (-screened_rates(k_Al25_to_Mg25)*Y(Al25) + screened_rates(k_Mg25_to_Al25)*Y(Mg25));

    ydot_nuc(Al26) =
        (-screened_rates(k_Al26_to_Mg26)*Y(Al26) + screened_rates(k_Mg26_to_Al26)*Y(Mg26));

    ydot_nuc(Al27) = 0.0_rt;

    ydot_nuc(Si28) = 0.0_rt;

    ydot_nuc(Si29) =
        (screened_rates(k_P29_to_Si29)*Y(P29) + -screened_rates(k_Si29_to_P29)*Y(Si29));

    ydot_nuc(Si30) =
        (screened_rates(k_P30_to_Si30)*Y(P30) + -screened_rates(k_Si30_to_P30)*Y(Si30));

    ydot_nuc(Si31) =
        (-screened_rates(k_Si31_to_P31)*Y(Si31) + screened_rates(k_P31_to_Si31)*Y(P31));

    ydot_nuc(Si32) =
        (-screened_rates(k_Si32_to_P32)*Y(Si32) + screened_rates(k_P32_to_Si32)*Y(P32));

    ydot_nuc(P29) =
        (-screened_rates(k_P29_to_Si29)*Y(P29) + screened_rates(k_Si29_to_P29)*Y(Si29));

    ydot_nuc(P30) =
        (-screened_rates(k_P30_to_Si30)*Y(P30) + screened_rates(k_Si30_to_P30)*Y(Si30));

    ydot_nuc(P31) =
        (screened_rates(k_Si31_to_P31)*Y(Si31) + -screened_rates(k_P31_to_Si31)*Y(P31));

    ydot_nuc(P32) =
        (-screened_rates(k_P32_to_S32)*Y(P32) + screened_rates(k_S32_to_P32)*Y(S32)) +
        (screened_rates(k_Si32_to_P32)*Y(Si32) + -screened_rates(k_P32_to_Si32)*Y(P32));

    ydot_nuc(P33) =
        (-screened_rates(k_P33_to_S33)*Y(P33) + screened_rates(k_S33_to_P33)*Y(S33));

    ydot_nuc(S32) =
        (screened_rates(k_P32_to_S32)*Y(P32) + -screened_rates(k_S32_to_P32)*Y(S32));

    ydot_nuc(S33) =
        (screened_rates(k_Cl33_to_S33)*Y(Cl33) + -screened_rates(k_S33_to_Cl33)*Y(S33)) +
        (screened_rates(k_P33_to_S33)*Y(P33) + -screened_rates(k_S33_to_P33)*Y(S33));

    ydot_nuc(S34) =
        (screened_rates(k_Cl34_to_S34)*Y(Cl34) + -screened_rates(k_S34_to_Cl34)*Y(S34));

    ydot_nuc(S35) =
        (-screened_rates(k_S35_to_Cl35)*Y(S35) + screened_rates(k_Cl35_to_S35)*Y(Cl35));

    ydot_nuc(S36) =
        (screened_rates(k_Cl36_to_S36)*Y(Cl36) + -screened_rates(k_S36_to_Cl36)*Y(S36));

    ydot_nuc(Cl33) =
        (-screened_rates(k_Cl33_to_S33)*Y(Cl33) + screened_rates(k_S33_to_Cl33)*Y(S33));

    ydot_nuc(Cl34) =
        (-screened_rates(k_Cl34_to_S34)*Y(Cl34) + screened_rates(k_S34_to_Cl34)*Y(S34));

    ydot_nuc(Cl35) =
        (screened_rates(k_S35_to_Cl35)*Y(S35) + -screened_rates(k_Cl35_to_S35)*Y(Cl35));

    ydot_nuc(Cl36) =
        (-screened_rates(k_Cl36_to_Ar36)*Y(Cl36) + screened_rates(k_Ar36_to_Cl36)*Y(Ar36)) +
        (-screened_rates(k_Cl36_to_S36)*Y(Cl36) + screened_rates(k_S36_to_Cl36)*Y(S36));

    ydot_nuc(Cl37) =
        (screened_rates(k_Ar37_to_Cl37)*Y(Ar37) + -screened_rates(k_Cl37_to_Ar37)*Y(Cl37));

    ydot_nuc(Ar36) =
        (screened_rates(k_Cl36_to_Ar36)*Y(Cl36) + -screened_rates(k_Ar36_to_Cl36)*Y(Ar36));

    ydot_nuc(Ar37) =
        (-screened_rates(k_Ar37_to_Cl37)*Y(Ar37) + screened_rates(k_Cl37_to_Ar37)*Y(Cl37)) +
        (screened_rates(k_K37_to_Ar37)*Y(K37) + -screened_rates(k_Ar37_to_K37)*Y(Ar37));

    ydot_nuc(Ar38) =
        (screened_rates(k_K38_to_Ar38)*Y(K38) + -screened_rates(k_Ar38_to_K38)*Y(Ar38));

    ydot_nuc(Ar39) =
        (-screened_rates(k_Ar39_to_K39)*Y(Ar39) + screened_rates(k_K39_to_Ar39)*Y(K39));

    ydot_nuc(Ar40) =
        (screened_rates(k_K40_to_Ar40)*Y(K40) + -screened_rates(k_Ar40_to_K40)*Y(Ar40));

    ydot_nuc(K37) =
        (-screened_rates(k_K37_to_Ar37)*Y(K37) + screened_rates(k_Ar37_to_K37)*Y(Ar37));

    ydot_nuc(K38) =
        (-screened_rates(k_K38_to_Ar38)*Y(K38) + screened_rates(k_Ar38_to_K38)*Y(Ar38));

    ydot_nuc(K39) =
        (screened_rates(k_Ar39_to_K39)*Y(Ar39) + -screened_rates(k_K39_to_Ar39)*Y(K39));

    ydot_nuc(K40) =
        (-screened_rates(k_K40_to_Ar40)*Y(K40) + screened_rates(k_Ar40_to_K40)*Y(Ar40)) +
        (-screened_rates(k_K40_to_Ca40)*Y(K40) + screened_rates(k_Ca40_to_K40)*Y(Ca40));

    ydot_nuc(K41) =
        (screened_rates(k_Ca41_to_K41)*Y(Ca41) + -screened_rates(k_K41_to_Ca41)*Y(K41));

    ydot_nuc(Ca40) =
        (screened_rates(k_K40_to_Ca40)*Y(K40) + -screened_rates(k_Ca40_to_K40)*Y(Ca40));

    ydot_nuc(Ca41) =
        (-screened_rates(k_Ca41_to_K41)*Y(Ca41) + screened_rates(k_K41_to_Ca41)*Y(K41));

    ydot_nuc(Ca42) = 0.0_rt;

    ydot_nuc(Ca43) =
        (screened_rates(k_Sc43_to_Ca43)*Y(Sc43) + -screened_rates(k_Ca43_to_Sc43)*Y(Ca43));

    ydot_nuc(Ca44) =
        (screened_rates(k_Sc44_to_Ca44)*Y(Sc44) + -screened_rates(k_Ca44_to_Sc44)*Y(Ca44));

    ydot_nuc(Ca45) =
        (-screened_rates(k_Ca45_to_Sc45)*Y(Ca45) + screened_rates(k_Sc45_to_Ca45)*Y(Sc45));

    ydot_nuc(Ca46) =
        (screened_rates(k_Sc46_to_Ca46)*Y(Sc46) + -screened_rates(k_Ca46_to_Sc46)*Y(Ca46));

    ydot_nuc(Ca47) =
        (-screened_rates(k_Ca47_to_Sc47)*Y(Ca47) + screened_rates(k_Sc47_to_Ca47)*Y(Sc47));

    ydot_nuc(Ca48) =
        (-screened_rates(k_Ca48_to_Sc48)*Y(Ca48) + screened_rates(k_Sc48_to_Ca48)*Y(Sc48));

    ydot_nuc(Sc43) =
        (-screened_rates(k_Sc43_to_Ca43)*Y(Sc43) + screened_rates(k_Ca43_to_Sc43)*Y(Ca43));

    ydot_nuc(Sc44) =
        (-screened_rates(k_Sc44_to_Ca44)*Y(Sc44) + screened_rates(k_Ca44_to_Sc44)*Y(Ca44)) +
        (screened_rates(k_Ti44_to_Sc44)*Y(Ti44) + -screened_rates(k_Sc44_to_Ti44)*Y(Sc44));

    ydot_nuc(Sc45) =
        (screened_rates(k_Ca45_to_Sc45)*Y(Ca45) + -screened_rates(k_Sc45_to_Ca45)*Y(Sc45)) +
        (screened_rates(k_Ti45_to_Sc45)*Y(Ti45) + -screened_rates(k_Sc45_to_Ti45)*Y(Sc45));

    ydot_nuc(Sc46) =
        (-screened_rates(k_Sc46_to_Ca46)*Y(Sc46) + screened_rates(k_Ca46_to_Sc46)*Y(Ca46)) +
        (-screened_rates(k_Sc46_to_Ti46)*Y(Sc46) + screened_rates(k_Ti46_to_Sc46)*Y(Ti46));

    ydot_nuc(Sc47) =
        (screened_rates(k_Ca47_to_Sc47)*Y(Ca47) + -screened_rates(k_Sc47_to_Ca47)*Y(Sc47)) +
        (-screened_rates(k_Sc47_to_Ti47)*Y(Sc47) + screened_rates(k_Ti47_to_Sc47)*Y(Ti47));

    ydot_nuc(Sc48) =
        (screened_rates(k_Ca48_to_Sc48)*Y(Ca48) + -screened_rates(k_Sc48_to_Ca48)*Y(Sc48)) +
        (-screened_rates(k_Sc48_to_Ti48)*Y(Sc48) + screened_rates(k_Ti48_to_Sc48)*Y(Ti48));

    ydot_nuc(Sc49) =
        (-screened_rates(k_Sc49_to_Ti49)*Y(Sc49) + screened_rates(k_Ti49_to_Sc49)*Y(Ti49));

    ydot_nuc(Ti44) =
        (-screened_rates(k_Ti44_to_Sc44)*Y(Ti44) + screened_rates(k_Sc44_to_Ti44)*Y(Sc44));

    ydot_nuc(Ti45) =
        (-screened_rates(k_Ti45_to_Sc45)*Y(Ti45) + screened_rates(k_Sc45_to_Ti45)*Y(Sc45));

    ydot_nuc(Ti46) =
        (screened_rates(k_Sc46_to_Ti46)*Y(Sc46) + -screened_rates(k_Ti46_to_Sc46)*Y(Ti46)) +
        (screened_rates(k_V46_to_Ti46)*Y(V46) + -screened_rates(k_Ti46_to_V46)*Y(Ti46));

    ydot_nuc(Ti47) =
        (screened_rates(k_Sc47_to_Ti47)*Y(Sc47) + -screened_rates(k_Ti47_to_Sc47)*Y(Ti47)) +
        (screened_rates(k_V47_to_Ti47)*Y(V47) + -screened_rates(k_Ti47_to_V47)*Y(Ti47));

    ydot_nuc(Ti48) =
        (screened_rates(k_Sc48_to_Ti48)*Y(Sc48) + -screened_rates(k_Ti48_to_Sc48)*Y(Ti48)) +
        (screened_rates(k_V48_to_Ti48)*Y(V48) + -screened_rates(k_Ti48_to_V48)*Y(Ti48));

    ydot_nuc(Ti49) =
        (screened_rates(k_Sc49_to_Ti49)*Y(Sc49) + -screened_rates(k_Ti49_to_Sc49)*Y(Ti49)) +
        (screened_rates(k_V49_to_Ti49)*Y(V49) + -screened_rates(k_Ti49_to_V49)*Y(Ti49));

    ydot_nuc(Ti50) =
        (screened_rates(k_V50_to_Ti50)*Y(V50) + -screened_rates(k_Ti50_to_V50)*Y(Ti50));

    ydot_nuc(Ti51) =
        (-screened_rates(k_Ti51_to_V51)*Y(Ti51) + screened_rates(k_V51_to_Ti51)*Y(V51));

    ydot_nuc(V46) =
        (-screened_rates(k_V46_to_Ti46)*Y(V46) + screened_rates(k_Ti46_to_V46)*Y(Ti46));

    ydot_nuc(V47) =
        (-screened_rates(k_V47_to_Ti47)*Y(V47) + screened_rates(k_Ti47_to_V47)*Y(Ti47));

    ydot_nuc(V48) =
        (screened_rates(k_Cr48_to_V48)*Y(Cr48) + -screened_rates(k_V48_to_Cr48)*Y(V48)) +
        (-screened_rates(k_V48_to_Ti48)*Y(V48) + screened_rates(k_Ti48_to_V48)*Y(Ti48));

    ydot_nuc(V49) =
        (screened_rates(k_Cr49_to_V49)*Y(Cr49) + -screened_rates(k_V49_to_Cr49)*Y(V49)) +
        (-screened_rates(k_V49_to_Ti49)*Y(V49) + screened_rates(k_Ti49_to_V49)*Y(Ti49));

    ydot_nuc(V50) =
        (-screened_rates(k_V50_to_Cr50)*Y(V50) + screened_rates(k_Cr50_to_V50)*Y(Cr50)) +
        (-screened_rates(k_V50_to_Ti50)*Y(V50) + screened_rates(k_Ti50_to_V50)*Y(Ti50));

    ydot_nuc(V51) =
        (screened_rates(k_Cr51_to_V51)*Y(Cr51) + -screened_rates(k_V51_to_Cr51)*Y(V51)) +
        (screened_rates(k_Ti51_to_V51)*Y(Ti51) + -screened_rates(k_V51_to_Ti51)*Y(V51));

    ydot_nuc(V52) =
        (-screened_rates(k_V52_to_Cr52)*Y(V52) + screened_rates(k_Cr52_to_V52)*Y(Cr52));

    ydot_nuc(Cr48) =
        (-screened_rates(k_Cr48_to_V48)*Y(Cr48) + screened_rates(k_V48_to_Cr48)*Y(V48));

    ydot_nuc(Cr49) =
        (-screened_rates(k_Cr49_to_V49)*Y(Cr49) + screened_rates(k_V49_to_Cr49)*Y(V49));

    ydot_nuc(Cr50) =
        (screened_rates(k_Mn50_to_Cr50)*Y(Mn50) + -screened_rates(k_Cr50_to_Mn50)*Y(Cr50)) +
        (screened_rates(k_V50_to_Cr50)*Y(V50) + -screened_rates(k_Cr50_to_V50)*Y(Cr50));

    ydot_nuc(Cr51) =
        (-screened_rates(k_Cr51_to_V51)*Y(Cr51) + screened_rates(k_V51_to_Cr51)*Y(V51)) +
        (screened_rates(k_Mn51_to_Cr51)*Y(Mn51) + -screened_rates(k_Cr51_to_Mn51)*Y(Cr51));

    ydot_nuc(Cr52) =
        (screened_rates(k_Mn52_to_Cr52)*Y(Mn52) + -screened_rates(k_Cr52_to_Mn52)*Y(Cr52)) +
        (screened_rates(k_V52_to_Cr52)*Y(V52) + -screened_rates(k_Cr52_to_V52)*Y(Cr52));

    ydot_nuc(Cr53) =
        (screened_rates(k_Mn53_to_Cr53)*Y(Mn53) + -screened_rates(k_Cr53_to_Mn53)*Y(Cr53));

    ydot_nuc(Cr54) =
        (screened_rates(k_Mn54_to_Cr54)*Y(Mn54) + -screened_rates(k_Cr54_to_Mn54)*Y(Cr54));

    ydot_nuc(Mn50) =
        (-screened_rates(k_Mn50_to_Cr50)*Y(Mn50) + screened_rates(k_Cr50_to_Mn50)*Y(Cr50));

    ydot_nuc(Mn51) =
        (-screened_rates(k_Mn51_to_Cr51)*Y(Mn51) + screened_rates(k_Cr51_to_Mn51)*Y(Cr51));

    ydot_nuc(Mn52) =
        (screened_rates(k_Fe52_to_Mn52)*Y(Fe52) + -screened_rates(k_Mn52_to_Fe52)*Y(Mn52)) +
        (-screened_rates(k_Mn52_to_Cr52)*Y(Mn52) + screened_rates(k_Cr52_to_Mn52)*Y(Cr52));

    ydot_nuc(Mn53) =
        (screened_rates(k_Fe53_to_Mn53)*Y(Fe53) + -screened_rates(k_Mn53_to_Fe53)*Y(Mn53)) +
        (-screened_rates(k_Mn53_to_Cr53)*Y(Mn53) + screened_rates(k_Cr53_to_Mn53)*Y(Cr53));

    ydot_nuc(Mn54) =
        (-screened_rates(k_Mn54_to_Cr54)*Y(Mn54) + screened_rates(k_Cr54_to_Mn54)*Y(Cr54)) +
        (-screened_rates(k_Mn54_to_Fe54)*Y(Mn54) + screened_rates(k_Fe54_to_Mn54)*Y(Fe54));

    ydot_nuc(Mn55) =
        (screened_rates(k_Fe55_to_Mn55)*Y(Fe55) + -screened_rates(k_Mn55_to_Fe55)*Y(Mn55));

    ydot_nuc(Fe52) =
        (-screened_rates(k_Fe52_to_Mn52)*Y(Fe52) + screened_rates(k_Mn52_to_Fe52)*Y(Mn52));

    ydot_nuc(Fe53) =
        (screened_rates(k_Co53_to_Fe53)*Y(Co53) + -screened_rates(k_Fe53_to_Co53)*Y(Fe53)) +
        (-screened_rates(k_Fe53_to_Mn53)*Y(Fe53) + screened_rates(k_Mn53_to_Fe53)*Y(Mn53));

    ydot_nuc(Fe54) =
        (screened_rates(k_Co54_to_Fe54)*Y(Co54) + -screened_rates(k_Fe54_to_Co54)*Y(Fe54)) +
        (screened_rates(k_Mn54_to_Fe54)*Y(Mn54) + -screened_rates(k_Fe54_to_Mn54)*Y(Fe54));

    ydot_nuc(Fe55) =
        (screened_rates(k_Co55_to_Fe55)*Y(Co55) + -screened_rates(k_Fe55_to_Co55)*Y(Fe55)) +
        (-screened_rates(k_Fe55_to_Mn55)*Y(Fe55) + screened_rates(k_Mn55_to_Fe55)*Y(Mn55));

    ydot_nuc(Fe56) =
        (screened_rates(k_Co56_to_Fe56)*Y(Co56) + -screened_rates(k_Fe56_to_Co56)*Y(Fe56));

    ydot_nuc(Fe57) =
        (screened_rates(k_Co57_to_Fe57)*Y(Co57) + -screened_rates(k_Fe57_to_Co57)*Y(Fe57));

    ydot_nuc(Fe58) =
        (screened_rates(k_Co58_to_Fe58)*Y(Co58) + -screened_rates(k_Fe58_to_Co58)*Y(Fe58));

    ydot_nuc(Co53) =
        (-screened_rates(k_Co53_to_Fe53)*Y(Co53) + screened_rates(k_Fe53_to_Co53)*Y(Fe53));

    ydot_nuc(Co54) =
        (-screened_rates(k_Co54_to_Fe54)*Y(Co54) + screened_rates(k_Fe54_to_Co54)*Y(Fe54));

    ydot_nuc(Co55) =
        (-screened_rates(k_Co55_to_Fe55)*Y(Co55) + screened_rates(k_Fe55_to_Co55)*Y(Fe55));

    ydot_nuc(Co56) =
        (-screened_rates(k_Co56_to_Fe56)*Y(Co56) + screened_rates(k_Fe56_to_Co56)*Y(Fe56)) +
        (screened_rates(k_Ni56_to_Co56)*Y(Ni56) + -screened_rates(k_Co56_to_Ni56)*Y(Co56));

    ydot_nuc(Co57) =
        (-screened_rates(k_Co57_to_Fe57)*Y(Co57) + screened_rates(k_Fe57_to_Co57)*Y(Fe57)) +
        (screened_rates(k_Ni57_to_Co57)*Y(Ni57) + -screened_rates(k_Co57_to_Ni57)*Y(Co57));

    ydot_nuc(Co58) =
        (-screened_rates(k_Co58_to_Fe58)*Y(Co58) + screened_rates(k_Fe58_to_Co58)*Y(Fe58)) +
        (-screened_rates(k_Co58_to_Ni58)*Y(Co58) + screened_rates(k_Ni58_to_Co58)*Y(Ni58));

    ydot_nuc(Co59) =
        (screened_rates(k_Ni59_to_Co59)*Y(Ni59) + -screened_rates(k_Co59_to_Ni59)*Y(Co59));

    ydot_nuc(Ni56) =
        (-screened_rates(k_Ni56_to_Co56)*Y(Ni56) + screened_rates(k_Co56_to_Ni56)*Y(Co56));

    ydot_nuc(Ni57) =
        (screened_rates(k_Cu57_to_Ni57)*Y(Cu57) + -screened_rates(k_Ni57_to_Cu57)*Y(Ni57)) +
        (-screened_rates(k_Ni57_to_Co57)*Y(Ni57) + screened_rates(k_Co57_to_Ni57)*Y(Co57));

    ydot_nuc(Ni58) =
        (screened_rates(k_Co58_to_Ni58)*Y(Co58) + -screened_rates(k_Ni58_to_Co58)*Y(Ni58)) +
        (screened_rates(k_Cu58_to_Ni58)*Y(Cu58) + -screened_rates(k_Ni58_to_Cu58)*Y(Ni58));

    ydot_nuc(Ni59) =
        (screened_rates(k_Cu59_to_Ni59)*Y(Cu59) + -screened_rates(k_Ni59_to_Cu59)*Y(Ni59)) +
        (-screened_rates(k_Ni59_to_Co59)*Y(Ni59) + screened_rates(k_Co59_to_Ni59)*Y(Co59));

    ydot_nuc(Ni60) =
        (screened_rates(k_Cu60_to_Ni60)*Y(Cu60) + -screened_rates(k_Ni60_to_Cu60)*Y(Ni60));

    ydot_nuc(Ni61) =
        (screened_rates(k_Cu61_to_Ni61)*Y(Cu61) + -screened_rates(k_Ni61_to_Cu61)*Y(Ni61));

    ydot_nuc(Ni62) =
        (screened_rates(k_Cu62_to_Ni62)*Y(Cu62) + -screened_rates(k_Ni62_to_Cu62)*Y(Ni62));

    ydot_nuc(Ni63) =
        (-screened_rates(k_Ni63_to_Cu63)*Y(Ni63) + screened_rates(k_Cu63_to_Ni63)*Y(Cu63));

    ydot_nuc(Ni64) =
        (screened_rates(k_Cu64_to_Ni64)*Y(Cu64) + -screened_rates(k_Ni64_to_Cu64)*Y(Ni64));

    ydot_nuc(Cu57) =
        (-screened_rates(k_Cu57_to_Ni57)*Y(Cu57) + screened_rates(k_Ni57_to_Cu57)*Y(Ni57));

    ydot_nuc(Cu58) =
        (-screened_rates(k_Cu58_to_Ni58)*Y(Cu58) + screened_rates(k_Ni58_to_Cu58)*Y(Ni58));

    ydot_nuc(Cu59) =
        (-screened_rates(k_Cu59_to_Ni59)*Y(Cu59) + screened_rates(k_Ni59_to_Cu59)*Y(Ni59));

    ydot_nuc(Cu60) =
        (-screened_rates(k_Cu60_to_Ni60)*Y(Cu60) + screened_rates(k_Ni60_to_Cu60)*Y(Ni60)) +
        (screened_rates(k_Zn60_to_Cu60)*Y(Zn60) + -screened_rates(k_Cu60_to_Zn60)*Y(Cu60));

    ydot_nuc(Cu61) =
        (-screened_rates(k_Cu61_to_Ni61)*Y(Cu61) + screened_rates(k_Ni61_to_Cu61)*Y(Ni61)) +
        (screened_rates(k_Zn61_to_Cu61)*Y(Zn61) + -screened_rates(k_Cu61_to_Zn61)*Y(Cu61));

    ydot_nuc(Cu62) =
        (-screened_rates(k_Cu62_to_Ni62)*Y(Cu62) + screened_rates(k_Ni62_to_Cu62)*Y(Ni62)) +
        (screened_rates(k_Zn62_to_Cu62)*Y(Zn62) + -screened_rates(k_Cu62_to_Zn62)*Y(Cu62));

    ydot_nuc(Cu63) =
        (screened_rates(k_Ni63_to_Cu63)*Y(Ni63) + -screened_rates(k_Cu63_to_Ni63)*Y(Cu63)) +
        (screened_rates(k_Zn63_to_Cu63)*Y(Zn63) + -screened_rates(k_Cu63_to_Zn63)*Y(Cu63));

    ydot_nuc(Cu64) =
        (-screened_rates(k_Cu64_to_Ni64)*Y(Cu64) + screened_rates(k_Ni64_to_Cu64)*Y(Ni64)) +
        (-screened_rates(k_Cu64_to_Zn64)*Y(Cu64) + screened_rates(k_Zn64_to_Cu64)*Y(Zn64));

    ydot_nuc(Cu65) =
        (screened_rates(k_Zn65_to_Cu65)*Y(Zn65) + -screened_rates(k_Cu65_to_Zn65)*Y(Cu65));

    ydot_nuc(Zn59) = 0.0_rt;

    ydot_nuc(Zn60) =
        (-screened_rates(k_Zn60_to_Cu60)*Y(Zn60) + screened_rates(k_Cu60_to_Zn60)*Y(Cu60));

    ydot_nuc(Zn61) =
        (-screened_rates(k_Zn61_to_Cu61)*Y(Zn61) + screened_rates(k_Cu61_to_Zn61)*Y(Cu61));

    ydot_nuc(Zn62) =
        (screened_rates(k_Ga62_to_Zn62)*Y(Ga62) + -screened_rates(k_Zn62_to_Ga62)*Y(Zn62)) +
        (-screened_rates(k_Zn62_to_Cu62)*Y(Zn62) + screened_rates(k_Cu62_to_Zn62)*Y(Cu62));

    ydot_nuc(Zn63) =
        (screened_rates(k_Ga63_to_Zn63)*Y(Ga63) + -screened_rates(k_Zn63_to_Ga63)*Y(Zn63)) +
        (-screened_rates(k_Zn63_to_Cu63)*Y(Zn63) + screened_rates(k_Cu63_to_Zn63)*Y(Cu63));

    ydot_nuc(Zn64) =
        (screened_rates(k_Cu64_to_Zn64)*Y(Cu64) + -screened_rates(k_Zn64_to_Cu64)*Y(Zn64)) +
        (screened_rates(k_Ga64_to_Zn64)*Y(Ga64) + -screened_rates(k_Zn64_to_Ga64)*Y(Zn64));

    ydot_nuc(Zn65) =
        (-screened_rates(k_Zn65_to_Cu65)*Y(Zn65) + screened_rates(k_Cu65_to_Zn65)*Y(Cu65));

    ydot_nuc(Zn66) = 0.0_rt;

    ydot_nuc(Ga62) =
        (-screened_rates(k_Ga62_to_Zn62)*Y(Ga62) + screened_rates(k_Zn62_to_Ga62)*Y(Zn62));

    ydot_nuc(Ga63) =
        (-screened_rates(k_Ga63_to_Zn63)*Y(Ga63) + screened_rates(k_Zn63_to_Ga63)*Y(Zn63));

    ydot_nuc(Ga64) =
        (-screened_rates(k_Ga64_to_Zn64)*Y(Ga64) + screened_rates(k_Zn64_to_Ga64)*Y(Zn64)) +
        (screened_rates(k_Ge64_to_Ga64)*Y(Ge64) + -screened_rates(k_Ga64_to_Ge64)*Y(Ga64));

    ydot_nuc(Ge63) = 0.0_rt;

    ydot_nuc(Ge64) =
        (-screened_rates(k_Ge64_to_Ga64)*Y(Ge64) + screened_rates(k_Ga64_to_Ge64)*Y(Ga64));

    enuc_weak = rate_eval.enuc_weak;
}
#endif


AMREX_GPU_HOST_DEVICE AMREX_INLINE
void rhs_nuc(const burn_t& state,
             amrex::Array1D<amrex::Real, 1, neqs>& ydot_nuc,
             const amrex::Array1D<amrex::Real, 1, NumSpec>& Y,
             const amrex::Array1D<amrex::Real, 1, NumRates>& screened_rates) {

    using namespace Rates;

    ydot_nuc(N) =
        (-screened_rates(k_n_p_to_d)*Y(N)*Y(H1)*state.rho + screened_rates(k_d_to_n_p)*Y(H2)) +
        (-screened_rates(k_n_He3_to_He4)*Y(He3)*Y(N)*state.rho + screened_rates(k_He4_to_n_He3)*Y(He4)) +
        (-screened_rates(k_n_Li6_to_Li7)*Y(Li6)*Y(N)*state.rho + screened_rates(k_Li7_to_n_Li6)*Y(Li7)) +
        (-screened_rates(k_n_B10_to_B11)*Y(B10)*Y(N)*state.rho + screened_rates(k_B11_to_n_B10)*Y(B11)) +
        (-screened_rates(k_n_C12_to_C13)*Y(C12)*Y(N)*state.rho + screened_rates(k_C13_to_n_C12)*Y(C13)) +
        (-screened_rates(k_n_C13_to_C14)*Y(C13)*Y(N)*state.rho + screened_rates(k_C14_to_n_C13)*Y(C14)) +
        (-screened_rates(k_n_N13_to_N14)*Y(N13)*Y(N)*state.rho + screened_rates(k_N14_to_n_N13)*Y(N14)) +
        (-screened_rates(k_n_N14_to_N15)*Y(N14)*Y(N)*state.rho + screened_rates(k_N15_to_n_N14)*Y(N15)) +
        (-screened_rates(k_n_O14_to_O15)*Y(O14)*Y(N)*state.rho + screened_rates(k_O15_to_n_O14)*Y(O15)) +
        (-screened_rates(k_n_O15_to_O16)*Y(O15)*Y(N)*state.rho + screened_rates(k_O16_to_n_O15)*Y(O16)) +
        (-screened_rates(k_n_O16_to_O17)*Y(O16)*Y(N)*state.rho + screened_rates(k_O17_to_n_O16)*Y(O17)) +
        (-screened_rates(k_n_O17_to_O18)*Y(O17)*Y(N)*state.rho + screened_rates(k_O18_to_n_O17)*Y(O18)) +
        (-screened_rates(k_n_F17_to_F18)*Y(F17)*Y(N)*state.rho + screened_rates(k_F18_to_n_F17)*Y(F18)) +
        (-screened_rates(k_n_F18_to_F19)*Y(F18)*Y(N)*state.rho + screened_rates(k_F19_to_n_F18)*Y(F19)) +
        (-screened_rates(k_n_Ne18_to_Ne19)*Y(Ne18)*Y(N)*state.rho + screened_rates(k_Ne19_to_n_Ne18)*Y(Ne19)) +
        (-screened_rates(k_n_Ne19_to_Ne20)*Y(Ne19)*Y(N)*state.rho + screened_rates(k_Ne20_to_n_Ne19)*Y(Ne20)) +
        (-screened_rates(k_n_Ne20_to_Ne21)*Y(Ne20)*Y(N)*state.rho + screened_rates(k_Ne21_to_n_Ne20)*Y(Ne21)) +
        (-screened_rates(k_n_Ne21_to_Ne22)*Y(Ne21)*Y(N)*state.rho + screened_rates(k_Ne22_to_n_Ne21)*Y(Ne22)) +
        (-screened_rates(k_n_Na21_to_Na22)*Y(Na21)*Y(N)*state.rho + screened_rates(k_Na22_to_n_Na21)*Y(Na22)) +
        (-screened_rates(k_n_Na22_to_Na23)*Y(Na22)*Y(N)*state.rho + screened_rates(k_Na23_to_n_Na22)*Y(Na23)) +
        (-screened_rates(k_n_Mg23_to_Mg24)*Y(Mg23)*Y(N)*state.rho + screened_rates(k_Mg24_to_n_Mg23)*Y(Mg24)) +
        (-screened_rates(k_n_Mg24_to_Mg25)*Y(Mg24)*Y(N)*state.rho + screened_rates(k_Mg25_to_n_Mg24)*Y(Mg25)) +
        (-screened_rates(k_n_Mg25_to_Mg26)*Y(Mg25)*Y(N)*state.rho + screened_rates(k_Mg26_to_n_Mg25)*Y(Mg26)) +
        (-screened_rates(k_n_Al25_to_Al26)*Y(Al25)*Y(N)*state.rho + screened_rates(k_Al26_to_n_Al25)*Y(Al26)) +
        (-screened_rates(k_n_Al26_to_Al27)*Y(Al26)*Y(N)*state.rho + screened_rates(k_Al27_to_n_Al26)*Y(Al27)) +
        (-screened_rates(k_n_Si28_to_Si29)*Y(Si28)*Y(N)*state.rho + screened_rates(k_Si29_to_n_Si28)*Y(Si29)) +
        (-screened_rates(k_n_Si29_to_Si30)*Y(Si29)*Y(N)*state.rho + screened_rates(k_Si30_to_n_Si29)*Y(Si30)) +
        (-screened_rates(k_n_Si30_to_Si31)*Y(Si30)*Y(N)*state.rho + screened_rates(k_Si31_to_n_Si30)*Y(Si31)) +
        (-screened_rates(k_n_Si31_to_Si32)*Y(Si31)*Y(N)*state.rho + screened_rates(k_Si32_to_n_Si31)*Y(Si32)) +
        (-screened_rates(k_n_P29_to_P30)*Y(P29)*Y(N)*state.rho + screened_rates(k_P30_to_n_P29)*Y(P30)) +
        (-screened_rates(k_n_P30_to_P31)*Y(P30)*Y(N)*state.rho + screened_rates(k_P31_to_n_P30)*Y(P31)) +
        (-screened_rates(k_n_P31_to_P32)*Y(P31)*Y(N)*state.rho + screened_rates(k_P32_to_n_P31)*Y(P32)) +
        (-screened_rates(k_n_P32_to_P33)*Y(P32)*Y(N)*state.rho + screened_rates(k_P33_to_n_P32)*Y(P33)) +
        (-screened_rates(k_n_S32_to_S33)*Y(S32)*Y(N)*state.rho + screened_rates(k_S33_to_n_S32)*Y(S33)) +
        (-screened_rates(k_n_S33_to_S34)*Y(S33)*Y(N)*state.rho + screened_rates(k_S34_to_n_S33)*Y(S34)) +
        (-screened_rates(k_n_S34_to_S35)*Y(S34)*Y(N)*state.rho + screened_rates(k_S35_to_n_S34)*Y(S35)) +
        (-screened_rates(k_n_S35_to_S36)*Y(S35)*Y(N)*state.rho + screened_rates(k_S36_to_n_S35)*Y(S36)) +
        (-screened_rates(k_n_Cl33_to_Cl34)*Y(Cl33)*Y(N)*state.rho + screened_rates(k_Cl34_to_n_Cl33)*Y(Cl34)) +
        (-screened_rates(k_n_Cl34_to_Cl35)*Y(Cl34)*Y(N)*state.rho + screened_rates(k_Cl35_to_n_Cl34)*Y(Cl35)) +
        (-screened_rates(k_n_Cl35_to_Cl36)*Y(Cl35)*Y(N)*state.rho + screened_rates(k_Cl36_to_n_Cl35)*Y(Cl36)) +
        (-screened_rates(k_n_Cl36_to_Cl37)*Y(Cl36)*Y(N)*state.rho + screened_rates(k_Cl37_to_n_Cl36)*Y(Cl37)) +
        (-screened_rates(k_n_Ar36_to_Ar37)*Y(Ar36)*Y(N)*state.rho + screened_rates(k_Ar37_to_n_Ar36)*Y(Ar37)) +
        (-screened_rates(k_n_Ar37_to_Ar38)*Y(Ar37)*Y(N)*state.rho + screened_rates(k_Ar38_to_n_Ar37)*Y(Ar38)) +
        (-screened_rates(k_n_Ar38_to_Ar39)*Y(Ar38)*Y(N)*state.rho + screened_rates(k_Ar39_to_n_Ar38)*Y(Ar39)) +
        (-screened_rates(k_n_Ar39_to_Ar40)*Y(Ar39)*Y(N)*state.rho + screened_rates(k_Ar40_to_n_Ar39)*Y(Ar40)) +
        (-screened_rates(k_n_K37_to_K38)*Y(K37)*Y(N)*state.rho + screened_rates(k_K38_to_n_K37)*Y(K38)) +
        (-screened_rates(k_n_K38_to_K39)*Y(K38)*Y(N)*state.rho + screened_rates(k_K39_to_n_K38)*Y(K39)) +
        (-screened_rates(k_n_K39_to_K40)*Y(K39)*Y(N)*state.rho + screened_rates(k_K40_to_n_K39)*Y(K40)) +
        (-screened_rates(k_n_K40_to_K41)*Y(K40)*Y(N)*state.rho + screened_rates(k_K41_to_n_K40)*Y(K41)) +
        (-screened_rates(k_n_Ca40_to_Ca41)*Y(Ca40)*Y(N)*state.rho + screened_rates(k_Ca41_to_n_Ca40)*Y(Ca41)) +
        (-screened_rates(k_n_Ca41_to_Ca42)*Y(Ca41)*Y(N)*state.rho + screened_rates(k_Ca42_to_n_Ca41)*Y(Ca42)) +
        (-screened_rates(k_n_Ca42_to_Ca43)*Y(Ca42)*Y(N)*state.rho + screened_rates(k_Ca43_to_n_Ca42)*Y(Ca43)) +
        (-screened_rates(k_n_Ca43_to_Ca44)*Y(Ca43)*Y(N)*state.rho + screened_rates(k_Ca44_to_n_Ca43)*Y(Ca44)) +
        (-screened_rates(k_n_Ca44_to_Ca45)*Y(Ca44)*Y(N)*state.rho + screened_rates(k_Ca45_to_n_Ca44)*Y(Ca45)) +
        (-screened_rates(k_n_Ca45_to_Ca46)*Y(Ca45)*Y(N)*state.rho + screened_rates(k_Ca46_to_n_Ca45)*Y(Ca46)) +
        (-screened_rates(k_n_Ca46_to_Ca47)*Y(Ca46)*Y(N)*state.rho + screened_rates(k_Ca47_to_n_Ca46)*Y(Ca47)) +
        (-screened_rates(k_n_Ca47_to_Ca48)*Y(Ca47)*Y(N)*state.rho + screened_rates(k_Ca48_to_n_Ca47)*Y(Ca48)) +
        (-screened_rates(k_n_Sc43_to_Sc44)*Y(Sc43)*Y(N)*state.rho + screened_rates(k_Sc44_to_n_Sc43)*Y(Sc44)) +
        (-screened_rates(k_n_Sc44_to_Sc45)*Y(Sc44)*Y(N)*state.rho + screened_rates(k_Sc45_to_n_Sc44)*Y(Sc45)) +
        (-screened_rates(k_n_Sc45_to_Sc46)*Y(Sc45)*Y(N)*state.rho + screened_rates(k_Sc46_to_n_Sc45)*Y(Sc46)) +
        (-screened_rates(k_n_Sc46_to_Sc47)*Y(Sc46)*Y(N)*state.rho + screened_rates(k_Sc47_to_n_Sc46)*Y(Sc47)) +
        (-screened_rates(k_n_Sc47_to_Sc48)*Y(Sc47)*Y(N)*state.rho + screened_rates(k_Sc48_to_n_Sc47)*Y(Sc48)) +
        (-screened_rates(k_n_Sc48_to_Sc49)*Y(Sc48)*Y(N)*state.rho + screened_rates(k_Sc49_to_n_Sc48)*Y(Sc49)) +
        (-screened_rates(k_n_Ti44_to_Ti45)*Y(Ti44)*Y(N)*state.rho + screened_rates(k_Ti45_to_n_Ti44)*Y(Ti45)) +
        (-screened_rates(k_n_Ti45_to_Ti46)*Y(Ti45)*Y(N)*state.rho + screened_rates(k_Ti46_to_n_Ti45)*Y(Ti46)) +
        (-screened_rates(k_n_Ti46_to_Ti47)*Y(Ti46)*Y(N)*state.rho + screened_rates(k_Ti47_to_n_Ti46)*Y(Ti47)) +
        (-screened_rates(k_n_Ti47_to_Ti48)*Y(Ti47)*Y(N)*state.rho + screened_rates(k_Ti48_to_n_Ti47)*Y(Ti48)) +
        (-screened_rates(k_n_Ti48_to_Ti49)*Y(Ti48)*Y(N)*state.rho + screened_rates(k_Ti49_to_n_Ti48)*Y(Ti49)) +
        (-screened_rates(k_n_Ti49_to_Ti50)*Y(Ti49)*Y(N)*state.rho + screened_rates(k_Ti50_to_n_Ti49)*Y(Ti50)) +
        (-screened_rates(k_n_Ti50_to_Ti51)*Y(Ti50)*Y(N)*state.rho + screened_rates(k_Ti51_to_n_Ti50)*Y(Ti51)) +
        (-screened_rates(k_n_V46_to_V47)*Y(V46)*Y(N)*state.rho + screened_rates(k_V47_to_n_V46)*Y(V47)) +
        (-screened_rates(k_n_V47_to_V48)*Y(V47)*Y(N)*state.rho + screened_rates(k_V48_to_n_V47)*Y(V48)) +
        (-screened_rates(k_n_V48_to_V49)*Y(V48)*Y(N)*state.rho + screened_rates(k_V49_to_n_V48)*Y(V49)) +
        (-screened_rates(k_n_V49_to_V50)*Y(V49)*Y(N)*state.rho + screened_rates(k_V50_to_n_V49)*Y(V50)) +
        (-screened_rates(k_n_V50_to_V51)*Y(V50)*Y(N)*state.rho + screened_rates(k_V51_to_n_V50)*Y(V51)) +
        (-screened_rates(k_n_V51_to_V52)*Y(V51)*Y(N)*state.rho + screened_rates(k_V52_to_n_V51)*Y(V52)) +
        (-screened_rates(k_n_Cr48_to_Cr49)*Y(Cr48)*Y(N)*state.rho + screened_rates(k_Cr49_to_n_Cr48)*Y(Cr49)) +
        (-screened_rates(k_n_Cr49_to_Cr50)*Y(Cr49)*Y(N)*state.rho + screened_rates(k_Cr50_to_n_Cr49)*Y(Cr50)) +
        (-screened_rates(k_n_Cr50_to_Cr51)*Y(Cr50)*Y(N)*state.rho + screened_rates(k_Cr51_to_n_Cr50)*Y(Cr51)) +
        (-screened_rates(k_n_Cr51_to_Cr52)*Y(Cr51)*Y(N)*state.rho + screened_rates(k_Cr52_to_n_Cr51)*Y(Cr52)) +
        (-screened_rates(k_n_Cr52_to_Cr53)*Y(Cr52)*Y(N)*state.rho + screened_rates(k_Cr53_to_n_Cr52)*Y(Cr53)) +
        (-screened_rates(k_n_Cr53_to_Cr54)*Y(Cr53)*Y(N)*state.rho + screened_rates(k_Cr54_to_n_Cr53)*Y(Cr54)) +
        (-screened_rates(k_n_Mn50_to_Mn51)*Y(Mn50)*Y(N)*state.rho + screened_rates(k_Mn51_to_n_Mn50)*Y(Mn51)) +
        (-screened_rates(k_n_Mn51_to_Mn52)*Y(Mn51)*Y(N)*state.rho + screened_rates(k_Mn52_to_n_Mn51)*Y(Mn52)) +
        (-screened_rates(k_n_Mn52_to_Mn53)*Y(Mn52)*Y(N)*state.rho + screened_rates(k_Mn53_to_n_Mn52)*Y(Mn53)) +
        (-screened_rates(k_n_Mn53_to_Mn54)*Y(Mn53)*Y(N)*state.rho + screened_rates(k_Mn54_to_n_Mn53)*Y(Mn54)) +
        (-screened_rates(k_n_Mn54_to_Mn55)*Y(Mn54)*Y(N)*state.rho + screened_rates(k_Mn55_to_n_Mn54)*Y(Mn55)) +
        (-screened_rates(k_n_Fe52_to_Fe53)*Y(Fe52)*Y(N)*state.rho + screened_rates(k_Fe53_to_n_Fe52)*Y(Fe53)) +
        (-screened_rates(k_n_Fe53_to_Fe54)*Y(Fe53)*Y(N)*state.rho + screened_rates(k_Fe54_to_n_Fe53)*Y(Fe54)) +
        (-screened_rates(k_n_Fe54_to_Fe55)*Y(Fe54)*Y(N)*state.rho + screened_rates(k_Fe55_to_n_Fe54)*Y(Fe55)) +
        (-screened_rates(k_n_Fe55_to_Fe56)*Y(Fe55)*Y(N)*state.rho + screened_rates(k_Fe56_to_n_Fe55)*Y(Fe56)) +
        (-screened_rates(k_n_Fe56_to_Fe57)*Y(Fe56)*Y(N)*state.rho + screened_rates(k_Fe57_to_n_Fe56)*Y(Fe57)) +
        (-screened_rates(k_n_Fe57_to_Fe58)*Y(Fe57)*Y(N)*state.rho + screened_rates(k_Fe58_to_n_Fe57)*Y(Fe58)) +
        (-screened_rates(k_n_Co53_to_Co54)*Y(Co53)*Y(N)*state.rho + screened_rates(k_Co54_to_n_Co53)*Y(Co54)) +
        (-screened_rates(k_n_Co54_to_Co55)*Y(Co54)*Y(N)*state.rho + screened_rates(k_Co55_to_n_Co54)*Y(Co55)) +
        (-screened_rates(k_n_Co55_to_Co56)*Y(Co55)*Y(N)*state.rho + screened_rates(k_Co56_to_n_Co55)*Y(Co56)) +
        (-screened_rates(k_n_Co56_to_Co57)*Y(Co56)*Y(N)*state.rho + screened_rates(k_Co57_to_n_Co56)*Y(Co57)) +
        (-screened_rates(k_n_Co57_to_Co58)*Y(Co57)*Y(N)*state.rho + screened_rates(k_Co58_to_n_Co57)*Y(Co58)) +
        (-screened_rates(k_n_Co58_to_Co59)*Y(Co58)*Y(N)*state.rho + screened_rates(k_Co59_to_n_Co58)*Y(Co59)) +
        (-screened_rates(k_n_Ni56_to_Ni57)*Y(Ni56)*Y(N)*state.rho + screened_rates(k_Ni57_to_n_Ni56)*Y(Ni57)) +
        (-screened_rates(k_n_Ni57_to_Ni58)*Y(Ni57)*Y(N)*state.rho + screened_rates(k_Ni58_to_n_Ni57)*Y(Ni58)) +
        (-screened_rates(k_n_Ni58_to_Ni59)*Y(Ni58)*Y(N)*state.rho + screened_rates(k_Ni59_to_n_Ni58)*Y(Ni59)) +
        (-screened_rates(k_n_Ni59_to_Ni60)*Y(Ni59)*Y(N)*state.rho + screened_rates(k_Ni60_to_n_Ni59)*Y(Ni60)) +
        (-screened_rates(k_n_Ni60_to_Ni61)*Y(Ni60)*Y(N)*state.rho + screened_rates(k_Ni61_to_n_Ni60)*Y(Ni61)) +
        (-screened_rates(k_n_Ni61_to_Ni62)*Y(Ni61)*Y(N)*state.rho + screened_rates(k_Ni62_to_n_Ni61)*Y(Ni62)) +
        (-screened_rates(k_n_Ni62_to_Ni63)*Y(Ni62)*Y(N)*state.rho + screened_rates(k_Ni63_to_n_Ni62)*Y(Ni63)) +
        (-screened_rates(k_n_Ni63_to_Ni64)*Y(Ni63)*Y(N)*state.rho + screened_rates(k_Ni64_to_n_Ni63)*Y(Ni64)) +
        (-screened_rates(k_n_Cu57_to_Cu58)*Y(Cu57)*Y(N)*state.rho + screened_rates(k_Cu58_to_n_Cu57)*Y(Cu58)) +
        (-screened_rates(k_n_Cu58_to_Cu59)*Y(Cu58)*Y(N)*state.rho + screened_rates(k_Cu59_to_n_Cu58)*Y(Cu59)) +
        (-screened_rates(k_n_Cu59_to_Cu60)*Y(Cu59)*Y(N)*state.rho + screened_rates(k_Cu60_to_n_Cu59)*Y(Cu60)) +
        (-screened_rates(k_n_Cu60_to_Cu61)*Y(Cu60)*Y(N)*state.rho + screened_rates(k_Cu61_to_n_Cu60)*Y(Cu61)) +
        (-screened_rates(k_n_Cu61_to_Cu62)*Y(Cu61)*Y(N)*state.rho + screened_rates(k_Cu62_to_n_Cu61)*Y(Cu62)) +
        (-screened_rates(k_n_Cu62_to_Cu63)*Y(Cu62)*Y(N)*state.rho + screened_rates(k_Cu63_to_n_Cu62)*Y(Cu63)) +
        (-screened_rates(k_n_Cu63_to_Cu64)*Y(Cu63)*Y(N)*state.rho + screened_rates(k_Cu64_to_n_Cu63)*Y(Cu64)) +
        (-screened_rates(k_n_Cu64_to_Cu65)*Y(Cu64)*Y(N)*state.rho + screened_rates(k_Cu65_to_n_Cu64)*Y(Cu65)) +
        (-screened_rates(k_n_Zn59_to_Zn60)*Y(Zn59)*Y(N)*state.rho + screened_rates(k_Zn60_to_n_Zn59)*Y(Zn60)) +
        (-screened_rates(k_n_Zn60_to_Zn61)*Y(Zn60)*Y(N)*state.rho + screened_rates(k_Zn61_to_n_Zn60)*Y(Zn61)) +
        (-screened_rates(k_n_Zn61_to_Zn62)*Y(Zn61)*Y(N)*state.rho + screened_rates(k_Zn62_to_n_Zn61)*Y(Zn62)) +
        (-screened_rates(k_n_Zn62_to_Zn63)*Y(Zn62)*Y(N)*state.rho + screened_rates(k_Zn63_to_n_Zn62)*Y(Zn63)) +
        (-screened_rates(k_n_Zn63_to_Zn64)*Y(Zn63)*Y(N)*state.rho + screened_rates(k_Zn64_to_n_Zn63)*Y(Zn64)) +
        (-screened_rates(k_n_Zn64_to_Zn65)*Y(Zn64)*Y(N)*state.rho + screened_rates(k_Zn65_to_n_Zn64)*Y(Zn65)) +
        (-screened_rates(k_n_Zn65_to_Zn66)*Y(Zn65)*Y(N)*state.rho + screened_rates(k_Zn66_to_n_Zn65)*Y(Zn66)) +
        (-screened_rates(k_n_Ga62_to_Ga63)*Y(Ga62)*Y(N)*state.rho + screened_rates(k_Ga63_to_n_Ga62)*Y(Ga63)) +
        (-screened_rates(k_n_Ga63_to_Ga64)*Y(Ga63)*Y(N)*state.rho + screened_rates(k_Ga64_to_n_Ga63)*Y(Ga64)) +
        (-screened_rates(k_n_Ge63_to_Ge64)*Y(Ge63)*Y(N)*state.rho + screened_rates(k_Ge64_to_n_Ge63)*Y(Ge64)) +
        (0.5*screened_rates(k_d_d_to_n_He3)*amrex::Math::powi<2>(Y(H2))*state.rho + -screened_rates(k_n_He3_to_d_d)*Y(He3)*Y(N)*state.rho) +
        (screened_rates(k_d_Li6_to_n_Be7)*Y(Li6)*Y(H2)*state.rho + -screened_rates(k_n_Be7_to_d_Li6)*Y(Be7)*Y(N)*state.rho) +
        (-screened_rates(k_n_Be7_to_p_Li7)*Y(Be7)*Y(N)*state.rho + screened_rates(k_p_Li7_to_n_Be7)*Y(Li7)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Be7_to_He4_He4)*Y(Be7)*Y(N)*state.rho + 0.5*screened_rates(k_He4_He4_to_n_Be7)*amrex::Math::powi<2>(Y(He4))*state.rho) +
        (screened_rates(k_He4_Be9_to_n_C12)*Y(Be9)*Y(He4)*state.rho + -screened_rates(k_n_C12_to_He4_Be9)*Y(C12)*Y(N)*state.rho) +
        (-screened_rates(k_n_B10_to_He4_Li7)*Y(B10)*Y(N)*state.rho + screened_rates(k_He4_Li7_to_n_B10)*Y(He4)*Y(Li7)*state.rho) +
        (screened_rates(k_He4_B10_to_n_N13)*Y(B10)*Y(He4)*state.rho + -screened_rates(k_n_N13_to_He4_B10)*Y(N13)*Y(N)*state.rho) +
        (screened_rates(k_He4_B11_to_n_N14)*Y(B11)*Y(He4)*state.rho + -screened_rates(k_n_N14_to_He4_B11)*Y(N14)*Y(N)*state.rho) +
        (screened_rates(k_d_C13_to_n_N14)*Y(C13)*Y(H2)*state.rho + -screened_rates(k_n_N14_to_d_C13)*Y(N14)*Y(N)*state.rho) +
        (screened_rates(k_He4_C13_to_n_O16)*Y(C13)*Y(He4)*state.rho + -screened_rates(k_n_O16_to_He4_C13)*Y(O16)*Y(N)*state.rho) +
        (screened_rates(k_d_C14_to_n_N15)*Y(C14)*Y(H2)*state.rho + -screened_rates(k_n_N15_to_d_C14)*Y(N15)*Y(N)*state.rho) +
        (-screened_rates(k_n_N13_to_p_C13)*Y(N13)*Y(N)*state.rho + screened_rates(k_p_C13_to_n_N13)*Y(C13)*Y(H1)*state.rho) +
        (-screened_rates(k_n_N14_to_p_C14)*Y(N14)*Y(N)*state.rho + screened_rates(k_p_C14_to_n_N14)*Y(C14)*Y(H1)*state.rho) +
        (-screened_rates(k_n_O14_to_p_N14)*Y(O14)*Y(N)*state.rho + screened_rates(k_p_N14_to_n_O14)*Y(N14)*Y(H1)*state.rho) +
        (-screened_rates(k_n_O15_to_p_N15)*Y(O15)*Y(N)*state.rho + screened_rates(k_p_N15_to_n_O15)*Y(N15)*Y(H1)*state.rho) +
        (-screened_rates(k_n_O15_to_He4_C12)*Y(O15)*Y(N)*state.rho + screened_rates(k_He4_C12_to_n_O15)*Y(C12)*Y(He4)*state.rho) +
        (-screened_rates(k_n_O17_to_He4_C14)*Y(O17)*Y(N)*state.rho + screened_rates(k_He4_C14_to_n_O17)*Y(C14)*Y(He4)*state.rho) +
        (screened_rates(k_He4_O17_to_n_Ne20)*Y(He4)*Y(O17)*state.rho + -screened_rates(k_n_Ne20_to_He4_O17)*Y(Ne20)*Y(N)*state.rho) +
        (-screened_rates(k_n_F17_to_p_O17)*Y(F17)*Y(N)*state.rho + screened_rates(k_p_O17_to_n_F17)*Y(O17)*Y(H1)*state.rho) +
        (-screened_rates(k_n_F17_to_He4_N14)*Y(F17)*Y(N)*state.rho + screened_rates(k_He4_N14_to_n_F17)*Y(He4)*Y(N14)*state.rho) +
        (-screened_rates(k_n_F18_to_p_O18)*Y(F18)*Y(N)*state.rho + screened_rates(k_p_O18_to_n_F18)*Y(O18)*Y(H1)*state.rho) +
        (-screened_rates(k_n_F18_to_He4_N15)*Y(F18)*Y(N)*state.rho + screened_rates(k_He4_N15_to_n_F18)*Y(He4)*Y(N15)*state.rho) +
        (-screened_rates(k_n_Ne18_to_p_F18)*Y(Ne18)*Y(N)*state.rho + screened_rates(k_p_F18_to_n_Ne18)*Y(F18)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ne18_to_He4_O15)*Y(Ne18)*Y(N)*state.rho + screened_rates(k_He4_O15_to_n_Ne18)*Y(He4)*Y(O15)*state.rho) +
        (-screened_rates(k_n_Ne19_to_p_F19)*Y(Ne19)*Y(N)*state.rho + screened_rates(k_p_F19_to_n_Ne19)*Y(F19)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ne19_to_He4_O16)*Y(Ne19)*Y(N)*state.rho + screened_rates(k_He4_O16_to_n_Ne19)*Y(He4)*Y(O16)*state.rho) +
        (-screened_rates(k_n_Ne21_to_He4_O18)*Y(Ne21)*Y(N)*state.rho + screened_rates(k_He4_O18_to_n_Ne21)*Y(He4)*Y(O18)*state.rho) +
        (screened_rates(k_He4_Ne21_to_n_Mg24)*Y(He4)*Y(Ne21)*state.rho + -screened_rates(k_n_Mg24_to_He4_Ne21)*Y(Mg24)*Y(N)*state.rho) +
        (-screened_rates(k_n_Na21_to_p_Ne21)*Y(Na21)*Y(N)*state.rho + screened_rates(k_p_Ne21_to_n_Na21)*Y(Ne21)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Na21_to_He4_F18)*Y(Na21)*Y(N)*state.rho + screened_rates(k_He4_F18_to_n_Na21)*Y(F18)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Na22_to_p_Ne22)*Y(Na22)*Y(N)*state.rho + screened_rates(k_p_Ne22_to_n_Na22)*Y(Ne22)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Na22_to_He4_F19)*Y(Na22)*Y(N)*state.rho + screened_rates(k_He4_F19_to_n_Na22)*Y(F19)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Mg23_to_p_Na23)*Y(Mg23)*Y(N)*state.rho + screened_rates(k_p_Na23_to_n_Mg23)*Y(Na23)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Mg23_to_He4_Ne20)*Y(Mg23)*Y(N)*state.rho + screened_rates(k_He4_Ne20_to_n_Mg23)*Y(He4)*Y(Ne20)*state.rho) +
        (-screened_rates(k_n_Mg23_to_C12_C12)*Y(Mg23)*Y(N)*state.rho + 0.5*screened_rates(k_C12_C12_to_n_Mg23)*amrex::Math::powi<2>(Y(C12))*state.rho) +
        (-screened_rates(k_n_Mg25_to_He4_Ne22)*Y(Mg25)*Y(N)*state.rho + screened_rates(k_He4_Ne22_to_n_Mg25)*Y(He4)*Y(Ne22)*state.rho) +
        (screened_rates(k_He4_Mg25_to_n_Si28)*Y(He4)*Y(Mg25)*state.rho + -screened_rates(k_n_Si28_to_He4_Mg25)*Y(Si28)*Y(N)*state.rho) +
        (screened_rates(k_He4_Mg26_to_n_Si29)*Y(He4)*Y(Mg26)*state.rho + -screened_rates(k_n_Si29_to_He4_Mg26)*Y(Si29)*Y(N)*state.rho) +
        (-screened_rates(k_n_Al25_to_p_Mg25)*Y(Al25)*Y(N)*state.rho + screened_rates(k_p_Mg25_to_n_Al25)*Y(Mg25)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Al25_to_He4_Na22)*Y(Al25)*Y(N)*state.rho + screened_rates(k_He4_Na22_to_n_Al25)*Y(He4)*Y(Na22)*state.rho) +
        (-screened_rates(k_n_Al26_to_p_Mg26)*Y(Al26)*Y(N)*state.rho + screened_rates(k_p_Mg26_to_n_Al26)*Y(Mg26)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Al26_to_He4_Na23)*Y(Al26)*Y(N)*state.rho + screened_rates(k_He4_Na23_to_n_Al26)*Y(He4)*Y(Na23)*state.rho) +
        (screened_rates(k_p_Si31_to_n_P31)*Y(Si31)*Y(H1)*state.rho + -screened_rates(k_n_P31_to_p_Si31)*Y(P31)*Y(N)*state.rho) +
        (screened_rates(k_He4_Si31_to_n_S34)*Y(He4)*Y(Si31)*state.rho + -screened_rates(k_n_S34_to_He4_Si31)*Y(S34)*Y(N)*state.rho) +
        (-screened_rates(k_n_P29_to_p_Si29)*Y(P29)*Y(N)*state.rho + screened_rates(k_p_Si29_to_n_P29)*Y(Si29)*Y(H1)*state.rho) +
        (-screened_rates(k_n_P29_to_He4_Al26)*Y(P29)*Y(N)*state.rho + screened_rates(k_He4_Al26_to_n_P29)*Y(Al26)*Y(He4)*state.rho) +
        (-screened_rates(k_n_P30_to_p_Si30)*Y(P30)*Y(N)*state.rho + screened_rates(k_p_Si30_to_n_P30)*Y(Si30)*Y(H1)*state.rho) +
        (-screened_rates(k_n_P30_to_He4_Al27)*Y(P30)*Y(N)*state.rho + screened_rates(k_He4_Al27_to_n_P30)*Y(Al27)*Y(He4)*state.rho) +
        (-screened_rates(k_n_P32_to_p_Si32)*Y(P32)*Y(N)*state.rho + screened_rates(k_p_Si32_to_n_P32)*Y(Si32)*Y(H1)*state.rho) +
        (screened_rates(k_p_P32_to_n_S32)*Y(P32)*Y(H1)*state.rho + -screened_rates(k_n_S32_to_p_P32)*Y(S32)*Y(N)*state.rho) +
        (-screened_rates(k_n_S32_to_He4_Si29)*Y(S32)*Y(N)*state.rho + screened_rates(k_He4_Si29_to_n_S32)*Y(He4)*Y(Si29)*state.rho) +
        (-screened_rates(k_n_S33_to_p_P33)*Y(S33)*Y(N)*state.rho + screened_rates(k_p_P33_to_n_S33)*Y(P33)*Y(H1)*state.rho) +
        (-screened_rates(k_n_S33_to_He4_Si30)*Y(S33)*Y(N)*state.rho + screened_rates(k_He4_Si30_to_n_S33)*Y(He4)*Y(Si30)*state.rho) +
        (-screened_rates(k_n_S35_to_He4_Si32)*Y(S35)*Y(N)*state.rho + screened_rates(k_He4_Si32_to_n_S35)*Y(He4)*Y(Si32)*state.rho) +
        (screened_rates(k_He4_S35_to_n_Ar38)*Y(He4)*Y(S35)*state.rho + -screened_rates(k_n_Ar38_to_He4_S35)*Y(Ar38)*Y(N)*state.rho) +
        (-screened_rates(k_n_Cl33_to_p_S33)*Y(Cl33)*Y(N)*state.rho + screened_rates(k_p_S33_to_n_Cl33)*Y(S33)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cl33_to_He4_P30)*Y(Cl33)*Y(N)*state.rho + screened_rates(k_He4_P30_to_n_Cl33)*Y(He4)*Y(P30)*state.rho) +
        (-screened_rates(k_n_Cl34_to_p_S34)*Y(Cl34)*Y(N)*state.rho + screened_rates(k_p_S34_to_n_Cl34)*Y(S34)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cl34_to_He4_P31)*Y(Cl34)*Y(N)*state.rho + screened_rates(k_He4_P31_to_n_Cl34)*Y(He4)*Y(P31)*state.rho) +
        (-screened_rates(k_n_Cl35_to_p_S35)*Y(Cl35)*Y(N)*state.rho + screened_rates(k_p_S35_to_n_Cl35)*Y(S35)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cl35_to_He4_P32)*Y(Cl35)*Y(N)*state.rho + screened_rates(k_He4_P32_to_n_Cl35)*Y(He4)*Y(P32)*state.rho) +
        (-screened_rates(k_n_Cl36_to_p_S36)*Y(Cl36)*Y(N)*state.rho + screened_rates(k_p_S36_to_n_Cl36)*Y(S36)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cl36_to_He4_P33)*Y(Cl36)*Y(N)*state.rho + screened_rates(k_He4_P33_to_n_Cl36)*Y(He4)*Y(P33)*state.rho) +
        (-screened_rates(k_n_Ar36_to_p_Cl36)*Y(Ar36)*Y(N)*state.rho + screened_rates(k_p_Cl36_to_n_Ar36)*Y(Cl36)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ar36_to_He4_S33)*Y(Ar36)*Y(N)*state.rho + screened_rates(k_He4_S33_to_n_Ar36)*Y(He4)*Y(S33)*state.rho) +
        (-screened_rates(k_n_Ar37_to_p_Cl37)*Y(Ar37)*Y(N)*state.rho + screened_rates(k_p_Cl37_to_n_Ar37)*Y(Cl37)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ar37_to_He4_S34)*Y(Ar37)*Y(N)*state.rho + screened_rates(k_He4_S34_to_n_Ar37)*Y(He4)*Y(S34)*state.rho) +
        (-screened_rates(k_n_Ar39_to_He4_S36)*Y(Ar39)*Y(N)*state.rho + screened_rates(k_He4_S36_to_n_Ar39)*Y(He4)*Y(S36)*state.rho) +
        (-screened_rates(k_n_K37_to_p_Ar37)*Y(K37)*Y(N)*state.rho + screened_rates(k_p_Ar37_to_n_K37)*Y(Ar37)*Y(H1)*state.rho) +
        (-screened_rates(k_n_K37_to_He4_Cl34)*Y(K37)*Y(N)*state.rho + screened_rates(k_He4_Cl34_to_n_K37)*Y(Cl34)*Y(He4)*state.rho) +
        (-screened_rates(k_n_K38_to_p_Ar38)*Y(K38)*Y(N)*state.rho + screened_rates(k_p_Ar38_to_n_K38)*Y(Ar38)*Y(H1)*state.rho) +
        (-screened_rates(k_n_K38_to_He4_Cl35)*Y(K38)*Y(N)*state.rho + screened_rates(k_He4_Cl35_to_n_K38)*Y(Cl35)*Y(He4)*state.rho) +
        (-screened_rates(k_n_K39_to_p_Ar39)*Y(K39)*Y(N)*state.rho + screened_rates(k_p_Ar39_to_n_K39)*Y(Ar39)*Y(H1)*state.rho) +
        (-screened_rates(k_n_K39_to_He4_Cl36)*Y(K39)*Y(N)*state.rho + screened_rates(k_He4_Cl36_to_n_K39)*Y(Cl36)*Y(He4)*state.rho) +
        (-screened_rates(k_n_K40_to_p_Ar40)*Y(K40)*Y(N)*state.rho + screened_rates(k_p_Ar40_to_n_K40)*Y(Ar40)*Y(H1)*state.rho) +
        (-screened_rates(k_n_K40_to_He4_Cl37)*Y(K40)*Y(N)*state.rho + screened_rates(k_He4_Cl37_to_n_K40)*Y(Cl37)*Y(He4)*state.rho) +
        (screened_rates(k_p_K40_to_n_Ca40)*Y(K40)*Y(H1)*state.rho + -screened_rates(k_n_Ca40_to_p_K40)*Y(Ca40)*Y(N)*state.rho) +
        (-screened_rates(k_n_Ca40_to_He4_Ar37)*Y(Ca40)*Y(N)*state.rho + screened_rates(k_He4_Ar37_to_n_Ca40)*Y(Ar37)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Ca41_to_p_K41)*Y(Ca41)*Y(N)*state.rho + screened_rates(k_p_K41_to_n_Ca41)*Y(K41)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ca41_to_He4_Ar38)*Y(Ca41)*Y(N)*state.rho + screened_rates(k_He4_Ar38_to_n_Ca41)*Y(Ar38)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Ca42_to_He4_Ar39)*Y(Ca42)*Y(N)*state.rho + screened_rates(k_He4_Ar39_to_n_Ca42)*Y(Ar39)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Ca43_to_He4_Ar40)*Y(Ca43)*Y(N)*state.rho + screened_rates(k_He4_Ar40_to_n_Ca43)*Y(Ar40)*Y(He4)*state.rho) +
        (screened_rates(k_He4_Ca43_to_n_Ti46)*Y(Ca43)*Y(He4)*state.rho + -screened_rates(k_n_Ti46_to_He4_Ca43)*Y(Ti46)*Y(N)*state.rho) +
        (screened_rates(k_He4_Ca45_to_n_Ti48)*Y(Ca45)*Y(He4)*state.rho + -screened_rates(k_n_Ti48_to_He4_Ca45)*Y(Ti48)*Y(N)*state.rho) +
        (screened_rates(k_p_Ca47_to_n_Sc47)*Y(Ca47)*Y(H1)*state.rho + -screened_rates(k_n_Sc47_to_p_Ca47)*Y(Sc47)*Y(N)*state.rho) +
        (screened_rates(k_He4_Ca47_to_n_Ti50)*Y(Ca47)*Y(He4)*state.rho + -screened_rates(k_n_Ti50_to_He4_Ca47)*Y(Ti50)*Y(N)*state.rho) +
        (-screened_rates(k_n_Sc43_to_p_Ca43)*Y(Sc43)*Y(N)*state.rho + screened_rates(k_p_Ca43_to_n_Sc43)*Y(Ca43)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Sc43_to_He4_K40)*Y(Sc43)*Y(N)*state.rho + screened_rates(k_He4_K40_to_n_Sc43)*Y(He4)*Y(K40)*state.rho) +
        (-screened_rates(k_n_Sc44_to_p_Ca44)*Y(Sc44)*Y(N)*state.rho + screened_rates(k_p_Ca44_to_n_Sc44)*Y(Ca44)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Sc44_to_He4_K41)*Y(Sc44)*Y(N)*state.rho + screened_rates(k_He4_K41_to_n_Sc44)*Y(He4)*Y(K41)*state.rho) +
        (-screened_rates(k_n_Sc45_to_p_Ca45)*Y(Sc45)*Y(N)*state.rho + screened_rates(k_p_Ca45_to_n_Sc45)*Y(Ca45)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Sc46_to_p_Ca46)*Y(Sc46)*Y(N)*state.rho + screened_rates(k_p_Ca46_to_n_Sc46)*Y(Ca46)*Y(H1)*state.rho) +
        (screened_rates(k_p_Sc46_to_n_Ti46)*Y(Sc46)*Y(H1)*state.rho + -screened_rates(k_n_Ti46_to_p_Sc46)*Y(Ti46)*Y(N)*state.rho) +
        (screened_rates(k_He4_Sc46_to_n_V49)*Y(He4)*Y(Sc46)*state.rho + -screened_rates(k_n_V49_to_He4_Sc46)*Y(V49)*Y(N)*state.rho) +
        (-screened_rates(k_n_Sc48_to_p_Ca48)*Y(Sc48)*Y(N)*state.rho + screened_rates(k_p_Ca48_to_n_Sc48)*Y(Ca48)*Y(H1)*state.rho) +
        (screened_rates(k_p_Sc48_to_n_Ti48)*Y(Sc48)*Y(H1)*state.rho + -screened_rates(k_n_Ti48_to_p_Sc48)*Y(Ti48)*Y(N)*state.rho) +
        (screened_rates(k_He4_Sc48_to_n_V51)*Y(He4)*Y(Sc48)*state.rho + -screened_rates(k_n_V51_to_He4_Sc48)*Y(V51)*Y(N)*state.rho) +
        (screened_rates(k_p_Sc49_to_n_Ti49)*Y(Sc49)*Y(H1)*state.rho + -screened_rates(k_n_Ti49_to_p_Sc49)*Y(Ti49)*Y(N)*state.rho) +
        (-screened_rates(k_n_Ti44_to_p_Sc44)*Y(Ti44)*Y(N)*state.rho + screened_rates(k_p_Sc44_to_n_Ti44)*Y(Sc44)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ti44_to_He4_Ca41)*Y(Ti44)*Y(N)*state.rho + screened_rates(k_He4_Ca41_to_n_Ti44)*Y(Ca41)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Ti45_to_p_Sc45)*Y(Ti45)*Y(N)*state.rho + screened_rates(k_p_Sc45_to_n_Ti45)*Y(Sc45)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ti45_to_He4_Ca42)*Y(Ti45)*Y(N)*state.rho + screened_rates(k_He4_Ca42_to_n_Ti45)*Y(Ca42)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Ti47_to_p_Sc47)*Y(Ti47)*Y(N)*state.rho + screened_rates(k_p_Sc47_to_n_Ti47)*Y(Sc47)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ti47_to_He4_Ca44)*Y(Ti47)*Y(N)*state.rho + screened_rates(k_He4_Ca44_to_n_Ti47)*Y(Ca44)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Ti49_to_He4_Ca46)*Y(Ti49)*Y(N)*state.rho + screened_rates(k_He4_Ca46_to_n_Ti49)*Y(Ca46)*Y(He4)*state.rho) +
        (screened_rates(k_He4_Ti49_to_n_Cr52)*Y(He4)*Y(Ti49)*state.rho + -screened_rates(k_n_Cr52_to_He4_Ti49)*Y(Cr52)*Y(N)*state.rho) +
        (-screened_rates(k_n_Ti51_to_He4_Ca48)*Y(Ti51)*Y(N)*state.rho + screened_rates(k_He4_Ca48_to_n_Ti51)*Y(Ca48)*Y(He4)*state.rho) +
        (screened_rates(k_p_Ti51_to_n_V51)*Y(Ti51)*Y(H1)*state.rho + -screened_rates(k_n_V51_to_p_Ti51)*Y(V51)*Y(N)*state.rho) +
        (screened_rates(k_He4_Ti51_to_n_Cr54)*Y(He4)*Y(Ti51)*state.rho + -screened_rates(k_n_Cr54_to_He4_Ti51)*Y(Cr54)*Y(N)*state.rho) +
        (-screened_rates(k_n_V46_to_p_Ti46)*Y(V46)*Y(N)*state.rho + screened_rates(k_p_Ti46_to_n_V46)*Y(Ti46)*Y(H1)*state.rho) +
        (-screened_rates(k_n_V46_to_He4_Sc43)*Y(V46)*Y(N)*state.rho + screened_rates(k_He4_Sc43_to_n_V46)*Y(He4)*Y(Sc43)*state.rho) +
        (-screened_rates(k_n_V47_to_p_Ti47)*Y(V47)*Y(N)*state.rho + screened_rates(k_p_Ti47_to_n_V47)*Y(Ti47)*Y(H1)*state.rho) +
        (-screened_rates(k_n_V47_to_He4_Sc44)*Y(V47)*Y(N)*state.rho + screened_rates(k_He4_Sc44_to_n_V47)*Y(He4)*Y(Sc44)*state.rho) +
        (-screened_rates(k_n_V48_to_p_Ti48)*Y(V48)*Y(N)*state.rho + screened_rates(k_p_Ti48_to_n_V48)*Y(Ti48)*Y(H1)*state.rho) +
        (-screened_rates(k_n_V48_to_He4_Sc45)*Y(V48)*Y(N)*state.rho + screened_rates(k_He4_Sc45_to_n_V48)*Y(He4)*Y(Sc45)*state.rho) +
        (-screened_rates(k_n_V49_to_p_Ti49)*Y(V49)*Y(N)*state.rho + screened_rates(k_p_Ti49_to_n_V49)*Y(Ti49)*Y(H1)*state.rho) +
        (-screened_rates(k_n_V50_to_p_Ti50)*Y(V50)*Y(N)*state.rho + screened_rates(k_p_Ti50_to_n_V50)*Y(Ti50)*Y(H1)*state.rho) +
        (-screened_rates(k_n_V50_to_He4_Sc47)*Y(V50)*Y(N)*state.rho + screened_rates(k_He4_Sc47_to_n_V50)*Y(He4)*Y(Sc47)*state.rho) +
        (screened_rates(k_p_V50_to_n_Cr50)*Y(V50)*Y(H1)*state.rho + -screened_rates(k_n_Cr50_to_p_V50)*Y(Cr50)*Y(N)*state.rho) +
        (-screened_rates(k_n_V52_to_He4_Sc49)*Y(V52)*Y(N)*state.rho + screened_rates(k_He4_Sc49_to_n_V52)*Y(He4)*Y(Sc49)*state.rho) +
        (screened_rates(k_p_V52_to_n_Cr52)*Y(V52)*Y(H1)*state.rho + -screened_rates(k_n_Cr52_to_p_V52)*Y(Cr52)*Y(N)*state.rho) +
        (screened_rates(k_He4_V52_to_n_Mn55)*Y(He4)*Y(V52)*state.rho + -screened_rates(k_n_Mn55_to_He4_V52)*Y(Mn55)*Y(N)*state.rho) +
        (-screened_rates(k_n_Cr48_to_p_V48)*Y(Cr48)*Y(N)*state.rho + screened_rates(k_p_V48_to_n_Cr48)*Y(V48)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cr48_to_He4_Ti45)*Y(Cr48)*Y(N)*state.rho + screened_rates(k_He4_Ti45_to_n_Cr48)*Y(He4)*Y(Ti45)*state.rho) +
        (-screened_rates(k_n_Cr49_to_p_V49)*Y(Cr49)*Y(N)*state.rho + screened_rates(k_p_V49_to_n_Cr49)*Y(V49)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cr49_to_He4_Ti46)*Y(Cr49)*Y(N)*state.rho + screened_rates(k_He4_Ti46_to_n_Cr49)*Y(He4)*Y(Ti46)*state.rho) +
        (-screened_rates(k_n_Cr50_to_He4_Ti47)*Y(Cr50)*Y(N)*state.rho + screened_rates(k_He4_Ti47_to_n_Cr50)*Y(He4)*Y(Ti47)*state.rho) +
        (-screened_rates(k_n_Cr51_to_p_V51)*Y(Cr51)*Y(N)*state.rho + screened_rates(k_p_V51_to_n_Cr51)*Y(V51)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cr51_to_He4_Ti48)*Y(Cr51)*Y(N)*state.rho + screened_rates(k_He4_Ti48_to_n_Cr51)*Y(He4)*Y(Ti48)*state.rho) +
        (-screened_rates(k_n_Cr53_to_He4_Ti50)*Y(Cr53)*Y(N)*state.rho + screened_rates(k_He4_Ti50_to_n_Cr53)*Y(He4)*Y(Ti50)*state.rho) +
        (-screened_rates(k_n_Mn50_to_p_Cr50)*Y(Mn50)*Y(N)*state.rho + screened_rates(k_p_Cr50_to_n_Mn50)*Y(Cr50)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Mn50_to_He4_V47)*Y(Mn50)*Y(N)*state.rho + screened_rates(k_He4_V47_to_n_Mn50)*Y(He4)*Y(V47)*state.rho) +
        (-screened_rates(k_n_Mn51_to_p_Cr51)*Y(Mn51)*Y(N)*state.rho + screened_rates(k_p_Cr51_to_n_Mn51)*Y(Cr51)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Mn51_to_He4_V48)*Y(Mn51)*Y(N)*state.rho + screened_rates(k_He4_V48_to_n_Mn51)*Y(He4)*Y(V48)*state.rho) +
        (-screened_rates(k_n_Mn52_to_p_Cr52)*Y(Mn52)*Y(N)*state.rho + screened_rates(k_p_Cr52_to_n_Mn52)*Y(Cr52)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Mn52_to_He4_V49)*Y(Mn52)*Y(N)*state.rho + screened_rates(k_He4_V49_to_n_Mn52)*Y(He4)*Y(V49)*state.rho) +
        (-screened_rates(k_n_Mn53_to_p_Cr53)*Y(Mn53)*Y(N)*state.rho + screened_rates(k_p_Cr53_to_n_Mn53)*Y(Cr53)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Mn53_to_He4_V50)*Y(Mn53)*Y(N)*state.rho + screened_rates(k_He4_V50_to_n_Mn53)*Y(He4)*Y(V50)*state.rho) +
        (-screened_rates(k_n_Mn54_to_p_Cr54)*Y(Mn54)*Y(N)*state.rho + screened_rates(k_p_Cr54_to_n_Mn54)*Y(Cr54)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Mn54_to_He4_V51)*Y(Mn54)*Y(N)*state.rho + screened_rates(k_He4_V51_to_n_Mn54)*Y(He4)*Y(V51)*state.rho) +
        (-screened_rates(k_n_Fe52_to_p_Mn52)*Y(Fe52)*Y(N)*state.rho + screened_rates(k_p_Mn52_to_n_Fe52)*Y(Mn52)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Fe52_to_He4_Cr49)*Y(Fe52)*Y(N)*state.rho + screened_rates(k_He4_Cr49_to_n_Fe52)*Y(Cr49)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Fe53_to_p_Mn53)*Y(Fe53)*Y(N)*state.rho + screened_rates(k_p_Mn53_to_n_Fe53)*Y(Mn53)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Fe53_to_He4_Cr50)*Y(Fe53)*Y(N)*state.rho + screened_rates(k_He4_Cr50_to_n_Fe53)*Y(Cr50)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Fe54_to_p_Mn54)*Y(Fe54)*Y(N)*state.rho + screened_rates(k_p_Mn54_to_n_Fe54)*Y(Mn54)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Fe54_to_He4_Cr51)*Y(Fe54)*Y(N)*state.rho + screened_rates(k_He4_Cr51_to_n_Fe54)*Y(Cr51)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Fe55_to_p_Mn55)*Y(Fe55)*Y(N)*state.rho + screened_rates(k_p_Mn55_to_n_Fe55)*Y(Mn55)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Fe55_to_He4_Cr52)*Y(Fe55)*Y(N)*state.rho + screened_rates(k_He4_Cr52_to_n_Fe55)*Y(Cr52)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Fe56_to_He4_Cr53)*Y(Fe56)*Y(N)*state.rho + screened_rates(k_He4_Cr53_to_n_Fe56)*Y(Cr53)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Fe57_to_He4_Cr54)*Y(Fe57)*Y(N)*state.rho + screened_rates(k_He4_Cr54_to_n_Fe57)*Y(Cr54)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Co53_to_p_Fe53)*Y(Co53)*Y(N)*state.rho + screened_rates(k_p_Fe53_to_n_Co53)*Y(Fe53)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Co53_to_He4_Mn50)*Y(Co53)*Y(N)*state.rho + screened_rates(k_He4_Mn50_to_n_Co53)*Y(He4)*Y(Mn50)*state.rho) +
        (-screened_rates(k_n_Co54_to_p_Fe54)*Y(Co54)*Y(N)*state.rho + screened_rates(k_p_Fe54_to_n_Co54)*Y(Fe54)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Co54_to_He4_Mn51)*Y(Co54)*Y(N)*state.rho + screened_rates(k_He4_Mn51_to_n_Co54)*Y(He4)*Y(Mn51)*state.rho) +
        (-screened_rates(k_n_Co55_to_p_Fe55)*Y(Co55)*Y(N)*state.rho + screened_rates(k_p_Fe55_to_n_Co55)*Y(Fe55)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Co55_to_He4_Mn52)*Y(Co55)*Y(N)*state.rho + screened_rates(k_He4_Mn52_to_n_Co55)*Y(He4)*Y(Mn52)*state.rho) +
        (-screened_rates(k_n_Co56_to_p_Fe56)*Y(Co56)*Y(N)*state.rho + screened_rates(k_p_Fe56_to_n_Co56)*Y(Fe56)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Co56_to_He4_Mn53)*Y(Co56)*Y(N)*state.rho + screened_rates(k_He4_Mn53_to_n_Co56)*Y(He4)*Y(Mn53)*state.rho) +
        (-screened_rates(k_n_Co57_to_p_Fe57)*Y(Co57)*Y(N)*state.rho + screened_rates(k_p_Fe57_to_n_Co57)*Y(Fe57)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Co57_to_He4_Mn54)*Y(Co57)*Y(N)*state.rho + screened_rates(k_He4_Mn54_to_n_Co57)*Y(He4)*Y(Mn54)*state.rho) +
        (-screened_rates(k_n_Co58_to_p_Fe58)*Y(Co58)*Y(N)*state.rho + screened_rates(k_p_Fe58_to_n_Co58)*Y(Fe58)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Co58_to_He4_Mn55)*Y(Co58)*Y(N)*state.rho + screened_rates(k_He4_Mn55_to_n_Co58)*Y(He4)*Y(Mn55)*state.rho) +
        (-screened_rates(k_n_Ni56_to_p_Co56)*Y(Ni56)*Y(N)*state.rho + screened_rates(k_p_Co56_to_n_Ni56)*Y(Co56)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ni56_to_He4_Fe53)*Y(Ni56)*Y(N)*state.rho + screened_rates(k_He4_Fe53_to_n_Ni56)*Y(Fe53)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Ni57_to_p_Co57)*Y(Ni57)*Y(N)*state.rho + screened_rates(k_p_Co57_to_n_Ni57)*Y(Co57)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ni57_to_He4_Fe54)*Y(Ni57)*Y(N)*state.rho + screened_rates(k_He4_Fe54_to_n_Ni57)*Y(Fe54)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Ni58_to_p_Co58)*Y(Ni58)*Y(N)*state.rho + screened_rates(k_p_Co58_to_n_Ni58)*Y(Co58)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ni58_to_He4_Fe55)*Y(Ni58)*Y(N)*state.rho + screened_rates(k_He4_Fe55_to_n_Ni58)*Y(Fe55)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Ni59_to_p_Co59)*Y(Ni59)*Y(N)*state.rho + screened_rates(k_p_Co59_to_n_Ni59)*Y(Co59)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ni59_to_He4_Fe56)*Y(Ni59)*Y(N)*state.rho + screened_rates(k_He4_Fe56_to_n_Ni59)*Y(Fe56)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Ni60_to_He4_Fe57)*Y(Ni60)*Y(N)*state.rho + screened_rates(k_He4_Fe57_to_n_Ni60)*Y(Fe57)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Ni61_to_He4_Fe58)*Y(Ni61)*Y(N)*state.rho + screened_rates(k_He4_Fe58_to_n_Ni61)*Y(Fe58)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Cu57_to_p_Ni57)*Y(Cu57)*Y(N)*state.rho + screened_rates(k_p_Ni57_to_n_Cu57)*Y(Ni57)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cu57_to_He4_Co54)*Y(Cu57)*Y(N)*state.rho + screened_rates(k_He4_Co54_to_n_Cu57)*Y(Co54)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Cu58_to_p_Ni58)*Y(Cu58)*Y(N)*state.rho + screened_rates(k_p_Ni58_to_n_Cu58)*Y(Ni58)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cu58_to_He4_Co55)*Y(Cu58)*Y(N)*state.rho + screened_rates(k_He4_Co55_to_n_Cu58)*Y(Co55)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Cu59_to_p_Ni59)*Y(Cu59)*Y(N)*state.rho + screened_rates(k_p_Ni59_to_n_Cu59)*Y(Ni59)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cu59_to_He4_Co56)*Y(Cu59)*Y(N)*state.rho + screened_rates(k_He4_Co56_to_n_Cu59)*Y(Co56)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Cu60_to_p_Ni60)*Y(Cu60)*Y(N)*state.rho + screened_rates(k_p_Ni60_to_n_Cu60)*Y(Ni60)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cu60_to_He4_Co57)*Y(Cu60)*Y(N)*state.rho + screened_rates(k_He4_Co57_to_n_Cu60)*Y(Co57)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Cu61_to_p_Ni61)*Y(Cu61)*Y(N)*state.rho + screened_rates(k_p_Ni61_to_n_Cu61)*Y(Ni61)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cu61_to_He4_Co58)*Y(Cu61)*Y(N)*state.rho + screened_rates(k_He4_Co58_to_n_Cu61)*Y(Co58)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Cu62_to_p_Ni62)*Y(Cu62)*Y(N)*state.rho + screened_rates(k_p_Ni62_to_n_Cu62)*Y(Ni62)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cu62_to_He4_Co59)*Y(Cu62)*Y(N)*state.rho + screened_rates(k_He4_Co59_to_n_Cu62)*Y(Co59)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Cu63_to_p_Ni63)*Y(Cu63)*Y(N)*state.rho + screened_rates(k_p_Ni63_to_n_Cu63)*Y(Ni63)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cu64_to_p_Ni64)*Y(Cu64)*Y(N)*state.rho + screened_rates(k_p_Ni64_to_n_Cu64)*Y(Ni64)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Zn59_to_p_Cu59)*Y(Zn59)*Y(N)*state.rho + screened_rates(k_p_Cu59_to_n_Zn59)*Y(Cu59)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Zn59_to_He4_Ni56)*Y(Zn59)*Y(N)*state.rho + screened_rates(k_He4_Ni56_to_n_Zn59)*Y(He4)*Y(Ni56)*state.rho) +
        (-screened_rates(k_n_Zn60_to_p_Cu60)*Y(Zn60)*Y(N)*state.rho + screened_rates(k_p_Cu60_to_n_Zn60)*Y(Cu60)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Zn60_to_He4_Ni57)*Y(Zn60)*Y(N)*state.rho + screened_rates(k_He4_Ni57_to_n_Zn60)*Y(He4)*Y(Ni57)*state.rho) +
        (-screened_rates(k_n_Zn61_to_p_Cu61)*Y(Zn61)*Y(N)*state.rho + screened_rates(k_p_Cu61_to_n_Zn61)*Y(Cu61)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Zn61_to_He4_Ni58)*Y(Zn61)*Y(N)*state.rho + screened_rates(k_He4_Ni58_to_n_Zn61)*Y(He4)*Y(Ni58)*state.rho) +
        (-screened_rates(k_n_Zn62_to_p_Cu62)*Y(Zn62)*Y(N)*state.rho + screened_rates(k_p_Cu62_to_n_Zn62)*Y(Cu62)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Zn62_to_He4_Ni59)*Y(Zn62)*Y(N)*state.rho + screened_rates(k_He4_Ni59_to_n_Zn62)*Y(He4)*Y(Ni59)*state.rho) +
        (-screened_rates(k_n_Zn63_to_p_Cu63)*Y(Zn63)*Y(N)*state.rho + screened_rates(k_p_Cu63_to_n_Zn63)*Y(Cu63)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Zn63_to_He4_Ni60)*Y(Zn63)*Y(N)*state.rho + screened_rates(k_He4_Ni60_to_n_Zn63)*Y(He4)*Y(Ni60)*state.rho) +
        (-screened_rates(k_n_Zn64_to_p_Cu64)*Y(Zn64)*Y(N)*state.rho + screened_rates(k_p_Cu64_to_n_Zn64)*Y(Cu64)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Zn64_to_He4_Ni61)*Y(Zn64)*Y(N)*state.rho + screened_rates(k_He4_Ni61_to_n_Zn64)*Y(He4)*Y(Ni61)*state.rho) +
        (-screened_rates(k_n_Zn65_to_p_Cu65)*Y(Zn65)*Y(N)*state.rho + screened_rates(k_p_Cu65_to_n_Zn65)*Y(Cu65)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Zn65_to_He4_Ni62)*Y(Zn65)*Y(N)*state.rho + screened_rates(k_He4_Ni62_to_n_Zn65)*Y(He4)*Y(Ni62)*state.rho) +
        (-screened_rates(k_n_Zn66_to_He4_Ni63)*Y(Zn66)*Y(N)*state.rho + screened_rates(k_He4_Ni63_to_n_Zn66)*Y(He4)*Y(Ni63)*state.rho) +
        (-screened_rates(k_n_Ga62_to_p_Zn62)*Y(Ga62)*Y(N)*state.rho + screened_rates(k_p_Zn62_to_n_Ga62)*Y(Zn62)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ga62_to_He4_Cu59)*Y(Ga62)*Y(N)*state.rho + screened_rates(k_He4_Cu59_to_n_Ga62)*Y(Cu59)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Ga63_to_p_Zn63)*Y(Ga63)*Y(N)*state.rho + screened_rates(k_p_Zn63_to_n_Ga63)*Y(Zn63)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ga63_to_He4_Cu60)*Y(Ga63)*Y(N)*state.rho + screened_rates(k_He4_Cu60_to_n_Ga63)*Y(Cu60)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Ga64_to_p_Zn64)*Y(Ga64)*Y(N)*state.rho + screened_rates(k_p_Zn64_to_n_Ga64)*Y(Zn64)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ga64_to_He4_Cu61)*Y(Ga64)*Y(N)*state.rho + screened_rates(k_He4_Cu61_to_n_Ga64)*Y(Cu61)*Y(He4)*state.rho) +
        (-screened_rates(k_n_Ge63_to_p_Ga63)*Y(Ge63)*Y(N)*state.rho + screened_rates(k_p_Ga63_to_n_Ge63)*Y(Ga63)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ge63_to_He4_Zn60)*Y(Ge63)*Y(N)*state.rho + screened_rates(k_He4_Zn60_to_n_Ge63)*Y(He4)*Y(Zn60)*state.rho) +
        (-screened_rates(k_n_Ge64_to_p_Ga64)*Y(Ge64)*Y(N)*state.rho + screened_rates(k_p_Ga64_to_n_Ge64)*Y(Ga64)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ge64_to_He4_Zn61)*Y(Ge64)*Y(N)*state.rho + screened_rates(k_He4_Zn61_to_n_Ge64)*Y(He4)*Y(Zn61)*state.rho) +
        (screened_rates(k_d_Li7_to_n_He4_He4)*Y(Li7)*Y(H2)*state.rho + -0.5*screened_rates(k_n_He4_He4_to_d_Li7)*amrex::Math::powi<2>(Y(He4))*Y(N)*amrex::Math::powi<2>(state.rho)) +
        (-screened_rates(k_n_B8_to_p_He4_He4)*Y(B8)*Y(N)*state.rho + 0.5*screened_rates(k_p_He4_He4_to_n_B8)*amrex::Math::powi<2>(Y(He4))*Y(H1)*amrex::Math::powi<2>(state.rho)) +
        (screened_rates(k_He3_Li7_to_n_p_He4_He4)*Y(He3)*Y(Li7)*state.rho + -0.5*screened_rates(k_n_p_He4_He4_to_He3_Li7)*amrex::Math::powi<2>(Y(He4))*Y(N)*Y(H1)*amrex::Math::powi<3>(state.rho)) +
        (-screened_rates(k_n_p_He4_to_Li6)*Y(He4)*Y(N)*Y(H1)*amrex::Math::powi<2>(state.rho) + screened_rates(k_Li6_to_n_p_He4)*Y(Li6)) +
        (-0.5*screened_rates(k_n_He4_He4_to_Be9)*amrex::Math::powi<2>(Y(He4))*Y(N)*amrex::Math::powi<2>(state.rho) + screened_rates(k_Be9_to_n_He4_He4)*Y(Be9)) +
        (-0.5*screened_rates(k_n_p_p_to_p_d)*Y(N)*amrex::Math::powi<2>(Y(H1))*amrex::Math::powi<2>(state.rho) + screened_rates(k_p_d_to_n_p_p)*Y(H2)*Y(H1)*state.rho) +
        (-0.5*screened_rates(k_n_p_He4_He4_to_p_Be9)*amrex::Math::powi<2>(Y(He4))*Y(N)*Y(H1)*amrex::Math::powi<3>(state.rho) + screened_rates(k_p_Be9_to_n_p_He4_He4)*Y(Be9)*Y(H1)*state.rho) +
        (-screened_rates(k_n_to_p)*Y(N) + screened_rates(k_p_to_n)*Y(H1));

    ydot_nuc(H1) =
        screened_rates(k_Zn59_to_p_Ni58_weak_wc12)*Y(Zn59) +
        (-screened_rates(k_n_p_to_d)*Y(N)*Y(H1)*state.rho + screened_rates(k_d_to_n_p)*Y(H2)) +
        -screened_rates(k_p_p_to_d_weak_bet_pos_)*amrex::Math::powi<2>(Y(H1))*state.rho +
        -screened_rates(k_p_p_to_d_weak_electron_capture)*amrex::Math::powi<2>(Y(H1))*amrex::Math::powi<2>(state.rho)*state.y_e +
        (-screened_rates(k_p_d_to_He3)*Y(H2)*Y(H1)*state.rho + screened_rates(k_He3_to_p_d)*Y(He3)) +
        -screened_rates(k_p_He3_to_He4_weak_bet_pos_)*Y(He3)*Y(H1)*state.rho +
        (-screened_rates(k_p_Li6_to_Be7)*Y(Li6)*Y(H1)*state.rho + screened_rates(k_Be7_to_p_Li6)*Y(Be7)) +
        (-screened_rates(k_p_Be7_to_B8)*Y(Be7)*Y(H1)*state.rho + screened_rates(k_B8_to_p_Be7)*Y(B8)) +
        (-screened_rates(k_p_Be9_to_B10)*Y(Be9)*Y(H1)*state.rho + screened_rates(k_B10_to_p_Be9)*Y(B10)) +
        (-screened_rates(k_p_B11_to_C12)*Y(B11)*Y(H1)*state.rho + screened_rates(k_C12_to_p_B11)*Y(C12)) +
        (-screened_rates(k_p_C12_to_N13)*Y(C12)*Y(H1)*state.rho + screened_rates(k_N13_to_p_C12)*Y(N13)) +
        (-screened_rates(k_p_C13_to_N14)*Y(C13)*Y(H1)*state.rho + screened_rates(k_N14_to_p_C13)*Y(N14)) +
        (-screened_rates(k_p_C14_to_N15)*Y(C14)*Y(H1)*state.rho + screened_rates(k_N15_to_p_C14)*Y(N15)) +
        (-screened_rates(k_p_N13_to_O14)*Y(N13)*Y(H1)*state.rho + screened_rates(k_O14_to_p_N13)*Y(O14)) +
        (-screened_rates(k_p_N14_to_O15)*Y(N14)*Y(H1)*state.rho + screened_rates(k_O15_to_p_N14)*Y(O15)) +
        (-screened_rates(k_p_N15_to_O16)*Y(N15)*Y(H1)*state.rho + screened_rates(k_O16_to_p_N15)*Y(O16)) +
        (-screened_rates(k_p_O16_to_F17)*Y(O16)*Y(H1)*state.rho + screened_rates(k_F17_to_p_O16)*Y(F17)) +
        (-screened_rates(k_p_O17_to_F18)*Y(O17)*Y(H1)*state.rho + screened_rates(k_F18_to_p_O17)*Y(F18)) +
        (-screened_rates(k_p_O18_to_F19)*Y(O18)*Y(H1)*state.rho + screened_rates(k_F19_to_p_O18)*Y(F19)) +
        (-screened_rates(k_p_F17_to_Ne18)*Y(F17)*Y(H1)*state.rho + screened_rates(k_Ne18_to_p_F17)*Y(Ne18)) +
        (-screened_rates(k_p_F18_to_Ne19)*Y(F18)*Y(H1)*state.rho + screened_rates(k_Ne19_to_p_F18)*Y(Ne19)) +
        (-screened_rates(k_p_F19_to_Ne20)*Y(F19)*Y(H1)*state.rho + screened_rates(k_Ne20_to_p_F19)*Y(Ne20)) +
        (-screened_rates(k_p_Ne20_to_Na21)*Y(Ne20)*Y(H1)*state.rho + screened_rates(k_Na21_to_p_Ne20)*Y(Na21)) +
        (-screened_rates(k_p_Ne21_to_Na22)*Y(Ne21)*Y(H1)*state.rho + screened_rates(k_Na22_to_p_Ne21)*Y(Na22)) +
        (-screened_rates(k_p_Ne22_to_Na23)*Y(Ne22)*Y(H1)*state.rho + screened_rates(k_Na23_to_p_Ne22)*Y(Na23)) +
        (-screened_rates(k_p_Na22_to_Mg23)*Y(Na22)*Y(H1)*state.rho + screened_rates(k_Mg23_to_p_Na22)*Y(Mg23)) +
        (-screened_rates(k_p_Na23_to_Mg24)*Y(Na23)*Y(H1)*state.rho + screened_rates(k_Mg24_to_p_Na23)*Y(Mg24)) +
        (-screened_rates(k_p_Mg24_to_Al25)*Y(Mg24)*Y(H1)*state.rho + screened_rates(k_Al25_to_p_Mg24)*Y(Al25)) +
        (-screened_rates(k_p_Mg25_to_Al26)*Y(Mg25)*Y(H1)*state.rho + screened_rates(k_Al26_to_p_Mg25)*Y(Al26)) +
        (-screened_rates(k_p_Mg26_to_Al27)*Y(Mg26)*Y(H1)*state.rho + screened_rates(k_Al27_to_p_Mg26)*Y(Al27)) +
        (-screened_rates(k_p_Al27_to_Si28)*Y(Al27)*Y(H1)*state.rho + screened_rates(k_Si28_to_p_Al27)*Y(Si28)) +
        (-screened_rates(k_p_Si28_to_P29)*Y(Si28)*Y(H1)*state.rho + screened_rates(k_P29_to_p_Si28)*Y(P29)) +
        (-screened_rates(k_p_Si29_to_P30)*Y(Si29)*Y(H1)*state.rho + screened_rates(k_P30_to_p_Si29)*Y(P30)) +
        (-screened_rates(k_p_Si30_to_P31)*Y(Si30)*Y(H1)*state.rho + screened_rates(k_P31_to_p_Si30)*Y(P31)) +
        (-screened_rates(k_p_Si31_to_P32)*Y(Si31)*Y(H1)*state.rho + screened_rates(k_P32_to_p_Si31)*Y(P32)) +
        (-screened_rates(k_p_Si32_to_P33)*Y(Si32)*Y(H1)*state.rho + screened_rates(k_P33_to_p_Si32)*Y(P33)) +
        (-screened_rates(k_p_P31_to_S32)*Y(P31)*Y(H1)*state.rho + screened_rates(k_S32_to_p_P31)*Y(S32)) +
        (-screened_rates(k_p_P32_to_S33)*Y(P32)*Y(H1)*state.rho + screened_rates(k_S33_to_p_P32)*Y(S33)) +
        (-screened_rates(k_p_P33_to_S34)*Y(P33)*Y(H1)*state.rho + screened_rates(k_S34_to_p_P33)*Y(S34)) +
        (-screened_rates(k_p_S32_to_Cl33)*Y(S32)*Y(H1)*state.rho + screened_rates(k_Cl33_to_p_S32)*Y(Cl33)) +
        (-screened_rates(k_p_S33_to_Cl34)*Y(S33)*Y(H1)*state.rho + screened_rates(k_Cl34_to_p_S33)*Y(Cl34)) +
        (-screened_rates(k_p_S34_to_Cl35)*Y(S34)*Y(H1)*state.rho + screened_rates(k_Cl35_to_p_S34)*Y(Cl35)) +
        (-screened_rates(k_p_S35_to_Cl36)*Y(S35)*Y(H1)*state.rho + screened_rates(k_Cl36_to_p_S35)*Y(Cl36)) +
        (-screened_rates(k_p_S36_to_Cl37)*Y(S36)*Y(H1)*state.rho + screened_rates(k_Cl37_to_p_S36)*Y(Cl37)) +
        (-screened_rates(k_p_Cl35_to_Ar36)*Y(Cl35)*Y(H1)*state.rho + screened_rates(k_Ar36_to_p_Cl35)*Y(Ar36)) +
        (-screened_rates(k_p_Cl36_to_Ar37)*Y(Cl36)*Y(H1)*state.rho + screened_rates(k_Ar37_to_p_Cl36)*Y(Ar37)) +
        (-screened_rates(k_p_Cl37_to_Ar38)*Y(Cl37)*Y(H1)*state.rho + screened_rates(k_Ar38_to_p_Cl37)*Y(Ar38)) +
        (-screened_rates(k_p_Ar36_to_K37)*Y(Ar36)*Y(H1)*state.rho + screened_rates(k_K37_to_p_Ar36)*Y(K37)) +
        (-screened_rates(k_p_Ar37_to_K38)*Y(Ar37)*Y(H1)*state.rho + screened_rates(k_K38_to_p_Ar37)*Y(K38)) +
        (-screened_rates(k_p_Ar38_to_K39)*Y(Ar38)*Y(H1)*state.rho + screened_rates(k_K39_to_p_Ar38)*Y(K39)) +
        (-screened_rates(k_p_Ar39_to_K40)*Y(Ar39)*Y(H1)*state.rho + screened_rates(k_K40_to_p_Ar39)*Y(K40)) +
        (-screened_rates(k_p_Ar40_to_K41)*Y(Ar40)*Y(H1)*state.rho + screened_rates(k_K41_to_p_Ar40)*Y(K41)) +
        (-screened_rates(k_p_K39_to_Ca40)*Y(K39)*Y(H1)*state.rho + screened_rates(k_Ca40_to_p_K39)*Y(Ca40)) +
        (-screened_rates(k_p_K40_to_Ca41)*Y(K40)*Y(H1)*state.rho + screened_rates(k_Ca41_to_p_K40)*Y(Ca41)) +
        (-screened_rates(k_p_K41_to_Ca42)*Y(K41)*Y(H1)*state.rho + screened_rates(k_Ca42_to_p_K41)*Y(Ca42)) +
        (-screened_rates(k_p_Ca42_to_Sc43)*Y(Ca42)*Y(H1)*state.rho + screened_rates(k_Sc43_to_p_Ca42)*Y(Sc43)) +
        (-screened_rates(k_p_Ca43_to_Sc44)*Y(Ca43)*Y(H1)*state.rho + screened_rates(k_Sc44_to_p_Ca43)*Y(Sc44)) +
        (-screened_rates(k_p_Ca44_to_Sc45)*Y(Ca44)*Y(H1)*state.rho + screened_rates(k_Sc45_to_p_Ca44)*Y(Sc45)) +
        (-screened_rates(k_p_Ca45_to_Sc46)*Y(Ca45)*Y(H1)*state.rho + screened_rates(k_Sc46_to_p_Ca45)*Y(Sc46)) +
        (-screened_rates(k_p_Ca46_to_Sc47)*Y(Ca46)*Y(H1)*state.rho + screened_rates(k_Sc47_to_p_Ca46)*Y(Sc47)) +
        (-screened_rates(k_p_Ca47_to_Sc48)*Y(Ca47)*Y(H1)*state.rho + screened_rates(k_Sc48_to_p_Ca47)*Y(Sc48)) +
        (-screened_rates(k_p_Ca48_to_Sc49)*Y(Ca48)*Y(H1)*state.rho + screened_rates(k_Sc49_to_p_Ca48)*Y(Sc49)) +
        (-screened_rates(k_p_Sc43_to_Ti44)*Y(Sc43)*Y(H1)*state.rho + screened_rates(k_Ti44_to_p_Sc43)*Y(Ti44)) +
        (-screened_rates(k_p_Sc44_to_Ti45)*Y(Sc44)*Y(H1)*state.rho + screened_rates(k_Ti45_to_p_Sc44)*Y(Ti45)) +
        (-screened_rates(k_p_Sc45_to_Ti46)*Y(Sc45)*Y(H1)*state.rho + screened_rates(k_Ti46_to_p_Sc45)*Y(Ti46)) +
        (-screened_rates(k_p_Sc46_to_Ti47)*Y(Sc46)*Y(H1)*state.rho + screened_rates(k_Ti47_to_p_Sc46)*Y(Ti47)) +
        (-screened_rates(k_p_Sc47_to_Ti48)*Y(Sc47)*Y(H1)*state.rho + screened_rates(k_Ti48_to_p_Sc47)*Y(Ti48)) +
        (-screened_rates(k_p_Sc48_to_Ti49)*Y(Sc48)*Y(H1)*state.rho + screened_rates(k_Ti49_to_p_Sc48)*Y(Ti49)) +
        (-screened_rates(k_p_Sc49_to_Ti50)*Y(Sc49)*Y(H1)*state.rho + screened_rates(k_Ti50_to_p_Sc49)*Y(Ti50)) +
        (-screened_rates(k_p_Ti45_to_V46)*Y(Ti45)*Y(H1)*state.rho + screened_rates(k_V46_to_p_Ti45)*Y(V46)) +
        (-screened_rates(k_p_Ti46_to_V47)*Y(Ti46)*Y(H1)*state.rho + screened_rates(k_V47_to_p_Ti46)*Y(V47)) +
        (-screened_rates(k_p_Ti47_to_V48)*Y(Ti47)*Y(H1)*state.rho + screened_rates(k_V48_to_p_Ti47)*Y(V48)) +
        (-screened_rates(k_p_Ti48_to_V49)*Y(Ti48)*Y(H1)*state.rho + screened_rates(k_V49_to_p_Ti48)*Y(V49)) +
        (-screened_rates(k_p_Ti49_to_V50)*Y(Ti49)*Y(H1)*state.rho + screened_rates(k_V50_to_p_Ti49)*Y(V50)) +
        (-screened_rates(k_p_Ti50_to_V51)*Y(Ti50)*Y(H1)*state.rho + screened_rates(k_V51_to_p_Ti50)*Y(V51)) +
        (-screened_rates(k_p_Ti51_to_V52)*Y(Ti51)*Y(H1)*state.rho + screened_rates(k_V52_to_p_Ti51)*Y(V52)) +
        (-screened_rates(k_p_V47_to_Cr48)*Y(V47)*Y(H1)*state.rho + screened_rates(k_Cr48_to_p_V47)*Y(Cr48)) +
        (-screened_rates(k_p_V48_to_Cr49)*Y(V48)*Y(H1)*state.rho + screened_rates(k_Cr49_to_p_V48)*Y(Cr49)) +
        (-screened_rates(k_p_V49_to_Cr50)*Y(V49)*Y(H1)*state.rho + screened_rates(k_Cr50_to_p_V49)*Y(Cr50)) +
        (-screened_rates(k_p_V50_to_Cr51)*Y(V50)*Y(H1)*state.rho + screened_rates(k_Cr51_to_p_V50)*Y(Cr51)) +
        (-screened_rates(k_p_V51_to_Cr52)*Y(V51)*Y(H1)*state.rho + screened_rates(k_Cr52_to_p_V51)*Y(Cr52)) +
        (-screened_rates(k_p_V52_to_Cr53)*Y(V52)*Y(H1)*state.rho + screened_rates(k_Cr53_to_p_V52)*Y(Cr53)) +
        (-screened_rates(k_p_Cr49_to_Mn50)*Y(Cr49)*Y(H1)*state.rho + screened_rates(k_Mn50_to_p_Cr49)*Y(Mn50)) +
        (-screened_rates(k_p_Cr50_to_Mn51)*Y(Cr50)*Y(H1)*state.rho + screened_rates(k_Mn51_to_p_Cr50)*Y(Mn51)) +
        (-screened_rates(k_p_Cr51_to_Mn52)*Y(Cr51)*Y(H1)*state.rho + screened_rates(k_Mn52_to_p_Cr51)*Y(Mn52)) +
        (-screened_rates(k_p_Cr52_to_Mn53)*Y(Cr52)*Y(H1)*state.rho + screened_rates(k_Mn53_to_p_Cr52)*Y(Mn53)) +
        (-screened_rates(k_p_Cr53_to_Mn54)*Y(Cr53)*Y(H1)*state.rho + screened_rates(k_Mn54_to_p_Cr53)*Y(Mn54)) +
        (-screened_rates(k_p_Cr54_to_Mn55)*Y(Cr54)*Y(H1)*state.rho + screened_rates(k_Mn55_to_p_Cr54)*Y(Mn55)) +
        (-screened_rates(k_p_Mn51_to_Fe52)*Y(Mn51)*Y(H1)*state.rho + screened_rates(k_Fe52_to_p_Mn51)*Y(Fe52)) +
        (-screened_rates(k_p_Mn52_to_Fe53)*Y(Mn52)*Y(H1)*state.rho + screened_rates(k_Fe53_to_p_Mn52)*Y(Fe53)) +
        (-screened_rates(k_p_Mn53_to_Fe54)*Y(Mn53)*Y(H1)*state.rho + screened_rates(k_Fe54_to_p_Mn53)*Y(Fe54)) +
        (-screened_rates(k_p_Mn54_to_Fe55)*Y(Mn54)*Y(H1)*state.rho + screened_rates(k_Fe55_to_p_Mn54)*Y(Fe55)) +
        (-screened_rates(k_p_Mn55_to_Fe56)*Y(Mn55)*Y(H1)*state.rho + screened_rates(k_Fe56_to_p_Mn55)*Y(Fe56)) +
        (-screened_rates(k_p_Fe52_to_Co53)*Y(Fe52)*Y(H1)*state.rho + screened_rates(k_Co53_to_p_Fe52)*Y(Co53)) +
        (-screened_rates(k_p_Fe53_to_Co54)*Y(Fe53)*Y(H1)*state.rho + screened_rates(k_Co54_to_p_Fe53)*Y(Co54)) +
        (-screened_rates(k_p_Fe54_to_Co55)*Y(Fe54)*Y(H1)*state.rho + screened_rates(k_Co55_to_p_Fe54)*Y(Co55)) +
        (-screened_rates(k_p_Fe55_to_Co56)*Y(Fe55)*Y(H1)*state.rho + screened_rates(k_Co56_to_p_Fe55)*Y(Co56)) +
        (-screened_rates(k_p_Fe56_to_Co57)*Y(Fe56)*Y(H1)*state.rho + screened_rates(k_Co57_to_p_Fe56)*Y(Co57)) +
        (-screened_rates(k_p_Fe57_to_Co58)*Y(Fe57)*Y(H1)*state.rho + screened_rates(k_Co58_to_p_Fe57)*Y(Co58)) +
        (-screened_rates(k_p_Fe58_to_Co59)*Y(Fe58)*Y(H1)*state.rho + screened_rates(k_Co59_to_p_Fe58)*Y(Co59)) +
        (-screened_rates(k_p_Co55_to_Ni56)*Y(Co55)*Y(H1)*state.rho + screened_rates(k_Ni56_to_p_Co55)*Y(Ni56)) +
        (-screened_rates(k_p_Co56_to_Ni57)*Y(Co56)*Y(H1)*state.rho + screened_rates(k_Ni57_to_p_Co56)*Y(Ni57)) +
        (-screened_rates(k_p_Co57_to_Ni58)*Y(Co57)*Y(H1)*state.rho + screened_rates(k_Ni58_to_p_Co57)*Y(Ni58)) +
        (-screened_rates(k_p_Co58_to_Ni59)*Y(Co58)*Y(H1)*state.rho + screened_rates(k_Ni59_to_p_Co58)*Y(Ni59)) +
        (-screened_rates(k_p_Co59_to_Ni60)*Y(Co59)*Y(H1)*state.rho + screened_rates(k_Ni60_to_p_Co59)*Y(Ni60)) +
        (-screened_rates(k_p_Ni56_to_Cu57)*Y(Ni56)*Y(H1)*state.rho + screened_rates(k_Cu57_to_p_Ni56)*Y(Cu57)) +
        (-screened_rates(k_p_Ni57_to_Cu58)*Y(Ni57)*Y(H1)*state.rho + screened_rates(k_Cu58_to_p_Ni57)*Y(Cu58)) +
        (-screened_rates(k_p_Ni58_to_Cu59)*Y(Ni58)*Y(H1)*state.rho + screened_rates(k_Cu59_to_p_Ni58)*Y(Cu59)) +
        (-screened_rates(k_p_Ni59_to_Cu60)*Y(Ni59)*Y(H1)*state.rho + screened_rates(k_Cu60_to_p_Ni59)*Y(Cu60)) +
        (-screened_rates(k_p_Ni60_to_Cu61)*Y(Ni60)*Y(H1)*state.rho + screened_rates(k_Cu61_to_p_Ni60)*Y(Cu61)) +
        (-screened_rates(k_p_Ni61_to_Cu62)*Y(Ni61)*Y(H1)*state.rho + screened_rates(k_Cu62_to_p_Ni61)*Y(Cu62)) +
        (-screened_rates(k_p_Ni62_to_Cu63)*Y(Ni62)*Y(H1)*state.rho + screened_rates(k_Cu63_to_p_Ni62)*Y(Cu63)) +
        (-screened_rates(k_p_Ni63_to_Cu64)*Y(Ni63)*Y(H1)*state.rho + screened_rates(k_Cu64_to_p_Ni63)*Y(Cu64)) +
        (-screened_rates(k_p_Ni64_to_Cu65)*Y(Ni64)*Y(H1)*state.rho + screened_rates(k_Cu65_to_p_Ni64)*Y(Cu65)) +
        (-screened_rates(k_p_Cu58_to_Zn59)*Y(Cu58)*Y(H1)*state.rho + screened_rates(k_Zn59_to_p_Cu58)*Y(Zn59)) +
        (-screened_rates(k_p_Cu59_to_Zn60)*Y(Cu59)*Y(H1)*state.rho + screened_rates(k_Zn60_to_p_Cu59)*Y(Zn60)) +
        (-screened_rates(k_p_Cu60_to_Zn61)*Y(Cu60)*Y(H1)*state.rho + screened_rates(k_Zn61_to_p_Cu60)*Y(Zn61)) +
        (-screened_rates(k_p_Cu61_to_Zn62)*Y(Cu61)*Y(H1)*state.rho + screened_rates(k_Zn62_to_p_Cu61)*Y(Zn62)) +
        (-screened_rates(k_p_Cu62_to_Zn63)*Y(Cu62)*Y(H1)*state.rho + screened_rates(k_Zn63_to_p_Cu62)*Y(Zn63)) +
        (-screened_rates(k_p_Cu63_to_Zn64)*Y(Cu63)*Y(H1)*state.rho + screened_rates(k_Zn64_to_p_Cu63)*Y(Zn64)) +
        (-screened_rates(k_p_Cu64_to_Zn65)*Y(Cu64)*Y(H1)*state.rho + screened_rates(k_Zn65_to_p_Cu64)*Y(Zn65)) +
        (-screened_rates(k_p_Cu65_to_Zn66)*Y(Cu65)*Y(H1)*state.rho + screened_rates(k_Zn66_to_p_Cu65)*Y(Zn66)) +
        (-screened_rates(k_p_Zn61_to_Ga62)*Y(Zn61)*Y(H1)*state.rho + screened_rates(k_Ga62_to_p_Zn61)*Y(Ga62)) +
        (-screened_rates(k_p_Zn62_to_Ga63)*Y(Zn62)*Y(H1)*state.rho + screened_rates(k_Ga63_to_p_Zn62)*Y(Ga63)) +
        (-screened_rates(k_p_Zn63_to_Ga64)*Y(Zn63)*Y(H1)*state.rho + screened_rates(k_Ga64_to_p_Zn63)*Y(Ga64)) +
        (-screened_rates(k_p_Ga62_to_Ge63)*Y(Ga62)*Y(H1)*state.rho + screened_rates(k_Ge63_to_p_Ga62)*Y(Ge63)) +
        (-screened_rates(k_p_Ga63_to_Ge64)*Y(Ga63)*Y(H1)*state.rho + screened_rates(k_Ge64_to_p_Ga63)*Y(Ge64)) +
        (screened_rates(k_d_He3_to_p_He4)*Y(He3)*Y(H2)*state.rho + -screened_rates(k_p_He4_to_d_He3)*Y(He4)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Li6_to_He4_He3)*Y(Li6)*Y(H1)*state.rho + screened_rates(k_He4_He3_to_p_Li6)*Y(He3)*Y(He4)*state.rho) +
        (screened_rates(k_d_Li6_to_p_Li7)*Y(Li6)*Y(H2)*state.rho + -screened_rates(k_p_Li7_to_d_Li6)*Y(Li7)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Li7_to_He4_He4)*Y(Li7)*Y(H1)*state.rho + 0.5*screened_rates(k_He4_He4_to_p_Li7)*amrex::Math::powi<2>(Y(He4))*state.rho) +
        (screened_rates(k_n_Be7_to_p_Li7)*Y(Be7)*Y(N)*state.rho + -screened_rates(k_p_Li7_to_n_Be7)*Y(Li7)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Be9_to_He4_Li6)*Y(Be9)*Y(H1)*state.rho + screened_rates(k_He4_Li6_to_p_Be9)*Y(He4)*Y(Li6)*state.rho) +
        (-screened_rates(k_p_B10_to_He4_Be7)*Y(B10)*Y(H1)*state.rho + screened_rates(k_He4_Be7_to_p_B10)*Y(Be7)*Y(He4)*state.rho) +
        (screened_rates(k_He4_B10_to_p_C13)*Y(B10)*Y(He4)*state.rho + -screened_rates(k_p_C13_to_He4_B10)*Y(C13)*Y(H1)*state.rho) +
        (screened_rates(k_He4_B11_to_p_C14)*Y(B11)*Y(He4)*state.rho + -screened_rates(k_p_C14_to_He4_B11)*Y(C14)*Y(H1)*state.rho) +
        (0.5*screened_rates(k_C12_C12_to_p_Na23)*amrex::Math::powi<2>(Y(C12))*state.rho + -screened_rates(k_p_Na23_to_C12_C12)*Y(Na23)*Y(H1)*state.rho) +
        (screened_rates(k_n_N13_to_p_C13)*Y(N13)*Y(N)*state.rho + -screened_rates(k_p_C13_to_n_N13)*Y(C13)*Y(H1)*state.rho) +
        (screened_rates(k_He4_N13_to_p_O16)*Y(He4)*Y(N13)*state.rho + -screened_rates(k_p_O16_to_He4_N13)*Y(O16)*Y(H1)*state.rho) +
        (screened_rates(k_n_N14_to_p_C14)*Y(N14)*Y(N)*state.rho + -screened_rates(k_p_C14_to_n_N14)*Y(C14)*Y(H1)*state.rho) +
        (-screened_rates(k_p_N15_to_He4_C12)*Y(N15)*Y(H1)*state.rho + screened_rates(k_He4_C12_to_p_N15)*Y(C12)*Y(He4)*state.rho) +
        (screened_rates(k_n_O14_to_p_N14)*Y(O14)*Y(N)*state.rho + -screened_rates(k_p_N14_to_n_O14)*Y(N14)*Y(H1)*state.rho) +
        (screened_rates(k_He4_O14_to_p_F17)*Y(He4)*Y(O14)*state.rho + -screened_rates(k_p_F17_to_He4_O14)*Y(F17)*Y(H1)*state.rho) +
        (screened_rates(k_n_O15_to_p_N15)*Y(O15)*Y(N)*state.rho + -screened_rates(k_p_N15_to_n_O15)*Y(N15)*Y(H1)*state.rho) +
        (screened_rates(k_C12_O16_to_p_Al27)*Y(C12)*Y(O16)*state.rho + -screened_rates(k_p_Al27_to_C12_O16)*Y(Al27)*Y(H1)*state.rho) +
        (0.5*screened_rates(k_O16_O16_to_p_P31)*amrex::Math::powi<2>(Y(O16))*state.rho + -screened_rates(k_p_P31_to_O16_O16)*Y(P31)*Y(H1)*state.rho) +
        (-screened_rates(k_p_O17_to_He4_N14)*Y(O17)*Y(H1)*state.rho + screened_rates(k_He4_N14_to_p_O17)*Y(He4)*Y(N14)*state.rho) +
        (-screened_rates(k_p_O18_to_He4_N15)*Y(O18)*Y(H1)*state.rho + screened_rates(k_He4_N15_to_p_O18)*Y(He4)*Y(N15)*state.rho) +
        (screened_rates(k_n_F17_to_p_O17)*Y(F17)*Y(N)*state.rho + -screened_rates(k_p_O17_to_n_F17)*Y(O17)*Y(H1)*state.rho) +
        (screened_rates(k_He4_F17_to_p_Ne20)*Y(F17)*Y(He4)*state.rho + -screened_rates(k_p_Ne20_to_He4_F17)*Y(Ne20)*Y(H1)*state.rho) +
        (screened_rates(k_n_F18_to_p_O18)*Y(F18)*Y(N)*state.rho + -screened_rates(k_p_O18_to_n_F18)*Y(O18)*Y(H1)*state.rho) +
        (-screened_rates(k_p_F18_to_He4_O15)*Y(F18)*Y(H1)*state.rho + screened_rates(k_He4_O15_to_p_F18)*Y(He4)*Y(O15)*state.rho) +
        (screened_rates(k_He4_F18_to_p_Ne21)*Y(F18)*Y(He4)*state.rho + -screened_rates(k_p_Ne21_to_He4_F18)*Y(Ne21)*Y(H1)*state.rho) +
        (-screened_rates(k_p_F19_to_He4_O16)*Y(F19)*Y(H1)*state.rho + screened_rates(k_He4_O16_to_p_F19)*Y(He4)*Y(O16)*state.rho) +
        (screened_rates(k_He4_F19_to_p_Ne22)*Y(F19)*Y(He4)*state.rho + -screened_rates(k_p_Ne22_to_He4_F19)*Y(Ne22)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ne18_to_p_F18)*Y(Ne18)*Y(N)*state.rho + -screened_rates(k_p_F18_to_n_Ne18)*Y(F18)*Y(H1)*state.rho) +
        (screened_rates(k_He4_Ne18_to_p_Na21)*Y(He4)*Y(Ne18)*state.rho + -screened_rates(k_p_Na21_to_He4_Ne18)*Y(Na21)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ne19_to_p_F19)*Y(Ne19)*Y(N)*state.rho + -screened_rates(k_p_F19_to_n_Ne19)*Y(F19)*Y(H1)*state.rho) +
        (screened_rates(k_He4_Ne19_to_p_Na22)*Y(He4)*Y(Ne19)*state.rho + -screened_rates(k_p_Na22_to_He4_Ne19)*Y(Na22)*Y(H1)*state.rho) +
        (screened_rates(k_C12_Ne20_to_p_P31)*Y(C12)*Y(Ne20)*state.rho + -screened_rates(k_p_P31_to_C12_Ne20)*Y(P31)*Y(H1)*state.rho) +
        (screened_rates(k_n_Na21_to_p_Ne21)*Y(Na21)*Y(N)*state.rho + -screened_rates(k_p_Ne21_to_n_Na21)*Y(Ne21)*Y(H1)*state.rho) +
        (screened_rates(k_He4_Na21_to_p_Mg24)*Y(He4)*Y(Na21)*state.rho + -screened_rates(k_p_Mg24_to_He4_Na21)*Y(Mg24)*Y(H1)*state.rho) +
        (screened_rates(k_n_Na22_to_p_Ne22)*Y(Na22)*Y(N)*state.rho + -screened_rates(k_p_Ne22_to_n_Na22)*Y(Ne22)*Y(H1)*state.rho) +
        (screened_rates(k_He4_Na22_to_p_Mg25)*Y(He4)*Y(Na22)*state.rho + -screened_rates(k_p_Mg25_to_He4_Na22)*Y(Mg25)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Na23_to_He4_Ne20)*Y(Na23)*Y(H1)*state.rho + screened_rates(k_He4_Ne20_to_p_Na23)*Y(He4)*Y(Ne20)*state.rho) +
        (screened_rates(k_He4_Na23_to_p_Mg26)*Y(He4)*Y(Na23)*state.rho + -screened_rates(k_p_Mg26_to_He4_Na23)*Y(Mg26)*Y(H1)*state.rho) +
        (screened_rates(k_n_Mg23_to_p_Na23)*Y(Mg23)*Y(N)*state.rho + -screened_rates(k_p_Na23_to_n_Mg23)*Y(Na23)*Y(H1)*state.rho) +
        (screened_rates(k_He4_Mg23_to_p_Al26)*Y(He4)*Y(Mg23)*state.rho + -screened_rates(k_p_Al26_to_He4_Mg23)*Y(Al26)*Y(H1)*state.rho) +
        (screened_rates(k_n_Al25_to_p_Mg25)*Y(Al25)*Y(N)*state.rho + -screened_rates(k_p_Mg25_to_n_Al25)*Y(Mg25)*Y(H1)*state.rho) +
        (screened_rates(k_He4_Al25_to_p_Si28)*Y(Al25)*Y(He4)*state.rho + -screened_rates(k_p_Si28_to_He4_Al25)*Y(Si28)*Y(H1)*state.rho) +
        (screened_rates(k_n_Al26_to_p_Mg26)*Y(Al26)*Y(N)*state.rho + -screened_rates(k_p_Mg26_to_n_Al26)*Y(Mg26)*Y(H1)*state.rho) +
        (screened_rates(k_He4_Al26_to_p_Si29)*Y(Al26)*Y(He4)*state.rho + -screened_rates(k_p_Si29_to_He4_Al26)*Y(Si29)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Al27_to_He4_Mg24)*Y(Al27)*Y(H1)*state.rho + screened_rates(k_He4_Mg24_to_p_Al27)*Y(He4)*Y(Mg24)*state.rho) +
        (screened_rates(k_He4_Al27_to_p_Si30)*Y(Al27)*Y(He4)*state.rho + -screened_rates(k_p_Si30_to_He4_Al27)*Y(Si30)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Si31_to_n_P31)*Y(Si31)*Y(H1)*state.rho + screened_rates(k_n_P31_to_p_Si31)*Y(P31)*Y(N)*state.rho) +
        (screened_rates(k_n_P29_to_p_Si29)*Y(P29)*Y(N)*state.rho + -screened_rates(k_p_Si29_to_n_P29)*Y(Si29)*Y(H1)*state.rho) +
        (screened_rates(k_He4_P29_to_p_S32)*Y(He4)*Y(P29)*state.rho + -screened_rates(k_p_S32_to_He4_P29)*Y(S32)*Y(H1)*state.rho) +
        (screened_rates(k_n_P30_to_p_Si30)*Y(P30)*Y(N)*state.rho + -screened_rates(k_p_Si30_to_n_P30)*Y(Si30)*Y(H1)*state.rho) +
        (screened_rates(k_He4_P30_to_p_S33)*Y(He4)*Y(P30)*state.rho + -screened_rates(k_p_S33_to_He4_P30)*Y(S33)*Y(H1)*state.rho) +
        (-screened_rates(k_p_P31_to_He4_Si28)*Y(P31)*Y(H1)*state.rho + screened_rates(k_He4_Si28_to_p_P31)*Y(He4)*Y(Si28)*state.rho) +
        (screened_rates(k_He4_P31_to_p_S34)*Y(He4)*Y(P31)*state.rho + -screened_rates(k_p_S34_to_He4_P31)*Y(S34)*Y(H1)*state.rho) +
        (screened_rates(k_n_P32_to_p_Si32)*Y(P32)*Y(N)*state.rho + -screened_rates(k_p_Si32_to_n_P32)*Y(Si32)*Y(H1)*state.rho) +
        (-screened_rates(k_p_P32_to_n_S32)*Y(P32)*Y(H1)*state.rho + screened_rates(k_n_S32_to_p_P32)*Y(S32)*Y(N)*state.rho) +
        (-screened_rates(k_p_P32_to_He4_Si29)*Y(P32)*Y(H1)*state.rho + screened_rates(k_He4_Si29_to_p_P32)*Y(He4)*Y(Si29)*state.rho) +
        (-screened_rates(k_p_P33_to_He4_Si30)*Y(P33)*Y(H1)*state.rho + screened_rates(k_He4_Si30_to_p_P33)*Y(He4)*Y(Si30)*state.rho) +
        (screened_rates(k_n_S33_to_p_P33)*Y(S33)*Y(N)*state.rho + -screened_rates(k_p_P33_to_n_S33)*Y(P33)*Y(H1)*state.rho) +
        (-screened_rates(k_p_S35_to_He4_P32)*Y(S35)*Y(H1)*state.rho + screened_rates(k_He4_P32_to_p_S35)*Y(He4)*Y(P32)*state.rho) +
        (-screened_rates(k_p_S36_to_He4_P33)*Y(S36)*Y(H1)*state.rho + screened_rates(k_He4_P33_to_p_S36)*Y(He4)*Y(P33)*state.rho) +
        (screened_rates(k_n_Cl33_to_p_S33)*Y(Cl33)*Y(N)*state.rho + -screened_rates(k_p_S33_to_n_Cl33)*Y(S33)*Y(H1)*state.rho) +
        (screened_rates(k_He4_Cl33_to_p_Ar36)*Y(Cl33)*Y(He4)*state.rho + -screened_rates(k_p_Ar36_to_He4_Cl33)*Y(Ar36)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cl34_to_p_S34)*Y(Cl34)*Y(N)*state.rho + -screened_rates(k_p_S34_to_n_Cl34)*Y(S34)*Y(H1)*state.rho) +
        (screened_rates(k_He4_Cl34_to_p_Ar37)*Y(Cl34)*Y(He4)*state.rho + -screened_rates(k_p_Ar37_to_He4_Cl34)*Y(Ar37)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cl35_to_p_S35)*Y(Cl35)*Y(N)*state.rho + -screened_rates(k_p_S35_to_n_Cl35)*Y(S35)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Cl35_to_He4_S32)*Y(Cl35)*Y(H1)*state.rho + screened_rates(k_He4_S32_to_p_Cl35)*Y(He4)*Y(S32)*state.rho) +
        (screened_rates(k_He4_Cl35_to_p_Ar38)*Y(Cl35)*Y(He4)*state.rho + -screened_rates(k_p_Ar38_to_He4_Cl35)*Y(Ar38)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cl36_to_p_S36)*Y(Cl36)*Y(N)*state.rho + -screened_rates(k_p_S36_to_n_Cl36)*Y(S36)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Cl36_to_He4_S33)*Y(Cl36)*Y(H1)*state.rho + screened_rates(k_He4_S33_to_p_Cl36)*Y(He4)*Y(S33)*state.rho) +
        (-screened_rates(k_p_Cl37_to_He4_S34)*Y(Cl37)*Y(H1)*state.rho + screened_rates(k_He4_S34_to_p_Cl37)*Y(He4)*Y(S34)*state.rho) +
        (screened_rates(k_n_Ar36_to_p_Cl36)*Y(Ar36)*Y(N)*state.rho + -screened_rates(k_p_Cl36_to_n_Ar36)*Y(Cl36)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ar37_to_p_Cl37)*Y(Ar37)*Y(N)*state.rho + -screened_rates(k_p_Cl37_to_n_Ar37)*Y(Cl37)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Ar39_to_He4_Cl36)*Y(Ar39)*Y(H1)*state.rho + screened_rates(k_He4_Cl36_to_p_Ar39)*Y(Cl36)*Y(He4)*state.rho) +
        (-screened_rates(k_p_Ar40_to_He4_Cl37)*Y(Ar40)*Y(H1)*state.rho + screened_rates(k_He4_Cl37_to_p_Ar40)*Y(Cl37)*Y(He4)*state.rho) +
        (screened_rates(k_n_K37_to_p_Ar37)*Y(K37)*Y(N)*state.rho + -screened_rates(k_p_Ar37_to_n_K37)*Y(Ar37)*Y(H1)*state.rho) +
        (screened_rates(k_He4_K37_to_p_Ca40)*Y(He4)*Y(K37)*state.rho + -screened_rates(k_p_Ca40_to_He4_K37)*Y(Ca40)*Y(H1)*state.rho) +
        (screened_rates(k_n_K38_to_p_Ar38)*Y(K38)*Y(N)*state.rho + -screened_rates(k_p_Ar38_to_n_K38)*Y(Ar38)*Y(H1)*state.rho) +
        (screened_rates(k_He4_K38_to_p_Ca41)*Y(He4)*Y(K38)*state.rho + -screened_rates(k_p_Ca41_to_He4_K38)*Y(Ca41)*Y(H1)*state.rho) +
        (screened_rates(k_n_K39_to_p_Ar39)*Y(K39)*Y(N)*state.rho + -screened_rates(k_p_Ar39_to_n_K39)*Y(Ar39)*Y(H1)*state.rho) +
        (-screened_rates(k_p_K39_to_He4_Ar36)*Y(K39)*Y(H1)*state.rho + screened_rates(k_He4_Ar36_to_p_K39)*Y(Ar36)*Y(He4)*state.rho) +
        (screened_rates(k_n_K40_to_p_Ar40)*Y(K40)*Y(N)*state.rho + -screened_rates(k_p_Ar40_to_n_K40)*Y(Ar40)*Y(H1)*state.rho) +
        (-screened_rates(k_p_K40_to_n_Ca40)*Y(K40)*Y(H1)*state.rho + screened_rates(k_n_Ca40_to_p_K40)*Y(Ca40)*Y(N)*state.rho) +
        (-screened_rates(k_p_K40_to_He4_Ar37)*Y(K40)*Y(H1)*state.rho + screened_rates(k_He4_Ar37_to_p_K40)*Y(Ar37)*Y(He4)*state.rho) +
        (screened_rates(k_He4_K40_to_p_Ca43)*Y(He4)*Y(K40)*state.rho + -screened_rates(k_p_Ca43_to_He4_K40)*Y(Ca43)*Y(H1)*state.rho) +
        (-screened_rates(k_p_K41_to_He4_Ar38)*Y(K41)*Y(H1)*state.rho + screened_rates(k_He4_Ar38_to_p_K41)*Y(Ar38)*Y(He4)*state.rho) +
        (screened_rates(k_He4_K41_to_p_Ca44)*Y(He4)*Y(K41)*state.rho + -screened_rates(k_p_Ca44_to_He4_K41)*Y(Ca44)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ca41_to_p_K41)*Y(Ca41)*Y(N)*state.rho + -screened_rates(k_p_K41_to_n_Ca41)*Y(K41)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Ca42_to_He4_K39)*Y(Ca42)*Y(H1)*state.rho + screened_rates(k_He4_K39_to_p_Ca42)*Y(He4)*Y(K39)*state.rho) +
        (-screened_rates(k_p_Ca47_to_n_Sc47)*Y(Ca47)*Y(H1)*state.rho + screened_rates(k_n_Sc47_to_p_Ca47)*Y(Sc47)*Y(N)*state.rho) +
        (screened_rates(k_n_Sc43_to_p_Ca43)*Y(Sc43)*Y(N)*state.rho + -screened_rates(k_p_Ca43_to_n_Sc43)*Y(Ca43)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Sc43_to_He4_Ca40)*Y(Sc43)*Y(H1)*state.rho + screened_rates(k_He4_Ca40_to_p_Sc43)*Y(Ca40)*Y(He4)*state.rho) +
        (screened_rates(k_He4_Sc43_to_p_Ti46)*Y(He4)*Y(Sc43)*state.rho + -screened_rates(k_p_Ti46_to_He4_Sc43)*Y(Ti46)*Y(H1)*state.rho) +
        (screened_rates(k_n_Sc44_to_p_Ca44)*Y(Sc44)*Y(N)*state.rho + -screened_rates(k_p_Ca44_to_n_Sc44)*Y(Ca44)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Sc44_to_He4_Ca41)*Y(Sc44)*Y(H1)*state.rho + screened_rates(k_He4_Ca41_to_p_Sc44)*Y(Ca41)*Y(He4)*state.rho) +
        (screened_rates(k_He4_Sc44_to_p_Ti47)*Y(He4)*Y(Sc44)*state.rho + -screened_rates(k_p_Ti47_to_He4_Sc44)*Y(Ti47)*Y(H1)*state.rho) +
        (screened_rates(k_n_Sc45_to_p_Ca45)*Y(Sc45)*Y(N)*state.rho + -screened_rates(k_p_Ca45_to_n_Sc45)*Y(Ca45)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Sc45_to_He4_Ca42)*Y(Sc45)*Y(H1)*state.rho + screened_rates(k_He4_Ca42_to_p_Sc45)*Y(Ca42)*Y(He4)*state.rho) +
        (screened_rates(k_He4_Sc45_to_p_Ti48)*Y(He4)*Y(Sc45)*state.rho + -screened_rates(k_p_Ti48_to_He4_Sc45)*Y(Ti48)*Y(H1)*state.rho) +
        (screened_rates(k_n_Sc46_to_p_Ca46)*Y(Sc46)*Y(N)*state.rho + -screened_rates(k_p_Ca46_to_n_Sc46)*Y(Ca46)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Sc46_to_n_Ti46)*Y(Sc46)*Y(H1)*state.rho + screened_rates(k_n_Ti46_to_p_Sc46)*Y(Ti46)*Y(N)*state.rho) +
        (-screened_rates(k_p_Sc46_to_He4_Ca43)*Y(Sc46)*Y(H1)*state.rho + screened_rates(k_He4_Ca43_to_p_Sc46)*Y(Ca43)*Y(He4)*state.rho) +
        (screened_rates(k_He4_Sc46_to_p_Ti49)*Y(He4)*Y(Sc46)*state.rho + -screened_rates(k_p_Ti49_to_He4_Sc46)*Y(Ti49)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Sc47_to_He4_Ca44)*Y(Sc47)*Y(H1)*state.rho + screened_rates(k_He4_Ca44_to_p_Sc47)*Y(Ca44)*Y(He4)*state.rho) +
        (screened_rates(k_He4_Sc47_to_p_Ti50)*Y(He4)*Y(Sc47)*state.rho + -screened_rates(k_p_Ti50_to_He4_Sc47)*Y(Ti50)*Y(H1)*state.rho) +
        (screened_rates(k_n_Sc48_to_p_Ca48)*Y(Sc48)*Y(N)*state.rho + -screened_rates(k_p_Ca48_to_n_Sc48)*Y(Ca48)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Sc48_to_n_Ti48)*Y(Sc48)*Y(H1)*state.rho + screened_rates(k_n_Ti48_to_p_Sc48)*Y(Ti48)*Y(N)*state.rho) +
        (-screened_rates(k_p_Sc48_to_He4_Ca45)*Y(Sc48)*Y(H1)*state.rho + screened_rates(k_He4_Ca45_to_p_Sc48)*Y(Ca45)*Y(He4)*state.rho) +
        (screened_rates(k_He4_Sc48_to_p_Ti51)*Y(He4)*Y(Sc48)*state.rho + -screened_rates(k_p_Ti51_to_He4_Sc48)*Y(Ti51)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Sc49_to_n_Ti49)*Y(Sc49)*Y(H1)*state.rho + screened_rates(k_n_Ti49_to_p_Sc49)*Y(Ti49)*Y(N)*state.rho) +
        (-screened_rates(k_p_Sc49_to_He4_Ca46)*Y(Sc49)*Y(H1)*state.rho + screened_rates(k_He4_Ca46_to_p_Sc49)*Y(Ca46)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ti44_to_p_Sc44)*Y(Ti44)*Y(N)*state.rho + -screened_rates(k_p_Sc44_to_n_Ti44)*Y(Sc44)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ti45_to_p_Sc45)*Y(Ti45)*Y(N)*state.rho + -screened_rates(k_p_Sc45_to_n_Ti45)*Y(Sc45)*Y(H1)*state.rho) +
        (screened_rates(k_He4_Ti45_to_p_V48)*Y(He4)*Y(Ti45)*state.rho + -screened_rates(k_p_V48_to_He4_Ti45)*Y(V48)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ti47_to_p_Sc47)*Y(Ti47)*Y(N)*state.rho + -screened_rates(k_p_Sc47_to_n_Ti47)*Y(Sc47)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Ti51_to_n_V51)*Y(Ti51)*Y(H1)*state.rho + screened_rates(k_n_V51_to_p_Ti51)*Y(V51)*Y(N)*state.rho) +
        (screened_rates(k_n_V46_to_p_Ti46)*Y(V46)*Y(N)*state.rho + -screened_rates(k_p_Ti46_to_n_V46)*Y(Ti46)*Y(H1)*state.rho) +
        (screened_rates(k_He4_V46_to_p_Cr49)*Y(He4)*Y(V46)*state.rho + -screened_rates(k_p_Cr49_to_He4_V46)*Y(Cr49)*Y(H1)*state.rho) +
        (screened_rates(k_n_V47_to_p_Ti47)*Y(V47)*Y(N)*state.rho + -screened_rates(k_p_Ti47_to_n_V47)*Y(Ti47)*Y(H1)*state.rho) +
        (-screened_rates(k_p_V47_to_He4_Ti44)*Y(V47)*Y(H1)*state.rho + screened_rates(k_He4_Ti44_to_p_V47)*Y(He4)*Y(Ti44)*state.rho) +
        (screened_rates(k_He4_V47_to_p_Cr50)*Y(He4)*Y(V47)*state.rho + -screened_rates(k_p_Cr50_to_He4_V47)*Y(Cr50)*Y(H1)*state.rho) +
        (screened_rates(k_n_V48_to_p_Ti48)*Y(V48)*Y(N)*state.rho + -screened_rates(k_p_Ti48_to_n_V48)*Y(Ti48)*Y(H1)*state.rho) +
        (screened_rates(k_He4_V48_to_p_Cr51)*Y(He4)*Y(V48)*state.rho + -screened_rates(k_p_Cr51_to_He4_V48)*Y(Cr51)*Y(H1)*state.rho) +
        (screened_rates(k_n_V49_to_p_Ti49)*Y(V49)*Y(N)*state.rho + -screened_rates(k_p_Ti49_to_n_V49)*Y(Ti49)*Y(H1)*state.rho) +
        (-screened_rates(k_p_V49_to_He4_Ti46)*Y(V49)*Y(H1)*state.rho + screened_rates(k_He4_Ti46_to_p_V49)*Y(He4)*Y(Ti46)*state.rho) +
        (screened_rates(k_He4_V49_to_p_Cr52)*Y(He4)*Y(V49)*state.rho + -screened_rates(k_p_Cr52_to_He4_V49)*Y(Cr52)*Y(H1)*state.rho) +
        (screened_rates(k_n_V50_to_p_Ti50)*Y(V50)*Y(N)*state.rho + -screened_rates(k_p_Ti50_to_n_V50)*Y(Ti50)*Y(H1)*state.rho) +
        (-screened_rates(k_p_V50_to_n_Cr50)*Y(V50)*Y(H1)*state.rho + screened_rates(k_n_Cr50_to_p_V50)*Y(Cr50)*Y(N)*state.rho) +
        (-screened_rates(k_p_V50_to_He4_Ti47)*Y(V50)*Y(H1)*state.rho + screened_rates(k_He4_Ti47_to_p_V50)*Y(He4)*Y(Ti47)*state.rho) +
        (screened_rates(k_He4_V50_to_p_Cr53)*Y(He4)*Y(V50)*state.rho + -screened_rates(k_p_Cr53_to_He4_V50)*Y(Cr53)*Y(H1)*state.rho) +
        (-screened_rates(k_p_V51_to_He4_Ti48)*Y(V51)*Y(H1)*state.rho + screened_rates(k_He4_Ti48_to_p_V51)*Y(He4)*Y(Ti48)*state.rho) +
        (-screened_rates(k_p_V52_to_n_Cr52)*Y(V52)*Y(H1)*state.rho + screened_rates(k_n_Cr52_to_p_V52)*Y(Cr52)*Y(N)*state.rho) +
        (-screened_rates(k_p_V52_to_He4_Ti49)*Y(V52)*Y(H1)*state.rho + screened_rates(k_He4_Ti49_to_p_V52)*Y(He4)*Y(Ti49)*state.rho) +
        (screened_rates(k_n_Cr48_to_p_V48)*Y(Cr48)*Y(N)*state.rho + -screened_rates(k_p_V48_to_n_Cr48)*Y(V48)*Y(H1)*state.rho) +
        (screened_rates(k_He4_Cr48_to_p_Mn51)*Y(Cr48)*Y(He4)*state.rho + -screened_rates(k_p_Mn51_to_He4_Cr48)*Y(Mn51)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cr49_to_p_V49)*Y(Cr49)*Y(N)*state.rho + -screened_rates(k_p_V49_to_n_Cr49)*Y(V49)*Y(H1)*state.rho) +
        (screened_rates(k_He4_Cr49_to_p_Mn52)*Y(Cr49)*Y(He4)*state.rho + -screened_rates(k_p_Mn52_to_He4_Cr49)*Y(Mn52)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cr51_to_p_V51)*Y(Cr51)*Y(N)*state.rho + -screened_rates(k_p_V51_to_n_Cr51)*Y(V51)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Cr54_to_He4_V51)*Y(Cr54)*Y(H1)*state.rho + screened_rates(k_He4_V51_to_p_Cr54)*Y(He4)*Y(V51)*state.rho) +
        (screened_rates(k_n_Mn50_to_p_Cr50)*Y(Mn50)*Y(N)*state.rho + -screened_rates(k_p_Cr50_to_n_Mn50)*Y(Cr50)*Y(H1)*state.rho) +
        (screened_rates(k_He4_Mn50_to_p_Fe53)*Y(He4)*Y(Mn50)*state.rho + -screened_rates(k_p_Fe53_to_He4_Mn50)*Y(Fe53)*Y(H1)*state.rho) +
        (screened_rates(k_n_Mn51_to_p_Cr51)*Y(Mn51)*Y(N)*state.rho + -screened_rates(k_p_Cr51_to_n_Mn51)*Y(Cr51)*Y(H1)*state.rho) +
        (screened_rates(k_He4_Mn51_to_p_Fe54)*Y(He4)*Y(Mn51)*state.rho + -screened_rates(k_p_Fe54_to_He4_Mn51)*Y(Fe54)*Y(H1)*state.rho) +
        (screened_rates(k_n_Mn52_to_p_Cr52)*Y(Mn52)*Y(N)*state.rho + -screened_rates(k_p_Cr52_to_n_Mn52)*Y(Cr52)*Y(H1)*state.rho) +
        (screened_rates(k_He4_Mn52_to_p_Fe55)*Y(He4)*Y(Mn52)*state.rho + -screened_rates(k_p_Fe55_to_He4_Mn52)*Y(Fe55)*Y(H1)*state.rho) +
        (screened_rates(k_n_Mn53_to_p_Cr53)*Y(Mn53)*Y(N)*state.rho + -screened_rates(k_p_Cr53_to_n_Mn53)*Y(Cr53)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Mn53_to_He4_Cr50)*Y(Mn53)*Y(H1)*state.rho + screened_rates(k_He4_Cr50_to_p_Mn53)*Y(Cr50)*Y(He4)*state.rho) +
        (screened_rates(k_He4_Mn53_to_p_Fe56)*Y(He4)*Y(Mn53)*state.rho + -screened_rates(k_p_Fe56_to_He4_Mn53)*Y(Fe56)*Y(H1)*state.rho) +
        (screened_rates(k_n_Mn54_to_p_Cr54)*Y(Mn54)*Y(N)*state.rho + -screened_rates(k_p_Cr54_to_n_Mn54)*Y(Cr54)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Mn54_to_He4_Cr51)*Y(Mn54)*Y(H1)*state.rho + screened_rates(k_He4_Cr51_to_p_Mn54)*Y(Cr51)*Y(He4)*state.rho) +
        (-screened_rates(k_p_Mn55_to_He4_Cr52)*Y(Mn55)*Y(H1)*state.rho + screened_rates(k_He4_Cr52_to_p_Mn55)*Y(Cr52)*Y(He4)*state.rho) +
        (screened_rates(k_n_Fe52_to_p_Mn52)*Y(Fe52)*Y(N)*state.rho + -screened_rates(k_p_Mn52_to_n_Fe52)*Y(Mn52)*Y(H1)*state.rho) +
        (screened_rates(k_He4_Fe52_to_p_Co55)*Y(Fe52)*Y(He4)*state.rho + -screened_rates(k_p_Co55_to_He4_Fe52)*Y(Co55)*Y(H1)*state.rho) +
        (screened_rates(k_n_Fe53_to_p_Mn53)*Y(Fe53)*Y(N)*state.rho + -screened_rates(k_p_Mn53_to_n_Fe53)*Y(Mn53)*Y(H1)*state.rho) +
        (screened_rates(k_He4_Fe53_to_p_Co56)*Y(Fe53)*Y(He4)*state.rho + -screened_rates(k_p_Co56_to_He4_Fe53)*Y(Co56)*Y(H1)*state.rho) +
        (screened_rates(k_n_Fe54_to_p_Mn54)*Y(Fe54)*Y(N)*state.rho + -screened_rates(k_p_Mn54_to_n_Fe54)*Y(Mn54)*Y(H1)*state.rho) +
        (screened_rates(k_n_Fe55_to_p_Mn55)*Y(Fe55)*Y(N)*state.rho + -screened_rates(k_p_Mn55_to_n_Fe55)*Y(Mn55)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Fe57_to_He4_Mn54)*Y(Fe57)*Y(H1)*state.rho + screened_rates(k_He4_Mn54_to_p_Fe57)*Y(He4)*Y(Mn54)*state.rho) +
        (-screened_rates(k_p_Fe58_to_He4_Mn55)*Y(Fe58)*Y(H1)*state.rho + screened_rates(k_He4_Mn55_to_p_Fe58)*Y(He4)*Y(Mn55)*state.rho) +
        (screened_rates(k_n_Co53_to_p_Fe53)*Y(Co53)*Y(N)*state.rho + -screened_rates(k_p_Fe53_to_n_Co53)*Y(Fe53)*Y(H1)*state.rho) +
        (screened_rates(k_He4_Co53_to_p_Ni56)*Y(Co53)*Y(He4)*state.rho + -screened_rates(k_p_Ni56_to_He4_Co53)*Y(Ni56)*Y(H1)*state.rho) +
        (screened_rates(k_n_Co54_to_p_Fe54)*Y(Co54)*Y(N)*state.rho + -screened_rates(k_p_Fe54_to_n_Co54)*Y(Fe54)*Y(H1)*state.rho) +
        (screened_rates(k_He4_Co54_to_p_Ni57)*Y(Co54)*Y(He4)*state.rho + -screened_rates(k_p_Ni57_to_He4_Co54)*Y(Ni57)*Y(H1)*state.rho) +
        (screened_rates(k_n_Co55_to_p_Fe55)*Y(Co55)*Y(N)*state.rho + -screened_rates(k_p_Fe55_to_n_Co55)*Y(Fe55)*Y(H1)*state.rho) +
        (screened_rates(k_He4_Co55_to_p_Ni58)*Y(Co55)*Y(He4)*state.rho + -screened_rates(k_p_Ni58_to_He4_Co55)*Y(Ni58)*Y(H1)*state.rho) +
        (screened_rates(k_n_Co56_to_p_Fe56)*Y(Co56)*Y(N)*state.rho + -screened_rates(k_p_Fe56_to_n_Co56)*Y(Fe56)*Y(H1)*state.rho) +
        (screened_rates(k_He4_Co56_to_p_Ni59)*Y(Co56)*Y(He4)*state.rho + -screened_rates(k_p_Ni59_to_He4_Co56)*Y(Ni59)*Y(H1)*state.rho) +
        (screened_rates(k_n_Co57_to_p_Fe57)*Y(Co57)*Y(N)*state.rho + -screened_rates(k_p_Fe57_to_n_Co57)*Y(Fe57)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Co57_to_He4_Fe54)*Y(Co57)*Y(H1)*state.rho + screened_rates(k_He4_Fe54_to_p_Co57)*Y(Fe54)*Y(He4)*state.rho) +
        (screened_rates(k_He4_Co57_to_p_Ni60)*Y(Co57)*Y(He4)*state.rho + -screened_rates(k_p_Ni60_to_He4_Co57)*Y(Ni60)*Y(H1)*state.rho) +
        (screened_rates(k_n_Co58_to_p_Fe58)*Y(Co58)*Y(N)*state.rho + -screened_rates(k_p_Fe58_to_n_Co58)*Y(Fe58)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Co58_to_He4_Fe55)*Y(Co58)*Y(H1)*state.rho + screened_rates(k_He4_Fe55_to_p_Co58)*Y(Fe55)*Y(He4)*state.rho) +
        (-screened_rates(k_p_Co59_to_He4_Fe56)*Y(Co59)*Y(H1)*state.rho + screened_rates(k_He4_Fe56_to_p_Co59)*Y(Fe56)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ni56_to_p_Co56)*Y(Ni56)*Y(N)*state.rho + -screened_rates(k_p_Co56_to_n_Ni56)*Y(Co56)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ni57_to_p_Co57)*Y(Ni57)*Y(N)*state.rho + -screened_rates(k_p_Co57_to_n_Ni57)*Y(Co57)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ni58_to_p_Co58)*Y(Ni58)*Y(N)*state.rho + -screened_rates(k_p_Co58_to_n_Ni58)*Y(Co58)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ni59_to_p_Co59)*Y(Ni59)*Y(N)*state.rho + -screened_rates(k_p_Co59_to_n_Ni59)*Y(Co59)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Ni61_to_He4_Co58)*Y(Ni61)*Y(H1)*state.rho + screened_rates(k_He4_Co58_to_p_Ni61)*Y(Co58)*Y(He4)*state.rho) +
        (-screened_rates(k_p_Ni62_to_He4_Co59)*Y(Ni62)*Y(H1)*state.rho + screened_rates(k_He4_Co59_to_p_Ni62)*Y(Co59)*Y(He4)*state.rho) +
        (screened_rates(k_n_Cu57_to_p_Ni57)*Y(Cu57)*Y(N)*state.rho + -screened_rates(k_p_Ni57_to_n_Cu57)*Y(Ni57)*Y(H1)*state.rho) +
        (screened_rates(k_He4_Cu57_to_p_Zn60)*Y(Cu57)*Y(He4)*state.rho + -screened_rates(k_p_Zn60_to_He4_Cu57)*Y(Zn60)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cu58_to_p_Ni58)*Y(Cu58)*Y(N)*state.rho + -screened_rates(k_p_Ni58_to_n_Cu58)*Y(Ni58)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cu59_to_p_Ni59)*Y(Cu59)*Y(N)*state.rho + -screened_rates(k_p_Ni59_to_n_Cu59)*Y(Ni59)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Cu59_to_He4_Ni56)*Y(Cu59)*Y(H1)*state.rho + screened_rates(k_He4_Ni56_to_p_Cu59)*Y(He4)*Y(Ni56)*state.rho) +
        (screened_rates(k_n_Cu60_to_p_Ni60)*Y(Cu60)*Y(N)*state.rho + -screened_rates(k_p_Ni60_to_n_Cu60)*Y(Ni60)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Cu60_to_He4_Ni57)*Y(Cu60)*Y(H1)*state.rho + screened_rates(k_He4_Ni57_to_p_Cu60)*Y(He4)*Y(Ni57)*state.rho) +
        (screened_rates(k_n_Cu61_to_p_Ni61)*Y(Cu61)*Y(N)*state.rho + -screened_rates(k_p_Ni61_to_n_Cu61)*Y(Ni61)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Cu61_to_He4_Ni58)*Y(Cu61)*Y(H1)*state.rho + screened_rates(k_He4_Ni58_to_p_Cu61)*Y(He4)*Y(Ni58)*state.rho) +
        (screened_rates(k_n_Cu62_to_p_Ni62)*Y(Cu62)*Y(N)*state.rho + -screened_rates(k_p_Ni62_to_n_Cu62)*Y(Ni62)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Cu62_to_He4_Ni59)*Y(Cu62)*Y(H1)*state.rho + screened_rates(k_He4_Ni59_to_p_Cu62)*Y(He4)*Y(Ni59)*state.rho) +
        (screened_rates(k_n_Cu63_to_p_Ni63)*Y(Cu63)*Y(N)*state.rho + -screened_rates(k_p_Ni63_to_n_Cu63)*Y(Ni63)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Cu63_to_He4_Ni60)*Y(Cu63)*Y(H1)*state.rho + screened_rates(k_He4_Ni60_to_p_Cu63)*Y(He4)*Y(Ni60)*state.rho) +
        (screened_rates(k_n_Cu64_to_p_Ni64)*Y(Cu64)*Y(N)*state.rho + -screened_rates(k_p_Ni64_to_n_Cu64)*Y(Ni64)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Cu64_to_He4_Ni61)*Y(Cu64)*Y(H1)*state.rho + screened_rates(k_He4_Ni61_to_p_Cu64)*Y(He4)*Y(Ni61)*state.rho) +
        (-screened_rates(k_p_Cu65_to_He4_Ni62)*Y(Cu65)*Y(H1)*state.rho + screened_rates(k_He4_Ni62_to_p_Cu65)*Y(He4)*Y(Ni62)*state.rho) +
        (screened_rates(k_n_Zn59_to_p_Cu59)*Y(Zn59)*Y(N)*state.rho + -screened_rates(k_p_Cu59_to_n_Zn59)*Y(Cu59)*Y(H1)*state.rho) +
        (screened_rates(k_n_Zn60_to_p_Cu60)*Y(Zn60)*Y(N)*state.rho + -screened_rates(k_p_Cu60_to_n_Zn60)*Y(Cu60)*Y(H1)*state.rho) +
        (screened_rates(k_n_Zn61_to_p_Cu61)*Y(Zn61)*Y(N)*state.rho + -screened_rates(k_p_Cu61_to_n_Zn61)*Y(Cu61)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Zn61_to_He4_Cu58)*Y(Zn61)*Y(H1)*state.rho + screened_rates(k_He4_Cu58_to_p_Zn61)*Y(Cu58)*Y(He4)*state.rho) +
        (screened_rates(k_n_Zn62_to_p_Cu62)*Y(Zn62)*Y(N)*state.rho + -screened_rates(k_p_Cu62_to_n_Zn62)*Y(Cu62)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Zn62_to_He4_Cu59)*Y(Zn62)*Y(H1)*state.rho + screened_rates(k_He4_Cu59_to_p_Zn62)*Y(Cu59)*Y(He4)*state.rho) +
        (screened_rates(k_n_Zn63_to_p_Cu63)*Y(Zn63)*Y(N)*state.rho + -screened_rates(k_p_Cu63_to_n_Zn63)*Y(Cu63)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Zn63_to_He4_Cu60)*Y(Zn63)*Y(H1)*state.rho + screened_rates(k_He4_Cu60_to_p_Zn63)*Y(Cu60)*Y(He4)*state.rho) +
        (screened_rates(k_n_Zn64_to_p_Cu64)*Y(Zn64)*Y(N)*state.rho + -screened_rates(k_p_Cu64_to_n_Zn64)*Y(Cu64)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Zn64_to_He4_Cu61)*Y(Zn64)*Y(H1)*state.rho + screened_rates(k_He4_Cu61_to_p_Zn64)*Y(Cu61)*Y(He4)*state.rho) +
        (screened_rates(k_n_Zn65_to_p_Cu65)*Y(Zn65)*Y(N)*state.rho + -screened_rates(k_p_Cu65_to_n_Zn65)*Y(Cu65)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Zn65_to_He4_Cu62)*Y(Zn65)*Y(H1)*state.rho + screened_rates(k_He4_Cu62_to_p_Zn65)*Y(Cu62)*Y(He4)*state.rho) +
        (-screened_rates(k_p_Zn66_to_He4_Cu63)*Y(Zn66)*Y(H1)*state.rho + screened_rates(k_He4_Cu63_to_p_Zn66)*Y(Cu63)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ga62_to_p_Zn62)*Y(Ga62)*Y(N)*state.rho + -screened_rates(k_p_Zn62_to_n_Ga62)*Y(Zn62)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Ga62_to_He4_Zn59)*Y(Ga62)*Y(H1)*state.rho + screened_rates(k_He4_Zn59_to_p_Ga62)*Y(He4)*Y(Zn59)*state.rho) +
        (screened_rates(k_n_Ga63_to_p_Zn63)*Y(Ga63)*Y(N)*state.rho + -screened_rates(k_p_Zn63_to_n_Ga63)*Y(Zn63)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Ga63_to_He4_Zn60)*Y(Ga63)*Y(H1)*state.rho + screened_rates(k_He4_Zn60_to_p_Ga63)*Y(He4)*Y(Zn60)*state.rho) +
        (screened_rates(k_n_Ga64_to_p_Zn64)*Y(Ga64)*Y(N)*state.rho + -screened_rates(k_p_Zn64_to_n_Ga64)*Y(Zn64)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Ga64_to_He4_Zn61)*Y(Ga64)*Y(H1)*state.rho + screened_rates(k_He4_Zn61_to_p_Ga64)*Y(He4)*Y(Zn61)*state.rho) +
        (screened_rates(k_n_Ge63_to_p_Ga63)*Y(Ge63)*Y(N)*state.rho + -screened_rates(k_p_Ga63_to_n_Ge63)*Y(Ga63)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ge64_to_p_Ga64)*Y(Ge64)*Y(N)*state.rho + -screened_rates(k_p_Ga64_to_n_Ge64)*Y(Ga64)*Y(H1)*state.rho) +
        (screened_rates(k_He3_He3_to_p_p_He4)*amrex::Math::powi<2>(Y(He3))*state.rho + -screened_rates(k_p_p_He4_to_He3_He3)*Y(He4)*amrex::Math::powi<2>(Y(H1))*amrex::Math::powi<2>(state.rho)) +
        (screened_rates(k_d_Be7_to_p_He4_He4)*Y(Be7)*Y(H2)*state.rho + -0.5*screened_rates(k_p_He4_He4_to_d_Be7)*amrex::Math::powi<2>(Y(He4))*Y(H1)*amrex::Math::powi<2>(state.rho)) +
        (-screened_rates(k_p_Be9_to_d_He4_He4)*Y(Be9)*Y(H1)*state.rho + 0.5*screened_rates(k_d_He4_He4_to_p_Be9)*amrex::Math::powi<2>(Y(He4))*Y(H2)*amrex::Math::powi<2>(state.rho)) +
        (screened_rates(k_n_B8_to_p_He4_He4)*Y(B8)*Y(N)*state.rho + -0.5*screened_rates(k_p_He4_He4_to_n_B8)*amrex::Math::powi<2>(Y(He4))*Y(H1)*amrex::Math::powi<2>(state.rho)) +
        (-screened_rates(k_p_B11_to_He4_He4_He4)*Y(B11)*Y(H1)*state.rho + 0.16666666666666667*screened_rates(k_He4_He4_He4_to_p_B11)*amrex::Math::powi<3>(Y(He4))*amrex::Math::powi<2>(state.rho)) +
        (screened_rates(k_He3_Li7_to_n_p_He4_He4)*Y(He3)*Y(Li7)*state.rho + -0.5*screened_rates(k_n_p_He4_He4_to_He3_Li7)*amrex::Math::powi<2>(Y(He4))*Y(N)*Y(H1)*amrex::Math::powi<3>(state.rho)) +
        (2.0*screened_rates(k_He3_Be7_to_p_p_He4_He4)*Y(Be7)*Y(He3)*state.rho + -0.5*screened_rates(k_p_p_He4_He4_to_He3_Be7)*amrex::Math::powi<2>(Y(He4))*amrex::Math::powi<2>(Y(H1))*amrex::Math::powi<3>(state.rho)) +
        (-screened_rates(k_n_p_He4_to_Li6)*Y(He4)*Y(N)*Y(H1)*amrex::Math::powi<2>(state.rho) + screened_rates(k_Li6_to_n_p_He4)*Y(Li6)) +
        (-0.5*screened_rates(k_n_p_p_to_p_d)*Y(N)*amrex::Math::powi<2>(Y(H1))*amrex::Math::powi<2>(state.rho) + screened_rates(k_p_d_to_n_p_p)*Y(H2)*Y(H1)*state.rho) +
        (0.0e0_rt + 0.0e0_rt) +
        (screened_rates(k_n_to_p)*Y(N) + -screened_rates(k_p_to_n)*Y(H1));

    ydot_nuc(H2) =
        (screened_rates(k_n_p_to_d)*Y(N)*Y(H1)*state.rho + -screened_rates(k_d_to_n_p)*Y(H2)) +
        0.5*screened_rates(k_p_p_to_d_weak_bet_pos_)*amrex::Math::powi<2>(Y(H1))*state.rho +
        0.5*screened_rates(k_p_p_to_d_weak_electron_capture)*amrex::Math::powi<2>(Y(H1))*amrex::Math::powi<2>(state.rho)*state.y_e +
        (-screened_rates(k_p_d_to_He3)*Y(H2)*Y(H1)*state.rho + screened_rates(k_He3_to_p_d)*Y(He3)) +
        (-screened_rates(k_d_d_to_He4)*amrex::Math::powi<2>(Y(H2))*state.rho + 2.0*screened_rates(k_He4_to_d_d)*Y(He4)) +
        (-screened_rates(k_He4_d_to_Li6)*Y(He4)*Y(H2)*state.rho + screened_rates(k_Li6_to_He4_d)*Y(Li6)) +
        (-screened_rates(k_d_d_to_n_He3)*amrex::Math::powi<2>(Y(H2))*state.rho + 2.0*screened_rates(k_n_He3_to_d_d)*Y(He3)*Y(N)*state.rho) +
        (-screened_rates(k_d_He3_to_p_He4)*Y(He3)*Y(H2)*state.rho + screened_rates(k_p_He4_to_d_He3)*Y(He4)*Y(H1)*state.rho) +
        (-screened_rates(k_d_Li6_to_n_Be7)*Y(Li6)*Y(H2)*state.rho + screened_rates(k_n_Be7_to_d_Li6)*Y(Be7)*Y(N)*state.rho) +
        (-screened_rates(k_d_Li6_to_p_Li7)*Y(Li6)*Y(H2)*state.rho + screened_rates(k_p_Li7_to_d_Li6)*Y(Li7)*Y(H1)*state.rho) +
        (-screened_rates(k_d_C13_to_n_N14)*Y(C13)*Y(H2)*state.rho + screened_rates(k_n_N14_to_d_C13)*Y(N14)*Y(N)*state.rho) +
        (-screened_rates(k_d_C14_to_n_N15)*Y(C14)*Y(H2)*state.rho + screened_rates(k_n_N15_to_d_C14)*Y(N15)*Y(N)*state.rho) +
        (-screened_rates(k_d_Li7_to_n_He4_He4)*Y(Li7)*Y(H2)*state.rho + 0.5*screened_rates(k_n_He4_He4_to_d_Li7)*amrex::Math::powi<2>(Y(He4))*Y(N)*amrex::Math::powi<2>(state.rho)) +
        (-screened_rates(k_d_Be7_to_p_He4_He4)*Y(Be7)*Y(H2)*state.rho + 0.5*screened_rates(k_p_He4_He4_to_d_Be7)*amrex::Math::powi<2>(Y(He4))*Y(H1)*amrex::Math::powi<2>(state.rho)) +
        (screened_rates(k_p_Be9_to_d_He4_He4)*Y(Be9)*Y(H1)*state.rho + -0.5*screened_rates(k_d_He4_He4_to_p_Be9)*amrex::Math::powi<2>(Y(He4))*Y(H2)*amrex::Math::powi<2>(state.rho)) +
        (0.5*screened_rates(k_n_p_p_to_p_d)*Y(N)*amrex::Math::powi<2>(Y(H1))*amrex::Math::powi<2>(state.rho) + -screened_rates(k_p_d_to_n_p_p)*Y(H2)*Y(H1)*state.rho);

    ydot_nuc(He3) =
        (screened_rates(k_p_d_to_He3)*Y(H2)*Y(H1)*state.rho + -screened_rates(k_He3_to_p_d)*Y(He3)) +
        (-screened_rates(k_n_He3_to_He4)*Y(He3)*Y(N)*state.rho + screened_rates(k_He4_to_n_He3)*Y(He4)) +
        -screened_rates(k_p_He3_to_He4_weak_bet_pos_)*Y(He3)*Y(H1)*state.rho +
        (-screened_rates(k_He4_He3_to_Be7)*Y(He3)*Y(He4)*state.rho + screened_rates(k_Be7_to_He4_He3)*Y(Be7)) +
        (0.5*screened_rates(k_d_d_to_n_He3)*amrex::Math::powi<2>(Y(H2))*state.rho + -screened_rates(k_n_He3_to_d_d)*Y(He3)*Y(N)*state.rho) +
        (-screened_rates(k_d_He3_to_p_He4)*Y(He3)*Y(H2)*state.rho + screened_rates(k_p_He4_to_d_He3)*Y(He4)*Y(H1)*state.rho) +
        (screened_rates(k_p_Li6_to_He4_He3)*Y(Li6)*Y(H1)*state.rho + -screened_rates(k_He4_He3_to_p_Li6)*Y(He3)*Y(He4)*state.rho) +
        (-screened_rates(k_He3_He3_to_p_p_He4)*amrex::Math::powi<2>(Y(He3))*state.rho + screened_rates(k_p_p_He4_to_He3_He3)*Y(He4)*amrex::Math::powi<2>(Y(H1))*amrex::Math::powi<2>(state.rho)) +
        (-screened_rates(k_He3_Li7_to_n_p_He4_He4)*Y(He3)*Y(Li7)*state.rho + 0.5*screened_rates(k_n_p_He4_He4_to_He3_Li7)*amrex::Math::powi<2>(Y(He4))*Y(N)*Y(H1)*amrex::Math::powi<3>(state.rho)) +
        (-screened_rates(k_He3_Be7_to_p_p_He4_He4)*Y(Be7)*Y(He3)*state.rho + 0.25*screened_rates(k_p_p_He4_He4_to_He3_Be7)*amrex::Math::powi<2>(Y(He4))*amrex::Math::powi<2>(Y(H1))*amrex::Math::powi<3>(state.rho));

    ydot_nuc(He4) =
        2.0*screened_rates(k_B8_to_He4_He4_weak_wc12)*Y(B8) +
        (0.5*screened_rates(k_d_d_to_He4)*amrex::Math::powi<2>(Y(H2))*state.rho + -screened_rates(k_He4_to_d_d)*Y(He4)) +
        (-screened_rates(k_He4_d_to_Li6)*Y(He4)*Y(H2)*state.rho + screened_rates(k_Li6_to_He4_d)*Y(Li6)) +
        (screened_rates(k_n_He3_to_He4)*Y(He3)*Y(N)*state.rho + -screened_rates(k_He4_to_n_He3)*Y(He4)) +
        screened_rates(k_p_He3_to_He4_weak_bet_pos_)*Y(He3)*Y(H1)*state.rho +
        (-screened_rates(k_He4_He3_to_Be7)*Y(He3)*Y(He4)*state.rho + screened_rates(k_Be7_to_He4_He3)*Y(Be7)) +
        (-screened_rates(k_He4_Li6_to_B10)*Y(He4)*Y(Li6)*state.rho + screened_rates(k_B10_to_He4_Li6)*Y(B10)) +
        (-screened_rates(k_He4_Li7_to_B11)*Y(He4)*Y(Li7)*state.rho + screened_rates(k_B11_to_He4_Li7)*Y(B11)) +
        (-screened_rates(k_He4_C12_to_O16)*Y(C12)*Y(He4)*state.rho + screened_rates(k_O16_to_He4_C12)*Y(O16)) +
        (-screened_rates(k_He4_C14_to_O18)*Y(C14)*Y(He4)*state.rho + screened_rates(k_O18_to_He4_C14)*Y(O18)) +
        (-screened_rates(k_He4_N14_to_F18)*Y(He4)*Y(N14)*state.rho + screened_rates(k_F18_to_He4_N14)*Y(F18)) +
        (-screened_rates(k_He4_N15_to_F19)*Y(He4)*Y(N15)*state.rho + screened_rates(k_F19_to_He4_N15)*Y(F19)) +
        (-screened_rates(k_He4_O14_to_Ne18)*Y(He4)*Y(O14)*state.rho + screened_rates(k_Ne18_to_He4_O14)*Y(Ne18)) +
        (-screened_rates(k_He4_O15_to_Ne19)*Y(He4)*Y(O15)*state.rho + screened_rates(k_Ne19_to_He4_O15)*Y(Ne19)) +
        (-screened_rates(k_He4_O16_to_Ne20)*Y(He4)*Y(O16)*state.rho + screened_rates(k_Ne20_to_He4_O16)*Y(Ne20)) +
        (-screened_rates(k_He4_O17_to_Ne21)*Y(He4)*Y(O17)*state.rho + screened_rates(k_Ne21_to_He4_O17)*Y(Ne21)) +
        (-screened_rates(k_He4_O18_to_Ne22)*Y(He4)*Y(O18)*state.rho + screened_rates(k_Ne22_to_He4_O18)*Y(Ne22)) +
        (-screened_rates(k_He4_F17_to_Na21)*Y(F17)*Y(He4)*state.rho + screened_rates(k_Na21_to_He4_F17)*Y(Na21)) +
        (-screened_rates(k_He4_F18_to_Na22)*Y(F18)*Y(He4)*state.rho + screened_rates(k_Na22_to_He4_F18)*Y(Na22)) +
        (-screened_rates(k_He4_F19_to_Na23)*Y(F19)*Y(He4)*state.rho + screened_rates(k_Na23_to_He4_F19)*Y(Na23)) +
        (-screened_rates(k_He4_Ne19_to_Mg23)*Y(He4)*Y(Ne19)*state.rho + screened_rates(k_Mg23_to_He4_Ne19)*Y(Mg23)) +
        (-screened_rates(k_He4_Ne20_to_Mg24)*Y(He4)*Y(Ne20)*state.rho + screened_rates(k_Mg24_to_He4_Ne20)*Y(Mg24)) +
        (-screened_rates(k_He4_Ne21_to_Mg25)*Y(He4)*Y(Ne21)*state.rho + screened_rates(k_Mg25_to_He4_Ne21)*Y(Mg25)) +
        (-screened_rates(k_He4_Ne22_to_Mg26)*Y(He4)*Y(Ne22)*state.rho + screened_rates(k_Mg26_to_He4_Ne22)*Y(Mg26)) +
        (-screened_rates(k_He4_Na21_to_Al25)*Y(He4)*Y(Na21)*state.rho + screened_rates(k_Al25_to_He4_Na21)*Y(Al25)) +
        (-screened_rates(k_He4_Na22_to_Al26)*Y(He4)*Y(Na22)*state.rho + screened_rates(k_Al26_to_He4_Na22)*Y(Al26)) +
        (-screened_rates(k_He4_Na23_to_Al27)*Y(He4)*Y(Na23)*state.rho + screened_rates(k_Al27_to_He4_Na23)*Y(Al27)) +
        (-screened_rates(k_He4_Mg24_to_Si28)*Y(He4)*Y(Mg24)*state.rho + screened_rates(k_Si28_to_He4_Mg24)*Y(Si28)) +
        (-screened_rates(k_He4_Mg25_to_Si29)*Y(He4)*Y(Mg25)*state.rho + screened_rates(k_Si29_to_He4_Mg25)*Y(Si29)) +
        (-screened_rates(k_He4_Mg26_to_Si30)*Y(He4)*Y(Mg26)*state.rho + screened_rates(k_Si30_to_He4_Mg26)*Y(Si30)) +
        (-screened_rates(k_He4_Al25_to_P29)*Y(Al25)*Y(He4)*state.rho + screened_rates(k_P29_to_He4_Al25)*Y(P29)) +
        (-screened_rates(k_He4_Al26_to_P30)*Y(Al26)*Y(He4)*state.rho + screened_rates(k_P30_to_He4_Al26)*Y(P30)) +
        (-screened_rates(k_He4_Al27_to_P31)*Y(Al27)*Y(He4)*state.rho + screened_rates(k_P31_to_He4_Al27)*Y(P31)) +
        (-screened_rates(k_He4_Si28_to_S32)*Y(He4)*Y(Si28)*state.rho + screened_rates(k_S32_to_He4_Si28)*Y(S32)) +
        (-screened_rates(k_He4_Si29_to_S33)*Y(He4)*Y(Si29)*state.rho + screened_rates(k_S33_to_He4_Si29)*Y(S33)) +
        (-screened_rates(k_He4_Si30_to_S34)*Y(He4)*Y(Si30)*state.rho + screened_rates(k_S34_to_He4_Si30)*Y(S34)) +
        (-screened_rates(k_He4_Si31_to_S35)*Y(He4)*Y(Si31)*state.rho + screened_rates(k_S35_to_He4_Si31)*Y(S35)) +
        (-screened_rates(k_He4_Si32_to_S36)*Y(He4)*Y(Si32)*state.rho + screened_rates(k_S36_to_He4_Si32)*Y(S36)) +
        (-screened_rates(k_He4_P29_to_Cl33)*Y(He4)*Y(P29)*state.rho + screened_rates(k_Cl33_to_He4_P29)*Y(Cl33)) +
        (-screened_rates(k_He4_P30_to_Cl34)*Y(He4)*Y(P30)*state.rho + screened_rates(k_Cl34_to_He4_P30)*Y(Cl34)) +
        (-screened_rates(k_He4_P31_to_Cl35)*Y(He4)*Y(P31)*state.rho + screened_rates(k_Cl35_to_He4_P31)*Y(Cl35)) +
        (-screened_rates(k_He4_P32_to_Cl36)*Y(He4)*Y(P32)*state.rho + screened_rates(k_Cl36_to_He4_P32)*Y(Cl36)) +
        (-screened_rates(k_He4_P33_to_Cl37)*Y(He4)*Y(P33)*state.rho + screened_rates(k_Cl37_to_He4_P33)*Y(Cl37)) +
        (-screened_rates(k_He4_S32_to_Ar36)*Y(He4)*Y(S32)*state.rho + screened_rates(k_Ar36_to_He4_S32)*Y(Ar36)) +
        (-screened_rates(k_He4_S33_to_Ar37)*Y(He4)*Y(S33)*state.rho + screened_rates(k_Ar37_to_He4_S33)*Y(Ar37)) +
        (-screened_rates(k_He4_S34_to_Ar38)*Y(He4)*Y(S34)*state.rho + screened_rates(k_Ar38_to_He4_S34)*Y(Ar38)) +
        (-screened_rates(k_He4_S35_to_Ar39)*Y(He4)*Y(S35)*state.rho + screened_rates(k_Ar39_to_He4_S35)*Y(Ar39)) +
        (-screened_rates(k_He4_S36_to_Ar40)*Y(He4)*Y(S36)*state.rho + screened_rates(k_Ar40_to_He4_S36)*Y(Ar40)) +
        (-screened_rates(k_He4_Cl33_to_K37)*Y(Cl33)*Y(He4)*state.rho + screened_rates(k_K37_to_He4_Cl33)*Y(K37)) +
        (-screened_rates(k_He4_Cl34_to_K38)*Y(Cl34)*Y(He4)*state.rho + screened_rates(k_K38_to_He4_Cl34)*Y(K38)) +
        (-screened_rates(k_He4_Cl35_to_K39)*Y(Cl35)*Y(He4)*state.rho + screened_rates(k_K39_to_He4_Cl35)*Y(K39)) +
        (-screened_rates(k_He4_Cl36_to_K40)*Y(Cl36)*Y(He4)*state.rho + screened_rates(k_K40_to_He4_Cl36)*Y(K40)) +
        (-screened_rates(k_He4_Cl37_to_K41)*Y(Cl37)*Y(He4)*state.rho + screened_rates(k_K41_to_He4_Cl37)*Y(K41)) +
        (-screened_rates(k_He4_Ar36_to_Ca40)*Y(Ar36)*Y(He4)*state.rho + screened_rates(k_Ca40_to_He4_Ar36)*Y(Ca40)) +
        (-screened_rates(k_He4_Ar37_to_Ca41)*Y(Ar37)*Y(He4)*state.rho + screened_rates(k_Ca41_to_He4_Ar37)*Y(Ca41)) +
        (-screened_rates(k_He4_Ar38_to_Ca42)*Y(Ar38)*Y(He4)*state.rho + screened_rates(k_Ca42_to_He4_Ar38)*Y(Ca42)) +
        (-screened_rates(k_He4_Ar39_to_Ca43)*Y(Ar39)*Y(He4)*state.rho + screened_rates(k_Ca43_to_He4_Ar39)*Y(Ca43)) +
        (-screened_rates(k_He4_Ar40_to_Ca44)*Y(Ar40)*Y(He4)*state.rho + screened_rates(k_Ca44_to_He4_Ar40)*Y(Ca44)) +
        (-screened_rates(k_He4_K39_to_Sc43)*Y(He4)*Y(K39)*state.rho + screened_rates(k_Sc43_to_He4_K39)*Y(Sc43)) +
        (-screened_rates(k_He4_K40_to_Sc44)*Y(He4)*Y(K40)*state.rho + screened_rates(k_Sc44_to_He4_K40)*Y(Sc44)) +
        (-screened_rates(k_He4_K41_to_Sc45)*Y(He4)*Y(K41)*state.rho + screened_rates(k_Sc45_to_He4_K41)*Y(Sc45)) +
        (-screened_rates(k_He4_Ca40_to_Ti44)*Y(Ca40)*Y(He4)*state.rho + screened_rates(k_Ti44_to_He4_Ca40)*Y(Ti44)) +
        (-screened_rates(k_He4_Ca41_to_Ti45)*Y(Ca41)*Y(He4)*state.rho + screened_rates(k_Ti45_to_He4_Ca41)*Y(Ti45)) +
        (-screened_rates(k_He4_Ca42_to_Ti46)*Y(Ca42)*Y(He4)*state.rho + screened_rates(k_Ti46_to_He4_Ca42)*Y(Ti46)) +
        (-screened_rates(k_He4_Ca43_to_Ti47)*Y(Ca43)*Y(He4)*state.rho + screened_rates(k_Ti47_to_He4_Ca43)*Y(Ti47)) +
        (-screened_rates(k_He4_Ca44_to_Ti48)*Y(Ca44)*Y(He4)*state.rho + screened_rates(k_Ti48_to_He4_Ca44)*Y(Ti48)) +
        (-screened_rates(k_He4_Ca45_to_Ti49)*Y(Ca45)*Y(He4)*state.rho + screened_rates(k_Ti49_to_He4_Ca45)*Y(Ti49)) +
        (-screened_rates(k_He4_Ca46_to_Ti50)*Y(Ca46)*Y(He4)*state.rho + screened_rates(k_Ti50_to_He4_Ca46)*Y(Ti50)) +
        (-screened_rates(k_He4_Ca47_to_Ti51)*Y(Ca47)*Y(He4)*state.rho + screened_rates(k_Ti51_to_He4_Ca47)*Y(Ti51)) +
        (-screened_rates(k_He4_Sc43_to_V47)*Y(He4)*Y(Sc43)*state.rho + screened_rates(k_V47_to_He4_Sc43)*Y(V47)) +
        (-screened_rates(k_He4_Sc44_to_V48)*Y(He4)*Y(Sc44)*state.rho + screened_rates(k_V48_to_He4_Sc44)*Y(V48)) +
        (-screened_rates(k_He4_Sc45_to_V49)*Y(He4)*Y(Sc45)*state.rho + screened_rates(k_V49_to_He4_Sc45)*Y(V49)) +
        (-screened_rates(k_He4_Sc46_to_V50)*Y(He4)*Y(Sc46)*state.rho + screened_rates(k_V50_to_He4_Sc46)*Y(V50)) +
        (-screened_rates(k_He4_Sc47_to_V51)*Y(He4)*Y(Sc47)*state.rho + screened_rates(k_V51_to_He4_Sc47)*Y(V51)) +
        (-screened_rates(k_He4_Sc48_to_V52)*Y(He4)*Y(Sc48)*state.rho + screened_rates(k_V52_to_He4_Sc48)*Y(V52)) +
        (-screened_rates(k_He4_Ti44_to_Cr48)*Y(He4)*Y(Ti44)*state.rho + screened_rates(k_Cr48_to_He4_Ti44)*Y(Cr48)) +
        (-screened_rates(k_He4_Ti45_to_Cr49)*Y(He4)*Y(Ti45)*state.rho + screened_rates(k_Cr49_to_He4_Ti45)*Y(Cr49)) +
        (-screened_rates(k_He4_Ti46_to_Cr50)*Y(He4)*Y(Ti46)*state.rho + screened_rates(k_Cr50_to_He4_Ti46)*Y(Cr50)) +
        (-screened_rates(k_He4_Ti47_to_Cr51)*Y(He4)*Y(Ti47)*state.rho + screened_rates(k_Cr51_to_He4_Ti47)*Y(Cr51)) +
        (-screened_rates(k_He4_Ti48_to_Cr52)*Y(He4)*Y(Ti48)*state.rho + screened_rates(k_Cr52_to_He4_Ti48)*Y(Cr52)) +
        (-screened_rates(k_He4_Ti49_to_Cr53)*Y(He4)*Y(Ti49)*state.rho + screened_rates(k_Cr53_to_He4_Ti49)*Y(Cr53)) +
        (-screened_rates(k_He4_Ti50_to_Cr54)*Y(He4)*Y(Ti50)*state.rho + screened_rates(k_Cr54_to_He4_Ti50)*Y(Cr54)) +
        (-screened_rates(k_He4_V46_to_Mn50)*Y(He4)*Y(V46)*state.rho + screened_rates(k_Mn50_to_He4_V46)*Y(Mn50)) +
        (-screened_rates(k_He4_V47_to_Mn51)*Y(He4)*Y(V47)*state.rho + screened_rates(k_Mn51_to_He4_V47)*Y(Mn51)) +
        (-screened_rates(k_He4_V48_to_Mn52)*Y(He4)*Y(V48)*state.rho + screened_rates(k_Mn52_to_He4_V48)*Y(Mn52)) +
        (-screened_rates(k_He4_V49_to_Mn53)*Y(He4)*Y(V49)*state.rho + screened_rates(k_Mn53_to_He4_V49)*Y(Mn53)) +
        (-screened_rates(k_He4_V50_to_Mn54)*Y(He4)*Y(V50)*state.rho + screened_rates(k_Mn54_to_He4_V50)*Y(Mn54)) +
        (-screened_rates(k_He4_V51_to_Mn55)*Y(He4)*Y(V51)*state.rho + screened_rates(k_Mn55_to_He4_V51)*Y(Mn55)) +
        (-screened_rates(k_He4_Cr48_to_Fe52)*Y(Cr48)*Y(He4)*state.rho + screened_rates(k_Fe52_to_He4_Cr48)*Y(Fe52)) +
        (-screened_rates(k_He4_Cr49_to_Fe53)*Y(Cr49)*Y(He4)*state.rho + screened_rates(k_Fe53_to_He4_Cr49)*Y(Fe53)) +
        (-screened_rates(k_He4_Cr50_to_Fe54)*Y(Cr50)*Y(He4)*state.rho + screened_rates(k_Fe54_to_He4_Cr50)*Y(Fe54)) +
        (-screened_rates(k_He4_Cr51_to_Fe55)*Y(Cr51)*Y(He4)*state.rho + screened_rates(k_Fe55_to_He4_Cr51)*Y(Fe55)) +
        (-screened_rates(k_He4_Cr52_to_Fe56)*Y(Cr52)*Y(He4)*state.rho + screened_rates(k_Fe56_to_He4_Cr52)*Y(Fe56)) +
        (-screened_rates(k_He4_Cr53_to_Fe57)*Y(Cr53)*Y(He4)*state.rho + screened_rates(k_Fe57_to_He4_Cr53)*Y(Fe57)) +
        (-screened_rates(k_He4_Cr54_to_Fe58)*Y(Cr54)*Y(He4)*state.rho + screened_rates(k_Fe58_to_He4_Cr54)*Y(Fe58)) +
        (-screened_rates(k_He4_Mn50_to_Co54)*Y(He4)*Y(Mn50)*state.rho + screened_rates(k_Co54_to_He4_Mn50)*Y(Co54)) +
        (-screened_rates(k_He4_Mn51_to_Co55)*Y(He4)*Y(Mn51)*state.rho + screened_rates(k_Co55_to_He4_Mn51)*Y(Co55)) +
        (-screened_rates(k_He4_Mn52_to_Co56)*Y(He4)*Y(Mn52)*state.rho + screened_rates(k_Co56_to_He4_Mn52)*Y(Co56)) +
        (-screened_rates(k_He4_Mn53_to_Co57)*Y(He4)*Y(Mn53)*state.rho + screened_rates(k_Co57_to_He4_Mn53)*Y(Co57)) +
        (-screened_rates(k_He4_Mn54_to_Co58)*Y(He4)*Y(Mn54)*state.rho + screened_rates(k_Co58_to_He4_Mn54)*Y(Co58)) +
        (-screened_rates(k_He4_Mn55_to_Co59)*Y(He4)*Y(Mn55)*state.rho + screened_rates(k_Co59_to_He4_Mn55)*Y(Co59)) +
        (-screened_rates(k_He4_Fe52_to_Ni56)*Y(Fe52)*Y(He4)*state.rho + screened_rates(k_Ni56_to_He4_Fe52)*Y(Ni56)) +
        (-screened_rates(k_He4_Fe53_to_Ni57)*Y(Fe53)*Y(He4)*state.rho + screened_rates(k_Ni57_to_He4_Fe53)*Y(Ni57)) +
        (-screened_rates(k_He4_Fe54_to_Ni58)*Y(Fe54)*Y(He4)*state.rho + screened_rates(k_Ni58_to_He4_Fe54)*Y(Ni58)) +
        (-screened_rates(k_He4_Fe55_to_Ni59)*Y(Fe55)*Y(He4)*state.rho + screened_rates(k_Ni59_to_He4_Fe55)*Y(Ni59)) +
        (-screened_rates(k_He4_Fe56_to_Ni60)*Y(Fe56)*Y(He4)*state.rho + screened_rates(k_Ni60_to_He4_Fe56)*Y(Ni60)) +
        (-screened_rates(k_He4_Fe57_to_Ni61)*Y(Fe57)*Y(He4)*state.rho + screened_rates(k_Ni61_to_He4_Fe57)*Y(Ni61)) +
        (-screened_rates(k_He4_Fe58_to_Ni62)*Y(Fe58)*Y(He4)*state.rho + screened_rates(k_Ni62_to_He4_Fe58)*Y(Ni62)) +
        (-screened_rates(k_He4_Co53_to_Cu57)*Y(Co53)*Y(He4)*state.rho + screened_rates(k_Cu57_to_He4_Co53)*Y(Cu57)) +
        (-screened_rates(k_He4_Co54_to_Cu58)*Y(Co54)*Y(He4)*state.rho + screened_rates(k_Cu58_to_He4_Co54)*Y(Cu58)) +
        (-screened_rates(k_He4_Co55_to_Cu59)*Y(Co55)*Y(He4)*state.rho + screened_rates(k_Cu59_to_He4_Co55)*Y(Cu59)) +
        (-screened_rates(k_He4_Co56_to_Cu60)*Y(Co56)*Y(He4)*state.rho + screened_rates(k_Cu60_to_He4_Co56)*Y(Cu60)) +
        (-screened_rates(k_He4_Co57_to_Cu61)*Y(Co57)*Y(He4)*state.rho + screened_rates(k_Cu61_to_He4_Co57)*Y(Cu61)) +
        (-screened_rates(k_He4_Co58_to_Cu62)*Y(Co58)*Y(He4)*state.rho + screened_rates(k_Cu62_to_He4_Co58)*Y(Cu62)) +
        (-screened_rates(k_He4_Co59_to_Cu63)*Y(Co59)*Y(He4)*state.rho + screened_rates(k_Cu63_to_He4_Co59)*Y(Cu63)) +
        (-screened_rates(k_He4_Ni56_to_Zn60)*Y(He4)*Y(Ni56)*state.rho + screened_rates(k_Zn60_to_He4_Ni56)*Y(Zn60)) +
        (-screened_rates(k_He4_Ni57_to_Zn61)*Y(He4)*Y(Ni57)*state.rho + screened_rates(k_Zn61_to_He4_Ni57)*Y(Zn61)) +
        (-screened_rates(k_He4_Ni58_to_Zn62)*Y(He4)*Y(Ni58)*state.rho + screened_rates(k_Zn62_to_He4_Ni58)*Y(Zn62)) +
        (-screened_rates(k_He4_Ni59_to_Zn63)*Y(He4)*Y(Ni59)*state.rho + screened_rates(k_Zn63_to_He4_Ni59)*Y(Zn63)) +
        (-screened_rates(k_He4_Ni60_to_Zn64)*Y(He4)*Y(Ni60)*state.rho + screened_rates(k_Zn64_to_He4_Ni60)*Y(Zn64)) +
        (-screened_rates(k_He4_Ni61_to_Zn65)*Y(He4)*Y(Ni61)*state.rho + screened_rates(k_Zn65_to_He4_Ni61)*Y(Zn65)) +
        (-screened_rates(k_He4_Ni62_to_Zn66)*Y(He4)*Y(Ni62)*state.rho + screened_rates(k_Zn66_to_He4_Ni62)*Y(Zn66)) +
        (-screened_rates(k_He4_Cu58_to_Ga62)*Y(Cu58)*Y(He4)*state.rho + screened_rates(k_Ga62_to_He4_Cu58)*Y(Ga62)) +
        (-screened_rates(k_He4_Cu59_to_Ga63)*Y(Cu59)*Y(He4)*state.rho + screened_rates(k_Ga63_to_He4_Cu59)*Y(Ga63)) +
        (-screened_rates(k_He4_Cu60_to_Ga64)*Y(Cu60)*Y(He4)*state.rho + screened_rates(k_Ga64_to_He4_Cu60)*Y(Ga64)) +
        (-screened_rates(k_He4_Zn59_to_Ge63)*Y(He4)*Y(Zn59)*state.rho + screened_rates(k_Ge63_to_He4_Zn59)*Y(Ge63)) +
        (-screened_rates(k_He4_Zn60_to_Ge64)*Y(He4)*Y(Zn60)*state.rho + screened_rates(k_Ge64_to_He4_Zn60)*Y(Ge64)) +
        (screened_rates(k_d_He3_to_p_He4)*Y(He3)*Y(H2)*state.rho + -screened_rates(k_p_He4_to_d_He3)*Y(He4)*Y(H1)*state.rho) +
        (screened_rates(k_p_Li6_to_He4_He3)*Y(Li6)*Y(H1)*state.rho + -screened_rates(k_He4_He3_to_p_Li6)*Y(He3)*Y(He4)*state.rho) +
        (2.0*screened_rates(k_p_Li7_to_He4_He4)*Y(Li7)*Y(H1)*state.rho + -screened_rates(k_He4_He4_to_p_Li7)*amrex::Math::powi<2>(Y(He4))*state.rho) +
        (2.0*screened_rates(k_n_Be7_to_He4_He4)*Y(Be7)*Y(N)*state.rho + -screened_rates(k_He4_He4_to_n_Be7)*amrex::Math::powi<2>(Y(He4))*state.rho) +
        (screened_rates(k_p_Be9_to_He4_Li6)*Y(Be9)*Y(H1)*state.rho + -screened_rates(k_He4_Li6_to_p_Be9)*Y(He4)*Y(Li6)*state.rho) +
        (-screened_rates(k_He4_Be9_to_n_C12)*Y(Be9)*Y(He4)*state.rho + screened_rates(k_n_C12_to_He4_Be9)*Y(C12)*Y(N)*state.rho) +
        (screened_rates(k_n_B10_to_He4_Li7)*Y(B10)*Y(N)*state.rho + -screened_rates(k_He4_Li7_to_n_B10)*Y(He4)*Y(Li7)*state.rho) +
        (screened_rates(k_p_B10_to_He4_Be7)*Y(B10)*Y(H1)*state.rho + -screened_rates(k_He4_Be7_to_p_B10)*Y(Be7)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_B10_to_n_N13)*Y(B10)*Y(He4)*state.rho + screened_rates(k_n_N13_to_He4_B10)*Y(N13)*Y(N)*state.rho) +
        (-screened_rates(k_He4_B10_to_p_C13)*Y(B10)*Y(He4)*state.rho + screened_rates(k_p_C13_to_He4_B10)*Y(C13)*Y(H1)*state.rho) +
        (-screened_rates(k_He4_B11_to_n_N14)*Y(B11)*Y(He4)*state.rho + screened_rates(k_n_N14_to_He4_B11)*Y(N14)*Y(N)*state.rho) +
        (-screened_rates(k_He4_B11_to_p_C14)*Y(B11)*Y(He4)*state.rho + screened_rates(k_p_C14_to_He4_B11)*Y(C14)*Y(H1)*state.rho) +
        (0.5*screened_rates(k_C12_C12_to_He4_Ne20)*amrex::Math::powi<2>(Y(C12))*state.rho + -screened_rates(k_He4_Ne20_to_C12_C12)*Y(He4)*Y(Ne20)*state.rho) +
        (-screened_rates(k_He4_C13_to_n_O16)*Y(C13)*Y(He4)*state.rho + screened_rates(k_n_O16_to_He4_C13)*Y(O16)*Y(N)*state.rho) +
        (-screened_rates(k_He4_N13_to_p_O16)*Y(He4)*Y(N13)*state.rho + screened_rates(k_p_O16_to_He4_N13)*Y(O16)*Y(H1)*state.rho) +
        (screened_rates(k_p_N15_to_He4_C12)*Y(N15)*Y(H1)*state.rho + -screened_rates(k_He4_C12_to_p_N15)*Y(C12)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_O14_to_p_F17)*Y(He4)*Y(O14)*state.rho + screened_rates(k_p_F17_to_He4_O14)*Y(F17)*Y(H1)*state.rho) +
        (screened_rates(k_n_O15_to_He4_C12)*Y(O15)*Y(N)*state.rho + -screened_rates(k_He4_C12_to_n_O15)*Y(C12)*Y(He4)*state.rho) +
        (screened_rates(k_C12_O16_to_He4_Mg24)*Y(C12)*Y(O16)*state.rho + -screened_rates(k_He4_Mg24_to_C12_O16)*Y(He4)*Y(Mg24)*state.rho) +
        (0.5*screened_rates(k_O16_O16_to_He4_Si28)*amrex::Math::powi<2>(Y(O16))*state.rho + -screened_rates(k_He4_Si28_to_O16_O16)*Y(He4)*Y(Si28)*state.rho) +
        (screened_rates(k_n_O17_to_He4_C14)*Y(O17)*Y(N)*state.rho + -screened_rates(k_He4_C14_to_n_O17)*Y(C14)*Y(He4)*state.rho) +
        (screened_rates(k_p_O17_to_He4_N14)*Y(O17)*Y(H1)*state.rho + -screened_rates(k_He4_N14_to_p_O17)*Y(He4)*Y(N14)*state.rho) +
        (-screened_rates(k_He4_O17_to_n_Ne20)*Y(He4)*Y(O17)*state.rho + screened_rates(k_n_Ne20_to_He4_O17)*Y(Ne20)*Y(N)*state.rho) +
        (screened_rates(k_p_O18_to_He4_N15)*Y(O18)*Y(H1)*state.rho + -screened_rates(k_He4_N15_to_p_O18)*Y(He4)*Y(N15)*state.rho) +
        (screened_rates(k_n_F17_to_He4_N14)*Y(F17)*Y(N)*state.rho + -screened_rates(k_He4_N14_to_n_F17)*Y(He4)*Y(N14)*state.rho) +
        (-screened_rates(k_He4_F17_to_p_Ne20)*Y(F17)*Y(He4)*state.rho + screened_rates(k_p_Ne20_to_He4_F17)*Y(Ne20)*Y(H1)*state.rho) +
        (screened_rates(k_n_F18_to_He4_N15)*Y(F18)*Y(N)*state.rho + -screened_rates(k_He4_N15_to_n_F18)*Y(He4)*Y(N15)*state.rho) +
        (screened_rates(k_p_F18_to_He4_O15)*Y(F18)*Y(H1)*state.rho + -screened_rates(k_He4_O15_to_p_F18)*Y(He4)*Y(O15)*state.rho) +
        (-screened_rates(k_He4_F18_to_p_Ne21)*Y(F18)*Y(He4)*state.rho + screened_rates(k_p_Ne21_to_He4_F18)*Y(Ne21)*Y(H1)*state.rho) +
        (screened_rates(k_p_F19_to_He4_O16)*Y(F19)*Y(H1)*state.rho + -screened_rates(k_He4_O16_to_p_F19)*Y(He4)*Y(O16)*state.rho) +
        (-screened_rates(k_He4_F19_to_p_Ne22)*Y(F19)*Y(He4)*state.rho + screened_rates(k_p_Ne22_to_He4_F19)*Y(Ne22)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ne18_to_He4_O15)*Y(Ne18)*Y(N)*state.rho + -screened_rates(k_He4_O15_to_n_Ne18)*Y(He4)*Y(O15)*state.rho) +
        (-screened_rates(k_He4_Ne18_to_p_Na21)*Y(He4)*Y(Ne18)*state.rho + screened_rates(k_p_Na21_to_He4_Ne18)*Y(Na21)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ne19_to_He4_O16)*Y(Ne19)*Y(N)*state.rho + -screened_rates(k_He4_O16_to_n_Ne19)*Y(He4)*Y(O16)*state.rho) +
        (-screened_rates(k_He4_Ne19_to_p_Na22)*Y(He4)*Y(Ne19)*state.rho + screened_rates(k_p_Na22_to_He4_Ne19)*Y(Na22)*Y(H1)*state.rho) +
        (screened_rates(k_C12_Ne20_to_He4_Si28)*Y(C12)*Y(Ne20)*state.rho + -screened_rates(k_He4_Si28_to_C12_Ne20)*Y(He4)*Y(Si28)*state.rho) +
        (screened_rates(k_n_Ne21_to_He4_O18)*Y(Ne21)*Y(N)*state.rho + -screened_rates(k_He4_O18_to_n_Ne21)*Y(He4)*Y(O18)*state.rho) +
        (-screened_rates(k_He4_Ne21_to_n_Mg24)*Y(He4)*Y(Ne21)*state.rho + screened_rates(k_n_Mg24_to_He4_Ne21)*Y(Mg24)*Y(N)*state.rho) +
        (screened_rates(k_n_Na21_to_He4_F18)*Y(Na21)*Y(N)*state.rho + -screened_rates(k_He4_F18_to_n_Na21)*Y(F18)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Na21_to_p_Mg24)*Y(He4)*Y(Na21)*state.rho + screened_rates(k_p_Mg24_to_He4_Na21)*Y(Mg24)*Y(H1)*state.rho) +
        (screened_rates(k_n_Na22_to_He4_F19)*Y(Na22)*Y(N)*state.rho + -screened_rates(k_He4_F19_to_n_Na22)*Y(F19)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Na22_to_p_Mg25)*Y(He4)*Y(Na22)*state.rho + screened_rates(k_p_Mg25_to_He4_Na22)*Y(Mg25)*Y(H1)*state.rho) +
        (screened_rates(k_p_Na23_to_He4_Ne20)*Y(Na23)*Y(H1)*state.rho + -screened_rates(k_He4_Ne20_to_p_Na23)*Y(He4)*Y(Ne20)*state.rho) +
        (-screened_rates(k_He4_Na23_to_p_Mg26)*Y(He4)*Y(Na23)*state.rho + screened_rates(k_p_Mg26_to_He4_Na23)*Y(Mg26)*Y(H1)*state.rho) +
        (screened_rates(k_n_Mg23_to_He4_Ne20)*Y(Mg23)*Y(N)*state.rho + -screened_rates(k_He4_Ne20_to_n_Mg23)*Y(He4)*Y(Ne20)*state.rho) +
        (-screened_rates(k_He4_Mg23_to_p_Al26)*Y(He4)*Y(Mg23)*state.rho + screened_rates(k_p_Al26_to_He4_Mg23)*Y(Al26)*Y(H1)*state.rho) +
        (screened_rates(k_n_Mg25_to_He4_Ne22)*Y(Mg25)*Y(N)*state.rho + -screened_rates(k_He4_Ne22_to_n_Mg25)*Y(He4)*Y(Ne22)*state.rho) +
        (-screened_rates(k_He4_Mg25_to_n_Si28)*Y(He4)*Y(Mg25)*state.rho + screened_rates(k_n_Si28_to_He4_Mg25)*Y(Si28)*Y(N)*state.rho) +
        (-screened_rates(k_He4_Mg26_to_n_Si29)*Y(He4)*Y(Mg26)*state.rho + screened_rates(k_n_Si29_to_He4_Mg26)*Y(Si29)*Y(N)*state.rho) +
        (screened_rates(k_n_Al25_to_He4_Na22)*Y(Al25)*Y(N)*state.rho + -screened_rates(k_He4_Na22_to_n_Al25)*Y(He4)*Y(Na22)*state.rho) +
        (-screened_rates(k_He4_Al25_to_p_Si28)*Y(Al25)*Y(He4)*state.rho + screened_rates(k_p_Si28_to_He4_Al25)*Y(Si28)*Y(H1)*state.rho) +
        (screened_rates(k_n_Al26_to_He4_Na23)*Y(Al26)*Y(N)*state.rho + -screened_rates(k_He4_Na23_to_n_Al26)*Y(He4)*Y(Na23)*state.rho) +
        (-screened_rates(k_He4_Al26_to_p_Si29)*Y(Al26)*Y(He4)*state.rho + screened_rates(k_p_Si29_to_He4_Al26)*Y(Si29)*Y(H1)*state.rho) +
        (screened_rates(k_p_Al27_to_He4_Mg24)*Y(Al27)*Y(H1)*state.rho + -screened_rates(k_He4_Mg24_to_p_Al27)*Y(He4)*Y(Mg24)*state.rho) +
        (-screened_rates(k_He4_Al27_to_p_Si30)*Y(Al27)*Y(He4)*state.rho + screened_rates(k_p_Si30_to_He4_Al27)*Y(Si30)*Y(H1)*state.rho) +
        (-screened_rates(k_He4_Si31_to_n_S34)*Y(He4)*Y(Si31)*state.rho + screened_rates(k_n_S34_to_He4_Si31)*Y(S34)*Y(N)*state.rho) +
        (screened_rates(k_n_P29_to_He4_Al26)*Y(P29)*Y(N)*state.rho + -screened_rates(k_He4_Al26_to_n_P29)*Y(Al26)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_P29_to_p_S32)*Y(He4)*Y(P29)*state.rho + screened_rates(k_p_S32_to_He4_P29)*Y(S32)*Y(H1)*state.rho) +
        (screened_rates(k_n_P30_to_He4_Al27)*Y(P30)*Y(N)*state.rho + -screened_rates(k_He4_Al27_to_n_P30)*Y(Al27)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_P30_to_p_S33)*Y(He4)*Y(P30)*state.rho + screened_rates(k_p_S33_to_He4_P30)*Y(S33)*Y(H1)*state.rho) +
        (screened_rates(k_p_P31_to_He4_Si28)*Y(P31)*Y(H1)*state.rho + -screened_rates(k_He4_Si28_to_p_P31)*Y(He4)*Y(Si28)*state.rho) +
        (-screened_rates(k_He4_P31_to_p_S34)*Y(He4)*Y(P31)*state.rho + screened_rates(k_p_S34_to_He4_P31)*Y(S34)*Y(H1)*state.rho) +
        (screened_rates(k_p_P32_to_He4_Si29)*Y(P32)*Y(H1)*state.rho + -screened_rates(k_He4_Si29_to_p_P32)*Y(He4)*Y(Si29)*state.rho) +
        (screened_rates(k_p_P33_to_He4_Si30)*Y(P33)*Y(H1)*state.rho + -screened_rates(k_He4_Si30_to_p_P33)*Y(He4)*Y(Si30)*state.rho) +
        (screened_rates(k_n_S32_to_He4_Si29)*Y(S32)*Y(N)*state.rho + -screened_rates(k_He4_Si29_to_n_S32)*Y(He4)*Y(Si29)*state.rho) +
        (screened_rates(k_n_S33_to_He4_Si30)*Y(S33)*Y(N)*state.rho + -screened_rates(k_He4_Si30_to_n_S33)*Y(He4)*Y(Si30)*state.rho) +
        (screened_rates(k_n_S35_to_He4_Si32)*Y(S35)*Y(N)*state.rho + -screened_rates(k_He4_Si32_to_n_S35)*Y(He4)*Y(Si32)*state.rho) +
        (screened_rates(k_p_S35_to_He4_P32)*Y(S35)*Y(H1)*state.rho + -screened_rates(k_He4_P32_to_p_S35)*Y(He4)*Y(P32)*state.rho) +
        (-screened_rates(k_He4_S35_to_n_Ar38)*Y(He4)*Y(S35)*state.rho + screened_rates(k_n_Ar38_to_He4_S35)*Y(Ar38)*Y(N)*state.rho) +
        (screened_rates(k_p_S36_to_He4_P33)*Y(S36)*Y(H1)*state.rho + -screened_rates(k_He4_P33_to_p_S36)*Y(He4)*Y(P33)*state.rho) +
        (screened_rates(k_n_Cl33_to_He4_P30)*Y(Cl33)*Y(N)*state.rho + -screened_rates(k_He4_P30_to_n_Cl33)*Y(He4)*Y(P30)*state.rho) +
        (-screened_rates(k_He4_Cl33_to_p_Ar36)*Y(Cl33)*Y(He4)*state.rho + screened_rates(k_p_Ar36_to_He4_Cl33)*Y(Ar36)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cl34_to_He4_P31)*Y(Cl34)*Y(N)*state.rho + -screened_rates(k_He4_P31_to_n_Cl34)*Y(He4)*Y(P31)*state.rho) +
        (-screened_rates(k_He4_Cl34_to_p_Ar37)*Y(Cl34)*Y(He4)*state.rho + screened_rates(k_p_Ar37_to_He4_Cl34)*Y(Ar37)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cl35_to_He4_P32)*Y(Cl35)*Y(N)*state.rho + -screened_rates(k_He4_P32_to_n_Cl35)*Y(He4)*Y(P32)*state.rho) +
        (screened_rates(k_p_Cl35_to_He4_S32)*Y(Cl35)*Y(H1)*state.rho + -screened_rates(k_He4_S32_to_p_Cl35)*Y(He4)*Y(S32)*state.rho) +
        (-screened_rates(k_He4_Cl35_to_p_Ar38)*Y(Cl35)*Y(He4)*state.rho + screened_rates(k_p_Ar38_to_He4_Cl35)*Y(Ar38)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cl36_to_He4_P33)*Y(Cl36)*Y(N)*state.rho + -screened_rates(k_He4_P33_to_n_Cl36)*Y(He4)*Y(P33)*state.rho) +
        (screened_rates(k_p_Cl36_to_He4_S33)*Y(Cl36)*Y(H1)*state.rho + -screened_rates(k_He4_S33_to_p_Cl36)*Y(He4)*Y(S33)*state.rho) +
        (screened_rates(k_p_Cl37_to_He4_S34)*Y(Cl37)*Y(H1)*state.rho + -screened_rates(k_He4_S34_to_p_Cl37)*Y(He4)*Y(S34)*state.rho) +
        (screened_rates(k_n_Ar36_to_He4_S33)*Y(Ar36)*Y(N)*state.rho + -screened_rates(k_He4_S33_to_n_Ar36)*Y(He4)*Y(S33)*state.rho) +
        (screened_rates(k_n_Ar37_to_He4_S34)*Y(Ar37)*Y(N)*state.rho + -screened_rates(k_He4_S34_to_n_Ar37)*Y(He4)*Y(S34)*state.rho) +
        (screened_rates(k_n_Ar39_to_He4_S36)*Y(Ar39)*Y(N)*state.rho + -screened_rates(k_He4_S36_to_n_Ar39)*Y(He4)*Y(S36)*state.rho) +
        (screened_rates(k_p_Ar39_to_He4_Cl36)*Y(Ar39)*Y(H1)*state.rho + -screened_rates(k_He4_Cl36_to_p_Ar39)*Y(Cl36)*Y(He4)*state.rho) +
        (screened_rates(k_p_Ar40_to_He4_Cl37)*Y(Ar40)*Y(H1)*state.rho + -screened_rates(k_He4_Cl37_to_p_Ar40)*Y(Cl37)*Y(He4)*state.rho) +
        (screened_rates(k_n_K37_to_He4_Cl34)*Y(K37)*Y(N)*state.rho + -screened_rates(k_He4_Cl34_to_n_K37)*Y(Cl34)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_K37_to_p_Ca40)*Y(He4)*Y(K37)*state.rho + screened_rates(k_p_Ca40_to_He4_K37)*Y(Ca40)*Y(H1)*state.rho) +
        (screened_rates(k_n_K38_to_He4_Cl35)*Y(K38)*Y(N)*state.rho + -screened_rates(k_He4_Cl35_to_n_K38)*Y(Cl35)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_K38_to_p_Ca41)*Y(He4)*Y(K38)*state.rho + screened_rates(k_p_Ca41_to_He4_K38)*Y(Ca41)*Y(H1)*state.rho) +
        (screened_rates(k_n_K39_to_He4_Cl36)*Y(K39)*Y(N)*state.rho + -screened_rates(k_He4_Cl36_to_n_K39)*Y(Cl36)*Y(He4)*state.rho) +
        (screened_rates(k_p_K39_to_He4_Ar36)*Y(K39)*Y(H1)*state.rho + -screened_rates(k_He4_Ar36_to_p_K39)*Y(Ar36)*Y(He4)*state.rho) +
        (screened_rates(k_n_K40_to_He4_Cl37)*Y(K40)*Y(N)*state.rho + -screened_rates(k_He4_Cl37_to_n_K40)*Y(Cl37)*Y(He4)*state.rho) +
        (screened_rates(k_p_K40_to_He4_Ar37)*Y(K40)*Y(H1)*state.rho + -screened_rates(k_He4_Ar37_to_p_K40)*Y(Ar37)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_K40_to_p_Ca43)*Y(He4)*Y(K40)*state.rho + screened_rates(k_p_Ca43_to_He4_K40)*Y(Ca43)*Y(H1)*state.rho) +
        (screened_rates(k_p_K41_to_He4_Ar38)*Y(K41)*Y(H1)*state.rho + -screened_rates(k_He4_Ar38_to_p_K41)*Y(Ar38)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_K41_to_p_Ca44)*Y(He4)*Y(K41)*state.rho + screened_rates(k_p_Ca44_to_He4_K41)*Y(Ca44)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ca40_to_He4_Ar37)*Y(Ca40)*Y(N)*state.rho + -screened_rates(k_He4_Ar37_to_n_Ca40)*Y(Ar37)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ca41_to_He4_Ar38)*Y(Ca41)*Y(N)*state.rho + -screened_rates(k_He4_Ar38_to_n_Ca41)*Y(Ar38)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ca42_to_He4_Ar39)*Y(Ca42)*Y(N)*state.rho + -screened_rates(k_He4_Ar39_to_n_Ca42)*Y(Ar39)*Y(He4)*state.rho) +
        (screened_rates(k_p_Ca42_to_He4_K39)*Y(Ca42)*Y(H1)*state.rho + -screened_rates(k_He4_K39_to_p_Ca42)*Y(He4)*Y(K39)*state.rho) +
        (screened_rates(k_n_Ca43_to_He4_Ar40)*Y(Ca43)*Y(N)*state.rho + -screened_rates(k_He4_Ar40_to_n_Ca43)*Y(Ar40)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Ca43_to_n_Ti46)*Y(Ca43)*Y(He4)*state.rho + screened_rates(k_n_Ti46_to_He4_Ca43)*Y(Ti46)*Y(N)*state.rho) +
        (-screened_rates(k_He4_Ca45_to_n_Ti48)*Y(Ca45)*Y(He4)*state.rho + screened_rates(k_n_Ti48_to_He4_Ca45)*Y(Ti48)*Y(N)*state.rho) +
        (-screened_rates(k_He4_Ca47_to_n_Ti50)*Y(Ca47)*Y(He4)*state.rho + screened_rates(k_n_Ti50_to_He4_Ca47)*Y(Ti50)*Y(N)*state.rho) +
        (screened_rates(k_n_Sc43_to_He4_K40)*Y(Sc43)*Y(N)*state.rho + -screened_rates(k_He4_K40_to_n_Sc43)*Y(He4)*Y(K40)*state.rho) +
        (screened_rates(k_p_Sc43_to_He4_Ca40)*Y(Sc43)*Y(H1)*state.rho + -screened_rates(k_He4_Ca40_to_p_Sc43)*Y(Ca40)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Sc43_to_p_Ti46)*Y(He4)*Y(Sc43)*state.rho + screened_rates(k_p_Ti46_to_He4_Sc43)*Y(Ti46)*Y(H1)*state.rho) +
        (screened_rates(k_n_Sc44_to_He4_K41)*Y(Sc44)*Y(N)*state.rho + -screened_rates(k_He4_K41_to_n_Sc44)*Y(He4)*Y(K41)*state.rho) +
        (screened_rates(k_p_Sc44_to_He4_Ca41)*Y(Sc44)*Y(H1)*state.rho + -screened_rates(k_He4_Ca41_to_p_Sc44)*Y(Ca41)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Sc44_to_p_Ti47)*Y(He4)*Y(Sc44)*state.rho + screened_rates(k_p_Ti47_to_He4_Sc44)*Y(Ti47)*Y(H1)*state.rho) +
        (screened_rates(k_p_Sc45_to_He4_Ca42)*Y(Sc45)*Y(H1)*state.rho + -screened_rates(k_He4_Ca42_to_p_Sc45)*Y(Ca42)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Sc45_to_p_Ti48)*Y(He4)*Y(Sc45)*state.rho + screened_rates(k_p_Ti48_to_He4_Sc45)*Y(Ti48)*Y(H1)*state.rho) +
        (screened_rates(k_p_Sc46_to_He4_Ca43)*Y(Sc46)*Y(H1)*state.rho + -screened_rates(k_He4_Ca43_to_p_Sc46)*Y(Ca43)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Sc46_to_n_V49)*Y(He4)*Y(Sc46)*state.rho + screened_rates(k_n_V49_to_He4_Sc46)*Y(V49)*Y(N)*state.rho) +
        (-screened_rates(k_He4_Sc46_to_p_Ti49)*Y(He4)*Y(Sc46)*state.rho + screened_rates(k_p_Ti49_to_He4_Sc46)*Y(Ti49)*Y(H1)*state.rho) +
        (screened_rates(k_p_Sc47_to_He4_Ca44)*Y(Sc47)*Y(H1)*state.rho + -screened_rates(k_He4_Ca44_to_p_Sc47)*Y(Ca44)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Sc47_to_p_Ti50)*Y(He4)*Y(Sc47)*state.rho + screened_rates(k_p_Ti50_to_He4_Sc47)*Y(Ti50)*Y(H1)*state.rho) +
        (screened_rates(k_p_Sc48_to_He4_Ca45)*Y(Sc48)*Y(H1)*state.rho + -screened_rates(k_He4_Ca45_to_p_Sc48)*Y(Ca45)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Sc48_to_n_V51)*Y(He4)*Y(Sc48)*state.rho + screened_rates(k_n_V51_to_He4_Sc48)*Y(V51)*Y(N)*state.rho) +
        (-screened_rates(k_He4_Sc48_to_p_Ti51)*Y(He4)*Y(Sc48)*state.rho + screened_rates(k_p_Ti51_to_He4_Sc48)*Y(Ti51)*Y(H1)*state.rho) +
        (screened_rates(k_p_Sc49_to_He4_Ca46)*Y(Sc49)*Y(H1)*state.rho + -screened_rates(k_He4_Ca46_to_p_Sc49)*Y(Ca46)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ti44_to_He4_Ca41)*Y(Ti44)*Y(N)*state.rho + -screened_rates(k_He4_Ca41_to_n_Ti44)*Y(Ca41)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ti45_to_He4_Ca42)*Y(Ti45)*Y(N)*state.rho + -screened_rates(k_He4_Ca42_to_n_Ti45)*Y(Ca42)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Ti45_to_p_V48)*Y(He4)*Y(Ti45)*state.rho + screened_rates(k_p_V48_to_He4_Ti45)*Y(V48)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ti47_to_He4_Ca44)*Y(Ti47)*Y(N)*state.rho + -screened_rates(k_He4_Ca44_to_n_Ti47)*Y(Ca44)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ti49_to_He4_Ca46)*Y(Ti49)*Y(N)*state.rho + -screened_rates(k_He4_Ca46_to_n_Ti49)*Y(Ca46)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Ti49_to_n_Cr52)*Y(He4)*Y(Ti49)*state.rho + screened_rates(k_n_Cr52_to_He4_Ti49)*Y(Cr52)*Y(N)*state.rho) +
        (screened_rates(k_n_Ti51_to_He4_Ca48)*Y(Ti51)*Y(N)*state.rho + -screened_rates(k_He4_Ca48_to_n_Ti51)*Y(Ca48)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Ti51_to_n_Cr54)*Y(He4)*Y(Ti51)*state.rho + screened_rates(k_n_Cr54_to_He4_Ti51)*Y(Cr54)*Y(N)*state.rho) +
        (screened_rates(k_n_V46_to_He4_Sc43)*Y(V46)*Y(N)*state.rho + -screened_rates(k_He4_Sc43_to_n_V46)*Y(He4)*Y(Sc43)*state.rho) +
        (-screened_rates(k_He4_V46_to_p_Cr49)*Y(He4)*Y(V46)*state.rho + screened_rates(k_p_Cr49_to_He4_V46)*Y(Cr49)*Y(H1)*state.rho) +
        (screened_rates(k_n_V47_to_He4_Sc44)*Y(V47)*Y(N)*state.rho + -screened_rates(k_He4_Sc44_to_n_V47)*Y(He4)*Y(Sc44)*state.rho) +
        (screened_rates(k_p_V47_to_He4_Ti44)*Y(V47)*Y(H1)*state.rho + -screened_rates(k_He4_Ti44_to_p_V47)*Y(He4)*Y(Ti44)*state.rho) +
        (-screened_rates(k_He4_V47_to_p_Cr50)*Y(He4)*Y(V47)*state.rho + screened_rates(k_p_Cr50_to_He4_V47)*Y(Cr50)*Y(H1)*state.rho) +
        (screened_rates(k_n_V48_to_He4_Sc45)*Y(V48)*Y(N)*state.rho + -screened_rates(k_He4_Sc45_to_n_V48)*Y(He4)*Y(Sc45)*state.rho) +
        (-screened_rates(k_He4_V48_to_p_Cr51)*Y(He4)*Y(V48)*state.rho + screened_rates(k_p_Cr51_to_He4_V48)*Y(Cr51)*Y(H1)*state.rho) +
        (screened_rates(k_p_V49_to_He4_Ti46)*Y(V49)*Y(H1)*state.rho + -screened_rates(k_He4_Ti46_to_p_V49)*Y(He4)*Y(Ti46)*state.rho) +
        (-screened_rates(k_He4_V49_to_p_Cr52)*Y(He4)*Y(V49)*state.rho + screened_rates(k_p_Cr52_to_He4_V49)*Y(Cr52)*Y(H1)*state.rho) +
        (screened_rates(k_n_V50_to_He4_Sc47)*Y(V50)*Y(N)*state.rho + -screened_rates(k_He4_Sc47_to_n_V50)*Y(He4)*Y(Sc47)*state.rho) +
        (screened_rates(k_p_V50_to_He4_Ti47)*Y(V50)*Y(H1)*state.rho + -screened_rates(k_He4_Ti47_to_p_V50)*Y(He4)*Y(Ti47)*state.rho) +
        (-screened_rates(k_He4_V50_to_p_Cr53)*Y(He4)*Y(V50)*state.rho + screened_rates(k_p_Cr53_to_He4_V50)*Y(Cr53)*Y(H1)*state.rho) +
        (screened_rates(k_p_V51_to_He4_Ti48)*Y(V51)*Y(H1)*state.rho + -screened_rates(k_He4_Ti48_to_p_V51)*Y(He4)*Y(Ti48)*state.rho) +
        (screened_rates(k_n_V52_to_He4_Sc49)*Y(V52)*Y(N)*state.rho + -screened_rates(k_He4_Sc49_to_n_V52)*Y(He4)*Y(Sc49)*state.rho) +
        (screened_rates(k_p_V52_to_He4_Ti49)*Y(V52)*Y(H1)*state.rho + -screened_rates(k_He4_Ti49_to_p_V52)*Y(He4)*Y(Ti49)*state.rho) +
        (-screened_rates(k_He4_V52_to_n_Mn55)*Y(He4)*Y(V52)*state.rho + screened_rates(k_n_Mn55_to_He4_V52)*Y(Mn55)*Y(N)*state.rho) +
        (screened_rates(k_n_Cr48_to_He4_Ti45)*Y(Cr48)*Y(N)*state.rho + -screened_rates(k_He4_Ti45_to_n_Cr48)*Y(He4)*Y(Ti45)*state.rho) +
        (-screened_rates(k_He4_Cr48_to_p_Mn51)*Y(Cr48)*Y(He4)*state.rho + screened_rates(k_p_Mn51_to_He4_Cr48)*Y(Mn51)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cr49_to_He4_Ti46)*Y(Cr49)*Y(N)*state.rho + -screened_rates(k_He4_Ti46_to_n_Cr49)*Y(He4)*Y(Ti46)*state.rho) +
        (-screened_rates(k_He4_Cr49_to_p_Mn52)*Y(Cr49)*Y(He4)*state.rho + screened_rates(k_p_Mn52_to_He4_Cr49)*Y(Mn52)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cr50_to_He4_Ti47)*Y(Cr50)*Y(N)*state.rho + -screened_rates(k_He4_Ti47_to_n_Cr50)*Y(He4)*Y(Ti47)*state.rho) +
        (screened_rates(k_n_Cr51_to_He4_Ti48)*Y(Cr51)*Y(N)*state.rho + -screened_rates(k_He4_Ti48_to_n_Cr51)*Y(He4)*Y(Ti48)*state.rho) +
        (screened_rates(k_n_Cr53_to_He4_Ti50)*Y(Cr53)*Y(N)*state.rho + -screened_rates(k_He4_Ti50_to_n_Cr53)*Y(He4)*Y(Ti50)*state.rho) +
        (screened_rates(k_p_Cr54_to_He4_V51)*Y(Cr54)*Y(H1)*state.rho + -screened_rates(k_He4_V51_to_p_Cr54)*Y(He4)*Y(V51)*state.rho) +
        (screened_rates(k_n_Mn50_to_He4_V47)*Y(Mn50)*Y(N)*state.rho + -screened_rates(k_He4_V47_to_n_Mn50)*Y(He4)*Y(V47)*state.rho) +
        (-screened_rates(k_He4_Mn50_to_p_Fe53)*Y(He4)*Y(Mn50)*state.rho + screened_rates(k_p_Fe53_to_He4_Mn50)*Y(Fe53)*Y(H1)*state.rho) +
        (screened_rates(k_n_Mn51_to_He4_V48)*Y(Mn51)*Y(N)*state.rho + -screened_rates(k_He4_V48_to_n_Mn51)*Y(He4)*Y(V48)*state.rho) +
        (-screened_rates(k_He4_Mn51_to_p_Fe54)*Y(He4)*Y(Mn51)*state.rho + screened_rates(k_p_Fe54_to_He4_Mn51)*Y(Fe54)*Y(H1)*state.rho) +
        (screened_rates(k_n_Mn52_to_He4_V49)*Y(Mn52)*Y(N)*state.rho + -screened_rates(k_He4_V49_to_n_Mn52)*Y(He4)*Y(V49)*state.rho) +
        (-screened_rates(k_He4_Mn52_to_p_Fe55)*Y(He4)*Y(Mn52)*state.rho + screened_rates(k_p_Fe55_to_He4_Mn52)*Y(Fe55)*Y(H1)*state.rho) +
        (screened_rates(k_n_Mn53_to_He4_V50)*Y(Mn53)*Y(N)*state.rho + -screened_rates(k_He4_V50_to_n_Mn53)*Y(He4)*Y(V50)*state.rho) +
        (screened_rates(k_p_Mn53_to_He4_Cr50)*Y(Mn53)*Y(H1)*state.rho + -screened_rates(k_He4_Cr50_to_p_Mn53)*Y(Cr50)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Mn53_to_p_Fe56)*Y(He4)*Y(Mn53)*state.rho + screened_rates(k_p_Fe56_to_He4_Mn53)*Y(Fe56)*Y(H1)*state.rho) +
        (screened_rates(k_n_Mn54_to_He4_V51)*Y(Mn54)*Y(N)*state.rho + -screened_rates(k_He4_V51_to_n_Mn54)*Y(He4)*Y(V51)*state.rho) +
        (screened_rates(k_p_Mn54_to_He4_Cr51)*Y(Mn54)*Y(H1)*state.rho + -screened_rates(k_He4_Cr51_to_p_Mn54)*Y(Cr51)*Y(He4)*state.rho) +
        (screened_rates(k_p_Mn55_to_He4_Cr52)*Y(Mn55)*Y(H1)*state.rho + -screened_rates(k_He4_Cr52_to_p_Mn55)*Y(Cr52)*Y(He4)*state.rho) +
        (screened_rates(k_n_Fe52_to_He4_Cr49)*Y(Fe52)*Y(N)*state.rho + -screened_rates(k_He4_Cr49_to_n_Fe52)*Y(Cr49)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Fe52_to_p_Co55)*Y(Fe52)*Y(He4)*state.rho + screened_rates(k_p_Co55_to_He4_Fe52)*Y(Co55)*Y(H1)*state.rho) +
        (screened_rates(k_n_Fe53_to_He4_Cr50)*Y(Fe53)*Y(N)*state.rho + -screened_rates(k_He4_Cr50_to_n_Fe53)*Y(Cr50)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Fe53_to_p_Co56)*Y(Fe53)*Y(He4)*state.rho + screened_rates(k_p_Co56_to_He4_Fe53)*Y(Co56)*Y(H1)*state.rho) +
        (screened_rates(k_n_Fe54_to_He4_Cr51)*Y(Fe54)*Y(N)*state.rho + -screened_rates(k_He4_Cr51_to_n_Fe54)*Y(Cr51)*Y(He4)*state.rho) +
        (screened_rates(k_n_Fe55_to_He4_Cr52)*Y(Fe55)*Y(N)*state.rho + -screened_rates(k_He4_Cr52_to_n_Fe55)*Y(Cr52)*Y(He4)*state.rho) +
        (screened_rates(k_n_Fe56_to_He4_Cr53)*Y(Fe56)*Y(N)*state.rho + -screened_rates(k_He4_Cr53_to_n_Fe56)*Y(Cr53)*Y(He4)*state.rho) +
        (screened_rates(k_n_Fe57_to_He4_Cr54)*Y(Fe57)*Y(N)*state.rho + -screened_rates(k_He4_Cr54_to_n_Fe57)*Y(Cr54)*Y(He4)*state.rho) +
        (screened_rates(k_p_Fe57_to_He4_Mn54)*Y(Fe57)*Y(H1)*state.rho + -screened_rates(k_He4_Mn54_to_p_Fe57)*Y(He4)*Y(Mn54)*state.rho) +
        (screened_rates(k_p_Fe58_to_He4_Mn55)*Y(Fe58)*Y(H1)*state.rho + -screened_rates(k_He4_Mn55_to_p_Fe58)*Y(He4)*Y(Mn55)*state.rho) +
        (screened_rates(k_n_Co53_to_He4_Mn50)*Y(Co53)*Y(N)*state.rho + -screened_rates(k_He4_Mn50_to_n_Co53)*Y(He4)*Y(Mn50)*state.rho) +
        (-screened_rates(k_He4_Co53_to_p_Ni56)*Y(Co53)*Y(He4)*state.rho + screened_rates(k_p_Ni56_to_He4_Co53)*Y(Ni56)*Y(H1)*state.rho) +
        (screened_rates(k_n_Co54_to_He4_Mn51)*Y(Co54)*Y(N)*state.rho + -screened_rates(k_He4_Mn51_to_n_Co54)*Y(He4)*Y(Mn51)*state.rho) +
        (-screened_rates(k_He4_Co54_to_p_Ni57)*Y(Co54)*Y(He4)*state.rho + screened_rates(k_p_Ni57_to_He4_Co54)*Y(Ni57)*Y(H1)*state.rho) +
        (screened_rates(k_n_Co55_to_He4_Mn52)*Y(Co55)*Y(N)*state.rho + -screened_rates(k_He4_Mn52_to_n_Co55)*Y(He4)*Y(Mn52)*state.rho) +
        (-screened_rates(k_He4_Co55_to_p_Ni58)*Y(Co55)*Y(He4)*state.rho + screened_rates(k_p_Ni58_to_He4_Co55)*Y(Ni58)*Y(H1)*state.rho) +
        (screened_rates(k_n_Co56_to_He4_Mn53)*Y(Co56)*Y(N)*state.rho + -screened_rates(k_He4_Mn53_to_n_Co56)*Y(He4)*Y(Mn53)*state.rho) +
        (-screened_rates(k_He4_Co56_to_p_Ni59)*Y(Co56)*Y(He4)*state.rho + screened_rates(k_p_Ni59_to_He4_Co56)*Y(Ni59)*Y(H1)*state.rho) +
        (screened_rates(k_n_Co57_to_He4_Mn54)*Y(Co57)*Y(N)*state.rho + -screened_rates(k_He4_Mn54_to_n_Co57)*Y(He4)*Y(Mn54)*state.rho) +
        (screened_rates(k_p_Co57_to_He4_Fe54)*Y(Co57)*Y(H1)*state.rho + -screened_rates(k_He4_Fe54_to_p_Co57)*Y(Fe54)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Co57_to_p_Ni60)*Y(Co57)*Y(He4)*state.rho + screened_rates(k_p_Ni60_to_He4_Co57)*Y(Ni60)*Y(H1)*state.rho) +
        (screened_rates(k_n_Co58_to_He4_Mn55)*Y(Co58)*Y(N)*state.rho + -screened_rates(k_He4_Mn55_to_n_Co58)*Y(He4)*Y(Mn55)*state.rho) +
        (screened_rates(k_p_Co58_to_He4_Fe55)*Y(Co58)*Y(H1)*state.rho + -screened_rates(k_He4_Fe55_to_p_Co58)*Y(Fe55)*Y(He4)*state.rho) +
        (screened_rates(k_p_Co59_to_He4_Fe56)*Y(Co59)*Y(H1)*state.rho + -screened_rates(k_He4_Fe56_to_p_Co59)*Y(Fe56)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ni56_to_He4_Fe53)*Y(Ni56)*Y(N)*state.rho + -screened_rates(k_He4_Fe53_to_n_Ni56)*Y(Fe53)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ni57_to_He4_Fe54)*Y(Ni57)*Y(N)*state.rho + -screened_rates(k_He4_Fe54_to_n_Ni57)*Y(Fe54)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ni58_to_He4_Fe55)*Y(Ni58)*Y(N)*state.rho + -screened_rates(k_He4_Fe55_to_n_Ni58)*Y(Fe55)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ni59_to_He4_Fe56)*Y(Ni59)*Y(N)*state.rho + -screened_rates(k_He4_Fe56_to_n_Ni59)*Y(Fe56)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ni60_to_He4_Fe57)*Y(Ni60)*Y(N)*state.rho + -screened_rates(k_He4_Fe57_to_n_Ni60)*Y(Fe57)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ni61_to_He4_Fe58)*Y(Ni61)*Y(N)*state.rho + -screened_rates(k_He4_Fe58_to_n_Ni61)*Y(Fe58)*Y(He4)*state.rho) +
        (screened_rates(k_p_Ni61_to_He4_Co58)*Y(Ni61)*Y(H1)*state.rho + -screened_rates(k_He4_Co58_to_p_Ni61)*Y(Co58)*Y(He4)*state.rho) +
        (screened_rates(k_p_Ni62_to_He4_Co59)*Y(Ni62)*Y(H1)*state.rho + -screened_rates(k_He4_Co59_to_p_Ni62)*Y(Co59)*Y(He4)*state.rho) +
        (screened_rates(k_n_Cu57_to_He4_Co54)*Y(Cu57)*Y(N)*state.rho + -screened_rates(k_He4_Co54_to_n_Cu57)*Y(Co54)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Cu57_to_p_Zn60)*Y(Cu57)*Y(He4)*state.rho + screened_rates(k_p_Zn60_to_He4_Cu57)*Y(Zn60)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cu58_to_He4_Co55)*Y(Cu58)*Y(N)*state.rho + -screened_rates(k_He4_Co55_to_n_Cu58)*Y(Co55)*Y(He4)*state.rho) +
        (screened_rates(k_n_Cu59_to_He4_Co56)*Y(Cu59)*Y(N)*state.rho + -screened_rates(k_He4_Co56_to_n_Cu59)*Y(Co56)*Y(He4)*state.rho) +
        (screened_rates(k_p_Cu59_to_He4_Ni56)*Y(Cu59)*Y(H1)*state.rho + -screened_rates(k_He4_Ni56_to_p_Cu59)*Y(He4)*Y(Ni56)*state.rho) +
        (screened_rates(k_n_Cu60_to_He4_Co57)*Y(Cu60)*Y(N)*state.rho + -screened_rates(k_He4_Co57_to_n_Cu60)*Y(Co57)*Y(He4)*state.rho) +
        (screened_rates(k_p_Cu60_to_He4_Ni57)*Y(Cu60)*Y(H1)*state.rho + -screened_rates(k_He4_Ni57_to_p_Cu60)*Y(He4)*Y(Ni57)*state.rho) +
        (screened_rates(k_n_Cu61_to_He4_Co58)*Y(Cu61)*Y(N)*state.rho + -screened_rates(k_He4_Co58_to_n_Cu61)*Y(Co58)*Y(He4)*state.rho) +
        (screened_rates(k_p_Cu61_to_He4_Ni58)*Y(Cu61)*Y(H1)*state.rho + -screened_rates(k_He4_Ni58_to_p_Cu61)*Y(He4)*Y(Ni58)*state.rho) +
        (screened_rates(k_n_Cu62_to_He4_Co59)*Y(Cu62)*Y(N)*state.rho + -screened_rates(k_He4_Co59_to_n_Cu62)*Y(Co59)*Y(He4)*state.rho) +
        (screened_rates(k_p_Cu62_to_He4_Ni59)*Y(Cu62)*Y(H1)*state.rho + -screened_rates(k_He4_Ni59_to_p_Cu62)*Y(He4)*Y(Ni59)*state.rho) +
        (screened_rates(k_p_Cu63_to_He4_Ni60)*Y(Cu63)*Y(H1)*state.rho + -screened_rates(k_He4_Ni60_to_p_Cu63)*Y(He4)*Y(Ni60)*state.rho) +
        (screened_rates(k_p_Cu64_to_He4_Ni61)*Y(Cu64)*Y(H1)*state.rho + -screened_rates(k_He4_Ni61_to_p_Cu64)*Y(He4)*Y(Ni61)*state.rho) +
        (screened_rates(k_p_Cu65_to_He4_Ni62)*Y(Cu65)*Y(H1)*state.rho + -screened_rates(k_He4_Ni62_to_p_Cu65)*Y(He4)*Y(Ni62)*state.rho) +
        (screened_rates(k_n_Zn59_to_He4_Ni56)*Y(Zn59)*Y(N)*state.rho + -screened_rates(k_He4_Ni56_to_n_Zn59)*Y(He4)*Y(Ni56)*state.rho) +
        (screened_rates(k_n_Zn60_to_He4_Ni57)*Y(Zn60)*Y(N)*state.rho + -screened_rates(k_He4_Ni57_to_n_Zn60)*Y(He4)*Y(Ni57)*state.rho) +
        (screened_rates(k_n_Zn61_to_He4_Ni58)*Y(Zn61)*Y(N)*state.rho + -screened_rates(k_He4_Ni58_to_n_Zn61)*Y(He4)*Y(Ni58)*state.rho) +
        (screened_rates(k_p_Zn61_to_He4_Cu58)*Y(Zn61)*Y(H1)*state.rho + -screened_rates(k_He4_Cu58_to_p_Zn61)*Y(Cu58)*Y(He4)*state.rho) +
        (screened_rates(k_n_Zn62_to_He4_Ni59)*Y(Zn62)*Y(N)*state.rho + -screened_rates(k_He4_Ni59_to_n_Zn62)*Y(He4)*Y(Ni59)*state.rho) +
        (screened_rates(k_p_Zn62_to_He4_Cu59)*Y(Zn62)*Y(H1)*state.rho + -screened_rates(k_He4_Cu59_to_p_Zn62)*Y(Cu59)*Y(He4)*state.rho) +
        (screened_rates(k_n_Zn63_to_He4_Ni60)*Y(Zn63)*Y(N)*state.rho + -screened_rates(k_He4_Ni60_to_n_Zn63)*Y(He4)*Y(Ni60)*state.rho) +
        (screened_rates(k_p_Zn63_to_He4_Cu60)*Y(Zn63)*Y(H1)*state.rho + -screened_rates(k_He4_Cu60_to_p_Zn63)*Y(Cu60)*Y(He4)*state.rho) +
        (screened_rates(k_n_Zn64_to_He4_Ni61)*Y(Zn64)*Y(N)*state.rho + -screened_rates(k_He4_Ni61_to_n_Zn64)*Y(He4)*Y(Ni61)*state.rho) +
        (screened_rates(k_p_Zn64_to_He4_Cu61)*Y(Zn64)*Y(H1)*state.rho + -screened_rates(k_He4_Cu61_to_p_Zn64)*Y(Cu61)*Y(He4)*state.rho) +
        (screened_rates(k_n_Zn65_to_He4_Ni62)*Y(Zn65)*Y(N)*state.rho + -screened_rates(k_He4_Ni62_to_n_Zn65)*Y(He4)*Y(Ni62)*state.rho) +
        (screened_rates(k_p_Zn65_to_He4_Cu62)*Y(Zn65)*Y(H1)*state.rho + -screened_rates(k_He4_Cu62_to_p_Zn65)*Y(Cu62)*Y(He4)*state.rho) +
        (screened_rates(k_n_Zn66_to_He4_Ni63)*Y(Zn66)*Y(N)*state.rho + -screened_rates(k_He4_Ni63_to_n_Zn66)*Y(He4)*Y(Ni63)*state.rho) +
        (screened_rates(k_p_Zn66_to_He4_Cu63)*Y(Zn66)*Y(H1)*state.rho + -screened_rates(k_He4_Cu63_to_p_Zn66)*Y(Cu63)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ga62_to_He4_Cu59)*Y(Ga62)*Y(N)*state.rho + -screened_rates(k_He4_Cu59_to_n_Ga62)*Y(Cu59)*Y(He4)*state.rho) +
        (screened_rates(k_p_Ga62_to_He4_Zn59)*Y(Ga62)*Y(H1)*state.rho + -screened_rates(k_He4_Zn59_to_p_Ga62)*Y(He4)*Y(Zn59)*state.rho) +
        (screened_rates(k_n_Ga63_to_He4_Cu60)*Y(Ga63)*Y(N)*state.rho + -screened_rates(k_He4_Cu60_to_n_Ga63)*Y(Cu60)*Y(He4)*state.rho) +
        (screened_rates(k_p_Ga63_to_He4_Zn60)*Y(Ga63)*Y(H1)*state.rho + -screened_rates(k_He4_Zn60_to_p_Ga63)*Y(He4)*Y(Zn60)*state.rho) +
        (screened_rates(k_n_Ga64_to_He4_Cu61)*Y(Ga64)*Y(N)*state.rho + -screened_rates(k_He4_Cu61_to_n_Ga64)*Y(Cu61)*Y(He4)*state.rho) +
        (screened_rates(k_p_Ga64_to_He4_Zn61)*Y(Ga64)*Y(H1)*state.rho + -screened_rates(k_He4_Zn61_to_p_Ga64)*Y(He4)*Y(Zn61)*state.rho) +
        (screened_rates(k_n_Ge63_to_He4_Zn60)*Y(Ge63)*Y(N)*state.rho + -screened_rates(k_He4_Zn60_to_n_Ge63)*Y(He4)*Y(Zn60)*state.rho) +
        (screened_rates(k_n_Ge64_to_He4_Zn61)*Y(Ge64)*Y(N)*state.rho + -screened_rates(k_He4_Zn61_to_n_Ge64)*Y(He4)*Y(Zn61)*state.rho) +
        (0.5*screened_rates(k_He3_He3_to_p_p_He4)*amrex::Math::powi<2>(Y(He3))*state.rho + -0.5*screened_rates(k_p_p_He4_to_He3_He3)*Y(He4)*amrex::Math::powi<2>(Y(H1))*amrex::Math::powi<2>(state.rho)) +
        (2.0*screened_rates(k_d_Li7_to_n_He4_He4)*Y(Li7)*Y(H2)*state.rho + -screened_rates(k_n_He4_He4_to_d_Li7)*amrex::Math::powi<2>(Y(He4))*Y(N)*amrex::Math::powi<2>(state.rho)) +
        (2.0*screened_rates(k_d_Be7_to_p_He4_He4)*Y(Be7)*Y(H2)*state.rho + -screened_rates(k_p_He4_He4_to_d_Be7)*amrex::Math::powi<2>(Y(He4))*Y(H1)*amrex::Math::powi<2>(state.rho)) +
        (2.0*screened_rates(k_p_Be9_to_d_He4_He4)*Y(Be9)*Y(H1)*state.rho + -screened_rates(k_d_He4_He4_to_p_Be9)*amrex::Math::powi<2>(Y(He4))*Y(H2)*amrex::Math::powi<2>(state.rho)) +
        (2.0*screened_rates(k_n_B8_to_p_He4_He4)*Y(B8)*Y(N)*state.rho + -screened_rates(k_p_He4_He4_to_n_B8)*amrex::Math::powi<2>(Y(He4))*Y(H1)*amrex::Math::powi<2>(state.rho)) +
        (3.0*screened_rates(k_p_B11_to_He4_He4_He4)*Y(B11)*Y(H1)*state.rho + -0.5*screened_rates(k_He4_He4_He4_to_p_B11)*amrex::Math::powi<3>(Y(He4))*amrex::Math::powi<2>(state.rho)) +
        (2.0*screened_rates(k_He3_Li7_to_n_p_He4_He4)*Y(He3)*Y(Li7)*state.rho + -screened_rates(k_n_p_He4_He4_to_He3_Li7)*amrex::Math::powi<2>(Y(He4))*Y(N)*Y(H1)*amrex::Math::powi<3>(state.rho)) +
        (2.0*screened_rates(k_He3_Be7_to_p_p_He4_He4)*Y(Be7)*Y(He3)*state.rho + -0.5*screened_rates(k_p_p_He4_He4_to_He3_Be7)*amrex::Math::powi<2>(Y(He4))*amrex::Math::powi<2>(Y(H1))*amrex::Math::powi<3>(state.rho)) +
        (-screened_rates(k_n_p_He4_to_Li6)*Y(He4)*Y(N)*Y(H1)*amrex::Math::powi<2>(state.rho) + screened_rates(k_Li6_to_n_p_He4)*Y(Li6)) +
        (-screened_rates(k_n_He4_He4_to_Be9)*amrex::Math::powi<2>(Y(He4))*Y(N)*amrex::Math::powi<2>(state.rho) + 2.0*screened_rates(k_Be9_to_n_He4_He4)*Y(Be9)) +
        (-0.5*screened_rates(k_He4_He4_He4_to_C12)*amrex::Math::powi<3>(Y(He4))*amrex::Math::powi<2>(state.rho) + 3.0*screened_rates(k_C12_to_He4_He4_He4)*Y(C12)) +
        (-screened_rates(k_n_p_He4_He4_to_p_Be9)*amrex::Math::powi<2>(Y(He4))*Y(N)*Y(H1)*amrex::Math::powi<3>(state.rho) + 2.0*screened_rates(k_p_Be9_to_n_p_He4_He4)*Y(Be9)*Y(H1)*state.rho);

    ydot_nuc(Li6) =
        (screened_rates(k_He4_d_to_Li6)*Y(He4)*Y(H2)*state.rho + -screened_rates(k_Li6_to_He4_d)*Y(Li6)) +
        (-screened_rates(k_n_Li6_to_Li7)*Y(Li6)*Y(N)*state.rho + screened_rates(k_Li7_to_n_Li6)*Y(Li7)) +
        (-screened_rates(k_p_Li6_to_Be7)*Y(Li6)*Y(H1)*state.rho + screened_rates(k_Be7_to_p_Li6)*Y(Be7)) +
        (-screened_rates(k_He4_Li6_to_B10)*Y(He4)*Y(Li6)*state.rho + screened_rates(k_B10_to_He4_Li6)*Y(B10)) +
        (-screened_rates(k_p_Li6_to_He4_He3)*Y(Li6)*Y(H1)*state.rho + screened_rates(k_He4_He3_to_p_Li6)*Y(He3)*Y(He4)*state.rho) +
        (-screened_rates(k_d_Li6_to_n_Be7)*Y(Li6)*Y(H2)*state.rho + screened_rates(k_n_Be7_to_d_Li6)*Y(Be7)*Y(N)*state.rho) +
        (-screened_rates(k_d_Li6_to_p_Li7)*Y(Li6)*Y(H2)*state.rho + screened_rates(k_p_Li7_to_d_Li6)*Y(Li7)*Y(H1)*state.rho) +
        (screened_rates(k_p_Be9_to_He4_Li6)*Y(Be9)*Y(H1)*state.rho + -screened_rates(k_He4_Li6_to_p_Be9)*Y(He4)*Y(Li6)*state.rho) +
        (screened_rates(k_n_p_He4_to_Li6)*Y(He4)*Y(N)*Y(H1)*amrex::Math::powi<2>(state.rho) + -screened_rates(k_Li6_to_n_p_He4)*Y(Li6));

    ydot_nuc(Li7) =
        screened_rates(k_Be7_to_Li7_weak_electron_capture)*Y(Be7)*state.rho*state.y_e +
        (screened_rates(k_n_Li6_to_Li7)*Y(Li6)*Y(N)*state.rho + -screened_rates(k_Li7_to_n_Li6)*Y(Li7)) +
        (-screened_rates(k_He4_Li7_to_B11)*Y(He4)*Y(Li7)*state.rho + screened_rates(k_B11_to_He4_Li7)*Y(B11)) +
        (screened_rates(k_d_Li6_to_p_Li7)*Y(Li6)*Y(H2)*state.rho + -screened_rates(k_p_Li7_to_d_Li6)*Y(Li7)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Li7_to_He4_He4)*Y(Li7)*Y(H1)*state.rho + 0.5*screened_rates(k_He4_He4_to_p_Li7)*amrex::Math::powi<2>(Y(He4))*state.rho) +
        (screened_rates(k_n_Be7_to_p_Li7)*Y(Be7)*Y(N)*state.rho + -screened_rates(k_p_Li7_to_n_Be7)*Y(Li7)*Y(H1)*state.rho) +
        (screened_rates(k_n_B10_to_He4_Li7)*Y(B10)*Y(N)*state.rho + -screened_rates(k_He4_Li7_to_n_B10)*Y(He4)*Y(Li7)*state.rho) +
        (-screened_rates(k_d_Li7_to_n_He4_He4)*Y(Li7)*Y(H2)*state.rho + 0.5*screened_rates(k_n_He4_He4_to_d_Li7)*amrex::Math::powi<2>(Y(He4))*Y(N)*amrex::Math::powi<2>(state.rho)) +
        (-screened_rates(k_He3_Li7_to_n_p_He4_He4)*Y(He3)*Y(Li7)*state.rho + 0.5*screened_rates(k_n_p_He4_He4_to_He3_Li7)*amrex::Math::powi<2>(Y(He4))*Y(N)*Y(H1)*amrex::Math::powi<3>(state.rho));

    ydot_nuc(Be7) =
        -screened_rates(k_Be7_to_Li7_weak_electron_capture)*Y(Be7)*state.rho*state.y_e +
        (screened_rates(k_He4_He3_to_Be7)*Y(He3)*Y(He4)*state.rho + -screened_rates(k_Be7_to_He4_He3)*Y(Be7)) +
        (screened_rates(k_p_Li6_to_Be7)*Y(Li6)*Y(H1)*state.rho + -screened_rates(k_Be7_to_p_Li6)*Y(Be7)) +
        (-screened_rates(k_p_Be7_to_B8)*Y(Be7)*Y(H1)*state.rho + screened_rates(k_B8_to_p_Be7)*Y(B8)) +
        (screened_rates(k_d_Li6_to_n_Be7)*Y(Li6)*Y(H2)*state.rho + -screened_rates(k_n_Be7_to_d_Li6)*Y(Be7)*Y(N)*state.rho) +
        (-screened_rates(k_n_Be7_to_p_Li7)*Y(Be7)*Y(N)*state.rho + screened_rates(k_p_Li7_to_n_Be7)*Y(Li7)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Be7_to_He4_He4)*Y(Be7)*Y(N)*state.rho + 0.5*screened_rates(k_He4_He4_to_n_Be7)*amrex::Math::powi<2>(Y(He4))*state.rho) +
        (screened_rates(k_p_B10_to_He4_Be7)*Y(B10)*Y(H1)*state.rho + -screened_rates(k_He4_Be7_to_p_B10)*Y(Be7)*Y(He4)*state.rho) +
        (-screened_rates(k_d_Be7_to_p_He4_He4)*Y(Be7)*Y(H2)*state.rho + 0.5*screened_rates(k_p_He4_He4_to_d_Be7)*amrex::Math::powi<2>(Y(He4))*Y(H1)*amrex::Math::powi<2>(state.rho)) +
        (-screened_rates(k_He3_Be7_to_p_p_He4_He4)*Y(Be7)*Y(He3)*state.rho + 0.25*screened_rates(k_p_p_He4_He4_to_He3_Be7)*amrex::Math::powi<2>(Y(He4))*amrex::Math::powi<2>(Y(H1))*amrex::Math::powi<3>(state.rho));

    ydot_nuc(Be9) =
        (-screened_rates(k_p_Be9_to_B10)*Y(Be9)*Y(H1)*state.rho + screened_rates(k_B10_to_p_Be9)*Y(B10)) +
        (-screened_rates(k_p_Be9_to_He4_Li6)*Y(Be9)*Y(H1)*state.rho + screened_rates(k_He4_Li6_to_p_Be9)*Y(He4)*Y(Li6)*state.rho) +
        (-screened_rates(k_He4_Be9_to_n_C12)*Y(Be9)*Y(He4)*state.rho + screened_rates(k_n_C12_to_He4_Be9)*Y(C12)*Y(N)*state.rho) +
        (-screened_rates(k_p_Be9_to_d_He4_He4)*Y(Be9)*Y(H1)*state.rho + 0.5*screened_rates(k_d_He4_He4_to_p_Be9)*amrex::Math::powi<2>(Y(He4))*Y(H2)*amrex::Math::powi<2>(state.rho)) +
        (0.5*screened_rates(k_n_He4_He4_to_Be9)*amrex::Math::powi<2>(Y(He4))*Y(N)*amrex::Math::powi<2>(state.rho) + -screened_rates(k_Be9_to_n_He4_He4)*Y(Be9)) +
        (0.5*screened_rates(k_n_p_He4_He4_to_p_Be9)*amrex::Math::powi<2>(Y(He4))*Y(N)*Y(H1)*amrex::Math::powi<3>(state.rho) + -screened_rates(k_p_Be9_to_n_p_He4_He4)*Y(Be9)*Y(H1)*state.rho);

    ydot_nuc(B8) =
        -screened_rates(k_B8_to_He4_He4_weak_wc12)*Y(B8) +
        (screened_rates(k_p_Be7_to_B8)*Y(Be7)*Y(H1)*state.rho + -screened_rates(k_B8_to_p_Be7)*Y(B8)) +
        (-screened_rates(k_n_B8_to_p_He4_He4)*Y(B8)*Y(N)*state.rho + 0.5*screened_rates(k_p_He4_He4_to_n_B8)*amrex::Math::powi<2>(Y(He4))*Y(H1)*amrex::Math::powi<2>(state.rho));

    ydot_nuc(B10) =
        (screened_rates(k_He4_Li6_to_B10)*Y(He4)*Y(Li6)*state.rho + -screened_rates(k_B10_to_He4_Li6)*Y(B10)) +
        (screened_rates(k_p_Be9_to_B10)*Y(Be9)*Y(H1)*state.rho + -screened_rates(k_B10_to_p_Be9)*Y(B10)) +
        (-screened_rates(k_n_B10_to_B11)*Y(B10)*Y(N)*state.rho + screened_rates(k_B11_to_n_B10)*Y(B11)) +
        (-screened_rates(k_n_B10_to_He4_Li7)*Y(B10)*Y(N)*state.rho + screened_rates(k_He4_Li7_to_n_B10)*Y(He4)*Y(Li7)*state.rho) +
        (-screened_rates(k_p_B10_to_He4_Be7)*Y(B10)*Y(H1)*state.rho + screened_rates(k_He4_Be7_to_p_B10)*Y(Be7)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_B10_to_n_N13)*Y(B10)*Y(He4)*state.rho + screened_rates(k_n_N13_to_He4_B10)*Y(N13)*Y(N)*state.rho) +
        (-screened_rates(k_He4_B10_to_p_C13)*Y(B10)*Y(He4)*state.rho + screened_rates(k_p_C13_to_He4_B10)*Y(C13)*Y(H1)*state.rho);

    ydot_nuc(B11) =
        (screened_rates(k_He4_Li7_to_B11)*Y(He4)*Y(Li7)*state.rho + -screened_rates(k_B11_to_He4_Li7)*Y(B11)) +
        (screened_rates(k_n_B10_to_B11)*Y(B10)*Y(N)*state.rho + -screened_rates(k_B11_to_n_B10)*Y(B11)) +
        (-screened_rates(k_p_B11_to_C12)*Y(B11)*Y(H1)*state.rho + screened_rates(k_C12_to_p_B11)*Y(C12)) +
        (-screened_rates(k_He4_B11_to_n_N14)*Y(B11)*Y(He4)*state.rho + screened_rates(k_n_N14_to_He4_B11)*Y(N14)*Y(N)*state.rho) +
        (-screened_rates(k_He4_B11_to_p_C14)*Y(B11)*Y(He4)*state.rho + screened_rates(k_p_C14_to_He4_B11)*Y(C14)*Y(H1)*state.rho) +
        (-screened_rates(k_p_B11_to_He4_He4_He4)*Y(B11)*Y(H1)*state.rho + 0.16666666666666667*screened_rates(k_He4_He4_He4_to_p_B11)*amrex::Math::powi<3>(Y(He4))*amrex::Math::powi<2>(state.rho));

    ydot_nuc(C12) =
        (screened_rates(k_p_B11_to_C12)*Y(B11)*Y(H1)*state.rho + -screened_rates(k_C12_to_p_B11)*Y(C12)) +
        (-screened_rates(k_n_C12_to_C13)*Y(C12)*Y(N)*state.rho + screened_rates(k_C13_to_n_C12)*Y(C13)) +
        (-screened_rates(k_p_C12_to_N13)*Y(C12)*Y(H1)*state.rho + screened_rates(k_N13_to_p_C12)*Y(N13)) +
        (-screened_rates(k_He4_C12_to_O16)*Y(C12)*Y(He4)*state.rho + screened_rates(k_O16_to_He4_C12)*Y(O16)) +
        (screened_rates(k_He4_Be9_to_n_C12)*Y(Be9)*Y(He4)*state.rho + -screened_rates(k_n_C12_to_He4_Be9)*Y(C12)*Y(N)*state.rho) +
        (-screened_rates(k_C12_C12_to_p_Na23)*amrex::Math::powi<2>(Y(C12))*state.rho + 2.0*screened_rates(k_p_Na23_to_C12_C12)*Y(Na23)*Y(H1)*state.rho) +
        (-screened_rates(k_C12_C12_to_He4_Ne20)*amrex::Math::powi<2>(Y(C12))*state.rho + 2.0*screened_rates(k_He4_Ne20_to_C12_C12)*Y(He4)*Y(Ne20)*state.rho) +
        (screened_rates(k_p_N15_to_He4_C12)*Y(N15)*Y(H1)*state.rho + -screened_rates(k_He4_C12_to_p_N15)*Y(C12)*Y(He4)*state.rho) +
        (screened_rates(k_n_O15_to_He4_C12)*Y(O15)*Y(N)*state.rho + -screened_rates(k_He4_C12_to_n_O15)*Y(C12)*Y(He4)*state.rho) +
        (-screened_rates(k_C12_O16_to_p_Al27)*Y(C12)*Y(O16)*state.rho + screened_rates(k_p_Al27_to_C12_O16)*Y(Al27)*Y(H1)*state.rho) +
        (-screened_rates(k_C12_O16_to_He4_Mg24)*Y(C12)*Y(O16)*state.rho + screened_rates(k_He4_Mg24_to_C12_O16)*Y(He4)*Y(Mg24)*state.rho) +
        (-screened_rates(k_C12_Ne20_to_p_P31)*Y(C12)*Y(Ne20)*state.rho + screened_rates(k_p_P31_to_C12_Ne20)*Y(P31)*Y(H1)*state.rho) +
        (-screened_rates(k_C12_Ne20_to_He4_Si28)*Y(C12)*Y(Ne20)*state.rho + screened_rates(k_He4_Si28_to_C12_Ne20)*Y(He4)*Y(Si28)*state.rho) +
        (2.0*screened_rates(k_n_Mg23_to_C12_C12)*Y(Mg23)*Y(N)*state.rho + -screened_rates(k_C12_C12_to_n_Mg23)*amrex::Math::powi<2>(Y(C12))*state.rho) +
        (0.16666666666666667*screened_rates(k_He4_He4_He4_to_C12)*amrex::Math::powi<3>(Y(He4))*amrex::Math::powi<2>(state.rho) + -screened_rates(k_C12_to_He4_He4_He4)*Y(C12));

    ydot_nuc(C13) =
        screened_rates(k_N13_to_C13_weak_wc12)*Y(N13) +
        (screened_rates(k_n_C12_to_C13)*Y(C12)*Y(N)*state.rho + -screened_rates(k_C13_to_n_C12)*Y(C13)) +
        (-screened_rates(k_n_C13_to_C14)*Y(C13)*Y(N)*state.rho + screened_rates(k_C14_to_n_C13)*Y(C14)) +
        (-screened_rates(k_p_C13_to_N14)*Y(C13)*Y(H1)*state.rho + screened_rates(k_N14_to_p_C13)*Y(N14)) +
        (screened_rates(k_He4_B10_to_p_C13)*Y(B10)*Y(He4)*state.rho + -screened_rates(k_p_C13_to_He4_B10)*Y(C13)*Y(H1)*state.rho) +
        (-screened_rates(k_d_C13_to_n_N14)*Y(C13)*Y(H2)*state.rho + screened_rates(k_n_N14_to_d_C13)*Y(N14)*Y(N)*state.rho) +
        (-screened_rates(k_He4_C13_to_n_O16)*Y(C13)*Y(He4)*state.rho + screened_rates(k_n_O16_to_He4_C13)*Y(O16)*Y(N)*state.rho) +
        (screened_rates(k_n_N13_to_p_C13)*Y(N13)*Y(N)*state.rho + -screened_rates(k_p_C13_to_n_N13)*Y(C13)*Y(H1)*state.rho);

    ydot_nuc(C14) =
        -screened_rates(k_C14_to_N14_weak_wc12)*Y(C14) +
        (screened_rates(k_n_C13_to_C14)*Y(C13)*Y(N)*state.rho + -screened_rates(k_C14_to_n_C13)*Y(C14)) +
        (-screened_rates(k_p_C14_to_N15)*Y(C14)*Y(H1)*state.rho + screened_rates(k_N15_to_p_C14)*Y(N15)) +
        (-screened_rates(k_He4_C14_to_O18)*Y(C14)*Y(He4)*state.rho + screened_rates(k_O18_to_He4_C14)*Y(O18)) +
        (screened_rates(k_He4_B11_to_p_C14)*Y(B11)*Y(He4)*state.rho + -screened_rates(k_p_C14_to_He4_B11)*Y(C14)*Y(H1)*state.rho) +
        (-screened_rates(k_d_C14_to_n_N15)*Y(C14)*Y(H2)*state.rho + screened_rates(k_n_N15_to_d_C14)*Y(N15)*Y(N)*state.rho) +
        (screened_rates(k_n_N14_to_p_C14)*Y(N14)*Y(N)*state.rho + -screened_rates(k_p_C14_to_n_N14)*Y(C14)*Y(H1)*state.rho) +
        (screened_rates(k_n_O17_to_He4_C14)*Y(O17)*Y(N)*state.rho + -screened_rates(k_He4_C14_to_n_O17)*Y(C14)*Y(He4)*state.rho);

    ydot_nuc(N13) =
        -screened_rates(k_N13_to_C13_weak_wc12)*Y(N13) +
        (screened_rates(k_p_C12_to_N13)*Y(C12)*Y(H1)*state.rho + -screened_rates(k_N13_to_p_C12)*Y(N13)) +
        (-screened_rates(k_n_N13_to_N14)*Y(N13)*Y(N)*state.rho + screened_rates(k_N14_to_n_N13)*Y(N14)) +
        (-screened_rates(k_p_N13_to_O14)*Y(N13)*Y(H1)*state.rho + screened_rates(k_O14_to_p_N13)*Y(O14)) +
        (screened_rates(k_He4_B10_to_n_N13)*Y(B10)*Y(He4)*state.rho + -screened_rates(k_n_N13_to_He4_B10)*Y(N13)*Y(N)*state.rho) +
        (-screened_rates(k_n_N13_to_p_C13)*Y(N13)*Y(N)*state.rho + screened_rates(k_p_C13_to_n_N13)*Y(C13)*Y(H1)*state.rho) +
        (-screened_rates(k_He4_N13_to_p_O16)*Y(He4)*Y(N13)*state.rho + screened_rates(k_p_O16_to_He4_N13)*Y(O16)*Y(H1)*state.rho);

    ydot_nuc(N14) =
        screened_rates(k_C14_to_N14_weak_wc12)*Y(C14) +
        screened_rates(k_O14_to_N14_weak_wc12)*Y(O14) +
        (screened_rates(k_p_C13_to_N14)*Y(C13)*Y(H1)*state.rho + -screened_rates(k_N14_to_p_C13)*Y(N14)) +
        (screened_rates(k_n_N13_to_N14)*Y(N13)*Y(N)*state.rho + -screened_rates(k_N14_to_n_N13)*Y(N14)) +
        (-screened_rates(k_n_N14_to_N15)*Y(N14)*Y(N)*state.rho + screened_rates(k_N15_to_n_N14)*Y(N15)) +
        (-screened_rates(k_p_N14_to_O15)*Y(N14)*Y(H1)*state.rho + screened_rates(k_O15_to_p_N14)*Y(O15)) +
        (-screened_rates(k_He4_N14_to_F18)*Y(He4)*Y(N14)*state.rho + screened_rates(k_F18_to_He4_N14)*Y(F18)) +
        (screened_rates(k_He4_B11_to_n_N14)*Y(B11)*Y(He4)*state.rho + -screened_rates(k_n_N14_to_He4_B11)*Y(N14)*Y(N)*state.rho) +
        (screened_rates(k_d_C13_to_n_N14)*Y(C13)*Y(H2)*state.rho + -screened_rates(k_n_N14_to_d_C13)*Y(N14)*Y(N)*state.rho) +
        (-screened_rates(k_n_N14_to_p_C14)*Y(N14)*Y(N)*state.rho + screened_rates(k_p_C14_to_n_N14)*Y(C14)*Y(H1)*state.rho) +
        (screened_rates(k_n_O14_to_p_N14)*Y(O14)*Y(N)*state.rho + -screened_rates(k_p_N14_to_n_O14)*Y(N14)*Y(H1)*state.rho) +
        (screened_rates(k_p_O17_to_He4_N14)*Y(O17)*Y(H1)*state.rho + -screened_rates(k_He4_N14_to_p_O17)*Y(He4)*Y(N14)*state.rho) +
        (screened_rates(k_n_F17_to_He4_N14)*Y(F17)*Y(N)*state.rho + -screened_rates(k_He4_N14_to_n_F17)*Y(He4)*Y(N14)*state.rho);

    ydot_nuc(N15) =
        screened_rates(k_O15_to_N15_weak_wc12)*Y(O15) +
        (screened_rates(k_p_C14_to_N15)*Y(C14)*Y(H1)*state.rho + -screened_rates(k_N15_to_p_C14)*Y(N15)) +
        (screened_rates(k_n_N14_to_N15)*Y(N14)*Y(N)*state.rho + -screened_rates(k_N15_to_n_N14)*Y(N15)) +
        (-screened_rates(k_p_N15_to_O16)*Y(N15)*Y(H1)*state.rho + screened_rates(k_O16_to_p_N15)*Y(O16)) +
        (-screened_rates(k_He4_N15_to_F19)*Y(He4)*Y(N15)*state.rho + screened_rates(k_F19_to_He4_N15)*Y(F19)) +
        (screened_rates(k_d_C14_to_n_N15)*Y(C14)*Y(H2)*state.rho + -screened_rates(k_n_N15_to_d_C14)*Y(N15)*Y(N)*state.rho) +
        (-screened_rates(k_p_N15_to_He4_C12)*Y(N15)*Y(H1)*state.rho + screened_rates(k_He4_C12_to_p_N15)*Y(C12)*Y(He4)*state.rho) +
        (screened_rates(k_n_O15_to_p_N15)*Y(O15)*Y(N)*state.rho + -screened_rates(k_p_N15_to_n_O15)*Y(N15)*Y(H1)*state.rho) +
        (screened_rates(k_p_O18_to_He4_N15)*Y(O18)*Y(H1)*state.rho + -screened_rates(k_He4_N15_to_p_O18)*Y(He4)*Y(N15)*state.rho) +
        (screened_rates(k_n_F18_to_He4_N15)*Y(F18)*Y(N)*state.rho + -screened_rates(k_He4_N15_to_n_F18)*Y(He4)*Y(N15)*state.rho);

    ydot_nuc(O14) =
        -screened_rates(k_O14_to_N14_weak_wc12)*Y(O14) +
        (screened_rates(k_p_N13_to_O14)*Y(N13)*Y(H1)*state.rho + -screened_rates(k_O14_to_p_N13)*Y(O14)) +
        (-screened_rates(k_n_O14_to_O15)*Y(O14)*Y(N)*state.rho + screened_rates(k_O15_to_n_O14)*Y(O15)) +
        (-screened_rates(k_He4_O14_to_Ne18)*Y(He4)*Y(O14)*state.rho + screened_rates(k_Ne18_to_He4_O14)*Y(Ne18)) +
        (-screened_rates(k_n_O14_to_p_N14)*Y(O14)*Y(N)*state.rho + screened_rates(k_p_N14_to_n_O14)*Y(N14)*Y(H1)*state.rho) +
        (-screened_rates(k_He4_O14_to_p_F17)*Y(He4)*Y(O14)*state.rho + screened_rates(k_p_F17_to_He4_O14)*Y(F17)*Y(H1)*state.rho);

    ydot_nuc(O15) =
        -screened_rates(k_O15_to_N15_weak_wc12)*Y(O15) +
        (screened_rates(k_p_N14_to_O15)*Y(N14)*Y(H1)*state.rho + -screened_rates(k_O15_to_p_N14)*Y(O15)) +
        (screened_rates(k_n_O14_to_O15)*Y(O14)*Y(N)*state.rho + -screened_rates(k_O15_to_n_O14)*Y(O15)) +
        (-screened_rates(k_n_O15_to_O16)*Y(O15)*Y(N)*state.rho + screened_rates(k_O16_to_n_O15)*Y(O16)) +
        (-screened_rates(k_He4_O15_to_Ne19)*Y(He4)*Y(O15)*state.rho + screened_rates(k_Ne19_to_He4_O15)*Y(Ne19)) +
        (-screened_rates(k_n_O15_to_p_N15)*Y(O15)*Y(N)*state.rho + screened_rates(k_p_N15_to_n_O15)*Y(N15)*Y(H1)*state.rho) +
        (-screened_rates(k_n_O15_to_He4_C12)*Y(O15)*Y(N)*state.rho + screened_rates(k_He4_C12_to_n_O15)*Y(C12)*Y(He4)*state.rho) +
        (screened_rates(k_p_F18_to_He4_O15)*Y(F18)*Y(H1)*state.rho + -screened_rates(k_He4_O15_to_p_F18)*Y(He4)*Y(O15)*state.rho) +
        (screened_rates(k_n_Ne18_to_He4_O15)*Y(Ne18)*Y(N)*state.rho + -screened_rates(k_He4_O15_to_n_Ne18)*Y(He4)*Y(O15)*state.rho);

    ydot_nuc(O16) =
        (screened_rates(k_He4_C12_to_O16)*Y(C12)*Y(He4)*state.rho + -screened_rates(k_O16_to_He4_C12)*Y(O16)) +
        (screened_rates(k_p_N15_to_O16)*Y(N15)*Y(H1)*state.rho + -screened_rates(k_O16_to_p_N15)*Y(O16)) +
        (screened_rates(k_n_O15_to_O16)*Y(O15)*Y(N)*state.rho + -screened_rates(k_O16_to_n_O15)*Y(O16)) +
        (-screened_rates(k_n_O16_to_O17)*Y(O16)*Y(N)*state.rho + screened_rates(k_O17_to_n_O16)*Y(O17)) +
        (-screened_rates(k_p_O16_to_F17)*Y(O16)*Y(H1)*state.rho + screened_rates(k_F17_to_p_O16)*Y(F17)) +
        (-screened_rates(k_He4_O16_to_Ne20)*Y(He4)*Y(O16)*state.rho + screened_rates(k_Ne20_to_He4_O16)*Y(Ne20)) +
        (screened_rates(k_He4_C13_to_n_O16)*Y(C13)*Y(He4)*state.rho + -screened_rates(k_n_O16_to_He4_C13)*Y(O16)*Y(N)*state.rho) +
        (screened_rates(k_He4_N13_to_p_O16)*Y(He4)*Y(N13)*state.rho + -screened_rates(k_p_O16_to_He4_N13)*Y(O16)*Y(H1)*state.rho) +
        (-screened_rates(k_C12_O16_to_p_Al27)*Y(C12)*Y(O16)*state.rho + screened_rates(k_p_Al27_to_C12_O16)*Y(Al27)*Y(H1)*state.rho) +
        (-screened_rates(k_C12_O16_to_He4_Mg24)*Y(C12)*Y(O16)*state.rho + screened_rates(k_He4_Mg24_to_C12_O16)*Y(He4)*Y(Mg24)*state.rho) +
        (-screened_rates(k_O16_O16_to_p_P31)*amrex::Math::powi<2>(Y(O16))*state.rho + 2.0*screened_rates(k_p_P31_to_O16_O16)*Y(P31)*Y(H1)*state.rho) +
        (-screened_rates(k_O16_O16_to_He4_Si28)*amrex::Math::powi<2>(Y(O16))*state.rho + 2.0*screened_rates(k_He4_Si28_to_O16_O16)*Y(He4)*Y(Si28)*state.rho) +
        (screened_rates(k_p_F19_to_He4_O16)*Y(F19)*Y(H1)*state.rho + -screened_rates(k_He4_O16_to_p_F19)*Y(He4)*Y(O16)*state.rho) +
        (screened_rates(k_n_Ne19_to_He4_O16)*Y(Ne19)*Y(N)*state.rho + -screened_rates(k_He4_O16_to_n_Ne19)*Y(He4)*Y(O16)*state.rho);

    ydot_nuc(O17) =
        screened_rates(k_F17_to_O17_weak_wc12)*Y(F17) +
        (screened_rates(k_n_O16_to_O17)*Y(O16)*Y(N)*state.rho + -screened_rates(k_O17_to_n_O16)*Y(O17)) +
        (-screened_rates(k_n_O17_to_O18)*Y(O17)*Y(N)*state.rho + screened_rates(k_O18_to_n_O17)*Y(O18)) +
        (-screened_rates(k_p_O17_to_F18)*Y(O17)*Y(H1)*state.rho + screened_rates(k_F18_to_p_O17)*Y(F18)) +
        (-screened_rates(k_He4_O17_to_Ne21)*Y(He4)*Y(O17)*state.rho + screened_rates(k_Ne21_to_He4_O17)*Y(Ne21)) +
        (-screened_rates(k_n_O17_to_He4_C14)*Y(O17)*Y(N)*state.rho + screened_rates(k_He4_C14_to_n_O17)*Y(C14)*Y(He4)*state.rho) +
        (-screened_rates(k_p_O17_to_He4_N14)*Y(O17)*Y(H1)*state.rho + screened_rates(k_He4_N14_to_p_O17)*Y(He4)*Y(N14)*state.rho) +
        (-screened_rates(k_He4_O17_to_n_Ne20)*Y(He4)*Y(O17)*state.rho + screened_rates(k_n_Ne20_to_He4_O17)*Y(Ne20)*Y(N)*state.rho) +
        (screened_rates(k_n_F17_to_p_O17)*Y(F17)*Y(N)*state.rho + -screened_rates(k_p_O17_to_n_F17)*Y(O17)*Y(H1)*state.rho);

    ydot_nuc(O18) =
        screened_rates(k_F18_to_O18_weak_wc12)*Y(F18) +
        (screened_rates(k_He4_C14_to_O18)*Y(C14)*Y(He4)*state.rho + -screened_rates(k_O18_to_He4_C14)*Y(O18)) +
        (screened_rates(k_n_O17_to_O18)*Y(O17)*Y(N)*state.rho + -screened_rates(k_O18_to_n_O17)*Y(O18)) +
        (-screened_rates(k_p_O18_to_F19)*Y(O18)*Y(H1)*state.rho + screened_rates(k_F19_to_p_O18)*Y(F19)) +
        (-screened_rates(k_He4_O18_to_Ne22)*Y(He4)*Y(O18)*state.rho + screened_rates(k_Ne22_to_He4_O18)*Y(Ne22)) +
        (-screened_rates(k_p_O18_to_He4_N15)*Y(O18)*Y(H1)*state.rho + screened_rates(k_He4_N15_to_p_O18)*Y(He4)*Y(N15)*state.rho) +
        (screened_rates(k_n_F18_to_p_O18)*Y(F18)*Y(N)*state.rho + -screened_rates(k_p_O18_to_n_F18)*Y(O18)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ne21_to_He4_O18)*Y(Ne21)*Y(N)*state.rho + -screened_rates(k_He4_O18_to_n_Ne21)*Y(He4)*Y(O18)*state.rho);

    ydot_nuc(F17) =
        -screened_rates(k_F17_to_O17_weak_wc12)*Y(F17) +
        (screened_rates(k_p_O16_to_F17)*Y(O16)*Y(H1)*state.rho + -screened_rates(k_F17_to_p_O16)*Y(F17)) +
        (-screened_rates(k_n_F17_to_F18)*Y(F17)*Y(N)*state.rho + screened_rates(k_F18_to_n_F17)*Y(F18)) +
        (-screened_rates(k_p_F17_to_Ne18)*Y(F17)*Y(H1)*state.rho + screened_rates(k_Ne18_to_p_F17)*Y(Ne18)) +
        (-screened_rates(k_He4_F17_to_Na21)*Y(F17)*Y(He4)*state.rho + screened_rates(k_Na21_to_He4_F17)*Y(Na21)) +
        (screened_rates(k_He4_O14_to_p_F17)*Y(He4)*Y(O14)*state.rho + -screened_rates(k_p_F17_to_He4_O14)*Y(F17)*Y(H1)*state.rho) +
        (-screened_rates(k_n_F17_to_p_O17)*Y(F17)*Y(N)*state.rho + screened_rates(k_p_O17_to_n_F17)*Y(O17)*Y(H1)*state.rho) +
        (-screened_rates(k_n_F17_to_He4_N14)*Y(F17)*Y(N)*state.rho + screened_rates(k_He4_N14_to_n_F17)*Y(He4)*Y(N14)*state.rho) +
        (-screened_rates(k_He4_F17_to_p_Ne20)*Y(F17)*Y(He4)*state.rho + screened_rates(k_p_Ne20_to_He4_F17)*Y(Ne20)*Y(H1)*state.rho);

    ydot_nuc(F18) =
        -screened_rates(k_F18_to_O18_weak_wc12)*Y(F18) +
        screened_rates(k_Ne18_to_F18_weak_wc12)*Y(Ne18) +
        (screened_rates(k_He4_N14_to_F18)*Y(He4)*Y(N14)*state.rho + -screened_rates(k_F18_to_He4_N14)*Y(F18)) +
        (screened_rates(k_p_O17_to_F18)*Y(O17)*Y(H1)*state.rho + -screened_rates(k_F18_to_p_O17)*Y(F18)) +
        (screened_rates(k_n_F17_to_F18)*Y(F17)*Y(N)*state.rho + -screened_rates(k_F18_to_n_F17)*Y(F18)) +
        (-screened_rates(k_n_F18_to_F19)*Y(F18)*Y(N)*state.rho + screened_rates(k_F19_to_n_F18)*Y(F19)) +
        (-screened_rates(k_p_F18_to_Ne19)*Y(F18)*Y(H1)*state.rho + screened_rates(k_Ne19_to_p_F18)*Y(Ne19)) +
        (-screened_rates(k_He4_F18_to_Na22)*Y(F18)*Y(He4)*state.rho + screened_rates(k_Na22_to_He4_F18)*Y(Na22)) +
        (-screened_rates(k_n_F18_to_p_O18)*Y(F18)*Y(N)*state.rho + screened_rates(k_p_O18_to_n_F18)*Y(O18)*Y(H1)*state.rho) +
        (-screened_rates(k_n_F18_to_He4_N15)*Y(F18)*Y(N)*state.rho + screened_rates(k_He4_N15_to_n_F18)*Y(He4)*Y(N15)*state.rho) +
        (-screened_rates(k_p_F18_to_He4_O15)*Y(F18)*Y(H1)*state.rho + screened_rates(k_He4_O15_to_p_F18)*Y(He4)*Y(O15)*state.rho) +
        (-screened_rates(k_He4_F18_to_p_Ne21)*Y(F18)*Y(He4)*state.rho + screened_rates(k_p_Ne21_to_He4_F18)*Y(Ne21)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ne18_to_p_F18)*Y(Ne18)*Y(N)*state.rho + -screened_rates(k_p_F18_to_n_Ne18)*Y(F18)*Y(H1)*state.rho) +
        (screened_rates(k_n_Na21_to_He4_F18)*Y(Na21)*Y(N)*state.rho + -screened_rates(k_He4_F18_to_n_Na21)*Y(F18)*Y(He4)*state.rho);

    ydot_nuc(F19) =
        screened_rates(k_Ne19_to_F19_weak_wc12)*Y(Ne19) +
        (screened_rates(k_He4_N15_to_F19)*Y(He4)*Y(N15)*state.rho + -screened_rates(k_F19_to_He4_N15)*Y(F19)) +
        (screened_rates(k_p_O18_to_F19)*Y(O18)*Y(H1)*state.rho + -screened_rates(k_F19_to_p_O18)*Y(F19)) +
        (screened_rates(k_n_F18_to_F19)*Y(F18)*Y(N)*state.rho + -screened_rates(k_F19_to_n_F18)*Y(F19)) +
        (-screened_rates(k_p_F19_to_Ne20)*Y(F19)*Y(H1)*state.rho + screened_rates(k_Ne20_to_p_F19)*Y(Ne20)) +
        (-screened_rates(k_He4_F19_to_Na23)*Y(F19)*Y(He4)*state.rho + screened_rates(k_Na23_to_He4_F19)*Y(Na23)) +
        (-screened_rates(k_p_F19_to_He4_O16)*Y(F19)*Y(H1)*state.rho + screened_rates(k_He4_O16_to_p_F19)*Y(He4)*Y(O16)*state.rho) +
        (-screened_rates(k_He4_F19_to_p_Ne22)*Y(F19)*Y(He4)*state.rho + screened_rates(k_p_Ne22_to_He4_F19)*Y(Ne22)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ne19_to_p_F19)*Y(Ne19)*Y(N)*state.rho + -screened_rates(k_p_F19_to_n_Ne19)*Y(F19)*Y(H1)*state.rho) +
        (screened_rates(k_n_Na22_to_He4_F19)*Y(Na22)*Y(N)*state.rho + -screened_rates(k_He4_F19_to_n_Na22)*Y(F19)*Y(He4)*state.rho);

    ydot_nuc(Ne18) =
        -screened_rates(k_Ne18_to_F18_weak_wc12)*Y(Ne18) +
        (screened_rates(k_He4_O14_to_Ne18)*Y(He4)*Y(O14)*state.rho + -screened_rates(k_Ne18_to_He4_O14)*Y(Ne18)) +
        (screened_rates(k_p_F17_to_Ne18)*Y(F17)*Y(H1)*state.rho + -screened_rates(k_Ne18_to_p_F17)*Y(Ne18)) +
        (-screened_rates(k_n_Ne18_to_Ne19)*Y(Ne18)*Y(N)*state.rho + screened_rates(k_Ne19_to_n_Ne18)*Y(Ne19)) +
        (-screened_rates(k_n_Ne18_to_p_F18)*Y(Ne18)*Y(N)*state.rho + screened_rates(k_p_F18_to_n_Ne18)*Y(F18)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ne18_to_He4_O15)*Y(Ne18)*Y(N)*state.rho + screened_rates(k_He4_O15_to_n_Ne18)*Y(He4)*Y(O15)*state.rho) +
        (-screened_rates(k_He4_Ne18_to_p_Na21)*Y(He4)*Y(Ne18)*state.rho + screened_rates(k_p_Na21_to_He4_Ne18)*Y(Na21)*Y(H1)*state.rho);

    ydot_nuc(Ne19) =
        -screened_rates(k_Ne19_to_F19_weak_wc12)*Y(Ne19) +
        (screened_rates(k_He4_O15_to_Ne19)*Y(He4)*Y(O15)*state.rho + -screened_rates(k_Ne19_to_He4_O15)*Y(Ne19)) +
        (screened_rates(k_p_F18_to_Ne19)*Y(F18)*Y(H1)*state.rho + -screened_rates(k_Ne19_to_p_F18)*Y(Ne19)) +
        (screened_rates(k_n_Ne18_to_Ne19)*Y(Ne18)*Y(N)*state.rho + -screened_rates(k_Ne19_to_n_Ne18)*Y(Ne19)) +
        (-screened_rates(k_n_Ne19_to_Ne20)*Y(Ne19)*Y(N)*state.rho + screened_rates(k_Ne20_to_n_Ne19)*Y(Ne20)) +
        (-screened_rates(k_He4_Ne19_to_Mg23)*Y(He4)*Y(Ne19)*state.rho + screened_rates(k_Mg23_to_He4_Ne19)*Y(Mg23)) +
        (-screened_rates(k_n_Ne19_to_p_F19)*Y(Ne19)*Y(N)*state.rho + screened_rates(k_p_F19_to_n_Ne19)*Y(F19)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ne19_to_He4_O16)*Y(Ne19)*Y(N)*state.rho + screened_rates(k_He4_O16_to_n_Ne19)*Y(He4)*Y(O16)*state.rho) +
        (-screened_rates(k_He4_Ne19_to_p_Na22)*Y(He4)*Y(Ne19)*state.rho + screened_rates(k_p_Na22_to_He4_Ne19)*Y(Na22)*Y(H1)*state.rho);

    ydot_nuc(Ne20) =
        (screened_rates(k_He4_O16_to_Ne20)*Y(He4)*Y(O16)*state.rho + -screened_rates(k_Ne20_to_He4_O16)*Y(Ne20)) +
        (screened_rates(k_p_F19_to_Ne20)*Y(F19)*Y(H1)*state.rho + -screened_rates(k_Ne20_to_p_F19)*Y(Ne20)) +
        (screened_rates(k_n_Ne19_to_Ne20)*Y(Ne19)*Y(N)*state.rho + -screened_rates(k_Ne20_to_n_Ne19)*Y(Ne20)) +
        (-screened_rates(k_n_Ne20_to_Ne21)*Y(Ne20)*Y(N)*state.rho + screened_rates(k_Ne21_to_n_Ne20)*Y(Ne21)) +
        (-screened_rates(k_p_Ne20_to_Na21)*Y(Ne20)*Y(H1)*state.rho + screened_rates(k_Na21_to_p_Ne20)*Y(Na21)) +
        (-screened_rates(k_He4_Ne20_to_Mg24)*Y(He4)*Y(Ne20)*state.rho + screened_rates(k_Mg24_to_He4_Ne20)*Y(Mg24)) +
        (0.5*screened_rates(k_C12_C12_to_He4_Ne20)*amrex::Math::powi<2>(Y(C12))*state.rho + -screened_rates(k_He4_Ne20_to_C12_C12)*Y(He4)*Y(Ne20)*state.rho) +
        (screened_rates(k_He4_O17_to_n_Ne20)*Y(He4)*Y(O17)*state.rho + -screened_rates(k_n_Ne20_to_He4_O17)*Y(Ne20)*Y(N)*state.rho) +
        (screened_rates(k_He4_F17_to_p_Ne20)*Y(F17)*Y(He4)*state.rho + -screened_rates(k_p_Ne20_to_He4_F17)*Y(Ne20)*Y(H1)*state.rho) +
        (-screened_rates(k_C12_Ne20_to_p_P31)*Y(C12)*Y(Ne20)*state.rho + screened_rates(k_p_P31_to_C12_Ne20)*Y(P31)*Y(H1)*state.rho) +
        (-screened_rates(k_C12_Ne20_to_He4_Si28)*Y(C12)*Y(Ne20)*state.rho + screened_rates(k_He4_Si28_to_C12_Ne20)*Y(He4)*Y(Si28)*state.rho) +
        (screened_rates(k_p_Na23_to_He4_Ne20)*Y(Na23)*Y(H1)*state.rho + -screened_rates(k_He4_Ne20_to_p_Na23)*Y(He4)*Y(Ne20)*state.rho) +
        (screened_rates(k_n_Mg23_to_He4_Ne20)*Y(Mg23)*Y(N)*state.rho + -screened_rates(k_He4_Ne20_to_n_Mg23)*Y(He4)*Y(Ne20)*state.rho);

    ydot_nuc(Ne21) =
        (screened_rates(k_He4_O17_to_Ne21)*Y(He4)*Y(O17)*state.rho + -screened_rates(k_Ne21_to_He4_O17)*Y(Ne21)) +
        (screened_rates(k_n_Ne20_to_Ne21)*Y(Ne20)*Y(N)*state.rho + -screened_rates(k_Ne21_to_n_Ne20)*Y(Ne21)) +
        (-screened_rates(k_n_Ne21_to_Ne22)*Y(Ne21)*Y(N)*state.rho + screened_rates(k_Ne22_to_n_Ne21)*Y(Ne22)) +
        (-screened_rates(k_p_Ne21_to_Na22)*Y(Ne21)*Y(H1)*state.rho + screened_rates(k_Na22_to_p_Ne21)*Y(Na22)) +
        (-screened_rates(k_He4_Ne21_to_Mg25)*Y(He4)*Y(Ne21)*state.rho + screened_rates(k_Mg25_to_He4_Ne21)*Y(Mg25)) +
        (screened_rates(k_He4_F18_to_p_Ne21)*Y(F18)*Y(He4)*state.rho + -screened_rates(k_p_Ne21_to_He4_F18)*Y(Ne21)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ne21_to_He4_O18)*Y(Ne21)*Y(N)*state.rho + screened_rates(k_He4_O18_to_n_Ne21)*Y(He4)*Y(O18)*state.rho) +
        (-screened_rates(k_He4_Ne21_to_n_Mg24)*Y(He4)*Y(Ne21)*state.rho + screened_rates(k_n_Mg24_to_He4_Ne21)*Y(Mg24)*Y(N)*state.rho) +
        (screened_rates(k_n_Na21_to_p_Ne21)*Y(Na21)*Y(N)*state.rho + -screened_rates(k_p_Ne21_to_n_Na21)*Y(Ne21)*Y(H1)*state.rho) +
        (screened_rates(k_Na21_to_Ne21)*Y(Na21) + -screened_rates(k_Ne21_to_Na21)*Y(Ne21));

    ydot_nuc(Ne22) =
        (screened_rates(k_He4_O18_to_Ne22)*Y(He4)*Y(O18)*state.rho + -screened_rates(k_Ne22_to_He4_O18)*Y(Ne22)) +
        (screened_rates(k_n_Ne21_to_Ne22)*Y(Ne21)*Y(N)*state.rho + -screened_rates(k_Ne22_to_n_Ne21)*Y(Ne22)) +
        (-screened_rates(k_p_Ne22_to_Na23)*Y(Ne22)*Y(H1)*state.rho + screened_rates(k_Na23_to_p_Ne22)*Y(Na23)) +
        (-screened_rates(k_He4_Ne22_to_Mg26)*Y(He4)*Y(Ne22)*state.rho + screened_rates(k_Mg26_to_He4_Ne22)*Y(Mg26)) +
        (screened_rates(k_He4_F19_to_p_Ne22)*Y(F19)*Y(He4)*state.rho + -screened_rates(k_p_Ne22_to_He4_F19)*Y(Ne22)*Y(H1)*state.rho) +
        (screened_rates(k_n_Na22_to_p_Ne22)*Y(Na22)*Y(N)*state.rho + -screened_rates(k_p_Ne22_to_n_Na22)*Y(Ne22)*Y(H1)*state.rho) +
        (screened_rates(k_n_Mg25_to_He4_Ne22)*Y(Mg25)*Y(N)*state.rho + -screened_rates(k_He4_Ne22_to_n_Mg25)*Y(He4)*Y(Ne22)*state.rho) +
        (screened_rates(k_Na22_to_Ne22)*Y(Na22) + -screened_rates(k_Ne22_to_Na22)*Y(Ne22));

    ydot_nuc(Na21) =
        (screened_rates(k_He4_F17_to_Na21)*Y(F17)*Y(He4)*state.rho + -screened_rates(k_Na21_to_He4_F17)*Y(Na21)) +
        (screened_rates(k_p_Ne20_to_Na21)*Y(Ne20)*Y(H1)*state.rho + -screened_rates(k_Na21_to_p_Ne20)*Y(Na21)) +
        (-screened_rates(k_n_Na21_to_Na22)*Y(Na21)*Y(N)*state.rho + screened_rates(k_Na22_to_n_Na21)*Y(Na22)) +
        (-screened_rates(k_He4_Na21_to_Al25)*Y(He4)*Y(Na21)*state.rho + screened_rates(k_Al25_to_He4_Na21)*Y(Al25)) +
        (screened_rates(k_He4_Ne18_to_p_Na21)*Y(He4)*Y(Ne18)*state.rho + -screened_rates(k_p_Na21_to_He4_Ne18)*Y(Na21)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Na21_to_p_Ne21)*Y(Na21)*Y(N)*state.rho + screened_rates(k_p_Ne21_to_n_Na21)*Y(Ne21)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Na21_to_He4_F18)*Y(Na21)*Y(N)*state.rho + screened_rates(k_He4_F18_to_n_Na21)*Y(F18)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Na21_to_p_Mg24)*Y(He4)*Y(Na21)*state.rho + screened_rates(k_p_Mg24_to_He4_Na21)*Y(Mg24)*Y(H1)*state.rho) +
        (-screened_rates(k_Na21_to_Ne21)*Y(Na21) + screened_rates(k_Ne21_to_Na21)*Y(Ne21));

    ydot_nuc(Na22) =
        (screened_rates(k_He4_F18_to_Na22)*Y(F18)*Y(He4)*state.rho + -screened_rates(k_Na22_to_He4_F18)*Y(Na22)) +
        (screened_rates(k_p_Ne21_to_Na22)*Y(Ne21)*Y(H1)*state.rho + -screened_rates(k_Na22_to_p_Ne21)*Y(Na22)) +
        (screened_rates(k_n_Na21_to_Na22)*Y(Na21)*Y(N)*state.rho + -screened_rates(k_Na22_to_n_Na21)*Y(Na22)) +
        (-screened_rates(k_n_Na22_to_Na23)*Y(Na22)*Y(N)*state.rho + screened_rates(k_Na23_to_n_Na22)*Y(Na23)) +
        (-screened_rates(k_p_Na22_to_Mg23)*Y(Na22)*Y(H1)*state.rho + screened_rates(k_Mg23_to_p_Na22)*Y(Mg23)) +
        (-screened_rates(k_He4_Na22_to_Al26)*Y(He4)*Y(Na22)*state.rho + screened_rates(k_Al26_to_He4_Na22)*Y(Al26)) +
        (screened_rates(k_He4_Ne19_to_p_Na22)*Y(He4)*Y(Ne19)*state.rho + -screened_rates(k_p_Na22_to_He4_Ne19)*Y(Na22)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Na22_to_p_Ne22)*Y(Na22)*Y(N)*state.rho + screened_rates(k_p_Ne22_to_n_Na22)*Y(Ne22)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Na22_to_He4_F19)*Y(Na22)*Y(N)*state.rho + screened_rates(k_He4_F19_to_n_Na22)*Y(F19)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Na22_to_p_Mg25)*Y(He4)*Y(Na22)*state.rho + screened_rates(k_p_Mg25_to_He4_Na22)*Y(Mg25)*Y(H1)*state.rho) +
        (screened_rates(k_n_Al25_to_He4_Na22)*Y(Al25)*Y(N)*state.rho + -screened_rates(k_He4_Na22_to_n_Al25)*Y(He4)*Y(Na22)*state.rho) +
        (-screened_rates(k_Na22_to_Ne22)*Y(Na22) + screened_rates(k_Ne22_to_Na22)*Y(Ne22));

    ydot_nuc(Na23) =
        (screened_rates(k_He4_F19_to_Na23)*Y(F19)*Y(He4)*state.rho + -screened_rates(k_Na23_to_He4_F19)*Y(Na23)) +
        (screened_rates(k_p_Ne22_to_Na23)*Y(Ne22)*Y(H1)*state.rho + -screened_rates(k_Na23_to_p_Ne22)*Y(Na23)) +
        (screened_rates(k_n_Na22_to_Na23)*Y(Na22)*Y(N)*state.rho + -screened_rates(k_Na23_to_n_Na22)*Y(Na23)) +
        (-screened_rates(k_p_Na23_to_Mg24)*Y(Na23)*Y(H1)*state.rho + screened_rates(k_Mg24_to_p_Na23)*Y(Mg24)) +
        (-screened_rates(k_He4_Na23_to_Al27)*Y(He4)*Y(Na23)*state.rho + screened_rates(k_Al27_to_He4_Na23)*Y(Al27)) +
        (0.5*screened_rates(k_C12_C12_to_p_Na23)*amrex::Math::powi<2>(Y(C12))*state.rho + -screened_rates(k_p_Na23_to_C12_C12)*Y(Na23)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Na23_to_He4_Ne20)*Y(Na23)*Y(H1)*state.rho + screened_rates(k_He4_Ne20_to_p_Na23)*Y(He4)*Y(Ne20)*state.rho) +
        (-screened_rates(k_He4_Na23_to_p_Mg26)*Y(He4)*Y(Na23)*state.rho + screened_rates(k_p_Mg26_to_He4_Na23)*Y(Mg26)*Y(H1)*state.rho) +
        (screened_rates(k_n_Mg23_to_p_Na23)*Y(Mg23)*Y(N)*state.rho + -screened_rates(k_p_Na23_to_n_Mg23)*Y(Na23)*Y(H1)*state.rho) +
        (screened_rates(k_n_Al26_to_He4_Na23)*Y(Al26)*Y(N)*state.rho + -screened_rates(k_He4_Na23_to_n_Al26)*Y(He4)*Y(Na23)*state.rho) +
        (screened_rates(k_Mg23_to_Na23)*Y(Mg23) + -screened_rates(k_Na23_to_Mg23)*Y(Na23));

    ydot_nuc(Mg23) =
        (screened_rates(k_He4_Ne19_to_Mg23)*Y(He4)*Y(Ne19)*state.rho + -screened_rates(k_Mg23_to_He4_Ne19)*Y(Mg23)) +
        (screened_rates(k_p_Na22_to_Mg23)*Y(Na22)*Y(H1)*state.rho + -screened_rates(k_Mg23_to_p_Na22)*Y(Mg23)) +
        (-screened_rates(k_n_Mg23_to_Mg24)*Y(Mg23)*Y(N)*state.rho + screened_rates(k_Mg24_to_n_Mg23)*Y(Mg24)) +
        (-screened_rates(k_n_Mg23_to_p_Na23)*Y(Mg23)*Y(N)*state.rho + screened_rates(k_p_Na23_to_n_Mg23)*Y(Na23)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Mg23_to_He4_Ne20)*Y(Mg23)*Y(N)*state.rho + screened_rates(k_He4_Ne20_to_n_Mg23)*Y(He4)*Y(Ne20)*state.rho) +
        (-screened_rates(k_n_Mg23_to_C12_C12)*Y(Mg23)*Y(N)*state.rho + 0.5*screened_rates(k_C12_C12_to_n_Mg23)*amrex::Math::powi<2>(Y(C12))*state.rho) +
        (-screened_rates(k_He4_Mg23_to_p_Al26)*Y(He4)*Y(Mg23)*state.rho + screened_rates(k_p_Al26_to_He4_Mg23)*Y(Al26)*Y(H1)*state.rho) +
        (-screened_rates(k_Mg23_to_Na23)*Y(Mg23) + screened_rates(k_Na23_to_Mg23)*Y(Na23));

    ydot_nuc(Mg24) =
        (screened_rates(k_He4_Ne20_to_Mg24)*Y(He4)*Y(Ne20)*state.rho + -screened_rates(k_Mg24_to_He4_Ne20)*Y(Mg24)) +
        (screened_rates(k_p_Na23_to_Mg24)*Y(Na23)*Y(H1)*state.rho + -screened_rates(k_Mg24_to_p_Na23)*Y(Mg24)) +
        (screened_rates(k_n_Mg23_to_Mg24)*Y(Mg23)*Y(N)*state.rho + -screened_rates(k_Mg24_to_n_Mg23)*Y(Mg24)) +
        (-screened_rates(k_n_Mg24_to_Mg25)*Y(Mg24)*Y(N)*state.rho + screened_rates(k_Mg25_to_n_Mg24)*Y(Mg25)) +
        (-screened_rates(k_p_Mg24_to_Al25)*Y(Mg24)*Y(H1)*state.rho + screened_rates(k_Al25_to_p_Mg24)*Y(Al25)) +
        (-screened_rates(k_He4_Mg24_to_Si28)*Y(He4)*Y(Mg24)*state.rho + screened_rates(k_Si28_to_He4_Mg24)*Y(Si28)) +
        (screened_rates(k_C12_O16_to_He4_Mg24)*Y(C12)*Y(O16)*state.rho + -screened_rates(k_He4_Mg24_to_C12_O16)*Y(He4)*Y(Mg24)*state.rho) +
        (screened_rates(k_He4_Ne21_to_n_Mg24)*Y(He4)*Y(Ne21)*state.rho + -screened_rates(k_n_Mg24_to_He4_Ne21)*Y(Mg24)*Y(N)*state.rho) +
        (screened_rates(k_He4_Na21_to_p_Mg24)*Y(He4)*Y(Na21)*state.rho + -screened_rates(k_p_Mg24_to_He4_Na21)*Y(Mg24)*Y(H1)*state.rho) +
        (screened_rates(k_p_Al27_to_He4_Mg24)*Y(Al27)*Y(H1)*state.rho + -screened_rates(k_He4_Mg24_to_p_Al27)*Y(He4)*Y(Mg24)*state.rho);

    ydot_nuc(Mg25) =
        (screened_rates(k_He4_Ne21_to_Mg25)*Y(He4)*Y(Ne21)*state.rho + -screened_rates(k_Mg25_to_He4_Ne21)*Y(Mg25)) +
        (screened_rates(k_n_Mg24_to_Mg25)*Y(Mg24)*Y(N)*state.rho + -screened_rates(k_Mg25_to_n_Mg24)*Y(Mg25)) +
        (-screened_rates(k_n_Mg25_to_Mg26)*Y(Mg25)*Y(N)*state.rho + screened_rates(k_Mg26_to_n_Mg25)*Y(Mg26)) +
        (-screened_rates(k_p_Mg25_to_Al26)*Y(Mg25)*Y(H1)*state.rho + screened_rates(k_Al26_to_p_Mg25)*Y(Al26)) +
        (-screened_rates(k_He4_Mg25_to_Si29)*Y(He4)*Y(Mg25)*state.rho + screened_rates(k_Si29_to_He4_Mg25)*Y(Si29)) +
        (screened_rates(k_He4_Na22_to_p_Mg25)*Y(He4)*Y(Na22)*state.rho + -screened_rates(k_p_Mg25_to_He4_Na22)*Y(Mg25)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Mg25_to_He4_Ne22)*Y(Mg25)*Y(N)*state.rho + screened_rates(k_He4_Ne22_to_n_Mg25)*Y(He4)*Y(Ne22)*state.rho) +
        (-screened_rates(k_He4_Mg25_to_n_Si28)*Y(He4)*Y(Mg25)*state.rho + screened_rates(k_n_Si28_to_He4_Mg25)*Y(Si28)*Y(N)*state.rho) +
        (screened_rates(k_n_Al25_to_p_Mg25)*Y(Al25)*Y(N)*state.rho + -screened_rates(k_p_Mg25_to_n_Al25)*Y(Mg25)*Y(H1)*state.rho) +
        (screened_rates(k_Al25_to_Mg25)*Y(Al25) + -screened_rates(k_Mg25_to_Al25)*Y(Mg25));

    ydot_nuc(Mg26) =
        (screened_rates(k_He4_Ne22_to_Mg26)*Y(He4)*Y(Ne22)*state.rho + -screened_rates(k_Mg26_to_He4_Ne22)*Y(Mg26)) +
        (screened_rates(k_n_Mg25_to_Mg26)*Y(Mg25)*Y(N)*state.rho + -screened_rates(k_Mg26_to_n_Mg25)*Y(Mg26)) +
        (-screened_rates(k_p_Mg26_to_Al27)*Y(Mg26)*Y(H1)*state.rho + screened_rates(k_Al27_to_p_Mg26)*Y(Al27)) +
        (-screened_rates(k_He4_Mg26_to_Si30)*Y(He4)*Y(Mg26)*state.rho + screened_rates(k_Si30_to_He4_Mg26)*Y(Si30)) +
        (screened_rates(k_He4_Na23_to_p_Mg26)*Y(He4)*Y(Na23)*state.rho + -screened_rates(k_p_Mg26_to_He4_Na23)*Y(Mg26)*Y(H1)*state.rho) +
        (-screened_rates(k_He4_Mg26_to_n_Si29)*Y(He4)*Y(Mg26)*state.rho + screened_rates(k_n_Si29_to_He4_Mg26)*Y(Si29)*Y(N)*state.rho) +
        (screened_rates(k_n_Al26_to_p_Mg26)*Y(Al26)*Y(N)*state.rho + -screened_rates(k_p_Mg26_to_n_Al26)*Y(Mg26)*Y(H1)*state.rho) +
        (screened_rates(k_Al26_to_Mg26)*Y(Al26) + -screened_rates(k_Mg26_to_Al26)*Y(Mg26));

    ydot_nuc(Al25) =
        (screened_rates(k_He4_Na21_to_Al25)*Y(He4)*Y(Na21)*state.rho + -screened_rates(k_Al25_to_He4_Na21)*Y(Al25)) +
        (screened_rates(k_p_Mg24_to_Al25)*Y(Mg24)*Y(H1)*state.rho + -screened_rates(k_Al25_to_p_Mg24)*Y(Al25)) +
        (-screened_rates(k_n_Al25_to_Al26)*Y(Al25)*Y(N)*state.rho + screened_rates(k_Al26_to_n_Al25)*Y(Al26)) +
        (-screened_rates(k_He4_Al25_to_P29)*Y(Al25)*Y(He4)*state.rho + screened_rates(k_P29_to_He4_Al25)*Y(P29)) +
        (-screened_rates(k_n_Al25_to_p_Mg25)*Y(Al25)*Y(N)*state.rho + screened_rates(k_p_Mg25_to_n_Al25)*Y(Mg25)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Al25_to_He4_Na22)*Y(Al25)*Y(N)*state.rho + screened_rates(k_He4_Na22_to_n_Al25)*Y(He4)*Y(Na22)*state.rho) +
        (-screened_rates(k_He4_Al25_to_p_Si28)*Y(Al25)*Y(He4)*state.rho + screened_rates(k_p_Si28_to_He4_Al25)*Y(Si28)*Y(H1)*state.rho) +
        (-screened_rates(k_Al25_to_Mg25)*Y(Al25) + screened_rates(k_Mg25_to_Al25)*Y(Mg25));

    ydot_nuc(Al26) =
        (screened_rates(k_He4_Na22_to_Al26)*Y(He4)*Y(Na22)*state.rho + -screened_rates(k_Al26_to_He4_Na22)*Y(Al26)) +
        (screened_rates(k_p_Mg25_to_Al26)*Y(Mg25)*Y(H1)*state.rho + -screened_rates(k_Al26_to_p_Mg25)*Y(Al26)) +
        (screened_rates(k_n_Al25_to_Al26)*Y(Al25)*Y(N)*state.rho + -screened_rates(k_Al26_to_n_Al25)*Y(Al26)) +
        (-screened_rates(k_n_Al26_to_Al27)*Y(Al26)*Y(N)*state.rho + screened_rates(k_Al27_to_n_Al26)*Y(Al27)) +
        (-screened_rates(k_He4_Al26_to_P30)*Y(Al26)*Y(He4)*state.rho + screened_rates(k_P30_to_He4_Al26)*Y(P30)) +
        (screened_rates(k_He4_Mg23_to_p_Al26)*Y(He4)*Y(Mg23)*state.rho + -screened_rates(k_p_Al26_to_He4_Mg23)*Y(Al26)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Al26_to_p_Mg26)*Y(Al26)*Y(N)*state.rho + screened_rates(k_p_Mg26_to_n_Al26)*Y(Mg26)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Al26_to_He4_Na23)*Y(Al26)*Y(N)*state.rho + screened_rates(k_He4_Na23_to_n_Al26)*Y(He4)*Y(Na23)*state.rho) +
        (-screened_rates(k_He4_Al26_to_p_Si29)*Y(Al26)*Y(He4)*state.rho + screened_rates(k_p_Si29_to_He4_Al26)*Y(Si29)*Y(H1)*state.rho) +
        (screened_rates(k_n_P29_to_He4_Al26)*Y(P29)*Y(N)*state.rho + -screened_rates(k_He4_Al26_to_n_P29)*Y(Al26)*Y(He4)*state.rho) +
        (-screened_rates(k_Al26_to_Mg26)*Y(Al26) + screened_rates(k_Mg26_to_Al26)*Y(Mg26));

    ydot_nuc(Al27) =
        (screened_rates(k_He4_Na23_to_Al27)*Y(He4)*Y(Na23)*state.rho + -screened_rates(k_Al27_to_He4_Na23)*Y(Al27)) +
        (screened_rates(k_p_Mg26_to_Al27)*Y(Mg26)*Y(H1)*state.rho + -screened_rates(k_Al27_to_p_Mg26)*Y(Al27)) +
        (screened_rates(k_n_Al26_to_Al27)*Y(Al26)*Y(N)*state.rho + -screened_rates(k_Al27_to_n_Al26)*Y(Al27)) +
        (-screened_rates(k_p_Al27_to_Si28)*Y(Al27)*Y(H1)*state.rho + screened_rates(k_Si28_to_p_Al27)*Y(Si28)) +
        (-screened_rates(k_He4_Al27_to_P31)*Y(Al27)*Y(He4)*state.rho + screened_rates(k_P31_to_He4_Al27)*Y(P31)) +
        (screened_rates(k_C12_O16_to_p_Al27)*Y(C12)*Y(O16)*state.rho + -screened_rates(k_p_Al27_to_C12_O16)*Y(Al27)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Al27_to_He4_Mg24)*Y(Al27)*Y(H1)*state.rho + screened_rates(k_He4_Mg24_to_p_Al27)*Y(He4)*Y(Mg24)*state.rho) +
        (-screened_rates(k_He4_Al27_to_p_Si30)*Y(Al27)*Y(He4)*state.rho + screened_rates(k_p_Si30_to_He4_Al27)*Y(Si30)*Y(H1)*state.rho) +
        (screened_rates(k_n_P30_to_He4_Al27)*Y(P30)*Y(N)*state.rho + -screened_rates(k_He4_Al27_to_n_P30)*Y(Al27)*Y(He4)*state.rho);

    ydot_nuc(Si28) =
        (screened_rates(k_He4_Mg24_to_Si28)*Y(He4)*Y(Mg24)*state.rho + -screened_rates(k_Si28_to_He4_Mg24)*Y(Si28)) +
        (screened_rates(k_p_Al27_to_Si28)*Y(Al27)*Y(H1)*state.rho + -screened_rates(k_Si28_to_p_Al27)*Y(Si28)) +
        (-screened_rates(k_n_Si28_to_Si29)*Y(Si28)*Y(N)*state.rho + screened_rates(k_Si29_to_n_Si28)*Y(Si29)) +
        (-screened_rates(k_p_Si28_to_P29)*Y(Si28)*Y(H1)*state.rho + screened_rates(k_P29_to_p_Si28)*Y(P29)) +
        (-screened_rates(k_He4_Si28_to_S32)*Y(He4)*Y(Si28)*state.rho + screened_rates(k_S32_to_He4_Si28)*Y(S32)) +
        (0.5*screened_rates(k_O16_O16_to_He4_Si28)*amrex::Math::powi<2>(Y(O16))*state.rho + -screened_rates(k_He4_Si28_to_O16_O16)*Y(He4)*Y(Si28)*state.rho) +
        (screened_rates(k_C12_Ne20_to_He4_Si28)*Y(C12)*Y(Ne20)*state.rho + -screened_rates(k_He4_Si28_to_C12_Ne20)*Y(He4)*Y(Si28)*state.rho) +
        (screened_rates(k_He4_Mg25_to_n_Si28)*Y(He4)*Y(Mg25)*state.rho + -screened_rates(k_n_Si28_to_He4_Mg25)*Y(Si28)*Y(N)*state.rho) +
        (screened_rates(k_He4_Al25_to_p_Si28)*Y(Al25)*Y(He4)*state.rho + -screened_rates(k_p_Si28_to_He4_Al25)*Y(Si28)*Y(H1)*state.rho) +
        (screened_rates(k_p_P31_to_He4_Si28)*Y(P31)*Y(H1)*state.rho + -screened_rates(k_He4_Si28_to_p_P31)*Y(He4)*Y(Si28)*state.rho);

    ydot_nuc(Si29) =
        (screened_rates(k_He4_Mg25_to_Si29)*Y(He4)*Y(Mg25)*state.rho + -screened_rates(k_Si29_to_He4_Mg25)*Y(Si29)) +
        (screened_rates(k_n_Si28_to_Si29)*Y(Si28)*Y(N)*state.rho + -screened_rates(k_Si29_to_n_Si28)*Y(Si29)) +
        (-screened_rates(k_n_Si29_to_Si30)*Y(Si29)*Y(N)*state.rho + screened_rates(k_Si30_to_n_Si29)*Y(Si30)) +
        (-screened_rates(k_p_Si29_to_P30)*Y(Si29)*Y(H1)*state.rho + screened_rates(k_P30_to_p_Si29)*Y(P30)) +
        (-screened_rates(k_He4_Si29_to_S33)*Y(He4)*Y(Si29)*state.rho + screened_rates(k_S33_to_He4_Si29)*Y(S33)) +
        (screened_rates(k_He4_Mg26_to_n_Si29)*Y(He4)*Y(Mg26)*state.rho + -screened_rates(k_n_Si29_to_He4_Mg26)*Y(Si29)*Y(N)*state.rho) +
        (screened_rates(k_He4_Al26_to_p_Si29)*Y(Al26)*Y(He4)*state.rho + -screened_rates(k_p_Si29_to_He4_Al26)*Y(Si29)*Y(H1)*state.rho) +
        (screened_rates(k_n_P29_to_p_Si29)*Y(P29)*Y(N)*state.rho + -screened_rates(k_p_Si29_to_n_P29)*Y(Si29)*Y(H1)*state.rho) +
        (screened_rates(k_p_P32_to_He4_Si29)*Y(P32)*Y(H1)*state.rho + -screened_rates(k_He4_Si29_to_p_P32)*Y(He4)*Y(Si29)*state.rho) +
        (screened_rates(k_n_S32_to_He4_Si29)*Y(S32)*Y(N)*state.rho + -screened_rates(k_He4_Si29_to_n_S32)*Y(He4)*Y(Si29)*state.rho) +
        (screened_rates(k_P29_to_Si29)*Y(P29) + -screened_rates(k_Si29_to_P29)*Y(Si29));

    ydot_nuc(Si30) =
        (screened_rates(k_He4_Mg26_to_Si30)*Y(He4)*Y(Mg26)*state.rho + -screened_rates(k_Si30_to_He4_Mg26)*Y(Si30)) +
        (screened_rates(k_n_Si29_to_Si30)*Y(Si29)*Y(N)*state.rho + -screened_rates(k_Si30_to_n_Si29)*Y(Si30)) +
        (-screened_rates(k_n_Si30_to_Si31)*Y(Si30)*Y(N)*state.rho + screened_rates(k_Si31_to_n_Si30)*Y(Si31)) +
        (-screened_rates(k_p_Si30_to_P31)*Y(Si30)*Y(H1)*state.rho + screened_rates(k_P31_to_p_Si30)*Y(P31)) +
        (-screened_rates(k_He4_Si30_to_S34)*Y(He4)*Y(Si30)*state.rho + screened_rates(k_S34_to_He4_Si30)*Y(S34)) +
        (screened_rates(k_He4_Al27_to_p_Si30)*Y(Al27)*Y(He4)*state.rho + -screened_rates(k_p_Si30_to_He4_Al27)*Y(Si30)*Y(H1)*state.rho) +
        (screened_rates(k_n_P30_to_p_Si30)*Y(P30)*Y(N)*state.rho + -screened_rates(k_p_Si30_to_n_P30)*Y(Si30)*Y(H1)*state.rho) +
        (screened_rates(k_p_P33_to_He4_Si30)*Y(P33)*Y(H1)*state.rho + -screened_rates(k_He4_Si30_to_p_P33)*Y(He4)*Y(Si30)*state.rho) +
        (screened_rates(k_n_S33_to_He4_Si30)*Y(S33)*Y(N)*state.rho + -screened_rates(k_He4_Si30_to_n_S33)*Y(He4)*Y(Si30)*state.rho) +
        (screened_rates(k_P30_to_Si30)*Y(P30) + -screened_rates(k_Si30_to_P30)*Y(Si30));

    ydot_nuc(Si31) =
        (screened_rates(k_n_Si30_to_Si31)*Y(Si30)*Y(N)*state.rho + -screened_rates(k_Si31_to_n_Si30)*Y(Si31)) +
        (-screened_rates(k_n_Si31_to_Si32)*Y(Si31)*Y(N)*state.rho + screened_rates(k_Si32_to_n_Si31)*Y(Si32)) +
        (-screened_rates(k_p_Si31_to_P32)*Y(Si31)*Y(H1)*state.rho + screened_rates(k_P32_to_p_Si31)*Y(P32)) +
        (-screened_rates(k_He4_Si31_to_S35)*Y(He4)*Y(Si31)*state.rho + screened_rates(k_S35_to_He4_Si31)*Y(S35)) +
        (-screened_rates(k_p_Si31_to_n_P31)*Y(Si31)*Y(H1)*state.rho + screened_rates(k_n_P31_to_p_Si31)*Y(P31)*Y(N)*state.rho) +
        (-screened_rates(k_He4_Si31_to_n_S34)*Y(He4)*Y(Si31)*state.rho + screened_rates(k_n_S34_to_He4_Si31)*Y(S34)*Y(N)*state.rho) +
        (-screened_rates(k_Si31_to_P31)*Y(Si31) + screened_rates(k_P31_to_Si31)*Y(P31));

    ydot_nuc(Si32) =
        (screened_rates(k_n_Si31_to_Si32)*Y(Si31)*Y(N)*state.rho + -screened_rates(k_Si32_to_n_Si31)*Y(Si32)) +
        (-screened_rates(k_p_Si32_to_P33)*Y(Si32)*Y(H1)*state.rho + screened_rates(k_P33_to_p_Si32)*Y(P33)) +
        (-screened_rates(k_He4_Si32_to_S36)*Y(He4)*Y(Si32)*state.rho + screened_rates(k_S36_to_He4_Si32)*Y(S36)) +
        (screened_rates(k_n_P32_to_p_Si32)*Y(P32)*Y(N)*state.rho + -screened_rates(k_p_Si32_to_n_P32)*Y(Si32)*Y(H1)*state.rho) +
        (screened_rates(k_n_S35_to_He4_Si32)*Y(S35)*Y(N)*state.rho + -screened_rates(k_He4_Si32_to_n_S35)*Y(He4)*Y(Si32)*state.rho) +
        (-screened_rates(k_Si32_to_P32)*Y(Si32) + screened_rates(k_P32_to_Si32)*Y(P32));

    ydot_nuc(P29) =
        (screened_rates(k_He4_Al25_to_P29)*Y(Al25)*Y(He4)*state.rho + -screened_rates(k_P29_to_He4_Al25)*Y(P29)) +
        (screened_rates(k_p_Si28_to_P29)*Y(Si28)*Y(H1)*state.rho + -screened_rates(k_P29_to_p_Si28)*Y(P29)) +
        (-screened_rates(k_n_P29_to_P30)*Y(P29)*Y(N)*state.rho + screened_rates(k_P30_to_n_P29)*Y(P30)) +
        (-screened_rates(k_He4_P29_to_Cl33)*Y(He4)*Y(P29)*state.rho + screened_rates(k_Cl33_to_He4_P29)*Y(Cl33)) +
        (-screened_rates(k_n_P29_to_p_Si29)*Y(P29)*Y(N)*state.rho + screened_rates(k_p_Si29_to_n_P29)*Y(Si29)*Y(H1)*state.rho) +
        (-screened_rates(k_n_P29_to_He4_Al26)*Y(P29)*Y(N)*state.rho + screened_rates(k_He4_Al26_to_n_P29)*Y(Al26)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_P29_to_p_S32)*Y(He4)*Y(P29)*state.rho + screened_rates(k_p_S32_to_He4_P29)*Y(S32)*Y(H1)*state.rho) +
        (-screened_rates(k_P29_to_Si29)*Y(P29) + screened_rates(k_Si29_to_P29)*Y(Si29));

    ydot_nuc(P30) =
        (screened_rates(k_He4_Al26_to_P30)*Y(Al26)*Y(He4)*state.rho + -screened_rates(k_P30_to_He4_Al26)*Y(P30)) +
        (screened_rates(k_p_Si29_to_P30)*Y(Si29)*Y(H1)*state.rho + -screened_rates(k_P30_to_p_Si29)*Y(P30)) +
        (screened_rates(k_n_P29_to_P30)*Y(P29)*Y(N)*state.rho + -screened_rates(k_P30_to_n_P29)*Y(P30)) +
        (-screened_rates(k_n_P30_to_P31)*Y(P30)*Y(N)*state.rho + screened_rates(k_P31_to_n_P30)*Y(P31)) +
        (-screened_rates(k_He4_P30_to_Cl34)*Y(He4)*Y(P30)*state.rho + screened_rates(k_Cl34_to_He4_P30)*Y(Cl34)) +
        (-screened_rates(k_n_P30_to_p_Si30)*Y(P30)*Y(N)*state.rho + screened_rates(k_p_Si30_to_n_P30)*Y(Si30)*Y(H1)*state.rho) +
        (-screened_rates(k_n_P30_to_He4_Al27)*Y(P30)*Y(N)*state.rho + screened_rates(k_He4_Al27_to_n_P30)*Y(Al27)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_P30_to_p_S33)*Y(He4)*Y(P30)*state.rho + screened_rates(k_p_S33_to_He4_P30)*Y(S33)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cl33_to_He4_P30)*Y(Cl33)*Y(N)*state.rho + -screened_rates(k_He4_P30_to_n_Cl33)*Y(He4)*Y(P30)*state.rho) +
        (-screened_rates(k_P30_to_Si30)*Y(P30) + screened_rates(k_Si30_to_P30)*Y(Si30));

    ydot_nuc(P31) =
        (screened_rates(k_He4_Al27_to_P31)*Y(Al27)*Y(He4)*state.rho + -screened_rates(k_P31_to_He4_Al27)*Y(P31)) +
        (screened_rates(k_p_Si30_to_P31)*Y(Si30)*Y(H1)*state.rho + -screened_rates(k_P31_to_p_Si30)*Y(P31)) +
        (screened_rates(k_n_P30_to_P31)*Y(P30)*Y(N)*state.rho + -screened_rates(k_P31_to_n_P30)*Y(P31)) +
        (-screened_rates(k_n_P31_to_P32)*Y(P31)*Y(N)*state.rho + screened_rates(k_P32_to_n_P31)*Y(P32)) +
        (-screened_rates(k_p_P31_to_S32)*Y(P31)*Y(H1)*state.rho + screened_rates(k_S32_to_p_P31)*Y(S32)) +
        (-screened_rates(k_He4_P31_to_Cl35)*Y(He4)*Y(P31)*state.rho + screened_rates(k_Cl35_to_He4_P31)*Y(Cl35)) +
        (0.5*screened_rates(k_O16_O16_to_p_P31)*amrex::Math::powi<2>(Y(O16))*state.rho + -screened_rates(k_p_P31_to_O16_O16)*Y(P31)*Y(H1)*state.rho) +
        (screened_rates(k_C12_Ne20_to_p_P31)*Y(C12)*Y(Ne20)*state.rho + -screened_rates(k_p_P31_to_C12_Ne20)*Y(P31)*Y(H1)*state.rho) +
        (screened_rates(k_p_Si31_to_n_P31)*Y(Si31)*Y(H1)*state.rho + -screened_rates(k_n_P31_to_p_Si31)*Y(P31)*Y(N)*state.rho) +
        (-screened_rates(k_p_P31_to_He4_Si28)*Y(P31)*Y(H1)*state.rho + screened_rates(k_He4_Si28_to_p_P31)*Y(He4)*Y(Si28)*state.rho) +
        (-screened_rates(k_He4_P31_to_p_S34)*Y(He4)*Y(P31)*state.rho + screened_rates(k_p_S34_to_He4_P31)*Y(S34)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cl34_to_He4_P31)*Y(Cl34)*Y(N)*state.rho + -screened_rates(k_He4_P31_to_n_Cl34)*Y(He4)*Y(P31)*state.rho) +
        (screened_rates(k_Si31_to_P31)*Y(Si31) + -screened_rates(k_P31_to_Si31)*Y(P31));

    ydot_nuc(P32) =
        (screened_rates(k_p_Si31_to_P32)*Y(Si31)*Y(H1)*state.rho + -screened_rates(k_P32_to_p_Si31)*Y(P32)) +
        (screened_rates(k_n_P31_to_P32)*Y(P31)*Y(N)*state.rho + -screened_rates(k_P32_to_n_P31)*Y(P32)) +
        (-screened_rates(k_n_P32_to_P33)*Y(P32)*Y(N)*state.rho + screened_rates(k_P33_to_n_P32)*Y(P33)) +
        (-screened_rates(k_p_P32_to_S33)*Y(P32)*Y(H1)*state.rho + screened_rates(k_S33_to_p_P32)*Y(S33)) +
        (-screened_rates(k_He4_P32_to_Cl36)*Y(He4)*Y(P32)*state.rho + screened_rates(k_Cl36_to_He4_P32)*Y(Cl36)) +
        (-screened_rates(k_n_P32_to_p_Si32)*Y(P32)*Y(N)*state.rho + screened_rates(k_p_Si32_to_n_P32)*Y(Si32)*Y(H1)*state.rho) +
        (-screened_rates(k_p_P32_to_n_S32)*Y(P32)*Y(H1)*state.rho + screened_rates(k_n_S32_to_p_P32)*Y(S32)*Y(N)*state.rho) +
        (-screened_rates(k_p_P32_to_He4_Si29)*Y(P32)*Y(H1)*state.rho + screened_rates(k_He4_Si29_to_p_P32)*Y(He4)*Y(Si29)*state.rho) +
        (screened_rates(k_p_S35_to_He4_P32)*Y(S35)*Y(H1)*state.rho + -screened_rates(k_He4_P32_to_p_S35)*Y(He4)*Y(P32)*state.rho) +
        (screened_rates(k_n_Cl35_to_He4_P32)*Y(Cl35)*Y(N)*state.rho + -screened_rates(k_He4_P32_to_n_Cl35)*Y(He4)*Y(P32)*state.rho) +
        (-screened_rates(k_P32_to_S32)*Y(P32) + screened_rates(k_S32_to_P32)*Y(S32)) +
        (screened_rates(k_Si32_to_P32)*Y(Si32) + -screened_rates(k_P32_to_Si32)*Y(P32));

    ydot_nuc(P33) =
        (screened_rates(k_p_Si32_to_P33)*Y(Si32)*Y(H1)*state.rho + -screened_rates(k_P33_to_p_Si32)*Y(P33)) +
        (screened_rates(k_n_P32_to_P33)*Y(P32)*Y(N)*state.rho + -screened_rates(k_P33_to_n_P32)*Y(P33)) +
        (-screened_rates(k_p_P33_to_S34)*Y(P33)*Y(H1)*state.rho + screened_rates(k_S34_to_p_P33)*Y(S34)) +
        (-screened_rates(k_He4_P33_to_Cl37)*Y(He4)*Y(P33)*state.rho + screened_rates(k_Cl37_to_He4_P33)*Y(Cl37)) +
        (-screened_rates(k_p_P33_to_He4_Si30)*Y(P33)*Y(H1)*state.rho + screened_rates(k_He4_Si30_to_p_P33)*Y(He4)*Y(Si30)*state.rho) +
        (screened_rates(k_n_S33_to_p_P33)*Y(S33)*Y(N)*state.rho + -screened_rates(k_p_P33_to_n_S33)*Y(P33)*Y(H1)*state.rho) +
        (screened_rates(k_p_S36_to_He4_P33)*Y(S36)*Y(H1)*state.rho + -screened_rates(k_He4_P33_to_p_S36)*Y(He4)*Y(P33)*state.rho) +
        (screened_rates(k_n_Cl36_to_He4_P33)*Y(Cl36)*Y(N)*state.rho + -screened_rates(k_He4_P33_to_n_Cl36)*Y(He4)*Y(P33)*state.rho) +
        (-screened_rates(k_P33_to_S33)*Y(P33) + screened_rates(k_S33_to_P33)*Y(S33));

    ydot_nuc(S32) =
        (screened_rates(k_He4_Si28_to_S32)*Y(He4)*Y(Si28)*state.rho + -screened_rates(k_S32_to_He4_Si28)*Y(S32)) +
        (screened_rates(k_p_P31_to_S32)*Y(P31)*Y(H1)*state.rho + -screened_rates(k_S32_to_p_P31)*Y(S32)) +
        (-screened_rates(k_n_S32_to_S33)*Y(S32)*Y(N)*state.rho + screened_rates(k_S33_to_n_S32)*Y(S33)) +
        (-screened_rates(k_p_S32_to_Cl33)*Y(S32)*Y(H1)*state.rho + screened_rates(k_Cl33_to_p_S32)*Y(Cl33)) +
        (-screened_rates(k_He4_S32_to_Ar36)*Y(He4)*Y(S32)*state.rho + screened_rates(k_Ar36_to_He4_S32)*Y(Ar36)) +
        (screened_rates(k_He4_P29_to_p_S32)*Y(He4)*Y(P29)*state.rho + -screened_rates(k_p_S32_to_He4_P29)*Y(S32)*Y(H1)*state.rho) +
        (screened_rates(k_p_P32_to_n_S32)*Y(P32)*Y(H1)*state.rho + -screened_rates(k_n_S32_to_p_P32)*Y(S32)*Y(N)*state.rho) +
        (-screened_rates(k_n_S32_to_He4_Si29)*Y(S32)*Y(N)*state.rho + screened_rates(k_He4_Si29_to_n_S32)*Y(He4)*Y(Si29)*state.rho) +
        (screened_rates(k_p_Cl35_to_He4_S32)*Y(Cl35)*Y(H1)*state.rho + -screened_rates(k_He4_S32_to_p_Cl35)*Y(He4)*Y(S32)*state.rho) +
        (screened_rates(k_P32_to_S32)*Y(P32) + -screened_rates(k_S32_to_P32)*Y(S32));

    ydot_nuc(S33) =
        (screened_rates(k_He4_Si29_to_S33)*Y(He4)*Y(Si29)*state.rho + -screened_rates(k_S33_to_He4_Si29)*Y(S33)) +
        (screened_rates(k_p_P32_to_S33)*Y(P32)*Y(H1)*state.rho + -screened_rates(k_S33_to_p_P32)*Y(S33)) +
        (screened_rates(k_n_S32_to_S33)*Y(S32)*Y(N)*state.rho + -screened_rates(k_S33_to_n_S32)*Y(S33)) +
        (-screened_rates(k_n_S33_to_S34)*Y(S33)*Y(N)*state.rho + screened_rates(k_S34_to_n_S33)*Y(S34)) +
        (-screened_rates(k_p_S33_to_Cl34)*Y(S33)*Y(H1)*state.rho + screened_rates(k_Cl34_to_p_S33)*Y(Cl34)) +
        (-screened_rates(k_He4_S33_to_Ar37)*Y(He4)*Y(S33)*state.rho + screened_rates(k_Ar37_to_He4_S33)*Y(Ar37)) +
        (screened_rates(k_He4_P30_to_p_S33)*Y(He4)*Y(P30)*state.rho + -screened_rates(k_p_S33_to_He4_P30)*Y(S33)*Y(H1)*state.rho) +
        (-screened_rates(k_n_S33_to_p_P33)*Y(S33)*Y(N)*state.rho + screened_rates(k_p_P33_to_n_S33)*Y(P33)*Y(H1)*state.rho) +
        (-screened_rates(k_n_S33_to_He4_Si30)*Y(S33)*Y(N)*state.rho + screened_rates(k_He4_Si30_to_n_S33)*Y(He4)*Y(Si30)*state.rho) +
        (screened_rates(k_n_Cl33_to_p_S33)*Y(Cl33)*Y(N)*state.rho + -screened_rates(k_p_S33_to_n_Cl33)*Y(S33)*Y(H1)*state.rho) +
        (screened_rates(k_p_Cl36_to_He4_S33)*Y(Cl36)*Y(H1)*state.rho + -screened_rates(k_He4_S33_to_p_Cl36)*Y(He4)*Y(S33)*state.rho) +
        (screened_rates(k_n_Ar36_to_He4_S33)*Y(Ar36)*Y(N)*state.rho + -screened_rates(k_He4_S33_to_n_Ar36)*Y(He4)*Y(S33)*state.rho) +
        (screened_rates(k_Cl33_to_S33)*Y(Cl33) + -screened_rates(k_S33_to_Cl33)*Y(S33)) +
        (screened_rates(k_P33_to_S33)*Y(P33) + -screened_rates(k_S33_to_P33)*Y(S33));

    ydot_nuc(S34) =
        (screened_rates(k_He4_Si30_to_S34)*Y(He4)*Y(Si30)*state.rho + -screened_rates(k_S34_to_He4_Si30)*Y(S34)) +
        (screened_rates(k_p_P33_to_S34)*Y(P33)*Y(H1)*state.rho + -screened_rates(k_S34_to_p_P33)*Y(S34)) +
        (screened_rates(k_n_S33_to_S34)*Y(S33)*Y(N)*state.rho + -screened_rates(k_S34_to_n_S33)*Y(S34)) +
        (-screened_rates(k_n_S34_to_S35)*Y(S34)*Y(N)*state.rho + screened_rates(k_S35_to_n_S34)*Y(S35)) +
        (-screened_rates(k_p_S34_to_Cl35)*Y(S34)*Y(H1)*state.rho + screened_rates(k_Cl35_to_p_S34)*Y(Cl35)) +
        (-screened_rates(k_He4_S34_to_Ar38)*Y(He4)*Y(S34)*state.rho + screened_rates(k_Ar38_to_He4_S34)*Y(Ar38)) +
        (screened_rates(k_He4_Si31_to_n_S34)*Y(He4)*Y(Si31)*state.rho + -screened_rates(k_n_S34_to_He4_Si31)*Y(S34)*Y(N)*state.rho) +
        (screened_rates(k_He4_P31_to_p_S34)*Y(He4)*Y(P31)*state.rho + -screened_rates(k_p_S34_to_He4_P31)*Y(S34)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cl34_to_p_S34)*Y(Cl34)*Y(N)*state.rho + -screened_rates(k_p_S34_to_n_Cl34)*Y(S34)*Y(H1)*state.rho) +
        (screened_rates(k_p_Cl37_to_He4_S34)*Y(Cl37)*Y(H1)*state.rho + -screened_rates(k_He4_S34_to_p_Cl37)*Y(He4)*Y(S34)*state.rho) +
        (screened_rates(k_n_Ar37_to_He4_S34)*Y(Ar37)*Y(N)*state.rho + -screened_rates(k_He4_S34_to_n_Ar37)*Y(He4)*Y(S34)*state.rho) +
        (screened_rates(k_Cl34_to_S34)*Y(Cl34) + -screened_rates(k_S34_to_Cl34)*Y(S34));

    ydot_nuc(S35) =
        (screened_rates(k_He4_Si31_to_S35)*Y(He4)*Y(Si31)*state.rho + -screened_rates(k_S35_to_He4_Si31)*Y(S35)) +
        (screened_rates(k_n_S34_to_S35)*Y(S34)*Y(N)*state.rho + -screened_rates(k_S35_to_n_S34)*Y(S35)) +
        (-screened_rates(k_n_S35_to_S36)*Y(S35)*Y(N)*state.rho + screened_rates(k_S36_to_n_S35)*Y(S36)) +
        (-screened_rates(k_p_S35_to_Cl36)*Y(S35)*Y(H1)*state.rho + screened_rates(k_Cl36_to_p_S35)*Y(Cl36)) +
        (-screened_rates(k_He4_S35_to_Ar39)*Y(He4)*Y(S35)*state.rho + screened_rates(k_Ar39_to_He4_S35)*Y(Ar39)) +
        (-screened_rates(k_n_S35_to_He4_Si32)*Y(S35)*Y(N)*state.rho + screened_rates(k_He4_Si32_to_n_S35)*Y(He4)*Y(Si32)*state.rho) +
        (-screened_rates(k_p_S35_to_He4_P32)*Y(S35)*Y(H1)*state.rho + screened_rates(k_He4_P32_to_p_S35)*Y(He4)*Y(P32)*state.rho) +
        (-screened_rates(k_He4_S35_to_n_Ar38)*Y(He4)*Y(S35)*state.rho + screened_rates(k_n_Ar38_to_He4_S35)*Y(Ar38)*Y(N)*state.rho) +
        (screened_rates(k_n_Cl35_to_p_S35)*Y(Cl35)*Y(N)*state.rho + -screened_rates(k_p_S35_to_n_Cl35)*Y(S35)*Y(H1)*state.rho) +
        (-screened_rates(k_S35_to_Cl35)*Y(S35) + screened_rates(k_Cl35_to_S35)*Y(Cl35));

    ydot_nuc(S36) =
        (screened_rates(k_He4_Si32_to_S36)*Y(He4)*Y(Si32)*state.rho + -screened_rates(k_S36_to_He4_Si32)*Y(S36)) +
        (screened_rates(k_n_S35_to_S36)*Y(S35)*Y(N)*state.rho + -screened_rates(k_S36_to_n_S35)*Y(S36)) +
        (-screened_rates(k_p_S36_to_Cl37)*Y(S36)*Y(H1)*state.rho + screened_rates(k_Cl37_to_p_S36)*Y(Cl37)) +
        (-screened_rates(k_He4_S36_to_Ar40)*Y(He4)*Y(S36)*state.rho + screened_rates(k_Ar40_to_He4_S36)*Y(Ar40)) +
        (-screened_rates(k_p_S36_to_He4_P33)*Y(S36)*Y(H1)*state.rho + screened_rates(k_He4_P33_to_p_S36)*Y(He4)*Y(P33)*state.rho) +
        (screened_rates(k_n_Cl36_to_p_S36)*Y(Cl36)*Y(N)*state.rho + -screened_rates(k_p_S36_to_n_Cl36)*Y(S36)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ar39_to_He4_S36)*Y(Ar39)*Y(N)*state.rho + -screened_rates(k_He4_S36_to_n_Ar39)*Y(He4)*Y(S36)*state.rho) +
        (screened_rates(k_Cl36_to_S36)*Y(Cl36) + -screened_rates(k_S36_to_Cl36)*Y(S36));

    ydot_nuc(Cl33) =
        (screened_rates(k_He4_P29_to_Cl33)*Y(He4)*Y(P29)*state.rho + -screened_rates(k_Cl33_to_He4_P29)*Y(Cl33)) +
        (screened_rates(k_p_S32_to_Cl33)*Y(S32)*Y(H1)*state.rho + -screened_rates(k_Cl33_to_p_S32)*Y(Cl33)) +
        (-screened_rates(k_n_Cl33_to_Cl34)*Y(Cl33)*Y(N)*state.rho + screened_rates(k_Cl34_to_n_Cl33)*Y(Cl34)) +
        (-screened_rates(k_He4_Cl33_to_K37)*Y(Cl33)*Y(He4)*state.rho + screened_rates(k_K37_to_He4_Cl33)*Y(K37)) +
        (-screened_rates(k_n_Cl33_to_p_S33)*Y(Cl33)*Y(N)*state.rho + screened_rates(k_p_S33_to_n_Cl33)*Y(S33)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cl33_to_He4_P30)*Y(Cl33)*Y(N)*state.rho + screened_rates(k_He4_P30_to_n_Cl33)*Y(He4)*Y(P30)*state.rho) +
        (-screened_rates(k_He4_Cl33_to_p_Ar36)*Y(Cl33)*Y(He4)*state.rho + screened_rates(k_p_Ar36_to_He4_Cl33)*Y(Ar36)*Y(H1)*state.rho) +
        (-screened_rates(k_Cl33_to_S33)*Y(Cl33) + screened_rates(k_S33_to_Cl33)*Y(S33));

    ydot_nuc(Cl34) =
        (screened_rates(k_He4_P30_to_Cl34)*Y(He4)*Y(P30)*state.rho + -screened_rates(k_Cl34_to_He4_P30)*Y(Cl34)) +
        (screened_rates(k_p_S33_to_Cl34)*Y(S33)*Y(H1)*state.rho + -screened_rates(k_Cl34_to_p_S33)*Y(Cl34)) +
        (screened_rates(k_n_Cl33_to_Cl34)*Y(Cl33)*Y(N)*state.rho + -screened_rates(k_Cl34_to_n_Cl33)*Y(Cl34)) +
        (-screened_rates(k_n_Cl34_to_Cl35)*Y(Cl34)*Y(N)*state.rho + screened_rates(k_Cl35_to_n_Cl34)*Y(Cl35)) +
        (-screened_rates(k_He4_Cl34_to_K38)*Y(Cl34)*Y(He4)*state.rho + screened_rates(k_K38_to_He4_Cl34)*Y(K38)) +
        (-screened_rates(k_n_Cl34_to_p_S34)*Y(Cl34)*Y(N)*state.rho + screened_rates(k_p_S34_to_n_Cl34)*Y(S34)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cl34_to_He4_P31)*Y(Cl34)*Y(N)*state.rho + screened_rates(k_He4_P31_to_n_Cl34)*Y(He4)*Y(P31)*state.rho) +
        (-screened_rates(k_He4_Cl34_to_p_Ar37)*Y(Cl34)*Y(He4)*state.rho + screened_rates(k_p_Ar37_to_He4_Cl34)*Y(Ar37)*Y(H1)*state.rho) +
        (screened_rates(k_n_K37_to_He4_Cl34)*Y(K37)*Y(N)*state.rho + -screened_rates(k_He4_Cl34_to_n_K37)*Y(Cl34)*Y(He4)*state.rho) +
        (-screened_rates(k_Cl34_to_S34)*Y(Cl34) + screened_rates(k_S34_to_Cl34)*Y(S34));

    ydot_nuc(Cl35) =
        (screened_rates(k_He4_P31_to_Cl35)*Y(He4)*Y(P31)*state.rho + -screened_rates(k_Cl35_to_He4_P31)*Y(Cl35)) +
        (screened_rates(k_p_S34_to_Cl35)*Y(S34)*Y(H1)*state.rho + -screened_rates(k_Cl35_to_p_S34)*Y(Cl35)) +
        (screened_rates(k_n_Cl34_to_Cl35)*Y(Cl34)*Y(N)*state.rho + -screened_rates(k_Cl35_to_n_Cl34)*Y(Cl35)) +
        (-screened_rates(k_n_Cl35_to_Cl36)*Y(Cl35)*Y(N)*state.rho + screened_rates(k_Cl36_to_n_Cl35)*Y(Cl36)) +
        (-screened_rates(k_p_Cl35_to_Ar36)*Y(Cl35)*Y(H1)*state.rho + screened_rates(k_Ar36_to_p_Cl35)*Y(Ar36)) +
        (-screened_rates(k_He4_Cl35_to_K39)*Y(Cl35)*Y(He4)*state.rho + screened_rates(k_K39_to_He4_Cl35)*Y(K39)) +
        (-screened_rates(k_n_Cl35_to_p_S35)*Y(Cl35)*Y(N)*state.rho + screened_rates(k_p_S35_to_n_Cl35)*Y(S35)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cl35_to_He4_P32)*Y(Cl35)*Y(N)*state.rho + screened_rates(k_He4_P32_to_n_Cl35)*Y(He4)*Y(P32)*state.rho) +
        (-screened_rates(k_p_Cl35_to_He4_S32)*Y(Cl35)*Y(H1)*state.rho + screened_rates(k_He4_S32_to_p_Cl35)*Y(He4)*Y(S32)*state.rho) +
        (-screened_rates(k_He4_Cl35_to_p_Ar38)*Y(Cl35)*Y(He4)*state.rho + screened_rates(k_p_Ar38_to_He4_Cl35)*Y(Ar38)*Y(H1)*state.rho) +
        (screened_rates(k_n_K38_to_He4_Cl35)*Y(K38)*Y(N)*state.rho + -screened_rates(k_He4_Cl35_to_n_K38)*Y(Cl35)*Y(He4)*state.rho) +
        (screened_rates(k_S35_to_Cl35)*Y(S35) + -screened_rates(k_Cl35_to_S35)*Y(Cl35));

    ydot_nuc(Cl36) =
        (screened_rates(k_He4_P32_to_Cl36)*Y(He4)*Y(P32)*state.rho + -screened_rates(k_Cl36_to_He4_P32)*Y(Cl36)) +
        (screened_rates(k_p_S35_to_Cl36)*Y(S35)*Y(H1)*state.rho + -screened_rates(k_Cl36_to_p_S35)*Y(Cl36)) +
        (screened_rates(k_n_Cl35_to_Cl36)*Y(Cl35)*Y(N)*state.rho + -screened_rates(k_Cl36_to_n_Cl35)*Y(Cl36)) +
        (-screened_rates(k_n_Cl36_to_Cl37)*Y(Cl36)*Y(N)*state.rho + screened_rates(k_Cl37_to_n_Cl36)*Y(Cl37)) +
        (-screened_rates(k_p_Cl36_to_Ar37)*Y(Cl36)*Y(H1)*state.rho + screened_rates(k_Ar37_to_p_Cl36)*Y(Ar37)) +
        (-screened_rates(k_He4_Cl36_to_K40)*Y(Cl36)*Y(He4)*state.rho + screened_rates(k_K40_to_He4_Cl36)*Y(K40)) +
        (-screened_rates(k_n_Cl36_to_p_S36)*Y(Cl36)*Y(N)*state.rho + screened_rates(k_p_S36_to_n_Cl36)*Y(S36)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cl36_to_He4_P33)*Y(Cl36)*Y(N)*state.rho + screened_rates(k_He4_P33_to_n_Cl36)*Y(He4)*Y(P33)*state.rho) +
        (-screened_rates(k_p_Cl36_to_He4_S33)*Y(Cl36)*Y(H1)*state.rho + screened_rates(k_He4_S33_to_p_Cl36)*Y(He4)*Y(S33)*state.rho) +
        (screened_rates(k_n_Ar36_to_p_Cl36)*Y(Ar36)*Y(N)*state.rho + -screened_rates(k_p_Cl36_to_n_Ar36)*Y(Cl36)*Y(H1)*state.rho) +
        (screened_rates(k_p_Ar39_to_He4_Cl36)*Y(Ar39)*Y(H1)*state.rho + -screened_rates(k_He4_Cl36_to_p_Ar39)*Y(Cl36)*Y(He4)*state.rho) +
        (screened_rates(k_n_K39_to_He4_Cl36)*Y(K39)*Y(N)*state.rho + -screened_rates(k_He4_Cl36_to_n_K39)*Y(Cl36)*Y(He4)*state.rho) +
        (-screened_rates(k_Cl36_to_Ar36)*Y(Cl36) + screened_rates(k_Ar36_to_Cl36)*Y(Ar36)) +
        (-screened_rates(k_Cl36_to_S36)*Y(Cl36) + screened_rates(k_S36_to_Cl36)*Y(S36));

    ydot_nuc(Cl37) =
        (screened_rates(k_He4_P33_to_Cl37)*Y(He4)*Y(P33)*state.rho + -screened_rates(k_Cl37_to_He4_P33)*Y(Cl37)) +
        (screened_rates(k_p_S36_to_Cl37)*Y(S36)*Y(H1)*state.rho + -screened_rates(k_Cl37_to_p_S36)*Y(Cl37)) +
        (screened_rates(k_n_Cl36_to_Cl37)*Y(Cl36)*Y(N)*state.rho + -screened_rates(k_Cl37_to_n_Cl36)*Y(Cl37)) +
        (-screened_rates(k_p_Cl37_to_Ar38)*Y(Cl37)*Y(H1)*state.rho + screened_rates(k_Ar38_to_p_Cl37)*Y(Ar38)) +
        (-screened_rates(k_He4_Cl37_to_K41)*Y(Cl37)*Y(He4)*state.rho + screened_rates(k_K41_to_He4_Cl37)*Y(K41)) +
        (-screened_rates(k_p_Cl37_to_He4_S34)*Y(Cl37)*Y(H1)*state.rho + screened_rates(k_He4_S34_to_p_Cl37)*Y(He4)*Y(S34)*state.rho) +
        (screened_rates(k_n_Ar37_to_p_Cl37)*Y(Ar37)*Y(N)*state.rho + -screened_rates(k_p_Cl37_to_n_Ar37)*Y(Cl37)*Y(H1)*state.rho) +
        (screened_rates(k_p_Ar40_to_He4_Cl37)*Y(Ar40)*Y(H1)*state.rho + -screened_rates(k_He4_Cl37_to_p_Ar40)*Y(Cl37)*Y(He4)*state.rho) +
        (screened_rates(k_n_K40_to_He4_Cl37)*Y(K40)*Y(N)*state.rho + -screened_rates(k_He4_Cl37_to_n_K40)*Y(Cl37)*Y(He4)*state.rho) +
        (screened_rates(k_Ar37_to_Cl37)*Y(Ar37) + -screened_rates(k_Cl37_to_Ar37)*Y(Cl37));

    ydot_nuc(Ar36) =
        (screened_rates(k_He4_S32_to_Ar36)*Y(He4)*Y(S32)*state.rho + -screened_rates(k_Ar36_to_He4_S32)*Y(Ar36)) +
        (screened_rates(k_p_Cl35_to_Ar36)*Y(Cl35)*Y(H1)*state.rho + -screened_rates(k_Ar36_to_p_Cl35)*Y(Ar36)) +
        (-screened_rates(k_n_Ar36_to_Ar37)*Y(Ar36)*Y(N)*state.rho + screened_rates(k_Ar37_to_n_Ar36)*Y(Ar37)) +
        (-screened_rates(k_p_Ar36_to_K37)*Y(Ar36)*Y(H1)*state.rho + screened_rates(k_K37_to_p_Ar36)*Y(K37)) +
        (-screened_rates(k_He4_Ar36_to_Ca40)*Y(Ar36)*Y(He4)*state.rho + screened_rates(k_Ca40_to_He4_Ar36)*Y(Ca40)) +
        (screened_rates(k_He4_Cl33_to_p_Ar36)*Y(Cl33)*Y(He4)*state.rho + -screened_rates(k_p_Ar36_to_He4_Cl33)*Y(Ar36)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ar36_to_p_Cl36)*Y(Ar36)*Y(N)*state.rho + screened_rates(k_p_Cl36_to_n_Ar36)*Y(Cl36)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ar36_to_He4_S33)*Y(Ar36)*Y(N)*state.rho + screened_rates(k_He4_S33_to_n_Ar36)*Y(He4)*Y(S33)*state.rho) +
        (screened_rates(k_p_K39_to_He4_Ar36)*Y(K39)*Y(H1)*state.rho + -screened_rates(k_He4_Ar36_to_p_K39)*Y(Ar36)*Y(He4)*state.rho) +
        (screened_rates(k_Cl36_to_Ar36)*Y(Cl36) + -screened_rates(k_Ar36_to_Cl36)*Y(Ar36));

    ydot_nuc(Ar37) =
        (screened_rates(k_He4_S33_to_Ar37)*Y(He4)*Y(S33)*state.rho + -screened_rates(k_Ar37_to_He4_S33)*Y(Ar37)) +
        (screened_rates(k_p_Cl36_to_Ar37)*Y(Cl36)*Y(H1)*state.rho + -screened_rates(k_Ar37_to_p_Cl36)*Y(Ar37)) +
        (screened_rates(k_n_Ar36_to_Ar37)*Y(Ar36)*Y(N)*state.rho + -screened_rates(k_Ar37_to_n_Ar36)*Y(Ar37)) +
        (-screened_rates(k_n_Ar37_to_Ar38)*Y(Ar37)*Y(N)*state.rho + screened_rates(k_Ar38_to_n_Ar37)*Y(Ar38)) +
        (-screened_rates(k_p_Ar37_to_K38)*Y(Ar37)*Y(H1)*state.rho + screened_rates(k_K38_to_p_Ar37)*Y(K38)) +
        (-screened_rates(k_He4_Ar37_to_Ca41)*Y(Ar37)*Y(He4)*state.rho + screened_rates(k_Ca41_to_He4_Ar37)*Y(Ca41)) +
        (screened_rates(k_He4_Cl34_to_p_Ar37)*Y(Cl34)*Y(He4)*state.rho + -screened_rates(k_p_Ar37_to_He4_Cl34)*Y(Ar37)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ar37_to_p_Cl37)*Y(Ar37)*Y(N)*state.rho + screened_rates(k_p_Cl37_to_n_Ar37)*Y(Cl37)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ar37_to_He4_S34)*Y(Ar37)*Y(N)*state.rho + screened_rates(k_He4_S34_to_n_Ar37)*Y(He4)*Y(S34)*state.rho) +
        (screened_rates(k_n_K37_to_p_Ar37)*Y(K37)*Y(N)*state.rho + -screened_rates(k_p_Ar37_to_n_K37)*Y(Ar37)*Y(H1)*state.rho) +
        (screened_rates(k_p_K40_to_He4_Ar37)*Y(K40)*Y(H1)*state.rho + -screened_rates(k_He4_Ar37_to_p_K40)*Y(Ar37)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ca40_to_He4_Ar37)*Y(Ca40)*Y(N)*state.rho + -screened_rates(k_He4_Ar37_to_n_Ca40)*Y(Ar37)*Y(He4)*state.rho) +
        (-screened_rates(k_Ar37_to_Cl37)*Y(Ar37) + screened_rates(k_Cl37_to_Ar37)*Y(Cl37)) +
        (screened_rates(k_K37_to_Ar37)*Y(K37) + -screened_rates(k_Ar37_to_K37)*Y(Ar37));

    ydot_nuc(Ar38) =
        (screened_rates(k_He4_S34_to_Ar38)*Y(He4)*Y(S34)*state.rho + -screened_rates(k_Ar38_to_He4_S34)*Y(Ar38)) +
        (screened_rates(k_p_Cl37_to_Ar38)*Y(Cl37)*Y(H1)*state.rho + -screened_rates(k_Ar38_to_p_Cl37)*Y(Ar38)) +
        (screened_rates(k_n_Ar37_to_Ar38)*Y(Ar37)*Y(N)*state.rho + -screened_rates(k_Ar38_to_n_Ar37)*Y(Ar38)) +
        (-screened_rates(k_n_Ar38_to_Ar39)*Y(Ar38)*Y(N)*state.rho + screened_rates(k_Ar39_to_n_Ar38)*Y(Ar39)) +
        (-screened_rates(k_p_Ar38_to_K39)*Y(Ar38)*Y(H1)*state.rho + screened_rates(k_K39_to_p_Ar38)*Y(K39)) +
        (-screened_rates(k_He4_Ar38_to_Ca42)*Y(Ar38)*Y(He4)*state.rho + screened_rates(k_Ca42_to_He4_Ar38)*Y(Ca42)) +
        (screened_rates(k_He4_S35_to_n_Ar38)*Y(He4)*Y(S35)*state.rho + -screened_rates(k_n_Ar38_to_He4_S35)*Y(Ar38)*Y(N)*state.rho) +
        (screened_rates(k_He4_Cl35_to_p_Ar38)*Y(Cl35)*Y(He4)*state.rho + -screened_rates(k_p_Ar38_to_He4_Cl35)*Y(Ar38)*Y(H1)*state.rho) +
        (screened_rates(k_n_K38_to_p_Ar38)*Y(K38)*Y(N)*state.rho + -screened_rates(k_p_Ar38_to_n_K38)*Y(Ar38)*Y(H1)*state.rho) +
        (screened_rates(k_p_K41_to_He4_Ar38)*Y(K41)*Y(H1)*state.rho + -screened_rates(k_He4_Ar38_to_p_K41)*Y(Ar38)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ca41_to_He4_Ar38)*Y(Ca41)*Y(N)*state.rho + -screened_rates(k_He4_Ar38_to_n_Ca41)*Y(Ar38)*Y(He4)*state.rho) +
        (screened_rates(k_K38_to_Ar38)*Y(K38) + -screened_rates(k_Ar38_to_K38)*Y(Ar38));

    ydot_nuc(Ar39) =
        (screened_rates(k_He4_S35_to_Ar39)*Y(He4)*Y(S35)*state.rho + -screened_rates(k_Ar39_to_He4_S35)*Y(Ar39)) +
        (screened_rates(k_n_Ar38_to_Ar39)*Y(Ar38)*Y(N)*state.rho + -screened_rates(k_Ar39_to_n_Ar38)*Y(Ar39)) +
        (-screened_rates(k_n_Ar39_to_Ar40)*Y(Ar39)*Y(N)*state.rho + screened_rates(k_Ar40_to_n_Ar39)*Y(Ar40)) +
        (-screened_rates(k_p_Ar39_to_K40)*Y(Ar39)*Y(H1)*state.rho + screened_rates(k_K40_to_p_Ar39)*Y(K40)) +
        (-screened_rates(k_He4_Ar39_to_Ca43)*Y(Ar39)*Y(He4)*state.rho + screened_rates(k_Ca43_to_He4_Ar39)*Y(Ca43)) +
        (-screened_rates(k_n_Ar39_to_He4_S36)*Y(Ar39)*Y(N)*state.rho + screened_rates(k_He4_S36_to_n_Ar39)*Y(He4)*Y(S36)*state.rho) +
        (-screened_rates(k_p_Ar39_to_He4_Cl36)*Y(Ar39)*Y(H1)*state.rho + screened_rates(k_He4_Cl36_to_p_Ar39)*Y(Cl36)*Y(He4)*state.rho) +
        (screened_rates(k_n_K39_to_p_Ar39)*Y(K39)*Y(N)*state.rho + -screened_rates(k_p_Ar39_to_n_K39)*Y(Ar39)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ca42_to_He4_Ar39)*Y(Ca42)*Y(N)*state.rho + -screened_rates(k_He4_Ar39_to_n_Ca42)*Y(Ar39)*Y(He4)*state.rho) +
        (-screened_rates(k_Ar39_to_K39)*Y(Ar39) + screened_rates(k_K39_to_Ar39)*Y(K39));

    ydot_nuc(Ar40) =
        (screened_rates(k_He4_S36_to_Ar40)*Y(He4)*Y(S36)*state.rho + -screened_rates(k_Ar40_to_He4_S36)*Y(Ar40)) +
        (screened_rates(k_n_Ar39_to_Ar40)*Y(Ar39)*Y(N)*state.rho + -screened_rates(k_Ar40_to_n_Ar39)*Y(Ar40)) +
        (-screened_rates(k_p_Ar40_to_K41)*Y(Ar40)*Y(H1)*state.rho + screened_rates(k_K41_to_p_Ar40)*Y(K41)) +
        (-screened_rates(k_He4_Ar40_to_Ca44)*Y(Ar40)*Y(He4)*state.rho + screened_rates(k_Ca44_to_He4_Ar40)*Y(Ca44)) +
        (-screened_rates(k_p_Ar40_to_He4_Cl37)*Y(Ar40)*Y(H1)*state.rho + screened_rates(k_He4_Cl37_to_p_Ar40)*Y(Cl37)*Y(He4)*state.rho) +
        (screened_rates(k_n_K40_to_p_Ar40)*Y(K40)*Y(N)*state.rho + -screened_rates(k_p_Ar40_to_n_K40)*Y(Ar40)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ca43_to_He4_Ar40)*Y(Ca43)*Y(N)*state.rho + -screened_rates(k_He4_Ar40_to_n_Ca43)*Y(Ar40)*Y(He4)*state.rho) +
        (screened_rates(k_K40_to_Ar40)*Y(K40) + -screened_rates(k_Ar40_to_K40)*Y(Ar40));

    ydot_nuc(K37) =
        (screened_rates(k_He4_Cl33_to_K37)*Y(Cl33)*Y(He4)*state.rho + -screened_rates(k_K37_to_He4_Cl33)*Y(K37)) +
        (screened_rates(k_p_Ar36_to_K37)*Y(Ar36)*Y(H1)*state.rho + -screened_rates(k_K37_to_p_Ar36)*Y(K37)) +
        (-screened_rates(k_n_K37_to_K38)*Y(K37)*Y(N)*state.rho + screened_rates(k_K38_to_n_K37)*Y(K38)) +
        (-screened_rates(k_n_K37_to_p_Ar37)*Y(K37)*Y(N)*state.rho + screened_rates(k_p_Ar37_to_n_K37)*Y(Ar37)*Y(H1)*state.rho) +
        (-screened_rates(k_n_K37_to_He4_Cl34)*Y(K37)*Y(N)*state.rho + screened_rates(k_He4_Cl34_to_n_K37)*Y(Cl34)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_K37_to_p_Ca40)*Y(He4)*Y(K37)*state.rho + screened_rates(k_p_Ca40_to_He4_K37)*Y(Ca40)*Y(H1)*state.rho) +
        (-screened_rates(k_K37_to_Ar37)*Y(K37) + screened_rates(k_Ar37_to_K37)*Y(Ar37));

    ydot_nuc(K38) =
        (screened_rates(k_He4_Cl34_to_K38)*Y(Cl34)*Y(He4)*state.rho + -screened_rates(k_K38_to_He4_Cl34)*Y(K38)) +
        (screened_rates(k_p_Ar37_to_K38)*Y(Ar37)*Y(H1)*state.rho + -screened_rates(k_K38_to_p_Ar37)*Y(K38)) +
        (screened_rates(k_n_K37_to_K38)*Y(K37)*Y(N)*state.rho + -screened_rates(k_K38_to_n_K37)*Y(K38)) +
        (-screened_rates(k_n_K38_to_K39)*Y(K38)*Y(N)*state.rho + screened_rates(k_K39_to_n_K38)*Y(K39)) +
        (-screened_rates(k_n_K38_to_p_Ar38)*Y(K38)*Y(N)*state.rho + screened_rates(k_p_Ar38_to_n_K38)*Y(Ar38)*Y(H1)*state.rho) +
        (-screened_rates(k_n_K38_to_He4_Cl35)*Y(K38)*Y(N)*state.rho + screened_rates(k_He4_Cl35_to_n_K38)*Y(Cl35)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_K38_to_p_Ca41)*Y(He4)*Y(K38)*state.rho + screened_rates(k_p_Ca41_to_He4_K38)*Y(Ca41)*Y(H1)*state.rho) +
        (-screened_rates(k_K38_to_Ar38)*Y(K38) + screened_rates(k_Ar38_to_K38)*Y(Ar38));

    ydot_nuc(K39) =
        (screened_rates(k_He4_Cl35_to_K39)*Y(Cl35)*Y(He4)*state.rho + -screened_rates(k_K39_to_He4_Cl35)*Y(K39)) +
        (screened_rates(k_p_Ar38_to_K39)*Y(Ar38)*Y(H1)*state.rho + -screened_rates(k_K39_to_p_Ar38)*Y(K39)) +
        (screened_rates(k_n_K38_to_K39)*Y(K38)*Y(N)*state.rho + -screened_rates(k_K39_to_n_K38)*Y(K39)) +
        (-screened_rates(k_n_K39_to_K40)*Y(K39)*Y(N)*state.rho + screened_rates(k_K40_to_n_K39)*Y(K40)) +
        (-screened_rates(k_p_K39_to_Ca40)*Y(K39)*Y(H1)*state.rho + screened_rates(k_Ca40_to_p_K39)*Y(Ca40)) +
        (-screened_rates(k_He4_K39_to_Sc43)*Y(He4)*Y(K39)*state.rho + screened_rates(k_Sc43_to_He4_K39)*Y(Sc43)) +
        (-screened_rates(k_n_K39_to_p_Ar39)*Y(K39)*Y(N)*state.rho + screened_rates(k_p_Ar39_to_n_K39)*Y(Ar39)*Y(H1)*state.rho) +
        (-screened_rates(k_n_K39_to_He4_Cl36)*Y(K39)*Y(N)*state.rho + screened_rates(k_He4_Cl36_to_n_K39)*Y(Cl36)*Y(He4)*state.rho) +
        (-screened_rates(k_p_K39_to_He4_Ar36)*Y(K39)*Y(H1)*state.rho + screened_rates(k_He4_Ar36_to_p_K39)*Y(Ar36)*Y(He4)*state.rho) +
        (screened_rates(k_p_Ca42_to_He4_K39)*Y(Ca42)*Y(H1)*state.rho + -screened_rates(k_He4_K39_to_p_Ca42)*Y(He4)*Y(K39)*state.rho) +
        (screened_rates(k_Ar39_to_K39)*Y(Ar39) + -screened_rates(k_K39_to_Ar39)*Y(K39));

    ydot_nuc(K40) =
        (screened_rates(k_He4_Cl36_to_K40)*Y(Cl36)*Y(He4)*state.rho + -screened_rates(k_K40_to_He4_Cl36)*Y(K40)) +
        (screened_rates(k_p_Ar39_to_K40)*Y(Ar39)*Y(H1)*state.rho + -screened_rates(k_K40_to_p_Ar39)*Y(K40)) +
        (screened_rates(k_n_K39_to_K40)*Y(K39)*Y(N)*state.rho + -screened_rates(k_K40_to_n_K39)*Y(K40)) +
        (-screened_rates(k_n_K40_to_K41)*Y(K40)*Y(N)*state.rho + screened_rates(k_K41_to_n_K40)*Y(K41)) +
        (-screened_rates(k_p_K40_to_Ca41)*Y(K40)*Y(H1)*state.rho + screened_rates(k_Ca41_to_p_K40)*Y(Ca41)) +
        (-screened_rates(k_He4_K40_to_Sc44)*Y(He4)*Y(K40)*state.rho + screened_rates(k_Sc44_to_He4_K40)*Y(Sc44)) +
        (-screened_rates(k_n_K40_to_p_Ar40)*Y(K40)*Y(N)*state.rho + screened_rates(k_p_Ar40_to_n_K40)*Y(Ar40)*Y(H1)*state.rho) +
        (-screened_rates(k_n_K40_to_He4_Cl37)*Y(K40)*Y(N)*state.rho + screened_rates(k_He4_Cl37_to_n_K40)*Y(Cl37)*Y(He4)*state.rho) +
        (-screened_rates(k_p_K40_to_n_Ca40)*Y(K40)*Y(H1)*state.rho + screened_rates(k_n_Ca40_to_p_K40)*Y(Ca40)*Y(N)*state.rho) +
        (-screened_rates(k_p_K40_to_He4_Ar37)*Y(K40)*Y(H1)*state.rho + screened_rates(k_He4_Ar37_to_p_K40)*Y(Ar37)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_K40_to_p_Ca43)*Y(He4)*Y(K40)*state.rho + screened_rates(k_p_Ca43_to_He4_K40)*Y(Ca43)*Y(H1)*state.rho) +
        (screened_rates(k_n_Sc43_to_He4_K40)*Y(Sc43)*Y(N)*state.rho + -screened_rates(k_He4_K40_to_n_Sc43)*Y(He4)*Y(K40)*state.rho) +
        (-screened_rates(k_K40_to_Ar40)*Y(K40) + screened_rates(k_Ar40_to_K40)*Y(Ar40)) +
        (-screened_rates(k_K40_to_Ca40)*Y(K40) + screened_rates(k_Ca40_to_K40)*Y(Ca40));

    ydot_nuc(K41) =
        (screened_rates(k_He4_Cl37_to_K41)*Y(Cl37)*Y(He4)*state.rho + -screened_rates(k_K41_to_He4_Cl37)*Y(K41)) +
        (screened_rates(k_p_Ar40_to_K41)*Y(Ar40)*Y(H1)*state.rho + -screened_rates(k_K41_to_p_Ar40)*Y(K41)) +
        (screened_rates(k_n_K40_to_K41)*Y(K40)*Y(N)*state.rho + -screened_rates(k_K41_to_n_K40)*Y(K41)) +
        (-screened_rates(k_p_K41_to_Ca42)*Y(K41)*Y(H1)*state.rho + screened_rates(k_Ca42_to_p_K41)*Y(Ca42)) +
        (-screened_rates(k_He4_K41_to_Sc45)*Y(He4)*Y(K41)*state.rho + screened_rates(k_Sc45_to_He4_K41)*Y(Sc45)) +
        (-screened_rates(k_p_K41_to_He4_Ar38)*Y(K41)*Y(H1)*state.rho + screened_rates(k_He4_Ar38_to_p_K41)*Y(Ar38)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_K41_to_p_Ca44)*Y(He4)*Y(K41)*state.rho + screened_rates(k_p_Ca44_to_He4_K41)*Y(Ca44)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ca41_to_p_K41)*Y(Ca41)*Y(N)*state.rho + -screened_rates(k_p_K41_to_n_Ca41)*Y(K41)*Y(H1)*state.rho) +
        (screened_rates(k_n_Sc44_to_He4_K41)*Y(Sc44)*Y(N)*state.rho + -screened_rates(k_He4_K41_to_n_Sc44)*Y(He4)*Y(K41)*state.rho) +
        (screened_rates(k_Ca41_to_K41)*Y(Ca41) + -screened_rates(k_K41_to_Ca41)*Y(K41));

    ydot_nuc(Ca40) =
        (screened_rates(k_He4_Ar36_to_Ca40)*Y(Ar36)*Y(He4)*state.rho + -screened_rates(k_Ca40_to_He4_Ar36)*Y(Ca40)) +
        (screened_rates(k_p_K39_to_Ca40)*Y(K39)*Y(H1)*state.rho + -screened_rates(k_Ca40_to_p_K39)*Y(Ca40)) +
        (-screened_rates(k_n_Ca40_to_Ca41)*Y(Ca40)*Y(N)*state.rho + screened_rates(k_Ca41_to_n_Ca40)*Y(Ca41)) +
        (-screened_rates(k_He4_Ca40_to_Ti44)*Y(Ca40)*Y(He4)*state.rho + screened_rates(k_Ti44_to_He4_Ca40)*Y(Ti44)) +
        (screened_rates(k_He4_K37_to_p_Ca40)*Y(He4)*Y(K37)*state.rho + -screened_rates(k_p_Ca40_to_He4_K37)*Y(Ca40)*Y(H1)*state.rho) +
        (screened_rates(k_p_K40_to_n_Ca40)*Y(K40)*Y(H1)*state.rho + -screened_rates(k_n_Ca40_to_p_K40)*Y(Ca40)*Y(N)*state.rho) +
        (-screened_rates(k_n_Ca40_to_He4_Ar37)*Y(Ca40)*Y(N)*state.rho + screened_rates(k_He4_Ar37_to_n_Ca40)*Y(Ar37)*Y(He4)*state.rho) +
        (screened_rates(k_p_Sc43_to_He4_Ca40)*Y(Sc43)*Y(H1)*state.rho + -screened_rates(k_He4_Ca40_to_p_Sc43)*Y(Ca40)*Y(He4)*state.rho) +
        (screened_rates(k_K40_to_Ca40)*Y(K40) + -screened_rates(k_Ca40_to_K40)*Y(Ca40));

    ydot_nuc(Ca41) =
        (screened_rates(k_He4_Ar37_to_Ca41)*Y(Ar37)*Y(He4)*state.rho + -screened_rates(k_Ca41_to_He4_Ar37)*Y(Ca41)) +
        (screened_rates(k_p_K40_to_Ca41)*Y(K40)*Y(H1)*state.rho + -screened_rates(k_Ca41_to_p_K40)*Y(Ca41)) +
        (screened_rates(k_n_Ca40_to_Ca41)*Y(Ca40)*Y(N)*state.rho + -screened_rates(k_Ca41_to_n_Ca40)*Y(Ca41)) +
        (-screened_rates(k_n_Ca41_to_Ca42)*Y(Ca41)*Y(N)*state.rho + screened_rates(k_Ca42_to_n_Ca41)*Y(Ca42)) +
        (-screened_rates(k_He4_Ca41_to_Ti45)*Y(Ca41)*Y(He4)*state.rho + screened_rates(k_Ti45_to_He4_Ca41)*Y(Ti45)) +
        (screened_rates(k_He4_K38_to_p_Ca41)*Y(He4)*Y(K38)*state.rho + -screened_rates(k_p_Ca41_to_He4_K38)*Y(Ca41)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ca41_to_p_K41)*Y(Ca41)*Y(N)*state.rho + screened_rates(k_p_K41_to_n_Ca41)*Y(K41)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ca41_to_He4_Ar38)*Y(Ca41)*Y(N)*state.rho + screened_rates(k_He4_Ar38_to_n_Ca41)*Y(Ar38)*Y(He4)*state.rho) +
        (screened_rates(k_p_Sc44_to_He4_Ca41)*Y(Sc44)*Y(H1)*state.rho + -screened_rates(k_He4_Ca41_to_p_Sc44)*Y(Ca41)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ti44_to_He4_Ca41)*Y(Ti44)*Y(N)*state.rho + -screened_rates(k_He4_Ca41_to_n_Ti44)*Y(Ca41)*Y(He4)*state.rho) +
        (-screened_rates(k_Ca41_to_K41)*Y(Ca41) + screened_rates(k_K41_to_Ca41)*Y(K41));

    ydot_nuc(Ca42) =
        (screened_rates(k_He4_Ar38_to_Ca42)*Y(Ar38)*Y(He4)*state.rho + -screened_rates(k_Ca42_to_He4_Ar38)*Y(Ca42)) +
        (screened_rates(k_p_K41_to_Ca42)*Y(K41)*Y(H1)*state.rho + -screened_rates(k_Ca42_to_p_K41)*Y(Ca42)) +
        (screened_rates(k_n_Ca41_to_Ca42)*Y(Ca41)*Y(N)*state.rho + -screened_rates(k_Ca42_to_n_Ca41)*Y(Ca42)) +
        (-screened_rates(k_n_Ca42_to_Ca43)*Y(Ca42)*Y(N)*state.rho + screened_rates(k_Ca43_to_n_Ca42)*Y(Ca43)) +
        (-screened_rates(k_p_Ca42_to_Sc43)*Y(Ca42)*Y(H1)*state.rho + screened_rates(k_Sc43_to_p_Ca42)*Y(Sc43)) +
        (-screened_rates(k_He4_Ca42_to_Ti46)*Y(Ca42)*Y(He4)*state.rho + screened_rates(k_Ti46_to_He4_Ca42)*Y(Ti46)) +
        (-screened_rates(k_n_Ca42_to_He4_Ar39)*Y(Ca42)*Y(N)*state.rho + screened_rates(k_He4_Ar39_to_n_Ca42)*Y(Ar39)*Y(He4)*state.rho) +
        (-screened_rates(k_p_Ca42_to_He4_K39)*Y(Ca42)*Y(H1)*state.rho + screened_rates(k_He4_K39_to_p_Ca42)*Y(He4)*Y(K39)*state.rho) +
        (screened_rates(k_p_Sc45_to_He4_Ca42)*Y(Sc45)*Y(H1)*state.rho + -screened_rates(k_He4_Ca42_to_p_Sc45)*Y(Ca42)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ti45_to_He4_Ca42)*Y(Ti45)*Y(N)*state.rho + -screened_rates(k_He4_Ca42_to_n_Ti45)*Y(Ca42)*Y(He4)*state.rho);

    ydot_nuc(Ca43) =
        (screened_rates(k_He4_Ar39_to_Ca43)*Y(Ar39)*Y(He4)*state.rho + -screened_rates(k_Ca43_to_He4_Ar39)*Y(Ca43)) +
        (screened_rates(k_n_Ca42_to_Ca43)*Y(Ca42)*Y(N)*state.rho + -screened_rates(k_Ca43_to_n_Ca42)*Y(Ca43)) +
        (-screened_rates(k_n_Ca43_to_Ca44)*Y(Ca43)*Y(N)*state.rho + screened_rates(k_Ca44_to_n_Ca43)*Y(Ca44)) +
        (-screened_rates(k_p_Ca43_to_Sc44)*Y(Ca43)*Y(H1)*state.rho + screened_rates(k_Sc44_to_p_Ca43)*Y(Sc44)) +
        (-screened_rates(k_He4_Ca43_to_Ti47)*Y(Ca43)*Y(He4)*state.rho + screened_rates(k_Ti47_to_He4_Ca43)*Y(Ti47)) +
        (screened_rates(k_He4_K40_to_p_Ca43)*Y(He4)*Y(K40)*state.rho + -screened_rates(k_p_Ca43_to_He4_K40)*Y(Ca43)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ca43_to_He4_Ar40)*Y(Ca43)*Y(N)*state.rho + screened_rates(k_He4_Ar40_to_n_Ca43)*Y(Ar40)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Ca43_to_n_Ti46)*Y(Ca43)*Y(He4)*state.rho + screened_rates(k_n_Ti46_to_He4_Ca43)*Y(Ti46)*Y(N)*state.rho) +
        (screened_rates(k_n_Sc43_to_p_Ca43)*Y(Sc43)*Y(N)*state.rho + -screened_rates(k_p_Ca43_to_n_Sc43)*Y(Ca43)*Y(H1)*state.rho) +
        (screened_rates(k_p_Sc46_to_He4_Ca43)*Y(Sc46)*Y(H1)*state.rho + -screened_rates(k_He4_Ca43_to_p_Sc46)*Y(Ca43)*Y(He4)*state.rho) +
        (screened_rates(k_Sc43_to_Ca43)*Y(Sc43) + -screened_rates(k_Ca43_to_Sc43)*Y(Ca43));

    ydot_nuc(Ca44) =
        (screened_rates(k_He4_Ar40_to_Ca44)*Y(Ar40)*Y(He4)*state.rho + -screened_rates(k_Ca44_to_He4_Ar40)*Y(Ca44)) +
        (screened_rates(k_n_Ca43_to_Ca44)*Y(Ca43)*Y(N)*state.rho + -screened_rates(k_Ca44_to_n_Ca43)*Y(Ca44)) +
        (-screened_rates(k_n_Ca44_to_Ca45)*Y(Ca44)*Y(N)*state.rho + screened_rates(k_Ca45_to_n_Ca44)*Y(Ca45)) +
        (-screened_rates(k_p_Ca44_to_Sc45)*Y(Ca44)*Y(H1)*state.rho + screened_rates(k_Sc45_to_p_Ca44)*Y(Sc45)) +
        (-screened_rates(k_He4_Ca44_to_Ti48)*Y(Ca44)*Y(He4)*state.rho + screened_rates(k_Ti48_to_He4_Ca44)*Y(Ti48)) +
        (screened_rates(k_He4_K41_to_p_Ca44)*Y(He4)*Y(K41)*state.rho + -screened_rates(k_p_Ca44_to_He4_K41)*Y(Ca44)*Y(H1)*state.rho) +
        (screened_rates(k_n_Sc44_to_p_Ca44)*Y(Sc44)*Y(N)*state.rho + -screened_rates(k_p_Ca44_to_n_Sc44)*Y(Ca44)*Y(H1)*state.rho) +
        (screened_rates(k_p_Sc47_to_He4_Ca44)*Y(Sc47)*Y(H1)*state.rho + -screened_rates(k_He4_Ca44_to_p_Sc47)*Y(Ca44)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ti47_to_He4_Ca44)*Y(Ti47)*Y(N)*state.rho + -screened_rates(k_He4_Ca44_to_n_Ti47)*Y(Ca44)*Y(He4)*state.rho) +
        (screened_rates(k_Sc44_to_Ca44)*Y(Sc44) + -screened_rates(k_Ca44_to_Sc44)*Y(Ca44));

    ydot_nuc(Ca45) =
        (screened_rates(k_n_Ca44_to_Ca45)*Y(Ca44)*Y(N)*state.rho + -screened_rates(k_Ca45_to_n_Ca44)*Y(Ca45)) +
        (-screened_rates(k_n_Ca45_to_Ca46)*Y(Ca45)*Y(N)*state.rho + screened_rates(k_Ca46_to_n_Ca45)*Y(Ca46)) +
        (-screened_rates(k_p_Ca45_to_Sc46)*Y(Ca45)*Y(H1)*state.rho + screened_rates(k_Sc46_to_p_Ca45)*Y(Sc46)) +
        (-screened_rates(k_He4_Ca45_to_Ti49)*Y(Ca45)*Y(He4)*state.rho + screened_rates(k_Ti49_to_He4_Ca45)*Y(Ti49)) +
        (-screened_rates(k_He4_Ca45_to_n_Ti48)*Y(Ca45)*Y(He4)*state.rho + screened_rates(k_n_Ti48_to_He4_Ca45)*Y(Ti48)*Y(N)*state.rho) +
        (screened_rates(k_n_Sc45_to_p_Ca45)*Y(Sc45)*Y(N)*state.rho + -screened_rates(k_p_Ca45_to_n_Sc45)*Y(Ca45)*Y(H1)*state.rho) +
        (screened_rates(k_p_Sc48_to_He4_Ca45)*Y(Sc48)*Y(H1)*state.rho + -screened_rates(k_He4_Ca45_to_p_Sc48)*Y(Ca45)*Y(He4)*state.rho) +
        (-screened_rates(k_Ca45_to_Sc45)*Y(Ca45) + screened_rates(k_Sc45_to_Ca45)*Y(Sc45));

    ydot_nuc(Ca46) =
        (screened_rates(k_n_Ca45_to_Ca46)*Y(Ca45)*Y(N)*state.rho + -screened_rates(k_Ca46_to_n_Ca45)*Y(Ca46)) +
        (-screened_rates(k_n_Ca46_to_Ca47)*Y(Ca46)*Y(N)*state.rho + screened_rates(k_Ca47_to_n_Ca46)*Y(Ca47)) +
        (-screened_rates(k_p_Ca46_to_Sc47)*Y(Ca46)*Y(H1)*state.rho + screened_rates(k_Sc47_to_p_Ca46)*Y(Sc47)) +
        (-screened_rates(k_He4_Ca46_to_Ti50)*Y(Ca46)*Y(He4)*state.rho + screened_rates(k_Ti50_to_He4_Ca46)*Y(Ti50)) +
        (screened_rates(k_n_Sc46_to_p_Ca46)*Y(Sc46)*Y(N)*state.rho + -screened_rates(k_p_Ca46_to_n_Sc46)*Y(Ca46)*Y(H1)*state.rho) +
        (screened_rates(k_p_Sc49_to_He4_Ca46)*Y(Sc49)*Y(H1)*state.rho + -screened_rates(k_He4_Ca46_to_p_Sc49)*Y(Ca46)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ti49_to_He4_Ca46)*Y(Ti49)*Y(N)*state.rho + -screened_rates(k_He4_Ca46_to_n_Ti49)*Y(Ca46)*Y(He4)*state.rho) +
        (screened_rates(k_Sc46_to_Ca46)*Y(Sc46) + -screened_rates(k_Ca46_to_Sc46)*Y(Ca46));

    ydot_nuc(Ca47) =
        (screened_rates(k_n_Ca46_to_Ca47)*Y(Ca46)*Y(N)*state.rho + -screened_rates(k_Ca47_to_n_Ca46)*Y(Ca47)) +
        (-screened_rates(k_n_Ca47_to_Ca48)*Y(Ca47)*Y(N)*state.rho + screened_rates(k_Ca48_to_n_Ca47)*Y(Ca48)) +
        (-screened_rates(k_p_Ca47_to_Sc48)*Y(Ca47)*Y(H1)*state.rho + screened_rates(k_Sc48_to_p_Ca47)*Y(Sc48)) +
        (-screened_rates(k_He4_Ca47_to_Ti51)*Y(Ca47)*Y(He4)*state.rho + screened_rates(k_Ti51_to_He4_Ca47)*Y(Ti51)) +
        (-screened_rates(k_p_Ca47_to_n_Sc47)*Y(Ca47)*Y(H1)*state.rho + screened_rates(k_n_Sc47_to_p_Ca47)*Y(Sc47)*Y(N)*state.rho) +
        (-screened_rates(k_He4_Ca47_to_n_Ti50)*Y(Ca47)*Y(He4)*state.rho + screened_rates(k_n_Ti50_to_He4_Ca47)*Y(Ti50)*Y(N)*state.rho) +
        (-screened_rates(k_Ca47_to_Sc47)*Y(Ca47) + screened_rates(k_Sc47_to_Ca47)*Y(Sc47));

    ydot_nuc(Ca48) =
        (screened_rates(k_n_Ca47_to_Ca48)*Y(Ca47)*Y(N)*state.rho + -screened_rates(k_Ca48_to_n_Ca47)*Y(Ca48)) +
        (-screened_rates(k_p_Ca48_to_Sc49)*Y(Ca48)*Y(H1)*state.rho + screened_rates(k_Sc49_to_p_Ca48)*Y(Sc49)) +
        (screened_rates(k_n_Sc48_to_p_Ca48)*Y(Sc48)*Y(N)*state.rho + -screened_rates(k_p_Ca48_to_n_Sc48)*Y(Ca48)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ti51_to_He4_Ca48)*Y(Ti51)*Y(N)*state.rho + -screened_rates(k_He4_Ca48_to_n_Ti51)*Y(Ca48)*Y(He4)*state.rho) +
        (-screened_rates(k_Ca48_to_Sc48)*Y(Ca48) + screened_rates(k_Sc48_to_Ca48)*Y(Sc48));

    ydot_nuc(Sc43) =
        (screened_rates(k_He4_K39_to_Sc43)*Y(He4)*Y(K39)*state.rho + -screened_rates(k_Sc43_to_He4_K39)*Y(Sc43)) +
        (screened_rates(k_p_Ca42_to_Sc43)*Y(Ca42)*Y(H1)*state.rho + -screened_rates(k_Sc43_to_p_Ca42)*Y(Sc43)) +
        (-screened_rates(k_n_Sc43_to_Sc44)*Y(Sc43)*Y(N)*state.rho + screened_rates(k_Sc44_to_n_Sc43)*Y(Sc44)) +
        (-screened_rates(k_p_Sc43_to_Ti44)*Y(Sc43)*Y(H1)*state.rho + screened_rates(k_Ti44_to_p_Sc43)*Y(Ti44)) +
        (-screened_rates(k_He4_Sc43_to_V47)*Y(He4)*Y(Sc43)*state.rho + screened_rates(k_V47_to_He4_Sc43)*Y(V47)) +
        (-screened_rates(k_n_Sc43_to_p_Ca43)*Y(Sc43)*Y(N)*state.rho + screened_rates(k_p_Ca43_to_n_Sc43)*Y(Ca43)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Sc43_to_He4_K40)*Y(Sc43)*Y(N)*state.rho + screened_rates(k_He4_K40_to_n_Sc43)*Y(He4)*Y(K40)*state.rho) +
        (-screened_rates(k_p_Sc43_to_He4_Ca40)*Y(Sc43)*Y(H1)*state.rho + screened_rates(k_He4_Ca40_to_p_Sc43)*Y(Ca40)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Sc43_to_p_Ti46)*Y(He4)*Y(Sc43)*state.rho + screened_rates(k_p_Ti46_to_He4_Sc43)*Y(Ti46)*Y(H1)*state.rho) +
        (screened_rates(k_n_V46_to_He4_Sc43)*Y(V46)*Y(N)*state.rho + -screened_rates(k_He4_Sc43_to_n_V46)*Y(He4)*Y(Sc43)*state.rho) +
        (-screened_rates(k_Sc43_to_Ca43)*Y(Sc43) + screened_rates(k_Ca43_to_Sc43)*Y(Ca43));

    ydot_nuc(Sc44) =
        (screened_rates(k_He4_K40_to_Sc44)*Y(He4)*Y(K40)*state.rho + -screened_rates(k_Sc44_to_He4_K40)*Y(Sc44)) +
        (screened_rates(k_p_Ca43_to_Sc44)*Y(Ca43)*Y(H1)*state.rho + -screened_rates(k_Sc44_to_p_Ca43)*Y(Sc44)) +
        (screened_rates(k_n_Sc43_to_Sc44)*Y(Sc43)*Y(N)*state.rho + -screened_rates(k_Sc44_to_n_Sc43)*Y(Sc44)) +
        (-screened_rates(k_n_Sc44_to_Sc45)*Y(Sc44)*Y(N)*state.rho + screened_rates(k_Sc45_to_n_Sc44)*Y(Sc45)) +
        (-screened_rates(k_p_Sc44_to_Ti45)*Y(Sc44)*Y(H1)*state.rho + screened_rates(k_Ti45_to_p_Sc44)*Y(Ti45)) +
        (-screened_rates(k_He4_Sc44_to_V48)*Y(He4)*Y(Sc44)*state.rho + screened_rates(k_V48_to_He4_Sc44)*Y(V48)) +
        (-screened_rates(k_n_Sc44_to_p_Ca44)*Y(Sc44)*Y(N)*state.rho + screened_rates(k_p_Ca44_to_n_Sc44)*Y(Ca44)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Sc44_to_He4_K41)*Y(Sc44)*Y(N)*state.rho + screened_rates(k_He4_K41_to_n_Sc44)*Y(He4)*Y(K41)*state.rho) +
        (-screened_rates(k_p_Sc44_to_He4_Ca41)*Y(Sc44)*Y(H1)*state.rho + screened_rates(k_He4_Ca41_to_p_Sc44)*Y(Ca41)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Sc44_to_p_Ti47)*Y(He4)*Y(Sc44)*state.rho + screened_rates(k_p_Ti47_to_He4_Sc44)*Y(Ti47)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ti44_to_p_Sc44)*Y(Ti44)*Y(N)*state.rho + -screened_rates(k_p_Sc44_to_n_Ti44)*Y(Sc44)*Y(H1)*state.rho) +
        (screened_rates(k_n_V47_to_He4_Sc44)*Y(V47)*Y(N)*state.rho + -screened_rates(k_He4_Sc44_to_n_V47)*Y(He4)*Y(Sc44)*state.rho) +
        (-screened_rates(k_Sc44_to_Ca44)*Y(Sc44) + screened_rates(k_Ca44_to_Sc44)*Y(Ca44)) +
        (screened_rates(k_Ti44_to_Sc44)*Y(Ti44) + -screened_rates(k_Sc44_to_Ti44)*Y(Sc44));

    ydot_nuc(Sc45) =
        (screened_rates(k_He4_K41_to_Sc45)*Y(He4)*Y(K41)*state.rho + -screened_rates(k_Sc45_to_He4_K41)*Y(Sc45)) +
        (screened_rates(k_p_Ca44_to_Sc45)*Y(Ca44)*Y(H1)*state.rho + -screened_rates(k_Sc45_to_p_Ca44)*Y(Sc45)) +
        (screened_rates(k_n_Sc44_to_Sc45)*Y(Sc44)*Y(N)*state.rho + -screened_rates(k_Sc45_to_n_Sc44)*Y(Sc45)) +
        (-screened_rates(k_n_Sc45_to_Sc46)*Y(Sc45)*Y(N)*state.rho + screened_rates(k_Sc46_to_n_Sc45)*Y(Sc46)) +
        (-screened_rates(k_p_Sc45_to_Ti46)*Y(Sc45)*Y(H1)*state.rho + screened_rates(k_Ti46_to_p_Sc45)*Y(Ti46)) +
        (-screened_rates(k_He4_Sc45_to_V49)*Y(He4)*Y(Sc45)*state.rho + screened_rates(k_V49_to_He4_Sc45)*Y(V49)) +
        (-screened_rates(k_n_Sc45_to_p_Ca45)*Y(Sc45)*Y(N)*state.rho + screened_rates(k_p_Ca45_to_n_Sc45)*Y(Ca45)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Sc45_to_He4_Ca42)*Y(Sc45)*Y(H1)*state.rho + screened_rates(k_He4_Ca42_to_p_Sc45)*Y(Ca42)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Sc45_to_p_Ti48)*Y(He4)*Y(Sc45)*state.rho + screened_rates(k_p_Ti48_to_He4_Sc45)*Y(Ti48)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ti45_to_p_Sc45)*Y(Ti45)*Y(N)*state.rho + -screened_rates(k_p_Sc45_to_n_Ti45)*Y(Sc45)*Y(H1)*state.rho) +
        (screened_rates(k_n_V48_to_He4_Sc45)*Y(V48)*Y(N)*state.rho + -screened_rates(k_He4_Sc45_to_n_V48)*Y(He4)*Y(Sc45)*state.rho) +
        (screened_rates(k_Ca45_to_Sc45)*Y(Ca45) + -screened_rates(k_Sc45_to_Ca45)*Y(Sc45)) +
        (screened_rates(k_Ti45_to_Sc45)*Y(Ti45) + -screened_rates(k_Sc45_to_Ti45)*Y(Sc45));

    ydot_nuc(Sc46) =
        (screened_rates(k_p_Ca45_to_Sc46)*Y(Ca45)*Y(H1)*state.rho + -screened_rates(k_Sc46_to_p_Ca45)*Y(Sc46)) +
        (screened_rates(k_n_Sc45_to_Sc46)*Y(Sc45)*Y(N)*state.rho + -screened_rates(k_Sc46_to_n_Sc45)*Y(Sc46)) +
        (-screened_rates(k_n_Sc46_to_Sc47)*Y(Sc46)*Y(N)*state.rho + screened_rates(k_Sc47_to_n_Sc46)*Y(Sc47)) +
        (-screened_rates(k_p_Sc46_to_Ti47)*Y(Sc46)*Y(H1)*state.rho + screened_rates(k_Ti47_to_p_Sc46)*Y(Ti47)) +
        (-screened_rates(k_He4_Sc46_to_V50)*Y(He4)*Y(Sc46)*state.rho + screened_rates(k_V50_to_He4_Sc46)*Y(V50)) +
        (-screened_rates(k_n_Sc46_to_p_Ca46)*Y(Sc46)*Y(N)*state.rho + screened_rates(k_p_Ca46_to_n_Sc46)*Y(Ca46)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Sc46_to_n_Ti46)*Y(Sc46)*Y(H1)*state.rho + screened_rates(k_n_Ti46_to_p_Sc46)*Y(Ti46)*Y(N)*state.rho) +
        (-screened_rates(k_p_Sc46_to_He4_Ca43)*Y(Sc46)*Y(H1)*state.rho + screened_rates(k_He4_Ca43_to_p_Sc46)*Y(Ca43)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Sc46_to_n_V49)*Y(He4)*Y(Sc46)*state.rho + screened_rates(k_n_V49_to_He4_Sc46)*Y(V49)*Y(N)*state.rho) +
        (-screened_rates(k_He4_Sc46_to_p_Ti49)*Y(He4)*Y(Sc46)*state.rho + screened_rates(k_p_Ti49_to_He4_Sc46)*Y(Ti49)*Y(H1)*state.rho) +
        (-screened_rates(k_Sc46_to_Ca46)*Y(Sc46) + screened_rates(k_Ca46_to_Sc46)*Y(Ca46)) +
        (-screened_rates(k_Sc46_to_Ti46)*Y(Sc46) + screened_rates(k_Ti46_to_Sc46)*Y(Ti46));

    ydot_nuc(Sc47) =
        (screened_rates(k_p_Ca46_to_Sc47)*Y(Ca46)*Y(H1)*state.rho + -screened_rates(k_Sc47_to_p_Ca46)*Y(Sc47)) +
        (screened_rates(k_n_Sc46_to_Sc47)*Y(Sc46)*Y(N)*state.rho + -screened_rates(k_Sc47_to_n_Sc46)*Y(Sc47)) +
        (-screened_rates(k_n_Sc47_to_Sc48)*Y(Sc47)*Y(N)*state.rho + screened_rates(k_Sc48_to_n_Sc47)*Y(Sc48)) +
        (-screened_rates(k_p_Sc47_to_Ti48)*Y(Sc47)*Y(H1)*state.rho + screened_rates(k_Ti48_to_p_Sc47)*Y(Ti48)) +
        (-screened_rates(k_He4_Sc47_to_V51)*Y(He4)*Y(Sc47)*state.rho + screened_rates(k_V51_to_He4_Sc47)*Y(V51)) +
        (screened_rates(k_p_Ca47_to_n_Sc47)*Y(Ca47)*Y(H1)*state.rho + -screened_rates(k_n_Sc47_to_p_Ca47)*Y(Sc47)*Y(N)*state.rho) +
        (-screened_rates(k_p_Sc47_to_He4_Ca44)*Y(Sc47)*Y(H1)*state.rho + screened_rates(k_He4_Ca44_to_p_Sc47)*Y(Ca44)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Sc47_to_p_Ti50)*Y(He4)*Y(Sc47)*state.rho + screened_rates(k_p_Ti50_to_He4_Sc47)*Y(Ti50)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ti47_to_p_Sc47)*Y(Ti47)*Y(N)*state.rho + -screened_rates(k_p_Sc47_to_n_Ti47)*Y(Sc47)*Y(H1)*state.rho) +
        (screened_rates(k_n_V50_to_He4_Sc47)*Y(V50)*Y(N)*state.rho + -screened_rates(k_He4_Sc47_to_n_V50)*Y(He4)*Y(Sc47)*state.rho) +
        (screened_rates(k_Ca47_to_Sc47)*Y(Ca47) + -screened_rates(k_Sc47_to_Ca47)*Y(Sc47)) +
        (-screened_rates(k_Sc47_to_Ti47)*Y(Sc47) + screened_rates(k_Ti47_to_Sc47)*Y(Ti47));

    ydot_nuc(Sc48) =
        (screened_rates(k_p_Ca47_to_Sc48)*Y(Ca47)*Y(H1)*state.rho + -screened_rates(k_Sc48_to_p_Ca47)*Y(Sc48)) +
        (screened_rates(k_n_Sc47_to_Sc48)*Y(Sc47)*Y(N)*state.rho + -screened_rates(k_Sc48_to_n_Sc47)*Y(Sc48)) +
        (-screened_rates(k_n_Sc48_to_Sc49)*Y(Sc48)*Y(N)*state.rho + screened_rates(k_Sc49_to_n_Sc48)*Y(Sc49)) +
        (-screened_rates(k_p_Sc48_to_Ti49)*Y(Sc48)*Y(H1)*state.rho + screened_rates(k_Ti49_to_p_Sc48)*Y(Ti49)) +
        (-screened_rates(k_He4_Sc48_to_V52)*Y(He4)*Y(Sc48)*state.rho + screened_rates(k_V52_to_He4_Sc48)*Y(V52)) +
        (-screened_rates(k_n_Sc48_to_p_Ca48)*Y(Sc48)*Y(N)*state.rho + screened_rates(k_p_Ca48_to_n_Sc48)*Y(Ca48)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Sc48_to_n_Ti48)*Y(Sc48)*Y(H1)*state.rho + screened_rates(k_n_Ti48_to_p_Sc48)*Y(Ti48)*Y(N)*state.rho) +
        (-screened_rates(k_p_Sc48_to_He4_Ca45)*Y(Sc48)*Y(H1)*state.rho + screened_rates(k_He4_Ca45_to_p_Sc48)*Y(Ca45)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Sc48_to_n_V51)*Y(He4)*Y(Sc48)*state.rho + screened_rates(k_n_V51_to_He4_Sc48)*Y(V51)*Y(N)*state.rho) +
        (-screened_rates(k_He4_Sc48_to_p_Ti51)*Y(He4)*Y(Sc48)*state.rho + screened_rates(k_p_Ti51_to_He4_Sc48)*Y(Ti51)*Y(H1)*state.rho) +
        (screened_rates(k_Ca48_to_Sc48)*Y(Ca48) + -screened_rates(k_Sc48_to_Ca48)*Y(Sc48)) +
        (-screened_rates(k_Sc48_to_Ti48)*Y(Sc48) + screened_rates(k_Ti48_to_Sc48)*Y(Ti48));

    ydot_nuc(Sc49) =
        (screened_rates(k_p_Ca48_to_Sc49)*Y(Ca48)*Y(H1)*state.rho + -screened_rates(k_Sc49_to_p_Ca48)*Y(Sc49)) +
        (screened_rates(k_n_Sc48_to_Sc49)*Y(Sc48)*Y(N)*state.rho + -screened_rates(k_Sc49_to_n_Sc48)*Y(Sc49)) +
        (-screened_rates(k_p_Sc49_to_Ti50)*Y(Sc49)*Y(H1)*state.rho + screened_rates(k_Ti50_to_p_Sc49)*Y(Ti50)) +
        (-screened_rates(k_p_Sc49_to_n_Ti49)*Y(Sc49)*Y(H1)*state.rho + screened_rates(k_n_Ti49_to_p_Sc49)*Y(Ti49)*Y(N)*state.rho) +
        (-screened_rates(k_p_Sc49_to_He4_Ca46)*Y(Sc49)*Y(H1)*state.rho + screened_rates(k_He4_Ca46_to_p_Sc49)*Y(Ca46)*Y(He4)*state.rho) +
        (screened_rates(k_n_V52_to_He4_Sc49)*Y(V52)*Y(N)*state.rho + -screened_rates(k_He4_Sc49_to_n_V52)*Y(He4)*Y(Sc49)*state.rho) +
        (-screened_rates(k_Sc49_to_Ti49)*Y(Sc49) + screened_rates(k_Ti49_to_Sc49)*Y(Ti49));

    ydot_nuc(Ti44) =
        (screened_rates(k_He4_Ca40_to_Ti44)*Y(Ca40)*Y(He4)*state.rho + -screened_rates(k_Ti44_to_He4_Ca40)*Y(Ti44)) +
        (screened_rates(k_p_Sc43_to_Ti44)*Y(Sc43)*Y(H1)*state.rho + -screened_rates(k_Ti44_to_p_Sc43)*Y(Ti44)) +
        (-screened_rates(k_n_Ti44_to_Ti45)*Y(Ti44)*Y(N)*state.rho + screened_rates(k_Ti45_to_n_Ti44)*Y(Ti45)) +
        (-screened_rates(k_He4_Ti44_to_Cr48)*Y(He4)*Y(Ti44)*state.rho + screened_rates(k_Cr48_to_He4_Ti44)*Y(Cr48)) +
        (-screened_rates(k_n_Ti44_to_p_Sc44)*Y(Ti44)*Y(N)*state.rho + screened_rates(k_p_Sc44_to_n_Ti44)*Y(Sc44)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ti44_to_He4_Ca41)*Y(Ti44)*Y(N)*state.rho + screened_rates(k_He4_Ca41_to_n_Ti44)*Y(Ca41)*Y(He4)*state.rho) +
        (screened_rates(k_p_V47_to_He4_Ti44)*Y(V47)*Y(H1)*state.rho + -screened_rates(k_He4_Ti44_to_p_V47)*Y(He4)*Y(Ti44)*state.rho) +
        (-screened_rates(k_Ti44_to_Sc44)*Y(Ti44) + screened_rates(k_Sc44_to_Ti44)*Y(Sc44));

    ydot_nuc(Ti45) =
        (screened_rates(k_He4_Ca41_to_Ti45)*Y(Ca41)*Y(He4)*state.rho + -screened_rates(k_Ti45_to_He4_Ca41)*Y(Ti45)) +
        (screened_rates(k_p_Sc44_to_Ti45)*Y(Sc44)*Y(H1)*state.rho + -screened_rates(k_Ti45_to_p_Sc44)*Y(Ti45)) +
        (screened_rates(k_n_Ti44_to_Ti45)*Y(Ti44)*Y(N)*state.rho + -screened_rates(k_Ti45_to_n_Ti44)*Y(Ti45)) +
        (-screened_rates(k_n_Ti45_to_Ti46)*Y(Ti45)*Y(N)*state.rho + screened_rates(k_Ti46_to_n_Ti45)*Y(Ti46)) +
        (-screened_rates(k_p_Ti45_to_V46)*Y(Ti45)*Y(H1)*state.rho + screened_rates(k_V46_to_p_Ti45)*Y(V46)) +
        (-screened_rates(k_He4_Ti45_to_Cr49)*Y(He4)*Y(Ti45)*state.rho + screened_rates(k_Cr49_to_He4_Ti45)*Y(Cr49)) +
        (-screened_rates(k_n_Ti45_to_p_Sc45)*Y(Ti45)*Y(N)*state.rho + screened_rates(k_p_Sc45_to_n_Ti45)*Y(Sc45)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ti45_to_He4_Ca42)*Y(Ti45)*Y(N)*state.rho + screened_rates(k_He4_Ca42_to_n_Ti45)*Y(Ca42)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Ti45_to_p_V48)*Y(He4)*Y(Ti45)*state.rho + screened_rates(k_p_V48_to_He4_Ti45)*Y(V48)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cr48_to_He4_Ti45)*Y(Cr48)*Y(N)*state.rho + -screened_rates(k_He4_Ti45_to_n_Cr48)*Y(He4)*Y(Ti45)*state.rho) +
        (-screened_rates(k_Ti45_to_Sc45)*Y(Ti45) + screened_rates(k_Sc45_to_Ti45)*Y(Sc45));

    ydot_nuc(Ti46) =
        (screened_rates(k_He4_Ca42_to_Ti46)*Y(Ca42)*Y(He4)*state.rho + -screened_rates(k_Ti46_to_He4_Ca42)*Y(Ti46)) +
        (screened_rates(k_p_Sc45_to_Ti46)*Y(Sc45)*Y(H1)*state.rho + -screened_rates(k_Ti46_to_p_Sc45)*Y(Ti46)) +
        (screened_rates(k_n_Ti45_to_Ti46)*Y(Ti45)*Y(N)*state.rho + -screened_rates(k_Ti46_to_n_Ti45)*Y(Ti46)) +
        (-screened_rates(k_n_Ti46_to_Ti47)*Y(Ti46)*Y(N)*state.rho + screened_rates(k_Ti47_to_n_Ti46)*Y(Ti47)) +
        (-screened_rates(k_p_Ti46_to_V47)*Y(Ti46)*Y(H1)*state.rho + screened_rates(k_V47_to_p_Ti46)*Y(V47)) +
        (-screened_rates(k_He4_Ti46_to_Cr50)*Y(He4)*Y(Ti46)*state.rho + screened_rates(k_Cr50_to_He4_Ti46)*Y(Cr50)) +
        (screened_rates(k_He4_Ca43_to_n_Ti46)*Y(Ca43)*Y(He4)*state.rho + -screened_rates(k_n_Ti46_to_He4_Ca43)*Y(Ti46)*Y(N)*state.rho) +
        (screened_rates(k_He4_Sc43_to_p_Ti46)*Y(He4)*Y(Sc43)*state.rho + -screened_rates(k_p_Ti46_to_He4_Sc43)*Y(Ti46)*Y(H1)*state.rho) +
        (screened_rates(k_p_Sc46_to_n_Ti46)*Y(Sc46)*Y(H1)*state.rho + -screened_rates(k_n_Ti46_to_p_Sc46)*Y(Ti46)*Y(N)*state.rho) +
        (screened_rates(k_n_V46_to_p_Ti46)*Y(V46)*Y(N)*state.rho + -screened_rates(k_p_Ti46_to_n_V46)*Y(Ti46)*Y(H1)*state.rho) +
        (screened_rates(k_p_V49_to_He4_Ti46)*Y(V49)*Y(H1)*state.rho + -screened_rates(k_He4_Ti46_to_p_V49)*Y(He4)*Y(Ti46)*state.rho) +
        (screened_rates(k_n_Cr49_to_He4_Ti46)*Y(Cr49)*Y(N)*state.rho + -screened_rates(k_He4_Ti46_to_n_Cr49)*Y(He4)*Y(Ti46)*state.rho) +
        (screened_rates(k_Sc46_to_Ti46)*Y(Sc46) + -screened_rates(k_Ti46_to_Sc46)*Y(Ti46)) +
        (screened_rates(k_V46_to_Ti46)*Y(V46) + -screened_rates(k_Ti46_to_V46)*Y(Ti46));

    ydot_nuc(Ti47) =
        (screened_rates(k_He4_Ca43_to_Ti47)*Y(Ca43)*Y(He4)*state.rho + -screened_rates(k_Ti47_to_He4_Ca43)*Y(Ti47)) +
        (screened_rates(k_p_Sc46_to_Ti47)*Y(Sc46)*Y(H1)*state.rho + -screened_rates(k_Ti47_to_p_Sc46)*Y(Ti47)) +
        (screened_rates(k_n_Ti46_to_Ti47)*Y(Ti46)*Y(N)*state.rho + -screened_rates(k_Ti47_to_n_Ti46)*Y(Ti47)) +
        (-screened_rates(k_n_Ti47_to_Ti48)*Y(Ti47)*Y(N)*state.rho + screened_rates(k_Ti48_to_n_Ti47)*Y(Ti48)) +
        (-screened_rates(k_p_Ti47_to_V48)*Y(Ti47)*Y(H1)*state.rho + screened_rates(k_V48_to_p_Ti47)*Y(V48)) +
        (-screened_rates(k_He4_Ti47_to_Cr51)*Y(He4)*Y(Ti47)*state.rho + screened_rates(k_Cr51_to_He4_Ti47)*Y(Cr51)) +
        (screened_rates(k_He4_Sc44_to_p_Ti47)*Y(He4)*Y(Sc44)*state.rho + -screened_rates(k_p_Ti47_to_He4_Sc44)*Y(Ti47)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ti47_to_p_Sc47)*Y(Ti47)*Y(N)*state.rho + screened_rates(k_p_Sc47_to_n_Ti47)*Y(Sc47)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ti47_to_He4_Ca44)*Y(Ti47)*Y(N)*state.rho + screened_rates(k_He4_Ca44_to_n_Ti47)*Y(Ca44)*Y(He4)*state.rho) +
        (screened_rates(k_n_V47_to_p_Ti47)*Y(V47)*Y(N)*state.rho + -screened_rates(k_p_Ti47_to_n_V47)*Y(Ti47)*Y(H1)*state.rho) +
        (screened_rates(k_p_V50_to_He4_Ti47)*Y(V50)*Y(H1)*state.rho + -screened_rates(k_He4_Ti47_to_p_V50)*Y(He4)*Y(Ti47)*state.rho) +
        (screened_rates(k_n_Cr50_to_He4_Ti47)*Y(Cr50)*Y(N)*state.rho + -screened_rates(k_He4_Ti47_to_n_Cr50)*Y(He4)*Y(Ti47)*state.rho) +
        (screened_rates(k_Sc47_to_Ti47)*Y(Sc47) + -screened_rates(k_Ti47_to_Sc47)*Y(Ti47)) +
        (screened_rates(k_V47_to_Ti47)*Y(V47) + -screened_rates(k_Ti47_to_V47)*Y(Ti47));

    ydot_nuc(Ti48) =
        (screened_rates(k_He4_Ca44_to_Ti48)*Y(Ca44)*Y(He4)*state.rho + -screened_rates(k_Ti48_to_He4_Ca44)*Y(Ti48)) +
        (screened_rates(k_p_Sc47_to_Ti48)*Y(Sc47)*Y(H1)*state.rho + -screened_rates(k_Ti48_to_p_Sc47)*Y(Ti48)) +
        (screened_rates(k_n_Ti47_to_Ti48)*Y(Ti47)*Y(N)*state.rho + -screened_rates(k_Ti48_to_n_Ti47)*Y(Ti48)) +
        (-screened_rates(k_n_Ti48_to_Ti49)*Y(Ti48)*Y(N)*state.rho + screened_rates(k_Ti49_to_n_Ti48)*Y(Ti49)) +
        (-screened_rates(k_p_Ti48_to_V49)*Y(Ti48)*Y(H1)*state.rho + screened_rates(k_V49_to_p_Ti48)*Y(V49)) +
        (-screened_rates(k_He4_Ti48_to_Cr52)*Y(He4)*Y(Ti48)*state.rho + screened_rates(k_Cr52_to_He4_Ti48)*Y(Cr52)) +
        (screened_rates(k_He4_Ca45_to_n_Ti48)*Y(Ca45)*Y(He4)*state.rho + -screened_rates(k_n_Ti48_to_He4_Ca45)*Y(Ti48)*Y(N)*state.rho) +
        (screened_rates(k_He4_Sc45_to_p_Ti48)*Y(He4)*Y(Sc45)*state.rho + -screened_rates(k_p_Ti48_to_He4_Sc45)*Y(Ti48)*Y(H1)*state.rho) +
        (screened_rates(k_p_Sc48_to_n_Ti48)*Y(Sc48)*Y(H1)*state.rho + -screened_rates(k_n_Ti48_to_p_Sc48)*Y(Ti48)*Y(N)*state.rho) +
        (screened_rates(k_n_V48_to_p_Ti48)*Y(V48)*Y(N)*state.rho + -screened_rates(k_p_Ti48_to_n_V48)*Y(Ti48)*Y(H1)*state.rho) +
        (screened_rates(k_p_V51_to_He4_Ti48)*Y(V51)*Y(H1)*state.rho + -screened_rates(k_He4_Ti48_to_p_V51)*Y(He4)*Y(Ti48)*state.rho) +
        (screened_rates(k_n_Cr51_to_He4_Ti48)*Y(Cr51)*Y(N)*state.rho + -screened_rates(k_He4_Ti48_to_n_Cr51)*Y(He4)*Y(Ti48)*state.rho) +
        (screened_rates(k_Sc48_to_Ti48)*Y(Sc48) + -screened_rates(k_Ti48_to_Sc48)*Y(Ti48)) +
        (screened_rates(k_V48_to_Ti48)*Y(V48) + -screened_rates(k_Ti48_to_V48)*Y(Ti48));

    ydot_nuc(Ti49) =
        (screened_rates(k_He4_Ca45_to_Ti49)*Y(Ca45)*Y(He4)*state.rho + -screened_rates(k_Ti49_to_He4_Ca45)*Y(Ti49)) +
        (screened_rates(k_p_Sc48_to_Ti49)*Y(Sc48)*Y(H1)*state.rho + -screened_rates(k_Ti49_to_p_Sc48)*Y(Ti49)) +
        (screened_rates(k_n_Ti48_to_Ti49)*Y(Ti48)*Y(N)*state.rho + -screened_rates(k_Ti49_to_n_Ti48)*Y(Ti49)) +
        (-screened_rates(k_n_Ti49_to_Ti50)*Y(Ti49)*Y(N)*state.rho + screened_rates(k_Ti50_to_n_Ti49)*Y(Ti50)) +
        (-screened_rates(k_p_Ti49_to_V50)*Y(Ti49)*Y(H1)*state.rho + screened_rates(k_V50_to_p_Ti49)*Y(V50)) +
        (-screened_rates(k_He4_Ti49_to_Cr53)*Y(He4)*Y(Ti49)*state.rho + screened_rates(k_Cr53_to_He4_Ti49)*Y(Cr53)) +
        (screened_rates(k_He4_Sc46_to_p_Ti49)*Y(He4)*Y(Sc46)*state.rho + -screened_rates(k_p_Ti49_to_He4_Sc46)*Y(Ti49)*Y(H1)*state.rho) +
        (screened_rates(k_p_Sc49_to_n_Ti49)*Y(Sc49)*Y(H1)*state.rho + -screened_rates(k_n_Ti49_to_p_Sc49)*Y(Ti49)*Y(N)*state.rho) +
        (-screened_rates(k_n_Ti49_to_He4_Ca46)*Y(Ti49)*Y(N)*state.rho + screened_rates(k_He4_Ca46_to_n_Ti49)*Y(Ca46)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Ti49_to_n_Cr52)*Y(He4)*Y(Ti49)*state.rho + screened_rates(k_n_Cr52_to_He4_Ti49)*Y(Cr52)*Y(N)*state.rho) +
        (screened_rates(k_n_V49_to_p_Ti49)*Y(V49)*Y(N)*state.rho + -screened_rates(k_p_Ti49_to_n_V49)*Y(Ti49)*Y(H1)*state.rho) +
        (screened_rates(k_p_V52_to_He4_Ti49)*Y(V52)*Y(H1)*state.rho + -screened_rates(k_He4_Ti49_to_p_V52)*Y(He4)*Y(Ti49)*state.rho) +
        (screened_rates(k_Sc49_to_Ti49)*Y(Sc49) + -screened_rates(k_Ti49_to_Sc49)*Y(Ti49)) +
        (screened_rates(k_V49_to_Ti49)*Y(V49) + -screened_rates(k_Ti49_to_V49)*Y(Ti49));

    ydot_nuc(Ti50) =
        (screened_rates(k_He4_Ca46_to_Ti50)*Y(Ca46)*Y(He4)*state.rho + -screened_rates(k_Ti50_to_He4_Ca46)*Y(Ti50)) +
        (screened_rates(k_p_Sc49_to_Ti50)*Y(Sc49)*Y(H1)*state.rho + -screened_rates(k_Ti50_to_p_Sc49)*Y(Ti50)) +
        (screened_rates(k_n_Ti49_to_Ti50)*Y(Ti49)*Y(N)*state.rho + -screened_rates(k_Ti50_to_n_Ti49)*Y(Ti50)) +
        (-screened_rates(k_n_Ti50_to_Ti51)*Y(Ti50)*Y(N)*state.rho + screened_rates(k_Ti51_to_n_Ti50)*Y(Ti51)) +
        (-screened_rates(k_p_Ti50_to_V51)*Y(Ti50)*Y(H1)*state.rho + screened_rates(k_V51_to_p_Ti50)*Y(V51)) +
        (-screened_rates(k_He4_Ti50_to_Cr54)*Y(He4)*Y(Ti50)*state.rho + screened_rates(k_Cr54_to_He4_Ti50)*Y(Cr54)) +
        (screened_rates(k_He4_Ca47_to_n_Ti50)*Y(Ca47)*Y(He4)*state.rho + -screened_rates(k_n_Ti50_to_He4_Ca47)*Y(Ti50)*Y(N)*state.rho) +
        (screened_rates(k_He4_Sc47_to_p_Ti50)*Y(He4)*Y(Sc47)*state.rho + -screened_rates(k_p_Ti50_to_He4_Sc47)*Y(Ti50)*Y(H1)*state.rho) +
        (screened_rates(k_n_V50_to_p_Ti50)*Y(V50)*Y(N)*state.rho + -screened_rates(k_p_Ti50_to_n_V50)*Y(Ti50)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cr53_to_He4_Ti50)*Y(Cr53)*Y(N)*state.rho + -screened_rates(k_He4_Ti50_to_n_Cr53)*Y(He4)*Y(Ti50)*state.rho) +
        (screened_rates(k_V50_to_Ti50)*Y(V50) + -screened_rates(k_Ti50_to_V50)*Y(Ti50));

    ydot_nuc(Ti51) =
        (screened_rates(k_He4_Ca47_to_Ti51)*Y(Ca47)*Y(He4)*state.rho + -screened_rates(k_Ti51_to_He4_Ca47)*Y(Ti51)) +
        (screened_rates(k_n_Ti50_to_Ti51)*Y(Ti50)*Y(N)*state.rho + -screened_rates(k_Ti51_to_n_Ti50)*Y(Ti51)) +
        (-screened_rates(k_p_Ti51_to_V52)*Y(Ti51)*Y(H1)*state.rho + screened_rates(k_V52_to_p_Ti51)*Y(V52)) +
        (screened_rates(k_He4_Sc48_to_p_Ti51)*Y(He4)*Y(Sc48)*state.rho + -screened_rates(k_p_Ti51_to_He4_Sc48)*Y(Ti51)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ti51_to_He4_Ca48)*Y(Ti51)*Y(N)*state.rho + screened_rates(k_He4_Ca48_to_n_Ti51)*Y(Ca48)*Y(He4)*state.rho) +
        (-screened_rates(k_p_Ti51_to_n_V51)*Y(Ti51)*Y(H1)*state.rho + screened_rates(k_n_V51_to_p_Ti51)*Y(V51)*Y(N)*state.rho) +
        (-screened_rates(k_He4_Ti51_to_n_Cr54)*Y(He4)*Y(Ti51)*state.rho + screened_rates(k_n_Cr54_to_He4_Ti51)*Y(Cr54)*Y(N)*state.rho) +
        (-screened_rates(k_Ti51_to_V51)*Y(Ti51) + screened_rates(k_V51_to_Ti51)*Y(V51));

    ydot_nuc(V46) =
        (screened_rates(k_p_Ti45_to_V46)*Y(Ti45)*Y(H1)*state.rho + -screened_rates(k_V46_to_p_Ti45)*Y(V46)) +
        (-screened_rates(k_n_V46_to_V47)*Y(V46)*Y(N)*state.rho + screened_rates(k_V47_to_n_V46)*Y(V47)) +
        (-screened_rates(k_He4_V46_to_Mn50)*Y(He4)*Y(V46)*state.rho + screened_rates(k_Mn50_to_He4_V46)*Y(Mn50)) +
        (-screened_rates(k_n_V46_to_p_Ti46)*Y(V46)*Y(N)*state.rho + screened_rates(k_p_Ti46_to_n_V46)*Y(Ti46)*Y(H1)*state.rho) +
        (-screened_rates(k_n_V46_to_He4_Sc43)*Y(V46)*Y(N)*state.rho + screened_rates(k_He4_Sc43_to_n_V46)*Y(He4)*Y(Sc43)*state.rho) +
        (-screened_rates(k_He4_V46_to_p_Cr49)*Y(He4)*Y(V46)*state.rho + screened_rates(k_p_Cr49_to_He4_V46)*Y(Cr49)*Y(H1)*state.rho) +
        (-screened_rates(k_V46_to_Ti46)*Y(V46) + screened_rates(k_Ti46_to_V46)*Y(Ti46));

    ydot_nuc(V47) =
        (screened_rates(k_He4_Sc43_to_V47)*Y(He4)*Y(Sc43)*state.rho + -screened_rates(k_V47_to_He4_Sc43)*Y(V47)) +
        (screened_rates(k_p_Ti46_to_V47)*Y(Ti46)*Y(H1)*state.rho + -screened_rates(k_V47_to_p_Ti46)*Y(V47)) +
        (screened_rates(k_n_V46_to_V47)*Y(V46)*Y(N)*state.rho + -screened_rates(k_V47_to_n_V46)*Y(V47)) +
        (-screened_rates(k_n_V47_to_V48)*Y(V47)*Y(N)*state.rho + screened_rates(k_V48_to_n_V47)*Y(V48)) +
        (-screened_rates(k_p_V47_to_Cr48)*Y(V47)*Y(H1)*state.rho + screened_rates(k_Cr48_to_p_V47)*Y(Cr48)) +
        (-screened_rates(k_He4_V47_to_Mn51)*Y(He4)*Y(V47)*state.rho + screened_rates(k_Mn51_to_He4_V47)*Y(Mn51)) +
        (-screened_rates(k_n_V47_to_p_Ti47)*Y(V47)*Y(N)*state.rho + screened_rates(k_p_Ti47_to_n_V47)*Y(Ti47)*Y(H1)*state.rho) +
        (-screened_rates(k_n_V47_to_He4_Sc44)*Y(V47)*Y(N)*state.rho + screened_rates(k_He4_Sc44_to_n_V47)*Y(He4)*Y(Sc44)*state.rho) +
        (-screened_rates(k_p_V47_to_He4_Ti44)*Y(V47)*Y(H1)*state.rho + screened_rates(k_He4_Ti44_to_p_V47)*Y(He4)*Y(Ti44)*state.rho) +
        (-screened_rates(k_He4_V47_to_p_Cr50)*Y(He4)*Y(V47)*state.rho + screened_rates(k_p_Cr50_to_He4_V47)*Y(Cr50)*Y(H1)*state.rho) +
        (screened_rates(k_n_Mn50_to_He4_V47)*Y(Mn50)*Y(N)*state.rho + -screened_rates(k_He4_V47_to_n_Mn50)*Y(He4)*Y(V47)*state.rho) +
        (-screened_rates(k_V47_to_Ti47)*Y(V47) + screened_rates(k_Ti47_to_V47)*Y(Ti47));

    ydot_nuc(V48) =
        (screened_rates(k_He4_Sc44_to_V48)*Y(He4)*Y(Sc44)*state.rho + -screened_rates(k_V48_to_He4_Sc44)*Y(V48)) +
        (screened_rates(k_p_Ti47_to_V48)*Y(Ti47)*Y(H1)*state.rho + -screened_rates(k_V48_to_p_Ti47)*Y(V48)) +
        (screened_rates(k_n_V47_to_V48)*Y(V47)*Y(N)*state.rho + -screened_rates(k_V48_to_n_V47)*Y(V48)) +
        (-screened_rates(k_n_V48_to_V49)*Y(V48)*Y(N)*state.rho + screened_rates(k_V49_to_n_V48)*Y(V49)) +
        (-screened_rates(k_p_V48_to_Cr49)*Y(V48)*Y(H1)*state.rho + screened_rates(k_Cr49_to_p_V48)*Y(Cr49)) +
        (-screened_rates(k_He4_V48_to_Mn52)*Y(He4)*Y(V48)*state.rho + screened_rates(k_Mn52_to_He4_V48)*Y(Mn52)) +
        (screened_rates(k_He4_Ti45_to_p_V48)*Y(He4)*Y(Ti45)*state.rho + -screened_rates(k_p_V48_to_He4_Ti45)*Y(V48)*Y(H1)*state.rho) +
        (-screened_rates(k_n_V48_to_p_Ti48)*Y(V48)*Y(N)*state.rho + screened_rates(k_p_Ti48_to_n_V48)*Y(Ti48)*Y(H1)*state.rho) +
        (-screened_rates(k_n_V48_to_He4_Sc45)*Y(V48)*Y(N)*state.rho + screened_rates(k_He4_Sc45_to_n_V48)*Y(He4)*Y(Sc45)*state.rho) +
        (-screened_rates(k_He4_V48_to_p_Cr51)*Y(He4)*Y(V48)*state.rho + screened_rates(k_p_Cr51_to_He4_V48)*Y(Cr51)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cr48_to_p_V48)*Y(Cr48)*Y(N)*state.rho + -screened_rates(k_p_V48_to_n_Cr48)*Y(V48)*Y(H1)*state.rho) +
        (screened_rates(k_n_Mn51_to_He4_V48)*Y(Mn51)*Y(N)*state.rho + -screened_rates(k_He4_V48_to_n_Mn51)*Y(He4)*Y(V48)*state.rho) +
        (screened_rates(k_Cr48_to_V48)*Y(Cr48) + -screened_rates(k_V48_to_Cr48)*Y(V48)) +
        (-screened_rates(k_V48_to_Ti48)*Y(V48) + screened_rates(k_Ti48_to_V48)*Y(Ti48));

    ydot_nuc(V49) =
        (screened_rates(k_He4_Sc45_to_V49)*Y(He4)*Y(Sc45)*state.rho + -screened_rates(k_V49_to_He4_Sc45)*Y(V49)) +
        (screened_rates(k_p_Ti48_to_V49)*Y(Ti48)*Y(H1)*state.rho + -screened_rates(k_V49_to_p_Ti48)*Y(V49)) +
        (screened_rates(k_n_V48_to_V49)*Y(V48)*Y(N)*state.rho + -screened_rates(k_V49_to_n_V48)*Y(V49)) +
        (-screened_rates(k_n_V49_to_V50)*Y(V49)*Y(N)*state.rho + screened_rates(k_V50_to_n_V49)*Y(V50)) +
        (-screened_rates(k_p_V49_to_Cr50)*Y(V49)*Y(H1)*state.rho + screened_rates(k_Cr50_to_p_V49)*Y(Cr50)) +
        (-screened_rates(k_He4_V49_to_Mn53)*Y(He4)*Y(V49)*state.rho + screened_rates(k_Mn53_to_He4_V49)*Y(Mn53)) +
        (screened_rates(k_He4_Sc46_to_n_V49)*Y(He4)*Y(Sc46)*state.rho + -screened_rates(k_n_V49_to_He4_Sc46)*Y(V49)*Y(N)*state.rho) +
        (-screened_rates(k_n_V49_to_p_Ti49)*Y(V49)*Y(N)*state.rho + screened_rates(k_p_Ti49_to_n_V49)*Y(Ti49)*Y(H1)*state.rho) +
        (-screened_rates(k_p_V49_to_He4_Ti46)*Y(V49)*Y(H1)*state.rho + screened_rates(k_He4_Ti46_to_p_V49)*Y(He4)*Y(Ti46)*state.rho) +
        (-screened_rates(k_He4_V49_to_p_Cr52)*Y(He4)*Y(V49)*state.rho + screened_rates(k_p_Cr52_to_He4_V49)*Y(Cr52)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cr49_to_p_V49)*Y(Cr49)*Y(N)*state.rho + -screened_rates(k_p_V49_to_n_Cr49)*Y(V49)*Y(H1)*state.rho) +
        (screened_rates(k_n_Mn52_to_He4_V49)*Y(Mn52)*Y(N)*state.rho + -screened_rates(k_He4_V49_to_n_Mn52)*Y(He4)*Y(V49)*state.rho) +
        (screened_rates(k_Cr49_to_V49)*Y(Cr49) + -screened_rates(k_V49_to_Cr49)*Y(V49)) +
        (-screened_rates(k_V49_to_Ti49)*Y(V49) + screened_rates(k_Ti49_to_V49)*Y(Ti49));

    ydot_nuc(V50) =
        (screened_rates(k_He4_Sc46_to_V50)*Y(He4)*Y(Sc46)*state.rho + -screened_rates(k_V50_to_He4_Sc46)*Y(V50)) +
        (screened_rates(k_p_Ti49_to_V50)*Y(Ti49)*Y(H1)*state.rho + -screened_rates(k_V50_to_p_Ti49)*Y(V50)) +
        (screened_rates(k_n_V49_to_V50)*Y(V49)*Y(N)*state.rho + -screened_rates(k_V50_to_n_V49)*Y(V50)) +
        (-screened_rates(k_n_V50_to_V51)*Y(V50)*Y(N)*state.rho + screened_rates(k_V51_to_n_V50)*Y(V51)) +
        (-screened_rates(k_p_V50_to_Cr51)*Y(V50)*Y(H1)*state.rho + screened_rates(k_Cr51_to_p_V50)*Y(Cr51)) +
        (-screened_rates(k_He4_V50_to_Mn54)*Y(He4)*Y(V50)*state.rho + screened_rates(k_Mn54_to_He4_V50)*Y(Mn54)) +
        (-screened_rates(k_n_V50_to_p_Ti50)*Y(V50)*Y(N)*state.rho + screened_rates(k_p_Ti50_to_n_V50)*Y(Ti50)*Y(H1)*state.rho) +
        (-screened_rates(k_n_V50_to_He4_Sc47)*Y(V50)*Y(N)*state.rho + screened_rates(k_He4_Sc47_to_n_V50)*Y(He4)*Y(Sc47)*state.rho) +
        (-screened_rates(k_p_V50_to_n_Cr50)*Y(V50)*Y(H1)*state.rho + screened_rates(k_n_Cr50_to_p_V50)*Y(Cr50)*Y(N)*state.rho) +
        (-screened_rates(k_p_V50_to_He4_Ti47)*Y(V50)*Y(H1)*state.rho + screened_rates(k_He4_Ti47_to_p_V50)*Y(He4)*Y(Ti47)*state.rho) +
        (-screened_rates(k_He4_V50_to_p_Cr53)*Y(He4)*Y(V50)*state.rho + screened_rates(k_p_Cr53_to_He4_V50)*Y(Cr53)*Y(H1)*state.rho) +
        (screened_rates(k_n_Mn53_to_He4_V50)*Y(Mn53)*Y(N)*state.rho + -screened_rates(k_He4_V50_to_n_Mn53)*Y(He4)*Y(V50)*state.rho) +
        (-screened_rates(k_V50_to_Cr50)*Y(V50) + screened_rates(k_Cr50_to_V50)*Y(Cr50)) +
        (-screened_rates(k_V50_to_Ti50)*Y(V50) + screened_rates(k_Ti50_to_V50)*Y(Ti50));

    ydot_nuc(V51) =
        (screened_rates(k_He4_Sc47_to_V51)*Y(He4)*Y(Sc47)*state.rho + -screened_rates(k_V51_to_He4_Sc47)*Y(V51)) +
        (screened_rates(k_p_Ti50_to_V51)*Y(Ti50)*Y(H1)*state.rho + -screened_rates(k_V51_to_p_Ti50)*Y(V51)) +
        (screened_rates(k_n_V50_to_V51)*Y(V50)*Y(N)*state.rho + -screened_rates(k_V51_to_n_V50)*Y(V51)) +
        (-screened_rates(k_n_V51_to_V52)*Y(V51)*Y(N)*state.rho + screened_rates(k_V52_to_n_V51)*Y(V52)) +
        (-screened_rates(k_p_V51_to_Cr52)*Y(V51)*Y(H1)*state.rho + screened_rates(k_Cr52_to_p_V51)*Y(Cr52)) +
        (-screened_rates(k_He4_V51_to_Mn55)*Y(He4)*Y(V51)*state.rho + screened_rates(k_Mn55_to_He4_V51)*Y(Mn55)) +
        (screened_rates(k_He4_Sc48_to_n_V51)*Y(He4)*Y(Sc48)*state.rho + -screened_rates(k_n_V51_to_He4_Sc48)*Y(V51)*Y(N)*state.rho) +
        (screened_rates(k_p_Ti51_to_n_V51)*Y(Ti51)*Y(H1)*state.rho + -screened_rates(k_n_V51_to_p_Ti51)*Y(V51)*Y(N)*state.rho) +
        (-screened_rates(k_p_V51_to_He4_Ti48)*Y(V51)*Y(H1)*state.rho + screened_rates(k_He4_Ti48_to_p_V51)*Y(He4)*Y(Ti48)*state.rho) +
        (screened_rates(k_n_Cr51_to_p_V51)*Y(Cr51)*Y(N)*state.rho + -screened_rates(k_p_V51_to_n_Cr51)*Y(V51)*Y(H1)*state.rho) +
        (screened_rates(k_p_Cr54_to_He4_V51)*Y(Cr54)*Y(H1)*state.rho + -screened_rates(k_He4_V51_to_p_Cr54)*Y(He4)*Y(V51)*state.rho) +
        (screened_rates(k_n_Mn54_to_He4_V51)*Y(Mn54)*Y(N)*state.rho + -screened_rates(k_He4_V51_to_n_Mn54)*Y(He4)*Y(V51)*state.rho) +
        (screened_rates(k_Cr51_to_V51)*Y(Cr51) + -screened_rates(k_V51_to_Cr51)*Y(V51)) +
        (screened_rates(k_Ti51_to_V51)*Y(Ti51) + -screened_rates(k_V51_to_Ti51)*Y(V51));

    ydot_nuc(V52) =
        (screened_rates(k_He4_Sc48_to_V52)*Y(He4)*Y(Sc48)*state.rho + -screened_rates(k_V52_to_He4_Sc48)*Y(V52)) +
        (screened_rates(k_p_Ti51_to_V52)*Y(Ti51)*Y(H1)*state.rho + -screened_rates(k_V52_to_p_Ti51)*Y(V52)) +
        (screened_rates(k_n_V51_to_V52)*Y(V51)*Y(N)*state.rho + -screened_rates(k_V52_to_n_V51)*Y(V52)) +
        (-screened_rates(k_p_V52_to_Cr53)*Y(V52)*Y(H1)*state.rho + screened_rates(k_Cr53_to_p_V52)*Y(Cr53)) +
        (-screened_rates(k_n_V52_to_He4_Sc49)*Y(V52)*Y(N)*state.rho + screened_rates(k_He4_Sc49_to_n_V52)*Y(He4)*Y(Sc49)*state.rho) +
        (-screened_rates(k_p_V52_to_n_Cr52)*Y(V52)*Y(H1)*state.rho + screened_rates(k_n_Cr52_to_p_V52)*Y(Cr52)*Y(N)*state.rho) +
        (-screened_rates(k_p_V52_to_He4_Ti49)*Y(V52)*Y(H1)*state.rho + screened_rates(k_He4_Ti49_to_p_V52)*Y(He4)*Y(Ti49)*state.rho) +
        (-screened_rates(k_He4_V52_to_n_Mn55)*Y(He4)*Y(V52)*state.rho + screened_rates(k_n_Mn55_to_He4_V52)*Y(Mn55)*Y(N)*state.rho) +
        (-screened_rates(k_V52_to_Cr52)*Y(V52) + screened_rates(k_Cr52_to_V52)*Y(Cr52));

    ydot_nuc(Cr48) =
        (screened_rates(k_He4_Ti44_to_Cr48)*Y(He4)*Y(Ti44)*state.rho + -screened_rates(k_Cr48_to_He4_Ti44)*Y(Cr48)) +
        (screened_rates(k_p_V47_to_Cr48)*Y(V47)*Y(H1)*state.rho + -screened_rates(k_Cr48_to_p_V47)*Y(Cr48)) +
        (-screened_rates(k_n_Cr48_to_Cr49)*Y(Cr48)*Y(N)*state.rho + screened_rates(k_Cr49_to_n_Cr48)*Y(Cr49)) +
        (-screened_rates(k_He4_Cr48_to_Fe52)*Y(Cr48)*Y(He4)*state.rho + screened_rates(k_Fe52_to_He4_Cr48)*Y(Fe52)) +
        (-screened_rates(k_n_Cr48_to_p_V48)*Y(Cr48)*Y(N)*state.rho + screened_rates(k_p_V48_to_n_Cr48)*Y(V48)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cr48_to_He4_Ti45)*Y(Cr48)*Y(N)*state.rho + screened_rates(k_He4_Ti45_to_n_Cr48)*Y(He4)*Y(Ti45)*state.rho) +
        (-screened_rates(k_He4_Cr48_to_p_Mn51)*Y(Cr48)*Y(He4)*state.rho + screened_rates(k_p_Mn51_to_He4_Cr48)*Y(Mn51)*Y(H1)*state.rho) +
        (-screened_rates(k_Cr48_to_V48)*Y(Cr48) + screened_rates(k_V48_to_Cr48)*Y(V48));

    ydot_nuc(Cr49) =
        (screened_rates(k_He4_Ti45_to_Cr49)*Y(He4)*Y(Ti45)*state.rho + -screened_rates(k_Cr49_to_He4_Ti45)*Y(Cr49)) +
        (screened_rates(k_p_V48_to_Cr49)*Y(V48)*Y(H1)*state.rho + -screened_rates(k_Cr49_to_p_V48)*Y(Cr49)) +
        (screened_rates(k_n_Cr48_to_Cr49)*Y(Cr48)*Y(N)*state.rho + -screened_rates(k_Cr49_to_n_Cr48)*Y(Cr49)) +
        (-screened_rates(k_n_Cr49_to_Cr50)*Y(Cr49)*Y(N)*state.rho + screened_rates(k_Cr50_to_n_Cr49)*Y(Cr50)) +
        (-screened_rates(k_p_Cr49_to_Mn50)*Y(Cr49)*Y(H1)*state.rho + screened_rates(k_Mn50_to_p_Cr49)*Y(Mn50)) +
        (-screened_rates(k_He4_Cr49_to_Fe53)*Y(Cr49)*Y(He4)*state.rho + screened_rates(k_Fe53_to_He4_Cr49)*Y(Fe53)) +
        (screened_rates(k_He4_V46_to_p_Cr49)*Y(He4)*Y(V46)*state.rho + -screened_rates(k_p_Cr49_to_He4_V46)*Y(Cr49)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cr49_to_p_V49)*Y(Cr49)*Y(N)*state.rho + screened_rates(k_p_V49_to_n_Cr49)*Y(V49)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cr49_to_He4_Ti46)*Y(Cr49)*Y(N)*state.rho + screened_rates(k_He4_Ti46_to_n_Cr49)*Y(He4)*Y(Ti46)*state.rho) +
        (-screened_rates(k_He4_Cr49_to_p_Mn52)*Y(Cr49)*Y(He4)*state.rho + screened_rates(k_p_Mn52_to_He4_Cr49)*Y(Mn52)*Y(H1)*state.rho) +
        (screened_rates(k_n_Fe52_to_He4_Cr49)*Y(Fe52)*Y(N)*state.rho + -screened_rates(k_He4_Cr49_to_n_Fe52)*Y(Cr49)*Y(He4)*state.rho) +
        (-screened_rates(k_Cr49_to_V49)*Y(Cr49) + screened_rates(k_V49_to_Cr49)*Y(V49));

    ydot_nuc(Cr50) =
        (screened_rates(k_He4_Ti46_to_Cr50)*Y(He4)*Y(Ti46)*state.rho + -screened_rates(k_Cr50_to_He4_Ti46)*Y(Cr50)) +
        (screened_rates(k_p_V49_to_Cr50)*Y(V49)*Y(H1)*state.rho + -screened_rates(k_Cr50_to_p_V49)*Y(Cr50)) +
        (screened_rates(k_n_Cr49_to_Cr50)*Y(Cr49)*Y(N)*state.rho + -screened_rates(k_Cr50_to_n_Cr49)*Y(Cr50)) +
        (-screened_rates(k_n_Cr50_to_Cr51)*Y(Cr50)*Y(N)*state.rho + screened_rates(k_Cr51_to_n_Cr50)*Y(Cr51)) +
        (-screened_rates(k_p_Cr50_to_Mn51)*Y(Cr50)*Y(H1)*state.rho + screened_rates(k_Mn51_to_p_Cr50)*Y(Mn51)) +
        (-screened_rates(k_He4_Cr50_to_Fe54)*Y(Cr50)*Y(He4)*state.rho + screened_rates(k_Fe54_to_He4_Cr50)*Y(Fe54)) +
        (screened_rates(k_He4_V47_to_p_Cr50)*Y(He4)*Y(V47)*state.rho + -screened_rates(k_p_Cr50_to_He4_V47)*Y(Cr50)*Y(H1)*state.rho) +
        (screened_rates(k_p_V50_to_n_Cr50)*Y(V50)*Y(H1)*state.rho + -screened_rates(k_n_Cr50_to_p_V50)*Y(Cr50)*Y(N)*state.rho) +
        (-screened_rates(k_n_Cr50_to_He4_Ti47)*Y(Cr50)*Y(N)*state.rho + screened_rates(k_He4_Ti47_to_n_Cr50)*Y(He4)*Y(Ti47)*state.rho) +
        (screened_rates(k_n_Mn50_to_p_Cr50)*Y(Mn50)*Y(N)*state.rho + -screened_rates(k_p_Cr50_to_n_Mn50)*Y(Cr50)*Y(H1)*state.rho) +
        (screened_rates(k_p_Mn53_to_He4_Cr50)*Y(Mn53)*Y(H1)*state.rho + -screened_rates(k_He4_Cr50_to_p_Mn53)*Y(Cr50)*Y(He4)*state.rho) +
        (screened_rates(k_n_Fe53_to_He4_Cr50)*Y(Fe53)*Y(N)*state.rho + -screened_rates(k_He4_Cr50_to_n_Fe53)*Y(Cr50)*Y(He4)*state.rho) +
        (screened_rates(k_Mn50_to_Cr50)*Y(Mn50) + -screened_rates(k_Cr50_to_Mn50)*Y(Cr50)) +
        (screened_rates(k_V50_to_Cr50)*Y(V50) + -screened_rates(k_Cr50_to_V50)*Y(Cr50));

    ydot_nuc(Cr51) =
        (screened_rates(k_He4_Ti47_to_Cr51)*Y(He4)*Y(Ti47)*state.rho + -screened_rates(k_Cr51_to_He4_Ti47)*Y(Cr51)) +
        (screened_rates(k_p_V50_to_Cr51)*Y(V50)*Y(H1)*state.rho + -screened_rates(k_Cr51_to_p_V50)*Y(Cr51)) +
        (screened_rates(k_n_Cr50_to_Cr51)*Y(Cr50)*Y(N)*state.rho + -screened_rates(k_Cr51_to_n_Cr50)*Y(Cr51)) +
        (-screened_rates(k_n_Cr51_to_Cr52)*Y(Cr51)*Y(N)*state.rho + screened_rates(k_Cr52_to_n_Cr51)*Y(Cr52)) +
        (-screened_rates(k_p_Cr51_to_Mn52)*Y(Cr51)*Y(H1)*state.rho + screened_rates(k_Mn52_to_p_Cr51)*Y(Mn52)) +
        (-screened_rates(k_He4_Cr51_to_Fe55)*Y(Cr51)*Y(He4)*state.rho + screened_rates(k_Fe55_to_He4_Cr51)*Y(Fe55)) +
        (screened_rates(k_He4_V48_to_p_Cr51)*Y(He4)*Y(V48)*state.rho + -screened_rates(k_p_Cr51_to_He4_V48)*Y(Cr51)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cr51_to_p_V51)*Y(Cr51)*Y(N)*state.rho + screened_rates(k_p_V51_to_n_Cr51)*Y(V51)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cr51_to_He4_Ti48)*Y(Cr51)*Y(N)*state.rho + screened_rates(k_He4_Ti48_to_n_Cr51)*Y(He4)*Y(Ti48)*state.rho) +
        (screened_rates(k_n_Mn51_to_p_Cr51)*Y(Mn51)*Y(N)*state.rho + -screened_rates(k_p_Cr51_to_n_Mn51)*Y(Cr51)*Y(H1)*state.rho) +
        (screened_rates(k_p_Mn54_to_He4_Cr51)*Y(Mn54)*Y(H1)*state.rho + -screened_rates(k_He4_Cr51_to_p_Mn54)*Y(Cr51)*Y(He4)*state.rho) +
        (screened_rates(k_n_Fe54_to_He4_Cr51)*Y(Fe54)*Y(N)*state.rho + -screened_rates(k_He4_Cr51_to_n_Fe54)*Y(Cr51)*Y(He4)*state.rho) +
        (-screened_rates(k_Cr51_to_V51)*Y(Cr51) + screened_rates(k_V51_to_Cr51)*Y(V51)) +
        (screened_rates(k_Mn51_to_Cr51)*Y(Mn51) + -screened_rates(k_Cr51_to_Mn51)*Y(Cr51));

    ydot_nuc(Cr52) =
        (screened_rates(k_He4_Ti48_to_Cr52)*Y(He4)*Y(Ti48)*state.rho + -screened_rates(k_Cr52_to_He4_Ti48)*Y(Cr52)) +
        (screened_rates(k_p_V51_to_Cr52)*Y(V51)*Y(H1)*state.rho + -screened_rates(k_Cr52_to_p_V51)*Y(Cr52)) +
        (screened_rates(k_n_Cr51_to_Cr52)*Y(Cr51)*Y(N)*state.rho + -screened_rates(k_Cr52_to_n_Cr51)*Y(Cr52)) +
        (-screened_rates(k_n_Cr52_to_Cr53)*Y(Cr52)*Y(N)*state.rho + screened_rates(k_Cr53_to_n_Cr52)*Y(Cr53)) +
        (-screened_rates(k_p_Cr52_to_Mn53)*Y(Cr52)*Y(H1)*state.rho + screened_rates(k_Mn53_to_p_Cr52)*Y(Mn53)) +
        (-screened_rates(k_He4_Cr52_to_Fe56)*Y(Cr52)*Y(He4)*state.rho + screened_rates(k_Fe56_to_He4_Cr52)*Y(Fe56)) +
        (screened_rates(k_He4_Ti49_to_n_Cr52)*Y(He4)*Y(Ti49)*state.rho + -screened_rates(k_n_Cr52_to_He4_Ti49)*Y(Cr52)*Y(N)*state.rho) +
        (screened_rates(k_He4_V49_to_p_Cr52)*Y(He4)*Y(V49)*state.rho + -screened_rates(k_p_Cr52_to_He4_V49)*Y(Cr52)*Y(H1)*state.rho) +
        (screened_rates(k_p_V52_to_n_Cr52)*Y(V52)*Y(H1)*state.rho + -screened_rates(k_n_Cr52_to_p_V52)*Y(Cr52)*Y(N)*state.rho) +
        (screened_rates(k_n_Mn52_to_p_Cr52)*Y(Mn52)*Y(N)*state.rho + -screened_rates(k_p_Cr52_to_n_Mn52)*Y(Cr52)*Y(H1)*state.rho) +
        (screened_rates(k_p_Mn55_to_He4_Cr52)*Y(Mn55)*Y(H1)*state.rho + -screened_rates(k_He4_Cr52_to_p_Mn55)*Y(Cr52)*Y(He4)*state.rho) +
        (screened_rates(k_n_Fe55_to_He4_Cr52)*Y(Fe55)*Y(N)*state.rho + -screened_rates(k_He4_Cr52_to_n_Fe55)*Y(Cr52)*Y(He4)*state.rho) +
        (screened_rates(k_Mn52_to_Cr52)*Y(Mn52) + -screened_rates(k_Cr52_to_Mn52)*Y(Cr52)) +
        (screened_rates(k_V52_to_Cr52)*Y(V52) + -screened_rates(k_Cr52_to_V52)*Y(Cr52));

    ydot_nuc(Cr53) =
        (screened_rates(k_He4_Ti49_to_Cr53)*Y(He4)*Y(Ti49)*state.rho + -screened_rates(k_Cr53_to_He4_Ti49)*Y(Cr53)) +
        (screened_rates(k_p_V52_to_Cr53)*Y(V52)*Y(H1)*state.rho + -screened_rates(k_Cr53_to_p_V52)*Y(Cr53)) +
        (screened_rates(k_n_Cr52_to_Cr53)*Y(Cr52)*Y(N)*state.rho + -screened_rates(k_Cr53_to_n_Cr52)*Y(Cr53)) +
        (-screened_rates(k_n_Cr53_to_Cr54)*Y(Cr53)*Y(N)*state.rho + screened_rates(k_Cr54_to_n_Cr53)*Y(Cr54)) +
        (-screened_rates(k_p_Cr53_to_Mn54)*Y(Cr53)*Y(H1)*state.rho + screened_rates(k_Mn54_to_p_Cr53)*Y(Mn54)) +
        (-screened_rates(k_He4_Cr53_to_Fe57)*Y(Cr53)*Y(He4)*state.rho + screened_rates(k_Fe57_to_He4_Cr53)*Y(Fe57)) +
        (screened_rates(k_He4_V50_to_p_Cr53)*Y(He4)*Y(V50)*state.rho + -screened_rates(k_p_Cr53_to_He4_V50)*Y(Cr53)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cr53_to_He4_Ti50)*Y(Cr53)*Y(N)*state.rho + screened_rates(k_He4_Ti50_to_n_Cr53)*Y(He4)*Y(Ti50)*state.rho) +
        (screened_rates(k_n_Mn53_to_p_Cr53)*Y(Mn53)*Y(N)*state.rho + -screened_rates(k_p_Cr53_to_n_Mn53)*Y(Cr53)*Y(H1)*state.rho) +
        (screened_rates(k_n_Fe56_to_He4_Cr53)*Y(Fe56)*Y(N)*state.rho + -screened_rates(k_He4_Cr53_to_n_Fe56)*Y(Cr53)*Y(He4)*state.rho) +
        (screened_rates(k_Mn53_to_Cr53)*Y(Mn53) + -screened_rates(k_Cr53_to_Mn53)*Y(Cr53));

    ydot_nuc(Cr54) =
        (screened_rates(k_He4_Ti50_to_Cr54)*Y(He4)*Y(Ti50)*state.rho + -screened_rates(k_Cr54_to_He4_Ti50)*Y(Cr54)) +
        (screened_rates(k_n_Cr53_to_Cr54)*Y(Cr53)*Y(N)*state.rho + -screened_rates(k_Cr54_to_n_Cr53)*Y(Cr54)) +
        (-screened_rates(k_p_Cr54_to_Mn55)*Y(Cr54)*Y(H1)*state.rho + screened_rates(k_Mn55_to_p_Cr54)*Y(Mn55)) +
        (-screened_rates(k_He4_Cr54_to_Fe58)*Y(Cr54)*Y(He4)*state.rho + screened_rates(k_Fe58_to_He4_Cr54)*Y(Fe58)) +
        (screened_rates(k_He4_Ti51_to_n_Cr54)*Y(He4)*Y(Ti51)*state.rho + -screened_rates(k_n_Cr54_to_He4_Ti51)*Y(Cr54)*Y(N)*state.rho) +
        (-screened_rates(k_p_Cr54_to_He4_V51)*Y(Cr54)*Y(H1)*state.rho + screened_rates(k_He4_V51_to_p_Cr54)*Y(He4)*Y(V51)*state.rho) +
        (screened_rates(k_n_Mn54_to_p_Cr54)*Y(Mn54)*Y(N)*state.rho + -screened_rates(k_p_Cr54_to_n_Mn54)*Y(Cr54)*Y(H1)*state.rho) +
        (screened_rates(k_n_Fe57_to_He4_Cr54)*Y(Fe57)*Y(N)*state.rho + -screened_rates(k_He4_Cr54_to_n_Fe57)*Y(Cr54)*Y(He4)*state.rho) +
        (screened_rates(k_Mn54_to_Cr54)*Y(Mn54) + -screened_rates(k_Cr54_to_Mn54)*Y(Cr54));

    ydot_nuc(Mn50) =
        (screened_rates(k_He4_V46_to_Mn50)*Y(He4)*Y(V46)*state.rho + -screened_rates(k_Mn50_to_He4_V46)*Y(Mn50)) +
        (screened_rates(k_p_Cr49_to_Mn50)*Y(Cr49)*Y(H1)*state.rho + -screened_rates(k_Mn50_to_p_Cr49)*Y(Mn50)) +
        (-screened_rates(k_n_Mn50_to_Mn51)*Y(Mn50)*Y(N)*state.rho + screened_rates(k_Mn51_to_n_Mn50)*Y(Mn51)) +
        (-screened_rates(k_He4_Mn50_to_Co54)*Y(He4)*Y(Mn50)*state.rho + screened_rates(k_Co54_to_He4_Mn50)*Y(Co54)) +
        (-screened_rates(k_n_Mn50_to_p_Cr50)*Y(Mn50)*Y(N)*state.rho + screened_rates(k_p_Cr50_to_n_Mn50)*Y(Cr50)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Mn50_to_He4_V47)*Y(Mn50)*Y(N)*state.rho + screened_rates(k_He4_V47_to_n_Mn50)*Y(He4)*Y(V47)*state.rho) +
        (-screened_rates(k_He4_Mn50_to_p_Fe53)*Y(He4)*Y(Mn50)*state.rho + screened_rates(k_p_Fe53_to_He4_Mn50)*Y(Fe53)*Y(H1)*state.rho) +
        (screened_rates(k_n_Co53_to_He4_Mn50)*Y(Co53)*Y(N)*state.rho + -screened_rates(k_He4_Mn50_to_n_Co53)*Y(He4)*Y(Mn50)*state.rho) +
        (-screened_rates(k_Mn50_to_Cr50)*Y(Mn50) + screened_rates(k_Cr50_to_Mn50)*Y(Cr50));

    ydot_nuc(Mn51) =
        (screened_rates(k_He4_V47_to_Mn51)*Y(He4)*Y(V47)*state.rho + -screened_rates(k_Mn51_to_He4_V47)*Y(Mn51)) +
        (screened_rates(k_p_Cr50_to_Mn51)*Y(Cr50)*Y(H1)*state.rho + -screened_rates(k_Mn51_to_p_Cr50)*Y(Mn51)) +
        (screened_rates(k_n_Mn50_to_Mn51)*Y(Mn50)*Y(N)*state.rho + -screened_rates(k_Mn51_to_n_Mn50)*Y(Mn51)) +
        (-screened_rates(k_n_Mn51_to_Mn52)*Y(Mn51)*Y(N)*state.rho + screened_rates(k_Mn52_to_n_Mn51)*Y(Mn52)) +
        (-screened_rates(k_p_Mn51_to_Fe52)*Y(Mn51)*Y(H1)*state.rho + screened_rates(k_Fe52_to_p_Mn51)*Y(Fe52)) +
        (-screened_rates(k_He4_Mn51_to_Co55)*Y(He4)*Y(Mn51)*state.rho + screened_rates(k_Co55_to_He4_Mn51)*Y(Co55)) +
        (screened_rates(k_He4_Cr48_to_p_Mn51)*Y(Cr48)*Y(He4)*state.rho + -screened_rates(k_p_Mn51_to_He4_Cr48)*Y(Mn51)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Mn51_to_p_Cr51)*Y(Mn51)*Y(N)*state.rho + screened_rates(k_p_Cr51_to_n_Mn51)*Y(Cr51)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Mn51_to_He4_V48)*Y(Mn51)*Y(N)*state.rho + screened_rates(k_He4_V48_to_n_Mn51)*Y(He4)*Y(V48)*state.rho) +
        (-screened_rates(k_He4_Mn51_to_p_Fe54)*Y(He4)*Y(Mn51)*state.rho + screened_rates(k_p_Fe54_to_He4_Mn51)*Y(Fe54)*Y(H1)*state.rho) +
        (screened_rates(k_n_Co54_to_He4_Mn51)*Y(Co54)*Y(N)*state.rho + -screened_rates(k_He4_Mn51_to_n_Co54)*Y(He4)*Y(Mn51)*state.rho) +
        (-screened_rates(k_Mn51_to_Cr51)*Y(Mn51) + screened_rates(k_Cr51_to_Mn51)*Y(Cr51));

    ydot_nuc(Mn52) =
        (screened_rates(k_He4_V48_to_Mn52)*Y(He4)*Y(V48)*state.rho + -screened_rates(k_Mn52_to_He4_V48)*Y(Mn52)) +
        (screened_rates(k_p_Cr51_to_Mn52)*Y(Cr51)*Y(H1)*state.rho + -screened_rates(k_Mn52_to_p_Cr51)*Y(Mn52)) +
        (screened_rates(k_n_Mn51_to_Mn52)*Y(Mn51)*Y(N)*state.rho + -screened_rates(k_Mn52_to_n_Mn51)*Y(Mn52)) +
        (-screened_rates(k_n_Mn52_to_Mn53)*Y(Mn52)*Y(N)*state.rho + screened_rates(k_Mn53_to_n_Mn52)*Y(Mn53)) +
        (-screened_rates(k_p_Mn52_to_Fe53)*Y(Mn52)*Y(H1)*state.rho + screened_rates(k_Fe53_to_p_Mn52)*Y(Fe53)) +
        (-screened_rates(k_He4_Mn52_to_Co56)*Y(He4)*Y(Mn52)*state.rho + screened_rates(k_Co56_to_He4_Mn52)*Y(Co56)) +
        (screened_rates(k_He4_Cr49_to_p_Mn52)*Y(Cr49)*Y(He4)*state.rho + -screened_rates(k_p_Mn52_to_He4_Cr49)*Y(Mn52)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Mn52_to_p_Cr52)*Y(Mn52)*Y(N)*state.rho + screened_rates(k_p_Cr52_to_n_Mn52)*Y(Cr52)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Mn52_to_He4_V49)*Y(Mn52)*Y(N)*state.rho + screened_rates(k_He4_V49_to_n_Mn52)*Y(He4)*Y(V49)*state.rho) +
        (-screened_rates(k_He4_Mn52_to_p_Fe55)*Y(He4)*Y(Mn52)*state.rho + screened_rates(k_p_Fe55_to_He4_Mn52)*Y(Fe55)*Y(H1)*state.rho) +
        (screened_rates(k_n_Fe52_to_p_Mn52)*Y(Fe52)*Y(N)*state.rho + -screened_rates(k_p_Mn52_to_n_Fe52)*Y(Mn52)*Y(H1)*state.rho) +
        (screened_rates(k_n_Co55_to_He4_Mn52)*Y(Co55)*Y(N)*state.rho + -screened_rates(k_He4_Mn52_to_n_Co55)*Y(He4)*Y(Mn52)*state.rho) +
        (screened_rates(k_Fe52_to_Mn52)*Y(Fe52) + -screened_rates(k_Mn52_to_Fe52)*Y(Mn52)) +
        (-screened_rates(k_Mn52_to_Cr52)*Y(Mn52) + screened_rates(k_Cr52_to_Mn52)*Y(Cr52));

    ydot_nuc(Mn53) =
        (screened_rates(k_He4_V49_to_Mn53)*Y(He4)*Y(V49)*state.rho + -screened_rates(k_Mn53_to_He4_V49)*Y(Mn53)) +
        (screened_rates(k_p_Cr52_to_Mn53)*Y(Cr52)*Y(H1)*state.rho + -screened_rates(k_Mn53_to_p_Cr52)*Y(Mn53)) +
        (screened_rates(k_n_Mn52_to_Mn53)*Y(Mn52)*Y(N)*state.rho + -screened_rates(k_Mn53_to_n_Mn52)*Y(Mn53)) +
        (-screened_rates(k_n_Mn53_to_Mn54)*Y(Mn53)*Y(N)*state.rho + screened_rates(k_Mn54_to_n_Mn53)*Y(Mn54)) +
        (-screened_rates(k_p_Mn53_to_Fe54)*Y(Mn53)*Y(H1)*state.rho + screened_rates(k_Fe54_to_p_Mn53)*Y(Fe54)) +
        (-screened_rates(k_He4_Mn53_to_Co57)*Y(He4)*Y(Mn53)*state.rho + screened_rates(k_Co57_to_He4_Mn53)*Y(Co57)) +
        (-screened_rates(k_n_Mn53_to_p_Cr53)*Y(Mn53)*Y(N)*state.rho + screened_rates(k_p_Cr53_to_n_Mn53)*Y(Cr53)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Mn53_to_He4_V50)*Y(Mn53)*Y(N)*state.rho + screened_rates(k_He4_V50_to_n_Mn53)*Y(He4)*Y(V50)*state.rho) +
        (-screened_rates(k_p_Mn53_to_He4_Cr50)*Y(Mn53)*Y(H1)*state.rho + screened_rates(k_He4_Cr50_to_p_Mn53)*Y(Cr50)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Mn53_to_p_Fe56)*Y(He4)*Y(Mn53)*state.rho + screened_rates(k_p_Fe56_to_He4_Mn53)*Y(Fe56)*Y(H1)*state.rho) +
        (screened_rates(k_n_Fe53_to_p_Mn53)*Y(Fe53)*Y(N)*state.rho + -screened_rates(k_p_Mn53_to_n_Fe53)*Y(Mn53)*Y(H1)*state.rho) +
        (screened_rates(k_n_Co56_to_He4_Mn53)*Y(Co56)*Y(N)*state.rho + -screened_rates(k_He4_Mn53_to_n_Co56)*Y(He4)*Y(Mn53)*state.rho) +
        (screened_rates(k_Fe53_to_Mn53)*Y(Fe53) + -screened_rates(k_Mn53_to_Fe53)*Y(Mn53)) +
        (-screened_rates(k_Mn53_to_Cr53)*Y(Mn53) + screened_rates(k_Cr53_to_Mn53)*Y(Cr53));

    ydot_nuc(Mn54) =
        (screened_rates(k_He4_V50_to_Mn54)*Y(He4)*Y(V50)*state.rho + -screened_rates(k_Mn54_to_He4_V50)*Y(Mn54)) +
        (screened_rates(k_p_Cr53_to_Mn54)*Y(Cr53)*Y(H1)*state.rho + -screened_rates(k_Mn54_to_p_Cr53)*Y(Mn54)) +
        (screened_rates(k_n_Mn53_to_Mn54)*Y(Mn53)*Y(N)*state.rho + -screened_rates(k_Mn54_to_n_Mn53)*Y(Mn54)) +
        (-screened_rates(k_n_Mn54_to_Mn55)*Y(Mn54)*Y(N)*state.rho + screened_rates(k_Mn55_to_n_Mn54)*Y(Mn55)) +
        (-screened_rates(k_p_Mn54_to_Fe55)*Y(Mn54)*Y(H1)*state.rho + screened_rates(k_Fe55_to_p_Mn54)*Y(Fe55)) +
        (-screened_rates(k_He4_Mn54_to_Co58)*Y(He4)*Y(Mn54)*state.rho + screened_rates(k_Co58_to_He4_Mn54)*Y(Co58)) +
        (-screened_rates(k_n_Mn54_to_p_Cr54)*Y(Mn54)*Y(N)*state.rho + screened_rates(k_p_Cr54_to_n_Mn54)*Y(Cr54)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Mn54_to_He4_V51)*Y(Mn54)*Y(N)*state.rho + screened_rates(k_He4_V51_to_n_Mn54)*Y(He4)*Y(V51)*state.rho) +
        (-screened_rates(k_p_Mn54_to_He4_Cr51)*Y(Mn54)*Y(H1)*state.rho + screened_rates(k_He4_Cr51_to_p_Mn54)*Y(Cr51)*Y(He4)*state.rho) +
        (screened_rates(k_n_Fe54_to_p_Mn54)*Y(Fe54)*Y(N)*state.rho + -screened_rates(k_p_Mn54_to_n_Fe54)*Y(Mn54)*Y(H1)*state.rho) +
        (screened_rates(k_p_Fe57_to_He4_Mn54)*Y(Fe57)*Y(H1)*state.rho + -screened_rates(k_He4_Mn54_to_p_Fe57)*Y(He4)*Y(Mn54)*state.rho) +
        (screened_rates(k_n_Co57_to_He4_Mn54)*Y(Co57)*Y(N)*state.rho + -screened_rates(k_He4_Mn54_to_n_Co57)*Y(He4)*Y(Mn54)*state.rho) +
        (-screened_rates(k_Mn54_to_Cr54)*Y(Mn54) + screened_rates(k_Cr54_to_Mn54)*Y(Cr54)) +
        (-screened_rates(k_Mn54_to_Fe54)*Y(Mn54) + screened_rates(k_Fe54_to_Mn54)*Y(Fe54));

    ydot_nuc(Mn55) =
        (screened_rates(k_He4_V51_to_Mn55)*Y(He4)*Y(V51)*state.rho + -screened_rates(k_Mn55_to_He4_V51)*Y(Mn55)) +
        (screened_rates(k_p_Cr54_to_Mn55)*Y(Cr54)*Y(H1)*state.rho + -screened_rates(k_Mn55_to_p_Cr54)*Y(Mn55)) +
        (screened_rates(k_n_Mn54_to_Mn55)*Y(Mn54)*Y(N)*state.rho + -screened_rates(k_Mn55_to_n_Mn54)*Y(Mn55)) +
        (-screened_rates(k_p_Mn55_to_Fe56)*Y(Mn55)*Y(H1)*state.rho + screened_rates(k_Fe56_to_p_Mn55)*Y(Fe56)) +
        (-screened_rates(k_He4_Mn55_to_Co59)*Y(He4)*Y(Mn55)*state.rho + screened_rates(k_Co59_to_He4_Mn55)*Y(Co59)) +
        (screened_rates(k_He4_V52_to_n_Mn55)*Y(He4)*Y(V52)*state.rho + -screened_rates(k_n_Mn55_to_He4_V52)*Y(Mn55)*Y(N)*state.rho) +
        (-screened_rates(k_p_Mn55_to_He4_Cr52)*Y(Mn55)*Y(H1)*state.rho + screened_rates(k_He4_Cr52_to_p_Mn55)*Y(Cr52)*Y(He4)*state.rho) +
        (screened_rates(k_n_Fe55_to_p_Mn55)*Y(Fe55)*Y(N)*state.rho + -screened_rates(k_p_Mn55_to_n_Fe55)*Y(Mn55)*Y(H1)*state.rho) +
        (screened_rates(k_p_Fe58_to_He4_Mn55)*Y(Fe58)*Y(H1)*state.rho + -screened_rates(k_He4_Mn55_to_p_Fe58)*Y(He4)*Y(Mn55)*state.rho) +
        (screened_rates(k_n_Co58_to_He4_Mn55)*Y(Co58)*Y(N)*state.rho + -screened_rates(k_He4_Mn55_to_n_Co58)*Y(He4)*Y(Mn55)*state.rho) +
        (screened_rates(k_Fe55_to_Mn55)*Y(Fe55) + -screened_rates(k_Mn55_to_Fe55)*Y(Mn55));

    ydot_nuc(Fe52) =
        (screened_rates(k_He4_Cr48_to_Fe52)*Y(Cr48)*Y(He4)*state.rho + -screened_rates(k_Fe52_to_He4_Cr48)*Y(Fe52)) +
        (screened_rates(k_p_Mn51_to_Fe52)*Y(Mn51)*Y(H1)*state.rho + -screened_rates(k_Fe52_to_p_Mn51)*Y(Fe52)) +
        (-screened_rates(k_n_Fe52_to_Fe53)*Y(Fe52)*Y(N)*state.rho + screened_rates(k_Fe53_to_n_Fe52)*Y(Fe53)) +
        (-screened_rates(k_p_Fe52_to_Co53)*Y(Fe52)*Y(H1)*state.rho + screened_rates(k_Co53_to_p_Fe52)*Y(Co53)) +
        (-screened_rates(k_He4_Fe52_to_Ni56)*Y(Fe52)*Y(He4)*state.rho + screened_rates(k_Ni56_to_He4_Fe52)*Y(Ni56)) +
        (-screened_rates(k_n_Fe52_to_p_Mn52)*Y(Fe52)*Y(N)*state.rho + screened_rates(k_p_Mn52_to_n_Fe52)*Y(Mn52)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Fe52_to_He4_Cr49)*Y(Fe52)*Y(N)*state.rho + screened_rates(k_He4_Cr49_to_n_Fe52)*Y(Cr49)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Fe52_to_p_Co55)*Y(Fe52)*Y(He4)*state.rho + screened_rates(k_p_Co55_to_He4_Fe52)*Y(Co55)*Y(H1)*state.rho) +
        (-screened_rates(k_Fe52_to_Mn52)*Y(Fe52) + screened_rates(k_Mn52_to_Fe52)*Y(Mn52));

    ydot_nuc(Fe53) =
        (screened_rates(k_He4_Cr49_to_Fe53)*Y(Cr49)*Y(He4)*state.rho + -screened_rates(k_Fe53_to_He4_Cr49)*Y(Fe53)) +
        (screened_rates(k_p_Mn52_to_Fe53)*Y(Mn52)*Y(H1)*state.rho + -screened_rates(k_Fe53_to_p_Mn52)*Y(Fe53)) +
        (screened_rates(k_n_Fe52_to_Fe53)*Y(Fe52)*Y(N)*state.rho + -screened_rates(k_Fe53_to_n_Fe52)*Y(Fe53)) +
        (-screened_rates(k_n_Fe53_to_Fe54)*Y(Fe53)*Y(N)*state.rho + screened_rates(k_Fe54_to_n_Fe53)*Y(Fe54)) +
        (-screened_rates(k_p_Fe53_to_Co54)*Y(Fe53)*Y(H1)*state.rho + screened_rates(k_Co54_to_p_Fe53)*Y(Co54)) +
        (-screened_rates(k_He4_Fe53_to_Ni57)*Y(Fe53)*Y(He4)*state.rho + screened_rates(k_Ni57_to_He4_Fe53)*Y(Ni57)) +
        (screened_rates(k_He4_Mn50_to_p_Fe53)*Y(He4)*Y(Mn50)*state.rho + -screened_rates(k_p_Fe53_to_He4_Mn50)*Y(Fe53)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Fe53_to_p_Mn53)*Y(Fe53)*Y(N)*state.rho + screened_rates(k_p_Mn53_to_n_Fe53)*Y(Mn53)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Fe53_to_He4_Cr50)*Y(Fe53)*Y(N)*state.rho + screened_rates(k_He4_Cr50_to_n_Fe53)*Y(Cr50)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Fe53_to_p_Co56)*Y(Fe53)*Y(He4)*state.rho + screened_rates(k_p_Co56_to_He4_Fe53)*Y(Co56)*Y(H1)*state.rho) +
        (screened_rates(k_n_Co53_to_p_Fe53)*Y(Co53)*Y(N)*state.rho + -screened_rates(k_p_Fe53_to_n_Co53)*Y(Fe53)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ni56_to_He4_Fe53)*Y(Ni56)*Y(N)*state.rho + -screened_rates(k_He4_Fe53_to_n_Ni56)*Y(Fe53)*Y(He4)*state.rho) +
        (screened_rates(k_Co53_to_Fe53)*Y(Co53) + -screened_rates(k_Fe53_to_Co53)*Y(Fe53)) +
        (-screened_rates(k_Fe53_to_Mn53)*Y(Fe53) + screened_rates(k_Mn53_to_Fe53)*Y(Mn53));

    ydot_nuc(Fe54) =
        (screened_rates(k_He4_Cr50_to_Fe54)*Y(Cr50)*Y(He4)*state.rho + -screened_rates(k_Fe54_to_He4_Cr50)*Y(Fe54)) +
        (screened_rates(k_p_Mn53_to_Fe54)*Y(Mn53)*Y(H1)*state.rho + -screened_rates(k_Fe54_to_p_Mn53)*Y(Fe54)) +
        (screened_rates(k_n_Fe53_to_Fe54)*Y(Fe53)*Y(N)*state.rho + -screened_rates(k_Fe54_to_n_Fe53)*Y(Fe54)) +
        (-screened_rates(k_n_Fe54_to_Fe55)*Y(Fe54)*Y(N)*state.rho + screened_rates(k_Fe55_to_n_Fe54)*Y(Fe55)) +
        (-screened_rates(k_p_Fe54_to_Co55)*Y(Fe54)*Y(H1)*state.rho + screened_rates(k_Co55_to_p_Fe54)*Y(Co55)) +
        (-screened_rates(k_He4_Fe54_to_Ni58)*Y(Fe54)*Y(He4)*state.rho + screened_rates(k_Ni58_to_He4_Fe54)*Y(Ni58)) +
        (screened_rates(k_He4_Mn51_to_p_Fe54)*Y(He4)*Y(Mn51)*state.rho + -screened_rates(k_p_Fe54_to_He4_Mn51)*Y(Fe54)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Fe54_to_p_Mn54)*Y(Fe54)*Y(N)*state.rho + screened_rates(k_p_Mn54_to_n_Fe54)*Y(Mn54)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Fe54_to_He4_Cr51)*Y(Fe54)*Y(N)*state.rho + screened_rates(k_He4_Cr51_to_n_Fe54)*Y(Cr51)*Y(He4)*state.rho) +
        (screened_rates(k_n_Co54_to_p_Fe54)*Y(Co54)*Y(N)*state.rho + -screened_rates(k_p_Fe54_to_n_Co54)*Y(Fe54)*Y(H1)*state.rho) +
        (screened_rates(k_p_Co57_to_He4_Fe54)*Y(Co57)*Y(H1)*state.rho + -screened_rates(k_He4_Fe54_to_p_Co57)*Y(Fe54)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ni57_to_He4_Fe54)*Y(Ni57)*Y(N)*state.rho + -screened_rates(k_He4_Fe54_to_n_Ni57)*Y(Fe54)*Y(He4)*state.rho) +
        (screened_rates(k_Co54_to_Fe54)*Y(Co54) + -screened_rates(k_Fe54_to_Co54)*Y(Fe54)) +
        (screened_rates(k_Mn54_to_Fe54)*Y(Mn54) + -screened_rates(k_Fe54_to_Mn54)*Y(Fe54));

    ydot_nuc(Fe55) =
        (screened_rates(k_He4_Cr51_to_Fe55)*Y(Cr51)*Y(He4)*state.rho + -screened_rates(k_Fe55_to_He4_Cr51)*Y(Fe55)) +
        (screened_rates(k_p_Mn54_to_Fe55)*Y(Mn54)*Y(H1)*state.rho + -screened_rates(k_Fe55_to_p_Mn54)*Y(Fe55)) +
        (screened_rates(k_n_Fe54_to_Fe55)*Y(Fe54)*Y(N)*state.rho + -screened_rates(k_Fe55_to_n_Fe54)*Y(Fe55)) +
        (-screened_rates(k_n_Fe55_to_Fe56)*Y(Fe55)*Y(N)*state.rho + screened_rates(k_Fe56_to_n_Fe55)*Y(Fe56)) +
        (-screened_rates(k_p_Fe55_to_Co56)*Y(Fe55)*Y(H1)*state.rho + screened_rates(k_Co56_to_p_Fe55)*Y(Co56)) +
        (-screened_rates(k_He4_Fe55_to_Ni59)*Y(Fe55)*Y(He4)*state.rho + screened_rates(k_Ni59_to_He4_Fe55)*Y(Ni59)) +
        (screened_rates(k_He4_Mn52_to_p_Fe55)*Y(He4)*Y(Mn52)*state.rho + -screened_rates(k_p_Fe55_to_He4_Mn52)*Y(Fe55)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Fe55_to_p_Mn55)*Y(Fe55)*Y(N)*state.rho + screened_rates(k_p_Mn55_to_n_Fe55)*Y(Mn55)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Fe55_to_He4_Cr52)*Y(Fe55)*Y(N)*state.rho + screened_rates(k_He4_Cr52_to_n_Fe55)*Y(Cr52)*Y(He4)*state.rho) +
        (screened_rates(k_n_Co55_to_p_Fe55)*Y(Co55)*Y(N)*state.rho + -screened_rates(k_p_Fe55_to_n_Co55)*Y(Fe55)*Y(H1)*state.rho) +
        (screened_rates(k_p_Co58_to_He4_Fe55)*Y(Co58)*Y(H1)*state.rho + -screened_rates(k_He4_Fe55_to_p_Co58)*Y(Fe55)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ni58_to_He4_Fe55)*Y(Ni58)*Y(N)*state.rho + -screened_rates(k_He4_Fe55_to_n_Ni58)*Y(Fe55)*Y(He4)*state.rho) +
        (screened_rates(k_Co55_to_Fe55)*Y(Co55) + -screened_rates(k_Fe55_to_Co55)*Y(Fe55)) +
        (-screened_rates(k_Fe55_to_Mn55)*Y(Fe55) + screened_rates(k_Mn55_to_Fe55)*Y(Mn55));

    ydot_nuc(Fe56) =
        (screened_rates(k_He4_Cr52_to_Fe56)*Y(Cr52)*Y(He4)*state.rho + -screened_rates(k_Fe56_to_He4_Cr52)*Y(Fe56)) +
        (screened_rates(k_p_Mn55_to_Fe56)*Y(Mn55)*Y(H1)*state.rho + -screened_rates(k_Fe56_to_p_Mn55)*Y(Fe56)) +
        (screened_rates(k_n_Fe55_to_Fe56)*Y(Fe55)*Y(N)*state.rho + -screened_rates(k_Fe56_to_n_Fe55)*Y(Fe56)) +
        (-screened_rates(k_n_Fe56_to_Fe57)*Y(Fe56)*Y(N)*state.rho + screened_rates(k_Fe57_to_n_Fe56)*Y(Fe57)) +
        (-screened_rates(k_p_Fe56_to_Co57)*Y(Fe56)*Y(H1)*state.rho + screened_rates(k_Co57_to_p_Fe56)*Y(Co57)) +
        (-screened_rates(k_He4_Fe56_to_Ni60)*Y(Fe56)*Y(He4)*state.rho + screened_rates(k_Ni60_to_He4_Fe56)*Y(Ni60)) +
        (screened_rates(k_He4_Mn53_to_p_Fe56)*Y(He4)*Y(Mn53)*state.rho + -screened_rates(k_p_Fe56_to_He4_Mn53)*Y(Fe56)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Fe56_to_He4_Cr53)*Y(Fe56)*Y(N)*state.rho + screened_rates(k_He4_Cr53_to_n_Fe56)*Y(Cr53)*Y(He4)*state.rho) +
        (screened_rates(k_n_Co56_to_p_Fe56)*Y(Co56)*Y(N)*state.rho + -screened_rates(k_p_Fe56_to_n_Co56)*Y(Fe56)*Y(H1)*state.rho) +
        (screened_rates(k_p_Co59_to_He4_Fe56)*Y(Co59)*Y(H1)*state.rho + -screened_rates(k_He4_Fe56_to_p_Co59)*Y(Fe56)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ni59_to_He4_Fe56)*Y(Ni59)*Y(N)*state.rho + -screened_rates(k_He4_Fe56_to_n_Ni59)*Y(Fe56)*Y(He4)*state.rho) +
        (screened_rates(k_Co56_to_Fe56)*Y(Co56) + -screened_rates(k_Fe56_to_Co56)*Y(Fe56));

    ydot_nuc(Fe57) =
        (screened_rates(k_He4_Cr53_to_Fe57)*Y(Cr53)*Y(He4)*state.rho + -screened_rates(k_Fe57_to_He4_Cr53)*Y(Fe57)) +
        (screened_rates(k_n_Fe56_to_Fe57)*Y(Fe56)*Y(N)*state.rho + -screened_rates(k_Fe57_to_n_Fe56)*Y(Fe57)) +
        (-screened_rates(k_n_Fe57_to_Fe58)*Y(Fe57)*Y(N)*state.rho + screened_rates(k_Fe58_to_n_Fe57)*Y(Fe58)) +
        (-screened_rates(k_p_Fe57_to_Co58)*Y(Fe57)*Y(H1)*state.rho + screened_rates(k_Co58_to_p_Fe57)*Y(Co58)) +
        (-screened_rates(k_He4_Fe57_to_Ni61)*Y(Fe57)*Y(He4)*state.rho + screened_rates(k_Ni61_to_He4_Fe57)*Y(Ni61)) +
        (-screened_rates(k_n_Fe57_to_He4_Cr54)*Y(Fe57)*Y(N)*state.rho + screened_rates(k_He4_Cr54_to_n_Fe57)*Y(Cr54)*Y(He4)*state.rho) +
        (-screened_rates(k_p_Fe57_to_He4_Mn54)*Y(Fe57)*Y(H1)*state.rho + screened_rates(k_He4_Mn54_to_p_Fe57)*Y(He4)*Y(Mn54)*state.rho) +
        (screened_rates(k_n_Co57_to_p_Fe57)*Y(Co57)*Y(N)*state.rho + -screened_rates(k_p_Fe57_to_n_Co57)*Y(Fe57)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ni60_to_He4_Fe57)*Y(Ni60)*Y(N)*state.rho + -screened_rates(k_He4_Fe57_to_n_Ni60)*Y(Fe57)*Y(He4)*state.rho) +
        (screened_rates(k_Co57_to_Fe57)*Y(Co57) + -screened_rates(k_Fe57_to_Co57)*Y(Fe57));

    ydot_nuc(Fe58) =
        (screened_rates(k_He4_Cr54_to_Fe58)*Y(Cr54)*Y(He4)*state.rho + -screened_rates(k_Fe58_to_He4_Cr54)*Y(Fe58)) +
        (screened_rates(k_n_Fe57_to_Fe58)*Y(Fe57)*Y(N)*state.rho + -screened_rates(k_Fe58_to_n_Fe57)*Y(Fe58)) +
        (-screened_rates(k_p_Fe58_to_Co59)*Y(Fe58)*Y(H1)*state.rho + screened_rates(k_Co59_to_p_Fe58)*Y(Co59)) +
        (-screened_rates(k_He4_Fe58_to_Ni62)*Y(Fe58)*Y(He4)*state.rho + screened_rates(k_Ni62_to_He4_Fe58)*Y(Ni62)) +
        (-screened_rates(k_p_Fe58_to_He4_Mn55)*Y(Fe58)*Y(H1)*state.rho + screened_rates(k_He4_Mn55_to_p_Fe58)*Y(He4)*Y(Mn55)*state.rho) +
        (screened_rates(k_n_Co58_to_p_Fe58)*Y(Co58)*Y(N)*state.rho + -screened_rates(k_p_Fe58_to_n_Co58)*Y(Fe58)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ni61_to_He4_Fe58)*Y(Ni61)*Y(N)*state.rho + -screened_rates(k_He4_Fe58_to_n_Ni61)*Y(Fe58)*Y(He4)*state.rho) +
        (screened_rates(k_Co58_to_Fe58)*Y(Co58) + -screened_rates(k_Fe58_to_Co58)*Y(Fe58));

    ydot_nuc(Co53) =
        (screened_rates(k_p_Fe52_to_Co53)*Y(Fe52)*Y(H1)*state.rho + -screened_rates(k_Co53_to_p_Fe52)*Y(Co53)) +
        (-screened_rates(k_n_Co53_to_Co54)*Y(Co53)*Y(N)*state.rho + screened_rates(k_Co54_to_n_Co53)*Y(Co54)) +
        (-screened_rates(k_He4_Co53_to_Cu57)*Y(Co53)*Y(He4)*state.rho + screened_rates(k_Cu57_to_He4_Co53)*Y(Cu57)) +
        (-screened_rates(k_n_Co53_to_p_Fe53)*Y(Co53)*Y(N)*state.rho + screened_rates(k_p_Fe53_to_n_Co53)*Y(Fe53)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Co53_to_He4_Mn50)*Y(Co53)*Y(N)*state.rho + screened_rates(k_He4_Mn50_to_n_Co53)*Y(He4)*Y(Mn50)*state.rho) +
        (-screened_rates(k_He4_Co53_to_p_Ni56)*Y(Co53)*Y(He4)*state.rho + screened_rates(k_p_Ni56_to_He4_Co53)*Y(Ni56)*Y(H1)*state.rho) +
        (-screened_rates(k_Co53_to_Fe53)*Y(Co53) + screened_rates(k_Fe53_to_Co53)*Y(Fe53));

    ydot_nuc(Co54) =
        (screened_rates(k_He4_Mn50_to_Co54)*Y(He4)*Y(Mn50)*state.rho + -screened_rates(k_Co54_to_He4_Mn50)*Y(Co54)) +
        (screened_rates(k_p_Fe53_to_Co54)*Y(Fe53)*Y(H1)*state.rho + -screened_rates(k_Co54_to_p_Fe53)*Y(Co54)) +
        (screened_rates(k_n_Co53_to_Co54)*Y(Co53)*Y(N)*state.rho + -screened_rates(k_Co54_to_n_Co53)*Y(Co54)) +
        (-screened_rates(k_n_Co54_to_Co55)*Y(Co54)*Y(N)*state.rho + screened_rates(k_Co55_to_n_Co54)*Y(Co55)) +
        (-screened_rates(k_He4_Co54_to_Cu58)*Y(Co54)*Y(He4)*state.rho + screened_rates(k_Cu58_to_He4_Co54)*Y(Cu58)) +
        (-screened_rates(k_n_Co54_to_p_Fe54)*Y(Co54)*Y(N)*state.rho + screened_rates(k_p_Fe54_to_n_Co54)*Y(Fe54)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Co54_to_He4_Mn51)*Y(Co54)*Y(N)*state.rho + screened_rates(k_He4_Mn51_to_n_Co54)*Y(He4)*Y(Mn51)*state.rho) +
        (-screened_rates(k_He4_Co54_to_p_Ni57)*Y(Co54)*Y(He4)*state.rho + screened_rates(k_p_Ni57_to_He4_Co54)*Y(Ni57)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cu57_to_He4_Co54)*Y(Cu57)*Y(N)*state.rho + -screened_rates(k_He4_Co54_to_n_Cu57)*Y(Co54)*Y(He4)*state.rho) +
        (-screened_rates(k_Co54_to_Fe54)*Y(Co54) + screened_rates(k_Fe54_to_Co54)*Y(Fe54));

    ydot_nuc(Co55) =
        (screened_rates(k_He4_Mn51_to_Co55)*Y(He4)*Y(Mn51)*state.rho + -screened_rates(k_Co55_to_He4_Mn51)*Y(Co55)) +
        (screened_rates(k_p_Fe54_to_Co55)*Y(Fe54)*Y(H1)*state.rho + -screened_rates(k_Co55_to_p_Fe54)*Y(Co55)) +
        (screened_rates(k_n_Co54_to_Co55)*Y(Co54)*Y(N)*state.rho + -screened_rates(k_Co55_to_n_Co54)*Y(Co55)) +
        (-screened_rates(k_n_Co55_to_Co56)*Y(Co55)*Y(N)*state.rho + screened_rates(k_Co56_to_n_Co55)*Y(Co56)) +
        (-screened_rates(k_p_Co55_to_Ni56)*Y(Co55)*Y(H1)*state.rho + screened_rates(k_Ni56_to_p_Co55)*Y(Ni56)) +
        (-screened_rates(k_He4_Co55_to_Cu59)*Y(Co55)*Y(He4)*state.rho + screened_rates(k_Cu59_to_He4_Co55)*Y(Cu59)) +
        (screened_rates(k_He4_Fe52_to_p_Co55)*Y(Fe52)*Y(He4)*state.rho + -screened_rates(k_p_Co55_to_He4_Fe52)*Y(Co55)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Co55_to_p_Fe55)*Y(Co55)*Y(N)*state.rho + screened_rates(k_p_Fe55_to_n_Co55)*Y(Fe55)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Co55_to_He4_Mn52)*Y(Co55)*Y(N)*state.rho + screened_rates(k_He4_Mn52_to_n_Co55)*Y(He4)*Y(Mn52)*state.rho) +
        (-screened_rates(k_He4_Co55_to_p_Ni58)*Y(Co55)*Y(He4)*state.rho + screened_rates(k_p_Ni58_to_He4_Co55)*Y(Ni58)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cu58_to_He4_Co55)*Y(Cu58)*Y(N)*state.rho + -screened_rates(k_He4_Co55_to_n_Cu58)*Y(Co55)*Y(He4)*state.rho) +
        (-screened_rates(k_Co55_to_Fe55)*Y(Co55) + screened_rates(k_Fe55_to_Co55)*Y(Fe55));

    ydot_nuc(Co56) =
        (screened_rates(k_He4_Mn52_to_Co56)*Y(He4)*Y(Mn52)*state.rho + -screened_rates(k_Co56_to_He4_Mn52)*Y(Co56)) +
        (screened_rates(k_p_Fe55_to_Co56)*Y(Fe55)*Y(H1)*state.rho + -screened_rates(k_Co56_to_p_Fe55)*Y(Co56)) +
        (screened_rates(k_n_Co55_to_Co56)*Y(Co55)*Y(N)*state.rho + -screened_rates(k_Co56_to_n_Co55)*Y(Co56)) +
        (-screened_rates(k_n_Co56_to_Co57)*Y(Co56)*Y(N)*state.rho + screened_rates(k_Co57_to_n_Co56)*Y(Co57)) +
        (-screened_rates(k_p_Co56_to_Ni57)*Y(Co56)*Y(H1)*state.rho + screened_rates(k_Ni57_to_p_Co56)*Y(Ni57)) +
        (-screened_rates(k_He4_Co56_to_Cu60)*Y(Co56)*Y(He4)*state.rho + screened_rates(k_Cu60_to_He4_Co56)*Y(Cu60)) +
        (screened_rates(k_He4_Fe53_to_p_Co56)*Y(Fe53)*Y(He4)*state.rho + -screened_rates(k_p_Co56_to_He4_Fe53)*Y(Co56)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Co56_to_p_Fe56)*Y(Co56)*Y(N)*state.rho + screened_rates(k_p_Fe56_to_n_Co56)*Y(Fe56)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Co56_to_He4_Mn53)*Y(Co56)*Y(N)*state.rho + screened_rates(k_He4_Mn53_to_n_Co56)*Y(He4)*Y(Mn53)*state.rho) +
        (-screened_rates(k_He4_Co56_to_p_Ni59)*Y(Co56)*Y(He4)*state.rho + screened_rates(k_p_Ni59_to_He4_Co56)*Y(Ni59)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ni56_to_p_Co56)*Y(Ni56)*Y(N)*state.rho + -screened_rates(k_p_Co56_to_n_Ni56)*Y(Co56)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cu59_to_He4_Co56)*Y(Cu59)*Y(N)*state.rho + -screened_rates(k_He4_Co56_to_n_Cu59)*Y(Co56)*Y(He4)*state.rho) +
        (-screened_rates(k_Co56_to_Fe56)*Y(Co56) + screened_rates(k_Fe56_to_Co56)*Y(Fe56)) +
        (screened_rates(k_Ni56_to_Co56)*Y(Ni56) + -screened_rates(k_Co56_to_Ni56)*Y(Co56));

    ydot_nuc(Co57) =
        (screened_rates(k_He4_Mn53_to_Co57)*Y(He4)*Y(Mn53)*state.rho + -screened_rates(k_Co57_to_He4_Mn53)*Y(Co57)) +
        (screened_rates(k_p_Fe56_to_Co57)*Y(Fe56)*Y(H1)*state.rho + -screened_rates(k_Co57_to_p_Fe56)*Y(Co57)) +
        (screened_rates(k_n_Co56_to_Co57)*Y(Co56)*Y(N)*state.rho + -screened_rates(k_Co57_to_n_Co56)*Y(Co57)) +
        (-screened_rates(k_n_Co57_to_Co58)*Y(Co57)*Y(N)*state.rho + screened_rates(k_Co58_to_n_Co57)*Y(Co58)) +
        (-screened_rates(k_p_Co57_to_Ni58)*Y(Co57)*Y(H1)*state.rho + screened_rates(k_Ni58_to_p_Co57)*Y(Ni58)) +
        (-screened_rates(k_He4_Co57_to_Cu61)*Y(Co57)*Y(He4)*state.rho + screened_rates(k_Cu61_to_He4_Co57)*Y(Cu61)) +
        (-screened_rates(k_n_Co57_to_p_Fe57)*Y(Co57)*Y(N)*state.rho + screened_rates(k_p_Fe57_to_n_Co57)*Y(Fe57)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Co57_to_He4_Mn54)*Y(Co57)*Y(N)*state.rho + screened_rates(k_He4_Mn54_to_n_Co57)*Y(He4)*Y(Mn54)*state.rho) +
        (-screened_rates(k_p_Co57_to_He4_Fe54)*Y(Co57)*Y(H1)*state.rho + screened_rates(k_He4_Fe54_to_p_Co57)*Y(Fe54)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Co57_to_p_Ni60)*Y(Co57)*Y(He4)*state.rho + screened_rates(k_p_Ni60_to_He4_Co57)*Y(Ni60)*Y(H1)*state.rho) +
        (screened_rates(k_n_Ni57_to_p_Co57)*Y(Ni57)*Y(N)*state.rho + -screened_rates(k_p_Co57_to_n_Ni57)*Y(Co57)*Y(H1)*state.rho) +
        (screened_rates(k_n_Cu60_to_He4_Co57)*Y(Cu60)*Y(N)*state.rho + -screened_rates(k_He4_Co57_to_n_Cu60)*Y(Co57)*Y(He4)*state.rho) +
        (-screened_rates(k_Co57_to_Fe57)*Y(Co57) + screened_rates(k_Fe57_to_Co57)*Y(Fe57)) +
        (screened_rates(k_Ni57_to_Co57)*Y(Ni57) + -screened_rates(k_Co57_to_Ni57)*Y(Co57));

    ydot_nuc(Co58) =
        (screened_rates(k_He4_Mn54_to_Co58)*Y(He4)*Y(Mn54)*state.rho + -screened_rates(k_Co58_to_He4_Mn54)*Y(Co58)) +
        (screened_rates(k_p_Fe57_to_Co58)*Y(Fe57)*Y(H1)*state.rho + -screened_rates(k_Co58_to_p_Fe57)*Y(Co58)) +
        (screened_rates(k_n_Co57_to_Co58)*Y(Co57)*Y(N)*state.rho + -screened_rates(k_Co58_to_n_Co57)*Y(Co58)) +
        (-screened_rates(k_n_Co58_to_Co59)*Y(Co58)*Y(N)*state.rho + screened_rates(k_Co59_to_n_Co58)*Y(Co59)) +
        (-screened_rates(k_p_Co58_to_Ni59)*Y(Co58)*Y(H1)*state.rho + screened_rates(k_Ni59_to_p_Co58)*Y(Ni59)) +
        (-screened_rates(k_He4_Co58_to_Cu62)*Y(Co58)*Y(He4)*state.rho + screened_rates(k_Cu62_to_He4_Co58)*Y(Cu62)) +
        (-screened_rates(k_n_Co58_to_p_Fe58)*Y(Co58)*Y(N)*state.rho + screened_rates(k_p_Fe58_to_n_Co58)*Y(Fe58)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Co58_to_He4_Mn55)*Y(Co58)*Y(N)*state.rho + screened_rates(k_He4_Mn55_to_n_Co58)*Y(He4)*Y(Mn55)*state.rho) +
        (-screened_rates(k_p_Co58_to_He4_Fe55)*Y(Co58)*Y(H1)*state.rho + screened_rates(k_He4_Fe55_to_p_Co58)*Y(Fe55)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ni58_to_p_Co58)*Y(Ni58)*Y(N)*state.rho + -screened_rates(k_p_Co58_to_n_Ni58)*Y(Co58)*Y(H1)*state.rho) +
        (screened_rates(k_p_Ni61_to_He4_Co58)*Y(Ni61)*Y(H1)*state.rho + -screened_rates(k_He4_Co58_to_p_Ni61)*Y(Co58)*Y(He4)*state.rho) +
        (screened_rates(k_n_Cu61_to_He4_Co58)*Y(Cu61)*Y(N)*state.rho + -screened_rates(k_He4_Co58_to_n_Cu61)*Y(Co58)*Y(He4)*state.rho) +
        (-screened_rates(k_Co58_to_Fe58)*Y(Co58) + screened_rates(k_Fe58_to_Co58)*Y(Fe58)) +
        (-screened_rates(k_Co58_to_Ni58)*Y(Co58) + screened_rates(k_Ni58_to_Co58)*Y(Ni58));

    ydot_nuc(Co59) =
        (screened_rates(k_He4_Mn55_to_Co59)*Y(He4)*Y(Mn55)*state.rho + -screened_rates(k_Co59_to_He4_Mn55)*Y(Co59)) +
        (screened_rates(k_p_Fe58_to_Co59)*Y(Fe58)*Y(H1)*state.rho + -screened_rates(k_Co59_to_p_Fe58)*Y(Co59)) +
        (screened_rates(k_n_Co58_to_Co59)*Y(Co58)*Y(N)*state.rho + -screened_rates(k_Co59_to_n_Co58)*Y(Co59)) +
        (-screened_rates(k_p_Co59_to_Ni60)*Y(Co59)*Y(H1)*state.rho + screened_rates(k_Ni60_to_p_Co59)*Y(Ni60)) +
        (-screened_rates(k_He4_Co59_to_Cu63)*Y(Co59)*Y(He4)*state.rho + screened_rates(k_Cu63_to_He4_Co59)*Y(Cu63)) +
        (-screened_rates(k_p_Co59_to_He4_Fe56)*Y(Co59)*Y(H1)*state.rho + screened_rates(k_He4_Fe56_to_p_Co59)*Y(Fe56)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ni59_to_p_Co59)*Y(Ni59)*Y(N)*state.rho + -screened_rates(k_p_Co59_to_n_Ni59)*Y(Co59)*Y(H1)*state.rho) +
        (screened_rates(k_p_Ni62_to_He4_Co59)*Y(Ni62)*Y(H1)*state.rho + -screened_rates(k_He4_Co59_to_p_Ni62)*Y(Co59)*Y(He4)*state.rho) +
        (screened_rates(k_n_Cu62_to_He4_Co59)*Y(Cu62)*Y(N)*state.rho + -screened_rates(k_He4_Co59_to_n_Cu62)*Y(Co59)*Y(He4)*state.rho) +
        (screened_rates(k_Ni59_to_Co59)*Y(Ni59) + -screened_rates(k_Co59_to_Ni59)*Y(Co59));

    ydot_nuc(Ni56) =
        (screened_rates(k_He4_Fe52_to_Ni56)*Y(Fe52)*Y(He4)*state.rho + -screened_rates(k_Ni56_to_He4_Fe52)*Y(Ni56)) +
        (screened_rates(k_p_Co55_to_Ni56)*Y(Co55)*Y(H1)*state.rho + -screened_rates(k_Ni56_to_p_Co55)*Y(Ni56)) +
        (-screened_rates(k_n_Ni56_to_Ni57)*Y(Ni56)*Y(N)*state.rho + screened_rates(k_Ni57_to_n_Ni56)*Y(Ni57)) +
        (-screened_rates(k_p_Ni56_to_Cu57)*Y(Ni56)*Y(H1)*state.rho + screened_rates(k_Cu57_to_p_Ni56)*Y(Cu57)) +
        (-screened_rates(k_He4_Ni56_to_Zn60)*Y(He4)*Y(Ni56)*state.rho + screened_rates(k_Zn60_to_He4_Ni56)*Y(Zn60)) +
        (screened_rates(k_He4_Co53_to_p_Ni56)*Y(Co53)*Y(He4)*state.rho + -screened_rates(k_p_Ni56_to_He4_Co53)*Y(Ni56)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ni56_to_p_Co56)*Y(Ni56)*Y(N)*state.rho + screened_rates(k_p_Co56_to_n_Ni56)*Y(Co56)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ni56_to_He4_Fe53)*Y(Ni56)*Y(N)*state.rho + screened_rates(k_He4_Fe53_to_n_Ni56)*Y(Fe53)*Y(He4)*state.rho) +
        (screened_rates(k_p_Cu59_to_He4_Ni56)*Y(Cu59)*Y(H1)*state.rho + -screened_rates(k_He4_Ni56_to_p_Cu59)*Y(He4)*Y(Ni56)*state.rho) +
        (screened_rates(k_n_Zn59_to_He4_Ni56)*Y(Zn59)*Y(N)*state.rho + -screened_rates(k_He4_Ni56_to_n_Zn59)*Y(He4)*Y(Ni56)*state.rho) +
        (-screened_rates(k_Ni56_to_Co56)*Y(Ni56) + screened_rates(k_Co56_to_Ni56)*Y(Co56));

    ydot_nuc(Ni57) =
        (screened_rates(k_He4_Fe53_to_Ni57)*Y(Fe53)*Y(He4)*state.rho + -screened_rates(k_Ni57_to_He4_Fe53)*Y(Ni57)) +
        (screened_rates(k_p_Co56_to_Ni57)*Y(Co56)*Y(H1)*state.rho + -screened_rates(k_Ni57_to_p_Co56)*Y(Ni57)) +
        (screened_rates(k_n_Ni56_to_Ni57)*Y(Ni56)*Y(N)*state.rho + -screened_rates(k_Ni57_to_n_Ni56)*Y(Ni57)) +
        (-screened_rates(k_n_Ni57_to_Ni58)*Y(Ni57)*Y(N)*state.rho + screened_rates(k_Ni58_to_n_Ni57)*Y(Ni58)) +
        (-screened_rates(k_p_Ni57_to_Cu58)*Y(Ni57)*Y(H1)*state.rho + screened_rates(k_Cu58_to_p_Ni57)*Y(Cu58)) +
        (-screened_rates(k_He4_Ni57_to_Zn61)*Y(He4)*Y(Ni57)*state.rho + screened_rates(k_Zn61_to_He4_Ni57)*Y(Zn61)) +
        (screened_rates(k_He4_Co54_to_p_Ni57)*Y(Co54)*Y(He4)*state.rho + -screened_rates(k_p_Ni57_to_He4_Co54)*Y(Ni57)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ni57_to_p_Co57)*Y(Ni57)*Y(N)*state.rho + screened_rates(k_p_Co57_to_n_Ni57)*Y(Co57)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ni57_to_He4_Fe54)*Y(Ni57)*Y(N)*state.rho + screened_rates(k_He4_Fe54_to_n_Ni57)*Y(Fe54)*Y(He4)*state.rho) +
        (screened_rates(k_n_Cu57_to_p_Ni57)*Y(Cu57)*Y(N)*state.rho + -screened_rates(k_p_Ni57_to_n_Cu57)*Y(Ni57)*Y(H1)*state.rho) +
        (screened_rates(k_p_Cu60_to_He4_Ni57)*Y(Cu60)*Y(H1)*state.rho + -screened_rates(k_He4_Ni57_to_p_Cu60)*Y(He4)*Y(Ni57)*state.rho) +
        (screened_rates(k_n_Zn60_to_He4_Ni57)*Y(Zn60)*Y(N)*state.rho + -screened_rates(k_He4_Ni57_to_n_Zn60)*Y(He4)*Y(Ni57)*state.rho) +
        (screened_rates(k_Cu57_to_Ni57)*Y(Cu57) + -screened_rates(k_Ni57_to_Cu57)*Y(Ni57)) +
        (-screened_rates(k_Ni57_to_Co57)*Y(Ni57) + screened_rates(k_Co57_to_Ni57)*Y(Co57));

    ydot_nuc(Ni58) =
        screened_rates(k_Zn59_to_p_Ni58_weak_wc12)*Y(Zn59) +
        (screened_rates(k_He4_Fe54_to_Ni58)*Y(Fe54)*Y(He4)*state.rho + -screened_rates(k_Ni58_to_He4_Fe54)*Y(Ni58)) +
        (screened_rates(k_p_Co57_to_Ni58)*Y(Co57)*Y(H1)*state.rho + -screened_rates(k_Ni58_to_p_Co57)*Y(Ni58)) +
        (screened_rates(k_n_Ni57_to_Ni58)*Y(Ni57)*Y(N)*state.rho + -screened_rates(k_Ni58_to_n_Ni57)*Y(Ni58)) +
        (-screened_rates(k_n_Ni58_to_Ni59)*Y(Ni58)*Y(N)*state.rho + screened_rates(k_Ni59_to_n_Ni58)*Y(Ni59)) +
        (-screened_rates(k_p_Ni58_to_Cu59)*Y(Ni58)*Y(H1)*state.rho + screened_rates(k_Cu59_to_p_Ni58)*Y(Cu59)) +
        (-screened_rates(k_He4_Ni58_to_Zn62)*Y(He4)*Y(Ni58)*state.rho + screened_rates(k_Zn62_to_He4_Ni58)*Y(Zn62)) +
        (screened_rates(k_He4_Co55_to_p_Ni58)*Y(Co55)*Y(He4)*state.rho + -screened_rates(k_p_Ni58_to_He4_Co55)*Y(Ni58)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ni58_to_p_Co58)*Y(Ni58)*Y(N)*state.rho + screened_rates(k_p_Co58_to_n_Ni58)*Y(Co58)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ni58_to_He4_Fe55)*Y(Ni58)*Y(N)*state.rho + screened_rates(k_He4_Fe55_to_n_Ni58)*Y(Fe55)*Y(He4)*state.rho) +
        (screened_rates(k_n_Cu58_to_p_Ni58)*Y(Cu58)*Y(N)*state.rho + -screened_rates(k_p_Ni58_to_n_Cu58)*Y(Ni58)*Y(H1)*state.rho) +
        (screened_rates(k_p_Cu61_to_He4_Ni58)*Y(Cu61)*Y(H1)*state.rho + -screened_rates(k_He4_Ni58_to_p_Cu61)*Y(He4)*Y(Ni58)*state.rho) +
        (screened_rates(k_n_Zn61_to_He4_Ni58)*Y(Zn61)*Y(N)*state.rho + -screened_rates(k_He4_Ni58_to_n_Zn61)*Y(He4)*Y(Ni58)*state.rho) +
        (screened_rates(k_Co58_to_Ni58)*Y(Co58) + -screened_rates(k_Ni58_to_Co58)*Y(Ni58)) +
        (screened_rates(k_Cu58_to_Ni58)*Y(Cu58) + -screened_rates(k_Ni58_to_Cu58)*Y(Ni58));

    ydot_nuc(Ni59) =
        (screened_rates(k_He4_Fe55_to_Ni59)*Y(Fe55)*Y(He4)*state.rho + -screened_rates(k_Ni59_to_He4_Fe55)*Y(Ni59)) +
        (screened_rates(k_p_Co58_to_Ni59)*Y(Co58)*Y(H1)*state.rho + -screened_rates(k_Ni59_to_p_Co58)*Y(Ni59)) +
        (screened_rates(k_n_Ni58_to_Ni59)*Y(Ni58)*Y(N)*state.rho + -screened_rates(k_Ni59_to_n_Ni58)*Y(Ni59)) +
        (-screened_rates(k_n_Ni59_to_Ni60)*Y(Ni59)*Y(N)*state.rho + screened_rates(k_Ni60_to_n_Ni59)*Y(Ni60)) +
        (-screened_rates(k_p_Ni59_to_Cu60)*Y(Ni59)*Y(H1)*state.rho + screened_rates(k_Cu60_to_p_Ni59)*Y(Cu60)) +
        (-screened_rates(k_He4_Ni59_to_Zn63)*Y(He4)*Y(Ni59)*state.rho + screened_rates(k_Zn63_to_He4_Ni59)*Y(Zn63)) +
        (screened_rates(k_He4_Co56_to_p_Ni59)*Y(Co56)*Y(He4)*state.rho + -screened_rates(k_p_Ni59_to_He4_Co56)*Y(Ni59)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ni59_to_p_Co59)*Y(Ni59)*Y(N)*state.rho + screened_rates(k_p_Co59_to_n_Ni59)*Y(Co59)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ni59_to_He4_Fe56)*Y(Ni59)*Y(N)*state.rho + screened_rates(k_He4_Fe56_to_n_Ni59)*Y(Fe56)*Y(He4)*state.rho) +
        (screened_rates(k_n_Cu59_to_p_Ni59)*Y(Cu59)*Y(N)*state.rho + -screened_rates(k_p_Ni59_to_n_Cu59)*Y(Ni59)*Y(H1)*state.rho) +
        (screened_rates(k_p_Cu62_to_He4_Ni59)*Y(Cu62)*Y(H1)*state.rho + -screened_rates(k_He4_Ni59_to_p_Cu62)*Y(He4)*Y(Ni59)*state.rho) +
        (screened_rates(k_n_Zn62_to_He4_Ni59)*Y(Zn62)*Y(N)*state.rho + -screened_rates(k_He4_Ni59_to_n_Zn62)*Y(He4)*Y(Ni59)*state.rho) +
        (screened_rates(k_Cu59_to_Ni59)*Y(Cu59) + -screened_rates(k_Ni59_to_Cu59)*Y(Ni59)) +
        (-screened_rates(k_Ni59_to_Co59)*Y(Ni59) + screened_rates(k_Co59_to_Ni59)*Y(Co59));

    ydot_nuc(Ni60) =
        (screened_rates(k_He4_Fe56_to_Ni60)*Y(Fe56)*Y(He4)*state.rho + -screened_rates(k_Ni60_to_He4_Fe56)*Y(Ni60)) +
        (screened_rates(k_p_Co59_to_Ni60)*Y(Co59)*Y(H1)*state.rho + -screened_rates(k_Ni60_to_p_Co59)*Y(Ni60)) +
        (screened_rates(k_n_Ni59_to_Ni60)*Y(Ni59)*Y(N)*state.rho + -screened_rates(k_Ni60_to_n_Ni59)*Y(Ni60)) +
        (-screened_rates(k_n_Ni60_to_Ni61)*Y(Ni60)*Y(N)*state.rho + screened_rates(k_Ni61_to_n_Ni60)*Y(Ni61)) +
        (-screened_rates(k_p_Ni60_to_Cu61)*Y(Ni60)*Y(H1)*state.rho + screened_rates(k_Cu61_to_p_Ni60)*Y(Cu61)) +
        (-screened_rates(k_He4_Ni60_to_Zn64)*Y(He4)*Y(Ni60)*state.rho + screened_rates(k_Zn64_to_He4_Ni60)*Y(Zn64)) +
        (screened_rates(k_He4_Co57_to_p_Ni60)*Y(Co57)*Y(He4)*state.rho + -screened_rates(k_p_Ni60_to_He4_Co57)*Y(Ni60)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ni60_to_He4_Fe57)*Y(Ni60)*Y(N)*state.rho + screened_rates(k_He4_Fe57_to_n_Ni60)*Y(Fe57)*Y(He4)*state.rho) +
        (screened_rates(k_n_Cu60_to_p_Ni60)*Y(Cu60)*Y(N)*state.rho + -screened_rates(k_p_Ni60_to_n_Cu60)*Y(Ni60)*Y(H1)*state.rho) +
        (screened_rates(k_p_Cu63_to_He4_Ni60)*Y(Cu63)*Y(H1)*state.rho + -screened_rates(k_He4_Ni60_to_p_Cu63)*Y(He4)*Y(Ni60)*state.rho) +
        (screened_rates(k_n_Zn63_to_He4_Ni60)*Y(Zn63)*Y(N)*state.rho + -screened_rates(k_He4_Ni60_to_n_Zn63)*Y(He4)*Y(Ni60)*state.rho) +
        (screened_rates(k_Cu60_to_Ni60)*Y(Cu60) + -screened_rates(k_Ni60_to_Cu60)*Y(Ni60));

    ydot_nuc(Ni61) =
        (screened_rates(k_He4_Fe57_to_Ni61)*Y(Fe57)*Y(He4)*state.rho + -screened_rates(k_Ni61_to_He4_Fe57)*Y(Ni61)) +
        (screened_rates(k_n_Ni60_to_Ni61)*Y(Ni60)*Y(N)*state.rho + -screened_rates(k_Ni61_to_n_Ni60)*Y(Ni61)) +
        (-screened_rates(k_n_Ni61_to_Ni62)*Y(Ni61)*Y(N)*state.rho + screened_rates(k_Ni62_to_n_Ni61)*Y(Ni62)) +
        (-screened_rates(k_p_Ni61_to_Cu62)*Y(Ni61)*Y(H1)*state.rho + screened_rates(k_Cu62_to_p_Ni61)*Y(Cu62)) +
        (-screened_rates(k_He4_Ni61_to_Zn65)*Y(He4)*Y(Ni61)*state.rho + screened_rates(k_Zn65_to_He4_Ni61)*Y(Zn65)) +
        (-screened_rates(k_n_Ni61_to_He4_Fe58)*Y(Ni61)*Y(N)*state.rho + screened_rates(k_He4_Fe58_to_n_Ni61)*Y(Fe58)*Y(He4)*state.rho) +
        (-screened_rates(k_p_Ni61_to_He4_Co58)*Y(Ni61)*Y(H1)*state.rho + screened_rates(k_He4_Co58_to_p_Ni61)*Y(Co58)*Y(He4)*state.rho) +
        (screened_rates(k_n_Cu61_to_p_Ni61)*Y(Cu61)*Y(N)*state.rho + -screened_rates(k_p_Ni61_to_n_Cu61)*Y(Ni61)*Y(H1)*state.rho) +
        (screened_rates(k_p_Cu64_to_He4_Ni61)*Y(Cu64)*Y(H1)*state.rho + -screened_rates(k_He4_Ni61_to_p_Cu64)*Y(He4)*Y(Ni61)*state.rho) +
        (screened_rates(k_n_Zn64_to_He4_Ni61)*Y(Zn64)*Y(N)*state.rho + -screened_rates(k_He4_Ni61_to_n_Zn64)*Y(He4)*Y(Ni61)*state.rho) +
        (screened_rates(k_Cu61_to_Ni61)*Y(Cu61) + -screened_rates(k_Ni61_to_Cu61)*Y(Ni61));

    ydot_nuc(Ni62) =
        (screened_rates(k_He4_Fe58_to_Ni62)*Y(Fe58)*Y(He4)*state.rho + -screened_rates(k_Ni62_to_He4_Fe58)*Y(Ni62)) +
        (screened_rates(k_n_Ni61_to_Ni62)*Y(Ni61)*Y(N)*state.rho + -screened_rates(k_Ni62_to_n_Ni61)*Y(Ni62)) +
        (-screened_rates(k_n_Ni62_to_Ni63)*Y(Ni62)*Y(N)*state.rho + screened_rates(k_Ni63_to_n_Ni62)*Y(Ni63)) +
        (-screened_rates(k_p_Ni62_to_Cu63)*Y(Ni62)*Y(H1)*state.rho + screened_rates(k_Cu63_to_p_Ni62)*Y(Cu63)) +
        (-screened_rates(k_He4_Ni62_to_Zn66)*Y(He4)*Y(Ni62)*state.rho + screened_rates(k_Zn66_to_He4_Ni62)*Y(Zn66)) +
        (-screened_rates(k_p_Ni62_to_He4_Co59)*Y(Ni62)*Y(H1)*state.rho + screened_rates(k_He4_Co59_to_p_Ni62)*Y(Co59)*Y(He4)*state.rho) +
        (screened_rates(k_n_Cu62_to_p_Ni62)*Y(Cu62)*Y(N)*state.rho + -screened_rates(k_p_Ni62_to_n_Cu62)*Y(Ni62)*Y(H1)*state.rho) +
        (screened_rates(k_p_Cu65_to_He4_Ni62)*Y(Cu65)*Y(H1)*state.rho + -screened_rates(k_He4_Ni62_to_p_Cu65)*Y(He4)*Y(Ni62)*state.rho) +
        (screened_rates(k_n_Zn65_to_He4_Ni62)*Y(Zn65)*Y(N)*state.rho + -screened_rates(k_He4_Ni62_to_n_Zn65)*Y(He4)*Y(Ni62)*state.rho) +
        (screened_rates(k_Cu62_to_Ni62)*Y(Cu62) + -screened_rates(k_Ni62_to_Cu62)*Y(Ni62));

    ydot_nuc(Ni63) =
        (screened_rates(k_n_Ni62_to_Ni63)*Y(Ni62)*Y(N)*state.rho + -screened_rates(k_Ni63_to_n_Ni62)*Y(Ni63)) +
        (-screened_rates(k_n_Ni63_to_Ni64)*Y(Ni63)*Y(N)*state.rho + screened_rates(k_Ni64_to_n_Ni63)*Y(Ni64)) +
        (-screened_rates(k_p_Ni63_to_Cu64)*Y(Ni63)*Y(H1)*state.rho + screened_rates(k_Cu64_to_p_Ni63)*Y(Cu64)) +
        (screened_rates(k_n_Cu63_to_p_Ni63)*Y(Cu63)*Y(N)*state.rho + -screened_rates(k_p_Ni63_to_n_Cu63)*Y(Ni63)*Y(H1)*state.rho) +
        (screened_rates(k_n_Zn66_to_He4_Ni63)*Y(Zn66)*Y(N)*state.rho + -screened_rates(k_He4_Ni63_to_n_Zn66)*Y(He4)*Y(Ni63)*state.rho) +
        (-screened_rates(k_Ni63_to_Cu63)*Y(Ni63) + screened_rates(k_Cu63_to_Ni63)*Y(Cu63));

    ydot_nuc(Ni64) =
        (screened_rates(k_n_Ni63_to_Ni64)*Y(Ni63)*Y(N)*state.rho + -screened_rates(k_Ni64_to_n_Ni63)*Y(Ni64)) +
        (-screened_rates(k_p_Ni64_to_Cu65)*Y(Ni64)*Y(H1)*state.rho + screened_rates(k_Cu65_to_p_Ni64)*Y(Cu65)) +
        (screened_rates(k_n_Cu64_to_p_Ni64)*Y(Cu64)*Y(N)*state.rho + -screened_rates(k_p_Ni64_to_n_Cu64)*Y(Ni64)*Y(H1)*state.rho) +
        (screened_rates(k_Cu64_to_Ni64)*Y(Cu64) + -screened_rates(k_Ni64_to_Cu64)*Y(Ni64));

    ydot_nuc(Cu57) =
        (screened_rates(k_He4_Co53_to_Cu57)*Y(Co53)*Y(He4)*state.rho + -screened_rates(k_Cu57_to_He4_Co53)*Y(Cu57)) +
        (screened_rates(k_p_Ni56_to_Cu57)*Y(Ni56)*Y(H1)*state.rho + -screened_rates(k_Cu57_to_p_Ni56)*Y(Cu57)) +
        (-screened_rates(k_n_Cu57_to_Cu58)*Y(Cu57)*Y(N)*state.rho + screened_rates(k_Cu58_to_n_Cu57)*Y(Cu58)) +
        (-screened_rates(k_n_Cu57_to_p_Ni57)*Y(Cu57)*Y(N)*state.rho + screened_rates(k_p_Ni57_to_n_Cu57)*Y(Ni57)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cu57_to_He4_Co54)*Y(Cu57)*Y(N)*state.rho + screened_rates(k_He4_Co54_to_n_Cu57)*Y(Co54)*Y(He4)*state.rho) +
        (-screened_rates(k_He4_Cu57_to_p_Zn60)*Y(Cu57)*Y(He4)*state.rho + screened_rates(k_p_Zn60_to_He4_Cu57)*Y(Zn60)*Y(H1)*state.rho) +
        (-screened_rates(k_Cu57_to_Ni57)*Y(Cu57) + screened_rates(k_Ni57_to_Cu57)*Y(Ni57));

    ydot_nuc(Cu58) =
        (screened_rates(k_He4_Co54_to_Cu58)*Y(Co54)*Y(He4)*state.rho + -screened_rates(k_Cu58_to_He4_Co54)*Y(Cu58)) +
        (screened_rates(k_p_Ni57_to_Cu58)*Y(Ni57)*Y(H1)*state.rho + -screened_rates(k_Cu58_to_p_Ni57)*Y(Cu58)) +
        (screened_rates(k_n_Cu57_to_Cu58)*Y(Cu57)*Y(N)*state.rho + -screened_rates(k_Cu58_to_n_Cu57)*Y(Cu58)) +
        (-screened_rates(k_n_Cu58_to_Cu59)*Y(Cu58)*Y(N)*state.rho + screened_rates(k_Cu59_to_n_Cu58)*Y(Cu59)) +
        (-screened_rates(k_p_Cu58_to_Zn59)*Y(Cu58)*Y(H1)*state.rho + screened_rates(k_Zn59_to_p_Cu58)*Y(Zn59)) +
        (-screened_rates(k_He4_Cu58_to_Ga62)*Y(Cu58)*Y(He4)*state.rho + screened_rates(k_Ga62_to_He4_Cu58)*Y(Ga62)) +
        (-screened_rates(k_n_Cu58_to_p_Ni58)*Y(Cu58)*Y(N)*state.rho + screened_rates(k_p_Ni58_to_n_Cu58)*Y(Ni58)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cu58_to_He4_Co55)*Y(Cu58)*Y(N)*state.rho + screened_rates(k_He4_Co55_to_n_Cu58)*Y(Co55)*Y(He4)*state.rho) +
        (screened_rates(k_p_Zn61_to_He4_Cu58)*Y(Zn61)*Y(H1)*state.rho + -screened_rates(k_He4_Cu58_to_p_Zn61)*Y(Cu58)*Y(He4)*state.rho) +
        (-screened_rates(k_Cu58_to_Ni58)*Y(Cu58) + screened_rates(k_Ni58_to_Cu58)*Y(Ni58));

    ydot_nuc(Cu59) =
        screened_rates(k_Zn59_to_Cu59_weak_wc12)*Y(Zn59) +
        (screened_rates(k_He4_Co55_to_Cu59)*Y(Co55)*Y(He4)*state.rho + -screened_rates(k_Cu59_to_He4_Co55)*Y(Cu59)) +
        (screened_rates(k_p_Ni58_to_Cu59)*Y(Ni58)*Y(H1)*state.rho + -screened_rates(k_Cu59_to_p_Ni58)*Y(Cu59)) +
        (screened_rates(k_n_Cu58_to_Cu59)*Y(Cu58)*Y(N)*state.rho + -screened_rates(k_Cu59_to_n_Cu58)*Y(Cu59)) +
        (-screened_rates(k_n_Cu59_to_Cu60)*Y(Cu59)*Y(N)*state.rho + screened_rates(k_Cu60_to_n_Cu59)*Y(Cu60)) +
        (-screened_rates(k_p_Cu59_to_Zn60)*Y(Cu59)*Y(H1)*state.rho + screened_rates(k_Zn60_to_p_Cu59)*Y(Zn60)) +
        (-screened_rates(k_He4_Cu59_to_Ga63)*Y(Cu59)*Y(He4)*state.rho + screened_rates(k_Ga63_to_He4_Cu59)*Y(Ga63)) +
        (-screened_rates(k_n_Cu59_to_p_Ni59)*Y(Cu59)*Y(N)*state.rho + screened_rates(k_p_Ni59_to_n_Cu59)*Y(Ni59)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cu59_to_He4_Co56)*Y(Cu59)*Y(N)*state.rho + screened_rates(k_He4_Co56_to_n_Cu59)*Y(Co56)*Y(He4)*state.rho) +
        (-screened_rates(k_p_Cu59_to_He4_Ni56)*Y(Cu59)*Y(H1)*state.rho + screened_rates(k_He4_Ni56_to_p_Cu59)*Y(He4)*Y(Ni56)*state.rho) +
        (screened_rates(k_n_Zn59_to_p_Cu59)*Y(Zn59)*Y(N)*state.rho + -screened_rates(k_p_Cu59_to_n_Zn59)*Y(Cu59)*Y(H1)*state.rho) +
        (screened_rates(k_p_Zn62_to_He4_Cu59)*Y(Zn62)*Y(H1)*state.rho + -screened_rates(k_He4_Cu59_to_p_Zn62)*Y(Cu59)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ga62_to_He4_Cu59)*Y(Ga62)*Y(N)*state.rho + -screened_rates(k_He4_Cu59_to_n_Ga62)*Y(Cu59)*Y(He4)*state.rho) +
        (-screened_rates(k_Cu59_to_Ni59)*Y(Cu59) + screened_rates(k_Ni59_to_Cu59)*Y(Ni59));

    ydot_nuc(Cu60) =
        (screened_rates(k_He4_Co56_to_Cu60)*Y(Co56)*Y(He4)*state.rho + -screened_rates(k_Cu60_to_He4_Co56)*Y(Cu60)) +
        (screened_rates(k_p_Ni59_to_Cu60)*Y(Ni59)*Y(H1)*state.rho + -screened_rates(k_Cu60_to_p_Ni59)*Y(Cu60)) +
        (screened_rates(k_n_Cu59_to_Cu60)*Y(Cu59)*Y(N)*state.rho + -screened_rates(k_Cu60_to_n_Cu59)*Y(Cu60)) +
        (-screened_rates(k_n_Cu60_to_Cu61)*Y(Cu60)*Y(N)*state.rho + screened_rates(k_Cu61_to_n_Cu60)*Y(Cu61)) +
        (-screened_rates(k_p_Cu60_to_Zn61)*Y(Cu60)*Y(H1)*state.rho + screened_rates(k_Zn61_to_p_Cu60)*Y(Zn61)) +
        (-screened_rates(k_He4_Cu60_to_Ga64)*Y(Cu60)*Y(He4)*state.rho + screened_rates(k_Ga64_to_He4_Cu60)*Y(Ga64)) +
        (-screened_rates(k_n_Cu60_to_p_Ni60)*Y(Cu60)*Y(N)*state.rho + screened_rates(k_p_Ni60_to_n_Cu60)*Y(Ni60)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cu60_to_He4_Co57)*Y(Cu60)*Y(N)*state.rho + screened_rates(k_He4_Co57_to_n_Cu60)*Y(Co57)*Y(He4)*state.rho) +
        (-screened_rates(k_p_Cu60_to_He4_Ni57)*Y(Cu60)*Y(H1)*state.rho + screened_rates(k_He4_Ni57_to_p_Cu60)*Y(He4)*Y(Ni57)*state.rho) +
        (screened_rates(k_n_Zn60_to_p_Cu60)*Y(Zn60)*Y(N)*state.rho + -screened_rates(k_p_Cu60_to_n_Zn60)*Y(Cu60)*Y(H1)*state.rho) +
        (screened_rates(k_p_Zn63_to_He4_Cu60)*Y(Zn63)*Y(H1)*state.rho + -screened_rates(k_He4_Cu60_to_p_Zn63)*Y(Cu60)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ga63_to_He4_Cu60)*Y(Ga63)*Y(N)*state.rho + -screened_rates(k_He4_Cu60_to_n_Ga63)*Y(Cu60)*Y(He4)*state.rho) +
        (-screened_rates(k_Cu60_to_Ni60)*Y(Cu60) + screened_rates(k_Ni60_to_Cu60)*Y(Ni60)) +
        (screened_rates(k_Zn60_to_Cu60)*Y(Zn60) + -screened_rates(k_Cu60_to_Zn60)*Y(Cu60));

    ydot_nuc(Cu61) =
        (screened_rates(k_He4_Co57_to_Cu61)*Y(Co57)*Y(He4)*state.rho + -screened_rates(k_Cu61_to_He4_Co57)*Y(Cu61)) +
        (screened_rates(k_p_Ni60_to_Cu61)*Y(Ni60)*Y(H1)*state.rho + -screened_rates(k_Cu61_to_p_Ni60)*Y(Cu61)) +
        (screened_rates(k_n_Cu60_to_Cu61)*Y(Cu60)*Y(N)*state.rho + -screened_rates(k_Cu61_to_n_Cu60)*Y(Cu61)) +
        (-screened_rates(k_n_Cu61_to_Cu62)*Y(Cu61)*Y(N)*state.rho + screened_rates(k_Cu62_to_n_Cu61)*Y(Cu62)) +
        (-screened_rates(k_p_Cu61_to_Zn62)*Y(Cu61)*Y(H1)*state.rho + screened_rates(k_Zn62_to_p_Cu61)*Y(Zn62)) +
        (-screened_rates(k_n_Cu61_to_p_Ni61)*Y(Cu61)*Y(N)*state.rho + screened_rates(k_p_Ni61_to_n_Cu61)*Y(Ni61)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cu61_to_He4_Co58)*Y(Cu61)*Y(N)*state.rho + screened_rates(k_He4_Co58_to_n_Cu61)*Y(Co58)*Y(He4)*state.rho) +
        (-screened_rates(k_p_Cu61_to_He4_Ni58)*Y(Cu61)*Y(H1)*state.rho + screened_rates(k_He4_Ni58_to_p_Cu61)*Y(He4)*Y(Ni58)*state.rho) +
        (screened_rates(k_n_Zn61_to_p_Cu61)*Y(Zn61)*Y(N)*state.rho + -screened_rates(k_p_Cu61_to_n_Zn61)*Y(Cu61)*Y(H1)*state.rho) +
        (screened_rates(k_p_Zn64_to_He4_Cu61)*Y(Zn64)*Y(H1)*state.rho + -screened_rates(k_He4_Cu61_to_p_Zn64)*Y(Cu61)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ga64_to_He4_Cu61)*Y(Ga64)*Y(N)*state.rho + -screened_rates(k_He4_Cu61_to_n_Ga64)*Y(Cu61)*Y(He4)*state.rho) +
        (-screened_rates(k_Cu61_to_Ni61)*Y(Cu61) + screened_rates(k_Ni61_to_Cu61)*Y(Ni61)) +
        (screened_rates(k_Zn61_to_Cu61)*Y(Zn61) + -screened_rates(k_Cu61_to_Zn61)*Y(Cu61));

    ydot_nuc(Cu62) =
        (screened_rates(k_He4_Co58_to_Cu62)*Y(Co58)*Y(He4)*state.rho + -screened_rates(k_Cu62_to_He4_Co58)*Y(Cu62)) +
        (screened_rates(k_p_Ni61_to_Cu62)*Y(Ni61)*Y(H1)*state.rho + -screened_rates(k_Cu62_to_p_Ni61)*Y(Cu62)) +
        (screened_rates(k_n_Cu61_to_Cu62)*Y(Cu61)*Y(N)*state.rho + -screened_rates(k_Cu62_to_n_Cu61)*Y(Cu62)) +
        (-screened_rates(k_n_Cu62_to_Cu63)*Y(Cu62)*Y(N)*state.rho + screened_rates(k_Cu63_to_n_Cu62)*Y(Cu63)) +
        (-screened_rates(k_p_Cu62_to_Zn63)*Y(Cu62)*Y(H1)*state.rho + screened_rates(k_Zn63_to_p_Cu62)*Y(Zn63)) +
        (-screened_rates(k_n_Cu62_to_p_Ni62)*Y(Cu62)*Y(N)*state.rho + screened_rates(k_p_Ni62_to_n_Cu62)*Y(Ni62)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Cu62_to_He4_Co59)*Y(Cu62)*Y(N)*state.rho + screened_rates(k_He4_Co59_to_n_Cu62)*Y(Co59)*Y(He4)*state.rho) +
        (-screened_rates(k_p_Cu62_to_He4_Ni59)*Y(Cu62)*Y(H1)*state.rho + screened_rates(k_He4_Ni59_to_p_Cu62)*Y(He4)*Y(Ni59)*state.rho) +
        (screened_rates(k_n_Zn62_to_p_Cu62)*Y(Zn62)*Y(N)*state.rho + -screened_rates(k_p_Cu62_to_n_Zn62)*Y(Cu62)*Y(H1)*state.rho) +
        (screened_rates(k_p_Zn65_to_He4_Cu62)*Y(Zn65)*Y(H1)*state.rho + -screened_rates(k_He4_Cu62_to_p_Zn65)*Y(Cu62)*Y(He4)*state.rho) +
        (-screened_rates(k_Cu62_to_Ni62)*Y(Cu62) + screened_rates(k_Ni62_to_Cu62)*Y(Ni62)) +
        (screened_rates(k_Zn62_to_Cu62)*Y(Zn62) + -screened_rates(k_Cu62_to_Zn62)*Y(Cu62));

    ydot_nuc(Cu63) =
        (screened_rates(k_He4_Co59_to_Cu63)*Y(Co59)*Y(He4)*state.rho + -screened_rates(k_Cu63_to_He4_Co59)*Y(Cu63)) +
        (screened_rates(k_p_Ni62_to_Cu63)*Y(Ni62)*Y(H1)*state.rho + -screened_rates(k_Cu63_to_p_Ni62)*Y(Cu63)) +
        (screened_rates(k_n_Cu62_to_Cu63)*Y(Cu62)*Y(N)*state.rho + -screened_rates(k_Cu63_to_n_Cu62)*Y(Cu63)) +
        (-screened_rates(k_n_Cu63_to_Cu64)*Y(Cu63)*Y(N)*state.rho + screened_rates(k_Cu64_to_n_Cu63)*Y(Cu64)) +
        (-screened_rates(k_p_Cu63_to_Zn64)*Y(Cu63)*Y(H1)*state.rho + screened_rates(k_Zn64_to_p_Cu63)*Y(Zn64)) +
        (-screened_rates(k_n_Cu63_to_p_Ni63)*Y(Cu63)*Y(N)*state.rho + screened_rates(k_p_Ni63_to_n_Cu63)*Y(Ni63)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Cu63_to_He4_Ni60)*Y(Cu63)*Y(H1)*state.rho + screened_rates(k_He4_Ni60_to_p_Cu63)*Y(He4)*Y(Ni60)*state.rho) +
        (screened_rates(k_n_Zn63_to_p_Cu63)*Y(Zn63)*Y(N)*state.rho + -screened_rates(k_p_Cu63_to_n_Zn63)*Y(Cu63)*Y(H1)*state.rho) +
        (screened_rates(k_p_Zn66_to_He4_Cu63)*Y(Zn66)*Y(H1)*state.rho + -screened_rates(k_He4_Cu63_to_p_Zn66)*Y(Cu63)*Y(He4)*state.rho) +
        (screened_rates(k_Ni63_to_Cu63)*Y(Ni63) + -screened_rates(k_Cu63_to_Ni63)*Y(Cu63)) +
        (screened_rates(k_Zn63_to_Cu63)*Y(Zn63) + -screened_rates(k_Cu63_to_Zn63)*Y(Cu63));

    ydot_nuc(Cu64) =
        (screened_rates(k_p_Ni63_to_Cu64)*Y(Ni63)*Y(H1)*state.rho + -screened_rates(k_Cu64_to_p_Ni63)*Y(Cu64)) +
        (screened_rates(k_n_Cu63_to_Cu64)*Y(Cu63)*Y(N)*state.rho + -screened_rates(k_Cu64_to_n_Cu63)*Y(Cu64)) +
        (-screened_rates(k_n_Cu64_to_Cu65)*Y(Cu64)*Y(N)*state.rho + screened_rates(k_Cu65_to_n_Cu64)*Y(Cu65)) +
        (-screened_rates(k_p_Cu64_to_Zn65)*Y(Cu64)*Y(H1)*state.rho + screened_rates(k_Zn65_to_p_Cu64)*Y(Zn65)) +
        (-screened_rates(k_n_Cu64_to_p_Ni64)*Y(Cu64)*Y(N)*state.rho + screened_rates(k_p_Ni64_to_n_Cu64)*Y(Ni64)*Y(H1)*state.rho) +
        (-screened_rates(k_p_Cu64_to_He4_Ni61)*Y(Cu64)*Y(H1)*state.rho + screened_rates(k_He4_Ni61_to_p_Cu64)*Y(He4)*Y(Ni61)*state.rho) +
        (screened_rates(k_n_Zn64_to_p_Cu64)*Y(Zn64)*Y(N)*state.rho + -screened_rates(k_p_Cu64_to_n_Zn64)*Y(Cu64)*Y(H1)*state.rho) +
        (-screened_rates(k_Cu64_to_Ni64)*Y(Cu64) + screened_rates(k_Ni64_to_Cu64)*Y(Ni64)) +
        (-screened_rates(k_Cu64_to_Zn64)*Y(Cu64) + screened_rates(k_Zn64_to_Cu64)*Y(Zn64));

    ydot_nuc(Cu65) =
        (screened_rates(k_p_Ni64_to_Cu65)*Y(Ni64)*Y(H1)*state.rho + -screened_rates(k_Cu65_to_p_Ni64)*Y(Cu65)) +
        (screened_rates(k_n_Cu64_to_Cu65)*Y(Cu64)*Y(N)*state.rho + -screened_rates(k_Cu65_to_n_Cu64)*Y(Cu65)) +
        (-screened_rates(k_p_Cu65_to_Zn66)*Y(Cu65)*Y(H1)*state.rho + screened_rates(k_Zn66_to_p_Cu65)*Y(Zn66)) +
        (-screened_rates(k_p_Cu65_to_He4_Ni62)*Y(Cu65)*Y(H1)*state.rho + screened_rates(k_He4_Ni62_to_p_Cu65)*Y(He4)*Y(Ni62)*state.rho) +
        (screened_rates(k_n_Zn65_to_p_Cu65)*Y(Zn65)*Y(N)*state.rho + -screened_rates(k_p_Cu65_to_n_Zn65)*Y(Cu65)*Y(H1)*state.rho) +
        (screened_rates(k_Zn65_to_Cu65)*Y(Zn65) + -screened_rates(k_Cu65_to_Zn65)*Y(Cu65));

    ydot_nuc(Zn59) =
        -screened_rates(k_Zn59_to_Cu59_weak_wc12)*Y(Zn59) +
        -screened_rates(k_Zn59_to_p_Ni58_weak_wc12)*Y(Zn59) +
        (screened_rates(k_p_Cu58_to_Zn59)*Y(Cu58)*Y(H1)*state.rho + -screened_rates(k_Zn59_to_p_Cu58)*Y(Zn59)) +
        (-screened_rates(k_n_Zn59_to_Zn60)*Y(Zn59)*Y(N)*state.rho + screened_rates(k_Zn60_to_n_Zn59)*Y(Zn60)) +
        (-screened_rates(k_He4_Zn59_to_Ge63)*Y(He4)*Y(Zn59)*state.rho + screened_rates(k_Ge63_to_He4_Zn59)*Y(Ge63)) +
        (-screened_rates(k_n_Zn59_to_p_Cu59)*Y(Zn59)*Y(N)*state.rho + screened_rates(k_p_Cu59_to_n_Zn59)*Y(Cu59)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Zn59_to_He4_Ni56)*Y(Zn59)*Y(N)*state.rho + screened_rates(k_He4_Ni56_to_n_Zn59)*Y(He4)*Y(Ni56)*state.rho) +
        (screened_rates(k_p_Ga62_to_He4_Zn59)*Y(Ga62)*Y(H1)*state.rho + -screened_rates(k_He4_Zn59_to_p_Ga62)*Y(He4)*Y(Zn59)*state.rho);

    ydot_nuc(Zn60) =
        (screened_rates(k_He4_Ni56_to_Zn60)*Y(He4)*Y(Ni56)*state.rho + -screened_rates(k_Zn60_to_He4_Ni56)*Y(Zn60)) +
        (screened_rates(k_p_Cu59_to_Zn60)*Y(Cu59)*Y(H1)*state.rho + -screened_rates(k_Zn60_to_p_Cu59)*Y(Zn60)) +
        (screened_rates(k_n_Zn59_to_Zn60)*Y(Zn59)*Y(N)*state.rho + -screened_rates(k_Zn60_to_n_Zn59)*Y(Zn60)) +
        (-screened_rates(k_n_Zn60_to_Zn61)*Y(Zn60)*Y(N)*state.rho + screened_rates(k_Zn61_to_n_Zn60)*Y(Zn61)) +
        (-screened_rates(k_He4_Zn60_to_Ge64)*Y(He4)*Y(Zn60)*state.rho + screened_rates(k_Ge64_to_He4_Zn60)*Y(Ge64)) +
        (screened_rates(k_He4_Cu57_to_p_Zn60)*Y(Cu57)*Y(He4)*state.rho + -screened_rates(k_p_Zn60_to_He4_Cu57)*Y(Zn60)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Zn60_to_p_Cu60)*Y(Zn60)*Y(N)*state.rho + screened_rates(k_p_Cu60_to_n_Zn60)*Y(Cu60)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Zn60_to_He4_Ni57)*Y(Zn60)*Y(N)*state.rho + screened_rates(k_He4_Ni57_to_n_Zn60)*Y(He4)*Y(Ni57)*state.rho) +
        (screened_rates(k_p_Ga63_to_He4_Zn60)*Y(Ga63)*Y(H1)*state.rho + -screened_rates(k_He4_Zn60_to_p_Ga63)*Y(He4)*Y(Zn60)*state.rho) +
        (screened_rates(k_n_Ge63_to_He4_Zn60)*Y(Ge63)*Y(N)*state.rho + -screened_rates(k_He4_Zn60_to_n_Ge63)*Y(He4)*Y(Zn60)*state.rho) +
        (-screened_rates(k_Zn60_to_Cu60)*Y(Zn60) + screened_rates(k_Cu60_to_Zn60)*Y(Cu60));

    ydot_nuc(Zn61) =
        (screened_rates(k_He4_Ni57_to_Zn61)*Y(He4)*Y(Ni57)*state.rho + -screened_rates(k_Zn61_to_He4_Ni57)*Y(Zn61)) +
        (screened_rates(k_p_Cu60_to_Zn61)*Y(Cu60)*Y(H1)*state.rho + -screened_rates(k_Zn61_to_p_Cu60)*Y(Zn61)) +
        (screened_rates(k_n_Zn60_to_Zn61)*Y(Zn60)*Y(N)*state.rho + -screened_rates(k_Zn61_to_n_Zn60)*Y(Zn61)) +
        (-screened_rates(k_n_Zn61_to_Zn62)*Y(Zn61)*Y(N)*state.rho + screened_rates(k_Zn62_to_n_Zn61)*Y(Zn62)) +
        (-screened_rates(k_p_Zn61_to_Ga62)*Y(Zn61)*Y(H1)*state.rho + screened_rates(k_Ga62_to_p_Zn61)*Y(Ga62)) +
        (-screened_rates(k_n_Zn61_to_p_Cu61)*Y(Zn61)*Y(N)*state.rho + screened_rates(k_p_Cu61_to_n_Zn61)*Y(Cu61)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Zn61_to_He4_Ni58)*Y(Zn61)*Y(N)*state.rho + screened_rates(k_He4_Ni58_to_n_Zn61)*Y(He4)*Y(Ni58)*state.rho) +
        (-screened_rates(k_p_Zn61_to_He4_Cu58)*Y(Zn61)*Y(H1)*state.rho + screened_rates(k_He4_Cu58_to_p_Zn61)*Y(Cu58)*Y(He4)*state.rho) +
        (screened_rates(k_p_Ga64_to_He4_Zn61)*Y(Ga64)*Y(H1)*state.rho + -screened_rates(k_He4_Zn61_to_p_Ga64)*Y(He4)*Y(Zn61)*state.rho) +
        (screened_rates(k_n_Ge64_to_He4_Zn61)*Y(Ge64)*Y(N)*state.rho + -screened_rates(k_He4_Zn61_to_n_Ge64)*Y(He4)*Y(Zn61)*state.rho) +
        (-screened_rates(k_Zn61_to_Cu61)*Y(Zn61) + screened_rates(k_Cu61_to_Zn61)*Y(Cu61));

    ydot_nuc(Zn62) =
        (screened_rates(k_He4_Ni58_to_Zn62)*Y(He4)*Y(Ni58)*state.rho + -screened_rates(k_Zn62_to_He4_Ni58)*Y(Zn62)) +
        (screened_rates(k_p_Cu61_to_Zn62)*Y(Cu61)*Y(H1)*state.rho + -screened_rates(k_Zn62_to_p_Cu61)*Y(Zn62)) +
        (screened_rates(k_n_Zn61_to_Zn62)*Y(Zn61)*Y(N)*state.rho + -screened_rates(k_Zn62_to_n_Zn61)*Y(Zn62)) +
        (-screened_rates(k_n_Zn62_to_Zn63)*Y(Zn62)*Y(N)*state.rho + screened_rates(k_Zn63_to_n_Zn62)*Y(Zn63)) +
        (-screened_rates(k_p_Zn62_to_Ga63)*Y(Zn62)*Y(H1)*state.rho + screened_rates(k_Ga63_to_p_Zn62)*Y(Ga63)) +
        (-screened_rates(k_n_Zn62_to_p_Cu62)*Y(Zn62)*Y(N)*state.rho + screened_rates(k_p_Cu62_to_n_Zn62)*Y(Cu62)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Zn62_to_He4_Ni59)*Y(Zn62)*Y(N)*state.rho + screened_rates(k_He4_Ni59_to_n_Zn62)*Y(He4)*Y(Ni59)*state.rho) +
        (-screened_rates(k_p_Zn62_to_He4_Cu59)*Y(Zn62)*Y(H1)*state.rho + screened_rates(k_He4_Cu59_to_p_Zn62)*Y(Cu59)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ga62_to_p_Zn62)*Y(Ga62)*Y(N)*state.rho + -screened_rates(k_p_Zn62_to_n_Ga62)*Y(Zn62)*Y(H1)*state.rho) +
        (screened_rates(k_Ga62_to_Zn62)*Y(Ga62) + -screened_rates(k_Zn62_to_Ga62)*Y(Zn62)) +
        (-screened_rates(k_Zn62_to_Cu62)*Y(Zn62) + screened_rates(k_Cu62_to_Zn62)*Y(Cu62));

    ydot_nuc(Zn63) =
        (screened_rates(k_He4_Ni59_to_Zn63)*Y(He4)*Y(Ni59)*state.rho + -screened_rates(k_Zn63_to_He4_Ni59)*Y(Zn63)) +
        (screened_rates(k_p_Cu62_to_Zn63)*Y(Cu62)*Y(H1)*state.rho + -screened_rates(k_Zn63_to_p_Cu62)*Y(Zn63)) +
        (screened_rates(k_n_Zn62_to_Zn63)*Y(Zn62)*Y(N)*state.rho + -screened_rates(k_Zn63_to_n_Zn62)*Y(Zn63)) +
        (-screened_rates(k_n_Zn63_to_Zn64)*Y(Zn63)*Y(N)*state.rho + screened_rates(k_Zn64_to_n_Zn63)*Y(Zn64)) +
        (-screened_rates(k_p_Zn63_to_Ga64)*Y(Zn63)*Y(H1)*state.rho + screened_rates(k_Ga64_to_p_Zn63)*Y(Ga64)) +
        (-screened_rates(k_n_Zn63_to_p_Cu63)*Y(Zn63)*Y(N)*state.rho + screened_rates(k_p_Cu63_to_n_Zn63)*Y(Cu63)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Zn63_to_He4_Ni60)*Y(Zn63)*Y(N)*state.rho + screened_rates(k_He4_Ni60_to_n_Zn63)*Y(He4)*Y(Ni60)*state.rho) +
        (-screened_rates(k_p_Zn63_to_He4_Cu60)*Y(Zn63)*Y(H1)*state.rho + screened_rates(k_He4_Cu60_to_p_Zn63)*Y(Cu60)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ga63_to_p_Zn63)*Y(Ga63)*Y(N)*state.rho + -screened_rates(k_p_Zn63_to_n_Ga63)*Y(Zn63)*Y(H1)*state.rho) +
        (screened_rates(k_Ga63_to_Zn63)*Y(Ga63) + -screened_rates(k_Zn63_to_Ga63)*Y(Zn63)) +
        (-screened_rates(k_Zn63_to_Cu63)*Y(Zn63) + screened_rates(k_Cu63_to_Zn63)*Y(Cu63));

    ydot_nuc(Zn64) =
        (screened_rates(k_He4_Ni60_to_Zn64)*Y(He4)*Y(Ni60)*state.rho + -screened_rates(k_Zn64_to_He4_Ni60)*Y(Zn64)) +
        (screened_rates(k_p_Cu63_to_Zn64)*Y(Cu63)*Y(H1)*state.rho + -screened_rates(k_Zn64_to_p_Cu63)*Y(Zn64)) +
        (screened_rates(k_n_Zn63_to_Zn64)*Y(Zn63)*Y(N)*state.rho + -screened_rates(k_Zn64_to_n_Zn63)*Y(Zn64)) +
        (-screened_rates(k_n_Zn64_to_Zn65)*Y(Zn64)*Y(N)*state.rho + screened_rates(k_Zn65_to_n_Zn64)*Y(Zn65)) +
        (-screened_rates(k_n_Zn64_to_p_Cu64)*Y(Zn64)*Y(N)*state.rho + screened_rates(k_p_Cu64_to_n_Zn64)*Y(Cu64)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Zn64_to_He4_Ni61)*Y(Zn64)*Y(N)*state.rho + screened_rates(k_He4_Ni61_to_n_Zn64)*Y(He4)*Y(Ni61)*state.rho) +
        (-screened_rates(k_p_Zn64_to_He4_Cu61)*Y(Zn64)*Y(H1)*state.rho + screened_rates(k_He4_Cu61_to_p_Zn64)*Y(Cu61)*Y(He4)*state.rho) +
        (screened_rates(k_n_Ga64_to_p_Zn64)*Y(Ga64)*Y(N)*state.rho + -screened_rates(k_p_Zn64_to_n_Ga64)*Y(Zn64)*Y(H1)*state.rho) +
        (screened_rates(k_Cu64_to_Zn64)*Y(Cu64) + -screened_rates(k_Zn64_to_Cu64)*Y(Zn64)) +
        (screened_rates(k_Ga64_to_Zn64)*Y(Ga64) + -screened_rates(k_Zn64_to_Ga64)*Y(Zn64));

    ydot_nuc(Zn65) =
        (screened_rates(k_He4_Ni61_to_Zn65)*Y(He4)*Y(Ni61)*state.rho + -screened_rates(k_Zn65_to_He4_Ni61)*Y(Zn65)) +
        (screened_rates(k_p_Cu64_to_Zn65)*Y(Cu64)*Y(H1)*state.rho + -screened_rates(k_Zn65_to_p_Cu64)*Y(Zn65)) +
        (screened_rates(k_n_Zn64_to_Zn65)*Y(Zn64)*Y(N)*state.rho + -screened_rates(k_Zn65_to_n_Zn64)*Y(Zn65)) +
        (-screened_rates(k_n_Zn65_to_Zn66)*Y(Zn65)*Y(N)*state.rho + screened_rates(k_Zn66_to_n_Zn65)*Y(Zn66)) +
        (-screened_rates(k_n_Zn65_to_p_Cu65)*Y(Zn65)*Y(N)*state.rho + screened_rates(k_p_Cu65_to_n_Zn65)*Y(Cu65)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Zn65_to_He4_Ni62)*Y(Zn65)*Y(N)*state.rho + screened_rates(k_He4_Ni62_to_n_Zn65)*Y(He4)*Y(Ni62)*state.rho) +
        (-screened_rates(k_p_Zn65_to_He4_Cu62)*Y(Zn65)*Y(H1)*state.rho + screened_rates(k_He4_Cu62_to_p_Zn65)*Y(Cu62)*Y(He4)*state.rho) +
        (-screened_rates(k_Zn65_to_Cu65)*Y(Zn65) + screened_rates(k_Cu65_to_Zn65)*Y(Cu65));

    ydot_nuc(Zn66) =
        (screened_rates(k_He4_Ni62_to_Zn66)*Y(He4)*Y(Ni62)*state.rho + -screened_rates(k_Zn66_to_He4_Ni62)*Y(Zn66)) +
        (screened_rates(k_p_Cu65_to_Zn66)*Y(Cu65)*Y(H1)*state.rho + -screened_rates(k_Zn66_to_p_Cu65)*Y(Zn66)) +
        (screened_rates(k_n_Zn65_to_Zn66)*Y(Zn65)*Y(N)*state.rho + -screened_rates(k_Zn66_to_n_Zn65)*Y(Zn66)) +
        (-screened_rates(k_n_Zn66_to_He4_Ni63)*Y(Zn66)*Y(N)*state.rho + screened_rates(k_He4_Ni63_to_n_Zn66)*Y(He4)*Y(Ni63)*state.rho) +
        (-screened_rates(k_p_Zn66_to_He4_Cu63)*Y(Zn66)*Y(H1)*state.rho + screened_rates(k_He4_Cu63_to_p_Zn66)*Y(Cu63)*Y(He4)*state.rho);

    ydot_nuc(Ga62) =
        (screened_rates(k_He4_Cu58_to_Ga62)*Y(Cu58)*Y(He4)*state.rho + -screened_rates(k_Ga62_to_He4_Cu58)*Y(Ga62)) +
        (screened_rates(k_p_Zn61_to_Ga62)*Y(Zn61)*Y(H1)*state.rho + -screened_rates(k_Ga62_to_p_Zn61)*Y(Ga62)) +
        (-screened_rates(k_n_Ga62_to_Ga63)*Y(Ga62)*Y(N)*state.rho + screened_rates(k_Ga63_to_n_Ga62)*Y(Ga63)) +
        (-screened_rates(k_p_Ga62_to_Ge63)*Y(Ga62)*Y(H1)*state.rho + screened_rates(k_Ge63_to_p_Ga62)*Y(Ge63)) +
        (-screened_rates(k_n_Ga62_to_p_Zn62)*Y(Ga62)*Y(N)*state.rho + screened_rates(k_p_Zn62_to_n_Ga62)*Y(Zn62)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ga62_to_He4_Cu59)*Y(Ga62)*Y(N)*state.rho + screened_rates(k_He4_Cu59_to_n_Ga62)*Y(Cu59)*Y(He4)*state.rho) +
        (-screened_rates(k_p_Ga62_to_He4_Zn59)*Y(Ga62)*Y(H1)*state.rho + screened_rates(k_He4_Zn59_to_p_Ga62)*Y(He4)*Y(Zn59)*state.rho) +
        (-screened_rates(k_Ga62_to_Zn62)*Y(Ga62) + screened_rates(k_Zn62_to_Ga62)*Y(Zn62));

    ydot_nuc(Ga63) =
        screened_rates(k_Ge63_to_Ga63_weak_wc12)*Y(Ge63) +
        (screened_rates(k_He4_Cu59_to_Ga63)*Y(Cu59)*Y(He4)*state.rho + -screened_rates(k_Ga63_to_He4_Cu59)*Y(Ga63)) +
        (screened_rates(k_p_Zn62_to_Ga63)*Y(Zn62)*Y(H1)*state.rho + -screened_rates(k_Ga63_to_p_Zn62)*Y(Ga63)) +
        (screened_rates(k_n_Ga62_to_Ga63)*Y(Ga62)*Y(N)*state.rho + -screened_rates(k_Ga63_to_n_Ga62)*Y(Ga63)) +
        (-screened_rates(k_n_Ga63_to_Ga64)*Y(Ga63)*Y(N)*state.rho + screened_rates(k_Ga64_to_n_Ga63)*Y(Ga64)) +
        (-screened_rates(k_p_Ga63_to_Ge64)*Y(Ga63)*Y(H1)*state.rho + screened_rates(k_Ge64_to_p_Ga63)*Y(Ge64)) +
        (-screened_rates(k_n_Ga63_to_p_Zn63)*Y(Ga63)*Y(N)*state.rho + screened_rates(k_p_Zn63_to_n_Ga63)*Y(Zn63)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ga63_to_He4_Cu60)*Y(Ga63)*Y(N)*state.rho + screened_rates(k_He4_Cu60_to_n_Ga63)*Y(Cu60)*Y(He4)*state.rho) +
        (-screened_rates(k_p_Ga63_to_He4_Zn60)*Y(Ga63)*Y(H1)*state.rho + screened_rates(k_He4_Zn60_to_p_Ga63)*Y(He4)*Y(Zn60)*state.rho) +
        (screened_rates(k_n_Ge63_to_p_Ga63)*Y(Ge63)*Y(N)*state.rho + -screened_rates(k_p_Ga63_to_n_Ge63)*Y(Ga63)*Y(H1)*state.rho) +
        (-screened_rates(k_Ga63_to_Zn63)*Y(Ga63) + screened_rates(k_Zn63_to_Ga63)*Y(Zn63));

    ydot_nuc(Ga64) =
        (screened_rates(k_He4_Cu60_to_Ga64)*Y(Cu60)*Y(He4)*state.rho + -screened_rates(k_Ga64_to_He4_Cu60)*Y(Ga64)) +
        (screened_rates(k_p_Zn63_to_Ga64)*Y(Zn63)*Y(H1)*state.rho + -screened_rates(k_Ga64_to_p_Zn63)*Y(Ga64)) +
        (screened_rates(k_n_Ga63_to_Ga64)*Y(Ga63)*Y(N)*state.rho + -screened_rates(k_Ga64_to_n_Ga63)*Y(Ga64)) +
        (-screened_rates(k_n_Ga64_to_p_Zn64)*Y(Ga64)*Y(N)*state.rho + screened_rates(k_p_Zn64_to_n_Ga64)*Y(Zn64)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ga64_to_He4_Cu61)*Y(Ga64)*Y(N)*state.rho + screened_rates(k_He4_Cu61_to_n_Ga64)*Y(Cu61)*Y(He4)*state.rho) +
        (-screened_rates(k_p_Ga64_to_He4_Zn61)*Y(Ga64)*Y(H1)*state.rho + screened_rates(k_He4_Zn61_to_p_Ga64)*Y(He4)*Y(Zn61)*state.rho) +
        (screened_rates(k_n_Ge64_to_p_Ga64)*Y(Ge64)*Y(N)*state.rho + -screened_rates(k_p_Ga64_to_n_Ge64)*Y(Ga64)*Y(H1)*state.rho) +
        (-screened_rates(k_Ga64_to_Zn64)*Y(Ga64) + screened_rates(k_Zn64_to_Ga64)*Y(Zn64)) +
        (screened_rates(k_Ge64_to_Ga64)*Y(Ge64) + -screened_rates(k_Ga64_to_Ge64)*Y(Ga64));

    ydot_nuc(Ge63) =
        -screened_rates(k_Ge63_to_Ga63_weak_wc12)*Y(Ge63) +
        (screened_rates(k_He4_Zn59_to_Ge63)*Y(He4)*Y(Zn59)*state.rho + -screened_rates(k_Ge63_to_He4_Zn59)*Y(Ge63)) +
        (screened_rates(k_p_Ga62_to_Ge63)*Y(Ga62)*Y(H1)*state.rho + -screened_rates(k_Ge63_to_p_Ga62)*Y(Ge63)) +
        (-screened_rates(k_n_Ge63_to_Ge64)*Y(Ge63)*Y(N)*state.rho + screened_rates(k_Ge64_to_n_Ge63)*Y(Ge64)) +
        (-screened_rates(k_n_Ge63_to_p_Ga63)*Y(Ge63)*Y(N)*state.rho + screened_rates(k_p_Ga63_to_n_Ge63)*Y(Ga63)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ge63_to_He4_Zn60)*Y(Ge63)*Y(N)*state.rho + screened_rates(k_He4_Zn60_to_n_Ge63)*Y(He4)*Y(Zn60)*state.rho);

    ydot_nuc(Ge64) =
        (screened_rates(k_He4_Zn60_to_Ge64)*Y(He4)*Y(Zn60)*state.rho + -screened_rates(k_Ge64_to_He4_Zn60)*Y(Ge64)) +
        (screened_rates(k_p_Ga63_to_Ge64)*Y(Ga63)*Y(H1)*state.rho + -screened_rates(k_Ge64_to_p_Ga63)*Y(Ge64)) +
        (screened_rates(k_n_Ge63_to_Ge64)*Y(Ge63)*Y(N)*state.rho + -screened_rates(k_Ge64_to_n_Ge63)*Y(Ge64)) +
        (-screened_rates(k_n_Ge64_to_p_Ga64)*Y(Ge64)*Y(N)*state.rho + screened_rates(k_p_Ga64_to_n_Ge64)*Y(Ga64)*Y(H1)*state.rho) +
        (-screened_rates(k_n_Ge64_to_He4_Zn61)*Y(Ge64)*Y(N)*state.rho + screened_rates(k_He4_Zn61_to_n_Ge64)*Y(He4)*Y(Zn61)*state.rho) +
        (-screened_rates(k_Ge64_to_Ga64)*Y(Ge64) + screened_rates(k_Ga64_to_Ge64)*Y(Ga64));

}


AMREX_GPU_HOST_DEVICE AMREX_INLINE
void actual_rhs (burn_t& state, amrex::Array1D<amrex::Real, 1, neqs>& ydot)
{
    for (int i = 1; i <= neqs; ++i) {
        ydot(i) = 0.0_rt;
    }


    // Set molar abundances
    amrex::Array1D<amrex::Real, 1, NumSpec> Y;
    for (int i = 1; i <= NumSpec; ++i) {
        Y(i) = state.xn[i-1] * aion_inv[i-1];
    }

    // build the rates

    rate_t rate_eval;

    constexpr int do_T_derivatives = 0;

    evaluate_rates<do_T_derivatives, rate_t>(state, rate_eval);

    rhs_nuc(state, ydot, Y, rate_eval.screened_rates);

    // ion binding energy contributions

    amrex::Real enuc;
    ener_gener_rate(ydot, enuc);

    // include any weak rate neutrino losses
    enuc += rate_eval.enuc_weak;

    // Get the thermal neutrino losses

    amrex::Real sneut{};
#ifdef NEUTRINOS
    constexpr int do_derivatives{0};
    amrex::Real dsneutdt{}, dsneutdd{}, dsnuda{}, dsnudz{};

    neutrino_cooling<do_derivatives>(state.T, state.rho, state.abar, state.zbar,
                                     sneut, dsneutdt, dsneutdd, dsnuda, dsnudz);
#endif

    // Append the energy equation (this is erg/g/s)

    ydot(net_ienuc) = enuc - sneut;

}


template<class MatrixType>
AMREX_GPU_HOST_DEVICE AMREX_INLINE
void jac_nuc(const burn_t& state,
             MatrixType& jac,
             const amrex::Array1D<amrex::Real, 1, NumSpec>& Y,
             const amrex::Array1D<amrex::Real, 1, NumRates>& screened_rates)
{

    amrex::Real scratch;

    scratch = -screened_rates(k_n_Al25_to_Al26)*Y(Al25)*state.rho - screened_rates(k_n_Al25_to_He4_Na22)*Y(Al25)*state.rho - screened_rates(k_n_Al25_to_p_Mg25)*Y(Al25)*state.rho - screened_rates(k_n_Al26_to_Al27)*Y(Al26)*state.rho - screened_rates(k_n_Al26_to_He4_Na23)*Y(Al26)*state.rho - screened_rates(k_n_Al26_to_p_Mg26)*Y(Al26)*state.rho - screened_rates(k_n_Ar36_to_Ar37)*Y(Ar36)*state.rho - screened_rates(k_n_Ar36_to_He4_S33)*Y(Ar36)*state.rho - screened_rates(k_n_Ar36_to_p_Cl36)*Y(Ar36)*state.rho - screened_rates(k_n_Ar37_to_Ar38)*Y(Ar37)*state.rho - screened_rates(k_n_Ar37_to_He4_S34)*Y(Ar37)*state.rho - screened_rates(k_n_Ar37_to_p_Cl37)*Y(Ar37)*state.rho - screened_rates(k_n_Ar38_to_Ar39)*Y(Ar38)*state.rho - screened_rates(k_n_Ar38_to_He4_S35)*Y(Ar38)*state.rho - screened_rates(k_n_Ar39_to_Ar40)*Y(Ar39)*state.rho - screened_rates(k_n_Ar39_to_He4_S36)*Y(Ar39)*state.rho - screened_rates(k_n_B10_to_B11)*Y(B10)*state.rho - screened_rates(k_n_B10_to_He4_Li7)*Y(B10)*state.rho - screened_rates(k_n_B8_to_p_He4_He4)*Y(B8)*state.rho - screened_rates(k_n_Be7_to_He4_He4)*Y(Be7)*state.rho - screened_rates(k_n_Be7_to_d_Li6)*Y(Be7)*state.rho - screened_rates(k_n_Be7_to_p_Li7)*Y(Be7)*state.rho - screened_rates(k_n_C12_to_C13)*Y(C12)*state.rho - screened_rates(k_n_C12_to_He4_Be9)*Y(C12)*state.rho - screened_rates(k_n_C13_to_C14)*Y(C13)*state.rho - screened_rates(k_n_Ca40_to_Ca41)*Y(Ca40)*state.rho - screened_rates(k_n_Ca40_to_He4_Ar37)*Y(Ca40)*state.rho - screened_rates(k_n_Ca40_to_p_K40)*Y(Ca40)*state.rho - screened_rates(k_n_Ca41_to_Ca42)*Y(Ca41)*state.rho - screened_rates(k_n_Ca41_to_He4_Ar38)*Y(Ca41)*state.rho - screened_rates(k_n_Ca41_to_p_K41)*Y(Ca41)*state.rho - screened_rates(k_n_Ca42_to_Ca43)*Y(Ca42)*state.rho - screened_rates(k_n_Ca42_to_He4_Ar39)*Y(Ca42)*state.rho - screened_rates(k_n_Ca43_to_Ca44)*Y(Ca43)*state.rho - screened_rates(k_n_Ca43_to_He4_Ar40)*Y(Ca43)*state.rho - screened_rates(k_n_Ca44_to_Ca45)*Y(Ca44)*state.rho - screened_rates(k_n_Ca45_to_Ca46)*Y(Ca45)*state.rho - screened_rates(k_n_Ca46_to_Ca47)*Y(Ca46)*state.rho - screened_rates(k_n_Ca47_to_Ca48)*Y(Ca47)*state.rho - screened_rates(k_n_Cl33_to_Cl34)*Y(Cl33)*state.rho - screened_rates(k_n_Cl33_to_He4_P30)*Y(Cl33)*state.rho - screened_rates(k_n_Cl33_to_p_S33)*Y(Cl33)*state.rho - screened_rates(k_n_Cl34_to_Cl35)*Y(Cl34)*state.rho - screened_rates(k_n_Cl34_to_He4_P31)*Y(Cl34)*state.rho - screened_rates(k_n_Cl34_to_p_S34)*Y(Cl34)*state.rho - screened_rates(k_n_Cl35_to_Cl36)*Y(Cl35)*state.rho - screened_rates(k_n_Cl35_to_He4_P32)*Y(Cl35)*state.rho - screened_rates(k_n_Cl35_to_p_S35)*Y(Cl35)*state.rho - screened_rates(k_n_Cl36_to_Cl37)*Y(Cl36)*state.rho - screened_rates(k_n_Cl36_to_He4_P33)*Y(Cl36)*state.rho - screened_rates(k_n_Cl36_to_p_S36)*Y(Cl36)*state.rho - screened_rates(k_n_Co53_to_Co54)*Y(Co53)*state.rho - screened_rates(k_n_Co53_to_He4_Mn50)*Y(Co53)*state.rho - screened_rates(k_n_Co53_to_p_Fe53)*Y(Co53)*state.rho - screened_rates(k_n_Co54_to_Co55)*Y(Co54)*state.rho - screened_rates(k_n_Co54_to_He4_Mn51)*Y(Co54)*state.rho - screened_rates(k_n_Co54_to_p_Fe54)*Y(Co54)*state.rho - screened_rates(k_n_Co55_to_Co56)*Y(Co55)*state.rho - screened_rates(k_n_Co55_to_He4_Mn52)*Y(Co55)*state.rho - screened_rates(k_n_Co55_to_p_Fe55)*Y(Co55)*state.rho - screened_rates(k_n_Co56_to_Co57)*Y(Co56)*state.rho - screened_rates(k_n_Co56_to_He4_Mn53)*Y(Co56)*state.rho - screened_rates(k_n_Co56_to_p_Fe56)*Y(Co56)*state.rho - screened_rates(k_n_Co57_to_Co58)*Y(Co57)*state.rho - screened_rates(k_n_Co57_to_He4_Mn54)*Y(Co57)*state.rho - screened_rates(k_n_Co57_to_p_Fe57)*Y(Co57)*state.rho - screened_rates(k_n_Co58_to_Co59)*Y(Co58)*state.rho - screened_rates(k_n_Co58_to_He4_Mn55)*Y(Co58)*state.rho - screened_rates(k_n_Co58_to_p_Fe58)*Y(Co58)*state.rho - screened_rates(k_n_Cr48_to_Cr49)*Y(Cr48)*state.rho - screened_rates(k_n_Cr48_to_He4_Ti45)*Y(Cr48)*state.rho - screened_rates(k_n_Cr48_to_p_V48)*Y(Cr48)*state.rho - screened_rates(k_n_Cr49_to_Cr50)*Y(Cr49)*state.rho - screened_rates(k_n_Cr49_to_He4_Ti46)*Y(Cr49)*state.rho - screened_rates(k_n_Cr49_to_p_V49)*Y(Cr49)*state.rho - screened_rates(k_n_Cr50_to_Cr51)*Y(Cr50)*state.rho - screened_rates(k_n_Cr50_to_He4_Ti47)*Y(Cr50)*state.rho - screened_rates(k_n_Cr50_to_p_V50)*Y(Cr50)*state.rho - screened_rates(k_n_Cr51_to_Cr52)*Y(Cr51)*state.rho - screened_rates(k_n_Cr51_to_He4_Ti48)*Y(Cr51)*state.rho - screened_rates(k_n_Cr51_to_p_V51)*Y(Cr51)*state.rho - screened_rates(k_n_Cr52_to_Cr53)*Y(Cr52)*state.rho - screened_rates(k_n_Cr52_to_He4_Ti49)*Y(Cr52)*state.rho - screened_rates(k_n_Cr52_to_p_V52)*Y(Cr52)*state.rho - screened_rates(k_n_Cr53_to_Cr54)*Y(Cr53)*state.rho - screened_rates(k_n_Cr53_to_He4_Ti50)*Y(Cr53)*state.rho - screened_rates(k_n_Cr54_to_He4_Ti51)*Y(Cr54)*state.rho - screened_rates(k_n_Cu57_to_Cu58)*Y(Cu57)*state.rho - screened_rates(k_n_Cu57_to_He4_Co54)*Y(Cu57)*state.rho - screened_rates(k_n_Cu57_to_p_Ni57)*Y(Cu57)*state.rho - screened_rates(k_n_Cu58_to_Cu59)*Y(Cu58)*state.rho - screened_rates(k_n_Cu58_to_He4_Co55)*Y(Cu58)*state.rho - screened_rates(k_n_Cu58_to_p_Ni58)*Y(Cu58)*state.rho - screened_rates(k_n_Cu59_to_Cu60)*Y(Cu59)*state.rho - screened_rates(k_n_Cu59_to_He4_Co56)*Y(Cu59)*state.rho - screened_rates(k_n_Cu59_to_p_Ni59)*Y(Cu59)*state.rho - screened_rates(k_n_Cu60_to_Cu61)*Y(Cu60)*state.rho - screened_rates(k_n_Cu60_to_He4_Co57)*Y(Cu60)*state.rho - screened_rates(k_n_Cu60_to_p_Ni60)*Y(Cu60)*state.rho - screened_rates(k_n_Cu61_to_Cu62)*Y(Cu61)*state.rho - screened_rates(k_n_Cu61_to_He4_Co58)*Y(Cu61)*state.rho - screened_rates(k_n_Cu61_to_p_Ni61)*Y(Cu61)*state.rho - screened_rates(k_n_Cu62_to_Cu63)*Y(Cu62)*state.rho - screened_rates(k_n_Cu62_to_He4_Co59)*Y(Cu62)*state.rho - screened_rates(k_n_Cu62_to_p_Ni62)*Y(Cu62)*state.rho - screened_rates(k_n_Cu63_to_Cu64)*Y(Cu63)*state.rho - screened_rates(k_n_Cu63_to_p_Ni63)*Y(Cu63)*state.rho - screened_rates(k_n_Cu64_to_Cu65)*Y(Cu64)*state.rho - screened_rates(k_n_Cu64_to_p_Ni64)*Y(Cu64)*state.rho - screened_rates(k_n_F17_to_F18)*Y(F17)*state.rho - screened_rates(k_n_F17_to_He4_N14)*Y(F17)*state.rho - screened_rates(k_n_F17_to_p_O17)*Y(F17)*state.rho - screened_rates(k_n_F18_to_F19)*Y(F18)*state.rho - screened_rates(k_n_F18_to_He4_N15)*Y(F18)*state.rho - screened_rates(k_n_F18_to_p_O18)*Y(F18)*state.rho - screened_rates(k_n_Fe52_to_Fe53)*Y(Fe52)*state.rho - screened_rates(k_n_Fe52_to_He4_Cr49)*Y(Fe52)*state.rho - screened_rates(k_n_Fe52_to_p_Mn52)*Y(Fe52)*state.rho - screened_rates(k_n_Fe53_to_Fe54)*Y(Fe53)*state.rho - screened_rates(k_n_Fe53_to_He4_Cr50)*Y(Fe53)*state.rho - screened_rates(k_n_Fe53_to_p_Mn53)*Y(Fe53)*state.rho - screened_rates(k_n_Fe54_to_Fe55)*Y(Fe54)*state.rho - screened_rates(k_n_Fe54_to_He4_Cr51)*Y(Fe54)*state.rho - screened_rates(k_n_Fe54_to_p_Mn54)*Y(Fe54)*state.rho - screened_rates(k_n_Fe55_to_Fe56)*Y(Fe55)*state.rho - screened_rates(k_n_Fe55_to_He4_Cr52)*Y(Fe55)*state.rho - screened_rates(k_n_Fe55_to_p_Mn55)*Y(Fe55)*state.rho - screened_rates(k_n_Fe56_to_Fe57)*Y(Fe56)*state.rho - screened_rates(k_n_Fe56_to_He4_Cr53)*Y(Fe56)*state.rho - screened_rates(k_n_Fe57_to_Fe58)*Y(Fe57)*state.rho - screened_rates(k_n_Fe57_to_He4_Cr54)*Y(Fe57)*state.rho - screened_rates(k_n_Ga62_to_Ga63)*Y(Ga62)*state.rho - screened_rates(k_n_Ga62_to_He4_Cu59)*Y(Ga62)*state.rho - screened_rates(k_n_Ga62_to_p_Zn62)*Y(Ga62)*state.rho - screened_rates(k_n_Ga63_to_Ga64)*Y(Ga63)*state.rho - screened_rates(k_n_Ga63_to_He4_Cu60)*Y(Ga63)*state.rho - screened_rates(k_n_Ga63_to_p_Zn63)*Y(Ga63)*state.rho - screened_rates(k_n_Ga64_to_He4_Cu61)*Y(Ga64)*state.rho - screened_rates(k_n_Ga64_to_p_Zn64)*Y(Ga64)*state.rho - screened_rates(k_n_Ge63_to_Ge64)*Y(Ge63)*state.rho - screened_rates(k_n_Ge63_to_He4_Zn60)*Y(Ge63)*state.rho - screened_rates(k_n_Ge63_to_p_Ga63)*Y(Ge63)*state.rho - screened_rates(k_n_Ge64_to_He4_Zn61)*Y(Ge64)*state.rho - screened_rates(k_n_Ge64_to_p_Ga64)*Y(Ge64)*state.rho - screened_rates(k_n_He3_to_He4)*Y(He3)*state.rho - screened_rates(k_n_He3_to_d_d)*Y(He3)*state.rho - 0.5*screened_rates(k_n_He4_He4_to_Be9)*amrex::Math::powi<2>(Y(He4))*amrex::Math::powi<2>(state.rho) - 0.5*screened_rates(k_n_He4_He4_to_d_Li7)*amrex::Math::powi<2>(Y(He4))*amrex::Math::powi<2>(state.rho) - screened_rates(k_n_K37_to_He4_Cl34)*Y(K37)*state.rho - screened_rates(k_n_K37_to_K38)*Y(K37)*state.rho - screened_rates(k_n_K37_to_p_Ar37)*Y(K37)*state.rho - screened_rates(k_n_K38_to_He4_Cl35)*Y(K38)*state.rho - screened_rates(k_n_K38_to_K39)*Y(K38)*state.rho - screened_rates(k_n_K38_to_p_Ar38)*Y(K38)*state.rho - screened_rates(k_n_K39_to_He4_Cl36)*Y(K39)*state.rho - screened_rates(k_n_K39_to_K40)*Y(K39)*state.rho - screened_rates(k_n_K39_to_p_Ar39)*Y(K39)*state.rho - screened_rates(k_n_K40_to_He4_Cl37)*Y(K40)*state.rho - screened_rates(k_n_K40_to_K41)*Y(K40)*state.rho - screened_rates(k_n_K40_to_p_Ar40)*Y(K40)*state.rho - screened_rates(k_n_Li6_to_Li7)*Y(Li6)*state.rho - screened_rates(k_n_Mg23_to_C12_C12)*Y(Mg23)*state.rho - screened_rates(k_n_Mg23_to_He4_Ne20)*Y(Mg23)*state.rho - screened_rates(k_n_Mg23_to_Mg24)*Y(Mg23)*state.rho - screened_rates(k_n_Mg23_to_p_Na23)*Y(Mg23)*state.rho - screened_rates(k_n_Mg24_to_He4_Ne21)*Y(Mg24)*state.rho - screened_rates(k_n_Mg24_to_Mg25)*Y(Mg24)*state.rho - screened_rates(k_n_Mg25_to_He4_Ne22)*Y(Mg25)*state.rho - screened_rates(k_n_Mg25_to_Mg26)*Y(Mg25)*state.rho - screened_rates(k_n_Mn50_to_He4_V47)*Y(Mn50)*state.rho - screened_rates(k_n_Mn50_to_Mn51)*Y(Mn50)*state.rho - screened_rates(k_n_Mn50_to_p_Cr50)*Y(Mn50)*state.rho - screened_rates(k_n_Mn51_to_He4_V48)*Y(Mn51)*state.rho - screened_rates(k_n_Mn51_to_Mn52)*Y(Mn51)*state.rho - screened_rates(k_n_Mn51_to_p_Cr51)*Y(Mn51)*state.rho - screened_rates(k_n_Mn52_to_He4_V49)*Y(Mn52)*state.rho - screened_rates(k_n_Mn52_to_Mn53)*Y(Mn52)*state.rho - screened_rates(k_n_Mn52_to_p_Cr52)*Y(Mn52)*state.rho - screened_rates(k_n_Mn53_to_He4_V50)*Y(Mn53)*state.rho - screened_rates(k_n_Mn53_to_Mn54)*Y(Mn53)*state.rho - screened_rates(k_n_Mn53_to_p_Cr53)*Y(Mn53)*state.rho - screened_rates(k_n_Mn54_to_He4_V51)*Y(Mn54)*state.rho - screened_rates(k_n_Mn54_to_Mn55)*Y(Mn54)*state.rho - screened_rates(k_n_Mn54_to_p_Cr54)*Y(Mn54)*state.rho - screened_rates(k_n_Mn55_to_He4_V52)*Y(Mn55)*state.rho - screened_rates(k_n_N13_to_He4_B10)*Y(N13)*state.rho - screened_rates(k_n_N13_to_N14)*Y(N13)*state.rho - screened_rates(k_n_N13_to_p_C13)*Y(N13)*state.rho - screened_rates(k_n_N14_to_He4_B11)*Y(N14)*state.rho - screened_rates(k_n_N14_to_N15)*Y(N14)*state.rho - screened_rates(k_n_N14_to_d_C13)*Y(N14)*state.rho - screened_rates(k_n_N14_to_p_C14)*Y(N14)*state.rho - screened_rates(k_n_N15_to_d_C14)*Y(N15)*state.rho - screened_rates(k_n_Na21_to_He4_F18)*Y(Na21)*state.rho - screened_rates(k_n_Na21_to_Na22)*Y(Na21)*state.rho - screened_rates(k_n_Na21_to_p_Ne21)*Y(Na21)*state.rho - screened_rates(k_n_Na22_to_He4_F19)*Y(Na22)*state.rho - screened_rates(k_n_Na22_to_Na23)*Y(Na22)*state.rho - screened_rates(k_n_Na22_to_p_Ne22)*Y(Na22)*state.rho - screened_rates(k_n_Ne18_to_He4_O15)*Y(Ne18)*state.rho - screened_rates(k_n_Ne18_to_Ne19)*Y(Ne18)*state.rho - screened_rates(k_n_Ne18_to_p_F18)*Y(Ne18)*state.rho - screened_rates(k_n_Ne19_to_He4_O16)*Y(Ne19)*state.rho - screened_rates(k_n_Ne19_to_Ne20)*Y(Ne19)*state.rho - screened_rates(k_n_Ne19_to_p_F19)*Y(Ne19)*state.rho - screened_rates(k_n_Ne20_to_He4_O17)*Y(Ne20)*state.rho - screened_rates(k_n_Ne20_to_Ne21)*Y(Ne20)*state.rho - screened_rates(k_n_Ne21_to_He4_O18)*Y(Ne21)*state.rho - screened_rates(k_n_Ne21_to_Ne22)*Y(Ne21)*state.rho - screened_rates(k_n_Ni56_to_He4_Fe53)*Y(Ni56)*state.rho - screened_rates(k_n_Ni56_to_Ni57)*Y(Ni56)*state.rho - screened_rates(k_n_Ni56_to_p_Co56)*Y(Ni56)*state.rho - screened_rates(k_n_Ni57_to_He4_Fe54)*Y(Ni57)*state.rho - screened_rates(k_n_Ni57_to_Ni58)*Y(Ni57)*state.rho - screened_rates(k_n_Ni57_to_p_Co57)*Y(Ni57)*state.rho - screened_rates(k_n_Ni58_to_He4_Fe55)*Y(Ni58)*state.rho - screened_rates(k_n_Ni58_to_Ni59)*Y(Ni58)*state.rho - screened_rates(k_n_Ni58_to_p_Co58)*Y(Ni58)*state.rho - screened_rates(k_n_Ni59_to_He4_Fe56)*Y(Ni59)*state.rho - screened_rates(k_n_Ni59_to_Ni60)*Y(Ni59)*state.rho - screened_rates(k_n_Ni59_to_p_Co59)*Y(Ni59)*state.rho - screened_rates(k_n_Ni60_to_He4_Fe57)*Y(Ni60)*state.rho - screened_rates(k_n_Ni60_to_Ni61)*Y(Ni60)*state.rho - screened_rates(k_n_Ni61_to_He4_Fe58)*Y(Ni61)*state.rho - screened_rates(k_n_Ni61_to_Ni62)*Y(Ni61)*state.rho - screened_rates(k_n_Ni62_to_Ni63)*Y(Ni62)*state.rho - screened_rates(k_n_Ni63_to_Ni64)*Y(Ni63)*state.rho - screened_rates(k_n_O14_to_O15)*Y(O14)*state.rho - screened_rates(k_n_O14_to_p_N14)*Y(O14)*state.rho - screened_rates(k_n_O15_to_He4_C12)*Y(O15)*state.rho - screened_rates(k_n_O15_to_O16)*Y(O15)*state.rho - screened_rates(k_n_O15_to_p_N15)*Y(O15)*state.rho - screened_rates(k_n_O16_to_He4_C13)*Y(O16)*state.rho - screened_rates(k_n_O16_to_O17)*Y(O16)*state.rho - screened_rates(k_n_O17_to_He4_C14)*Y(O17)*state.rho - screened_rates(k_n_O17_to_O18)*Y(O17)*state.rho - screened_rates(k_n_P29_to_He4_Al26)*Y(P29)*state.rho - screened_rates(k_n_P29_to_P30)*Y(P29)*state.rho - screened_rates(k_n_P29_to_p_Si29)*Y(P29)*state.rho - screened_rates(k_n_P30_to_He4_Al27)*Y(P30)*state.rho - screened_rates(k_n_P30_to_P31)*Y(P30)*state.rho - screened_rates(k_n_P30_to_p_Si30)*Y(P30)*state.rho - screened_rates(k_n_P31_to_P32)*Y(P31)*state.rho - screened_rates(k_n_P31_to_p_Si31)*Y(P31)*state.rho - screened_rates(k_n_P32_to_P33)*Y(P32)*state.rho - screened_rates(k_n_P32_to_p_Si32)*Y(P32)*state.rho - screened_rates(k_n_S32_to_He4_Si29)*Y(S32)*state.rho - screened_rates(k_n_S32_to_S33)*Y(S32)*state.rho - screened_rates(k_n_S32_to_p_P32)*Y(S32)*state.rho - screened_rates(k_n_S33_to_He4_Si30)*Y(S33)*state.rho - screened_rates(k_n_S33_to_S34)*Y(S33)*state.rho - screened_rates(k_n_S33_to_p_P33)*Y(S33)*state.rho - screened_rates(k_n_S34_to_He4_Si31)*Y(S34)*state.rho - screened_rates(k_n_S34_to_S35)*Y(S34)*state.rho - screened_rates(k_n_S35_to_He4_Si32)*Y(S35)*state.rho - screened_rates(k_n_S35_to_S36)*Y(S35)*state.rho - screened_rates(k_n_Sc43_to_He4_K40)*Y(Sc43)*state.rho - screened_rates(k_n_Sc43_to_Sc44)*Y(Sc43)*state.rho - screened_rates(k_n_Sc43_to_p_Ca43)*Y(Sc43)*state.rho - screened_rates(k_n_Sc44_to_He4_K41)*Y(Sc44)*state.rho - screened_rates(k_n_Sc44_to_Sc45)*Y(Sc44)*state.rho - screened_rates(k_n_Sc44_to_p_Ca44)*Y(Sc44)*state.rho - screened_rates(k_n_Sc45_to_Sc46)*Y(Sc45)*state.rho - screened_rates(k_n_Sc45_to_p_Ca45)*Y(Sc45)*state.rho - screened_rates(k_n_Sc46_to_Sc47)*Y(Sc46)*state.rho - screened_rates(k_n_Sc46_to_p_Ca46)*Y(Sc46)*state.rho - screened_rates(k_n_Sc47_to_Sc48)*Y(Sc47)*state.rho - screened_rates(k_n_Sc47_to_p_Ca47)*Y(Sc47)*state.rho - screened_rates(k_n_Sc48_to_Sc49)*Y(Sc48)*state.rho - screened_rates(k_n_Sc48_to_p_Ca48)*Y(Sc48)*state.rho - screened_rates(k_n_Si28_to_He4_Mg25)*Y(Si28)*state.rho - screened_rates(k_n_Si28_to_Si29)*Y(Si28)*state.rho - screened_rates(k_n_Si29_to_He4_Mg26)*Y(Si29)*state.rho - screened_rates(k_n_Si29_to_Si30)*Y(Si29)*state.rho - screened_rates(k_n_Si30_to_Si31)*Y(Si30)*state.rho - screened_rates(k_n_Si31_to_Si32)*Y(Si31)*state.rho - screened_rates(k_n_Ti44_to_He4_Ca41)*Y(Ti44)*state.rho - screened_rates(k_n_Ti44_to_Ti45)*Y(Ti44)*state.rho - screened_rates(k_n_Ti44_to_p_Sc44)*Y(Ti44)*state.rho - screened_rates(k_n_Ti45_to_He4_Ca42)*Y(Ti45)*state.rho - screened_rates(k_n_Ti45_to_Ti46)*Y(Ti45)*state.rho - screened_rates(k_n_Ti45_to_p_Sc45)*Y(Ti45)*state.rho - screened_rates(k_n_Ti46_to_He4_Ca43)*Y(Ti46)*state.rho - screened_rates(k_n_Ti46_to_Ti47)*Y(Ti46)*state.rho - screened_rates(k_n_Ti46_to_p_Sc46)*Y(Ti46)*state.rho - screened_rates(k_n_Ti47_to_He4_Ca44)*Y(Ti47)*state.rho - screened_rates(k_n_Ti47_to_Ti48)*Y(Ti47)*state.rho - screened_rates(k_n_Ti47_to_p_Sc47)*Y(Ti47)*state.rho - screened_rates(k_n_Ti48_to_He4_Ca45)*Y(Ti48)*state.rho - screened_rates(k_n_Ti48_to_Ti49)*Y(Ti48)*state.rho - screened_rates(k_n_Ti48_to_p_Sc48)*Y(Ti48)*state.rho - screened_rates(k_n_Ti49_to_He4_Ca46)*Y(Ti49)*state.rho - screened_rates(k_n_Ti49_to_Ti50)*Y(Ti49)*state.rho - screened_rates(k_n_Ti49_to_p_Sc49)*Y(Ti49)*state.rho - screened_rates(k_n_Ti50_to_He4_Ca47)*Y(Ti50)*state.rho - screened_rates(k_n_Ti50_to_Ti51)*Y(Ti50)*state.rho - screened_rates(k_n_Ti51_to_He4_Ca48)*Y(Ti51)*state.rho - screened_rates(k_n_V46_to_He4_Sc43)*Y(V46)*state.rho - screened_rates(k_n_V46_to_V47)*Y(V46)*state.rho - screened_rates(k_n_V46_to_p_Ti46)*Y(V46)*state.rho - screened_rates(k_n_V47_to_He4_Sc44)*Y(V47)*state.rho - screened_rates(k_n_V47_to_V48)*Y(V47)*state.rho - screened_rates(k_n_V47_to_p_Ti47)*Y(V47)*state.rho - screened_rates(k_n_V48_to_He4_Sc45)*Y(V48)*state.rho - screened_rates(k_n_V48_to_V49)*Y(V48)*state.rho - screened_rates(k_n_V48_to_p_Ti48)*Y(V48)*state.rho - screened_rates(k_n_V49_to_He4_Sc46)*Y(V49)*state.rho - screened_rates(k_n_V49_to_V50)*Y(V49)*state.rho - screened_rates(k_n_V49_to_p_Ti49)*Y(V49)*state.rho - screened_rates(k_n_V50_to_He4_Sc47)*Y(V50)*state.rho - screened_rates(k_n_V50_to_V51)*Y(V50)*state.rho - screened_rates(k_n_V50_to_p_Ti50)*Y(V50)*state.rho - screened_rates(k_n_V51_to_He4_Sc48)*Y(V51)*state.rho - screened_rates(k_n_V51_to_V52)*Y(V51)*state.rho - screened_rates(k_n_V51_to_p_Ti51)*Y(V51)*state.rho - screened_rates(k_n_V52_to_He4_Sc49)*Y(V52)*state.rho - screened_rates(k_n_Zn59_to_He4_Ni56)*Y(Zn59)*state.rho - screened_rates(k_n_Zn59_to_Zn60)*Y(Zn59)*state.rho - screened_rates(k_n_Zn59_to_p_Cu59)*Y(Zn59)*state.rho - screened_rates(k_n_Zn60_to_He4_Ni57)*Y(Zn60)*state.rho - screened_rates(k_n_Zn60_to_Zn61)*Y(Zn60)*state.rho - screened_rates(k_n_Zn60_to_p_Cu60)*Y(Zn60)*state.rho - screened_rates(k_n_Zn61_to_He4_Ni58)*Y(Zn61)*state.rho - screened_rates(k_n_Zn61_to_Zn62)*Y(Zn61)*state.rho - screened_rates(k_n_Zn61_to_p_Cu61)*Y(Zn61)*state.rho - screened_rates(k_n_Zn62_to_He4_Ni59)*Y(Zn62)*state.rho - screened_rates(k_n_Zn62_to_Zn63)*Y(Zn62)*state.rho - screened_rates(k_n_Zn62_to_p_Cu62)*Y(Zn62)*state.rho - screened_rates(k_n_Zn63_to_He4_Ni60)*Y(Zn63)*state.rho - screened_rates(k_n_Zn63_to_Zn64)*Y(Zn63)*state.rho - screened_rates(k_n_Zn63_to_p_Cu63)*Y(Zn63)*state.rho - screened_rates(k_n_Zn64_to_He4_Ni61)*Y(Zn64)*state.rho - screened_rates(k_n_Zn64_to_Zn65)*Y(Zn64)*state.rho - screened_rates(k_n_Zn64_to_p_Cu64)*Y(Zn64)*state.rho - screened_rates(k_n_Zn65_to_He4_Ni62)*Y(Zn65)*state.rho - screened_rates(k_n_Zn65_to_Zn66)*Y(Zn65)*state.rho - screened_rates(k_n_Zn65_to_p_Cu65)*Y(Zn65)*state.rho - screened_rates(k_n_Zn66_to_He4_Ni63)*Y(Zn66)*state.rho - 0.5*screened_rates(k_n_p_He4_He4_to_He3_Li7)*amrex::Math::powi<2>(Y(He4))*Y(H1)*amrex::Math::powi<3>(state.rho) - 0.5*screened_rates(k_n_p_He4_He4_to_p_Be9)*amrex::Math::powi<2>(Y(He4))*Y(H1)*amrex::Math::powi<3>(state.rho) - screened_rates(k_n_p_He4_to_Li6)*Y(He4)*Y(H1)*amrex::Math::powi<2>(state.rho) - 0.5*screened_rates(k_n_p_p_to_p_d)*amrex::Math::powi<2>(Y(H1))*amrex::Math::powi<2>(state.rho) - screened_rates(k_n_p_to_d)*Y(H1)*state.rho - screened_rates(k_n_to_p);
    jac.set(N, N, scratch);

    scratch = -0.5*screened_rates(k_n_p_He4_He4_to_He3_Li7)*amrex::Math::powi<2>(Y(He4))*Y(N)*amrex::Math::powi<3>(state.rho) - 0.5*screened_rates(k_n_p_He4_He4_to_p_Be9)*amrex::Math::powi<2>(Y(He4))*Y(N)*amrex::Math::powi<3>(state.rho) - screened_rates(k_n_p_He4_to_Li6)*Y(He4)*Y(N)*amrex::Math::powi<2>(state.rho) - 1.0*screened_rates(k_n_p_p_to_p_d)*Y(N)*Y(H1)*amrex::Math::powi<2>(state.rho) - screened_rates(k_n_p_to_d)*Y(N)*state.rho + screened_rates(k_p_Ar37_to_n_K37)*Y(Ar37)*state.rho + screened_rates(k_p_Ar38_to_n_K38)*Y(Ar38)*state.rho + screened_rates(k_p_Ar39_to_n_K39)*Y(Ar39)*state.rho + screened_rates(k_p_Ar40_to_n_K40)*Y(Ar40)*state.rho + screened_rates(k_p_Be9_to_n_p_He4_He4)*Y(Be9)*state.rho + screened_rates(k_p_C13_to_n_N13)*Y(C13)*state.rho + screened_rates(k_p_C14_to_n_N14)*Y(C14)*state.rho + screened_rates(k_p_Ca43_to_n_Sc43)*Y(Ca43)*state.rho + screened_rates(k_p_Ca44_to_n_Sc44)*Y(Ca44)*state.rho + screened_rates(k_p_Ca45_to_n_Sc45)*Y(Ca45)*state.rho + screened_rates(k_p_Ca46_to_n_Sc46)*Y(Ca46)*state.rho + screened_rates(k_p_Ca47_to_n_Sc47)*Y(Ca47)*state.rho + screened_rates(k_p_Ca48_to_n_Sc48)*Y(Ca48)*state.rho + screened_rates(k_p_Cl36_to_n_Ar36)*Y(Cl36)*state.rho + screened_rates(k_p_Cl37_to_n_Ar37)*Y(Cl37)*state.rho + screened_rates(k_p_Co56_to_n_Ni56)*Y(Co56)*state.rho + screened_rates(k_p_Co57_to_n_Ni57)*Y(Co57)*state.rho + screened_rates(k_p_Co58_to_n_Ni58)*Y(Co58)*state.rho + screened_rates(k_p_Co59_to_n_Ni59)*Y(Co59)*state.rho + screened_rates(k_p_Cr50_to_n_Mn50)*Y(Cr50)*state.rho + screened_rates(k_p_Cr51_to_n_Mn51)*Y(Cr51)*state.rho + screened_rates(k_p_Cr52_to_n_Mn52)*Y(Cr52)*state.rho + screened_rates(k_p_Cr53_to_n_Mn53)*Y(Cr53)*state.rho + screened_rates(k_p_Cr54_to_n_Mn54)*Y(Cr54)*state.rho + screened_rates(k_p_Cu59_to_n_Zn59)*Y(Cu59)*state.rho + screened_rates(k_p_Cu60_to_n_Zn60)*Y(Cu60)*state.rho + screened_rates(k_p_Cu61_to_n_Zn61)*Y(Cu61)*state.rho + screened_rates(k_p_Cu62_to_n_Zn62)*Y(Cu62)*state.rho + screened_rates(k_p_Cu63_to_n_Zn63)*Y(Cu63)*state.rho + screened_rates(k_p_Cu64_to_n_Zn64)*Y(Cu64)*state.rho + screened_rates(k_p_Cu65_to_n_Zn65)*Y(Cu65)*state.rho + screened_rates(k_p_F18_to_n_Ne18)*Y(F18)*state.rho + screened_rates(k_p_F19_to_n_Ne19)*Y(F19)*state.rho + screened_rates(k_p_Fe53_to_n_Co53)*Y(Fe53)*state.rho + screened_rates(k_p_Fe54_to_n_Co54)*Y(Fe54)*state.rho + screened_rates(k_p_Fe55_to_n_Co55)*Y(Fe55)*state.rho + screened_rates(k_p_Fe56_to_n_Co56)*Y(Fe56)*state.rho + screened_rates(k_p_Fe57_to_n_Co57)*Y(Fe57)*state.rho + screened_rates(k_p_Fe58_to_n_Co58)*Y(Fe58)*state.rho + screened_rates(k_p_Ga63_to_n_Ge63)*Y(Ga63)*state.rho + screened_rates(k_p_Ga64_to_n_Ge64)*Y(Ga64)*state.rho + 0.5*screened_rates(k_p_He4_He4_to_n_B8)*amrex::Math::powi<2>(Y(He4))*amrex::Math::powi<2>(state.rho) + screened_rates(k_p_K40_to_n_Ca40)*Y(K40)*state.rho + screened_rates(k_p_K41_to_n_Ca41)*Y(K41)*state.rho + screened_rates(k_p_Li7_to_n_Be7)*Y(Li7)*state.rho + screened_rates(k_p_Mg25_to_n_Al25)*Y(Mg25)*state.rho + screened_rates(k_p_Mg26_to_n_Al26)*Y(Mg26)*state.rho + screened_rates(k_p_Mn52_to_n_Fe52)*Y(Mn52)*state.rho + screened_rates(k_p_Mn53_to_n_Fe53)*Y(Mn53)*state.rho + screened_rates(k_p_Mn54_to_n_Fe54)*Y(Mn54)*state.rho + screened_rates(k_p_Mn55_to_n_Fe55)*Y(Mn55)*state.rho + screened_rates(k_p_N14_to_n_O14)*Y(N14)*state.rho + screened_rates(k_p_N15_to_n_O15)*Y(N15)*state.rho + screened_rates(k_p_Na23_to_n_Mg23)*Y(Na23)*state.rho + screened_rates(k_p_Ne21_to_n_Na21)*Y(Ne21)*state.rho + screened_rates(k_p_Ne22_to_n_Na22)*Y(Ne22)*state.rho + screened_rates(k_p_Ni57_to_n_Cu57)*Y(Ni57)*state.rho + screened_rates(k_p_Ni58_to_n_Cu58)*Y(Ni58)*state.rho + screened_rates(k_p_Ni59_to_n_Cu59)*Y(Ni59)*state.rho + screened_rates(k_p_Ni60_to_n_Cu60)*Y(Ni60)*state.rho + screened_rates(k_p_Ni61_to_n_Cu61)*Y(Ni61)*state.rho + screened_rates(k_p_Ni62_to_n_Cu62)*Y(Ni62)*state.rho + screened_rates(k_p_Ni63_to_n_Cu63)*Y(Ni63)*state.rho + screened_rates(k_p_Ni64_to_n_Cu64)*Y(Ni64)*state.rho + screened_rates(k_p_O17_to_n_F17)*Y(O17)*state.rho + screened_rates(k_p_O18_to_n_F18)*Y(O18)*state.rho + screened_rates(k_p_P32_to_n_S32)*Y(P32)*state.rho + screened_rates(k_p_P33_to_n_S33)*Y(P33)*state.rho + screened_rates(k_p_S33_to_n_Cl33)*Y(S33)*state.rho + screened_rates(k_p_S34_to_n_Cl34)*Y(S34)*state.rho + screened_rates(k_p_S35_to_n_Cl35)*Y(S35)*state.rho + screened_rates(k_p_S36_to_n_Cl36)*Y(S36)*state.rho + screened_rates(k_p_Sc44_to_n_Ti44)*Y(Sc44)*state.rho + screened_rates(k_p_Sc45_to_n_Ti45)*Y(Sc45)*state.rho + screened_rates(k_p_Sc46_to_n_Ti46)*Y(Sc46)*state.rho + screened_rates(k_p_Sc47_to_n_Ti47)*Y(Sc47)*state.rho + screened_rates(k_p_Sc48_to_n_Ti48)*Y(Sc48)*state.rho + screened_rates(k_p_Sc49_to_n_Ti49)*Y(Sc49)*state.rho + screened_rates(k_p_Si29_to_n_P29)*Y(Si29)*state.rho + screened_rates(k_p_Si30_to_n_P30)*Y(Si30)*state.rho + screened_rates(k_p_Si31_to_n_P31)*Y(Si31)*state.rho + screened_rates(k_p_Si32_to_n_P32)*Y(Si32)*state.rho + screened_rates(k_p_Ti46_to_n_V46)*Y(Ti46)*state.rho + screened_rates(k_p_Ti47_to_n_V47)*Y(Ti47)*state.rho + screened_rates(k_p_Ti48_to_n_V48)*Y(Ti48)*state.rho + screened_rates(k_p_Ti49_to_n_V49)*Y(Ti49)*state.rho + screened_rates(k_p_Ti50_to_n_V50)*Y(Ti50)*state.rho + screened_rates(k_p_Ti51_to_n_V51)*Y(Ti51)*state.rho + screened_rates(k_p_V48_to_n_Cr48)*Y(V48)*state.rho + screened_rates(k_p_V49_to_n_Cr49)*Y(V49)*state.rho + screened_rates(k_p_V50_to_n_Cr50)*Y(V50)*state.rho + screened_rates(k_p_V51_to_n_Cr51)*Y(V51)*state.rho + screened_rates(k_p_V52_to_n_Cr52)*Y(V52)*state.rho + screened_rates(k_p_Zn62_to_n_Ga62)*Y(Zn62)*state.rho + screened_rates(k_p_Zn63_to_n_Ga63)*Y(Zn63)*state.rho + screened_rates(k_p_Zn64_to_n_Ga64)*Y(Zn64)*state.rho + screened_rates(k_p_d_to_n_p_p)*Y(H2)*state.rho + screened_rates(k_p_to_n);
    jac.set(N, H1, scratch);

    scratch = screened_rates(k_d_C13_to_n_N14)*Y(C13)*state.rho + screened_rates(k_d_C14_to_n_N15)*Y(C14)*state.rho + screened_rates(k_d_Li6_to_n_Be7)*Y(Li6)*state.rho + screened_rates(k_d_Li7_to_n_He4_He4)*Y(Li7)*state.rho + 1.0*screened_rates(k_d_d_to_n_He3)*Y(H2)*state.rho + screened_rates(k_d_to_n_p) + screened_rates(k_p_d_to_n_p_p)*Y(H1)*state.rho;
    jac.set(N, H2, scratch);

    scratch = screened_rates(k_He3_Li7_to_n_p_He4_He4)*Y(Li7)*state.rho - screened_rates(k_n_He3_to_He4)*Y(N)*state.rho - screened_rates(k_n_He3_to_d_d)*Y(N)*state.rho;
    jac.set(N, He3, scratch);

    scratch = screened_rates(k_He4_Al26_to_n_P29)*Y(Al26)*state.rho + screened_rates(k_He4_Al27_to_n_P30)*Y(Al27)*state.rho + screened_rates(k_He4_Ar37_to_n_Ca40)*Y(Ar37)*state.rho + screened_rates(k_He4_Ar38_to_n_Ca41)*Y(Ar38)*state.rho + screened_rates(k_He4_Ar39_to_n_Ca42)*Y(Ar39)*state.rho + screened_rates(k_He4_Ar40_to_n_Ca43)*Y(Ar40)*state.rho + screened_rates(k_He4_B10_to_n_N13)*Y(B10)*state.rho + screened_rates(k_He4_B11_to_n_N14)*Y(B11)*state.rho + screened_rates(k_He4_Be9_to_n_C12)*Y(Be9)*state.rho + screened_rates(k_He4_C12_to_n_O15)*Y(C12)*state.rho + screened_rates(k_He4_C13_to_n_O16)*Y(C13)*state.rho + screened_rates(k_He4_C14_to_n_O17)*Y(C14)*state.rho + screened_rates(k_He4_Ca41_to_n_Ti44)*Y(Ca41)*state.rho + screened_rates(k_He4_Ca42_to_n_Ti45)*Y(Ca42)*state.rho + screened_rates(k_He4_Ca43_to_n_Ti46)*Y(Ca43)*state.rho + screened_rates(k_He4_Ca44_to_n_Ti47)*Y(Ca44)*state.rho + screened_rates(k_He4_Ca45_to_n_Ti48)*Y(Ca45)*state.rho + screened_rates(k_He4_Ca46_to_n_Ti49)*Y(Ca46)*state.rho + screened_rates(k_He4_Ca47_to_n_Ti50)*Y(Ca47)*state.rho + screened_rates(k_He4_Ca48_to_n_Ti51)*Y(Ca48)*state.rho + screened_rates(k_He4_Cl34_to_n_K37)*Y(Cl34)*state.rho + screened_rates(k_He4_Cl35_to_n_K38)*Y(Cl35)*state.rho + screened_rates(k_He4_Cl36_to_n_K39)*Y(Cl36)*state.rho + screened_rates(k_He4_Cl37_to_n_K40)*Y(Cl37)*state.rho + screened_rates(k_He4_Co54_to_n_Cu57)*Y(Co54)*state.rho + screened_rates(k_He4_Co55_to_n_Cu58)*Y(Co55)*state.rho + screened_rates(k_He4_Co56_to_n_Cu59)*Y(Co56)*state.rho + screened_rates(k_He4_Co57_to_n_Cu60)*Y(Co57)*state.rho + screened_rates(k_He4_Co58_to_n_Cu61)*Y(Co58)*state.rho + screened_rates(k_He4_Co59_to_n_Cu62)*Y(Co59)*state.rho + screened_rates(k_He4_Cr49_to_n_Fe52)*Y(Cr49)*state.rho + screened_rates(k_He4_Cr50_to_n_Fe53)*Y(Cr50)*state.rho + screened_rates(k_He4_Cr51_to_n_Fe54)*Y(Cr51)*state.rho + screened_rates(k_He4_Cr52_to_n_Fe55)*Y(Cr52)*state.rho + screened_rates(k_He4_Cr53_to_n_Fe56)*Y(Cr53)*state.rho + screened_rates(k_He4_Cr54_to_n_Fe57)*Y(Cr54)*state.rho + screened_rates(k_He4_Cu59_to_n_Ga62)*Y(Cu59)*state.rho + screened_rates(k_He4_Cu60_to_n_Ga63)*Y(Cu60)*state.rho + screened_rates(k_He4_Cu61_to_n_Ga64)*Y(Cu61)*state.rho + screened_rates(k_He4_F18_to_n_Na21)*Y(F18)*state.rho + screened_rates(k_He4_F19_to_n_Na22)*Y(F19)*state.rho + screened_rates(k_He4_Fe53_to_n_Ni56)*Y(Fe53)*state.rho + screened_rates(k_He4_Fe54_to_n_Ni57)*Y(Fe54)*state.rho + screened_rates(k_He4_Fe55_to_n_Ni58)*Y(Fe55)*state.rho + screened_rates(k_He4_Fe56_to_n_Ni59)*Y(Fe56)*state.rho + screened_rates(k_He4_Fe57_to_n_Ni60)*Y(Fe57)*state.rho + screened_rates(k_He4_Fe58_to_n_Ni61)*Y(Fe58)*state.rho + 1.0*screened_rates(k_He4_He4_to_n_Be7)*Y(He4)*state.rho + screened_rates(k_He4_K40_to_n_Sc43)*Y(K40)*state.rho + screened_rates(k_He4_K41_to_n_Sc44)*Y(K41)*state.rho + screened_rates(k_He4_Li7_to_n_B10)*Y(Li7)*state.rho + screened_rates(k_He4_Mg25_to_n_Si28)*Y(Mg25)*state.rho + screened_rates(k_He4_Mg26_to_n_Si29)*Y(Mg26)*state.rho + screened_rates(k_He4_Mn50_to_n_Co53)*Y(Mn50)*state.rho + screened_rates(k_He4_Mn51_to_n_Co54)*Y(Mn51)*state.rho + screened_rates(k_He4_Mn52_to_n_Co55)*Y(Mn52)*state.rho + screened_rates(k_He4_Mn53_to_n_Co56)*Y(Mn53)*state.rho + screened_rates(k_He4_Mn54_to_n_Co57)*Y(Mn54)*state.rho + screened_rates(k_He4_Mn55_to_n_Co58)*Y(Mn55)*state.rho + screened_rates(k_He4_N14_to_n_F17)*Y(N14)*state.rho + screened_rates(k_He4_N15_to_n_F18)*Y(N15)*state.rho + screened_rates(k_He4_Na22_to_n_Al25)*Y(Na22)*state.rho + screened_rates(k_He4_Na23_to_n_Al26)*Y(Na23)*state.rho + screened_rates(k_He4_Ne20_to_n_Mg23)*Y(Ne20)*state.rho + screened_rates(k_He4_Ne21_to_n_Mg24)*Y(Ne21)*state.rho + screened_rates(k_He4_Ne22_to_n_Mg25)*Y(Ne22)*state.rho + screened_rates(k_He4_Ni56_to_n_Zn59)*Y(Ni56)*state.rho + screened_rates(k_He4_Ni57_to_n_Zn60)*Y(Ni57)*state.rho + screened_rates(k_He4_Ni58_to_n_Zn61)*Y(Ni58)*state.rho + screened_rates(k_He4_Ni59_to_n_Zn62)*Y(Ni59)*state.rho + screened_rates(k_He4_Ni60_to_n_Zn63)*Y(Ni60)*state.rho + screened_rates(k_He4_Ni61_to_n_Zn64)*Y(Ni61)*state.rho + screened_rates(k_He4_Ni62_to_n_Zn65)*Y(Ni62)*state.rho + screened_rates(k_He4_Ni63_to_n_Zn66)*Y(Ni63)*state.rho + screened_rates(k_He4_O15_to_n_Ne18)*Y(O15)*state.rho + screened_rates(k_He4_O16_to_n_Ne19)*Y(O16)*state.rho + screened_rates(k_He4_O17_to_n_Ne20)*Y(O17)*state.rho + screened_rates(k_He4_O18_to_n_Ne21)*Y(O18)*state.rho + screened_rates(k_He4_P30_to_n_Cl33)*Y(P30)*state.rho + screened_rates(k_He4_P31_to_n_Cl34)*Y(P31)*state.rho + screened_rates(k_He4_P32_to_n_Cl35)*Y(P32)*state.rho + screened_rates(k_He4_P33_to_n_Cl36)*Y(P33)*state.rho + screened_rates(k_He4_S33_to_n_Ar36)*Y(S33)*state.rho + screened_rates(k_He4_S34_to_n_Ar37)*Y(S34)*state.rho + screened_rates(k_He4_S35_to_n_Ar38)*Y(S35)*state.rho + screened_rates(k_He4_S36_to_n_Ar39)*Y(S36)*state.rho + screened_rates(k_He4_Sc43_to_n_V46)*Y(Sc43)*state.rho + screened_rates(k_He4_Sc44_to_n_V47)*Y(Sc44)*state.rho + screened_rates(k_He4_Sc45_to_n_V48)*Y(Sc45)*state.rho + screened_rates(k_He4_Sc46_to_n_V49)*Y(Sc46)*state.rho + screened_rates(k_He4_Sc47_to_n_V50)*Y(Sc47)*state.rho + screened_rates(k_He4_Sc48_to_n_V51)*Y(Sc48)*state.rho + screened_rates(k_He4_Sc49_to_n_V52)*Y(Sc49)*state.rho + screened_rates(k_He4_Si29_to_n_S32)*Y(Si29)*state.rho + screened_rates(k_He4_Si30_to_n_S33)*Y(Si30)*state.rho + screened_rates(k_He4_Si31_to_n_S34)*Y(Si31)*state.rho + screened_rates(k_He4_Si32_to_n_S35)*Y(Si32)*state.rho + screened_rates(k_He4_Ti45_to_n_Cr48)*Y(Ti45)*state.rho + screened_rates(k_He4_Ti46_to_n_Cr49)*Y(Ti46)*state.rho + screened_rates(k_He4_Ti47_to_n_Cr50)*Y(Ti47)*state.rho + screened_rates(k_He4_Ti48_to_n_Cr51)*Y(Ti48)*state.rho + screened_rates(k_He4_Ti49_to_n_Cr52)*Y(Ti49)*state.rho + screened_rates(k_He4_Ti50_to_n_Cr53)*Y(Ti50)*state.rho + screened_rates(k_He4_Ti51_to_n_Cr54)*Y(Ti51)*state.rho + screened_rates(k_He4_V47_to_n_Mn50)*Y(V47)*state.rho + screened_rates(k_He4_V48_to_n_Mn51)*Y(V48)*state.rho + screened_rates(k_He4_V49_to_n_Mn52)*Y(V49)*state.rho + screened_rates(k_He4_V50_to_n_Mn53)*Y(V50)*state.rho + screened_rates(k_He4_V51_to_n_Mn54)*Y(V51)*state.rho + screened_rates(k_He4_V52_to_n_Mn55)*Y(V52)*state.rho + screened_rates(k_He4_Zn60_to_n_Ge63)*Y(Zn60)*state.rho + screened_rates(k_He4_Zn61_to_n_Ge64)*Y(Zn61)*state.rho + screened_rates(k_He4_to_n_He3) - 1.0*screened_rates(k_n_He4_He4_to_Be9)*Y(He4)*Y(N)*amrex::Math::powi<2>(state.rho) - 1.0*screened_rates(k_n_He4_He4_to_d_Li7)*Y(He4)*Y(N)*amrex::Math::powi<2>(state.rho) - 1.0*screened_rates(k_n_p_He4_He4_to_He3_Li7)*Y(He4)*Y(N)*Y(H1)*amrex::Math::powi<3>(state.rho) - 1.0*screened_rates(k_n_p_He4_He4_to_p_Be9)*Y(He4)*Y(N)*Y(H1)*amrex::Math::powi<3>(state.rho) - screened_rates(k_n_p_He4_to_Li6)*Y(N)*Y(H1)*amrex::Math::powi<2>(state.rho) + 1.0*screened_rates(k_p_He4_He4_to_n_B8)*Y(He4)*Y(H1)*amrex::Math::powi<2>(state.rho);
    jac.set(N, He4, scratch);

    scratch = screened_rates(k_Li6_to_n_p_He4) + screened_rates(k_d_Li6_to_n_Be7)*Y(H2)*state.rho - screened_rates(k_n_Li6_to_Li7)*Y(N)*state.rho;
    jac.set(N, Li6, scratch);

    scratch = screened_rates(k_He3_Li7_to_n_p_He4_He4)*Y(He3)*state.rho + screened_rates(k_He4_Li7_to_n_B10)*Y(He4)*state.rho + screened_rates(k_Li7_to_n_Li6) + screened_rates(k_d_Li7_to_n_He4_He4)*Y(H2)*state.rho + screened_rates(k_p_Li7_to_n_Be7)*Y(H1)*state.rho;
    jac.set(N, Li7, scratch);

    scratch = -screened_rates(k_n_Be7_to_He4_He4)*Y(N)*state.rho - screened_rates(k_n_Be7_to_d_Li6)*Y(N)*state.rho - screened_rates(k_n_Be7_to_p_Li7)*Y(N)*state.rho;
    jac.set(N, Be7, scratch);

    scratch = screened_rates(k_Be9_to_n_He4_He4) + screened_rates(k_He4_Be9_to_n_C12)*Y(He4)*state.rho + screened_rates(k_p_Be9_to_n_p_He4_He4)*Y(H1)*state.rho;
    jac.set(N, Be9, scratch);

    scratch = -screened_rates(k_n_B8_to_p_He4_He4)*Y(N)*state.rho;
    jac.set(N, B8, scratch);

    scratch = screened_rates(k_He4_B10_to_n_N13)*Y(He4)*state.rho - screened_rates(k_n_B10_to_B11)*Y(N)*state.rho - screened_rates(k_n_B10_to_He4_Li7)*Y(N)*state.rho;
    jac.set(N, B10, scratch);

    scratch = screened_rates(k_B11_to_n_B10) + screened_rates(k_He4_B11_to_n_N14)*Y(He4)*state.rho;
    jac.set(N, B11, scratch);

    scratch = 1.0*screened_rates(k_C12_C12_to_n_Mg23)*Y(C12)*state.rho + screened_rates(k_He4_C12_to_n_O15)*Y(He4)*state.rho - screened_rates(k_n_C12_to_C13)*Y(N)*state.rho - screened_rates(k_n_C12_to_He4_Be9)*Y(N)*state.rho;
    jac.set(N, C12, scratch);

    scratch = screened_rates(k_C13_to_n_C12) + screened_rates(k_He4_C13_to_n_O16)*Y(He4)*state.rho + screened_rates(k_d_C13_to_n_N14)*Y(H2)*state.rho - screened_rates(k_n_C13_to_C14)*Y(N)*state.rho + screened_rates(k_p_C13_to_n_N13)*Y(H1)*state.rho;
    jac.set(N, C13, scratch);

    scratch = screened_rates(k_C14_to_n_C13) + screened_rates(k_He4_C14_to_n_O17)*Y(He4)*state.rho + screened_rates(k_d_C14_to_n_N15)*Y(H2)*state.rho + screened_rates(k_p_C14_to_n_N14)*Y(H1)*state.rho;
    jac.set(N, C14, scratch);

    scratch = -screened_rates(k_n_N13_to_He4_B10)*Y(N)*state.rho - screened_rates(k_n_N13_to_N14)*Y(N)*state.rho - screened_rates(k_n_N13_to_p_C13)*Y(N)*state.rho;
    jac.set(N, N13, scratch);

    scratch = screened_rates(k_He4_N14_to_n_F17)*Y(He4)*state.rho + screened_rates(k_N14_to_n_N13) - screened_rates(k_n_N14_to_He4_B11)*Y(N)*state.rho - screened_rates(k_n_N14_to_N15)*Y(N)*state.rho - screened_rates(k_n_N14_to_d_C13)*Y(N)*state.rho - screened_rates(k_n_N14_to_p_C14)*Y(N)*state.rho + screened_rates(k_p_N14_to_n_O14)*Y(H1)*state.rho;
    jac.set(N, N14, scratch);

    scratch = screened_rates(k_He4_N15_to_n_F18)*Y(He4)*state.rho + screened_rates(k_N15_to_n_N14) - screened_rates(k_n_N15_to_d_C14)*Y(N)*state.rho + screened_rates(k_p_N15_to_n_O15)*Y(H1)*state.rho;
    jac.set(N, N15, scratch);

    scratch = -screened_rates(k_n_O14_to_O15)*Y(N)*state.rho - screened_rates(k_n_O14_to_p_N14)*Y(N)*state.rho;
    jac.set(N, O14, scratch);

    scratch = screened_rates(k_He4_O15_to_n_Ne18)*Y(He4)*state.rho + screened_rates(k_O15_to_n_O14) - screened_rates(k_n_O15_to_He4_C12)*Y(N)*state.rho - screened_rates(k_n_O15_to_O16)*Y(N)*state.rho - screened_rates(k_n_O15_to_p_N15)*Y(N)*state.rho;
    jac.set(N, O15, scratch);

    scratch = screened_rates(k_He4_O16_to_n_Ne19)*Y(He4)*state.rho + screened_rates(k_O16_to_n_O15) - screened_rates(k_n_O16_to_He4_C13)*Y(N)*state.rho - screened_rates(k_n_O16_to_O17)*Y(N)*state.rho;
    jac.set(N, O16, scratch);

    scratch = screened_rates(k_He4_O17_to_n_Ne20)*Y(He4)*state.rho + screened_rates(k_O17_to_n_O16) - screened_rates(k_n_O17_to_He4_C14)*Y(N)*state.rho - screened_rates(k_n_O17_to_O18)*Y(N)*state.rho + screened_rates(k_p_O17_to_n_F17)*Y(H1)*state.rho;
    jac.set(N, O17, scratch);

    scratch = screened_rates(k_He4_O18_to_n_Ne21)*Y(He4)*state.rho + screened_rates(k_O18_to_n_O17) + screened_rates(k_p_O18_to_n_F18)*Y(H1)*state.rho;
    jac.set(N, O18, scratch);

    scratch = -screened_rates(k_n_F17_to_F18)*Y(N)*state.rho - screened_rates(k_n_F17_to_He4_N14)*Y(N)*state.rho - screened_rates(k_n_F17_to_p_O17)*Y(N)*state.rho;
    jac.set(N, F17, scratch);

    scratch = screened_rates(k_F18_to_n_F17) + screened_rates(k_He4_F18_to_n_Na21)*Y(He4)*state.rho - screened_rates(k_n_F18_to_F19)*Y(N)*state.rho - screened_rates(k_n_F18_to_He4_N15)*Y(N)*state.rho - screened_rates(k_n_F18_to_p_O18)*Y(N)*state.rho + screened_rates(k_p_F18_to_n_Ne18)*Y(H1)*state.rho;
    jac.set(N, F18, scratch);

    scratch = screened_rates(k_F19_to_n_F18) + screened_rates(k_He4_F19_to_n_Na22)*Y(He4)*state.rho + screened_rates(k_p_F19_to_n_Ne19)*Y(H1)*state.rho;
    jac.set(N, F19, scratch);

    scratch = -screened_rates(k_n_Ne18_to_He4_O15)*Y(N)*state.rho - screened_rates(k_n_Ne18_to_Ne19)*Y(N)*state.rho - screened_rates(k_n_Ne18_to_p_F18)*Y(N)*state.rho;
    jac.set(N, Ne18, scratch);

    scratch = screened_rates(k_Ne19_to_n_Ne18) - screened_rates(k_n_Ne19_to_He4_O16)*Y(N)*state.rho - screened_rates(k_n_Ne19_to_Ne20)*Y(N)*state.rho - screened_rates(k_n_Ne19_to_p_F19)*Y(N)*state.rho;
    jac.set(N, Ne19, scratch);

    scratch = screened_rates(k_He4_Ne20_to_n_Mg23)*Y(He4)*state.rho + screened_rates(k_Ne20_to_n_Ne19) - screened_rates(k_n_Ne20_to_He4_O17)*Y(N)*state.rho - screened_rates(k_n_Ne20_to_Ne21)*Y(N)*state.rho;
    jac.set(N, Ne20, scratch);

    scratch = screened_rates(k_He4_Ne21_to_n_Mg24)*Y(He4)*state.rho + screened_rates(k_Ne21_to_n_Ne20) - screened_rates(k_n_Ne21_to_He4_O18)*Y(N)*state.rho - screened_rates(k_n_Ne21_to_Ne22)*Y(N)*state.rho + screened_rates(k_p_Ne21_to_n_Na21)*Y(H1)*state.rho;
    jac.set(N, Ne21, scratch);

    scratch = screened_rates(k_He4_Ne22_to_n_Mg25)*Y(He4)*state.rho + screened_rates(k_Ne22_to_n_Ne21) + screened_rates(k_p_Ne22_to_n_Na22)*Y(H1)*state.rho;
    jac.set(N, Ne22, scratch);

    scratch = -screened_rates(k_n_Na21_to_He4_F18)*Y(N)*state.rho - screened_rates(k_n_Na21_to_Na22)*Y(N)*state.rho - screened_rates(k_n_Na21_to_p_Ne21)*Y(N)*state.rho;
    jac.set(N, Na21, scratch);

    scratch = screened_rates(k_He4_Na22_to_n_Al25)*Y(He4)*state.rho + screened_rates(k_Na22_to_n_Na21) - screened_rates(k_n_Na22_to_He4_F19)*Y(N)*state.rho - screened_rates(k_n_Na22_to_Na23)*Y(N)*state.rho - screened_rates(k_n_Na22_to_p_Ne22)*Y(N)*state.rho;
    jac.set(N, Na22, scratch);

    scratch = screened_rates(k_He4_Na23_to_n_Al26)*Y(He4)*state.rho + screened_rates(k_Na23_to_n_Na22) + screened_rates(k_p_Na23_to_n_Mg23)*Y(H1)*state.rho;
    jac.set(N, Na23, scratch);

    scratch = -screened_rates(k_n_Mg23_to_C12_C12)*Y(N)*state.rho - screened_rates(k_n_Mg23_to_He4_Ne20)*Y(N)*state.rho - screened_rates(k_n_Mg23_to_Mg24)*Y(N)*state.rho - screened_rates(k_n_Mg23_to_p_Na23)*Y(N)*state.rho;
    jac.set(N, Mg23, scratch);

    scratch = screened_rates(k_Mg24_to_n_Mg23) - screened_rates(k_n_Mg24_to_He4_Ne21)*Y(N)*state.rho - screened_rates(k_n_Mg24_to_Mg25)*Y(N)*state.rho;
    jac.set(N, Mg24, scratch);

    scratch = screened_rates(k_He4_Mg25_to_n_Si28)*Y(He4)*state.rho + screened_rates(k_Mg25_to_n_Mg24) - screened_rates(k_n_Mg25_to_He4_Ne22)*Y(N)*state.rho - screened_rates(k_n_Mg25_to_Mg26)*Y(N)*state.rho + screened_rates(k_p_Mg25_to_n_Al25)*Y(H1)*state.rho;
    jac.set(N, Mg25, scratch);

    scratch = screened_rates(k_He4_Mg26_to_n_Si29)*Y(He4)*state.rho + screened_rates(k_Mg26_to_n_Mg25) + screened_rates(k_p_Mg26_to_n_Al26)*Y(H1)*state.rho;
    jac.set(N, Mg26, scratch);

    scratch = -screened_rates(k_n_Al25_to_Al26)*Y(N)*state.rho - screened_rates(k_n_Al25_to_He4_Na22)*Y(N)*state.rho - screened_rates(k_n_Al25_to_p_Mg25)*Y(N)*state.rho;
    jac.set(N, Al25, scratch);

    scratch = screened_rates(k_Al26_to_n_Al25) + screened_rates(k_He4_Al26_to_n_P29)*Y(He4)*state.rho - screened_rates(k_n_Al26_to_Al27)*Y(N)*state.rho - screened_rates(k_n_Al26_to_He4_Na23)*Y(N)*state.rho - screened_rates(k_n_Al26_to_p_Mg26)*Y(N)*state.rho;
    jac.set(N, Al26, scratch);

    scratch = screened_rates(k_Al27_to_n_Al26) + screened_rates(k_He4_Al27_to_n_P30)*Y(He4)*state.rho;
    jac.set(N, Al27, scratch);

    scratch = -screened_rates(k_n_Si28_to_He4_Mg25)*Y(N)*state.rho - screened_rates(k_n_Si28_to_Si29)*Y(N)*state.rho;
    jac.set(N, Si28, scratch);

    scratch = screened_rates(k_He4_Si29_to_n_S32)*Y(He4)*state.rho + screened_rates(k_Si29_to_n_Si28) - screened_rates(k_n_Si29_to_He4_Mg26)*Y(N)*state.rho - screened_rates(k_n_Si29_to_Si30)*Y(N)*state.rho + screened_rates(k_p_Si29_to_n_P29)*Y(H1)*state.rho;
    jac.set(N, Si29, scratch);

    scratch = screened_rates(k_He4_Si30_to_n_S33)*Y(He4)*state.rho + screened_rates(k_Si30_to_n_Si29) - screened_rates(k_n_Si30_to_Si31)*Y(N)*state.rho + screened_rates(k_p_Si30_to_n_P30)*Y(H1)*state.rho;
    jac.set(N, Si30, scratch);

    scratch = screened_rates(k_He4_Si31_to_n_S34)*Y(He4)*state.rho + screened_rates(k_Si31_to_n_Si30) - screened_rates(k_n_Si31_to_Si32)*Y(N)*state.rho + screened_rates(k_p_Si31_to_n_P31)*Y(H1)*state.rho;
    jac.set(N, Si31, scratch);

    scratch = screened_rates(k_He4_Si32_to_n_S35)*Y(He4)*state.rho + screened_rates(k_Si32_to_n_Si31) + screened_rates(k_p_Si32_to_n_P32)*Y(H1)*state.rho;
    jac.set(N, Si32, scratch);

    scratch = -screened_rates(k_n_P29_to_He4_Al26)*Y(N)*state.rho - screened_rates(k_n_P29_to_P30)*Y(N)*state.rho - screened_rates(k_n_P29_to_p_Si29)*Y(N)*state.rho;
    jac.set(N, P29, scratch);

    scratch = screened_rates(k_He4_P30_to_n_Cl33)*Y(He4)*state.rho + screened_rates(k_P30_to_n_P29) - screened_rates(k_n_P30_to_He4_Al27)*Y(N)*state.rho - screened_rates(k_n_P30_to_P31)*Y(N)*state.rho - screened_rates(k_n_P30_to_p_Si30)*Y(N)*state.rho;
    jac.set(N, P30, scratch);

    scratch = screened_rates(k_He4_P31_to_n_Cl34)*Y(He4)*state.rho + screened_rates(k_P31_to_n_P30) - screened_rates(k_n_P31_to_P32)*Y(N)*state.rho - screened_rates(k_n_P31_to_p_Si31)*Y(N)*state.rho;
    jac.set(N, P31, scratch);

    scratch = screened_rates(k_He4_P32_to_n_Cl35)*Y(He4)*state.rho + screened_rates(k_P32_to_n_P31) - screened_rates(k_n_P32_to_P33)*Y(N)*state.rho - screened_rates(k_n_P32_to_p_Si32)*Y(N)*state.rho + screened_rates(k_p_P32_to_n_S32)*Y(H1)*state.rho;
    jac.set(N, P32, scratch);

    scratch = screened_rates(k_He4_P33_to_n_Cl36)*Y(He4)*state.rho + screened_rates(k_P33_to_n_P32) + screened_rates(k_p_P33_to_n_S33)*Y(H1)*state.rho;
    jac.set(N, P33, scratch);

    scratch = -screened_rates(k_n_S32_to_He4_Si29)*Y(N)*state.rho - screened_rates(k_n_S32_to_S33)*Y(N)*state.rho - screened_rates(k_n_S32_to_p_P32)*Y(N)*state.rho;
    jac.set(N, S32, scratch);

    scratch = screened_rates(k_He4_S33_to_n_Ar36)*Y(He4)*state.rho + screened_rates(k_S33_to_n_S32) - screened_rates(k_n_S33_to_He4_Si30)*Y(N)*state.rho - screened_rates(k_n_S33_to_S34)*Y(N)*state.rho - screened_rates(k_n_S33_to_p_P33)*Y(N)*state.rho + screened_rates(k_p_S33_to_n_Cl33)*Y(H1)*state.rho;
    jac.set(N, S33, scratch);

    scratch = screened_rates(k_He4_S34_to_n_Ar37)*Y(He4)*state.rho + screened_rates(k_S34_to_n_S33) - screened_rates(k_n_S34_to_He4_Si31)*Y(N)*state.rho - screened_rates(k_n_S34_to_S35)*Y(N)*state.rho + screened_rates(k_p_S34_to_n_Cl34)*Y(H1)*state.rho;
    jac.set(N, S34, scratch);

    scratch = screened_rates(k_He4_S35_to_n_Ar38)*Y(He4)*state.rho + screened_rates(k_S35_to_n_S34) - screened_rates(k_n_S35_to_He4_Si32)*Y(N)*state.rho - screened_rates(k_n_S35_to_S36)*Y(N)*state.rho + screened_rates(k_p_S35_to_n_Cl35)*Y(H1)*state.rho;
    jac.set(N, S35, scratch);

    scratch = screened_rates(k_He4_S36_to_n_Ar39)*Y(He4)*state.rho + screened_rates(k_S36_to_n_S35) + screened_rates(k_p_S36_to_n_Cl36)*Y(H1)*state.rho;
    jac.set(N, S36, scratch);

    scratch = -screened_rates(k_n_Cl33_to_Cl34)*Y(N)*state.rho - screened_rates(k_n_Cl33_to_He4_P30)*Y(N)*state.rho - screened_rates(k_n_Cl33_to_p_S33)*Y(N)*state.rho;
    jac.set(N, Cl33, scratch);

    scratch = screened_rates(k_Cl34_to_n_Cl33) + screened_rates(k_He4_Cl34_to_n_K37)*Y(He4)*state.rho - screened_rates(k_n_Cl34_to_Cl35)*Y(N)*state.rho - screened_rates(k_n_Cl34_to_He4_P31)*Y(N)*state.rho - screened_rates(k_n_Cl34_to_p_S34)*Y(N)*state.rho;
    jac.set(N, Cl34, scratch);

    scratch = screened_rates(k_Cl35_to_n_Cl34) + screened_rates(k_He4_Cl35_to_n_K38)*Y(He4)*state.rho - screened_rates(k_n_Cl35_to_Cl36)*Y(N)*state.rho - screened_rates(k_n_Cl35_to_He4_P32)*Y(N)*state.rho - screened_rates(k_n_Cl35_to_p_S35)*Y(N)*state.rho;
    jac.set(N, Cl35, scratch);

    scratch = screened_rates(k_Cl36_to_n_Cl35) + screened_rates(k_He4_Cl36_to_n_K39)*Y(He4)*state.rho - screened_rates(k_n_Cl36_to_Cl37)*Y(N)*state.rho - screened_rates(k_n_Cl36_to_He4_P33)*Y(N)*state.rho - screened_rates(k_n_Cl36_to_p_S36)*Y(N)*state.rho + screened_rates(k_p_Cl36_to_n_Ar36)*Y(H1)*state.rho;
    jac.set(N, Cl36, scratch);

    scratch = screened_rates(k_Cl37_to_n_Cl36) + screened_rates(k_He4_Cl37_to_n_K40)*Y(He4)*state.rho + screened_rates(k_p_Cl37_to_n_Ar37)*Y(H1)*state.rho;
    jac.set(N, Cl37, scratch);

    scratch = -screened_rates(k_n_Ar36_to_Ar37)*Y(N)*state.rho - screened_rates(k_n_Ar36_to_He4_S33)*Y(N)*state.rho - screened_rates(k_n_Ar36_to_p_Cl36)*Y(N)*state.rho;
    jac.set(N, Ar36, scratch);

    scratch = screened_rates(k_Ar37_to_n_Ar36) + screened_rates(k_He4_Ar37_to_n_Ca40)*Y(He4)*state.rho - screened_rates(k_n_Ar37_to_Ar38)*Y(N)*state.rho - screened_rates(k_n_Ar37_to_He4_S34)*Y(N)*state.rho - screened_rates(k_n_Ar37_to_p_Cl37)*Y(N)*state.rho + screened_rates(k_p_Ar37_to_n_K37)*Y(H1)*state.rho;
    jac.set(N, Ar37, scratch);

    scratch = screened_rates(k_Ar38_to_n_Ar37) + screened_rates(k_He4_Ar38_to_n_Ca41)*Y(He4)*state.rho - screened_rates(k_n_Ar38_to_Ar39)*Y(N)*state.rho - screened_rates(k_n_Ar38_to_He4_S35)*Y(N)*state.rho + screened_rates(k_p_Ar38_to_n_K38)*Y(H1)*state.rho;
    jac.set(N, Ar38, scratch);

    scratch = screened_rates(k_Ar39_to_n_Ar38) + screened_rates(k_He4_Ar39_to_n_Ca42)*Y(He4)*state.rho - screened_rates(k_n_Ar39_to_Ar40)*Y(N)*state.rho - screened_rates(k_n_Ar39_to_He4_S36)*Y(N)*state.rho + screened_rates(k_p_Ar39_to_n_K39)*Y(H1)*state.rho;
    jac.set(N, Ar39, scratch);

    scratch = screened_rates(k_Ar40_to_n_Ar39) + screened_rates(k_He4_Ar40_to_n_Ca43)*Y(He4)*state.rho + screened_rates(k_p_Ar40_to_n_K40)*Y(H1)*state.rho;
    jac.set(N, Ar40, scratch);

    scratch = -screened_rates(k_n_K37_to_He4_Cl34)*Y(N)*state.rho - screened_rates(k_n_K37_to_K38)*Y(N)*state.rho - screened_rates(k_n_K37_to_p_Ar37)*Y(N)*state.rho;
    jac.set(N, K37, scratch);

    scratch = screened_rates(k_K38_to_n_K37) - screened_rates(k_n_K38_to_He4_Cl35)*Y(N)*state.rho - screened_rates(k_n_K38_to_K39)*Y(N)*state.rho - screened_rates(k_n_K38_to_p_Ar38)*Y(N)*state.rho;
    jac.set(N, K38, scratch);

    scratch = screened_rates(k_K39_to_n_K38) - screened_rates(k_n_K39_to_He4_Cl36)*Y(N)*state.rho - screened_rates(k_n_K39_to_K40)*Y(N)*state.rho - screened_rates(k_n_K39_to_p_Ar39)*Y(N)*state.rho;
    jac.set(N, K39, scratch);

    scratch = screened_rates(k_He4_K40_to_n_Sc43)*Y(He4)*state.rho + screened_rates(k_K40_to_n_K39) - screened_rates(k_n_K40_to_He4_Cl37)*Y(N)*state.rho - screened_rates(k_n_K40_to_K41)*Y(N)*state.rho - screened_rates(k_n_K40_to_p_Ar40)*Y(N)*state.rho + screened_rates(k_p_K40_to_n_Ca40)*Y(H1)*state.rho;
    jac.set(N, K40, scratch);

    scratch = screened_rates(k_He4_K41_to_n_Sc44)*Y(He4)*state.rho + screened_rates(k_K41_to_n_K40) + screened_rates(k_p_K41_to_n_Ca41)*Y(H1)*state.rho;
    jac.set(N, K41, scratch);

    scratch = -screened_rates(k_n_Ca40_to_Ca41)*Y(N)*state.rho - screened_rates(k_n_Ca40_to_He4_Ar37)*Y(N)*state.rho - screened_rates(k_n_Ca40_to_p_K40)*Y(N)*state.rho;
    jac.set(N, Ca40, scratch);

    scratch = screened_rates(k_Ca41_to_n_Ca40) + screened_rates(k_He4_Ca41_to_n_Ti44)*Y(He4)*state.rho - screened_rates(k_n_Ca41_to_Ca42)*Y(N)*state.rho - screened_rates(k_n_Ca41_to_He4_Ar38)*Y(N)*state.rho - screened_rates(k_n_Ca41_to_p_K41)*Y(N)*state.rho;
    jac.set(N, Ca41, scratch);

    scratch = screened_rates(k_Ca42_to_n_Ca41) + screened_rates(k_He4_Ca42_to_n_Ti45)*Y(He4)*state.rho - screened_rates(k_n_Ca42_to_Ca43)*Y(N)*state.rho - screened_rates(k_n_Ca42_to_He4_Ar39)*Y(N)*state.rho;
    jac.set(N, Ca42, scratch);

    scratch = screened_rates(k_Ca43_to_n_Ca42) + screened_rates(k_He4_Ca43_to_n_Ti46)*Y(He4)*state.rho - screened_rates(k_n_Ca43_to_Ca44)*Y(N)*state.rho - screened_rates(k_n_Ca43_to_He4_Ar40)*Y(N)*state.rho + screened_rates(k_p_Ca43_to_n_Sc43)*Y(H1)*state.rho;
    jac.set(N, Ca43, scratch);

    scratch = screened_rates(k_Ca44_to_n_Ca43) + screened_rates(k_He4_Ca44_to_n_Ti47)*Y(He4)*state.rho - screened_rates(k_n_Ca44_to_Ca45)*Y(N)*state.rho + screened_rates(k_p_Ca44_to_n_Sc44)*Y(H1)*state.rho;
    jac.set(N, Ca44, scratch);

    scratch = screened_rates(k_Ca45_to_n_Ca44) + screened_rates(k_He4_Ca45_to_n_Ti48)*Y(He4)*state.rho - screened_rates(k_n_Ca45_to_Ca46)*Y(N)*state.rho + screened_rates(k_p_Ca45_to_n_Sc45)*Y(H1)*state.rho;
    jac.set(N, Ca45, scratch);

    scratch = screened_rates(k_Ca46_to_n_Ca45) + screened_rates(k_He4_Ca46_to_n_Ti49)*Y(He4)*state.rho - screened_rates(k_n_Ca46_to_Ca47)*Y(N)*state.rho + screened_rates(k_p_Ca46_to_n_Sc46)*Y(H1)*state.rho;
    jac.set(N, Ca46, scratch);

    scratch = screened_rates(k_Ca47_to_n_Ca46) + screened_rates(k_He4_Ca47_to_n_Ti50)*Y(He4)*state.rho - screened_rates(k_n_Ca47_to_Ca48)*Y(N)*state.rho + screened_rates(k_p_Ca47_to_n_Sc47)*Y(H1)*state.rho;
    jac.set(N, Ca47, scratch);

    scratch = screened_rates(k_Ca48_to_n_Ca47) + screened_rates(k_He4_Ca48_to_n_Ti51)*Y(He4)*state.rho + screened_rates(k_p_Ca48_to_n_Sc48)*Y(H1)*state.rho;
    jac.set(N, Ca48, scratch);

    scratch = screened_rates(k_He4_Sc43_to_n_V46)*Y(He4)*state.rho - screened_rates(k_n_Sc43_to_He4_K40)*Y(N)*state.rho - screened_rates(k_n_Sc43_to_Sc44)*Y(N)*state.rho - screened_rates(k_n_Sc43_to_p_Ca43)*Y(N)*state.rho;
    jac.set(N, Sc43, scratch);

    scratch = screened_rates(k_He4_Sc44_to_n_V47)*Y(He4)*state.rho + screened_rates(k_Sc44_to_n_Sc43) - screened_rates(k_n_Sc44_to_He4_K41)*Y(N)*state.rho - screened_rates(k_n_Sc44_to_Sc45)*Y(N)*state.rho - screened_rates(k_n_Sc44_to_p_Ca44)*Y(N)*state.rho + screened_rates(k_p_Sc44_to_n_Ti44)*Y(H1)*state.rho;
    jac.set(N, Sc44, scratch);

    scratch = screened_rates(k_He4_Sc45_to_n_V48)*Y(He4)*state.rho + screened_rates(k_Sc45_to_n_Sc44) - screened_rates(k_n_Sc45_to_Sc46)*Y(N)*state.rho - screened_rates(k_n_Sc45_to_p_Ca45)*Y(N)*state.rho + screened_rates(k_p_Sc45_to_n_Ti45)*Y(H1)*state.rho;
    jac.set(N, Sc45, scratch);

    scratch = screened_rates(k_He4_Sc46_to_n_V49)*Y(He4)*state.rho + screened_rates(k_Sc46_to_n_Sc45) - screened_rates(k_n_Sc46_to_Sc47)*Y(N)*state.rho - screened_rates(k_n_Sc46_to_p_Ca46)*Y(N)*state.rho + screened_rates(k_p_Sc46_to_n_Ti46)*Y(H1)*state.rho;
    jac.set(N, Sc46, scratch);

    scratch = screened_rates(k_He4_Sc47_to_n_V50)*Y(He4)*state.rho + screened_rates(k_Sc47_to_n_Sc46) - screened_rates(k_n_Sc47_to_Sc48)*Y(N)*state.rho - screened_rates(k_n_Sc47_to_p_Ca47)*Y(N)*state.rho + screened_rates(k_p_Sc47_to_n_Ti47)*Y(H1)*state.rho;
    jac.set(N, Sc47, scratch);

    scratch = screened_rates(k_He4_Sc48_to_n_V51)*Y(He4)*state.rho + screened_rates(k_Sc48_to_n_Sc47) - screened_rates(k_n_Sc48_to_Sc49)*Y(N)*state.rho - screened_rates(k_n_Sc48_to_p_Ca48)*Y(N)*state.rho + screened_rates(k_p_Sc48_to_n_Ti48)*Y(H1)*state.rho;
    jac.set(N, Sc48, scratch);

    scratch = screened_rates(k_He4_Sc49_to_n_V52)*Y(He4)*state.rho + screened_rates(k_Sc49_to_n_Sc48) + screened_rates(k_p_Sc49_to_n_Ti49)*Y(H1)*state.rho;
    jac.set(N, Sc49, scratch);

    scratch = -screened_rates(k_n_Ti44_to_He4_Ca41)*Y(N)*state.rho - screened_rates(k_n_Ti44_to_Ti45)*Y(N)*state.rho - screened_rates(k_n_Ti44_to_p_Sc44)*Y(N)*state.rho;
    jac.set(N, Ti44, scratch);

    scratch = screened_rates(k_He4_Ti45_to_n_Cr48)*Y(He4)*state.rho + screened_rates(k_Ti45_to_n_Ti44) - screened_rates(k_n_Ti45_to_He4_Ca42)*Y(N)*state.rho - screened_rates(k_n_Ti45_to_Ti46)*Y(N)*state.rho - screened_rates(k_n_Ti45_to_p_Sc45)*Y(N)*state.rho;
    jac.set(N, Ti45, scratch);

    scratch = screened_rates(k_He4_Ti46_to_n_Cr49)*Y(He4)*state.rho + screened_rates(k_Ti46_to_n_Ti45) - screened_rates(k_n_Ti46_to_He4_Ca43)*Y(N)*state.rho - screened_rates(k_n_Ti46_to_Ti47)*Y(N)*state.rho - screened_rates(k_n_Ti46_to_p_Sc46)*Y(N)*state.rho + screened_rates(k_p_Ti46_to_n_V46)*Y(H1)*state.rho;
    jac.set(N, Ti46, scratch);

    scratch = screened_rates(k_He4_Ti47_to_n_Cr50)*Y(He4)*state.rho + screened_rates(k_Ti47_to_n_Ti46) - screened_rates(k_n_Ti47_to_He4_Ca44)*Y(N)*state.rho - screened_rates(k_n_Ti47_to_Ti48)*Y(N)*state.rho - screened_rates(k_n_Ti47_to_p_Sc47)*Y(N)*state.rho + screened_rates(k_p_Ti47_to_n_V47)*Y(H1)*state.rho;
    jac.set(N, Ti47, scratch);

    scratch = screened_rates(k_He4_Ti48_to_n_Cr51)*Y(He4)*state.rho + screened_rates(k_Ti48_to_n_Ti47) - screened_rates(k_n_Ti48_to_He4_Ca45)*Y(N)*state.rho - screened_rates(k_n_Ti48_to_Ti49)*Y(N)*state.rho - screened_rates(k_n_Ti48_to_p_Sc48)*Y(N)*state.rho + screened_rates(k_p_Ti48_to_n_V48)*Y(H1)*state.rho;
    jac.set(N, Ti48, scratch);

    scratch = screened_rates(k_He4_Ti49_to_n_Cr52)*Y(He4)*state.rho + screened_rates(k_Ti49_to_n_Ti48) - screened_rates(k_n_Ti49_to_He4_Ca46)*Y(N)*state.rho - screened_rates(k_n_Ti49_to_Ti50)*Y(N)*state.rho - screened_rates(k_n_Ti49_to_p_Sc49)*Y(N)*state.rho + screened_rates(k_p_Ti49_to_n_V49)*Y(H1)*state.rho;
    jac.set(N, Ti49, scratch);

    scratch = screened_rates(k_He4_Ti50_to_n_Cr53)*Y(He4)*state.rho + screened_rates(k_Ti50_to_n_Ti49) - screened_rates(k_n_Ti50_to_He4_Ca47)*Y(N)*state.rho - screened_rates(k_n_Ti50_to_Ti51)*Y(N)*state.rho + screened_rates(k_p_Ti50_to_n_V50)*Y(H1)*state.rho;
    jac.set(N, Ti50, scratch);

    scratch = screened_rates(k_He4_Ti51_to_n_Cr54)*Y(He4)*state.rho + screened_rates(k_Ti51_to_n_Ti50) - screened_rates(k_n_Ti51_to_He4_Ca48)*Y(N)*state.rho + screened_rates(k_p_Ti51_to_n_V51)*Y(H1)*state.rho;
    jac.set(N, Ti51, scratch);

    scratch = -screened_rates(k_n_V46_to_He4_Sc43)*Y(N)*state.rho - screened_rates(k_n_V46_to_V47)*Y(N)*state.rho - screened_rates(k_n_V46_to_p_Ti46)*Y(N)*state.rho;
    jac.set(N, V46, scratch);

    scratch = screened_rates(k_He4_V47_to_n_Mn50)*Y(He4)*state.rho + screened_rates(k_V47_to_n_V46) - screened_rates(k_n_V47_to_He4_Sc44)*Y(N)*state.rho - screened_rates(k_n_V47_to_V48)*Y(N)*state.rho - screened_rates(k_n_V47_to_p_Ti47)*Y(N)*state.rho;
    jac.set(N, V47, scratch);

    scratch = screened_rates(k_He4_V48_to_n_Mn51)*Y(He4)*state.rho + screened_rates(k_V48_to_n_V47) - screened_rates(k_n_V48_to_He4_Sc45)*Y(N)*state.rho - screened_rates(k_n_V48_to_V49)*Y(N)*state.rho - screened_rates(k_n_V48_to_p_Ti48)*Y(N)*state.rho + screened_rates(k_p_V48_to_n_Cr48)*Y(H1)*state.rho;
    jac.set(N, V48, scratch);

    scratch = screened_rates(k_He4_V49_to_n_Mn52)*Y(He4)*state.rho + screened_rates(k_V49_to_n_V48) - screened_rates(k_n_V49_to_He4_Sc46)*Y(N)*state.rho - screened_rates(k_n_V49_to_V50)*Y(N)*state.rho - screened_rates(k_n_V49_to_p_Ti49)*Y(N)*state.rho + screened_rates(k_p_V49_to_n_Cr49)*Y(H1)*state.rho;
    jac.set(N, V49, scratch);

    scratch = screened_rates(k_He4_V50_to_n_Mn53)*Y(He4)*state.rho + screened_rates(k_V50_to_n_V49) - screened_rates(k_n_V50_to_He4_Sc47)*Y(N)*state.rho - screened_rates(k_n_V50_to_V51)*Y(N)*state.rho - screened_rates(k_n_V50_to_p_Ti50)*Y(N)*state.rho + screened_rates(k_p_V50_to_n_Cr50)*Y(H1)*state.rho;
    jac.set(N, V50, scratch);

    scratch = screened_rates(k_He4_V51_to_n_Mn54)*Y(He4)*state.rho + screened_rates(k_V51_to_n_V50) - screened_rates(k_n_V51_to_He4_Sc48)*Y(N)*state.rho - screened_rates(k_n_V51_to_V52)*Y(N)*state.rho - screened_rates(k_n_V51_to_p_Ti51)*Y(N)*state.rho + screened_rates(k_p_V51_to_n_Cr51)*Y(H1)*state.rho;
    jac.set(N, V51, scratch);

    scratch = screened_rates(k_He4_V52_to_n_Mn55)*Y(He4)*state.rho + screened_rates(k_V52_to_n_V51) - screened_rates(k_n_V52_to_He4_Sc49)*Y(N)*state.rho + screened_rates(k_p_V52_to_n_Cr52)*Y(H1)*state.rho;
    jac.set(N, V52, scratch);

    scratch = -screened_rates(k_n_Cr48_to_Cr49)*Y(N)*state.rho - screened_rates(k_n_Cr48_to_He4_Ti45)*Y(N)*state.rho - screened_rates(k_n_Cr48_to_p_V48)*Y(N)*state.rho;
    jac.set(N, Cr48, scratch);

    scratch = screened_rates(k_Cr49_to_n_Cr48) + screened_rates(k_He4_Cr49_to_n_Fe52)*Y(He4)*state.rho - screened_rates(k_n_Cr49_to_Cr50)*Y(N)*state.rho - screened_rates(k_n_Cr49_to_He4_Ti46)*Y(N)*state.rho - screened_rates(k_n_Cr49_to_p_V49)*Y(N)*state.rho;
    jac.set(N, Cr49, scratch);

    scratch = screened_rates(k_Cr50_to_n_Cr49) + screened_rates(k_He4_Cr50_to_n_Fe53)*Y(He4)*state.rho - screened_rates(k_n_Cr50_to_Cr51)*Y(N)*state.rho - screened_rates(k_n_Cr50_to_He4_Ti47)*Y(N)*state.rho - screened_rates(k_n_Cr50_to_p_V50)*Y(N)*state.rho + screened_rates(k_p_Cr50_to_n_Mn50)*Y(H1)*state.rho;
    jac.set(N, Cr50, scratch);

    scratch = screened_rates(k_Cr51_to_n_Cr50) + screened_rates(k_He4_Cr51_to_n_Fe54)*Y(He4)*state.rho - screened_rates(k_n_Cr51_to_Cr52)*Y(N)*state.rho - screened_rates(k_n_Cr51_to_He4_Ti48)*Y(N)*state.rho - screened_rates(k_n_Cr51_to_p_V51)*Y(N)*state.rho + screened_rates(k_p_Cr51_to_n_Mn51)*Y(H1)*state.rho;
    jac.set(N, Cr51, scratch);

    scratch = screened_rates(k_Cr52_to_n_Cr51) + screened_rates(k_He4_Cr52_to_n_Fe55)*Y(He4)*state.rho - screened_rates(k_n_Cr52_to_Cr53)*Y(N)*state.rho - screened_rates(k_n_Cr52_to_He4_Ti49)*Y(N)*state.rho - screened_rates(k_n_Cr52_to_p_V52)*Y(N)*state.rho + screened_rates(k_p_Cr52_to_n_Mn52)*Y(H1)*state.rho;
    jac.set(N, Cr52, scratch);

    scratch = screened_rates(k_Cr53_to_n_Cr52) + screened_rates(k_He4_Cr53_to_n_Fe56)*Y(He4)*state.rho - screened_rates(k_n_Cr53_to_Cr54)*Y(N)*state.rho - screened_rates(k_n_Cr53_to_He4_Ti50)*Y(N)*state.rho + screened_rates(k_p_Cr53_to_n_Mn53)*Y(H1)*state.rho;
    jac.set(N, Cr53, scratch);

    scratch = screened_rates(k_Cr54_to_n_Cr53) + screened_rates(k_He4_Cr54_to_n_Fe57)*Y(He4)*state.rho - screened_rates(k_n_Cr54_to_He4_Ti51)*Y(N)*state.rho + screened_rates(k_p_Cr54_to_n_Mn54)*Y(H1)*state.rho;
    jac.set(N, Cr54, scratch);

    scratch = screened_rates(k_He4_Mn50_to_n_Co53)*Y(He4)*state.rho - screened_rates(k_n_Mn50_to_He4_V47)*Y(N)*state.rho - screened_rates(k_n_Mn50_to_Mn51)*Y(N)*state.rho - screened_rates(k_n_Mn50_to_p_Cr50)*Y(N)*state.rho;
    jac.set(N, Mn50, scratch);

    scratch = screened_rates(k_He4_Mn51_to_n_Co54)*Y(He4)*state.rho + screened_rates(k_Mn51_to_n_Mn50) - screened_rates(k_n_Mn51_to_He4_V48)*Y(N)*state.rho - screened_rates(k_n_Mn51_to_Mn52)*Y(N)*state.rho - screened_rates(k_n_Mn51_to_p_Cr51)*Y(N)*state.rho;
    jac.set(N, Mn51, scratch);

    scratch = screened_rates(k_He4_Mn52_to_n_Co55)*Y(He4)*state.rho + screened_rates(k_Mn52_to_n_Mn51) - screened_rates(k_n_Mn52_to_He4_V49)*Y(N)*state.rho - screened_rates(k_n_Mn52_to_Mn53)*Y(N)*state.rho - screened_rates(k_n_Mn52_to_p_Cr52)*Y(N)*state.rho + screened_rates(k_p_Mn52_to_n_Fe52)*Y(H1)*state.rho;
    jac.set(N, Mn52, scratch);

    scratch = screened_rates(k_He4_Mn53_to_n_Co56)*Y(He4)*state.rho + screened_rates(k_Mn53_to_n_Mn52) - screened_rates(k_n_Mn53_to_He4_V50)*Y(N)*state.rho - screened_rates(k_n_Mn53_to_Mn54)*Y(N)*state.rho - screened_rates(k_n_Mn53_to_p_Cr53)*Y(N)*state.rho + screened_rates(k_p_Mn53_to_n_Fe53)*Y(H1)*state.rho;
    jac.set(N, Mn53, scratch);

    scratch = screened_rates(k_He4_Mn54_to_n_Co57)*Y(He4)*state.rho + screened_rates(k_Mn54_to_n_Mn53) - screened_rates(k_n_Mn54_to_He4_V51)*Y(N)*state.rho - screened_rates(k_n_Mn54_to_Mn55)*Y(N)*state.rho - screened_rates(k_n_Mn54_to_p_Cr54)*Y(N)*state.rho + screened_rates(k_p_Mn54_to_n_Fe54)*Y(H1)*state.rho;
    jac.set(N, Mn54, scratch);

    scratch = screened_rates(k_He4_Mn55_to_n_Co58)*Y(He4)*state.rho + screened_rates(k_Mn55_to_n_Mn54) - screened_rates(k_n_Mn55_to_He4_V52)*Y(N)*state.rho + screened_rates(k_p_Mn55_to_n_Fe55)*Y(H1)*state.rho;
    jac.set(N, Mn55, scratch);

    scratch = -screened_rates(k_n_Fe52_to_Fe53)*Y(N)*state.rho - screened_rates(k_n_Fe52_to_He4_Cr49)*Y(N)*state.rho - screened_rates(k_n_Fe52_to_p_Mn52)*Y(N)*state.rho;
    jac.set(N, Fe52, scratch);

    scratch = screened_rates(k_Fe53_to_n_Fe52) + screened_rates(k_He4_Fe53_to_n_Ni56)*Y(He4)*state.rho - screened_rates(k_n_Fe53_to_Fe54)*Y(N)*state.rho - screened_rates(k_n_Fe53_to_He4_Cr50)*Y(N)*state.rho - screened_rates(k_n_Fe53_to_p_Mn53)*Y(N)*state.rho + screened_rates(k_p_Fe53_to_n_Co53)*Y(H1)*state.rho;
    jac.set(N, Fe53, scratch);

    scratch = screened_rates(k_Fe54_to_n_Fe53) + screened_rates(k_He4_Fe54_to_n_Ni57)*Y(He4)*state.rho - screened_rates(k_n_Fe54_to_Fe55)*Y(N)*state.rho - screened_rates(k_n_Fe54_to_He4_Cr51)*Y(N)*state.rho - screened_rates(k_n_Fe54_to_p_Mn54)*Y(N)*state.rho + screened_rates(k_p_Fe54_to_n_Co54)*Y(H1)*state.rho;
    jac.set(N, Fe54, scratch);

    scratch = screened_rates(k_Fe55_to_n_Fe54) + screened_rates(k_He4_Fe55_to_n_Ni58)*Y(He4)*state.rho - screened_rates(k_n_Fe55_to_Fe56)*Y(N)*state.rho - screened_rates(k_n_Fe55_to_He4_Cr52)*Y(N)*state.rho - screened_rates(k_n_Fe55_to_p_Mn55)*Y(N)*state.rho + screened_rates(k_p_Fe55_to_n_Co55)*Y(H1)*state.rho;
    jac.set(N, Fe55, scratch);

    scratch = screened_rates(k_Fe56_to_n_Fe55) + screened_rates(k_He4_Fe56_to_n_Ni59)*Y(He4)*state.rho - screened_rates(k_n_Fe56_to_Fe57)*Y(N)*state.rho - screened_rates(k_n_Fe56_to_He4_Cr53)*Y(N)*state.rho + screened_rates(k_p_Fe56_to_n_Co56)*Y(H1)*state.rho;
    jac.set(N, Fe56, scratch);

    scratch = screened_rates(k_Fe57_to_n_Fe56) + screened_rates(k_He4_Fe57_to_n_Ni60)*Y(He4)*state.rho - screened_rates(k_n_Fe57_to_Fe58)*Y(N)*state.rho - screened_rates(k_n_Fe57_to_He4_Cr54)*Y(N)*state.rho + screened_rates(k_p_Fe57_to_n_Co57)*Y(H1)*state.rho;
    jac.set(N, Fe57, scratch);

    scratch = screened_rates(k_Fe58_to_n_Fe57) + screened_rates(k_He4_Fe58_to_n_Ni61)*Y(He4)*state.rho + screened_rates(k_p_Fe58_to_n_Co58)*Y(H1)*state.rho;
    jac.set(N, Fe58, scratch);

    scratch = -screened_rates(k_n_Co53_to_Co54)*Y(N)*state.rho - screened_rates(k_n_Co53_to_He4_Mn50)*Y(N)*state.rho - screened_rates(k_n_Co53_to_p_Fe53)*Y(N)*state.rho;
    jac.set(N, Co53, scratch);

    scratch = screened_rates(k_Co54_to_n_Co53) + screened_rates(k_He4_Co54_to_n_Cu57)*Y(He4)*state.rho - screened_rates(k_n_Co54_to_Co55)*Y(N)*state.rho - screened_rates(k_n_Co54_to_He4_Mn51)*Y(N)*state.rho - screened_rates(k_n_Co54_to_p_Fe54)*Y(N)*state.rho;
    jac.set(N, Co54, scratch);

    scratch = screened_rates(k_Co55_to_n_Co54) + screened_rates(k_He4_Co55_to_n_Cu58)*Y(He4)*state.rho - screened_rates(k_n_Co55_to_Co56)*Y(N)*state.rho - screened_rates(k_n_Co55_to_He4_Mn52)*Y(N)*state.rho - screened_rates(k_n_Co55_to_p_Fe55)*Y(N)*state.rho;
    jac.set(N, Co55, scratch);

    scratch = screened_rates(k_Co56_to_n_Co55) + screened_rates(k_He4_Co56_to_n_Cu59)*Y(He4)*state.rho - screened_rates(k_n_Co56_to_Co57)*Y(N)*state.rho - screened_rates(k_n_Co56_to_He4_Mn53)*Y(N)*state.rho - screened_rates(k_n_Co56_to_p_Fe56)*Y(N)*state.rho + screened_rates(k_p_Co56_to_n_Ni56)*Y(H1)*state.rho;
    jac.set(N, Co56, scratch);

    scratch = screened_rates(k_Co57_to_n_Co56) + screened_rates(k_He4_Co57_to_n_Cu60)*Y(He4)*state.rho - screened_rates(k_n_Co57_to_Co58)*Y(N)*state.rho - screened_rates(k_n_Co57_to_He4_Mn54)*Y(N)*state.rho - screened_rates(k_n_Co57_to_p_Fe57)*Y(N)*state.rho + screened_rates(k_p_Co57_to_n_Ni57)*Y(H1)*state.rho;
    jac.set(N, Co57, scratch);

    scratch = screened_rates(k_Co58_to_n_Co57) + screened_rates(k_He4_Co58_to_n_Cu61)*Y(He4)*state.rho - screened_rates(k_n_Co58_to_Co59)*Y(N)*state.rho - screened_rates(k_n_Co58_to_He4_Mn55)*Y(N)*state.rho - screened_rates(k_n_Co58_to_p_Fe58)*Y(N)*state.rho + screened_rates(k_p_Co58_to_n_Ni58)*Y(H1)*state.rho;
    jac.set(N, Co58, scratch);

    scratch = screened_rates(k_Co59_to_n_Co58) + screened_rates(k_He4_Co59_to_n_Cu62)*Y(He4)*state.rho + screened_rates(k_p_Co59_to_n_Ni59)*Y(H1)*state.rho;
    jac.set(N, Co59, scratch);

    scratch = screened_rates(k_He4_Ni56_to_n_Zn59)*Y(He4)*state.rho - screened_rates(k_n_Ni56_to_He4_Fe53)*Y(N)*state.rho - screened_rates(k_n_Ni56_to_Ni57)*Y(N)*state.rho - screened_rates(k_n_Ni56_to_p_Co56)*Y(N)*state.rho;
    jac.set(N, Ni56, scratch);

    scratch = screened_rates(k_He4_Ni57_to_n_Zn60)*Y(He4)*state.rho + screened_rates(k_Ni57_to_n_Ni56) - screened_rates(k_n_Ni57_to_He4_Fe54)*Y(N)*state.rho - screened_rates(k_n_Ni57_to_Ni58)*Y(N)*state.rho - screened_rates(k_n_Ni57_to_p_Co57)*Y(N)*state.rho + screened_rates(k_p_Ni57_to_n_Cu57)*Y(H1)*state.rho;
    jac.set(N, Ni57, scratch);

    scratch = screened_rates(k_He4_Ni58_to_n_Zn61)*Y(He4)*state.rho + screened_rates(k_Ni58_to_n_Ni57) - screened_rates(k_n_Ni58_to_He4_Fe55)*Y(N)*state.rho - screened_rates(k_n_Ni58_to_Ni59)*Y(N)*state.rho - screened_rates(k_n_Ni58_to_p_Co58)*Y(N)*state.rho + screened_rates(k_p_Ni58_to_n_Cu58)*Y(H1)*state.rho;
    jac.set(N, Ni58, scratch);

    scratch = screened_rates(k_He4_Ni59_to_n_Zn62)*Y(He4)*state.rho + screened_rates(k_Ni59_to_n_Ni58) - screened_rates(k_n_Ni59_to_He4_Fe56)*Y(N)*state.rho - screened_rates(k_n_Ni59_to_Ni60)*Y(N)*state.rho - screened_rates(k_n_Ni59_to_p_Co59)*Y(N)*state.rho + screened_rates(k_p_Ni59_to_n_Cu59)*Y(H1)*state.rho;
    jac.set(N, Ni59, scratch);

    scratch = screened_rates(k_He4_Ni60_to_n_Zn63)*Y(He4)*state.rho + screened_rates(k_Ni60_to_n_Ni59) - screened_rates(k_n_Ni60_to_He4_Fe57)*Y(N)*state.rho - screened_rates(k_n_Ni60_to_Ni61)*Y(N)*state.rho + screened_rates(k_p_Ni60_to_n_Cu60)*Y(H1)*state.rho;
    jac.set(N, Ni60, scratch);

    scratch = screened_rates(k_He4_Ni61_to_n_Zn64)*Y(He4)*state.rho + screened_rates(k_Ni61_to_n_Ni60) - screened_rates(k_n_Ni61_to_He4_Fe58)*Y(N)*state.rho - screened_rates(k_n_Ni61_to_Ni62)*Y(N)*state.rho + screened_rates(k_p_Ni61_to_n_Cu61)*Y(H1)*state.rho;
    jac.set(N, Ni61, scratch);

    scratch = screened_rates(k_He4_Ni62_to_n_Zn65)*Y(He4)*state.rho + screened_rates(k_Ni62_to_n_Ni61) - screened_rates(k_n_Ni62_to_Ni63)*Y(N)*state.rho + screened_rates(k_p_Ni62_to_n_Cu62)*Y(H1)*state.rho;
    jac.set(N, Ni62, scratch);

    scratch = screened_rates(k_He4_Ni63_to_n_Zn66)*Y(He4)*state.rho + screened_rates(k_Ni63_to_n_Ni62) - screened_rates(k_n_Ni63_to_Ni64)*Y(N)*state.rho + screened_rates(k_p_Ni63_to_n_Cu63)*Y(H1)*state.rho;
    jac.set(N, Ni63, scratch);

    scratch = screened_rates(k_Ni64_to_n_Ni63) + screened_rates(k_p_Ni64_to_n_Cu64)*Y(H1)*state.rho;
    jac.set(N, Ni64, scratch);

    scratch = -screened_rates(k_n_Cu57_to_Cu58)*Y(N)*state.rho - screened_rates(k_n_Cu57_to_He4_Co54)*Y(N)*state.rho - screened_rates(k_n_Cu57_to_p_Ni57)*Y(N)*state.rho;
    jac.set(N, Cu57, scratch);

    scratch = screened_rates(k_Cu58_to_n_Cu57) - screened_rates(k_n_Cu58_to_Cu59)*Y(N)*state.rho - screened_rates(k_n_Cu58_to_He4_Co55)*Y(N)*state.rho - screened_rates(k_n_Cu58_to_p_Ni58)*Y(N)*state.rho;
    jac.set(N, Cu58, scratch);

    scratch = screened_rates(k_Cu59_to_n_Cu58) + screened_rates(k_He4_Cu59_to_n_Ga62)*Y(He4)*state.rho - screened_rates(k_n_Cu59_to_Cu60)*Y(N)*state.rho - screened_rates(k_n_Cu59_to_He4_Co56)*Y(N)*state.rho - screened_rates(k_n_Cu59_to_p_Ni59)*Y(N)*state.rho + screened_rates(k_p_Cu59_to_n_Zn59)*Y(H1)*state.rho;
    jac.set(N, Cu59, scratch);

    scratch = screened_rates(k_Cu60_to_n_Cu59) + screened_rates(k_He4_Cu60_to_n_Ga63)*Y(He4)*state.rho - screened_rates(k_n_Cu60_to_Cu61)*Y(N)*state.rho - screened_rates(k_n_Cu60_to_He4_Co57)*Y(N)*state.rho - screened_rates(k_n_Cu60_to_p_Ni60)*Y(N)*state.rho + screened_rates(k_p_Cu60_to_n_Zn60)*Y(H1)*state.rho;
    jac.set(N, Cu60, scratch);

    scratch = screened_rates(k_Cu61_to_n_Cu60) + screened_rates(k_He4_Cu61_to_n_Ga64)*Y(He4)*state.rho - screened_rates(k_n_Cu61_to_Cu62)*Y(N)*state.rho - screened_rates(k_n_Cu61_to_He4_Co58)*Y(N)*state.rho - screened_rates(k_n_Cu61_to_p_Ni61)*Y(N)*state.rho + screened_rates(k_p_Cu61_to_n_Zn61)*Y(H1)*state.rho;
    jac.set(N, Cu61, scratch);

    scratch = screened_rates(k_Cu62_to_n_Cu61) - screened_rates(k_n_Cu62_to_Cu63)*Y(N)*state.rho - screened_rates(k_n_Cu62_to_He4_Co59)*Y(N)*state.rho - screened_rates(k_n_Cu62_to_p_Ni62)*Y(N)*state.rho + screened_rates(k_p_Cu62_to_n_Zn62)*Y(H1)*state.rho;
    jac.set(N, Cu62, scratch);

    scratch = screened_rates(k_Cu63_to_n_Cu62) - screened_rates(k_n_Cu63_to_Cu64)*Y(N)*state.rho - screened_rates(k_n_Cu63_to_p_Ni63)*Y(N)*state.rho + screened_rates(k_p_Cu63_to_n_Zn63)*Y(H1)*state.rho;
    jac.set(N, Cu63, scratch);

    scratch = screened_rates(k_Cu64_to_n_Cu63) - screened_rates(k_n_Cu64_to_Cu65)*Y(N)*state.rho - screened_rates(k_n_Cu64_to_p_Ni64)*Y(N)*state.rho + screened_rates(k_p_Cu64_to_n_Zn64)*Y(H1)*state.rho;
    jac.set(N, Cu64, scratch);

    scratch = screened_rates(k_Cu65_to_n_Cu64) + screened_rates(k_p_Cu65_to_n_Zn65)*Y(H1)*state.rho;
    jac.set(N, Cu65, scratch);

    scratch = -screened_rates(k_n_Zn59_to_He4_Ni56)*Y(N)*state.rho - screened_rates(k_n_Zn59_to_Zn60)*Y(N)*state.rho - screened_rates(k_n_Zn59_to_p_Cu59)*Y(N)*state.rho;
    jac.set(N, Zn59, scratch);

    scratch = screened_rates(k_He4_Zn60_to_n_Ge63)*Y(He4)*state.rho + screened_rates(k_Zn60_to_n_Zn59) - screened_rates(k_n_Zn60_to_He4_Ni57)*Y(N)*state.rho - screened_rates(k_n_Zn60_to_Zn61)*Y(N)*state.rho - screened_rates(k_n_Zn60_to_p_Cu60)*Y(N)*state.rho;
    jac.set(N, Zn60, scratch);

    scratch = screened_rates(k_He4_Zn61_to_n_Ge64)*Y(He4)*state.rho + screened_rates(k_Zn61_to_n_Zn60) - screened_rates(k_n_Zn61_to_He4_Ni58)*Y(N)*state.rho - screened_rates(k_n_Zn61_to_Zn62)*Y(N)*state.rho - screened_rates(k_n_Zn61_to_p_Cu61)*Y(N)*state.rho;
    jac.set(N, Zn61, scratch);

    scratch = screened_rates(k_Zn62_to_n_Zn61) - screened_rates(k_n_Zn62_to_He4_Ni59)*Y(N)*state.rho - screened_rates(k_n_Zn62_to_Zn63)*Y(N)*state.rho - screened_rates(k_n_Zn62_to_p_Cu62)*Y(N)*state.rho + screened_rates(k_p_Zn62_to_n_Ga62)*Y(H1)*state.rho;
    jac.set(N, Zn62, scratch);

    scratch = screened_rates(k_Zn63_to_n_Zn62) - screened_rates(k_n_Zn63_to_He4_Ni60)*Y(N)*state.rho - screened_rates(k_n_Zn63_to_Zn64)*Y(N)*state.rho - screened_rates(k_n_Zn63_to_p_Cu63)*Y(N)*state.rho + screened_rates(k_p_Zn63_to_n_Ga63)*Y(H1)*state.rho;
    jac.set(N, Zn63, scratch);

    scratch = screened_rates(k_Zn64_to_n_Zn63) - screened_rates(k_n_Zn64_to_He4_Ni61)*Y(N)*state.rho - screened_rates(k_n_Zn64_to_Zn65)*Y(N)*state.rho - screened_rates(k_n_Zn64_to_p_Cu64)*Y(N)*state.rho + screened_rates(k_p_Zn64_to_n_Ga64)*Y(H1)*state.rho;
    jac.set(N, Zn64, scratch);

    scratch = screened_rates(k_Zn65_to_n_Zn64) - screened_rates(k_n_Zn65_to_He4_Ni62)*Y(N)*state.rho - screened_rates(k_n_Zn65_to_Zn66)*Y(N)*state.rho - screened_rates(k_n_Zn65_to_p_Cu65)*Y(N)*state.rho;
    jac.set(N, Zn65, scratch);

    scratch = screened_rates(k_Zn66_to_n_Zn65) - screened_rates(k_n_Zn66_to_He4_Ni63)*Y(N)*state.rho;
    jac.set(N, Zn66, scratch);

    scratch = -screened_rates(k_n_Ga62_to_Ga63)*Y(N)*state.rho - screened_rates(k_n_Ga62_to_He4_Cu59)*Y(N)*state.rho - screened_rates(k_n_Ga62_to_p_Zn62)*Y(N)*state.rho;
    jac.set(N, Ga62, scratch);

    scratch = screened_rates(k_Ga63_to_n_Ga62) - screened_rates(k_n_Ga63_to_Ga64)*Y(N)*state.rho - screened_rates(k_n_Ga63_to_He4_Cu60)*Y(N)*state.rho - screened_rates(k_n_Ga63_to_p_Zn63)*Y(N)*state.rho + screened_rates(k_p_Ga63_to_n_Ge63)*Y(H1)*state.rho;
    jac.set(N, Ga63, scratch);

    scratch = screened_rates(k_Ga64_to_n_Ga63) - screened_rates(k_n_Ga64_to_He4_Cu61)*Y(N)*state.rho - screened_rates(k_n_Ga64_to_p_Zn64)*Y(N)*state.rho + screened_rates(k_p_Ga64_to_n_Ge64)*Y(H1)*state.rho;
    jac.set(N, Ga64, scratch);

    scratch = -screened_rates(k_n_Ge63_to_Ge64)*Y(N)*state.rho - screened_rates(k_n_Ge63_to_He4_Zn60)*Y(N)*state.rho - screened_rates(k_n_Ge63_to_p_Ga63)*Y(N)*state.rho;
    jac.set(N, Ge63, scratch);

    scratch = screened_rates(k_Ge64_to_n_Ge63) - screened_rates(k_n_Ge64_to_He4_Zn61)*Y(N)*state.rho - screened_rates(k_n_Ge64_to_p_Ga64)*Y(N)*state.rho;
    jac.set(N, Ge64, scratch);

    scratch = screened_rates(k_n_Al25_to_p_Mg25)*Y(Al25)*state.rho + screened_rates(k_n_Al26_to_p_Mg26)*Y(Al26)*state.rho + screened_rates(k_n_Ar36_to_p_Cl36)*Y(Ar36)*state.rho + screened_rates(k_n_Ar37_to_p_Cl37)*Y(Ar37)*state.rho + screened_rates(k_n_B8_to_p_He4_He4)*Y(B8)*state.rho + screened_rates(k_n_Be7_to_p_Li7)*Y(Be7)*state.rho + screened_rates(k_n_Ca40_to_p_K40)*Y(Ca40)*state.rho + screened_rates(k_n_Ca41_to_p_K41)*Y(Ca41)*state.rho + screened_rates(k_n_Cl33_to_p_S33)*Y(Cl33)*state.rho + screened_rates(k_n_Cl34_to_p_S34)*Y(Cl34)*state.rho + screened_rates(k_n_Cl35_to_p_S35)*Y(Cl35)*state.rho + screened_rates(k_n_Cl36_to_p_S36)*Y(Cl36)*state.rho + screened_rates(k_n_Co53_to_p_Fe53)*Y(Co53)*state.rho + screened_rates(k_n_Co54_to_p_Fe54)*Y(Co54)*state.rho + screened_rates(k_n_Co55_to_p_Fe55)*Y(Co55)*state.rho + screened_rates(k_n_Co56_to_p_Fe56)*Y(Co56)*state.rho + screened_rates(k_n_Co57_to_p_Fe57)*Y(Co57)*state.rho + screened_rates(k_n_Co58_to_p_Fe58)*Y(Co58)*state.rho + screened_rates(k_n_Cr48_to_p_V48)*Y(Cr48)*state.rho + screened_rates(k_n_Cr49_to_p_V49)*Y(Cr49)*state.rho + screened_rates(k_n_Cr50_to_p_V50)*Y(Cr50)*state.rho + screened_rates(k_n_Cr51_to_p_V51)*Y(Cr51)*state.rho + screened_rates(k_n_Cr52_to_p_V52)*Y(Cr52)*state.rho + screened_rates(k_n_Cu57_to_p_Ni57)*Y(Cu57)*state.rho + screened_rates(k_n_Cu58_to_p_Ni58)*Y(Cu58)*state.rho + screened_rates(k_n_Cu59_to_p_Ni59)*Y(Cu59)*state.rho + screened_rates(k_n_Cu60_to_p_Ni60)*Y(Cu60)*state.rho + screened_rates(k_n_Cu61_to_p_Ni61)*Y(Cu61)*state.rho + screened_rates(k_n_Cu62_to_p_Ni62)*Y(Cu62)*state.rho + screened_rates(k_n_Cu63_to_p_Ni63)*Y(Cu63)*state.rho + screened_rates(k_n_Cu64_to_p_Ni64)*Y(Cu64)*state.rho + screened_rates(k_n_F17_to_p_O17)*Y(F17)*state.rho + screened_rates(k_n_F18_to_p_O18)*Y(F18)*state.rho + screened_rates(k_n_Fe52_to_p_Mn52)*Y(Fe52)*state.rho + screened_rates(k_n_Fe53_to_p_Mn53)*Y(Fe53)*state.rho + screened_rates(k_n_Fe54_to_p_Mn54)*Y(Fe54)*state.rho + screened_rates(k_n_Fe55_to_p_Mn55)*Y(Fe55)*state.rho + screened_rates(k_n_Ga62_to_p_Zn62)*Y(Ga62)*state.rho + screened_rates(k_n_Ga63_to_p_Zn63)*Y(Ga63)*state.rho + screened_rates(k_n_Ga64_to_p_Zn64)*Y(Ga64)*state.rho + screened_rates(k_n_Ge63_to_p_Ga63)*Y(Ge63)*state.rho + screened_rates(k_n_Ge64_to_p_Ga64)*Y(Ge64)*state.rho + screened_rates(k_n_K37_to_p_Ar37)*Y(K37)*state.rho + screened_rates(k_n_K38_to_p_Ar38)*Y(K38)*state.rho + screened_rates(k_n_K39_to_p_Ar39)*Y(K39)*state.rho + screened_rates(k_n_K40_to_p_Ar40)*Y(K40)*state.rho + screened_rates(k_n_Mg23_to_p_Na23)*Y(Mg23)*state.rho + screened_rates(k_n_Mn50_to_p_Cr50)*Y(Mn50)*state.rho + screened_rates(k_n_Mn51_to_p_Cr51)*Y(Mn51)*state.rho + screened_rates(k_n_Mn52_to_p_Cr52)*Y(Mn52)*state.rho + screened_rates(k_n_Mn53_to_p_Cr53)*Y(Mn53)*state.rho + screened_rates(k_n_Mn54_to_p_Cr54)*Y(Mn54)*state.rho + screened_rates(k_n_N13_to_p_C13)*Y(N13)*state.rho + screened_rates(k_n_N14_to_p_C14)*Y(N14)*state.rho + screened_rates(k_n_Na21_to_p_Ne21)*Y(Na21)*state.rho + screened_rates(k_n_Na22_to_p_Ne22)*Y(Na22)*state.rho + screened_rates(k_n_Ne18_to_p_F18)*Y(Ne18)*state.rho + screened_rates(k_n_Ne19_to_p_F19)*Y(Ne19)*state.rho + screened_rates(k_n_Ni56_to_p_Co56)*Y(Ni56)*state.rho + screened_rates(k_n_Ni57_to_p_Co57)*Y(Ni57)*state.rho + screened_rates(k_n_Ni58_to_p_Co58)*Y(Ni58)*state.rho + screened_rates(k_n_Ni59_to_p_Co59)*Y(Ni59)*state.rho + screened_rates(k_n_O14_to_p_N14)*Y(O14)*state.rho + screened_rates(k_n_O15_to_p_N15)*Y(O15)*state.rho + screened_rates(k_n_P29_to_p_Si29)*Y(P29)*state.rho + screened_rates(k_n_P30_to_p_Si30)*Y(P30)*state.rho + screened_rates(k_n_P31_to_p_Si31)*Y(P31)*state.rho + screened_rates(k_n_P32_to_p_Si32)*Y(P32)*state.rho + screened_rates(k_n_S32_to_p_P32)*Y(S32)*state.rho + screened_rates(k_n_S33_to_p_P33)*Y(S33)*state.rho + screened_rates(k_n_Sc43_to_p_Ca43)*Y(Sc43)*state.rho + screened_rates(k_n_Sc44_to_p_Ca44)*Y(Sc44)*state.rho + screened_rates(k_n_Sc45_to_p_Ca45)*Y(Sc45)*state.rho + screened_rates(k_n_Sc46_to_p_Ca46)*Y(Sc46)*state.rho + screened_rates(k_n_Sc47_to_p_Ca47)*Y(Sc47)*state.rho + screened_rates(k_n_Sc48_to_p_Ca48)*Y(Sc48)*state.rho + screened_rates(k_n_Ti44_to_p_Sc44)*Y(Ti44)*state.rho + screened_rates(k_n_Ti45_to_p_Sc45)*Y(Ti45)*state.rho + screened_rates(k_n_Ti46_to_p_Sc46)*Y(Ti46)*state.rho + screened_rates(k_n_Ti47_to_p_Sc47)*Y(Ti47)*state.rho + screened_rates(k_n_Ti48_to_p_Sc48)*Y(Ti48)*state.rho + screened_rates(k_n_Ti49_to_p_Sc49)*Y(Ti49)*state.rho + screened_rates(k_n_V46_to_p_Ti46)*Y(V46)*state.rho + screened_rates(k_n_V47_to_p_Ti47)*Y(V47)*state.rho + screened_rates(k_n_V48_to_p_Ti48)*Y(V48)*state.rho + screened_rates(k_n_V49_to_p_Ti49)*Y(V49)*state.rho + screened_rates(k_n_V50_to_p_Ti50)*Y(V50)*state.rho + screened_rates(k_n_V51_to_p_Ti51)*Y(V51)*state.rho + screened_rates(k_n_Zn59_to_p_Cu59)*Y(Zn59)*state.rho + screened_rates(k_n_Zn60_to_p_Cu60)*Y(Zn60)*state.rho + screened_rates(k_n_Zn61_to_p_Cu61)*Y(Zn61)*state.rho + screened_rates(k_n_Zn62_to_p_Cu62)*Y(Zn62)*state.rho + screened_rates(k_n_Zn63_to_p_Cu63)*Y(Zn63)*state.rho + screened_rates(k_n_Zn64_to_p_Cu64)*Y(Zn64)*state.rho + screened_rates(k_n_Zn65_to_p_Cu65)*Y(Zn65)*state.rho - 0.5*screened_rates(k_n_p_He4_He4_to_He3_Li7)*amrex::Math::powi<2>(Y(He4))*Y(H1)*amrex::Math::powi<3>(state.rho) - screened_rates(k_n_p_He4_to_Li6)*Y(He4)*Y(H1)*amrex::Math::powi<2>(state.rho) - 1.0*screened_rates(k_n_p_p_to_p_d)*amrex::Math::powi<2>(Y(H1))*amrex::Math::powi<2>(state.rho) - screened_rates(k_n_p_to_d)*Y(H1)*state.rho + screened_rates(k_n_to_p);
    jac.set(H1, N, scratch);

    scratch = -0.5*screened_rates(k_n_p_He4_He4_to_He3_Li7)*amrex::Math::powi<2>(Y(He4))*Y(N)*amrex::Math::powi<3>(state.rho) - screened_rates(k_n_p_He4_to_Li6)*Y(He4)*Y(N)*amrex::Math::powi<2>(state.rho) - 2.0*screened_rates(k_n_p_p_to_p_d)*Y(N)*Y(H1)*amrex::Math::powi<2>(state.rho) - screened_rates(k_n_p_to_d)*Y(N)*state.rho - screened_rates(k_p_Al26_to_He4_Mg23)*Y(Al26)*state.rho - screened_rates(k_p_Al27_to_C12_O16)*Y(Al27)*state.rho - screened_rates(k_p_Al27_to_He4_Mg24)*Y(Al27)*state.rho - screened_rates(k_p_Al27_to_Si28)*Y(Al27)*state.rho - screened_rates(k_p_Ar36_to_He4_Cl33)*Y(Ar36)*state.rho - screened_rates(k_p_Ar36_to_K37)*Y(Ar36)*state.rho - screened_rates(k_p_Ar37_to_He4_Cl34)*Y(Ar37)*state.rho - screened_rates(k_p_Ar37_to_K38)*Y(Ar37)*state.rho - screened_rates(k_p_Ar37_to_n_K37)*Y(Ar37)*state.rho - screened_rates(k_p_Ar38_to_He4_Cl35)*Y(Ar38)*state.rho - screened_rates(k_p_Ar38_to_K39)*Y(Ar38)*state.rho - screened_rates(k_p_Ar38_to_n_K38)*Y(Ar38)*state.rho - screened_rates(k_p_Ar39_to_He4_Cl36)*Y(Ar39)*state.rho - screened_rates(k_p_Ar39_to_K40)*Y(Ar39)*state.rho - screened_rates(k_p_Ar39_to_n_K39)*Y(Ar39)*state.rho - screened_rates(k_p_Ar40_to_He4_Cl37)*Y(Ar40)*state.rho - screened_rates(k_p_Ar40_to_K41)*Y(Ar40)*state.rho - screened_rates(k_p_Ar40_to_n_K40)*Y(Ar40)*state.rho - screened_rates(k_p_B10_to_He4_Be7)*Y(B10)*state.rho - screened_rates(k_p_B11_to_C12)*Y(B11)*state.rho - screened_rates(k_p_B11_to_He4_He4_He4)*Y(B11)*state.rho - screened_rates(k_p_Be7_to_B8)*Y(Be7)*state.rho - screened_rates(k_p_Be9_to_B10)*Y(Be9)*state.rho - screened_rates(k_p_Be9_to_He4_Li6)*Y(Be9)*state.rho - screened_rates(k_p_Be9_to_d_He4_He4)*Y(Be9)*state.rho - screened_rates(k_p_C12_to_N13)*Y(C12)*state.rho - screened_rates(k_p_C13_to_He4_B10)*Y(C13)*state.rho - screened_rates(k_p_C13_to_N14)*Y(C13)*state.rho - screened_rates(k_p_C13_to_n_N13)*Y(C13)*state.rho - screened_rates(k_p_C14_to_He4_B11)*Y(C14)*state.rho - screened_rates(k_p_C14_to_N15)*Y(C14)*state.rho - screened_rates(k_p_C14_to_n_N14)*Y(C14)*state.rho - screened_rates(k_p_Ca40_to_He4_K37)*Y(Ca40)*state.rho - screened_rates(k_p_Ca41_to_He4_K38)*Y(Ca41)*state.rho - screened_rates(k_p_Ca42_to_He4_K39)*Y(Ca42)*state.rho - screened_rates(k_p_Ca42_to_Sc43)*Y(Ca42)*state.rho - screened_rates(k_p_Ca43_to_He4_K40)*Y(Ca43)*state.rho - screened_rates(k_p_Ca43_to_Sc44)*Y(Ca43)*state.rho - screened_rates(k_p_Ca43_to_n_Sc43)*Y(Ca43)*state.rho - screened_rates(k_p_Ca44_to_He4_K41)*Y(Ca44)*state.rho - screened_rates(k_p_Ca44_to_Sc45)*Y(Ca44)*state.rho - screened_rates(k_p_Ca44_to_n_Sc44)*Y(Ca44)*state.rho - screened_rates(k_p_Ca45_to_Sc46)*Y(Ca45)*state.rho - screened_rates(k_p_Ca45_to_n_Sc45)*Y(Ca45)*state.rho - screened_rates(k_p_Ca46_to_Sc47)*Y(Ca46)*state.rho - screened_rates(k_p_Ca46_to_n_Sc46)*Y(Ca46)*state.rho - screened_rates(k_p_Ca47_to_Sc48)*Y(Ca47)*state.rho - screened_rates(k_p_Ca47_to_n_Sc47)*Y(Ca47)*state.rho - screened_rates(k_p_Ca48_to_Sc49)*Y(Ca48)*state.rho - screened_rates(k_p_Ca48_to_n_Sc48)*Y(Ca48)*state.rho - screened_rates(k_p_Cl35_to_Ar36)*Y(Cl35)*state.rho - screened_rates(k_p_Cl35_to_He4_S32)*Y(Cl35)*state.rho - screened_rates(k_p_Cl36_to_Ar37)*Y(Cl36)*state.rho - screened_rates(k_p_Cl36_to_He4_S33)*Y(Cl36)*state.rho - screened_rates(k_p_Cl36_to_n_Ar36)*Y(Cl36)*state.rho - screened_rates(k_p_Cl37_to_Ar38)*Y(Cl37)*state.rho - screened_rates(k_p_Cl37_to_He4_S34)*Y(Cl37)*state.rho - screened_rates(k_p_Cl37_to_n_Ar37)*Y(Cl37)*state.rho - screened_rates(k_p_Co55_to_He4_Fe52)*Y(Co55)*state.rho - screened_rates(k_p_Co55_to_Ni56)*Y(Co55)*state.rho - screened_rates(k_p_Co56_to_He4_Fe53)*Y(Co56)*state.rho - screened_rates(k_p_Co56_to_Ni57)*Y(Co56)*state.rho - screened_rates(k_p_Co56_to_n_Ni56)*Y(Co56)*state.rho - screened_rates(k_p_Co57_to_He4_Fe54)*Y(Co57)*state.rho - screened_rates(k_p_Co57_to_Ni58)*Y(Co57)*state.rho - screened_rates(k_p_Co57_to_n_Ni57)*Y(Co57)*state.rho - screened_rates(k_p_Co58_to_He4_Fe55)*Y(Co58)*state.rho - screened_rates(k_p_Co58_to_Ni59)*Y(Co58)*state.rho - screened_rates(k_p_Co58_to_n_Ni58)*Y(Co58)*state.rho - screened_rates(k_p_Co59_to_He4_Fe56)*Y(Co59)*state.rho - screened_rates(k_p_Co59_to_Ni60)*Y(Co59)*state.rho - screened_rates(k_p_Co59_to_n_Ni59)*Y(Co59)*state.rho - screened_rates(k_p_Cr49_to_He4_V46)*Y(Cr49)*state.rho - screened_rates(k_p_Cr49_to_Mn50)*Y(Cr49)*state.rho - screened_rates(k_p_Cr50_to_He4_V47)*Y(Cr50)*state.rho - screened_rates(k_p_Cr50_to_Mn51)*Y(Cr50)*state.rho - screened_rates(k_p_Cr50_to_n_Mn50)*Y(Cr50)*state.rho - screened_rates(k_p_Cr51_to_He4_V48)*Y(Cr51)*state.rho - screened_rates(k_p_Cr51_to_Mn52)*Y(Cr51)*state.rho - screened_rates(k_p_Cr51_to_n_Mn51)*Y(Cr51)*state.rho - screened_rates(k_p_Cr52_to_He4_V49)*Y(Cr52)*state.rho - screened_rates(k_p_Cr52_to_Mn53)*Y(Cr52)*state.rho - screened_rates(k_p_Cr52_to_n_Mn52)*Y(Cr52)*state.rho - screened_rates(k_p_Cr53_to_He4_V50)*Y(Cr53)*state.rho - screened_rates(k_p_Cr53_to_Mn54)*Y(Cr53)*state.rho - screened_rates(k_p_Cr53_to_n_Mn53)*Y(Cr53)*state.rho - screened_rates(k_p_Cr54_to_He4_V51)*Y(Cr54)*state.rho - screened_rates(k_p_Cr54_to_Mn55)*Y(Cr54)*state.rho - screened_rates(k_p_Cr54_to_n_Mn54)*Y(Cr54)*state.rho - screened_rates(k_p_Cu58_to_Zn59)*Y(Cu58)*state.rho - screened_rates(k_p_Cu59_to_He4_Ni56)*Y(Cu59)*state.rho - screened_rates(k_p_Cu59_to_Zn60)*Y(Cu59)*state.rho - screened_rates(k_p_Cu59_to_n_Zn59)*Y(Cu59)*state.rho - screened_rates(k_p_Cu60_to_He4_Ni57)*Y(Cu60)*state.rho - screened_rates(k_p_Cu60_to_Zn61)*Y(Cu60)*state.rho - screened_rates(k_p_Cu60_to_n_Zn60)*Y(Cu60)*state.rho - screened_rates(k_p_Cu61_to_He4_Ni58)*Y(Cu61)*state.rho - screened_rates(k_p_Cu61_to_Zn62)*Y(Cu61)*state.rho - screened_rates(k_p_Cu61_to_n_Zn61)*Y(Cu61)*state.rho - screened_rates(k_p_Cu62_to_He4_Ni59)*Y(Cu62)*state.rho - screened_rates(k_p_Cu62_to_Zn63)*Y(Cu62)*state.rho - screened_rates(k_p_Cu62_to_n_Zn62)*Y(Cu62)*state.rho - screened_rates(k_p_Cu63_to_He4_Ni60)*Y(Cu63)*state.rho - screened_rates(k_p_Cu63_to_Zn64)*Y(Cu63)*state.rho - screened_rates(k_p_Cu63_to_n_Zn63)*Y(Cu63)*state.rho - screened_rates(k_p_Cu64_to_He4_Ni61)*Y(Cu64)*state.rho - screened_rates(k_p_Cu64_to_Zn65)*Y(Cu64)*state.rho - screened_rates(k_p_Cu64_to_n_Zn64)*Y(Cu64)*state.rho - screened_rates(k_p_Cu65_to_He4_Ni62)*Y(Cu65)*state.rho - screened_rates(k_p_Cu65_to_Zn66)*Y(Cu65)*state.rho - screened_rates(k_p_Cu65_to_n_Zn65)*Y(Cu65)*state.rho - screened_rates(k_p_F17_to_He4_O14)*Y(F17)*state.rho - screened_rates(k_p_F17_to_Ne18)*Y(F17)*state.rho - screened_rates(k_p_F18_to_He4_O15)*Y(F18)*state.rho - screened_rates(k_p_F18_to_Ne19)*Y(F18)*state.rho - screened_rates(k_p_F18_to_n_Ne18)*Y(F18)*state.rho - screened_rates(k_p_F19_to_He4_O16)*Y(F19)*state.rho - screened_rates(k_p_F19_to_Ne20)*Y(F19)*state.rho - screened_rates(k_p_F19_to_n_Ne19)*Y(F19)*state.rho - screened_rates(k_p_Fe52_to_Co53)*Y(Fe52)*state.rho - screened_rates(k_p_Fe53_to_Co54)*Y(Fe53)*state.rho - screened_rates(k_p_Fe53_to_He4_Mn50)*Y(Fe53)*state.rho - screened_rates(k_p_Fe53_to_n_Co53)*Y(Fe53)*state.rho - screened_rates(k_p_Fe54_to_Co55)*Y(Fe54)*state.rho - screened_rates(k_p_Fe54_to_He4_Mn51)*Y(Fe54)*state.rho - screened_rates(k_p_Fe54_to_n_Co54)*Y(Fe54)*state.rho - screened_rates(k_p_Fe55_to_Co56)*Y(Fe55)*state.rho - screened_rates(k_p_Fe55_to_He4_Mn52)*Y(Fe55)*state.rho - screened_rates(k_p_Fe55_to_n_Co55)*Y(Fe55)*state.rho - screened_rates(k_p_Fe56_to_Co57)*Y(Fe56)*state.rho - screened_rates(k_p_Fe56_to_He4_Mn53)*Y(Fe56)*state.rho - screened_rates(k_p_Fe56_to_n_Co56)*Y(Fe56)*state.rho - screened_rates(k_p_Fe57_to_Co58)*Y(Fe57)*state.rho - screened_rates(k_p_Fe57_to_He4_Mn54)*Y(Fe57)*state.rho - screened_rates(k_p_Fe57_to_n_Co57)*Y(Fe57)*state.rho - screened_rates(k_p_Fe58_to_Co59)*Y(Fe58)*state.rho - screened_rates(k_p_Fe58_to_He4_Mn55)*Y(Fe58)*state.rho - screened_rates(k_p_Fe58_to_n_Co58)*Y(Fe58)*state.rho - screened_rates(k_p_Ga62_to_Ge63)*Y(Ga62)*state.rho - screened_rates(k_p_Ga62_to_He4_Zn59)*Y(Ga62)*state.rho - screened_rates(k_p_Ga63_to_Ge64)*Y(Ga63)*state.rho - screened_rates(k_p_Ga63_to_He4_Zn60)*Y(Ga63)*state.rho - screened_rates(k_p_Ga63_to_n_Ge63)*Y(Ga63)*state.rho - screened_rates(k_p_Ga64_to_He4_Zn61)*Y(Ga64)*state.rho - screened_rates(k_p_Ga64_to_n_Ge64)*Y(Ga64)*state.rho - screened_rates(k_p_He3_to_He4_weak_bet_pos_)*Y(He3)*state.rho - 0.5*screened_rates(k_p_He4_He4_to_d_Be7)*amrex::Math::powi<2>(Y(He4))*amrex::Math::powi<2>(state.rho) - 0.5*screened_rates(k_p_He4_He4_to_n_B8)*amrex::Math::powi<2>(Y(He4))*amrex::Math::powi<2>(state.rho) - screened_rates(k_p_He4_to_d_He3)*Y(He4)*state.rho - screened_rates(k_p_K39_to_Ca40)*Y(K39)*state.rho - screened_rates(k_p_K39_to_He4_Ar36)*Y(K39)*state.rho - screened_rates(k_p_K40_to_Ca41)*Y(K40)*state.rho - screened_rates(k_p_K40_to_He4_Ar37)*Y(K40)*state.rho - screened_rates(k_p_K40_to_n_Ca40)*Y(K40)*state.rho - screened_rates(k_p_K41_to_Ca42)*Y(K41)*state.rho - screened_rates(k_p_K41_to_He4_Ar38)*Y(K41)*state.rho - screened_rates(k_p_K41_to_n_Ca41)*Y(K41)*state.rho - screened_rates(k_p_Li6_to_Be7)*Y(Li6)*state.rho - screened_rates(k_p_Li6_to_He4_He3)*Y(Li6)*state.rho - screened_rates(k_p_Li7_to_He4_He4)*Y(Li7)*state.rho - screened_rates(k_p_Li7_to_d_Li6)*Y(Li7)*state.rho - screened_rates(k_p_Li7_to_n_Be7)*Y(Li7)*state.rho - screened_rates(k_p_Mg24_to_Al25)*Y(Mg24)*state.rho - screened_rates(k_p_Mg24_to_He4_Na21)*Y(Mg24)*state.rho - screened_rates(k_p_Mg25_to_Al26)*Y(Mg25)*state.rho - screened_rates(k_p_Mg25_to_He4_Na22)*Y(Mg25)*state.rho - screened_rates(k_p_Mg25_to_n_Al25)*Y(Mg25)*state.rho - screened_rates(k_p_Mg26_to_Al27)*Y(Mg26)*state.rho - screened_rates(k_p_Mg26_to_He4_Na23)*Y(Mg26)*state.rho - screened_rates(k_p_Mg26_to_n_Al26)*Y(Mg26)*state.rho - screened_rates(k_p_Mn51_to_Fe52)*Y(Mn51)*state.rho - screened_rates(k_p_Mn51_to_He4_Cr48)*Y(Mn51)*state.rho - screened_rates(k_p_Mn52_to_Fe53)*Y(Mn52)*state.rho - screened_rates(k_p_Mn52_to_He4_Cr49)*Y(Mn52)*state.rho - screened_rates(k_p_Mn52_to_n_Fe52)*Y(Mn52)*state.rho - screened_rates(k_p_Mn53_to_Fe54)*Y(Mn53)*state.rho - screened_rates(k_p_Mn53_to_He4_Cr50)*Y(Mn53)*state.rho - screened_rates(k_p_Mn53_to_n_Fe53)*Y(Mn53)*state.rho - screened_rates(k_p_Mn54_to_Fe55)*Y(Mn54)*state.rho - screened_rates(k_p_Mn54_to_He4_Cr51)*Y(Mn54)*state.rho - screened_rates(k_p_Mn54_to_n_Fe54)*Y(Mn54)*state.rho - screened_rates(k_p_Mn55_to_Fe56)*Y(Mn55)*state.rho - screened_rates(k_p_Mn55_to_He4_Cr52)*Y(Mn55)*state.rho - screened_rates(k_p_Mn55_to_n_Fe55)*Y(Mn55)*state.rho - screened_rates(k_p_N13_to_O14)*Y(N13)*state.rho - screened_rates(k_p_N14_to_O15)*Y(N14)*state.rho - screened_rates(k_p_N14_to_n_O14)*Y(N14)*state.rho - screened_rates(k_p_N15_to_He4_C12)*Y(N15)*state.rho - screened_rates(k_p_N15_to_O16)*Y(N15)*state.rho - screened_rates(k_p_N15_to_n_O15)*Y(N15)*state.rho - screened_rates(k_p_Na21_to_He4_Ne18)*Y(Na21)*state.rho - screened_rates(k_p_Na22_to_He4_Ne19)*Y(Na22)*state.rho - screened_rates(k_p_Na22_to_Mg23)*Y(Na22)*state.rho - screened_rates(k_p_Na23_to_C12_C12)*Y(Na23)*state.rho - screened_rates(k_p_Na23_to_He4_Ne20)*Y(Na23)*state.rho - screened_rates(k_p_Na23_to_Mg24)*Y(Na23)*state.rho - screened_rates(k_p_Na23_to_n_Mg23)*Y(Na23)*state.rho - screened_rates(k_p_Ne20_to_He4_F17)*Y(Ne20)*state.rho - screened_rates(k_p_Ne20_to_Na21)*Y(Ne20)*state.rho - screened_rates(k_p_Ne21_to_He4_F18)*Y(Ne21)*state.rho - screened_rates(k_p_Ne21_to_Na22)*Y(Ne21)*state.rho - screened_rates(k_p_Ne21_to_n_Na21)*Y(Ne21)*state.rho - screened_rates(k_p_Ne22_to_He4_F19)*Y(Ne22)*state.rho - screened_rates(k_p_Ne22_to_Na23)*Y(Ne22)*state.rho - screened_rates(k_p_Ne22_to_n_Na22)*Y(Ne22)*state.rho - screened_rates(k_p_Ni56_to_Cu57)*Y(Ni56)*state.rho - screened_rates(k_p_Ni56_to_He4_Co53)*Y(Ni56)*state.rho - screened_rates(k_p_Ni57_to_Cu58)*Y(Ni57)*state.rho - screened_rates(k_p_Ni57_to_He4_Co54)*Y(Ni57)*state.rho - screened_rates(k_p_Ni57_to_n_Cu57)*Y(Ni57)*state.rho - screened_rates(k_p_Ni58_to_Cu59)*Y(Ni58)*state.rho - screened_rates(k_p_Ni58_to_He4_Co55)*Y(Ni58)*state.rho - screened_rates(k_p_Ni58_to_n_Cu58)*Y(Ni58)*state.rho - screened_rates(k_p_Ni59_to_Cu60)*Y(Ni59)*state.rho - screened_rates(k_p_Ni59_to_He4_Co56)*Y(Ni59)*state.rho - screened_rates(k_p_Ni59_to_n_Cu59)*Y(Ni59)*state.rho - screened_rates(k_p_Ni60_to_Cu61)*Y(Ni60)*state.rho - screened_rates(k_p_Ni60_to_He4_Co57)*Y(Ni60)*state.rho - screened_rates(k_p_Ni60_to_n_Cu60)*Y(Ni60)*state.rho - screened_rates(k_p_Ni61_to_Cu62)*Y(Ni61)*state.rho - screened_rates(k_p_Ni61_to_He4_Co58)*Y(Ni61)*state.rho - screened_rates(k_p_Ni61_to_n_Cu61)*Y(Ni61)*state.rho - screened_rates(k_p_Ni62_to_Cu63)*Y(Ni62)*state.rho - screened_rates(k_p_Ni62_to_He4_Co59)*Y(Ni62)*state.rho - screened_rates(k_p_Ni62_to_n_Cu62)*Y(Ni62)*state.rho - screened_rates(k_p_Ni63_to_Cu64)*Y(Ni63)*state.rho - screened_rates(k_p_Ni63_to_n_Cu63)*Y(Ni63)*state.rho - screened_rates(k_p_Ni64_to_Cu65)*Y(Ni64)*state.rho - screened_rates(k_p_Ni64_to_n_Cu64)*Y(Ni64)*state.rho - screened_rates(k_p_O16_to_F17)*Y(O16)*state.rho - screened_rates(k_p_O16_to_He4_N13)*Y(O16)*state.rho - screened_rates(k_p_O17_to_F18)*Y(O17)*state.rho - screened_rates(k_p_O17_to_He4_N14)*Y(O17)*state.rho - screened_rates(k_p_O17_to_n_F17)*Y(O17)*state.rho - screened_rates(k_p_O18_to_F19)*Y(O18)*state.rho - screened_rates(k_p_O18_to_He4_N15)*Y(O18)*state.rho - screened_rates(k_p_O18_to_n_F18)*Y(O18)*state.rho - screened_rates(k_p_P31_to_C12_Ne20)*Y(P31)*state.rho - screened_rates(k_p_P31_to_He4_Si28)*Y(P31)*state.rho - screened_rates(k_p_P31_to_O16_O16)*Y(P31)*state.rho - screened_rates(k_p_P31_to_S32)*Y(P31)*state.rho - screened_rates(k_p_P32_to_He4_Si29)*Y(P32)*state.rho - screened_rates(k_p_P32_to_S33)*Y(P32)*state.rho - screened_rates(k_p_P32_to_n_S32)*Y(P32)*state.rho - screened_rates(k_p_P33_to_He4_Si30)*Y(P33)*state.rho - screened_rates(k_p_P33_to_S34)*Y(P33)*state.rho - screened_rates(k_p_P33_to_n_S33)*Y(P33)*state.rho - screened_rates(k_p_S32_to_Cl33)*Y(S32)*state.rho - screened_rates(k_p_S32_to_He4_P29)*Y(S32)*state.rho - screened_rates(k_p_S33_to_Cl34)*Y(S33)*state.rho - screened_rates(k_p_S33_to_He4_P30)*Y(S33)*state.rho - screened_rates(k_p_S33_to_n_Cl33)*Y(S33)*state.rho - screened_rates(k_p_S34_to_Cl35)*Y(S34)*state.rho - screened_rates(k_p_S34_to_He4_P31)*Y(S34)*state.rho - screened_rates(k_p_S34_to_n_Cl34)*Y(S34)*state.rho - screened_rates(k_p_S35_to_Cl36)*Y(S35)*state.rho - screened_rates(k_p_S35_to_He4_P32)*Y(S35)*state.rho - screened_rates(k_p_S35_to_n_Cl35)*Y(S35)*state.rho - screened_rates(k_p_S36_to_Cl37)*Y(S36)*state.rho - screened_rates(k_p_S36_to_He4_P33)*Y(S36)*state.rho - screened_rates(k_p_S36_to_n_Cl36)*Y(S36)*state.rho - screened_rates(k_p_Sc43_to_He4_Ca40)*Y(Sc43)*state.rho - screened_rates(k_p_Sc43_to_Ti44)*Y(Sc43)*state.rho - screened_rates(k_p_Sc44_to_He4_Ca41)*Y(Sc44)*state.rho - screened_rates(k_p_Sc44_to_Ti45)*Y(Sc44)*state.rho - screened_rates(k_p_Sc44_to_n_Ti44)*Y(Sc44)*state.rho - screened_rates(k_p_Sc45_to_He4_Ca42)*Y(Sc45)*state.rho - screened_rates(k_p_Sc45_to_Ti46)*Y(Sc45)*state.rho - screened_rates(k_p_Sc45_to_n_Ti45)*Y(Sc45)*state.rho - screened_rates(k_p_Sc46_to_He4_Ca43)*Y(Sc46)*state.rho - screened_rates(k_p_Sc46_to_Ti47)*Y(Sc46)*state.rho - screened_rates(k_p_Sc46_to_n_Ti46)*Y(Sc46)*state.rho - screened_rates(k_p_Sc47_to_He4_Ca44)*Y(Sc47)*state.rho - screened_rates(k_p_Sc47_to_Ti48)*Y(Sc47)*state.rho - screened_rates(k_p_Sc47_to_n_Ti47)*Y(Sc47)*state.rho - screened_rates(k_p_Sc48_to_He4_Ca45)*Y(Sc48)*state.rho - screened_rates(k_p_Sc48_to_Ti49)*Y(Sc48)*state.rho - screened_rates(k_p_Sc48_to_n_Ti48)*Y(Sc48)*state.rho - screened_rates(k_p_Sc49_to_He4_Ca46)*Y(Sc49)*state.rho - screened_rates(k_p_Sc49_to_Ti50)*Y(Sc49)*state.rho - screened_rates(k_p_Sc49_to_n_Ti49)*Y(Sc49)*state.rho - screened_rates(k_p_Si28_to_He4_Al25)*Y(Si28)*state.rho - screened_rates(k_p_Si28_to_P29)*Y(Si28)*state.rho - screened_rates(k_p_Si29_to_He4_Al26)*Y(Si29)*state.rho - screened_rates(k_p_Si29_to_P30)*Y(Si29)*state.rho - screened_rates(k_p_Si29_to_n_P29)*Y(Si29)*state.rho - screened_rates(k_p_Si30_to_He4_Al27)*Y(Si30)*state.rho - screened_rates(k_p_Si30_to_P31)*Y(Si30)*state.rho - screened_rates(k_p_Si30_to_n_P30)*Y(Si30)*state.rho - screened_rates(k_p_Si31_to_P32)*Y(Si31)*state.rho - screened_rates(k_p_Si31_to_n_P31)*Y(Si31)*state.rho - screened_rates(k_p_Si32_to_P33)*Y(Si32)*state.rho - screened_rates(k_p_Si32_to_n_P32)*Y(Si32)*state.rho - screened_rates(k_p_Ti45_to_V46)*Y(Ti45)*state.rho - screened_rates(k_p_Ti46_to_He4_Sc43)*Y(Ti46)*state.rho - screened_rates(k_p_Ti46_to_V47)*Y(Ti46)*state.rho - screened_rates(k_p_Ti46_to_n_V46)*Y(Ti46)*state.rho - screened_rates(k_p_Ti47_to_He4_Sc44)*Y(Ti47)*state.rho - screened_rates(k_p_Ti47_to_V48)*Y(Ti47)*state.rho - screened_rates(k_p_Ti47_to_n_V47)*Y(Ti47)*state.rho - screened_rates(k_p_Ti48_to_He4_Sc45)*Y(Ti48)*state.rho - screened_rates(k_p_Ti48_to_V49)*Y(Ti48)*state.rho - screened_rates(k_p_Ti48_to_n_V48)*Y(Ti48)*state.rho - screened_rates(k_p_Ti49_to_He4_Sc46)*Y(Ti49)*state.rho - screened_rates(k_p_Ti49_to_V50)*Y(Ti49)*state.rho - screened_rates(k_p_Ti49_to_n_V49)*Y(Ti49)*state.rho - screened_rates(k_p_Ti50_to_He4_Sc47)*Y(Ti50)*state.rho - screened_rates(k_p_Ti50_to_V51)*Y(Ti50)*state.rho - screened_rates(k_p_Ti50_to_n_V50)*Y(Ti50)*state.rho - screened_rates(k_p_Ti51_to_He4_Sc48)*Y(Ti51)*state.rho - screened_rates(k_p_Ti51_to_V52)*Y(Ti51)*state.rho - screened_rates(k_p_Ti51_to_n_V51)*Y(Ti51)*state.rho - screened_rates(k_p_V47_to_Cr48)*Y(V47)*state.rho - screened_rates(k_p_V47_to_He4_Ti44)*Y(V47)*state.rho - screened_rates(k_p_V48_to_Cr49)*Y(V48)*state.rho - screened_rates(k_p_V48_to_He4_Ti45)*Y(V48)*state.rho - screened_rates(k_p_V48_to_n_Cr48)*Y(V48)*state.rho - screened_rates(k_p_V49_to_Cr50)*Y(V49)*state.rho - screened_rates(k_p_V49_to_He4_Ti46)*Y(V49)*state.rho - screened_rates(k_p_V49_to_n_Cr49)*Y(V49)*state.rho - screened_rates(k_p_V50_to_Cr51)*Y(V50)*state.rho - screened_rates(k_p_V50_to_He4_Ti47)*Y(V50)*state.rho - screened_rates(k_p_V50_to_n_Cr50)*Y(V50)*state.rho - screened_rates(k_p_V51_to_Cr52)*Y(V51)*state.rho - screened_rates(k_p_V51_to_He4_Ti48)*Y(V51)*state.rho - screened_rates(k_p_V51_to_n_Cr51)*Y(V51)*state.rho - screened_rates(k_p_V52_to_Cr53)*Y(V52)*state.rho - screened_rates(k_p_V52_to_He4_Ti49)*Y(V52)*state.rho - screened_rates(k_p_V52_to_n_Cr52)*Y(V52)*state.rho - screened_rates(k_p_Zn60_to_He4_Cu57)*Y(Zn60)*state.rho - screened_rates(k_p_Zn61_to_Ga62)*Y(Zn61)*state.rho - screened_rates(k_p_Zn61_to_He4_Cu58)*Y(Zn61)*state.rho - screened_rates(k_p_Zn62_to_Ga63)*Y(Zn62)*state.rho - screened_rates(k_p_Zn62_to_He4_Cu59)*Y(Zn62)*state.rho - screened_rates(k_p_Zn62_to_n_Ga62)*Y(Zn62)*state.rho - screened_rates(k_p_Zn63_to_Ga64)*Y(Zn63)*state.rho - screened_rates(k_p_Zn63_to_He4_Cu60)*Y(Zn63)*state.rho - screened_rates(k_p_Zn63_to_n_Ga63)*Y(Zn63)*state.rho - screened_rates(k_p_Zn64_to_He4_Cu61)*Y(Zn64)*state.rho - screened_rates(k_p_Zn64_to_n_Ga64)*Y(Zn64)*state.rho - screened_rates(k_p_Zn65_to_He4_Cu62)*Y(Zn65)*state.rho - screened_rates(k_p_Zn66_to_He4_Cu63)*Y(Zn66)*state.rho - screened_rates(k_p_d_to_He3)*Y(H2)*state.rho + 2*screened_rates(k_p_d_to_n_p_p)*Y(H2)*state.rho - 1.0*screened_rates(k_p_p_He4_He4_to_He3_Be7)*amrex::Math::powi<2>(Y(He4))*Y(H1)*amrex::Math::powi<3>(state.rho) - 2.0*screened_rates(k_p_p_He4_to_He3_He3)*Y(He4)*Y(H1)*amrex::Math::powi<2>(state.rho) - 2.0*screened_rates(k_p_p_to_d_weak_bet_pos_)*Y(H1)*state.rho - 2.0*screened_rates(k_p_p_to_d_weak_electron_capture)*Y(H1)*amrex::Math::powi<2>(state.rho)*state.y_e - screened_rates(k_p_to_n);
    jac.set(H1, H1, scratch);

    scratch = screened_rates(k_d_Be7_to_p_He4_He4)*Y(Be7)*state.rho + screened_rates(k_d_He3_to_p_He4)*Y(He3)*state.rho + 0.5*screened_rates(k_d_He4_He4_to_p_Be9)*amrex::Math::powi<2>(Y(He4))*amrex::Math::powi<2>(state.rho) + screened_rates(k_d_Li6_to_p_Li7)*Y(Li6)*state.rho + screened_rates(k_d_to_n_p) - screened_rates(k_p_d_to_He3)*Y(H1)*state.rho + 2*screened_rates(k_p_d_to_n_p_p)*Y(H1)*state.rho;
    jac.set(H1, H2, scratch);

    scratch = 2.0*screened_rates(k_He3_Be7_to_p_p_He4_He4)*Y(Be7)*state.rho + 2.0*screened_rates(k_He3_He3_to_p_p_He4)*Y(He3)*state.rho + screened_rates(k_He3_Li7_to_n_p_He4_He4)*Y(Li7)*state.rho + screened_rates(k_He3_to_p_d) + screened_rates(k_He4_He3_to_p_Li6)*Y(He4)*state.rho + screened_rates(k_d_He3_to_p_He4)*Y(H2)*state.rho - screened_rates(k_p_He3_to_He4_weak_bet_pos_)*Y(H1)*state.rho;
    jac.set(H1, He3, scratch);

    scratch = screened_rates(k_He4_Al25_to_p_Si28)*Y(Al25)*state.rho + screened_rates(k_He4_Al26_to_p_Si29)*Y(Al26)*state.rho + screened_rates(k_He4_Al27_to_p_Si30)*Y(Al27)*state.rho + screened_rates(k_He4_Ar36_to_p_K39)*Y(Ar36)*state.rho + screened_rates(k_He4_Ar37_to_p_K40)*Y(Ar37)*state.rho + screened_rates(k_He4_Ar38_to_p_K41)*Y(Ar38)*state.rho + screened_rates(k_He4_B10_to_p_C13)*Y(B10)*state.rho + screened_rates(k_He4_B11_to_p_C14)*Y(B11)*state.rho + screened_rates(k_He4_Be7_to_p_B10)*Y(Be7)*state.rho + screened_rates(k_He4_C12_to_p_N15)*Y(C12)*state.rho + screened_rates(k_He4_Ca40_to_p_Sc43)*Y(Ca40)*state.rho + screened_rates(k_He4_Ca41_to_p_Sc44)*Y(Ca41)*state.rho + screened_rates(k_He4_Ca42_to_p_Sc45)*Y(Ca42)*state.rho + screened_rates(k_He4_Ca43_to_p_Sc46)*Y(Ca43)*state.rho + screened_rates(k_He4_Ca44_to_p_Sc47)*Y(Ca44)*state.rho + screened_rates(k_He4_Ca45_to_p_Sc48)*Y(Ca45)*state.rho + screened_rates(k_He4_Ca46_to_p_Sc49)*Y(Ca46)*state.rho + screened_rates(k_He4_Cl33_to_p_Ar36)*Y(Cl33)*state.rho + screened_rates(k_He4_Cl34_to_p_Ar37)*Y(Cl34)*state.rho + screened_rates(k_He4_Cl35_to_p_Ar38)*Y(Cl35)*state.rho + screened_rates(k_He4_Cl36_to_p_Ar39)*Y(Cl36)*state.rho + screened_rates(k_He4_Cl37_to_p_Ar40)*Y(Cl37)*state.rho + screened_rates(k_He4_Co53_to_p_Ni56)*Y(Co53)*state.rho + screened_rates(k_He4_Co54_to_p_Ni57)*Y(Co54)*state.rho + screened_rates(k_He4_Co55_to_p_Ni58)*Y(Co55)*state.rho + screened_rates(k_He4_Co56_to_p_Ni59)*Y(Co56)*state.rho + screened_rates(k_He4_Co57_to_p_Ni60)*Y(Co57)*state.rho + screened_rates(k_He4_Co58_to_p_Ni61)*Y(Co58)*state.rho + screened_rates(k_He4_Co59_to_p_Ni62)*Y(Co59)*state.rho + screened_rates(k_He4_Cr48_to_p_Mn51)*Y(Cr48)*state.rho + screened_rates(k_He4_Cr49_to_p_Mn52)*Y(Cr49)*state.rho + screened_rates(k_He4_Cr50_to_p_Mn53)*Y(Cr50)*state.rho + screened_rates(k_He4_Cr51_to_p_Mn54)*Y(Cr51)*state.rho + screened_rates(k_He4_Cr52_to_p_Mn55)*Y(Cr52)*state.rho + screened_rates(k_He4_Cu57_to_p_Zn60)*Y(Cu57)*state.rho + screened_rates(k_He4_Cu58_to_p_Zn61)*Y(Cu58)*state.rho + screened_rates(k_He4_Cu59_to_p_Zn62)*Y(Cu59)*state.rho + screened_rates(k_He4_Cu60_to_p_Zn63)*Y(Cu60)*state.rho + screened_rates(k_He4_Cu61_to_p_Zn64)*Y(Cu61)*state.rho + screened_rates(k_He4_Cu62_to_p_Zn65)*Y(Cu62)*state.rho + screened_rates(k_He4_Cu63_to_p_Zn66)*Y(Cu63)*state.rho + screened_rates(k_He4_F17_to_p_Ne20)*Y(F17)*state.rho + screened_rates(k_He4_F18_to_p_Ne21)*Y(F18)*state.rho + screened_rates(k_He4_F19_to_p_Ne22)*Y(F19)*state.rho + screened_rates(k_He4_Fe52_to_p_Co55)*Y(Fe52)*state.rho + screened_rates(k_He4_Fe53_to_p_Co56)*Y(Fe53)*state.rho + screened_rates(k_He4_Fe54_to_p_Co57)*Y(Fe54)*state.rho + screened_rates(k_He4_Fe55_to_p_Co58)*Y(Fe55)*state.rho + screened_rates(k_He4_Fe56_to_p_Co59)*Y(Fe56)*state.rho + screened_rates(k_He4_He3_to_p_Li6)*Y(He3)*state.rho + 0.5*screened_rates(k_He4_He4_He4_to_p_B11)*amrex::Math::powi<2>(Y(He4))*amrex::Math::powi<2>(state.rho) + 1.0*screened_rates(k_He4_He4_to_p_Li7)*Y(He4)*state.rho + screened_rates(k_He4_K37_to_p_Ca40)*Y(K37)*state.rho + screened_rates(k_He4_K38_to_p_Ca41)*Y(K38)*state.rho + screened_rates(k_He4_K39_to_p_Ca42)*Y(K39)*state.rho + screened_rates(k_He4_K40_to_p_Ca43)*Y(K40)*state.rho + screened_rates(k_He4_K41_to_p_Ca44)*Y(K41)*state.rho + screened_rates(k_He4_Li6_to_p_Be9)*Y(Li6)*state.rho + screened_rates(k_He4_Mg23_to_p_Al26)*Y(Mg23)*state.rho + screened_rates(k_He4_Mg24_to_p_Al27)*Y(Mg24)*state.rho + screened_rates(k_He4_Mn50_to_p_Fe53)*Y(Mn50)*state.rho + screened_rates(k_He4_Mn51_to_p_Fe54)*Y(Mn51)*state.rho + screened_rates(k_He4_Mn52_to_p_Fe55)*Y(Mn52)*state.rho + screened_rates(k_He4_Mn53_to_p_Fe56)*Y(Mn53)*state.rho + screened_rates(k_He4_Mn54_to_p_Fe57)*Y(Mn54)*state.rho + screened_rates(k_He4_Mn55_to_p_Fe58)*Y(Mn55)*state.rho + screened_rates(k_He4_N13_to_p_O16)*Y(N13)*state.rho + screened_rates(k_He4_N14_to_p_O17)*Y(N14)*state.rho + screened_rates(k_He4_N15_to_p_O18)*Y(N15)*state.rho + screened_rates(k_He4_Na21_to_p_Mg24)*Y(Na21)*state.rho + screened_rates(k_He4_Na22_to_p_Mg25)*Y(Na22)*state.rho + screened_rates(k_He4_Na23_to_p_Mg26)*Y(Na23)*state.rho + screened_rates(k_He4_Ne18_to_p_Na21)*Y(Ne18)*state.rho + screened_rates(k_He4_Ne19_to_p_Na22)*Y(Ne19)*state.rho + screened_rates(k_He4_Ne20_to_p_Na23)*Y(Ne20)*state.rho + screened_rates(k_He4_Ni56_to_p_Cu59)*Y(Ni56)*state.rho + screened_rates(k_He4_Ni57_to_p_Cu60)*Y(Ni57)*state.rho + screened_rates(k_He4_Ni58_to_p_Cu61)*Y(Ni58)*state.rho + screened_rates(k_He4_Ni59_to_p_Cu62)*Y(Ni59)*state.rho + screened_rates(k_He4_Ni60_to_p_Cu63)*Y(Ni60)*state.rho + screened_rates(k_He4_Ni61_to_p_Cu64)*Y(Ni61)*state.rho + screened_rates(k_He4_Ni62_to_p_Cu65)*Y(Ni62)*state.rho + screened_rates(k_He4_O14_to_p_F17)*Y(O14)*state.rho + screened_rates(k_He4_O15_to_p_F18)*Y(O15)*state.rho + screened_rates(k_He4_O16_to_p_F19)*Y(O16)*state.rho + screened_rates(k_He4_P29_to_p_S32)*Y(P29)*state.rho + screened_rates(k_He4_P30_to_p_S33)*Y(P30)*state.rho + screened_rates(k_He4_P31_to_p_S34)*Y(P31)*state.rho + screened_rates(k_He4_P32_to_p_S35)*Y(P32)*state.rho + screened_rates(k_He4_P33_to_p_S36)*Y(P33)*state.rho + screened_rates(k_He4_S32_to_p_Cl35)*Y(S32)*state.rho + screened_rates(k_He4_S33_to_p_Cl36)*Y(S33)*state.rho + screened_rates(k_He4_S34_to_p_Cl37)*Y(S34)*state.rho + screened_rates(k_He4_Sc43_to_p_Ti46)*Y(Sc43)*state.rho + screened_rates(k_He4_Sc44_to_p_Ti47)*Y(Sc44)*state.rho + screened_rates(k_He4_Sc45_to_p_Ti48)*Y(Sc45)*state.rho + screened_rates(k_He4_Sc46_to_p_Ti49)*Y(Sc46)*state.rho + screened_rates(k_He4_Sc47_to_p_Ti50)*Y(Sc47)*state.rho + screened_rates(k_He4_Sc48_to_p_Ti51)*Y(Sc48)*state.rho + screened_rates(k_He4_Si28_to_p_P31)*Y(Si28)*state.rho + screened_rates(k_He4_Si29_to_p_P32)*Y(Si29)*state.rho + screened_rates(k_He4_Si30_to_p_P33)*Y(Si30)*state.rho + screened_rates(k_He4_Ti44_to_p_V47)*Y(Ti44)*state.rho + screened_rates(k_He4_Ti45_to_p_V48)*Y(Ti45)*state.rho + screened_rates(k_He4_Ti46_to_p_V49)*Y(Ti46)*state.rho + screened_rates(k_He4_Ti47_to_p_V50)*Y(Ti47)*state.rho + screened_rates(k_He4_Ti48_to_p_V51)*Y(Ti48)*state.rho + screened_rates(k_He4_Ti49_to_p_V52)*Y(Ti49)*state.rho + screened_rates(k_He4_V46_to_p_Cr49)*Y(V46)*state.rho + screened_rates(k_He4_V47_to_p_Cr50)*Y(V47)*state.rho + screened_rates(k_He4_V48_to_p_Cr51)*Y(V48)*state.rho + screened_rates(k_He4_V49_to_p_Cr52)*Y(V49)*state.rho + screened_rates(k_He4_V50_to_p_Cr53)*Y(V50)*state.rho + screened_rates(k_He4_V51_to_p_Cr54)*Y(V51)*state.rho + screened_rates(k_He4_Zn59_to_p_Ga62)*Y(Zn59)*state.rho + screened_rates(k_He4_Zn60_to_p_Ga63)*Y(Zn60)*state.rho + screened_rates(k_He4_Zn61_to_p_Ga64)*Y(Zn61)*state.rho + 1.0*screened_rates(k_d_He4_He4_to_p_Be9)*Y(He4)*Y(H2)*amrex::Math::powi<2>(state.rho) - 1.0*screened_rates(k_n_p_He4_He4_to_He3_Li7)*Y(He4)*Y(N)*Y(H1)*amrex::Math::powi<3>(state.rho) - screened_rates(k_n_p_He4_to_Li6)*Y(N)*Y(H1)*amrex::Math::powi<2>(state.rho) - 1.0*screened_rates(k_p_He4_He4_to_d_Be7)*Y(He4)*Y(H1)*amrex::Math::powi<2>(state.rho) - 1.0*screened_rates(k_p_He4_He4_to_n_B8)*Y(He4)*Y(H1)*amrex::Math::powi<2>(state.rho) - screened_rates(k_p_He4_to_d_He3)*Y(H1)*state.rho - 1.0*screened_rates(k_p_p_He4_He4_to_He3_Be7)*Y(He4)*amrex::Math::powi<2>(Y(H1))*amrex::Math::powi<3>(state.rho) - screened_rates(k_p_p_He4_to_He3_He3)*amrex::Math::powi<2>(Y(H1))*amrex::Math::powi<2>(state.rho);
    jac.set(H1, He4, scratch);

    scratch = screened_rates(k_He4_Li6_to_p_Be9)*Y(He4)*state.rho + screened_rates(k_Li6_to_n_p_He4) + screened_rates(k_d_Li6_to_p_Li7)*Y(H2)*state.rho - screened_rates(k_p_Li6_to_Be7)*Y(H1)*state.rho - screened_rates(k_p_Li6_to_He4_He3)*Y(H1)*state.rho;
    jac.set(H1, Li6, scratch);

    scratch = screened_rates(k_He3_Li7_to_n_p_He4_He4)*Y(He3)*state.rho - screened_rates(k_p_Li7_to_He4_He4)*Y(H1)*state.rho - screened_rates(k_p_Li7_to_d_Li6)*Y(H1)*state.rho - screened_rates(k_p_Li7_to_n_Be7)*Y(H1)*state.rho;
    jac.set(H1, Li7, scratch);

    scratch = screened_rates(k_Be7_to_p_Li6) + 2.0*screened_rates(k_He3_Be7_to_p_p_He4_He4)*Y(He3)*state.rho + screened_rates(k_He4_Be7_to_p_B10)*Y(He4)*state.rho + screened_rates(k_d_Be7_to_p_He4_He4)*Y(H2)*state.rho + screened_rates(k_n_Be7_to_p_Li7)*Y(N)*state.rho - screened_rates(k_p_Be7_to_B8)*Y(H1)*state.rho;
    jac.set(H1, Be7, scratch);

    scratch = -screened_rates(k_p_Be9_to_B10)*Y(H1)*state.rho - screened_rates(k_p_Be9_to_He4_Li6)*Y(H1)*state.rho - screened_rates(k_p_Be9_to_d_He4_He4)*Y(H1)*state.rho;
    jac.set(H1, Be9, scratch);

    scratch = screened_rates(k_B8_to_p_Be7) + screened_rates(k_n_B8_to_p_He4_He4)*Y(N)*state.rho;
    jac.set(H1, B8, scratch);

    scratch = screened_rates(k_B10_to_p_Be9) + screened_rates(k_He4_B10_to_p_C13)*Y(He4)*state.rho - screened_rates(k_p_B10_to_He4_Be7)*Y(H1)*state.rho;
    jac.set(H1, B10, scratch);

    scratch = screened_rates(k_He4_B11_to_p_C14)*Y(He4)*state.rho - screened_rates(k_p_B11_to_C12)*Y(H1)*state.rho - screened_rates(k_p_B11_to_He4_He4_He4)*Y(H1)*state.rho;
    jac.set(H1, B11, scratch);

    scratch = 1.0*screened_rates(k_C12_C12_to_p_Na23)*Y(C12)*state.rho + screened_rates(k_C12_Ne20_to_p_P31)*Y(Ne20)*state.rho + screened_rates(k_C12_O16_to_p_Al27)*Y(O16)*state.rho + screened_rates(k_C12_to_p_B11) + screened_rates(k_He4_C12_to_p_N15)*Y(He4)*state.rho - screened_rates(k_p_C12_to_N13)*Y(H1)*state.rho;
    jac.set(H1, C12, scratch);

    scratch = -screened_rates(k_p_C13_to_He4_B10)*Y(H1)*state.rho - screened_rates(k_p_C13_to_N14)*Y(H1)*state.rho - screened_rates(k_p_C13_to_n_N13)*Y(H1)*state.rho;
    jac.set(H1, C13, scratch);

    scratch = -screened_rates(k_p_C14_to_He4_B11)*Y(H1)*state.rho - screened_rates(k_p_C14_to_N15)*Y(H1)*state.rho - screened_rates(k_p_C14_to_n_N14)*Y(H1)*state.rho;
    jac.set(H1, C14, scratch);

    scratch = screened_rates(k_He4_N13_to_p_O16)*Y(He4)*state.rho + screened_rates(k_N13_to_p_C12) + screened_rates(k_n_N13_to_p_C13)*Y(N)*state.rho - screened_rates(k_p_N13_to_O14)*Y(H1)*state.rho;
    jac.set(H1, N13, scratch);

    scratch = screened_rates(k_He4_N14_to_p_O17)*Y(He4)*state.rho + screened_rates(k_N14_to_p_C13) + screened_rates(k_n_N14_to_p_C14)*Y(N)*state.rho - screened_rates(k_p_N14_to_O15)*Y(H1)*state.rho - screened_rates(k_p_N14_to_n_O14)*Y(H1)*state.rho;
    jac.set(H1, N14, scratch);

    scratch = screened_rates(k_He4_N15_to_p_O18)*Y(He4)*state.rho + screened_rates(k_N15_to_p_C14) - screened_rates(k_p_N15_to_He4_C12)*Y(H1)*state.rho - screened_rates(k_p_N15_to_O16)*Y(H1)*state.rho - screened_rates(k_p_N15_to_n_O15)*Y(H1)*state.rho;
    jac.set(H1, N15, scratch);

    scratch = screened_rates(k_He4_O14_to_p_F17)*Y(He4)*state.rho + screened_rates(k_O14_to_p_N13) + screened_rates(k_n_O14_to_p_N14)*Y(N)*state.rho;
    jac.set(H1, O14, scratch);

    scratch = screened_rates(k_He4_O15_to_p_F18)*Y(He4)*state.rho + screened_rates(k_O15_to_p_N14) + screened_rates(k_n_O15_to_p_N15)*Y(N)*state.rho;
    jac.set(H1, O15, scratch);

    scratch = screened_rates(k_C12_O16_to_p_Al27)*Y(C12)*state.rho + screened_rates(k_He4_O16_to_p_F19)*Y(He4)*state.rho + 1.0*screened_rates(k_O16_O16_to_p_P31)*Y(O16)*state.rho + screened_rates(k_O16_to_p_N15) - screened_rates(k_p_O16_to_F17)*Y(H1)*state.rho - screened_rates(k_p_O16_to_He4_N13)*Y(H1)*state.rho;
    jac.set(H1, O16, scratch);

    scratch = -screened_rates(k_p_O17_to_F18)*Y(H1)*state.rho - screened_rates(k_p_O17_to_He4_N14)*Y(H1)*state.rho - screened_rates(k_p_O17_to_n_F17)*Y(H1)*state.rho;
    jac.set(H1, O17, scratch);

    scratch = -screened_rates(k_p_O18_to_F19)*Y(H1)*state.rho - screened_rates(k_p_O18_to_He4_N15)*Y(H1)*state.rho - screened_rates(k_p_O18_to_n_F18)*Y(H1)*state.rho;
    jac.set(H1, O18, scratch);

    scratch = screened_rates(k_F17_to_p_O16) + screened_rates(k_He4_F17_to_p_Ne20)*Y(He4)*state.rho + screened_rates(k_n_F17_to_p_O17)*Y(N)*state.rho - screened_rates(k_p_F17_to_He4_O14)*Y(H1)*state.rho - screened_rates(k_p_F17_to_Ne18)*Y(H1)*state.rho;
    jac.set(H1, F17, scratch);

    scratch = screened_rates(k_F18_to_p_O17) + screened_rates(k_He4_F18_to_p_Ne21)*Y(He4)*state.rho + screened_rates(k_n_F18_to_p_O18)*Y(N)*state.rho - screened_rates(k_p_F18_to_He4_O15)*Y(H1)*state.rho - screened_rates(k_p_F18_to_Ne19)*Y(H1)*state.rho - screened_rates(k_p_F18_to_n_Ne18)*Y(H1)*state.rho;
    jac.set(H1, F18, scratch);

    scratch = screened_rates(k_F19_to_p_O18) + screened_rates(k_He4_F19_to_p_Ne22)*Y(He4)*state.rho - screened_rates(k_p_F19_to_He4_O16)*Y(H1)*state.rho - screened_rates(k_p_F19_to_Ne20)*Y(H1)*state.rho - screened_rates(k_p_F19_to_n_Ne19)*Y(H1)*state.rho;
    jac.set(H1, F19, scratch);

    scratch = screened_rates(k_He4_Ne18_to_p_Na21)*Y(He4)*state.rho + screened_rates(k_Ne18_to_p_F17) + screened_rates(k_n_Ne18_to_p_F18)*Y(N)*state.rho;
    jac.set(H1, Ne18, scratch);

    scratch = screened_rates(k_He4_Ne19_to_p_Na22)*Y(He4)*state.rho + screened_rates(k_Ne19_to_p_F18) + screened_rates(k_n_Ne19_to_p_F19)*Y(N)*state.rho;
    jac.set(H1, Ne19, scratch);

    scratch = screened_rates(k_C12_Ne20_to_p_P31)*Y(C12)*state.rho + screened_rates(k_He4_Ne20_to_p_Na23)*Y(He4)*state.rho + screened_rates(k_Ne20_to_p_F19) - screened_rates(k_p_Ne20_to_He4_F17)*Y(H1)*state.rho - screened_rates(k_p_Ne20_to_Na21)*Y(H1)*state.rho;
    jac.set(H1, Ne20, scratch);

    scratch = -screened_rates(k_p_Ne21_to_He4_F18)*Y(H1)*state.rho - screened_rates(k_p_Ne21_to_Na22)*Y(H1)*state.rho - screened_rates(k_p_Ne21_to_n_Na21)*Y(H1)*state.rho;
    jac.set(H1, Ne21, scratch);

    scratch = -screened_rates(k_p_Ne22_to_He4_F19)*Y(H1)*state.rho - screened_rates(k_p_Ne22_to_Na23)*Y(H1)*state.rho - screened_rates(k_p_Ne22_to_n_Na22)*Y(H1)*state.rho;
    jac.set(H1, Ne22, scratch);

    scratch = screened_rates(k_He4_Na21_to_p_Mg24)*Y(He4)*state.rho + screened_rates(k_Na21_to_p_Ne20) + screened_rates(k_n_Na21_to_p_Ne21)*Y(N)*state.rho - screened_rates(k_p_Na21_to_He4_Ne18)*Y(H1)*state.rho;
    jac.set(H1, Na21, scratch);

    scratch = screened_rates(k_He4_Na22_to_p_Mg25)*Y(He4)*state.rho + screened_rates(k_Na22_to_p_Ne21) + screened_rates(k_n_Na22_to_p_Ne22)*Y(N)*state.rho - screened_rates(k_p_Na22_to_He4_Ne19)*Y(H1)*state.rho - screened_rates(k_p_Na22_to_Mg23)*Y(H1)*state.rho;
    jac.set(H1, Na22, scratch);

    scratch = screened_rates(k_He4_Na23_to_p_Mg26)*Y(He4)*state.rho + screened_rates(k_Na23_to_p_Ne22) - screened_rates(k_p_Na23_to_C12_C12)*Y(H1)*state.rho - screened_rates(k_p_Na23_to_He4_Ne20)*Y(H1)*state.rho - screened_rates(k_p_Na23_to_Mg24)*Y(H1)*state.rho - screened_rates(k_p_Na23_to_n_Mg23)*Y(H1)*state.rho;
    jac.set(H1, Na23, scratch);

    scratch = screened_rates(k_He4_Mg23_to_p_Al26)*Y(He4)*state.rho + screened_rates(k_Mg23_to_p_Na22) + screened_rates(k_n_Mg23_to_p_Na23)*Y(N)*state.rho;
    jac.set(H1, Mg23, scratch);

    scratch = screened_rates(k_He4_Mg24_to_p_Al27)*Y(He4)*state.rho + screened_rates(k_Mg24_to_p_Na23) - screened_rates(k_p_Mg24_to_Al25)*Y(H1)*state.rho - screened_rates(k_p_Mg24_to_He4_Na21)*Y(H1)*state.rho;
    jac.set(H1, Mg24, scratch);

    scratch = -screened_rates(k_p_Mg25_to_Al26)*Y(H1)*state.rho - screened_rates(k_p_Mg25_to_He4_Na22)*Y(H1)*state.rho - screened_rates(k_p_Mg25_to_n_Al25)*Y(H1)*state.rho;
    jac.set(H1, Mg25, scratch);

    scratch = -screened_rates(k_p_Mg26_to_Al27)*Y(H1)*state.rho - screened_rates(k_p_Mg26_to_He4_Na23)*Y(H1)*state.rho - screened_rates(k_p_Mg26_to_n_Al26)*Y(H1)*state.rho;
    jac.set(H1, Mg26, scratch);

    scratch = screened_rates(k_Al25_to_p_Mg24) + screened_rates(k_He4_Al25_to_p_Si28)*Y(He4)*state.rho + screened_rates(k_n_Al25_to_p_Mg25)*Y(N)*state.rho;
    jac.set(H1, Al25, scratch);

    scratch = screened_rates(k_Al26_to_p_Mg25) + screened_rates(k_He4_Al26_to_p_Si29)*Y(He4)*state.rho + screened_rates(k_n_Al26_to_p_Mg26)*Y(N)*state.rho - screened_rates(k_p_Al26_to_He4_Mg23)*Y(H1)*state.rho;
    jac.set(H1, Al26, scratch);

    scratch = screened_rates(k_Al27_to_p_Mg26) + screened_rates(k_He4_Al27_to_p_Si30)*Y(He4)*state.rho - screened_rates(k_p_Al27_to_C12_O16)*Y(H1)*state.rho - screened_rates(k_p_Al27_to_He4_Mg24)*Y(H1)*state.rho - screened_rates(k_p_Al27_to_Si28)*Y(H1)*state.rho;
    jac.set(H1, Al27, scratch);

    scratch = screened_rates(k_He4_Si28_to_p_P31)*Y(He4)*state.rho + screened_rates(k_Si28_to_p_Al27) - screened_rates(k_p_Si28_to_He4_Al25)*Y(H1)*state.rho - screened_rates(k_p_Si28_to_P29)*Y(H1)*state.rho;
    jac.set(H1, Si28, scratch);

    scratch = screened_rates(k_He4_Si29_to_p_P32)*Y(He4)*state.rho - screened_rates(k_p_Si29_to_He4_Al26)*Y(H1)*state.rho - screened_rates(k_p_Si29_to_P30)*Y(H1)*state.rho - screened_rates(k_p_Si29_to_n_P29)*Y(H1)*state.rho;
    jac.set(H1, Si29, scratch);

    scratch = screened_rates(k_He4_Si30_to_p_P33)*Y(He4)*state.rho - screened_rates(k_p_Si30_to_He4_Al27)*Y(H1)*state.rho - screened_rates(k_p_Si30_to_P31)*Y(H1)*state.rho - screened_rates(k_p_Si30_to_n_P30)*Y(H1)*state.rho;
    jac.set(H1, Si30, scratch);

    scratch = -screened_rates(k_p_Si31_to_P32)*Y(H1)*state.rho - screened_rates(k_p_Si31_to_n_P31)*Y(H1)*state.rho;
    jac.set(H1, Si31, scratch);

    scratch = -screened_rates(k_p_Si32_to_P33)*Y(H1)*state.rho - screened_rates(k_p_Si32_to_n_P32)*Y(H1)*state.rho;
    jac.set(H1, Si32, scratch);

    scratch = screened_rates(k_He4_P29_to_p_S32)*Y(He4)*state.rho + screened_rates(k_P29_to_p_Si28) + screened_rates(k_n_P29_to_p_Si29)*Y(N)*state.rho;
    jac.set(H1, P29, scratch);

    scratch = screened_rates(k_He4_P30_to_p_S33)*Y(He4)*state.rho + screened_rates(k_P30_to_p_Si29) + screened_rates(k_n_P30_to_p_Si30)*Y(N)*state.rho;
    jac.set(H1, P30, scratch);

    scratch = screened_rates(k_He4_P31_to_p_S34)*Y(He4)*state.rho + screened_rates(k_P31_to_p_Si30) + screened_rates(k_n_P31_to_p_Si31)*Y(N)*state.rho - screened_rates(k_p_P31_to_C12_Ne20)*Y(H1)*state.rho - screened_rates(k_p_P31_to_He4_Si28)*Y(H1)*state.rho - screened_rates(k_p_P31_to_O16_O16)*Y(H1)*state.rho - screened_rates(k_p_P31_to_S32)*Y(H1)*state.rho;
    jac.set(H1, P31, scratch);

    scratch = screened_rates(k_He4_P32_to_p_S35)*Y(He4)*state.rho + screened_rates(k_P32_to_p_Si31) + screened_rates(k_n_P32_to_p_Si32)*Y(N)*state.rho - screened_rates(k_p_P32_to_He4_Si29)*Y(H1)*state.rho - screened_rates(k_p_P32_to_S33)*Y(H1)*state.rho - screened_rates(k_p_P32_to_n_S32)*Y(H1)*state.rho;
    jac.set(H1, P32, scratch);

    scratch = screened_rates(k_He4_P33_to_p_S36)*Y(He4)*state.rho + screened_rates(k_P33_to_p_Si32) - screened_rates(k_p_P33_to_He4_Si30)*Y(H1)*state.rho - screened_rates(k_p_P33_to_S34)*Y(H1)*state.rho - screened_rates(k_p_P33_to_n_S33)*Y(H1)*state.rho;
    jac.set(H1, P33, scratch);

    scratch = screened_rates(k_He4_S32_to_p_Cl35)*Y(He4)*state.rho + screened_rates(k_S32_to_p_P31) + screened_rates(k_n_S32_to_p_P32)*Y(N)*state.rho - screened_rates(k_p_S32_to_Cl33)*Y(H1)*state.rho - screened_rates(k_p_S32_to_He4_P29)*Y(H1)*state.rho;
    jac.set(H1, S32, scratch);

    scratch = screened_rates(k_He4_S33_to_p_Cl36)*Y(He4)*state.rho + screened_rates(k_S33_to_p_P32) + screened_rates(k_n_S33_to_p_P33)*Y(N)*state.rho - screened_rates(k_p_S33_to_Cl34)*Y(H1)*state.rho - screened_rates(k_p_S33_to_He4_P30)*Y(H1)*state.rho - screened_rates(k_p_S33_to_n_Cl33)*Y(H1)*state.rho;
    jac.set(H1, S33, scratch);

    scratch = screened_rates(k_He4_S34_to_p_Cl37)*Y(He4)*state.rho + screened_rates(k_S34_to_p_P33) - screened_rates(k_p_S34_to_Cl35)*Y(H1)*state.rho - screened_rates(k_p_S34_to_He4_P31)*Y(H1)*state.rho - screened_rates(k_p_S34_to_n_Cl34)*Y(H1)*state.rho;
    jac.set(H1, S34, scratch);

    scratch = -screened_rates(k_p_S35_to_Cl36)*Y(H1)*state.rho - screened_rates(k_p_S35_to_He4_P32)*Y(H1)*state.rho - screened_rates(k_p_S35_to_n_Cl35)*Y(H1)*state.rho;
    jac.set(H1, S35, scratch);

    scratch = -screened_rates(k_p_S36_to_Cl37)*Y(H1)*state.rho - screened_rates(k_p_S36_to_He4_P33)*Y(H1)*state.rho - screened_rates(k_p_S36_to_n_Cl36)*Y(H1)*state.rho;
    jac.set(H1, S36, scratch);

    scratch = screened_rates(k_Cl33_to_p_S32) + screened_rates(k_He4_Cl33_to_p_Ar36)*Y(He4)*state.rho + screened_rates(k_n_Cl33_to_p_S33)*Y(N)*state.rho;
    jac.set(H1, Cl33, scratch);

    scratch = screened_rates(k_Cl34_to_p_S33) + screened_rates(k_He4_Cl34_to_p_Ar37)*Y(He4)*state.rho + screened_rates(k_n_Cl34_to_p_S34)*Y(N)*state.rho;
    jac.set(H1, Cl34, scratch);

    scratch = screened_rates(k_Cl35_to_p_S34) + screened_rates(k_He4_Cl35_to_p_Ar38)*Y(He4)*state.rho + screened_rates(k_n_Cl35_to_p_S35)*Y(N)*state.rho - screened_rates(k_p_Cl35_to_Ar36)*Y(H1)*state.rho - screened_rates(k_p_Cl35_to_He4_S32)*Y(H1)*state.rho;
    jac.set(H1, Cl35, scratch);

    scratch = screened_rates(k_Cl36_to_p_S35) + screened_rates(k_He4_Cl36_to_p_Ar39)*Y(He4)*state.rho + screened_rates(k_n_Cl36_to_p_S36)*Y(N)*state.rho - screened_rates(k_p_Cl36_to_Ar37)*Y(H1)*state.rho - screened_rates(k_p_Cl36_to_He4_S33)*Y(H1)*state.rho - screened_rates(k_p_Cl36_to_n_Ar36)*Y(H1)*state.rho;
    jac.set(H1, Cl36, scratch);

    scratch = screened_rates(k_Cl37_to_p_S36) + screened_rates(k_He4_Cl37_to_p_Ar40)*Y(He4)*state.rho - screened_rates(k_p_Cl37_to_Ar38)*Y(H1)*state.rho - screened_rates(k_p_Cl37_to_He4_S34)*Y(H1)*state.rho - screened_rates(k_p_Cl37_to_n_Ar37)*Y(H1)*state.rho;
    jac.set(H1, Cl37, scratch);

    scratch = screened_rates(k_Ar36_to_p_Cl35) + screened_rates(k_He4_Ar36_to_p_K39)*Y(He4)*state.rho + screened_rates(k_n_Ar36_to_p_Cl36)*Y(N)*state.rho - screened_rates(k_p_Ar36_to_He4_Cl33)*Y(H1)*state.rho - screened_rates(k_p_Ar36_to_K37)*Y(H1)*state.rho;
    jac.set(H1, Ar36, scratch);

    scratch = screened_rates(k_Ar37_to_p_Cl36) + screened_rates(k_He4_Ar37_to_p_K40)*Y(He4)*state.rho + screened_rates(k_n_Ar37_to_p_Cl37)*Y(N)*state.rho - screened_rates(k_p_Ar37_to_He4_Cl34)*Y(H1)*state.rho - screened_rates(k_p_Ar37_to_K38)*Y(H1)*state.rho - screened_rates(k_p_Ar37_to_n_K37)*Y(H1)*state.rho;
    jac.set(H1, Ar37, scratch);

    scratch = screened_rates(k_Ar38_to_p_Cl37) + screened_rates(k_He4_Ar38_to_p_K41)*Y(He4)*state.rho - screened_rates(k_p_Ar38_to_He4_Cl35)*Y(H1)*state.rho - screened_rates(k_p_Ar38_to_K39)*Y(H1)*state.rho - screened_rates(k_p_Ar38_to_n_K38)*Y(H1)*state.rho;
    jac.set(H1, Ar38, scratch);

    scratch = -screened_rates(k_p_Ar39_to_He4_Cl36)*Y(H1)*state.rho - screened_rates(k_p_Ar39_to_K40)*Y(H1)*state.rho - screened_rates(k_p_Ar39_to_n_K39)*Y(H1)*state.rho;
    jac.set(H1, Ar39, scratch);

    scratch = -screened_rates(k_p_Ar40_to_He4_Cl37)*Y(H1)*state.rho - screened_rates(k_p_Ar40_to_K41)*Y(H1)*state.rho - screened_rates(k_p_Ar40_to_n_K40)*Y(H1)*state.rho;
    jac.set(H1, Ar40, scratch);

    scratch = screened_rates(k_He4_K37_to_p_Ca40)*Y(He4)*state.rho + screened_rates(k_K37_to_p_Ar36) + screened_rates(k_n_K37_to_p_Ar37)*Y(N)*state.rho;
    jac.set(H1, K37, scratch);

    scratch = screened_rates(k_He4_K38_to_p_Ca41)*Y(He4)*state.rho + screened_rates(k_K38_to_p_Ar37) + screened_rates(k_n_K38_to_p_Ar38)*Y(N)*state.rho;
    jac.set(H1, K38, scratch);

    scratch = screened_rates(k_He4_K39_to_p_Ca42)*Y(He4)*state.rho + screened_rates(k_K39_to_p_Ar38) + screened_rates(k_n_K39_to_p_Ar39)*Y(N)*state.rho - screened_rates(k_p_K39_to_Ca40)*Y(H1)*state.rho - screened_rates(k_p_K39_to_He4_Ar36)*Y(H1)*state.rho;
    jac.set(H1, K39, scratch);

    scratch = screened_rates(k_He4_K40_to_p_Ca43)*Y(He4)*state.rho + screened_rates(k_K40_to_p_Ar39) + screened_rates(k_n_K40_to_p_Ar40)*Y(N)*state.rho - screened_rates(k_p_K40_to_Ca41)*Y(H1)*state.rho - screened_rates(k_p_K40_to_He4_Ar37)*Y(H1)*state.rho - screened_rates(k_p_K40_to_n_Ca40)*Y(H1)*state.rho;
    jac.set(H1, K40, scratch);

    scratch = screened_rates(k_He4_K41_to_p_Ca44)*Y(He4)*state.rho + screened_rates(k_K41_to_p_Ar40) - screened_rates(k_p_K41_to_Ca42)*Y(H1)*state.rho - screened_rates(k_p_K41_to_He4_Ar38)*Y(H1)*state.rho - screened_rates(k_p_K41_to_n_Ca41)*Y(H1)*state.rho;
    jac.set(H1, K41, scratch);

    scratch = screened_rates(k_Ca40_to_p_K39) + screened_rates(k_He4_Ca40_to_p_Sc43)*Y(He4)*state.rho + screened_rates(k_n_Ca40_to_p_K40)*Y(N)*state.rho - screened_rates(k_p_Ca40_to_He4_K37)*Y(H1)*state.rho;
    jac.set(H1, Ca40, scratch);

    scratch = screened_rates(k_Ca41_to_p_K40) + screened_rates(k_He4_Ca41_to_p_Sc44)*Y(He4)*state.rho + screened_rates(k_n_Ca41_to_p_K41)*Y(N)*state.rho - screened_rates(k_p_Ca41_to_He4_K38)*Y(H1)*state.rho;
    jac.set(H1, Ca41, scratch);

    scratch = screened_rates(k_Ca42_to_p_K41) + screened_rates(k_He4_Ca42_to_p_Sc45)*Y(He4)*state.rho - screened_rates(k_p_Ca42_to_He4_K39)*Y(H1)*state.rho - screened_rates(k_p_Ca42_to_Sc43)*Y(H1)*state.rho;
    jac.set(H1, Ca42, scratch);

    scratch = screened_rates(k_He4_Ca43_to_p_Sc46)*Y(He4)*state.rho - screened_rates(k_p_Ca43_to_He4_K40)*Y(H1)*state.rho - screened_rates(k_p_Ca43_to_Sc44)*Y(H1)*state.rho - screened_rates(k_p_Ca43_to_n_Sc43)*Y(H1)*state.rho;
    jac.set(H1, Ca43, scratch);

    scratch = screened_rates(k_He4_Ca44_to_p_Sc47)*Y(He4)*state.rho - screened_rates(k_p_Ca44_to_He4_K41)*Y(H1)*state.rho - screened_rates(k_p_Ca44_to_Sc45)*Y(H1)*state.rho - screened_rates(k_p_Ca44_to_n_Sc44)*Y(H1)*state.rho;
    jac.set(H1, Ca44, scratch);

    scratch = screened_rates(k_He4_Ca45_to_p_Sc48)*Y(He4)*state.rho - screened_rates(k_p_Ca45_to_Sc46)*Y(H1)*state.rho - screened_rates(k_p_Ca45_to_n_Sc45)*Y(H1)*state.rho;
    jac.set(H1, Ca45, scratch);

    scratch = screened_rates(k_He4_Ca46_to_p_Sc49)*Y(He4)*state.rho - screened_rates(k_p_Ca46_to_Sc47)*Y(H1)*state.rho - screened_rates(k_p_Ca46_to_n_Sc46)*Y(H1)*state.rho;
    jac.set(H1, Ca46, scratch);

    scratch = -screened_rates(k_p_Ca47_to_Sc48)*Y(H1)*state.rho - screened_rates(k_p_Ca47_to_n_Sc47)*Y(H1)*state.rho;
    jac.set(H1, Ca47, scratch);

    scratch = -screened_rates(k_p_Ca48_to_Sc49)*Y(H1)*state.rho - screened_rates(k_p_Ca48_to_n_Sc48)*Y(H1)*state.rho;
    jac.set(H1, Ca48, scratch);

    scratch = screened_rates(k_He4_Sc43_to_p_Ti46)*Y(He4)*state.rho + screened_rates(k_Sc43_to_p_Ca42) + screened_rates(k_n_Sc43_to_p_Ca43)*Y(N)*state.rho - screened_rates(k_p_Sc43_to_He4_Ca40)*Y(H1)*state.rho - screened_rates(k_p_Sc43_to_Ti44)*Y(H1)*state.rho;
    jac.set(H1, Sc43, scratch);

    scratch = screened_rates(k_He4_Sc44_to_p_Ti47)*Y(He4)*state.rho + screened_rates(k_Sc44_to_p_Ca43) + screened_rates(k_n_Sc44_to_p_Ca44)*Y(N)*state.rho - screened_rates(k_p_Sc44_to_He4_Ca41)*Y(H1)*state.rho - screened_rates(k_p_Sc44_to_Ti45)*Y(H1)*state.rho - screened_rates(k_p_Sc44_to_n_Ti44)*Y(H1)*state.rho;
    jac.set(H1, Sc44, scratch);

    scratch = screened_rates(k_He4_Sc45_to_p_Ti48)*Y(He4)*state.rho + screened_rates(k_Sc45_to_p_Ca44) + screened_rates(k_n_Sc45_to_p_Ca45)*Y(N)*state.rho - screened_rates(k_p_Sc45_to_He4_Ca42)*Y(H1)*state.rho - screened_rates(k_p_Sc45_to_Ti46)*Y(H1)*state.rho - screened_rates(k_p_Sc45_to_n_Ti45)*Y(H1)*state.rho;
    jac.set(H1, Sc45, scratch);

    scratch = screened_rates(k_He4_Sc46_to_p_Ti49)*Y(He4)*state.rho + screened_rates(k_Sc46_to_p_Ca45) + screened_rates(k_n_Sc46_to_p_Ca46)*Y(N)*state.rho - screened_rates(k_p_Sc46_to_He4_Ca43)*Y(H1)*state.rho - screened_rates(k_p_Sc46_to_Ti47)*Y(H1)*state.rho - screened_rates(k_p_Sc46_to_n_Ti46)*Y(H1)*state.rho;
    jac.set(H1, Sc46, scratch);

    scratch = screened_rates(k_He4_Sc47_to_p_Ti50)*Y(He4)*state.rho + screened_rates(k_Sc47_to_p_Ca46) + screened_rates(k_n_Sc47_to_p_Ca47)*Y(N)*state.rho - screened_rates(k_p_Sc47_to_He4_Ca44)*Y(H1)*state.rho - screened_rates(k_p_Sc47_to_Ti48)*Y(H1)*state.rho - screened_rates(k_p_Sc47_to_n_Ti47)*Y(H1)*state.rho;
    jac.set(H1, Sc47, scratch);

    scratch = screened_rates(k_He4_Sc48_to_p_Ti51)*Y(He4)*state.rho + screened_rates(k_Sc48_to_p_Ca47) + screened_rates(k_n_Sc48_to_p_Ca48)*Y(N)*state.rho - screened_rates(k_p_Sc48_to_He4_Ca45)*Y(H1)*state.rho - screened_rates(k_p_Sc48_to_Ti49)*Y(H1)*state.rho - screened_rates(k_p_Sc48_to_n_Ti48)*Y(H1)*state.rho;
    jac.set(H1, Sc48, scratch);

    scratch = screened_rates(k_Sc49_to_p_Ca48) - screened_rates(k_p_Sc49_to_He4_Ca46)*Y(H1)*state.rho - screened_rates(k_p_Sc49_to_Ti50)*Y(H1)*state.rho - screened_rates(k_p_Sc49_to_n_Ti49)*Y(H1)*state.rho;
    jac.set(H1, Sc49, scratch);

    scratch = screened_rates(k_He4_Ti44_to_p_V47)*Y(He4)*state.rho + screened_rates(k_Ti44_to_p_Sc43) + screened_rates(k_n_Ti44_to_p_Sc44)*Y(N)*state.rho;
    jac.set(H1, Ti44, scratch);

    scratch = screened_rates(k_He4_Ti45_to_p_V48)*Y(He4)*state.rho + screened_rates(k_Ti45_to_p_Sc44) + screened_rates(k_n_Ti45_to_p_Sc45)*Y(N)*state.rho - screened_rates(k_p_Ti45_to_V46)*Y(H1)*state.rho;
    jac.set(H1, Ti45, scratch);

    scratch = screened_rates(k_He4_Ti46_to_p_V49)*Y(He4)*state.rho + screened_rates(k_Ti46_to_p_Sc45) + screened_rates(k_n_Ti46_to_p_Sc46)*Y(N)*state.rho - screened_rates(k_p_Ti46_to_He4_Sc43)*Y(H1)*state.rho - screened_rates(k_p_Ti46_to_V47)*Y(H1)*state.rho - screened_rates(k_p_Ti46_to_n_V46)*Y(H1)*state.rho;
    jac.set(H1, Ti46, scratch);

    scratch = screened_rates(k_He4_Ti47_to_p_V50)*Y(He4)*state.rho + screened_rates(k_Ti47_to_p_Sc46) + screened_rates(k_n_Ti47_to_p_Sc47)*Y(N)*state.rho - screened_rates(k_p_Ti47_to_He4_Sc44)*Y(H1)*state.rho - screened_rates(k_p_Ti47_to_V48)*Y(H1)*state.rho - screened_rates(k_p_Ti47_to_n_V47)*Y(H1)*state.rho;
    jac.set(H1, Ti47, scratch);

    scratch = screened_rates(k_He4_Ti48_to_p_V51)*Y(He4)*state.rho + screened_rates(k_Ti48_to_p_Sc47) + screened_rates(k_n_Ti48_to_p_Sc48)*Y(N)*state.rho - screened_rates(k_p_Ti48_to_He4_Sc45)*Y(H1)*state.rho - screened_rates(k_p_Ti48_to_V49)*Y(H1)*state.rho - screened_rates(k_p_Ti48_to_n_V48)*Y(H1)*state.rho;
    jac.set(H1, Ti48, scratch);

    scratch = screened_rates(k_He4_Ti49_to_p_V52)*Y(He4)*state.rho + screened_rates(k_Ti49_to_p_Sc48) + screened_rates(k_n_Ti49_to_p_Sc49)*Y(N)*state.rho - screened_rates(k_p_Ti49_to_He4_Sc46)*Y(H1)*state.rho - screened_rates(k_p_Ti49_to_V50)*Y(H1)*state.rho - screened_rates(k_p_Ti49_to_n_V49)*Y(H1)*state.rho;
    jac.set(H1, Ti49, scratch);

    scratch = screened_rates(k_Ti50_to_p_Sc49) - screened_rates(k_p_Ti50_to_He4_Sc47)*Y(H1)*state.rho - screened_rates(k_p_Ti50_to_V51)*Y(H1)*state.rho - screened_rates(k_p_Ti50_to_n_V50)*Y(H1)*state.rho;
    jac.set(H1, Ti50, scratch);

    scratch = -screened_rates(k_p_Ti51_to_He4_Sc48)*Y(H1)*state.rho - screened_rates(k_p_Ti51_to_V52)*Y(H1)*state.rho - screened_rates(k_p_Ti51_to_n_V51)*Y(H1)*state.rho;
    jac.set(H1, Ti51, scratch);

    scratch = screened_rates(k_He4_V46_to_p_Cr49)*Y(He4)*state.rho + screened_rates(k_V46_to_p_Ti45) + screened_rates(k_n_V46_to_p_Ti46)*Y(N)*state.rho;
    jac.set(H1, V46, scratch);

    scratch = screened_rates(k_He4_V47_to_p_Cr50)*Y(He4)*state.rho + screened_rates(k_V47_to_p_Ti46) + screened_rates(k_n_V47_to_p_Ti47)*Y(N)*state.rho - screened_rates(k_p_V47_to_Cr48)*Y(H1)*state.rho - screened_rates(k_p_V47_to_He4_Ti44)*Y(H1)*state.rho;
    jac.set(H1, V47, scratch);

    scratch = screened_rates(k_He4_V48_to_p_Cr51)*Y(He4)*state.rho + screened_rates(k_V48_to_p_Ti47) + screened_rates(k_n_V48_to_p_Ti48)*Y(N)*state.rho - screened_rates(k_p_V48_to_Cr49)*Y(H1)*state.rho - screened_rates(k_p_V48_to_He4_Ti45)*Y(H1)*state.rho - screened_rates(k_p_V48_to_n_Cr48)*Y(H1)*state.rho;
    jac.set(H1, V48, scratch);

    scratch = screened_rates(k_He4_V49_to_p_Cr52)*Y(He4)*state.rho + screened_rates(k_V49_to_p_Ti48) + screened_rates(k_n_V49_to_p_Ti49)*Y(N)*state.rho - screened_rates(k_p_V49_to_Cr50)*Y(H1)*state.rho - screened_rates(k_p_V49_to_He4_Ti46)*Y(H1)*state.rho - screened_rates(k_p_V49_to_n_Cr49)*Y(H1)*state.rho;
    jac.set(H1, V49, scratch);

    scratch = screened_rates(k_He4_V50_to_p_Cr53)*Y(He4)*state.rho + screened_rates(k_V50_to_p_Ti49) + screened_rates(k_n_V50_to_p_Ti50)*Y(N)*state.rho - screened_rates(k_p_V50_to_Cr51)*Y(H1)*state.rho - screened_rates(k_p_V50_to_He4_Ti47)*Y(H1)*state.rho - screened_rates(k_p_V50_to_n_Cr50)*Y(H1)*state.rho;
    jac.set(H1, V50, scratch);

    scratch = screened_rates(k_He4_V51_to_p_Cr54)*Y(He4)*state.rho + screened_rates(k_V51_to_p_Ti50) + screened_rates(k_n_V51_to_p_Ti51)*Y(N)*state.rho - screened_rates(k_p_V51_to_Cr52)*Y(H1)*state.rho - screened_rates(k_p_V51_to_He4_Ti48)*Y(H1)*state.rho - screened_rates(k_p_V51_to_n_Cr51)*Y(H1)*state.rho;
    jac.set(H1, V51, scratch);

    scratch = screened_rates(k_V52_to_p_Ti51) - screened_rates(k_p_V52_to_Cr53)*Y(H1)*state.rho - screened_rates(k_p_V52_to_He4_Ti49)*Y(H1)*state.rho - screened_rates(k_p_V52_to_n_Cr52)*Y(H1)*state.rho;
    jac.set(H1, V52, scratch);

    scratch = screened_rates(k_Cr48_to_p_V47) + screened_rates(k_He4_Cr48_to_p_Mn51)*Y(He4)*state.rho + screened_rates(k_n_Cr48_to_p_V48)*Y(N)*state.rho;
    jac.set(H1, Cr48, scratch);

    scratch = screened_rates(k_Cr49_to_p_V48) + screened_rates(k_He4_Cr49_to_p_Mn52)*Y(He4)*state.rho + screened_rates(k_n_Cr49_to_p_V49)*Y(N)*state.rho - screened_rates(k_p_Cr49_to_He4_V46)*Y(H1)*state.rho - screened_rates(k_p_Cr49_to_Mn50)*Y(H1)*state.rho;
    jac.set(H1, Cr49, scratch);

    scratch = screened_rates(k_Cr50_to_p_V49) + screened_rates(k_He4_Cr50_to_p_Mn53)*Y(He4)*state.rho + screened_rates(k_n_Cr50_to_p_V50)*Y(N)*state.rho - screened_rates(k_p_Cr50_to_He4_V47)*Y(H1)*state.rho - screened_rates(k_p_Cr50_to_Mn51)*Y(H1)*state.rho - screened_rates(k_p_Cr50_to_n_Mn50)*Y(H1)*state.rho;
    jac.set(H1, Cr50, scratch);

    scratch = screened_rates(k_Cr51_to_p_V50) + screened_rates(k_He4_Cr51_to_p_Mn54)*Y(He4)*state.rho + screened_rates(k_n_Cr51_to_p_V51)*Y(N)*state.rho - screened_rates(k_p_Cr51_to_He4_V48)*Y(H1)*state.rho - screened_rates(k_p_Cr51_to_Mn52)*Y(H1)*state.rho - screened_rates(k_p_Cr51_to_n_Mn51)*Y(H1)*state.rho;
    jac.set(H1, Cr51, scratch);

    scratch = screened_rates(k_Cr52_to_p_V51) + screened_rates(k_He4_Cr52_to_p_Mn55)*Y(He4)*state.rho + screened_rates(k_n_Cr52_to_p_V52)*Y(N)*state.rho - screened_rates(k_p_Cr52_to_He4_V49)*Y(H1)*state.rho - screened_rates(k_p_Cr52_to_Mn53)*Y(H1)*state.rho - screened_rates(k_p_Cr52_to_n_Mn52)*Y(H1)*state.rho;
    jac.set(H1, Cr52, scratch);

    scratch = screened_rates(k_Cr53_to_p_V52) - screened_rates(k_p_Cr53_to_He4_V50)*Y(H1)*state.rho - screened_rates(k_p_Cr53_to_Mn54)*Y(H1)*state.rho - screened_rates(k_p_Cr53_to_n_Mn53)*Y(H1)*state.rho;
    jac.set(H1, Cr53, scratch);

    scratch = -screened_rates(k_p_Cr54_to_He4_V51)*Y(H1)*state.rho - screened_rates(k_p_Cr54_to_Mn55)*Y(H1)*state.rho - screened_rates(k_p_Cr54_to_n_Mn54)*Y(H1)*state.rho;
    jac.set(H1, Cr54, scratch);

    scratch = screened_rates(k_He4_Mn50_to_p_Fe53)*Y(He4)*state.rho + screened_rates(k_Mn50_to_p_Cr49) + screened_rates(k_n_Mn50_to_p_Cr50)*Y(N)*state.rho;
    jac.set(H1, Mn50, scratch);

    scratch = screened_rates(k_He4_Mn51_to_p_Fe54)*Y(He4)*state.rho + screened_rates(k_Mn51_to_p_Cr50) + screened_rates(k_n_Mn51_to_p_Cr51)*Y(N)*state.rho - screened_rates(k_p_Mn51_to_Fe52)*Y(H1)*state.rho - screened_rates(k_p_Mn51_to_He4_Cr48)*Y(H1)*state.rho;
    jac.set(H1, Mn51, scratch);

    scratch = screened_rates(k_He4_Mn52_to_p_Fe55)*Y(He4)*state.rho + screened_rates(k_Mn52_to_p_Cr51) + screened_rates(k_n_Mn52_to_p_Cr52)*Y(N)*state.rho - screened_rates(k_p_Mn52_to_Fe53)*Y(H1)*state.rho - screened_rates(k_p_Mn52_to_He4_Cr49)*Y(H1)*state.rho - screened_rates(k_p_Mn52_to_n_Fe52)*Y(H1)*state.rho;
    jac.set(H1, Mn52, scratch);

    scratch = screened_rates(k_He4_Mn53_to_p_Fe56)*Y(He4)*state.rho + screened_rates(k_Mn53_to_p_Cr52) + screened_rates(k_n_Mn53_to_p_Cr53)*Y(N)*state.rho - screened_rates(k_p_Mn53_to_Fe54)*Y(H1)*state.rho - screened_rates(k_p_Mn53_to_He4_Cr50)*Y(H1)*state.rho - screened_rates(k_p_Mn53_to_n_Fe53)*Y(H1)*state.rho;
    jac.set(H1, Mn53, scratch);

    scratch = screened_rates(k_He4_Mn54_to_p_Fe57)*Y(He4)*state.rho + screened_rates(k_Mn54_to_p_Cr53) + screened_rates(k_n_Mn54_to_p_Cr54)*Y(N)*state.rho - screened_rates(k_p_Mn54_to_Fe55)*Y(H1)*state.rho - screened_rates(k_p_Mn54_to_He4_Cr51)*Y(H1)*state.rho - screened_rates(k_p_Mn54_to_n_Fe54)*Y(H1)*state.rho;
    jac.set(H1, Mn54, scratch);

    scratch = screened_rates(k_He4_Mn55_to_p_Fe58)*Y(He4)*state.rho + screened_rates(k_Mn55_to_p_Cr54) - screened_rates(k_p_Mn55_to_Fe56)*Y(H1)*state.rho - screened_rates(k_p_Mn55_to_He4_Cr52)*Y(H1)*state.rho - screened_rates(k_p_Mn55_to_n_Fe55)*Y(H1)*state.rho;
    jac.set(H1, Mn55, scratch);

    scratch = screened_rates(k_Fe52_to_p_Mn51) + screened_rates(k_He4_Fe52_to_p_Co55)*Y(He4)*state.rho + screened_rates(k_n_Fe52_to_p_Mn52)*Y(N)*state.rho - screened_rates(k_p_Fe52_to_Co53)*Y(H1)*state.rho;
    jac.set(H1, Fe52, scratch);

    scratch = screened_rates(k_Fe53_to_p_Mn52) + screened_rates(k_He4_Fe53_to_p_Co56)*Y(He4)*state.rho + screened_rates(k_n_Fe53_to_p_Mn53)*Y(N)*state.rho - screened_rates(k_p_Fe53_to_Co54)*Y(H1)*state.rho - screened_rates(k_p_Fe53_to_He4_Mn50)*Y(H1)*state.rho - screened_rates(k_p_Fe53_to_n_Co53)*Y(H1)*state.rho;
    jac.set(H1, Fe53, scratch);

    scratch = screened_rates(k_Fe54_to_p_Mn53) + screened_rates(k_He4_Fe54_to_p_Co57)*Y(He4)*state.rho + screened_rates(k_n_Fe54_to_p_Mn54)*Y(N)*state.rho - screened_rates(k_p_Fe54_to_Co55)*Y(H1)*state.rho - screened_rates(k_p_Fe54_to_He4_Mn51)*Y(H1)*state.rho - screened_rates(k_p_Fe54_to_n_Co54)*Y(H1)*state.rho;
    jac.set(H1, Fe54, scratch);

    scratch = screened_rates(k_Fe55_to_p_Mn54) + screened_rates(k_He4_Fe55_to_p_Co58)*Y(He4)*state.rho + screened_rates(k_n_Fe55_to_p_Mn55)*Y(N)*state.rho - screened_rates(k_p_Fe55_to_Co56)*Y(H1)*state.rho - screened_rates(k_p_Fe55_to_He4_Mn52)*Y(H1)*state.rho - screened_rates(k_p_Fe55_to_n_Co55)*Y(H1)*state.rho;
    jac.set(H1, Fe55, scratch);

    scratch = screened_rates(k_Fe56_to_p_Mn55) + screened_rates(k_He4_Fe56_to_p_Co59)*Y(He4)*state.rho - screened_rates(k_p_Fe56_to_Co57)*Y(H1)*state.rho - screened_rates(k_p_Fe56_to_He4_Mn53)*Y(H1)*state.rho - screened_rates(k_p_Fe56_to_n_Co56)*Y(H1)*state.rho;
    jac.set(H1, Fe56, scratch);

    scratch = -screened_rates(k_p_Fe57_to_Co58)*Y(H1)*state.rho - screened_rates(k_p_Fe57_to_He4_Mn54)*Y(H1)*state.rho - screened_rates(k_p_Fe57_to_n_Co57)*Y(H1)*state.rho;
    jac.set(H1, Fe57, scratch);

    scratch = -screened_rates(k_p_Fe58_to_Co59)*Y(H1)*state.rho - screened_rates(k_p_Fe58_to_He4_Mn55)*Y(H1)*state.rho - screened_rates(k_p_Fe58_to_n_Co58)*Y(H1)*state.rho;
    jac.set(H1, Fe58, scratch);

    scratch = screened_rates(k_Co53_to_p_Fe52) + screened_rates(k_He4_Co53_to_p_Ni56)*Y(He4)*state.rho + screened_rates(k_n_Co53_to_p_Fe53)*Y(N)*state.rho;
    jac.set(H1, Co53, scratch);

    scratch = screened_rates(k_Co54_to_p_Fe53) + screened_rates(k_He4_Co54_to_p_Ni57)*Y(He4)*state.rho + screened_rates(k_n_Co54_to_p_Fe54)*Y(N)*state.rho;
    jac.set(H1, Co54, scratch);

    scratch = screened_rates(k_Co55_to_p_Fe54) + screened_rates(k_He4_Co55_to_p_Ni58)*Y(He4)*state.rho + screened_rates(k_n_Co55_to_p_Fe55)*Y(N)*state.rho - screened_rates(k_p_Co55_to_He4_Fe52)*Y(H1)*state.rho - screened_rates(k_p_Co55_to_Ni56)*Y(H1)*state.rho;
    jac.set(H1, Co55, scratch);

    scratch = screened_rates(k_Co56_to_p_Fe55) + screened_rates(k_He4_Co56_to_p_Ni59)*Y(He4)*state.rho + screened_rates(k_n_Co56_to_p_Fe56)*Y(N)*state.rho - screened_rates(k_p_Co56_to_He4_Fe53)*Y(H1)*state.rho - screened_rates(k_p_Co56_to_Ni57)*Y(H1)*state.rho - screened_rates(k_p_Co56_to_n_Ni56)*Y(H1)*state.rho;
    jac.set(H1, Co56, scratch);

    scratch = screened_rates(k_Co57_to_p_Fe56) + screened_rates(k_He4_Co57_to_p_Ni60)*Y(He4)*state.rho + screened_rates(k_n_Co57_to_p_Fe57)*Y(N)*state.rho - screened_rates(k_p_Co57_to_He4_Fe54)*Y(H1)*state.rho - screened_rates(k_p_Co57_to_Ni58)*Y(H1)*state.rho - screened_rates(k_p_Co57_to_n_Ni57)*Y(H1)*state.rho;
    jac.set(H1, Co57, scratch);

    scratch = screened_rates(k_Co58_to_p_Fe57) + screened_rates(k_He4_Co58_to_p_Ni61)*Y(He4)*state.rho + screened_rates(k_n_Co58_to_p_Fe58)*Y(N)*state.rho - screened_rates(k_p_Co58_to_He4_Fe55)*Y(H1)*state.rho - screened_rates(k_p_Co58_to_Ni59)*Y(H1)*state.rho - screened_rates(k_p_Co58_to_n_Ni58)*Y(H1)*state.rho;
    jac.set(H1, Co58, scratch);

    scratch = screened_rates(k_Co59_to_p_Fe58) + screened_rates(k_He4_Co59_to_p_Ni62)*Y(He4)*state.rho - screened_rates(k_p_Co59_to_He4_Fe56)*Y(H1)*state.rho - screened_rates(k_p_Co59_to_Ni60)*Y(H1)*state.rho - screened_rates(k_p_Co59_to_n_Ni59)*Y(H1)*state.rho;
    jac.set(H1, Co59, scratch);

    scratch = screened_rates(k_He4_Ni56_to_p_Cu59)*Y(He4)*state.rho + screened_rates(k_Ni56_to_p_Co55) + screened_rates(k_n_Ni56_to_p_Co56)*Y(N)*state.rho - screened_rates(k_p_Ni56_to_Cu57)*Y(H1)*state.rho - screened_rates(k_p_Ni56_to_He4_Co53)*Y(H1)*state.rho;
    jac.set(H1, Ni56, scratch);

    scratch = screened_rates(k_He4_Ni57_to_p_Cu60)*Y(He4)*state.rho + screened_rates(k_Ni57_to_p_Co56) + screened_rates(k_n_Ni57_to_p_Co57)*Y(N)*state.rho - screened_rates(k_p_Ni57_to_Cu58)*Y(H1)*state.rho - screened_rates(k_p_Ni57_to_He4_Co54)*Y(H1)*state.rho - screened_rates(k_p_Ni57_to_n_Cu57)*Y(H1)*state.rho;
    jac.set(H1, Ni57, scratch);

    scratch = screened_rates(k_He4_Ni58_to_p_Cu61)*Y(He4)*state.rho + screened_rates(k_Ni58_to_p_Co57) + screened_rates(k_n_Ni58_to_p_Co58)*Y(N)*state.rho - screened_rates(k_p_Ni58_to_Cu59)*Y(H1)*state.rho - screened_rates(k_p_Ni58_to_He4_Co55)*Y(H1)*state.rho - screened_rates(k_p_Ni58_to_n_Cu58)*Y(H1)*state.rho;
    jac.set(H1, Ni58, scratch);

    scratch = screened_rates(k_He4_Ni59_to_p_Cu62)*Y(He4)*state.rho + screened_rates(k_Ni59_to_p_Co58) + screened_rates(k_n_Ni59_to_p_Co59)*Y(N)*state.rho - screened_rates(k_p_Ni59_to_Cu60)*Y(H1)*state.rho - screened_rates(k_p_Ni59_to_He4_Co56)*Y(H1)*state.rho - screened_rates(k_p_Ni59_to_n_Cu59)*Y(H1)*state.rho;
    jac.set(H1, Ni59, scratch);

    scratch = screened_rates(k_He4_Ni60_to_p_Cu63)*Y(He4)*state.rho + screened_rates(k_Ni60_to_p_Co59) - screened_rates(k_p_Ni60_to_Cu61)*Y(H1)*state.rho - screened_rates(k_p_Ni60_to_He4_Co57)*Y(H1)*state.rho - screened_rates(k_p_Ni60_to_n_Cu60)*Y(H1)*state.rho;
    jac.set(H1, Ni60, scratch);

    scratch = screened_rates(k_He4_Ni61_to_p_Cu64)*Y(He4)*state.rho - screened_rates(k_p_Ni61_to_Cu62)*Y(H1)*state.rho - screened_rates(k_p_Ni61_to_He4_Co58)*Y(H1)*state.rho - screened_rates(k_p_Ni61_to_n_Cu61)*Y(H1)*state.rho;
    jac.set(H1, Ni61, scratch);

    scratch = screened_rates(k_He4_Ni62_to_p_Cu65)*Y(He4)*state.rho - screened_rates(k_p_Ni62_to_Cu63)*Y(H1)*state.rho - screened_rates(k_p_Ni62_to_He4_Co59)*Y(H1)*state.rho - screened_rates(k_p_Ni62_to_n_Cu62)*Y(H1)*state.rho;
    jac.set(H1, Ni62, scratch);

    scratch = -screened_rates(k_p_Ni63_to_Cu64)*Y(H1)*state.rho - screened_rates(k_p_Ni63_to_n_Cu63)*Y(H1)*state.rho;
    jac.set(H1, Ni63, scratch);

    scratch = -screened_rates(k_p_Ni64_to_Cu65)*Y(H1)*state.rho - screened_rates(k_p_Ni64_to_n_Cu64)*Y(H1)*state.rho;
    jac.set(H1, Ni64, scratch);

    scratch = screened_rates(k_Cu57_to_p_Ni56) + screened_rates(k_He4_Cu57_to_p_Zn60)*Y(He4)*state.rho + screened_rates(k_n_Cu57_to_p_Ni57)*Y(N)*state.rho;
    jac.set(H1, Cu57, scratch);

    scratch = screened_rates(k_Cu58_to_p_Ni57) + screened_rates(k_He4_Cu58_to_p_Zn61)*Y(He4)*state.rho + screened_rates(k_n_Cu58_to_p_Ni58)*Y(N)*state.rho - screened_rates(k_p_Cu58_to_Zn59)*Y(H1)*state.rho;
    jac.set(H1, Cu58, scratch);

    scratch = screened_rates(k_Cu59_to_p_Ni58) + screened_rates(k_He4_Cu59_to_p_Zn62)*Y(He4)*state.rho + screened_rates(k_n_Cu59_to_p_Ni59)*Y(N)*state.rho - screened_rates(k_p_Cu59_to_He4_Ni56)*Y(H1)*state.rho - screened_rates(k_p_Cu59_to_Zn60)*Y(H1)*state.rho - screened_rates(k_p_Cu59_to_n_Zn59)*Y(H1)*state.rho;
    jac.set(H1, Cu59, scratch);

    scratch = screened_rates(k_Cu60_to_p_Ni59) + screened_rates(k_He4_Cu60_to_p_Zn63)*Y(He4)*state.rho + screened_rates(k_n_Cu60_to_p_Ni60)*Y(N)*state.rho - screened_rates(k_p_Cu60_to_He4_Ni57)*Y(H1)*state.rho - screened_rates(k_p_Cu60_to_Zn61)*Y(H1)*state.rho - screened_rates(k_p_Cu60_to_n_Zn60)*Y(H1)*state.rho;
    jac.set(H1, Cu60, scratch);

    scratch = screened_rates(k_Cu61_to_p_Ni60) + screened_rates(k_He4_Cu61_to_p_Zn64)*Y(He4)*state.rho + screened_rates(k_n_Cu61_to_p_Ni61)*Y(N)*state.rho - screened_rates(k_p_Cu61_to_He4_Ni58)*Y(H1)*state.rho - screened_rates(k_p_Cu61_to_Zn62)*Y(H1)*state.rho - screened_rates(k_p_Cu61_to_n_Zn61)*Y(H1)*state.rho;
    jac.set(H1, Cu61, scratch);

    scratch = screened_rates(k_Cu62_to_p_Ni61) + screened_rates(k_He4_Cu62_to_p_Zn65)*Y(He4)*state.rho + screened_rates(k_n_Cu62_to_p_Ni62)*Y(N)*state.rho - screened_rates(k_p_Cu62_to_He4_Ni59)*Y(H1)*state.rho - screened_rates(k_p_Cu62_to_Zn63)*Y(H1)*state.rho - screened_rates(k_p_Cu62_to_n_Zn62)*Y(H1)*state.rho;
    jac.set(H1, Cu62, scratch);

    scratch = screened_rates(k_Cu63_to_p_Ni62) + screened_rates(k_He4_Cu63_to_p_Zn66)*Y(He4)*state.rho + screened_rates(k_n_Cu63_to_p_Ni63)*Y(N)*state.rho - screened_rates(k_p_Cu63_to_He4_Ni60)*Y(H1)*state.rho - screened_rates(k_p_Cu63_to_Zn64)*Y(H1)*state.rho - screened_rates(k_p_Cu63_to_n_Zn63)*Y(H1)*state.rho;
    jac.set(H1, Cu63, scratch);

    scratch = screened_rates(k_Cu64_to_p_Ni63) + screened_rates(k_n_Cu64_to_p_Ni64)*Y(N)*state.rho - screened_rates(k_p_Cu64_to_He4_Ni61)*Y(H1)*state.rho - screened_rates(k_p_Cu64_to_Zn65)*Y(H1)*state.rho - screened_rates(k_p_Cu64_to_n_Zn64)*Y(H1)*state.rho;
    jac.set(H1, Cu64, scratch);

    scratch = screened_rates(k_Cu65_to_p_Ni64) - screened_rates(k_p_Cu65_to_He4_Ni62)*Y(H1)*state.rho - screened_rates(k_p_Cu65_to_Zn66)*Y(H1)*state.rho - screened_rates(k_p_Cu65_to_n_Zn65)*Y(H1)*state.rho;
    jac.set(H1, Cu65, scratch);

    scratch = screened_rates(k_He4_Zn59_to_p_Ga62)*Y(He4)*state.rho + screened_rates(k_Zn59_to_p_Cu58) + screened_rates(k_Zn59_to_p_Ni58_weak_wc12) + screened_rates(k_n_Zn59_to_p_Cu59)*Y(N)*state.rho;
    jac.set(H1, Zn59, scratch);

    scratch = screened_rates(k_He4_Zn60_to_p_Ga63)*Y(He4)*state.rho + screened_rates(k_Zn60_to_p_Cu59) + screened_rates(k_n_Zn60_to_p_Cu60)*Y(N)*state.rho - screened_rates(k_p_Zn60_to_He4_Cu57)*Y(H1)*state.rho;
    jac.set(H1, Zn60, scratch);

    scratch = screened_rates(k_He4_Zn61_to_p_Ga64)*Y(He4)*state.rho + screened_rates(k_Zn61_to_p_Cu60) + screened_rates(k_n_Zn61_to_p_Cu61)*Y(N)*state.rho - screened_rates(k_p_Zn61_to_Ga62)*Y(H1)*state.rho - screened_rates(k_p_Zn61_to_He4_Cu58)*Y(H1)*state.rho;
    jac.set(H1, Zn61, scratch);

    scratch = screened_rates(k_Zn62_to_p_Cu61) + screened_rates(k_n_Zn62_to_p_Cu62)*Y(N)*state.rho - screened_rates(k_p_Zn62_to_Ga63)*Y(H1)*state.rho - screened_rates(k_p_Zn62_to_He4_Cu59)*Y(H1)*state.rho - screened_rates(k_p_Zn62_to_n_Ga62)*Y(H1)*state.rho;
    jac.set(H1, Zn62, scratch);

    scratch = screened_rates(k_Zn63_to_p_Cu62) + screened_rates(k_n_Zn63_to_p_Cu63)*Y(N)*state.rho - screened_rates(k_p_Zn63_to_Ga64)*Y(H1)*state.rho - screened_rates(k_p_Zn63_to_He4_Cu60)*Y(H1)*state.rho - screened_rates(k_p_Zn63_to_n_Ga63)*Y(H1)*state.rho;
    jac.set(H1, Zn63, scratch);

    scratch = screened_rates(k_Zn64_to_p_Cu63) + screened_rates(k_n_Zn64_to_p_Cu64)*Y(N)*state.rho - screened_rates(k_p_Zn64_to_He4_Cu61)*Y(H1)*state.rho - screened_rates(k_p_Zn64_to_n_Ga64)*Y(H1)*state.rho;
    jac.set(H1, Zn64, scratch);

    scratch = screened_rates(k_Zn65_to_p_Cu64) + screened_rates(k_n_Zn65_to_p_Cu65)*Y(N)*state.rho - screened_rates(k_p_Zn65_to_He4_Cu62)*Y(H1)*state.rho;
    jac.set(H1, Zn65, scratch);

    scratch = screened_rates(k_Zn66_to_p_Cu65) - screened_rates(k_p_Zn66_to_He4_Cu63)*Y(H1)*state.rho;
    jac.set(H1, Zn66, scratch);

    scratch = screened_rates(k_Ga62_to_p_Zn61) + screened_rates(k_n_Ga62_to_p_Zn62)*Y(N)*state.rho - screened_rates(k_p_Ga62_to_Ge63)*Y(H1)*state.rho - screened_rates(k_p_Ga62_to_He4_Zn59)*Y(H1)*state.rho;
    jac.set(H1, Ga62, scratch);

    scratch = screened_rates(k_Ga63_to_p_Zn62) + screened_rates(k_n_Ga63_to_p_Zn63)*Y(N)*state.rho - screened_rates(k_p_Ga63_to_Ge64)*Y(H1)*state.rho - screened_rates(k_p_Ga63_to_He4_Zn60)*Y(H1)*state.rho - screened_rates(k_p_Ga63_to_n_Ge63)*Y(H1)*state.rho;
    jac.set(H1, Ga63, scratch);

    scratch = screened_rates(k_Ga64_to_p_Zn63) + screened_rates(k_n_Ga64_to_p_Zn64)*Y(N)*state.rho - screened_rates(k_p_Ga64_to_He4_Zn61)*Y(H1)*state.rho - screened_rates(k_p_Ga64_to_n_Ge64)*Y(H1)*state.rho;
    jac.set(H1, Ga64, scratch);

    scratch = screened_rates(k_Ge63_to_p_Ga62) + screened_rates(k_n_Ge63_to_p_Ga63)*Y(N)*state.rho;
    jac.set(H1, Ge63, scratch);

    scratch = screened_rates(k_Ge64_to_p_Ga63) + screened_rates(k_n_Ge64_to_p_Ga64)*Y(N)*state.rho;
    jac.set(H1, Ge64, scratch);

    scratch = screened_rates(k_n_Be7_to_d_Li6)*Y(Be7)*state.rho + 2.0*screened_rates(k_n_He3_to_d_d)*Y(He3)*state.rho + 0.5*screened_rates(k_n_He4_He4_to_d_Li7)*amrex::Math::powi<2>(Y(He4))*amrex::Math::powi<2>(state.rho) + screened_rates(k_n_N14_to_d_C13)*Y(N14)*state.rho + screened_rates(k_n_N15_to_d_C14)*Y(N15)*state.rho + 0.5*screened_rates(k_n_p_p_to_p_d)*amrex::Math::powi<2>(Y(H1))*amrex::Math::powi<2>(state.rho) + screened_rates(k_n_p_to_d)*Y(H1)*state.rho;
    jac.set(H2, N, scratch);

    scratch = 1.0*screened_rates(k_n_p_p_to_p_d)*Y(N)*Y(H1)*amrex::Math::powi<2>(state.rho) + screened_rates(k_n_p_to_d)*Y(N)*state.rho + screened_rates(k_p_Be9_to_d_He4_He4)*Y(Be9)*state.rho + 0.5*screened_rates(k_p_He4_He4_to_d_Be7)*amrex::Math::powi<2>(Y(He4))*amrex::Math::powi<2>(state.rho) + screened_rates(k_p_He4_to_d_He3)*Y(He4)*state.rho + screened_rates(k_p_Li7_to_d_Li6)*Y(Li7)*state.rho - screened_rates(k_p_d_to_He3)*Y(H2)*state.rho - screened_rates(k_p_d_to_n_p_p)*Y(H2)*state.rho + 1.0*screened_rates(k_p_p_to_d_weak_bet_pos_)*Y(H1)*state.rho + 1.0*screened_rates(k_p_p_to_d_weak_electron_capture)*Y(H1)*amrex::Math::powi<2>(state.rho)*state.y_e;
    jac.set(H2, H1, scratch);

    scratch = -screened_rates(k_He4_d_to_Li6)*Y(He4)*state.rho - screened_rates(k_d_Be7_to_p_He4_He4)*Y(Be7)*state.rho - screened_rates(k_d_C13_to_n_N14)*Y(C13)*state.rho - screened_rates(k_d_C14_to_n_N15)*Y(C14)*state.rho - screened_rates(k_d_He3_to_p_He4)*Y(He3)*state.rho - 0.5*screened_rates(k_d_He4_He4_to_p_Be9)*amrex::Math::powi<2>(Y(He4))*amrex::Math::powi<2>(state.rho) - screened_rates(k_d_Li6_to_n_Be7)*Y(Li6)*state.rho - screened_rates(k_d_Li6_to_p_Li7)*Y(Li6)*state.rho - screened_rates(k_d_Li7_to_n_He4_He4)*Y(Li7)*state.rho - 2.0*screened_rates(k_d_d_to_He4)*Y(H2)*state.rho - 2.0*screened_rates(k_d_d_to_n_He3)*Y(H2)*state.rho - screened_rates(k_d_to_n_p) - screened_rates(k_p_d_to_He3)*Y(H1)*state.rho - screened_rates(k_p_d_to_n_p_p)*Y(H1)*state.rho;
    jac.set(H2, H2, scratch);

    scratch = screened_rates(k_He3_to_p_d) - screened_rates(k_d_He3_to_p_He4)*Y(H2)*state.rho + 2.0*screened_rates(k_n_He3_to_d_d)*Y(N)*state.rho;
    jac.set(H2, He3, scratch);

    scratch = -screened_rates(k_He4_d_to_Li6)*Y(H2)*state.rho + 2.0*screened_rates(k_He4_to_d_d) - 1.0*screened_rates(k_d_He4_He4_to_p_Be9)*Y(He4)*Y(H2)*amrex::Math::powi<2>(state.rho) + 1.0*screened_rates(k_n_He4_He4_to_d_Li7)*Y(He4)*Y(N)*amrex::Math::powi<2>(state.rho) + 1.0*screened_rates(k_p_He4_He4_to_d_Be7)*Y(He4)*Y(H1)*amrex::Math::powi<2>(state.rho) + screened_rates(k_p_He4_to_d_He3)*Y(H1)*state.rho;
    jac.set(H2, He4, scratch);

    scratch = screened_rates(k_Li6_to_He4_d) - screened_rates(k_d_Li6_to_n_Be7)*Y(H2)*state.rho - screened_rates(k_d_Li6_to_p_Li7)*Y(H2)*state.rho;
    jac.set(H2, Li6, scratch);

    scratch = -screened_rates(k_d_Li7_to_n_He4_He4)*Y(H2)*state.rho + screened_rates(k_p_Li7_to_d_Li6)*Y(H1)*state.rho;
    jac.set(H2, Li7, scratch);

    scratch = -screened_rates(k_d_Be7_to_p_He4_He4)*Y(H2)*state.rho + screened_rates(k_n_Be7_to_d_Li6)*Y(N)*state.rho;
    jac.set(H2, Be7, scratch);

    scratch = screened_rates(k_p_Be9_to_d_He4_He4)*Y(H1)*state.rho;
    jac.set(H2, Be9, scratch);

    scratch = -screened_rates(k_d_C13_to_n_N14)*Y(H2)*state.rho;
    jac.set(H2, C13, scratch);

    scratch = -screened_rates(k_d_C14_to_n_N15)*Y(H2)*state.rho;
    jac.set(H2, C14, scratch);

    scratch = screened_rates(k_n_N14_to_d_C13)*Y(N)*state.rho;
    jac.set(H2, N14, scratch);

    scratch = screened_rates(k_n_N15_to_d_C14)*Y(N)*state.rho;
    jac.set(H2, N15, scratch);

    scratch = -screened_rates(k_n_He3_to_He4)*Y(He3)*state.rho - screened_rates(k_n_He3_to_d_d)*Y(He3)*state.rho + 0.5*screened_rates(k_n_p_He4_He4_to_He3_Li7)*amrex::Math::powi<2>(Y(He4))*Y(H1)*amrex::Math::powi<3>(state.rho);
    jac.set(He3, N, scratch);

    scratch = 0.5*screened_rates(k_n_p_He4_He4_to_He3_Li7)*amrex::Math::powi<2>(Y(He4))*Y(N)*amrex::Math::powi<3>(state.rho) - screened_rates(k_p_He3_to_He4_weak_bet_pos_)*Y(He3)*state.rho + screened_rates(k_p_He4_to_d_He3)*Y(He4)*state.rho + screened_rates(k_p_Li6_to_He4_He3)*Y(Li6)*state.rho + screened_rates(k_p_d_to_He3)*Y(H2)*state.rho + 0.5*screened_rates(k_p_p_He4_He4_to_He3_Be7)*amrex::Math::powi<2>(Y(He4))*Y(H1)*amrex::Math::powi<3>(state.rho) + 2.0*screened_rates(k_p_p_He4_to_He3_He3)*Y(He4)*Y(H1)*amrex::Math::powi<2>(state.rho);
    jac.set(He3, H1, scratch);

    scratch = -screened_rates(k_d_He3_to_p_He4)*Y(He3)*state.rho + 1.0*screened_rates(k_d_d_to_n_He3)*Y(H2)*state.rho + screened_rates(k_p_d_to_He3)*Y(H1)*state.rho;
    jac.set(He3, H2, scratch);

    scratch = -screened_rates(k_He3_Be7_to_p_p_He4_He4)*Y(Be7)*state.rho - 2.0*screened_rates(k_He3_He3_to_p_p_He4)*Y(He3)*state.rho - screened_rates(k_He3_Li7_to_n_p_He4_He4)*Y(Li7)*state.rho - screened_rates(k_He3_to_p_d) - screened_rates(k_He4_He3_to_Be7)*Y(He4)*state.rho - screened_rates(k_He4_He3_to_p_Li6)*Y(He4)*state.rho - screened_rates(k_d_He3_to_p_He4)*Y(H2)*state.rho - screened_rates(k_n_He3_to_He4)*Y(N)*state.rho - screened_rates(k_n_He3_to_d_d)*Y(N)*state.rho - screened_rates(k_p_He3_to_He4_weak_bet_pos_)*Y(H1)*state.rho;
    jac.set(He3, He3, scratch);

    scratch = -screened_rates(k_He4_He3_to_Be7)*Y(He3)*state.rho - screened_rates(k_He4_He3_to_p_Li6)*Y(He3)*state.rho + screened_rates(k_He4_to_n_He3) + 1.0*screened_rates(k_n_p_He4_He4_to_He3_Li7)*Y(He4)*Y(N)*Y(H1)*amrex::Math::powi<3>(state.rho) + screened_rates(k_p_He4_to_d_He3)*Y(H1)*state.rho + 0.5*screened_rates(k_p_p_He4_He4_to_He3_Be7)*Y(He4)*amrex::Math::powi<2>(Y(H1))*amrex::Math::powi<3>(state.rho) + screened_rates(k_p_p_He4_to_He3_He3)*amrex::Math::powi<2>(Y(H1))*amrex::Math::powi<2>(state.rho);
    jac.set(He3, He4, scratch);

    scratch = screened_rates(k_p_Li6_to_He4_He3)*Y(H1)*state.rho;
    jac.set(He3, Li6, scratch);

    scratch = -screened_rates(k_He3_Li7_to_n_p_He4_He4)*Y(He3)*state.rho;
    jac.set(He3, Li7, scratch);

    scratch = screened_rates(k_Be7_to_He4_He3) - screened_rates(k_He3_Be7_to_p_p_He4_He4)*Y(He3)*state.rho;
    jac.set(He3, Be7, scratch);

    scratch = screened_rates(k_n_Al25_to_He4_Na22)*Y(Al25)*state.rho + screened_rates(k_n_Al26_to_He4_Na23)*Y(Al26)*state.rho + screened_rates(k_n_Ar36_to_He4_S33)*Y(Ar36)*state.rho + screened_rates(k_n_Ar37_to_He4_S34)*Y(Ar37)*state.rho + screened_rates(k_n_Ar38_to_He4_S35)*Y(Ar38)*state.rho + screened_rates(k_n_Ar39_to_He4_S36)*Y(Ar39)*state.rho + screened_rates(k_n_B10_to_He4_Li7)*Y(B10)*state.rho + 2.0*screened_rates(k_n_B8_to_p_He4_He4)*Y(B8)*state.rho + 2.0*screened_rates(k_n_Be7_to_He4_He4)*Y(Be7)*state.rho + screened_rates(k_n_C12_to_He4_Be9)*Y(C12)*state.rho + screened_rates(k_n_Ca40_to_He4_Ar37)*Y(Ca40)*state.rho + screened_rates(k_n_Ca41_to_He4_Ar38)*Y(Ca41)*state.rho + screened_rates(k_n_Ca42_to_He4_Ar39)*Y(Ca42)*state.rho + screened_rates(k_n_Ca43_to_He4_Ar40)*Y(Ca43)*state.rho + screened_rates(k_n_Cl33_to_He4_P30)*Y(Cl33)*state.rho + screened_rates(k_n_Cl34_to_He4_P31)*Y(Cl34)*state.rho + screened_rates(k_n_Cl35_to_He4_P32)*Y(Cl35)*state.rho + screened_rates(k_n_Cl36_to_He4_P33)*Y(Cl36)*state.rho + screened_rates(k_n_Co53_to_He4_Mn50)*Y(Co53)*state.rho + screened_rates(k_n_Co54_to_He4_Mn51)*Y(Co54)*state.rho + screened_rates(k_n_Co55_to_He4_Mn52)*Y(Co55)*state.rho + screened_rates(k_n_Co56_to_He4_Mn53)*Y(Co56)*state.rho + screened_rates(k_n_Co57_to_He4_Mn54)*Y(Co57)*state.rho + screened_rates(k_n_Co58_to_He4_Mn55)*Y(Co58)*state.rho + screened_rates(k_n_Cr48_to_He4_Ti45)*Y(Cr48)*state.rho + screened_rates(k_n_Cr49_to_He4_Ti46)*Y(Cr49)*state.rho + screened_rates(k_n_Cr50_to_He4_Ti47)*Y(Cr50)*state.rho + screened_rates(k_n_Cr51_to_He4_Ti48)*Y(Cr51)*state.rho + screened_rates(k_n_Cr52_to_He4_Ti49)*Y(Cr52)*state.rho + screened_rates(k_n_Cr53_to_He4_Ti50)*Y(Cr53)*state.rho + screened_rates(k_n_Cr54_to_He4_Ti51)*Y(Cr54)*state.rho + screened_rates(k_n_Cu57_to_He4_Co54)*Y(Cu57)*state.rho + screened_rates(k_n_Cu58_to_He4_Co55)*Y(Cu58)*state.rho + screened_rates(k_n_Cu59_to_He4_Co56)*Y(Cu59)*state.rho + screened_rates(k_n_Cu60_to_He4_Co57)*Y(Cu60)*state.rho + screened_rates(k_n_Cu61_to_He4_Co58)*Y(Cu61)*state.rho + screened_rates(k_n_Cu62_to_He4_Co59)*Y(Cu62)*state.rho + screened_rates(k_n_F17_to_He4_N14)*Y(F17)*state.rho + screened_rates(k_n_F18_to_He4_N15)*Y(F18)*state.rho + screened_rates(k_n_Fe52_to_He4_Cr49)*Y(Fe52)*state.rho + screened_rates(k_n_Fe53_to_He4_Cr50)*Y(Fe53)*state.rho + screened_rates(k_n_Fe54_to_He4_Cr51)*Y(Fe54)*state.rho + screened_rates(k_n_Fe55_to_He4_Cr52)*Y(Fe55)*state.rho + screened_rates(k_n_Fe56_to_He4_Cr53)*Y(Fe56)*state.rho + screened_rates(k_n_Fe57_to_He4_Cr54)*Y(Fe57)*state.rho + screened_rates(k_n_Ga62_to_He4_Cu59)*Y(Ga62)*state.rho + screened_rates(k_n_Ga63_to_He4_Cu60)*Y(Ga63)*state.rho + screened_rates(k_n_Ga64_to_He4_Cu61)*Y(Ga64)*state.rho + screened_rates(k_n_Ge63_to_He4_Zn60)*Y(Ge63)*state.rho + screened_rates(k_n_Ge64_to_He4_Zn61)*Y(Ge64)*state.rho + screened_rates(k_n_He3_to_He4)*Y(He3)*state.rho - screened_rates(k_n_He4_He4_to_Be9)*amrex::Math::powi<2>(Y(He4))*amrex::Math::powi<2>(state.rho) - screened_rates(k_n_He4_He4_to_d_Li7)*amrex::Math::powi<2>(Y(He4))*amrex::Math::powi<2>(state.rho) + screened_rates(k_n_K37_to_He4_Cl34)*Y(K37)*state.rho + screened_rates(k_n_K38_to_He4_Cl35)*Y(K38)*state.rho + screened_rates(k_n_K39_to_He4_Cl36)*Y(K39)*state.rho + screened_rates(k_n_K40_to_He4_Cl37)*Y(K40)*state.rho + screened_rates(k_n_Mg23_to_He4_Ne20)*Y(Mg23)*state.rho + screened_rates(k_n_Mg24_to_He4_Ne21)*Y(Mg24)*state.rho + screened_rates(k_n_Mg25_to_He4_Ne22)*Y(Mg25)*state.rho + screened_rates(k_n_Mn50_to_He4_V47)*Y(Mn50)*state.rho + screened_rates(k_n_Mn51_to_He4_V48)*Y(Mn51)*state.rho + screened_rates(k_n_Mn52_to_He4_V49)*Y(Mn52)*state.rho + screened_rates(k_n_Mn53_to_He4_V50)*Y(Mn53)*state.rho + screened_rates(k_n_Mn54_to_He4_V51)*Y(Mn54)*state.rho + screened_rates(k_n_Mn55_to_He4_V52)*Y(Mn55)*state.rho + screened_rates(k_n_N13_to_He4_B10)*Y(N13)*state.rho + screened_rates(k_n_N14_to_He4_B11)*Y(N14)*state.rho + screened_rates(k_n_Na21_to_He4_F18)*Y(Na21)*state.rho + screened_rates(k_n_Na22_to_He4_F19)*Y(Na22)*state.rho + screened_rates(k_n_Ne18_to_He4_O15)*Y(Ne18)*state.rho + screened_rates(k_n_Ne19_to_He4_O16)*Y(Ne19)*state.rho + screened_rates(k_n_Ne20_to_He4_O17)*Y(Ne20)*state.rho + screened_rates(k_n_Ne21_to_He4_O18)*Y(Ne21)*state.rho + screened_rates(k_n_Ni56_to_He4_Fe53)*Y(Ni56)*state.rho + screened_rates(k_n_Ni57_to_He4_Fe54)*Y(Ni57)*state.rho + screened_rates(k_n_Ni58_to_He4_Fe55)*Y(Ni58)*state.rho + screened_rates(k_n_Ni59_to_He4_Fe56)*Y(Ni59)*state.rho + screened_rates(k_n_Ni60_to_He4_Fe57)*Y(Ni60)*state.rho + screened_rates(k_n_Ni61_to_He4_Fe58)*Y(Ni61)*state.rho + screened_rates(k_n_O15_to_He4_C12)*Y(O15)*state.rho + screened_rates(k_n_O16_to_He4_C13)*Y(O16)*state.rho + screened_rates(k_n_O17_to_He4_C14)*Y(O17)*state.rho + screened_rates(k_n_P29_to_He4_Al26)*Y(P29)*state.rho + screened_rates(k_n_P30_to_He4_Al27)*Y(P30)*state.rho + screened_rates(k_n_S32_to_He4_Si29)*Y(S32)*state.rho + screened_rates(k_n_S33_to_He4_Si30)*Y(S33)*state.rho + screened_rates(k_n_S34_to_He4_Si31)*Y(S34)*state.rho + screened_rates(k_n_S35_to_He4_Si32)*Y(S35)*state.rho + screened_rates(k_n_Sc43_to_He4_K40)*Y(Sc43)*state.rho + screened_rates(k_n_Sc44_to_He4_K41)*Y(Sc44)*state.rho + screened_rates(k_n_Si28_to_He4_Mg25)*Y(Si28)*state.rho + screened_rates(k_n_Si29_to_He4_Mg26)*Y(Si29)*state.rho + screened_rates(k_n_Ti44_to_He4_Ca41)*Y(Ti44)*state.rho + screened_rates(k_n_Ti45_to_He4_Ca42)*Y(Ti45)*state.rho + screened_rates(k_n_Ti46_to_He4_Ca43)*Y(Ti46)*state.rho + screened_rates(k_n_Ti47_to_He4_Ca44)*Y(Ti47)*state.rho + screened_rates(k_n_Ti48_to_He4_Ca45)*Y(Ti48)*state.rho + screened_rates(k_n_Ti49_to_He4_Ca46)*Y(Ti49)*state.rho + screened_rates(k_n_Ti50_to_He4_Ca47)*Y(Ti50)*state.rho + screened_rates(k_n_Ti51_to_He4_Ca48)*Y(Ti51)*state.rho + screened_rates(k_n_V46_to_He4_Sc43)*Y(V46)*state.rho + screened_rates(k_n_V47_to_He4_Sc44)*Y(V47)*state.rho + screened_rates(k_n_V48_to_He4_Sc45)*Y(V48)*state.rho + screened_rates(k_n_V49_to_He4_Sc46)*Y(V49)*state.rho + screened_rates(k_n_V50_to_He4_Sc47)*Y(V50)*state.rho + screened_rates(k_n_V51_to_He4_Sc48)*Y(V51)*state.rho + screened_rates(k_n_V52_to_He4_Sc49)*Y(V52)*state.rho + screened_rates(k_n_Zn59_to_He4_Ni56)*Y(Zn59)*state.rho + screened_rates(k_n_Zn60_to_He4_Ni57)*Y(Zn60)*state.rho + screened_rates(k_n_Zn61_to_He4_Ni58)*Y(Zn61)*state.rho + screened_rates(k_n_Zn62_to_He4_Ni59)*Y(Zn62)*state.rho + screened_rates(k_n_Zn63_to_He4_Ni60)*Y(Zn63)*state.rho + screened_rates(k_n_Zn64_to_He4_Ni61)*Y(Zn64)*state.rho + screened_rates(k_n_Zn65_to_He4_Ni62)*Y(Zn65)*state.rho + screened_rates(k_n_Zn66_to_He4_Ni63)*Y(Zn66)*state.rho - screened_rates(k_n_p_He4_He4_to_He3_Li7)*amrex::Math::powi<2>(Y(He4))*Y(H1)*amrex::Math::powi<3>(state.rho) - screened_rates(k_n_p_He4_He4_to_p_Be9)*amrex::Math::powi<2>(Y(He4))*Y(H1)*amrex::Math::powi<3>(state.rho) - screened_rates(k_n_p_He4_to_Li6)*Y(He4)*Y(H1)*amrex::Math::powi<2>(state.rho);
    jac.set(He4, N, scratch);

    scratch = -screened_rates(k_n_p_He4_He4_to_He3_Li7)*amrex::Math::powi<2>(Y(He4))*Y(N)*amrex::Math::powi<3>(state.rho) - screened_rates(k_n_p_He4_He4_to_p_Be9)*amrex::Math::powi<2>(Y(He4))*Y(N)*amrex::Math::powi<3>(state.rho) - screened_rates(k_n_p_He4_to_Li6)*Y(He4)*Y(N)*amrex::Math::powi<2>(state.rho) + screened_rates(k_p_Al26_to_He4_Mg23)*Y(Al26)*state.rho + screened_rates(k_p_Al27_to_He4_Mg24)*Y(Al27)*state.rho + screened_rates(k_p_Ar36_to_He4_Cl33)*Y(Ar36)*state.rho + screened_rates(k_p_Ar37_to_He4_Cl34)*Y(Ar37)*state.rho + screened_rates(k_p_Ar38_to_He4_Cl35)*Y(Ar38)*state.rho + screened_rates(k_p_Ar39_to_He4_Cl36)*Y(Ar39)*state.rho + screened_rates(k_p_Ar40_to_He4_Cl37)*Y(Ar40)*state.rho + screened_rates(k_p_B10_to_He4_Be7)*Y(B10)*state.rho + 3.0*screened_rates(k_p_B11_to_He4_He4_He4)*Y(B11)*state.rho + screened_rates(k_p_Be9_to_He4_Li6)*Y(Be9)*state.rho + 2.0*screened_rates(k_p_Be9_to_d_He4_He4)*Y(Be9)*state.rho + 2.0*screened_rates(k_p_Be9_to_n_p_He4_He4)*Y(Be9)*state.rho + screened_rates(k_p_C13_to_He4_B10)*Y(C13)*state.rho + screened_rates(k_p_C14_to_He4_B11)*Y(C14)*state.rho + screened_rates(k_p_Ca40_to_He4_K37)*Y(Ca40)*state.rho + screened_rates(k_p_Ca41_to_He4_K38)*Y(Ca41)*state.rho + screened_rates(k_p_Ca42_to_He4_K39)*Y(Ca42)*state.rho + screened_rates(k_p_Ca43_to_He4_K40)*Y(Ca43)*state.rho + screened_rates(k_p_Ca44_to_He4_K41)*Y(Ca44)*state.rho + screened_rates(k_p_Cl35_to_He4_S32)*Y(Cl35)*state.rho + screened_rates(k_p_Cl36_to_He4_S33)*Y(Cl36)*state.rho + screened_rates(k_p_Cl37_to_He4_S34)*Y(Cl37)*state.rho + screened_rates(k_p_Co55_to_He4_Fe52)*Y(Co55)*state.rho + screened_rates(k_p_Co56_to_He4_Fe53)*Y(Co56)*state.rho + screened_rates(k_p_Co57_to_He4_Fe54)*Y(Co57)*state.rho + screened_rates(k_p_Co58_to_He4_Fe55)*Y(Co58)*state.rho + screened_rates(k_p_Co59_to_He4_Fe56)*Y(Co59)*state.rho + screened_rates(k_p_Cr49_to_He4_V46)*Y(Cr49)*state.rho + screened_rates(k_p_Cr50_to_He4_V47)*Y(Cr50)*state.rho + screened_rates(k_p_Cr51_to_He4_V48)*Y(Cr51)*state.rho + screened_rates(k_p_Cr52_to_He4_V49)*Y(Cr52)*state.rho + screened_rates(k_p_Cr53_to_He4_V50)*Y(Cr53)*state.rho + screened_rates(k_p_Cr54_to_He4_V51)*Y(Cr54)*state.rho + screened_rates(k_p_Cu59_to_He4_Ni56)*Y(Cu59)*state.rho + screened_rates(k_p_Cu60_to_He4_Ni57)*Y(Cu60)*state.rho + screened_rates(k_p_Cu61_to_He4_Ni58)*Y(Cu61)*state.rho + screened_rates(k_p_Cu62_to_He4_Ni59)*Y(Cu62)*state.rho + screened_rates(k_p_Cu63_to_He4_Ni60)*Y(Cu63)*state.rho + screened_rates(k_p_Cu64_to_He4_Ni61)*Y(Cu64)*state.rho + screened_rates(k_p_Cu65_to_He4_Ni62)*Y(Cu65)*state.rho + screened_rates(k_p_F17_to_He4_O14)*Y(F17)*state.rho + screened_rates(k_p_F18_to_He4_O15)*Y(F18)*state.rho + screened_rates(k_p_F19_to_He4_O16)*Y(F19)*state.rho + screened_rates(k_p_Fe53_to_He4_Mn50)*Y(Fe53)*state.rho + screened_rates(k_p_Fe54_to_He4_Mn51)*Y(Fe54)*state.rho + screened_rates(k_p_Fe55_to_He4_Mn52)*Y(Fe55)*state.rho + screened_rates(k_p_Fe56_to_He4_Mn53)*Y(Fe56)*state.rho + screened_rates(k_p_Fe57_to_He4_Mn54)*Y(Fe57)*state.rho + screened_rates(k_p_Fe58_to_He4_Mn55)*Y(Fe58)*state.rho + screened_rates(k_p_Ga62_to_He4_Zn59)*Y(Ga62)*state.rho + screened_rates(k_p_Ga63_to_He4_Zn60)*Y(Ga63)*state.rho + screened_rates(k_p_Ga64_to_He4_Zn61)*Y(Ga64)*state.rho + screened_rates(k_p_He3_to_He4_weak_bet_pos_)*Y(He3)*state.rho - screened_rates(k_p_He4_He4_to_d_Be7)*amrex::Math::powi<2>(Y(He4))*amrex::Math::powi<2>(state.rho) - screened_rates(k_p_He4_He4_to_n_B8)*amrex::Math::powi<2>(Y(He4))*amrex::Math::powi<2>(state.rho) - screened_rates(k_p_He4_to_d_He3)*Y(He4)*state.rho + screened_rates(k_p_K39_to_He4_Ar36)*Y(K39)*state.rho + screened_rates(k_p_K40_to_He4_Ar37)*Y(K40)*state.rho + screened_rates(k_p_K41_to_He4_Ar38)*Y(K41)*state.rho + screened_rates(k_p_Li6_to_He4_He3)*Y(Li6)*state.rho + 2.0*screened_rates(k_p_Li7_to_He4_He4)*Y(Li7)*state.rho + screened_rates(k_p_Mg24_to_He4_Na21)*Y(Mg24)*state.rho + screened_rates(k_p_Mg25_to_He4_Na22)*Y(Mg25)*state.rho + screened_rates(k_p_Mg26_to_He4_Na23)*Y(Mg26)*state.rho + screened_rates(k_p_Mn51_to_He4_Cr48)*Y(Mn51)*state.rho + screened_rates(k_p_Mn52_to_He4_Cr49)*Y(Mn52)*state.rho + screened_rates(k_p_Mn53_to_He4_Cr50)*Y(Mn53)*state.rho + screened_rates(k_p_Mn54_to_He4_Cr51)*Y(Mn54)*state.rho + screened_rates(k_p_Mn55_to_He4_Cr52)*Y(Mn55)*state.rho + screened_rates(k_p_N15_to_He4_C12)*Y(N15)*state.rho + screened_rates(k_p_Na21_to_He4_Ne18)*Y(Na21)*state.rho + screened_rates(k_p_Na22_to_He4_Ne19)*Y(Na22)*state.rho + screened_rates(k_p_Na23_to_He4_Ne20)*Y(Na23)*state.rho + screened_rates(k_p_Ne20_to_He4_F17)*Y(Ne20)*state.rho + screened_rates(k_p_Ne21_to_He4_F18)*Y(Ne21)*state.rho + screened_rates(k_p_Ne22_to_He4_F19)*Y(Ne22)*state.rho + screened_rates(k_p_Ni56_to_He4_Co53)*Y(Ni56)*state.rho + screened_rates(k_p_Ni57_to_He4_Co54)*Y(Ni57)*state.rho + screened_rates(k_p_Ni58_to_He4_Co55)*Y(Ni58)*state.rho + screened_rates(k_p_Ni59_to_He4_Co56)*Y(Ni59)*state.rho + screened_rates(k_p_Ni60_to_He4_Co57)*Y(Ni60)*state.rho + screened_rates(k_p_Ni61_to_He4_Co58)*Y(Ni61)*state.rho + screened_rates(k_p_Ni62_to_He4_Co59)*Y(Ni62)*state.rho + screened_rates(k_p_O16_to_He4_N13)*Y(O16)*state.rho + screened_rates(k_p_O17_to_He4_N14)*Y(O17)*state.rho + screened_rates(k_p_O18_to_He4_N15)*Y(O18)*state.rho + screened_rates(k_p_P31_to_He4_Si28)*Y(P31)*state.rho + screened_rates(k_p_P32_to_He4_Si29)*Y(P32)*state.rho + screened_rates(k_p_P33_to_He4_Si30)*Y(P33)*state.rho + screened_rates(k_p_S32_to_He4_P29)*Y(S32)*state.rho + screened_rates(k_p_S33_to_He4_P30)*Y(S33)*state.rho + screened_rates(k_p_S34_to_He4_P31)*Y(S34)*state.rho + screened_rates(k_p_S35_to_He4_P32)*Y(S35)*state.rho + screened_rates(k_p_S36_to_He4_P33)*Y(S36)*state.rho + screened_rates(k_p_Sc43_to_He4_Ca40)*Y(Sc43)*state.rho + screened_rates(k_p_Sc44_to_He4_Ca41)*Y(Sc44)*state.rho + screened_rates(k_p_Sc45_to_He4_Ca42)*Y(Sc45)*state.rho + screened_rates(k_p_Sc46_to_He4_Ca43)*Y(Sc46)*state.rho + screened_rates(k_p_Sc47_to_He4_Ca44)*Y(Sc47)*state.rho + screened_rates(k_p_Sc48_to_He4_Ca45)*Y(Sc48)*state.rho + screened_rates(k_p_Sc49_to_He4_Ca46)*Y(Sc49)*state.rho + screened_rates(k_p_Si28_to_He4_Al25)*Y(Si28)*state.rho + screened_rates(k_p_Si29_to_He4_Al26)*Y(Si29)*state.rho + screened_rates(k_p_Si30_to_He4_Al27)*Y(Si30)*state.rho + screened_rates(k_p_Ti46_to_He4_Sc43)*Y(Ti46)*state.rho + screened_rates(k_p_Ti47_to_He4_Sc44)*Y(Ti47)*state.rho + screened_rates(k_p_Ti48_to_He4_Sc45)*Y(Ti48)*state.rho + screened_rates(k_p_Ti49_to_He4_Sc46)*Y(Ti49)*state.rho + screened_rates(k_p_Ti50_to_He4_Sc47)*Y(Ti50)*state.rho + screened_rates(k_p_Ti51_to_He4_Sc48)*Y(Ti51)*state.rho + screened_rates(k_p_V47_to_He4_Ti44)*Y(V47)*state.rho + screened_rates(k_p_V48_to_He4_Ti45)*Y(V48)*state.rho + screened_rates(k_p_V49_to_He4_Ti46)*Y(V49)*state.rho + screened_rates(k_p_V50_to_He4_Ti47)*Y(V50)*state.rho + screened_rates(k_p_V51_to_He4_Ti48)*Y(V51)*state.rho + screened_rates(k_p_V52_to_He4_Ti49)*Y(V52)*state.rho + screened_rates(k_p_Zn60_to_He4_Cu57)*Y(Zn60)*state.rho + screened_rates(k_p_Zn61_to_He4_Cu58)*Y(Zn61)*state.rho + screened_rates(k_p_Zn62_to_He4_Cu59)*Y(Zn62)*state.rho + screened_rates(k_p_Zn63_to_He4_Cu60)*Y(Zn63)*state.rho + screened_rates(k_p_Zn64_to_He4_Cu61)*Y(Zn64)*state.rho + screened_rates(k_p_Zn65_to_He4_Cu62)*Y(Zn65)*state.rho + screened_rates(k_p_Zn66_to_He4_Cu63)*Y(Zn66)*state.rho - 1.0*screened_rates(k_p_p_He4_He4_to_He3_Be7)*amrex::Math::powi<2>(Y(He4))*Y(H1)*amrex::Math::powi<3>(state.rho) - 1.0*screened_rates(k_p_p_He4_to_He3_He3)*Y(He4)*Y(H1)*amrex::Math::powi<2>(state.rho);
    jac.set(He4, H1, scratch);

    scratch = -screened_rates(k_He4_d_to_Li6)*Y(He4)*state.rho + 2.0*screened_rates(k_d_Be7_to_p_He4_He4)*Y(Be7)*state.rho + screened_rates(k_d_He3_to_p_He4)*Y(He3)*state.rho - screened_rates(k_d_He4_He4_to_p_Be9)*amrex::Math::powi<2>(Y(He4))*amrex::Math::powi<2>(state.rho) + 2.0*screened_rates(k_d_Li7_to_n_He4_He4)*Y(Li7)*state.rho + 1.0*screened_rates(k_d_d_to_He4)*Y(H2)*state.rho;
    jac.set(He4, H2, scratch);

    scratch = 2.0*screened_rates(k_He3_Be7_to_p_p_He4_He4)*Y(Be7)*state.rho + 1.0*screened_rates(k_He3_He3_to_p_p_He4)*Y(He3)*state.rho + 2.0*screened_rates(k_He3_Li7_to_n_p_He4_He4)*Y(Li7)*state.rho - screened_rates(k_He4_He3_to_Be7)*Y(He4)*state.rho - screened_rates(k_He4_He3_to_p_Li6)*Y(He4)*state.rho + screened_rates(k_d_He3_to_p_He4)*Y(H2)*state.rho + screened_rates(k_n_He3_to_He4)*Y(N)*state.rho + screened_rates(k_p_He3_to_He4_weak_bet_pos_)*Y(H1)*state.rho;
    jac.set(He4, He3, scratch);

    scratch = -screened_rates(k_He4_Al25_to_P29)*Y(Al25)*state.rho - screened_rates(k_He4_Al25_to_p_Si28)*Y(Al25)*state.rho - screened_rates(k_He4_Al26_to_P30)*Y(Al26)*state.rho - screened_rates(k_He4_Al26_to_n_P29)*Y(Al26)*state.rho - screened_rates(k_He4_Al26_to_p_Si29)*Y(Al26)*state.rho - screened_rates(k_He4_Al27_to_P31)*Y(Al27)*state.rho - screened_rates(k_He4_Al27_to_n_P30)*Y(Al27)*state.rho - screened_rates(k_He4_Al27_to_p_Si30)*Y(Al27)*state.rho - screened_rates(k_He4_Ar36_to_Ca40)*Y(Ar36)*state.rho - screened_rates(k_He4_Ar36_to_p_K39)*Y(Ar36)*state.rho - screened_rates(k_He4_Ar37_to_Ca41)*Y(Ar37)*state.rho - screened_rates(k_He4_Ar37_to_n_Ca40)*Y(Ar37)*state.rho - screened_rates(k_He4_Ar37_to_p_K40)*Y(Ar37)*state.rho - screened_rates(k_He4_Ar38_to_Ca42)*Y(Ar38)*state.rho - screened_rates(k_He4_Ar38_to_n_Ca41)*Y(Ar38)*state.rho - screened_rates(k_He4_Ar38_to_p_K41)*Y(Ar38)*state.rho - screened_rates(k_He4_Ar39_to_Ca43)*Y(Ar39)*state.rho - screened_rates(k_He4_Ar39_to_n_Ca42)*Y(Ar39)*state.rho - screened_rates(k_He4_Ar40_to_Ca44)*Y(Ar40)*state.rho - screened_rates(k_He4_Ar40_to_n_Ca43)*Y(Ar40)*state.rho - screened_rates(k_He4_B10_to_n_N13)*Y(B10)*state.rho - screened_rates(k_He4_B10_to_p_C13)*Y(B10)*state.rho - screened_rates(k_He4_B11_to_n_N14)*Y(B11)*state.rho - screened_rates(k_He4_B11_to_p_C14)*Y(B11)*state.rho - screened_rates(k_He4_Be7_to_p_B10)*Y(Be7)*state.rho - screened_rates(k_He4_Be9_to_n_C12)*Y(Be9)*state.rho - screened_rates(k_He4_C12_to_O16)*Y(C12)*state.rho - screened_rates(k_He4_C12_to_n_O15)*Y(C12)*state.rho - screened_rates(k_He4_C12_to_p_N15)*Y(C12)*state.rho - screened_rates(k_He4_C13_to_n_O16)*Y(C13)*state.rho - screened_rates(k_He4_C14_to_O18)*Y(C14)*state.rho - screened_rates(k_He4_C14_to_n_O17)*Y(C14)*state.rho - screened_rates(k_He4_Ca40_to_Ti44)*Y(Ca40)*state.rho - screened_rates(k_He4_Ca40_to_p_Sc43)*Y(Ca40)*state.rho - screened_rates(k_He4_Ca41_to_Ti45)*Y(Ca41)*state.rho - screened_rates(k_He4_Ca41_to_n_Ti44)*Y(Ca41)*state.rho - screened_rates(k_He4_Ca41_to_p_Sc44)*Y(Ca41)*state.rho - screened_rates(k_He4_Ca42_to_Ti46)*Y(Ca42)*state.rho - screened_rates(k_He4_Ca42_to_n_Ti45)*Y(Ca42)*state.rho - screened_rates(k_He4_Ca42_to_p_Sc45)*Y(Ca42)*state.rho - screened_rates(k_He4_Ca43_to_Ti47)*Y(Ca43)*state.rho - screened_rates(k_He4_Ca43_to_n_Ti46)*Y(Ca43)*state.rho - screened_rates(k_He4_Ca43_to_p_Sc46)*Y(Ca43)*state.rho - screened_rates(k_He4_Ca44_to_Ti48)*Y(Ca44)*state.rho - screened_rates(k_He4_Ca44_to_n_Ti47)*Y(Ca44)*state.rho - screened_rates(k_He4_Ca44_to_p_Sc47)*Y(Ca44)*state.rho - screened_rates(k_He4_Ca45_to_Ti49)*Y(Ca45)*state.rho - screened_rates(k_He4_Ca45_to_n_Ti48)*Y(Ca45)*state.rho - screened_rates(k_He4_Ca45_to_p_Sc48)*Y(Ca45)*state.rho - screened_rates(k_He4_Ca46_to_Ti50)*Y(Ca46)*state.rho - screened_rates(k_He4_Ca46_to_n_Ti49)*Y(Ca46)*state.rho - screened_rates(k_He4_Ca46_to_p_Sc49)*Y(Ca46)*state.rho - screened_rates(k_He4_Ca47_to_Ti51)*Y(Ca47)*state.rho - screened_rates(k_He4_Ca47_to_n_Ti50)*Y(Ca47)*state.rho - screened_rates(k_He4_Ca48_to_n_Ti51)*Y(Ca48)*state.rho - screened_rates(k_He4_Cl33_to_K37)*Y(Cl33)*state.rho - screened_rates(k_He4_Cl33_to_p_Ar36)*Y(Cl33)*state.rho - screened_rates(k_He4_Cl34_to_K38)*Y(Cl34)*state.rho - screened_rates(k_He4_Cl34_to_n_K37)*Y(Cl34)*state.rho - screened_rates(k_He4_Cl34_to_p_Ar37)*Y(Cl34)*state.rho - screened_rates(k_He4_Cl35_to_K39)*Y(Cl35)*state.rho - screened_rates(k_He4_Cl35_to_n_K38)*Y(Cl35)*state.rho - screened_rates(k_He4_Cl35_to_p_Ar38)*Y(Cl35)*state.rho - screened_rates(k_He4_Cl36_to_K40)*Y(Cl36)*state.rho - screened_rates(k_He4_Cl36_to_n_K39)*Y(Cl36)*state.rho - screened_rates(k_He4_Cl36_to_p_Ar39)*Y(Cl36)*state.rho - screened_rates(k_He4_Cl37_to_K41)*Y(Cl37)*state.rho - screened_rates(k_He4_Cl37_to_n_K40)*Y(Cl37)*state.rho - screened_rates(k_He4_Cl37_to_p_Ar40)*Y(Cl37)*state.rho - screened_rates(k_He4_Co53_to_Cu57)*Y(Co53)*state.rho - screened_rates(k_He4_Co53_to_p_Ni56)*Y(Co53)*state.rho - screened_rates(k_He4_Co54_to_Cu58)*Y(Co54)*state.rho - screened_rates(k_He4_Co54_to_n_Cu57)*Y(Co54)*state.rho - screened_rates(k_He4_Co54_to_p_Ni57)*Y(Co54)*state.rho - screened_rates(k_He4_Co55_to_Cu59)*Y(Co55)*state.rho - screened_rates(k_He4_Co55_to_n_Cu58)*Y(Co55)*state.rho - screened_rates(k_He4_Co55_to_p_Ni58)*Y(Co55)*state.rho - screened_rates(k_He4_Co56_to_Cu60)*Y(Co56)*state.rho - screened_rates(k_He4_Co56_to_n_Cu59)*Y(Co56)*state.rho - screened_rates(k_He4_Co56_to_p_Ni59)*Y(Co56)*state.rho - screened_rates(k_He4_Co57_to_Cu61)*Y(Co57)*state.rho - screened_rates(k_He4_Co57_to_n_Cu60)*Y(Co57)*state.rho - screened_rates(k_He4_Co57_to_p_Ni60)*Y(Co57)*state.rho - screened_rates(k_He4_Co58_to_Cu62)*Y(Co58)*state.rho - screened_rates(k_He4_Co58_to_n_Cu61)*Y(Co58)*state.rho - screened_rates(k_He4_Co58_to_p_Ni61)*Y(Co58)*state.rho - screened_rates(k_He4_Co59_to_Cu63)*Y(Co59)*state.rho - screened_rates(k_He4_Co59_to_n_Cu62)*Y(Co59)*state.rho - screened_rates(k_He4_Co59_to_p_Ni62)*Y(Co59)*state.rho - screened_rates(k_He4_Cr48_to_Fe52)*Y(Cr48)*state.rho - screened_rates(k_He4_Cr48_to_p_Mn51)*Y(Cr48)*state.rho - screened_rates(k_He4_Cr49_to_Fe53)*Y(Cr49)*state.rho - screened_rates(k_He4_Cr49_to_n_Fe52)*Y(Cr49)*state.rho - screened_rates(k_He4_Cr49_to_p_Mn52)*Y(Cr49)*state.rho - screened_rates(k_He4_Cr50_to_Fe54)*Y(Cr50)*state.rho - screened_rates(k_He4_Cr50_to_n_Fe53)*Y(Cr50)*state.rho - screened_rates(k_He4_Cr50_to_p_Mn53)*Y(Cr50)*state.rho - screened_rates(k_He4_Cr51_to_Fe55)*Y(Cr51)*state.rho - screened_rates(k_He4_Cr51_to_n_Fe54)*Y(Cr51)*state.rho - screened_rates(k_He4_Cr51_to_p_Mn54)*Y(Cr51)*state.rho - screened_rates(k_He4_Cr52_to_Fe56)*Y(Cr52)*state.rho - screened_rates(k_He4_Cr52_to_n_Fe55)*Y(Cr52)*state.rho - screened_rates(k_He4_Cr52_to_p_Mn55)*Y(Cr52)*state.rho - screened_rates(k_He4_Cr53_to_Fe57)*Y(Cr53)*state.rho - screened_rates(k_He4_Cr53_to_n_Fe56)*Y(Cr53)*state.rho - screened_rates(k_He4_Cr54_to_Fe58)*Y(Cr54)*state.rho - screened_rates(k_He4_Cr54_to_n_Fe57)*Y(Cr54)*state.rho - screened_rates(k_He4_Cu57_to_p_Zn60)*Y(Cu57)*state.rho - screened_rates(k_He4_Cu58_to_Ga62)*Y(Cu58)*state.rho - screened_rates(k_He4_Cu58_to_p_Zn61)*Y(Cu58)*state.rho - screened_rates(k_He4_Cu59_to_Ga63)*Y(Cu59)*state.rho - screened_rates(k_He4_Cu59_to_n_Ga62)*Y(Cu59)*state.rho - screened_rates(k_He4_Cu59_to_p_Zn62)*Y(Cu59)*state.rho - screened_rates(k_He4_Cu60_to_Ga64)*Y(Cu60)*state.rho - screened_rates(k_He4_Cu60_to_n_Ga63)*Y(Cu60)*state.rho - screened_rates(k_He4_Cu60_to_p_Zn63)*Y(Cu60)*state.rho - screened_rates(k_He4_Cu61_to_n_Ga64)*Y(Cu61)*state.rho - screened_rates(k_He4_Cu61_to_p_Zn64)*Y(Cu61)*state.rho - screened_rates(k_He4_Cu62_to_p_Zn65)*Y(Cu62)*state.rho - screened_rates(k_He4_Cu63_to_p_Zn66)*Y(Cu63)*state.rho - screened_rates(k_He4_F17_to_Na21)*Y(F17)*state.rho - screened_rates(k_He4_F17_to_p_Ne20)*Y(F17)*state.rho - screened_rates(k_He4_F18_to_Na22)*Y(F18)*state.rho - screened_rates(k_He4_F18_to_n_Na21)*Y(F18)*state.rho - screened_rates(k_He4_F18_to_p_Ne21)*Y(F18)*state.rho - screened_rates(k_He4_F19_to_Na23)*Y(F19)*state.rho - screened_rates(k_He4_F19_to_n_Na22)*Y(F19)*state.rho - screened_rates(k_He4_F19_to_p_Ne22)*Y(F19)*state.rho - screened_rates(k_He4_Fe52_to_Ni56)*Y(Fe52)*state.rho - screened_rates(k_He4_Fe52_to_p_Co55)*Y(Fe52)*state.rho - screened_rates(k_He4_Fe53_to_Ni57)*Y(Fe53)*state.rho - screened_rates(k_He4_Fe53_to_n_Ni56)*Y(Fe53)*state.rho - screened_rates(k_He4_Fe53_to_p_Co56)*Y(Fe53)*state.rho - screened_rates(k_He4_Fe54_to_Ni58)*Y(Fe54)*state.rho - screened_rates(k_He4_Fe54_to_n_Ni57)*Y(Fe54)*state.rho - screened_rates(k_He4_Fe54_to_p_Co57)*Y(Fe54)*state.rho - screened_rates(k_He4_Fe55_to_Ni59)*Y(Fe55)*state.rho - screened_rates(k_He4_Fe55_to_n_Ni58)*Y(Fe55)*state.rho - screened_rates(k_He4_Fe55_to_p_Co58)*Y(Fe55)*state.rho - screened_rates(k_He4_Fe56_to_Ni60)*Y(Fe56)*state.rho - screened_rates(k_He4_Fe56_to_n_Ni59)*Y(Fe56)*state.rho - screened_rates(k_He4_Fe56_to_p_Co59)*Y(Fe56)*state.rho - screened_rates(k_He4_Fe57_to_Ni61)*Y(Fe57)*state.rho - screened_rates(k_He4_Fe57_to_n_Ni60)*Y(Fe57)*state.rho - screened_rates(k_He4_Fe58_to_Ni62)*Y(Fe58)*state.rho - screened_rates(k_He4_Fe58_to_n_Ni61)*Y(Fe58)*state.rho - screened_rates(k_He4_He3_to_Be7)*Y(He3)*state.rho - screened_rates(k_He4_He3_to_p_Li6)*Y(He3)*state.rho - 1.5*screened_rates(k_He4_He4_He4_to_C12)*amrex::Math::powi<2>(Y(He4))*amrex::Math::powi<2>(state.rho) - 1.5*screened_rates(k_He4_He4_He4_to_p_B11)*amrex::Math::powi<2>(Y(He4))*amrex::Math::powi<2>(state.rho) - 2.0*screened_rates(k_He4_He4_to_n_Be7)*Y(He4)*state.rho - 2.0*screened_rates(k_He4_He4_to_p_Li7)*Y(He4)*state.rho - screened_rates(k_He4_K37_to_p_Ca40)*Y(K37)*state.rho - screened_rates(k_He4_K38_to_p_Ca41)*Y(K38)*state.rho - screened_rates(k_He4_K39_to_Sc43)*Y(K39)*state.rho - screened_rates(k_He4_K39_to_p_Ca42)*Y(K39)*state.rho - screened_rates(k_He4_K40_to_Sc44)*Y(K40)*state.rho - screened_rates(k_He4_K40_to_n_Sc43)*Y(K40)*state.rho - screened_rates(k_He4_K40_to_p_Ca43)*Y(K40)*state.rho - screened_rates(k_He4_K41_to_Sc45)*Y(K41)*state.rho - screened_rates(k_He4_K41_to_n_Sc44)*Y(K41)*state.rho - screened_rates(k_He4_K41_to_p_Ca44)*Y(K41)*state.rho - screened_rates(k_He4_Li6_to_B10)*Y(Li6)*state.rho - screened_rates(k_He4_Li6_to_p_Be9)*Y(Li6)*state.rho - screened_rates(k_He4_Li7_to_B11)*Y(Li7)*state.rho - screened_rates(k_He4_Li7_to_n_B10)*Y(Li7)*state.rho - screened_rates(k_He4_Mg23_to_p_Al26)*Y(Mg23)*state.rho - screened_rates(k_He4_Mg24_to_C12_O16)*Y(Mg24)*state.rho - screened_rates(k_He4_Mg24_to_Si28)*Y(Mg24)*state.rho - screened_rates(k_He4_Mg24_to_p_Al27)*Y(Mg24)*state.rho - screened_rates(k_He4_Mg25_to_Si29)*Y(Mg25)*state.rho - screened_rates(k_He4_Mg25_to_n_Si28)*Y(Mg25)*state.rho - screened_rates(k_He4_Mg26_to_Si30)*Y(Mg26)*state.rho - screened_rates(k_He4_Mg26_to_n_Si29)*Y(Mg26)*state.rho - screened_rates(k_He4_Mn50_to_Co54)*Y(Mn50)*state.rho - screened_rates(k_He4_Mn50_to_n_Co53)*Y(Mn50)*state.rho - screened_rates(k_He4_Mn50_to_p_Fe53)*Y(Mn50)*state.rho - screened_rates(k_He4_Mn51_to_Co55)*Y(Mn51)*state.rho - screened_rates(k_He4_Mn51_to_n_Co54)*Y(Mn51)*state.rho - screened_rates(k_He4_Mn51_to_p_Fe54)*Y(Mn51)*state.rho - screened_rates(k_He4_Mn52_to_Co56)*Y(Mn52)*state.rho - screened_rates(k_He4_Mn52_to_n_Co55)*Y(Mn52)*state.rho - screened_rates(k_He4_Mn52_to_p_Fe55)*Y(Mn52)*state.rho - screened_rates(k_He4_Mn53_to_Co57)*Y(Mn53)*state.rho - screened_rates(k_He4_Mn53_to_n_Co56)*Y(Mn53)*state.rho - screened_rates(k_He4_Mn53_to_p_Fe56)*Y(Mn53)*state.rho - screened_rates(k_He4_Mn54_to_Co58)*Y(Mn54)*state.rho - screened_rates(k_He4_Mn54_to_n_Co57)*Y(Mn54)*state.rho - screened_rates(k_He4_Mn54_to_p_Fe57)*Y(Mn54)*state.rho - screened_rates(k_He4_Mn55_to_Co59)*Y(Mn55)*state.rho - screened_rates(k_He4_Mn55_to_n_Co58)*Y(Mn55)*state.rho - screened_rates(k_He4_Mn55_to_p_Fe58)*Y(Mn55)*state.rho - screened_rates(k_He4_N13_to_p_O16)*Y(N13)*state.rho - screened_rates(k_He4_N14_to_F18)*Y(N14)*state.rho - screened_rates(k_He4_N14_to_n_F17)*Y(N14)*state.rho - screened_rates(k_He4_N14_to_p_O17)*Y(N14)*state.rho - screened_rates(k_He4_N15_to_F19)*Y(N15)*state.rho - screened_rates(k_He4_N15_to_n_F18)*Y(N15)*state.rho - screened_rates(k_He4_N15_to_p_O18)*Y(N15)*state.rho - screened_rates(k_He4_Na21_to_Al25)*Y(Na21)*state.rho - screened_rates(k_He4_Na21_to_p_Mg24)*Y(Na21)*state.rho - screened_rates(k_He4_Na22_to_Al26)*Y(Na22)*state.rho - screened_rates(k_He4_Na22_to_n_Al25)*Y(Na22)*state.rho - screened_rates(k_He4_Na22_to_p_Mg25)*Y(Na22)*state.rho - screened_rates(k_He4_Na23_to_Al27)*Y(Na23)*state.rho - screened_rates(k_He4_Na23_to_n_Al26)*Y(Na23)*state.rho - screened_rates(k_He4_Na23_to_p_Mg26)*Y(Na23)*state.rho - screened_rates(k_He4_Ne18_to_p_Na21)*Y(Ne18)*state.rho - screened_rates(k_He4_Ne19_to_Mg23)*Y(Ne19)*state.rho - screened_rates(k_He4_Ne19_to_p_Na22)*Y(Ne19)*state.rho - screened_rates(k_He4_Ne20_to_C12_C12)*Y(Ne20)*state.rho - screened_rates(k_He4_Ne20_to_Mg24)*Y(Ne20)*state.rho - screened_rates(k_He4_Ne20_to_n_Mg23)*Y(Ne20)*state.rho - screened_rates(k_He4_Ne20_to_p_Na23)*Y(Ne20)*state.rho - screened_rates(k_He4_Ne21_to_Mg25)*Y(Ne21)*state.rho - screened_rates(k_He4_Ne21_to_n_Mg24)*Y(Ne21)*state.rho - screened_rates(k_He4_Ne22_to_Mg26)*Y(Ne22)*state.rho - screened_rates(k_He4_Ne22_to_n_Mg25)*Y(Ne22)*state.rho - screened_rates(k_He4_Ni56_to_Zn60)*Y(Ni56)*state.rho - screened_rates(k_He4_Ni56_to_n_Zn59)*Y(Ni56)*state.rho - screened_rates(k_He4_Ni56_to_p_Cu59)*Y(Ni56)*state.rho - screened_rates(k_He4_Ni57_to_Zn61)*Y(Ni57)*state.rho - screened_rates(k_He4_Ni57_to_n_Zn60)*Y(Ni57)*state.rho - screened_rates(k_He4_Ni57_to_p_Cu60)*Y(Ni57)*state.rho - screened_rates(k_He4_Ni58_to_Zn62)*Y(Ni58)*state.rho - screened_rates(k_He4_Ni58_to_n_Zn61)*Y(Ni58)*state.rho - screened_rates(k_He4_Ni58_to_p_Cu61)*Y(Ni58)*state.rho - screened_rates(k_He4_Ni59_to_Zn63)*Y(Ni59)*state.rho - screened_rates(k_He4_Ni59_to_n_Zn62)*Y(Ni59)*state.rho - screened_rates(k_He4_Ni59_to_p_Cu62)*Y(Ni59)*state.rho - screened_rates(k_He4_Ni60_to_Zn64)*Y(Ni60)*state.rho - screened_rates(k_He4_Ni60_to_n_Zn63)*Y(Ni60)*state.rho - screened_rates(k_He4_Ni60_to_p_Cu63)*Y(Ni60)*state.rho - screened_rates(k_He4_Ni61_to_Zn65)*Y(Ni61)*state.rho - screened_rates(k_He4_Ni61_to_n_Zn64)*Y(Ni61)*state.rho - screened_rates(k_He4_Ni61_to_p_Cu64)*Y(Ni61)*state.rho - screened_rates(k_He4_Ni62_to_Zn66)*Y(Ni62)*state.rho - screened_rates(k_He4_Ni62_to_n_Zn65)*Y(Ni62)*state.rho - screened_rates(k_He4_Ni62_to_p_Cu65)*Y(Ni62)*state.rho - screened_rates(k_He4_Ni63_to_n_Zn66)*Y(Ni63)*state.rho - screened_rates(k_He4_O14_to_Ne18)*Y(O14)*state.rho - screened_rates(k_He4_O14_to_p_F17)*Y(O14)*state.rho - screened_rates(k_He4_O15_to_Ne19)*Y(O15)*state.rho - screened_rates(k_He4_O15_to_n_Ne18)*Y(O15)*state.rho - screened_rates(k_He4_O15_to_p_F18)*Y(O15)*state.rho - screened_rates(k_He4_O16_to_Ne20)*Y(O16)*state.rho - screened_rates(k_He4_O16_to_n_Ne19)*Y(O16)*state.rho - screened_rates(k_He4_O16_to_p_F19)*Y(O16)*state.rho - screened_rates(k_He4_O17_to_Ne21)*Y(O17)*state.rho - screened_rates(k_He4_O17_to_n_Ne20)*Y(O17)*state.rho - screened_rates(k_He4_O18_to_Ne22)*Y(O18)*state.rho - screened_rates(k_He4_O18_to_n_Ne21)*Y(O18)*state.rho - screened_rates(k_He4_P29_to_Cl33)*Y(P29)*state.rho - screened_rates(k_He4_P29_to_p_S32)*Y(P29)*state.rho - screened_rates(k_He4_P30_to_Cl34)*Y(P30)*state.rho - screened_rates(k_He4_P30_to_n_Cl33)*Y(P30)*state.rho - screened_rates(k_He4_P30_to_p_S33)*Y(P30)*state.rho - screened_rates(k_He4_P31_to_Cl35)*Y(P31)*state.rho - screened_rates(k_He4_P31_to_n_Cl34)*Y(P31)*state.rho - screened_rates(k_He4_P31_to_p_S34)*Y(P31)*state.rho - screened_rates(k_He4_P32_to_Cl36)*Y(P32)*state.rho - screened_rates(k_He4_P32_to_n_Cl35)*Y(P32)*state.rho - screened_rates(k_He4_P32_to_p_S35)*Y(P32)*state.rho - screened_rates(k_He4_P33_to_Cl37)*Y(P33)*state.rho - screened_rates(k_He4_P33_to_n_Cl36)*Y(P33)*state.rho - screened_rates(k_He4_P33_to_p_S36)*Y(P33)*state.rho - screened_rates(k_He4_S32_to_Ar36)*Y(S32)*state.rho - screened_rates(k_He4_S32_to_p_Cl35)*Y(S32)*state.rho - screened_rates(k_He4_S33_to_Ar37)*Y(S33)*state.rho - screened_rates(k_He4_S33_to_n_Ar36)*Y(S33)*state.rho - screened_rates(k_He4_S33_to_p_Cl36)*Y(S33)*state.rho - screened_rates(k_He4_S34_to_Ar38)*Y(S34)*state.rho - screened_rates(k_He4_S34_to_n_Ar37)*Y(S34)*state.rho - screened_rates(k_He4_S34_to_p_Cl37)*Y(S34)*state.rho - screened_rates(k_He4_S35_to_Ar39)*Y(S35)*state.rho - screened_rates(k_He4_S35_to_n_Ar38)*Y(S35)*state.rho - screened_rates(k_He4_S36_to_Ar40)*Y(S36)*state.rho - screened_rates(k_He4_S36_to_n_Ar39)*Y(S36)*state.rho - screened_rates(k_He4_Sc43_to_V47)*Y(Sc43)*state.rho - screened_rates(k_He4_Sc43_to_n_V46)*Y(Sc43)*state.rho - screened_rates(k_He4_Sc43_to_p_Ti46)*Y(Sc43)*state.rho - screened_rates(k_He4_Sc44_to_V48)*Y(Sc44)*state.rho - screened_rates(k_He4_Sc44_to_n_V47)*Y(Sc44)*state.rho - screened_rates(k_He4_Sc44_to_p_Ti47)*Y(Sc44)*state.rho - screened_rates(k_He4_Sc45_to_V49)*Y(Sc45)*state.rho - screened_rates(k_He4_Sc45_to_n_V48)*Y(Sc45)*state.rho - screened_rates(k_He4_Sc45_to_p_Ti48)*Y(Sc45)*state.rho - screened_rates(k_He4_Sc46_to_V50)*Y(Sc46)*state.rho - screened_rates(k_He4_Sc46_to_n_V49)*Y(Sc46)*state.rho - screened_rates(k_He4_Sc46_to_p_Ti49)*Y(Sc46)*state.rho - screened_rates(k_He4_Sc47_to_V51)*Y(Sc47)*state.rho - screened_rates(k_He4_Sc47_to_n_V50)*Y(Sc47)*state.rho - screened_rates(k_He4_Sc47_to_p_Ti50)*Y(Sc47)*state.rho - screened_rates(k_He4_Sc48_to_V52)*Y(Sc48)*state.rho - screened_rates(k_He4_Sc48_to_n_V51)*Y(Sc48)*state.rho - screened_rates(k_He4_Sc48_to_p_Ti51)*Y(Sc48)*state.rho - screened_rates(k_He4_Sc49_to_n_V52)*Y(Sc49)*state.rho - screened_rates(k_He4_Si28_to_C12_Ne20)*Y(Si28)*state.rho - screened_rates(k_He4_Si28_to_O16_O16)*Y(Si28)*state.rho - screened_rates(k_He4_Si28_to_S32)*Y(Si28)*state.rho - screened_rates(k_He4_Si28_to_p_P31)*Y(Si28)*state.rho - screened_rates(k_He4_Si29_to_S33)*Y(Si29)*state.rho - screened_rates(k_He4_Si29_to_n_S32)*Y(Si29)*state.rho - screened_rates(k_He4_Si29_to_p_P32)*Y(Si29)*state.rho - screened_rates(k_He4_Si30_to_S34)*Y(Si30)*state.rho - screened_rates(k_He4_Si30_to_n_S33)*Y(Si30)*state.rho - screened_rates(k_He4_Si30_to_p_P33)*Y(Si30)*state.rho - screened_rates(k_He4_Si31_to_S35)*Y(Si31)*state.rho - screened_rates(k_He4_Si31_to_n_S34)*Y(Si31)*state.rho - screened_rates(k_He4_Si32_to_S36)*Y(Si32)*state.rho - screened_rates(k_He4_Si32_to_n_S35)*Y(Si32)*state.rho - screened_rates(k_He4_Ti44_to_Cr48)*Y(Ti44)*state.rho - screened_rates(k_He4_Ti44_to_p_V47)*Y(Ti44)*state.rho - screened_rates(k_He4_Ti45_to_Cr49)*Y(Ti45)*state.rho - screened_rates(k_He4_Ti45_to_n_Cr48)*Y(Ti45)*state.rho - screened_rates(k_He4_Ti45_to_p_V48)*Y(Ti45)*state.rho - screened_rates(k_He4_Ti46_to_Cr50)*Y(Ti46)*state.rho - screened_rates(k_He4_Ti46_to_n_Cr49)*Y(Ti46)*state.rho - screened_rates(k_He4_Ti46_to_p_V49)*Y(Ti46)*state.rho - screened_rates(k_He4_Ti47_to_Cr51)*Y(Ti47)*state.rho - screened_rates(k_He4_Ti47_to_n_Cr50)*Y(Ti47)*state.rho - screened_rates(k_He4_Ti47_to_p_V50)*Y(Ti47)*state.rho - screened_rates(k_He4_Ti48_to_Cr52)*Y(Ti48)*state.rho - screened_rates(k_He4_Ti48_to_n_Cr51)*Y(Ti48)*state.rho - screened_rates(k_He4_Ti48_to_p_V51)*Y(Ti48)*state.rho - screened_rates(k_He4_Ti49_to_Cr53)*Y(Ti49)*state.rho - screened_rates(k_He4_Ti49_to_n_Cr52)*Y(Ti49)*state.rho - screened_rates(k_He4_Ti49_to_p_V52)*Y(Ti49)*state.rho - screened_rates(k_He4_Ti50_to_Cr54)*Y(Ti50)*state.rho - screened_rates(k_He4_Ti50_to_n_Cr53)*Y(Ti50)*state.rho - screened_rates(k_He4_Ti51_to_n_Cr54)*Y(Ti51)*state.rho - screened_rates(k_He4_V46_to_Mn50)*Y(V46)*state.rho - screened_rates(k_He4_V46_to_p_Cr49)*Y(V46)*state.rho - screened_rates(k_He4_V47_to_Mn51)*Y(V47)*state.rho - screened_rates(k_He4_V47_to_n_Mn50)*Y(V47)*state.rho - screened_rates(k_He4_V47_to_p_Cr50)*Y(V47)*state.rho - screened_rates(k_He4_V48_to_Mn52)*Y(V48)*state.rho - screened_rates(k_He4_V48_to_n_Mn51)*Y(V48)*state.rho - screened_rates(k_He4_V48_to_p_Cr51)*Y(V48)*state.rho - screened_rates(k_He4_V49_to_Mn53)*Y(V49)*state.rho - screened_rates(k_He4_V49_to_n_Mn52)*Y(V49)*state.rho - screened_rates(k_He4_V49_to_p_Cr52)*Y(V49)*state.rho - screened_rates(k_He4_V50_to_Mn54)*Y(V50)*state.rho - screened_rates(k_He4_V50_to_n_Mn53)*Y(V50)*state.rho - screened_rates(k_He4_V50_to_p_Cr53)*Y(V50)*state.rho - screened_rates(k_He4_V51_to_Mn55)*Y(V51)*state.rho - screened_rates(k_He4_V51_to_n_Mn54)*Y(V51)*state.rho - screened_rates(k_He4_V51_to_p_Cr54)*Y(V51)*state.rho - screened_rates(k_He4_V52_to_n_Mn55)*Y(V52)*state.rho - screened_rates(k_He4_Zn59_to_Ge63)*Y(Zn59)*state.rho - screened_rates(k_He4_Zn59_to_p_Ga62)*Y(Zn59)*state.rho - screened_rates(k_He4_Zn60_to_Ge64)*Y(Zn60)*state.rho - screened_rates(k_He4_Zn60_to_n_Ge63)*Y(Zn60)*state.rho - screened_rates(k_He4_Zn60_to_p_Ga63)*Y(Zn60)*state.rho - screened_rates(k_He4_Zn61_to_n_Ge64)*Y(Zn61)*state.rho - screened_rates(k_He4_Zn61_to_p_Ga64)*Y(Zn61)*state.rho - screened_rates(k_He4_d_to_Li6)*Y(H2)*state.rho - screened_rates(k_He4_to_d_d) - screened_rates(k_He4_to_n_He3) - 2.0*screened_rates(k_d_He4_He4_to_p_Be9)*Y(He4)*Y(H2)*amrex::Math::powi<2>(state.rho) - 2.0*screened_rates(k_n_He4_He4_to_Be9)*Y(He4)*Y(N)*amrex::Math::powi<2>(state.rho) - 2.0*screened_rates(k_n_He4_He4_to_d_Li7)*Y(He4)*Y(N)*amrex::Math::powi<2>(state.rho) - 2.0*screened_rates(k_n_p_He4_He4_to_He3_Li7)*Y(He4)*Y(N)*Y(H1)*amrex::Math::powi<3>(state.rho) - 2.0*screened_rates(k_n_p_He4_He4_to_p_Be9)*Y(He4)*Y(N)*Y(H1)*amrex::Math::powi<3>(state.rho) - screened_rates(k_n_p_He4_to_Li6)*Y(N)*Y(H1)*amrex::Math::powi<2>(state.rho) - 2.0*screened_rates(k_p_He4_He4_to_d_Be7)*Y(He4)*Y(H1)*amrex::Math::powi<2>(state.rho) - 2.0*screened_rates(k_p_He4_He4_to_n_B8)*Y(He4)*Y(H1)*amrex::Math::powi<2>(state.rho) - screened_rates(k_p_He4_to_d_He3)*Y(H1)*state.rho - 1.0*screened_rates(k_p_p_He4_He4_to_He3_Be7)*Y(He4)*amrex::Math::powi<2>(Y(H1))*amrex::Math::powi<3>(state.rho) - 0.5*screened_rates(k_p_p_He4_to_He3_He3)*amrex::Math::powi<2>(Y(H1))*amrex::Math::powi<2>(state.rho);
    jac.set(He4, He4, scratch);

    scratch = -screened_rates(k_He4_Li6_to_B10)*Y(He4)*state.rho - screened_rates(k_He4_Li6_to_p_Be9)*Y(He4)*state.rho + screened_rates(k_Li6_to_He4_d) + screened_rates(k_Li6_to_n_p_He4) + screened_rates(k_p_Li6_to_He4_He3)*Y(H1)*state.rho;
    jac.set(He4, Li6, scratch);

    scratch = 2.0*screened_rates(k_He3_Li7_to_n_p_He4_He4)*Y(He3)*state.rho - screened_rates(k_He4_Li7_to_B11)*Y(He4)*state.rho - screened_rates(k_He4_Li7_to_n_B10)*Y(He4)*state.rho + 2.0*screened_rates(k_d_Li7_to_n_He4_He4)*Y(H2)*state.rho + 2.0*screened_rates(k_p_Li7_to_He4_He4)*Y(H1)*state.rho;
    jac.set(He4, Li7, scratch);

    scratch = screened_rates(k_Be7_to_He4_He3) + 2.0*screened_rates(k_He3_Be7_to_p_p_He4_He4)*Y(He3)*state.rho - screened_rates(k_He4_Be7_to_p_B10)*Y(He4)*state.rho + 2.0*screened_rates(k_d_Be7_to_p_He4_He4)*Y(H2)*state.rho + 2.0*screened_rates(k_n_Be7_to_He4_He4)*Y(N)*state.rho;
    jac.set(He4, Be7, scratch);

    scratch = 2.0*screened_rates(k_Be9_to_n_He4_He4) - screened_rates(k_He4_Be9_to_n_C12)*Y(He4)*state.rho + screened_rates(k_p_Be9_to_He4_Li6)*Y(H1)*state.rho + 2.0*screened_rates(k_p_Be9_to_d_He4_He4)*Y(H1)*state.rho + 2.0*screened_rates(k_p_Be9_to_n_p_He4_He4)*Y(H1)*state.rho;
    jac.set(He4, Be9, scratch);

    scratch = 2.0*screened_rates(k_B8_to_He4_He4_weak_wc12) + 2.0*screened_rates(k_n_B8_to_p_He4_He4)*Y(N)*state.rho;
    jac.set(He4, B8, scratch);

    scratch = screened_rates(k_B10_to_He4_Li6) - screened_rates(k_He4_B10_to_n_N13)*Y(He4)*state.rho - screened_rates(k_He4_B10_to_p_C13)*Y(He4)*state.rho + screened_rates(k_n_B10_to_He4_Li7)*Y(N)*state.rho + screened_rates(k_p_B10_to_He4_Be7)*Y(H1)*state.rho;
    jac.set(He4, B10, scratch);

    scratch = screened_rates(k_B11_to_He4_Li7) - screened_rates(k_He4_B11_to_n_N14)*Y(He4)*state.rho - screened_rates(k_He4_B11_to_p_C14)*Y(He4)*state.rho + 3.0*screened_rates(k_p_B11_to_He4_He4_He4)*Y(H1)*state.rho;
    jac.set(He4, B11, scratch);

    scratch = 1.0*screened_rates(k_C12_C12_to_He4_Ne20)*Y(C12)*state.rho + screened_rates(k_C12_Ne20_to_He4_Si28)*Y(Ne20)*state.rho + screened_rates(k_C12_O16_to_He4_Mg24)*Y(O16)*state.rho + 3.0*screened_rates(k_C12_to_He4_He4_He4) - screened_rates(k_He4_C12_to_O16)*Y(He4)*state.rho - screened_rates(k_He4_C12_to_n_O15)*Y(He4)*state.rho - screened_rates(k_He4_C12_to_p_N15)*Y(He4)*state.rho + screened_rates(k_n_C12_to_He4_Be9)*Y(N)*state.rho;
    jac.set(He4, C12, scratch);

    scratch = -screened_rates(k_He4_C13_to_n_O16)*Y(He4)*state.rho + screened_rates(k_p_C13_to_He4_B10)*Y(H1)*state.rho;
    jac.set(He4, C13, scratch);

    scratch = -screened_rates(k_He4_C14_to_O18)*Y(He4)*state.rho - screened_rates(k_He4_C14_to_n_O17)*Y(He4)*state.rho + screened_rates(k_p_C14_to_He4_B11)*Y(H1)*state.rho;
    jac.set(He4, C14, scratch);

    scratch = -screened_rates(k_He4_N13_to_p_O16)*Y(He4)*state.rho + screened_rates(k_n_N13_to_He4_B10)*Y(N)*state.rho;
    jac.set(He4, N13, scratch);

    scratch = -screened_rates(k_He4_N14_to_F18)*Y(He4)*state.rho - screened_rates(k_He4_N14_to_n_F17)*Y(He4)*state.rho - screened_rates(k_He4_N14_to_p_O17)*Y(He4)*state.rho + screened_rates(k_n_N14_to_He4_B11)*Y(N)*state.rho;
    jac.set(He4, N14, scratch);

    scratch = -screened_rates(k_He4_N15_to_F19)*Y(He4)*state.rho - screened_rates(k_He4_N15_to_n_F18)*Y(He4)*state.rho - screened_rates(k_He4_N15_to_p_O18)*Y(He4)*state.rho + screened_rates(k_p_N15_to_He4_C12)*Y(H1)*state.rho;
    jac.set(He4, N15, scratch);

    scratch = -screened_rates(k_He4_O14_to_Ne18)*Y(He4)*state.rho - screened_rates(k_He4_O14_to_p_F17)*Y(He4)*state.rho;
    jac.set(He4, O14, scratch);

    scratch = -screened_rates(k_He4_O15_to_Ne19)*Y(He4)*state.rho - screened_rates(k_He4_O15_to_n_Ne18)*Y(He4)*state.rho - screened_rates(k_He4_O15_to_p_F18)*Y(He4)*state.rho + screened_rates(k_n_O15_to_He4_C12)*Y(N)*state.rho;
    jac.set(He4, O15, scratch);

    scratch = screened_rates(k_C12_O16_to_He4_Mg24)*Y(C12)*state.rho - screened_rates(k_He4_O16_to_Ne20)*Y(He4)*state.rho - screened_rates(k_He4_O16_to_n_Ne19)*Y(He4)*state.rho - screened_rates(k_He4_O16_to_p_F19)*Y(He4)*state.rho + 1.0*screened_rates(k_O16_O16_to_He4_Si28)*Y(O16)*state.rho + screened_rates(k_O16_to_He4_C12) + screened_rates(k_n_O16_to_He4_C13)*Y(N)*state.rho + screened_rates(k_p_O16_to_He4_N13)*Y(H1)*state.rho;
    jac.set(He4, O16, scratch);

    scratch = -screened_rates(k_He4_O17_to_Ne21)*Y(He4)*state.rho - screened_rates(k_He4_O17_to_n_Ne20)*Y(He4)*state.rho + screened_rates(k_n_O17_to_He4_C14)*Y(N)*state.rho + screened_rates(k_p_O17_to_He4_N14)*Y(H1)*state.rho;
    jac.set(He4, O17, scratch);

    scratch = -screened_rates(k_He4_O18_to_Ne22)*Y(He4)*state.rho - screened_rates(k_He4_O18_to_n_Ne21)*Y(He4)*state.rho + screened_rates(k_O18_to_He4_C14) + screened_rates(k_p_O18_to_He4_N15)*Y(H1)*state.rho;
    jac.set(He4, O18, scratch);

    scratch = -screened_rates(k_He4_F17_to_Na21)*Y(He4)*state.rho - screened_rates(k_He4_F17_to_p_Ne20)*Y(He4)*state.rho + screened_rates(k_n_F17_to_He4_N14)*Y(N)*state.rho + screened_rates(k_p_F17_to_He4_O14)*Y(H1)*state.rho;
    jac.set(He4, F17, scratch);

    scratch = screened_rates(k_F18_to_He4_N14) - screened_rates(k_He4_F18_to_Na22)*Y(He4)*state.rho - screened_rates(k_He4_F18_to_n_Na21)*Y(He4)*state.rho - screened_rates(k_He4_F18_to_p_Ne21)*Y(He4)*state.rho + screened_rates(k_n_F18_to_He4_N15)*Y(N)*state.rho + screened_rates(k_p_F18_to_He4_O15)*Y(H1)*state.rho;
    jac.set(He4, F18, scratch);

    scratch = screened_rates(k_F19_to_He4_N15) - screened_rates(k_He4_F19_to_Na23)*Y(He4)*state.rho - screened_rates(k_He4_F19_to_n_Na22)*Y(He4)*state.rho - screened_rates(k_He4_F19_to_p_Ne22)*Y(He4)*state.rho + screened_rates(k_p_F19_to_He4_O16)*Y(H1)*state.rho;
    jac.set(He4, F19, scratch);

    scratch = -screened_rates(k_He4_Ne18_to_p_Na21)*Y(He4)*state.rho + screened_rates(k_Ne18_to_He4_O14) + screened_rates(k_n_Ne18_to_He4_O15)*Y(N)*state.rho;
    jac.set(He4, Ne18, scratch);

    scratch = -screened_rates(k_He4_Ne19_to_Mg23)*Y(He4)*state.rho - screened_rates(k_He4_Ne19_to_p_Na22)*Y(He4)*state.rho + screened_rates(k_Ne19_to_He4_O15) + screened_rates(k_n_Ne19_to_He4_O16)*Y(N)*state.rho;
    jac.set(He4, Ne19, scratch);

    scratch = screened_rates(k_C12_Ne20_to_He4_Si28)*Y(C12)*state.rho - screened_rates(k_He4_Ne20_to_C12_C12)*Y(He4)*state.rho - screened_rates(k_He4_Ne20_to_Mg24)*Y(He4)*state.rho - screened_rates(k_He4_Ne20_to_n_Mg23)*Y(He4)*state.rho - screened_rates(k_He4_Ne20_to_p_Na23)*Y(He4)*state.rho + screened_rates(k_Ne20_to_He4_O16) + screened_rates(k_n_Ne20_to_He4_O17)*Y(N)*state.rho + screened_rates(k_p_Ne20_to_He4_F17)*Y(H1)*state.rho;
    jac.set(He4, Ne20, scratch);

    scratch = -screened_rates(k_He4_Ne21_to_Mg25)*Y(He4)*state.rho - screened_rates(k_He4_Ne21_to_n_Mg24)*Y(He4)*state.rho + screened_rates(k_Ne21_to_He4_O17) + screened_rates(k_n_Ne21_to_He4_O18)*Y(N)*state.rho + screened_rates(k_p_Ne21_to_He4_F18)*Y(H1)*state.rho;
    jac.set(He4, Ne21, scratch);

    scratch = -screened_rates(k_He4_Ne22_to_Mg26)*Y(He4)*state.rho - screened_rates(k_He4_Ne22_to_n_Mg25)*Y(He4)*state.rho + screened_rates(k_Ne22_to_He4_O18) + screened_rates(k_p_Ne22_to_He4_F19)*Y(H1)*state.rho;
    jac.set(He4, Ne22, scratch);

    scratch = -screened_rates(k_He4_Na21_to_Al25)*Y(He4)*state.rho - screened_rates(k_He4_Na21_to_p_Mg24)*Y(He4)*state.rho + screened_rates(k_Na21_to_He4_F17) + screened_rates(k_n_Na21_to_He4_F18)*Y(N)*state.rho + screened_rates(k_p_Na21_to_He4_Ne18)*Y(H1)*state.rho;
    jac.set(He4, Na21, scratch);

    scratch = -screened_rates(k_He4_Na22_to_Al26)*Y(He4)*state.rho - screened_rates(k_He4_Na22_to_n_Al25)*Y(He4)*state.rho - screened_rates(k_He4_Na22_to_p_Mg25)*Y(He4)*state.rho + screened_rates(k_Na22_to_He4_F18) + screened_rates(k_n_Na22_to_He4_F19)*Y(N)*state.rho + screened_rates(k_p_Na22_to_He4_Ne19)*Y(H1)*state.rho;
    jac.set(He4, Na22, scratch);

    scratch = -screened_rates(k_He4_Na23_to_Al27)*Y(He4)*state.rho - screened_rates(k_He4_Na23_to_n_Al26)*Y(He4)*state.rho - screened_rates(k_He4_Na23_to_p_Mg26)*Y(He4)*state.rho + screened_rates(k_Na23_to_He4_F19) + screened_rates(k_p_Na23_to_He4_Ne20)*Y(H1)*state.rho;
    jac.set(He4, Na23, scratch);

    scratch = -screened_rates(k_He4_Mg23_to_p_Al26)*Y(He4)*state.rho + screened_rates(k_Mg23_to_He4_Ne19) + screened_rates(k_n_Mg23_to_He4_Ne20)*Y(N)*state.rho;
    jac.set(He4, Mg23, scratch);

    scratch = -screened_rates(k_He4_Mg24_to_C12_O16)*Y(He4)*state.rho - screened_rates(k_He4_Mg24_to_Si28)*Y(He4)*state.rho - screened_rates(k_He4_Mg24_to_p_Al27)*Y(He4)*state.rho + screened_rates(k_Mg24_to_He4_Ne20) + screened_rates(k_n_Mg24_to_He4_Ne21)*Y(N)*state.rho + screened_rates(k_p_Mg24_to_He4_Na21)*Y(H1)*state.rho;
    jac.set(He4, Mg24, scratch);

    scratch = -screened_rates(k_He4_Mg25_to_Si29)*Y(He4)*state.rho - screened_rates(k_He4_Mg25_to_n_Si28)*Y(He4)*state.rho + screened_rates(k_Mg25_to_He4_Ne21) + screened_rates(k_n_Mg25_to_He4_Ne22)*Y(N)*state.rho + screened_rates(k_p_Mg25_to_He4_Na22)*Y(H1)*state.rho;
    jac.set(He4, Mg25, scratch);

    scratch = -screened_rates(k_He4_Mg26_to_Si30)*Y(He4)*state.rho - screened_rates(k_He4_Mg26_to_n_Si29)*Y(He4)*state.rho + screened_rates(k_Mg26_to_He4_Ne22) + screened_rates(k_p_Mg26_to_He4_Na23)*Y(H1)*state.rho;
    jac.set(He4, Mg26, scratch);

    scratch = screened_rates(k_Al25_to_He4_Na21) - screened_rates(k_He4_Al25_to_P29)*Y(He4)*state.rho - screened_rates(k_He4_Al25_to_p_Si28)*Y(He4)*state.rho + screened_rates(k_n_Al25_to_He4_Na22)*Y(N)*state.rho;
    jac.set(He4, Al25, scratch);

    scratch = screened_rates(k_Al26_to_He4_Na22) - screened_rates(k_He4_Al26_to_P30)*Y(He4)*state.rho - screened_rates(k_He4_Al26_to_n_P29)*Y(He4)*state.rho - screened_rates(k_He4_Al26_to_p_Si29)*Y(He4)*state.rho + screened_rates(k_n_Al26_to_He4_Na23)*Y(N)*state.rho + screened_rates(k_p_Al26_to_He4_Mg23)*Y(H1)*state.rho;
    jac.set(He4, Al26, scratch);

    scratch = screened_rates(k_Al27_to_He4_Na23) - screened_rates(k_He4_Al27_to_P31)*Y(He4)*state.rho - screened_rates(k_He4_Al27_to_n_P30)*Y(He4)*state.rho - screened_rates(k_He4_Al27_to_p_Si30)*Y(He4)*state.rho + screened_rates(k_p_Al27_to_He4_Mg24)*Y(H1)*state.rho;
    jac.set(He4, Al27, scratch);

    scratch = -screened_rates(k_He4_Si28_to_C12_Ne20)*Y(He4)*state.rho - screened_rates(k_He4_Si28_to_O16_O16)*Y(He4)*state.rho - screened_rates(k_He4_Si28_to_S32)*Y(He4)*state.rho - screened_rates(k_He4_Si28_to_p_P31)*Y(He4)*state.rho + screened_rates(k_Si28_to_He4_Mg24) + screened_rates(k_n_Si28_to_He4_Mg25)*Y(N)*state.rho + screened_rates(k_p_Si28_to_He4_Al25)*Y(H1)*state.rho;
    jac.set(He4, Si28, scratch);

    scratch = -screened_rates(k_He4_Si29_to_S33)*Y(He4)*state.rho - screened_rates(k_He4_Si29_to_n_S32)*Y(He4)*state.rho - screened_rates(k_He4_Si29_to_p_P32)*Y(He4)*state.rho + screened_rates(k_Si29_to_He4_Mg25) + screened_rates(k_n_Si29_to_He4_Mg26)*Y(N)*state.rho + screened_rates(k_p_Si29_to_He4_Al26)*Y(H1)*state.rho;
    jac.set(He4, Si29, scratch);

    scratch = -screened_rates(k_He4_Si30_to_S34)*Y(He4)*state.rho - screened_rates(k_He4_Si30_to_n_S33)*Y(He4)*state.rho - screened_rates(k_He4_Si30_to_p_P33)*Y(He4)*state.rho + screened_rates(k_Si30_to_He4_Mg26) + screened_rates(k_p_Si30_to_He4_Al27)*Y(H1)*state.rho;
    jac.set(He4, Si30, scratch);

    scratch = -screened_rates(k_He4_Si31_to_S35)*Y(He4)*state.rho - screened_rates(k_He4_Si31_to_n_S34)*Y(He4)*state.rho;
    jac.set(He4, Si31, scratch);

    scratch = -screened_rates(k_He4_Si32_to_S36)*Y(He4)*state.rho - screened_rates(k_He4_Si32_to_n_S35)*Y(He4)*state.rho;
    jac.set(He4, Si32, scratch);

    scratch = -screened_rates(k_He4_P29_to_Cl33)*Y(He4)*state.rho - screened_rates(k_He4_P29_to_p_S32)*Y(He4)*state.rho + screened_rates(k_P29_to_He4_Al25) + screened_rates(k_n_P29_to_He4_Al26)*Y(N)*state.rho;
    jac.set(He4, P29, scratch);

    scratch = -screened_rates(k_He4_P30_to_Cl34)*Y(He4)*state.rho - screened_rates(k_He4_P30_to_n_Cl33)*Y(He4)*state.rho - screened_rates(k_He4_P30_to_p_S33)*Y(He4)*state.rho + screened_rates(k_P30_to_He4_Al26) + screened_rates(k_n_P30_to_He4_Al27)*Y(N)*state.rho;
    jac.set(He4, P30, scratch);

    scratch = -screened_rates(k_He4_P31_to_Cl35)*Y(He4)*state.rho - screened_rates(k_He4_P31_to_n_Cl34)*Y(He4)*state.rho - screened_rates(k_He4_P31_to_p_S34)*Y(He4)*state.rho + screened_rates(k_P31_to_He4_Al27) + screened_rates(k_p_P31_to_He4_Si28)*Y(H1)*state.rho;
    jac.set(He4, P31, scratch);

    scratch = -screened_rates(k_He4_P32_to_Cl36)*Y(He4)*state.rho - screened_rates(k_He4_P32_to_n_Cl35)*Y(He4)*state.rho - screened_rates(k_He4_P32_to_p_S35)*Y(He4)*state.rho + screened_rates(k_p_P32_to_He4_Si29)*Y(H1)*state.rho;
    jac.set(He4, P32, scratch);

    scratch = -screened_rates(k_He4_P33_to_Cl37)*Y(He4)*state.rho - screened_rates(k_He4_P33_to_n_Cl36)*Y(He4)*state.rho - screened_rates(k_He4_P33_to_p_S36)*Y(He4)*state.rho + screened_rates(k_p_P33_to_He4_Si30)*Y(H1)*state.rho;
    jac.set(He4, P33, scratch);

    scratch = -screened_rates(k_He4_S32_to_Ar36)*Y(He4)*state.rho - screened_rates(k_He4_S32_to_p_Cl35)*Y(He4)*state.rho + screened_rates(k_S32_to_He4_Si28) + screened_rates(k_n_S32_to_He4_Si29)*Y(N)*state.rho + screened_rates(k_p_S32_to_He4_P29)*Y(H1)*state.rho;
    jac.set(He4, S32, scratch);

    scratch = -screened_rates(k_He4_S33_to_Ar37)*Y(He4)*state.rho - screened_rates(k_He4_S33_to_n_Ar36)*Y(He4)*state.rho - screened_rates(k_He4_S33_to_p_Cl36)*Y(He4)*state.rho + screened_rates(k_S33_to_He4_Si29) + screened_rates(k_n_S33_to_He4_Si30)*Y(N)*state.rho + screened_rates(k_p_S33_to_He4_P30)*Y(H1)*state.rho;
    jac.set(He4, S33, scratc