import numpy as np
#import matplotlib.pyplot as plt
import scipy as sp
import itertools as itools
from math import factorial
from sympy.utilities.iterables import multiset_permutations
# functions to deal with ordinary (not hardcore) bosons in second quantization
# the bases are constructed according to Ponomarev ordering
# see Raventos et al., J. Phys. B: At. Mol. Opt. Phys. 50 (2017) 113001 
 


def PascalTriangle(Npart, Ns):
  pas=np.zeros((Npart+1, Ns+1), dtype=int)
  pas[0,:]=1
  for i in range(1,Npart+1):
    for j in range(1,Ns+1):
      pas[i,j]=np.sum(pas[:(i+1), j-1])
  return pas

# occupation number vector to configuration (stored as indices of occupied orbitals)
def OccToConf(occ):
 Ns=len(occ)
 Npart=sum(occ)
 conf=np.zeros(Npart, dtype=int)
 k=Npart-1
 for i in range(Ns):
   for j in range(occ[i]):
     conf[k]=i
     k=k-1
 return conf

# configuration (stored as indices of occupied orbitals) to occupation number vector
def ConfToOcc(conf, Ns):
 occ=np.zeros(Ns, dtype=int)
 for i in conf:
  occ[i]+=1
 return occ

# configuration to index in the basis (no angular momentum conservation)
def ConfToInd(conf,Ns, pas):
  Npart=len(conf)
  ind=0
  for i in range(Npart):
    i2=Ns-(conf[i]+1)
    ind+=pas[i+1, i2]
  return ind


# configuration to index in the basis (no angular momentum conservation)
def IndToConf(ind, Npart, Ns,pas):
  ind2=ind+1
  i2=Npart #index for the configuration states,
  i=Npart
  m=0
  conf=np.zeros(Npart, dtype=int)
  while((m<Ns)and(i>=0)):
    if(ind2-pas[i,Ns-m-1]>0):
      ind2=ind2-pas[i,Ns-m-1]
      conf[i2-1]=m # appedning a state to configuration starting from the last one
      i2-=1
      m=0
      i-=1
    else:
      m+=1
  return conf

# matrix of transformation from hardcore bosons (two orbitals per atom, used in the NonHermitianAngularMomentumED.py)  to ordinary bosons
def TwoOrbHCToOrdinaryBosonsU(basisHC, pas):
  Nsite=len(basisHC[0])
  Ns=2*Nsite
  ind1s=[]
  ind2s=[]
  vals=[]
  for iHC, occHC in enumerate(basisHC):
    confOB=[]
    for i in range(Nsite-1, -1,-1):
      o=occHC[i]
      if(o>0):
        confOB.append(2*i+o-1)
    ind=ConfToInd(confOB,Ns, pas)
    #print(occHC, confOB, ind, iHC)
    ind1s.append(ind)
    ind2s.append(iHC)
    vals.append(1)
  Npart=len(confOB)
  NconfOB=pas[Npart, Ns]
  NconfHC=len(basisHC)
  U=sp.sparse.coo_matrix((vals, (ind1s, ind2s)), shape=[NconfOB,NconfHC])
  U=U.tocsr() 
  return U

# matrix of basis transformation for ordinary bosons in the case when the single-particle wavefunctins (orbitals) change to neworbs      
def ChangeBasis(neworbs,Npart,Ns,  pas):
  Nconf=pas[Npart, Ns]
  U=np.zeros((Nconf, Nconf), dtype=complex)
  #perms=list(itools.permutations(range(Npart)))
  #print(perms)
  for iconf1 in range(Nconf):
    conf1=IndToConf(iconf1, Npart, Ns,pas)
    occ1=ConfToOcc(conf1, Ns)
    pref1=1
    for io, o in enumerate(occ1):
      pref1/=np.sqrt(factorial(o))
    for iconf2 in range(Nconf):
      conf2=IndToConf(iconf2, Npart, Ns,pas)
      occ2=ConfToOcc(conf2, Ns)
      pref2=1
      for io, o in enumerate(occ2):
        pref2*=np.sqrt(factorial(o))
      elem=0
      for perm in multiset_permutations(conf2):
        term=1
        for ip,p in enumerate(perm):
          ind1=conf1[ip]
          ind2=p
          term*=np.conj(neworbs[ind2, ind1])
        elem+=term
        #print(term)
      U[iconf1, iconf2]=pref1*pref2*elem
  return U


# create basis divided into 1D momentum sectors
def BasisWith1DMomentum(Npart, Ns, pas, ks, Kmax):
  Nconf=pas[Npart, Ns]
  mombasis=[]
  for i in range(Kmax):
    mombasis.append([])
  numlist=[]
  for iconf in range(Nconf): 
    conf=IndToConf(iconf, Npart, Ns,pas)
    K=0
    for c in conf:
      K+=ks[c]
    K=int(K % Kmax)
    mombasis[K].append(conf)
    numlist.append(len(mombasis[K])-1)
  return mombasis, numlist

# transformation matrix from a basis without momentum (e.g. site basis) to a basis with momentum conservation, defined by orbitals neworbs
def ChangeBasisWith1DMomentum(neworbs,K, mombasis, Npart,Ns,  pas):
  Nconf=pas[Npart, Ns]
  NconfK=len(mombasis[K])
  U=np.zeros((NconfK, Nconf), dtype=complex)
  #perms=list(itools.permutations(range(Npart)))
  #print(perms)
  for iconfK in range(NconfK):
    conf1=mombasis[K][iconfK]
    #conf1=IndToConf(iconf1, Npart, Ns,pas)
    occ1=ConfToOcc(conf1, Ns)
    pref1=1
    for io, o in enumerate(occ1):
      pref1/=np.sqrt(factorial(o))
    for iconf2 in range(Nconf):
      conf2=IndToConf(iconf2, Npart, Ns,pas)
      occ2=ConfToOcc(conf2, Ns)
      pref2=1
      for io, o in enumerate(occ2):
        pref2*=np.sqrt(factorial(o))
      elem=0
      for perm in multiset_permutations(conf2):
        term=1
        for ip,p in enumerate(perm):
          ind1=conf1[ip]
          ind2=p
          term*=np.conj(neworbs[ind2, ind1])
        elem+=term
        #print(term)
      U[iconfK, iconf2]=pref1*pref2*elem
  return U
