import numpy as np
from scipy.sparse.linalg import eigsh, eigs
import matplotlib.pyplot as plt
import sys
import os
import inspect
import math
import json

# import ED functions
currentdir = os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))
parentdir = os.path.dirname(currentdir)
sys.path.insert(0, parentdir) 
import NonHermitianAngularMomentumED as ed
import OrdinaryBosons as ob


##################################################################################################
##  Start calculations
##################################################################################################

# ---------------- system parameters ---------
lambda0=1 #atomic transition wavelength
k0=2*np.pi/lambda0 # atomic transition wavenumber
d0=0.5  # dimensionless lattice constant
d=d0*lambda0
hermitian=True # if diagonalize the Hermitian part of the Hamiltonian (keep True otherwise the orbitals will be nonorthogonal) 
D0=0.6 # system size 
D=D0*d
simplify=False #use simplified model instead of the full model
Vquad0=0 # rescaled confining potential strength (not useful for D0=0.6)
Npart=2 # number of particles (note: some parts of the code work for two particles only)
Lmax=6 # Lmax-fold rotational symmetry
rtrunc=-1 #truncation radius of interaction (-1 if no truncation)
B0=12 # rescaled magnetic field
ComputeOrbitalDecomposition=True # if compute the decomposition of ED eigenstates in the orbital basis
w0=1 # waist parameter of the LG modes
Lgs=2 # will compute emission from this subspace

B=B0/d**3/k0**3 # non-rescaled magnetic field
Vquad=Vquad0/d**5/k0**3 # non-rescaled harmonic potential

latvec=[[d*np.sqrt(3)/2.,-d*0.5], [d*np.sqrt(3)/2.,d*0.5] ] #lattice vectors
V=0
q=2

orb=[[1./3.,1./3.], [2./3.,2./3.]] # positions of atoms in the unit cell (in the basis of lattice vectors)

output_dict={} #dictionary of results to be saved in the output file 
output_dict["params"]={"d0": d0, "D0":D0, "Vquad0":Vquad0, "B0":B0, "Npart":Npart, "w0":w0, "hermitian":hermitian, "Lgs":Lgs}

# get site positions and plot them
xs,ys=ed.GetLatticePointsSegment(D0, latvec, orb,d)
Nseg=len(xs)
Nsite=Nseg*Lmax
Ns=2*Nsite
xall=[]
yall=[]
for i in range(Lmax):
  xres,yres=ed.RotateAll(xs,ys, i, Lmax)
  plt.scatter(xres,yres)
  xall.extend(xres)
  yall.extend(yres)
pos=np.transpose(np.array([xall,yall]))

ax=plt.gca()
ax.set_aspect("equal")

# get interactions defining the Hamiltonian
if(simplify):
  interactions=ed.RealDipolarInteractionFiniteCircularSimplifiedTransform(pos, B, Vquad,k0, rtrunc, Lmax)
else:
  interactions=ed.RealDipolarInteractionFiniteCircularFullTransform(pos, B, Vquad,k0, rtrunc, Lmax,hermitian)


################################################################################
##  Find single-particle eigenstates
################################################################################

basisSP=ed.GenerateBasis(1, Nsite)
mombasisSP = ed.FindAngularMomentumEigenstates(basisSP, 1, Lmax)


evalsSPL=[]
evecsSPL=[]
evecssiteSPL=[]
neval=6
evalsSP_all=[]
LsSP_all=[]
numbersSP_all=[]
evecssiteSP2=np.zeros((Ns,Ns), dtype=complex)
k=0
sp_dicts=[]
for L in range(Lmax):
  H=ed.GenerateCRSHamiltonianWithAngularMomentum(L,interactions,basisSP,Lmax, mombasisSP)
  Hdense=H.todense()
  eval, evec=np.linalg.eig(Hdense)
  arg=np.argsort(np.real(eval))
  eval=eval[arg]
  evec=evec[:,arg]
  evec=np.array(evec)
  evalsSP_all.extend(eval)
  LsSP_all.extend(np.ones(len(eval))*L)
  numbersSP_all.extend(np.arange(len(eval)))
  evalsSPL.append(eval)
  evecsSPL.append(evec)
  evecsite=ed.ConvertEigenstates(evec, basisSP, mombasisSP, L,Lmax)
  evecssiteSPL.append(evecsite)
  for i in range(len(eval)):
    evecssiteSP2[:,k]=evecsite[:,i]
    k+=1
  Vsite=ed.OriginalPhases(evecsite, basisSP, Lmax, Nsite)
  sp_dicts.append({"L":L, "E":list(np.real(eval))})

plt.figure()
plt.scatter(LsSP_all, np.real(evalsSP_all))

output_dict["single particle spectrum"]=sp_dicts





################################################################################
##  Find many-particle eigenstates
################################################################################

basis=ed.GenerateBasis(Npart, Nsite)
mombasis = ed.FindAngularMomentumEigenstates(basis, Npart, Lmax)

evalsL=[]
evecsL=[]
evecssiteL=[]
neval=10
evals_all=[]
Ls_all=[]
mp_dicts=[]
for L in range(Lmax):
  H=ed.GenerateCRSHamiltonianWithAngularMomentum(L,interactions,basis,Lmax, mombasis)
  if (len(mombasis[0][L])<20):
    Hdense=H.todense()
    eval, evec=np.linalg.eig(Hdense)
    arg=np.argsort(np.real(eval))
    eval=eval[arg]
    evec=evec[:,arg]
    evec=np.array(evec)
    evals_all.extend(eval)
    Ls_all.extend(np.ones(len(eval))*L)
    evalsL.append(eval)
    evecsL.append(evec)
    evecsite=ed.ConvertEigenstates(evec, basis, mombasis, L,Lmax)
    evecssiteL.append(evecsite)
  else:
    eval, evec=eigs(H, k=min(len(mombasis[0][L])-1,neval), which = 'SR', return_eigenvectors=True)
    arg=np.argsort(np.real(eval))
    eval=eval[arg]
    evec=evec[:,arg]
    evals_all.extend(eval)
    Ls_all.extend(np.ones(len(eval))*L)
    evalsL.append(eval)
    evecsL.append(evec)
    evecsite=ed.ConvertEigenstates(evec, basis, mombasis, L,Lmax)
    evecssiteL.append(evecsite)
  mp_dicts.append({"L":L, "E":list(np.real(eval))})

output_dict["many particle spectrum"]=mp_dicts
plt.figure()
plt.scatter(Ls_all, np.real(evals_all))


################################################################################
## expectation values of orbital occupations
###############################################################################

# generate the orbital occupation operators n^0_L=b_{0L}^{\dagger}b_{0L}
# uses the same function as to generate Hamiltonian, so the off-diagonal elements
# (in the site basis) are "hoppings" and diagonal ones are "onsite"


orb_hams=[]
orbs=[]
for lorb in range(Lmax):
  gs=evecssiteSPL[lorb][:,0] # the orbital, only taking into account the lowest one per L
  orbs.append(gs)
  orb_int=[]
  orb_hams0=[]
  for i in range(Nsite):
    for j in range(Nsite):
      if(i!=j):
        for dir1 in range(2):
          for dir2 in range(2):
            orb_int.append(["hopping", i, j, dir1,dir2, gs[2*i+dir1]*np.conj(gs[2*j+dir2])])
      else:
        for dir1 in range(2):
          for dir2 in range(2):
            if(dir1==dir2):
              orb_int.append(["onsite", i, dir1, np.abs(gs[2*i+dir1]**2)])
            else:
              orb_int.append(["hopping", i, i, dir1,dir2, gs[2*i+dir1]*np.conj(gs[2*i+dir2])])
  ham0=ed.GenerateSparseHamiltonianNoMomentumCoLex(orb_int,basis, Nsite, Npart)
  orb_hams.append(ham0)


Nstate=evecssiteL[Lgs].shape[1]


#evaluate the expectation value of the orbital occupation operator
occ_dicts=[]
for i in range(Nstate): 
  gs=evecssiteL[Lgs][:,i]
  orb_occ=[]
  for lorb in range(Lmax):
    ham=orb_hams[lorb]
    ham.size
    orb_occ.append( np.conj(gs) @ ham @ gs)
  occ_dicts.append({"state index": i, "occ":list(np.real(orb_occ))})

output_dict["occ"]={"comment:":"Number of particles occupying the lowest single-particle orbital at given angular momentum, from L=0 to L=5", "data":occ_dicts}

###############################################################################
#  Emission spectrum
###############################################################################

def GaussianModeAll(w0, l, pol, pos):
  # evaluate a gaussian mode of any polarization at all sites
  # the output alternates - and + polarization to match the - and + orbitals in the site basis
  Nsite=pos.shape[0]
  field=np.zeros((2*Nsite), dtype=complex)
  for i in range(Nsite):
    field[2*i]=GaussianMode(w0, l, pos[i,:])*pol[0]
    field[2*i+1]=GaussianMode(w0, l, pos[i,:])*pol[1]
  return field

def GaussianMode(w0, l, point):
  # evaluate a p=0 Gaussian mode at a single point in space
  x,y=point
  rho=np.sqrt(x**2+y**2)
  E0=np.sqrt(2/(np.math.factorial(abs(l))*np.pi))/w0
  phi=np.arctan2(y,x)
  return E0*(rho*np.sqrt(2)/w0)**np.abs(l)*np.exp(-rho**2/w0**2)*np.exp(-1j*l*phi)


# generate the mode population operator E^dagger E (the same way as the orbital occupation, only that the role of the orbital wavefunction is played by the mode function)
orb_hams=[[],[]]
lorbs=list(range(-6, 7))
for ipol, pol in enumerate([[1,0],[0,1]]):
  for lorb in lorbs:
    gs=GaussianModeAll(w0, lorb, pol, pos)
    orbs.append(gs)
    orb_int=[]
    orb_hams0=[]
    for i in range(Nsite):
      for j in range(Nsite):
        if(i!=j):
          for dir1 in range(2):
            for dir2 in range(2):
              orb_int.append(["hopping", i, j, dir1,dir2, gs[2*i+dir1]*np.conj(gs[2*j+dir2])])
        else:
          for dir1 in range(2):
            for dir2 in range(2):
              if(dir1==dir2):
                orb_int.append(["onsite", i, dir1, np.abs(gs[2*i+dir1]**2)])
              else:
                orb_int.append(["hopping", i, i, dir1,dir2, gs[2*i+dir1]*np.conj(gs[2*i+dir2])])
    ham0=ed.GenerateSparseHamiltonianNoMomentumCoLex(orb_int,basis, Nsite, Npart)
    orb_hams[ipol].append(ham0)

evecssite_orig=ed.OriginalPhases(evecssiteL[Lgs], basis, Lmax, Nsite) # undo the rotation-symmetry-restoring transformation

#evaluate the expectation value of the mode population operator 
occ_dicts=[]
for i in range(Nstate): 
  gs=evecssite_orig[:,i]
  orb_occ_minus=[]
  orb_occ_plus=[]
  for iorb,lorb in enumerate(lorbs):
    ham=orb_hams[0][iorb]
    orb_occ_minus.append( np.conj(gs) @ ham @ gs)
    ham=orb_hams[1][iorb]
    orb_occ_plus.append( np.conj(gs) @ ham @ gs)
  occ_dicts.append({"state index": i, "occ_minus":list(np.real(orb_occ_minus)),  "occ_plus":list(np.real(orb_occ_plus))})

output_dict["single photon emission"]={"comment:":"Expectation value of the number of photons emitted into a Laguerre-Gauss mode with width w0 and respective angular momentum from ls array", "ls":lorbs, "data":occ_dicts}

################################################################################
## expectation values of two-photon emission and two-photon amplitude
###############################################################################


# pairs of modes for which to calculate two-photon emission
if(Lgs==2):
  lpairs=[
  [0,0],
  [-1,1],
  [-2,2],
  [-3,3],
  [-4,4],
  [-5,5],
  [-6,6],
  [0,6],
  [1,5],
  [-1,-5],
  [0,-6],
  ]
if(Lgs==4): 
  lpairs=[
  [1,1],
  [0,2],
  [-1,3],
  [-2,4],
  [-3,5],
  [-4,6],
  [-4,0],
  #[0,-6],
  ]

# compute the expectation values
Nstate=evecssite_orig.shape[1]
twopart_emission=[]
twopart_amplitude=[]
for i in range(Nstate):
  twopart_emission.append([])
  twopart_amplitude.append([])
for lpair in lpairs:
  l1,l2=lpair
  orb1=GaussianModeAll(w0, l1, [1,0], pos)
  orb2=GaussianModeAll(w0, l2, [1,0], pos)
  phi=ed.TwoParticleAmplitudeVector2part(orb1, orb2, basis) #the vector <0|E_{l_1} E_{l_2}
  for i in range(Nstate):
    gs=evecssite_orig[:,i] 
    amp= np.conj(phi) @ gs #<0|E_{l_1} E_{l_2}|gs>
    twopart_emission[i].append(np.abs(amp)**2)
    twopart_amplitude[i].append(amp)

# add to the dictionary to be saved in the output file
data_dicts=[]
for i in range(Nstate):
  data_dicts.append({"state index": i, "amplitude abs":list(np.abs(twopart_amplitude[i])), "amplitude phase":list(np.angle(np.array(twopart_amplitude[i])/twopart_amplitude[i][0]))})

twoph_dict={}
twoph_dict["comment"]="Two-photon amplitude \langle |E^{\dagger}_{l_1}  E^{\dagger}_{l_2} | \psi_i \rangle where the E's genrate Laguerre-Gauss mode with width w0 and respective angular momenta from lpairs list"
twoph_dict["lpairs"]=lpairs
twoph_dict["data"]=data_dicts

output_dict["two-photon emission"]=twoph_dict


################################################################################
## Map 2-particle eigenvectors to orbital basis
###############################################################################


if (ComputeOrbitalDecomposition):
  pas=ob.PascalTriangle(Npart, Ns) # pascal triangle is used to generate ordinary boson basis
  Nconf=pas[Npart, Ns]
  U=ob.TwoOrbHCToOrdinaryBosonsU(basis, pas) # matrix of transformation of the hardcore bosons in site basis to ordinary bosons in site basis (larger basis, different ordering of vectors)
  mombasisOB,numlistOB=ob.BasisWith1DMomentum(Npart, Ns, pas, LsSP_all, Lmax) # ordinary boson angular momentum basis (orbital occupations with total angular momentum L)
  subspaces=[]
  for L in range(Lmax):
    U3=ob.ChangeBasisWith1DMomentum(evecssiteSP2, L , mombasisOB, Npart,Ns,  pas) #matrix of transformation between ordinary boson site and orbital basis
    U3new=U3 @ U # total matrix of transformation
    evecorb_OB_L = U3new @ evecssiteL[L] # transform eigenvectors into orbital basis
    #write ordinary boson orbital basis in a format [[L1, i1], [L2, i2]] for human-readable output
    basis2=[]
    for conf in mombasisOB[L]:
      confL=[]
      for c in conf:
        confL.append([int(LsSP_all[c]), int(numbersSP_all[c])])
      basis2.append(confL)
    orbbasis_dict={"L":L, "basis": basis2} # add basis to the ouput dictionary
    # add transformed states to the output dictionary
    states=[]
    for i in range(evecssiteL[L].shape[1]):
      states.append({"i":i, "E":np.real(evalsL[L][i]), "wfn real": list(np.real(evecorb_OB_L[:,i])), "wfn imag": list(np.imag(evecorb_OB_L[:,i]))   })
    orbbasis_dict["states"]=states
    subspaces.append(orbbasis_dict)
  output_dict["Fock basis coeffs"]=subspaces

# save the output dictionary
f= open("output_OBC_d0={:.4f}_D0={:.4f}_Npart={}_B0={:.4f}_Vquad0={:.4f}_Lgs{}.json".format(d0, D0,Npart,B0,Vquad0, Lgs), "w")
json.dump(output_dict, f, indent=2)
f.close()

plt.show()
