import numpy as np
import matplotlib.pyplot as plt
import json
import matplotlib as mpl
from string import ascii_lowercase

# ---------------- system parameters ---------
lambda0=1 #atomic transition wavelength
k0=2*np.pi/lambda0 # atomic transition wavenumber
d0=0.3  # dimensionless lattice constant
d=d0*lambda0
hermitian=True # if diagonalize the Hermitian part of the Hamiltonian (keep True otherwise the orbitals will be nonorthogonal) 
D0=0.6 # system size 
D=D0*d
simplify=False #use simplified model instead of the full model
Vquad0=0 # rescaled confining potential strength (not useful for D0=0.6)
Npart=2 # number of particles (note: some parts of the code work for two particles only)
Lmax=6 # Lmax-fold rotational symmetry
rtrunc=-1 #truncation radius of interaction (-1 if no truncation)
B0=12 # rescaled magnetic field
w0=1 # waist parameter of the LG modes
Lgs=2 # will compute emission from this subspace
fsize=16 #font size in the plot

# set plot font
plt.rcParams['font.family'] = 'serif'
plt.rcParams['font.serif'] = ['CMU Serif']
plt.rcParams['mathtext.fontset'] = 'cm'

#load the f's for rescaling
proj=np.loadtxt("modeprojections_OBC_d0={:.4f}_herm{}_simp{}_D0={:.4f}_Npart={}_B0={:.4f}_Vquad0={:.4f}_rtrunc={:.4f}.txt".format(d0, hermitian,simplify, D0,1,B0,Vquad0,rtrunc))
lproj=proj[:,0].astype(int)

# load the single photon emissions
f= open("output_OBC_d0={:.4f}_D0={:.4f}_Npart={}_B0={:.4f}_Vquad0={:.4f}_Lgs{}.json".format(d0, D0,Npart,B0,Vquad0, Lgs))
data_dict=json.load(f)
f.close()
sp=data_dict["single photon emission"]
ls=sp["ls"]
i0=ls.index(0)
Nstate=2 # how many states to plot 

if (ls==list(lproj)): #check if the values of l in both files are the same, otherwise the code will crash
    scale_minus=proj[:,1]

fig, ax=plt.subplots(Nstate, 2, sharex=True,sharey="col",figsize=(6, 4)) #define subplots

################################################################
##  single photon emissions, minus polarization
################################################################

for istate,state in enumerate(sp["data"]):
  if(istate<Nstate):
    ax[istate,1].bar(ls, state["occ_minus"]/scale_minus) # rescaled intensity
    ax[istate,0].bar(ls, state["occ_minus"]) # non-rescaled intensity
    ax[istate,0].set_yscale("log")
    ax[istate,0].set_ylabel(r"$I^{-}_l$", fontsize=fsize)
    ax[istate,1].set_ylabel(r"$\tilde{I}^{-}_l$", fontsize=fsize)
    ax[istate,0].tick_params(axis='both', which='major', labelsize=fsize)
    ax[istate,1].tick_params(axis='both', which='major', labelsize=fsize)
  if(istate==Nstate-1):
    ax[istate,0].set_xlabel(r"$l$", fontsize=fsize)
    ax[istate,1].set_xlabel(r"$l$", fontsize=fsize)


################################################################
##  orbital occupations
################################################################

# get the L corresponding to l
inds=[]
for l in ls:
  L=(1+l)%Lmax
  inds.append(L)


occ=data_dict["occ"]
for istate,state in enumerate(occ["data"]):
  if(istate<Nstate):
    occ0=state["occ"]
    #else:
    occ1=[]
    for i in inds:
      occ1.append(occ0[i])
    ax[istate, 1].bar(ls, occ1,fill=False, edgecolor="r")

################################################################
## set plot parameters, add labels etc
################################################################
ax[0,0].set_xlim(-3.5, 3.5)
ax[0,0].set_ylim(10**(-7), 0)
#OutputSignaturesFigure
plt.tight_layout()
k=0
for i in range(Nstate):
 for j in range(2):
  ax[i,j].text(-0.4, 0.95, "("+ascii_lowercase[k]+")", transform=ax[i,j].transAxes, fontsize=fsize)
  k+=1
plt.savefig("Figure5.pdf")
plt.show()
