import numpy as np
import matplotlib.pyplot as plt
import matplotlib.patches as mpatches
from string import ascii_lowercase

# parameters of the systems in (a) and (b): maximum m of orbitals for which the LLL-like structure is checked, was the Hamiltonian Hermitian, D0, B0, rtrunc 
params_all=[
[4, True, 2.1, 12, -1],
[9, True, 2.1, 12, -1],
        ]

#set font size and fonts for the plot
fsize=13
plt.rcParams['font.family'] = 'serif'
plt.rcParams['font.serif'] = ['CMU Serif']
plt.rcParams['mathtext.fontset'] = 'cm'

# define subplots
npar=len(params_all)
fig,ax=plt.subplots(1,npar, squeeze=False, figsize=(5,3), sharex=True, sharey=True)

# read data and plot phase diagrams
for ip,params in enumerate(params_all):
  max_orbital, hermitian, D0,B0,rtrunc=params
  fname="phasediagram_landau_OBC_maxo={}_herm{}_D0={:.4f}_B0={:.4f}_rtrunc={:.4f}.txt".format(max_orbital, hermitian, D0,B0,rtrunc)
  #read the x and y axes limits and steps from the file header
  f=open(fname)
  lines=f.readlines()
  line=lines[0].split()
  linef=[]
  for i in range(1,7):
    linef.append(float(line[i]))
  d0min, d0max, dd0, Vquad0min, Vquad0max, dVquad0=linef
  landau_conditions=np.loadtxt(fname) # read the map of LLL-like structure
  im=ax[0,ip].imshow(np.transpose(landau_conditions), extent=[d0min-dd0/2., d0max+dd0/2., Vquad0min-dVquad0/2., Vquad0max+dVquad0/2.], origin="lower", rasterized=True, interpolation="none") #plot
  ax[0,ip].set_aspect("auto") # turn off aspect ratio 1


# set plot labels
ax[0,0].set_ylabel(r"$V_\mathrm{harm0}$", fontsize=fsize, labelpad=-5)
for i in range(npar):
  ax[0,i].set_xlabel(r"$d_0$", fontsize=fsize, labelpad=0)
  ax[0,i].text(0.05, 0.90, "("+ascii_lowercase[i]+")", transform=ax[0,i].transAxes,  bbox=dict(facecolor='w'), fontsize=fsize)
  ax[0,i].tick_params(axis='both', which='major', labelsize=fsize)


# create legend
values=[0,1]
labels=["not LLL-like", "LLL-like"]
colors = [ im.cmap(im.norm(value)) for value in values] # get the colors of the values, according to the colormap used by imshow
patches = [ mpatches.Patch(color=colors[i], label=labels[i] ) for i in range(len(values)) ] # create a patch (proxy artist) for every color 
ax[0,1].legend(handles=patches, loc=4, fontsize=fsize)# put those patched as legend-handles into the legend

plt.subplots_adjust(left=0.1, right=0.99, bottom=0.16, top=0.98, wspace=0.1)
plt.savefig("Figure13.pdf")

plt.show()


