import numpy as np
import matplotlib as mpl
import matplotlib.pyplot as plt
import json
from string import ascii_lowercase

#------------------------- system parameters ------------------------
lambda0=1 #atomic transition wavelength
k0=2*np.pi/lambda0 # atomic transition wavenumber
d0=0.3  # dimensionless lattice constant
d=d0*lambda0
D0=0.6 # system size 
D=D0*d
Vquad0=0 # rescaled confining potential strength (not useful for D0=0.6)
Npart=2 # maximum number of particles (note: some parts of the code work for two particles only)
Lmax=6 # Lmax-fold rotational symmetry
rtrunc=-1 #truncation radius of interaction (-1 if no truncation)
B0=12 # rescaled magnetic field
w0=1 # waist parameter of the LG modes
Lgs=2 # will compute emission from this subspace
omega=0.1 # Rabi frequency in the units of gamma0
dip1=1 # dipole element of state -
dip2=1 # dipole element oi state +
#drive="gaussian"
drive="uniform" #type of drive (gaussian or uniform)
Ltarget=1 # single-particle angular momentum sector that is accessed by driving
pol=0 # polarization (0=-, 1=+)
w0=1 # w0 of the measurement Gaussian modes
fsize=16 # font size for the plot

# write polarization as a vector
if(pol==0):
  pol2=[1,0]
if(pol==1):
  pol2=[0,1]

# set plot fonts
plt.rcParams['font.family'] = 'serif'
plt.rcParams['font.serif'] = ['CMU Serif']
plt.rcParams['mathtext.fontset'] = 'cm'

# load f_l 
proj=np.loadtxt("modeprojections_OBC_d0={:.4f}_herm{}_simpFalse_D0={:.4f}_Npart={}_B0={:.4f}_Vquad0={:.4f}_rtrunc={:.4f}.txt".format(d0, True, D0,1,B0,Vquad0,rtrunc))
lproj=proj[:,0].astype(int)

# part of the filename that depends on driving type
if(drive=="gaussian"):
  l=Ltarget+(2*pol-1)
  drivestr="g_l={}_w={:.2f}".format(l,w0)
if(drive=="uniform"):
  drivestr="u"

#load driven system results
fname="driving_ang_pol_"+drivestr+"_omega={:.4f}_pol{:.4f}_{:.4f}_Ltarget={}_d0={:.4f}_D0={:.4f}_B0={:.4f}_Vquad0={:.4f}.json".format(omega, pol2[0], pol2[1],Ltarget, d0,D0,B0,Vquad0)
f=open(fname)
data=json.load(f)
f.close()

# extract data from file
deltas=data["deltas"]
deltas0=np.array(deltas)*d**3*k0**3
ndeltas=len(deltas)
part=len(data["subspace_results"])
twopart_data=data["subspace_results"][1]
nlpairs=len(twopart_data["two-photon amplitudes"])
lpair0=twopart_data["two-photon amplitudes"][0]
lpair1=twopart_data["two-photon amplitudes"][1]
twopart_amp_ref=np.array(lpair0["real"])+1j*np.array(lpair0["imag"])


colors = mpl.colormaps.get_cmap('inferno').resampled(5).colors
coeffs=[]
exponent=-5
scale=10**exponent # the result will be plotted in the units of scale

# calculate and rescale the intensitues
for ipair,lpair in enumerate(twopart_data["two-photon amplitudes"]):
  twopart_amp=np.array(lpair["real"])+1j*np.array(lpair["imag"])
  if(ipair<4):
    l1,l2=lpair["lpair"]
    ind1=list(lproj).index(l1)
    ind2=list(lproj).index(l2)
    coeff=np.abs(twopart_amp)**2/proj[ind1,1]/proj[ind2,1]
    if(l1==l2):
      coeff/=2
    coeffs.append(coeff)
coeffs_norm=sum(coeffs)

# plot the intensities
for ipair,lpair in enumerate(twopart_data["two-photon amplitudes"]):
  if(ipair<4):
    l1,l2=lpair["lpair"]
    #plt.plot(deltas, coeffs[ipair]/coeffs_norm, label="{}, {}".format(l1, l2), c=colors[ipair])
    plt.plot(deltas0, coeffs[ipair]/scale, label=r"$l_1,l_2 = {}, {}$".format(l1, l2), c=colors[ipair])
plt.legend(loc="upper right", fontsize=fsize)

overlap=data["subspace_results"][1]["states"][0]["overlaps"]
imax=np.argmax(overlap)
deltamax=deltas0[imax]
plt.axvline(deltamax, linestyle="dotted", c="C2")
plt.ylabel(r"$\tilde{I}_{l_1, l_2}$", fontsize=fsize)
plt.ylim([-0.1, 8])


plt.xlabel(r"$\Delta_0$", fontsize=fsize)
ax=plt.gca()
ax.tick_params(axis='both', which='major', labelsize=fsize)
#ax.yaxis.offsetText.set_fontsize(fsize)
ax.text(0.0, 1.00,  r"$\times 10^{{{}}}$".format(exponent)   ,  transform=ax.transAxes, fontsize=fsize)

plt.tight_layout()
plt.savefig("Figure10.pdf")

plt.show()


