#include <stdio.h>
#include <string.h>

#include "esp_log.h"
#include "freertos/FreeRTOS.h"
#include "freertos/semphr.h"

#include "memory.h"
#include "utils.h"

#include "../include/main-thread.h"
#include "../include/protocol.h"

# define TASK_ITEM_QUEUE_LENGTH   5

typedef enum {
    TASK_RESET,
    TASK_CALL_MAIN,
    TASK_CALL_EVENT,
    TASK_SEND_PROFILE
} task_t;

typedef union {
    task_t task;

    struct task_call_main {
        task_t task;
        int32_t id;
        void* address;
    } call_main;

    struct task_event_call {
        task_t task;
        value_t fn;
    } call_event;

    struct task_send_profile {
        task_t task;
        uint8_t fid;
        char* profile;
    } send_profile;
} task_item_u;

static QueueHandle_t task_item_queue;

static bool is_code_installed(bs_memory_layout_t* memory_layout) {
    uint8_t* dflash_address = (uint8_t*)memory_layout->dflash_address;
    return dflash_address[0] == 1;
}

static void main_thread_init(bs_memory_layout_t* memory_layout) {
    BS_LOG_INFO("Initialize main thread")
    gc_initialize();
    bs_memory_init();
    bs_memory_get_layout(memory_layout);
    task_item_queue = xQueueCreate(TASK_ITEM_QUEUE_LENGTH, sizeof(task_item_u));
    if (is_code_installed(memory_layout)) {
        BS_LOG_INFO("Execute installed code");
        uint32_t dram_code_length = *(uint32_t*)(memory_layout->dflash_address + 1);
        bs_protocol_read((uint8_t*)memory_layout->dflash_address + 5, dram_code_length);
    } else {
        BS_LOG_INFO("Reset memory")
        bs_memory_reset();
    }
}

static void main_thread_reset() {
    BS_LOG_INFO("Reset main thread")
    bs_memory_ram_reset();
    gc_initialize();
    xQueueReset(task_item_queue);
}

static float task_call_main(int32_t id, void* address) {
    BS_LOG_INFO("Call main, address: %p, id: %d", (int)address, (int)id)
    uint64_t start_us = bs_timer_get_time_us();
    try_and_catch(address);
    uint64_t end_us = bs_timer_get_time_us();
    return (float)(end_us - start_us) / 1000.0;
}

static void task_call_event(value_t fn) {
    BS_LOG_INFO("Call event")
    ((void (*)(value_t))gc_function_object_ptr(fn, 0))(get_obj_property(fn, 2));
}

static void task_send_profile(uint8_t fid, char* profile) {
    BS_LOG_INFO("Send profile")
    bs_protocol_write_profile(fid, profile);
    free(profile);
}

void main_thread(void *arg) {
    bs_memory_layout_t memory_layout;
    main_thread_init(&memory_layout);

    while (true) {
        task_item_u task_item;
        xQueueReceive(task_item_queue, &task_item, portMAX_DELAY);

        switch (task_item.task) {
            case TASK_CALL_MAIN:
                float execution_time = task_call_main(task_item.call_main.id, task_item.call_main.address);
                bs_protocol_write_execution_time(task_item.call_main.id, execution_time);
                break;
            case TASK_CALL_EVENT:
                task_call_event(task_item.call_event.fn);
                break;
            case TASK_SEND_PROFILE:
                task_send_profile(task_item.send_profile.fid, task_item.send_profile.profile);
                break;
            case TASK_RESET:
                main_thread_reset();
                bs_protocol_write_memory_layout(&memory_layout);
                break;
            default:
                BS_LOG_INFO("Unknown task command.")
                break;
            }
    }
}

void bs_main_thread_init() {
    xTaskCreatePinnedToCore(main_thread, "bs_main_thread", 4096 * 16, NULL, 1, NULL, 0);
}

void bs_main_thread_reset() {
    task_item_u task_item;
    task_item.task = TASK_RESET;
    xQueueSend(task_item_queue, &task_item, portMAX_DELAY);
}

void bs_main_thread_set_main(int32_t id, void* address) {
    task_item_u task_item;
    task_item.call_main.task = TASK_CALL_MAIN;
    task_item.call_main.id = id;
    task_item.call_main.address = address;
    xQueueSend(task_item_queue, &task_item, portMAX_DELAY);
}

void bs_main_thread_set_event(value_t fn) {
    task_item_u task_item;
    task_item.call_event.task = TASK_CALL_EVENT;
    task_item.call_event.fn = fn;
    xQueueSend(task_item_queue, &task_item, portMAX_DELAY);
}

void bs_main_thread_set_event_from_isr(void* fn) {
    portBASE_TYPE yield = pdFALSE;
    task_item_u task_item;
    task_item.call_event.task = TASK_CALL_EVENT;
    task_item.call_event.fn = (value_t)fn;
    xQueueSendFromISR(task_item_queue, &task_item, &yield);
}

void bs_main_thread_set_profile(uint8_t fid, char* profile) {
    uint32_t len = strlen(profile) + 1;
    char* profile2 = (char*)malloc(len);
    strcpy(profile2, profile);
    portBASE_TYPE yield = pdFALSE;
    task_item_u task_item;
    task_item.call_event.task = TASK_SEND_PROFILE;
    task_item.send_profile.fid = fid;
    task_item.send_profile.profile = profile2;
    xQueueSendFromISR(task_item_queue, &task_item, &yield);
}