"""Quick Demonstration of Enhanced Quantum-Classical Boundary Emergence Simulator.

This script provides a simple example of how to use the enhanced simulator
with all the new features. Run this to see the simulator in action!
"""

import numpy as np
import matplotlib.pyplot as plt
from simulation_control import QuantumClassicalSimulator

def run_demo():
    """Run a quick demonstration of the enhanced simulator."""
    try:
        print("🔬 Enhanced Quantum-Classical Boundary Emergence Simulator Demo")
        print("=" * 70)

        # Configure a basic demonstration
        config = {
            # Basic parameters
            'grid_size': 256,
            'x_range': (-8, 8),
            'dt': 0.01,
            'n_steps': 300,

            # Test different correction strengths
            'lambda_values': [0.0, 0.02, 0.08],

            # Enable some advanced features
            'correction_type': 'adaptive',
            'use_lindblad': True,
            'gamma_x': 0.005,
            'gamma_p': 0.005,

            # Ensemble for statistical analysis
            'use_ensemble': True,
            'ensemble_size': 5,  # Small for quick demo

            # Diagnostics
            'diagnostic_sampling': 10,
            'save_wigner_evolution': False,  # Skip for speed
            'create_boundary_map': True,

            # Output
            'output_dir': 'demo_results',
            'save_raw_data': False
        }

        print("Configuration:")
        print(f"  - Grid size: {config['grid_size']}")
        print(f"  - Time steps: {config['n_steps']}")
        print(f"  - Lambda values: {config['lambda_values']}")
        print(f"  - Correction type: {config['correction_type']}")
        print(f"  - Lindblad decoherence: {config['use_lindblad']}")
        print(f"  - Ensemble size: {config['ensemble_size']}")
        print()

        # Create and run simulator
        print("Initializing enhanced simulator...")
        try:
            simulator = QuantumClassicalSimulator(config)
        except Exception as e:
            print(f"❌ Failed to initialize simulator: {e}")
            print("Falling back to basic configuration...")
            # Try with minimal config
            basic_config = {
                'grid_size': 128,
                'x_range': (-5, 5),
                'dt': 0.02,
                'n_steps': 100,
                'lambda_values': [0.0, 0.05],
                'output_dir': 'demo_results_fallback'
            }
            simulator = QuantumClassicalSimulator(basic_config)

        print("Running parameter sweep...")
        try:
            results, output_dir = simulator.run_full_parameter_sweep()
        except Exception as e:
            print(f"❌ Parameter sweep failed: {e}")
            print("Attempting single simulation fallback...")
            # Try single simulation as fallback
            results = [simulator.run_single_simulation(0.0, save_evolution=False)]
            output_dir = simulator.save_results(results)

        print(f"\n✅ Demo completed successfully!")
        print(f"Results saved to: {output_dir}")

        # Create a summary plot with error handling
        try:
            create_summary_plot(results)
        except Exception as e:
            print(f"⚠️ Warning: Could not create summary plot: {e}")
            print("Demo completed but visualization failed.")

        return results

    except Exception as e:
        print(f"\n❌ Demo failed with error: {e}")
        print("\nTrying minimal demonstration...")
        return run_minimal_demo()

def create_summary_plot(results):
    """Create a summary plot showing key results."""
    try:
        print("\nCreating summary visualization...")

        if not results or len(results) == 0:
            print("⚠️ Warning: No results to plot")
            return

        # Validate matplotlib availability
        try:
            import matplotlib.pyplot as plt
        except ImportError:
            print("⚠️ Warning: matplotlib not available, skipping plots")
            return

        try:
            fig, axes = plt.subplots(2, 2, figsize=(12, 10))
        except Exception as e:
            print(f"⚠️ Warning: Could not create plot figure: {e}")
            return

        for result in results:
            if 'boundary_map' in result:
                # This is the boundary map data
                boundary_data = result['boundary_map']
                plot_boundary_map(axes[0, 0], boundary_data)
                continue

            # Individual simulation results
            lambda_val = result['lambda']
            diagnostics = result.get('averaged_diagnostics', result.get('diagnostics', []))

            if not diagnostics:
                continue

            times = [d['time'] for d in diagnostics if 'time' in d]

            # Plot Shannon entropy evolution
            shannon_entropies = [d.get('shannon_entropy', 0) for d in diagnostics]
            axes[0, 1].plot(times, shannon_entropies, label=f'λ = {lambda_val:.3f}', linewidth=2)

            # Plot Von Neumann entropy evolution (if available)
            vn_entropies = [d.get('von_neumann_entropy', 0) for d in diagnostics]
            if any(vn > 0 for vn in vn_entropies):
                axes[1, 0].plot(times, vn_entropies, label=f'λ = {lambda_val:.3f}', linewidth=2)

            # Plot position evolution
            positions = [d.get('position_expectation', 0) for d in diagnostics]
            axes[1, 1].plot(times, positions, label=f'λ = {lambda_val:.3f}', linewidth=2)

        # Customize plots
        axes[0, 1].set_title('Shannon Entropy Evolution')
        axes[0, 1].set_xlabel('Time')
        axes[0, 1].set_ylabel('Shannon Entropy')
        axes[0, 1].legend()
        axes[0, 1].grid(True, alpha=0.3)

        axes[1, 0].set_title('Von Neumann Entropy Evolution')
        axes[1, 0].set_xlabel('Time')
        axes[1, 0].set_ylabel('Von Neumann Entropy')
        axes[1, 0].legend()
        axes[1, 0].grid(True, alpha=0.3)

        axes[1, 1].set_title('Position Expectation Value')
        axes[1, 1].set_xlabel('Time')
        axes[1, 1].set_ylabel('⟨x⟩')
        axes[1, 1].legend()
        axes[1, 1].grid(True, alpha=0.3)

        plt.suptitle('Enhanced Quantum-Classical Simulator Demo Results', fontsize=14)
        plt.tight_layout()

        try:
            plt.savefig('demo_summary.png', dpi=150, bbox_inches='tight')
            print("Summary plot saved as: demo_summary.png")
        except Exception as e:
            print(f"⚠️ Warning: Could not save plot: {e}")

    except Exception as e:
        print(f"⚠️ Warning: Plot creation failed: {e}")
        print("Continuing without visualization...")

def plot_boundary_map(ax, boundary_data):
    """Plot the quantum-classical boundary map."""
    lambda_vals = boundary_data['lambda_values']

    # Try to find Shannon entropy data
    if 'shannon_entropy' in boundary_data['metrics']:
        metric_data = boundary_data['metrics']['shannon_entropy']

        # Create a simple final value plot
        final_values = boundary_data['final_metrics']['shannon_entropy']
        ax.plot(lambda_vals, final_values, 'bo-', linewidth=2, markersize=6)
        ax.set_title('Quantum-Classical Boundary Map')
        ax.set_xlabel('Correction Strength λ')
        ax.set_ylabel('Final Shannon Entropy')
        ax.grid(True, alpha=0.3)
    else:
        ax.text(0.5, 0.5, 'Boundary Map\n(Data Not Available)',
               ha='center', va='center', transform=ax.transAxes)
        ax.set_title('Boundary Map')

def test_individual_components():
    """Test individual components of the simulator."""
    print("\n🔧 Testing individual components...")

    failed_components = []

    try:

        # Test physics module
        print("  Testing physics module...")
        try:
            from physics import QuantumPhysics
            physics = QuantumPhysics(grid_size=64)
            psi = np.exp(-physics.x**2)
            psi = physics.normalize_wavefunction(psi)
            V = physics.double_well_potential()
            psi_evolved = physics.split_step_evolution(psi, V, 0.01)
            Q = physics.quantum_potential(psi)
            print("    ✅ Physics module working")
        except Exception as e:
            print(f"    ❌ Physics module failed: {e}")
            failed_components.append("physics")

        # Test diagnostics module
        print("  Testing diagnostics module...")
        try:
            from diagnostics import QuantumDiagnostics
            diag = QuantumDiagnostics(physics.x, physics.dx)
            entropy = diag.shannon_entropy(psi)
            rho = physics.wavefunction_to_density_matrix(psi)
            vn_entropy = diag.von_neumann_entropy(rho)
            print("    ✅ Diagnostics module working")
        except Exception as e:
            print(f"    ❌ Diagnostics module failed: {e}")
            failed_components.append("diagnostics")

        # Test phase space module
        print("  Testing phase space module...")
        try:
            from phase_space import PhaseSpaceAnalysis
            phase_space = PhaseSpaceAnalysis(physics.x[:32], physics.k[:32])  # Smaller for speed
            wigner = phase_space.compute_wigner_function(psi[:32])
            print("    ✅ Phase space module working")
        except Exception as e:
            print(f"    ❌ Phase space module failed: {e}")
            failed_components.append("phase_space")

        # Test correction module
        print("  Testing correction module...")
        try:
            from correction import CorrectionMechanism
            correction = CorrectionMechanism(physics.x, 0.01)
            V_corr, strength = correction.basic_correction(psi, 0.0, 0.1)
            print("    ✅ Correction module working")
        except Exception as e:
            print(f"    ❌ Correction module failed: {e}")
            failed_components.append("correction")

        if failed_components:
            print(f"  ⚠️ Some components failed: {failed_components}")
            print("  System may have limited functionality.")
        else:
            print("  🎉 All components working correctly!")

    except Exception as e:
        print(f"  ❌ Component testing failed: {e}")
        print("  System may not be functional.")

if __name__ == "__main__":
    # Test individual components first
    test_individual_components()

    # Run the full demo
    try:
        results = run_demo()
        print("\n🎉 Demo completed successfully!")
        print("Try running: python Enhanced_Quantum_Classical_Simulator.py --preset basic")
        print("Or explore the test suite: python test_suite.py")
    except Exception as e:
        print(f"\n❌ Demo failed with error: {e}")
        import traceback
        traceback.print_exc()
        print("\nTrying minimal demonstration...")
        run_minimal_demo()

def run_minimal_demo():
    """Run a minimal demonstration with maximum error tolerance."""
    print("\n🔄 Running minimal demonstration...")

    try:
        # Test basic numpy operations
        import numpy as np
        x = np.linspace(-5, 5, 100)
        psi = np.exp(-x**2)
        psi = psi / np.sqrt(np.trapz(np.abs(psi)**2, x))

        print("✅ Basic numpy operations working")
        print(f"✅ Created normalized wavefunction with norm: {np.trapz(np.abs(psi)**2, x):.6f}")

        # Test simple potential
        V = 0.25 * x**4 - 2 * x**2
        print("✅ Created double-well potential")

        # Test basic integration
        expectation_x = np.trapz(x * np.abs(psi)**2, x)
        print(f"✅ Calculated position expectation: {expectation_x:.6f}")

        print("\n🎉 Minimal demo completed successfully!")
        print("The system can perform basic operations.")

        return {'status': 'minimal_success', 'expectation_x': expectation_x}

    except Exception as e:
        print(f"\n❌ Even minimal demo failed: {e}")
        print("The system has severe issues that need to be addressed.")
        return {'status': 'complete_failure', 'error': str(e)}