#!/usr/bin/env python3
"""
Physics Analysis: Quantum-Classical Boundary Emergence Verification
==================================================================

This script focuses on the PHYSICS rather than robustness - specifically
examining whether the system actually demonstrates quantum-to-classical
boundary emergence as intended.
"""

import numpy as np
import matplotlib.pyplot as plt
import warnings
from typing import Dict, List, Tuple
import time

def analyze_quantum_classical_transition():
    """Analyze whether the system shows quantum-to-classical transition."""
    print("🔬 ANALYZING QUANTUM-CLASSICAL BOUNDARY EMERGENCE")
    print("=" * 60)

    from physics import QuantumPhysics
    from correction import CorrectionMechanism
    from input_validator import create_safe_config

    # Create configuration for physics analysis
    config = create_safe_config(grid_size=256, time_steps=200)

    physics = QuantumPhysics(grid_size=config['grid_size'], x_range=(-8, 8))
    correction = CorrectionMechanism(physics.x, config['dt'])

    # Test different scenarios to see quantum-classical transition
    scenarios = [
        {
            'name': 'No Correction (Pure Quantum)',
            'lambda': 0.0,
            'initial_state': 'asymmetric_superposition',
            'description': 'Should show pure quantum evolution'
        },
        {
            'name': 'Weak Correction',
            'lambda': 0.02,
            'initial_state': 'asymmetric_superposition',
            'description': 'Should show slight classicalization'
        },
        {
            'name': 'Medium Correction',
            'lambda': 0.08,
            'initial_state': 'asymmetric_superposition',
            'description': 'Should show clear transition'
        },
        {
            'name': 'Strong Correction',
            'lambda': 0.2,
            'initial_state': 'asymmetric_superposition',
            'description': 'Should show strong classical behavior'
        }
    ]

    results = []

    for scenario in scenarios:
        print(f"\n📊 {scenario['name']} (λ = {scenario['lambda']})")
        print(f"   {scenario['description']}")

        # Create asymmetric initial state to see dynamics
        if scenario['initial_state'] == 'asymmetric_superposition':
            # Create superposition of left and right Gaussians with slight asymmetry
            psi_left = 0.6 * np.exp(-(physics.x + 2.5)**2)  # Left component (stronger)
            psi_right = 0.4 * np.exp(-(physics.x - 2.5)**2)  # Right component (weaker)
            psi = psi_left + 1j * psi_right  # Add phase difference
        else:
            # Default symmetric state
            psi = np.exp(-physics.x**2)

        psi = physics.normalize_wavefunction(psi)
        psi_initial = psi.copy()

        # Track evolution
        evolution_data = {
            'times': [],
            'positions': [],
            'position_uncertainties': [],
            'shannon_entropies': [],
            'quantum_potentials': [],
            'norms': [],
            'asymmetries': [],
            'wavefunction_snapshots': []
        }

        lambda_val = scenario['lambda']

        # Evolution loop
        for step in range(config['n_steps']):
            current_time = step * config['dt']

            # Quantum evolution
            potential = physics.double_well_potential()
            psi = physics.split_step_evolution(psi, potential, config['dt'])

            # Apply correction if needed
            if lambda_val > 0:
                x_classical = 0.0  # Classical reference point
                V_corr, _ = correction.basic_correction(psi, x_classical, lambda_val)

                if not (np.any(np.isnan(V_corr)) or np.any(np.isinf(V_corr))):
                    correction_factor = np.exp(-1j * V_corr * config['dt'])
                    if not (np.any(np.isnan(correction_factor)) or np.any(np.isinf(correction_factor))):
                        psi *= correction_factor

            psi = physics.normalize_wavefunction(psi)

            # Collect detailed diagnostics every few steps
            if step % 5 == 0:
                # Position expectation and uncertainty
                prob_density = np.abs(psi)**2
                position = np.trapz(physics.x * prob_density, physics.x)
                position_sq = np.trapz(physics.x**2 * prob_density, physics.x)
                position_uncertainty = np.sqrt(position_sq - position**2)

                # Shannon entropy (measure of delocalization)
                prob_density_norm = prob_density / np.sum(prob_density)
                shannon_entropy = -np.sum(prob_density_norm * np.log(prob_density_norm + 1e-16))

                # Quantum potential
                Q = physics.quantum_potential(psi)
                Q_mean = np.mean(np.abs(Q))

                # Asymmetry measure (how much the wavefunction favors one side)
                left_prob = np.trapz(prob_density[physics.x < 0], physics.x[physics.x < 0])
                right_prob = np.trapz(prob_density[physics.x > 0], physics.x[physics.x > 0])
                asymmetry = abs(left_prob - right_prob)

                # Norm check
                norm = np.sqrt(np.trapz(prob_density, physics.x))

                evolution_data['times'].append(current_time)
                evolution_data['positions'].append(position)
                evolution_data['position_uncertainties'].append(position_uncertainty)
                evolution_data['shannon_entropies'].append(shannon_entropy)
                evolution_data['quantum_potentials'].append(Q_mean)
                evolution_data['norms'].append(norm)
                evolution_data['asymmetries'].append(asymmetry)

                # Store snapshots for visualization
                if step % 20 == 0:
                    evolution_data['wavefunction_snapshots'].append({
                        'time': current_time,
                        'psi': psi.copy(),
                        'prob_density': prob_density.copy()
                    })

        # Analysis of this scenario
        final_analysis = analyze_scenario_results(evolution_data, scenario)
        scenario_result = {
            'scenario': scenario,
            'evolution_data': evolution_data,
            'analysis': final_analysis
        }

        results.append(scenario_result)

        # Print key findings
        print(f"   📈 Final position: {evolution_data['positions'][-1]:6.3f}")
        print(f"   📊 Final uncertainty: {evolution_data['position_uncertainties'][-1]:6.3f}")
        print(f"   🌀 Final Shannon entropy: {evolution_data['shannon_entropies'][-1]:6.3f}")
        print(f"   ⚖️ Final asymmetry: {evolution_data['asymmetries'][-1]:6.3f}")
        print(f"   🎯 Quantum→Classical transition: {final_analysis['shows_transition']}")

    return results

def analyze_scenario_results(evolution_data, scenario):
    """Analyze whether a scenario shows quantum-to-classical transition."""

    # Key indicators of quantum-to-classical transition:
    # 1. Decreasing Shannon entropy (more localized)
    # 2. Decreasing position uncertainty (more classical-like)
    # 3. Stabilizing position (classical trajectory)
    # 4. Decreasing quantum potential fluctuations

    times = np.array(evolution_data['times'])
    positions = np.array(evolution_data['positions'])
    uncertainties = np.array(evolution_data['position_uncertainties'])
    entropies = np.array(evolution_data['shannon_entropies'])
    q_potentials = np.array(evolution_data['quantum_potentials'])

    analysis = {}

    # Check for entropy reduction (localization)
    initial_entropy = entropies[0] if len(entropies) > 0 else 0
    final_entropy = entropies[-1] if len(entropies) > 0 else 0
    entropy_reduction = initial_entropy - final_entropy
    analysis['entropy_reduction'] = entropy_reduction
    analysis['entropy_decreases'] = entropy_reduction > 0.1

    # Check for uncertainty reduction
    initial_uncertainty = uncertainties[0] if len(uncertainties) > 0 else 0
    final_uncertainty = uncertainties[-1] if len(uncertainties) > 0 else 0
    uncertainty_reduction = initial_uncertainty - final_uncertainty
    analysis['uncertainty_reduction'] = uncertainty_reduction
    analysis['uncertainty_decreases'] = uncertainty_reduction > 0.1

    # Check for position stabilization
    if len(positions) > 10:
        late_positions = positions[-10:]  # Last 10 points
        position_stability = np.std(late_positions)
        analysis['position_stability'] = position_stability
        analysis['position_stabilized'] = position_stability < 0.5
    else:
        analysis['position_stability'] = float('inf')
        analysis['position_stabilized'] = False

    # Check for quantum potential reduction
    if len(q_potentials) > 1:
        initial_q_pot = np.mean(q_potentials[:5])
        final_q_pot = np.mean(q_potentials[-5:])
        q_potential_reduction = initial_q_pot - final_q_pot
        analysis['q_potential_reduction'] = q_potential_reduction
        analysis['q_potential_decreases'] = q_potential_reduction > 0
    else:
        analysis['q_potential_reduction'] = 0
        analysis['q_potential_decreases'] = False

    # Overall assessment
    transition_indicators = sum([
        analysis['entropy_decreases'],
        analysis['uncertainty_decreases'],
        analysis['position_stabilized'],
        analysis['q_potential_decreases']
    ])

    analysis['transition_score'] = transition_indicators
    analysis['shows_transition'] = transition_indicators >= 2

    # Correction effectiveness (only for lambda > 0)
    if scenario['lambda'] > 0:
        analysis['correction_effective'] = analysis['shows_transition']
    else:
        analysis['correction_effective'] = None

    return analysis

def create_physics_visualization(results):
    """Create visualizations showing the quantum-classical transition."""
    print("\n📊 Creating Physics Visualization...")

    try:
        fig, axes = plt.subplots(2, 3, figsize=(18, 12))

        colors = ['blue', 'green', 'orange', 'red']

        for i, result in enumerate(results):
            color = colors[i % len(colors)]
            scenario = result['scenario']
            data = result['evolution_data']

            times = data['times']

            # Plot 1: Position evolution
            axes[0, 0].plot(times, data['positions'],
                           label=f"λ = {scenario['lambda']}",
                           color=color, linewidth=2)

            # Plot 2: Position uncertainty
            axes[0, 1].plot(times, data['position_uncertainties'],
                           label=f"λ = {scenario['lambda']}",
                           color=color, linewidth=2)

            # Plot 3: Shannon entropy
            axes[0, 2].plot(times, data['shannon_entropies'],
                           label=f"λ = {scenario['lambda']}",
                           color=color, linewidth=2)

            # Plot 4: Quantum potential
            axes[1, 0].plot(times, data['quantum_potentials'],
                           label=f"λ = {scenario['lambda']}",
                           color=color, linewidth=2)

            # Plot 5: Asymmetry
            axes[1, 1].plot(times, data['asymmetries'],
                           label=f"λ = {scenario['lambda']}",
                           color=color, linewidth=2)

            # Plot 6: Norm (should stay ~1)
            axes[1, 2].plot(times, data['norms'],
                           label=f"λ = {scenario['lambda']}",
                           color=color, linewidth=2)

        # Customize plots
        titles = [
            'Position Evolution ⟨x⟩', 'Position Uncertainty Δx', 'Shannon Entropy',
            'Quantum Potential |Q|', 'Left-Right Asymmetry', 'Normalization'
        ]

        for ax, title in zip(axes.flat, titles):
            ax.set_title(title, fontsize=12)
            ax.set_xlabel('Time')
            ax.legend()
            ax.grid(True, alpha=0.3)

        plt.suptitle('Quantum-Classical Boundary Emergence Analysis', fontsize=16)
        plt.tight_layout()
        plt.savefig('quantum_classical_analysis.png', dpi=150, bbox_inches='tight')
        print("✅ Visualization saved as: quantum_classical_analysis.png")

    except Exception as e:
        print(f"⚠️ Could not create visualization: {e}")

def generate_physics_report(results):
    """Generate a report on the physics findings."""
    print("\n📋 QUANTUM-CLASSICAL BOUNDARY EMERGENCE REPORT")
    print("=" * 60)

    # Check if we see the expected physics
    transition_found = False
    lambda_effectiveness = []

    print("\n🔍 TRANSITION ANALYSIS BY CORRECTION STRENGTH:")

    for result in results:
        scenario = result['scenario']
        analysis = result['analysis']

        print(f"\n   {scenario['name']} (λ = {scenario['lambda']}):")
        print(f"     Entropy reduction: {analysis['entropy_reduction']:6.3f} ({'✅' if analysis['entropy_decreases'] else '❌'})")
        print(f"     Uncertainty reduction: {analysis['uncertainty_reduction']:6.3f} ({'✅' if analysis['uncertainty_decreases'] else '❌'})")
        print(f"     Position stability: {analysis['position_stability']:6.3f} ({'✅' if analysis['position_stabilized'] else '❌'})")
        print(f"     Quantum potential ↓: {'✅' if analysis['q_potential_decreases'] else '❌'}")
        print(f"     Transition score: {analysis['transition_score']}/4")
        print(f"     Shows transition: {'✅ YES' if analysis['shows_transition'] else '❌ NO'}")

        if analysis['shows_transition']:
            transition_found = True

        if scenario['lambda'] > 0:
            lambda_effectiveness.append({
                'lambda': scenario['lambda'],
                'effective': analysis['correction_effective'],
                'score': analysis['transition_score']
            })

    # Overall assessment
    print(f"\n🎯 OVERALL ASSESSMENT:")
    print(f"   Quantum-classical transition observed: {'✅ YES' if transition_found else '❌ NO'}")

    if lambda_effectiveness:
        effective_corrections = [x for x in lambda_effectiveness if x['effective']]
        print(f"   Effective correction strengths: {len(effective_corrections)}/{len(lambda_effectiveness)}")

        if effective_corrections:
            best_lambda = max(effective_corrections, key=lambda x: x['score'])
            print(f"   Most effective λ: {best_lambda['lambda']} (score: {best_lambda['score']}/4)")

    # Scientific conclusions
    print(f"\n🧪 SCIENTIFIC CONCLUSIONS:")

    if transition_found:
        print("   ✅ System successfully demonstrates quantum-classical boundary emergence")
        print("   ✅ Correction mechanisms effectively induce classicalization")
        print("   ✅ Observable transition from quantum superposition to classical localization")
        print("   ✅ Physics goals achieved")
    else:
        print("   ❌ Limited evidence of quantum-classical boundary emergence")
        print("   ⚠️  May need longer simulation times or different parameters")
        print("   ⚠️  Initial conditions might need adjustment")
        print("   ⚠️  Correction mechanisms may need strengthening")

    return transition_found

if __name__ == "__main__":
    print("🔬 QUANTUM-CLASSICAL BOUNDARY EMERGENCE PHYSICS ANALYSIS")
    print("Focus: Does the system demonstrate the intended physics?")
    print("=" * 70)

    # Suppress warnings for cleaner output
    warnings.filterwarnings("ignore")

    start_time = time.time()

    try:
        # Run physics analysis
        results = analyze_quantum_classical_transition()

        # Create visualizations
        create_physics_visualization(results)

        # Generate physics report
        physics_success = generate_physics_report(results)

        elapsed = time.time() - start_time
        print(f"\n⏱️ Analysis completed in {elapsed:.2f} seconds")

        if physics_success:
            print("\n🎉 SUCCESS: System demonstrates quantum-classical boundary emergence!")
            print("The physics goals have been achieved.")
        else:
            print("\n🤔 MIXED RESULTS: System is robust but physics demonstration needs improvement.")
            print("Consider adjusting simulation parameters or initial conditions.")

    except Exception as e:
        print(f"\n❌ Analysis failed: {e}")
        import traceback
        traceback.print_exc()