import argparse
import json
import os
import time
from pathlib import Path

try:
    import yaml
except Exception:
    yaml = None

def _deep_update(a, b):
    for k, v in b.items():
        if isinstance(v, dict) and isinstance(a.get(k), dict):
            _deep_update(a[k], v)
        else:
            a[k] = v
    return a

def _as_number_or_bool_or_str(x):
    if isinstance(x, (int, float, bool)):
        return x
    s = str(x)
    if s.lower() in ["true", "false"]:
        return s.lower() == "true"
    try:
        if "." in s:
            return float(s)
        return int(s)
    except Exception:
        pass
    try:
        return json.loads(s)
    except Exception:
        return s

def _set_deep(cfg, keypath, value):
    keys = keypath.split(".")
    d = cfg
    for k in keys[:-1]:
        if k not in d or not isinstance(d[k], dict):
            d[k] = {}
        d = d[k]
    d[keys[-1]] = value

def default_cfg():
    return {
        "device": "cuda",
        "outdir": "outputs",
        "seed": 0,
        "env": {
            "rate_hz": 100.0,
            "dt": 0.01,
            "cycle_phases": ["lower","cut","lift","dump"],
            "energy_scope": "hydraulic_pump_equiv",
            "valve_limits": {"min": 0.0, "max": 1.0, "rate_per_step": 0.02},
            "pump_limits": {"min_mpa": 5.0, "max_mpa": 35.0, "rate_mpa_per_step": 0.5},
            "accumulator": {"type": "bladder", "gas_volume_L": 10.0, "soc_window": [0.0,1.0], "op_window_mpa": [12.0, 28.0], "precharge_ratio": 0.9},
            "system_pressure_window_mpa": [5.0, 35.0],
            "oil_temp_range_C": [35.0, 65.0],
            "viscosity_range_Pa_s": [0.050, 0.020],
            "safety": {"rail_lo_mpa": 5.0, "rail_hi_mpa": 35.0, "acc_lo_mpa": 12.0, "acc_hi_mpa": 28.0, "spool_rate_per_step": 0.02, "pump_rate_mpa_per_step": 0.5, "pressure_margin_mpa": 0.2, "soc_margin": 0.05},
            "excavator": {
                "class_t": 21,
                "joints": ["boom","arm","bucket"],
                "pump_rated_Lpm": 220.0,
                "pump_rpm": 1800.0,
                "relief_mpa": [32.0, 35.0],
                "valve": "pressure_compensated_sectional",
                "angles_deg": {"boom":[-5, 65], "arm":[-45, 75], "bucket":[-100, 30]}
            }
        },
        "uncertainty": {
            "duration_s": 1200.0,
            "rate_hz": 100.0,
            "seeds": [i for i in range(100, 140)],
            "environment_temperature_C": {"start": 18.0, "end": 34.0, "ramp_until_s": 720.0, "hold_after": True},
            "oil_temp_lag_s": 180.0,
            "oil_temp_clamp_C": [35.0, 65.0],
            "viscosity_map_Pa_s": {"hot": 0.020, "cold": 0.050, "T_hot_C": 65.0, "T_cold_C": 35.0},
            "soil_segments_s": [240.0, 180.0, 300.0, 240.0, 240.0],
            "soil_means": [0.9, 1.2, 1.0, 1.3, 0.8],
            "soil_noise": {"amp_rel": 0.10, "corr_time_s": 0.5},
            "wear_drift": {"valve_gain_drop": 0.06, "cyl_leak_Lpm_at_rated": 0.4, "pump_eta_drop": 0.03},
            "sensor_noise": {"pressure_std_fs": 0.005, "pressure_bias_rw_fs": 0.002, "angle_std_deg": 0.15, "angle_bias_rw_deg": 0.5},
            "acc_precharge_offset": {"uniform_rel": 0.10},
            "valve_hysteresis": 0.03,
            "valve_deadband": 0.02,
            "operator": {"delay_ms_range": [80,160], "amp_scale_range": [0.95, 1.05]},
            "constraints": {"acc_window_mpa": [12.0, 28.0], "sys_window_mpa": [5.0, 35.0]}
        },
        "pf": {
            "num_particles": 1024,
            "ess_threshold_frac": 0.5,
            "resampling": "stratified",
            "process_noise": {"soil": 0.002, "acc_offset": 0.0005, "pump_eta": 0.0002, "valve_gain": 0.0005, "cyl_leak": 0.0005, "bias_pressure": 0.0002, "bias_angle": 0.002, "oil_T": 0.1},
            "obs_noise": {"pressure_std": 0.05, "angle_std_deg": 0.15}
        },
        "rl": {
            "gamma": 0.999,
            "gae_lambda": 0.95,
            "ent_coef": 0.01,
            "net": {"actor_hidden": [256,256], "critic_hidden": [256,256]},
            "optim": {"actor_lr_init": 1e-4, "actor_lr_final": 1e-5, "critic_lr_init": 3e-4, "critic_lr_final": 3e-5},
            "batch_size": 8,
            "episode_seconds": 60.0,
            "clip_grad": 1.0
        },
        "train": {
            "windows": 40
        },
        "eval": {
            "seeds": 20,
            "extension_cycles": 120,
            "save_dir": "outputs",
            "enable_stats": True
        },
        "figs": {
            "dir": "outputs/figs",
            "dpi": 160,
            "style": "default",
            "same_page": True
        },
        "logging": {
            "dir": "outputs/logs",
            "per_step_parquet": True,
            "level": "INFO"
        }
    }

def load_cfg(path=None):
    cfg = default_cfg()
    if path is None:
        return cfg
    p = Path(path)
    if not p.exists():
        return cfg
    if p.suffix.lower() in [".yml", ".yaml"]:
        if yaml is None:
            return cfg
        with open(p, "r", encoding="utf-8") as f:
            user = yaml.safe_load(f)
    else:
        with open(p, "r", encoding="utf-8") as f:
            user = json.load(f)
    if user is None:
        return cfg
    return _deep_update(cfg, user)

def override_from_cli(cfg, argv=None):
    parser = argparse.ArgumentParser(add_help=False)
    parser.add_argument("--device", type=str)
    parser.add_argument("--outdir", type=str)
    parser.add_argument("--seed", type=int)
    parser.add_argument("--set", action="append", default=[])
    args, _ = parser.parse_known_args(argv)
    if args.device is not None:
        cfg["device"] = args.device
    if args.outdir is not None:
        cfg["outdir"] = args.outdir
        cfg["eval"]["save_dir"] = args.outdir
        cfg["figs"]["dir"] = os.path.join(args.outdir, "figs")
        cfg["logging"]["dir"] = os.path.join(args.outdir, "logs")
    if args.seed is not None:
        cfg["seed"] = int(args.seed)
    for kv in args.__dict__.get("set", []) or []:
        if "=" not in kv:
            continue
        k, v = kv.split("=", 1)
        _set_deep(cfg, k.strip(), _as_number_or_bool_or_str(v.strip()))
    return cfg

def freeze(cfg, outdir=None, tag=None):
    outdir = outdir or cfg.get("outdir", "outputs")
    Path(outdir).mkdir(parents=True, exist_ok=True)
    stamp = time.strftime("%Y%m%d_%H%M%S")
    name = f"config_frozen_{stamp}" + (f"_{tag}" if tag else "") + ".json"
    path = os.path.join(outdir, name)
    with open(path, "w", encoding="utf-8") as f:
        json.dump(cfg, f, indent=2, ensure_ascii=False)
    return path

def seeds_list(cfg, n=None):
    base = int(cfg.get("seed", 0))
    want = int(n if n is not None else cfg.get("eval", {}).get("seeds", 20))
    return [base + i for i in range(want)]

def assemble_working_config(path=None, argv=None):
    cfg = load_cfg(path)
    cfg = override_from_cli(cfg, argv)
    Path(cfg["outdir"]).mkdir(parents=True, exist_ok=True)
    Path(cfg["figs"]["dir"]).mkdir(parents=True, exist_ok=True)
    Path(cfg["logging"]["dir"]).mkdir(parents=True, exist_ok=True)
    return cfg

if __name__ == "__main__":
    import sys
    cfg = assemble_working_config(path=sys.argv[1] if len(sys.argv)>1 and not sys.argv[1].startswith("--") else None, argv=sys.argv[1:] if len(sys.argv)>1 else None)
    p = freeze(cfg, cfg.get("outdir","outputs"))
    print(p)
