import numpy as np
import pandas as pd

class BootstrapPF:
    def __init__(self, cfg=None):
        self.cfg = cfg or {}
        self.N = int(self.cfg.get("pf", {}).get("num_particles", 1024))
        self.ess_threshold = float(self.cfg.get("pf", {}).get("ess_frac", 0.5))
        self.resample_mode = str(self.cfg.get("pf", {}).get("resample", "stratified"))
        self.dt = float(self.cfg.get("env", {}).get("dt", 0.01))
        self.rate_hz = float(self.cfg.get("env", {}).get("rate_hz", 100.0))
        self.rng = np.random.RandomState(self.cfg.get("seed", 0))
        self.particles = None
        self.weights = None
        self.prev_spools = None
        self.t = 0
        self.log_rows = []
        self.noise = {}
        self.noise["angles_sigma"] = float(self.cfg.get("sensor", {}).get("angles_sigma_deg", 0.15))
        self.noise["press_sigma_mpa"] = float(self.cfg.get("sensor", {}).get("press_sigma_mpa", 0.2))
        self.noise["pump_press_sigma_mpa"] = float(self.cfg.get("sensor", {}).get("pump_press_sigma_mpa", 0.2))
        self.noise["oilT_sigma_C"] = float(self.cfg.get("sensor", {}).get("oilT_sigma_C", 0.5))
        self.bounds = {}
        self.bounds["soil"] = (0.6, 1.5)
        self.bounds["acc_offset"] = (-0.10, 0.10)
        self.bounds["valve_gain_scale"] = (0.90, 1.00)
        self.bounds["cyl_leak_scale"] = (0.00, 0.40)
        self.bounds["pump_eta"] = (0.95, 1.00)
        self.bounds["pb_pump"] = (-0.08, 0.08)
        self.bounds["pb_ports"] = (-0.08, 0.08)
        self.bounds["enc_bias"] = (-0.5, 0.5)
        self.bounds["oil_T"] = (35.0, 65.0)
        self.bounds["mu"] = (0.020, 0.050)
        self.bounds["soc"] = (0.0, 1.0)
        self.hyst = float(self.cfg.get("uncertainty", {}).get("valve_hysteresis", 0.03))
        self.dead = float(self.cfg.get("uncertainty", {}).get("valve_deadband", 0.02))
        self.rail_lo = 5.0
        self.rail_hi = 35.0
        self.acc_lo = 12.0
        self.acc_hi = 28.0

    def _linmap(self, x, x0, x1, y0, y1):
        return y0 + (np.clip(x, x0, x1) - x0) * (y1 - y0) / (x1 - x0 + 1e-9)

    def _clamp(self, x, lo, hi):
        return np.minimum(np.maximum(x, lo), hi)

    def _mu_from_oilT(self, T):
        return self._linmap(T, 35.0, 65.0, 0.050, 0.020)

    def init_from_obs(self, obs):
        self.particles = {}
        self.particles["soil"] = self.rng.uniform(0.8, 1.2, size=self.N)
        self.particles["acc_offset"] = self.rng.uniform(-0.10, 0.10, size=self.N)
        self.particles["valve_gain_scale"] = self.rng.uniform(0.96, 1.00, size=self.N)
        self.particles["cyl_leak_scale"] = self.rng.uniform(0.00, 0.10, size=self.N)
        self.particles["pump_eta"] = self.rng.uniform(0.97, 1.00, size=self.N)
        self.particles["pb_pump"] = self.rng.normal(0.0, 0.01, size=self.N)
        self.particles["pb_ports"] = self.rng.normal(0.0, 0.01, size=(self.N, 6))
        self.particles["enc_bias"] = self.rng.normal(0.0, 0.1, size=self.N)
        oilT0 = float(obs["oil_T"])
        self.particles["oil_T"] = self._clamp(self.rng.normal(oilT0, 0.5, size=self.N), *self.bounds["oil_T"])
        self.particles["mu"] = self._clamp(self._mu_from_oilT(self.particles["oil_T"]), *self.bounds["mu"])
        self.particles["soc"] = self.rng.uniform(0.4, 0.6, size=self.N)
        self.particles["P_pump"] = self.rng.uniform(18.0, 24.0, size=self.N)
        self.particles["phase_code"] = self.rng.choice([0,1,2,3], size=self.N, p=[0.3,0.3,0.2,0.2])
        self.weights = np.full(self.N, 1.0 / self.N, dtype=np.float64)
        self.prev_spools = np.zeros((self.N, 3), dtype=np.float32)
        self.t = 0
        return self.estimate()

    def _spool_hyst_map(self, sp, prev):
        out = np.zeros_like(sp)
        for i in range(sp.shape[0]):
            for j in range(3):
                c = sp[i, j]
                p = prev[i, j]
                if abs(c) < self.dead:
                    eff = 0.0
                else:
                    if c > p + self.hyst:
                        eff = c - 0.5*self.hyst
                    elif c < p - self.hyst:
                        eff = c + 0.5*self.hyst
                    else:
                        eff = p
                out[i, j] = np.clip(eff, 0.0, 1.0)
        return out

    def predict(self, u, dt=None):
        if self.particles is None:
            raise RuntimeError("init_from_obs must be called first")
        dtt = float(self.dt if dt is None else dt)
        sp_raw = np.broadcast_to(np.asarray(u[:3], dtype=np.float32), (self.N, 3)).copy()
        sp_eff = self._spool_hyst_map(sp_raw, self.prev_spools)
        self.prev_spools = sp_eff
        pump_sp = float(u[3])
        pump_target = self._clamp(pump_sp * self.rail_hi, self.rail_lo, self.rail_hi)
        a_gain = -0.06 / (20.0*60.0*self.rate_hz)
        self.particles["valve_gain_scale"] = self._clamp(self.particles["valve_gain_scale"] + a_gain + self.rng.normal(0.0, 1e-5, size=self.N), *self.bounds["valve_gain_scale"])
        a_leak = 0.40 / (20.0*60.0*self.rate_hz)
        self.particles["cyl_leak_scale"] = self._clamp(self.particles["cyl_leak_scale"] + a_leak + self.rng.normal(0.0, 1e-4, size=self.N), *self.bounds["cyl_leak_scale"])
        a_pump = -0.03 / (20.0*60.0*self.rate_hz)
        self.particles["pump_eta"] = self._clamp(self.particles["pump_eta"] + a_pump + self.rng.normal(0.0, 1e-5, size=self.N), *self.bounds["pump_eta"])
        self.particles["pb_pump"] = self._clamp(self.particles["pb_pump"] + self.rng.normal(0.0, 1e-4, size=self.N), *self.bounds["pb_pump"])
        self.particles["pb_ports"] = self._clamp(self.particles["pb_ports"] + self.rng.normal(0.0, 1e-4, size=(self.N,6)), *self.bounds["pb_ports"])
        self.particles["enc_bias"] = self._clamp(self.particles["enc_bias"] + self.rng.normal(0.0, 1e-3, size=self.N), *self.bounds["enc_bias"])
        alpha_T = 1.0 - (1.0 / max(1.0, 180.0*self.rate_hz))
        self.particles["oil_T"] = self._clamp(alpha_T*self.particles["oil_T"] + (1.0-alpha_T)*self.particles["oil_T"] + self.rng.normal(0.0, 0.02, size=self.N), *self.bounds["oil_T"])
        self.particles["mu"] = self._clamp(self._mu_from_oilT(self.particles["oil_T"]), *self.bounds["mu"])
        drift_soil = self.rng.normal(0.0, 5e-4, size=self.N)
        self.particles["soil"] = self._clamp(self.particles["soil"] + drift_soil, *self.bounds["soil"])
        self.particles["acc_offset"] = self._clamp(self.particles["acc_offset"] + self.rng.normal(0.0, 1e-5, size=self.N), *self.bounds["acc_offset"])
        phase = self.particles["phase_code"]
        loads = np.zeros((self.N,3))
        base_lower = np.array([10.0, 8.0, 6.0])
        base_cut = np.array([20.0, 22.0, 18.0])
        base_lift = np.array([26.0, 18.0, 12.0])
        base_dump = np.array([12.0, 10.0, 8.0])
        loads[phase==0] = base_lower
        loads[phase==1] = base_cut
        loads[phase==2] = base_lift
        loads[phase==3] = base_dump
        loads = loads * self.particles["soil"][:,None]
        loads = np.clip(loads, 5.0, 32.0)
        P_prev = self.particles["P_pump"]
        P_new = P_prev + 0.5*(pump_target - P_prev)
        P_new = self._clamp(P_new, self.rail_lo, self.rail_hi)
        self.particles["P_pump"] = P_new
        soc = self.particles["soc"]
        soc += 0.002 * np.mean(sp_eff, axis=1) * (P_new/28.0) * (phase==0)
        soc -= 0.0025 * np.mean(sp_eff, axis=1) * (np.clip(loads[:,0],1.0,40.0)/26.0) * (phase==2)
        soc = self._clamp(soc, *self.bounds["soc"])
        self.particles["soc"] = soc
        self.t += 1

    def _predict_observation(self):
        press_ports = np.zeros((self.N, 6))
        phase = self.particles["phase_code"]
        base_lower = np.array([10.0, 8.0, 6.0])
        base_cut = np.array([20.0, 22.0, 18.0])
        base_lift = np.array([26.0, 18.0, 12.0])
        base_dump = np.array([12.0, 10.0, 8.0])
        loads = np.zeros((self.N,3))
        loads[phase==0] = base_lower
        loads[phase==1] = base_cut
        loads[phase==2] = base_lift
        loads[phase==3] = base_dump
        loads = loads * self.particles["soil"][:,None]
        loads = np.clip(loads, 5.0, 32.0)
        A = np.clip(loads + 2.0, self.rail_lo, self.rail_hi)
        B = np.clip(loads - 2.0, self.rail_lo, self.rail_hi)
        press_ports[:, :3] = A
        press_ports[:, 3:] = B
        press_ports += self.particles["pb_ports"]
        pump_press = self.particles["P_pump"] + self.particles["pb_pump"]
        oil_T = self.particles["oil_T"]
        angles = self.rng.normal(0.0, 0.5, size=(self.N,3)) + self.particles["enc_bias"][:,None]
        return angles, press_ports, pump_press, oil_T

    def update(self, obs):
        angles_obs = np.asarray(obs["angles"]).reshape(-1)
        press_ports_obs = np.asarray(obs["press_ports"]).reshape(-1)
        pump_press_obs = float(obs["pump_press"])
        oil_T_obs = float(obs["oil_T"])
        angles_pred, press_ports_pred, pump_press_pred, oil_T_pred = self._predict_observation()
        s_ang = self.noise["angles_sigma"]
        s_pr = self.noise["press_sigma_mpa"]
        s_pp = self.noise["pump_press_sigma_mpa"]
        s_T = self.noise["oilT_sigma_C"]
        da = angles_obs[None,:] - angles_pred
        dp = press_ports_obs[None,:] - press_ports_pred
        dP = pump_press_obs - pump_press_pred
        dT = oil_T_obs - oil_T_pred
        la = -0.5*np.sum((da/s_ang)**2, axis=1)
        lp = -0.5*np.sum((dp/s_pr)**2, axis=1)
        lP = -0.5*(dP/s_pp)**2
        lT = -0.5*(dT/s_T)**2
        loglik = la + lp + lP + lT
        m = np.max(loglik)
        w = self.weights * np.exp(loglik - m)
        w_sum = np.sum(w) + 1e-12
        w = w / w_sum
        self.weights = w
        ess = 1.0 / np.sum(w**2)
        resampled = False
        if ess < self.ess_threshold * self.N:
            self._resample()
            resampled = True
            ess = 1.0 / np.sum(self.weights**2)
        return {"ESS": float(ess), "resampled": bool(resampled)}

    def _resample(self):
        if self.resample_mode == "multinomial":
            idx = self.rng.choice(self.N, size=self.N, p=self.weights)
        else:
            positions = (np.arange(self.N) + self.rng.rand()) / self.N
            cumsum = np.cumsum(self.weights)
            idx = np.zeros(self.N, dtype=np.int64)
            i, j = 0, 0
            while i < self.N:
                if positions[i] < cumsum[j]:
                    idx[i] = j
                    i += 1
                else:
                    j += 1
        for k in list(self.particles.keys()):
            if isinstance(self.particles[k], np.ndarray):
                self.particles[k] = self.particles[k][idx].copy()
        self.prev_spools = self.prev_spools[idx].copy()
        self.weights.fill(1.0 / self.N)

    def estimate(self):
        if self.particles is None:
            return {}
        fields = ["soil","acc_offset","valve_gain_scale","cyl_leak_scale","pump_eta","pb_pump","enc_bias","oil_T","mu","soc","P_pump"]
        means = {}
        stds = {}
        w = self.weights
        for f in fields:
            x = self.particles[f]
            if x.ndim == 1:
                m = float(np.sum(w * x))
                v = float(np.sum(w * (x - m)**2))
                means[f] = m
                stds[f] = float(np.sqrt(max(0.0, v)))
        ports_bias_mean = np.sum(self.particles["pb_ports"] * w[:,None], axis=0)
        ports_bias_var = np.sum(((self.particles["pb_ports"] - ports_bias_mean)**2) * w[:,None], axis=0)
        means["pb_ports"] = ports_bias_mean.astype(float).tolist()
        stds["pb_ports"] = np.sqrt(np.maximum(ports_bias_var, 0.0)).astype(float).tolist()
        return {"mean": means, "std": stds}

    def log_step(self, step_idx, obs, diag=None, truth=None):
        est = self.estimate()
        row = {}
        row["step"] = int(step_idx)
        row["soil_mean"] = est["mean"]["soil"]
        row["soil_std"] = est["std"]["soil"]
        row["acc_offset_mean"] = est["mean"]["acc_offset"]
        row["acc_offset_std"] = est["std"]["acc_offset"]
        row["pump_eta_mean"] = est["mean"]["pump_eta"]
        row["valve_gain_mean"] = est["mean"]["valve_gain_scale"]
        row["cyl_leak_mean"] = est["mean"]["cyl_leak_scale"]
        row["enc_bias_mean"] = est["mean"]["enc_bias"]
        row["pb_pump_mean"] = est["mean"]["pb_pump"]
        row["oil_T_mean"] = est["mean"]["oil_T"]
        row["mu_mean"] = est["mean"]["mu"]
        row["soc_mean"] = est["mean"]["soc"]
        row["P_pump_mean"] = est["mean"]["P_pump"]
        if diag is not None:
            row["ESS"] = float(diag.get("ESS", np.nan))
            row["resampled"] = int(diag.get("resampled", False))
        if truth is not None:
            row["soil_true"] = float(truth.get("soil", np.nan))
            row["acc_offset_true"] = float(truth.get("acc_offset", np.nan))
            row["oil_T_true"] = float(truth.get("oil_T", np.nan))
            row["P_pump_true"] = float(truth.get("P_pump", np.nan))
            row["soc_true"] = float(truth.get("SoC", np.nan)) if "SoC" in truth else float(truth.get("soc", np.nan))
        self.log_rows.append(row)
        return row

    def to_dataframe(self):
        if not self.log_rows:
            return pd.DataFrame()
        return pd.DataFrame(self.log_rows)

if __name__ == "__main__":
    rng = np.random.RandomState(0)
    obs0 = {"angles": np.array([30.0, 0.0, -20.0], dtype=np.float32), "press_ports": np.array([18,17,16,14,13,12], dtype=np.float32), "pump_press": 20.0, "pump_rpm": 1800.0, "pump_disp": 0.6, "oil_T": 40.0}
    pf = BootstrapPF({"pf":{"num_particles":512,"ess_frac":0.5,"resample":"stratified"}, "env":{"dt":0.01,"rate_hz":100.0}})
    pf.init_from_obs(obs0)
    for k in range(1000):
        u = np.array([0.2,0.2,0.2,0.6], dtype=np.float32)
        pf.predict(u, 0.01)
        obsk = {"angles": obs0["angles"] + rng.randn(3)*0.1, "press_ports": obs0["press_ports"] + rng.randn(6)*0.2, "pump_press": 20.0 + rng.randn()*0.2, "pump_rpm": 1800.0, "pump_disp": 0.6, "oil_T": 40.0 + rng.randn()*0.2}
        diag = pf.update(obsk)
        pf.log_step(k, obsk, diag=diag)
    df = pf.to_dataframe()
    df.to_csv("belief_log.csv", index=False)
    m = pf.estimate()
    print(m["mean"]["soil"], m["mean"]["acc_offset"])
