# molmass/elements_descriptions.py

"""Set the description attribute on all elements."""

try:
    from .elements import ELEMENTS
except ImportError:
    from elements import ELEMENTS  # type: ignore[no-redef]

ELEMENTS['H'].description = (
    'Colourless, odourless gaseous chemical element. Lightest and most '
    'abundant element in the universe. Present in water and in all organic '
    'compounds. Chemically reacts with most elements. Discovered by Henry '
    'Cavendish in 1776.'
)
ELEMENTS['He'].description = (
    'Colourless, odourless gaseous nonmetallic element. Belongs to group '
    '18 of the periodic table. Lowest boiling point of all elements and '
    'can only be solidified under pressure. Chemically inert, no known '
    'compounds. Discovered in the solar spectrum in 1868 by Lockyer.'
)
ELEMENTS['Li'].description = (
    'Socket silvery metal. First member of group 1 of the periodic table. '
    'Lithium salts are used in psychomedicine.'
)
ELEMENTS['Be'].description = (
    'Grey metallic element of group 2 of the periodic table. Is toxic and '
    'can cause severe lung diseases and dermatitis. Shows high covalent '
    'character. It was isolated independently by F. Wohler and A.A. Bussy '
    'in 1828.'
)
ELEMENTS['B'].description = (
    'An element of group 13 of the periodic table. There are two '
    'allotropes, amorphous boron is a brown power, but metallic boron is '
    "black. The metallic form is hard (9.3 on Mohs' scale) and a bad "
    'conductor in room temperatures. It is never found free in nature. '
    'Boron-10 is used in nuclear reactor control rods and shields. It was '
    'discovered in 1808 by Sir Humphry Davy and by J.L. Gay-Lussac and '
    'L.J. Thenard.'
)
ELEMENTS['C'].description = (
    'Carbon is a member of group 14 of the periodic table. It has three '
    'allotropic forms of it, diamonds, graphite and fullerite. Carbon-14 '
    'is commonly used in radioactive dating. Carbon occurs in all organic '
    'life and is the basis of organic chemistry. Carbon has the '
    'interesting chemical property of being able to bond with itself, and '
    'a wide variety of other elements.'
)
ELEMENTS['N'].description = (
    'Colourless, gaseous element which belongs to group 15 of the '
    'periodic table. Constitutes ~78% of the atmosphere and is an '
    'essential part of the ecosystem. Nitrogen for industrial purposes is '
    'acquired by the fractional distillation of liquid air. Chemically '
    'inactive, reactive generally only at high temperatures or in '
    'electrical discharges. It was discovered in 1772 by D. Rutherford.'
)
ELEMENTS['O'].description = (
    'A colourless, odourless gaseous element belonging to group 16 of the '
    'periodic table. It is the most abundant element present in the '
    "earth's crust. It also makes up 20.8% of the Earth's atmosphere. For "
    'industrial purposes, it is separated from liquid air by fractional '
    'distillation. It is used in high temperature welding, and in '
    'breathing. It commonly comes in the form of Oxygen, but is found as '
    'Ozone in the upper atmosphere. It was discovered by Priestley in '
    '1774.'
)
ELEMENTS['F'].description = (
    'A poisonous pale yellow gaseous element belonging to group 17 of the '
    'periodic table (The halogens). It is the most chemically reactive and '
    'electronegative element. It is highly dangerous, causing severe '
    'chemical burns on contact with flesh. Fluorine was identified by '
    'Scheele in 1771 and first isolated by Moissan in 1886.'
)
ELEMENTS['Ne'].description = (
    'Colourless gaseous element of group 18 on the periodic table (noble '
    'gases). Neon occurs in the atmosphere, and comprises 0.0018% of the '
    'volume of the atmosphere. It has a distinct reddish glow when used in '
    'discharge tubes and neon based lamps. It forms almost no chemical '
    'compounds. Neon was discovered in 1898 by Sir William Ramsey and M.W. '
    'Travers.'
)
ELEMENTS['Na'].description = (
    'Soft silvery reactive element belonging to group 1 of the periodic '
    'table (alkali metals). It is highly reactive, oxidizing in air and '
    'reacting violently with water, forcing it to be kept under oil. It '
    'was first isolated by Humphrey Davy in 1807.'
)
ELEMENTS['Mg'].description = (
    'Silvery metallic element belonging to group 2 of the periodic table '
    '(alkaline-earth metals). It is essential for living organisms, and is '
    'used in a number of light alloys. Chemically very reactive, it forms '
    'a protective oxide coating when exposed to air and burns with an '
    'intense white flame. It also reacts with sulphur, nitrogen and the '
    'halogens. First isolated by Bussy in 1828.'
)
ELEMENTS['Al'].description = (
    'Silvery-white lustrous metallic element of group 3 of the periodic '
    'table. Highly reactive but protected by a thin transparent layer of '
    'the oxide which quickly forms in air. There are many alloys of '
    'aluminum, as well as a good number of industrial uses. Makes up 8.1% '
    "of the Earth's crust, by weight. Isolated in 1825 by H.C. Oersted."
)
ELEMENTS['Si'].description = (
    'Metalloid element belonging to group 14 of the periodic table. It is '
    "the second most abundant element in the Earth's crust, making up "
    '25.7% of it by weight. Chemically less reactive than carbon. First '
    'identified by Lavoisier in 1787 and first isolated in 1823 by '
    'Berzelius.'
)
ELEMENTS['P'].description = (
    'Non-metallic element belonging to group 15 of the periodic table. '
    'Has a multiple allotropic forms. Essential element for living '
    'organisms. It was discovered by Brandt in 1669.'
)
ELEMENTS['S'].description = (
    'Yellow, nonmetallic element belonging to group 16 of the periodic '
    'table. It is an essential element in living organisms, needed in the '
    'amino acids cysteine and methionine, and hence in many proteins. '
    'Absorbed by plants from the soil as sulphate ion.'
)
ELEMENTS['Cl'].description = (
    'Halogen element. Poisonous greenish-yellow gas. Occurs widely in '
    'nature as sodium chloride in seawater. Reacts directly with many '
    'elements and compounds, strong oxidizing agent. Discovered by Karl '
    'Scheele in 1774. Humphrey David confirmed it as an element in 1810.'
)
ELEMENTS['Ar'].description = (
    'Monatomic noble gas. Makes up 0.93% of the air. Colourless, '
    'odorless. Is inert and has no true compounds. Lord Rayleigh and Sir '
    'william Ramsey identified argon in 1894.'
)
ELEMENTS['K'].description = (
    'Soft silvery metallic element belonging to group 1 of the periodic '
    'table (alkali metals). Occurs naturally in seawater and a many '
    'minerals. Highly reactive, chemically, it resembles sodium in its '
    'behavior and compounds. Discovered by Sir Humphry Davy in 1807.'
)
ELEMENTS['Ca'].description = (
    'Soft grey metallic element belonging to group 2 of the periodic '
    'table. Used a reducing agent in the extraction of thorium, zirconium '
    'and uranium. Essential element for living organisms.'
)
ELEMENTS['Sc'].description = (
    'Rare soft silvery metallic element belonging to group 3 of the '
    'periodic table. There are ten isotopes, nine of which are radioactive '
    'and have short half-lives. Predicted in 1869 by Mendeleev, isolated '
    'by Nilson in 1879.'
)
ELEMENTS['Ti'].description = (
    'White metallic transition element. Occurs in numerous minerals. Used '
    'in strong, light corrosion-resistant alloys. Forms a passive oxide '
    'coating when exposed to air. First discovered by Gregor in 1789.'
)
ELEMENTS['V'].description = (
    'Soft and ductile, bright white metal. Good resistance to corrosion '
    'by alkalis, sulphuric and hydrochloric acid. It oxidizes readily '
    'about 933K. There are two naturally occurring isotopes of vanadium, '
    'and 5 radioisotopes, V-49 having the longest half-life at 337 days. '
    'Vanadium has nuclear applications, the foil is used in cladding '
    'titanium to steel, and vanadium-gallium tape is used to produce a '
    'superconductive magnet. Originally discovered by Andres Manuel del '
    'Rio of Mexico City in 1801. His discovery went unheeded, however, and '
    'in 1820, Nils Gabriel Sefstron of Sweden rediscovered it. Metallic '
    'vanadium was isolated by Henry Enfield Roscoe in 1867. The name '
    'vanadium comes from Vanadis, a goddess of Scandinavian mythology. '
    'Silvery-white metallic transition element. Vanadium is essential to '
    'Ascidians. Rats and chickens are also known to require it. Metal '
    'powder is a fire hazard, and vanadium compounds should be considered '
    'highly toxic. May cause lung cancer if inhaled.'
)
ELEMENTS['Cr'].description = (
    'Hard silvery transition element. Used in decorative electroplating. '
    'Discovered in 1797 by Vauquelin.'
)
ELEMENTS['Mn'].description = (
    'Grey brittle metallic transition element. Rather electropositive, '
    'combines with some non-metals when heated. Discovered in 1774 by '
    'Scheele.'
)
ELEMENTS['Fe'].description = (
    'Silvery malleable and ductile metallic transition element. Has nine '
    "isotopes and is the fourth most abundant element in the earth's "
    'crust. Required by living organisms as a trace element (used in '
    'hemoglobin in humans.) Quite reactive, oxidizes in moist air, '
    'displaces hydrogen from dilute acids and combines with nonmetallic '
    'elements.'
)
ELEMENTS['Co'].description = (
    'Light grey transition element. Some meteorites contain small amounts '
    'of metallic cobalt. Generally alloyed for use. Mammals require small '
    'amounts of cobalt salts. Cobalt-60, an artificially produced '
    'radioactive isotope of Cobalt is an important radioactive tracer and '
    'cancer-treatment agent. Discovered by G. Brandt in 1737.'
)
ELEMENTS['Ni'].description = (
    'Malleable ductile silvery metallic transition element. Discovered by '
    'A.F. Cronstedt in 1751.'
)
ELEMENTS['Cu'].description = (
    "Red-brown transition element. Known by the Romans as 'cuprum.' "
    'Extracted and used for thousands of years. Malleable, ductile and an '
    'excellent conductor of heat and electricity. When in moist '
    'conditions, a greenish layer forms on the outside.'
)
ELEMENTS['Zn'].description = (
    'Blue-white metallic element. Occurs in multiple compounds naturally. '
    'Five stable isotopes are six radioactive isotopes have been found. '
    'Chemically a reactive metal, combines with oxygen and other '
    'non-metals, reacts with dilute acids to release hydrogen.'
)
ELEMENTS['Ga'].description = (
    'Soft silvery metallic element, belongs to group 13 of the periodic '
    'table. The two stable isotopes are Ga-69 and Ga-71. Eight radioactive '
    'isotopes are known, all having short half-lives. Gallium Arsenide is '
    'used as a semiconductor. Corrodes most other metals by diffusing into '
    'their lattice. First identified by Francois Lecoq de Boisbaudran in '
    '1875.'
)
ELEMENTS['Ge'].description = (
    'Lustrous hard metalloid element, belongs to group 14 of the periodic '
    'table. Forms a large number of organometallic compounds. Predicted by '
    'Mendeleev in 1871, it was actually found in 1886 by Winkler.'
)
ELEMENTS['As'].description = (
    'Metalloid element of group 15. There are three allotropes, yellow, '
    'black, and grey. Reacts with halogens, concentrated oxidizing acids '
    'and hot alkalis. Albertus Magnus is believed to have been the first '
    'to isolate the element in 1250.'
)
ELEMENTS['Se'].description = (
    'Metalloid element, belongs to group 16 of the periodic table. '
    'Multiple allotropic forms exist. Chemically resembles sulphur. '
    'Discovered in 1817 by Jons J. Berzelius.'
)
ELEMENTS['Br'].description = (
    'Halogen element. Red volatile liquid at room temperature. Its '
    'reactivity is somewhere between chlorine and iodine. Harmful to human '
    'tissue in a liquid state, the vapour irritates eyes and throat. '
    'Discovered in 1826 by Antoine Balard.'
)
ELEMENTS['Kr'].description = (
    'Colorless gaseous element, belongs to the noble gases. Occurs in the '
    'air, 0.0001% by volume. It can be extracted from liquid air by '
    'fractional distillation. Generally not isolated, but used with other '
    'inert gases in fluorescent lamps. Five natural isotopes, and five '
    'radioactive isotopes. Kr-85, the most stable radioactive isotope, has '
    'a half-life of 10.76 years and is produced in fission reactors. '
    'Practically inert, though known to form compounds with Fluorine.'
)
ELEMENTS['Rb'].description = (
    'Soft silvery metallic element, belongs to group 1 of the periodic '
    'table. Rb-97, the naturally occurring isotope, is radioactive. It is '
    'highly reactive, with properties similar to other elements in group '
    '1, like igniting spontaneously in air. Discovered spectroscopically '
    'in 1861 by W. Bunsen and G.R. Kirchoff.'
)
ELEMENTS['Sr'].description = (
    'Soft yellowish metallic element, belongs to group 2 of the periodic '
    'table. Highly reactive chemically. Sr-90 is present in radioactive '
    'fallout and has a half-life of 28 years. Discovered in 1798 by '
    'Klaproth and Hope, isolated in 1808 by Humphry Davy.'
)
ELEMENTS['Y'].description = (
    'Silvery-grey metallic element of group 3 on the periodic table. '
    'Found in uranium ores. The only natural isotope is Y-89, there are 14 '
    'other artificial isotopes. Chemically resembles the lanthanoids. '
    'Stable in the air below 400 degrees, celsius. Discovered in 1828 by '
    'Friedrich Wohler.'
)
ELEMENTS['Zr'].description = (
    'Grey-white metallic transition element. Five natural isotopes and '
    'six radioactive isotopes are known. Used in nuclear reactors for a '
    'Neutron absorber. Discovered in 1789 by Martin Klaproth, isolated in '
    '1824 by Berzelius.'
)
ELEMENTS['Nb'].description = (
    'Soft, ductile grey-blue metallic transition element. Used in special '
    'steels and in welded joints to increase strength. Combines with '
    'halogens and oxidizes in air at 200 degrees celsius. Discovered by '
    'Charles Hatchett in 1801 and isolated by Blomstrand in 1864. Called '
    'Columbium originally.'
)
ELEMENTS['Mo'].description = (
    'Silvery-white, hard metallic transition element. It is chemically '
    'unreactive and is not affected by most acids. It oxidizes at high '
    'temperatures. There are seven natural isotopes, and four '
    'radioisotopes, Mo-93 being the most stable with a half-life of 3500 '
    'years. Molybdenum is used in almost all high-strength steels, it has '
    'nuclear applications, and is a catalyst in petroleum refining. '
    'Discovered in 1778 by Carl Welhelm Scheele of Sweden. Impure metal '
    'was prepared in 1782 by Peter Jacob Hjelm. The name comes from the '
    'Greek word molybdos which means lead. Trace amounts of molybdenum are '
    'required for all known forms of life. All molybdenum compounds should '
    'be considered highly toxic, and will also cause severe birth defects.'
)
ELEMENTS['Tc'].description = (
    'Radioactive metallic transition element. Can be detected in some '
    'stars and the fission products of uranium. First made by Perrier and '
    'Segre by bombarding molybdenum with deutrons, giving them Tc-97. '
    'Tc-99 is the most stable isotope with a half-life of 2.6*10^6 years. '
    'Sixteen isotopes are known. Organic technetium compounds are used in '
    'bone imaging. Chemical properties are intermediate between rhenium '
    'and manganese.'
)
ELEMENTS['Ru'].description = (
    'Hard white metallic transition element. Found with platinum, used as '
    'a catalyst in some platinum alloys. Dissolves in fused alkalis, and '
    'is not attacked by acids. Reacts with halogens and oxygen at high '
    'temperatures. Isolated in 1844 by K.K. Klaus.'
)
ELEMENTS['Rh'].description = (
    'Silvery white metallic transition element. Found with platinum and '
    'used in some platinum alloys. Not attacked by acids, dissolves only '
    'in aqua regia. Discovered in 1803 by W.H. Wollaston.'
)
ELEMENTS['Pd'].description = (
    'Soft white ductile transition element. Found with some copper and '
    'nickel ores. Does not react with oxygen at normal temperatures. '
    'Dissolves slowly in hydrochloric acid. Discovered in 1803 by W.H. '
    'Wollaston.'
)
ELEMENTS['Ag'].description = (
    'White lustrous soft metallic transition element. Found in both its '
    'elemental form and in minerals. Used in jewellery, tableware and so '
    'on. Less reactive than silver, chemically.'
)
ELEMENTS['Cd'].description = (
    'Soft bluish metal belonging to group 12 of the periodic table. '
    'Extremely toxic even in low concentrations. Chemically similar to '
    'zinc, but lends itself to more complex compounds. Discovered in 1817 '
    'by F. Stromeyer.'
)
ELEMENTS['In'].description = (
    'Soft silvery element belonging to group 13 of the periodic table. '
    'The most common natural isotope is In-115, which has a half-life of '
    '6*10^4 years. Five other radioisotopes exist. Discovered in 1863 by '
    'Reich and Richter.'
)
ELEMENTS['Sn'].description = (
    'Silvery malleable metallic element belonging to group 14 of the '
    'periodic table. Twenty-six isotopes are known, five of which are '
    'radioactive. Chemically reactive. Combines directly with chlorine and '
    'oxygen and displaces hydrogen from dilute acids.'
)
ELEMENTS['Sb'].description = (
    'Element of group 15. Multiple allotropic forms. The stable form of '
    'antimony is a blue-white metal. Yellow and black antimony are '
    'unstable non-metals. Used in flame-proofing, paints, ceramics, '
    'enamels, and rubber. Attacked by oxidizing acids and halogens. First '
    'reported by Tholden in 1450.'
)
ELEMENTS['Te'].description = (
    'Silvery metalloid element of group 16. Eight natural isotopes, nine '
    'radioactive isotopes. Used in semiconductors and to a degree in some '
    'steels. Chemistry is similar to Sulphur. Discovered in 1782 by Franz '
    'Miller.'
)
ELEMENTS['I'].description = (
    'Dark violet nonmetallic element, belongs to group 17 of the periodic '
    'table. Insoluble in water. Required as a trace element for living '
    'organisms. One stable isotope, I-127 exists, in addition to fourteen '
    'radioactive isotopes. Chemically the least reactive of the halogens, '
    'and the most electropositive metallic halogen. Discovered in 1812 by '
    'Courtois.'
)
ELEMENTS['Xe'].description = (
    'Colourless, odourless gas belonging to group 18 on the periodic '
    'table (the noble gases.) Nine natural isotopes and seven radioactive '
    'isotopes are known. Xenon was part of the first noble-gas compound '
    'synthesized. Several others involving Xenon have been found since '
    'then. Xenon was discovered by Ramsey and Travers in 1898.'
)
ELEMENTS['Cs'].description = (
    'Soft silvery-white metallic element belonging to group 1 of the '
    'periodic table. One of the three metals which are liquid at room '
    'temperature. Cs-133 is the natural, and only stable, isotope. Fifteen '
    'other radioisotopes exist. Caesium reacts explosively with cold '
    'water, and ice at temperatures above 157K. Caesium hydroxide is the '
    'strongest base known. Caesium is the most electropositive, most '
    'alkaline and has the least ionization potential of all the elements. '
    'Known uses include the basis of atomic clocks, catalyst for the '
    'hydrogenation of some organic compounds, and in photoelectric cells. '
    'Caesium was discovered by Gustav Kirchoff and Robert Bunsen in '
    'Germany in 1860 spectroscopically. Its identification was based upon '
    'the bright blue lines in its spectrum. The name comes from the latin '
    'word caesius, which means sky blue. Caesium should be considered '
    'highly toxic. Some of the radioisotopes are even more toxic.'
)
ELEMENTS['Ba'].description = (
    'Silvery-white reactive element, belonging to group 2 of the periodic '
    'table. Soluble barium compounds are extremely poisonous. Identified '
    'in 1774 by Karl Scheele and extracted in 1808 by Humphry Davy.'
)
ELEMENTS['La'].description = (
    '(From the Greek word lanthanein, to line hidden) Silvery metallic '
    'element belonging to group 3 of the periodic table and oft considered '
    'to be one of the lanthanoids. Found in some rare-earth minerals. '
    'Twenty-five natural isotopes exist. La-139 which is stable, and '
    'La-138 which has a half-life of 10^10 to 10^15 years. The other '
    'twenty-three isotopes are radioactive. It resembles the lanthanoids '
    'chemically. Lanthanum has a low to moderate level of toxicity, and '
    'should be handled with care. Discovered in 1839 by C.G. Mosander.'
)
ELEMENTS['Ce'].description = (
    'Silvery metallic element, belongs to the lanthanoids. Four natural '
    'isotopes exist, and fifteen radioactive isotopes have been '
    'identified. Used in some rare-earth alloys. The oxidized form is used '
    'in the glass industry. Discovered by Martin .H. Klaproth in 1803.'
)
ELEMENTS['Pr'].description = (
    'Soft silvery metallic element, belongs to the lanthanoids. Only '
    'natural isotope is Pr-141 which is not radioactive. Fourteen '
    'radioactive isotopes have been artificially produced. Used in '
    'rare-earth alloys. Discovered in 1885 by C.A. von Welsbach.'
)
ELEMENTS['Nd'].description = (
    'Soft bright silvery metallic element, belongs to the lanthanoids. '
    'Seven natural isotopes, Nd-144 being the only radioactive one with a '
    'half-life of 10^10 to 10^15 years. Six artificial radioisotopes have '
    'been produced. The metal is used in glass works to color class a '
    'shade of violet-purple and make it dichroic. One of the more reactive '
    'rare-earth metals, quickly reacts with air. Used in some rare-earth '
    "alloys. Neodymium is used to color the glass used in welder's "
    'glasses. Neodymium is also used in very powerful, permanent magnets '
    '(Nd2Fe14B). Discovered by Carl F. Auer von Welsbach in Austria in '
    '1885 by separating didymium into its elemental components '
    "Praseodymium and neodymium. The name comes from the Greek words 'neos "
    "didymos' which means 'new twin'. Neodymium should be considered "
    'highly toxic, however evidence would seem to show that it acts as '
    'little more than a skin and eye irritant. The dust however, presents '
    'a fire and explosion hazard.'
)
ELEMENTS['Pm'].description = (
    'Soft silvery metallic element, belongs to the lanthanoids. Pm-147, '
    'the only natural isotope, is radioactive and has a half-life of 252 '
    'years. Eighteen radioisotopes have been produced, but all have very '
    'short half-lives. Found only in nuclear decay waste. Pm-147 is of '
    'interest as a beta-decay source, however Pm-146 and Pm-148 have to be '
    'removed from it first, as they generate gamma radiation. Discovered '
    'by J.A. Marinsky, L.E. Glendenin and C.D. Coryell in 1947.'
)
ELEMENTS['Sm'].description = (
    'Soft silvery metallic element, belongs to the lanthanoids. Seven '
    'natural isotopes, Sm-147 is the only radioisotope, and has a '
    'half-life of 2.5*10^11 years. Used for making special alloys needed '
    'in the production of nuclear reactors. Also used as a neutron '
    'absorber. Small quantities of samarium oxide is used in special '
    'optical glasses. The largest use of the element is its ferromagnetic '
    'alloy which produces permanent magnets that are five times stronger '
    'than magnets produced by any other material. Discovered by Francois '
    'Lecoq de Boisbaudran in 1879.'
)
ELEMENTS['Eu'].description = (
    'Soft silvery metallic element belonging to the lanthanoids. Eu-151 '
    'and Eu-153 are the only two stable isotopes, both of which are '
    'Neutron absorbers. Discovered in 1889 by Sir William Crookes.'
)
ELEMENTS['Gd'].description = (
    'Soft silvery metallic element belonging to the lanthanoids. Seven '
    'natural, stable isotopes are known in addition to eleven artificial '
    'isotopes. Gd-155 and Gd-157 and the best neutron absorbers of all '
    'elements. Gadolinium compounds are used in electronics. Discovered by '
    'J.C.G Marignac in 1880.'
)
ELEMENTS['Tb'].description = (
    'Silvery metallic element belonging to the lanthanoids. Tb-159 is the '
    'only stable isotope, there are seventeen artificial isotopes. '
    'Discovered by G.G. Mosander in 1843.'
)
ELEMENTS['Dy'].description = (
    'Metallic with a bright silvery-white lustre. Dysprosium belongs to '
    'the lanthanoids. It is relatively stable in air at room temperatures, '
    'it will however dissolve in mineral acids, evolving hydrogen. It is '
    'found in from rare-earth minerals. There are seven natural isotopes '
    'of dysprosium, and eight radioisotopes, Dy-154 being the most stable '
    'with a half-life of 3*10^6 years. Dysprosium is used as a neutron '
    'absorber in nuclear fission reactions, and in compact disks. It was '
    'discovered by Paul Emile Lecoq de Boisbaudran in 1886 in France. Its '
    'name comes from the Greek word dysprositos, which means hard to '
    'obtain.'
)
ELEMENTS['Ho'].description = (
    'Relatively soft and malleable silvery-white metallic element, which '
    'is stable in dry air at room temperature. It oxidizes in moist air '
    'and at high temperatures. It belongs to the lanthanoids. A rare-earth '
    'metal, it is found in the minerals monazite and gadolinite. It '
    'possesses unusual magnetic properties. One natural isotope, Ho-165 '
    'exists, six radioisotopes exist, the most stable being Ho-163 with a '
    'half-life of 4570 years. Holmium is used in some metal alloys, it is '
    'also said to stimulate the metabolism. Discovered by Per Theodor '
    'Cleve and J.L. Soret in Switzerland in 1879. The name homium comes '
    'from the Greek word Holmia which means Sweden. While all holmium '
    'compounds should be considered highly toxic, initial evidence seems '
    "to indicate that they do not pose much danger. The metal's dust "
    'however, is a fire hazard.'
)
ELEMENTS['Er'].description = (
    'Soft silvery metallic element which belongs to the lanthanoids. Six '
    'natural isotopes that are stable. Twelve artificial isotopes are '
    'known. Used in nuclear technology as a neutron absorber. It is being '
    'investigated for other possible uses. Discovered by Carl G. Mosander '
    'in 1843.'
)
ELEMENTS['Tm'].description = (
    'Soft grey metallic element that belongs to the lanthanoids. One '
    'natural isotope exists, Tm-169, and seventeen artificial isotopes '
    'have been produced. No known uses for the element. Discovered in 1879 '
    'by Per Theodor Cleve.'
)
ELEMENTS['Yb'].description = (
    'Silvery metallic element of the lanthanoids. Seven natural isotopes '
    'and ten artificial isotopes are known. Used in certain steels. '
    'Discovered by J.D.G. Marignac in 1878.'
)
ELEMENTS['Lu'].description = (
    'Silvery-white rare-earth metal which is relatively stable in air. It '
    'happens to be the most expensive rare-earth metal. Its found with '
    'almost all rare-earth metals, but is very difficult to separate from '
    'other elements. Least abundant of all natural elements. Used in metal '
    'alloys, and as a catalyst in various processes. There are two '
    'natural, stable isotopes, and seven radioisotopes, the most stable '
    'being Lu-174 with a half-life of 3.3 years. The separation of '
    'lutetium from Ytterbium was described by Georges Urbain in 1907. It '
    'was discovered at approximately the same time by Carl Auer von '
    'Welsbach. The name comes from the Greek word lutetia which means '
    'Paris.'
)
ELEMENTS['Hf'].description = (
    'Silvery lustrous metallic transition element. Used in tungsten '
    'alloys in filaments and electrodes, also acts as a neutron absorber. '
    'First reported by Urbain in 1911, existence was finally established '
    'in 1923 by D. Coster, G.C. de Hevesy in 1923.'
)
ELEMENTS['Ta'].description = (
    'Heavy blue-grey metallic transition element. Ta-181 is a stable '
    'isotope, and Ta-180 is a radioactive isotope, with a half-life in '
    'excess of 10^7 years. Used in surgery as it is unreactive. Forms a '
    'passive oxide layer in air. Identified in 1802 by Ekeberg and '
    'isolated in 1820 by Jons J. Berzelius.'
)
ELEMENTS['W'].description = (
    'White or grey metallic transition element,formerly called Wolfram. '
    'Forms a protective oxide in air and can be oxidized at high '
    'temperature. First isolated by Jose and Fausto de Elhuyer in 1783.'
)
ELEMENTS['Re'].description = (
    'Silvery-white metallic transition element. Obtained as a by-product '
    'of molybdenum refinement. Rhenium-molybdenum alloys are '
    'superconducting.'
)
ELEMENTS['Os'].description = (
    'Hard blue-white metallic transition element. Found with platinum and '
    'used in some alloys with platinum and iridium.'
)
ELEMENTS['Ir'].description = (
    'Very hard and brittle, silvery metallic transition element. It has a '
    'yellowish cast to it. Salts of iridium are highly colored. It is the '
    'most corrosion resistant metal known, not attacked by any acid, but '
    'is attacked by molten salts. There are two natural isotopes of '
    'iridium, and 4 radioisotopes, the most stable being Ir-192 with a '
    'half-life of 73.83 days. Ir-192 decays into Platinum, while the other '
    'radioisotopes decay into Osmium. Iridium is used in high temperature '
    'apparatus, electrical contacts, and as a hardening agent for '
    'platinumpy. Discovered in 1803 by Smithson Tennant in England. The '
    'name comes from the Greek word iris, which means rainbow. Iridium '
    'metal is generally non-toxic due to its relative unreactivity, but '
    'iridium compounds should be considered highly toxic.'
)
ELEMENTS['Pt'].description = (
    'Attractive greyish-white metal. When pure, it is malleable and '
    'ductile. Does not oxidize in air, insoluble in hydrochloric and '
    'nitric acid. Corroded by halogens, cyandies, sulphur and alkalis. '
    'Hydrogen and Oxygen react explosively in the presence of platinumpy. '
    'There are six stable isotopes and three radioisotopes, the most '
    'stable being Pt-193 with a half-life of 60 years. Platinum is used in '
    'jewelry, laboratory equipment, electrical contacts, dentistry, and '
    'anti-pollution devices in cars. PtCl2(NH3)2 is used to treat some '
    'forms of cancer. Platinum-Cobalt alloys have magnetic properties. It '
    'is also used in the definition of the Standard Hydrogen Electrode. '
    'Discovered by Antonio de Ulloa in South America in 1735. The name '
    'comes from the Spanish word platina which means silver. Platinum '
    'metal is generally not a health concern due to its unreactivity, '
    'however platinum compounds should be considered highly toxic.'
)
ELEMENTS['Au'].description = (
    'Gold is gold colored. It is the most malleable and ductile metal '
    'known. There is only one stable isotope of gold, and five '
    'radioisotopes of gold, Au-195 being the most stable with a half-life '
    'of 186 days. Gold is used as a monetary standard, in jewelry, '
    'dentistry, electronics. Au-198 is used in treating cancer and some '
    'other medical conditions. Gold has been known to exist as far back as '
    '2600 BC. Gold comes from the Anglo-Saxon word gold. Its symbol, Au, '
    'comes from the Latin word aurum, which means gold. Gold is not '
    'particularly toxic, however it is known to cause damage to the liver '
    'and kidneys in some.'
)
ELEMENTS['Hg'].description = (
    'Heavy silvery liquid metallic element, belongs to the zinc group. '
    'Used in thermometers, barometers and other scientific apparatus. Less '
    'reactive than zinc and cadmium, does not displace hydrogen from '
    'acids. Forms a number of complexes and organomercury compounds.'
)
ELEMENTS['Tl'].description = (
    'Pure, unreacted thallium appears silvery-white and exhibits a '
    'metallic lustre. Upon reacting with air, it begins to turn '
    'bluish-grey and looks like lead. It is very malleable, and can be cut '
    'with a knife. There are two stable isotopes, and four radioisotopes, '
    'Tl-204 being the most stable with a half-life of 3.78 years. Thallium '
    "sulphate was used as a rodenticide. Thallium sulphine's conductivity "
    'changes with exposure to infrared light, this gives it a use in '
    'infrared detectors. Discovered by Sir William Crookes via '
    'spectroscopy. Its name comes from the Greek word thallos, which means '
    'green twig. Thallium and its compounds are toxic and can cause '
    'cancer.'
)
ELEMENTS['Pb'].description = (
    'Heavy dull grey ductile metallic element, belongs to group 14. Used '
    'in building construction, lead-place accumulators, bullets and shot, '
    'and is part of solder, pewter, bearing metals, type metals and '
    'fusible alloys.'
)
ELEMENTS['Bi'].description = (
    'White crystalline metal with a pink tinge, belongs to group 15. Most '
    'diamagnetic of all metals and has the lowest thermal conductivity of '
    'all the elements except mercury. Lead-free bismuth compounds are used '
    'in cosmetics and medical procedures. Burns in the air and produces a '
    'blue flame. In 1753, C.G. Junine first demonstrated that it was '
    'different from lead.'
)
ELEMENTS['Po'].description = (
    'Rare radioactive metallic element, belongs to group 16 of the '
    'periodic table. Over 30 known isotopes exist, the most of all '
    'elements. Po-209 has a half-life of 103 years. Possible uses in '
    'heating spacecraft. Discovered by Marie Curie in 1898 in a sample of '
    'pitchblende.'
)
ELEMENTS['At'].description = (
    'Radioactive halogen element. Occurs naturally from uranium and '
    'thorium decay. At least 20 known isotopes. At-210, the most stable, '
    'has a half-life of 8.3 hours. Synthesized by nuclear bombardment in '
    '1940 by D.R. Corson, K.R. MacKenzie and E. Segre at the University of '
    'California.'
)
ELEMENTS['Rn'].description = (
    'Colorless radioactive gaseous element, belongs to the noble gases. '
    'Of the twenty known isotopes, the most stable is Rn-222 with a '
    'half-life of 3.8 days. Formed by the radioactive decay of Radium-226. '
    'Radon itself decays into Polonium. Used in radiotherapy. As a noble '
    'gas, it is effectively inert, though radon fluoride has been '
    'synthesized. First isolated in 1908 by Ramsey and Gray.'
)
ELEMENTS['Fr'].description = (
    'Radioactive element, belongs to group 1 of the periodic table. Found '
    'in uranium and thorium ores. The 22 known isotopes are all '
    'radioactive, with the most stable being Fr-223. Its existence was '
    'confirmed in 1939 by Marguerite Perey.'
)
ELEMENTS['Ra'].description = (
    'Radioactive metallic transuranic element, belongs to group 2 of the '
    'periodic table. Most stable isotope, Ra-226 has a half-life of 1602 '
    'years, which decays into radon. Isolated from pitchblende in 1898 '
    'Marie and Pierre Curie.'
)
ELEMENTS['Ac'].description = (
    'Silvery radioactive metallic element, belongs to group 3 of the '
    'periodic table. The most stable isotope, Ac-227, has a half-life of '
    '217 years. Ac-228 (half-life of 6.13 hours) also occurs in nature. '
    'There are 22 other artificial isotopes, all radioactive and having '
    'very short half-lives. Chemistry similar to lanthanumpy. Used as a '
    'source of alpha particles. Discovered by A. Debierne in 1899.'
)
ELEMENTS['Th'].description = (
    'Grey radioactive metallic element. Belongs to actinoids. Found in '
    'monazite sand in Brazil, India and the US. Thorium-232 has a '
    'half-life of 1.39x10^10 years. Can be used as a nuclear fuel for '
    'breeder reactors. Thorium-232 captures slow Neutrons and breeds '
    'uranium-233. Discovered by Jons J. Berzelius in 1829.'
)
ELEMENTS['Pa'].description = (
    'Radioactive metallic element, belongs to the actinoids. The most '
    'stable isotope, Pa-231 has a half-life of 2.43*10^4 years. At least '
    '10 other radioactive isotopes are known. No practical applications '
    'are known. Discovered in 1917 by Lise Meitner and Otto Hahn.'
)
ELEMENTS['U'].description = (
    'White radioactive metallic element belonging to the actinoids. Three '
    'natural isotopes, U-238, U-235 and U-234. Uranium-235 is used as the '
    'fuel for nuclear reactors and weapons. Discovered by Martin H. '
    'Klaproth in 1789.'
)
ELEMENTS['Np'].description = (
    'Radioactive metallic transuranic element, belongs to the actinoids. '
    'Np-237, the most stable isotope, has a half-life of 2.2*10^6 years '
    'and is a by product of nuclear reactors. The other known isotopes '
    'have mass numbers 229 through 236, and 238 through 241. Np-236 has a '
    'half-life of 5*10^3 years. First produced by Edwin M. McMillan and '
    'P.H. Abelson in 1940.'
)
ELEMENTS['Pu'].description = (
    'Dense silvery radioactive metallic transuranic element, belongs to '
    'the actinoids. Pu-244 is the most stable isotope with a half-life of '
    '7.6*10^7 years. Thirteen isotopes are known. Pu-239 is the most '
    'important, it undergoes nuclear fission with slow neutrons and is '
    'hence important to nuclear weapons and reactors. Plutonium production '
    'is monitored down to the gram to prevent military misuse. First '
    'produced by Gleen T. Seaborg, Edwin M. McMillan, J.W. Kennedy and '
    'A.C. Wahl in 1940.'
)
ELEMENTS['Am'].description = (
    'Radioactive metallic transuranic element, belongs to the actinoids. '
    'Ten known isotopes. Am-243 is the most stable isotope, with a '
    'half-life of 7.95*10^3 years. Discovered by Glenn T. Seaborg and '
    'associates in 1945, it was obtained by bombarding Uranium-238 with '
    'alpha particles.'
)
ELEMENTS['Cm'].description = (
    'Radioactive metallic transuranic element. Belongs to actinoid '
    'series. Nine known isotopes, Cm-247 has a half-life of 1.64*10^7 '
    'years. First identified by Glenn T. Seaborg and associates in 1944, '
    'first produced by L.B. Werner and I. Perlman in 1947 by bombarding '
    'americium-241 with Neutrons. Named for Marie Curie.'
)
ELEMENTS['Bk'].description = (
    'Radioactive metallic transuranic element. Belongs to actinoid '
    'series. Eight known isotopes, the most common Bk-247, has a half-life '
    'of 1.4*10^3 years. First produced by Glenn T. Seaborg and associates '
    'in 1949 by bombarding americium-241 with alpha particles.'
)
ELEMENTS['Cf'].description = (
    'Radioactive metallic transuranic element. Belongs to actinoid '
    'series. Cf-251 has a half life of about 700 years. Nine isotopes are '
    'known. Cf-252 is an intense Neutron source, which makes it an intense '
    'Neutron source and gives it a use in Neutron activation analysis and '
    'a possible use as a radiation source in medicine. First produced by '
    'Glenn T. Seaborg and associates in 1950.'
)
ELEMENTS['Es'].description = (
    'Appearance is unknown, however it is most probably metallic and '
    'silver or gray in color. Radioactive metallic transuranic element '
    'belonging to the actinoids. Es-254 has the longest half-life of the '
    'eleven known isotopes at 270 days. First identified by Albert Ghiorso '
    'and associates in the debris of the 1952 hydrogen bomb explosion. In '
    '1961 the first microgram quantities of Es-232 were separated. While '
    'einsteinium never exists naturally, if a sufficient amount was '
    'assembled, it would pose a radiation hazard.'
)
ELEMENTS['Fm'].description = (
    'Radioactive metallic transuranic element, belongs to the actinoids. '
    'Ten known isotopes, most stable is Fm-257 with a half-life of 10 '
    'days. First identified by Albert Ghiorso and associates in the debris '
    'of the first hydrogen-bomb explosion in 1952.'
)
ELEMENTS['Md'].description = (
    'Radioactive metallic transuranic element. Belongs to the actinoid '
    'series. Only known isotope, Md-256 has a half-life of 1.3 hours. '
    'First identified by Glenn T. Seaborg, Albert Ghiorso and associates '
    'in 1955. Alternative name Unnilunium has been proposed. Named after '
    "the 'inventor' of the periodic table, Dmitri Mendeleev."
)
ELEMENTS['No'].description = (
    'Radioactive metallic transuranic element, belongs to the actinoids. '
    'Seven known isotopes exist, the most stable being No-254 with a '
    'half-life of 255 seconds. First identified with certainty by Albert '
    'Ghiorso and Glenn T. Seaborg in 1966. Unnilbium has been proposed as '
    'an alternative name.'
)
ELEMENTS['Lr'].description = (
    'Appearance unknown, however it is most likely silvery-white or grey '
    'and metallic. Lawrencium is a synthetic rare-earth metal. There are '
    'eight known radioisotopes, the most stable being Lr-262 with a '
    'half-life of 3.6 hours. Due to the short half-life of lawrencium, and '
    'its radioactivity, there are no known uses for it. Identified by '
    'Albert Ghiorso in 1961 at Berkeley. It was produced by bombarding '
    'californium with boron ions. The name is temporary IUPAC '
    'nomenclature, the origin of the name comes from Ernest O. Lawrence, '
    'the inventor of the cyclotron. If sufficient amounts of lawrencium '
    'were produced, it would pose a radiation hazard.'
)
ELEMENTS['Rf'].description = (
    'Radioactive transactinide element. Expected to have similar chemical '
    'properties to those displayed by hafnium. Rf-260 was discovered by '
    'the Joint Nuclear Research Institute at Dubna (U.S.S.R.) in 1964. '
    'Researchers at Berkeley discovered Unq-257 and Unq-258 in 1964.'
)
ELEMENTS['Db'].description = (
    'Also known as Hahnium, Ha. Radioactive transactinide element. '
    'Half-life of 1.6s. Discovered in 1970 by Berkeley researchers. So '
    'far, seven isotopes have been discovered.'
)
ELEMENTS['Sg'].description = (
    'Half-life of 0.9 +/- 0.2 s. Discovered by the Joint Institute for '
    'Nuclear Research at Dubna (U.S.S.R.) in June of 1974. Its existence '
    'was confirmed by the Lawrence Berkeley Laboratory and Livermore '
    'National Laboratory in September of 1974.'
)
ELEMENTS['Bh'].description = (
    'Radioactive transition metal. Half-life of approximately 1/500 s. '
    'Discovered by the Joint Institute for Nuclear Research at Dubna '
    '(U.S.S.R.) in 1976. Confirmed by West German physicists at the Heavy '
    'Ion Research Laboratory at Darmstadt.'
)
ELEMENTS['Hs'].description = (
    'Radioactive transition metal first synthesized in 1984 by a German '
    'research team led by Peter Armbruster and Gottfried Muenzenberg at '
    'the Institute for Heavy Ion Research at Darmstadt.'
)
ELEMENTS['Mt'].description = (
    'Half-life of approximately 5 ms. The creation of this element '
    'demonstrated that fusion techniques could indeed be used to make new, '
    'heavy nuclei. Made and identified by physicists of the Heavy Ion '
    'Research Laboratory, Darmstadt, West Germany in 1982. Named in honor '
    'of Lise Meitner, the Austrian physicist.'
)

if __name__ == '__main__':
    try:
        from .elements import word_wrap
    except ImportError:
        from elements import word_wrap  # type: ignore[no-redef]

    for ele in ELEMENTS:
        descr = word_wrap(
            ele.description, linelen=70, indent=0, joinstr=' "\n    "'
        )
        print(
            f'ELEMENTS[{ele.symbol!r}].description = ' f'(\n    "{descr}"\n)'
        )
