#!/bin/bash

# Check for number of arguments and assign variables
if [ "$#" -lt 1 ]; then
    echo "Usage: $0 <number_of_clients> [key_storage_directory]"
    exit 1
fi

CLIENT_COUNT="$1"
KEY_DIR="${2:-wireguard-keys}"  # Default to 'wireguard-keys' if no directory is specified

# Validate that the number of clients is a positive integer
if ! [[ "$CLIENT_COUNT" =~ ^[0-9]+$ ]]; then
    echo "Error: Number of clients must be a positive integer."
    exit 1
fi

# Create directory for keys if it does not exist
mkdir -p "$KEY_DIR"
cd "$KEY_DIR" || exit

# Generate server keys
echo "Generating server keys..."
mkdir -p server
cd server || exit
wg genkey | tee private-key-server | wg pubkey > public-key-server
cd .. || exit

# Create subdirectory for client keys
mkdir -p clients
cd clients || exit

# Generate client keys
echo "Generating client keys..."
for i in $(seq 1 "$CLIENT_COUNT"); do
    wg genkey | tee "private-key-client-$i" | wg pubkey > "public-key-client-$i"
    echo "Keys for client $i generated."
done

echo "All keys have been generated successfully in $KEY_DIR."
 
