import argparse
import base64
import json
import time
import os
import re

FORMAT = "cbor"

def serialize(data):
    """Serializes data into JSON or CBOR format based on the global FORMAT setting."""
    if FORMAT == "json":
        return json.dumps(data)
    elif FORMAT == "cbor":
        try:
            import cbor2
        except ImportError:
            raise ImportError("cbor2 is required for CBOR serialization. Install it using: pip install cbor2")
        return cbor2.dumps(data)
    else:
        raise ValueError(f"Unsupported format: {FORMAT}")

def output(data):
    """Outputs the data either as string or raw bytes to the standard output."""
    if FORMAT == "json":
        print(data)
    elif FORMAT == "cbor":
        import sys
        sys.stdout.buffer.write(data)
    else:
        raise ValueError(f"Unsupported format: {FORMAT}")

def get_key_hex(key_path):
    with open(key_path, 'r') as key_file:
        key = base64.b64decode(key_file.read()).hex()

    return key

def setup_event(private_key):
    """
    Create a JSON object that represents the setup event.

    :param private_key: The private key of the user in hex format.
    """

    event = {
	    "time": time.time_ns(),
	    "event": {
            "name": "pair",
            "type": "function",
            "args": [
                {
                    "name": "setup",
                    "type": "function",
                    "args": [
                        {"value": f"0x{private_key}", "type": "constant"},
                    ]
                },
                {"type": "function", "name": "pair"}
            ]
       }
    }
    return serialize(event)

def init_event(private_key, public_key):
    """
    Create a JSON object that represents the init event.

    :param private_key: The private key of the user in hex format.
    :param public_key: The public key of the user in hex format.
    """

    event = {
	    "time": time.time_ns(),
	    "event": {
            "name": "pair",
            "type": "function",
            "args": [
                {
                    "name": "init",
                    "type": "function",
                    "args": [
                        {"value": f"0x{private_key}", "type": "constant"},
                        {"value": f"0x{public_key}", "type": "constant"}
                    ]
                },
                {"type": "function", "name": "pair"}
            ]
       }
    }
    return serialize(event)

def get_first_n_public_keys(directory, n):
    # List all files in the directory
    all_files = os.listdir(directory)
    
    # Filter out public key files
    public_key_files = [f for f in all_files if re.match(r'public-key-client-\d+', f)]

    # Sort the files numerically
    public_key_files_sorted = sorted(public_key_files, key=lambda x: int(re.search(r'\d+', x).group()))
    
    # Get the first n public key files
    first_n_public_keys = public_key_files_sorted[:n]
    
    return first_n_public_keys

def main():
    global FORMAT

    parser = argparse.ArgumentParser()
    parser.add_argument("--format", help="Output format to use", default=FORMAT)
    parser.add_argument('private_key_path', help='Path to the private key file')
    parser.add_argument('public_key_dir', help='Path to the directory containing public key files')
    parser.add_argument('client_count', help='The number of clients to use')
    args = parser.parse_args()

    FORMAT = args.format

    private_key = get_key_hex(args.private_key_path)
    output(setup_event(private_key))

    for filename in get_first_n_public_keys(args.public_key_dir, int(args.client_count)):
        if filename.startswith("public-key"):
            public_key_path = os.path.join(args.public_key_dir, filename)
            if os.path.isfile(public_key_path):
                public_key = get_key_hex(public_key_path)
                output(init_event(private_key, public_key))

if __name__ == "__main__":
    main()
