import argparse
import base64
import json
import time
import os
import re

def get_key_hex(key_path):
    with open(key_path, 'r') as key_file:
        key = base64.b64decode(key_file.read()).hex()

    return key

def setup_event(private_key):
    """
    Create a JSON object that represents the setup event.

    :param private_key: The private key of the user in hex format.
    """

    event = {
	    "time": time.time_ns(),
	    "event": {
            "name": "pair",
            "type": "function",
            "args": [
                {
                    "name": "setup",
                    "type": "function",
                    "args": [
                        {"value": f"0x{private_key}", "type": "constant"},
                    ]
                },
                {"type": "function", "name": "pair"}
            ]
       }
    }
    return json.dumps(event)

def init_event(private_key, public_key):
    """
    Create a JSON object that represents the init event.

    :param private_key: The private key of the user in hex format.
    :param public_key: The public key of the user in hex format.
    """

    event = {
	    "time": time.time_ns(),
	    "event": {
            "name": "pair",
            "type": "function",
            "args": [
                {
                    "name": "init",
                    "type": "function",
                    "args": [
                        {"value": f"0x{private_key}", "type": "constant"},
                        {"value": f"0x{public_key}", "type": "constant"}
                    ]
                },
                {"type": "function", "name": "pair"}
            ]
       }
    }
    return json.dumps(event)

def get_first_n_public_keys(directory, n):
    # List all files in the directory
    all_files = os.listdir(directory)

    
    # Filter out public key files
    public_key_files = [f for f in all_files if re.match(r'public-key-client-\d+', f)]

    # Sort the files numerically
    public_key_files_sorted = sorted(public_key_files, key=lambda x: int(re.search(r'\d+', x).group()))
    
    # Get the first n public key files
    first_n_public_keys = public_key_files_sorted[:n]
    
    return first_n_public_keys

def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('private_key_path', help='Path to the private key file')
    parser.add_argument('public_key_path', help='Path to the public key file')
    args = parser.parse_args()

    private_key = get_key_hex(args.private_key_path)
    print(setup_event(private_key))

    public_key = get_key_hex(args.public_key_path)
    print(init_event(private_key, public_key))

if __name__ == "__main__":
    main()
