import os
import shutil
import time
import json
import re
from datetime import datetime
from pathlib import Path
from watchdog.observers import Observer
from watchdog.events import FileSystemEventHandler

LOG_DIR = "/tmp"
SPECMON_LOG = "specmon.log"
TIME_REWRITE = "time-rewrite.log"
TIME_MONITOR = "time-monitor.log"
SERVER_LOG = "server.log"
CLIENTS_LOG = "clients.log"
FLAGS = ["server-done", "clients-done"]

CLIENT_COUNT = os.getenv("CLIENT_COUNT", "unknwon")

def max_rss(output):
    # Search for the line containing the maximum RSS
    match = re.search(r'Maximum resident set size \(kbytes\): (\d+)', output)

    # If the line was found, return the number as an integer
    if match:
        return int(match.group(1))

    # If the line was not found, return None
    return None

def monitor_stats(stdout):
    stats = json.loads(stdout)

    # Durations are reported in nanoseconds, convert to milliseconds.
    # Memory usage is reported in kilobytes, convert to megabytes.
    return {
        "latency": stats["avg_latency_received"] / 1e6,
        "processing_time": stats["avg_processing_time"] / 1e6,
        "total_time": stats["total_time"] / 1e6,
    }

def print_results(case_study, num, runs, online, results):
    mode = 'online' if online else 'offline'

    print(f"Evaluation result ({case_study}, {mode}, nums={num}, runs={runs})")
    
    # Print latency stats if available
    if 'latency' in results:
        print(f"  Latency:                {results['latency']:10.2f} ms")
    else:
        print(f"  Latency:                {'N/A':>10}")
    
    if 'processing_time' in results:
        print(f"  Processing time:        {results['processing_time']:10.2f} ms")
    else:
        print(f"  Processing time:        {'N/A':>10}")
    
    if 'total_time' in results:
        print(f"  Total time:             {results['total_time']:10.2f} ms")
    else:
        print(f"  Total time:             {'N/A':>10}")
    
    if 'memory_usage_monitor' in results:
        print(f"  Memory usage (monitor): {results['memory_usage_monitor']:10.2f} MB")
    else:
        print(f"  Memory usage (monitor): {'N/A':>10}")
    
    if 'memory_usage_rewrite' in results:
        print(f"  Memory usage (rewrite): {results['memory_usage_rewrite']:10.2f} MB")
    else:
        print(f"  Memory usage (rewrite): {'N/A':>10}")
    
    print()

def process():
    stats = dict()

    # Process specmon log for monitor stats
    specmon_path = Path(LOG_DIR, SPECMON_LOG)
    if specmon_path.exists():
        try:
            with open(specmon_path, 'r') as log_file:
                res = "".join(log_file.readlines()[1:])
                stats.update(monitor_stats(res))
        except Exception as e:
            print(f"Warning: Error reading {SPECMON_LOG}: {e}")
    else:
        print(f"Warning: {SPECMON_LOG} not found, skipping monitor stats")

    # Process time monitor log for memory usage
    time_monitor_path = Path(LOG_DIR, TIME_MONITOR)
    if time_monitor_path.exists():
        try:
            with open(time_monitor_path, 'r') as log_file:
                memory_usage = max_rss(log_file.read())
                if memory_usage is not None:
                    stats["memory_usage_monitor"] = memory_usage / 1e3
        except Exception as e:
            print(f"Warning: Error reading {TIME_MONITOR}: {e}")
    else:
        print(f"Warning: {TIME_MONITOR} not found, skipping monitor memory usage")

    # Process time rewrite log for memory usage
    time_rewrite_path = Path(LOG_DIR, TIME_REWRITE)
    if time_rewrite_path.exists():
        try:
            with open(time_rewrite_path, 'r') as log_file:
                memory_usage = max_rss(log_file.read())
                if memory_usage is not None:
                    stats["memory_usage_rewrite"] = memory_usage / 1e3
        except Exception as e:
            print(f"Warning: Error reading {TIME_REWRITE}: {e}")
    else:
        print(f"Warning: {TIME_REWRITE} not found, skipping rewrite memory usage")

    return stats

def cleanup():
    out_dir = f"{LOG_DIR}/{CLIENT_COUNT}-{datetime.now().strftime('%Y-%m-%d_%H-%M-%S')}"
    os.makedirs(out_dir)

    # Static files to move
    files_to_move = [Path(LOG_DIR, CLIENTS_LOG),
                     Path(LOG_DIR, SERVER_LOG),
                     Path(LOG_DIR, SPECMON_LOG),
                     Path(LOG_DIR, TIME_REWRITE),
                     Path(LOG_DIR, TIME_MONITOR)]
    
    # Find all client log files (client-X.log pattern)
    log_dir_path = Path(LOG_DIR)
    client_logs = list(log_dir_path.glob("client-*.log"))
    files_to_move.extend(client_logs)
    
    # Find all pre-trace client log files (pre-trace-client-X.log pattern)
    pre_trace_client_logs = list(log_dir_path.glob("pre-trace-client-*.log"))
    files_to_move.extend(pre_trace_client_logs)
    
    # Add server pre-trace log
    server_pre_trace = Path(LOG_DIR, "server-pre-trace.log")
    if server_pre_trace.exists():
        files_to_move.append(server_pre_trace)
    
    files_to_delete = [Path(LOG_DIR, flag) for flag in FLAGS]

    for file_path in files_to_move:
       if os.path.isfile(file_path):
           try:
               shutil.move(file_path, out_dir)
           except Exception as e:
               print(f"Error moving {file_path}: {e}")

    for file_path in files_to_delete:
        if os.path.isfile(file_path):
            try:
                os.remove(file_path)
            except Exception as e:
                print(f"Error deleting {file_path}: {e}")


def wait_for_flags(flag_files):
    while True:
        if all(Path(LOG_DIR, flag).exists() for flag in flag_files):
            print_results("wireguard", CLIENT_COUNT, 1, True, process())
            break
        time.sleep(5)

if __name__ == "__main__":
    if not os.path.exists(LOG_DIR):
        os.makedirs(LOG_DIR)

    wait_for_flags(FLAGS)

    cleanup()
