#!/usr/bin/env bash

SERVER_PUBLIC_KEY="$(cat /etc/wireguard/server/public-key-server)"
SERVER_ENDPOINT="$(getent hosts wireguard-server | awk '{ print $1 }')":51820
SERVER_IP="10.0.0.1"

printf "IP Address of server container: %s\n" "$SERVER_ENDPOINT"

# Ensure /etc/iproute2 exists
mkdir -p /etc/iproute2

# Enable IP forwarding
sysctl -w net.ipv4.ip_forward=1

# Decrease GC impact
export GOGC=200

sleep 5

for i in $(seq 1 "$CLIENT_COUNT"); do
    CLIENT_INTERFACE="wg0-$i"
    CLIENT_IP="10.0.0.$((1 + i))"
    CLIENT_PORT=$((51820 + i))

    # Save pre-trace for clients
    python3 pre-trace-client.py "/etc/wireguard/clients/private-key-client-$i" "/etc/wireguard/server/public-key-server" > "/tmp/pre-trace-client-$i.log"

    # Set up the network interface
    GO_ANNOTATE_LOG_TARGET=/tmp/client-$i.log wireguard-go "$CLIENT_INTERFACE"
    ip -4 address add dev "$CLIENT_INTERFACE" "$CLIENT_IP"

    # Set private key for the client
    wg set "$CLIENT_INTERFACE" private-key "/etc/wireguard/clients/private-key-client-$i"

    # Set the listening port
    wg set "$CLIENT_INTERFACE" listen-port "$CLIENT_PORT"

    # Add the server as peer
    wg set "$CLIENT_INTERFACE" peer "$SERVER_PUBLIC_KEY" allowed-ips "$SERVER_IP/32" endpoint "$SERVER_ENDPOINT"

    # Bring the interface up
    ip -4 link set mtu 1420 up dev "$CLIENT_INTERFACE"

    # Create routing table for each client
    echo "$i $CLIENT_INTERFACE" >> /etc/iproute2/rt_tables
    ip -4 rule add from "$CLIENT_IP"/32 table "$CLIENT_INTERFACE"
    ip -4 route add "$SERVER_IP"/32 dev "$CLIENT_INTERFACE" table "$CLIENT_INTERFACE"

    echo "Configured client $i with IP $CLIENT_IP and port $CLIENT_PORT"
done

# Signal server that clients are ready
touch /tmp/clients-done

# Wait for the server to finish
while [ ! -f /tmp/server-done ]; do
    echo "Waiting for server to finish"
    sleep 1
done
