# SpecMon: Unifying Verification and Monitoring for WireGuard

This artifact contains a complete reproducible implementation of the WireGuard case study for SpecMon, demonstrating the creation and use of a **unified model** for both formal verification and runtime monitoring. This work addresses the critical problem of specification divergence between verification and monitoring models.

The artifact includes Docker images and files for the WireGuard case study, providing instructions to reproduce the unified modeling and runtime monitoring evaluation.

## Key Contribution

**Unified Specification**: This work creates a single Tamarin model of WireGuard that serves both for formal verification and runtime monitoring with SpecMon. This unification ensures that the specification proven correct is the same one used for monitoring, preventing dangerous divergence between verification and monitoring models.

**Bridging Abstraction Gap**: The unified model bridges the gap between the symbolic world of verification and the concrete world of implementation through careful use of Tamarin's preprocessor, allowing format strings for monitoring and abstract tuples for verification from the same source.

## Artifact Structure

The artifact is structured as follows:

```
├── LICENSE                      # AGPL-3.0 license
├── README.md                    # This file
├── images/                      # Pre-built Docker images as tar archives
│   ├── wireguard-server.tar.gz
│   ├── wireguard-clients.tar.gz
│   └── wireguard-eval.tar.gz
├── docker/                      # Docker configuration and evaluation setup
│   ├── Dockerfile               # Main container build file
│   ├── Dockerfile-eval          # Evaluation container
│   ├── docker-compose.yml       # Multi-container orchestration
│   ├── start-server.sh          # WireGuard server startup script
│   ├── start-clients.sh         # WireGuard clients startup script
│   ├── wireguard-eval.py        # Evaluation orchestration script
│   └── README-*.md              # Container-specific documentation
├── modules/patches/             # Source code patches for instrumentation
│   ├── blake2s.patch            # Blake2s crypto modifications
│   ├── chacha20poly1305.patch   # ChaCha20Poly1305 crypto modifications
│   ├── fix_internal_module.patch # Go module path fixes
│   └── ipv4_batch.patch         # Network batching optimizations
├── monitor/                     # SpecMon monitoring specifications
│   ├── go-rewrite.spthy         # Trace-rewriting model file
│   ├── wireguard.spthy          # Unified WireGuard model file
│   ├── pre-trace.py             # Script for generating setup events (server)
│   └── pre-trace-client.py      # Script for generating setup events (clients)
└── scripts/
    └── generate-keys.sh         # WireGuard key generation utility
```

## Requirements

The following dependencies are required for performing the evaluation:

- `bash`
- `docker` with the Compose plugin

Installation instructions for Docker and the Compose plugin are available in the Docker documentation: [Docker](https://docs.docker.com/engine/install), [Docker Compose](https://docs.docker.com/compose/install).

## Claims

This artifact demonstrates:
1. **Unified modeling**: A single Tamarin specification serves both formal verification and runtime monitoring
2. **Runtime monitoring**: Live monitoring of WireGuard protocol compliance using the unified model
3. **Violation detection**: Detection of deliberately introduced protocol violations
4. **Verification compatibility**: The unified model successfully verifies all original security properties

## Evaluation

### Hardware

The evaluation has been tested on modern multi-core systems with sufficient memory. Performance may vary based on hardware specifications and Docker overhead.

### Expectations

The evaluation demonstrates real-time protocol monitoring using the unified model. Results focus on protocol compliance verification rather than performance metrics.

**Expected Evaluation Time:** 30 seconds to 3 minutes depending on client count  
**Expected Verification Time:** ~100 seconds for all security properties  
**System Requirements:** 8GB+ RAM (recommended: 16GB for large client counts), multi-core CPU

### WireGuard Unified Model Evaluation (5 human-minutes + 10 compute-minutes)

The environment is provided as three Docker images managed by Docker Compose which include all the necessary tools and files. The Docker images are provided as compressed tar archives and can be loaded by executing:

```bash
docker load -i images/wireguard-server.tar.gz
docker load -i images/wireguard-clients.tar.gz
docker load -i images/wireguard-eval.tar.gz
```

The number of WireGuard clients can be specified when starting the containers:

```bash
CLIENT_COUNT=<number> docker compose up
```

in the `docker` directory containing the `docker-compose.yml` file.

This will run the evaluation with `<number>` of clients, demonstrating the unified model in action. At the end, the output should be similar to:

```
Evaluation result (wireguard, online, nums=10, runs=1)
  Latency:                     18.66 ms
  Processing time:              0.51 ms
  Total time:               36593.47 ms
  Memory usage (monitor):      25.61 MB
  Memory usage (rewrite):        N/A
Evaluation completed successfully
```

**Example:**
```bash
CLIENT_COUNT=10 docker compose up   # Default evaluation (~45 seconds)
```

**Alternative: Build from Source**

Alternatively, the images can be built from source and run:

```bash
cd docker
docker compose build
CLIENT_COUNT=<number> docker compose up
```

### Results

Results are automatically saved to `docker/logs/<timestamp>/`:
- `client-$i.log` - Individual client monitoring events (CBOR format, where $i is the client number)
- `time-monitor.log` - Performance information and timing data
- `specmon.log` - Monitoring statistics in JSON format

Each successful run demonstrates that the WireGuard implementation adheres to the unified model, with SpecMon accepting all protocol events generated during normal operation.

### Verifying Security Properties

The unified model can also be used for formal verification with Tamarin. To verify all security properties of the WireGuard model:

```bash
tamarin-prover --prove monitor/wireguard.spthy
```

This will verify all lemmas in the unified model, demonstrating that the same specification used for monitoring also provides formal security guarantees.

**Requirements:** Tamarin Prover installation. Instructions available at: [Tamarin Installation Guide](https://tamarin-prover.com/manual/master/book/002_installation.html)

## Unified Model Features

The unified model addresses key differences between verification and monitoring:

- **Message format**: Uses format strings for monitoring while maintaining tuple abstraction for verification
- **Key handling**: Loads concrete keys from files for monitoring vs. fresh names for verification
- **Protocol precision**: More precise than the original reference model with full key derivation and multi-message exchanges
- **Preprocessor selection**: Tamarin preprocessor selects appropriate rules and macros for each use case

## License

This artifact is licensed under AGPL-3.0 due to its incorporation of SpecMon (AGPL-3.0) as the core monitoring component. Other components include go-annotate (MIT), WireGuard-Go (MIT), and Go standard library (BSD-style). The AGPL-3.0 license ensures compatibility with SpecMon while properly incorporating the MIT-licensed components.

## Related Projects

- **SpecMon**: https://github.com/specmon/specmon
- **WireGuard Protocol**: https://www.wireguard.com/protocol/