function [data,info,status] = readWAVtoACC(varargin)
% readWAVtoACC reads WAV files generated by OmGui from AX3 or AX6 devices to Acceleration, Light & Temp
% Date: 2019-12-25 02:58, (c) 2019, Pasan Hettiarachchi, Uppsala University
% 
% varargin{1}=filename (required)
% varargin{2} if varargin{2}='info' only reads WAV metadata otherwise
% reads the complete file
% varargin{3} The start time as a datenum value
% varargin{4} The stop time as a datenum value
% Outputs:
% data: The structure similar to AX3_readfile (ACC, LIGHT and TEMP data)
% info: Structure somewhat similar to AX3_readfile info object
% status: 'OK' if everything is ok, an explanatory message (or exception) otherwise

% Copyright (c) 2019, Pasan Hettiarachchi .
% All rights reserved.
% 
% Redistribution and use in source and binary forms, with or without 
% modification, are permitted provided that the following conditions are met: 
% 1. Redistributions of source code must retain the above copyright notice, 
%    this list of conditions and the following disclaimer.
% 2. Redistributions in binary form must reproduce the above copyright notice, 
%    this list of conditions and the following disclaimer in the documentation 
%    and/or other materials provided with the distribution.
% 3. Neither the name of the copyright holder nor the names of its contributors
%    may be used to endorse or promote products derived from this software without
%    specific prior written permission.
% 
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
% POSSIBILITY OF SUCH DAMAGE. 
try
    % Assigns input arguments
    if nargin<2 || nargin>4
        status='Invalid arguments, Probably no filename or mode given';
        data=[];
        info=[];
        return;
    else
        filename=varargin{1};
        infoMode=strcmp(varargin{2},'info');
        if nargin==3
            start=varargin{3};
            stop=[];
        elseif nargin==4
            start= varargin{3};
            stop=varargin{4};
        else
            start= [];
            stop=[];
        end
    end
    
    
    
    % find audio file metadata
    signalInfo = audioinfo(filename);
    
    
    
    % find scaling factors
    %     x_scale=sscanf(string(regexp(signalInfo.Comment,'Scale-1: \d+','Match')),'Scale-1:%d');
    %     y_scale=sscanf(string(regexp(signalInfo.Comment,'Scale-2: \d+','Match')),'Scale-2:%d');
    %     z_scale=sscanf(string(regexp(signalInfo.Comment,'Scale-3: \d+','Match')),'Scale-3:%d');

    x_scale=sscanf(signalInfo.Comment(strfind(signalInfo.Comment,'Scale-1'):end),'Scale-1:%d');
    y_scale=sscanf(signalInfo.Comment(strfind(signalInfo.Comment,'Scale-2'):end),'Scale-2:%d');
    z_scale=sscanf(signalInfo.Comment(strfind(signalInfo.Comment,'Scale-3'):end),'Scale-3:%d');
    % find start time
    startTime=sscanf(signalInfo.Comment,'Time: %23c');
    startTime=datenum(startTime,'yyyy-mm-dd HH:MM:SS.FFF');
    %find deviceId and sessionId
    deviceID=sscanf(signalInfo.Artist,'Id: %d');
    sessionID=sscanf(signalInfo.Title,'Session: %d');
    
    %find the start and stop sample numbers to read
    Fs=signalInfo.SampleRate;
    
    time_ACC = startTime + (0:signalInfo.TotalSamples-1)'/Fs/86400;% The samples for light and temperature
    
    %information as a structure
    info.start.mtime=time_ACC(1);
    info.stop.mtime=time_ACC(end);
    info.start.str=datestr(time_ACC(1),'yyyy-mm-dd HH:MM:SS.FFF');
    info.stop.str=datestr(time_ACC(end),'yyyy-mm-dd HH:MM:SS.FFF');
    info.deviceId=deviceID;
    info.sessionId=sessionID;
    info.sampleRate=Fs;
    
    if infoMode
        data=[];
        status='OK';
        return;
    end
    
    % find start sample number from given start time
    if isempty(start)
        startSmpl=1;
    else
        startSmpl=find(time_ACC>=start,1,'first');
        if isempty(startSmpl)
          startSmpl=1;
        end
    end
    % find end sample number from given stop time
    if isempty(stop)
        endSmpl= signalInfo.TotalSamples;
    else
        endSmpl=find(time_ACC<=stop,1,'last');
        if isempty(endSmpl)
            endSmpl=signalInfo.TotalSamples;
        end
    end
    
    % read audio file with 'native' option. The data is read as it is
    [signal, ~]=audioread(filename,[startSmpl,endSmpl],'native');
    % find Acc values byapplying scaling factors and converting to double
    AccX=double(signal(:,1))* x_scale/32768;
    AccY=double(signal(:,2))* y_scale/32768;
    AccZ=double(signal(:,3))* z_scale/32768;
    
    %create the time axis for Acc
    time_ACC = time_ACC(startSmpl:endSmpl);% The samples for light and temperature
    %create the 1s samples auxiliary time axis for LIGHT, TEMP, BATT
    smpl_LT=1:Fs:length(signal)-2;
    time_LT=time_ACC(smpl_LT);
   
    LT_Ch=signalInfo.NumChannels;
    
    %light is just typecasted to double. The last 10 bits is data
    LIGHT=double(bitand(typecast(signal(smpl_LT+1,LT_Ch),'uint16'),0b0000001111111111));
    %TEMP is typecasted to double. The last 10 bits is data
    TEMP=double(bitand(typecast(signal(smpl_LT+2,LT_Ch),'uint16'),0b0000001111111111));
    % raw TEMP= (A1*TEMP+ B1)/C1 % conversion of temperature in C back to raw values
    A1= 1000; B1=50000; C1= 75000/256.0;
    % TEMP value is rescaled to Celcius scale
    TEMP=(TEMP*C1-B1)/A1;
    
    % data as a structure
    data.ACC=[time_ACC,AccX,AccY,AccZ];
    data.LIGHT=[time_LT,LIGHT];
    data.TEMP=[time_LT,TEMP];
    
    
    status='OK';
catch ME
    
    status=ME.message;
    data=[];
    info=[];
    %rethrow(ME);
end
end

