package de.upb.deblometer;
import com.google.gson.Gson;
import com.itextpdf.kernel.colors.ColorConstants;
import com.itextpdf.kernel.pdf.PdfDocument;
import com.itextpdf.layout.element.Cell;
import com.itextpdf.layout.properties.TextAlignment;

import java.io.*;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.*;

import com.itextpdf.kernel.pdf.PdfWriter;
import com.itextpdf.layout.Document;
import com.itextpdf.layout.element.Paragraph;
import com.itextpdf.layout.element.Table;

import static de.upb.deblometer.MemberExtraction.*;

public class Deblometer {
    public static void main(String[] args) {

        List<List<AnalysisResult>> LanguageFeatureResults = new ArrayList<>();

        //input paths for the analysis process
        String groundTruthFolder = "src/main/resources/GroundTruth";
        String debloatedJarsFolder = "src/main/resources/DebloatedJars";
        String pdfResultDestination = "src/main/resources/ProguardResults.pdf";
        String csvResultDestination = "src/main/resources/ProguardResults.csv";
        String ResultDestination = "src/main/resources/Results/";

        List<String> languageFeatureList = new ArrayList<>();
        List<String> jsonLocationList = new ArrayList<>();
        List<String> DebloatedJarsLocationList = new ArrayList<>();




        path_filename_extractor(groundTruthFolder, languageFeatureList, jsonLocationList);
        DebloatedJarsLocationList = findJarFiles(debloatedJarsFolder);

        Gson gson = new Gson();

        for (int i = 0; i < languageFeatureList.size(); i++) {
            String jsonLocation = jsonLocationList.get(i);

            try (FileReader reader = new FileReader(jsonLocation)) {
                GroundTruth gt = gson.fromJson(reader, GroundTruth.class);
                List<AnalysisResult> results = createAnalysisList(gt, DebloatedJarsLocationList.get(i));
                LanguageFeatureResults.add(results);
            } catch (Exception e) {
                System.err.println("Error processing index " + i + ": " + e.getMessage());

                List<AnalysisResult> crashedResults = new ArrayList<>();
                for (int j = 0; j < 4; j++) {
                    crashedResults.add(new AnalysisResult(true));
                }
                LanguageFeatureResults.add(crashedResults);
            }
        }

        generatePdf(pdfResultDestination, LanguageFeatureResults, languageFeatureList);
        generateCsv(csvResultDestination, LanguageFeatureResults, languageFeatureList);
        generateMemberCsv(ResultDestination, LanguageFeatureResults, languageFeatureList);
    }

    public static void path_filename_extractor(String groundTruthFolder, List<String> languageFeatureList, List<String> jsonLocationList){
        try {
            Files.walk(Paths.get(groundTruthFolder))
                    .filter(Files::isRegularFile)
                    .filter(path -> path.toString().endsWith(".json"))
                    .forEach(path -> {
                        String fileNameWithoutExtension = path.getFileName().toString().replaceAll("_json", "").replaceFirst("[.][^.]+$", "");
                        languageFeatureList.add(fileNameWithoutExtension);
                        jsonLocationList.add(path.toString());
                    });
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * Returns a list of all .jar file paths within the specified folder
     *
     * @param folderPath the folder to search for .jar files
     * @return a list of paths to .jar files in the folder
     */
    public static List<String> findJarFiles(String folderPath) {
        List<String> jarPaths = new ArrayList<>();
        File folder = new File(folderPath);

        if (!folder.exists() || !folder.isDirectory()) {
            System.out.println("Invalid folder path: " + folderPath);
            return jarPaths;
        }

        File[] files = folder.listFiles();
        if (files == null) return jarPaths;

        for (File file : files) {
            if (file.isFile() && file.getName().endsWith(".jar")) {
                jarPaths.add(file.getAbsolutePath());
            }
        }
        return jarPaths;
    }

    public static <T> int areAllElementsInList(List<T> groundTruth, List<T> targetList) {
        int notDebloatedElements = 0;

        for (T element1 : groundTruth) {
            if (targetList.contains(element1)) {
                notDebloatedElements++;
            }
        }

        return notDebloatedElements;
    }

    public static List<AnalysisResult> createAnalysisList(GroundTruth gt, String debloatedJarLocation){
        List<AnalysisResult> result = new ArrayList<>();

        //Creating analysisObject for Classes in debloated Jar
        List<String> groundTruthRequiredClassspecifier = new ArrayList<>();
        List<String> groundTruthBloatedClassspecifier = new ArrayList<>();
        gt.getLF().getClassGroundTruth().getRequired().forEach(n -> groundTruthRequiredClassspecifier.add(n.getAllAttributesAsString()));
        gt.getLF().getClassGroundTruth().getBloated().forEach(n -> groundTruthBloatedClassspecifier.add(n.getAllAttributesAsString()));

        List<String> debloatedClassSpecifiers = new ArrayList<>();
        extractClass(debloatedJarLocation).forEach(n -> debloatedClassSpecifiers.add(n.getAllAttributesAsString()));

        int presentRequiredClasses = areAllElementsInList(groundTruthRequiredClassspecifier,debloatedClassSpecifiers);
        int presentBloatedClasses = areAllElementsInList(groundTruthBloatedClassspecifier,debloatedClassSpecifiers);
        AnalysisResult classResult = new AnalysisResult("Class",groundTruthRequiredClassspecifier.size(),groundTruthBloatedClassspecifier.size(),presentRequiredClasses,presentBloatedClasses);
        result.add(classResult);

        //Creating analysisObject for Methods in debloated Jar
        List<String> groundTruthRequiredMethodspecifier = new ArrayList<>();
        List<String> groundTruthBloatedMethodspecifier = new ArrayList<>();
        gt.getLF().getMethodGroundTruth().getRequired().forEach(n -> groundTruthRequiredMethodspecifier.add(n.getAllAttributesAsString()));
        gt.getLF().getMethodGroundTruth().getBloated().forEach(n -> groundTruthBloatedMethodspecifier.add(n.getAllAttributesAsString()));

        List<String> debloatedMethodSpecifiers = new ArrayList<>();
        extractMethod(debloatedJarLocation).forEach(n -> debloatedMethodSpecifiers.add(n.getAllAttributesAsString()));

        //Creating analysisObject for Fields in debloated Jar
        int presentRequiredMethods = areAllElementsInList(groundTruthRequiredMethodspecifier,debloatedMethodSpecifiers);
        int presentBloatedMethods = areAllElementsInList(groundTruthBloatedMethodspecifier,debloatedMethodSpecifiers);
        AnalysisResult methodResult = new AnalysisResult("Method",groundTruthRequiredMethodspecifier.size(),groundTruthBloatedMethodspecifier.size(),presentRequiredMethods,presentBloatedMethods);
        result.add(methodResult);

        List<String> groundTruthRequiredFieldspecifier = new ArrayList<>();
        List<String> groundTruthBloatedFieldspecifier = new ArrayList<>();
        gt.getLF().getFieldGroundTruth().getRequired().forEach(n -> groundTruthRequiredFieldspecifier.add(n.getAllAttributesAsString()));
        gt.getLF().getFieldGroundTruth().getBloated().forEach(n -> groundTruthBloatedFieldspecifier.add(n.getAllAttributesAsString()));

        List<String> debloatedFieldSpecifiers = new ArrayList<>();
        extractField(debloatedJarLocation).forEach(n -> debloatedFieldSpecifiers.add(n.getAllAttributesAsString()));

        int presentRequiredFields = areAllElementsInList(groundTruthRequiredFieldspecifier,debloatedFieldSpecifiers);
        int presentBloatedFields = areAllElementsInList(groundTruthBloatedFieldspecifier,debloatedFieldSpecifiers);
        AnalysisResult fieldResult = new AnalysisResult("Field",groundTruthRequiredFieldspecifier.size(),groundTruthBloatedFieldspecifier.size(),presentRequiredFields,presentBloatedFields);
        result.add(fieldResult);

        //Creating analysisObject for summation over all members in debloated Jar
        int required_sum = groundTruthRequiredClassspecifier.size() + groundTruthRequiredMethodspecifier.size() + groundTruthRequiredFieldspecifier.size();
        int bloated_sum = groundTruthBloatedClassspecifier.size() + groundTruthBloatedMethodspecifier.size() + groundTruthBloatedFieldspecifier.size();
        int presentRequiredMembers = presentRequiredClasses + presentRequiredMethods + presentRequiredFields;
        int presentBloatedMembers = presentBloatedClasses + presentBloatedMethods + presentBloatedFields;
        AnalysisResult sum = new AnalysisResult("Sum",required_sum,bloated_sum,presentRequiredMembers,presentBloatedMembers);
        result.add(sum);

        return result;
    }

    public static void generatePdf(String dest, List<List<AnalysisResult>> languageFeatureResults, List<String> languageFeatureList) {
        try {
            PdfWriter writer = new PdfWriter(new FileOutputStream(dest));
            Document document = new Document(new PdfDocument(writer));

            for (int i = 0; i < languageFeatureResults.size(); i++) {
                List<AnalysisResult> analysisResults = languageFeatureResults.get(i);
                String title = "Language Feature : " + languageFeatureList.get(i);

                document.add(new Paragraph(title).setBold().setFontSize(14));

                Table table = new Table(new float[]{1, 1, 1, 1, 1, 1, 1, 1});
                table.setWidth(100);

                table.addHeaderCell("Type");
                table.addHeaderCell("Groundtruth: Required");
                table.addHeaderCell("Groundtruth: Bloated");
                table.addHeaderCell("Present Debloated");
                table.addHeaderCell("False Positives");
                table.addHeaderCell("False Negatives");
                table.addHeaderCell("Soundness");
                table.addHeaderCell("Precision");

                for (AnalysisResult result : analysisResults) {
                    if (result.isCrashed()) {
                        Cell crashCell = new Cell(1, 8)
                                .add(new Paragraph("⚠️ Analysis crashed — no data available"))
                                .setFontColor(ColorConstants.RED)
                                .setTextAlignment(TextAlignment.CENTER);
                        table.addCell(crashCell);
                    } else {
                        table.addCell(result.getType());
                        table.addCell(String.valueOf(result.getRequired()));
                        table.addCell(String.valueOf(result.getBloated()));
                        table.addCell(result.getNotDebloated_required() + "/" + result.getRequired());
                        table.addCell(String.valueOf(result.getFalse_positives()));
                        table.addCell(String.valueOf(result.getFalse_negatives()));
                        table.addCell(String.valueOf(result.getSoundness()));
                        table.addCell(String.valueOf(result.getPrecision()));
                    }
                }

                document.add(table);
                document.add(new Paragraph("\n"));
            }

            // Explanations
            document.add(new Paragraph(" Groundtruth Required: Amount of setted class members that shouldn’t be debloated"));
            document.add(new Paragraph(" Groundtruth Bloated: Amount of setted class members that should be debloated"));
            document.add(new Paragraph(" Present Debloated: Necessary class members still inside debloated Jar"));
            document.add(new Paragraph(" False Positives: Falsely not-debloated unrequired class members"));
            document.add(new Paragraph(" False Negatives: Falsely debloated required class members"));
            document.add(new Paragraph(" Soundness: Correctly not-debloated required class members (%)"));
            document.add(new Paragraph(" Precision: Correctly debloated class members over total considered (%)"));

            document.close();
            System.out.println("PDF created successfully at: " + dest);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public static void generateCsv(String dest, List<List<AnalysisResult>> languageFeatureResults, List<String> languageFeatureList) {
        try (FileWriter csvWriter = new FileWriter(dest)) {
            // Write header
            csvWriter.append("Language Feature Type,Type,Required,Bloated,Present Debloated,False Positives,False Negatives,Soundness,Precision\n");

            for (int i = 0; i < languageFeatureResults.size(); i++) {
                List<AnalysisResult> analysisResults = languageFeatureResults.get(i);
                String featureType = languageFeatureList.get(i);

                for (AnalysisResult result : analysisResults) {
                    StringBuilder row = new StringBuilder();
                    row.append(featureType).append(",");

                    if (result.isCrashed()) {
                        row.append("CRASHED, , , , , , ,\n");
                    } else {
                        row.append(result.getType()).append(",");
                        row.append(result.getRequired()).append(",");
                        row.append(result.getBloated()).append(",");
                        row.append(result.getNotDebloated_required()).append("/").append(result.getRequired()).append(",");
                        row.append(result.getFalse_positives()).append(",");
                        row.append(result.getFalse_negatives()).append(",");
                        row.append(result.getSoundness()).append(",");
                        row.append(result.getPrecision()).append("\n");
                    }

                    csvWriter.append(row.toString());
                }
            }

            csvWriter.flush();
            System.out.println("CSV created successfully at: " + dest);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public static void generateMemberCsv(String destFolder, List<List<AnalysisResult>> languageFeatureResults, List<String> languageFeatureList) {
        // Types you expect to handle
        String[] types = {"Class", "Method", "Field", "Sum"};
        Map<String, FileWriter> writers = new HashMap<>();

        try {
            // Initialize a writer for each type
            for (String type : types) {
                FileWriter writer = new FileWriter(destFolder + File.separator + type + ".csv");
                writer.append("Language Feature Type,Type,Required,Bloated,Present Debloated,False Positives,False Negatives,Soundness,Precision\n");
                writers.put(type, writer);
            }

            for (int i = 0; i < languageFeatureResults.size(); i++) {
                List<AnalysisResult> analysisResults = languageFeatureResults.get(i);
                String featureType = languageFeatureList.get(i);

                for (AnalysisResult result : analysisResults) {
                    String resultType = result.getType();
                    FileWriter writer = writers.get(resultType);

                    if (writer == null) {
                        System.err.println("Unknown type: " + resultType + ". Skipping entry.");
                        continue;
                    }

                    StringBuilder row = new StringBuilder();
                    row.append(featureType).append(",");

                    if (result.isCrashed()) {
                        row.append("CRASHED, , , , , , ,\n");
                    } else {
                        row.append(resultType).append(",");
                        row.append(result.getRequired()).append(",");
                        row.append(result.getBloated()).append(",");
                        row.append(result.getNotDebloated_required()).append("/").append(result.getRequired()).append(",");
                        row.append(result.getFalse_positives()).append(",");
                        row.append(result.getFalse_negatives()).append(",");
                        row.append(result.getSoundness()).append(",");
                        row.append(result.getPrecision()).append("\n");
                    }

                    writer.append(row.toString());
                }
            }

            for (FileWriter writer : writers.values()) {
                writer.flush();
                writer.close();
            }

            System.out.println("CSV files created successfully in folder: " + destFolder);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

}