library(CMDdemux)
library(Seurat)
library(scran)
library(scater)
library(deMULTIplex2)
library(demuxmix)
library(cellhashR)
library(DropletUtils)
library(stringr)
library(tidyverse)
library(clValid)

source(~/bench.R)
load("~/otb2.hash.count.rdata")
load("~/otb2.gex.count.rdata")
otb2.donor <- read.csv("~/ovarian_tumour_donors_b2.csv")

# 1. Vireo
otb2.donor.assign <- otb2.donor$genetic_donor
otb2.donor.assign <- gsub(" ", "-", otb2.donor.assign)
names(otb2.donor.assign) <- gsub("-", ".", otb2.donor$X)
otb2.demux.result <- data.frame("Vireo" = otb2.donor.assign[colnames(otb2.hash.count)])

# 2. CMDdemux
otb2.clr.norm <- LocalCLRNorm(otb2.hash.count)
otb2.kmed.cl <- KmedCluster(otb2.clr.norm) 
otb2.cl.dist <- EuclideanClusterDist(otb2.clr.norm, otb2.kmed.cl)
otb2.noncore <- DefineNonCore(otb2.cl.dist, otb2.kmed.cl)
otb2.cluster.assign <- LabelClusterHTO(otb2.clr.norm, otb2.kmed.cl, otb2.noncore, "medoids")
otb2.md.mat <- CalculateMD(otb2.clr.norm, otb2.noncore, otb2.kmed.cl, otb2.cluster.assign)
otb2.outlier.assign <- AssignOutlierDrop(otb2.md.mat, md_auto = TRUE)
otb2.cmddemux.assign <- CMDdemuxClass(otb2.md.mat, otb2.hash.count, otb2.outlier.assign,  TRUE, otb2.gex.count, 9, 10)
otb2.demux.result$CMDdemux <- otb2.cmddemux.assign$demux_global_class

# 3. HTOdemux
ot.obj <- CreateSeuratObject(counts = otb2.hash.count)
ot.obj[["HTO"]] <- CreateAssayObject(counts = otb2.hash.count)
ot.obj <- NormalizeData(ot.obj, assay = "HTO", normalization.method = "CLR")
ot.obj <- HTODemux(ot.obj, assay = "HTO", positive.quantile = 0.99)
otb2.demux.result$HTODemux <- ot.obj$hash.ID[rownames(otb2.demux.result)]

# 4. GMM-Demux
# Prepare input data
otb2.gmm.input <- t(otb2.hash.count)
write.csv(otb2.gmm.input, "~/OT.gmm.input.csv", quote=F)
# Command: GMM-demux -c ~/OT.gmm.input.csv OT-E,OT-F,OT-G,OT-H -x OT-E,OT-F,OT-G,OT-H -f .
otb2.gmm.output <- read.csv("~/GMM_full.csv")
otb2.gmm.config <- read.table("~/GMM_full.config", header = FALSE, sep = ",")
otb2.gmm.demux <- GMM_demux_class(otb2.gmm.output, otb2.gmm.config, otb2.hash.count)
otb2.demux.result$`GMM-Demux` <- otb2.gmm.demux

# 5. deMULTIplex2
otb2.demultiplex2.output <- demultiplexTags(otb2.gmm.input, plot.diagnostics = FALSE, seed = 2024)
otb2.demultiplex2.assign <- deMULTIplex2_class(otb2.demultiplex2.output)
otb2.demux.result$deMULTIplex2 <- otb2.demultiplex2.assign

# 6. demuxEM
otb2.hash.write <- as.data.frame(otb2.hash.count) %>% rownames_to_column('Antibody')
write.csv(otb2.hash.write, "~/otb2.hash.write.csv", quote=F)
write10xCounts("~/otb2.gex.h5", otb2.gex.count, version='3')
# demuxEM -p 8 --random-state 2024 ~/otb2.gex.h5 ~/otb2.hash.write.csv OT_demuxEM
otb2.demuxEM.out1 <- read.table(file = "~/OT.demuxEM.demux.txt", header = TRUE)
otb2.demuxEM.out2 <- read.table(file = "~/OT.demuxEM.assign.txt", sep = "\t", header = TRUE)
otb2.demuxEM.assign <- demuxEM_class(otb2.demuxEM.out1, otb2.demuxEM.out2, otb2.hash.count, TRUE)
otb2.demux.result$demuxEM <- otb2.demuxEM.assign[rownames(otb2.demux.result)]

# 7. demuxmix
otb2.gex.genes <- colSums(otb2.gex.count > 0)
otb2.demuxmix.model <- demuxmix(hto = otb2.hash.count, rna = otb2.gex.genes)
otb2.demuxmix.labels <- dmmClassify(otb2.demuxmix.model)
otb2.demuxmix.assign <- demuxmix_class(otb2.demuxmix.labels)
otb2.demux.result$demuxmix <- otb2.demuxmix.assign[rownames(otb2.demux.result)]

# 8. hashedDrops
otb2.hasheddrops.output <- hashedDrops(otb2.hash.count)
otb2.hasheddrops.assign <- hashedDrops_class(otb2.hasheddrops.output, otb2.hash.count)
otb2.demux.result$hashedDrops <- otb2.hasheddrops.assign[rownames(otb2.demux.result)]

# 9. BFF
otb2.bff.output <- GenerateCellHashingCalls(barcodeMatrix = otb2.hash.count, methods = c("bff_raw", "bff_cluster"))
otb2.demux.result$BFF_raw <- otb2.bff.output$bff_raw
otb2.demux.result$BFF_cluster <- otb2.bff.output$bff_cluster

# Dimensional reduction for visualization
otb2.sce <- SingleCellExperiment(assays = list(hto = otb2.hash.count, clr = otb2.clr.norm))
otb2.sce <- runTSNE(otb2.sce,exprs_values = "clr")
otb2.sce <- runUMAP(otb2.sce,exprs_values = "clr")

# Concordance of each method with the ground truth
otb2.label.levels <- c(rownames(otb2.hash.count), "Doublet", "Negative", "Uncertain")
otb2.bench.concord <- ConfusionMatPlot(otb2.demux.result, "Vireo", otb2.label.levels)

# Overall concordance
otb2.ov.concord <- ConcordanceDF(otb2.demux.result, "Vireo")

# Precision, recall, F, MCC
otb2.evaluation <- GTStats(otb2.demux.result, "Vireo", otb2.label.levels[which(!otb2.label.levels %in% c("Uncertain", "Negative"))])

# Average precision, recall, F, MCC
otb2.evaluation.avg <- GTStatsAvgGTStatsBar(otb2.demux.result, "Vireo", otb2.label.levels[which(!otb2.label.levels %in% "Uncertain")], "macro")

# Library size
otb2.gex.lib <- log(colSums(otb2.gex.count))
otb2.hto.lib <- log(colSums(otb2.hash.count)+1)

# Summary of HTO library size ratio
otb2.htoratio.sum <- LibRatioSum(otb2.hto.lib, otb2.demux.result)

# Summary of gex library size ratio
otb2.gexratio.sum <- LibRatioSum(otb2.gex.lib, otb2.demux.result)
