%%%%%%%%%%%%%%%%%%%%%%%%%%% August 2025 %%%%%%%%%%%%%%%%%%%%%%%%%%

% Yanchao Jiang, Pierluigi Poggiolini, Politecnico di Torino
%
% Copyright (C) 2025 The Authors - all rights reserved
% Creative Commons Attribution 4.0 International
%
% acknowledgements for technical contributions to:
% Yifeng Gao, Soochow University
% Tianchun Zhu, Politecnico di Torino
%
% acknowledgements for sponsorship to CISCO Systems and for
% advice and guidance to Fabrizio Forghieri and Stefano Piciaccia

% PCFM implementation in UWB systems with powerful backward Raman
% amplification. Currently, we consider SCI and XCI contributions.
% The program is provided in a multispan system with both NLI and
% ASE (Raman + DFA) noises.

% PCFM reference:
% [1] Pierluigi Poggiolini and Yanchao Jiang, “Recent Advances
% in Real-Time Models for UWB Transmission Systems,” OFC2025,
% Tu3K.2, San Francisco, US, 2025.


clear;close all;clc
% start with adding paths
main_path = './';
addpath(main_path)
% windows
addpath(strcat(main_path,'/functions_windows'))
% linux
% addpath(strcat(main_path,'/functions_linux'))
addpath(strcat(main_path,'/parameters'))

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% step 1: set system parameters
parameters_struct = PCFM_parameters;

% step 2: calculate spatial power profiles(SPPs)
% and fit by 9th-degree polynomial
% Two main fields 'SPPs_z' and 'fitting_coeffiis' are added:
% 'SPPs_z' is a 1*Nspan cell:
%   Last column: z [m], containing the location along one span
%   other columns: signal SPPs
% 'fitting_coeffiis' is a 3D matrix with the following
% dimensions:
%   rows: spans
%   colums: channels
%   pages:  polynomial degrees from 0 to 9
parameters_struct = PCFM_SPP(parameters_struct);
% Note: you can also comment it out and directly provide
% these new fields.
% If the algorithm fails during stress testing,
% please solve it using a conventional solver,
% such as bvp4c.
if isfield(parameters_struct, 'SPPs_z') == 0
    parameters_struct.SPPs_z = input('Entering: ');
    parameters_struct.fitting_coeffi = input('Entering: ');
end

% step 3: calculate NLI power
% SCI contribution
PSCI_PCFM = SCI_calculator(parameters_struct);
% XCI contribution
PXCI_PCFM = XCI_calculator(parameters_struct);
% NLI power
PNLI_PCFM = PSCI_PCFM + PXCI_PCFM;

% step 4: calculate ASE power
[PASE_RA, PASE_DFA] = ASE_calculator(parameters_struct);
PASE = PASE_DFA+PASE_RA;

% GSNRs
% the signal power
P_CH_start = parameters_struct.P_signal_W(2:end,:);
% OSNRs
OSNR_DFA_dB = 10*log10(P_CH_start./PASE_DFA);
OSNR_RA_dB = 10*log10(P_CH_start./PASE_RA);
OSNR_dB = 10*log10(P_CH_start./PASE);
% nonlinear GSNRs
GSNR_SCI_dB = 10*log10(P_CH_start./PSCI_PCFM);
GSNR_XCI_dB = 10*log10(P_CH_start./PXCI_PCFM);
GSNR_NLI_dB = 10*log10(P_CH_start./PNLI_PCFM);
% total GSNR
GSNR_dB = 10*log10(P_CH_start./(PASE+PNLI_PCFM));

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% figures
close all;

% 1. 3D SPPs
% channels per band
Nch_per_band = parameters_struct.Nch_per_band;
Nch_Lband = Nch_per_band(1);
Nch_Cband = Nch_per_band(2);
Nch_Sband = Nch_per_band(3);
Nch_signal = sum(Nch_per_band);

% span selection: which span number the user
% wants the graphical output to be about
while true
    % Create a dialog box for user input
    prompt = {'Enter span (1 to 10):'};
    dlgtitle = 'Input Parameter';
    dims = [1 35];
    definput = {'1'};
    answer = inputdlg(prompt, dlgtitle, dims, definput);

    % Check if user cancelled
    if isempty(answer)
        fprintf('Input cancelled. Using default span: 1\n');
        span_selected = 1; % Set default or handle as needed
        break;
    end

    % Convert and validate input
    span_selected = str2double(answer{1});
    if ~isnan(span_selected) && span_selected >= 1 && span_selected <= 10
        break; % Valid input, exit loop
    else
        % Show warning dialog for invalid input
        uiwait(warndlg('Span must be a number between 1 and 10.', 'Invalid Input'));
    end
end

% Display the selected span
fprintf('Selected span: %d\n', span_selected);

% SPPs and z
SPPs = 10*log10(1e3*...
    parameters_struct.SPPs_z{span_selected}(:,1:Nch_signal));
z = parameters_struct.SPPs_z{span_selected}(:,end) * 1e-3;
% WDM center frequencies in THz
freq_grid = parameters_struct.center_signal*1e-12;

[xx, yy] = meshgrid(freq_grid, z);
for i_band = 1:numel(Nch_per_band)
    if i_band == 1
        nch = 1:Nch_Lband;
    elseif i_band == 2
        nch = Nch_Lband+1:Nch_Lband+Nch_Cband;
    elseif i_band == 3
        nch = Nch_Lband+Nch_Cband+1:Nch_signal;
    end
    figure(1);
    mesh(xx(:,nch),yy(:,nch),SPPs(:,nch))
    hold on
end
xlabel('optical frequency [THz]', 'Rotation', 13,...
    'VerticalAlignment','baseline')
ylabel('distance into span [km]', 'Units','normalized', ...
    'Position',[0.2 +0.03 0],'Rotation', -25);
zlabel('power per channel [dBm]')
set(gca, 'YDir','reverse')
grid on; box on
set(gca,'FontSize',16)
set(gcf,'position',[100,100,850,650])
xlim([184, 204]); xticks(184:4:204)
yticks(0:20:100)
title("spatial power profiles of all channels for span " + string(span_selected));


% 2.1 launch power per channel
P_launch = parameters_struct.P_signal_W(span_selected,:);
figure(2)
set(gcf,'position',[100,100,1700,800])

subplot(221)
plot(freq_grid, 10*log10(P_launch*1e3),...
    'k.','MarkerSize',8)
xlabel('optical frequency [THz]')
ylabel('power [dBm]')
grid on
set(gca,'FontSize',16)
xlim([184 204]);xticks(184:4:204)
title('launch power per channel')

% 2.2-2.4 GSNR 
figure(2); hold on
for i_band = 1:numel(Nch_per_band)
    if i_band == 1
        nch = 1:Nch_Lband;
    elseif i_band == 2
        nch = Nch_Lband+1:Nch_Lband+Nch_Cband;
    elseif i_band == 3
        nch = Nch_Lband+Nch_Cband+1:Nch_signal;
    end

    % OSNR
    subplot(222)
    plot(freq_grid(nch), OSNR_DFA_dB(span_selected,nch),'g--','LineWidth',2)
    hold on
    plot(freq_grid(nch), OSNR_RA_dB(span_selected,nch),'g:','LineWidth',2)
    plot(freq_grid(nch), OSNR_dB(span_selected,nch),'g','LineWidth',2)

    % GSNR_NLI
    subplot(223)
    plot(freq_grid(nch), GSNR_SCI_dB(span_selected,nch),'r--','LineWidth',2)
    hold on
    plot(freq_grid(nch), GSNR_XCI_dB(span_selected,nch),'r:','LineWidth',2)
    plot(freq_grid(nch), GSNR_NLI_dB(span_selected,nch),'r','LineWidth',2)

    % GSNR
    subplot(224)
    plot(freq_grid(nch), OSNR_dB(span_selected,nch),'g','LineWidth',2)
    hold on
    plot(freq_grid(nch), GSNR_NLI_dB(span_selected,nch),'r','LineWidth',2)
    plot(freq_grid(nch), GSNR_dB(span_selected,nch),'b','LineWidth',2)
end
subplot(222)
xlabel('optical frequency [THz]')
ylabel('signal-to-noise ratio [dB]')
legend('GSNR_{DFA}','GSNR_{RA}','GSNR_{ASE}', ...
    'location', 'southwest','Orientation','horizontal')
set(gca,'FontSize',16)
grid on
xlim([184 204]);xticks(184:4:204)
% ylim([15 40])
title('GSNR due to ASE: DFAs, Raman amps, both')

subplot(223)
xlabel('optical frequency [THz]')
ylabel('signal-to-noise ratio [dB]')
legend('GSNR_{SCI}','GSNR_{XCI}', 'GSNR_{NLI}',...
    'location', 'southwest','Orientation','horizontal')
set(gca,'FontSize',16)
grid on
xlim([184 204]);xticks(184:4:204)
% ylim([15 40])
title('GSNR due to non-linearity: SCI, XCI, total NLI')


subplot(224)
xlabel('optical frequency [THz]')
ylabel('signal-to-noise ratio [dB]')
legend('GSNR_{ASE}','GSNR_{NLI}', 'GSNR',...
    'location', 'southwest','Orientation','horizontal')
set(gca,'FontSize',16)
grid on
xlim([184 204]);xticks(184:4:204)
% ylim([10 30])
title('GSNR due to ASE, NLI, both (total GSNR)')


% Get current positions
pos1 = get(subplot(2, 2, 1), 'Position'); % Top-left
pos2 = get(subplot(2, 2, 2), 'Position'); % Top-right
pos3 = get(subplot(2, 2, 3), 'Position'); % Bottom-left
pos4 = get(subplot(2, 2, 4), 'Position'); % Bottom-right

% Define vertical gap (e.g., 0.1 or increase to 0.15 for more space)
vertical_gap = 0.02;

% Adjust positions for top plots
new_pos1 = [pos1(1), pos1(2) + vertical_gap, pos1(3), pos1(4)]; % Move top-left up
new_pos2 = [pos2(1), pos2(2) + vertical_gap, pos2(3), pos2(4)]; % Move top-right up

% Adjust positions for bottom plots
new_pos3 = [pos3(1), pos3(2) - vertical_gap, pos3(3), pos3(4)]; % Move bottom-left down
new_pos4 = [pos4(1), pos4(2) - vertical_gap, pos4(3), pos4(4)]; % Move bottom-right down

% Apply new positions
set(subplot(2, 2, 1), 'Position', new_pos1);
set(subplot(2, 2, 2), 'Position', new_pos2);
set(subplot(2, 2, 3), 'Position', new_pos3);
set(subplot(2, 2, 4), 'Position', new_pos4);


% 3. per-channel throughput
figure(3)
set(gcf,'position',[950,100,850,800])
subplot(211)
GSNR_ref = 5:0.1:30;
load('GMI_GSNR.mat')
GMI_ref = GMI(GSNR_ref);
plot(GSNR_ref, GMI_ref, 'm', 'LineWidth', 2)
xlabel('GSNR over the symbol rate [dB]')
ylabel('[bits/symbol]')
set(gca,'FontSize',16)
grid on
ylim([3 10.2])
title('net transceiver information rate')

% per-channel throughput: Gb/s
Th_ch = GMI(GSNR_dB(span_selected,:)) .*...
    parameters_struct.CH_BR * 1e-9;
subplot(212)
plot(freq_grid, Th_ch,'.', 'color',[0.1 1 0.1])
xlabel('optical frequency [THz]')
ylabel('throughput [Gb/s]')
set(gca,'FontSize',16)
grid on
xlim([184 204]);xticks(184:4:204)
ylim([600 1000])
title('net throughput per channel')

% Get current positions
pos1 = get(subplot(2, 1, 1), 'Position'); % Top plot
pos2 = get(subplot(2, 1, 2), 'Position'); % Bottom plot

% Define vertical gap (e.g., 0.1 or increase to 0.15 for more space)
vertical_gap = 0.02;

% Adjust positions for top plot
new_pos1 = [pos1(1), pos1(2) + vertical_gap, pos1(3), pos1(4)]; % Move top plot up

% Adjust positions for bottom plot
new_pos2 = [pos2(1), pos2(2) - vertical_gap, pos2(3), pos2(4)]; % Move bottom plot down

% Apply new positions
set(subplot(2, 1, 1), 'Position', new_pos1);
set(subplot(2, 1, 2), 'Position', new_pos2);


figure(1)