function parameters_struct = PCFM_parameters

% Yanchao Jiang, Pierluigi Poggiolini, Politecnico di Torino
% 
% Copyright (C) 2025 The Authors - all rights reserved
% Creative Commons Attribution 4.0 International
% 
% acknowledgements for technical contributions to:
% Yifeng Gao, Soochow University 
% Tianchun Zhu, Politecnico di Torino
%
% acknowledgements for sponsorship to CISCO Systems and for
% advice and guidance to Fabrizio Forghieri and Stefano Piciaccia
%
% for details on the model, see:
% [1] Pierluigi Poggiolini and Yanchao Jiang, “Recent Advances 
% in Real-Time Models for UWB Transmission Systems,” OFC2025, 
% Tu3K.2, San Francisco, US, 2025.


% The system parameters are set and saved in a structure.
% We provide an example to calculate NLI GSNR in a C+L+S system.
% The fiber characteristics (attenuation, dispersion, and 
% nonlinearity) are taken from C+L experiment in [2] and then 
% extended to the S band [3].
% [2] Y. Jiang, A. Nespola, A. Tanzi, S. Piciaccia, M. Zefreh, 
% F. Forghieri, P. Poggiolini, "Experimental Test of a Closed-Form 
% EGN Model  Over C+L Bands," in Journal of Lightwave Technology, 
% vol. 43, no. 2, pp. 439-449, 2025.
% [3] Y. Jiang, J. Sarkis, A. Nespola, F. Forghieri, S. Piciaccia,
% A. Tanzi, M. Zefreh, P. Poggiolini,"Optimization of Long-Haul 
% C+L+S Systems by Means of a Closed Form EGN Model," in Photonics 
% Technology Letters, vol. 36, no. 18, pp. 1129-1132, 2024.
% Other references:
% [4] Y. Jiang, J. Sarkis, S. Piciaccia, F. Forghieri, P. Poggiolini,
% "Signal and Backward Raman Pump Power Optimization
% in Multi-Band Systems Using Fast Power Profile Estimation," in 
% Journal of Lightwave Technology, doi: 10.1109/JLT.2025.3585684.
% [5] M. Zefreh, F. Forghieri, S. Piciaccia, P. Poggiolini,
% "Accurate Closed-Form Real-Time EGN Model Formula Leveraging
% Machine-Learning Over 8500 Thoroughly Randomized Full C-Band
% Systems,” in Journal of Lightwave Technology, vol. 38, no. 18,
% pp. 4987-4999, 2020.

% Here, an example with 10 identical spans is provided. 
% However, it can be used with non-uniform spans through 
% modifying the setup.

%%%%%%%%%%%%% WDM signal %%%%%%%%%%%%%
% identical WDM setup is assumed across the link
% L + C + S
% number of channels in L band
Nch_Lband = 50;     
% number of channels in C band
Nch_Cband = 50;     
% number of channels in S band
Nch_Sband = 50;    
% total number of channels 
Nch_signal = Nch_Cband + Nch_Lband + Nch_Sband;
parameters_struct.Nch_signal = Nch_signal;

% identical channels are assumed 
% roll-off
parameters_struct.roll_off = 0.1*ones(1,Nch_signal);   
% symbol rate [Baud]
parameters_struct.CH_BR = 100e9*ones(1,Nch_signal);    
% channel spacing
channel_spacing = 118.75e9; 
% WDM center frequenies [Hz]
% L band
center_Lband = 184.5000e12:channel_spacing:190.31875e12;
% C band
center_Cband = 190.7500e12:channel_spacing:196.56875e12;
% S band
center_Sband = 197.0000e12:channel_spacing:202.81875e12;
% L + C + S
parameters_struct.center_signal = ...
    [center_Lband, center_Cband, center_Sband];

% launch power: cubic polynomial per band
load('signal_launch_power_cubic.mat', 'p')
% C band
gd_C = linspace(-1,1,Nch_Cband);
P_mW_C = 10.^((p(1)+p(2)*gd_C+p(3)*gd_C.^2+p(4)*gd_C.^3)/10);
%P_mW_C = 11.^((p(1)+p(2)*gd_C+p(3)*gd_C.^2+p(4)*gd_C.^3)/10);
% L band
gd_L = linspace(-1,1,Nch_Lband);
P_mW_L = 10.^((p(5)+p(6)*gd_L+p(7)*gd_L.^2+p(8)*gd_L.^3)/10);
% S band
gd_S = linspace(-1,1,Nch_Sband);
P_mW_S = 10.^((p(9)+p(10)*gd_S+p(11)*gd_S.^2+p(12)*gd_S.^3)/10);
% L + C + S
P_signal_mW = [P_mW_L, P_mW_C, P_mW_S];     % per-channel power [mW]
              

% forward-propagating signals: % 1:forward; -1: backward
parameters_struct.direction_signal  = ones(1,Nch_signal);     

%%%%%%%%%%%%% Link Parameters %%%%%%%%%%%%%
% we use 10 identical spans
% number of spans
Nspan = 10;
parameters_struct.Nspan = Nspan; 
% span length [km]
parameters_struct.Ls_km = repmat(100, Nspan, 1);      
% last row is the power at the receiver
parameters_struct.P_signal_W = ones(Nspan+1,1) * P_signal_mW * 1e-3; 

% fiber characteristics
% attenuation [dB/km]  
load('alpha_CLS.mat', 'alpha_signal_dB_per_km')    
% field alpha [1/m] 
parameters_struct.alpha_signal = ...
    ones(Nspan,1) * alpha_signal_dB_per_km/(20000*log10(exp(1))); 

% dispersion
% light speed [m/s]
c = 299792458;                      
parameters_struct.c = c;
% the reference frequency [Hz], where beta2, beta3 and beta4 
% are defined (Taylor series expansion of dispersion). 
% 191.2THz is taken from [1]
parameters_struct.freq_beta = 191.2e12;   
 % beta2 [ps^2/km]        
parameters_struct.beta2_ps_2_per_km = repmat(-22.4043, Nspan, 1);   
% beta3 [ps^3/km]                            
parameters_struct.beta3_ps_3_per_km = repmat(0.1358, Nspan, 1); 
% beta4 [ps^4/km]
parameters_struct.beta4_ps_4_per_km = repmat(-5.4338*1e-4, Nspan, 1);    

% nonlinearity
% refractive index [m^2/W]
parameters_struct.n2 = repmat(2.5310e-20, Nspan, 1);  
% numerical apperture NA 
parameters_struct.NA = repmat(0.124, Nspan, 1);       
% fiber core radius [m]
parameters_struct.a_fiber = repmat(4.1e-6, Nspan, 1); 

%%%%%%%%%%%%% Pump Parameters %%%%%%%%%%%%%
% If there are no pumps, please only set pump power much lower, 
% such as -100dBm each
% number of pumps
Nch_pump = 3;       
parameters_struct.Nch_pump = Nch_pump;
% pump frequency [Hz]
parameters_struct.freq_pump_Hz = ...
    [205.1, 211.5, 214.0] * 1e12;   
% pump power [dBm]
P_pump_dBm = [21.5, 27.7, 26.6];   
% pump power [W]
parameters_struct.P_pump_W =  ...
    ones(Nspan+1,1) * 10.^(P_pump_dBm/10) *1e-3;              
% pump attenuation [dB/km]
load('alpha_CLS.mat', 'alpha_pump_dB_per_km')   
% [1/km]
parameters_struct.alpha_pump = ...
    ones(Nspan,1) * alpha_pump_dB_per_km/(20000*log10(exp(1)));     

% backward-propagating pumps: 1:forward; -1: backward
parameters_struct.direction_pump  = -1*ones(1,Nch_pump);      

%%%%%%%%%%%%% modulation format %%%%%%%%%%%%%
% GN shaped is assumed
parameters_struct.PHI = zeros(1,Nch_signal); % 4-th order correlation

%%%%%%%%%%%%% ISRS Parameters %%%%%%%%%%%%%
% identical Raman gain curve is assumed
% pump wavelength for measuring Raman gain efficiency curve [nm]
lambda_Raman_nm = repmat(1453, Nspan, 1);        
% pump frequency [Hz]
parameters_struct.freq_Raman = c./lambda_Raman_nm * 1e9; 
% the peak value [1/W/m]
C_r_max = 4.50e-4*ones(Nspan,1);    
% normailized Raman Gain efficiency 
load('Raman_Gain_efficiency.mat', 'RGE_normalization', 'freq_THz')
% frequency gap [Hz]
parameters_struct.freq_C_R = ones(Nspan,1) * freq_THz*1e12;        
% actual Raman Gain efficiency [1/W/m]
parameters_struct.C_R = C_r_max * RGE_normalization; 

%%%%%%%%%%%%% SPP Polynomial Fitting %%%%%%%%%%%%%
% spatial resolution for SPPs calculation
parameters_struct.deltaz = 100; %
% the degree of polynomial: 9th
parameters_struct.fitting_degree = 9;
% Note: If you want to use a lower (K<9) degree,  
% please set other coefficients (K+1-th to 9th) 
% as 0 when calling SCI calculator


%%%%%%%%%%%%% machine-learning factor %%%%%%%%%%%%%
% the machine-learning factor, in table IV of [3]
parameters_struct.ML_factor = ...
    [+1.0436e0, -1.1878e0, +1.0573e0, -1.8309e+1,...
    +1.6665e0,  -1.0020e0,  +9.0933e0,  +6.6420e-3,...
    +8.4481e-1, -1.8530e0,  +9.4539e-1, -1.5421e+1,...
    +1.0229e0,  -1.1440e0,  +1.1393e-2, +3.8070e+5,...
    +1.4785e+3, -2.2593e0,  -6.7997e-1, +2.0215e0,...
    -2.9781e-1, +5.5130e-1, -3.6718e-1, +1.1486e0];

%%%%%%%%%%%%% ASE noise  %%%%%%%%%%%%%
% Boltzmann constant, J/K
parameters_struct.k_Boltz = 1.380649 * 1e-23; 
% Planck constant, J/Hz
parameters_struct.h_Planck = 6.62607015 * 1e-34;
% room temperature of the system, K
parameters_struct.T_room = 300;
% DFA noise figure
NF_dB_span = [ones(1,Nch_Lband)*6, ones(1,Nch_Cband)*5,...
    ones(1,Nch_Sband)*6];
parameters_struct.NF_dB = ones(Nspan,1) * NF_dB_span;
% The number of channels per band
parameters_struct.Nband = 3;
parameters_struct.Nch_per_band = [Nch_Lband, Nch_Cband, Nch_Sband];