MODULE usrdef_istate
   !!======================================================================
   !!                   ***  MODULE  usrdef_istate   ***
   !!
   !!                     ===  GYRE configuration  ===
   !!
   !! User defined : set the initial state of a user configuration
   !!======================================================================
   !! History :  4.0  ! 2016-03  (S. Flavoni) Original code
   !!                 ! 2020-11  (S. Techene, G. Madec) separate tsuv from ssh
   !!----------------------------------------------------------------------

   !!----------------------------------------------------------------------
   !!  usr_def_istate : initial state in Temperature and salinity
   !!----------------------------------------------------------------------
   USE par_oce        ! ocean space and time domain
   USE phycst         ! physical constants
   !
   USE in_out_manager ! I/O manager
   USE lib_mpp        ! MPP library
   
   IMPLICIT NONE
   PRIVATE

   PUBLIC   usr_def_istate       ! called in istate.F90
   PUBLIC   usr_def_istate_ssh   ! called by domqco.F90

   !! * Substitutions
#  include "do_loop_substitute.h90"
   !!----------------------------------------------------------------------
   !! NEMO/OCE 4.0 , NEMO Consortium (2018)
   !! $Id: usrdef_istate.F90 14834 2021-05-11 09:24:44Z hadcv $ 
   !! Software governed by the CeCILL license (see ./LICENSE)
   !!----------------------------------------------------------------------
CONTAINS
  
   SUBROUTINE usr_def_istate( pdept, ptmask, pts, pu, pv )
      !!----------------------------------------------------------------------
      !!                   ***  ROUTINE usr_def_istate  ***
      !! 
      !! ** Purpose :   Initialization of the dynamics and tracers
      !!                Here BLACK SEA initialisation from an average T,S profile
      !!                from the World Ocean Database, typical for 1950-1960
      !!                The 59 vertical layers are the ones from the Copernicus Marine BGC product
      !!
      !! ** Method  : - set temperature field
      !!              - set salinity   field
      !!----------------------------------------------------------------------
      REAL(wp), DIMENSION(jpi,jpj,jpk)     , INTENT(in   ) ::   pdept   ! depth of t-point               [m]
      REAL(wp), DIMENSION(jpi,jpj,jpk)     , INTENT(in   ) ::   ptmask  ! t-point ocean mask             [m]
      REAL(wp), DIMENSION(jpi,jpj,jpk,jpts), INTENT(  out) ::   pts     ! T & S fields      [Celsius ; g/kg]
      REAL(wp), DIMENSION(jpi,jpj,jpk)     , INTENT(  out) ::   pu      ! i-component of the velocity  [m/s] 
      REAL(wp), DIMENSION(jpi,jpj,jpk)     , INTENT(  out) ::   pv      ! j-component of the velocity  [m/s] 
      REAL(wp), DIMENSION(jpk)                             ::   woatem, woasal ! WOA18 T & S profiles for 1955-1965, interpolated on NEMO levels
      !
      INTEGER :: ji, jj, jk  ! dummy loop indices
      !!----------------------------------------------------------------------
      !
      IF(lwp) WRITE(numout,*)
      IF(lwp) WRITE(numout,*) 'usr_def_istate : analytical definition of initial state '
      IF(lwp) WRITE(numout,*) '~~~~~~~~~~~~~~   Ocean at rest, with an horizontally uniform T and S profiles'
      !
      woatem=(/ 14.599493, 14.545625, 14.489419, 14.430452, 14.368225, 14.302148, 14.231524, 14.155534, 14.075779, 13.992708, &
      & 13.901473, 13.800604, 13.688368, 13.516724, 13.275599, 13.002717, 12.678128, 12.280921, 11.822970, 11.279382, &
      & 10.664289, 9.979589, 9.310913, 8.639817, 8.060335, 7.694469, 7.712594, 7.810150, 7.933596, 8.138269, &
      & 8.289204, 8.416208, 8.508797, 8.594493, 8.672205, 8.727485, 8.769637, 8.800839, 8.822793, 8.841622, &
      & 8.860068, 8.874296, 8.884415, 8.899540, 8.911026, 8.916259, 8.923292, 8.934126, 8.950013, 8.981992, &
      & 8.980091, 9.000399, 9.031312, 9.054758, 9.069003, 9.062764, 9.109193 /)
      !woatem=(/ 14.537313, 13.945683, 13.130260, 11.993332, 10.871042, 9.887828, 9.105650, 8.493643, 8.026244, 7.716018, &
      !      &  7.667011, 7.745811, 7.834168, 7.910037, 8.024994, 8.141339, 8.245052, 8.383411, 8.531593, 8.660633, 8.764582, &
      !      &  8.826050, 8.862161, 8.888103, 8.911188, 8.927335, 8.964386, 9.005014, 9.037939, 9.086773, 9.107373 /)
      woasal=(/ 18.296755, 18.295851, 18.294907, 18.293917, 18.292870, 18.291759, 18.290573, 18.289297, 18.289032, 18.290453, &
      & 18.292013, 18.293739, 18.295658, 18.299265, 18.304842, 18.311155, 18.318071, 18.325485, 18.334930, 18.351902, &
      & 18.373014, 18.400095, 18.438684, 18.490292, 18.580013, 18.718994, 18.993856, 19.284704, 19.568300, 19.968851, &
      & 20.295698, 20.607157, 20.847206, 21.081947, 21.295712, 21.450703, 21.608093, 21.729280, 21.813873, 21.884333, &
      & 21.962202, 22.040527, 22.107304, 22.170010, 22.207468, 22.238749, 22.260239, 22.287645, 22.304136, 22.324173, &
      & 22.330648, 22.341002, 22.344540, 22.349445, 22.349287, 22.354301, 22.364779 /)
      ! woasal=(/ 18.281960, 18.309126, 18.343681, 18.379904, 18.415457, 18.465836, 18.525776, 18.599001, 18.696314, 18.815296, &
      !       & 18.955173, 19.221689, 19.488852, 19.741365, 20.042246, 20.324991, 20.556086, 20.812122, 21.090899, 21.353283, &
      !       & 21.585978, 21.755325, 21.914541, 22.058243, 22.187738, 22.243891, 22.285482, 22.305632, 22.306728, 22.319605, 22.345110 /)
      !
      IF(lwp) WRITE(numout,*) 'WOA SAL PROFILE: ',woasal
      !
      pu  (:,:,:) = 0._wp           ! ocean at rest
      pv  (:,:,:) = 0._wp
      !
      DO_3D( nn_hls, nn_hls, nn_hls, nn_hls, 1, jpk )   ! horizontally uniform T & S profiles
         !pts(ji,jj,jk,jp_tem) =  (  (  16. - 12. * TANH( (pdept(ji,jj,jk) - 400) / 700 ) )   &
         !     &           * (-TANH( (500. - pdept(ji,jj,jk)) / 150. ) + 1.) / 2.             &
         !     &           + ( 15. * ( 1. - TANH( (pdept(ji,jj,jk)-50.) / 1500.) )            &
         !     &           - 1.4 * TANH((pdept(ji,jj,jk)-100.) / 100.)                        &
         !     &           + 7.  * (1500. - pdept(ji,jj,jk) ) / 1500.)                        &
         !     &           * (-TANH( (pdept(ji,jj,jk) - 500.) / 150.) + 1.) / 2.  ) * ptmask(ji,jj,jk)
         !
         !pts(ji,jj,jk,jp_sal) =  (  (  36.25 - 1.13 * TANH( (pdept(ji,jj,jk) - 305) / 460 ) )  &
         !     &         * (-TANH((500. - pdept(ji,jj,jk)) / 150.) + 1.) / 2                  &
         !     &         + ( 35.55 + 1.25 * (5000. - pdept(ji,jj,jk)) / 5000.                 &
         !     &         - 1.62 * TANH( (pdept(ji,jj,jk) - 60.  ) / 650. )                    &
         !     &         + 0.2  * TANH( (pdept(ji,jj,jk) - 35.  ) / 100. )                    &
         !     &         + 0.2  * TANH( (pdept(ji,jj,jk) - 1000.) / 5000.) )                  &
         !     &         * (-TANH( (pdept(ji,jj,jk) - 500.) / 150.) + 1.) / 2  ) * ptmask(ji,jj,jk)
         pts(ji,jj,jk,jp_tem) = woatem(jk) * ptmask(ji,jj,jk)
         pts(ji,jj,jk,jp_sal) = woasal(jk) * ptmask(ji,jj,jk)
         !
      END_3D
      !   
   END SUBROUTINE usr_def_istate

   
   SUBROUTINE usr_def_istate_ssh( ptmask, pssh )
      !!----------------------------------------------------------------------
      !!                   ***  ROUTINE usr_def_istate_ssh  ***
      !! 
      !! ** Purpose :   Initialization of ssh
      !!
      !! ** Method  :   Set ssh as null, ptmask is required for test cases
      !!----------------------------------------------------------------------
      REAL(wp), DIMENSION(jpi,jpj,jpk)     , INTENT(in   ) ::   ptmask  ! t-point ocean mask   [m]
      REAL(wp), DIMENSION(jpi,jpj)         , INTENT(  out) ::   pssh    ! sea-surface height   [m]
      !!----------------------------------------------------------------------
      !
      IF(lwp) WRITE(numout,*)
      IF(lwp) WRITE(numout,*) 'usr_def_istate_ssh : GYRE configuration, analytical definition of initial state'
      IF(lwp) WRITE(numout,*) '~~~~~~~~~~~~~~~~~~   Ocean at rest, ssh is zero'
      !
      ! Sea level:
      pssh(:,:) = 0._wp
      !
   END SUBROUTINE usr_def_istate_ssh

   !!======================================================================
END MODULE usrdef_istate
