MODULE stopar
#ifdef key_top
#include "bamhbi.h90"
#endif
   !!======================================================================
   !!                       ***  MODULE  stopar  ***
   !! Stochastic parameters : definition and time stepping
   !!=====================================================================
   !! History :  3.3  ! 2011-10 (J.-M. Brankart)  Original code
   !!----------------------------------------------------------------------

   !!----------------------------------------------------------------------
   !!   sto_par       : update the stochastic parameters
   !!   sto_par_init  : define the stochastic parameterization
   !!   sto_rst_read  : read restart file for stochastic parameters
   !!   sto_rst_write : write restart file for stochastic parameters
   !!   sto_par_white : fill input array with white Gaussian noise
   !!   sto_par_flt   : apply horizontal Laplacian filter to input array
   !!----------------------------------------------------------------------
   USE storng          ! random number generator (external module)
   USE par_oce         ! ocean parameters
   USE dom_oce         ! ocean space and time domain variables
   USE lbclnk          ! lateral boundary conditions (or mpp link)
   USE in_out_manager  ! I/O manager
   USE iom             ! I/O module
   USE fldread
   USE lib_mpp
#ifdef key_top
   USE trc, only : numnat_cfg
   USE bamhbi_params
#endif

   IMPLICIT NONE
   PRIVATE

   PUBLIC   sto_par_init    ! called by nemogcm.F90
   PUBLIC   sto_par         ! called by step.F90
   PUBLIC   sto_rst_write   ! called by step.F90

#if ! defined key_mpi_off
   INCLUDE 'mpif.h'
#endif

   
   LOGICAL           :: ln_rst_clock = .FALSE.
   LOGICAL           :: ln_rststo = .FALSE.  ! restart stochastic parameters from restart file
   LOGICAL           :: ln_rstseed = .FALSE. ! read seed of RNG from restart file
   CHARACTER(len=32) :: cn_storst_in  = "restart_sto"    ! suffix of sto restart name (input)
   CHARACTER(len=32) :: cn_storst_out = "restart_sto"    ! suffix of sto restart name (output)
   INTEGER           :: numstor, numstow     ! logical unit for restart (read and write)

   INTEGER           :: jpsto0d = 0          ! number of 0D stochastic parameters, L.V.
   INTEGER           :: jpsto2d = 0          ! number of 2D stochastic parameters
   INTEGER           :: jpsto3d = 0          ! number of 3D stochastic parameters

   REAL(wp), PUBLIC, DIMENSION(:),       ALLOCATABLE :: sto0d
   REAL(wp), PUBLIC, DIMENSION(:,:,:),   ALLOCATABLE :: sto2d       ! 2D stochastic parameters
   REAL(wp), PUBLIC, DIMENSION(:,:,:,:), ALLOCATABLE :: sto3d       ! 3D stochastic parameters
   REAL(wp),         DIMENSION(:,:),     ALLOCATABLE :: sto_tmp     ! temporary workspace
   REAL(wp),         DIMENSION(:,:),     ALLOCATABLE :: sto0d_abc
   REAL(wp),         DIMENSION(:,:),     ALLOCATABLE :: sto2d_abc   ! a, b, c parameters (for 2D arrays)
   REAL(wp),         DIMENSION(:,:),     ALLOCATABLE :: sto3d_abc   ! a, b, c parameters (for 3D arrays)
   REAL(wp), PUBLIC, DIMENSION(:),       ALLOCATABLE :: sto0d_ave   ! mean value (for scalars)
   REAL(wp), PUBLIC, DIMENSION(:),       ALLOCATABLE :: sto2d_ave   ! mean value (for 2D arrays)
   REAL(wp), PUBLIC, DIMENSION(:),       ALLOCATABLE :: sto3d_ave   ! mean value (for 3D arrays)
   REAL(wp),         DIMENSION(:),       ALLOCATABLE :: sto0d_std   ! standard deviation (for scalars)
   REAL(wp),         DIMENSION(:),       ALLOCATABLE :: sto2d_std   ! standard deviation (for 2D arrays)
   REAL(wp),         DIMENSION(:),       ALLOCATABLE :: sto3d_std   ! standard deviation (for 3D arrays)
   REAL(wp),         DIMENSION(:),       ALLOCATABLE :: sto0d_lim   ! limitation factor (for scalars)
   REAL(wp),         DIMENSION(:),       ALLOCATABLE :: sto2d_lim   ! limitation factor (for 2D arrays)
   REAL(wp),         DIMENSION(:),       ALLOCATABLE :: sto3d_lim   ! limitation factor (for 3D arrays)
   REAL(wp),         DIMENSION(:),       ALLOCATABLE :: sto2d_xycor ! spatial correlation (for 2D arrays and/or 3D arrays for every level in vertical)
   REAL(wp),         DIMENSION(:),       ALLOCATABLE :: sto0d_tcor  ! time correlation (for scalars)
   REAL(wp),         DIMENSION(:),       ALLOCATABLE :: sto2d_tcor  ! time correlation (for 2D arrays)
   REAL(wp),         DIMENSION(:),       ALLOCATABLE :: sto3d_tcor  ! time correlation (for 3D arrays)
   INTEGER,          DIMENSION(:),       ALLOCATABLE :: sto0d_ord   ! order of autoregressive process
   INTEGER,          DIMENSION(:),       ALLOCATABLE :: sto2d_ord   ! order of autoregressive process
   INTEGER,          DIMENSION(:),       ALLOCATABLE :: sto3d_ord   ! order of autoregressive process
   REAL(wp)                                          :: sto_tmp0    ! temporary scalar
   
   CHARACTER(len=lca), DIMENSION(:),       ALLOCATABLE :: sto2d_typ  ! nature of grid point (T, U, V, W, F, I)
   CHARACTER(len=lca), DIMENSION(:),       ALLOCATABLE :: sto3d_typ  ! nature of grid point (T, U, V, W, F, I)
   REAL(wp),         DIMENSION(:),       ALLOCATABLE :: sto2d_sgn  ! control of the sign accross the north fold
   REAL(wp),         DIMENSION(:),       ALLOCATABLE :: sto3d_sgn  ! control of the sign accross the north fold
   INTEGER,          DIMENSION(:),       ALLOCATABLE :: sto2d_flt  ! number of passes of Laplacian filter
   INTEGER,          DIMENSION(:),       ALLOCATABLE :: sto3d_flt  ! number of passes of Laplacian filter
   REAL(wp),         DIMENSION(:),       ALLOCATABLE :: sto2d_fac  ! factor to restore std after filtering
   REAL(wp),         DIMENSION(:),       ALLOCATABLE :: sto3d_fac  ! factor to restore std after filtering
   
   LOGICAL, PUBLIC :: ln_sto_any = .FALSE.
   
   LOGICAL, PUBLIC :: ln_sto_ldf = .FALSE.    ! stochastic lateral diffusion
   INTEGER, PUBLIC :: jsto_ldf                ! index of lateral diffusion stochastic parameter
   REAL(wp)        :: rn_ldf_std              ! lateral diffusion standard deviation (in percent)
   REAL(wp)        :: rn_ldf_tcor             ! lateral diffusion correlation timescale (in timesteps)

   LOGICAL, PUBLIC :: ln_sto_hpg = .FALSE.    ! stochastic horizontal pressure gradient
   INTEGER, PUBLIC :: jsto_hpgi               ! index of stochastic hpg parameter (i direction)
   INTEGER, PUBLIC :: jsto_hpgj               ! index of stochastic hpg parameter (j direction)
   REAL(wp)        :: rn_hpg_std              ! density gradient standard deviation (in percent)
   REAL(wp)        :: rn_hpg_tcor             ! density gradient correlation timescale (in timesteps)

   LOGICAL, PUBLIC :: ln_sto_pstar = .FALSE.  ! stochastic ice strength
   INTEGER, PUBLIC :: jsto_pstar              ! index of stochastic ice strength
   REAL(wp), PUBLIC:: rn_pstar_std            ! ice strength standard deviation (in percent)
   REAL(wp)        :: rn_pstar_tcor           ! ice strength correlation timescale (in timesteps)
   INTEGER         :: nn_pstar_flt = 0        ! number of passes of Laplacian filter
   INTEGER         :: nn_pstar_ord = 1        ! order of autoregressive processes

   LOGICAL, PUBLIC :: ln_sto_trd = .FALSE.    ! stochastic model trend
   INTEGER, PUBLIC :: jsto_trd                ! index of stochastic trend parameter
   REAL(wp),PUBLIC :: rn_trd_std              ! trend standard deviation (in percent)
   REAL(wp)        :: rn_trd_tcor             ! trend correlation timescale (in timesteps)
   REAL(wp)        :: rn_trd_xycor = 1.0_wp   ! trend spatial correlation (radius in grid points)
   REAL(wp)        :: rn_trd_lim = 3.0_wp     ! limitation factor
   INTEGER         :: nn_trd_flt = 0          ! number of passes of Laplacian filter

   LOGICAL, PUBLIC :: ln_sto_eos = .FALSE.    ! stochastic equation of state
   INTEGER, PUBLIC :: nn_sto_eos = 1          ! number of degrees of freedom in stochastic equation of state
   INTEGER, PUBLIC, DIMENSION(:), ALLOCATABLE :: jsto_eosi ! index of stochastic eos parameter (i direction)
   INTEGER, PUBLIC, DIMENSION(:), ALLOCATABLE :: jsto_eosj ! index of stochastic eos parameter (j direction)
   INTEGER, PUBLIC, DIMENSION(:), ALLOCATABLE :: jsto_eosk ! index of stochastic eos parameter (k direction)
   REAL(wp)        :: rn_eos_stdxy            ! random walk horz. standard deviation (in grid points)
   REAL(wp)        :: rn_eos_stdz             ! random walk vert. standard deviation (in grid points)
   REAL(wp)        :: rn_eos_tcor             ! random walk correlation timescale (in timesteps)
   REAL(wp)        :: rn_eos_lim = 3.0_wp     ! limitation factor
   INTEGER         :: nn_eos_flt = 0          ! number of passes of Laplacian filter
   INTEGER         :: nn_eos_ord = 1          ! order of autoregressive processes

   LOGICAL, PUBLIC :: ln_sto_trc = .FALSE.    ! stochastic tracer dynamics
   INTEGER, PUBLIC :: nn_sto_trc = 1          ! number of degrees of freedom in stochastic tracer dynamics
   INTEGER, PUBLIC, DIMENSION(:), ALLOCATABLE :: jsto_trci ! index of stochastic trc parameter (i direction)
   INTEGER, PUBLIC, DIMENSION(:), ALLOCATABLE :: jsto_trcj ! index of stochastic trc parameter (j direction)
   INTEGER, PUBLIC, DIMENSION(:), ALLOCATABLE :: jsto_trck ! index of stochastic trc parameter (k direction)
   REAL(wp)        :: rn_trc_stdxy            ! random walk horz. standard deviation (in grid points)
   REAL(wp)        :: rn_trc_stdz             ! random walk vert. standard deviation (in grid points)
   REAL(wp)        :: rn_trc_tcor             ! random walk correlation timescale (in timesteps)
   REAL(wp)        :: rn_trc_lim = 3.0_wp     ! limitation factor
   INTEGER         :: nn_trc_flt = 0          ! number of passes of Laplacian filter
   INTEGER         :: nn_trc_ord = 1          ! order of autoregressive processes

   LOGICAL, PUBLIC :: ln_sto_cdo  = .FALSE.   ! stochastic ocean drag coefficient momentum Cd
   LOGICAL, PUBLIC :: ln_sto_turb = .FALSE.   ! stochastic ocean drag coefficient turbulence Ch, Ce (activation dependency with Cd) 
   INTEGER, PUBLIC :: jsto_cdo                ! index of ocean drag coef. stochastic parameter
   REAL(wp)        :: rn_cdo_std              ! ocean drag coef. standard deviation (in percent)
   REAL(wp)        :: rn_cdo_tcor             ! ocean drag coef. correlation timescale (in timesteps)
   REAL(wp)        :: rn_cdo_xycor = 1.0_wp   ! ocean drag coef. spatial correlation (radius in grid points)
   REAL(wp)        :: rn_cdo_lim = 3.0_wp     ! limitation factor
   INTEGER         :: nn_cdo_flt = 0          ! number of passes of Laplacian filter
   INTEGER         :: nn_cdo_ord = 1          ! order of autoregressive processes

   LOGICAL, PUBLIC :: ln_sto_bfr  = .FALSE.   ! stochastic ocean bottom friction coefficient
   INTEGER, PUBLIC :: jsto_bfr                ! index of ocean bottom friction coef. stochastic parameter
   REAL(wp)        :: rn_bfr_std              ! ocean bottom friction coef. standard deviation (in percent)
   REAL(wp)        :: rn_bfr_tcor             ! ocean bottom friction coef. correlation timescale (in timesteps)
   REAL(wp)        :: rn_bfr_xycor = 1.0_wp   ! ocean bottom friction coef. spatial correlation (radius in grid points)
   REAL(wp)        :: rn_bfr_lim = 3.0_wp     ! limitation factor
   INTEGER         :: nn_bfr_flt = 0          ! number of passes of Laplacian filter
   INTEGER         :: nn_bfr_ord = 1          ! order of autoregressive processes

   LOGICAL, PUBLIC :: ln_sto_rnf  = .FALSE.   ! stochastic runoff
   INTEGER, PUBLIC :: jsto_rnf                ! index of runoff stochastic parameter
   REAL(wp)        :: rn_rnf_std              ! runoff standard deviation (in percent)
   REAL(wp)        :: rn_rnf_tcor             ! runoff correlation timescale (in timesteps)
   REAL(wp)        :: rn_rnf_xycor = 1.0_wp   ! runoff spatial correlation (radius in grid points)
   REAL(wp)        :: rn_rnf_lim = 3.0_wp     ! limitation factor
   INTEGER         :: nn_rnf_flt = 0          ! number of passes of Laplacian filter
   INTEGER         :: nn_rnf_ord = 1          ! order of autoregressive processes
   
   LOGICAL, PUBLIC :: ln_sto_kpar = .FALSE.   ! stochastic kpar coefficient
   INTEGER, PUBLIC :: jsto_kpar               ! index of kpar coef. stochastic parameter
   REAL(wp), PUBLIC:: rn_kpar_std             ! kpar coef. standard deviation (in percent)
   REAL(wp)        :: rn_kpar_tcor            ! kpar coef. correlation timescale (in timesteps)
   REAL(wp)        :: rn_kpar_xycor = 1.0_wp  ! kpar coef. spatial correlation (radius in grid points)
   REAL(wp)        :: rn_kpar_lim = 3.0_wp    ! limitation factor
   INTEGER         :: nn_kpar_flt = 0         ! number of passes of Laplacian filter
   INTEGER         :: nn_kpar_ord = 1         ! order of autoregressive processes

   LOGICAL, PUBLIC :: ln_sto_wnd = .FALSE.    ! stochastic u,v wind forcing
   INTEGER, PUBLIC :: jsto_wnd                ! index of u,v wind forcing stochastic parameter
   REAL(wp)        :: rn_wnd_std              ! u,v wind forcing standard deviation (in percent)
   REAL(wp)        :: rn_wnd_tcor             ! u,v wind forcing correlation timescale (in timesteps)
   REAL(wp)        :: rn_wnd_xycor = 1.0_wp   ! u,v wind forcing spatial correlation (radius in grid points)
   REAL(wp)        :: rn_wnd_lim = 3.0_wp     ! limitation factor
   INTEGER         :: nn_wnd_flt = 0          ! number of passes of Laplacian filter
   INTEGER         :: nn_wnd_ord = 1          ! order of autoregressive processes

   LOGICAL, PUBLIC :: ln_sto_tair = .FALSE.   ! stochastic air temperature
   INTEGER, PUBLIC :: jsto_tair               ! index of air temperature stochastic parameter
   REAL(wp)        :: rn_tair_std             ! air temperature standard deviation (in percent)
   REAL(wp)        :: rn_tair_tcor            ! air temperature correlation timescale (in timesteps)
   REAL(wp)        :: rn_tair_xycor = 1.0_wp  ! air temperature spatial correlation (radius in grid points)
   REAL(wp)        :: rn_tair_lim = 3.0_wp    ! limitation factor
   INTEGER         :: nn_tair_flt = 0         ! number of passes of Laplacian filter
   INTEGER         :: nn_tair_ord = 1         ! order of autoregressive processes

   LOGICAL, PUBLIC :: ln_sto_slp = .FALSE.    ! stochastic sea level pressure
   INTEGER, PUBLIC :: jsto_slp                ! index of sea level pressure stochastic parameter
   REAL(wp)        :: rn_slp_std              ! sea level pressure standard deviation (in percent)
   REAL(wp)        :: rn_slp_tcor             ! sea level pressure correlation timescale (in timesteps)
   REAL(wp)        :: rn_slp_xycor = 1.0_wp   ! sea level pressure spatial correlation (radius in grid points)
   REAL(wp)        :: rn_slp_lim = 3.0_wp     ! limitation factor
   INTEGER         :: nn_slp_flt = 0          ! number of passes of Laplacian filter
   INTEGER         :: nn_slp_ord = 1          ! order of autoregressive processes

   LOGICAL, PUBLIC       :: ln_sto_wnd_eofs = .FALSE.
   INTEGER, PUBLIC       :: jsto_wnd_eofs
   REAL(wp)              :: rn_wnd_eofs_std           ! u,v wind forcing standard deviation (in percent)
   REAL(wp)              :: rn_wnd_eofs_tcor          ! u,v wind forcing correlation timescale (in timesteps)
   REAL(wp)              :: rn_wnd_eofs_lim = 3.0_wp  ! limitation factor
   INTEGER               :: nn_wnd_eofs_ord = 1       ! order of autoregressive processes
   CHARACTER(len=64)     :: cn_wnd_eofs_file          ! eofs filename with variables S???.u10 and S???.v10
   CHARACTER(len=32)     :: cn_wnd_eofs_vars          ! comma-separated list of netcdf variables to read
   INTEGER, PUBLIC       :: nn_wnd_eofs     = 25      ! how many EOFs to use
   REAL(wp), PUBLIC, DIMENSION(:,:,:,:), ALLOCATABLE :: wnd_eofs ! (i,j,rank,variable)
   INTEGER, PUBLIC, SAVE :: iw_u10                    ! index, in the weights structure, of the weights for some variables, will be set by blkmfs

   LOGICAL, PUBLIC       :: ln_sto_tair_eofs = .FALSE.
   INTEGER, PUBLIC       :: jsto_tair_eofs
   REAL(wp)              :: rn_tair_eofs_std          ! T2M forcing standard deviation (in percent)
   REAL(wp)              :: rn_tair_eofs_tcor         ! T2M forcing correlation timescale (in timesteps)
   REAL(wp)              :: rn_tair_eofs_lim = 3.0_wp ! limitation factor
   INTEGER               :: nn_tair_eofs_ord = 1      ! order of autoregressive processes                                                                                                                        
   CHARACTER(len=64)     :: cn_tair_eofs_file         ! eofs filename with variables S???.t2m
   CHARACTER(len=32)     :: cn_tair_eofs_vars         ! comma-separated list of netcdf variables to read
   INTEGER, PUBLIC       :: nn_tair_eofs     = 25     ! how many EOFs to use
   REAL(wp), PUBLIC, DIMENSION(:,:,:), ALLOCATABLE :: tair_eofs ! (i,j,rank,variable)
   INTEGER, PUBLIC, SAVE :: iw_tair                   ! index, in the weights structure, of the weights for t2m, will be set by blkmfs

   LOGICAL, PUBLIC       :: ln_sto_tcc_eofs = .FALSE.
   INTEGER, PUBLIC       :: jsto_tcc_eofs
   REAL(wp)              :: rn_tcc_eofs_std           ! TCC forcing standard deviation (in percent)
   REAL(wp)              :: rn_tcc_eofs_tcor          ! TCC forcing correlation timescale (in timesteps)
   REAL(wp)              :: rn_tcc_eofs_lim = 3.0_wp  ! limitation factor
   INTEGER               :: nn_tcc_eofs_ord = 1       ! order of autoregressive processes
   CHARACTER(len=64)     :: cn_tcc_eofs_file          ! eofs filename with variables S???.tcc
   CHARACTER(len=32)     :: cn_tcc_eofs_vars          ! comma-separated list of netcdf variables to read
   INTEGER, PUBLIC       :: nn_tcc_eofs     = 25      ! how many EOFs to use
   REAL(wp), PUBLIC, DIMENSION(:,:,:), ALLOCATABLE :: tcc_eofs ! (i,j,rank,variable)
   INTEGER, PUBLIC, SAVE :: iw_tcc                    ! index, in the weights structure, of the weights for tcc, will be set by blkmfs

   LOGICAL, PUBLIC       :: ln_sto_prec_eofs = .FALSE.
   INTEGER, PUBLIC       :: jsto_prec_eofs
   REAL(wp)              :: rn_prec_eofs_std          ! PREC standard deviation (in percent)
   REAL(wp)              :: rn_prec_eofs_tcor         ! PREC correlation timescale (in timesteps)
   REAL(wp)              :: rn_prec_eofs_lim = 3.0_wp ! limitation factor
   INTEGER               :: nn_prec_eofs_ord = 1      ! order of autoregressive processes
   CHARACTER(len=64)     :: cn_prec_eofs_file         ! eofs filename with variables S???.prec
   CHARACTER(len=32)     :: cn_prec_eofs_vars         ! comma-separated list of netcdf variables to read
   INTEGER, PUBLIC       :: nn_prec_eofs     = 25     ! how many EOFs to use
   REAL(wp), PUBLIC, DIMENSION(:,:,:), ALLOCATABLE :: prec_eofs ! (i,j,rank,variable)
   INTEGER, PUBLIC, SAVE :: iw_prec                   ! index, in the weights structure, of the weights for prec, will be set by blkmfs

   LOGICAL, PUBLIC       :: ln_sto_wnd_fft = .FALSE.
   INTEGER, PUBLIC       :: jsto_wnd_fft
   INTEGER, PUBLIC       :: nn_wnd_fft                ! how many modes to use
   REAL(wp)              :: rn_wnd_fft_std            ! WIND std dev (in percent)
   REAL(wp)              :: rn_wnd_fft_lim = 5.0_wp   ! limitation factor
   INTEGER               :: nn_wnd_fft_ord = 1        ! order of autoregressive processes
   CHARACTER(len=64)     :: cn_wnd_fft_file           ! filename with Fourier periods and modes
   CHARACTER(len=32)     :: cn_wnd_fft_T              ! netcdf variable containing the period of modes
   CHARACTER(len=32)     :: cn_wnd_fft_vars           ! comma-separated list of netcdf variables to read
   REAL(wp), PUBLIC, ALLOCATABLE, SAVE, DIMENSION(:) :: fft_periods
   REAL(wp), PUBLIC, DIMENSION(:,:,:,:), ALLOCATABLE :: wnd_ffts ! (i,j,rank,variable)

   LOGICAL, PUBLIC :: ln_sto_cdom  = .FALSE.           ! stochastic CDOM absorption 
   INTEGER, PUBLIC :: jsto_cdom                       ! index of CDOM absorption stochastic parameter  
   REAL(wp)        :: rn_cdom_ave                     ! CDOM absorption average (in percent)
   REAL(wp)        :: rn_cdom_std                     ! CDOM absorption standard deviation (in percent) 
   REAL(wp)        :: rn_cdom_tcor                    ! CDOM absorption correlation timescale (in timesteps)
   REAL(wp)        :: rn_cdom_xycor = 1.0_wp          ! CDMO absorption spatial correlation (radius in grid points)   
   REAL(wp)        :: rn_cdom_lim = 3.0_wp            ! limitation factor    
   INTEGER         :: nn_cdom_flt = 5                 ! number of passes of Laplacian filter
   INTEGER         :: nn_cdom_ord = 1                 ! order of autoregressive processes        

   LOGICAL, PUBLIC :: ln_sto_expcdom  = .FALSE.          ! stochastic CDOM absorption                                                  
   INTEGER, PUBLIC :: jsto_expcdom                       ! index of CDOM absorption stochastic parameter                                                      
   REAL(wp)        :: rn_expcdom_ave                     ! CDOM absorption average (in percent)                                                                         
   REAL(wp)        :: rn_expcdom_std                     ! CDOM absorption standard deviation (in percent)                                                                  
   REAL(wp)        :: rn_expcdom_tcor                    ! CDOM absorption correlation timescale (in timesteps)                                                               
   REAL(wp)        :: rn_expcdom_xycor = 1.0_wp          ! CDOM absorption spatial correlation (radius in grid points)                                                           
   REAL(wp)        :: rn_expcdom_lim = 3.0_wp            ! limitation factor                                                              
   INTEGER         :: nn_expcdom_flt = 5                 ! number of passes of Laplacian filter   
   INTEGER         :: nn_expcdom_ord = 1                 ! order of autoregressive processes

#ifdef key_top
   LOGICAL, PUBLIC :: ln_sto_rnf_bio  = .FALSE.       ! stochastic bio runoff
   INTEGER, PUBLIC :: jsto_rnf_bio                    ! index of bio runoff stochastic parameter
   REAL(wp)        :: rn_rnfbio_std                   ! bio runoff standard deviation (in percent)
   REAL(wp)        :: rn_rnfbio_tcor                  ! bio runoff correlation timescale (in timesteps)
   REAL(wp)        :: rn_rnfbio_xycor = 1.0_wp        ! bio runoff spatial correlation (radius in grid points)
   REAL(wp)        :: rn_rnfbio_lim = 3.0_wp          ! limitation factor
   INTEGER         :: nn_rnfbio_flt = 0               ! number of passes of Laplacian filter
   INTEGER         :: nn_rnfbio_ord = 1               ! order of autoregressive processes

   LOGICAL, PUBLIC :: ln_sto_depo  = .FALSE.          ! stochastic atmospheric deposition
   INTEGER, PUBLIC :: jsto_depo                       ! index of atmospheric deposition stochastic parameter
   REAL(wp)        :: rn_depo_std                     ! atmospheric deposition standard deviation (in percent)
   REAL(wp)        :: rn_depo_tcor                    ! atmospheric deposition correlation timescale (in timesteps)
   REAL(wp)        :: rn_depo_xycor = 1.0_wp          ! atmospheric deposition spatial correlation (radius in grid points)
   REAL(wp)        :: rn_depo_lim = 3.0_wp            ! limitation factor
   INTEGER         :: nn_depo_flt = 0                 ! number of passes of Laplacian filter
   INTEGER         :: nn_depo_ord = 1                 ! order of autoregressive processes

   INTEGER, SAVE   :: bamhbi_sto_cpt = 0
   real(wp) :: sto_MaxgrazingrateMesoZoo, sto_alphaPIDiatoms, sto_Ass_Eff_OnCarbon, sto_efficiency_growth_MesoZoo
   real(wp) :: sto_HalfSatMort_MesoZoo, sto_Capt_eff_Gelatinous_mesozoo, sto_MaxgrazingrateGelatinous, sto_Capt_eff_MesoZoo_Diatoms
   real(wp) :: sto_Ass_Eff_OnNitrogen, sto_MuMaxDiatoms, sto_SiNrDiatoms, sto_MinNCrDiatoms, sto_MortalityDiatoms
   real(wp) :: sto_Messy_feeding_MicroZoo, sto_Messy_feeding_MesoZoo
   real(wp) :: sto_SinkingRateDiatomsmin, sto_SinkingRateDiatomsmax, sto_MortalityFlagellates, sto_MortalityEmiliana, sto_ksNHs
   real(wp) :: sto_MuMaxFlagellates, sto_MuMaxEmiliana, sto_Q10Phy, sto_Q10PhyDiatoms, sto_Q10Zoo
   real(wp) :: sto_nlin_mort_MESozoo
   real(wp) :: sto_dzetabio,sto_etabio,sto_lightabsB,sto_Q10chem
#endif

   !! * Substitutions
#  include "do_loop_substitute.h90"
   !!----------------------------------------------------------------------
   !! NEMO/OCE 4.0 , NEMO Consortium (2018)
   !! $Id: stopar.F90 13295 2020-07-10 18:24:21Z acc $
   !! Software governed by the CeCILL license (see ./LICENSE)
   !!----------------------------------------------------------------------
CONTAINS

   SUBROUTINE sto_par( kt )
      !!----------------------------------------------------------------------
      !!                  ***  ROUTINE sto_par  ***
      !!
      !! ** Purpose :   update the stochastic parameters
      !!
      !! ** Method  :   model basic stochastic parameters
      !!                as a first order autoregressive process AR(1),
      !!                governed by the equation:
      !!                   X(t) = a * X(t-1) + b * w + c
      !!                where the parameters a, b and c are related
      !!                to expected value, standard deviation
      !!                and time correlation (all stationary in time) by:
      !!                   E   [X(t)]        = c / ( 1 - a )
      !!                   STD [X(t)]        = b / SQRT( 1 - a * a )
      !!                   COR [X(t),X(t-k)] = a ** k
      !!                and w is a Gaussian white noise.
      !!
      !!                Higher order autoregressive proces can be optionally generated
      !!                by replacing the white noise by a lower order process.
      !!
      !!                1) The statistics of the stochastic parameters (X) are assumed
      !!                constant in space (homogeneous) and time (stationary).
      !!                This could be generalized by replacing the constant
      !!                a, b, c parameters by functions of space and time.
      !!
      !!                2) The computation is performed independently for every model
      !!                grid point, which corresponds to assume that the stochastic
      !!                parameters are uncorrelated in space.
      !!                This could be generalized by including a spatial filter: Y = Filt[ X ]
      !!                (possibly non-homgeneous and non-stationary) in the computation,
      !!                or by solving an elliptic equation: L[ Y ] = X.
      !!
      !!                3) The stochastic model for the parameters could also
      !!                be generalized to depend on the current state of the ocean (not done here).
      !!----------------------------------------------------------------------
      INTEGER, INTENT( in ) ::   kt   ! ocean time-step index
      !!
      INTEGER  :: ji, jj, jk, jsto, jflt, ios, ipos, jvar
      REAL(wp) :: stomax
      REAL(KIND=8) :: gran
      CHARACTER(LEN=16) :: eofvar     ! split comma-separated cn_wnd_eofs_vars
      CHARACTER(LEN=3)  :: eofnn
      CHARACTER(LEN=32) :: eofncvar   ! e.g. S001.u10
      REAL(wp), DIMENSION(:,:,:), ALLOCATABLE :: temp_eof
      !!----------------------------------------------------------------------
      !
      ! Update 0D stochastic arrays
      !
      do jsto = 1, jpsto0d
        sto_tmp0 = sto0d(jsto)
        IF ( sto0d_ord(jsto) == 1 ) THEN    ! draw new N(0,1) number
          CALL kiss_gaussian( gran ) 
          sto0d(jsto) = gran
        ELSE    ! Use previous process (one order lower) instead of white noise
          sto0d(jsto) = sto0d(jsto-1)
        ENDIF
        ! Multiply white noise (or lower order process) by b --> b * w
        sto0d(jsto) = sto0d(jsto) * sto0d_abc(jsto,2)
        !! increase wind perturbation intensity in second half of medium-term forcast starting in April
        !!if (jsto.ge.jsto_wnd_fft .and. jsto.le.jsto_wnd_fft+nn_wnd_fft) &
        !!     sto0d(jsto) = sto0d(jsto) * (1.0_wp+3.555/( 1.0+exp(-((nday_year-106.0)-15.0)) )) ! 106 days to start from 15/April, then 15 days to have sigmoid acting on days 10:20
        ! Update autoregressive processes --> a * X(t-1) + b * w
        sto0d(jsto) = sto0d(jsto) + sto_tmp0 * sto0d_abc(jsto,1)
        ! Add parameter c --> a * X(t-1) + b * w + c
        sto0d(jsto) = sto0d(jsto) + sto0d_abc(jsto,3)
        ! Limit random parameter anomalies to std times the limitation factor
        stomax = sto0d_std(jsto) * sto0d_lim(jsto)
        sto0d(jsto) = sto0d(jsto) - sto0d_ave(jsto)
        sto0d(jsto) = SIGN(MIN(stomax,ABS(sto0d(jsto))),sto0d(jsto))
        sto0d(jsto) = sto0d(jsto) + sto0d_ave(jsto)
      end do
      if (jpsto0d>0) call sto_share_0d( )
      !
      ! Update 2D stochastic arrays
      !
      DO jsto = 1, jpsto2d
        ! Store array from previous time step
        sto_tmp(:,:) = sto2d(:,:,jsto)

        IF ( sto2d_ord(jsto) == 1 ) THEN
          ! Draw new random numbers from N(0,1) --> w
          IF ( sto2d_xycor(jsto) > 1._wp ) THEN
            ! Introduce random horiz. 2D Gaussian structures --> w 
            CALL sto_par_xygau( jsto, sto2d(:,:,jsto) )
          ELSE
            ! Draw new random numbers from N(0,1) --> w
            CALL sto_par_white( sto2d(:,:,jsto) )
          ENDIF
          ! Apply horizontal Laplacian filter to w
          DO jflt = 1, sto2d_flt(jsto)
            CALL lbc_lnk( 'stopar', sto2d(:,:,jsto), sto2d_typ(jsto), sto2d_sgn(jsto) )
            CALL sto_par_flt( sto2d(:,:,jsto) )
          END DO
          ! Factor to restore standard deviation after filtering
          sto2d(:,:,jsto) = sto2d(:,:,jsto) * sto2d_fac(jsto)
        ELSE
          ! Use previous process (one order lower) instead of white noise
          sto2d(:,:,jsto) = sto2d(:,:,jsto-1)
        ENDIF

        ! Multiply white noise (or lower order process) by b --> b * w
        sto2d(:,:,jsto) = sto2d(:,:,jsto) * sto2d_abc(jsto,2)
        ! Update autoregressive processes --> a * X(t-1) + b * w
        sto2d(:,:,jsto) = sto2d(:,:,jsto) + sto_tmp(:,:) * sto2d_abc(jsto,1)
        ! Add parameter c --> a * X(t-1) + b * w + c
        sto2d(:,:,jsto) = sto2d(:,:,jsto) + sto2d_abc(jsto,3)
        ! Limit random parameter anomalies to std times the limitation factor
        stomax = sto2d_std(jsto) * sto2d_lim(jsto)
        sto2d(:,:,jsto) = sto2d(:,:,jsto) - sto2d_ave(jsto)
        sto2d(:,:,jsto) = SIGN(MIN(stomax,ABS(sto2d(:,:,jsto))),sto2d(:,:,jsto))
        sto2d(:,:,jsto) = sto2d(:,:,jsto) + sto2d_ave(jsto)

        ! Lateral boundary conditions on sto2d
        CALL lbc_lnk( 'stopar', sto2d(:,:,jsto), sto2d_typ(jsto), sto2d_sgn(jsto) )
      END DO
      !
      ! Update 3D stochastic arrays
      !
      DO jsto = 1, jpsto3d
         DO jk = 1, jpk
           ! Store array from previous time step
           sto_tmp(:,:) = sto3d(:,:,jk,jsto)

           IF ( sto3d_ord(jsto) == 1 ) THEN
             ! Draw new random numbers from N(0,1) --> w
             IF ( sto2d_xycor(jpsto2d+jsto) > 1._wp ) THEN
               ! Introduce random horiz. 2D Gaussian structures --> w
               CALL sto_par_xygau( jpsto2d+jsto, sto3d(:,:,jk,jsto) )
             ELSE
               ! Draw new random numbers from N(0,1) --> w
               CALL sto_par_white( sto3d(:,:,jk,jsto) )
             ENDIF
             ! Apply horizontal Laplacian filter to w
             DO jflt = 1, sto3d_flt(jsto)
               CALL lbc_lnk( 'stopar', sto3d(:,:,jk,jsto), sto3d_typ(jsto), sto3d_sgn(jsto) )
               CALL sto_par_flt( sto3d(:,:,jk,jsto) )
             END DO
             ! Factor to restore standard deviation after filtering
             sto3d(:,:,jk,jsto) = sto3d(:,:,jk,jsto) * sto3d_fac(jsto)
           ELSE
             ! Use previous process (one order lower) instead of white noise
             sto3d(:,:,jk,jsto) = sto3d(:,:,jk,jsto-1)
           ENDIF

           ! Multiply white noise by b --> b * w
           sto3d(:,:,jk,jsto) = sto3d(:,:,jk,jsto) * sto3d_abc(jsto,2)
           ! Update autoregressive processes --> a * X(t-1) + b * w
           sto3d(:,:,jk,jsto) = sto3d(:,:,jk,jsto) + sto_tmp(:,:) * sto3d_abc(jsto,1)
           ! Add parameter c --> a * X(t-1) + b * w + c
           sto3d(:,:,jk,jsto) = sto3d(:,:,jk,jsto) + sto3d_abc(jsto,3)
           ! Limit random parameters anomalies to std times the limitation factor
           stomax = sto3d_std(jsto) * sto3d_lim(jsto)
           sto3d(:,:,jk,jsto) = sto3d(:,:,jk,jsto) - sto3d_ave(jsto)
           sto3d(:,:,jk,jsto) = SIGN(MIN(stomax,ABS(sto3d(:,:,jk,jsto))),sto3d(:,:,jk,jsto))
           sto3d(:,:,jk,jsto) = sto3d(:,:,jk,jsto) + sto3d_ave(jsto)
         END DO
         ! Lateral boundary conditions on sto3d
         CALL lbc_lnk( 'stopar', sto3d(:,:,:,jsto), sto3d_typ(jsto), sto3d_sgn(jsto) )
      END DO

      !
      ! At model start, if required, load EOFs, FFTs, ... and interpolate them
      !
      if (kt.eq.nit000.and.(ln_sto_wnd_eofs.or.ln_sto_tair_eofs.or.ln_sto_tcc_eofs.or.ln_sto_prec_eofs)) then
         allocate(temp_eof(jpi,jpj,1))
         if (ln_sto_wnd_eofs) then
           call iom_open(cn_wnd_eofs_file, ios)
           do jvar=1,2
              ipos=index(cn_wnd_eofs_vars,',')
              if (ipos.eq.0) call ctl_stop('cn_wnd_eofs_vars expected to contain a comma !')
              if (jvar==1) eofvar=cn_wnd_eofs_vars(1:ipos-1)
              if (jvar==2) eofvar=cn_wnd_eofs_vars(ipos+1:len(trim(cn_wnd_eofs_vars)))
              eofncvar='S000.'//trim(eofvar)
              do jsto = 1, nn_wnd_eofs
                 write(eofncvar(2:4),'(i3.3)') jsto
                 !if (lwp) write(numout,*) 'reading variable ',trim(eofncvar),' with weight set ',iw_u10
                 !call iom_get( ios, jpdom_autoglo, trim(eofncvar) , wnd_eofs(:,:,jsto,jvar) )  ! works only if variable doesn't need spatial interpolations
                 call fld_interp(ios,trim(eofncvar),iw_u10,1,temp_eof,1,'')
                 wnd_eofs(:,:,jsto,jvar)=temp_eof(:,:,1)
              end do
           end do
         end if
         if (ln_sto_tair_eofs) then
           call iom_open(cn_tair_eofs_file, ios)
           eofncvar='S000.'//trim(cn_tair_eofs_vars)
           do jsto = 1, nn_tair_eofs
                 write(eofncvar(2:4),'(i3.3)') jsto
                 call fld_interp(ios,trim(eofncvar),iw_tair,1,temp_eof,1,'')
                 tair_eofs(:,:,jsto)=temp_eof(:,:,1)
            end do
         end if
         if (ln_sto_tcc_eofs) then
           call iom_open(cn_tcc_eofs_file, ios)
           eofncvar='S000.'//trim(cn_tcc_eofs_vars)
           do jsto = 1, nn_tcc_eofs
                 write(eofncvar(2:4),'(i3.3)') jsto
                 call fld_interp(ios,trim(eofncvar),iw_tcc,1,temp_eof,1,'')
                 tcc_eofs(:,:,jsto)=temp_eof(:,:,1)
            end do
         end if
         if (ln_sto_prec_eofs) then
           call iom_open(cn_prec_eofs_file, ios)
           eofncvar='S000.'//trim(cn_prec_eofs_vars)
           do jsto = 1, nn_prec_eofs
                 write(eofncvar(2:4),'(i3.3)') jsto
                 call fld_interp(ios,trim(eofncvar),iw_prec,1,temp_eof,1,'')
                 prec_eofs(:,:,jsto)=temp_eof(:,:,1)
            end do
         end if
         deallocate(temp_eof)
      end if
      if (kt.eq.nit000.and.(ln_sto_wnd_fft)) then
         allocate(temp_eof(jpi,jpj,1))
         call iom_open(cn_wnd_fft_file, ios)
         do jvar=1,2
           ipos=index(cn_wnd_fft_vars,',')
           if (ipos.eq.0) call ctl_stop('cn_wnd_eofs_vars expected to contain a comma !')
           if (jvar==1) eofvar=cn_wnd_fft_vars(1:ipos-1)
           if (jvar==2) eofvar=cn_wnd_fft_vars(ipos+1:len(trim(cn_wnd_fft_vars)))
           eofncvar=trim(eofvar)
           do jsto = 1, nn_wnd_fft
             write(eofncvar(7:9),'(i3.3)') jsto
             call fld_interp(ios,trim(eofncvar),iw_u10,1,temp_eof,1,'')
             wnd_ffts(:,:,jsto,jvar)=temp_eof(:,:,1)
           end do
         end do
         deallocate(temp_eof)
      end if
      
   END SUBROUTINE sto_par


   SUBROUTINE sto_par_init
      !!----------------------------------------------------------------------
      !!                  ***  ROUTINE sto_par_init  ***
      !!
      !! ** Purpose :   define the stochastic parameterization
      !!----------------------------------------------------------------------
      NAMELIST/namsto/ ln_rst_clock, ln_rststo, ln_rstseed, cn_storst_in, cn_storst_out, &
        &              ln_sto_ldf, rn_ldf_std, rn_ldf_tcor, &
        &              ln_sto_hpg, rn_hpg_std, rn_hpg_tcor, &
        &              ln_sto_pstar, rn_pstar_std, rn_pstar_tcor, nn_pstar_flt, nn_pstar_ord, &
        &              ln_sto_trd, rn_trd_std, rn_trd_tcor, rn_trd_xycor, rn_trd_lim, nn_trd_flt, &
        &              ln_sto_eos, nn_sto_eos, rn_eos_stdxy, rn_eos_stdz, &
        &              rn_eos_tcor, nn_eos_ord, nn_eos_flt, rn_eos_lim, &
        &              ln_sto_trc, nn_sto_trc, rn_trc_stdxy, rn_trc_stdz, &
        &              rn_trc_tcor, nn_trc_ord, nn_trc_flt, rn_trc_lim, &
        &              ln_sto_cdo, ln_sto_turb, rn_cdo_std, rn_cdo_tcor, rn_cdo_xycor, & 
        &              rn_cdo_lim, nn_cdo_flt, nn_cdo_ord, & 
        &              ln_sto_bfr, rn_bfr_std, rn_bfr_tcor, rn_bfr_xycor, & 
        &              rn_bfr_lim, nn_bfr_flt, nn_bfr_ord, & 
        &              ln_sto_rnf, rn_rnf_std, rn_rnf_tcor, rn_rnf_xycor, &
        &              rn_rnf_lim, nn_rnf_flt, nn_rnf_ord, &
        &              ln_sto_kpar, rn_kpar_std, rn_kpar_tcor, rn_kpar_xycor, & 
        &              rn_kpar_lim, nn_kpar_flt, nn_kpar_ord, & 
        &              ln_sto_wnd, rn_wnd_std, rn_wnd_tcor, rn_wnd_xycor, & 
        &              rn_wnd_lim, nn_wnd_flt, nn_wnd_ord, &
        &              ln_sto_tair, rn_tair_std, rn_tair_tcor, rn_tair_xycor, & 
        &              rn_tair_lim, nn_tair_flt, nn_tair_ord, & 
        &              ln_sto_slp, rn_slp_std, rn_slp_tcor, rn_slp_xycor, & 
        &              rn_slp_lim, nn_slp_flt, nn_slp_ord, & 
        &              ln_sto_wnd_eofs, rn_wnd_eofs_std, rn_wnd_eofs_tcor, &    ! L.V. : EOFs perturbation
        &              rn_wnd_eofs_lim, nn_wnd_eofs_ord, cn_wnd_eofs_file, &
        &              cn_wnd_eofs_vars, nn_wnd_eofs, &
        &              ln_sto_tair_eofs, rn_tair_eofs_std, rn_tair_eofs_tcor, &
        &              rn_tair_eofs_lim, nn_tair_eofs_ord, cn_tair_eofs_file, &
        &              cn_tair_eofs_vars, nn_tair_eofs, &
        &              ln_sto_tcc_eofs, rn_tcc_eofs_std, rn_tcc_eofs_tcor, &
        &              rn_tcc_eofs_lim, nn_tcc_eofs_ord, cn_tcc_eofs_file, &
        &              cn_tcc_eofs_vars, nn_tcc_eofs, &
        &              ln_sto_prec_eofs, rn_prec_eofs_std, rn_prec_eofs_tcor, &
        &              rn_prec_eofs_lim, nn_prec_eofs_ord, cn_prec_eofs_file, &
        &              cn_prec_eofs_vars, nn_prec_eofs, &
        &              ln_sto_wnd_fft, nn_wnd_fft, rn_wnd_fft_std, nn_wnd_fft_ord, &  ! L.V. : Fourier pertubation
        &              rn_wnd_fft_lim, cn_wnd_fft_file, cn_wnd_fft_T, cn_wnd_fft_vars
#ifdef key_top
      NAMELIST/namsto_bio/ &
        &              ln_sto_rnf_bio, rn_rnfbio_std, rn_rnfbio_tcor, rn_rnfbio_xycor, &
        &              rn_rnfbio_lim, nn_rnfbio_flt, nn_rnfbio_ord, &
        &              ln_sto_depo, rn_depo_std, rn_depo_tcor, rn_depo_xycor, &
        &              rn_depo_lim, nn_depo_flt, nn_depo_ord
      NAMELIST/namsto_bamhbi/ &
        &              sto_MaxgrazingrateMesoZoo, sto_alphaPIDiatoms, sto_Ass_Eff_OnCarbon, sto_efficiency_growth_MesoZoo, sto_nlin_mort_MESozoo, sto_HalfSatMort_MesoZoo, &
        &              sto_Capt_eff_Gelatinous_mesozoo, sto_MaxgrazingrateGelatinous, sto_Capt_eff_MesoZoo_Diatoms, sto_Ass_Eff_OnNitrogen, sto_MuMaxDiatoms, &
        &              sto_SiNrDiatoms, sto_MinNCrDiatoms, sto_MortalityDiatoms, sto_Messy_feeding_MicroZoo, sto_Messy_feeding_MesoZoo, sto_SinkingRateDiatomsmin, &
        &              sto_SinkingRateDiatomsmax, sto_MortalityFlagellates, sto_MortalityEmiliana, sto_ksNHs, sto_MuMaxFlagellates, sto_MuMaxEmiliana, &
        &              sto_Q10Phy, sto_Q10PhyDiatoms, sto_Q10Zoo, &
        &              sto_dzetabio,sto_etabio,sto_lightabsB,sto_Q10chem
#endif
      !!----------------------------------------------------------------------
      INTEGER         :: jsto, jmem, jarea, jdof, jord, jordm1, jk, jflt
      INTEGER(KIND=8) :: zseed1, zseed2, zseed3, zseed4
      REAL(wp)        :: rinflate
      INTEGER         ::   ios                 ! Local integer output status for namelist read
      INTEGER         :: ierr, fftncid, vidOK
      REAL(KIND=8)    :: gran, k, mean,variance, alpha, beta, theta
      INTEGER         :: daytosecond=86400
      
      ! 0a) read namelists, process them
      ! -------------------------------
      READ  ( numnam_ref, namsto, IOSTAT = ios, ERR = 901)
901   IF( ios /= 0 ) CALL ctl_nam ( ios , 'namsto in reference namelist' )

      READ  ( numnam_cfg, namsto, IOSTAT = ios, ERR = 902 )
902   IF( ios >  0 ) CALL ctl_nam ( ios , 'namsto in configuration namelist' )
      IF(lwm) WRITE ( numond, namsto )

#ifdef key_top
      READ  ( numnat_cfg, namsto_bio, IOSTAT = ios, ERR = 903 )
903   IF( ios >  0 ) CALL ctl_nam ( ios , 'namsto_bio in top configuration namelist' )
      IF(lwm) WRITE ( numond, namsto )
      READ  ( numnat_cfg, namsto_bamhbi, IOSTAT = ios, ERR = 904 )
904   IF( ios >  0 ) CALL ctl_nam ( ios , 'namsto_bamhbi in top configuration namelist' )
      IF(lwm) WRITE ( numond, namsto )
#endif

#ifdef key_top
      bamhbi_sto_cpt=0
      if (sto_MaxgrazingrateMesoZoo.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_alphaPIDiatoms.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_Ass_Eff_OnCarbon.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_efficiency_growth_MesoZoo.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_nlin_mort_MESozoo.ne.0.0) bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_HalfSatMort_MesoZoo.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_Capt_eff_Gelatinous_mesozoo.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_MaxgrazingrateGelatinous.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_Capt_eff_MesoZoo_Diatoms.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_Ass_Eff_OnNitrogen.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_MuMaxDiatoms.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_SiNrDiatoms.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_MinNCrDiatoms.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_MortalityDiatoms.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_Messy_feeding_MicroZoo.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_Messy_feeding_MesoZoo.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_SinkingRateDiatomsmin.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_SinkingRateDiatomsmax.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_MortalityFlagellates.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_MortalityEmiliana.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_ksNHs.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_MuMaxFlagellates.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_MuMaxEmiliana.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_Q10Phy.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_Q10PhyDiatoms.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_Q10Zoo.ne.0.0)    bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_dzetabio.ne.0.0)   bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_etabio.ne.0.0)   bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_lightabsB.ne.0.0)   bamhbi_sto_cpt=bamhbi_sto_cpt+1
      if (sto_Q10chem.ne.0.0)   bamhbi_sto_cpt=bamhbi_sto_cpt+1
#endif
      ln_sto_any=ln_sto_eos.or.ln_sto_trd.or.ln_sto_cdo.or.ln_sto_bfr.or.ln_sto_rnf.or.ln_sto_wnd.or.ln_sto_tair.or.ln_sto_slp.or. &
#ifdef key_top
       &    ln_sto_rnf_bio.or.ln_sto_depo.or. &
#endif
       &    ln_sto_wnd_eofs.or.ln_sto_tair_eofs.or.ln_sto_tcc_eofs.or.ln_sto_prec_eofs.or.ln_sto_wnd_fft
      
      IF( .NOT.ln_sto_any ) THEN   ! no use of stochastic parameterization
         IF(lwp) THEN
            WRITE(numout,*)
            WRITE(numout,*) 'sto_par_init : NO use of stochastic parameterization'
            WRITE(numout,*) '~~~~~~~~~~~~'
         ENDIF
         RETURN
      ENDIF


      ! 0b) Initialize seeds for random number generator
      ! ------------------------------------------------
      ! using different seeds for different processors (jarea)
      ! and different ensemble members (jmem)
      CALL kiss_reset(reset_from_clock=ln_rst_clock)
      DO jarea = 1, narea
         !DO jmem = 0, nmember
         zseed1 = kiss() ; zseed2 = kiss() ; zseed3 = kiss() ; zseed4 = kiss()
         !END DO
      END DO
      CALL kiss_seed( zseed1, zseed2, zseed3, zseed4 )


      ! 0c) print out perturbation options in ocean.output
      ! ---------------------------------------------------
      IF(lwp) THEN
         WRITE(numout,*)
         WRITE(numout,*) 'sto_par_init : stochastic parameterization'
         WRITE(numout,*) '~~~~~~~~~~~~'
         WRITE(numout,*) '   Namelist namsto : stochastic parameterization'
         WRITE(numout,*) '      seed from clock or from fixed value     ln_rst_clock  = ', ln_rst_clock
         WRITE(numout,*) '      restart stochastic parameters           ln_rststo     = ', ln_rststo
         WRITE(numout,*) '      read seed of RNG from restart file      ln_rstseed    = ', ln_rstseed
         WRITE(numout,*) '      suffix of sto restart name (input)      cn_storst_in  = ', cn_storst_in
         WRITE(numout,*) '      suffix of sto restart name (output)     cn_storst_out = ', cn_storst_out

         ! WRITE(numout,*) '      stochastic lateral diffusion            ln_sto_ldf    = ', ln_sto_ldf
         ! WRITE(numout,*) '      lateral diffusion std (in percent)      rn_ldf_std    = ', rn_ldf_std
         ! WRITE(numout,*) '      lateral diffusion tcor (in timesteps)   rn_ldf_tcor   = ', rn_ldf_tcor

         ! WRITE(numout,*) '      stochastic horizontal pressure gradient ln_sto_hpg    = ', ln_sto_hpg
         ! WRITE(numout,*) '      density gradient std (in percent)       rn_hpg_std    = ', rn_hpg_std
         ! WRITE(numout,*) '      density gradient tcor (in timesteps)    rn_hpg_tcor   = ', rn_hpg_tcor

         ! WRITE(numout,*) '      stochastic ice strength                 ln_sto_pstar  = ', ln_sto_pstar
         ! WRITE(numout,*) '      ice strength std (in percent)           rn_pstar_std  = ', rn_pstar_std
         ! WRITE(numout,*) '      ice strength tcor (in timesteps)        rn_pstar_tcor = ', rn_pstar_tcor
         ! WRITE(numout,*) '      order of autoregressive  processes      nn_pstar_ord  = ', nn_pstar_ord
         ! WRITE(numout,*) '      passes of Laplacian filter              nn_pstar_flt  = ', nn_pstar_flt

         WRITE(numout,*) '   stochastic trend                           ln_sto_trd    = ', ln_sto_trd
         if (ln_sto_trd) then
           WRITE(numout,*) '      trend std (in percent)                  rn_trd_std    = ', rn_trd_std
           WRITE(numout,*) '      trend tcor (in timesteps)               rn_trd_tcor   = ', rn_trd_tcor
           WRITE(numout,*) '      trend xycor (in grid points)            rn_trd_xycor  = ', rn_trd_xycor
           WRITE(numout,*) '      passes of Laplacian filter              nn_trd_flt    = ', nn_trd_flt
           WRITE(numout,*) '      limitation factor                       rn_trd_lim    = ', rn_trd_lim
         end if
	 
         WRITE(numout,*) '   stochastic equation of state               ln_sto_eos    = ', ln_sto_eos
         if (ln_sto_eos) then
           WRITE(numout,*) '      number of degrees of freedom            nn_sto_eos    = ', nn_sto_eos
           WRITE(numout,*) '      random walk horz. std (in grid points)  rn_eos_stdxy  = ', rn_eos_stdxy
           WRITE(numout,*) '      random walk vert. std (in grid points)  rn_eos_stdz   = ', rn_eos_stdz
           WRITE(numout,*) '      random walk tcor (in timesteps)         rn_eos_tcor   = ', rn_eos_tcor
           WRITE(numout,*) '      order of autoregressive  processes      nn_eos_ord    = ', nn_eos_ord
           WRITE(numout,*) '      passes of Laplacian filter              nn_eos_flt    = ', nn_eos_flt
           WRITE(numout,*) '      limitation factor                       rn_eos_lim    = ', rn_eos_lim
         end if

         WRITE(numout,*) '   stochastic tracers dynamics                ln_sto_trc    = ', ln_sto_trc
         if (ln_sto_trc) then
           WRITE(numout,*) '      number of degrees of freedom            nn_sto_trc    = ', nn_sto_trc
           WRITE(numout,*) '      random walk horz. std (in grid points)  rn_trc_stdxy  = ', rn_trc_stdxy
           WRITE(numout,*) '      random walk vert. std (in grid points)  rn_trc_stdz   = ', rn_trc_stdz
           WRITE(numout,*) '      random walk tcor (in timesteps)         rn_trc_tcor   = ', rn_trc_tcor
           WRITE(numout,*) '      order of autoregressive  processes      nn_trc_ord    = ', nn_trc_ord
           WRITE(numout,*) '      passes of Laplacian filter              nn_trc_flt    = ', nn_trc_flt
           WRITE(numout,*) '      limitation factor                       rn_trc_lim    = ', rn_trc_lim
         end if

         WRITE(numout,*) '   stochastic ocean drag coef. mom. Cd        ln_sto_cdo    = ', ln_sto_cdo
         WRITE(numout,*) '   stochastic ocean drag coef. turb.Ch,Ce     ln_sto_turb   = ', ln_sto_turb
         if (ln_sto_cdo) then
           WRITE(numout,*) '      ocean drag coef. std (in percent)       rn_cdo_std    = ', rn_cdo_std
           WRITE(numout,*) '      ocean drag coef. tcor (in timesteps)    rn_cdo_tcor   = ', rn_cdo_tcor
           WRITE(numout,*) '      ocean drag coef. xycor (in grid points) rn_cdo_xycor  = ', rn_cdo_xycor
           WRITE(numout,*) '      order of autoregressive  processes      nn_cdo_ord    = ', nn_cdo_ord
           WRITE(numout,*) '      passes of Laplacian filter              nn_cdo_flt    = ', nn_cdo_flt
           WRITE(numout,*) '      limitation factor                       rn_cdo_lim    = ', rn_cdo_lim
         end if

         WRITE(numout,*) '   stochastic ocean bottom friction coef.  ln_sto_bfr       = ', ln_sto_bfr
         if (ln_sto_bfr) then
           WRITE(numout,*) '      ocean bot. fric. std (in percent)       rn_bfr_std    = ', rn_bfr_std
           WRITE(numout,*) '      ocean bot. fric. tcor (in timesteps)    rn_bfr_tcor   = ', rn_bfr_tcor
           WRITE(numout,*) '      ocean bot. fric. xycor(in grid points)  rn_bfr_xycor  = ', rn_bfr_xycor
           WRITE(numout,*) '      order of autoregressive  processes      nn_bfr_ord    = ', nn_bfr_ord
           WRITE(numout,*) '      passes of Laplacian filter              nn_bfr_flt    = ', nn_bfr_flt
           WRITE(numout,*) '      limitation factor                       rn_bfr_lim    = ', rn_bfr_lim
         end if

         WRITE(numout,*) '   stochastic ocean river runoff           ln_sto_rnf       = ', ln_sto_rnf
         if (ln_sto_rnf) then
           WRITE(numout,*) '      river runoff std (in percent)           rn_rnf_std    = ', rn_rnf_std
           WRITE(numout,*) '      river runoff tcor (in timesteps)        rn_rnf_tcor   = ', rn_rnf_tcor
           WRITE(numout,*) '      river runoff xycor(in grid points)      rn_rnf_xycor  = ', rn_rnf_xycor
           WRITE(numout,*) '      order of autoregressive  processes      nn_rnf_ord    = ', nn_rnf_ord
           WRITE(numout,*) '      passes of Laplacian filter              nn_rnf_flt    = ', nn_rnf_flt
           WRITE(numout,*) '      limitation factor                       rn_rnf_lim    = ', rn_rnf_lim
         end if

         WRITE(numout,*) '   stochastic kpar coef.                      ln_sto_kpar   = ', ln_sto_kpar
         if (ln_sto_kpar) then
           WRITE(numout,*) '      kpar coef. std (in percent)             rn_kpar_std   = ', rn_kpar_std
           WRITE(numout,*) '      kpar coef. tcor (in timesteps)          rn_kpar_tcor  = ', rn_kpar_tcor
           WRITE(numout,*) '      kpar coef. xycor(in grid points)        rn_kpar_xycor = ', rn_kpar_xycor
           WRITE(numout,*) '      order of autoregressive  processes      nn_kpar_ord   = ', nn_kpar_ord
           WRITE(numout,*) '      passes of Laplacian filter              nn_kpar_flt   = ', nn_kpar_flt
           WRITE(numout,*) '      limitation factor                       rn_kpar_lim   = ', rn_kpar_lim
         end if

         WRITE(numout,*) '   stochastic u,v wind forcing                ln_sto_wnd    = ', ln_sto_wnd
         if (ln_sto_wnd) then
           WRITE(numout,*) '      u,v wind forcing std (in percent)       rn_wnd_std    = ', rn_wnd_std
           WRITE(numout,*) '      u,v wind forcing tcor (in timesteps)    rn_wnd_tcor   = ', rn_wnd_tcor
           WRITE(numout,*) '      u,v wind forcing xycor(in grid points)  rn_wnd_xycor  = ', rn_wnd_xycor
           WRITE(numout,*) '      order of autoregressive  processes      nn_wnd_ord    = ', nn_wnd_ord
           WRITE(numout,*) '      passes of Laplacian filter              nn_wnd_flt    = ', nn_wnd_flt
           WRITE(numout,*) '      limitation factor                       rn_wnd_lim    = ', rn_wnd_lim
         end if
        
         WRITE(numout,*) '   stochastic air temperature                 ln_sto_tair   = ', ln_sto_tair
         if (ln_sto_tair) then
           WRITE(numout,*) '      air temperature std (in percent)        rn_tair_std   = ', rn_tair_std
           WRITE(numout,*) '      air temperature tcor (in timesteps)     rn_tair_tcor  = ', rn_tair_tcor
           WRITE(numout,*) '      air temperature xycor(in grid points)   rn_tair_xycor = ', rn_tair_xycor
           WRITE(numout,*) '      order of autoregressive  processes      nn_tair_ord   = ', nn_tair_ord
           WRITE(numout,*) '      passes of Laplacian filter              nn_tair_flt   = ', nn_tair_flt
           WRITE(numout,*) '      limitation factor                       rn_tair_lim   = ', rn_tair_lim
         end if
           
         WRITE(numout,*) '   stochastic sea level pressure              ln_sto_slp    = ', ln_sto_slp
         if (ln_sto_slp) then
           WRITE(numout,*) '      sea level pres. std (in percent)        rn_slp_std    = ', rn_slp_std
           WRITE(numout,*) '      sea level pres. tcor (in timesteps)     rn_slp_tcor   = ', rn_slp_tcor
           WRITE(numout,*) '      sea level pres. xycor(in grid points)   rn_slp_xycor  = ', rn_slp_xycor
           WRITE(numout,*) '      order of autoregressive  processes      nn_slp_ord    = ', nn_slp_ord
           WRITE(numout,*) '      passes of Laplacian filter              nn_slp_flt    = ', nn_slp_flt
           WRITE(numout,*) '      limitation factor                       rn_slp_lim    = ', rn_slp_lim
         end if

         WRITE(numout,*) '   stochastic wind forcing based on EOFS      ln_sto_wnd_eofs    = ', ln_sto_wnd_eofs
         if (ln_sto_wnd_eofs) then
           WRITE(numout,*) '      u,v wind forcing std (in percent)       rn_wnd_eofs_std    = ', rn_wnd_eofs_std
           WRITE(numout,*) '      u,v wind forcing tcor (in timesteps)    rn_wnd_eofs_tcor   = ', rn_wnd_eofs_tcor
           WRITE(numout,*) '      order of autoregressive  processes      nn_wnd_eofs_ord    = ', nn_wnd_eofs_ord
           WRITE(numout,*) '      limitation factor                       rn_wnd_eofs_lim    = ', rn_wnd_eofs_lim
           WRITE(numout,*) '      EOFS to use:                            nn_wnd_eofs        = ', nn_wnd_eofs
         end if

         WRITE(numout,*) '   stochastic tair forcing based on EOFS      ln_sto_tair_eofs   = ', ln_sto_tair_eofs
         if (ln_sto_tair_eofs) then
           WRITE(numout,*) '      tair forcing std (in percent)           rn_tair_eofs_std   = ', rn_tair_eofs_std
           WRITE(numout,*) '      tair forcing tcor (in timesteps)        rn_tair_eofs_tcor  = ', rn_tair_eofs_tcor
           WRITE(numout,*) '      order of autoregressive processes       nn_tair_eofs_ord   = ', nn_tair_eofs_ord
           WRITE(numout,*) '      limitation factor                       rn_tair_eofs_lim   = ', rn_tair_eofs_lim
           WRITE(numout,*) '      EOFS to use:                            nn_tair_eofs       = ', nn_tair_eofs
         end if
        
         WRITE(numout,*) '   stochastic tcc forcing based on EOFS       ln_sto_tcc_eofs    = ', ln_sto_tcc_eofs
         if (ln_sto_tcc_eofs) then
           WRITE(numout,*) '      tcc forcing std (in percent)            rn_tcc_eofs_std    = ', rn_tcc_eofs_std
           WRITE(numout,*) '      tcc forcing tcor (in timesteps     )    rn_tcc_eofs_tcor   = ', rn_tcc_eofs_tcor
           WRITE(numout,*) '      order of autoregressive  processes      nn_tcc_eofs_ord    = ', nn_tcc_eofs_ord
           WRITE(numout,*) '      limitation factor                       rn_tcc_eofs_lim    = ', rn_tcc_eofs_lim
           WRITE(numout,*) '      EOFS to use:                            nn_tcc_eofs        = ', nn_tcc_eofs
         end if
           
         WRITE(numout,*) '   stochastic precipitations based on EOFS    ln_sto_prec_eofs    = ', ln_sto_prec_eofs
         if (ln_sto_prec_eofs) then
           WRITE(numout,*) '      precipitations std (in percent)         rn_prec_eofs_std    = ', rn_prec_eofs_std
           WRITE(numout,*) '      precipitations tcor (in timesteps)      rn_prec_eofs_tcor   = ', rn_prec_eofs_tcor
           WRITE(numout,*) '      order of autoregressive  processes      nn_prec_eofs_ord    = ', nn_prec_eofs_ord
           WRITE(numout,*) '      limitation factor                       rn_prec_eofs_lim    = ', rn_prec_eofs_lim
           WRITE(numout,*) '      EOFS to use:                            nn_prec_eofs        = ', nn_prec_eofs
         end if

         WRITE(numout,*) '   stochastic wind based on Fourier modes ln_sto_wnd_fft   = ', ln_sto_wnd_fft
         if (ln_sto_wnd_fft) then
           WRITE(numout,*) '      Fourier modes to use:                   nn_wnd_fft     = ', nn_wnd_fft
           WRITE(numout,*) '      wind std (in percent)                   rn_wnd_fft_std = ', rn_wnd_fft_std
           WRITE(numout,*) '      order of autoregressive  processes      nn_wnd_fft_ord = ', nn_wnd_fft_ord
           WRITE(numout,*) '      limitation factor                       rn_wnd_fft_lim = ', rn_wnd_fft_lim
         end if

#ifdef key_top
         WRITE(numout,*) '   stochastic ocean river BIO runoff       ln_sto_rnf_bio   = ', ln_sto_rnf_bio
         if (ln_sto_rnf_bio) then
           WRITE(numout,*) '      river runoff std (in percent)           rn_rnfbio_std    = ', rn_rnfbio_std
           WRITE(numout,*) '      river runoff tcor (in timesteps)        rn_rnfbio_tcor   = ', rn_rnfbio_tcor
           WRITE(numout,*) '      river runoff xycor(in grid points)      rn_rnfbio_xycor  = ', rn_rnfbio_xycor
           WRITE(numout,*) '      order of autoregressive  processes      nn_rnfbio_ord    = ', nn_rnfbio_ord
           WRITE(numout,*) '      passes of Laplacian filter              nn_rnfbio_flt    = ', nn_rnfbio_flt
           WRITE(numout,*) '      limitation factor                       rn_rnfbio_lim    = ', rn_rnfbio_lim
         end if

         WRITE(numout,*) '   stochastic ocean atmos. deposition      ln_sto_depo      = ', ln_sto_depo
         if (ln_sto_depo) then
           WRITE(numout,*) '      atmos. depo std (in percent)            rn_depo_std    = ', rn_depo_std
           WRITE(numout,*) '      atmos. depo tcor (in timesteps)         rn_depo_tcor   = ', rn_depo_tcor
           WRITE(numout,*) '      atmos. depo xycor(in grid points)       rn_depo_xycor  = ', rn_depo_xycor
           WRITE(numout,*) '      order of autoregressive  processes      nn_depo_ord    = ', nn_depo_ord
           WRITE(numout,*) '      passes of Laplacian filter              nn_depo_flt    = ', nn_depo_flt
           WRITE(numout,*) '      limitation factor                       rn_depo_lim    = ', rn_depo_lim
         end if
  
         WRITE(numout,*) '   stochastic BAMHBI model parameters                       : ', bamhbi_sto_cpt
#endif
      ENDIF

      IF(lwp) WRITE(numout,*)
      IF(lwp) WRITE(numout,*) '   stochastic parameterization :'

      ! Set number of 0D stochastic arrays, L.V.
      jpsto0d = 0
      if (ln_sto_wnd_eofs) THEN
         if(lwp) write(numout,*) '       - stochastic wind using EOFs'
         jsto_wnd_eofs = jpsto0d + 1
         jpsto0d = jpsto0d + nn_wnd_eofs
      else
         nn_wnd_eofs=0
      endif
      if (ln_sto_tair_eofs) THEN
         if(lwp) write(numout,*) '       - stochastic T2M using EOFs'
         jsto_tair_eofs = jpsto0d + 1
         jpsto0d = jpsto0d + nn_tair_eofs
      else
         nn_tair_eofs=0
      endif
      if (ln_sto_tcc_eofs) THEN
         if(lwp) write(numout,*) '       - stochastic TCC using EOFs'
         jsto_tcc_eofs = jpsto0d + 1
         jpsto0d = jpsto0d + nn_tcc_eofs
      else
         nn_tcc_eofs=0
      endif
      if (ln_sto_prec_eofs) THEN
         if(lwp) write(numout,*) '       - stochastic prec using EOFs'
         jsto_prec_eofs = jpsto0d + 1
         jpsto0d = jpsto0d + nn_prec_eofs
      else
         nn_prec_eofs=0
      endif
      if (ln_sto_wnd_fft) THEN
         if(lwp) write(numout,*) '       - stochastic wind using FFTs'
         jsto_wnd_fft = jpsto0d + 1
         jpsto0d = jpsto0d + nn_wnd_fft
      else
         nn_wnd_fft=0
      endif
      
      ! Set number of 2D stochastic arrays
      jpsto2d = 0
      IF( ln_sto_ldf ) THEN
         IF(lwp) WRITE(numout,*) '       - stochastic lateral diffusion'
         jpsto2d   = jpsto2d + 1
         jsto_ldf  = jpsto2d
      ENDIF
      IF( ln_sto_pstar ) THEN
         IF(lwp) WRITE(numout,*) '       - stochastic ice strength'
         jpsto2d    = jpsto2d + 1 * nn_pstar_ord
         jsto_pstar = jpsto2d
      ENDIF
      IF( ln_sto_eos ) THEN
         IF ( lk_agrif ) CALL ctl_stop('EOS stochastic parametrization is not compatible with AGRIF')
         IF(lwp) WRITE(numout,*) '       - stochastic equation of state'
         ALLOCATE(jsto_eosi(nn_sto_eos))
         ALLOCATE(jsto_eosj(nn_sto_eos))
         ALLOCATE(jsto_eosk(nn_sto_eos))
         DO jdof = 1, nn_sto_eos
            jpsto2d   = jpsto2d + 3 * nn_eos_ord
            jsto_eosi(jdof) = jpsto2d - 2 * nn_eos_ord
            jsto_eosj(jdof) = jpsto2d - 1 * nn_eos_ord
            jsto_eosk(jdof) = jpsto2d
         END DO
      ELSE
         nn_sto_eos = 0
      ENDIF
      IF( ln_sto_trc ) THEN
         IF(lwp) WRITE(numout,*) '       - stochastic tracers dynamics'
         ALLOCATE(jsto_trci(nn_sto_trc))
         ALLOCATE(jsto_trcj(nn_sto_trc))
         ALLOCATE(jsto_trck(nn_sto_trc))
         DO jdof = 1, nn_sto_trc
            jpsto2d   = jpsto2d + 3 * nn_trc_ord
            jsto_trci(jdof) = jpsto2d - 2 * nn_trc_ord
            jsto_trcj(jdof) = jpsto2d - 1 * nn_trc_ord
            jsto_trck(jdof) = jpsto2d
         END DO
      ELSE
         nn_sto_trc = 0
      ENDIF
      IF( ln_sto_cdo ) THEN
         IF ( lk_agrif ) CALL ctl_stop('stochastic ocean drag coefficient have not been tested with AGRIF')
         IF(lwp) WRITE(numout,*) '       - stochastic ocean drag coefficient'
         jpsto2d   = jpsto2d + 1 * nn_cdo_ord
         jsto_cdo  = jpsto2d
         IF(lwp) WRITE(numout,*) ' index of 2D stochastic parameters jpsto2d: jsto_cdo = ', jsto_cdo
      ENDIF
      IF( ln_sto_bfr ) THEN
         IF ( lk_agrif ) CALL ctl_stop('stochastic ocean bottom friction coefficient has not been tested with AGRIF')
         IF(lwp) WRITE(numout,*) '       - stochastic ocean drag coefficient'
         jpsto2d   = jpsto2d + 1 * nn_bfr_ord
         jsto_bfr  = jpsto2d
         IF(lwp) WRITE(numout,*) ' index of 2D stochastic parameters jpsto2d: jsto_bfr = ', jsto_bfr
      ENDIF
      IF( ln_sto_rnf ) THEN
         IF ( lk_agrif ) CALL ctl_stop('stochastic river runoff has not been tested with AGRIF')
         IF(lwp) WRITE(numout,*) '       - stochastic river runoff'
         jpsto2d   = jpsto2d + 1 * nn_rnf_ord
         jsto_rnf  = jpsto2d
         IF(lwp) WRITE(numout,*) ' index of 2D stochastic parameters jpsto2d: jsto_rnf = ', jsto_rnf
      ENDIF
      IF( ln_sto_kpar ) THEN
         IF ( lk_agrif ) CALL ctl_stop('stochastic kpar coefficient has not been tested with AGRIF')
         IF(lwp) WRITE(numout,*) '       - stochastic kpar coefficient'
         jpsto2d   = jpsto2d + 1 * nn_kpar_ord
         jsto_kpar = jpsto2d
         IF(lwp) WRITE(numout,*) ' index of 2D stochastic parameters jpsto2d: jsto_kpar = ', jsto_kpar
      ENDIF
      IF( ln_sto_wnd ) THEN
         IF ( lk_agrif ) CALL ctl_stop('stochastic u,v wind forcing has not been tested with AGRIF')
         IF(lwp) WRITE(numout,*) '       - stochastic u,v wind forcing'
         jpsto2d   = jpsto2d + 1 * nn_wnd_ord
         jsto_wnd  = jpsto2d
         IF(lwp) WRITE(numout,*) ' index of 2D stochastic parameters jpsto2d: jsto_wnd = ', jsto_wnd
      ENDIF
      IF( ln_sto_tair ) THEN
         IF ( lk_agrif ) CALL ctl_stop('stochastic air temperature has not been tested with AGRIF')
         IF(lwp) WRITE(numout,*) '       - stochastic air temperature'
         jpsto2d   = jpsto2d + 1 * nn_tair_ord
         jsto_tair = jpsto2d
         IF(lwp) WRITE(numout,*) ' index of 2D stochastic parameters jpsto2d: jsto_tair = ', jsto_tair
      ENDIF
      IF( ln_sto_slp ) THEN
         IF ( lk_agrif ) CALL ctl_stop('stochastic sea level pressure has not been tested with AGRIF')
         IF(lwp) WRITE(numout,*) '       - stochastic sea level pressure'
         jpsto2d   = jpsto2d + 1 * nn_slp_ord
         jsto_slp  = jpsto2d
         IF(lwp) WRITE(numout,*) ' index of 2D stochastic parameters jpsto2d: jsto_slp = ', jsto_slp
      ENDIF
#ifdef key_top
      IF( ln_sto_rnf_bio ) THEN
         IF ( lk_agrif ) CALL ctl_stop('stochastic river BIO runoff has not been tested with AGRIF')
         IF(lwp) WRITE(numout,*) '       - stochastic river BIO runoff'
         jpsto2d   = jpsto2d + 1 * nn_rnfbio_ord
         jsto_rnf_bio  = jpsto2d
         IF(lwp) WRITE(numout,*) ' index of 2D stochastic parameters jpsto2d: jsto_rnfbio = ', jsto_rnf_bio
      ENDIF
      IF( ln_sto_depo ) THEN
         IF ( lk_agrif ) CALL ctl_stop('stochastic atmospheric deposition has not been tested with AGRIF')
         IF(lwp) WRITE(numout,*) '       - stochastic atmospheric deposition'
         jpsto2d   = jpsto2d + 1 * nn_depo_ord
         jsto_depo  = jpsto2d
         IF(lwp) WRITE(numout,*) ' index of 2D stochastic parameters jpsto2d: jsto_depo = ', jsto_depo
      ENDIF
#endif

      ! Set number of 3D stochastic arrays
      jpsto3d = 0
      IF( ln_sto_hpg ) THEN
         IF(lwp) WRITE(numout,*) '       - stochastic horizontal pressure gradient'
         jpsto3d   = jpsto3d + 2
         jsto_hpgi = jpsto3d - 1
         jsto_hpgj = jpsto3d
      ENDIF
      IF( ln_sto_trd ) THEN
         IF(lwp) WRITE(numout,*) '       - stochastic trend'
         jpsto3d   = jpsto3d + 1
         jsto_trd  = jpsto3d
      ENDIF

      ! Allocate 0D stochastic arrays
      if (jpsto0d > 0 ) then
         allocate ( sto0d(jpsto0d) )
         ALLOCATE ( sto0d_abc(jpsto0d,3) )
         ALLOCATE ( sto0d_ave(jpsto0d) )
         ALLOCATE ( sto0d_std(jpsto0d) )
         ALLOCATE ( sto0d_lim(jpsto0d) )
         ALLOCATE ( sto0d_tcor(jpsto0d) )
         ALLOCATE ( sto0d_ord(jpsto0d) )
      end if
      
      ! Allocate 2D stochastic arrays
      IF ( jpsto2d > 0 ) THEN
         ALLOCATE ( sto2d(jpi,jpj,jpsto2d) )
         ALLOCATE ( sto2d_abc(jpsto2d,3) )
         ALLOCATE ( sto2d_ave(jpsto2d) )
         ALLOCATE ( sto2d_std(jpsto2d) )
         ALLOCATE ( sto2d_lim(jpsto2d) )
         IF ( jpsto3d == 0 ) ALLOCATE ( sto2d_xycor(jpsto2d) )
         ALLOCATE ( sto2d_tcor(jpsto2d) )
         ALLOCATE ( sto2d_ord(jpsto2d) )
         ALLOCATE ( sto2d_typ(jpsto2d) )
         ALLOCATE ( sto2d_sgn(jpsto2d) )
         ALLOCATE ( sto2d_flt(jpsto2d) )
         ALLOCATE ( sto2d_fac(jpsto2d) )
      ENDIF

      ! Allocate 3D stochastic arrays
      IF ( jpsto3d > 0 ) THEN
         ALLOCATE ( sto3d(jpi,jpj,jpk,jpsto3d) )
         ALLOCATE ( sto3d_abc(jpsto3d,3) )
         ALLOCATE ( sto3d_ave(jpsto3d) )
         ALLOCATE ( sto3d_std(jpsto3d) )
         ALLOCATE ( sto3d_lim(jpsto3d) )
         ALLOCATE ( sto2d_xycor(jpsto2d+jpsto3d) )
         ALLOCATE ( sto3d_tcor(jpsto3d) )
         ALLOCATE ( sto3d_ord(jpsto3d) )
         ALLOCATE ( sto3d_typ(jpsto3d) )
         ALLOCATE ( sto3d_sgn(jpsto3d) )
         ALLOCATE ( sto3d_flt(jpsto3d) )
         ALLOCATE ( sto3d_fac(jpsto3d) )
      ENDIF

      ! Allocate temporary workspace
      IF ( jpsto2d > 0 .OR. jpsto3d > 0 ) THEN
         ALLOCATE ( sto_tmp(jpi,jpj) ) ; sto_tmp(:,:) = 0._wp
      ENDIF

      ! allocate EOFs and/or FFTs
      if (ln_sto_wnd_eofs) then
        allocate(wnd_eofs(jpi,jpj,nn_wnd_eofs,2),stat=ierr)
        if (ierr.ne.0) call ctl_stop('allocation error for wnd_eofs')
      end if
      if (ln_sto_tair_eofs) then
        allocate(tair_eofs(jpi,jpj,nn_tair_eofs),stat=ierr)
        if (ierr.ne.0) call ctl_stop('allocation error for tair_eofs')
      end if
      if (ln_sto_tcc_eofs) then
        allocate(tcc_eofs(jpi,jpj,nn_tcc_eofs),stat=ierr)
        if (ierr.ne.0) call ctl_stop('allocation error for tcc_eofs')
      end if
      if (ln_sto_prec_eofs) then
        allocate(prec_eofs(jpi,jpj,nn_prec_eofs),stat=ierr)
        if (ierr.ne.0) call ctl_stop('allocation error for prec_eofs')
      end if
      if (ln_sto_wnd_fft) then
        allocate(wnd_ffts(jpi,jpj,nn_wnd_fft,2),stat=ierr)
        if (ierr.ne.0) call ctl_stop('allocation error for wnd_fft')
      end if      
      ! for FFT perturbations, also allocate and read array of periods
      if (ln_sto_wnd_fft) then
         call iom_nf90_open(cn_wnd_fft_file, fftncid, .false., .true.)
         vidOK = iom_nf90_varid(fftncid,trim(cn_wnd_fft_T),1)
         allocate(fft_periods(nn_wnd_fft))
         call iom_nf90_get(fftncid, 1,1,(/1/),(/nn_wnd_fft/),1,1,1,1,fft_periods)
         !call iom_nf90_close(fftncid) ! this gives an error at the end of the simulation
      end if
      
      
      ! 1) For every stochastic parameter:
      ! ----------------------------------
      ! - set nature of grid point and control of the sign
      !       across the north fold (sto2d_typ, sto2d_sgn)
      ! - set number of passes of Laplacian filter (sto2d_flt)
      ! - set order of every autoregressive process (sto2d_ord)
      DO jsto = 1, jpsto0d
         sto0d_ord(jsto) = 1
         if ( jsto .ge. jsto_wnd_eofs .and. jsto .lt. (jsto_wnd_eofs+nn_wnd_eofs) ) then
            sto0d_ord(jsto) = nn_wnd_eofs_ord
         end if
         if ( jsto .ge. jsto_tair_eofs .and. jsto .lt. (jsto_tair_eofs+nn_tair_eofs) ) then
            sto0d_ord(jsto) = nn_tair_eofs_ord
         end if
         if ( jsto .ge. jsto_tcc_eofs .and. jsto .lt. (jsto_tcc_eofs+nn_tcc_eofs) ) then
            sto0d_ord(jsto) = nn_tcc_eofs_ord
         end if
         if ( jsto .ge. jsto_prec_eofs .and. jsto .lt. (jsto_prec_eofs+nn_prec_eofs) ) then
            sto0d_ord(jsto) = nn_prec_eofs_ord
         end if
         if ( jsto .ge. jsto_wnd_fft .and. jsto .lt. (jsto_wnd_fft+nn_wnd_fft) ) then
            sto0d_ord(jsto) = nn_wnd_fft_ord
         end if
      end do
      !
      DO jsto = 1, jpsto2d
         sto2d_typ(jsto) = 'T'
         sto2d_sgn(jsto) = 1._wp
         sto2d_flt(jsto) = 0
         sto2d_ord(jsto) = 1
         DO jord = 0, nn_pstar_ord-1
            IF ( jsto+jord == jsto_pstar ) THEN ! Stochastic ice strength (ave=1)
               sto2d_ord(jsto) = nn_pstar_ord - jord
               sto2d_flt(jsto) = nn_pstar_flt
            ENDIF
         ENDDO
         DO jdof = 1, nn_sto_eos
         DO jord = 0, nn_eos_ord-1
            IF ( jsto+jord == jsto_eosi(jdof) ) THEN ! Stochastic equation of state i (ave=0)
               sto2d_ord(jsto) = nn_eos_ord - jord
               sto2d_sgn(jsto) = -1._wp
               sto2d_flt(jsto) = nn_eos_flt
            ENDIF
            IF ( jsto+jord == jsto_eosj(jdof) ) THEN ! Stochastic equation of state j (ave=0)
               sto2d_ord(jsto) = nn_eos_ord - jord
               sto2d_sgn(jsto) = -1._wp
               sto2d_flt(jsto) = nn_eos_flt
            ENDIF
            IF ( jsto+jord == jsto_eosk(jdof) ) THEN ! Stochastic equation of state k (ave=0)
               sto2d_ord(jsto) = nn_eos_ord - jord
               sto2d_flt(jsto) = nn_eos_flt
            ENDIF
         END DO
         END DO
         DO jdof = 1, nn_sto_trc
         DO jord = 0, nn_trc_ord-1
            IF ( jsto+jord == jsto_trci(jdof) ) THEN ! Stochastic tracers dynamics i (ave=0)
               sto2d_ord(jsto) = nn_trc_ord - jord
               sto2d_sgn(jsto) = -1._wp
               sto2d_flt(jsto) = nn_trc_flt
            ENDIF
            IF ( jsto+jord == jsto_trcj(jdof) ) THEN ! Stochastic tracers dynamics j (ave=0)
               sto2d_ord(jsto) = nn_trc_ord - jord
               sto2d_sgn(jsto) = -1._wp
               sto2d_flt(jsto) = nn_trc_flt
            ENDIF
            IF ( jsto+jord == jsto_trck(jdof) ) THEN ! Stochastic tracers dynamics k (ave=0)
               sto2d_ord(jsto) = nn_trc_ord - jord
               sto2d_flt(jsto) = nn_trc_flt
            ENDIF
         END DO
         END DO
         DO jord = 0, nn_cdo_ord-1
            IF ( jsto+jord == jsto_cdo ) THEN ! Stochastic ocean drag coef.
               sto2d_ord(jsto) = nn_cdo_ord - jord
               sto2d_flt(jsto) = nn_cdo_flt
            ENDIF
         ENDDO
         DO jord = 0, nn_bfr_ord-1
            IF ( jsto+jord == jsto_bfr ) THEN ! Stochastic ocean bottom friction coef.
               sto2d_ord(jsto) = nn_bfr_ord - jord
               sto2d_flt(jsto) = nn_bfr_flt
            ENDIF
         ENDDO
         DO jord = 0, nn_rnf_ord-1
            IF ( jsto+jord == jsto_rnf ) THEN ! Stochastic river runoff
               sto2d_ord(jsto) = nn_rnf_ord - jord
               sto2d_flt(jsto) = nn_rnf_flt
            ENDIF
         ENDDO
         DO jord = 0, nn_kpar_ord-1
            IF ( jsto+jord == jsto_kpar ) THEN ! Stochastic kpar coef.
               sto2d_ord(jsto) = nn_kpar_ord - jord
               sto2d_flt(jsto) = nn_kpar_flt
            ENDIF
         ENDDO
         DO jord = 0, nn_wnd_ord-1
            IF ( jsto+jord == jsto_wnd ) THEN ! Stochastic u,v wind forcing
               sto2d_ord(jsto) = nn_wnd_ord - jord
               sto2d_flt(jsto) = nn_wnd_flt
            ENDIF
         ENDDO
         DO jord = 0, nn_tair_ord-1
            IF ( jsto+jord == jsto_tair ) THEN ! Stochastic air temperature
               sto2d_ord(jsto) = nn_tair_ord - jord
               sto2d_flt(jsto) = nn_tair_flt
            ENDIF
         ENDDO
         DO jord = 0, nn_slp_ord-1
            IF ( jsto+jord == jsto_slp ) THEN ! Stochastic sea level pressure
               sto2d_ord(jsto) = nn_slp_ord - jord
               sto2d_flt(jsto) = nn_slp_flt
            ENDIF
         ENDDO
#ifdef key_top
         DO jord = 0, nn_rnfbio_ord-1
            IF ( jsto+jord == jsto_rnf_bio ) THEN ! Stochastic river runoff
               sto2d_ord(jsto) = nn_rnfbio_ord - jord
               sto2d_flt(jsto) = nn_rnfbio_flt
            ENDIF
         ENDDO
         DO jord = 0, nn_depo_ord-1
            IF ( jsto+jord == jsto_depo ) THEN ! Stochastic atmospheric deposition
               sto2d_ord(jsto) = nn_depo_ord - jord
               sto2d_flt(jsto) = nn_depo_flt
            ENDIF
         ENDDO

#endif
         sto2d_fac(jsto) = sto_par_flt_fac ( sto2d_flt(jsto) )
      END DO
      !
      DO jsto = 1, jpsto3d
         sto3d_typ(jsto) = 'T'
         sto3d_sgn(jsto) = 1._wp
         sto3d_flt(jsto) = 0
         sto3d_ord(jsto) = 1
         IF ( jsto == jsto_hpgi ) THEN ! Stochastic density gradient i (ave=1)
            sto3d_typ(jsto) = 'U'
         ENDIF
         IF ( jsto == jsto_hpgj ) THEN ! Stochastic density gradient j (ave=1)
            sto3d_typ(jsto) = 'V'
         ENDIF
         sto3d_fac(jsto) = sto_par_flt_fac ( sto3d_flt(jsto) )
      END DO

      ! 2) For every stochastic parameter:
      ! ----------------------------------
      ! set average, standard deviation and time correlation
      do jsto = 1, jpsto0d
         sto0d_ave(jsto)  = 0._wp
         sto0d_std(jsto)  = 1._wp
         sto0d_tcor(jsto) = 1._wp
         sto0d_lim(jsto)  = 3._wp
         sto0d_ord(jsto)  = 1
         if (jsto .ge. jsto_wnd_eofs .and. jsto .lt. (jsto_wnd_eofs+nn_wnd_eofs) ) then
            sto0d_std(jsto) = rn_wnd_eofs_std
            sto0d_tcor(jsto)= rn_wnd_eofs_tcor
            sto0d_lim(jsto) = rn_wnd_eofs_lim
            sto0d_ord(jsto) = nn_wnd_eofs_ord
         end if
         if (jsto .ge. jsto_tair_eofs .and. jsto .lt. (jsto_tair_eofs+nn_tair_eofs) ) then
            sto0d_std(jsto) = rn_tair_eofs_std
            sto0d_tcor(jsto)= rn_tair_eofs_tcor
            sto0d_lim(jsto) = rn_tair_eofs_lim
            sto0d_ord(jsto) = nn_tair_eofs_ord
         end if
         if (jsto .ge. jsto_tcc_eofs .and. jsto .lt. (jsto_tcc_eofs+nn_tcc_eofs) ) then
            sto0d_std(jsto) = rn_tcc_eofs_std
            sto0d_tcor(jsto)= rn_tcc_eofs_tcor
            sto0d_lim(jsto) = rn_tcc_eofs_lim
            sto0d_ord(jsto) = nn_tcc_eofs_ord
         end if
         if (jsto .ge. jsto_prec_eofs .and. jsto .lt. (jsto_prec_eofs+nn_prec_eofs) ) then
            sto0d_std(jsto) = rn_prec_eofs_std
            sto0d_tcor(jsto)= rn_prec_eofs_tcor
            sto0d_lim(jsto) = rn_prec_eofs_lim
            sto0d_ord(jsto) = nn_prec_eofs_ord
         end if
         if (jsto .ge. jsto_wnd_fft .and. jsto .lt. (jsto_wnd_fft+nn_wnd_fft) ) then
            sto0d_std(jsto) = rn_wnd_fft_std
            sto0d_tcor(jsto)= 3600 * fft_periods(jsto-jsto_wnd_fft+1) / rdt ! periods are in [hour], tcor is in [timesteps]
            sto0d_lim(jsto) = rn_wnd_fft_lim
            sto0d_ord(jsto) = nn_wnd_fft_ord
         end if
      end do
      !
      DO jsto = 1, jpsto2d
         sto2d_ave(jsto)  = 0._wp
         sto2d_std(jsto)  = 1._wp
         sto2d_xycor(jsto)= 1._wp
         sto2d_tcor(jsto) = 1._wp
         sto2d_lim(jsto)  = 3._wp
         IF ( jsto == jsto_ldf  ) THEN ! Stochastic lateral diffusion (ave=1)
            sto2d_ave(jsto)  = 1._wp
            sto2d_std(jsto)  = rn_ldf_std
            sto2d_tcor(jsto) = rn_ldf_tcor
         ENDIF
         DO jord = 0, nn_pstar_ord-1
            IF ( jsto+jord == jsto_pstar ) THEN ! Stochastic ice strength (ave=1)
               sto2d_std(jsto) = 1._wp
               sto2d_tcor(jsto) = rn_pstar_tcor
            ENDIF
         ENDDO
         DO jdof = 1, nn_sto_eos
         DO jord = 0, nn_eos_ord-1
            IF ( jsto+jord == jsto_eosi(jdof) ) THEN ! Stochastic equation of state i (ave=0)
               sto2d_std(jsto)  = rn_eos_stdxy
               sto2d_tcor(jsto) = rn_eos_tcor
               sto2d_lim(jsto)  = rn_eos_lim
            ENDIF
            IF ( jsto+jord == jsto_eosj(jdof) ) THEN ! Stochastic equation of state j (ave=0)
               sto2d_std(jsto)  = rn_eos_stdxy
               sto2d_tcor(jsto) = rn_eos_tcor
               sto2d_lim(jsto)  = rn_eos_lim
            ENDIF
            IF ( jsto+jord == jsto_eosk(jdof) ) THEN ! Stochastic equation of state k (ave=0)
               sto2d_std(jsto)  = rn_eos_stdz
               sto2d_tcor(jsto) = rn_eos_tcor
               sto2d_lim(jsto)  = rn_eos_lim
            ENDIF
         END DO
         END DO
         DO jdof = 1, nn_sto_trc
         DO jord = 0, nn_trc_ord-1
            IF ( jsto+jord == jsto_trci(jdof) ) THEN ! Stochastic tracer dynamics i (ave=0)
               sto2d_std(jsto)  = rn_trc_stdxy
               sto2d_tcor(jsto) = rn_trc_tcor
               sto2d_lim(jsto)  = rn_trc_lim
            ENDIF
            IF ( jsto+jord == jsto_trcj(jdof) ) THEN ! Stochastic tracer dynamics j (ave=0)
               sto2d_std(jsto)  = rn_trc_stdxy
               sto2d_tcor(jsto) = rn_trc_tcor
               sto2d_lim(jsto)  = rn_trc_lim
            ENDIF
            IF ( jsto+jord == jsto_trck(jdof) ) THEN ! Stochastic tracer dynamics k (ave=0)
               sto2d_std(jsto)  = rn_trc_stdz
               sto2d_tcor(jsto) = rn_trc_tcor
               sto2d_lim(jsto)  = rn_trc_lim
            ENDIF
         END DO
         END DO
         DO jord = 0, nn_cdo_ord-1
            IF ( jsto+jord == jsto_cdo  ) THEN ! Stochastic ocean drag coef.
               sto2d_ave(jsto)  = 1._wp
               sto2d_std(jsto)  = rn_cdo_std
               sto2d_tcor(jsto) = rn_cdo_tcor
               sto2d_xycor(jsto) = rn_cdo_xycor
               sto2d_lim(jsto)  = rn_cdo_lim
            ENDIF
         END DO
         DO jord = 0, nn_bfr_ord-1
            IF ( jsto+jord == jsto_bfr  ) THEN ! Stochastic ocean bottom friction coef.
               sto2d_ave(jsto)  = 1._wp
               sto2d_std(jsto)  = rn_bfr_std
               sto2d_tcor(jsto) = rn_bfr_tcor
               sto2d_xycor(jsto) = rn_bfr_xycor
               sto2d_lim(jsto)  = rn_bfr_lim
            ENDIF
         END DO
         DO jord = 0, nn_rnf_ord-1
            IF ( jsto+jord == jsto_rnf  ) THEN ! Stochastic river runoff
               sto2d_ave(jsto)  = 1._wp
               sto2d_std(jsto)  = rn_rnf_std
               sto2d_tcor(jsto) = rn_rnf_tcor
               sto2d_xycor(jsto) = rn_rnf_xycor
               sto2d_lim(jsto)  = rn_rnf_lim
            ENDIF
         END DO
         DO jord = 0, nn_kpar_ord-1
            IF ( jsto+jord == jsto_kpar  ) THEN ! Stochastic kpar coef.
               sto2d_ave(jsto)  = 1._wp
               sto2d_std(jsto)  = rn_kpar_std
               sto2d_tcor(jsto) = rn_kpar_tcor
               sto2d_xycor(jsto) = rn_kpar_xycor
               sto2d_lim(jsto)  = rn_kpar_lim
            ENDIF
         END DO
         DO jord = 0, nn_wnd_ord-1
            IF ( jsto+jord == jsto_wnd  ) THEN ! Stochastic u,v wind forcing
               sto2d_ave(jsto)  = 1._wp
               sto2d_std(jsto)  = rn_wnd_std
               sto2d_tcor(jsto) = rn_wnd_tcor
               sto2d_xycor(jsto) = rn_wnd_xycor
               sto2d_lim(jsto)  = rn_wnd_lim
            ENDIF
         END DO
         DO jord = 0, nn_tair_ord-1
            IF ( jsto+jord == jsto_tair  ) THEN ! Stochastic air temperature
               sto2d_ave(jsto)  = 1._wp
               sto2d_std(jsto)  = rn_tair_std
               sto2d_tcor(jsto) = rn_tair_tcor
               sto2d_xycor(jsto) = rn_tair_xycor
               sto2d_lim(jsto)  = rn_tair_lim
            ENDIF
         END DO
         DO jord = 0, nn_slp_ord-1
            IF ( jsto+jord == jsto_slp  ) THEN ! Stochastic sea level pressure
               sto2d_ave(jsto)  = 1._wp
               sto2d_std(jsto)  = rn_slp_std
               sto2d_tcor(jsto) = rn_slp_tcor
               sto2d_xycor(jsto) = rn_slp_xycor
               sto2d_lim(jsto)  = rn_slp_lim
            ENDIF
         END DO
#ifdef key_top
         DO jord = 0, nn_rnfbio_ord-1
            IF ( jsto+jord == jsto_rnf_bio  ) THEN ! Stochastic river BIO runoff
               sto2d_ave(jsto)  = 1._wp
               sto2d_std(jsto)  = rn_rnfbio_std
               sto2d_tcor(jsto) = rn_rnfbio_tcor
               sto2d_xycor(jsto) = rn_rnfbio_xycor
               sto2d_lim(jsto)  = rn_rnfbio_lim
            ENDIF
         END DO
         DO jord = 0, nn_depo_ord-1
            IF ( jsto+jord == jsto_depo  ) THEN ! Stochastic atmospheric deposition
               sto2d_ave(jsto)  = 1._wp
               sto2d_std(jsto)  = rn_depo_std
               sto2d_tcor(jsto) = rn_depo_tcor
               sto2d_xycor(jsto) = rn_depo_xycor
               sto2d_lim(jsto)  = rn_depo_lim
            ENDIF
         END DO

#endif
      END DO
      !
      DO jsto = 1, jpsto3d
         sto3d_ave(jsto)  = 0._wp
         sto3d_std(jsto)  = 1._wp
         sto2d_xycor(jpsto2d+jsto) = 1._wp
         sto3d_tcor(jsto) = 1._wp
         sto3d_lim(jsto)  = 3._wp
         IF ( jsto == jsto_hpgi ) THEN ! Stochastic density gradient i (ave=1)
            sto3d_ave(jsto)  = 1._wp
            sto3d_std(jsto)  = rn_hpg_std
            sto3d_tcor(jsto) = rn_hpg_tcor
         ENDIF
         IF ( jsto == jsto_hpgj ) THEN ! Stochastic density gradient j (ave=1)
            sto3d_ave(jsto)  = 1._wp
            sto3d_std(jsto)  = rn_hpg_std
            sto3d_tcor(jsto) = rn_hpg_tcor
         ENDIF
         IF ( jsto == jsto_trd ) THEN ! Stochastic trend (ave=1)
            sto3d_ave(jsto)  = 1._wp
            sto3d_std(jsto)  = rn_trd_std
            sto3d_tcor(jsto) = rn_trd_tcor
            sto2d_xycor(jpsto2d+jsto) = rn_trd_xycor
            sto3d_lim(jsto)   = rn_trd_lim
         ENDIF
      END DO

      ! 3) For every stochastic parameter:
      ! ----------------------------------
      ! - compute parameters (a, b, c) of the AR1 autoregressive process
      !   from expected value (ave), standard deviation (std)
      !   and time correlation (tcor):
      !     a = EXP ( - 1 / tcor )           --> sto2d_abc(:,1)
      !     b = std * SQRT( 1 - a * a )      --> sto2d_abc(:,2)
      !     c = ave * ( 1 - a )              --> sto2d_abc(:,3)
      ! - for higher order processes (ARn, n>1), use approximate formula
      !   for the b parameter (valid for tcor>>1 time step)
      do jsto = 1, jpsto0d
         IF ( sto0d_tcor(jsto) == 0._wp ) THEN
            sto0d_abc(jsto,1) = 0._wp
         ELSE
            sto0d_abc(jsto,1) = EXP ( - 1._wp / sto0d_tcor(jsto) )
         ENDIF
         IF ( sto0d_ord(jsto) == 1 ) THEN      ! Exact formula for 1st order process
            rinflate = sto0d_std(jsto)
         ELSE
            ! Approximate formula, valid for tcor >> 1
            jordm1 = sto0d_ord(jsto) - 1
            rinflate = SQRT ( REAL( jordm1 , wp ) / REAL( 2*(2*jordm1-1) , wp ) )
         ENDIF
         sto0d_abc(jsto,2) = rinflate * SQRT ( 1._wp - sto0d_abc(jsto,1) &
                                                     * sto0d_abc(jsto,1) )
         sto0d_abc(jsto,3) = sto0d_ave(jsto) * ( 1._wp - sto0d_abc(jsto,1) )
      end do
      !
      DO jsto = 1, jpsto2d
         IF ( sto2d_tcor(jsto) == 0._wp ) THEN
            sto2d_abc(jsto,1) = 0._wp
         ELSE
            sto2d_abc(jsto,1) = EXP ( - 1._wp / sto2d_tcor(jsto) )
         ENDIF
         IF ( sto2d_ord(jsto) == 1 ) THEN      ! Exact formula for 1st order process
            rinflate = sto2d_std(jsto)
         ELSE
            ! Approximate formula, valid for tcor >> 1
            jordm1 = sto2d_ord(jsto) - 1
            rinflate = SQRT ( REAL( jordm1 , wp ) / REAL( 2*(2*jordm1-1) , wp ) )
         ENDIF
         sto2d_abc(jsto,2) = rinflate * SQRT ( 1._wp - sto2d_abc(jsto,1) &
                                                     * sto2d_abc(jsto,1) )
         sto2d_abc(jsto,3) = sto2d_ave(jsto) * ( 1._wp - sto2d_abc(jsto,1) )
      END DO
      !
      DO jsto = 1, jpsto3d
         IF ( sto3d_tcor(jsto) == 0._wp ) THEN
            sto3d_abc(jsto,1) = 0._wp
         ELSE
            sto3d_abc(jsto,1) = EXP ( - 1._wp / sto3d_tcor(jsto) )
         ENDIF
         IF ( sto3d_ord(jsto) == 1 ) THEN      ! Exact formula for 1st order process
            rinflate = sto3d_std(jsto)
         ELSE
            ! Approximate formula, valid for tcor >> 1
            jordm1 = sto3d_ord(jsto) - 1
            rinflate = SQRT ( REAL( jordm1 , wp ) / REAL( 2*(2*jordm1-1) , wp ) )
         ENDIF
         sto3d_abc(jsto,2) = rinflate * SQRT ( 1._wp - sto3d_abc(jsto,1) &
                                                     * sto3d_abc(jsto,1) )
         sto3d_abc(jsto,3) = sto3d_ave(jsto) * ( 1._wp - sto3d_abc(jsto,1) )
      END DO

      ! 4) Initialize stochastic parameters to: ave + std * w
      ! -----------------------------------------------------
      do jsto = 1, jpsto0d
         CALL kiss_gaussian( gran )
         sto0d(jsto)=gran
         ! Limit random parameter to the limitation factor
         sto0d(jsto) = SIGN(MIN(sto0d_lim(jsto),ABS(sto0d(jsto))),sto0d(jsto))
         ! Multiply by standard devation and add average value
         sto0d(jsto) = sto0d(jsto) * sto0d_std(jsto) + sto0d_ave(jsto)
      end do
      ! the value should be the same for all processors
      if (jpsto0d>0) call sto_share_0d( )
      !
      DO jsto = 1, jpsto2d
         IF ( sto2d_xycor(jsto) > 1._wp ) THEN
            ! Introduce random horiz. 2D Gaussian structures --> w
            CALL sto_par_xygau( jsto, sto2d(:,:,jsto) )
         ELSE
            ! Draw random numbers from N(0,1) --> w
            CALL sto_par_white( sto2d(:,:,jsto) )
         ENDIF
         ! Apply horizontal Laplacian filter to w
         DO jflt = 1, sto2d_flt(jsto)
            CALL lbc_lnk( 'stopar', sto2d(:,:,jsto), sto2d_typ(jsto), sto2d_sgn(jsto) )
            CALL sto_par_flt( sto2d(:,:,jsto) )
         END DO
         ! Factor to restore standard deviation after filtering
         sto2d(:,:,jsto) = sto2d(:,:,jsto) * sto2d_fac(jsto)
         ! Limit random parameter to the limitation factor
         sto2d(:,:,jsto) = SIGN(MIN(sto2d_lim(jsto),ABS(sto2d(:,:,jsto))),sto2d(:,:,jsto))
         ! Multiply by standard devation and add average value
         sto2d(:,:,jsto) = sto2d(:,:,jsto) * sto2d_std(jsto) + sto2d_ave(jsto)
      END DO
      !
      DO jsto = 1, jpsto3d
         DO jk = 1, jpk
            IF ( sto2d_xycor(jpsto2d+jsto) > 1._wp ) THEN
               ! Introduce random horiz. 2D Gaussian structures --> w
               CALL sto_par_xygau( jpsto2d+jsto, sto3d(:,:,jk,jsto) )
            ELSE
            ! Draw random numbers from N(0,1) --> w
            CALL sto_par_white( sto3d(:,:,jk,jsto) )
            ENDIF
            ! Apply horizontal Laplacian filter to w
            DO jflt = 1, sto3d_flt(jsto)
               CALL lbc_lnk( 'stopar', sto3d(:,:,jk,jsto), sto3d_typ(jsto), sto3d_sgn(jsto) )
               CALL sto_par_flt( sto3d(:,:,jk,jsto) )
            END DO
            ! Factor to restore standard deviation after filtering
            sto3d(:,:,jk,jsto) = sto3d(:,:,jk,jsto) * sto3d_fac(jsto)
            ! Limit random parameter to the limitation factor
            sto3d(:,:,jk,jsto) = SIGN(MIN(sto3d_lim(jsto),ABS(sto3d(:,:,jk,jsto))),sto3d(:,:,jk,jsto))
            ! Multiply by standard devation and add average value
            sto3d(:,:,jk,jsto) = sto3d(:,:,jk,jsto) * sto3d_std(jsto) + sto3d_ave(jsto)
         END DO
      END DO

#ifdef key_top
      ! 4bis) Randomize bamhbi parameters (once and for all)
      ! ----------------------------------------------------
      ! for a Gamma distribution G(k,theta) : (1) decide k (see Prieur et al), (2) compute theta = (sto/mu * mu) / sqrt(k), (3) draw in Gamma(k,1) and multiply with theta
      ! for a Beta distribution Beta(alpha,beta) , if E(X)=µ and Var(X)=v, then  (1) alpha=m*(m*(1-m)/v-1) ; beta=(1-m)*(m*(1-m)/v-1), (2) draw in Beta(alpha,beta)
      !
      if (sto_MaxgrazingrateMesoZoo.ne.0.0) then
         !call kiss_gaussian( gran ) ;  MaxgrazingrateMesoZoo = MaxgrazingrateMesoZoo * min(max(0.0,max(1-2*sto_MaxgrazingrateMesoZoo,(1+sto_MaxgrazingrateMesoZoo*gran))),1+2*sto_MaxgrazingrateMesoZoo)
         k=1/sto_MaxgrazingrateMesoZoo**2 ; theta = sto_MaxgrazingrateMesoZoo * (MaxgrazingrateMesoZoo*daytosecond) / sqrt(k)
         call kiss_gamma (gran,k) ; MaxgrazingrateMesoZoo = theta * gran / daytosecond ;
         if (lwp) write(numout,*) 'MaxgrazingrateMesoZoo perturbed: ', MaxGrazingRateMesoZoo
      end if
      if (sto_alphaPIDiatoms.ne.0.0) then
         k=1/sto_alphaPIDiatoms**2 ; theta = sto_alphaPIDiatoms * (alphaPIDiatoms*daytosecond) / sqrt(k)
         call kiss_gamma (gran,k) ; alphaPIDiatoms = theta * gran / daytosecond ;
         if (lwp) write(numout,*) 'alphaPIDiatoms perturbed: ', alphapidiatoms
      end if
      if (sto_Ass_Eff_OnCarbon.ne.0.0) then
         mean=Ass_Eff_OnCarbon ; variance=(sto_Ass_Eff_OnCarbon*mean)**2 ;      alpha=mean*(mean*(1-mean)/variance-1) ; beta=(1-mean)*(mean*(1-mean)/variance-1)
         call kiss_beta( Ass_Eff_OnCarbon , alpha,beta )
         if (lwp) write(numout,*) 'Ass_Eff_OnCarbon perturbed: ', Ass_Eff_OnCarbon
      end if
      if (sto_efficiency_growth_MesoZoo.ne.0.0) then
         mean=efficiency_growth_MesoZoo ; variance=(sto_efficiency_growth_MesoZoo*mean)**2 ;      alpha=mean*(mean*(1-mean)/variance-1) ; beta=(1-mean)*(mean*(1-mean)/variance-1)
         call kiss_beta( efficiency_growth_MesoZoo , alpha,beta )
         if (lwp) write(numout,*) 'efficiency_growth_MesoZoo perturbed: ', efficiency_growth_MesoZoo
      end if
      if (sto_nlin_mort_MESozoo.ne.0.0) then
         k=1/sto_nlin_mort_MESozoo**2 ; theta = sto_nlin_mort_MESozoo * nlin_mort_MESozoo / sqrt(k)
         call kiss_gamma (gran,k) ; nlin_mort_MESozoo = theta * gran
         if (lwp) write(numout,*) 'nlin_mort_MESozoo perturbed: ',nlin_mort_MESozoo
      end if
      if (sto_HalfSatMort_MesoZoo.ne.0.0) then
         k=1/sto_HalfSatMort_MESozoo**2 ; theta = sto_HalfSatMort_MESozoo * HalfSatMort_MESozoo / sqrt(k)
         call kiss_gamma (gran,k) ; HalfSatMort_MesoZoo = theta * gran
         if (lwp) write(numout,*) 'HalfSatMort_MesoZoo perturbed: ', HalfSatMort_MesoZoo
      end if
      if (sto_Capt_eff_Gelatinous_mesozoo.ne.0.0) then
         mean=Capt_eff_Gelatinous_mesozoo ; variance=(sto_Capt_eff_Gelatinous_mesozoo*mean)**2
         if (mean.gt.0.0.and.mean.lt.1.0) then
            alpha=mean*(mean*(1-mean)/variance-1) ; beta=(1-mean)*(mean*(1-mean)/variance-1)
            call kiss_beta( Capt_eff_Gelatinous_mesozoo , alpha,beta )
         elseif (mean.eq.1.0) then
            call kiss_gaussian( gran )
            Capt_eff_gelatinous_mesozoo=mean-(abs(gran)*sqrt(variance))
         elseif (mean.eq.0.0) then
            call kiss_gaussian( gran )
            Capt_eff_gelatinous_mesozoo=abs(gran)*sqrt(variance)
         else
            if (lwp) write(numout,*) 'I don''t know how to generate a random Capt_eff_Gelatinous_mesozoo'
            call ctl_stop( 'STOP', 'sto_par: invalid random generation for Capt_eff_Gelatinous_mesozoo' )
         endif
         if (lwp) write(numout,*) 'Capt_eff_Gelatinous_mesozoo perturbed: ', Capt_eff_Gelatinous_mesozoo
      end if
      if (sto_MaxgrazingrateGelatinous.ne.0.0) then
         k=1/sto_MaxgrazingrateGelatinous**2 ; theta = sto_MaxgrazingrateGelatinous * (MaxgrazingrateGelatinous*daytosecond) / sqrt(k)
         call kiss_gamma (gran,k) ; MaxgrazingrateGelatinous = theta * gran / daytosecond ;
         if (lwp) write(numout,*) 'MaxgrazingrateGelatinous perturbed: ', MaxgrazingrateGelatinous
      end if
      if (sto_Capt_eff_MesoZoo_Diatoms.ne.0.0) then
         mean=Capt_eff_MesoZoo_Diatoms ; variance=(sto_Capt_eff_mesozoo_diatoms*mean)**2
         if (mean.gt.0.0.and.mean.lt.1.0) then
            alpha=mean*(mean*(1-mean)/variance-1) ; beta=(1-mean)*(mean*(1-mean)/variance-1)
            call kiss_beta( Capt_eff_mesozoo_diatoms , alpha,beta )
         elseif (mean.eq.1.0) then
            call kiss_gaussian( gran )
            Capt_eff_mesozoo_diatoms=mean-(abs(gran)*sqrt(variance))
         elseif (mean.eq.0.0) then
            call kiss_gaussian( gran )
            Capt_eff_mesozoo_diatoms=abs(gran)*sqrt(variance)
         else
            if (lwp) write(numout,*) 'I don''t know how to generate a random Capt_eff_mesozoo_diatoms'
            call ctl_stop( 'STOP', 'sto_par: invalid random generation for Capt_eff_mesozoo_diatoms' )
         endif            
         if (lwp) write(numout,*) 'Capt_eff_MesoZoo_Diatoms perturbed: ', Capt_eff_mesozoo_diatoms
      end if
      if (sto_Ass_Eff_OnNitrogen.ne.0.0) then
         mean=Ass_Eff_OnNitrogen ; variance=(sto_Ass_Eff_OnNitrogen*mean)**2 ;      alpha=mean*(mean*(1-mean)/variance-1) ; beta=(1-mean)*(mean*(1-mean)/variance-1)
         call kiss_beta( Ass_Eff_OnNitrogen , alpha,beta )
         if (lwp) write(numout,*) 'Ass_Eff_OnNitrogen perturbed: ', Ass_Eff_OnNitrogen
      end if
      if (sto_MuMaxDiatoms.ne.0.0) then
         k=1/sto_MuMaxDiatoms**2 ; theta = sto_MuMaxDiatoms * (MuMaxDiatoms*daytosecond) / sqrt(k)
         call kiss_gamma (gran,k) ; MuMaxDiatoms = theta * gran / daytosecond
         if (lwp) write(numout,*) 'MuMaxDiatoms perturbed: ', MuMaxDiatoms
      end if
      if (sto_SiNrDiatoms.ne.0.0) then ! k =25.0
         k=1/sto_SiNrDiatoms**2 ; theta = sto_SiNrDiatoms * SiNrDiatoms / sqrt(k)
         call kiss_gamma( gran,k ) ;  SiNrDiatoms = theta * gran
         if (lwp) write(numout,*) 'SiNrDiatoms perturbed: ', SiNrDiatoms
      end if
      if (sto_MinNCrDiatoms.ne.0.0) then
         k=1/sto_MinNCrDiatoms**2 ; theta = sto_MinNCrDiatoms * MinNCrDiatoms / sqrt(k)
         call kiss_gamma( gran,k ) ;  MinNCrDiatoms = theta * gran
         if (lwp) write(numout,*) 'MinNCrDiatoms perturbed: ', MinNCrDiatoms
      end if
      if (sto_MortalityDiatoms.ne.0.0) then
         k=1/sto_MortalityDiatoms**2 ; theta = sto_MortalityDiatoms * (MortalityDiatoms*daytosecond) / sqrt(k)
         call kiss_gamma( gran,k ) ;  MortalityDiatoms = theta * gran / daytosecond
         if (lwp) write(numout,*) 'MortalityDiatoms perturbed: ', MortalityDiatoms
      end if
      if (sto_Messy_feeding_MicroZoo.ne.0.0) then
         mean=Messy_feeding_MicroZoo ; variance=(sto_Messy_feeding_MicroZoo*mean)**2 ;      alpha=mean*(mean*(1-mean)/variance-1) ; beta=(1-mean)*(mean*(1-mean)/variance-1)
         !if (lwp) write(numout,*) "alpha,beta=",alpha,beta
         call kiss_beta( Messy_feeding_MicroZoo , alpha,beta )
         if (lwp) write(numout,*) 'Messy_feeding_MicroZoo perturbed: ', Messy_feeding_MicroZoo
      end if
      if (sto_Messy_feeding_MesoZoo.ne.0.0) then
         mean=Messy_feeding_MesoZoo ; variance=(sto_Messy_feeding_MesoZoo*mean)**2 ;      alpha=mean*(mean*(1-mean)/variance-1) ; beta=(1-mean)*(mean*(1-mean)/variance-1)
         call kiss_beta( Messy_feeding_MesoZoo , alpha,beta )
         if (lwp) write(numout,*) 'Messy_feeding_MesoZoo perturbed: ', Messy_feeding_MesoZoo
      end if
      if (sto_SinkingRateDiatomsmin.ne.0.0) then
         k=1/sto_SinkingRateDiatomsmin**2 ; theta = sto_SinkingRateDiatomsmin * (SinkingRateDiatomsmin*daytosecond) / sqrt(k)
         call kiss_gamma( gran,k ) ;  SinkingRateDiatomsmin = theta * gran / daytosecond
         if (lwp) write(numout,*) 'SinkingRateDiatomsmin perturbed: ', SinkingRateDiatomsmin
      end if
      if (sto_SinkingRateDiatomsmax.ne.0.0) then
         k=1/sto_SinkingRateDiatomsmax**2 ; theta = sto_SinkingRateDiatomsmax * (SinkingRateDiatomsmax*daytosecond) / sqrt(k)
         call kiss_gamma( gran,k ) ;  SinkingRateDiatomsmax = theta * gran / daytosecond
         if (lwp) write(numout,*) 'SinkingRateDiatomsmax perturbed: ', SinkingRateDiatomsmax
      end if
      if (sto_SinkingRateDiatomsmax.ne.0.0 .or. sto_SinkingRateDiatomsmin.ne.0.0) then
         if (SinkingRateDiatomsmin.gt.SinkingRateDiatomsmax) then
            mean=0.5*(SinkingRateDiatomsmin+SinkingRateDiatomsmax)
            SinkingRateDiatomsmin=mean*0.95
            SinkingRateDiatomsmax=mean*1.05
         end if
      end if
      if (sto_MortalityFlagellates.eq.9999.) then
         MortalityFlagellates = MortalityDiatoms
      elseif(sto_MortalityFlagellates.ne.0.0) then
         k=1/sto_MortalityFlagellates**2 ; theta = sto_MortalityFlagellates * (MortalityFlagellates*daytosecond) / sqrt(k)
         call kiss_gamma( gran,k ) ;  MortalityFlagellates = theta * gran / daytosecond
         if (lwp) write(numout,*) 'MortalityFlagellates perturbed: ', MortalityFlagellates
      end if
      if (sto_MortalityEmiliana.eq.9999.) then
         MortalityEmiliana = MortalityDiatoms
      elseif(sto_MortalityEmiliana.ne.0.0) then
         k=1/sto_MortalityEmiliana**2 ; theta = sto_MortalityEmiliana * (MortalityEmiliana*daytosecond) / sqrt(k)
         call kiss_gamma( gran,k ) ;  MortalityEmiliana = theta * gran / daytosecond
         if (lwp) write(numout,*) 'MortalityEmiliana perturbed: ', MortalityEmiliana
      end if
      if (sto_ksNHs.ne.0.0) then
         call kiss_gaussian( gran ) ;  ksNHs = ksNHs * min(max(0.0,max(1-2*sto_ksNHs,(1+sto_ksNHs*gran))),1+2*sto_ksNHs)
         ksNHsFlagellates=ksNHsFlagellates*ksNHs ; ksNHsEmiliana=ksNHsEmiliana*ksNHs ; ksNHsDiatoms=ksNHsDiatoms*ksNHs
         if (lwp) write(numout,*) 'ksNHs (all phyto groups) multiplicative perturbation : ', min(max(0.0,max(1-2*sto_ksNHs,(1+sto_ksNHs*gran))),1+2*sto_ksNHs)
      end if
      if (sto_MuMaxFlagellates.ne.0.0) then
         k=1/sto_MuMaxFlagellates**2 ; theta = sto_MuMaxFlagellates * (MuMaxFlagellates*daytosecond) / sqrt(k)
         call kiss_gamma (gran,k) ; MuMaxFlagellates = theta * gran / daytosecond
         if (lwp) write(numout,*) 'MuMaxFlagellates perturbed: ', MuMaxFlagellates
      end if
      if (sto_MuMaxEmiliana.ne.0.0) then
         k=1/sto_MuMaxEmiliana**2 ; theta = sto_MuMaxEmiliana * (MuMaxEmiliana*daytosecond) / sqrt(k)
         call kiss_gamma (gran,k) ; MuMaxEmiliana = theta * gran / daytosecond
         if (lwp) write(numout,*) 'MuMaxEmiliana perturbed: ', MuMaxEmiliana
      end if
      if (sto_Q10Phy.ne.0.0) then
         k=10 ; theta = log(Q10Phy) / k  ! when Q10Phy=1.8, then  :  k=4 -->  std(Q10Phy)/Q10Phy = 37.5%   ; k=5 --> 31.7% ;  k=10 --> 20.2%  ; k=20 --> 13.7%
         call kiss_gamma (gran,k) ; Q10Phy=exp( gran * theta )
         if (lwp) write(numout,*) 'Q10Phy perturbed: ', Q10Phy   ! sto_Q10Phy is not actually used (apart from checking that it's not zero) ...
      end if
      if (sto_Q10PhyDiatoms.eq.9999.) then
         Q10PhyDiatoms = Q10Phy
      elseif (sto_Q10PhyDiatoms.ne.0.0) then
         k=10 ; theta = log(Q10PhyDiatoms) / k
         call kiss_gamma( gran,k ) ;  Q10PhyDiatoms = exp( gran * theta )
         if (lwp) write(numout,*) 'Q10PhyDiatoms perturbed: ', Q10PhyDiatoms
      end if
      if (sto_Q10Zoo.eq.9999.) then
         Q10Zoo = Q10Phy
      elseif (sto_Q10Zoo.ne.0.0) then
         k=10 ; theta = log(Q10Zoo) / k
         call kiss_gamma( gran,k ) ;  Q10Zoo = exp( gran * theta )
         if (lwp) write(numout,*) 'Q10Zoo perturbed: ', Q10Zoo
      end if
      if (sto_dzetabio.ne.0.0) then
         k=1/sto_dzetabio**2 ; theta = sto_dzetabio * dzetabio / sqrt(k)
         call kiss_gamma( gran,k ) ; dzetabio = gran*theta
         if (lwp) write(numout,*) 'dzetabio perturbed: ', dzetabio
      end if
      if (sto_etabio.ne.0.0) then
         k=1/sto_etabio**2 ; theta = sto_etabio * etabio / sqrt(k)
         call kiss_gamma( gran,k ) ; etabio = gran*theta
          if (lwp) write(numout,*) 'etabio perturbed: ', etabio
      end if
      if (sto_lightAbsB.ne.0.0) then
         mean=lightAbsB ; variance=(sto_lightAbsB*mean)**2 ;      alpha=mean*(mean*(1-mean)/variance-1) ; beta=(1-mean)*(mean*(1-mean)/variance-1)
         call kiss_beta( lightAbsB , alpha,beta )
         if (lwp) write(numout,*) 'LightAbsB perturbed: ', LightAbsB
      end if
      if (sto_Q10Chem.eq.9999.) then
         Q10Chem = Q10Phy
      elseif (sto_Q10Chem.ne.0.0) then
         k=10 ; theta = log(Q10Chem) / k
         call kiss_gamma( gran,k ) ;  Q10Chem = exp( gran * theta )
         if (lwp) write(numout,*) 'Q10Chem perturbed: ', Q10Chem
      end if
#endif

      ! 5) Restart stochastic parameters from file
      ! ------------------------------------------
      IF( ln_rstart .AND. ln_rststo ) CALL sto_rst_read

   END SUBROUTINE sto_par_init


   SUBROUTINE sto_rst_read
      !!----------------------------------------------------------------------
      !!                  ***  ROUTINE sto_rst_read  ***
      !!
      !! ** Purpose :   read stochastic parameters from restart file
      !!----------------------------------------------------------------------
      INTEGER             ::   jsto, jseed
      INTEGER             ::   iomvarid
      INTEGER             ::   idg                 ! number of digits
      INTEGER(KIND=8)     ::   ziseed(4)           ! RNG seeds in integer type
      REAL(KIND=dp)       ::   zrseed(4)           ! RNG seeds in double-precision (with same bits to save in restart)
      CHARACTER(LEN=9)    ::   clsto0d='sto0d_000' ! stochastic parameter variable name
      CHARACTER(LEN=9)    ::   clsto2d='sto2d_000' ! stochastic parameter variable name
      CHARACTER(LEN=9)    ::   clsto3d='sto3d_000' ! stochastic parameter variable name
      CHARACTER(LEN=15)   ::   clseed='seed0_0000' ! seed variable name
      CHARACTER(LEN=6)    ::   clfmt               ! writing format
      !!----------------------------------------------------------------------

      IF ( &
#ifdef key_top
     &  bamhbi_sto_cpt > 0  .OR.  &
#endif
     &       jpsto0d > 0 .OR. jpsto2d > 0 .OR. jpsto3d > 0 ) THEN

         IF(lwp) THEN
            WRITE(numout,*)
            WRITE(numout,*) 'sto_rst_read : read stochastic parameters from restart file'
            WRITE(numout,*) '~~~~~~~~~~~~'
         ENDIF

         ! Open the restart file
         CALL iom_open( cn_storst_in, numstor )

         if (numstor>0) then   ! if the file does not exist, we do nothing -- stochastic parameters (already created) are not overwritten by restart values
           ! Get stochastic parameters from restart file:
           ! 0D stochastic parameters
           if (lwp) write(numout,*) 'reading sto0d values from restart: ',jpsto0d 
           do jsto = 1 , jpsto0d
             WRITE(clsto0d(7:9),'(i3.3)') jsto
             iomvarid=iom_varid(numstor,clsto0d,ldstop=.FALSE.)
             if (iomvarid.ne.0)  CALL iom_get( numstor, clsto0d , sto0d(jsto) )
           END DO
           ! 2D stochastic parameters
           DO jsto = 1 , jpsto2d
              WRITE(clsto2d(7:9),'(i3.3)') jsto
              iomvarid=iom_varid(numstor,clsto2d,ldstop=.FALSE.)
              if (iomvarid.ne.0) CALL iom_get( numstor, jpdom_auto, clsto2d , sto2d(:,:,jsto) )
           END DO
           ! 3D stochastic parameters
           DO jsto = 1 , jpsto3d
              WRITE(clsto3d(7:9),'(i3.3)') jsto
              iomvarid=iom_varid(numstor,clsto3d,ldstop=.FALSE.)
              if (iomvarid.ne.0) CALL iom_get( numstor, jpdom_auto, clsto3d , sto3d(:,:,:,jsto) )
           END DO

           IF (ln_rstseed) THEN
              ! Get saved state of the random number generator
              idg = MAX( INT(LOG10(REAL(jpnij,wp))) + 1, 4 )        ! how many digits to we need to write? min=4, max=9
              WRITE(clfmt, "('(i', i1, '.', i1, ')')") idg, idg     ! "(ix.x)"
              DO jseed = 1 , 4
                 WRITE(clseed(5:5)      ,'(i1.1)') jseed
                 WRITE(clseed(7:7+idg-1),  clfmt ) narea
                 CALL iom_get( numstor, clseed(1:7+idg-1) , zrseed(jseed) )
              END DO
              ziseed = TRANSFER( zrseed , ziseed)
              CALL kiss_seed( ziseed(1) , ziseed(2) , ziseed(3) , ziseed(4) )
           ENDIF

#ifdef key_top
           if (lwp) write(numout,*) 'loading bamhbi parameter values from restart'
           ! BAMHBI parameters
           if (sto_MaxgrazingrateMesoZoo.ne.0.0)         then
              iomvarid=iom_varid(numstor,'MaxgrazingrateMesoZoo',ldstop=.FALSE.)
              if (iomvarid.ne.0)               CALL iom_get( numstor, 'MaxgrazingrateMesoZoo', MaxgrazingrateMesoZoo )
           end if
           if (sto_alphaPIDiatoms.ne.0.0)                then
              iomvarid=iom_varid(numstor,'alphaPIDiatoms',ldstop=.FALSE.)
              if (iomvarid.ne.0)               CALL iom_get( numstor, 'alphaPIDiatoms', alphaPIDiatoms )
           end if
           if (sto_Ass_Eff_OnCarbon.ne.0.0)              then
              iomvarid=iom_varid(numstor,'Ass_Eff_OnCarbon',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'Ass_Eff_OnCarbon', Ass_Eff_OnCarbon )
           end if
           if (sto_efficiency_growth_MesoZoo.ne.0.0)     then
              iomvarid=iom_varid(numstor,'efficiency_growth_MesoZoo',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'efficiency_growth_MesoZoo', efficiency_growth_MesoZoo )
           end if
           if (sto_nlin_mort_MESozoo.ne.0.0)     then
              iomvarid=iom_varid(numstor,'nlin_mort_MESozoo',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'nlin_mort_MESozoo', nlin_mort_MESozoo )
           end if
           if (sto_HalfSatMort_MesoZoo.ne.0.0)           then
              iomvarid=iom_varid(numstor,'HalfSatMort_MesoZoo',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'HalfSatMort_MesoZoo', HalfSatMort_MesoZoo )
           end if
           if (sto_Capt_eff_Gelatinous_mesozoo.ne.0.0)   then
              iomvarid=iom_varid(numstor,'Capt_eff_Gelatinous_mesozoo',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'Capt_eff_Gelatinous_mesozoo', Capt_eff_Gelatinous_mesozoo )
           end if
           if (sto_MaxgrazingrateGelatinous.ne.0.0)      then
              iomvarid=iom_varid(numstor,'MaxgrazingrateGelatinous',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'MaxgrazingrateGelatinous', MaxgrazingrateGelatinous )
           end if
           if (sto_Capt_eff_MesoZoo_Diatoms.ne.0.0)      then
              iomvarid=iom_varid(numstor,'Capt_eff_MesoZoo_Diatoms',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'Capt_eff_MesoZoo_Diatoms', Capt_eff_MesoZoo_Diatoms )
           end if
           if (sto_Ass_Eff_OnNitrogen.ne.0.0)            then
              iomvarid=iom_varid(numstor,'Ass_Eff_OnNitrogen',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'Ass_Eff_OnNitrogen', Ass_Eff_OnNitrogen )
           end if
           if (sto_MuMaxDiatoms.ne.0.0)                  then
              iomvarid=iom_varid(numstor,'MuMaxDiatoms',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'MuMaxDiatoms', MuMaxDiatoms )
           end if
           if (sto_SiNrDiatoms.ne.0.0)                   then
              iomvarid=iom_varid(numstor,'SiNrDiatoms',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'SiNrDiatoms', SiNrDiatoms )
           end if
           if (sto_MinNCrDiatoms.ne.0.0)                 then
              iomvarid=iom_varid(numstor,'MinNCrDiatoms',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'MinNCrDiatoms', MinNCrDiatoms )
           end if
           if (sto_MortalityDiatoms.ne.0.0)              then
              iomvarid=iom_varid(numstor,'MortalityDiatoms',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'MortalityDiatoms', MortalityDiatoms )
           end if
           if (sto_Messy_feeding_MicroZoo.ne.0.0)        then
              iomvarid=iom_varid(numstor,'Messy_feeding_MicroZoo',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'Messy_feeding_MicroZoo', Messy_feeding_MicroZoo )
           end if
           if (sto_Messy_feeding_MesoZoo.ne.0.0)         then
              iomvarid=iom_varid(numstor,'Messy_feeding_MesoZoo',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'Messy_feeding_MesoZoo', Messy_feeding_MesoZoo )
           end if
           if (sto_SinkingRateDiatomsmin.ne.0.0)         then
              iomvarid=iom_varid(numstor,'SinkingRateDiatomsmin',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'SinkingRateDiatomsmin', SinkingRateDiatomsmin )
           end if
           if (sto_SinkingRateDiatomsmax.ne.0.0)         then
              iomvarid=iom_varid(numstor,'SinkingRateDiatomsmax',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'SinkingRateDiatomsmax', SinkingRateDiatomsmax )
           end if
           if (sto_MortalityFlagellates.ne.0.0)          then
              iomvarid=iom_varid(numstor,'MortalityFlagellates',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'MortalityFlagellates', MortalityFlagellates )
           end if
           if (sto_MortalityEmiliana.ne.0.0)             then
              iomvarid=iom_varid(numstor,'MortalityEmiliana',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'MortalityEmiliana', MortalityEmiliana )
           end if
           if (sto_ksNHs.ne.0.0)                         then
              iomvarid=iom_varid(numstor,'ksNHsEmiliana',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'ksNHsEmiliana', ksNHsEmiliana )
           end if
           if (sto_ksNHs.ne.0.0)                         then
              iomvarid=iom_varid(numstor,'ksNHsFlagellates',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'ksNHsFlagellates', ksNHsFlagellates )
           end if
           if (sto_ksNHs.ne.0.0)                         then
              iomvarid=iom_varid(numstor,'ksNHsDiatoms',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'ksNHsDiatoms', ksNHsDiatoms )
           end if
           if (sto_MuMaxFlagellates.ne.0.0)              then
              iomvarid=iom_varid(numstor,'MuMaxFlagellates',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'MuMaxFlagellates', MuMaxFlagellates )
           end if
           if (sto_MuMaxEmiliana.ne.0.0)                 then
              iomvarid=iom_varid(numstor,'MuMaxEmiliana',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'MuMaxEmiliana', MuMaxEmiliana )
           end if
           if (sto_Q10Phy.ne.0.0)                        then
              iomvarid=iom_varid(numstor,'Q10Phy',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'Q10Phy', Q10Phy )
           end if
           if (sto_Q10PhyDiatoms.ne.0.0)                 then
              iomvarid=iom_varid(numstor,'Q10PhyDiatoms',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'Q10PhyDiatoms', Q10PhyDiatoms )
           end if
           if (sto_Q10Zoo.ne.0.0)                        then
              iomvarid=iom_varid(numstor,'Q10Zoo',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'Q10Zoo', Q10Zoo )
           end if
           if (sto_dzetabio.ne.0.0)                      then
              iomvarid=iom_varid(numstor,'dzetabio',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'dzetabio', dzetabio )
           end if
           if (sto_etabio.ne.0.0)                        then
              iomvarid=iom_varid(numstor,'etabio',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'etabio', etabio )
           end if
           if (sto_lightAbsB.ne.0.0)                     then
              iomvarid=iom_varid(numstor,'LightAbsB',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'LightAbsB', LightAbsB )
           end if
           if (sto_Q10Chem.ne.0.0)                        then
              iomvarid=iom_varid(numstor,'Q10Chem',ldstop=.FALSE.)
              if (iomvarid.ne.0)              CALL iom_get( numstor, 'Q10Chem', Q10Chem )
           end if
#endif
           
           ! Close the restart file
           CALL iom_close( numstor )
        endif
         
      ENDIF

   END SUBROUTINE sto_rst_read


   SUBROUTINE sto_rst_write( kt )
      !!----------------------------------------------------------------------
      !!                  ***  ROUTINE sto_rst_write  ***
      !!
      !! ** Purpose :   write stochastic parameters in restart file
      !!----------------------------------------------------------------------
      INTEGER, INTENT(in) ::   kt     ! ocean time-step
      !!
      INTEGER             ::   jsto, jseed
      INTEGER             ::   idg                 ! number of digits
      INTEGER(KIND=8)     ::   ziseed(4)           ! RNG seeds in integer type
      REAL(KIND=dp)       ::   zrseed(4)           ! RNG seeds in double-precision (with same bits to save in restart)
      CHARACTER(LEN=20)   ::   clkt                ! ocean time-step defined as a character
      CHARACTER(LEN=50)   ::   clname              ! restart file name
      CHARACTER(LEN=9)    ::   clsto0d='sto0d_000' ! stochastic parameter variable name
      CHARACTER(LEN=9)    ::   clsto2d='sto2d_000' ! stochastic parameter variable name
      CHARACTER(LEN=9)    ::   clsto3d='sto3d_000' ! stochastic parameter variable name
      CHARACTER(LEN=15)   ::   clseed='seed0_0000' ! seed variable name
      CHARACTER(LEN=6)    ::   clfmt               ! writing format
      !!----------------------------------------------------------------------
      real(wp) :: temporary

      IF( .NOT. ln_rst_list .AND. nn_stock == -1 ) RETURN   ! we will never do any restart
      
      IF ( &
#ifdef key_top
     &  bamhbi_sto_cpt > 0  .OR.  &
#endif
     &       jpsto0d > 0 .OR. jpsto2d > 0 .OR. jpsto3d > 0 ) THEN

         IF( kt == nitrst .OR. kt == nitend ) THEN
            IF(lwp) THEN
               WRITE(numout,*)
               WRITE(numout,*) 'sto_rst_write : write stochastic parameters in restart file'
               WRITE(numout,*) '~~~~~~~~~~~~~'
            ENDIF
         ENDIF

         ! Put stochastic parameters in restart files
         ! (as opened at previous timestep, see below)
         IF( kt > nit000) THEN
         IF( kt == nitrst .OR. kt == nitend ) THEN
            ! get and save current state of the random number generator
            CALL kiss_state( ziseed(1) , ziseed(2) , ziseed(3) , ziseed(4) )
            zrseed = TRANSFER( ziseed , zrseed)
            idg = MAX( INT(LOG10(REAL(jpnij,wp))) + 1, 4 )        ! how many digits to we need to write? min=4, max=9
            WRITE(clfmt, "('(i', i1, '.', i1, ')')") idg, idg     ! "(ix.x)"
            DO jseed = 1 , 4
               WRITE(clseed(5:5)      ,'(i1.1)') jseed
               WRITE(clseed(7:7+idg-1),  clfmt ) narea
               CALL iom_rstput( kt, nitrst, numstow, clseed(1:7+idg-1), zrseed(jseed) )
            END DO
            ! 0D stochastic parameters
            DO jsto = 1 , jpsto0d
               WRITE(clsto0d(7:9),'(i3.3)') jsto
               temporary=sto0d(jsto)
               CALL iom_rstput( kt, nitrst, numstow, clsto0d , temporary )
            END DO
            ! 2D stochastic parameters
            DO jsto = 1 , jpsto2d
               WRITE(clsto2d(7:9),'(i3.3)') jsto
               CALL iom_rstput( kt, nitrst, numstow, clsto2d , sto2d(:,:,jsto) )
            END DO
            ! 3D stochastic parameters
            DO jsto = 1 , jpsto3d
               WRITE(clsto3d(7:9),'(i3.3)') jsto
               CALL iom_rstput( kt, nitrst, numstow, clsto3d , sto3d(:,:,:,jsto) )
            END DO
#ifdef key_top
            ! BAMHBI parameters
            if (sto_MaxgrazingrateMesoZoo.ne.0.0)         CALL iom_rstput( kt, nitrst, numstow, 'MaxgrazingrateMesoZoo', MaxgrazingrateMesoZoo )
            if (sto_alphaPIDiatoms.ne.0.0)                CALL iom_rstput( kt, nitrst, numstow, 'alphaPIDiatoms', alphaPIDiatoms )
            if (sto_Ass_Eff_OnCarbon.ne.0.0)              CALL iom_rstput( kt, nitrst, numstow, 'Ass_Eff_OnCarbon', Ass_Eff_OnCarbon )
            if (sto_efficiency_growth_MesoZoo.ne.0.0)     CALL iom_rstput( kt, nitrst, numstow, 'efficiency_growth_MesoZoo', efficiency_growth_MesoZoo )
            if (sto_nlin_mort_MESozoo.ne.0.0)             CALL iom_rstput( kt, nitrst, numstow, 'nlin_mort_MESozoo', nlin_mort_MESozoo )
            if (sto_HalfSatMort_MesoZoo.ne.0.0)           CALL iom_rstput( kt, nitrst, numstow, 'HalfSatMort_MesoZoo', HalfSatMort_MesoZoo )
            if (sto_Capt_eff_Gelatinous_mesozoo.ne.0.0)   CALL iom_rstput( kt, nitrst, numstow, 'Capt_eff_Gelatinous_mesozoo', Capt_eff_Gelatinous_mesozoo )
            if (sto_MaxgrazingrateGelatinous.ne.0.0)      CALL iom_rstput( kt, nitrst, numstow, 'MaxgrazingrateGelatinous', MaxgrazingrateGelatinous )
            if (sto_Capt_eff_MesoZoo_Diatoms.ne.0.0)      CALL iom_rstput( kt, nitrst, numstow, 'Capt_eff_MesoZoo_Diatoms', Capt_eff_MesoZoo_Diatoms )
            if (sto_Ass_Eff_OnNitrogen.ne.0.0)            CALL iom_rstput( kt, nitrst, numstow, 'Ass_Eff_OnNitrogen', Ass_Eff_OnNitrogen )
            if (sto_MuMaxDiatoms.ne.0.0)                  CALL iom_rstput( kt, nitrst, numstow, 'MuMaxDiatoms', MuMaxDiatoms )
            if (sto_SiNrDiatoms.ne.0.0)                   CALL iom_rstput( kt, nitrst, numstow, 'SiNrDiatoms', SiNrDiatoms )
            if (sto_MinNCrDiatoms.ne.0.0)                 CALL iom_rstput( kt, nitrst, numstow, 'MinNCrDiatoms', MinNCrDiatoms )
            if (sto_MortalityDiatoms.ne.0.0)              CALL iom_rstput( kt, nitrst, numstow, 'MortalityDiatoms', MortalityDiatoms )
            if (sto_Messy_feeding_MicroZoo.ne.0.0)        CALL iom_rstput( kt, nitrst, numstow, 'Messy_feeding_MicroZoo', Messy_feeding_MicroZoo )
            if (sto_Messy_feeding_MesoZoo.ne.0.0)         CALL iom_rstput( kt, nitrst, numstow, 'Messy_feeding_MesoZoo', Messy_feeding_MesoZoo )
            if (sto_SinkingRateDiatomsmin.ne.0.0)         CALL iom_rstput( kt, nitrst, numstow, 'SinkingRateDiatomsmin', SinkingRateDiatomsmin )
            if (sto_SinkingRateDiatomsmax.ne.0.0)         CALL iom_rstput( kt, nitrst, numstow, 'SinkingRateDiatomsmax', SinkingRateDiatomsmax )
            if (sto_MortalityFlagellates.ne.0.0)          CALL iom_rstput( kt, nitrst, numstow, 'MortalityFlagellates', MortalityFlagellates )
            if (sto_MortalityEmiliana.ne.0.0)             CALL iom_rstput( kt, nitrst, numstow, 'MortalityEmiliana', MortalityEmiliana )
            if (sto_ksNHs.ne.0.0)                         CALL iom_rstput( kt, nitrst, numstow, 'ksNHsEmiliana', ksNHsEmiliana )
            if (sto_ksNHs.ne.0.0)                         CALL iom_rstput( kt, nitrst, numstow, 'ksNHsFlagellates', ksNHsFlagellates )
            if (sto_ksNHs.ne.0.0)                         CALL iom_rstput( kt, nitrst, numstow, 'ksNHsDiatoms', ksNHsDiatoms )
            if (sto_MuMaxFlagellates.ne.0.0)              CALL iom_rstput( kt, nitrst, numstow, 'MuMaxFlagellates', MuMaxFlagellates )
            if (sto_MuMaxEmiliana.ne.0.0)                 CALL iom_rstput( kt, nitrst, numstow, 'MuMaxEmiliana', MuMaxEmiliana )
            if (sto_Q10Phy.ne.0.0)                        CALL iom_rstput( kt, nitrst, numstow, 'Q10Phy', Q10Phy )
            if (sto_Q10PhyDiatoms.ne.0.0)                 CALL iom_rstput( kt, nitrst, numstow, 'Q10PhyDiatoms', Q10PhyDiatoms )
            if (sto_Q10Zoo.ne.0.0)                        CALL iom_rstput( kt, nitrst, numstow, 'Q10Zoo', Q10Zoo )
            if (sto_dzetabio.ne.0.0)                      CALL iom_rstput( kt, nitrst, numstow, 'dzetabio', dzetabio )
            if (sto_etabio.ne.0.0)                        CALL iom_rstput( kt, nitrst, numstow, 'etabio', etabio )
            if (sto_LightAbsB.ne.0.0)                     CALL iom_rstput( kt, nitrst, numstow, 'LightAbsB', LightAbsB )
            if (sto_Q10Chem.ne.0.0)                       CALL iom_rstput( kt, nitrst, numstow, 'Q10Chem', Q10Chem )
#endif
            ! close the restart file
            CALL iom_close( numstow )
         ENDIF
         ENDIF

         ! Open the restart file one timestep before writing restart
         IF( kt < nitend) THEN
         IF( kt == nitrst - 1 .OR. nn_stock == 1 .OR. kt == nitend-1 ) THEN
            ! create the filename
            IF( nitrst > 999999999 ) THEN   ;   WRITE(clkt, *       ) nitrst
            ELSE                            ;   WRITE(clkt, '(i8.8)') nitrst
            ENDIF
            clname = TRIM(cexper)//"_"//TRIM(ADJUSTL(clkt))//"_"//TRIM(cn_storst_out)
            ! print information
            IF(lwp) THEN
               WRITE(numout,*) '             open stochastic parameters restart file: '//clname
               IF( kt == nitrst - 1 ) THEN
                  WRITE(numout,*) '             kt = nitrst - 1 = ', kt
               ELSE
                  WRITE(numout,*) '             kt = '             , kt
               ENDIF
            ENDIF
            ! open the restart file
            CALL iom_open( clname, numstow, ldwrt = .TRUE. )
         ENDIF
         ENDIF

      ENDIF

   END SUBROUTINE sto_rst_write


   SUBROUTINE sto_par_white( psto )
      !!----------------------------------------------------------------------
      !!                  ***  ROUTINE sto_par_white  ***
      !!
      !! ** Purpose :   fill input array with white Gaussian noise
      !!----------------------------------------------------------------------
      REAL(wp), DIMENSION(jpi,jpj), INTENT(out)           ::   psto
      !!
      INTEGER  :: ji, jj
      REAL(wp) :: gran   ! Gaussian random number (forced KIND=8 as in kiss_gaussian)

      DO_2D( 1, 1, 1, 1 )
         CALL kiss_gaussian( gran )
         psto(ji,jj) = gran
      END_2D

   END SUBROUTINE sto_par_white


   SUBROUTINE sto_par_flt( psto )
      !!----------------------------------------------------------------------
      !!                  ***  ROUTINE sto_par_flt  ***
      !!
      !! ** Purpose :   apply horizontal Laplacian filter to input array
      !!----------------------------------------------------------------------
      REAL(wp), DIMENSION(jpi,jpj), INTENT(out)           ::   psto
      !!
      INTEGER  :: ji, jj

      DO_2D( 0, 0, 0, 0 )
         psto(ji,jj) = 0.5_wp * psto(ji,jj) + 0.125_wp * &
                           &  ( psto(ji-1,jj) + psto(ji+1,jj) +  &
                           &    psto(ji,jj-1) + psto(ji,jj+1) )
      END_2D

   END SUBROUTINE sto_par_flt


   FUNCTION sto_par_flt_fac( kpasses )
      !!----------------------------------------------------------------------
      !!                  ***  FUNCTION sto_par_flt_fac  ***
      !!
      !! ** Purpose :   compute factor to restore standard deviation
      !!                as a function of the number of passes
      !!                of the Laplacian filter
      !!----------------------------------------------------------------------
      INTEGER, INTENT(in) :: kpasses
      REAL(wp) :: sto_par_flt_fac
      !!
      INTEGER :: jpasses, ji, jj, jflti, jfltj
      INTEGER, DIMENSION(-1:1,-1:1) :: pflt0
      REAL(wp), DIMENSION(:,:), ALLOCATABLE :: pfltb
      REAL(wp), DIMENSION(:,:), ALLOCATABLE :: pflta
      REAL(wp) :: ratio

      pflt0(-1,-1) = 0 ; pflt0(-1,0) = 1 ; pflt0(-1,1) = 0
      pflt0( 0,-1) = 1 ; pflt0( 0,0) = 4 ; pflt0( 0,1) = 1
      pflt0( 1,-1) = 0 ; pflt0( 1,0) = 1 ; pflt0( 1,1) = 0

      ALLOCATE(pfltb(-kpasses-1:kpasses+1,-kpasses-1:kpasses+1))
      ALLOCATE(pflta(-kpasses-1:kpasses+1,-kpasses-1:kpasses+1))

      pfltb(:,:) = 0
      pfltb(0,0) = 1
      DO jpasses = 1, kpasses
        pflta(:,:) = 0
        DO jflti= -1, 1
        DO jfltj= -1, 1
          DO ji= -kpasses, kpasses
          DO jj= -kpasses, kpasses
            pflta(ji,jj) = pflta(ji,jj) + pfltb(ji+jflti,jj+jfltj) * pflt0(jflti,jfltj)
          ENDDO
          ENDDO
        ENDDO
        ENDDO
        pfltb(:,:) = pflta(:,:)
      ENDDO

      ratio = SUM(pfltb(:,:))
      ratio = ratio * ratio / SUM(pfltb(:,:)*pfltb(:,:))
      ratio = SQRT(ratio)

      DEALLOCATE(pfltb,pflta)

      sto_par_flt_fac = ratio

   END FUNCTION sto_par_flt_fac

   SUBROUTINE sto_par_xygau( jsto_xygau, psto )
      !!----------------------------------------------------------------------
      !!                  ***  ROUTINE sto_par_xygau  ***
      !! 
      !! ** Purpose : fill input array with horiz. 2D Gaussian structures;
      !!              the elliptic 2D Gaussian function with diagonal
      !!              covariance matrix [ stdx^2   0 
      !!                                    0    stdy^2 ] is expressed as:
      !!
      !!                  0  = ( 1 / ( 2 * pi * stdx * stdy ) ) * 
      !!                       EXP( -0.5 * ( ( ( x - xo )^2 / stdx^2 ) + 
      !!                                     ( ( y - yo )^2 / stdy^2 ) ) )
      !!
      !!              where in general stdx /= stdy; the Gaussian function
      !!              used here is an Apodization function (i.e. a tappering/
      !!              filtering method) where the amplitude in each xo, yo
      !!              center position ( 1 / ( 2 * pi * stdx * stdy ) )
      !!              is replaced by a random Gaussian number with N(0,1)
      !!
      !! ** Method  : (1) create 1D array of global positions, scales and 
      !!                  amplitudes on zero proc
      !!              (2) broadcast to the other procs
      !!                  (mpi_barrier, mpi_bcast)
      !!              (3) apply perturbation in local using global positions;
      !!                  we solve and sum the elliptic 2D Gaussian equation
      !!                  for several positions in order to create a complex
      !!                  pattern of perturbations (i.e. modes); the method
      !!                  introduces modal patterns with variable/anisotropic
      !!                  correlation length in x,y-axis (eliminating 
      !!                  cross-land perturbations is under development) 
      !!              (4) normalize with global mean & std
      !!                  (mpi_allreduce)
      !!              (5) the final horiz. 2D Gaus. stochastic field w,
      !!                  for AR processes, is declared in the Eq.:
      !!                  X(t) = a * X(t-1) + b * w + c
      !!
      !! ** Note    : (1) do not allocate 2D global array for computational
      !!                  time and resources issues
      !!----------------------------------------------------------------------
      REAL(wp)    , DIMENSION(jpi,jpj), INTENT(out) :: psto           ! local stochastic field
      REAL(wp)    , DIMENSION(:)      , ALLOCATABLE :: xsto_o, ysto_o ! global center of each mode
      REAL(wp)    , DIMENSION(:)      , ALLOCATABLE :: l_stdx, l_stdy ! horiz. length of perturbations
      REAL(KIND=8), DIMENSION(:)      , ALLOCATABLE :: psto_ampl      ! amplitude of perturbations
      REAL(KIND=8)         :: ampl   ! Gaussian random number N(0,1) (forced KIND=8 as in kiss_gaussian)
      REAL(KIND=8)         :: u1, u2 ! uniform  random numbers [0 1] (forced KIND=8 as in kiss_uniform)
      INTEGER              :: ji, jj, jn    ! dummy loop indices
      INTEGER              :: nmod          ! number of modal patterns
      INTEGER              :: min_nmod = 2  ! constrain the minimum number of modes; e.g. 1(2,3...) => uni(bi,tri...)modal
      INTEGER, INTENT(in)  :: jsto_xygau    ! perturbed variable index; declare "radius in grid points" of horizontal 2D Gaussian
                                            ! structures, with 1std of the total length and no truncation: sto2d_xycor(jsto)
                                            ! Other variables can be declared also, e.g:
                                            ! "grid type, sign": sto2d_typ(jsto), sto2d_sgn(jsto); 
                                            ! "limitation factor": sto2d_lim(jsto);
                                            ! For 3D stochastic fields we implement 2D stochastic fields in each level, e.g:
                                            ! sto2d_xycor(jpsto2d+jsto), sto3d_typ(jsto), sto3d_sgn(jsto), sto3d_lim(jsto)
      INTEGER :: nn_ens_size=1    ! LUC ADDED THIS WHILE WAITING FOR THE ENSEMBLE CAPABILITY
#if ! defined key_mpi_off
      INTEGER  :: ierr, localcomm
      REAL(wp) :: mean_glo, std_glo, mean_loc, std_loc ! scalar values used in normalization
      ! choose communicator to control the mean & std for each member or for the ensemble
      localcomm = mpi_comm_oce ! communicator per member
!      localcomm = mpi_comm_all ! communicator of all members
#endif

      ! Calculate number of modal patterns
      nmod = INT(     FLOAT(jpiglo * jpjglo) / &
                 MIN((FLOAT(jpiglo * jpjglo) / FLOAT(min_nmod)), rpi * sto2d_xycor(jsto_xygau)**2) )
!      ! print stochastic diagnostics in ocean.output (note: walltime consuming print)
!      IF(lwp) THEN
!            WRITE(numout,*)
!            WRITE(numout,*) 'sto_par_xygau : start solving the elliptic 2D Gaussian function'
!            WRITE(numout,*) '~~~~~~~~~~~~~'
!            WRITE(numout,*)
!            WRITE(numout,*) ' -- total number of modes in global domain -- '
!            WRITE(numout,*) '    nmod = ', nmod
!      ENDIF
      ! random positions in global where we apply the horiz. 2D Gaussian structures;
      ALLOCATE( xsto_o(nmod) )
      ALLOCATE( ysto_o(nmod) )
      ! variable/anisotropic lengths of modal patterns in x,y-axis by 50%
      ALLOCATE( l_stdx(nmod) )
      ALLOCATE( l_stdy(nmod) )
      ! amplitude in each center xsto_o, ysto_o of the modes in the global positions
      ALLOCATE( psto_ampl(nmod) )
      IF ( mpprank == 0 ) THEN ! calculate 1D global values on zero proc and broadcast to the other procs
         DO jn = 1, nmod
            ! uniformly placed modes in global domain
            CALL kiss_uniform(u1)
            CALL kiss_uniform(u2)
            xsto_o(jn) = u1 * ( FLOAT(jpiglo) - 1._wp ) + 1._wp
            ysto_o(jn) = u2 * ( FLOAT(jpjglo) - 1._wp ) + 1._wp
            ! variable/anisotropic lengths in x,y-axis = uniform_random * sqrt(2) * radius(1std of the total length)
            CALL kiss_uniform(u1)
            CALL kiss_uniform(u2)
            l_stdx(jn) = ( .5_wp + u1 ) * ( SQRT(2._wp) * &
                       MIN(SQRT((FLOAT(jpiglo * jpjglo) / FLOAT(min_nmod)) / rpi), sto2d_xycor(jsto_xygau)) ) ! constrained radius
            l_stdy(jn) = ( .5_wp + u2 ) * ( SQRT(2._wp) * &
                       MIN(SQRT((FLOAT(jpiglo * jpjglo) / FLOAT(min_nmod)) / rpi), sto2d_xycor(jsto_xygau)) ) ! constrained radius
            ! Gaussian N(0,1) amplitudes in xsto_o, ysto_o (tappering/filtering method)
            CALL kiss_gaussian(ampl)
            IF ( ampl .GT. 3._wp ) THEN ! strictly constrain amplitude distribution between [-3 +3]
               DO WHILE ( ampl .LE. 3._wp )
                  CALL kiss_gaussian(ampl)
               END DO
            ENDIF
            psto_ampl(jn) = ampl
!            ! print stochastic diagnostics in ocean.output (note: walltime consuming print)
!            IF(lwp) THEN
!               WRITE(numout,*) ' -- specific position in global domain -- '
!               WRITE(numout,*) '    jn = ', jn
!               WRITE(numout,*) ' -- uniformly placed position in global domain -- '
!               WRITE(numout,*) '    xsto_o = ', xsto_o(jn)
!               WRITE(numout,*) '    ysto_o = ', ysto_o(jn)
!               WRITE(numout,*) ' -- variable/anisotropic length of perturbation in x,y-axis -- '
!               WRITE(numout,*) '    l_stdx = ', l_stdx(jn)
!               WRITE(numout,*) '    l_stdy = ', l_stdy(jn)
!               WRITE(numout,*) ' -- Gaussian N(0,1) perturbation amplitude in center position xsto_o, ysto_o -- '
!               WRITE(numout,*) '    psto_ampl = ', psto_ampl(jn)
!            ENDIF
         END DO
      ENDIF
#if ! defined key_mpi_off
      CALL mpi_barrier( localcomm, ierr ) ! wait to broadcast from zero proc to the other procs
      CALL   mpi_bcast(    xsto_o(1:nmod), nmod, mpi_double_precision, 0, localcomm, ierr)
      CALL   mpi_bcast(    ysto_o(1:nmod), nmod, mpi_double_precision, 0, localcomm, ierr)
      CALL   mpi_bcast(    l_stdx(1:nmod), nmod, mpi_double_precision, 0, localcomm, ierr)
      CALL   mpi_bcast(    l_stdy(1:nmod), nmod, mpi_double_precision, 0, localcomm, ierr)
      CALL   mpi_bcast( psto_ampl(1:nmod), nmod, mpi_double_precision, 0, localcomm, ierr)
#endif
      !
      ! apply perturbations in local with horiz. 2D Gaussian structures in global positions
      psto(:,:) = 0._wp ! necessary to initialize before the sum of all modes
      DO jj = 2, jpj-1
         DO ji = 2, jpi-1
            DO jn = 1, nmod ! the sum loop for all modes in global positions creates a complex pattern
                  psto(ji,jj) = psto(ji,jj) + psto_ampl(jn) * EXP( -1._wp * ( & 
                          ( ( FLOAT(mig(ji)) - xsto_o(jn) )**2 / l_stdx(jn)**2 ) + &
                          ( ( FLOAT(mjg(jj)) - ysto_o(jn) )**2 / l_stdy(jn)**2 ) ) )
            END DO
         END DO
      END DO
#if ! defined key_mpi_off
      ! local /= global
      ! calculate the global mean
      mean_loc = 0._wp
      DO jj = 2, jpj-1
         DO ji = 2, jpi-1
            mean_loc = mean_loc + psto(ji,jj)
         END DO
      END DO
      mean_loc = mean_loc / FLOAT((jpi - 2) * (jpj - 2))
      CALL mpi_allreduce(mean_loc, mean_glo, 1, mpi_double_precision, mpi_sum, localcomm, ierr)
      IF     ( localcomm == mpi_comm_oce ) THEN
         mean_glo = mean_glo / FLOAT(jpnij)
      ELSE !IF ( localcomm == mpi_comm_all ) THEN
         mean_glo = mean_glo / FLOAT(jpnij * nn_ens_size)
      ENDIF
      ! calculate the global std
      std_loc = 0._wp
      DO jj = 2, jpj-1
         DO ji = 2, jpi-1
            std_loc = std_loc + (psto(ji,jj) - mean_glo)**2
         END DO
      END DO
      std_loc = std_loc / FLOAT((jpi - 2) * (jpj - 2))
      CALL mpi_allreduce(std_loc, std_glo, 1, mpi_double_precision, mpi_sum, localcomm, ierr)
      IF     ( localcomm == mpi_comm_oce ) THEN
         std_glo = std_glo / FLOAT(jpnij)
      ELSE !IF ( localcomm == mpi_comm_all ) THEN
         std_glo = std_glo / FLOAT(jpnij * nn_ens_size)
      ENDIF
      std_glo = sqrt(std_glo)
#else
      ! local = global (not implemented for ensembles)
#endif
      ! normalize the local stochastic field using the global mean & std
      DO jj = 2, jpj-1
         DO ji = 2, jpi-1
            psto(ji,jj) = psto(ji,jj) - mean_glo                          ! default value: sto2d_lim(jsto) = 3._wp
            psto(ji,jj) = psto(ji,jj) / std_glo                           ! later/outside subroutine: cutoff between [-sto2d_lim +sto2d_lim]
            IF ( jsto_xygau > jpsto2d ) THEN                              ! now/inside subroutine:    smooth between [-sto2d_lim +sto2d_lim]
               psto(ji,jj) = psto(ji,jj) * ( sto3d_lim(jsto_xygau-jpsto2d) / 3._wp )
            ELSE
               psto(ji,jj) = psto(ji,jj) * ( sto2d_lim(jsto_xygau)         / 3._wp )
            ENDIF
         END DO
      END DO
      DEALLOCATE( psto_ampl, l_stdx, l_stdy, xsto_o, ysto_o )
!      ! print stochastic diagnostics in ocean.output (note: walltime consuming print)
!      IF(lwp) THEN
!            WRITE(numout,*)
!            WRITE(numout,*) 'end solving the elliptic 2D Gaussian function'
!            WRITE(numout,*)
!      ENDIF

   END SUBROUTINE sto_par_xygau

   SUBROUTINE sto_share_0d ( )
#if ! defined key_mpi_off
     INTEGER  :: ierr, localcomm
     ! choose communicator to control the mean & std for each member or for the ensemble
     localcomm = mpi_comm_oce ! communicator per member
     !localcomm = mpi_comm_all ! communicator of all members
     CALL mpi_barrier( localcomm, ierr ) ! wait to broadcast from zero proc to the other procs
     CALL   mpi_bcast( sto0d(1:jpsto0d), jpsto0d, mpi_double_precision, 0, localcomm, ierr)
#endif
    END SUBROUTINE sto_share_0d


END MODULE stopar

