MODULE diaccc
   !!======================================================================
   !!                       ***  MODULE  diaccc  ***
   !!======================================================================

   !!----------------------------------------------------------------------
   !!   dia_ccc       : Diagnose the conservation of Cold intermediate layer Cold Content
   !!   dia_ccc_init  : Initialization of the conservation diagnostic
   !!----------------------------------------------------------------------
   USE oce             ! ocean dynamics and tracers
   USE dom_oce         ! ocean space and time domain
   USE phycst          ! physical constants
   USE sbc_oce         ! surface thermohaline fluxes
   USE domvvl          ! vertical scale factors
   !
   USE iom             ! I/O manager
   USE in_out_manager  ! I/O manager
   USE lib_fortran     ! glob_sum
   USE lib_mpp         ! distributed memory computing library
   USE eosbn2

   IMPLICIT NONE
   PRIVATE

   PUBLIC   dia_ccc        ! routine called by step.F90
   PUBLIC   dia_ccc_init   ! routine called by nemogcm

   REAL(wp), DIMENSION(:,:)  , ALLOCATABLE ::   ccc, mld_bs, phi_t, phi_s
   
   !! * Substitutions
#  include "do_loop_substitute.h90"
#  include "domzgr_substitute.h90"

   !!----------------------------------------------------------------------
   !! NEMO/OPA 4.2 , L.V.
   !! $Id$: diaccc.F90 2023-05-18
   !! Software governed by the CeCILL licence     (NEMOGCM/NEMO_CeCILL.txt)
   !!----------------------------------------------------------------------
CONTAINS

   SUBROUTINE dia_ccc( kt , Kmm )
      !!---------------------------------------------------------------------------
      !!                  ***  ROUTINE dia_ccc  ***
      !!     
      !! ** Purpose: Compute the ocean Cold Intermediate Water cold content
      !!                     the MLD according to Black Sea method
      !!                     PHI-T and PHI-S diagnostic
      !!	
      !!---------------------------------------------------------------------------
      INTEGER, INTENT(in) ::   kt   ! ocean time-step index
      INTEGER, INTENT(in) ::   Kmm  ! ocean time level index
      !
      INTEGER    ::   ji, jj, jk                  ! dummy loop indice
      REAL(wp)   ::   rho_10m, a
      REAL(wp)   ::   ijdepth,tbarre,sbarre,rhoTbarre,rhoSbarre,rhoTbarreSbarre
      !!---------------------------------------------------------------------------
      !
        if (iom_use("rho")) CALL iom_put(   'rho'   , rhop )
        if (iom_use("CILCC")) then
        ccc=0.0_wp
        DO_2D_OVR( nn_hls, nn_hls, nn_hls, nn_hls )
            ijdepth=gdepw(ji,jj,mbkt(ji,jj),Kmm)
            if ( ijdepth > 50._wp ) then
               do jk = 1, jpk
                if ( tmask(ji,jj,jk).eq.1 .and. rhop(ji,jj,jk).ge.1014._wp .and. ts(ji,jj,jk,jp_tem,Kmm).lt.8.35_wp ) &
                  ccc(ji,jj)=ccc(ji,jj) + rhop(ji,jj,jk)*rcp*(8.35_wp-ts(ji,jj,jk,jp_tem,Kmm))*e3t(ji,jj,jk,Kmm)
              end do
            end if
        END_2D
        CALL iom_put(   'CILCC' , ccc  )
        end if
        !
        if (iom_use("mld_bs")) then
        mld_bs=0.0_wp
        DO_2D_OVR( nn_hls, nn_hls, nn_hls, nn_hls )
            if ((tmask(ji,jj,1).eq.1).and.gdept(ji,jj,mbkt(ji,jj),Kmm)>10._wp) then
              jk=1   ! assume uppermost layer depth(ji,jj,1) < 10 meter !
              do while (gdept(ji,jj,jk+1,Kmm)<10 .and. tmask(ji,jj,jk+1).eq.1)
                 jk=jk+1
              end do
              rho_10m = ( (gdept(ji,jj,jk+1,Kmm)-10)*rhop(ji,jj,jk) + (10-gdept(ji,jj,jk,Kmm))*rhop(ji,jj,jk+1) ) / ( gdept(ji,jj,jk+1,Kmm)-gdept(ji,jj,jk,Kmm) )
              !jk=0 ! don't start from surface, start from 10m
              do while ( (tmask(ji,jj,jk+1).eq.1) .and. ((rhop(ji,jj,jk+1)-rho_10m)<0.125) )
                jk=jk+1
              end do
              mld_bs(ji,jj) = 0.5*(gdept(ji,jj,jk,Kmm)+gdept(ji,jj,jk+1,Kmm))
              ! mld_bs(ji,jj) = ( (rhop(ji,jj,jk+1)-rho_10m)*gdept_0(ji,jj,jk) +(rho_10m-rhop(ji,jj,jk))*gdept_0(ji,jj,jk+1) ) / (rhop(ji,jj,jk+1)-rhop(ji,jj,jk))
            !else mld value stays at 0.0
            end if
        END_2D
        CALL iom_put(   'mld_bs' , mld_bs  )
        end if
      !
      ! Stratification metrics like Capet 2013 - Black Sea Hypoxia paper (based on potential energy anomaly)
      if (iom_use("phi_t") .OR. iom_use("phi_s")) then
        phi_t=0.0_wp ; phi_s=0.0_wp
        DO_2D_OVR( nn_hls, nn_hls, nn_hls, nn_hls )
            ijdepth = gdepw(ji,jj,mbkt(ji,jj)+1,Kmm) ! depth of current column
            tbarre=0.0_wp ; sbarre=0.0_wp
            do jk = 1, mbkt(ji,jj)
              tbarre = tbarre + e3t(ji,jj,jk,Kmm) * ts(ji,jj,jk,jp_tem,Kmm)
              sbarre = sbarre + e3t(ji,jj,jk,Kmm) * ts(ji,jj,jk,jp_sal,Kmm)
            end do
            tbarre = tbarre / ijdepth
            sbarre = sbarre / ijdepth
            do jk = 1, mbkt(ji,jj)
              call eos(tbarre,sbarre,gdept(ji,jj,jk,Kmm),rhoTbarreSbarre)
              call eos(ts(ji,jj,jk,jp_tem,Kmm),sbarre,gdept(ji,jj,jk,Kmm),rhoSbarre)
              phi_t(ji,jj) = phi_t(ji,jj) + gdept(ji,jj,jk,Kmm) * ( rhoSbarre-rhoTbarreSbarre ) * e3t(ji,jj,jk,Kmm) 
              call eos(tbarre,ts(ji,jj,jk,jp_sal,Kmm),gdept(ji,jj,jk,Kmm),rhoTbarre)
              phi_s(ji,jj) = phi_s(ji,jj) + gdept(ji,jj,jk,Kmm) * ( rhoTbarre-rhoTbarreSbarre) * e3t(ji,jj,jk,Kmm)
              !at first sight equation differs from capet 2013, but it is effectively the same equation (two equivalent ways of expressing phi_s)
            end do
            phi_t(ji,jj) = grav / ijdepth * phi_t(ji,jj) 
            phi_s(ji,jj) = grav / ijdepth * phi_s(ji,jj)
        END_2D
        call iom_put( 'phi_t' , phi_t )
        call iom_put( 'phi_s' , phi_s )
      end if

   END SUBROUTINE dia_ccc


   SUBROUTINE dia_ccc_init
      !!---------------------------------------------------------------------------
      !!                  ***  ROUTINE dia_ccc  ***
      !!     
      !! ** Purpose: Initialization for the CCC and stratification metrics
      !! Computation only performed if corresponding fields in file_def-top included
      !!	
      !!---------------------------------------------------------------------------
      INTEGER ::   ierror   ! local integer
      INTEGER ::   ios
      !
      !!----------------------------------------------------------------------
      !
      ierror=0
      IF (iom_use("CILCC")) THEN
      ALLOCATE( ccc(jpi,jpj), mld_bs(jpi,jpj), STAT=ierror )
      IF( ierror > 0 ) THEN
         CALL ctl_stop( 'dia_ccc: unable to allocate memory' )
         RETURN
      ENDIF
      ccc(:,:) = 0._wp
      mld_bs(:,:) = 0._wp
      ENDIF
      IF (iom_use("phi_t") .OR. iom_use("phi_s")) THEN
      ALLOCATE( phi_t(jpi,jpj), phi_s(jpi,jpj), STAT=ierror )
      IF( ierror > 0 ) THEN
         CALL ctl_stop( 'dia_ccc: unable to allocate memory' )
         RETURN
      ENDIF
      phi_t(:,:) = 0._wp
      phi_s(:,:) = 0._wp
      ENDIF
   END SUBROUTINE dia_ccc_init

   !!======================================================================
END MODULE diaccc
